/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_test_ird_fixtures.h"
#include "sph_ird_instrument_model.h"
#include "sph_ird_distortion_map.h"
#include "sph_ird_tags.h"
#include "sph_utils.h"
#include "sph_ird_keywords.h"
#include "sph_common_keywords.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

static void cutest_sph_ird_distortion_map_test_verify(const double[],
                                                      const double[]);

static
void cutest_sph_ird_distortion_map_test_it(const cpl_polynomial* polyx,
                                           const cpl_polynomial* polyy);


static cpl_frame*
sph_test_ird_fixtures_create_dist_map_raw_frame(const cpl_polynomial*,
                                                const cpl_polynomial*,
                                                sph_ird_instrument_model*,
                                                int, double, cpl_frame**)
    CPL_ATTR_ALLOC;

static
void cutest_sph_ird_distortion_map_test_it(const cpl_polynomial* polyx,
                                           const cpl_polynomial* polyy) {
    const int ndet = 1024;
    cpl_frame* rawframe = NULL;
    cpl_frameset* rawframeset = NULL;
    sph_ird_instrument_model* model = NULL;
    sph_distortion_model * distmodel;
    cpl_polynomial* distpolyx;
    cpl_polynomial* distpolyy;
    cpl_parameterlist* parlist = NULL;
    sph_ird_distortion_map* recipe = NULL;
    cpl_frame* point_pattern_frame = NULL;
    cpl_propertylist* plist = NULL;
    int ivalue;
    int order;
    double dvalue, rcoeff;
    cpl_size pows[2];
    cpl_error_code code;
    cpl_boolean has_2d_residual = CPL_TRUE;

    rawframeset = cpl_frameset_new();
    model = sph_test_ird_fixtures_instrument_model(2 * ndet, ndet);
    cpl_test_nonnull( model );
    rawframe = sph_test_ird_fixtures_create_dist_map_raw_frame(polyx, polyy,
                                                               model, 63, 2.0,
            &point_pattern_frame);
    cpl_test_nonnull( rawframe );
    cpl_frameset_insert(rawframeset, rawframe);
    cpl_frameset_insert(rawframeset, point_pattern_frame);

    parlist = sph_ird_distortion_map_create_paramlist();
    recipe = sph_ird_distortion_map_new(rawframeset, parlist);
    //recipe->polyfit_order = 2;
    recipe->full_qc = CPL_TRUE;

    order = recipe->polyfit_order;

    cpl_test_eq_error(sph_ird_distortion_map_run(recipe), CPL_ERROR_NONE);
    sph_ird_instrument_model_delete(model);
    sph_ird_distortion_map_delete(recipe);
    cpl_frameset_delete(rawframeset);
    cpl_parameterlist_delete(parlist);

    distmodel =
        sph_distortion_model_load("distortion_map.fits", 0,
                                             SPH_IRD_KEYWORD_DISTMAP_LEFT_COEFFX,
                                             SPH_IRD_KEYWORD_DISTMAP_LEFT_COEFFY);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(distmodel);

    distpolyx = sph_distortion_model_get_poly_x(distmodel);
    cpl_test_error(CPL_ERROR_NONE);

    distpolyy = sph_distortion_model_get_poly_y(distmodel);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull(distpolyx);
    cpl_test_nonnull(distpolyy);

    cpl_test_eq(cpl_polynomial_get_dimension(distpolyx), 2);
    cpl_test_eq(cpl_polynomial_get_dimension(distpolyy), 2);

    cpl_test_eq(cpl_polynomial_get_degree(distpolyx), order);
    cpl_test_eq(cpl_polynomial_get_degree(distpolyy), order);

    code = cpl_polynomial_subtract(distpolyx, polyx, distpolyx);
    if (code == CPL_ERROR_UNSUPPORTED_MODE) {
        cpl_test_error(CPL_ERROR_UNSUPPORTED_MODE);
        cpl_msg_warning(cpl_func, "2D-polynomial residual not (yet) supported "
                        "(by CPL ver. %s)", cpl_version_get_version());

        has_2d_residual = CPL_FALSE;

    } else {
        cpl_polynomial* zerodist = cpl_polynomial_new(2);
        cpl_vector* vxy = cpl_vector_new(2);
        double* dvxy = cpl_vector_get_data(vxy);
        const double ndetc = ndet / 2.0 + 0.5;
        double maxresidx = 0.0;
        double maxresidy = 0.0;
        int imaxresidx = 0;
        int imaxresidy = 0;

        code = cpl_polynomial_subtract(distpolyy, polyy, distpolyy);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        cpl_test_polynomial_abs(distpolyx, zerodist, 0.02);
        cpl_test_polynomial_abs(distpolyy, zerodist, 0.005);

        if (cpl_test_get_failed() || cpl_msg_get_level() <= CPL_MSG_INFO) {
            cpl_polynomial_dump(polyx, stderr);
            cpl_polynomial_dump(distpolyx, stderr);
            cpl_polynomial_dump(polyy, stderr);
            cpl_polynomial_dump(distpolyy, stderr);
        }

        for (int i = 0; i < ndet; i++) {
            dvxy[0] = i - ndetc;
            for (int j = 0; j < ndet; j++) {
                double dx, dy;

                dvxy[1] = j - ndetc;

                dx = cpl_polynomial_eval(distpolyx, vxy);
                dy = cpl_polynomial_eval(distpolyy, vxy);

                if (fabs(dx) > fabs(maxresidx)) {
                    maxresidx = dx;
                    imaxresidx = i;
                }
                if (fabs(dy) > fabs(maxresidy)) {
                    maxresidy = dy;
                    imaxresidy = j;
                }
            }
            cpl_msg_info(cpl_func, "Maximum distortion X-residual (X=%d/%d) "
                         "[pixel]: %g", 1 + imaxresidx, ndet, maxresidx);
            cpl_msg_info(cpl_func, "Maximum distortion Y-residual (Y=%d/%d)"
                         "[pixel]: %g", 1 + imaxresidy, ndet, maxresidy);
        }

        cpl_vector_delete(vxy);
        cpl_polynomial_delete(zerodist);
    }

    /* Verify certain, individual coefficients */

    pows[0] = 0;
    pows[1] = 0;
    rcoeff = cpl_polynomial_get_coeff(distpolyx, pows);
    dvalue = cpl_polynomial_get_coeff(polyx,     pows);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(dvalue, -rcoeff, has_2d_residual ? 1e-4 : 5e-3);

    rcoeff = cpl_polynomial_get_coeff(distpolyy, pows);
    dvalue = cpl_polynomial_get_coeff(polyy,     pows);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(dvalue, -rcoeff, has_2d_residual ? 3e-5 : 8e-3);

    pows[0] = 1;
    pows[1] = 0;
    rcoeff = cpl_polynomial_get_coeff(distpolyx, pows);
    dvalue = cpl_polynomial_get_coeff(polyx,     pows);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(dvalue, -rcoeff, 0.04);

    pows[0] = 0;
    pows[1] = 1;
    rcoeff = cpl_polynomial_get_coeff(distpolyy, pows);
    dvalue = cpl_polynomial_get_coeff(polyy,     pows);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(dvalue, -rcoeff, 0.01);

    sph_distortion_model_delete(distmodel);

    /* Verify other parts of product */
    plist = cpl_propertylist_load("distortion_map.fits", 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( plist );

    ivalue = cpl_propertylist_has(plist,"EXTNAME");
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_eq(ivalue, 1);
    if (ivalue == 1) {
        const char * cvalue = cpl_propertylist_get_string(plist,"EXTNAME");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(cvalue);
        cpl_test_zero(strncmp(cvalue, "LEFT.DISTX.IMAGE", 10));
    }
    cpl_propertylist_delete(plist);
    plist = NULL;

    plist = cpl_propertylist_load("distortion_map.fits", 4);
    cpl_test_nonnull( plist );
    ivalue = cpl_propertylist_has(plist,"EXTNAME");
    cpl_test_error(CPL_ERROR_NONE);

    if (ivalue == 1) {
        const char * cvalue = cpl_propertylist_get_string(plist,"EXTNAME");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(cvalue);
        cpl_test_zero(strncmp(cvalue, "LEFT.DISTY.IMAGE", 10));
    }
    cpl_propertylist_delete(plist);

    plist = cpl_propertylist_load("distortion_map.fits", 8);
    cpl_test_nonnull( plist );
    ivalue = cpl_propertylist_has(plist,"EXTNAME");
    cpl_test_error(CPL_ERROR_NONE);
    if (ivalue == 1) {
        const char * cvalue = cpl_propertylist_get_string(plist,"EXTNAME");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(cvalue);
        cpl_test_zero(strncmp(cvalue, "RIGHT.DISTX.IMAGE", 10));
    }
    cpl_propertylist_delete(plist);

    return;
}

static
void cutest_sph_ird_distortion_map_verify_result(const cpl_polynomial* polyx,
                                                 const cpl_polynomial* polyy) {
    cpl_frame* rawframe = NULL;
    cpl_frameset* rawframeset = NULL;
    sph_ird_instrument_model* model = NULL;
    cpl_parameterlist* parlist = NULL;
    sph_ird_distortion_map* recipe = NULL;
    cpl_frame* point_pattern_frame = NULL;
    cpl_frame* dummyframe = NULL;
    cpl_propertylist* plist = NULL;
    cpl_image* distimx = NULL;
    cpl_image* distimy = NULL;
    cpl_error_code code;
    double dvalue, rcoeff;
    cpl_size pows[2];
    cpl_image* qc_image;

    plist = cpl_propertylist_load("distortion_map.fits", 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(plist);
    qc_image = cpl_image_load("sph_ird_distortion_map_qc.fits",
                              CPL_TYPE_UNSPECIFIED, 0, 4);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(qc_image);
    code = cpl_image_save(qc_image, "qc_image.fits", CPL_TYPE_UNSPECIFIED,
                          plist, CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_image_delete(qc_image);
    cpl_propertylist_delete(plist);

    model = sph_test_ird_fixtures_instrument_model(2048, 1024);
    cpl_test_nonnull( model );
    // Just need the point pattern here
    dummyframe = sph_test_ird_fixtures_create_dist_map_raw_frame(polyx, polyy,
                                                                 model, 63, 2.0,
            &point_pattern_frame);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(dummyframe);
    cpl_frame_delete(dummyframe);

    rawframeset = cpl_frameset_new();
    cpl_test_nonnull( rawframeset );
    rawframe = cpl_frame_new();
    cpl_test_nonnull( rawframe );
    code = cpl_frame_set_filename(rawframe, "qc_image.fits");
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_frame_set_group(rawframe, CPL_FRAME_GROUP_RAW);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_frame_set_tag(rawframe, SPH_IRD_TAG_DISTORTION_MAP_RAW);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_frameset_insert(rawframeset, rawframe);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_frameset_insert(rawframeset, point_pattern_frame);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    parlist = sph_ird_distortion_map_create_paramlist();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( parlist );
    recipe = sph_ird_distortion_map_new(rawframeset, parlist);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( recipe );
    recipe->polyfit_order = 2;
    code = sph_ird_distortion_map_run(recipe);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    plist = cpl_propertylist_load("distortion_map.fits", 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( plist );

    dvalue = cpl_propertylist_get_double(plist,
                                         SPH_IRD_KEYWORD_DISTMAP_LEFT_COEFFX
                                         " 0_0");
    cpl_test_error(CPL_ERROR_NONE);
    pows[0] = 0; pows[1] = 0;
    rcoeff = cpl_polynomial_get_coeff(polyx, pows);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(dvalue, -rcoeff, 1e-7);

    dvalue = cpl_propertylist_get_double(plist,
                                         SPH_IRD_KEYWORD_DISTMAP_LEFT_COEFFX
                                         " 1_0");
    cpl_test_error(CPL_ERROR_NONE);
    pows[0] = 1; pows[1] = 0;
    rcoeff = cpl_polynomial_get_coeff(polyx, pows);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(dvalue, -rcoeff, 0.01);

    dvalue = cpl_propertylist_get_double(plist,
                                         SPH_IRD_KEYWORD_DISTMAP_LEFT_COEFFX
                                         " 0_1");
    cpl_test_error(CPL_ERROR_NONE);
    pows[0] = 0; pows[1] = 1;
    rcoeff = cpl_polynomial_get_coeff(polyx, pows);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(dvalue, -rcoeff, 0.004);

    distimx = cpl_image_load("distortion_map.fits", CPL_TYPE_DOUBLE, 0, 0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( distimx );

    distimy = cpl_image_load("distortion_map.fits", CPL_TYPE_DOUBLE, 0, 4);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( distimy );

    cpl_test_abs( cpl_image_get_min(distimx), 0.0, 0.015);
    cpl_test_abs( cpl_image_get_max(distimx), 0.0, 0.015);
    cpl_test_abs( cpl_image_get_min(distimy), 0.0, 0.03);
    cpl_test_abs( cpl_image_get_min(distimy), 0.0, 0.03);

    cpl_image_delete(distimx);
    distimx = NULL;
    cpl_image_delete(distimy);
    distimy = NULL;

    distimx = cpl_image_load("distortion_map.fits", CPL_TYPE_DOUBLE, 0, 8);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( distimx );

    distimy = cpl_image_load("distortion_map.fits", CPL_TYPE_DOUBLE, 0, 12);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( distimy );

    cpl_test_abs( cpl_image_get_min(distimx), 0.0, 0.01);
    cpl_test_abs( cpl_image_get_max(distimx), 0.0, 0.01);
    cpl_test_abs( cpl_image_get_min(distimy), 0.0, 0.02);
    cpl_test_abs( cpl_image_get_min(distimy), 0.0, 0.02);

    cpl_image_delete(distimx);
    distimx = NULL;
    cpl_image_delete(distimy);
    distimy = NULL;

    cpl_propertylist_delete(plist);
    plist = NULL;

    sph_ird_instrument_model_delete(model);
    sph_ird_distortion_map_delete(recipe);
    cpl_frameset_delete(rawframeset);
    cpl_parameterlist_delete(parlist);
    return;
}

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_ird_distortion_map_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_ird_distortion_map_clean_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;

    const double xcoeffs1[] = {0.0, -0.00178,
                              0.00765, -2.002e-7,
                              0.0, 0.0};

    const double ycoeffs1[] = {0.0, 0.001658,
                              0.001536, -1.499e-6,
                              0.0, 0.0};

    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("A cutest_sph_ird_distortion_map testsuite",
            cutest_sph_ird_distortion_map_init_testsuite,
            cutest_sph_ird_distortion_map_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }

    if (cpl_version_get_major() > 7 ||
        (cpl_version_get_major() == 7 && (cpl_version_get_minor() > 1 ||
                                          (cpl_version_get_minor() == 1 &&
                                           cpl_version_get_micro() > 0)))) {
        cutest_sph_ird_distortion_map_test_verify(xcoeffs1, ycoeffs1);
    } else {
        cpl_msg_warning(cpl_func, "Skipping tests with too old CPL: %s",
                        cpl_version_get_version());
    }

    return sph_test_end();
}

static cpl_frame*
sph_test_ird_fixtures_create_dist_map_raw_frame(const cpl_polynomial* polyx,
                                                const cpl_polynomial* polyy,
                                                sph_ird_instrument_model* model,
        int points_per_side, double fwhm, cpl_frame** pp_frame) {
    cpl_frame* result = NULL;
    int pix = 0;
    int piy = 0;
    sph_test_ngc_ir_simulator* ngc = NULL;
    cpl_image* im = NULL;
    double deltax = 0.0;
    double deltay = 0.0;
    const double xc = 0.5 + 0.25 * model->detsize_pixels_x;
    const double yc = 0.5 + 0.50 * model->detsize_pixels_y;
    cpl_image* rawim = NULL;
    FILE* fp = NULL;
    char* ppname = NULL;
    cpl_vector* vxy = cpl_vector_new(2);

    ngc = sph_test_ngc_ir_simulator_new(model->detsize_pixels_x,
            model->detsize_pixels_y / 2);
    ngc->mean_bias = ngc->mean_dark = ngc->sigma_bias = ngc->sigma_dark =
            ngc->ron = 0.0;

    result = sph_filemanager_create_temp_frame("DISTMAPTEST",
            SPH_IRD_TAG_DISTORTION_MAP_RAW, CPL_FRAME_GROUP_RAW);
    cpl_test_error(CPL_ERROR_NONE);

    im = sph_test_image_tools_create_flat_image_double(
            model->detsize_pixels_x / 2, model->detsize_pixels_y, 0.0);
    cpl_test_lt(255, cpl_image_get_size_x(im));
    cpl_test_lt(255, cpl_image_get_size_y(im));

    cpl_test_eq(2 * (int)(xc - 0.5), cpl_image_get_size_x(im));
    cpl_test_eq(2 * (int)(yc - 0.5), cpl_image_get_size_y(im));
    deltax = (cpl_image_get_size_x(im) - 50.0) / points_per_side;
    deltay = (cpl_image_get_size_y(im) - 50.0) / points_per_side;

    if (pp_frame) {
        *pp_frame = sph_filemanager_create_temp_frame("DISTMAPTEST_PP",
                SPH_IRD_TAG_POINT_PATTERN, CPL_FRAME_GROUP_CALIB);
        ppname = sph_filemanager_filename_new_ext(
                cpl_frame_get_filename(*pp_frame), "txt");

        cpl_frame_set_filename(*pp_frame, ppname);
        cpl_free(ppname);
        ppname = NULL;
        fp = fopen(cpl_frame_get_filename(*pp_frame), "w");
    }
    for (pix = -points_per_side/2; pix <= points_per_side/2; pix++) {
        for (piy = -points_per_side/2; piy <= points_per_side/2; piy++) {
            /* Ensure that central mask is centered on detector */
            const double x = xc + pix * deltax;
            const double y = yc + piy * deltay;
            double dx, dy;

            cpl_vector_set(vxy, 0, x - xc);
            cpl_vector_set(vxy, 1, y - yc);
            dx = cpl_polynomial_eval(polyx, vxy);
            dy = cpl_polynomial_eval(polyy, vxy);
            if (fp) {
                fprintf(fp, "%10.3f     %10.3f\n", x - xc , y - yc);
            }
            sph_test_image_tools_add_gauss(im, x + dx, y + dy, 2.0, 1000000.0);
        }
    }

    rawim = sph_ird_instrument_model_assemble_image(model, im, im);
    cpl_image_add_scalar(rawim, 100.0);
    sph_test_image_tools_apply_poisson_noise(rawim, NULL);
    cpl_msg_info(cpl_func, "Saving distorted (raw) image (w. pixel-range %g < "
                 "%g): %s", cpl_image_get_min(rawim), cpl_image_get_max(rawim),
                 cpl_frame_get_filename(result));
    cpl_image_save(rawim, cpl_frame_get_filename(result), CPL_BPP_32_SIGNED,
            NULL, CPL_IO_CREATE);
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(rawim);
    rawim = NULL;

    sph_test_ngc_ir_simulator_delete(ngc);
    ngc = NULL;

    if (fp) {
        fclose(fp);
        fp = NULL;
    }

    cpl_vector_delete(vxy);

    return result;
}

static void cutest_sph_ird_distortion_map_test_verify(const double xcoeffs[],
                                                      const double ycoeffs[])
{

    cpl_polynomial* polyx = cpl_polynomial_new(2);
    cpl_polynomial* polyy = cpl_polynomial_new(2);
    cpl_size pows[2];

    pows[0] = 0; pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, xcoeffs[0]);
    cpl_polynomial_set_coeff(polyy, pows, ycoeffs[0]);
    pows[0] = 0; pows[1] = 1;
    cpl_polynomial_set_coeff(polyx, pows, xcoeffs[1]);
    cpl_polynomial_set_coeff(polyy, pows, ycoeffs[1]);
    pows[0] = 1; pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, xcoeffs[2]);
    cpl_polynomial_set_coeff(polyy, pows, ycoeffs[2]);
    pows[0] = 1; pows[1] = 1;
    cpl_polynomial_set_coeff(polyx, pows, xcoeffs[3]);
    cpl_polynomial_set_coeff(polyy, pows, ycoeffs[3]);
    pows[0] = 0; pows[1] = 2;
    cpl_polynomial_set_coeff(polyx, pows, xcoeffs[4]);
    cpl_polynomial_set_coeff(polyy, pows, ycoeffs[4]);
    pows[0] = 2; pows[1] = 0;
    cpl_polynomial_set_coeff(polyx, pows, xcoeffs[5]);
    cpl_polynomial_set_coeff(polyy, pows, ycoeffs[5]);

    cutest_sph_ird_distortion_map_test_it(polyx, polyy);
    cutest_sph_ird_distortion_map_verify_result(polyx, polyy);

    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);


}


/**@}*/
