/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_ird_tags.h"
#include "sph_ird_common_science.h"
#include "sph_ird_star_center.h"
#include "sph_ird_star_center.c"
#include "sph_ird_star_center_run.c"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

static
int cutest_init_testsuite(void);
static
int cutest_clean_testsuite(void);
static
void cutest_sph_ird_star_center_run_no_sources(void);
static
void cutest_sph_ird_star_center_get_center_wbpix(void);
static
void cutest_sph_ird_star_center_get_center_with_bright_center(void);
static
void cutest_sph_ird_star_center_run_coords(void);
static
void cutest_sph_ird_star_center_table_output_format(void);
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testsuite for star center recipe",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing no sources found",
                    cutest_sph_ird_star_center_run_no_sources)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing masking of bright center",
                    cutest_sph_ird_star_center_get_center_with_bright_center)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing correct coordinate system",
                    cutest_sph_ird_star_center_run_coords)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing effect of badpixels",
                    cutest_sph_ird_star_center_get_center_wbpix)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing correct output format",
                    cutest_sph_ird_star_center_table_output_format)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_star_center_run_no_sources(void) {
    cpl_image* im = NULL;
    cpl_frame* aframe = NULL;
    sph_ird_star_center* recipe = NULL;
    cpl_frameset* inframes = NULL;
    cpl_parameterlist* inparams = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    im = sph_test_image_tools_create_flat_image_double(2048, 1024, 100.0);
    cpl_test_nonnull( im );
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    aframe = sph_test_frame_image(im, CPL_BPP_32_SIGNED,
            SPH_IRD_TAG_STAR_CENTER_RAW, CPL_FRAME_GROUP_RAW);
    sph_test_frame_add_key_double(aframe, 0, "MJD-OBS", 12223.0003);
    sph_test_frame_add_key_double(aframe, 0, "EXPTIME", 2.0);
    cpl_image_delete(im);
    im = NULL;
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, cpl_frame_duplicate(aframe));
    cpl_frame_set_tag(aframe, SPH_IRD_TAG_WAFFLE_IMAGE_RAW);
    cpl_frameset_insert(inframes, aframe);

    dark = sph_master_frame_new(2048, 1024);
    flat = sph_master_frame_new(2048, 1024);
    cpl_image_add_scalar(flat->image, 1.0);
    cpl_image_add_scalar(dark->image, 10.0);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER",
            SPH_IRD_TAG_DARK_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(dark, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER",
            SPH_IRD_TAG_FLAT_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(flat, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);

    inparams = sph_ird_star_center_create_paramlist();
    recipe = sph_ird_star_center_new(inframes, inparams);
    cpl_test_nonnull( recipe );
    cpl_test_noneq( sph_ird_star_center_run(recipe), CPL_ERROR_NONE);
    cpl_error_reset();
    sph_ird_star_center_delete(recipe);
    recipe = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    return;
}

static
void cutest_sph_ird_star_center_get_center_wbpix(void) {
    cpl_image* im = NULL;
    cpl_frame* aframe = NULL;
    sph_ird_star_center* recipe = NULL;
    cpl_frameset* inframes = NULL;
    cpl_parameterlist* inparams = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_fctable* fctab = NULL;
    char* fname = NULL;
    char* fname1 = NULL;
    double bgvalue = 10.0;
    double peakvalue = 35000.0;
    double vpeak = 0.0;

    im = sph_test_image_tools_create_flat_image_double(2048, 1024, 0.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im,
            cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    vpeak = cpl_image_get_max(im);
    cpl_image_multiply_scalar(im, peakvalue / vpeak);
    cpl_image_add_scalar(im, bgvalue);

    cpl_test_nonnull( im );
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    cpl_image_set(im, 511, 511, -100.0);
    cpl_image_set(im, 515, 512, 10000.0);
    aframe = sph_test_frame_image(im, CPL_BPP_32_SIGNED,
            SPH_IRD_TAG_WAFFLE_IMAGE_RAW, CPL_FRAME_GROUP_RAW);
    sph_test_frame_add_key_string(aframe, 0, "DET FRAM UTC",
            "2013-05-18T13:43:21");

    sph_test_frame_add_key_double(aframe, 0, "EXPTIME", 2.0);
    cpl_image_delete(im);
    im = NULL;
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, aframe);

    dark = sph_master_frame_new(2048, 1024);
    flat = sph_master_frame_new(2048, 1024);
    cpl_image_add_scalar(flat->image, 1.0);
    cpl_image_add_scalar(dark->image, 10.0);
    cpl_image_set(dark->badpixelmap, 511, 511, 1);
    cpl_image_set(dark->badpixelmap, 515, 512, 1);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_DARK",
            SPH_IRD_TAG_DARK_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(dark, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_FLAT",
            SPH_IRD_TAG_FLAT_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(flat, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);

    inparams = sph_ird_star_center_create_paramlist();
    recipe = sph_ird_star_center_new(inframes, inparams);
    cpl_test_nonnull( recipe );
    cpl_test_eq_error(sph_ird_star_center_run(recipe), CPL_ERROR_NONE);
    cpl_error_reset();
    fname1 = sph_filemanager_new_filename_from_base(
            cpl_frame_get_filename(cpl_frameset_get_position(inframes, 0)),
            "fctable");
    fctab = sph_fctable_load_fits_IRDIS("star_center.fits");
    cpl_test_nonnull( fctab );
    cpl_test_eq(sph_fctable_get_size(fctab), 1);
    cpl_test_abs(cpl_vector_get(fctab->centresL_x,0), 512, 0.01);
    cpl_test_abs(cpl_vector_get(fctab->centresL_y,0), 512, 0.01);
    cpl_test_abs(cpl_vector_get(fctab->centresR_x,0), 512, 0.01);
    cpl_test_abs(cpl_vector_get(fctab->centresR_y,0), 512, 0.01);
    sph_ird_star_center_delete(recipe);
    recipe = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_free(fname);
    fname = NULL;
    cpl_free(fname1);
    fname1 = NULL;
    return;
}

static
void cutest_sph_ird_star_center_get_center_with_bright_center(void) {
    cpl_image* im = NULL;
    cpl_frame* aframe = NULL;
    sph_ird_star_center* recipe = NULL;
    cpl_frameset* inframes = NULL;
    cpl_parameterlist* inparams = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_fctable* fctab = NULL;
    char* fname = NULL;
    char* fname1 = NULL;
    double bgvalue = 10.0;
    double peakvalue = 35000.0;
    double vpeak = 0.0;
    cpl_image* mask_image = NULL;
    int xx = 0;
    int yy = 0;
    im = sph_test_image_tools_create_flat_image_double(2048, 1024, 0.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 20.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 - 19.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 20.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 - 19.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im,
            cpl_image_get_size_y(im) / 2.0 + 1024.0 + 20.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im,
            cpl_image_get_size_y(im) / 2.0 + 1024.0 - 19.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im,
            cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 20.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im,
            cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 - 19.5, 4.0, 1000000.0);

    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 3.5,
            cpl_image_get_size_y(im) / 2.0 + 4.5, 2.0, 1000000.0);
    sph_test_image_tools_add_gauss(im,
            cpl_image_get_size_y(im) / 2.0 + 1024.0 + 3.5,
            cpl_image_get_size_y(im) / 2.0 + 1.5, 2.0, 1000000.0);

    vpeak = cpl_image_get_max(im);
    cpl_image_multiply_scalar(im, peakvalue / vpeak);
    cpl_image_add_scalar(im, bgvalue);

    cpl_test_nonnull( im );
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    aframe = sph_test_frame_image(im, CPL_BPP_32_SIGNED,
            SPH_IRD_TAG_WAFFLE_IMAGE_RAW, CPL_FRAME_GROUP_RAW);
    sph_test_frame_add_key_string(aframe, 0, "DET FRAM UTC",
            "2013-05-18T13:43:21");
    sph_test_frame_add_key_double(aframe, 0, "EXPTIME", 2.0);
    cpl_image_delete(im);
    im = NULL;
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, aframe);

    dark = sph_master_frame_new(2048, 1024);
    flat = sph_master_frame_new(2048, 1024);
    cpl_image_add_scalar(flat->image, 1.0);
    cpl_image_add_scalar(dark->image, 10.0);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_DARK",
            SPH_IRD_TAG_DARK_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(dark, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_FLAT",
            SPH_IRD_TAG_FLAT_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(flat, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);

    mask_image = cpl_image_new(2048, 1024, CPL_TYPE_INT);
    for (xx = 0; xx < 20; ++xx) {
        for (yy = 0; yy < 20; ++yy) {
            cpl_image_set(mask_image, xx + 513 - 10, yy + 513 - 10, 1.0);
            cpl_image_set(mask_image, xx + 513 + 1024 - 10, yy + 513 - 10, 1.0);
        }
    }
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_MASK",
            SPH_IRD_TAG_STATIC_WAFFLEMAP_CALIB, CPL_FRAME_GROUP_CALIB);
    cpl_image_save(mask_image, cpl_frame_get_filename(aframe), CPL_TYPE_INT,
            NULL, CPL_IO_CREATE);
    cpl_frameset_insert(inframes, aframe);

    inparams = sph_ird_star_center_create_paramlist();
    recipe = sph_ird_star_center_new(inframes, inparams);
    cpl_test_nonnull( recipe );
    cpl_test_eq_error(sph_ird_star_center_run(recipe), CPL_ERROR_NONE);
    cpl_error_reset();
    fname1 = sph_filemanager_new_filename_from_base(
            cpl_frame_get_filename(cpl_frameset_get_position(inframes, 0)),
            "00000");
    fname = sph_filemanager_filename_new_ext(fname1, "fits");
    fctab = sph_fctable_load_fits_IRDIS("star_center.fits");
    cpl_test_nonnull( fctab );
    cpl_test_eq(sph_fctable_get_size(fctab), 1);
    cpl_test_abs(cpl_vector_get(fctab->centresL_x,0), 512, 0.01);
    cpl_test_abs(cpl_vector_get(fctab->centresL_y,0), 512, 0.01);
    cpl_test_abs(cpl_vector_get(fctab->centresR_x,0), 512, 0.01);
    cpl_test_abs(cpl_vector_get(fctab->centresR_y,0), 512, 0.01);
    sph_ird_star_center_delete(recipe);
    recipe = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    cpl_image_delete(mask_image);
    mask_image = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_free(fname);
    fname = NULL;
    cpl_free(fname1);
    fname1 = NULL;
    return;
}

static
void cutest_sph_ird_star_center_run_coords(void) {
    cpl_image* im = NULL;
    cpl_frame* aframe = NULL;
    sph_ird_star_center* recipe = NULL;
    cpl_frameset* inframes = NULL;
    cpl_parameterlist* inparams = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_fctable* fctab = NULL;
    char* fname = NULL;
    char* fname1 = NULL;
    double bgvalue = 10.0;
    double peakvalue = 35000.0;
    double vpeak = 0.0;

    cpl_test_error(CPL_ERROR_NONE);
    im = sph_test_image_tools_create_flat_image_double(2048, 1024, 0.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im,
            cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    vpeak = cpl_image_get_max(im);
    cpl_image_multiply_scalar(im, peakvalue / vpeak);
    cpl_image_add_scalar(im, bgvalue);

    cpl_test_nonnull( im );
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    aframe = sph_test_frame_image(im, CPL_BPP_32_SIGNED,
            SPH_IRD_TAG_WAFFLE_IMAGE_RAW, CPL_FRAME_GROUP_RAW);
    sph_test_frame_add_key_string(aframe, 0, "DET FRAM UTC",
            "2013-05-18T13:43:21");
    sph_test_frame_add_key_double(aframe, 0, "EXPTIME", 2.0);

    cpl_image_delete(im);
    im = NULL;
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, aframe);

    dark = sph_master_frame_new(2048, 1024);
    flat = sph_master_frame_new(2048, 1024);
    cpl_image_add_scalar(flat->image, 1.0);
    cpl_image_add_scalar(dark->image, 10.0);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_DARK",
            SPH_IRD_TAG_DARK_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(dark, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_FLAT",
            SPH_IRD_TAG_FLAT_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(flat, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);

    inparams = sph_ird_star_center_create_paramlist();
    recipe = sph_ird_star_center_new(inframes, inparams);
    cpl_test_nonnull( recipe );
    recipe->tot_fctab = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq_error(sph_ird_star_center_run(recipe), CPL_ERROR_NONE);
    cpl_error_reset();
    fname1 = sph_filemanager_new_filename_from_base(
            cpl_frame_get_filename(cpl_frameset_get_position(inframes, 0)),
            "fctable");
    fctab = sph_fctable_load_fits_IRDIS("star_center.fits");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( fctab );
    cpl_test_eq(sph_fctable_get_size(fctab), 1);
    cpl_test_abs(cpl_vector_get(fctab->centresL_x,0), 512, 0.01);
    cpl_test_abs(cpl_vector_get(fctab->centresL_y,0), 512, 0.01);
    cpl_test_abs(cpl_vector_get(fctab->centresR_x,0), 512, 0.01);
    cpl_test_abs(cpl_vector_get(fctab->centresR_y,0), 512, 0.01);
    sph_ird_star_center_delete(recipe);
    recipe = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_free(fname);
    fname = NULL;
    cpl_free(fname1);
    fname1 = NULL;
    return;
}

static
void cutest_sph_ird_star_center_table_output_format(void) {
    cpl_image* im = NULL;
    cpl_frame* aframe = NULL;
    sph_ird_star_center* recipe = NULL;
    cpl_frameset* inframes = NULL;
    cpl_parameterlist* inparams = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_fctable* fctab = NULL;
    char* fname = NULL;
    char* fname1 = NULL;
    double bgvalue = 10.0;
    double peakvalue = 35000.0;
    double vpeak = 0.0;
    cpl_table* ftab = NULL;

    cpl_test_error(CPL_ERROR_NONE);
    im = sph_test_image_tools_create_flat_image_double(2048, 1024, 0.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    sph_test_image_tools_add_gauss(im,
            cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 4.0, 1000000.0);
    vpeak = cpl_image_get_max(im);
    cpl_image_multiply_scalar(im, peakvalue / vpeak);
    cpl_image_add_scalar(im, bgvalue);

    cpl_test_nonnull( im );
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    aframe = sph_test_frame_image(im, CPL_BPP_32_SIGNED,
            SPH_IRD_TAG_WAFFLE_IMAGE_RAW, CPL_FRAME_GROUP_RAW);
    sph_test_frame_add_key_string(aframe, 0, "DET FRAM UTC",
            "2013-05-18T13:43:21");
    sph_test_frame_add_key_double(aframe, 0, "EXPTIME", 2.0);

    cpl_image_delete(im);
    im = NULL;
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, aframe);

    dark = sph_master_frame_new(2048, 1024);
    flat = sph_master_frame_new(2048, 1024);
    cpl_image_add_scalar(flat->image, 1.0);
    cpl_image_add_scalar(dark->image, 10.0);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_DARK",
            SPH_IRD_TAG_DARK_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(dark, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_FLAT",
            SPH_IRD_TAG_FLAT_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(flat, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);

    inparams = sph_ird_star_center_create_paramlist();
    recipe = sph_ird_star_center_new(inframes, inparams);
    cpl_test_nonnull( recipe );
    recipe->tot_fctab = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq_error(sph_ird_star_center_run(recipe), CPL_ERROR_NONE);
    cpl_error_reset();
    fname1 = sph_filemanager_new_filename_from_base(
            cpl_frame_get_filename(cpl_frameset_get_position(inframes, 0)),
            "fctable");
    ftab = cpl_table_load("star_center.fits", 1, 0);
    cpl_test_assert(ftab);
    cpl_test_error(CPL_ERROR_NONE);
    if (ftab) {
        cpl_test_eq(cpl_table_get_ncol(ftab), 7);
        if (cpl_table_get_ncol(ftab) == 7) {
            cpl_test_eq_string(cpl_table_get_column_name(ftab), "TIME_UT");
            cpl_test_eq_string(cpl_table_get_column_name(NULL),
                    "CENTRE LEFT X");
            cpl_test_eq_string(cpl_table_get_column_name(NULL),
                    "CENTRE LEFT Y");
            cpl_test_eq_string(cpl_table_get_column_name(NULL),
                    "CENTRE RIGHT X");
            cpl_test_eq_string(cpl_table_get_column_name(NULL),
                    "CENTRE RIGHT Y");
            cpl_test_eq_string(cpl_table_get_column_name(NULL),
                    "DMS POS X");
            cpl_test_eq_string(cpl_table_get_column_name(NULL),
                    "DMS POS Y");
            cpl_test_eq(cpl_table_get_column_type(ftab, "TIME_UT"),
                    CPL_TYPE_STRING);
            cpl_test_eq(cpl_table_get_column_type(ftab, "CENTRE LEFT X"),
                    CPL_TYPE_DOUBLE);
            cpl_test_eq(cpl_table_get_column_type(ftab, "CENTRE LEFT Y"),
                    CPL_TYPE_DOUBLE);
            cpl_test_eq(cpl_table_get_column_type(ftab, "CENTRE RIGHT X"),
                    CPL_TYPE_DOUBLE);
            cpl_test_eq(cpl_table_get_column_type(ftab, "CENTRE RIGHT Y"),
                    CPL_TYPE_DOUBLE);
            cpl_test_eq(cpl_table_get_column_type(ftab, "DMS POS X"),
                    CPL_TYPE_DOUBLE);
            cpl_test_eq(cpl_table_get_column_type(ftab, "DMS POS Y"),
                    CPL_TYPE_DOUBLE);
        }
        cpl_table_delete(ftab);
        ftab = NULL;
    }
    sph_ird_star_center_delete(recipe);
    recipe = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_free(fname);
    fname = NULL;
    cpl_free(fname1);
    fname1 = NULL;
    return;
}

/**@}*/
