/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "cutest_zpl_intensity_flat_imaging.h"
#include "sph_zpl_intensity_flat_imaging.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_intensity_flat_imaging_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     *--------------------------------------------------------------------*/
    sph_test_nop_code();
    return CPL_ERROR_NONE;
}

static int cutest_clean_intensity_flat_imaging_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat_imaging
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_intensity_flat_imaging_run_1(void) {
    sph_zpl_intensity_flat_imaging* zpl_intensity_flat_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube2 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 6;
    //int                         		nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         		ny                  = 100; // -//-
    double dV = 100.0;
    double image_odd = 2500.0;
    double image_even = 2000.0;
    cpl_propertylist* pl = NULL;
    cpl_parameter* parameter_collapse = NULL;

    inparams = sph_zpl_intensity_flat_imaging_create_paramlist();
    //set "zpl.intensity_flat.collapse" parameter to FALSE
    parameter_collapse = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat_imaging.collapse");
    cpl_parameter_set_bool(parameter_collapse, 0);

    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_imaging_cube_set(&zexpcube, image_odd,
            image_even, dV, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_IFF_IMAGING_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);

    rerr = cpl_frameset_insert(inframes, zexpcube);

    sph_test_simulator_create_zplpreproc_imaging_cube_set(&zexpcube2, image_odd,
            image_even, dV, nframes);
    cpl_frame_set_tag(zexpcube2, SPH_ZPL_TAG_IFF_IMAGING_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube2, CPL_FRAME_GROUP_RAW);

    rerr = cpl_frameset_insert(inframes, zexpcube2);

    //printf("raw frame: %s\n", cpl_frame_get_filename(cpl_frameset_get_first( zpl_intensity_flat->rawframes )));
    //printf("rawframes #: %d\n", cpl_frameset_get_size( zpl_intensity_flat->rawframes ));

    zpl_intensity_flat_imaging = sph_zpl_intensity_flat_imaging_new(inframes,
            inparams);
    cpl_test_nonnull( zpl_intensity_flat_imaging );

    rerr = sph_zpl_intensity_flat_imaging_run(zpl_intensity_flat_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_intensity_flat_imaging->intensity_flat_imaging_outfilename_cam1,
            0);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, pframe_names_leakage.bad_pixels_name),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            1.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, pframe_names_leakage.mean_name),
            1.0, 1e-10);

    cpl_propertylist_delete(pl);

    pl = cpl_propertylist_load(
            zpl_intensity_flat_imaging->intensity_flat_imaging_outfilename_cam2,
            0);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, pframe_names_leakage.bad_pixels_name),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            1.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, pframe_names_leakage.mean_name),
            1.0, 1e-10);

    cpl_propertylist_delete(pl);

    rerr = sph_zpl_intensity_flat_imaging_delete(zpl_intensity_flat_imaging);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat_imaging with bias and dark
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_intensity_flat_imaging_run_2(void) {
    sph_zpl_intensity_flat_imaging* zpl_intensity_flat_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* master_bias_imaging_frame = NULL;
    cpl_frame* master_dark_imaging_frame = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 6;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-
    double dV = 100.0;
    double image_odd = 2500.0;
    double image_even = 2000.0;
    cpl_propertylist* pl = NULL;

    double iframe_bias = 4.0;
    double pframe_bias = 1.0;
    cpl_parameter* parameter_collapse = NULL;

    inparams = sph_zpl_intensity_flat_imaging_create_paramlist();
    //set "zpl.intensity_flat.collapse" parameter to FALSE
    parameter_collapse = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat_imaging.collapse");
    cpl_parameter_set_bool(parameter_collapse, 0);

    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_imaging_cube_set(&zexpcube, image_odd,
            image_even, dV, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_IFF_IMAGING_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube);

    //create master bias imaging (double image frame)
    master_bias_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_fi_master_bias.fits.tmp", nx, ny, iframe_bias,
                    pframe_bias);
    cpl_frame_set_tag(master_bias_imaging_frame,
            SPH_ZPL_TAG_BIAS_IMAGING_CALIB_CAM1);
    cpl_frame_set_group(master_bias_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_bias_imaging_frame);

    //create master dark imaging (double image frame)
    master_dark_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_fi_master_dark.fits.tmp", nx, ny, iframe_bias + 10.,
                    pframe_bias + 10.);
    cpl_frame_set_tag(master_dark_imaging_frame,
            SPH_ZPL_TAG_DARK_IMAGING_CALIB_CAM1);
    cpl_frame_set_group(master_dark_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_dark_imaging_frame);

    //printf("raw frame: %s\n", cpl_frame_get_filename(cpl_frameset_get_first( zpl_intensity_flat->rawframes )));
    //printf("rawframes #: %d\n", cpl_frameset_get_size( zpl_intensity_flat->rawframes ));

    zpl_intensity_flat_imaging = sph_zpl_intensity_flat_imaging_new(inframes,
            inparams);
    cpl_test_nonnull( zpl_intensity_flat_imaging );

    rerr = sph_zpl_intensity_flat_imaging_run(zpl_intensity_flat_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_intensity_flat_imaging->intensity_flat_imaging_outfilename_cam1,
            0);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, pframe_names_leakage.bad_pixels_name),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            1.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, pframe_names_leakage.mean_name),
            1.0, 1e-10);

    cpl_propertylist_delete(pl);
    rerr = sph_zpl_intensity_flat_imaging_delete(zpl_intensity_flat_imaging);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat_imaging
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_intensity_flat_imaging_run_3(void) {
    sph_zpl_intensity_flat_imaging* zpl_intensity_flat_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 6;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-
    double dV = 100.0;
    double image_odd = 2500.0;
    double image_even = 2000.0;
    cpl_propertylist* pl = NULL;
    cpl_parameter* parameter_collapse = NULL;
    cpl_parameter* parameter_sigma_clip = NULL;
    cpl_parameter* keep_intermediate = NULL;

    cpl_frame* master_bias_imaging_frame = NULL;
    cpl_frame* master_dark_imaging_frame = NULL;
    double iframe_bias = 4.0;
    double pframe_bias = 1.0;

    char outfilename[256];

    inparams = sph_zpl_intensity_flat_imaging_create_paramlist();

    //set "zpl.intensity_flat.collapse" parameter to TRUE
    parameter_collapse = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat_imaging.collapse");
    cpl_parameter_set_bool(parameter_collapse, 1);
    parameter_sigma_clip = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat_imaging.sigma_clip");
    cpl_parameter_set_double(parameter_sigma_clip, 3.0);
    keep_intermediate = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat_imaging.keep_intermediate");
    cpl_parameter_set_bool(keep_intermediate, 1);

    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_imaging_cube_set(&zexpcube, image_odd,
            image_even, dV, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_IFF_IMAGING_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);

    rerr = cpl_frameset_insert(inframes, zexpcube);

    //create master bias imaging (double image frame)
    master_bias_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_fi_master_bias.fits.tmp", nx, ny, iframe_bias,
                    pframe_bias);
    cpl_frame_set_tag(master_bias_imaging_frame,
            SPH_ZPL_TAG_BIAS_IMAGING_CALIB_CAM2);
    cpl_frame_set_group(master_bias_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_bias_imaging_frame);

    //create master dark imaging (double image frame)
    master_dark_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_fi_master_dark.fits.tmp", nx, ny, iframe_bias + 10.,
                    pframe_bias + 10.);
    cpl_frame_set_tag(master_dark_imaging_frame,
            SPH_ZPL_TAG_DARK_IMAGING_CALIB_CAM2);
    cpl_frame_set_group(master_dark_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_dark_imaging_frame);

    //printf("raw frame: %s\n", cpl_frame_get_filename(cpl_frameset_get_first( zpl_intensity_flat->rawframes )));
    //printf("rawframes #: %d\n", cpl_frameset_get_size( zpl_intensity_flat->rawframes ));

    zpl_intensity_flat_imaging = sph_zpl_intensity_flat_imaging_new(inframes,
            inparams);
    cpl_test_nonnull( zpl_intensity_flat_imaging );

    rerr = sph_zpl_intensity_flat_imaging_run(zpl_intensity_flat_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    sprintf(
            outfilename,
            "%s-qc-non-normalized.fits",
            zpl_intensity_flat_imaging->intensity_flat_imaging_outfilename_cam2);

    pl = cpl_propertylist_load(outfilename, 0);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, pframe_names_leakage.bad_pixels_name),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            2750.0-2*iframe_bias-10, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, pframe_names_leakage.mean_name),
            2250.0-2*pframe_bias-10, 1e-10);

    cpl_propertylist_delete(pl);

    pl = cpl_propertylist_load(
            zpl_intensity_flat_imaging->intensity_flat_imaging_outfilename_cam2,
            0);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, pframe_names_leakage.bad_pixels_name),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            1, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, pframe_names_leakage.mean_name),
            1, 1e-10);

    cpl_propertylist_delete(pl);
    rerr = sph_zpl_intensity_flat_imaging_delete(zpl_intensity_flat_imaging);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_intensity_flat_imaging
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_intensity_flat_imaging_run_4(void) {
    sph_zpl_intensity_flat_imaging* zpl_intensity_flat_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube_1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
//    int                         		nframes             = 6;
//    int                         		nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
//    int                         		ny                  = 100; // -//-
//    double              dV						= 100.0;
//    double 				 image_odd				= 2500.0;
//    double				 image_even				= 2000.0;
    cpl_propertylist* pl = NULL;
    cpl_parameter* parameter_collapse = NULL;
    cpl_parameter* parameter_sigma_clip = NULL;
    cpl_parameter* keep_intermediate = NULL;
    cpl_frame* master_bias_imaging_frame = NULL;
    cpl_frame* master_dark_imaging_frame = NULL;
    double iframe_bias = 4.0;
    double pframe_bias = 1.0;
    double iframe_dark = 10.0;
    double pframe_dark = 11.0;
    int nz = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-
    double iframe = 34;
    double pframe = 39;
    double iframe1 = 44;
    double pframe1 = 49;
    double ovsc[4] = { 4.0, 1.0, 6.0, 2.0 };
    double ovsc_1[4] = { 9.0, 1.0, 10.0, 2.0 };
//    double 						ovsc[4]		= {0.0, 1.0, 0.0, 2.0};
//    double 						ovsc_1[4]	= {0.0, 1.0, 0.0, 2.0};

    double adu1_zero = 0;
    double adu2_zero = 0;

    char outfilename[256];

    inparams = sph_zpl_intensity_flat_imaging_create_paramlist();

    //set "zpl.intensity_flat.collapse" parameter to TRUE
    parameter_collapse = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat_imaging.collapse");
    cpl_parameter_set_bool(parameter_collapse, 1);
    parameter_sigma_clip = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat_imaging.sigma_clip");
    cpl_parameter_set_double(parameter_sigma_clip, 3.0);
    keep_intermediate = cpl_parameterlist_find(inparams,
            "zpl.intensity_flat_imaging.keep_intermediate");
    cpl_parameter_set_bool(keep_intermediate, 1);

    inframes = cpl_frameset_new();
    //sph_test_simulator_create_zplpreproc_imaging_cube_set( &zexpcube, image_odd, image_even, dV, nframes );

    // zpl frame cube with overscan table
    zexpcube = sph_test_simulator_create_zpl_exposure_imaging_overscans_cube(nx,
            ny, nz, "test_iff_imaging_zplexp_ovsc_cube", iframe, pframe, ovsc,
            1, 1);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_IFF_IMAGING_PREPROC_CAM1);
    rerr = cpl_frameset_insert(inframes, zexpcube);

    // zpl frame cube with overscan table
    zexpcube_1 = sph_test_simulator_create_zpl_exposure_imaging_overscans_cube(
            nx, ny, nz, "test_iff_imaging_zplexp_ovsc_cube_1", iframe1, pframe1,
            ovsc_1, 1, 0);
    cpl_frame_set_tag(zexpcube_1, SPH_ZPL_TAG_IFF_IMAGING_PREPROC_CAM1);
    rerr = cpl_frameset_insert(inframes, zexpcube_1);

    //create master bias imaging (double image frame)
    master_bias_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_fi_master_bias.fits.tmp", nx, ny, iframe_bias,
                    pframe_bias);
    cpl_frame_set_tag(master_bias_imaging_frame,
            SPH_ZPL_TAG_BIAS_IMAGING_CALIB_CAM1);
    cpl_frame_set_group(master_bias_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_bias_imaging_frame);

    //create master dark imaging (double image frame)
    master_dark_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_fi_master_dark.fits.tmp", nx, ny, iframe_dark,
                    pframe_dark);
    cpl_frame_set_tag(master_dark_imaging_frame,
            SPH_ZPL_TAG_DARK_IMAGING_CALIB_CAM1);
    cpl_frame_set_group(master_dark_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_dark_imaging_frame);

    //printf("raw frame: %s\n", cpl_frame_get_filename(cpl_frameset_get_first( zpl_intensity_flat->rawframes )));
    //printf("rawframes #: %d\n", cpl_frameset_get_size( zpl_intensity_flat->rawframes ));

    zpl_intensity_flat_imaging = sph_zpl_intensity_flat_imaging_new(inframes,
            inparams);
    cpl_test_nonnull( zpl_intensity_flat_imaging );

    rerr = sph_zpl_intensity_flat_imaging_run(zpl_intensity_flat_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    //test for non-normalized output
    sprintf(
            outfilename,
            "%s-qc-non-normalized.fits",
            zpl_intensity_flat_imaging->intensity_flat_imaging_outfilename_cam1);
    pl = cpl_propertylist_load(outfilename, 0);

    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, pframe_names_leakage.bad_pixels_name),
            0.0, 1e-10);

    adu1_zero = (34. - ovsc[0] + 35. - (ovsc[0] + 1) + 36. - (ovsc[0] + 2) + 44.
            - ovsc_1[0] + 45. - ovsc_1[0] + 46. - ovsc_1[0]) / 6.;
    adu2_zero = (34. - ovsc[2] + 35. - (ovsc[2] + 1) + 36. - (ovsc[2] + 2) + 44.
            - ovsc_1[2] + 45. - ovsc_1[2] + 46. - ovsc_1[2]) / 6.;
    //printf("0.5*(adu1_zero+adu2_zero) = %f\n",  0.5*(adu1_zero+adu2_zero));
    //printf("adu1_zero = %f\n",  adu1_zero );
    //printf("adu2_zero = %f\n",  adu2_zero );
    cpl_test_abs(
            cpl_propertylist_get_double( pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            0.5*(adu1_zero+adu2_zero)-iframe_bias-iframe_dark, 1e-10);

    adu1_zero = (39. - ovsc[0] + 40. - (ovsc[0] + 1) + 41. - (ovsc[0] + 2) + 49.
            - ovsc_1[0] + 50. - ovsc_1[0] + 51. - ovsc_1[0]) / 6.;
    adu2_zero = (39. - ovsc[2] + 40. - (ovsc[2] + 1) + 41. - (ovsc[2] + 2) + 49.
            - ovsc_1[2] + 50. - ovsc_1[2] + 51. - ovsc_1[2]) / 6.;

    cpl_test_abs(
            cpl_propertylist_get_double( pl, pframe_names_leakage.mean_name ),
            0.5*(adu1_zero+adu2_zero)-pframe_bias-pframe_dark, 1e-10);
    cpl_propertylist_delete(pl);

    //test for normalized output
    pl = cpl_propertylist_load(
            zpl_intensity_flat_imaging->intensity_flat_imaging_outfilename_cam1,
            0);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD),
            0.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_long(pl, pframe_names_leakage.bad_pixels_name),
            0.0, 1e-10);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            1, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, pframe_names_leakage.mean_name),
            1, 1e-10);

    cpl_propertylist_delete(pl);

    rerr = sph_zpl_intensity_flat_imaging_delete(zpl_intensity_flat_imaging);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_intensity_flat recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;

    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("intensity_flat_imaging_test",
            cutest_init_intensity_flat_imaging_testsuite,
            cutest_clean_intensity_flat_imaging_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite,
                    "sph_intensity_flat_imaging_run_1: pre-processed frames",
                    cutest_zpl_intensity_flat_imaging_run_1)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_intensity_flat_imaging_run_2: raw(pre-processed) frames, bias and dark calibrations",
                    cutest_zpl_intensity_flat_imaging_run_2)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_intensity_flat_imaging_run_3: simple collapse method (no linear fitting)",
                    cutest_zpl_intensity_flat_imaging_run_3)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_intensity_flat_imaging_run_4: simple collapse method on the pre-processed data with overscan table + bias + dark",
                    cutest_zpl_intensity_flat_imaging_run_4)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
