/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "cutest_zpl_master_bias_imaging.h"
#include "sph_zpl_master_bias_imaging.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_master_bias_imaging_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     *--------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static int cutest_clean_master_bias_imaging_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_bias_imaging
 @note  It is fed by raw pre-processed frames, namely one cube without overscan table
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_master_bias_imaging_run_1(void) {
    sph_zpl_master_bias_imaging* zpl_master_bias_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube2 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-

    inparams = sph_zpl_master_bias_imaging_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_imaging_cube(&zexpcube, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_BIAS_IMAGING_PREPROC_CAM1);
    // cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert(inframes, zexpcube);

    sph_test_simulator_create_zplpreproc_imaging_cube(&zexpcube2, nframes);
    cpl_frame_set_tag(zexpcube2, SPH_ZPL_TAG_BIAS_IMAGING_PREPROC_CAM2);
    rerr = cpl_frameset_insert(inframes, zexpcube2);

    //printf("Call master_bias_imaging_new\n...");
    zpl_master_bias_imaging = sph_zpl_master_bias_imaging_new(inframes,
            inparams);
    cpl_test_nonnull( zpl_master_bias_imaging );

    //printf("Call master_bias_imaging_run\n...");

    rerr = sph_zpl_master_bias_imaging_run(zpl_master_bias_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_abs(
            cpl_image_get_absflux(zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam1->iframe->rmsmap)/(nx*ny),
            sqrt(2./(nframes-1.5)), 1e-5);
    cpl_test_abs(
            cpl_propertylist_get_double(zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam1->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            20.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam1->qclist, pframe_names_leakage.mean_name),
            21.0, 1e-20);

    cpl_test_abs(
            cpl_image_get_absflux(zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam2->iframe->rmsmap)/(nx*ny),
            sqrt(2./(nframes-1.5)), 1e-5);
    cpl_test_abs(
            cpl_propertylist_get_double(zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            20.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam2->qclist, pframe_names_leakage.mean_name),
            21.0, 1e-20);

    rerr = sph_zpl_master_bias_imaging_delete(zpl_master_bias_imaging);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_bias_imaging
 @note  It is fed by raw pre-processed frames, namely 2 cubes without overscan table
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_master_bias_imaging_run_2(void) {
    sph_zpl_master_bias_imaging* zpl_master_bias_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;

    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    //int                         		nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         		ny                  = 100; // -//-
    double iframe = 34.0;
    double pframe = 39.0;
    double iframe1 = 44.0;
    double pframe1 = 49.0;

    inparams = sph_zpl_master_bias_imaging_create_paramlist();

    //simulate 2 frames (cubes) of the zpl exposure imaging type
    inframes = cpl_frameset_new();
    //create first cube
    sph_test_simulator_create_zplpreproc_imaging_cube_set(&zexpcube, iframe,
            pframe, 1.0, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_BIAS_IMAGING_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube);
    ///create second cube
    sph_test_simulator_create_zplpreproc_imaging_cube_set(&zexpcube1, iframe1,
            pframe1, 1.0, nframes);
    cpl_frame_set_tag(zexpcube1, SPH_ZPL_TAG_BIAS_IMAGING_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube1, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube1);

    //printf("Call master_bias_imaging_new\n...");
    zpl_master_bias_imaging = sph_zpl_master_bias_imaging_new(inframes,
            inparams);
    cpl_test_nonnull( zpl_master_bias_imaging );

    //printf("Call master_bias_imaging_run\n...");

    rerr = sph_zpl_master_bias_imaging_run(zpl_master_bias_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    //cpl_test_abs( cpl_image_get_absflux(zpl_master_bias_imaging->master_bias_imaging_doubleimage->iframe->rmsmap)/(nx*ny), sqrt(2./(nframes-1.0)), 1e-5 );
    cpl_test_abs(
            cpl_propertylist_get_double(zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam2->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            40.0, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double(zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam2->qclist, pframe_names_leakage.mean_name),
            45.0, 1e-20);

    rerr = sph_zpl_master_bias_imaging_delete(zpl_master_bias_imaging);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_bias_imaging
 @note  It is fed by raw pre-processed frames, namely 2 cubes with overscan tables
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_master_bias_imaging_run_3(void) {
    sph_zpl_master_bias_imaging* zpl_master_bias_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube_1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nz = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-
    int iframe = 34;
    int pframe = 39;
    int iframe1 = 44;
    int pframe1 = 49;
    double ovsc[4] = { 4.0, 1.0, 5.0, 2.0 };
    double ovsc_1[4] = { 9.0, 1.0, 10.0, 2.0 };
    double adu1_zero = 0;
    double adu2_zero = 0;

    inparams = sph_zpl_master_bias_imaging_create_paramlist();
    inframes = cpl_frameset_new();

    // zpl frame cube with overscan table
    zexpcube = sph_test_simulator_create_zpl_exposure_imaging_overscans_cube(nx,
            ny, nz, "test_master_bias_imaging_zplexp_ovsc_cube", iframe, pframe,
            ovsc, 1, 1);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_BIAS_IMAGING_PREPROC_CAM1);
    rerr = cpl_frameset_insert(inframes, zexpcube);

    // zpl frame cube with overscan table
    zexpcube_1 = sph_test_simulator_create_zpl_exposure_imaging_overscans_cube(
            nx, ny, nz, "test_master_bias_imaging_zplexp_ovsc_cube_1", iframe1,
            pframe1, ovsc_1, 1, 0);
    cpl_frame_set_tag(zexpcube_1, SPH_ZPL_TAG_BIAS_IMAGING_PREPROC_CAM1);
    rerr = cpl_frameset_insert(inframes, zexpcube_1);

    zpl_master_bias_imaging = sph_zpl_master_bias_imaging_new(inframes,
            inparams);
    cpl_test_nonnull( zpl_master_bias_imaging );

    rerr = sph_zpl_master_bias_imaging_run(zpl_master_bias_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    adu1_zero = (34. - ovsc[0] + 35. - (ovsc[0] + 1) + 36. - (ovsc[0] + 2) + 44.
            - ovsc_1[0] + 45. - ovsc_1[0] + 46. - ovsc_1[0]) / 6.;
    adu2_zero = (34. - ovsc[2] + 35. - (ovsc[2] + 1) + 36. - (ovsc[2] + 2) + 44.
            - ovsc_1[2] + 45. - ovsc_1[2] + 46. - ovsc_1[2]) / 6.;

    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam1->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            0.5*(adu1_zero+adu2_zero), 1e-10);

    adu1_zero = (39. - ovsc[0] + 40. - (ovsc[0] + 1) + 41. - (ovsc[0] + 2) + 49.
            - ovsc_1[0] + 50. - ovsc_1[0] + 51. - ovsc_1[0]) / 6.;
    adu2_zero = (39. - ovsc[2] + 40. - (ovsc[2] + 1) + 41. - (ovsc[2] + 2) + 49.
            - ovsc_1[2] + 50. - ovsc_1[2] + 51. - ovsc_1[2]) / 6.;

    cpl_test_abs(
            cpl_propertylist_get_double( zpl_master_bias_imaging->master_bias_imaging_doubleimage_cam1->qclist, pframe_names_leakage.mean_name ),
            0.5*(adu1_zero+adu2_zero), 1e-10);

    rerr = sph_zpl_master_bias_imaging_delete(zpl_master_bias_imaging);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;

    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("master_bias_imaging_test",
            cutest_init_master_bias_imaging_testsuite,
            cutest_clean_master_bias_imaging_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_bias_master_bias_imaging_run_1: 1 pre-processed cubes without overscan tables",
                    cutest_zpl_master_bias_imaging_run_1)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_bias_master_bias_imaging_run_2: 2 pre-processed cubes without overscan tables",
                    cutest_zpl_master_bias_imaging_run_2)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_bias_master_bias_imaging_run_3: 2 pre-processed cubes with overscan tables",
                    cutest_zpl_master_bias_imaging_run_3)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
