/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>

#include "cutest_zpl_subtract_dark_scaled.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"
#include "sph_common_keywords.h"
#include "sph_utils.h"
#include "sph_test_zpl_simulator.h"
#include "sph_zpl_subtract_dark_scaled.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}

const int NX = 256;
const int NY = 256;
const int NBADPIX = 10;
const double BADPIXVAL = 100000000;

/*----------------------------------------------------------------------------*/
/**
 @brief    Helper function to set up a frame with a gradient and
           some bad pixels.
 */
/*----------------------------------------------------------------------------*/
static
void setup_frame_with_badpix(sph_master_frame* frame, double val) {
    sph_test_image_tools_add_gradient_x(frame->image, -val, val);
    for (int i=1; i<=NBADPIX; i++) {
      cpl_image_set(frame->badpixelmap, i, i, 1);
      cpl_image_set(frame->image, i, i, BADPIXVAL);
    }
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Helper function to set up a frame with a gradient and
           some pixels with bad values, but not flagged as bad pixels.
 */
/*----------------------------------------------------------------------------*/
static
void setup_frame_without_badpix(sph_master_frame* frame, double val) {
    sph_test_image_tools_add_gradient_x(frame->image, -val, val);
    for (int i=1; i<=NBADPIX; i++) {
      cpl_image_set(frame->image, i, i, BADPIXVAL);
    }
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Macro to do some standard tests on an output frame.
           Implemented as a macro, not a function, so that the output log
           has more meaningful line numbers.
 */
/*----------------------------------------------------------------------------*/
#define check_frame(frame) do { \
        int bpix = 0; \
        for (int i=1; i<=NBADPIX; i++) { \
            cpl_test_eq(cpl_image_get(frame->badpixelmap, i, i, &bpix), 1); \
        } \
        cpl_test_eq(sph_master_frame_get_nbads(frame), NBADPIX); \
        cpl_test_abs(sph_master_frame_get_absflux(frame, NULL), 0.0, 0.0001); \
    } while (0);

/*----------------------------------------------------------------------------*/
/**
 @brief    Tests dark subtraction of two frames without DIT information.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_double_no_dit_info(void) {
    sph_double_image* dark = NULL;
    sph_double_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    sph_master_frame* idark = dark->iframe;
    sph_master_frame* pdark = dark->pframe;

    setup_frame_with_badpix(idark, 100.0);
    setup_frame_with_badpix(pdark, 100.0);

    raw = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    sph_master_frame* iraw = raw->iframe;
    sph_master_frame* praw = raw->pframe;

    setup_frame_without_badpix(iraw, 100.0);
    setup_frame_without_badpix(praw, 100.0);

    cpl_test_eq(sph_zpl_subtract_dark_double_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(iraw);
    check_frame(praw);

    sph_double_image_delete(raw);
    raw = NULL;
    sph_double_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Tests dark subtraction of two frames with matching DITs.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_double_dit_short_match(void) {
    sph_double_image* dark = NULL;
    sph_double_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            1.7);
    sph_master_frame* idark = dark->iframe;
    sph_master_frame* pdark = dark->pframe;

    setup_frame_with_badpix(idark, 100.0);
    setup_frame_with_badpix(pdark, 100.0);

    raw = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            1.7);
    sph_master_frame* iraw = raw->iframe;
    sph_master_frame* praw = raw->pframe;

    setup_frame_without_badpix(iraw, 100.0);
    setup_frame_without_badpix(praw, 100.0);

    cpl_test_eq(sph_zpl_subtract_dark_double_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(iraw);
    check_frame(praw);

    sph_double_image_delete(raw);
    raw = NULL;
    sph_double_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Tests dark subtraction where the dark is half the exposure time
          of the raw frame.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_double_dit_short_no_match(void) {
    sph_double_image* dark = NULL;
    sph_double_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            1.7);
    sph_master_frame* idark = dark->iframe;
    sph_master_frame* pdark = dark->pframe;

    setup_frame_with_badpix(idark, 50.0);
    setup_frame_with_badpix(pdark, 50.0);

    raw = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            3.4);
    sph_master_frame* iraw = raw->iframe;
    sph_master_frame* praw = raw->pframe;

    setup_frame_without_badpix(iraw, 100.0);
    setup_frame_without_badpix(praw, 100.0);

    cpl_test_eq(sph_zpl_subtract_dark_double_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(iraw);
    check_frame(praw);

    sph_double_image_delete(raw);
    raw = NULL;
    sph_double_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Tests dark subtraction where both frames have bad DITs.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_double_bad_dit(void) {
    sph_double_image* dark = NULL;
    sph_double_image* raw = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    dark = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            -1.7);

    raw = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            -3.4);

    rerr = sph_zpl_subtract_dark_double_image_scaled(raw, dark);
    cpl_test_eq_error(rerr, CPL_ERROR_ILLEGAL_INPUT);

    sph_double_image_delete(raw);
    raw = NULL;
    sph_double_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Tests dark subtraction where the dark exposure is short and the
           raw frame is long.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_double_dit_long_no_match_1(void) {
    sph_double_image* dark = NULL;
    sph_double_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            10.0);
    sph_master_frame* idark = dark->iframe;
    sph_master_frame* pdark = dark->pframe;

    setup_frame_with_badpix(idark, 1.0);
    setup_frame_with_badpix(pdark, 1.0);

    raw = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            1000.0);
    sph_master_frame* iraw = raw->iframe;
    sph_master_frame* praw = raw->pframe;

    setup_frame_without_badpix(iraw, 100.0);
    setup_frame_without_badpix(praw, 100.0);

    cpl_test_eq(sph_zpl_subtract_dark_double_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(iraw);
    check_frame(praw);

    sph_double_image_delete(raw);
    raw = NULL;
    sph_double_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Tests dark subtraction where the dark exposure is long and the
           raw frame is short.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_double_dit_long_no_match_2(void) {
    sph_double_image* dark = NULL;
    sph_double_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            1000.0);
    sph_master_frame* idark = dark->iframe;
    sph_master_frame* pdark = dark->pframe;

    setup_frame_with_badpix(idark, 100.0);
    setup_frame_with_badpix(pdark, 100.0);

    raw = sph_test_zpl_simulator_create_double_image(NX, NY, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            10.0);
    sph_master_frame* iraw = raw->iframe;
    sph_master_frame* praw = raw->pframe;

    setup_frame_without_badpix(iraw, 1.0);
    setup_frame_without_badpix(praw, 1.0);

    cpl_test_eq(sph_zpl_subtract_dark_double_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(iraw);
    check_frame(praw);

    sph_double_image_delete(raw);
    raw = NULL;
    sph_double_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Tests dark subtraction of two frames without DIT information.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_quad_no_dit_info(void) {
    sph_quad_image* dark = NULL;
    sph_quad_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    sph_master_frame* zidark = dark->zero_image->iframe;
    sph_master_frame* zpdark = dark->zero_image->pframe;
    sph_master_frame* pidark = dark->pi_image->iframe;
    sph_master_frame* ppdark = dark->pi_image->pframe;

    setup_frame_with_badpix(zidark, 100.0);
    setup_frame_with_badpix(zpdark, 100.0);
    setup_frame_with_badpix(pidark, 100.0);
    setup_frame_with_badpix(ppdark, 100.0);

    raw = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    sph_master_frame* ziraw = raw->zero_image->iframe;
    sph_master_frame* zpraw = raw->zero_image->pframe;
    sph_master_frame* piraw = raw->pi_image->iframe;
    sph_master_frame* ppraw = raw->pi_image->pframe;

    setup_frame_without_badpix(ziraw, 100.0);
    setup_frame_without_badpix(zpraw, 100.0);
    setup_frame_without_badpix(piraw, 100.0);
    setup_frame_without_badpix(ppraw, 100.0);

    cpl_test_eq(sph_zpl_subtract_dark_quad_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(ziraw);
    check_frame(zpraw);
    check_frame(piraw);
    check_frame(ppraw);

    sph_quad_image_delete(raw);
    raw = NULL;
    sph_quad_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Tests dark subtraction of two frames with matching DITs.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_quad_dit_short_match(void) {
    sph_quad_image* dark = NULL;
    sph_quad_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            1.7);
    sph_master_frame* zidark = dark->zero_image->iframe;
    sph_master_frame* zpdark = dark->zero_image->pframe;
    sph_master_frame* pidark = dark->pi_image->iframe;
    sph_master_frame* ppdark = dark->pi_image->pframe;

    setup_frame_with_badpix(zidark, 100.0);
    setup_frame_with_badpix(zpdark, 100.0);
    setup_frame_with_badpix(pidark, 100.0);
    setup_frame_with_badpix(ppdark, 100.0);

    raw = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            1.7);
    sph_master_frame* ziraw = raw->zero_image->iframe;
    sph_master_frame* zpraw = raw->zero_image->pframe;
    sph_master_frame* piraw = raw->pi_image->iframe;
    sph_master_frame* ppraw = raw->pi_image->pframe;

    setup_frame_without_badpix(ziraw, 100.0);
    setup_frame_without_badpix(zpraw, 100.0);
    setup_frame_without_badpix(piraw, 100.0);
    setup_frame_without_badpix(ppraw, 100.0);

    cpl_test_eq(sph_zpl_subtract_dark_quad_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(ziraw);
    check_frame(zpraw);
    check_frame(piraw);
    check_frame(ppraw);

    sph_quad_image_delete(raw);
    raw = NULL;
    sph_quad_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Tests dark subtraction where the dark is half the exposure time
          of the raw frame.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_quad_dit_short_no_match(void) {
    sph_quad_image* dark = NULL;
    sph_quad_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            1.7);
    sph_master_frame* zidark = dark->zero_image->iframe;
    sph_master_frame* zpdark = dark->zero_image->pframe;
    sph_master_frame* pidark = dark->pi_image->iframe;
    sph_master_frame* ppdark = dark->pi_image->pframe;

    setup_frame_with_badpix(zidark, 50.0);
    setup_frame_with_badpix(zpdark, 50.0);
    setup_frame_with_badpix(pidark, 50.0);
    setup_frame_with_badpix(ppdark, 50.0);

    raw = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            3.4);
    sph_master_frame* ziraw = raw->zero_image->iframe;
    sph_master_frame* zpraw = raw->zero_image->pframe;
    sph_master_frame* piraw = raw->pi_image->iframe;
    sph_master_frame* ppraw = raw->pi_image->pframe;

    setup_frame_without_badpix(ziraw, 100.0);
    setup_frame_without_badpix(zpraw, 100.0);
    setup_frame_without_badpix(piraw, 100.0);
    setup_frame_without_badpix(ppraw, 100.0);

    cpl_test_eq(sph_zpl_subtract_dark_quad_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(ziraw);
    check_frame(zpraw);
    check_frame(piraw);
    check_frame(ppraw);

    sph_quad_image_delete(raw);
    raw = NULL;
    sph_quad_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Tests dark subtraction where both frames have bad DITs.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_quad_bad_dit(void) {
    sph_quad_image* dark = NULL;
    sph_quad_image* raw = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    dark = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            -1.7);

    raw = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            -3.4);

    rerr = sph_zpl_subtract_dark_quad_image_scaled(raw, dark);
    cpl_test_eq_error(rerr, CPL_ERROR_ILLEGAL_INPUT);

    sph_quad_image_delete(raw);
    raw = NULL;
    sph_quad_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Tests dark subtraction where the dark exposure is short and the
           raw frame is long.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_quad_dit_long_no_match_1(void) {
    sph_quad_image* dark = NULL;
    sph_quad_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            10.0);
    sph_master_frame* zidark = dark->zero_image->iframe;
    sph_master_frame* zpdark = dark->zero_image->pframe;
    sph_master_frame* pidark = dark->pi_image->iframe;
    sph_master_frame* ppdark = dark->pi_image->pframe;

    setup_frame_with_badpix(zidark, 1.0);
    setup_frame_with_badpix(zpdark, 1.0);
    setup_frame_with_badpix(pidark, 1.0);
    setup_frame_with_badpix(ppdark, 1.0);

    raw = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            1000.0);
    sph_master_frame* ziraw = raw->zero_image->iframe;
    sph_master_frame* zpraw = raw->zero_image->pframe;
    sph_master_frame* piraw = raw->pi_image->iframe;
    sph_master_frame* ppraw = raw->pi_image->pframe;

    setup_frame_without_badpix(ziraw, 100.0);
    setup_frame_without_badpix(zpraw, 100.0);
    setup_frame_without_badpix(piraw, 100.0);
    setup_frame_without_badpix(ppraw, 100.0);

    cpl_test_eq(sph_zpl_subtract_dark_quad_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(ziraw);
    check_frame(zpraw);
    check_frame(piraw);
    check_frame(ppraw);

    sph_quad_image_delete(raw);
    raw = NULL;
    sph_quad_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Tests dark subtraction where the dark exposure is long and the
           raw frame is short.
 */
/*----------------------------------------------------------------------------*/
void cutest_sph_zpl_subtract_dark_quad_dit_long_no_match_2(void) {
    sph_quad_image* dark = NULL;
    sph_quad_image* raw = NULL;

    dark = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    dark->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(dark->properties, SPH_COMMON_KEYWORD_DIT,
            1000.0);
    sph_master_frame* zidark = dark->zero_image->iframe;
    sph_master_frame* zpdark = dark->zero_image->pframe;
    sph_master_frame* pidark = dark->pi_image->iframe;
    sph_master_frame* ppdark = dark->pi_image->pframe;

    setup_frame_with_badpix(zidark, 100.0);
    setup_frame_with_badpix(zpdark, 100.0);
    setup_frame_with_badpix(pidark, 100.0);
    setup_frame_with_badpix(ppdark, 100.0);

    raw = sph_test_zpl_simulator_create_quad_image(NX, NY, 0.0, 0.0, 0.0, 0.0);
    raw->properties = cpl_propertylist_new();
    cpl_propertylist_update_double(raw->properties, SPH_COMMON_KEYWORD_DIT,
            10.0);
    sph_master_frame* ziraw = raw->zero_image->iframe;
    sph_master_frame* zpraw = raw->zero_image->pframe;
    sph_master_frame* piraw = raw->pi_image->iframe;
    sph_master_frame* ppraw = raw->pi_image->pframe;

    setup_frame_without_badpix(ziraw, 1.0);
    setup_frame_without_badpix(zpraw, 1.0);
    setup_frame_without_badpix(piraw, 1.0);
    setup_frame_without_badpix(ppraw, 1.0);

    cpl_test_eq(sph_zpl_subtract_dark_quad_image_scaled(raw, dark),
            CPL_ERROR_NONE);
    check_frame(ziraw);
    check_frame(zpraw);
    check_frame(piraw);
    check_frame(ppraw);

    sph_quad_image_delete(raw);
    raw = NULL;
    sph_quad_image_delete(dark);
    dark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testuite ZPL subtract dark", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark double w/o DIT info",
                    cutest_sph_zpl_subtract_dark_double_no_dit_info)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark double short dit (match)",
                    cutest_sph_zpl_subtract_dark_double_dit_short_match)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark double short dit (no match)",
                    cutest_sph_zpl_subtract_dark_double_dit_short_no_match)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark double bad dit",
                    cutest_sph_zpl_subtract_dark_double_bad_dit)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark double long raw dit (no match)",
                    cutest_sph_zpl_subtract_dark_double_dit_long_no_match_1)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark double long dark dit (no match)",
                    cutest_sph_zpl_subtract_dark_double_dit_long_no_match_2)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark quad w/o DIT info",
                    cutest_sph_zpl_subtract_dark_quad_no_dit_info)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark quad short dit (match)",
                    cutest_sph_zpl_subtract_dark_quad_dit_short_match)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark quad short dit (no match)",
                    cutest_sph_zpl_subtract_dark_quad_dit_short_no_match)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark quad bad dit",
                    cutest_sph_zpl_subtract_dark_quad_bad_dit)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark quad long raw dit (no match)",
                    cutest_sph_zpl_subtract_dark_quad_dit_long_no_match_1)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing subtract dark quad long dark dit (no match)",
                    cutest_sph_zpl_subtract_dark_quad_dit_long_no_match_2)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
