/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_badpixelco.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/*-                            Declarations of private functions              */
/*----------------------------------------------------------------------------*/

static
int cutest_sph_badpixelco_init_testsuite(void);

static
int cutest_sph_badpixelco_clean_testsuite(void);

static
void
cutest_sph_badpixelco_apply_to_raw_cubes(void);

static
void cutest_sph_badpixelco_fill_cx_cy_vectors_returns_vectors_size_1_when_no_badpix(void);

static
void cutest_sph_badpixelco_fill_cx_cy_vectors_returns_vectors_filled_with_correct_bpix_coords(void);

static
void cutest_sph_badpixelco_correct_single_pixel_does_correct_correction(void);

static
void cutest_sph_badpixelco_correct_get_medians_for_rows_returns_vector(void);

static
void correct_single_pixel_entire_row_returns_original_image_if_no_pix_above_threshold(void);

static
void correct_single_pixel_entire_row_returns_changed_image_if_pix_above_threshold(void);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/


/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                      */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_badpixelco_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_badpixelco_clean_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_badpixelco_fill_cx_cy_vectors_returns_vectors_size_1_when_no_badpix(void) {
	cpl_image* badimage = NULL;
	cpl_vector* cx = NULL;
	cpl_vector* cy = NULL;
	cpl_size badcount = 0;

	badimage = sph_test_image_tools_create_flat_image_int(2048,2048,0);
	badcount = sph_badpixelco_fill_cx_cy_vectors( badimage, &cx, &cy );
	cpl_test_nonnull( cx );
	cpl_test_nonnull( cy );
	cpl_test_zero(badcount);
	cpl_test_eq(cpl_vector_get_size(cx), 1 );
	cpl_test_eq(cpl_vector_get_size(cy), 1 );
	cpl_image_delete( badimage );
	cpl_vector_delete(cx);
	cpl_vector_delete(cy);
}
static
void cutest_sph_badpixelco_fill_cx_cy_vectors_returns_vectors_filled_with_correct_bpix_coords(void) {
	cpl_image* badimage = NULL;
	cpl_vector* cx = NULL;
	cpl_vector* cy = NULL;
	cpl_size badcount = 0;

	badimage = sph_test_image_tools_create_flat_image_int(2048,2048,0);
	cpl_image_set(badimage,1,1,1);
	cpl_image_set(badimage,1,2048,1);
	cpl_image_set(badimage,2048,1,1);
	cpl_image_set(badimage,2048,2048,1);
	cpl_image_set(badimage,1024,1024,1);

	badcount = sph_badpixelco_fill_cx_cy_vectors( badimage, &cx, &cy );
	cpl_test_nonnull( cx );
	cpl_test_nonnull( cy );
	cpl_test_eq(badcount, 5 );
	cpl_test_eq(cpl_vector_get_size(cx), 5 );
	cpl_test_eq(cpl_vector_get_size(cy), 5 );

	cpl_test_zero(cpl_vector_get(cx,0));
	cpl_test_zero(cpl_vector_get(cy,0));
	cpl_test_eq(cpl_vector_get(cx,1), 2047);
	cpl_test_zero(cpl_vector_get(cy,1));
	cpl_test_eq(cpl_vector_get(cx,2), 1023);
	cpl_test_eq(cpl_vector_get(cy,2), 1023);
	cpl_test_zero(cpl_vector_get(cx,3));
	cpl_test_eq(cpl_vector_get(cy,3), 2047);
	cpl_test_eq(cpl_vector_get(cx,4), 2047);
	cpl_test_eq(cpl_vector_get(cy,4), 2047);
	cpl_image_delete( badimage );
	cpl_vector_delete(cx);
	cpl_vector_delete(cy);
}

static
void cutest_sph_badpixelco_correct_single_pixel_does_correct_correction(void) {
	cpl_image* badimage = NULL;
	cpl_image* rawimage = NULL;
	cpl_image* outimage = NULL;
	cpl_vector* cx = NULL;
	cpl_vector* cy = NULL;
	cpl_size badcount = 0;
	int badpix = 0;

	cpl_error_code result = CPL_ERROR_NONE;

	rawimage = sph_test_image_tools_create_flat_image_double(2048,2048,1.0);
	outimage = sph_test_image_tools_create_flat_image_double(2048,2048,10.0);
	badimage = sph_test_image_tools_create_flat_image_int(2048,2048,0);
	cpl_image_set(badimage,1024,1024,1);
	cpl_image_set(rawimage,1024,1024,0);
	badcount = sph_badpixelco_fill_cx_cy_vectors( badimage, &cx, &cy );
	cpl_test_eq(badcount, 1 );
	result = sph_badpixelco_correct_single_pixel(badimage,rawimage,1024,1024,outimage);

	cpl_test_abs( cpl_image_get(outimage,1024,1024,&badpix), 1.0, 0.0001);
	cpl_test_zero(badpix);
	cpl_test_eq_error(result, CPL_ERROR_NONE );

	cpl_image_save(outimage,"badpix_out.fits",CPL_TYPE_UNSPECIFIED,NULL,CPL_IO_CREATE);
	cpl_image_delete( rawimage );
	cpl_image_delete( badimage );
	cpl_image_delete( outimage );
	cpl_vector_delete(cx);
	cpl_vector_delete(cy);
}

static
void cutest_sph_badpixelco_correct_get_medians_for_rows_returns_vector(void) {
	cpl_image* rawimage = NULL;
	cpl_vector* medians = NULL;

	rawimage = sph_test_image_tools_create_flat_image_double(2048,2048,1.0);

	medians = sph_badpixelco_get_medians_for_rows( rawimage, 100 );
	cpl_test_eq(cpl_vector_get_size(medians), 2048);
	cpl_test_abs( cpl_vector_get(medians, 324), 1.0, 0.000001);

	cpl_vector_delete(medians);
	cpl_image_delete(rawimage);
}

static
void correct_single_pixel_entire_row_returns_original_image_if_no_pix_above_threshold(void) {
	cpl_image* rawimage = NULL;
	cpl_image* corrected_image = NULL;
	cpl_vector* medians = NULL;

	rawimage = sph_test_image_tools_create_flat_image_double(2048,2048,1.0);
	corrected_image = cpl_image_duplicate(rawimage);

	medians = sph_badpixelco_get_medians_for_rows( rawimage, 100 );

	sph_badpixelco_correct_single_pixel_entire_row(rawimage,medians,0.0,1.0,10,10,corrected_image);

	cpl_image_subtract(corrected_image,rawimage);

	cpl_test_abs( cpl_image_get_absflux(corrected_image), 0.0, 0.00000001);

	cpl_vector_delete(medians);
	cpl_image_delete(rawimage);
	cpl_image_delete(corrected_image);
}

static
void correct_single_pixel_entire_row_returns_changed_image_if_pix_above_threshold(void) {
	cpl_image* rawimage = NULL;
	cpl_image* corrected_image = NULL;
	cpl_vector* medians = NULL;

	rawimage = sph_test_image_tools_create_flat_image_double(2048,2048,1.0);
	cpl_image_set(rawimage,500,500,10000.0);
	corrected_image = cpl_image_duplicate(rawimage);

	medians = sph_badpixelco_get_medians_for_rows( rawimage, 100 );

	sph_badpixelco_correct_single_pixel_entire_row(rawimage,medians,0.0,1.0,500,500,corrected_image);

	cpl_image_subtract(corrected_image,rawimage);

	cpl_test_lt(0.00000001, fabs(cpl_image_get_absflux(corrected_image)));
	cpl_test_abs( cpl_image_get_absflux(corrected_image), 10000.0 - 1.0, 0.00000001);

	cpl_vector_delete(medians);
	cpl_image_delete(rawimage);
	cpl_image_delete(corrected_image);
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_badpixelco_apply_to_raw_cubes(void) {
	cpl_imagelist* imlist = cpl_imagelist_new();
	//for ( cpl_size ii = 0; ii < 2; ++ii ) {
        cpl_image* image =
            sph_test_image_tools_create_striped_image(2048, 2048, 10,
                                                      CPL_TYPE_DOUBLE,
                                                      0.0, 1000.0);
		//cpl_imagelist_set(imlist,im,ii);
	//}

	cpl_image_turn(image,1);
	sph_test_image_tools_add_gradient_x(image,0.0,5000.0);
	sph_test_image_tools_add_gradient_y(image,0.0,5000.0);
    /* Small relative reduction in num. args (like below) clears Valgrind error
    sph_test_image_tools_add_crazypixels(image, 1e5, 1.0e7, NULL); */
    sph_test_image_tools_add_crazypixels(image,1000000,1.0e10,NULL);
	{
	cpl_frame* aframe = sph_test_cubify_image(image,"IFS RAW",CPL_FRAME_GROUP_RAW);
	cpl_frameset* allframes = cpl_frameset_new();

	cpl_frameset_insert(allframes,aframe);

	sph_badpixelco_apply_to_raw_cube(allframes,aframe,100,5.0,500.0);

	cpl_image_delete(image);
	cpl_imagelist_delete(imlist);
	cpl_frameset_delete(allframes);
	}
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_badpixelco",
            cutest_sph_badpixelco_init_testsuite,
            cutest_sph_badpixelco_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_badpixelco_apply",
                    cutest_sph_badpixelco_apply_to_raw_cubes)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_badpixelco_fill_cx_cy_vectors_returns_vectors_size_1_when_no_badpix",
            		cutest_sph_badpixelco_fill_cx_cy_vectors_returns_vectors_size_1_when_no_badpix))
    {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_badpixelco_fill_cx_cy_vectors_returns_vectors_filled_with_correct_bpix_coords",
            		cutest_sph_badpixelco_fill_cx_cy_vectors_returns_vectors_filled_with_correct_bpix_coords))
    {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "cutest_sph_badpixelco_correct_single_pixel_does_correct_correction",
            		cutest_sph_badpixelco_correct_single_pixel_does_correct_correction))
    {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "cutest_sph_badpixelco_correct_get_medians_for_rows_returns_vector",
            		cutest_sph_badpixelco_correct_get_medians_for_rows_returns_vector))
    {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "correct_single_pixel_entire_row_returns_original_image_if_no_pix_above_threshold",
            		correct_single_pixel_entire_row_returns_original_image_if_no_pix_above_threshold))
    {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "correct_single_pixel_entire_row_returns_changed_image_if_pix_above_threshold",
            		correct_single_pixel_entire_row_returns_changed_image_if_pix_above_threshold))
    {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
