/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_dataset.h"
#include "sph_dataset_stats.h"
#include "sph_test_image_tools.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test.h"
#include "sph_utils.h"

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>


/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and 
 *                                  associated functions.               
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
sph_dataset*
cutest_util_dataset_new_load(void) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        cpl_vector_set(vals, ii, (double) ii);
    }

    cpl_test_eq(cpl_vector_get( lams, 0 ), 1.0);
    cpl_test_eq(cpl_vector_get( lams, 99 ), 100.0);
    return dat;
}

static
sph_dataset*
cutest_util_dataset_new_gauss(void) {
    sph_dataset* dat = NULL;
    cpl_vector* lams = NULL;
    cpl_vector* vals = NULL;
    double val = 0.0;
    int ii = 0;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    lams = dat->xvalues;
    vals = dat->yvalues;
    cpl_test_nonnull( lams );
    cpl_test_nonnull( vals );

    for (ii = 0; ii < cpl_vector_get_size(lams); ++ii) {
        val = cpl_vector_get(lams, ii) - 50.0;
        val = exp(-val * val / 25.0);
        cpl_vector_set(vals, ii, val);
    }

    return dat;
}

static
int cutest_init_dataset_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_dataset_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_new function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_new(void) {
    sph_dataset* dat = NULL;

    /* Setup and run ...*/
    dat = cutest_util_dataset_new_load();

    /*Verify */
    cpl_test_nonnull( dat );
    sph_dataset_delete(dat);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_new_pixel_histogram function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_new_pixel_histogram(void) {
    sph_dataset* dat = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;

    /* Setup */
    cpl_error_reset();

    image = cpl_image_new(248, 248, CPL_TYPE_DOUBLE);

    //cpl_image_fill_gaussian(image, 124, 124, 1.0, 1.0, 1.0 );
    cpl_image_set(image, 100, 100, 10.0);
    cpl_image_set(image, 101, 100, 10.0);
    cpl_image_set(image, 102, 100, 30.0);
    cpl_image_set(image, 103, 100, 30.0);
    cpl_image_set(image, 104, 100, 40.0);

    /* Run ... */
    dat = sph_dataset_new_pixel_histogram(image, cpl_image_get_min(image),
            cpl_image_get_max(image), 100);

    cpl_test_nonnull( dat );

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( dat->xvalues );
    cpl_test_nonnull( dat->yvalues );
    cpl_test_eq(cpl_vector_get( dat->xvalues, 0 ), 0.0);
    cpl_test_eq(cpl_vector_get( dat->xvalues, 99 ), 40.0);
    cpl_test_eq((int)cpl_vector_get( dat->yvalues, 0 ), 248 * 248 - 5);
    cpl_test_abs( cpl_vector_get( dat->yvalues, 2 ), 0.0, 0.001);
    cpl_test_eq(sph_dataset_get_value( dat, 30.0 ), 2.0);
    sph_dataset_delete(dat);
    cpl_image_delete(image);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_new_pixel_histogram function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_collapse(void) {
    sph_dataset* dat = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    cpl_vector* peaks = NULL;
    sph_dataset_stats* datstat = NULL;
    /* Setup */
    cpl_error_reset();

    image = cpl_image_new(248, 248, CPL_TYPE_DOUBLE);

    sph_test_image_tools_add_noise(image, 1.0, NULL);
    //cpl_image_fill_gaussian(image, 124, 124, 1.0, 1.0, 1.0 );
    cpl_image_set(image, 100, 100, 10000.0);
    cpl_image_set(image, 101, 100, 10000.0);
    cpl_image_set(image, 102, 100, 30000.0);
    cpl_image_set(image, 100, 101, 10000.0);
    cpl_image_set(image, 101, 101, 10000.0);
    cpl_image_set(image, 102, 101, 30000.0);
    cpl_image_set(image, 100, 102, 10000.0);
    cpl_image_set(image, 101, 102, 10000.0);
    cpl_image_set(image, 102, 102, 30000.0);

    cpl_image_set(image, 100, 150, 10000.0);
    cpl_image_set(image, 101, 150, 10000.0);
    cpl_image_set(image, 102, 150, 30000.0);
    cpl_image_set(image, 100, 151, 10000.0);
    cpl_image_set(image, 101, 151, 10000.0);
    cpl_image_set(image, 102, 151, 30000.0);
    cpl_image_set(image, 100, 152, 10000.0);
    cpl_image_set(image, 101, 152, 10000.0);
    cpl_image_set(image, 102, 152, 30000.0);

    cpl_image_set(image, 100, 170, 10000.0);
    cpl_image_set(image, 101, 170, 10000.0);
    cpl_image_set(image, 102, 170, 30000.0);
    cpl_image_set(image, 100, 171, 10000.0);
    cpl_image_set(image, 101, 171, 10000.0);
    cpl_image_set(image, 102, 171, 30000.0);
    cpl_image_set(image, 100, 172, 10000.0);
    cpl_image_set(image, 101, 172, 10000.0);
    cpl_image_set(image, 102, 172, 30000.0);

    /* Run ... */
    dat = sph_dataset_new(10, 0.0, 10.0);

    cpl_test_nonnull( dat );
    sph_dataset_collapse(dat, image, 0.0, 247.0);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( dat->xvalues );
    cpl_test_nonnull( dat->yvalues );
    //cpl_vector_dump(dat->xvalues,stdout);
    //cpl_vector_dump(dat->yvalues,stdout);
    datstat = sph_dataset_stats_new(dat);
    sph_dataset_stats_find_peaks(datstat, 1.0, 1);
    peaks = sph_dataset_stats_get_peaks_as_cpl_vector(datstat);
    //cpl_vector_dump(peaks,stdout);
    cpl_vector_delete(peaks);
    sph_dataset_stats_delete(datstat);
    cpl_image_delete(image);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_save function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_save(void) {
    sph_dataset* dat = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cpl_error_reset();
    dat = cutest_util_dataset_new_load();
    cpl_test_nonnull( dat );
    cpl_test_error(CPL_ERROR_NONE);

    /* Run ... */
    rerr = sph_dataset_save(dat, "test_dat.fits");

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    sph_dataset_delete(dat);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_get_max_xpos function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_get_max_xpos(void) {
    sph_dataset* dat = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    double val = 0.0;
    /* Setup */
    cpl_error_reset();
    dat = cutest_util_dataset_new_gauss();
    cpl_test_nonnull( dat );
    cpl_test_error(CPL_ERROR_NONE);

    /* Run ... */
    val = sph_dataset_get_max_xpos(dat);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs( val, 50.0, 0.5);
    sph_dataset_delete(dat);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_get_index function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_get_index(void) {
    sph_dataset* dat = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int val = 0;
    /* Setup */
    cpl_error_reset();
    dat = cutest_util_dataset_new_load();
    cpl_test_nonnull( dat );
    cpl_test_error(CPL_ERROR_NONE);

    /* Run ... */
    val = sph_dataset_get_index(dat, 50.5);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(val, 50);
    sph_dataset_delete(dat);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_add_value function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_add_value(void) {
    sph_dataset* dat = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    double val = 0;
    /* Setup */
    cpl_error_reset();
    dat = cutest_util_dataset_new_load();
    cpl_test_nonnull( dat );
    cpl_test_error(CPL_ERROR_NONE);

    /* Run ... */
    rerr = sph_dataset_add_value(dat, 50.5, 200.0);
    val = sph_dataset_get_value(dat, 50.5);

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs( val, 250, 0.5);
    sph_dataset_delete(dat);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_load function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_load(void) {
    sph_dataset* dat = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /* Setup */
    cpl_error_reset();

    /* Run ... */
    dat = sph_dataset_load("test_dat.fits");
    cpl_test_nonnull( dat );

    /* Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( dat->xvalues );
    cpl_test_nonnull( dat->yvalues );
    cpl_test_eq(cpl_vector_get( dat->xvalues, 0 ), 1.0);
    cpl_test_eq(cpl_vector_get( dat->xvalues, 99 ), 100.0);
    cpl_test_eq(cpl_vector_get( dat->yvalues, 0 ), 0.0);
    cpl_test_eq(cpl_vector_get( dat->yvalues, 99 ), 99.0);
    sph_dataset_delete(dat);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_dataset_delete function. 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_dataset_delete(void) {
    sph_dataset* dat = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    dat = sph_dataset_new(100, 1.0, 100.0);
    cpl_test_nonnull( dat );

    rerr = sph_dataset_delete(dat);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("dataset_test", cutest_init_dataset_testsuite,
            cutest_clean_dataset_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL == sph_test_do(pSuite, "sph_dataset_new", cutest_dataset_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_collapse",
                    cutest_dataset_collapse)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_dataset_save", cutest_dataset_save)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_dataset_load", cutest_dataset_load)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_get_max_xpos",
                    cutest_dataset_get_max_xpos)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_get_index",
                    cutest_dataset_get_index)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_add_value",
                    cutest_dataset_add_value)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_new_pixel_histogram",
                    cutest_dataset_new_pixel_histogram)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_dataset_delete", cutest_dataset_delete)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
