/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_differential_imaging.h"
#include "sph_utils.h"
#include "sph_time.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    return sph_end_test();
}
static
void cutest_sph_differential_imaging_fixture_calibs(int nx, int ny,
        sph_master_frame** dark, sph_master_frame** flat) {

    if (dark) {
        cpl_image* darkim = 
            sph_test_image_tools_create_flat_image_double(nx, ny, 52.0);
        *dark = sph_master_frame_new_from_cpl_image(darkim);
        cpl_image_delete(darkim);
        darkim = NULL;
    }
    if (flat) {
        cpl_image* flatim = 
        sph_test_image_tools_create_flat_image_double(nx, ny, 1.0);

        sph_test_image_tools_multiply_gradient_x(flatim, 0.1, 10.0);
        sph_test_image_tools_multiply_gradient_y(flatim, 0.1, 20.0);
        *flat = sph_master_frame_new_from_cpl_image(flatim);
        cpl_image_delete(flatim);
        flatim = NULL;
    }
    return;
}

static cpl_frameset*
cutest_sph_differential_imaging_fixture_no_star(int nx, int ny) {
    cpl_frameset* frameset = NULL;
    int ii = 0;
    int maxf = 10;
    cpl_frame* aframe = NULL;
    cpl_image* animage = NULL;
    frameset = cpl_frameset_new();
    for (ii = 0; ii < maxf; ++ii) {
        animage = sph_test_image_tools_create_flat_image_double(nx, ny,
                60000.0);
        sph_test_image_tools_multiply_gradient_x(animage, 0.1, 10.0);
        sph_test_image_tools_multiply_gradient_y(animage, 0.1, 20.0);
        cpl_image_add_scalar(animage, 52.0);
        //sph_test_image_tools_apply_poisson_noise(animage,NULL);
        aframe = sph_test_frame_image(animage, CPL_BPP_32_SIGNED,
                "TESTING_RAW_SCIENCE_FRAME", CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frameset, aframe);
        cpl_image_delete(animage);
        animage = NULL;
    }

    return frameset;
}

static cpl_frameset*
cutest_sph_differential_imaging_fixture_standard_adi(int nx, int ny) {
    cpl_frameset* frameset = NULL;
    int ii = 0;
    int maxf = 10;
    cpl_frame* aframe = NULL;
    cpl_image* animage = NULL;
    frameset = cpl_frameset_new();
    for (ii = 0; ii < maxf; ++ii) {
        animage = sph_test_image_tools_create_flat_image_double(nx, ny, 600.0);
        sph_test_image_tools_add_gauss(animage, nx / 2.0 + 0.5, ny / 2.0 + 0.5,
                nx / 100.0, 100000.0);
        sph_test_image_tools_add_gauss(animage, nx / 2.0 + 20.5, ny / 2.0 + 0.5,
                nx / 100.0, 10000.0);
        sph_test_image_tools_multiply_gradient_x(animage, 0.1, 10.0);
        sph_test_image_tools_multiply_gradient_y(animage, 0.1, 20.0);
        cpl_image_add_scalar(animage, 52.0);
        sph_test_image_tools_apply_poisson_noise(animage, NULL);
        aframe = sph_test_frame_image(animage, CPL_BPP_32_SIGNED,
                "TESTING_RAW_SCIENCE_FRAME", CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frameset, aframe);
        cpl_image_delete(animage);
        animage = NULL;
    }

    return frameset;
}
static cpl_frameset*
cutest_sph_differential_imaging_fixture_already_reduced(int nx, int ny) {
    cpl_frameset* frameset = NULL;
    int ii = 0;
    int maxf = 10;
    cpl_frame* aframe = NULL;
    cpl_image* animage = NULL;
    frameset = cpl_frameset_new();
    for (ii = 0; ii < maxf; ++ii) {
        animage = sph_test_image_tools_create_flat_image_double(nx, ny, 600.0);
        sph_test_image_tools_add_gauss(animage, nx / 2.0 + 0.5, ny / 2.0 + 0.5,
                nx / 100.0, 100000.0);
        sph_test_image_tools_add_gauss(animage, nx / 2.0 + 20.5, ny / 2.0 + 0.5,
                nx / 100.0, 10000.0);
        sph_test_image_tools_apply_poisson_noise(animage, NULL);
        aframe = sph_test_frame_image(animage, CPL_BPP_32_SIGNED,
                "TESTING_RED_SCIENCE_FRAME", CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frameset, aframe);
        cpl_image_delete(animage);
        animage = NULL;
        animage = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
        cpl_image_save(animage, cpl_frame_get_filename(aframe),
                CPL_BPP_32_SIGNED, NULL, CPL_IO_EXTEND);
        cpl_image_delete(animage);
        animage = NULL;
    }
    return frameset;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_differential_imaging_new_test(void) {
    sph_differential_imaging* diffmod = NULL;
    cpl_frameset* inframeset = NULL;
    diffmod = sph_differential_imaging_new(inframeset);
    cpl_test_nonnull( diffmod );
    sph_differential_imaging_delete(diffmod);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_differential_imaging_get_next_raw_test(void) {
    sph_differential_imaging* diffmod = NULL;
    cpl_frameset* inframeset = NULL;
    cpl_image* im = NULL;
    int c = 0;

    inframeset = cutest_sph_differential_imaging_fixture_standard_adi(256, 256);
    diffmod = sph_differential_imaging_new(inframeset);
    cpl_test_nonnull( diffmod );
    im = sph_differential_imaging_get_next_raw(diffmod);
    while (im) {
        cpl_image_delete(im);
        im = NULL;
        im = sph_differential_imaging_get_next_raw(diffmod);
        c++;
    }cpl_test_eq(c, 10);
    sph_differential_imaging_delete(diffmod);
    diffmod = NULL;
    cpl_frameset_delete(inframeset);
    inframeset = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_differential_imaging_get_next_calib_raw_test(void) {
    sph_differential_imaging* diffmod = NULL;
    cpl_frameset* inframeset = NULL;
    cpl_image* im = NULL;
    int c = 0;
    int nx = 256;
    int ny = 256;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;

    inframeset = cutest_sph_differential_imaging_fixture_no_star(nx, ny);
    cutest_sph_differential_imaging_fixture_calibs(nx, ny, &dark, &flat);
    diffmod = sph_differential_imaging_new(inframeset);
    cpl_test_nonnull( diffmod );
    sph_differential_imaging_set_calibrations(diffmod, dark, flat, NULL);
    im = sph_differential_imaging_get_next_calibrated_raw(diffmod, NULL, NULL);
    while (im) {
        cpl_image_delete(im);
        im = NULL;
        im = sph_differential_imaging_get_next_calibrated_raw(diffmod, NULL,
                NULL);
        if (im) {
            cpl_test_abs(cpl_image_get_mean(im), 60000.0, 10.0);
            cpl_test_abs(cpl_image_get_median(im), 60000.0, 10.0);
            cpl_test_abs(cpl_image_get_max(im), 60000.0, 10.0);
            cpl_test_abs(cpl_image_get_min(im), 60000.0, 100.0);
        }
        c++;
    }cpl_test_eq(c, 10);
    sph_differential_imaging_delete(diffmod);
    diffmod = NULL;
    cpl_frameset_delete(inframeset);
    inframeset = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_differential_imaging_calc_speckles_test(void) {
    sph_master_frame* flat = NULL;
    sph_master_frame* dark = NULL;
    cpl_frameset* inframes = NULL;
    int nx = 512;
    int ny = 512;
    sph_distortion_model* distmodel = NULL;
    cpl_frameset* speckle_frameset = NULL;

    inframes = cutest_sph_differential_imaging_fixture_already_reduced(nx, ny);
    sph_time_start_timer(1);
    speckle_frameset = sph_differential_imaging_calculate_speckles(inframes, 3,
            SPH_COLL_ALG_MEAN, NULL);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Timer time: %f", sph_time_stop_timer(1));
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_assert(speckle_frameset);
    cpl_test_eq(cpl_frameset_get_size(speckle_frameset), 4);
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_distortion_model_delete(distmodel);
    distmodel = NULL;
    cpl_frameset_delete(speckle_frameset);
    speckle_frameset = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_differential_imaging_calc_speckles_test_warray(void) {
    sph_master_frame* flat = NULL;
    sph_master_frame* dark = NULL;
    cpl_frameset* inframes = NULL;
    int nx = 512;
    int ny = 512;
    sph_distortion_model* distmodel = NULL;
    cpl_frameset* speckle_frameset = NULL;
    cpl_array* array = NULL;
    int bp = 0;

    inframes = cutest_sph_differential_imaging_fixture_already_reduced(nx, ny);
    sph_time_start_timer(1);
    speckle_frameset = sph_differential_imaging_calculate_speckles(inframes, 3,
            SPH_COLL_ALG_MEAN, &array);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Timer time: %f", sph_time_stop_timer(1));
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_assert(speckle_frameset);
    cpl_test_assert(array);
    cpl_test_eq(cpl_array_get_size(array), 10);
    cpl_test_zero(cpl_array_get_int(array,0,&bp));
    cpl_test_zero(cpl_array_get_int(array,1,&bp));
    cpl_test_zero(cpl_array_get_int(array,2,&bp));
    cpl_test_eq(cpl_array_get_int(array,3,&bp), 1);
    cpl_test_eq(cpl_array_get_int(array,4,&bp), 1);
    cpl_test_eq(cpl_array_get_int(array,5,&bp), 1);
    cpl_test_eq(cpl_array_get_int(array,6,&bp), 2);
    cpl_test_eq(cpl_array_get_int(array,7,&bp), 2);
    cpl_test_eq(cpl_array_get_int(array,8,&bp), 2);
    cpl_test_eq(cpl_array_get_int(array,9,&bp), 3);
    cpl_test_eq(cpl_frameset_get_size(speckle_frameset), 4);
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_distortion_model_delete(distmodel);
    distmodel = NULL;
    cpl_frameset_delete(speckle_frameset);
    speckle_frameset = NULL;
    cpl_array_delete(array);
    array = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("A testsuite", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing constructor",
                    cutest_sph_differential_imaging_new_test)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing getting next raw",
                    cutest_sph_differential_imaging_get_next_raw_test)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing getting next calib raw",
                    cutest_sph_differential_imaging_get_next_calib_raw_test)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing calculating speckles",
                    cutest_sph_differential_imaging_calc_speckles_test)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing calculating speckles w array",
                    cutest_sph_differential_imaging_calc_speckles_test_warray)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
