/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_distortion_map.h"
#include "sph_point_pattern.h"
#include "sph_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

static sph_point_pattern* cutest_create_pp(int np) {
    sph_point_pattern* pp = sph_point_pattern_new_(np);
    int ii;
    for (ii = 0; ii < np; ++ii) {
        sph_point_pattern_add_point(pp, ii * 10.0, ii * 10.0);
    }
    return pp;
}
static sph_point_pattern* cutest_create_pp_offset(int np, double off) {
    sph_point_pattern* pp = sph_point_pattern_new_(np);
    int ii;
    for (ii = 0; ii < np; ++ii) {
        sph_point_pattern_add_point(pp, ii * 10.0 + off, ii * 10.0 - off);
    }
    return pp;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_constructor_two_point_sets(void) {
    sph_point_pattern* ppA = cutest_create_pp(10);
    sph_point_pattern* ppB = cutest_create_pp_offset(10, 0.2);
    sph_distortion_map* map = NULL;

    cpl_test_zero(sph_distortion_map_get_size(NULL));
    cpl_test_error(CPL_ERROR_NONE);

    map = sph_distortion_map_new(ppA, ppB);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( map );
    sph_distortion_map_delete_multi(map);

    cpl_test_eq(sph_distortion_map_get_size(map), 10);
    cpl_test_error(CPL_ERROR_NONE);
    sph_distortion_map_delete(map);
    sph_point_pattern_delete(ppA);
    sph_point_pattern_delete(ppB);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Distortion Map Testsuite", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite,
                    "Test Distortion Map constructor, two point sets as input",
                    cutest_test_constructor_two_point_sets)) {
        return sph_test_get_error();
    }
    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
