/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_extract_angles.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_ird_instrument_model.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"
#include "sph_fits.h"
#include "sph_time.h"
#include "sph_skycalc.h"
#include "sph_coord.h"
#include "sph_keyword_manager.h"
#include "sph_common_keywords.h"
#include <cpl.h>

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

static
void cutest_sph_extract_angles_test_bad_input(void);
static
void cutest_sph_extract_angles_test_irdis(void);
static
int cutest_init_testsuite(void);
static
int cutest_clean_testsuite(void);

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Exctract angles functionality testsuite",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite, "Testing for null input errors",
                    cutest_sph_extract_angles_test_bad_input)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing IRDIS angles correctly extracted",
                    cutest_sph_extract_angles_test_irdis)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/*----------------------------------------------------------------------------*/
/*-                            TESTS                                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_extract_angles_test_bad_input(void) {
    cpl_image* testimage = NULL;
    cpl_frame* input_frame = NULL;
    cpl_vector* angles = NULL;

    testimage = sph_test_image_tools_create_grid_image(256, 256, 10, 10,
            CPL_TYPE_INT, 10000, 2.0);

    cpl_test_nonnull( testimage );

    input_frame = sph_test_cubify_image(testimage, "TEST_CUBE_TAG",
            CPL_FRAME_GROUP_RAW);

    angles = sph_extract_angles_from_cube(NULL, 0.0, 0.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(angles);
    angles = sph_extract_angles_from_cube(input_frame, 0.0, 0.0);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_null(angles);

    cpl_image_delete(testimage);
    testimage = NULL;
    cpl_frame_delete(input_frame);
    input_frame = NULL;
    cpl_vector_delete(angles);
    angles = NULL;

    cpl_error_reset();
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_extract_angles_test_irdis(void) {
    cpl_image* testimage = NULL;
    cpl_frame* input_frame = NULL;
    cpl_vector* angles = NULL;
    double mjd = 0.0;

    testimage = sph_test_image_tools_create_grid_image(256, 256, 10, 10,
            CPL_TYPE_INT, 10000, 2.0);

    cpl_test_nonnull( testimage );

    input_frame = sph_test_cubify_image(testimage, "TEST_CUBE_TAG",
            CPL_FRAME_GROUP_RAW);

    sph_test_frame_add_datestamp(input_frame);

    sph_test_frame_add_key_string(input_frame, 0,
            SPH_COMMON_KEYWORD_DET_FRAM_UTC, "2000-01-05T13:00:00");

    sph_time_mjd_from_string(&mjd, "2000-01-05T12:00:00");

    sph_test_frame_add_key_double(input_frame, 0, SPH_COMMON_KEYWORD_MJD_OBS,
            mjd);

    sph_test_frame_add_key_double(input_frame, 0, SPH_COMMON_KEYWORD_EPOCH,
                                  2000);
    sph_test_frame_add_key_double(input_frame, 0, SPH_COMMON_KEYWORD_DEC, 0.0);
    sph_test_frame_add_key_double(input_frame, 0, SPH_COMMON_KEYWORD_RA, 0.0);

    angles = sph_extract_angles_from_cube(input_frame,
                                      SPH_IRD_INSTRUMENT_MODEL_DEROT_OFFSETELEV,
                                          0.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( angles );

    cpl_test_eq(cpl_vector_get_size(angles), 1);

    // TODO: insert check for actual values once we have verified with some real data

    cpl_image_delete(testimage);
    testimage = NULL;
    cpl_frame_delete(input_frame);
    input_frame = NULL;
    cpl_vector_delete(angles);
    angles = NULL;

    cpl_error_reset();
    return;
}
/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}

/**@}*/
