/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_line.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_line_testsuite(void) {
    return sph_test_nop_code();
}

static
int cutest_clean_line_testsuite(void) {
    sph_error_dump(0);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_line_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_line_new(void) {
    sph_line* line = NULL;

    line = sph_line_new(0.0, 0.0, 1.0, 1.0);

    cpl_test_nonnull( line );

    sph_line_delete(line);
    line = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_line_intersect_test function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_line_intersect_test(void) {
    sph_line* lineA = NULL;
    sph_line* lineB = NULL;
    sph_line* lineC = NULL;
    sph_line* lineD = NULL;
    sph_line* lineE = NULL;
    sph_line* lineF = NULL;
    int result = 0;

    lineA = sph_line_new(0.0, 0.0, 1.0, 1.0);
    lineB = sph_line_new(0.0, 1.0, 1.0, 0.0);
    lineC = sph_line_new(1.0, 0.0, 2.0, 1.0);
    lineD = sph_line_new(0.0, 10.0, 10.0, 0.0);
    lineE = sph_line_new(10.0, 0.0, 20.0, 0.0);
    lineF = sph_line_new(20.0, 0.0, 30.0, 0.0);

    cpl_test_nonnull( lineA );
    cpl_test_nonnull( lineB );
    cpl_test_nonnull( lineC );
    cpl_test_nonnull( lineD );
    cpl_test_nonnull( lineE );
    cpl_test_nonnull( lineF );

    result = sph_line_intersect_test(lineA, lineB);
    cpl_test_eq(result, 1);
    result = sph_line_intersect_test(lineA, lineC);
    cpl_test_zero(result);
    result = sph_line_intersect_test(lineA, lineD);
    cpl_test_zero(result);
    result = sph_line_intersect_test(lineE, lineD);
    cpl_test_eq(result, 1);
    result = sph_line_intersect_test(lineD, lineE);
    cpl_test_eq(result, 1);
    result = sph_line_intersect_test(lineF, lineE);
    cpl_test_eq(result, 1);
    result = sph_line_intersect_test(lineE, lineF);
    cpl_test_eq(result, 1);
    result = sph_line_intersect_test(lineE, lineE);
    cpl_test_eq(result, 1);
    sph_line_delete(lineA);
    lineA = NULL;
    sph_line_delete(lineB);
    lineB = NULL;
    sph_line_delete(lineC);
    lineC = NULL;
    sph_line_delete(lineD);
    lineD = NULL;
    sph_line_delete(lineE);
    lineE = NULL;
    sph_line_delete(lineF);
    lineF = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_line_intersect_point function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_line_intersect_point(void) {
    sph_line* lineA = NULL;
    sph_line* lineB = NULL;
    sph_line* lineC = NULL;
    sph_line* lineD = NULL;
    sph_line* lineE = NULL;
    sph_line* lineF = NULL;
    sph_point* result = NULL;
    cpl_error_code code;

    lineA = sph_line_new(0.0, 0.0, 1.0, 1.0);
    lineB = sph_line_new(0.0, 1.0, 1.0, 0.0);
    lineC = sph_line_new(1.0, 0.0, 2.0, 1.0);
    lineD = sph_line_new(0.0, -10.0, -10.0, 0.0);
    lineE = sph_line_new(-10.0, 0.0, -20.0, 0.0);
    lineF = sph_line_new(-20.0, 0.0, -30.0, 0.0);

    cpl_test_nonnull( lineA );
    cpl_test_nonnull( lineB );
    cpl_test_nonnull( lineC );
    cpl_test_nonnull( lineD );
    cpl_test_nonnull( lineE );
    cpl_test_nonnull( lineF );
    result = sph_point_new(1.0, 1.0);
    cpl_test_nonnull( result );

    code = sph_line_intersect_point(result, lineA, lineB);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs( result->x, 0.5, 1.0e-10);
    cpl_test_abs( result->y, 0.5, 1.0e-10);

    code = sph_line_intersect_point(result, lineA, lineD);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs( result->x, -5.0, 1.0e-10);
    cpl_test_abs( result->y, -5.0, 1.0e-10);

    code = sph_line_intersect_point(result, lineE, lineD);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs( result->x, -10.0, 1.0e-10);
    cpl_test_abs( result->y, 0.0, 1.0e-10);

    code = sph_line_intersect_point(result, lineD, lineE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs( result->x, -10.0, 1.0e-10);
    cpl_test_abs( result->y, 0.0, 1.0e-10);

    code = sph_line_intersect_point(result, lineE, lineF);
    cpl_test_eq_error(code, CPL_ERROR_DIVISION_BY_ZERO);
    cpl_test_abs( result->x, 0.0, 0.0);
    cpl_test_abs( result->y, 0.0, 0.0);

    code = sph_line_intersect_point(result, lineE, lineE);
    cpl_test_eq_error(code, CPL_ERROR_DIVISION_BY_ZERO);
    cpl_test_abs( result->x, 0.0, 0.0);
    cpl_test_abs( result->y, 0.0, 0.0);

    sph_point_delete(result);
    sph_line_delete(lineA);
    sph_line_delete(lineB);
    sph_line_delete(lineC);
    sph_line_delete(lineD);
    sph_line_delete(lineE);
    sph_line_delete(lineF);
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_line_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_line_delete(void) {
    sph_line* line = NULL;

    line = sph_line_new(0.0, 0.0, 1.0, 1.0);

    cpl_test_nonnull( line );

    sph_line_delete(line);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("line_test", cutest_init_line_testsuite,
            cutest_clean_line_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL == sph_test_do(pSuite, "sph_line_new", cutest_line_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_line_intersect_test",
                    cutest_line_intersect_test)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_line_intersect_point",
                    cutest_line_intersect_point)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "sph_line_delete", cutest_line_delete)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
