/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_overlap_calculator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

#include <math.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and 
 *                                  associated functions.               
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static sph_overlap_calculator*
cutest_overlap_calculator_help_create_zero(void) {
    sph_overlap_calculator* overlap_calc = NULL;

    overlap_calc = cpl_calloc(1, sizeof(sph_overlap_calculator));
    cpl_test_nonnull( overlap_calc );
    return overlap_calc;
}
static sph_overlap_calculator*
cutest_overlap_calculator_help_create_simple(void) {
    sph_overlap_calculator* overlap_calc = NULL;
    int xx = 0;
    int yy = 0;
    overlap_calc = cpl_calloc(1, sizeof(sph_overlap_calculator));
    cpl_test_nonnull( overlap_calc );
    overlap_calc->resolution = 20;
    overlap_calc->im = cpl_image_new(20, 20, CPL_TYPE_DOUBLE);
    for (xx = 0; xx < 20; ++xx) {
        for (yy = 0; yy < 20; ++yy) {
            cpl_image_reject(overlap_calc->im, xx + 1, yy + 1);
        }
    }
    overlap_calc->poly = sph_polygon_new();
    sph_polygon_add_point(overlap_calc->poly, 0.0, 0.0);
    sph_polygon_add_point(overlap_calc->poly, 1.0, 0.0);
    sph_polygon_add_point(overlap_calc->poly, 1.0, 1.0);
    sph_polygon_add_point(overlap_calc->poly, 0.0, 1.0);
    overlap_calc->maxx = 1.0;
    overlap_calc->maxy = 1.0;
    overlap_calc->minx = -1.0;
    overlap_calc->miny = -1.0;
    overlap_calc->dy = 2.0 / 20.0;
    overlap_calc->dx = 2.0 / 20.0;
    return overlap_calc;
}

static sph_polygon*
cutest_overlap_calculator_help_create_hex(double t) {
    sph_polygon* poly = NULL;
    poly = sph_polygon_new();
    sph_polygon_add_point(poly, 0.0, -t);
    sph_polygon_add_point(poly, 0.5 * CPL_MATH_SQRT3 * t, -0.5 * t);
    sph_polygon_add_point(poly, 0.5 * CPL_MATH_SQRT3 * t, 0.5 * t);
    sph_polygon_add_point(poly, 0.0, t);
    sph_polygon_add_point(poly, -0.5 * CPL_MATH_SQRT3 * t, 0.5 * t);
    sph_polygon_add_point(poly, -0.5 * CPL_MATH_SQRT3 * t, -0.5 * t);
    return poly;
}
static
int cutest_sph_overlap_calculator_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int cutest_sph_overlap_calculator_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_INFO);
    return sph_end_test();
}

static
void cutest_overlap_calculator_test_new(void) {
    sph_overlap_calculator* overlap_calc = NULL;
    sph_polygon* poly = NULL;

    overlap_calc = sph_overlap_calculator_new(NULL, 0.0, -1);
    cpl_test_null(overlap_calc);
    cpl_test_noneq( cpl_error_get_code(), CPL_ERROR_NONE);
    cpl_error_reset();
    poly = cutest_overlap_calculator_help_create_hex(1.0);
    overlap_calc = sph_overlap_calculator_new(poly, 1.0, -1);
    cpl_test_nonnull( overlap_calc );
    cpl_test_assert(overlap_calc->im);
    cpl_test_assert(overlap_calc->poly);
    cpl_test_error(CPL_ERROR_NONE);
    sph_polygon_delete(poly);
    poly = NULL;
    sph_overlap_calculator_delete(overlap_calc);
}
static
void cutest_overlap_calculator_test_get_overlap(void) {
    sph_overlap_calculator* over_calc =
            cutest_overlap_calculator_help_create_simple();
    double result = 0.0;

    cpl_test_error(CPL_ERROR_NONE);
    result = sph_overlap_calculator_get_overlap(over_calc, 0.0, 0.0);

    cpl_test_abs(result, 1.0, 0.000000001);
    cpl_test_error(CPL_ERROR_NONE);
    sph_overlap_calculator_delete(over_calc);
    cpl_error_reset();
}

static
void cutest_overlap_calculator_test_precalc(void) {
    sph_overlap_calculator* over_calc =
            cutest_overlap_calculator_help_create_simple();
    double result = 0.0;
    cpl_mask* m = NULL;
    cpl_test_error(CPL_ERROR_NONE);

    sph_overlap_calculator_precalc(over_calc);
    cpl_test_error(CPL_ERROR_NONE);
    m = cpl_image_get_bpm(over_calc->im);
    cpl_test_nonnull( m );
    cpl_test( cpl_mask_is_empty(m));
    result = sph_overlap_calculator_get_overlap(over_calc, 0.0, 0.0);

    cpl_test_abs(result, 1.0, 0.1);
    cpl_test_error(CPL_ERROR_NONE);
    sph_overlap_calculator_delete(over_calc);
    cpl_error_reset();
}

static
void cutest_overlap_calculator_test_get_overlap_test_accuracy(void) {
    sph_overlap_calculator* over_calc = NULL;
    sph_polygon_workspace* ws = NULL;
    sph_polygon* hex = NULL;
    sph_polygon* pixelpoly = NULL;
    gsl_rng* pRNG = NULL;
    int ii = 0;
    double offx = 0.0;
    double offy = 0.0;
    double lx = 0.0;
    double ly = 0.0;
    double result = 0.0;
    double comparison = 0.0;

    pRNG = gsl_rng_alloc(gsl_rng_taus);

    ws = sph_polygon_workspace_new();
    hex = cutest_overlap_calculator_help_create_hex(0.5);
    lx = sph_polygon_get_left(hex);
    ly = sph_polygon_get_bottom(hex);
    over_calc = sph_overlap_calculator_new(hex, 2.0, -1);

    cpl_test_error(CPL_ERROR_NONE);

    for (ii = 0; ii < 1000000; ++ii) {
        offx = gsl_ran_flat(pRNG, -1.0, 1.0);
        offy = gsl_ran_flat(pRNG, -1.0, 1.0);
        pixelpoly = sph_polygon_new();
        sph_polygon_add_point(pixelpoly, offx + lx, offy + ly);
        sph_polygon_add_point(pixelpoly, offx + 1.0 + lx, offy + 0.0 + ly);
        sph_polygon_add_point(pixelpoly, offx + 1.0 + lx, offy + 1.0 + ly);
        sph_polygon_add_point(pixelpoly, offx + 0.0 + lx, offy + 1.0 + ly);
        result = sph_overlap_calculator_get_overlap(over_calc, offx, offy);
        comparison = sph_polygon_calculate_overlap(hex, pixelpoly, ws);
        if (fabs(result - comparison) > 0.03) {
            printf(
                    "Res: %d, At offset: %f, %f -> result:%f, comp: %f, diff: %f\n",
                    over_calc->resolution, offx, offy, result, comparison,
                    fabs(result - comparison));
            sph_test_later("Unaccurate value");
        }
        sph_polygon_delete(pixelpoly);
        pixelpoly = NULL;
    }

    gsl_rng_free(pRNG);
    cpl_test_error(CPL_ERROR_NONE);
    sph_overlap_calculator_delete(over_calc);
    sph_polygon_workspace_delete(ws);
    sph_polygon_delete(hex);
    cpl_error_reset();
}

static
void cutest_overlap_calculator_test_delete(void) {
    sph_overlap_calculator* overlap_calc = NULL;

    overlap_calc = cutest_overlap_calculator_help_create_simple();
    sph_overlap_calculator_delete(overlap_calc);
    cpl_test_error(CPL_ERROR_NONE);
    overlap_calc = cutest_overlap_calculator_help_create_zero();
    sph_overlap_calculator_delete(overlap_calc);
    cpl_test_error(CPL_ERROR_NONE);
    sph_overlap_calculator_delete(NULL);
    cpl_test_error(CPL_ERROR_NONE);
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("sph_overlap_calculator_testsuite",
            cutest_sph_overlap_calculator_init_testsuite,
            cutest_sph_overlap_calculator_clean_testsuite);

    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_overlap_calculator_new",
                    cutest_overlap_calculator_test_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_overlap_calculator_get_overlap",
                    cutest_overlap_calculator_test_get_overlap)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_overlap_calculator_get_overlap_test_accuracy",
                    cutest_overlap_calculator_test_get_overlap_test_accuracy)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_overlap_calculator_delete",
                    cutest_overlap_calculator_test_delete)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_overlap_calculator_precalc",
                    cutest_overlap_calculator_test_precalc)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
