/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>
#include <float.h>

#include "sph_pixel_description_table.h"
#include "sph_keyword_manager.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"

extern sph_error_code
sph_pixel_description_table_set_ids__(sph_pixel_description_table* pdt,
        sph_ifs_lenslet_model* model);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_pixel_description_table    unit tests for sph_pixel_description_table
 * and associated functions.
 */
/*----------------------------------------------------------------------------*/
static sph_ifs_lenslet_model*
cutest_pixel_description_table_create_very_small_lenslet_model__(void) {
    sph_ifs_lenslet_model* lensmodel = NULL;
    lensmodel = sph_ifs_lenslet_model_new();
    cpl_test_nonnull( lensmodel );
    //lensmodel->zero_offsetx = 0.0;
    //lensmodel->zero_offsety = 0.0;
    lensmodel->detsize_pixels = 128;
    lensmodel->lenslets_per_side = 3;
    return lensmodel;
}

static sph_pixel_description_table*
cutest_pixel_description_table_new_from_model_very_small__(double dx, double dy) {
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* lensmodel = NULL;
    /* Setup and run ...*/

    lensmodel =
            cutest_pixel_description_table_create_very_small_lenslet_model__();
    pdt = sph_pixel_description_table_new_from_model(lensmodel, dx, dy);
    sph_ifs_lenslet_model_delete(lensmodel);
    return pdt;
}

static
int cutest_pixel_description_table_verify_pdts_identical__(
        sph_pixel_description_table* pdt, sph_pixel_description_table* pdt2) {
    int xx = 0;
    int yy = 0;
    cpl_image* iddiff = NULL;
    cpl_image* wdiff = NULL;
    sph_pixel_descriptor* pdescr = NULL;
    sph_pixel_descriptor* pdescr2 = NULL;
    int offx = 0;
    int offy = 0;
    int result = 1;

    iddiff = cpl_image_new(pdt->nx, pdt->ny, CPL_TYPE_DOUBLE);
    wdiff = cpl_image_new(pdt->nx, pdt->ny, CPL_TYPE_DOUBLE);

    offx = (int) pdt2->offx - (int) pdt->offx;
    offy = (int) pdt2->offy - (int) pdt->offy;

    for (xx = 0; xx < pdt->nx; ++xx) {
        for (yy = 0; yy < pdt->ny; ++yy) {
            pdescr = sph_pixel_description_table_get_descriptor(pdt, xx, yy);
            if (pdescr) {
                if (xx + offx >= 0 && yy + offy >= 0 && xx + offx < pdt->nx
                        && yy + offy < pdt->ny) {
                    pdescr2 = sph_pixel_description_table_get_descriptor(pdt2,
                            xx + offx, yy + offy);
                    if (!pdescr2)
                        result = -1;
                    else if (pdescr->specid && pdescr2->specid) {
                        if (pdescr->lensid != pdescr2->lensid) {
                            result = -2;
                            cpl_image_set(iddiff, xx + 1, yy + 1,
                                    pdescr->lensid - pdescr2->lensid);
                        }
                        if (fabs(pdescr->wavelength - pdescr2->wavelength)
                                > 0.0001) {
                            result = -3;
                            cpl_image_set(wdiff, xx + 1, yy + 1,
                                    pdescr->wavelength - pdescr2->wavelength);
                        }
                    }
                }
            } else
                result = -1;
        }
    }
    if (result != 1) {
        sph_pixel_description_table_save(pdt, "UNMATCHING_PDT_1.fits", NULL);
        sph_pixel_description_table_save(pdt2, "UNMATCHING_PDT_2.fits", NULL);
        sph_test_frame_image(iddiff, CPL_TYPE_DOUBLE, "TEST",
                CPL_FRAME_GROUP_RAW);
        sph_test_frame_image(wdiff, CPL_TYPE_DOUBLE, "TEST",
                CPL_FRAME_GROUP_RAW);
    }
    cpl_image_delete(iddiff);
    cpl_image_delete(wdiff);
    return result;
}

/**@{*/
static
int cutest_init_pixel_description_table_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    cpl_error_reset();
    return 0;
}

static
int cutest_clean_pixel_description_table_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_pixel_description_table_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_new(void) {
    sph_pixel_description_table* model = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/

    model = sph_pixel_description_table_new(10, 10, 0.0, 0.0);

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( model );
    cpl_test_eq(model->nx, 10);
    cpl_test_eq(model->ny, 10);
    cpl_test_zero(model->idlongest);
    cpl_test_eq(model->maxlength, -1);
    cpl_test_zero(model->nregions);
    cpl_test_abs( model->offx, 0.0, 0.0);
    cpl_test_abs( model->offy, 0.0, 0.0);

    sph_pixel_description_table_delete(model);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_pixel_description_table_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_new_from_model_very_small(void) {
    sph_pixel_description_table* pdt = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* maskimage = NULL;
    cpl_image* wavimage = NULL;
    cpl_mask* mask = NULL;
    int absflux = 0;
    /* Setup and run ...*/
    pdt = cutest_pixel_description_table_new_from_model_very_small__(0.0, 0.0);

    cpl_test_nonnull( pdt );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( pdt );
    cpl_test_eq(pdt->nx, 128);
    cpl_test_eq(pdt->ny, 128);
    cpl_test_abs( pdt->offx, 0.0, 0.0);
    cpl_test_abs( pdt->offy, 0.0, 0.0);

    mask = sph_pixel_description_table_get_mask(pdt);
    maskimage = cpl_image_new_from_mask(mask);
    cpl_mask_delete(mask);
    mask = NULL;
    cpl_image_save(maskimage, "pdt_mask.fits", CPL_TYPE_FLOAT, NULL,
            CPL_IO_DEFAULT);

    absflux = cpl_image_get_flux(maskimage);
    cpl_test_eq(absflux, 5712);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Flux in PDT is: %d", absflux);

    wavimage = sph_pixel_description_table_get_wavimage(pdt);
    cpl_image_save(wavimage, "pdt_wavim.fits", CPL_TYPE_FLOAT, NULL,
            CPL_IO_DEFAULT);
    cpl_test_nonnull( wavimage );
    cpl_test_lt(1.0, cpl_image_get_flux(wavimage)/absflux);

    sph_pixel_description_table_delete(pdt);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(wavimage);
    cpl_image_delete(maskimage);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_pixel_description_table_new function.
 
 This bug was really a toughie. The problem was actually in the sph_polynomial
 _distortion_map saving and loading which used to cut off all coefficients
 after the 3rd when saving. This lead to to a possible difference between
 distortion maps before saving to after saving and reloading again.
 
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_new_from_model_save_load_bug(void) {
    sph_pixel_description_table* pdt = NULL;
    sph_pixel_description_table* pdt2 = NULL;
    sph_ifs_lenslet_model* lensmodel = NULL;
    sph_ifs_lenslet_model* lensmodel2 = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* maskimage = NULL;
    cpl_mask* mask = NULL;
    double absflux = 0;
    cpl_size pows[2] = { 0, 0 };
    cpl_propertylist* pl = NULL;
    int iverify;
    /* Setup and run ...*/

    lensmodel = sph_ifs_lenslet_model_new();
    lensmodel->speclength_pixels = 39.0;
    lensmodel->specwidth_pixels = 4.94181266489202;
    lensmodel->rotangle = -10.8933946491309;
    lensmodel->zero_offsetx = 2.05216774814812;
    lensmodel->zero_offsety = 5.35259167926597;
    lensmodel->stretch_x = 1.0023960780714;
    lensmodel->stretch_y = 1.0023960780714;
    lensmodel->minlambda = 0.95;
    lensmodel->maxlambda = 1.35;
    lensmodel->dispersion = (lensmodel->maxlambda - lensmodel->minlambda)
            / (lensmodel->speclength_pixels - 1);
    lensmodel->detsize_pixels = 256;
    lensmodel->detsize_microns = lensmodel->pixsize_microns
            * lensmodel->detsize_pixels;
    lensmodel->speclength_microns = lensmodel->pixsize_microns
            * lensmodel->speclength_pixels;
    lensmodel->specwidth_microns = lensmodel->pixsize_microns
            * lensmodel->specwidth_pixels;
    lensmodel->detector_distortion = sph_distortion_model_new(NULL,
            NULL);

    pows[0] = 0;
    pows[1] = 4;
    cpl_polynomial_set_coeff(lensmodel->detector_distortion->polyx, pows,
            1.47581883794879E-11);
    cpl_polynomial_set_coeff(lensmodel->detector_distortion->polyy, pows,
            -9.02606306813477E-11);
    pows[0] = 4;
    pows[1] = 0;
    cpl_polynomial_set_coeff(lensmodel->detector_distortion->polyx, pows,
            1.47581883794879E-11);
    cpl_polynomial_set_coeff(lensmodel->detector_distortion->polyy, pows,
            -9.02606306813477E-11);

    pl = sph_ifs_lenslet_model_get_as_propertylist(lensmodel);
    lensmodel2 = sph_ifs_lenslet_model_new_from_propertylist(pl);

    cpl_test_abs(lensmodel->detsize_microns,
            lensmodel2->detsize_microns, 1.0e-15);
    cpl_test_abs(lensmodel->dispersion, lensmodel2->dispersion,
            1.0e-15);
    cpl_test_abs(lensmodel->lensize_microns,
            lensmodel2->lensize_microns, 1.0e-15);
    cpl_test_abs(lensmodel->maxlambda, lensmodel2->maxlambda,
            1.0e-15);
    cpl_test_abs(lensmodel->minlambda, lensmodel2->minlambda,
            1.0e-15);
    cpl_test_abs(lensmodel->pixsize_microns,
            lensmodel2->pixsize_microns, 1.0e-15);
    cpl_test_abs(lensmodel->rotangle, lensmodel2->rotangle, 1.0e-15);
    cpl_test_abs(lensmodel->speclength_microns,
            lensmodel2->speclength_microns, 1.0e-15);
    cpl_test_abs(lensmodel->specwidth_microns,
            lensmodel2->specwidth_microns, 1.0e-15);
    cpl_test_abs(lensmodel->speclength_pixels,
            lensmodel2->speclength_pixels, 1.0e-15);
    cpl_test_abs(lensmodel->specwidth_pixels,
            lensmodel2->specwidth_pixels, 1.0e-15);
    cpl_test_abs(lensmodel->stretch_x, lensmodel2->stretch_x,
            1.0e-15);
    cpl_test_abs(lensmodel->stretch_y, lensmodel2->stretch_y,
            1.0e-15);
    cpl_test_abs(lensmodel->zero_offsetx, lensmodel2->zero_offsetx,
            1.0e-15);
    cpl_test_abs(lensmodel->zero_offsety, lensmodel2->zero_offsety,
            1.0e-15);
    cpl_test_eq(lensmodel->detsize_pixels, lensmodel2->detsize_pixels);
    cpl_test_eq(lensmodel->lenslets_per_side,
            lensmodel2->lenslets_per_side);

    pdt = sph_pixel_description_table_new_from_model(lensmodel, 0.0, 0.0);
    cpl_test_nonnull( pdt );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_pixel_description_table_save(pdt, "pdt.fits", lensmodel);

    pdt2 = sph_pixel_description_table_load("pdt.fits");

    iverify = cutest_pixel_description_table_verify_pdts_identical__(pdt, pdt2);
    cpl_test(iverify);

    sph_pixel_description_table_delete(pdt2);
    pdt2 = NULL;

    /*Verify */
    cpl_test_nonnull( pdt );
    cpl_test_eq(pdt->nx, 256);
    cpl_test_eq(pdt->ny, 256);
    cpl_test_abs( pdt->offx, 0.0, 0.0);
    cpl_test_abs( pdt->offy, 0.0, 0.0);

    mask = sph_pixel_description_table_get_mask(pdt);
    maskimage = cpl_image_new_from_mask(mask);

    absflux = cpl_image_get_flux(maskimage);
    cpl_test_lt(5572, absflux);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Flux in PDT is: %f", absflux);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(maskimage);
    maskimage = NULL;
    cpl_mask_delete(mask);
    mask = NULL;
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    sph_ifs_lenslet_model_delete(lensmodel);
    lensmodel = NULL;

    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = sph_keyword_manager_load_properties("pdt.fits", 0);
    lensmodel = sph_ifs_lenslet_model_new_from_propertylist(pl);
    pdt = sph_pixel_description_table_new_from_model(lensmodel, 0.0, 0.0);
    sph_pixel_description_table_save(pdt, "pdt2.fits", lensmodel);

    mask = sph_pixel_description_table_get_mask(pdt);
    maskimage = cpl_image_new_from_mask(mask);

    cpl_test_abs( cpl_image_get_flux(maskimage), absflux, 0.001);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Flux in PDT is: %f", cpl_image_get_flux(maskimage));
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(maskimage);
    maskimage = NULL;
    cpl_mask_delete(mask);
    mask = NULL;
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    sph_ifs_lenslet_model_delete(lensmodel);
    lensmodel = NULL;

    cpl_propertylist_delete(pl);
    pl = NULL;
    pl = sph_keyword_manager_load_properties("pdt2.fits", 0);
    lensmodel = sph_ifs_lenslet_model_new_from_propertylist(pl);
    pdt = sph_pixel_description_table_new_from_model(lensmodel, 0.0, 0.0);
    sph_pixel_description_table_save(pdt, "pdt3.fits", lensmodel);

    mask = sph_pixel_description_table_get_mask(pdt);
    maskimage = cpl_image_new_from_mask(mask);

    cpl_test_abs( cpl_image_get_flux(maskimage), absflux, 0.001);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Flux in PDT is: %f", cpl_image_get_flux(maskimage));
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(maskimage);
    maskimage = NULL;
    cpl_mask_delete(mask);
    mask = NULL;
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    sph_ifs_lenslet_model_delete(lensmodel);
    lensmodel = NULL;
    cpl_propertylist_delete(pl);
    sph_ifs_lenslet_model_delete(lensmodel2);
    return;
}
static
void cutest_pixel_description_table_collapse_line_total(void) {
    sph_pixel_description_table* pdt = NULL;
    cpl_image* image = NULL;
    cpl_vector* wavs = NULL;
    int vv = 0;
    sph_master_frame* mframe = NULL;
    double val = 0.0;
    double rms = 0.0;
    double weight = 0.0;
    double delta_lambda = 0.0;
    int bpix = 0;

    image = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    sph_test_image_tools_add_in_window(image, 150, 150, 160, 190, 10.0);
    pdt = sph_pixel_description_table_new_from_image(image, 0.0, 0.0, 1.0);
    cpl_test_nonnull( pdt );

    wavs = cpl_vector_new(190 - 150 + 1);
    for (vv = 0; vv < cpl_vector_get_size(wavs); ++vv) {
        cpl_vector_set(wavs, vv, 0.5 + 0.1 * vv);
    }cpl_test_eq(
            sph_pixel_description_table_set_spectra_wavelengths(pdt,1,wavs,0),
            CPL_ERROR_NONE);
    mframe = sph_master_frame_new_empty();

    mframe->image = image;
    mframe->rmsmap = sph_test_image_tools_create_flat_image_double(256, 256,
            0.0);
    mframe->ncombmap = sph_test_image_tools_create_flat_image_double(256, 256,
            1.0);
    mframe->badpixelmap = sph_test_image_tools_create_flat_image_int(256, 256,
            0);

    cpl_image_set(mframe->badpixelmap, 150, 171, 1.0);
    cpl_image_set(mframe->badpixelmap, 160, 171, 1.0);

    cpl_test_eq(
            sph_pixel_description_table_collapse_line_total(pdt,mframe,170,149,159,&val,&bpix,&rms,&weight,&delta_lambda),
            CPL_ERROR_NONE);
    cpl_test_abs( val, 10.0, 0.0001);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Val was:%f ", val);
    cpl_test_abs( bpix, 2.0, 0.0001);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "bpix was:%d ", bpix);
    cpl_test_abs( rms, 0.0, 0.0001);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "rms was:%f ", rms);
    cpl_test_abs( weight, 9.0, 0.0001);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "weight was:%f ", weight);
    sph_master_frame_delete(mframe);
    cpl_vector_delete(wavs);
    sph_pixel_description_table_delete(pdt);
}

static
void cutest_pixel_description_table_collapse_line_total_noisy(void) {
    sph_pixel_description_table* pdt = NULL;
    cpl_image* image = NULL;
    cpl_vector* wavs = NULL;
    int vv = 0;
    sph_master_frame* mframe = NULL;
    double val = 0.0;
    double rms = 0.0;
    double weight = 0.0;
    double delta_lambda = 0.0;
    int bpix = 0;
    gsl_rng* pRNG = NULL;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    image = sph_test_image_tools_create_flat_image_double(256, 256, 0.0);
    sph_test_image_tools_add_in_window(image, 150, 150, 160, 190, 100.0);
    sph_test_image_tools_add_noise(image, 1.0, pRNG);
    pdt = sph_pixel_description_table_new_from_image(image, 0.0, 0.0, 99.0);
    cpl_test_nonnull( pdt );

    wavs = cpl_vector_new(190 - 150 + 1);
    for (vv = 0; vv < cpl_vector_get_size(wavs); ++vv) {
        cpl_vector_set(wavs, vv, 0.5 + 0.1 * vv);
    }cpl_test_eq(
            sph_pixel_description_table_set_spectra_wavelengths( pdt,1,wavs,0),
            CPL_ERROR_NONE);
    mframe = sph_master_frame_new_empty();

    mframe->image = image;
    mframe->rmsmap = sph_test_image_tools_create_flat_image_double(256, 256,
            0.0);
    mframe->ncombmap = sph_test_image_tools_create_flat_image_double(256, 256,
            1.0);
    mframe->badpixelmap = sph_test_image_tools_create_flat_image_int(256, 256,
            0);

    cpl_image_set(mframe->badpixelmap, 150, 171, 1.0);
    cpl_image_set(mframe->badpixelmap, 160, 171, 1.0);

    cpl_test_eq(
            sph_pixel_description_table_collapse_line_total(pdt,mframe,170,149,159,&val,&bpix,&rms,&weight,&delta_lambda),
            CPL_ERROR_NONE);
    cpl_test_abs( val, 100.0, 5.0);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Val was:%f ", val);
    cpl_test_abs( bpix, 2.0, 0.0001);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "bpix was:%d ", bpix);
    cpl_test_abs( rms, 1.0, 0.5);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "rms was:%f ", rms);
    cpl_test_abs( weight, 9.0, 0.0001);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "weight was:%f ", weight);
    sph_master_frame_delete(mframe);
    cpl_vector_delete(wavs);
    sph_pixel_description_table_delete(pdt);
    gsl_rng_free(pRNG);
    pRNG = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_pixel_description_table_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_new_shift(void) {
    sph_pixel_description_table* pdt = NULL;
    sph_pixel_description_table* pdt_shifted = NULL;
    sph_ifs_lenslet_model* lensmodel = NULL;
    cpl_vector* wavs = NULL;
    /* Setup and run ...*/

    lensmodel = sph_test_create_small_lenslet_model();
    cpl_test_nonnull( lensmodel );
    lensmodel->lenslets_per_side = 5;
    pdt = sph_pixel_description_table_new_from_model(lensmodel, 0.0, 0.0);

    cpl_test_nonnull( pdt );
    cpl_test_error(CPL_ERROR_NONE);

    sph_pixel_description_table_save(pdt, "pdt_new_unshift.fits", lensmodel);
    wavs = sph_pixel_description_table_get_wavs(pdt, 17);
    cpl_vector_fill(wavs, 0.0);
    sph_pixel_description_table_set_spectra_wavelengths(pdt, 17, wavs, 0);
    pdt_shifted = sph_pixel_description_table_new_shift(pdt, lensmodel, 1.4,
            1.7);
    /*Verify */
    cpl_test_nonnull( pdt_shifted );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs( pdt_shifted->offx, 1.4, DBL_EPSILON);
    cpl_test_abs( pdt_shifted->offy, 1.7, DBL_EPSILON);

    sph_pixel_description_table_save(pdt_shifted, "pdt_new_shift.fits",
            lensmodel);

    cpl_vector_delete(wavs);
    sph_pixel_description_table_delete(pdt);
    sph_pixel_description_table_delete(pdt_shifted);
    sph_ifs_lenslet_model_delete(lensmodel);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_pixel_description_table_nonlin_correction function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_correction(void) {
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* lensmodel = NULL;
    cpl_image* wavs = NULL;
    int bpix = 0;
    double old_lambda = 0.0;
    /* Setup and run ...*/

    lensmodel = sph_test_create_small_lenslet_model();
    cpl_test_nonnull( lensmodel );
    lensmodel->lenslets_per_side = 5;
    pdt = sph_pixel_description_table_new_from_model(lensmodel, 0.0, 0.0);

    cpl_test_nonnull( pdt );
    cpl_test_error(CPL_ERROR_NONE);
    
    wavs = sph_pixel_description_table_get_wavimage( pdt );
    old_lambda = cpl_image_get(wavs, 166, 146, &bpix);
    cpl_image_delete(wavs); wavs = NULL;
    cpl_test_eq_error(sph_pixel_description_table_correct_non_linear(pdt, 0.67754, -1.4464, 0.75754), CPL_ERROR_NONE);
    
    wavs = sph_pixel_description_table_get_wavimage( pdt );
    cpl_test_abs( cpl_image_get(wavs, 166, 146, &bpix), old_lambda + 0.67754 * old_lambda * old_lambda - 1.4464 * old_lambda + 0.75754, 0.0001);
    sph_pixel_description_table_save(pdt, "pdt_new_unshift.fits", lensmodel);
    sph_pixel_description_table_delete(pdt);
    sph_ifs_lenslet_model_delete(lensmodel);
    cpl_image_delete(wavs);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_pixel_description_table_save function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_save(void) {
    sph_pixel_description_table* model = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/

    model = sph_pixel_description_table_new(10, 10, 0.0, 0.0);

    cpl_test_nonnull( model );

    rerr = sph_pixel_description_table_save(model, "testpdt.fits", NULL);

    /*Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_pixel_description_table_delete(model);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_pixel_description_table_load function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_load(void) {
    sph_pixel_description_table* model = NULL;

    /* Setup and run ...*/

    model = sph_pixel_description_table_load("testpdt.fits");

    cpl_test_nonnull( model );
    /*Verify */

    cpl_test_nonnull( model );
    cpl_test_eq(model->nx, 10);
    cpl_test_eq(model->ny, 10);
    cpl_test_zero(model->idlongest);
    cpl_test_eq(model->maxlength, -1);
    cpl_test_zero(model->nregions);
    cpl_test_abs( model->offx, 0.0, 0.0);
    cpl_test_abs( model->offy, 0.0, 0.0);

    sph_pixel_description_table_delete(model);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_extract_spectrum_null_wav(void) {
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* lensmodel = NULL;
    sph_spectrum* spec = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* im = NULL;
    sph_master_frame* mf = NULL;
    double dl = 0.0;
    /* Setup and run ...*/

    lensmodel = sph_test_create_small_lenslet_model();
    cpl_test_nonnull( lensmodel );
    pdt = sph_pixel_description_table_new_from_model(lensmodel, 0.0, 0.0);

    cpl_test_nonnull( pdt );
    sph_pixel_description_table_save(pdt, "pdt10.fits", lensmodel);

    im = sph_test_image_tools_create_flat_image_double(256, 256, 122.0);
    cpl_test_nonnull( im );

    mf = sph_master_frame_new_from_cpl_image(im);
    cpl_test_nonnull( mf );
    spec = sph_pixel_description_table_extract_spectrum_mframe(pdt, 100, mf,
            NULL);
    cpl_test_nonnull( spec );

    cpl_test_eq( cpl_vector_get_size(spec->wavelengths),
            (int)lensmodel->speclength_pixels);

    cpl_test_abs( cpl_vector_get(spec->wavelengths,0),
            lensmodel->minlambda, lensmodel->dispersion);
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Spec->wavelengths[0]=%f, minlmabda = %f, disperions=%f", cpl_vector_get(spec->wavelengths,0), lensmodel->minlambda, lensmodel->dispersion);
    dl = (cpl_vector_get(spec->wavelengths, 1)
            - cpl_vector_get(spec->wavelengths, 0)) * 1.1;

    cpl_test_abs(
            cpl_vector_get(spec->wavelengths, cpl_vector_get_size(spec->wavelengths)-1),
            lensmodel->maxlambda, dl * 2.0);

    cpl_test_eq( cpl_vector_get_size(spec->spec),
            (int)lensmodel->speclength_pixels);

    cpl_test_abs( cpl_vector_get(spec->spec,0), 122.0, 0.001);

    cpl_test_abs(
            cpl_vector_get(spec->spec, cpl_vector_get_size(spec->spec)-1),
            122.0, 0.001);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_master_frame_delete(mf);
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    cpl_image_delete(im);
    sph_spectrum_delete(spec);
    sph_ifs_lenslet_model_delete(lensmodel);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_extract_spectrum_mframe_noisy(void) {
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* lensmodel = NULL;
    sph_spectrum* spec = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* im = NULL;
    sph_master_frame* mf = NULL;
    cpl_vector* wavs = NULL;
    int specid = 0;
    cpl_array* hist = NULL;
    int bp = 0;
    cpl_size maxpos = 0;
    /* Setup and run ...*/

    lensmodel = sph_ifs_lenslet_model_new();
    lensmodel->lenslets_per_side = 30;
    lensmodel->detsize_pixels = 1024;
    cpl_test_nonnull( lensmodel );
    pdt = sph_pixel_description_table_new_from_model(lensmodel, 0.0, 0.0);

    cpl_test_nonnull( pdt );

    im = sph_test_image_tools_create_flat_image_double(
            lensmodel->detsize_pixels, lensmodel->detsize_pixels, 122.0);
    sph_test_image_tools_add_noise(im, 5.0, NULL);
    cpl_test_nonnull( im );
    mf = sph_master_frame_new_from_cpl_image(im);
    cpl_test_nonnull( mf );

    wavs = sph_ifs_lenslet_model_get_lambda_vector(lensmodel);
    cpl_test_nonnull( wavs );
    hist = cpl_array_new(40, CPL_TYPE_DOUBLE);
    for (specid = 0; specid < pdt->nregions; ++specid) {
        spec = sph_pixel_description_table_extract_spectrum_mframe(pdt,
                specid + 1, mf, wavs);
        cpl_test_nonnull( spec );
        maxpos = (cpl_vector_get_median(spec->rms) / 5.0
                * cpl_array_get_size(hist));
        if (maxpos >= 0 && maxpos < cpl_array_get_size(hist)) {
            cpl_array_set(hist, maxpos, cpl_array_get(hist, maxpos, &bp) + 1.0);
        }
        sph_spectrum_delete(spec);
    }

    cpl_array_get_maxpos(hist, &maxpos);
    cpl_test_abs( maxpos * 5.0 / cpl_array_get_size(hist),
            5.0 / sqrt(2.0), 1.0);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Median RMS: %f", maxpos * 5.0 / cpl_array_get_size(hist));
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_array_delete(hist);
    sph_ifs_lenslet_model_delete(lensmodel);
    cpl_image_delete(im);
    sph_master_frame_delete(mf);
    sph_pixel_description_table_delete(pdt);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_extract_spectrum_mframe(void) {
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* lensmodel = NULL;
    sph_spectrum* spec = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* im = NULL;
    sph_master_frame* mf = NULL;
    cpl_vector* wavs = NULL;
    double dl = 0.0;
    double width;
    /* Setup and run ...*/

    lensmodel = sph_test_create_small_lenslet_model();
    cpl_test_nonnull( lensmodel );
    pdt = sph_pixel_description_table_new_from_model(lensmodel, 0.0, 0.0);

    cpl_test_nonnull( pdt );

    im = sph_test_image_tools_create_flat_image_double(256, 256, 122.0);
    cpl_test_nonnull( im );
    mf = sph_master_frame_new_from_cpl_image(im);
    cpl_test_nonnull( mf );

    cpl_image_delete(mf->ncombmap);
    cpl_image_delete(mf->rmsmap);
    mf->ncombmap = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    mf->rmsmap = sph_test_image_tools_create_flat_image_double(256, 256, 55.0);

    wavs = sph_ifs_lenslet_model_get_lambda_vector(lensmodel);
    cpl_test_nonnull( wavs );

    cpl_test_abs( cpl_vector_get(wavs,0), lensmodel->minlambda,
            0.001);

    dl = (cpl_vector_get(wavs, 1) - cpl_vector_get(wavs, 0)) * 1.1;
    cpl_test_abs(
            cpl_vector_get(wavs, (int)lensmodel->speclength_pixels - 2 ),
            lensmodel->maxlambda, dl * 2.0);

    cpl_test_eq( cpl_vector_get_size(wavs),
            (int)lensmodel->speclength_pixels);

    spec = sph_pixel_description_table_extract_spectrum_mframe(pdt, 10, mf,
            wavs);
    cpl_test_nonnull( spec );
    cpl_test_eq( cpl_vector_get_size(spec->wavelengths),
            (int)lensmodel->speclength_pixels);
    cpl_test_abs( cpl_vector_get(spec->wavelengths,0),
            lensmodel->minlambda, 0.001);
    cpl_test_abs(
            cpl_vector_get(spec->wavelengths, (int)lensmodel->speclength_pixels - 1),
            lensmodel->maxlambda, dl * 4.0);

    width = (1 + pdt->regions[spec->specid - 1]->maxx
            - pdt->regions[spec->specid - 1]->minx);

    cpl_test_lt(0.0, width);

    cpl_test_eq( cpl_vector_get_size(spec->spec),
            (int)lensmodel->speclength_pixels);
    cpl_test_abs( cpl_vector_get(spec->spec,0), 122.0, 0.001);
    cpl_test_abs(
            cpl_vector_get(spec->spec, (int)(lensmodel->speclength_pixels-1)),
            122.0, 0.001);
    cpl_test_abs( cpl_vector_get_min(spec->spec), 122.0, 0.001);
    cpl_test_abs( cpl_vector_get_mean(spec->spec), 122.0, 0.001);
    cpl_test_abs( cpl_vector_get_max(spec->spec), 122.0, 0.001);

    cpl_test_eq( cpl_vector_get_size(spec->rms),
            (int)lensmodel->speclength_pixels);
    cpl_test_abs( cpl_vector_get(spec->rms,0), 0.0, 0.001);
    cpl_test_abs(
            cpl_vector_get(spec->rms, (int)(lensmodel->speclength_pixels-1)),
            0.0, 0.001);

    cpl_test_eq( cpl_vector_get_size(spec->ncomb),
            (int)lensmodel->speclength_pixels);
    cpl_test_abs( cpl_vector_get(spec->ncomb,0), 4.0, 0.001);
    cpl_test_abs(
            cpl_vector_get(spec->ncomb, (int)(lensmodel->speclength_pixels-1)),
            4.0, 0.001);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_image_delete(im);
    sph_master_frame_delete(mf);
    sph_spectrum_delete(spec);
    sph_pixel_description_table_delete(pdt);
    sph_ifs_lenslet_model_delete(lensmodel);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_pixel_description_table_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_pixel_description_table_delete(void) {
    sph_pixel_description_table* model = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/

    model = sph_pixel_description_table_new(10, 10, 0.0, 0.0);

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( model );
    sph_pixel_description_table_delete(model);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("sph_pixel_description_table_test",
            cutest_init_pixel_description_table_testsuite,
            cutest_clean_pixel_description_table_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
        == sph_test_do(pSuite,
                       "sph_pixel_description_table_new_from_model_save_load_bug",
                       cutest_pixel_description_table_new_from_model_save_load_bug)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_pixel_description_table_correction",
                    cutest_pixel_description_table_correction)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_pixel_description_table_new",
                    cutest_pixel_description_table_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_pixel_description_table_save",
                    cutest_pixel_description_table_save)) {
        return sph_test_get_error();
    }
    if (NULL
        == sph_test_do(pSuite, "sph_pixel_description_table_load",
                       cutest_pixel_description_table_load)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_pixel_description_table_new_from_model_very_small",
                    cutest_pixel_description_table_new_from_model_very_small)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_pixel_description_table_new_shift",
                    cutest_pixel_description_table_new_shift)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_pixel_description_table_extract_spectrum_null_wav",
                    cutest_pixel_description_table_extract_spectrum_null_wav)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_pixel_description_table_extract_spectrum_mframe",
                    cutest_pixel_description_table_extract_spectrum_mframe)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_pixel_description_table_extract_spectrum_mframe_noisy",
                    cutest_pixel_description_table_extract_spectrum_mframe_noisy)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_pixel_description_table_collapse_line_total",
                    cutest_pixel_description_table_collapse_line_total)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_pixel_description_table_collapse_line_total_noisy",
                    cutest_pixel_description_table_collapse_line_total_noisy)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_pixel_description_table_delete",
                    cutest_pixel_description_table_delete)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    //CU_console_run_tests();
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
