/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_fitting.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_vector.h>
#include "sph_pixel_polyfit_table.h"
#include "sph_ptc.h"
#include "sph_framecombination.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/
static cpl_parameterlist* cutest_sph_ptc_create_paramlist(void) {
    cpl_parameterlist* result = NULL;
    cpl_parameter* p = NULL;
    result = cpl_parameterlist_new();

    p =
            cpl_parameter_new_enum(
                    "ifs.ron.coll_alg",
                    CPL_TYPE_INT,
                    "The collapse algorithm to use. 0 = Mean, 1 = Median, 2 = Clean mean. ",
                    "ifs.ron", 2, 3, 0, 1, 2);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    p = cpl_parameter_new_range("clean_mean.reject_high", CPL_TYPE_INT,
            "The clean mean reject pixels on high end. ", NULL, 0, 0, 20);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    p = cpl_parameter_new_range("clean_mean.reject_low", CPL_TYPE_INT,
            "The clean mean reject pixels on low end. ", NULL, 0, 0, 20);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    return result;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ptc_fit(void) {
    cpl_imagelist* imlist = NULL;
    cpl_imagelist* errlist = NULL;
    sph_pixel_polyfit_table* fittab = NULL;
    int nx = 256;
    int ny = 256;
    int order = 1;
    int nims = 10;
    int ii = 0;

    imlist = cpl_imagelist_new();
    errlist = cpl_imagelist_new();

    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(
                imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny,
                        ii * 1000.0 + 10.0), ii);
        cpl_imagelist_set(
                errlist,
                sph_test_image_tools_create_flat_image_double(nx, ny,
                        ii * 1000.0 + 10.0), ii);

    }

    fittab = sph_pixel_polyfit_table_new(nx, ny, order);

    sph_ptc_fit(fittab, 0, 0, imlist, errlist, order);

    //cpl_table_dump(fittab->table, 1, 10, stdout);

    sph_pixel_polyfit_table_delete(fittab);
    cpl_imagelist_delete(imlist);
    cpl_imagelist_delete(errlist);
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ptc_fit_test2(void) {
    cpl_imagelist* imlist = NULL;
    cpl_imagelist* vallist = NULL;
    cpl_imagelist* errlist = NULL;
    sph_pixel_polyfit_table* fittab = NULL;
    int nx = 256;
    int ny = 256;
    int order = 1;
    int nims = 10;
    int ii = 0;
    gsl_rng* pRNG = NULL;
    cpl_parameterlist* params = NULL;
    sph_master_frame* mframe = NULL;
    int xx = 0;
    int yy = 0;
    double w = 0.0;
    double v = 0.0;
    double totw = 0.0;
    double totv = 0.0;
    double toterr = 0.0;
    int bp = 0;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    params = cutest_sph_ptc_create_paramlist();

    vallist = cpl_imagelist_new();
    errlist = cpl_imagelist_new();

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(
                imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny,
                        100.0 + 0.0), ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    mframe = sph_framecombination_master_frame_new_collapse(imlist,
            SPH_COLL_ALG_CLEAN_MEAN, params);
    cpl_test_nonnull( mframe );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_imagelist_set(vallist, cpl_image_duplicate(mframe->image), 0);
    cpl_imagelist_set(errlist, cpl_image_power_create(mframe->rmsmap, 2.0), 0);
    cpl_imagelist_delete(imlist);
    sph_master_frame_delete(mframe);
    mframe = NULL;

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(
                imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny,
                        200.0 + 0.0), ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    mframe = sph_framecombination_master_frame_new_collapse(imlist,
            SPH_COLL_ALG_CLEAN_MEAN, params);
    cpl_test_nonnull( mframe );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_imagelist_set(vallist, cpl_image_duplicate(mframe->image), 1);
    cpl_imagelist_set(errlist, cpl_image_power_create(mframe->rmsmap, 2.0), 1);
    cpl_imagelist_delete(imlist);
    sph_master_frame_delete(mframe);
    mframe = NULL;

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(
                imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny,
                        300.0 + 0.0), ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    mframe = sph_framecombination_master_frame_new_collapse(imlist,
            SPH_COLL_ALG_CLEAN_MEAN, params);
    cpl_test_nonnull( mframe );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_imagelist_set(vallist, cpl_image_duplicate(mframe->image), 2);
    cpl_imagelist_set(errlist, cpl_image_power_create(mframe->rmsmap, 2.0), 2);
    cpl_imagelist_delete(imlist);
    sph_master_frame_delete(mframe);
    mframe = NULL;

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(
                imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny,
                        500.0 + 0.0), ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    mframe = sph_framecombination_master_frame_new_collapse(imlist,
            SPH_COLL_ALG_CLEAN_MEAN, params);
    cpl_test_nonnull( mframe );
    cpl_test_error(CPL_ERROR_NONE);
    cpl_imagelist_set(vallist, cpl_image_duplicate(mframe->image), 2);
    cpl_imagelist_set(errlist, cpl_image_power_create(mframe->rmsmap, 2.0), 2);
    cpl_imagelist_delete(imlist);
    sph_master_frame_delete(mframe);
    mframe = NULL;

    fittab = sph_pixel_polyfit_table_new(nx, ny, order);

    sph_ptc_fit(fittab, 0, 0, vallist, errlist, order);

    //cpl_table_dump(fittab->table, 1, 10, stdout);

    mframe = sph_pixel_polyfit_table_get_coeff_as_image(fittab, 0);
    SPH_ERROR_RAISE_INFO(
            1,
            "Zero coefficient: %f, %f", cpl_image_get_mean(mframe->image), cpl_image_get_stdev(mframe->image));
    totv = 0;
    totw = 0;
    for (xx = 0; xx < cpl_image_get_size_x(mframe->image); ++xx) {
        for (yy = 0; yy < cpl_image_get_size_y(mframe->image); ++yy) {
            v = cpl_image_get(mframe->image, xx + 1, yy + 1, &bp);
            w = cpl_image_get(mframe->rmsmap, xx + 1, yy + 1, &bp);
            if (w) {
                totv += v / exp(w * w);
                totw += 1.0 / exp(w * w);
            }
        }
    }
    totv /= totw;
    toterr = 0.0;
    for (xx = 0; xx < cpl_image_get_size_x(mframe->image); ++xx) {
        for (yy = 0; yy < cpl_image_get_size_y(mframe->image); ++yy) {
            v = cpl_image_get(mframe->image, xx + 1, yy + 1, &bp);
            w = cpl_image_get(mframe->rmsmap, xx + 1, yy + 1, &bp);
            if (w) {
                toterr += (v - totv) * (v - totv) / exp(w * w);
            }
        }
    }
    toterr = toterr / totw;
    SPH_ERROR_RAISE_INFO(1, "Zero coefficient: %f, %f", totv, sqrt(toterr));

    sph_master_frame_delete(mframe);

    mframe = sph_pixel_polyfit_table_get_coeff_as_image(fittab, 1);
    SPH_ERROR_RAISE_INFO(
            1,
            "First coefficient: %f, %f", cpl_image_get_mean(mframe->image), cpl_image_get_stdev(mframe->image));

    totv = 0;
    totw = 0;
    for (xx = 0; xx < cpl_image_get_size_x(mframe->image); ++xx) {
        for (yy = 0; yy < cpl_image_get_size_y(mframe->image); ++yy) {
            v = cpl_image_get(mframe->image, xx + 1, yy + 1, &bp);
            w = cpl_image_get(mframe->rmsmap, xx + 1, yy + 1, &bp);
            if (w) {
                totv += v / exp(w * w);
                totw += 1.0 / exp(w * w);
            }
        }
    }
    totv /= totw;
    toterr = 0.0;
    for (xx = 0; xx < cpl_image_get_size_x(mframe->image); ++xx) {
        for (yy = 0; yy < cpl_image_get_size_y(mframe->image); ++yy) {
            v = cpl_image_get(mframe->image, xx + 1, yy + 1, &bp);
            w = cpl_image_get(mframe->rmsmap, xx + 1, yy + 1, &bp);
            if (w) {
                toterr += (v - totv) * (v - totv) / exp(w * w);
            }
        }
    }
    toterr = toterr / totw;
    SPH_ERROR_RAISE_INFO(1, "First coefficient: %f, %f", totv, sqrt(toterr));

    sph_master_frame_delete(mframe);

    sph_pixel_polyfit_table_save_cube(fittab, "fitcube2.fits");
    sph_pixel_polyfit_table_delete(fittab);
    cpl_parameterlist_delete(params);
    cpl_imagelist_delete(vallist);
    vallist = NULL;
    cpl_imagelist_delete(errlist);
    errlist = NULL;
    gsl_rng_free(pRNG);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ptc_create_full_testA(void) {
    cpl_imagelist* imlist = NULL;
    sph_pixel_polyfit_table* fittab = NULL;
    int nx = 256;
    int ny = 256;
    int nims = 10;
    int ii = 0;
    gsl_rng* pRNG = NULL;
    cpl_frame* tmpframe = NULL;
    cpl_frameset* frameset = NULL;
    cpl_parameterlist* params = NULL;
    cpl_frameset* collapsed_frames = NULL;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    frameset = cpl_frameset_new();

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny, 1000.0),
                ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    tmpframe = sph_filemanager_create_temp_frame("ptc_test", "IFS GAIN RAW",
            CPL_FRAME_GROUP_RAW);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(tmpframe),
            CPL_BPP_32_SIGNED, NULL, CPL_IO_CREATE);
    cpl_frameset_insert(frameset, tmpframe);
    cpl_imagelist_delete(imlist);

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny, 2000.0),
                ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    tmpframe = sph_filemanager_create_temp_frame("ptc_test", "IFS GAIN RAW",
            CPL_FRAME_GROUP_RAW);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(tmpframe),
            CPL_BPP_32_SIGNED, NULL, CPL_IO_CREATE);
    cpl_frameset_insert(frameset, tmpframe);
    cpl_imagelist_delete(imlist);

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny, 4000.0),
                ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    tmpframe = sph_filemanager_create_temp_frame("ptc_test", "IFS GAIN RAW",
            CPL_FRAME_GROUP_RAW);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(tmpframe),
            CPL_BPP_32_SIGNED, NULL, CPL_IO_CREATE);
    cpl_frameset_insert(frameset, tmpframe);
    cpl_imagelist_delete(imlist);

    params = cutest_sph_ptc_create_paramlist();
    collapsed_frames = sph_framecombination_collapse_cubes(frameset,
            SPH_COLL_ALG_CLEAN_MEAN, params, "TEST", "ATEST");
    fittab = sph_ptc_create_full(collapsed_frames, 1);

    cpl_test_nonnull( fittab );
    //cpl_table_dump(fittab->table, 1, 10, stdout);
    sph_pixel_polyfit_table_save_cube(fittab, "fitcube.fits");
    sph_pixel_polyfit_table_delete(fittab);
    cpl_parameterlist_delete(params);
    cpl_frameset_delete(frameset);
    cpl_frameset_delete(collapsed_frames);
    gsl_rng_free(pRNG);
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ptc_calculate_means(void) {
    cpl_imagelist* imlist = NULL;
    cpl_vector* means = NULL;
    cpl_vector* errs = NULL;
    int nx = 256;
    int ny = 256;
    int nims = 3;
    int ii = 0;
    gsl_rng* pRNG = NULL;
    cpl_frame* tmpframe = NULL;
    cpl_frameset* frameset = NULL;
    cpl_parameterlist* params = NULL;
    cpl_polynomial* poly = NULL;
    double red = 0.0;
    cpl_size p0 = 0;
    cpl_size p1 = 1;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    frameset = cpl_frameset_new();

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny, 1000.0),
                ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    tmpframe = sph_filemanager_create_temp_frame("ptc_test", "IFS GAIN RAW",
            CPL_FRAME_GROUP_RAW);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(tmpframe),
            CPL_BPP_32_SIGNED, NULL, CPL_IO_CREATE);
    cpl_frameset_insert(frameset, tmpframe);
    cpl_imagelist_delete(imlist);

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny, 2000.0),
                ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    tmpframe = sph_filemanager_create_temp_frame("ptc_test", "IFS GAIN RAW",
            CPL_FRAME_GROUP_RAW);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(tmpframe),
            CPL_BPP_32_SIGNED, NULL, CPL_IO_CREATE);
    cpl_frameset_insert(frameset, tmpframe);
    cpl_imagelist_delete(imlist);

    imlist = cpl_imagelist_new();
    for (ii = 0; ii < nims; ++ii) {
        cpl_imagelist_set(imlist,
                sph_test_image_tools_create_flat_image_double(nx, ny, 4000.0),
                ii);
        sph_test_image_tools_apply_poisson_noise(cpl_imagelist_get(imlist, ii),
                pRNG);
    }
    tmpframe = sph_filemanager_create_temp_frame("ptc_test", "IFS GAIN RAW",
            CPL_FRAME_GROUP_RAW);
    cpl_imagelist_save(imlist, cpl_frame_get_filename(tmpframe),
            CPL_BPP_32_SIGNED, NULL, CPL_IO_CREATE);
    cpl_frameset_insert(frameset, tmpframe);
    cpl_imagelist_delete(imlist);

    params = cutest_sph_ptc_create_paramlist();
    means = sph_ptc_calculate_means(frameset, SPH_COLL_ALG_CLEAN_MEAN, params,
            NULL, NULL, &errs);

    cpl_test_nonnull( means );
    cpl_test_nonnull( errs );

    cpl_vector_power(errs, 2.0);
    //cpl_vector_multiply_scalar(errs,sqrt((double)(nims)/(double)(nims - 1.0)));
    //cpl_vector_dump(means,stdout);
    //cpl_vector_dump(errs,stdout);
    poly = sph_fitting_fit_poly1d(means, errs, NULL, 0, 1, 0, 0.0, &red);

    cpl_test_nonnull( poly );
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Fit: %f + %f * signal with red. chi %f", cpl_polynomial_get_coeff(poly,&p0), cpl_polynomial_get_coeff(poly,&p1), red);
    cpl_vector_delete(means);
    cpl_vector_delete(errs);

    cpl_polynomial_delete(poly);

    cpl_parameterlist_delete(params);
    cpl_frameset_delete(frameset);
    gsl_rng_free(pRNG);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_ptc", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL == sph_test_do(pSuite, "Testing ptc_fit", cutest_sph_ptc_fit)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing ptc_fit2", cutest_sph_ptc_fit_test2)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing ptc_create_full",
                    cutest_sph_ptc_create_full_testA)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing ptc_calculate_means",
                    cutest_sph_ptc_calculate_means)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
