/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_strehl.h"

#include "irplib_strehl.h"

#include <assert.h>
#include <string.h>
#include <stdio.h>
#ifdef HAVE_UNISTD_H
/* Needed for access() */
#include <unistd.h>
#endif

/*-----------------------------------------------------------------------------
 Define
 -----------------------------------------------------------------------------*/

#define SPH_IRD_TAG_PHOT_TABLE_CALIB "IRD_PHOT_STAR_TABLE"

#ifndef SPH_STREHL_STDCAT_IRDIS
#define SPH_STREHL_STDCAT_IRDIS "spher-calib/ird/sph_ird_phot_star_table.fits"
#endif

#ifndef SPH_STREHL_TABLE_IRDIS
#define SPH_STREHL_TABLE_IRDIS "spher-calib/ird/sph_ird_filt_table.fits"
#endif

#ifndef SPH_STREHL_TABLE_ZIMPOL
#define SPH_STREHL_TABLE_ZIMPOL "spher-calib/zpl/sph_zpl_filt_table.fits"
#endif

#define SPH_STREHL_RAW "sph_strehl.fits"

#ifndef SPH_STREHL_NX
/* A pretty much random, even number */
#define SPH_STREHL_NX 798
#endif

#ifndef SPH_STREHL_NY
/* A pretty much random number */
#define SPH_STREHL_NY 877
#endif

/* For the purpose of the test, using ten times bigger pixel scales, so can
 * use smaller images (otherwise the test runs for ages on some machines)
 */
#ifndef CUTEST_SPH_STREHL_PSCALE_IRDIS
#define CUTEST_SPH_STREHL_PSCALE_IRDIS 10.0 * SPH_STREHL_PSCALE_IRDIS
#endif

#ifndef CUTEST_SPH_STREHL_PSCALE_ZIMPOL
#define CUTEST_SPH_STREHL_PSCALE_ZIMPOL 10.0 * SPH_STREHL_PSCALE_ZIMPOL
#endif

/*-----------------------------------------------------------------------------
 Function declarations
 -----------------------------------------------------------------------------*/

static void sph_strehl_test_one(const char *, const char *, int)
    CPL_ATTR_NONNULL;
static void sph_strehl_fill_header(cpl_propertylist *, int)
    CPL_ATTR_NONNULL;
static void sph_strehl_save_raw(cpl_frame *, const cpl_image *, const char *,
        int)
    CPL_ATTR_NONNULL;

static void sph_phot_test_cat(const char *, const char *, const char *,
                              const char *)
    CPL_ATTR_NONNULL;


static
cpl_error_code sph_strehl_fill_parlist(cpl_parameterlist *,
                                       const char        *);

static
cpl_error_code sph_strehl_fill_parlist_double(cpl_parameterlist *,
                                              const char        *,
                                              const char        *,
                                              const char        *,
                                              const char        *,
                                              double             );

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char* argv[]) {

    char * base;
    char *end;
    int ndir = 5; /* Depth of source directory */

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    assert(argc > 0);

    base = cpl_strdup(argv[0]);

    while ((end = strrchr(base, '/')) > base && --ndir >= 0) {
        *end = 0;
    }
    if (end != NULL) {
        end[1] = 0; /* Truncate after '/' */

        sph_strehl_test_one(base, SPH_STREHL_TABLE_IRDIS, 'I');
        sph_strehl_test_one(base, SPH_STREHL_TABLE_ZIMPOL, 'Z');
        sph_phot_test_cat(base, SPH_STREHL_STDCAT_IRDIS, "IRDIS",
                          SPH_IRD_TAG_PHOT_TABLE_CALIB);
    }

    cpl_free(base);

    return cpl_test_end(0);
}

/**@}*/

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Test one arm
 @param path   Initial path to filter frame
 @param base   Filter frame (with remainder of path)
 @param arm    Arm indicator ('I'/'Z')
 @return   void

 */
/*----------------------------------------------------------------------------*/
static void sph_strehl_test_one(const char * path, const char * base, int arm) {

    char * file = cpl_sprintf("%s%s", path, base);
    const int no_file =
#ifdef HAVE_ACCESS
        access(file, F_OK | R_OK)
#else
        1
#endif
        ;

    if (no_file) {
        cpl_msg_info(cpl_func, "Not testing %s", file);
    } else {
        cpl_frame * filterframe = cpl_frame_new();
        cpl_frame * rawframe = cpl_frame_new();
        cpl_propertylist * qclist = cpl_propertylist_new();
        cpl_parameterlist * parlist = cpl_parameterlist_new();
        cpl_image * imtmp = cpl_image_new(SPH_STREHL_NX, SPH_STREHL_NY,
                CPL_TYPE_FLOAT);
        cpl_image * image = cpl_image_new(SPH_STREHL_NX, SPH_STREHL_NY,
                CPL_TYPE_FLOAT);
        FILE * stream =
                cpl_msg_get_level() > CPL_MSG_INFO ?
                        fopen("/dev/null", "a") : stdout;
        cpl_error_code code;

        cpl_msg_info(cpl_func, "Testing %s", file);
        cpl_test_nonnull(stream);

        code = cpl_frame_set_filename(filterframe, file);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        code = cpl_frame_set_group(filterframe, CPL_FRAME_GROUP_CALIB);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        code = cpl_image_fill_noise_uniform(image, 0.0, 1.0);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        if (arm == 'I') {
            /* FIXME: Use pixel scale (2" = 163pixel) */
            const double sig = 2.2;

            if (cpl_msg_get_level() <= CPL_MSG_DEBUG) {
                /* Fail on pure noise */
                sph_strehl_save_raw(rawframe, image, SPH_STREHL_RAW, arm);
                code = sph_strehl_irdis_and_append(qclist, qclist, image, image, rawframe,
                        filterframe, "test", parlist, NULL, NULL);

                cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_OUTPUT);
            }

            /* FIXME: Use something better than a Gaussian */
            code = cpl_image_fill_gaussian(imtmp, SPH_STREHL_NX / 4,
                    SPH_STREHL_NY / 2, 1000.0, sig, sig);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            code = cpl_image_add(image, imtmp);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            code = cpl_image_fill_gaussian(imtmp, 3 * SPH_STREHL_NX / 4,
                    SPH_STREHL_NY / 2, 10000.0, sig, sig);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            code = cpl_image_add(image, imtmp);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            sph_strehl_save_raw(rawframe, image, SPH_STREHL_RAW, arm);

            code = sph_strehl_fill_parlist(parlist, "test");
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            code = sph_strehl_irdis_and_append(qclist, qclist, image, image, rawframe, filterframe,
                    "test", parlist, NULL, NULL);

            cpl_test_eq_error(code, CPL_ERROR_NONE);

        } else if (arm == 'Z') {
            const double sig = 2.2; /* FIXME: Use pixel scale */

            sph_strehl_qc_pars strehl_qc_pars1, strehl_qc_pars2;
            if (cpl_msg_get_level() <= CPL_MSG_DEBUG) {
                /* Fail on pure noise */
                sph_strehl_save_raw(rawframe, image, SPH_STREHL_RAW, arm);

                code = sph_strehl_zimpol(image, rawframe,
                        filterframe, "test", parlist, CPL_TRUE, &strehl_qc_pars1);
                cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_OUTPUT);

                code = sph_strehl_zimpol(image, rawframe,
                        filterframe, "test", parlist, CPL_FALSE, &strehl_qc_pars2);
                cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_OUTPUT);
            }

            /* FIXME: Use something better than a Gaussian */
            code = cpl_image_fill_gaussian(imtmp, SPH_STREHL_NX / 2,
                    SPH_STREHL_NY / 2, 10000.0, sig, sig);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            code = cpl_image_add(image, imtmp);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            sph_strehl_save_raw(rawframe, image, SPH_STREHL_RAW, arm);

            code = sph_strehl_fill_parlist(parlist, "test");
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            code = sph_strehl_zimpol(image, rawframe,
                    filterframe, "test", parlist, CPL_TRUE, &strehl_qc_pars1);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            code = sph_strehl_zimpol(image, rawframe,
                    filterframe, "test", parlist, CPL_FALSE, &strehl_qc_pars2);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            //pars for 1 and 2 have the same name, test only one case
            code = sph_strehl_fill_qc_pars_zimpol(qclist, &strehl_qc_pars1);
			cpl_test_eq_error(code, CPL_ERROR_NONE);
        }

        cpl_propertylist_dump(qclist, stream);

        cpl_image_delete(image);
        cpl_image_delete(imtmp);
        cpl_frame_delete(rawframe);
        cpl_frame_delete(filterframe);
        cpl_propertylist_delete(qclist);
        cpl_parameterlist_delete(parlist);

        if (stream != stdout)
            cpl_test_zero( fclose(stream));

        if (cpl_test_get_failed() == 0)
            cpl_test_zero(remove(SPH_STREHL_RAW));
    }

    cpl_free(file);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Fill propertylist corresponding to a raw frame
 @param self   Propertylist to fill
 @param arm    Arm indicator ('I'/'Z')
 @return   void

 */
/*----------------------------------------------------------------------------*/
static void sph_strehl_fill_header(cpl_propertylist * self, int arm) {

    cpl_error_code code;

    double scale_h = arm == 'I' ? CUTEST_SPH_STREHL_PSCALE_IRDIS
    		: CUTEST_SPH_STREHL_PSCALE_ZIMPOL;
    //the scale in the header is in different units and must divided by 2 if ZIMPOL
    scale_h *= 1e3;
    if(arm != 'I')
    	scale_h *= 2.0;

    code = cpl_propertylist_append_double(self, SPH_STREHL_PSCALE,
            scale_h);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Set filter both both arms */
    code = cpl_propertylist_append_string(self, SPH_STREHL_FILTER_IRD1,
            "CntK1");
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_propertylist_append_string(self, SPH_STREHL_FILTER_IRD2,
            "CLEAR");
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_propertylist_append_string(self, SPH_STREHL_FILTER_ZPL0, "HeI");
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_propertylist_append_string(self, SPH_STREHL_FILTER_ZPL1, "OPEN");
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_propertylist_append_string(self, SPH_STREHL_FILTER_ZPL2, "Ha");
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Create a raw frame
 @param self     The frame to fill
 @param image    Image to save (only dimensions are used)
 @param filename The name of the FITS file
 @param arm    Arm indicator ('I'/'Z')
 @return   void

 */
/*----------------------------------------------------------------------------*/
static void sph_strehl_save_raw(cpl_frame * self, const cpl_image * image,
        const char * filename, int arm) {

    cpl_propertylist * plist = cpl_propertylist_new();
    cpl_error_code code;

    sph_strehl_fill_header(plist, arm);

    code = cpl_image_save(image, filename, CPL_TYPE_UNSPECIFIED, plist,
            CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_frame_set_filename(self, filename);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_frame_set_group(self, CPL_FRAME_GROUP_RAW);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_propertylist_delete(plist);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Make a few tests on a standard star catalogue
 @param path     The directory path
 @param base     The catalogue file name
 @param arm      The instrument arm, e.g. "IRDIS"
 @param catg     The product category, e.g. "IRD_PHOT_STAR_TABLE"
 @return   void

 */
/*----------------------------------------------------------------------------*/
static void sph_phot_test_cat(const char * path, const char * base,
                              const char * arm, const char * catg)
{
    char * file = cpl_sprintf("%s%s", path, base);
    const int no_file =
#ifdef HAVE_ACCESS
        access(file, F_OK | R_OK)
#else
        1
#endif
        ;

    if (no_file) {
        cpl_msg_info(cpl_func, "Not testing %s", file);
    } else {

        cpl_propertylist * pl_first;
        cpl_propertylist * pl_second;
        cpl_table        * tab_stdcat;
        cpl_size           nrows, nok;
        cpl_error_code     code;
        const char       * strval;
        int                intval;
        cpl_type           typeval;
        FILE             * stream =
            cpl_msg_get_level() > CPL_MSG_INFO ?
            fopen("/dev/null", "a") : stdout;

        cpl_msg_info(cpl_func, "Testing %s", base);
        cpl_test_nonnull(stream);

        cpl_test_fits(file);

        pl_first   = cpl_propertylist_load(file, 0);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(pl_first);

        strval = cpl_propertylist_get_string(pl_first, "INSTRUME");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq_string(strval, "SPHERE");

        strval = cpl_propertylist_get_string(pl_first, "ESO SEQ ARM");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq_string(strval, arm);

        strval = cpl_propertylist_get_string(pl_first, "ESO PRO CATG");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq_string(strval, catg);

        pl_second  = cpl_propertylist_load(file, 1);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(pl_second);

        tab_stdcat = cpl_table_load(file, 1, 1);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_nonnull(tab_stdcat);

        nrows = cpl_table_get_nrow(tab_stdcat);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_lt(0, nrows);

        intval = cpl_table_has_column(tab_stdcat, "Target");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(intval, 1);

        intval = cpl_table_has_column(tab_stdcat, "RA");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(intval, 1);

        intval = cpl_table_has_column(tab_stdcat, "DEC");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(intval, 1);

        typeval = cpl_table_get_column_type(tab_stdcat, "Target");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(typeval, CPL_TYPE_STRING);

        typeval = cpl_table_get_column_type(tab_stdcat, "RA");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(typeval, CPL_TYPE_DOUBLE);

        typeval = cpl_table_get_column_type(tab_stdcat, "DEC");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(typeval, CPL_TYPE_DOUBLE);

        strval = cpl_table_get_column_unit(tab_stdcat, "RA");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq_string(strval, "DEG");

        strval = cpl_table_get_column_unit(tab_stdcat, "DEC");
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq_string(strval, "DEG");

        nok = cpl_table_and_selected_double(tab_stdcat, "RA",
                                            CPL_GREATER_THAN, 0.0);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(nok, nrows);
        code = cpl_table_select_all(tab_stdcat);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        nok = cpl_table_and_selected_double(tab_stdcat, "RA",
                                            CPL_LESS_THAN, 360.0);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(nok, nrows);
        code = cpl_table_select_all(tab_stdcat);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        nok = cpl_table_and_selected_double(tab_stdcat, "DEC",
                                            CPL_GREATER_THAN, -90.0);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(nok, nrows);
        code = cpl_table_select_all(tab_stdcat);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        nok = cpl_table_and_selected_double(tab_stdcat, "DEC",
                                            CPL_LESS_THAN, 90.0);
        cpl_test_error(CPL_ERROR_NONE);
        cpl_test_eq(nok, nrows);
        code = cpl_table_select_all(tab_stdcat);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        cpl_table_dump_structure(tab_stdcat, stream);

        cpl_propertylist_delete(pl_first);
        cpl_propertylist_delete(pl_second);
        cpl_table_delete(tab_stdcat);

        if (stream != stdout)
            cpl_test_zero( fclose(stream));
    }

    cpl_free(file);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Insert a double parameter into a parameterlist
 @param self    Parameterlist to append  parameter to
 @param recipe  Recipe name (for parameter access)
 @return   CPL_ERROR_NONE or the relevant CPL error code on error

 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code sph_strehl_fill_parlist_double(cpl_parameterlist * self,
                                              const char        * recipe,
                                              const char        * longname,
                                              const char        * shortname,
                                              const char        * mantext,
                                              double              defaultvalue)
{

    const char* arm = "ird"; /* FIXME: Support also ZIMPOL at some point */

    char * context = cpl_sprintf("%s.%s", arm, recipe);
    char * paramname = cpl_sprintf("%s.%s.%s", arm, recipe, longname);
    cpl_parameter * p =
        cpl_parameter_new_value(paramname, CPL_TYPE_DOUBLE, mantext,
                                context, defaultvalue);
    cpl_free(context);
    cpl_free(paramname);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, shortname);
    cpl_parameterlist_append(self, p);

    return CPL_ERROR_NONE;

}

/*----------------------------------------------------------------------------*/
/**
 @brief Insert the Strehl parameters to a parameterlist
 @param self    Parameterlist to append Strehl parameters to
 @param recipe  Recipe name (for parameter access)
 @return   CPL_ERROR_NONE or the relevant CPL error code on error

 */
/*----------------------------------------------------------------------------*/
static
cpl_error_code sph_strehl_fill_parlist(cpl_parameterlist * self,
                                       const char       * recipe)
{

    if (sph_strehl_fill_parlist_double(self, recipe, "star_r", "star_r",
                                       "The star radius [arcsecond]",
                                       IRPLIB_STREHL_STAR_RADIUS))
        return cpl_error_set_where(cpl_func);


    if (sph_strehl_fill_parlist_double(self, recipe, "bg_r1", "bg_r1",
                                       "The internal radius of the background "
                                       "[arcsecond]",
                                       IRPLIB_STREHL_BACKGROUND_R1))
        return cpl_error_set_where(cpl_func);


    if (sph_strehl_fill_parlist_double(self, recipe, "bg_r2", "bg_r2",
                                       "The external radius of the background "
                                       "[arcsecond]",
                                       IRPLIB_STREHL_BACKGROUND_R2))
        return cpl_error_set_where(cpl_func);

    return CPL_ERROR_NONE;
}
