/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_utils.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_zpl_simulator.h"
#include "sph_test_image_tools.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_time.h"
static
void cutest_sph_time_test_it(void);

static
void cutest_sph_time_test_get_times(void);

static
int cutest_sph_time_clean_testsuite(void);

static
int cutest_sph_time_init_testsuite(void);

static void sph_radec_raddeg_to_iso8601string_test(void);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("A cutest_sph_time testsuite",
            cutest_sph_time_init_testsuite, cutest_sph_time_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing function", cutest_sph_time_test_it)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing function: sph_time_test_get_times",
                    cutest_sph_time_test_get_times)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_time_test_it(void) {
    return;
}

static
void cutest_sph_time_test_get_times(void) {
    double mjd_start = 0.0;
    double mjd_stop = 0.0;
    double mjd_frame_ut_et = 0.0;
    const char* filename = "sph_time_get_times.fits";
    double mjd_obs_et = 56799.00106531; // 2014-05-22T00:01:32.0429
    const char* frame_ut_string_et = "2014-05-22T00:04:57.0520";

    sph_time_mjd_from_string(&mjd_frame_ut_et, frame_ut_string_et);

    cpl_frame* inframe = NULL;

    inframe = sph_test_zpl_simulator_create_double_image_frame(filename, 10, 10,
            5.0, 6.0);
    sph_time_get_times(inframe, &mjd_start, &mjd_stop);

    cpl_test_abs( mjd_start, mjd_obs_et, 1e-10);
    cpl_test_abs( mjd_stop, mjd_frame_ut_et, 1e-10);

    sph_radec_raddeg_to_iso8601string_test();

    cpl_frame_delete(inframe);

    return;
}

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_time_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_time_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Test the SPHERE function
 */
/*----------------------------------------------------------------------------*/
static void sph_radec_raddeg_to_iso8601string_test(void)
{

    cpl_error_code code;
    char           iso8601string[256];


    code = sph_radec_deg_to_iso8601string(0.0, NULL, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = sph_radec_rad_to_iso8601string(0.0, NULL, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = sph_radec_deg_to_iso8601string(0.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_msg_info(cpl_func, "ZERO: %s", iso8601string);

    code = sph_radec_rad_to_iso8601string(0.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_msg_info(cpl_func, "ZERO: %s", iso8601string);

    code = sph_radec_deg_to_iso8601string(400.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    code = sph_radec_rad_to_iso8601string(400.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    code = sph_radec_deg_to_iso8601string(90.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_msg_info(cpl_func, "90: %s", iso8601string);

    code = sph_radec_deg_to_iso8601string(-90.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_msg_info(cpl_func, "-90: %s", iso8601string);

    code = sph_radec_deg_to_iso8601string(50.0 / 3.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_msg_info(cpl_func, "50/3: %s", iso8601string);

    code = sph_radec_deg_to_iso8601string(50.0 / 7.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_msg_info(cpl_func, "50/7: %s", iso8601string);

    code = sph_radec_deg_to_iso8601string(-50.0 / 3.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_msg_info(cpl_func, "-50/3: %s", iso8601string);

    code = sph_radec_deg_to_iso8601string(-50.0 / 7.0, iso8601string, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_msg_info(cpl_func, "-50/7: %s", iso8601string);

}


/**@}*/
