from edps import (
    task, match_rules, alternative_associated_inputs, FilterMode, ReportInput,
    qc0, qc1calib, science, calchecker,
    THREE_WEEKS
)

from .irdis_datasources import *
from .irdis_subwkf import (
    process_ird_dark_background, process_ird_standard_img, process_ird_science_img_pol,
)
from .sphere_rules import TEN_DAYS

"""
# Naming conventions
- Classification rules for raw frames start with `cls_`
- Classification rules for products have the same name as the PRO.CATG
- DataSources start with `raw_`
- Tasks start with `task_`
"""

__title__ = "SPHERE IRDIS workflow"
########
# TASK #
########
# imaging cascade

# all
task_ird_dark, task_ird_ins_bg, task_ird_sky_bg, task_ird_static_bpm = process_ird_dark_background()

# Associate suitable background (sky preferred, next internal background, next dark)
# obs_background = (alternative_associated_inputs(sort_keys=[kwd.mjd_obs])
# sorts w mjd-obs to replicate reflex behaviour, but we want to give preference as indicated

obs_background = (alternative_associated_inputs()
                  .with_associated_input(task_ird_sky_bg, [IRD_SKY_BG])
                  .with_associated_input(task_ird_ins_bg, [IRD_INS_BG])
                  .with_associated_input(task_ird_dark, [IRD_MASTER_DARK]))

obs_background_optional = (alternative_associated_inputs()
                           .with_associated_input(task_ird_sky_bg, [IRD_SKY_BG], min_ret=0)
                           .with_associated_input(task_ird_ins_bg, [IRD_INS_BG], min_ret=0)
                           .with_associated_input(task_ird_dark, [IRD_MASTER_DARK], min_ret=0))

# Detector monitoring (determine gain and read-out noise) not used for science processing
# task_ird_rongain = (task("irdis_ron_gain")
#                    .with_recipe("sph_ird_gain")
#                    .with_main_input(ird_raw_gain)
#                    .with_meta_targets([qc1calib, calchecker])
#                    .build())

# Determine pixel-to-pixel sensitivity variations (quantum efficiency, filter defects, etc)
task_ird_flat = (task("irdis_flat_field")
                 .with_recipe("sph_ird_instrument_flat")
                 .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                 .with_report("sphere_img_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
                 .with_main_input(ird_raw_flat)
                 .with_alternative_associated_inputs(obs_background_optional)
                 .with_associated_input(task_ird_static_bpm, [IRD_STATIC_BADPIXELMAP], min_ret=0)
                 .with_input_filter(IRD_INS_BG_FIT, mode=FilterMode.REJECT)
                 .with_meta_targets([qc1calib])
                 .build())

# Determine internal distortion from images of pinhole mask
task_ird_distortion_img = (task("irdis_distortion_map")
                           .with_recipe("sph_ird_distortion_map")
                           .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                           .with_main_input(ird_raw_distortion)
                           .with_associated_input(ird_point_pattern_table, min_ret=1)
                           .with_meta_targets([qc1calib])
                           .build())

# Process standard star observations (standard_flux used for zeropoints, standard astrometry not analysed)
task_ird_standard_flux, task_ird_standard_astrometry = process_ird_standard_img(task_ird_dark,
                                                                                task_ird_static_bpm,
                                                                                task_ird_ins_bg,
                                                                                task_ird_sky_bg,
                                                                                task_ird_flat, task_ird_distortion_img)

# Process science on-sky calibration data for imaging (classical, dual band, and polarimetric)
task_ird_coronagraph_center, task_ird_science_flux_img, task_ird_science_flux_pol = \
    process_ird_science_img_pol(task_ird_dark, task_ird_static_bpm, task_ird_ins_bg, task_ird_sky_bg, task_ird_flat,
                                task_ird_distortion_img)

# Actual science data, may be observed with or without coronagraph
task_ird_science_img = (task("irdis_science_imaging")
                        .with_recipe("sph_ird_science_dbi")
                        .with_main_input(ird_raw_science_img)
                        .with_alternative_associated_inputs(obs_background)
                        .with_associated_input(task_ird_flat, [IRD_FLAT_FIELD], min_ret=1)
                        .with_associated_input(task_ird_static_bpm, [IRD_STATIC_BADPIXELMAP], min_ret=0)
                        .with_associated_input(task_ird_distortion_img, [IRD_DISTORTION_MAP], min_ret=1)
                        .with_associated_input(task_ird_coronagraph_center, [IRD_STAR_CENTER],
                                               min_ret=0)
                        .with_associated_input(task_ird_science_flux_img, min_ret=0)
                        .with_associated_input(ird_filter_table, min_ret=0)
                        .with_input_filter(IRD_SCIENCE_DBI, IRD_SCIENCE_DBI_LEFT, IRD_INS_BG_FIT,
                                           IRD_SCIENCE_DBI_RIGHT, mode=FilterMode.REJECT)
                        .with_meta_targets([qc0, science, calchecker])
                        .build())

# sph_ird_science_dpi accepts products of task_ird_science_center only if taken BEFORE the science
match_ird_coronagraph_pol = (match_rules()
                             .with_match_keywords([kwd.obs_start, kwd.ins1_filt_name, kwd.ins1_opti2_name],
                                                  time_range=RelativeTimeRange(-0.4, 0)))

# Actual science data, may be observed with or without coronagraph
assoc_ird_flats = (match_rules()
                   .with_match_keywords(rules.matchkwd_ird_flat, time_range=ONE_DAY, level=0)
                   .with_match_keywords(rules.matchkwd_ird_flat, time_range=TEN_DAYS, level=1))

task_ird_science_pol = (task("irdis_science_polarimetry")
                        .with_recipe("sph_ird_science_dpi")
                        .with_main_input(ird_raw_science_pol)
                        .with_alternative_associated_inputs(obs_background)
                        .with_associated_input(task_ird_flat, [IRD_FLAT_FIELD], min_ret=1, match_rules=assoc_ird_flats)
                        .with_associated_input(task_ird_static_bpm, [IRD_STATIC_BADPIXELMAP], min_ret=0)
                        .with_associated_input(task_ird_distortion_img, [IRD_DISTORTION_MAP], min_ret=1)
                        .with_associated_input(task_ird_coronagraph_center, [IRD_STAR_CENTER],
                                               match_rules=match_ird_coronagraph_pol, min_ret=0)
                        .with_associated_input(task_ird_science_flux_pol, min_ret=0)
                        .with_input_filter(IRD_SCIENCE_DPI, IRD_SCIENCE_DPI_LEFT, IRD_INS_BG_FIT,
                                           IRD_SCIENCE_DPI_RIGHT, mode=FilterMode.REJECT)
                        .with_meta_targets([qc0, science, calchecker])
                        .build())

# Determine wavelength calibration for long-slit spectra
associate_flat_to_wavecal_lss = (match_rules()
                                 .with_match_keywords([kwd.ins1_filt_name, kwd.ins1_opti1_name, kwd.ins1_opti2_name],
                                                      time_range=TEN_DAYS, level=0)
                                 .with_match_keywords([kwd.ins1_filt_name, kwd.ins1_opti1_name, kwd.ins1_opti2_name],
                                                      time_range=THREE_WEEKS, level=1))

task_ird_wavecal_lss = (task("irdis_wavecal_spectroscopy")
                        .with_recipe("sph_ird_wave_calib")
                        .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                        .with_main_input(ird_raw_wavecal_lss)
                        .with_alternative_associated_inputs(obs_background_optional)
                        .with_associated_input(task_ird_flat, [IRD_FLAT_FIELD], min_ret=1,
                                               match_rules=associate_flat_to_wavecal_lss)
                        .with_input_filter(IRD_INS_BG_FIT, mode=FilterMode.REJECT)
                        .with_meta_targets([qc1calib, calchecker])
                        .build())
# Process science data (long-slit spectroscopy)
task_ird_science_lss = (task("irdis_science_spectroscopy")
                        .with_recipe("sph_ird_science_spectroscopy")
                        .with_main_input(ird_raw_science_lss)
                        .with_alternative_associated_inputs(obs_background)
                        .with_associated_input(task_ird_flat, [IRD_FLAT_FIELD], min_ret=1)
                        .with_associated_input(task_ird_static_bpm, [IRD_STATIC_BADPIXELMAP], min_ret=0)
                        # for calselector only
                        .with_associated_input(task_ird_wavecal_lss, [IRD_WAVECALIB], min_ret=0)
                        # filter out unwanted products
                        .with_input_filter(IRD_WAVECALIB, IRD_INS_BG_FIT, mode=FilterMode.REJECT)
                        .with_meta_targets([qc0, science, calchecker])
                        .build())
