# SPDX-License-Identifier: BSD-3-Clause
import os
from adari_core.data_libs.master_formatcheck import MasterFormatCheckReport
from adari_core.plots.panel import Panel
from adari_core.plots.points import ScatterPlot
from adari_core.plots.text import TextPlot
from adari_core.utils.utils import fetch_kw_or_default
from .xshooter_utils import XshooterSetupInfo, XshooterReportMixin

class XshooterFormatCheckReport(XshooterReportMixin, MasterFormatCheckReport):
    def __init__(self):
        super().__init__("xshooter_formatcheck")
        self.central_region_size = 200

    def parse_sof(self):
        # building one report set
        arc_line_list_uvb = None
        arc_line_list_vis = None
        arc_line_list_nir = None

        fmtchk_on_uvb = None
        fmtchk_on_vis = None
        fmtchk_on_nir = None

        for filename, catg in self.inputs:
            if catg == "FMTCHK_RESID_TAB_LINES_UVB":
                arc_line_list_uvb = filename
                self.category_label = "UVB"
            if catg == "FMTCHK_RESID_TAB_LINES_VIS":
                arc_line_list_vis = filename
                self.category_label = "VIS"
            if catg == "FMTCHK_RESID_TAB_LINES_NIR":
                arc_line_list_nir = filename
                self.category_label = "NIR"

            if catg == "FMTCHK_ON_UVB":
                fmtchk_on_uvb = filename
            if catg == "FMTCHK_ON_VIS":
                fmtchk_on_vis = filename
            if catg == "FMTCHK_ON_NIR":
                fmtchk_on_nir = filename

        # Build and return the (one) file name list
        file_lists = []
        if arc_line_list_uvb is not None and fmtchk_on_uvb is not None:
            # for ext in ['WIN1','WIN2']:
            file_lists.append(
                {
                    "line_guess": arc_line_list_uvb,
                    "arc_lamp": fmtchk_on_uvb,
                }
            )
        if arc_line_list_vis is not None and fmtchk_on_vis is not None:
            file_lists.append(
                {
                    "line_guess": arc_line_list_vis,
                    "arc_lamp": fmtchk_on_vis,
                }
            )
        if arc_line_list_nir is not None and fmtchk_on_nir is not None:
            file_lists.append(
                {
                    "line_guess": arc_line_list_nir,
                    "arc_lamp": fmtchk_on_nir,
                }
            )
        return file_lists

    def generate_panels(self, **kwargs):
        panels = {}
        self.metadata = XshooterSetupInfo.order_prediction(
            list(self.hdus[0].values())[0]
        )

        exts = []
        data_ext = []

        plot_params = {
            "ext": 1,
            "mask_data": "Flag",
            "mask": 0,
            "pos_plot_x": "Order",
            "pos_plot_y": "YGauss",
            "pos_plot_limits": "x",
            "xres_plot_x": "YGauss",
            "xres_plot_y": "ResidXmodel",
            "yres_plot_x": "YGauss",
            "yres_plot_y": "ResidYmodel",
            "fres_plot_x": "ResidXmodel",
            "fres_plot_y": "ResidYmodel",
            "raw_cutX_vclip": "percentile",
            "raw_cutX_vclip_kwargs": {"percentile": 95.0},
            "raw_cutX": "y",
            "raw_hist_x": "X",
            "raw_hist_y": "YNEW",
            "overlay_x": "XGauss",
            "overlay_y": "YGauss",
            "arc_yoffset": False,
            "raw_cutX_rot": None,
            "raw_cutX_flip": None,
        }

        for file_lists in self.hdus:
            exts.append("PRIMARY")
            data_ext.append([0])

        new_panels = super().generate_panels(
            arc_lamp_ext=exts, data_ext=data_ext, plot_params=plot_params
        )
        extra_panels = {}

        for i, (panel, panel_descr) in enumerate(new_panels.items()):

            vspace = 0.3

            file_lists = self.hdus[i]
            mdata_hdul = file_lists["line_guess"]
            mdata_ext = "PRIMARY"
            # Text Plot
            if "FMTCHK_RESID_TAB_LINES" in file_lists["line_guess"][
                "PRIMARY"
            ].header.get("HIERARCH ESO PRO CATG"):

                t1 = TextPlot(columns=1, v_space=vspace)
                fname = os.path.basename(str(mdata_hdul.filename()))

                col1 = (
                    str(mdata_hdul[mdata_ext].header.get("INSTRUME")),
                    "EXTNAME: N/A",
                    "PRO CATG: "
                    + str(mdata_hdul[mdata_ext].header.get("HIERARCH ESO PRO CATG")),
                    "FILE NAME: " + fname,
                    "RAW1 NAME: "
                    + str(
                        mdata_hdul["PRIMARY"].header.get(
                            "HIERARCH ESO PRO REC1 RAW1 NAME"
                        )
                    ),
                )
                t1.add_data(col1)
                panel.assign_plot(t1, 0, 0, xext=2)

                t2 = TextPlot(columns=1, v_space=vspace, xext=1)
                col2 = self.metadata

                t2.add_data(col2)
                panel.assign_plot(t2, 2, 0, xext=1)
            else:

                t1 = TextPlot(columns=1, v_space=vspace)
                fname = os.path.basename(str(mdata_hdul.filename()))

                col1 = (
                    str(mdata_hdul[mdata_ext].header.get("INSTRUME")),
                    "EXTNAME: N/A",
                    "PRO CATG: "
                    + str(mdata_hdul[mdata_ext].header.get("HIERARCH ESO PRO CATG")),
                    "FILE NAME: " + fname,
                    "RAW1 NAME: "
                    + str(
                        mdata_hdul["PRIMARY"].header.get(
                            "HIERARCH ESO PRO REC1 RAW1 NAME"
                        )
                    ),
                )
                t1.add_data(col1)
                panel.assign_plot(t1, 0, 0, xext=2)

                t2 = TextPlot(columns=1, v_space=vspace, xext=1)
                col2 = self.metadata
                t2.add_data(col2)
                panel.assign_plot(t2, 2, 0, xext=1)

            panel_descr["report_description"] = (
                f"XSHOOTER formatcheck panel - " f"{panel_descr['ext']}"
            )
            panel_descr["report_name"] = "XSHOOTER_" + panel_descr["report_name"]

            ### Additional panel for sigma plots

            pextra = Panel(4, 3, height_ratios=[1, 4, 4])

            pextra.assign_plot(t1, 0, 0, xext=2)
            pextra.assign_plot(t2, 2, 0, xext=1)

            rplot_size = 1.0
            line_guess_hdul = self.hdus[i]["line_guess"]
            arc_lamp_hdul = self.hdus[i]["arc_lamp"]
            line_guess_procatg = fetch_kw_or_default(
                line_guess_hdul["PRIMARY"], "HIERARCH ESO PRO CATG"
            )

            mask = line_guess_hdul[1].data["Flag"] == 0
            selplot = line_guess_hdul[1].data[mask]

            # Sigma in x (line width) vs. x position
            sx_x_plot = ScatterPlot(
                title="Line width along X",
                x_label="XGauss",
                y_label="SigmaXGauss / pixel",
                markersize=rplot_size,
                legend=False,
            )
            sx_x_plot.add_data(
                (
                    selplot["XGauss"],
                    selplot["SigmaXGauss"],
                ),
                color="black",
                label="Sigma X",
            )
            pextra.assign_plot(sx_x_plot, 0, 1, xext=2, yext=2)

            # Sigma in y (line width) vs. y position
            sy_y_plot = ScatterPlot(
                title="Line width along Y",
                x_label="YGauss",
                y_label="SigmaYGauss / pixel",
                markersize=rplot_size,
                legend=False,
            )
            sy_y_plot.add_data(
                (
                    selplot["YGauss"],
                    selplot["SigmaYGauss"],
                ),
                color="black",
                label="Sigma Y",
            )
            pextra.assign_plot(sy_y_plot, 2, 1, xext=2, yext=2)

            ext = 0
            input_files = [line_guess_hdul.filename(), arc_lamp_hdul.filename()] 

            addme2 = {
                "ext": "PRIMARY",
                "line_guess": line_guess_hdul.filename(),
                "arc_lamp": arc_lamp_hdul.filename(),
                "arc_lamp_ext": "PRIMARY",
                "report_name": f"XSHOOTER_{line_guess_procatg.lower()}_{str(ext).lower()}_sigma",
                "report_description": f"XSHOOTER formatcheck panel extra" f"{ext}",
                "report_tags": [],
                "input_files": input_files,
            }

            extra_panels[pextra] = addme2

        panels = {**new_panels, **extra_panels}
        return panels


rep = XshooterFormatCheckReport()
