import os
from adari_core.plots.panel import Panel
from adari_core.plots.text import TextPlot
from adari_core.utils.utils import fetch_kw_or_default
from adari_core.utils.utils import read_idp_spectrum
from adari_core.data_libs.master_spec_science import MasterSpecScienceReport

from .xshooter_utils import XshooterReportMixin

class XshooterScienceReport(XshooterReportMixin, MasterSpecScienceReport):
    def __init__(self):
        super().__init__("xshooter_science")

    def parse_sof(self):
        sci_spec = None

        input_tags = ["SCI_SLIT_FLUX_IDP_UVB", "SCI_SLIT_FLUX_IDP_VIS", "SCI_SLIT_FLUX_IDP_NIR", 
                      "TELL_SLIT_FLUX_IDP_UVB", "TELL_SLIT_FLUX_IDP_VIS", "TELL_SLIT_FLUX_IDP_NIR"]

        for filename, catg in self.inputs:
            if catg in input_tags:
                sci_spec = filename

        file_dict = {}
        if sci_spec is not None:
            file_dict["science"] = sci_spec

        return [
            file_dict,
        ]

    def generate_panels(self, **kwargs):
        panels = {}
        
        science = self.hdus[0]["science"]
        science_fname = os.path.basename(str(science.filename()))

        spec_wave, spec_flux, spec_flux_unit = read_idp_spectrum(science["SPECTRUM"])
        self.snr = science["SPECTRUM"].data["SNR"][0]
        self.qual = science["SPECTRUM"].data["QUAL"][0]

        p = Panel(6, 5, height_ratios=[1, 7, 1, 3, 1], y_stretch=0.84)
   
        arm = fetch_kw_or_default(science["PRIMARY"], "ESO SEQ ARM", default="N/A")
        
        binning = 0.5  # nm
        
        if arm == "NIR":
            self.zoom_range = 5.  # nm width
        else:
            self.zoom_range = 2.5  # nm width
        
        # Create the main spectrum plot
        low_wave = low_flux = high_wave = high_flux = None

        specplot, lowhighplot = self.generate_spec_plot(
            spec_wave,
            spec_flux,
            spec_flux_unit,
            binning,
            low_wave,
            low_flux,
            high_wave,
            high_flux,
        )
        p.assign_plot(specplot, 0, 1, xext=6)

        # Generate S/N plot
        snrplot = self.generate_snr_plot(spec_wave, self.snr, ylim=True)
        p.assign_plot(snrplot, 0, 2, xext=6)

        # Add spectral zoom plots
        zoomplots = self.generate_spec_zooms(
            spec_wave, spec_flux, spec_flux_unit,
        )
        xpos = 0
        for pl in zoomplots:
            if arm == "NIR":
                pl.x_major_ticks = 1.
            p.assign_plot(pl, xpos, 3, xext=2)
            xpos += 2

        # Upper Text Plot
        vspace = 0.4
        t0 = TextPlot(columns=1, v_space=vspace)
        col0 = (
            str(fetch_kw_or_default(science["PRIMARY"], "INSTRUME", default="N/A"))
            + " science product preview",
            "Product: "
            + str(
                fetch_kw_or_default(science["PRIMARY"], "ESO PRO CATG", default="N/A")
            ),
            "Raw file: "
            + str(
                fetch_kw_or_default(
                    science["PRIMARY"], "ESO PRO REC1 RAW1 NAME", default="N/A"
                )
            ),
            "MJD-OBS: "
            + str(fetch_kw_or_default(science["PRIMARY"], "MJD-OBS", default="N/A")),
        )
        t0.add_data(col0, fontsize=13)
        p.assign_plot(t0, 0, 0, xext=1)

        t1 = TextPlot(columns=1, v_space=vspace)
        col1 = (
            "Target: "
            + str(fetch_kw_or_default(science["PRIMARY"], "OBJECT", default="N/A")),
            "OB ID: "
            + str(fetch_kw_or_default(science["PRIMARY"], "ESO OBS ID", default="N/A")),
            "OB NAME: "
            + str(
                fetch_kw_or_default(science["PRIMARY"], "ESO OBS NAME", default="N/A")
            ),
            "TPL ID: "
            + str(fetch_kw_or_default(science["PRIMARY"], "ESO TPL ID", default="N/A")),
            "RUN ID: "
            + str(
                fetch_kw_or_default(
                    science["PRIMARY"], "ESO OBS PROG ID", default="N/A"
                )
            ),
        )
        t1.add_data(col1, fontsize=13)
        p.assign_plot(t1, 2, 0, xext=1)

        #Find slit
        slit = "N/A"
        if arm == "UVB":
            slit = fetch_kw_or_default(science["PRIMARY"], "ESO INS OPTI3 NAME", default="N/A")
        elif arm == "VIS":
            slit = fetch_kw_or_default(science["PRIMARY"], "ESO INS OPTI4 NAME", default="N/A")
        elif arm == "NIR":
            slit = fetch_kw_or_default(science["PRIMARY"], "ESO INS OPTI5 NAME", default="N/A")

        t2 = TextPlot(columns=1, v_space=vspace)
        col2 = (
            "PRO TECH: "
            + str(
                fetch_kw_or_default(science["PRIMARY"], "ESO PRO TECH", default="N/A")
            ),
            "Arm: " + str(arm),
            "Slit: " + str(slit),
            "Binning: "
            + str(
                fetch_kw_or_default(science["PRIMARY"], "ESO DET WIN1 BINX", default="N/A")
            )
            + " x "
            + str(
                fetch_kw_or_default(science["PRIMARY"], "ESO DET WIN1 BINY", default="N/A")
            ),
        )
        t2.add_data(col2, fontsize=13)
        p.assign_plot(t2, 4, 0, xext=1)

        # Bottom Text Plot
        vspace = 0.4
        t4 = TextPlot(columns=1, v_space=vspace)
        col4 = ()
        col4 += (
            "Exp. time [s]: "
            + "%.1f"
            % fetch_kw_or_default(science["PRIMARY"], "TEXPTIME", default="N/A"),
            "N exposures: "
            + "%i"
            % fetch_kw_or_default(science["PRIMARY"], "NCOMBINE", default="N/A"),
            "S/N average: "
            + "%.2f"
            % fetch_kw_or_default(
                science["PRIMARY"], "ESO QC MEAN S2N", default="N/A"
            ),
            "Seeing: "
            + str(fetch_kw_or_default(
                science["PRIMARY"], "ESO QC FWHM AVG", default="N/A")
            ),
            "Airmass: "
            + str(fetch_kw_or_default(
                science["PRIMARY"], "ESO QC AIRM AVG", default="N/A")
            ),
        )
        t4.add_data(col4, fontsize=13)
        p.assign_plot(t4, 0, 4, xext=1)

        t5 = TextPlot(columns=1, v_space=vspace)
        col5 = ()
        col5 += (
            "Lambda bin [nm]: "
            + "%.5f"
            % fetch_kw_or_default(science["PRIMARY"], "SPEC_BIN", default="N/A"),
            "Lambda start [nm]: "
            + "%.5f"
            % fetch_kw_or_default(
                science["PRIMARY"], "WAVELMIN", default="N/A"
            ),
            "Lambda end [nm]: "
            + "%.5f"
            % fetch_kw_or_default(
                science["PRIMARY"], "WAVELMAX", default="N/A"
            ),
            "N pix sat: "
            + str(fetch_kw_or_default(
                science["PRIMARY"], "ESO QC NUM SAT", default="N/A")
            ),
        )
        t5.add_data(col5, fontsize=13)
        p.assign_plot(t5, 2, 4, xext=1)

        t6 = TextPlot(columns=1, v_space=vspace)
        col6 = ()
        col6 += (
            "Water vapour: "
            + str(fetch_kw_or_default(
                science["PRIMARY"], "ESO QC IWV AVG", default="N/A")
            ),
            "N cosmic ray hits: "
            + str(
                fetch_kw_or_default(
                science["PRIMARY"], "ESO QC NCRH AVG", default="N/A")
            ),
            "Mean flux: "
            + "%.2f"
            % fetch_kw_or_default(
                science["PRIMARY"], "ESO QC MEAN RED", default="N/A"
            ),
            "RMS of flux: "
            + "%.2f"
            % fetch_kw_or_default(
                science["PRIMARY"], "ESO QC RMS RED", default="N/A"
            ),
        )
        t6.add_data(col6, fontsize=13)
        p.assign_plot(t6, 4, 4, xext=1)

        input_files = [science.filename()]
        for k in [key for key in self.hdus[0] if key.startswith('raw')]:
            input_files.append(self.hdus[0][k].filename())
        addme = {
            "report_name": f"XSHOOTER_{str(science_fname).removesuffix('.fits').lower()}",
            "report_description": "Science panel",
            "report_tags": [],
            "report_prodcatg": "ANCILLARY.PREVIEW",
            "input_files": input_files,
        }

        panels[p] = addme

        return panels


rep = XshooterScienceReport()

