import numpy as np
import os

from adari_core.report import AdariReportBase
from adari_core.plots.panel import Panel
from adari_core.plots.images import ImagePlot, CentralImagePlot
from adari_core.plots.points import ScatterPlot
from adari_core.plots.histogram import HistogramPlot
from adari_core.plots.cut import CutPlot
from adari_core.plots.text import TextPlot
from adari_core.utils.utils import fetch_kw_or_default
from .xshooter_utils import XshooterSetupInfo, XshooterReportMixin

class XshooterWavelengthCalibReport(XshooterReportMixin, AdariReportBase):
    #    raw_extensions = {"blue": "PRIMARY", "redu": "CCD-20", "redl": "CCD-44"}

    def __init__(self):

        super().__init__("xshooter_wavelength")

    def parse_sof(self):

        line_table_uvb = None
        line_table_vis = None
        line_table_nir = None

        line_table_cen = None
        line_table_up = None
        line_table_down = None

        raw_uvb = None
        raw_vis = None
        raw_nir = None

        arc_lamp_uvb = None
        arc_lamp_vis = None
        arc_lamp_nir = None

        for filename, catg in self.inputs:

            if (
                catg == "WAVE_RESID_TAB_LINES_NIR"
                or catg == "WAVE_RESID_TAB_LINES_VIS"
                or catg == "WAVE_RESID_TAB_LINES_UVB"
            ):
                self.mode_label = "2DMAP"
            elif (
                catg == "TILT_TAB_SLIT_VIS"
                or catg == "TILT_TAB_SLIT_NIR"
                or catg == "TILT_TAB_SLIT_UVB"
            ):
                self.mode_label = "SLIT"
            else:
                if (
                    catg == "TILT_TAB_UP_IFU_VIS"
                    or catg == "TILT_TAB_UP_IFU_NIR"
                    or catg == "TILT_TAB_UP_IFU_UVB"
                ):
                    self.mode_label = "IFU"
                    line_table_up = filename
                if (
                    catg == "TILT_TAB_CEN_IFU_VIS"
                    or catg == "TILT_TAB_CEN_IFU_NIR"
                    or catg == "TILT_TAB_CEN_IFU_UVB"
                ):
                    self.mode_label = "IFU"
                    line_table_cen = filename
                if (
                    catg == "TILT_TAB_DOWN_IFU_VIS"
                    or catg == "TILT_TAB_DOWN_IFU_NIR"
                    or catg == "TILT_TAB_DOWN_IFU_UVB"
                ):
                    self.mode_label = "IFU"
                    line_table_down = filename

            if (
                catg == "TILT_TAB_SLIT_UVB"
                or catg == "WAVE_RESID_TAB_LINES_UVB"
                or catg == "TILT_TAB_DOWN_IFU_UVB"
            ):
                if self.mode_label != "IFU":
                    line_table_uvb = filename
                self.category_label = "UVB"
            elif (
                catg == "TILT_TAB_SLIT_VIS"
                or catg == "WAVE_RESID_TAB_LINES_VIS"
                or catg == "TILT_TAB_DOWN_IFU_VIS"
            ):
                if self.mode_label != "IFU":
                    line_table_vis = filename
                self.category_label = "VIS"
            elif (
                catg == "TILT_TAB_SLIT_NIR"
                or catg == "WAVE_RESID_TAB_LINES_NIR"
                or catg == "TILT_TAB_DOWN_IFU_NIR"
            ):
                if self.mode_label != "IFU":
                    line_table_nir = filename
                self.category_label = "NIR"

                file_lists = []
                if (
                    line_table_uvb is not None
                    and arc_lamp_uvb is not None
                    and raw_uvb is not None
                ):
                    file_lists.append(
                        {
                            "raw": raw_uvb,
                            "line_table": line_table_uvb,
                            "arc_lamp": arc_lamp_uvb,
                        }
                    )
                if (
                    line_table_vis is not None
                    and arc_lamp_vis is not None
                    and raw_vis is not None
                ):
                    file_lists.append(
                        {
                            "raw": raw_vis,
                            "line_table": line_table_vis,
                            "arc_lamp": arc_lamp_vis,
                        }
                    )
                if (
                    line_table_nir is not None
                    and arc_lamp_nir is not None
                    and raw_nir is not None
                ):
                    file_lists.append(
                        {
                            "raw": raw_nir,
                            "line_table": line_table_nir,
                            "arc_lamp": arc_lamp_nir,
                        }
                    )

            if catg == "WAVE_UVB" or catg == "ARC_IFU_UVB" or catg == "ARC_SLIT_UVB":
                raw_uvb = filename
            elif catg == "WAVE_VIS" or catg == "ARC_SLIT_VIS" or catg == "ARC_IFU_VIS":
                raw_vis = filename
            elif (
                catg == "WAVE_NIR_ON"
                or catg == "ARC_SLIT_NIR_ON"
                or catg == "ARC_IFU_NIR_ON"
            ):
                raw_nir = filename

            if (
                catg == "ARC_SLIT_ON_UVB"
                or catg == "ARC_IFU_ON_UVB"
                or catg == "WAVE_ON_UVB"
            ):
                arc_lamp_uvb = filename
            elif (
                catg == "ARC_SLIT_ON_VIS"
                or catg == "ARC_IFU_ON_VIS"
                or catg == "WAVE_ON_VIS"
            ):
                arc_lamp_vis = filename
            elif (
                catg == "ARC_SLIT_ON_NIR"
                or catg == "ARC_IFU_ON_NIR"
                or catg == "WAVE_ON_NIR"
            ):
                arc_lamp_nir = filename

        file_lists = []
        if (
            line_table_uvb is not None
            and arc_lamp_uvb is not None
            and raw_uvb is not None
        ):
            file_lists.append(
                {
                    "raw": raw_uvb,
                    "line_table": line_table_uvb,
                    "arc_lamp": arc_lamp_uvb,
                }
            )
        elif (
            line_table_up is not None
            and arc_lamp_uvb is not None
            and raw_uvb is not None
        ):
            file_lists.append(
                {
                    "raw": raw_uvb,
                    "line_table_up": line_table_up,
                    "line_table_down": line_table_down,
                    "line_table_cen": line_table_cen,
                    "arc_lamp": arc_lamp_uvb,
                }
            )

        if (
            line_table_vis is not None
            and arc_lamp_vis is not None
            and raw_vis is not None
        ):
            file_lists.append(
                {
                    "raw": raw_vis,
                    "line_table": line_table_vis,
                    "arc_lamp": arc_lamp_vis,
                }
            )
        elif (
            line_table_up is not None
            and arc_lamp_vis is not None
            and raw_vis is not None
        ):
            file_lists.append(
                {
                    "raw": raw_vis,
                    "line_table_up": line_table_up,
                    "line_table_down": line_table_down,
                    "line_table_cen": line_table_cen,
                    "arc_lamp": arc_lamp_vis,
                }
            )

        if (
            line_table_nir is not None
            and arc_lamp_nir is not None
            and raw_nir is not None
        ):
            file_lists.append(
                {
                    "raw": raw_nir,
                    "line_table": line_table_nir,
                    "arc_lamp": arc_lamp_nir,
                }
            )
        elif (
            line_table_up is not None
            and arc_lamp_nir is not None
            and raw_nir is not None
        ):
            file_lists.append(
                {
                    "raw": raw_nir,
                    "line_table_up": line_table_up,
                    "line_table_down": line_table_down,
                    "line_table_cen": line_table_cen,
                    "arc_lamp": arc_lamp_nir,
                }
            )

        return file_lists

    def report_panels_2dmap(self, p):

        fname = os.path.basename(str(self.line_table_hdul.filename()))
        hdr = self.line_table_hdul["PRIMARY"].header
        t1 = TextPlot(columns=1, v_space=self.vspace)
        t2 = TextPlot(columns=1, v_space=self.vspace, xext=1)
        px = self.px
        py = self.py
        mask = self.line_table_hdul[1].data["FLAG"] == 0
        selplot = self.line_table_hdul[1].data[mask]

        col1 = (
            str(hdr.get("INSTRUME")),
            "EXTNAME: " + "PRIMARY",
            "PRO CATG: " + str(hdr.get("HIERARCH ESO PRO CATG")),
            "FILE NAME: " + fname,
            "RAW1 NAME: " + str(hdr.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
        )

        self.metadata = XshooterSetupInfo.wavelength_calibration_2d(
            self.line_table_hdul
        )

        # 1 Cut in cross-dispersion direction (at centre of dispersion direction) through full raw file.
        arc_plot = ImagePlot(
            self.arc_lamp_image_data,
            title="Arc plot",
            v_clip="percentile",
            v_clip_kwargs={"percentile": 95.0},
        )
        arc_center = CentralImagePlot(
            arc_plot,
            extent=self.center_size,
            title="raw center",
            v_clip="percentile",
            v_clip_kwargs={"percentile": 95.0},
        )

        cutpos = arc_plot.get_data_coord(np.floor(arc_plot.data.shape[0] // 2), "y")
        arc_cutX = CutPlot(
            "y",
            title="Arc: cut at Y {}".format(cutpos),
            x_label="x",
            y_label=fetch_kw_or_default(
                self.arc_lamp_hdul["PRIMARY"], "BUNIT", default="ADU"
            ),
            legend=False,
        )
        arc_cutX.add_data(arc_plot, cutpos, color="red", label="arc")
        p.assign_plot(arc_cutX, 0, 1)

        # 2 Same for central region
        cutpos_c = arc_center.get_data_coord(
            np.floor(arc_center.data.shape[0] // 2), "y"
        )
        arc_cen_cutX = CutPlot(
            "y",
            title="Arc central region: cut at Y {}".format(cutpos),
            x_label="x",
            y_label=fetch_kw_or_default(
                self.arc_lamp_hdul["PRIMARY"],
                "BUNIT",
                default="ADU",
            ),
            legend=False,
        )
        arc_cen_cutX.add_data(arc_center, cutpos_c, label="arc", color="red")
        p.assign_plot(arc_cen_cutX, 1, 1)

        # 3 Histogram Plot
        raw_plot = ImagePlot(
            self.raw_data,
            title="Raw plot",
            v_clip="percentile",
            v_clip_kwargs={"percentile": 95.0},
        )

        raw_hist = HistogramPlot(
            raw_data=raw_plot.data,
            title="Raw value counts",
            bins=50,
            v_min=-5000,
            v_max=70000,
            legend=False,
        )
        p.assign_plot(raw_hist, 1, 2)

        # 4 Fit residuals in x (ResidXmodel) vs. Ycen
        resorder_plot = ScatterPlot(
            title="Fit residuals x",
            x_label="YGauss",
            y_label="Residuals in x direction",
            markersize=self.rplot_size,
            legend=False,
        )

        resorder_plot.add_data(
            (selplot["YGauss"], selplot["ResidXmodel"]),
            marker="circle",
            label="Used lines",
            color="red",
        )
        p.assign_plot(resorder_plot, 2, 1)

        # 5 Fit residuals in y (ResidXmodel) vs. Ycen
        resorder_plot = ScatterPlot(
            title="Fit residuals y",
            x_label="YGauss",
            y_label="Residuals in y direction",
            markersize=self.rplot_size,
            legend=False,
        )

        resorder_plot.add_data(
            (selplot["YGauss"], selplot["ResidYmodel"]),
            marker="circle",
            label="Used lines",
            color="red",
        )
        p.assign_plot(resorder_plot, 2, 2)

        # 6 Line width in x direction (SigmaXGauss)vs. XGauss

        line_width_plot = ScatterPlot(
            title="Fit line width x",
            x_label="XGauss",
            y_label="Line width in x direction",
            markersize=self.rplot_size,
            legend=False,
        )

        line_width_plot.add_data(
            (selplot["XGauss"], selplot["SigmaXGauss"]),
            marker="circle",
            label="Used lines",
            color="red",
        )
        p.assign_plot(line_width_plot, 3, 1)

        # 7 Line width in y direction (SigmaYGauss)vs. YGauss

        line_width_plot = ScatterPlot(
            title="Fit line width y",
            x_label="YGauss",
            y_label="Line width in y direction",
            markersize=self.rplot_size,
            legend=False,
        )

        line_width_plot.add_data(
            (selplot["YGauss"], selplot["SigmaYGauss"]),
            marker="circle",
            label="Used lines",
            color="red",
        )
        p.assign_plot(line_width_plot, 3, 2)

        t1.add_data(col1)
        p.assign_plot(t1, px, py, xext=2)

        px = px + 2
        t2 = TextPlot(columns=1, v_space=self.vspace, xext=1)
        col2 = self.metadata
        t2.add_data(col2)
        p.assign_plot(t2, px, py, xext=1)

    def report_panels_ifu(self, p):

        fname = os.path.basename(str(self.line_table_cen.filename()))
        t1 = TextPlot(columns=1, v_space=self.vspace)
        t2 = TextPlot(columns=1, v_space=self.vspace, xext=1)
        px = self.px
        py = self.py
        mask_cen = self.line_table_cen[1].data["FLAG"] == 0
        selplot_cen = self.line_table_cen[1].data[mask_cen]
        mask_up = self.line_table_up[1].data["FLAG"] == 0
        selplot_up = self.line_table_up[1].data[mask_up]
        mask_down = self.line_table_down[1].data["FLAG"] == 0
        selplot_down = self.line_table_down[1].data[mask_down]

        col1 = (
            str(self.line_table_cen["PRIMARY"].header.get("INSTRUME")),
            "EXTNAME: " + "PRIMARY",
            "PRO CATG: "
            + str(self.line_table_cen["PRIMARY"].header.get("HIERARCH ESO PRO CATG")),
            "FILE NAME: " + fname,
            "RAW1 NAME: "
            + str(
                self.line_table_cen["PRIMARY"].header.get(
                    "HIERARCH ESO PRO REC1 RAW1 NAME"
                )
            ),
        )

        self.metadata = XshooterSetupInfo.arc(self.line_table_cen)
        # 1. Position of the arc lines that are used for the wavelength solution.
        # Y axis being dispersion direction (in pixels), x axis being order number.
        xmin = max(selplot_cen["Order"]) + 1
        xmax = min(selplot_cen["Order"]) - 1

        pos_plot = ScatterPlot(
            title="Line positions",
            x_label="Order",
            y_label="Ycen",
            markersize=self.rplot_size,
            legend=True,
        )
        pos_plot.add_data(
            (selplot_cen["ORDER"], selplot_cen["Ycen"]),
            marker="circle",
            label="cen",
            color="blue",
        )
        pos_plot.add_data(
            (selplot_up["ORDER"], selplot_up["Ycen"]),
            marker="circle",
            label="up",
            color="green",
        )
        pos_plot.add_data(
            (selplot_down["ORDER"], selplot_down["Ycen"]),
            marker="circle",
            label="down",
            color="red",
        )

        # pos_plot.add_data((line_table.data['X'], line_table.data['Order']), label='Used lines', color='black')
        # flip the yaxis limits as per required convention
        pos_plot.set_xlim(xmin, xmax)
        p.assign_plot(pos_plot, 0, 1)

        # 2 Resolving power (SPECRES)of the lines vs.Ycen.

        pos_plot = ScatterPlot(
            title="Resolving power vs. dispersion",
            x_label="Ycen",
            y_label="Resolving power (SPECRES)",
            markersize=self.rplot_size,
            legend=True,
        )
        pos_plot.add_data(
            (selplot_cen["Ycen"], selplot_cen["SPECRES"]),
            marker="circle",
            label="cen",
            color="blue",
        )
        pos_plot.add_data(
            (selplot_up["Ycen"], selplot_up["SPECRES"]),
            marker="circle",
            label="up",
            color="green",
        )
        pos_plot.add_data(
            (selplot_down["Ycen"], selplot_down["SPECRES"]),
            marker="circle",
            label="down",
            color="red",
        )

        p.assign_plot(pos_plot, 0, 2)

        # 3 Cut in cross-dispersion direction (at centre of dispersion direction) through arc lamp file.
        arc_plot = ImagePlot(
            self.arc_lamp_image_data,
            title="Arc plot",
            v_clip="percentile",
            v_clip_kwargs={"percentile": 95.0},
        )
        arc_center = CentralImagePlot(
            arc_plot,
            extent=self.center_size,
            title="raw center",
            v_clip="percentile",
            v_clip_kwargs={"percentile": 95.0},
        )

        cutpos = arc_plot.get_data_coord(np.floor(arc_plot.data.shape[0] // 2), "y")
        arc_cutX = CutPlot(
            "y",
            title="Arc: cut at Y {}".format(cutpos),
            x_label="x",
            y_label=fetch_kw_or_default(
                self.arc_lamp_hdul["PRIMARY"], "BUNIT", default="ADU"
            ),
            legend=False,
        )
        arc_cutX.add_data(arc_plot, cutpos, color="red", label="arc")
        p.assign_plot(arc_cutX, 1, 1)

        # 4 Same for central region
        cutpos_c = arc_center.get_data_coord(
            np.floor(arc_center.data.shape[0] // 2), "y"
        )
        arc_cen_cutX = CutPlot(
            "y",
            title="Arc central region: cut at Y {}".format(cutpos),
            x_label="x",
            y_label=fetch_kw_or_default(
                self.arc_lamp_hdul["PRIMARY"], "BUNIT", default="ADU"
            ),
            legend=False,
        )
        arc_cen_cutX.add_data(arc_center, cutpos_c, label="arc", color="red")
        p.assign_plot(arc_cen_cutX, 1, 2)

        # 5 Histogram Plot
        raw_plot = ImagePlot(
            self.raw_data,
            title="Raw plot",
            v_clip="percentile",
            v_clip_kwargs={"percentile": 95.0},
        )

        raw_hist = HistogramPlot(
            raw_data=raw_plot.data,
            title="Raw value counts",
            bins=50,
            v_min=-5000,
            v_max=70000,
            legend=False,
        )
        p.assign_plot(raw_hist, 2, 1)

        # 6 Resolving power (SPECRES) vs. tilt(TILT, in degrees).
        resorder_plot = ScatterPlot(
            title="Resolving power vs. tilt",
            x_label="TILT",
            y_label="Resolving power (SPECRES)",
            markersize=self.rplot_size,
            legend=True,
        )

        resorder_plot.add_data(
            (selplot_cen["TILT"], selplot_cen["SPECRES"]),
            marker="circle",
            label="cen",
            color="blue",
        )
        resorder_plot.add_data(
            (selplot_up["TILT"], selplot_up["SPECRES"]),
            marker="circle",
            label="up",
            color="green",
        )
        resorder_plot.add_data(
            (selplot_down["TILT"], selplot_down["SPECRES"]),
            marker="circle",
            label="down",
            color="red",
        )
        p.assign_plot(resorder_plot, 2, 2)

        # 7 Shift along y (SHIFT_Y) axis vs. Ycen.Range for y axis: -5, 5.
        respos_plot = ScatterPlot(
            title="Fit residuals",
            x_label="Ycen",
            y_label="Shift along y (SHIFT_Y)",
            markersize=self.rplot_size,
            y_min=-5,
            y_max=5,
            legend=True,
        )
        respos_plot.add_data(
            (selplot_cen["Ycen"], selplot_cen["SHIFT_Y"]),
            marker="circle",
            label="cen",
            color="blue",
        )
        respos_plot.add_data(
            (selplot_up["Ycen"], selplot_up["SHIFT_Y"]),
            marker="circle",
            label="up",
            color="green",
        )
        respos_plot.add_data(
            (selplot_down["Ycen"], selplot_down["SHIFT_Y"]),
            marker="circle",
            label="down",
            color="red",
        )
        p.assign_plot(respos_plot, 3, 1)

        # 8 Same, dynamic scale for y axis.
        respos_plot = ScatterPlot(
            title="Fit residuals (dynamic scale)",
            x_label="Ycen",
            y_label="Shift along y (SHIFT_Y)",
            markersize=self.rplot_size,
            legend=True,
        )

        respos_plot.add_data(
            (selplot_cen["Ycen"], selplot_cen["SHIFT_Y"]),
            marker="circle",
            label="cen",
            color="blue",
        )
        respos_plot.add_data(
            (selplot_up["Ycen"], selplot_up["SHIFT_Y"]),
            marker="circle",
            label="up",
            color="green",
        )
        respos_plot.add_data(
            (selplot_down["Ycen"], selplot_down["SHIFT_Y"]),
            marker="circle",
            label="down",
            color="red",
        )

        p.assign_plot(respos_plot, 3, 2)

        t1.add_data(col1)
        p.assign_plot(t1, px, py, xext=2)

        px = px + 2
        t2 = TextPlot(columns=1, v_space=self.vspace, xext=1)
        col2 = self.metadata
        t2.add_data(col2)
        p.assign_plot(t2, px, py, xext=1)

    def report_panels_wavecal(self, p):

        fname = os.path.basename(str(self.line_table_hdul.filename()))
        t1 = TextPlot(columns=1, v_space=self.vspace)
        t2 = TextPlot(columns=1, v_space=self.vspace, xext=1)
        px = self.px
        py = self.py
        mask = self.line_table_hdul[1].data["FLAG"] == 0
        selplot = self.line_table_hdul[1].data[mask]

        col1 = (
            str(self.line_table_hdul["PRIMARY"].header.get("INSTRUME")),
            "EXTNAME: " + "PRIMARY",
            "PRO CATG: "
            + str(self.line_table_hdul["PRIMARY"].header.get("HIERARCH ESO PRO CATG")),
            "FILE NAME: " + fname,
            "RAW1 NAME: "
            + str(
                self.line_table_hdul["PRIMARY"].header.get(
                    "HIERARCH ESO PRO REC1 RAW1 NAME"
                )
            ),
        )

        self.metadata = XshooterSetupInfo.arc(self.line_table_hdul)

        # 1. Position of the arc lines that are used for the wavelength solution.
        # Y axis being dispersion direction (in pixels), x axis being order number.
        xmin = max(selplot["Order"]) + 1
        xmax = min(selplot["Order"]) - 1

        pos_plot = ScatterPlot(
            title="Line positions",
            x_label="Order",
            y_label="Ycen",
            markersize=self.rplot_size,
            legend=False,
        )
        pos_plot.add_data(
            (selplot["ORDER"], selplot["Ycen"]),
            marker="circle",
            label="Used lines",
            color="red",
        )
        # pos_plot.add_data((line_table.data['X'], line_table.data['Order']), label='Used lines', color='black')
        # flip the yaxis limits as per required convention
        pos_plot.set_xlim(xmin, xmax)
        p.assign_plot(pos_plot, 0, 1)

        # 2 Resolving power (SPECRES)of the lines vs.Ycen.

        pos_plot = ScatterPlot(
            title="Resolving power vs. dispersion",
            x_label="Ycen",
            y_label="Resolving power (SPECRES)",
            markersize=self.rplot_size,
            legend=False,
        )
        pos_plot.add_data(
            (selplot["Ycen"], selplot["SPECRES"]),
            marker="circle",
            label="Used lines",
            color="red",
        )
        p.assign_plot(pos_plot, 0, 2)

        # 3 Cut in cross-dispersion direction (at centre of dispersion direction) through arc lamp file.
        arc_plot = ImagePlot(
            self.arc_lamp_image_data,
            title="Arc plot",
            v_clip="percentile",
            v_clip_kwargs={"percentile": 95.0},
        )
        arc_center = CentralImagePlot(
            arc_plot,
            extent=self.center_size,
            title="raw center",
            v_clip="percentile",
            v_clip_kwargs={"percentile": 95.0},
        )

        cutpos = arc_plot.get_data_coord(np.floor(arc_plot.data.shape[0] // 2), "y")
        arc_cutX = CutPlot(
            "y",
            title="Arc: cut at Y {}".format(cutpos),
            x_label="x",
            y_label=fetch_kw_or_default(
                self.arc_lamp_hdul["PRIMARY"], "BUNIT", default="ADU"
            ),
            legend=False,
        )
        arc_cutX.add_data(arc_plot, cutpos, color="red", label="arc")
        p.assign_plot(arc_cutX, 1, 1)

        # 4 Same for central region
        cutpos_c = arc_center.get_data_coord(
            np.floor(arc_center.data.shape[0] // 2), "y"
        )
        arc_cen_cutX = CutPlot(
            "y",
            title="Arc central region: cut at Y {}".format(cutpos),
            x_label="x",
            y_label=fetch_kw_or_default(
                self.arc_lamp_hdul["PRIMARY"], "BUNIT", default="ADU"
            ),
            legend=False,
        )
        arc_cen_cutX.add_data(arc_center, cutpos_c, label="arc", color="red")
        p.assign_plot(arc_cen_cutX, 1, 2)

        # 5 Histogram Plot
        raw_plot = ImagePlot(
            self.raw_data,
            title="Raw plot",
            v_clip="percentile",
            v_clip_kwargs={"percentile": 95.0},
        )

        raw_hist = HistogramPlot(
            raw_data=raw_plot.data,
            title="Raw value counts",
            bins=50,
            v_min=-5000,
            v_max=70000,
            legend=False,
        )
        p.assign_plot(raw_hist, 2, 1)

        # 6 Resolving power (SPECRES) vs. tilt(TILT, in degrees).
        resorder_plot = ScatterPlot(
            title="Resolving power vs. tilt",
            x_label="TILT",
            y_label="Resolving power (SPECRES)",
            markersize=self.rplot_size,
            legend=False,
        )

        resorder_plot.add_data(
            (selplot["TILT"], selplot["SPECRES"]),
            marker="circle",
            label="Used lines",
            color="red",
        )
        p.assign_plot(resorder_plot, 2, 2)

        # 7 Shift along y (SHIFT_Y) axis vs. Ycen.Range for y axis: -5, 5.
        respos_plot = ScatterPlot(
            title="Fit residuals",
            x_label="Ycen",
            y_label="Shift along y (SHIFT_Y)",
            markersize=self.rplot_size,
            y_min=-5,
            y_max=5,
            legend=False,
        )

        respos_plot.add_data(
            (selplot["Ycen"], selplot["SHIFT_Y"]),
            marker="circle",
            label="Used lines",
            color="red",
        )
        p.assign_plot(respos_plot, 3, 1)

        # 8 Same, dynamic scale for y axis.
        respos_plot = ScatterPlot(
            title="Fit residuals (dynamic scale)",
            x_label="Ycen",
            y_label="Shift along y (SHIFT_Y)",
            markersize=self.rplot_size,
            legend=False,
        )

        respos_plot.add_data(
            (selplot["Ycen"], selplot["SHIFT_Y"]),
            marker="circle",
            label="Used lines",
            color="red",
        )
        p.assign_plot(respos_plot, 3, 2)

        t1.add_data(col1)
        p.assign_plot(t1, px, py, xext=2)

        px = px + 2
        t2 = TextPlot(columns=1, v_space=self.vspace, xext=1)
        col2 = self.metadata
        t2.add_data(col2)
        p.assign_plot(t2, px, py, xext=1)

    def generate_panels(self, lplot_msize=0.05, ext=0, **kwargs):

        panels = {}
        self.center_size = 200
        self.raw_hdul = self.hdus[0]["raw"]
        self.raw_data = self.raw_hdul["PRIMARY"].data
        if self.mode_label == "IFU":
            self.line_table_cen = self.hdus[0]["line_table_cen"]
            self.line_table_up = self.hdus[0]["line_table_up"]
            self.line_table_down = self.hdus[0]["line_table_down"]
            table_fname_cen = self.line_table_cen.filename()
            self.table_fname_cen = os.path.basename(table_fname_cen)
            table_fname_up = self.line_table_cen.filename()
            self.table_fname_up = os.path.basename(table_fname_up)
            table_fname_down = self.line_table_cen.filename()
            self.table_fname_down = os.path.basename(table_fname_down)
            arm = self.line_table_cen["PRIMARY"].header.get("HIERARCH ESO INS PATH")
            procatg = self.line_table_cen["PRIMARY"].header["HIERARCH ESO PRO CATG"]
        else:
            self.line_table_hdul = self.hdus[0]["line_table"]
            table_fname = self.line_table_hdul.filename()
            self.table_fname = os.path.basename(table_fname)
            arm = self.line_table_hdul["PRIMARY"].header.get("HIERARCH ESO INS PATH")
            procatg = self.line_table_hdul["PRIMARY"].header["HIERARCH ESO PRO CATG"]

        if arm is None:
            raise ValueError(
                "XSHOOTER spectrograph arm [HIERARCH ESO INS PATH] not found"
            )
        arm = arm.lower().strip()
        # Table containing the the lines used for computing the wavelength solution
        # and resolving power (R)
        self.arc_lamp_hdul = self.hdus[0]["arc_lamp"]
        self.arc_lamp_image_data = self.arc_lamp_hdul["PRIMARY"].data
        ext = "PRIMARY"

        p = Panel(4, 3, height_ratios=[1, 4, 4])
        # Text Plot
        # Column 1
        # Used to keep track of position in Panel
        self.px = 0
        self.py = 0
        # which hdul and ext to use
        self.vspace = 0.3
        self.rplot_size = 1

        if self.mode_label == "2DMAP":
            self.report_panels_2dmap(p)
            filename = self.line_table_hdul.filename()
        elif self.mode_label == "IFU":
            self.report_panels_ifu(p)
            filename = self.line_table_cen.filename()
        else:
            self.report_panels_wavecal(p)
            filename = self.line_table_hdul.filename()

        input_files = [self.raw_hdul.filename(), self.arc_lamp_hdul.filename()]
        if self.mode_label == "IFU":
            input_files.append(table_fname_cen)
            input_files.append(table_fname_up)
            input_files.append(table_fname_down)
        else:
            input_files.append(table_fname)
        addme = {
            "ext": ext,
            "line_guess": filename,
            "arc_lamp": self.arc_lamp_hdul.filename(),
            "arc_lamp_ext": ext,
            "report_name": f"XSHOOTER_{procatg.lower()}_{str(ext).lower()}",
            "report_description": f"Wavelength cal panel - (" f"{ext})",
            "report_tags": [],
            "input_files": input_files,
        }

        panels[p] = addme
        return panels


rep = XshooterWavelengthCalibReport()
