from edps import classification_rule

from . import xshooter_keywords as kwd
from . import xshooter_rules as rules

# Basic rule dictionaries
is_xsh = {kwd.instrume: ["XSHOOTER", "SHOOT"]}
is_calib = {**is_xsh, kwd.dpr_catg: "CALIB"}
is_science = {**is_xsh, kwd.dpr_catg: "SCIENCE"}
is_acquisition = {**is_xsh, kwd.dpr_catg: "ACQUISITION"}

# Arm-specific dictionaries
is_uvb = {kwd.seq_arm: "UVB"}
is_vis = {kwd.seq_arm: "VIS"}
is_nir = {kwd.seq_arm: "NIR"}
is_agc = {kwd.seq_arm: "AGC"}  # Simplified from the function

# Mode-specific dictionaries
is_ifu = {kwd.ins_opti2_id: "IFU"}
is_slit = {kwd.ins_opti2_id: "SLOT"}

# Type-specific dictionaries
is_linearity_on = {kwd.dpr_type: "FLAT,LINEARITY,DETCHAR"}
is_image = {kwd.dpr_tech: "IMAGE"}
is_linearity_ir_off = {kwd.dpr_type: "FLAT,LINEARITY,DETCHAR"}
is_linearity_opt_off = {kwd.dpr_type: ["BIAS,LINEARITY,DETCHAR", "BIAS,DETCHECK"]}
is_bias = {kwd.dpr_type: "BIAS"}
is_dark = {kwd.dpr_type: "DARK"}
is_sky = {kwd.dpr_type: "SKY"}

# Combined arm and category dictionaries
is_calib_uvb = {**is_calib, **is_uvb}
is_calib_vis = {**is_calib, **is_vis}
is_calib_nir = {**is_calib, **is_nir}
is_calib_agc = {**is_calib, **is_agc}

# Format check dictionaries
is_thar_lamp = {kwd.ins_lamp1_name: "ThAr_Lamp"}
is_lamp_fmtchk = {kwd.dpr_type: "LAMP,FMTCHK"}
is_lamp_orderdef = {kwd.dpr_type: "LAMP,ORDERDEF"}
is_lamp_dorderdef = {kwd.dpr_type: "LAMP,DORDERDEF"}
is_lamp_qorderdef = {kwd.dpr_type: "LAMP,QORDERDEF"}
is_lamp_flat = {kwd.dpr_type: "LAMP,FLAT"}
is_lamp_dflat = {kwd.dpr_type: "LAMP,DFLAT"}
is_lamp_qflat = {kwd.dpr_type: "LAMP,QFLAT"}

is_echelle = {kwd.dpr_tech: "ECHELLE"}
is_echelle_pinhole = {kwd.dpr_tech: "ECHELLE,PINHOLE"}
is_echelle_multi_pinhole = {kwd.dpr_tech: "ECHELLE,MULTI-PINHOLE"}
is_echelle_ifu_nodding = {kwd.dpr_tech: "ECHELLE,IFU,NODDING"}
is_echelle_slit_stare = {kwd.dpr_tech: "ECHELLE,SLIT,STARE"}
is_echelle_slit_nodding = {kwd.dpr_tech: "ECHELLE,SLIT,NODDING"}
is_echelle_slit_offset = {kwd.dpr_tech: "ECHELLE,SLIT,OFFSET"}
is_echelle_slit_mapping = {kwd.dpr_tech: "ECHELLE,SLIT,MAPPING"}
is_echelle_ifu_mapping = {kwd.dpr_tech: "ECHELLE,IFU,MAPPING"}
is_echelle_slit = {kwd.dpr_tech: "ECHELLE,SLIT"}
is_echelle_ifu = {kwd.dpr_tech: "ECHELLE,IFU"}
is_echelle_ifu_offset = {kwd.dpr_tech: "ECHELLE,IFU,OFFSET"}
is_echelle_ifu_stare = {kwd.dpr_tech: "ECHELLE,IFU,STARE"}

is_afc_generic = {
    **is_calib,
    **is_echelle,
    kwd.dpr_type: "LAMP,AFC",
    kwd.tpl_id: [
        "SHOOT_gen_cal_AFC",
        "SHOOT_slt_AFC",
        "XSHOOTER_slt_AFC",
        "XSHOOTER_gen_cal_AFC",
    ],
}

# Pin dictionaries
is_opti3_pin_05 = {kwd.ins_opti3_name: "Pin_0.5"}
is_opti4_pin_05 = {kwd.ins_opti4_name: "Pin_0.5"}
is_opti5_pin_05 = {kwd.ins_opti5_name: "Pin_0.5"}

# Lamp dictionaries
is_vis_ff_lamp = {kwd.ins_lamp5_name: "VIS_FF_lamp"}
is_nir_ff_lamp = {kwd.ins_lamp6_name: "NIR_FF_lamp"}

# Combined mode dictionaries
is_echelle_slit_on = {**is_slit, **is_echelle_slit}
is_echelle_slit_off = {**is_slit, **is_image}
is_echelle_ifu_on = {**is_ifu, **is_echelle_ifu}
is_echelle_ifu_off = {**is_ifu, **is_image}

# Sky slit offset dictionaries
is_sky_slit_off = {**is_science, **is_sky, **is_echelle_slit_offset}
is_sky_slit_off_uvb = {**is_sky_slit_off, **is_uvb}
is_sky_slit_off_vis = {**is_sky_slit_off, **is_vis}
is_sky_slit_off_nir = {**is_sky_slit_off, **is_nir}

# wave and arcs
is_lamp_wave = {**is_calib, kwd.dpr_type: "LAMP,WAVE"}
is_lamp_afc = {**is_calib, kwd.dpr_type: "LAMP,AFC"}
is_arc_slit_on = {**is_lamp_wave, **is_echelle_slit_on, **is_thar_lamp}
is_arc_slit_nir_on = {
    **is_lamp_wave,
    **is_echelle_slit_on,
    **is_nir,
    kwd.ins_lamp2_name: "ArHgNeXe_Lamp",
}
is_arc_ifu_on = {**is_lamp_wave, **is_echelle_ifu_on, **is_thar_lamp}
is_arc_ifu_off = {**is_lamp_wave, **is_echelle_ifu_off, **is_thar_lamp}
is_arc_ifu_nir_on = {
    **is_lamp_wave,
    **is_echelle_ifu_on,
    **is_nir,
    kwd.ins_lamp2_name: "ArHgNeXe_Lamp",
}
is_arc_ifu_nir_off = {**is_lamp_wave, **is_echelle_ifu_off, **is_nir}

is_flex = {**is_lamp_afc, **is_echelle, kwd.ins_opti2_name: "SLOT"}
is_flex_slit = {
    **is_flex,
    kwd.tpl_id: ["SHOOT_slt_acq", "XSHOOTER_slt_acq", "XSHOOTER_slt_acq_rrm"],
}
is_flex_ifu = {
    **is_flex,
    kwd.tpl_id: ["SHOOT_ifu_acq", "XSHOOTER_ifu_acq", "XSHOOTER_ifu_acq_rrm"],
}

# ---------------- CLASSIFICATION RULES -----------------

# ---------------- RAW DATA -----------------------------

linearity_detmon_on_uvb_class = classification_rule(
    "ON_RAW", {**is_calib_uvb, **is_linearity_on}
)
linearity_detmon_off_uvb_class = classification_rule(
    "OFF_RAW", {**is_calib_uvb, **is_linearity_opt_off}
)
linearity_detmon_on_vis_class = classification_rule(
    "ON_RAW", {**is_calib_vis, **is_linearity_on}
)
linearity_detmon_off_vis_class = classification_rule(
    "OFF_RAW", {**is_calib_vis, **is_linearity_opt_off}
)
linearity_detmon_on_nir_class = classification_rule(
    "ON_RAW", {**is_calib_nir, **is_linearity_on, kwd.ins_lamp6_name: "NIR_FF_lamp"}
)
linearity_detmon_off_nir_class = classification_rule(
    "OFF_RAW", {**is_calib_nir, **is_image, **is_linearity_ir_off}
)
linearity_detmon_on_agc_class = classification_rule(
    "ON_RAW", {**is_calib_agc, kwd.dpr_type: "FLAT,LINEARITY,DETCHAR"}
)
linearity_detmon_off_agc_class = classification_rule(
    "OFF_RAW", {**is_calib_agc, **is_linearity_opt_off}
)

bias_uvb_class = classification_rule(
    "BIAS_UVB", {**is_calib_uvb, **is_image, **is_bias}
)
bias_vis_class = classification_rule(
    "BIAS_VIS", {**is_calib_vis, **is_image, **is_bias}
)
bias_agc_class = classification_rule(
    "BIAS_AGC", {**is_calib_agc, **is_image, **is_bias}
)

dark_uvb_class = classification_rule("DARK_UVB", rules.is_dark_uvb)
dark_vis_class = classification_rule("DARK_VIS", rules.is_dark_vis)
dark_nir_class = classification_rule(
    "DARK_NIR", {**is_calib_nir, **is_image, **is_dark}
)
dark_agc_class = classification_rule(
    "DARK_AGC", {**is_calib_agc, **is_image, **is_dark}
)

fmtchk_uvb_class = classification_rule(
    "FMTCHK_UVB",
    {**is_calib_uvb, **is_echelle_pinhole, **is_thar_lamp, **is_lamp_fmtchk},
)
fmtchk_vis_class = classification_rule(
    "FMTCHK_VIS",
    {**is_calib_vis, **is_echelle_pinhole, **is_thar_lamp, **is_lamp_fmtchk},
)
fmtchk_nir_on_class = classification_rule("FMTCHK_NIR_ON", rules.is_fmtchk_on_nir)
fmtchk_nir_off_class = classification_rule("FMTCHK_NIR_OFF", rules.is_fmtchk_off_nir)

orderdef_uvb_class = classification_rule(
    "ORDERDEF_D2_UVB",
    {
        **is_calib_uvb,
        **is_lamp_dorderdef,
        **is_echelle_pinhole,
        kwd.det_binx: 1,
        kwd.det_biny: 1,
        **is_opti3_pin_05,
    },
)
orderdef_uvb_qth_class = classification_rule(
    "ORDERDEF_QTH_UVB",
    {
        **is_calib_uvb,
        **is_lamp_qorderdef,
        **is_echelle_pinhole,
        kwd.det_binx: 1,
        kwd.det_biny: 1,
        **is_opti3_pin_05,
        kwd.ins_lamp4_name: "UVB_High",
    },
)
orderdef_vis_class = classification_rule(
    "ORDERDEF_VIS",
    {
        **is_calib_vis,
        **is_lamp_orderdef,
        **is_echelle_pinhole,
        **is_vis_ff_lamp,
        kwd.det_binx: 1,
        kwd.det_biny: 1,
        **is_opti4_pin_05,
    },
)
orderdef_nir_class = classification_rule(
    "ORDERDEF_NIR_ON",
    {
        **is_calib_nir,
        **is_lamp_orderdef,
        **is_echelle_pinhole,
        **is_nir_ff_lamp,
        **is_opti5_pin_05,
    },
)
orderdef_nir_off_class = classification_rule(
    "ORDERDEF_NIR_OFF",
    {**is_calib_nir, **is_image, **is_lamp_orderdef, **is_opti5_pin_05},
)

flat_d2_slit_uvb_class = classification_rule(
    "FLAT_D2_SLIT_UVB",
    {
        **is_calib_uvb,
        **is_echelle_slit_on,
        **is_lamp_dflat,
        kwd.ins_lamp3_name: "UVB_Low_D2",
    },
)
flat_qth_slit_uvb_class = classification_rule(
    "FLAT_QTH_SLIT_UVB",
    {
        **is_calib_uvb,
        **is_echelle_slit_on,
        **is_lamp_qflat,
        kwd.ins_lamp4_name: "UVB_High",
    },
)
flat_slit_vis_class = classification_rule(
    "FLAT_SLIT_VIS",
    {**is_calib_vis, **is_echelle_slit_on, **is_lamp_flat, **is_vis_ff_lamp},
)
flat_slit_nir_on_class = classification_rule(
    "FLAT_SLIT_NIR_ON",
    {**is_calib_nir, **is_echelle_slit_on, **is_lamp_flat, **is_nir_ff_lamp},
)
flat_slit_nir_off_class = classification_rule(
    "FLAT_SLIT_NIR_OFF", {**is_calib_nir, **is_echelle_slit_off, **is_lamp_flat}
)
flat_d2_ifu_uvb_class = classification_rule("FLAT_D2_IFU_UVB", rules.is_flat_d2_ifu_uvb)
flat_qth_ifu_uvb_class = classification_rule(
    "FLAT_QTH_IFU_UVB", rules.is_flat_qth_ifu_uvb
)

flat_ifu_vis_class = classification_rule("FLAT_IFU_VIS", rules.is_flat_ifu_vis)
flat_ifu_nir_on_class = classification_rule("FLAT_IFU_NIR_ON", rules.is_flat_ifu_nir_on)
flat_ifu_nir_off_class = classification_rule(
    "FLAT_IFU_NIR_OFF", rules.is_flat_ifu_nir_off
)
# HC flats are not used in the reduction, but I might still want to classify them
flat_qth_ifu_uvb_hc_class = classification_rule(
    "FLAT_QTH_IFU_UVB", rules.is_flat_qth_ifu_uvb_hc
)
flat_d2_ifu_uvb_hc_class = classification_rule(
    "FLAT_D2_IFU_UVB", rules.is_flat_d2_ifu_uvb_hc
)
flat_ifu_vis_hc_class = classification_rule("FLAT_IFU_VIS", rules.is_flat_ifu_vis_hc)
flat_ifu_nir_on_hc_class = classification_rule(
    "FLAT_IFU_NIR_ON", rules.is_flat_ifu_nir_on_hc
)
flat_ifu_nir_off_hc_class = classification_rule(
    "FLAT_IFU_NIR_OFF", rules.is_flat_ifu_nir_off_hc
)
skyflat_agc_class = classification_rule("SKYFLAT_AGC", {**is_calib_agc, **is_image})
wave_uvb_class = classification_rule(
    "WAVE_UVB", {**is_echelle_multi_pinhole, **is_lamp_wave, **is_uvb}
)
wave_vis_class = classification_rule(
    "WAVE_VIS", {**is_echelle_multi_pinhole, **is_lamp_wave, **is_vis}
)
wave_nir_on_class = classification_rule(
    "WAVE_NIR_ON", {**is_echelle_multi_pinhole, **is_lamp_wave, **is_nir}
)
wave_nir_off_class = classification_rule(
    "WAVE_NIR_OFF",
    {**is_lamp_wave, **is_image, **is_nir, kwd.ins_opti5_name: "Pin_row"},
)

arc_slit_uvb_class = classification_rule("ARC_SLIT_UVB", {**is_arc_slit_on, **is_uvb})
arc_slit_vis_class = classification_rule("ARC_SLIT_VIS", {**is_arc_slit_on, **is_vis})
arc_slit_nir_on_class = classification_rule("ARC_SLIT_NIR_ON", {**is_arc_slit_nir_on})
arc_slit_nir_off_class = classification_rule(
    "ARC_SLIT_NIR_OFF", rules.is_arc_slit_nir_off
)
arc_ifu_uvb_class = classification_rule("ARC_IFU_UVB", {**is_arc_ifu_on, **is_uvb})
arc_ifu_vis_class = classification_rule("ARC_IFU_VIS", {**is_arc_ifu_on, **is_vis})
arc_ifu_nir_on_class = classification_rule("ARC_IFU_NIR_ON", {**is_arc_ifu_nir_on})
arc_ifu_nir_off_class = classification_rule("ARC_IFU_NIR_OFF", {**is_arc_ifu_nir_off})

adc_slit_uvb = classification_rule("ADC_SLIT_UVB", rules.is_adc_slit_uvb)
adc_slit_vis = classification_rule("ADC_SLIT_VIS", rules.is_adc_slit_vis)
adc_slit_nir = classification_rule("ADC_SLIT_NIR", rules.is_adc_slit_nir)
afc_ifu_uvb = classification_rule("AFC_IFU_UVB", rules.is_afc_ifu_uvb)
afc_ifu_vis = classification_rule("AFC_IFU_VIS", rules.is_afc_ifu_vis)
afc_ifu_nir = classification_rule("AFC_IFU_NIR", rules.is_afc_ifu_nir)
afc_generic_uvb = classification_rule("AFC_GENERIC_UVB", {**is_afc_generic, **is_uvb})
afc_generic_vis = classification_rule("AFC_GENERIC_VIS", {**is_afc_generic, **is_vis})
afc_generic_nir = classification_rule("AFC_GENERIC_NIR", {**is_afc_generic, **is_nir})

flex_slit_uvb_class = classification_rule(
    "AFC_ATT_UVB", {**is_flex_slit, **is_opti3_pin_05, **is_uvb}
)
flex_slit_vis_class = classification_rule(
    "AFC_ATT_VIS", {**is_flex_slit, **is_opti4_pin_05, **is_vis}
)
flex_slit_nir_class = classification_rule(
    "AFC_ATT_NIR", {**is_flex_slit, **is_opti5_pin_05, **is_nir}
)
flex_ifu_uvb_class = classification_rule(
    "AFC_ATT_UVB", {**is_flex_ifu, **is_opti3_pin_05, **is_uvb}
)
flex_ifu_vis_class = classification_rule(
    "AFC_ATT_VIS", {**is_flex_ifu, **is_opti4_pin_05, **is_vis}
)
flex_ifu_nir_class = classification_rule(
    "AFC_ATT_NIR", {**is_flex_ifu, **is_opti5_pin_05, **is_nir}
)

is_std_flux_slit_off = {**is_calib, **is_echelle_slit_offset, kwd.dpr_type: "STD,FLUX"}
raw_std_slit_off_uvb_class = classification_rule(
    "STD_FLUX_SLIT_OFFSET_UVB", {**is_std_flux_slit_off, **is_uvb}
)
raw_std_slit_off_vis_class = classification_rule(
    "STD_FLUX_SLIT_OFFSET_VIS", {**is_std_flux_slit_off, **is_vis}
)
raw_std_slit_off_nir_class = classification_rule(
    "STD_FLUX_SLIT_OFFSET_NIR", {**is_std_flux_slit_off, **is_nir}
)

is_sky_flux_slit_off = {**is_calib, **is_echelle_slit_offset, kwd.dpr_type: "STD,SKY"}
sky_flux_slit_off_uvb_class = classification_rule(
    "SKY_SLIT_UVB", {**is_sky_flux_slit_off, **is_uvb}
)
sky_flux_slit_off_vis_class = classification_rule(
    "SKY_SLIT_VIS", {**is_sky_flux_slit_off, **is_vis}
)
sky_flux_slit_off_nir_class = classification_rule(
    "SKY_SLIT_NIR", {**is_sky_flux_slit_off, **is_nir}
)

is_std_flux_slit_nod = {**is_calib, **is_echelle_slit_nodding, kwd.dpr_type: "STD,FLUX"}
raw_std_slit_nod_uvb_class = classification_rule(
    "STD_FLUX_SLIT_NOD_UVB", {**is_std_flux_slit_nod, **is_uvb}
)
raw_std_slit_nod_vis_class = classification_rule(
    "STD_FLUX_SLIT_NOD_VIS", {**is_std_flux_slit_nod, **is_vis}
)
raw_std_slit_nod_nir_class = classification_rule(
    "STD_FLUX_SLIT_NOD_NIR", {**is_std_flux_slit_nod, **is_nir}
)

is_std_flux_slit_stare = {**is_calib, **is_echelle_slit_stare, kwd.dpr_type: "STD,FLUX"}
raw_std_slit_stare_uvb_class = classification_rule(
    "STD_FLUX_SLIT_STARE_UVB", {**is_std_flux_slit_stare, **is_uvb}
)
raw_std_slit_stare_vis_class = classification_rule(
    "STD_FLUX_SLIT_STARE_VIS", {**is_std_flux_slit_stare, **is_vis}
)
raw_std_slit_stare_nir_class = classification_rule(
    "STD_FLUX_SLIT_STARE_NIR", {**is_std_flux_slit_stare, **is_nir}
)

# IFU STANDARD STARS ARE TREATED AS SCIENCE TARGETS
is_std_flux_ifu_off = {**is_calib, **is_echelle_ifu_offset, kwd.dpr_type: "STD,FLUX"}
raw_std_ifu_off_uvb_class = classification_rule(
    "OBJECT_IFU_OFFSET_UVB", {**is_std_flux_ifu_off, **is_uvb}
)
raw_std_ifu_off_vis_class = classification_rule(
    "OBJECT_IFU_OFFSET_VIS", {**is_std_flux_ifu_off, **is_vis}
)
raw_std_ifu_off_nir_class = classification_rule(
    "OBJECT_IFU_OFFSET_NIR", {**is_std_flux_ifu_off, **is_nir}
)

is_sky_flux_ifu_off = {**is_calib, **is_echelle_ifu_offset, kwd.dpr_type: "STD,SKY"}
sky_flux_ifu_off_uvb_class = classification_rule(
    "SKY_IFU_UVB", {**is_sky_flux_ifu_off, **is_uvb}
)
sky_flux_ifu_off_vis_class = classification_rule(
    "SKY_IFU_VIS", {**is_sky_flux_ifu_off, **is_vis}
)
sky_flux_ifu_off_nir_class = classification_rule(
    "SKY_IFU_NIR", {**is_sky_flux_ifu_off, **is_nir}
)

is_std_flux_ifu_nod = {**is_calib, **is_echelle_ifu_nodding, kwd.dpr_type: "STD,FLUX"}
raw_std_ifu_nod_uvb_class = classification_rule(
    "OBJECT_IFU_NOD_UVB", {**is_std_flux_ifu_nod, **is_uvb}
)
raw_std_ifu_nod_vis_class = classification_rule(
    "OBJECT_IFU_NOD_VIS", {**is_std_flux_ifu_nod, **is_vis}
)
raw_std_ifu_nod_nir_class = classification_rule(
    "OBJECT_IFU_NOD_NIR", {**is_std_flux_ifu_nod, **is_nir}
)

is_std_flux_ifu_stare = {**is_calib, **is_echelle_ifu_stare, kwd.dpr_type: "STD,FLUX"}
raw_std_ifu_stare_uvb_class = classification_rule(
    "OBJECT_IFU_STARE_UVB", {**is_std_flux_ifu_stare, **is_uvb}
)
raw_std_ifu_stare_vis_class = classification_rule(
    "OBJECT_IFU_STARE_VIS", {**is_std_flux_ifu_stare, **is_vis}
)
raw_std_ifu_stare_nir_class = classification_rule(
    "OBJECT_IFU_STARE_NIR", {**is_std_flux_ifu_stare, **is_nir}
)

is_std_tell = {**is_calib, kwd.dpr_type: "STD,TELLURIC"}
is_std_tell_slit_stare = {
    **is_std_tell,
    **is_echelle_slit_stare,
    kwd.tpl_id: ["SHOOT_slt_cal_TelluricStdStare", "XSHOOTER_slt_cal_TelluricStdStare"],
}
is_std_tell_slit_offset = {
    **is_std_tell,
    **is_echelle_slit_offset,
    kwd.tpl_id: ["SHOOT_slt_cal_TelluricStdOff", "XSHOOTER_slt_cal_TelluricStdOff"],
}
is_std_tell_slit_nod = {
    **is_std_tell,
    **is_echelle_slit_nodding,
    kwd.tpl_id: ["SHOOT_slt_cal_TelluricStdNod", "XSHOOTER_slt_cal_TelluricStdNod"],
}

raw_tell_slit_stare_uvb_class = classification_rule(
    "STD_TELL_SLIT_STARE_UVB", {**is_std_tell_slit_stare, **is_uvb}
)
raw_tell_slit_stare_vis_class = classification_rule(
    "STD_TELL_SLIT_STARE_VIS", {**is_std_tell_slit_stare, **is_vis}
)
raw_tell_slit_stare_nir_class = classification_rule(
    "STD_TELL_SLIT_STARE_NIR", {**is_std_tell_slit_stare, **is_nir}
)
raw_tell_slit_offset_uvb_class = classification_rule(
    "STD_TELL_SLIT_OFFSET_UVB", {**is_std_tell_slit_offset, **is_uvb}
)
raw_tell_slit_offset_vis_class = classification_rule(
    "STD_TELL_SLIT_OFFSET_VIS", {**is_std_tell_slit_offset, **is_vis}
)
raw_tell_slit_offset_nir_class = classification_rule(
    "STD_TELL_SLIT_OFFSET_NIR", {**is_std_tell_slit_offset, **is_nir}
)
raw_tell_slit_nod_uvb_class = classification_rule(
    "STD_TELL_SLIT_NOD_UVB", {**is_std_tell_slit_nod, **is_uvb}
)
raw_tell_slit_nod_vis_class = classification_rule(
    "STD_TELL_SLIT_NOD_VIS", {**is_std_tell_slit_nod, **is_vis}
)
raw_tell_slit_nod_nir_class = classification_rule(
    "STD_TELL_SLIT_NOD_NIR", {**is_std_tell_slit_nod, **is_nir}
)

is_sci_slit_stare = {
    **is_science,
    kwd.dpr_type: ["OBJECT", "SKY"],
    kwd.dpr_tech: ["ECHELLE,SLIT,STARE", "ECHELLE,SLIT,MAPPING"],
}
is_sci_ifu_stare = {
    **is_science,
    kwd.dpr_type: "OBJECT",
    kwd.dpr_tech: ["ECHELLE,IFU,STARE", "ECHELLE,IFU,MAPPING"],
}
raw_sci_slit_stare_uvb_class = classification_rule(
    "OBJECT_SLIT_STARE_UVB", {**is_sci_slit_stare, **is_uvb}
)
raw_sci_slit_stare_vis_class = classification_rule(
    "OBJECT_SLIT_STARE_VIS", {**is_sci_slit_stare, **is_vis}
)
raw_sci_slit_stare_nir_class = classification_rule(
    "OBJECT_SLIT_STARE_NIR", {**is_sci_slit_stare, **is_nir}
)
raw_sci_ifu_stare_uvb_class = classification_rule(
    "OBJECT_IFU_STARE_UVB", {**is_sci_ifu_stare, **is_uvb}
)
raw_sci_ifu_stare_vis_class = classification_rule(
    "OBJECT_IFU_STARE_VIS", {**is_sci_ifu_stare, **is_vis}
)
raw_sci_ifu_stare_nir_class = classification_rule(
    "OBJECT_IFU_STARE_NIR", {**is_sci_ifu_stare, **is_nir}
)

is_sci_slit_offset = {**is_science, **is_echelle_slit_offset, kwd.dpr_type: "OBJECT"}
raw_sci_slit_off_uvb_class = classification_rule(
    "OBJECT_SLIT_OFFSET_UVB", {**is_sci_slit_offset, **is_uvb}
)
raw_sci_slit_off_vis_class = classification_rule(
    "OBJECT_SLIT_OFFSET_VIS", {**is_sci_slit_offset, **is_vis}
)
raw_sci_slit_off_nir_class = classification_rule(
    "OBJECT_SLIT_OFFSET_NIR", {**is_sci_slit_offset, **is_nir}
)
raw_sky_slit_off_uvb_class = classification_rule("SKY_SLIT_UVB", is_sky_slit_off_uvb)
raw_sky_slit_off_vis_class = classification_rule("SKY_SLIT_VIS", is_sky_slit_off_vis)
raw_sky_slit_off_nir_class = classification_rule("SKY_SLIT_NIR", is_sky_slit_off_nir)

is_sci_ifu_off = {**is_science, **is_echelle_ifu_offset, kwd.dpr_type: "OBJECT"}
raw_sci_ifu_offset_uvb_class = classification_rule(
    "OBJECT_IFU_OFFSET_UVB", {**is_sci_ifu_off, **is_uvb}
)
raw_sci_ifu_offset_vis_class = classification_rule(
    "OBJECT_IFU_OFFSET_VIS", {**is_sci_ifu_off, **is_vis}
)
raw_sci_ifu_offset_nir_class = classification_rule(
    "OBJECT_IFU_OFFSET_NIR", {**is_sci_ifu_off, **is_nir}
)

is_sky_ifu_offset = {**is_science, **is_echelle_ifu_offset, kwd.dpr_type: ["SKY"]}
is_sky_ifu_mapping = {**is_science, **is_echelle_ifu_mapping, kwd.dpr_type: ["SKY"]}
raw_sky_ifu_offset_uvb_class = classification_rule(
    "SKY_IFU_UVB", {**is_sky_ifu_offset, **is_uvb}
)
raw_sky_ifu_offset_vis_class = classification_rule(
    "SKY_IFU_VIS", {**is_sky_ifu_offset, **is_vis}
)
raw_sky_ifu_offset_nir_class = classification_rule(
    "SKY_IFU_NIR", {**is_sky_ifu_offset, **is_nir}
)
raw_sky_ifu_mapping_uvb_class = classification_rule(
    "SKY_IFU_MAP_UVB", {**is_sky_ifu_mapping, **is_uvb}
)
raw_sky_ifu_mapping_vis_class = classification_rule(
    "SKY_IFU_MAP_VIS", {**is_sky_ifu_mapping, **is_vis}
)
raw_sky_ifu_mapping_nir_class = classification_rule(
    "SKY_IFU_MAP_NIR", {**is_sky_ifu_mapping, **is_nir}
)

is_sci_slit_nod = {**is_science, **is_echelle_slit_nodding, kwd.dpr_type: "OBJECT"}
is_sci_ifu_nod = {**is_science, **is_echelle_ifu_nodding, kwd.dpr_type: "OBJECT"}
raw_sci_slit_nod_uvb_class = classification_rule(
    "OBJECT_SLIT_NOD_UVB", {**is_sci_slit_nod, **is_uvb}
)
raw_sci_slit_nod_vis_class = classification_rule(
    "OBJECT_SLIT_NOD_VIS", {**is_sci_slit_nod, **is_vis}
)
raw_sci_slit_nod_nir_class = classification_rule(
    "OBJECT_SLIT_NOD_NIR", {**is_sci_slit_nod, **is_nir}
)
raw_sci_ifu_nod_uvb_class = classification_rule(
    "SCI_IFU_NOD_UVB", {**is_sci_ifu_nod, **is_uvb}
)
raw_sci_ifu_nod_vis_class = classification_rule(
    "SCI_IFU_NOD_VIS", {**is_sci_ifu_nod, **is_vis}
)
raw_sci_ifu_nod_nir_class = classification_rule(
    "SCI_IFU_NOD_NIR", {**is_sci_ifu_nod, **is_nir}
)

# SONO ARRIVATO QUI
is_tell_ifu = {**is_calib, **is_ifu, kwd.dpr_type: "STD,TELLURIC"}
is_tell_ifu_nod = {**is_tell_ifu, **is_echelle_ifu_nodding}
is_tell_ifu_stare = {**is_tell_ifu, **is_echelle_ifu_stare}
is_tell_ifu_offset = {**is_tell_ifu, **is_echelle_ifu_offset}
raw_tell_ifu_nod_uvb_class = classification_rule(
    "SCI_IFU_NOD_UVB", {**is_tell_ifu_nod, **is_uvb}
)
raw_tell_ifu_nod_vis_class = classification_rule(
    "SCI_IFU_NOD_VIS", {**is_tell_ifu_nod, **is_vis}
)
raw_tell_ifu_nod_nir_class = classification_rule(
    "SCI_IFU_NOD_NIR", {**is_tell_ifu_nod, **is_nir}
)
raw_tell_ifu_stare_uvb_class = classification_rule(
    "SCI_IFU_STARE_UVB", {**is_tell_ifu_stare, **is_uvb}
)
raw_tell_ifu_stare_vis_class = classification_rule(
    "SCI_IFU_STARE_VIS", {**is_tell_ifu_stare, **is_vis}
)
raw_tell_ifu_stare_nir_class = classification_rule(
    "SCI_IFU_STARE_NIR", {**is_tell_ifu_stare, **is_nir}
)
raw_tell_ifu_offset_uvb_class = classification_rule(
    "SCI_IFU_OFFSET_UVB", {**is_tell_ifu_offset, **is_uvb}
)
raw_tell_ifu_offset_vis_class = classification_rule(
    "SCI_IFU_OFFSET_VIS", {**is_tell_ifu_offset, **is_vis}
)
raw_tell_ifu_offset_nir_class = classification_rule(
    "SCI_IFU_OFFSET_NIR", {**is_tell_ifu_offset, **is_nir}
)

acquisition_class = classification_rule(
    "ACQUISITION", {**is_xsh, kwd.dpr_catg: "ACQUISITION"}
)
SCI_SLIT_FLUX_IDP_UVB = classification_rule(
    "SPECTRUM_1D", {kwd.pro_catg: "SCI_SLIT_FLUX_IDP_UVB"}
)
SCI_SLIT_FLUX_IDP_VIS = classification_rule(
    "SPECTRUM_1D", {kwd.pro_catg: "SCI_SLIT_FLUX_IDP_VIS"}
)
SCI_SLIT_FLUX_IDP_NIR = classification_rule(
    "SPECTRUM_1D", {kwd.pro_catg: "SCI_SLIT_FLUX_IDP_NIR"}
)
SCI_SLIT_IDP_UVB = classification_rule("SPECTRUM_1D", {kwd.pro_catg: "SCI_SLIT_IDP_UVB"})
SCI_SLIT_IDP_VIS = classification_rule("SPECTRUM_1D", {kwd.pro_catg: "SCI_SLIT_IDP_VIS"})
SCI_SLIT_IDP_NIR = classification_rule("SPECTRUM_1D", {kwd.pro_catg: "SCI_SLIT_IDP_NIR"})

# ---------------- STATIC DATA PRODUCTS --------------------------------
bp_map_rp_uvb_class = classification_rule("BP_MAP_RP_UVB")
bp_map_rp_vis_class = classification_rule("BP_MAP_RP_VIS")
bp_map_rp_nir_class = classification_rule("BP_MAP_RP_NIR")
bp_map_rp_agc_class = classification_rule("BP_MAP_RP_AGC")

xsh_mod_cfg_tab_uvb_class = classification_rule("XSH_MOD_CFG_TAB_UVB")
xsh_mod_cfg_tab_vis_class = classification_rule("XSH_MOD_CFG_TAB_VIS")
xsh_mod_cfg_tab_nir_class = classification_rule("XSH_MOD_CFG_TAB_NIR")
xsh_mod_cfg_opt_2d_uvb_class = classification_rule("XSH_MOD_CFG_OPT_2D_UVB")
xsh_mod_cfg_opt_2d_vis_class = classification_rule("XSH_MOD_CFG_OPT_2D_VIS")
xsh_mod_cfg_opt_2d_nir_class = classification_rule("XSH_MOD_CFG_OPT_2D_NIR")
theo_tab_uvb_class = classification_rule("THEO_TAB_SING_UVB")
theo_tab_vis_class = classification_rule("THEO_TAB_SING_VIS")
theo_tab_nir_class = classification_rule("THEO_TAB_SING_NIR")

flux_std_catalog_uvb_class = classification_rule("FLUX_STD_CATALOG_UVB")
flux_std_catalog_vis_class = classification_rule("FLUX_STD_CATALOG_VIS")
flux_std_catalog_nir_class = classification_rule("FLUX_STD_CATALOG_NIR")

arc_line_list_afc_uvb_class = classification_rule("ARC_LINE_LIST_AFC_UVB")
arc_line_list_afc_vis_class = classification_rule("ARC_LINE_LIST_AFC_VIS")
arc_line_list_afc_nir_class = classification_rule("ARC_LINE_LIST_AFC_NIR")

atmos_ext_uvb_class = classification_rule("ATMOS_EXT_UVB")
atmos_ext_vis_class = classification_rule("ATMOS_EXT_VIS")
atmos_ext_nir_class = classification_rule("ATMOS_EXT_NIR")

tell_mod_cat_uvb_class = classification_rule("TELL_MOD_CAT_UVB")
tell_mod_cat_vis_class = classification_rule("TELL_MOD_CAT_VIS")
tell_mod_cat_nir_class = classification_rule("TELL_MOD_CAT_NIR")
tell_mask_uvb_class = classification_rule("TELL_MASK_UVB")
tell_mask_vis_class = classification_rule("TELL_MASK_VIS")
tell_mask_nir_class = classification_rule("TELL_MASK_NIR")

sky_line_list_uvb_class = classification_rule("SKY_LINE_LIST_UVB")
sky_line_list_vis_class = classification_rule("SKY_LINE_LIST_VIS")
sky_line_list_nir_class = classification_rule("SKY_LINE_LIST_NIR")

sky_slit_uvb_class = classification_rule("SKY_SLIT_UVB")
sky_slit_vis_class = classification_rule("SKY_SLIT_VIS")
sky_slit_nir_class = classification_rule("SKY_SLIT_NIR")

arc_line_list_uvb_class = classification_rule("ARC_LINE_LIST_UVB")
arc_line_list_vis_class = classification_rule("ARC_LINE_LIST_VIS")
arc_line_list_nir_class = classification_rule("ARC_LINE_LIST_NIR")

resp_fit_points_cat_uvb_class = classification_rule("RESP_FIT_POINTS_CAT_UVB")
resp_fit_points_cat_vis_class = classification_rule("RESP_FIT_POINTS_CAT_VIS")
resp_fit_points_cat_nir_class = classification_rule("RESP_FIT_POINTS_CAT_NIR")

spectral_format_tab_uvb_class = classification_rule("SPECTRAL_FORMAT_TAB_UVB")
spectral_format_tab_vis_class = classification_rule("SPECTRAL_FORMAT_TAB_VIS")
spectral_format_tab_nir_class = classification_rule("SPECTRAL_FORMAT_TAB_NIR")
spectral_format_tab_nir_jh_class = classification_rule("SPECTRAL_FORMAT_TAB_JH_NIR")

mresponse_merge1d_slit_uvb_class = classification_rule("RESPONSE_MERGE1D_SLIT_UVB")
mresponse_merge1d_slit_vis_class = classification_rule("RESPONSE_MERGE1D_SLIT_VIS")
mresponse_merge1d_slit_nir_class = classification_rule("RESPONSE_MERGE1D_SLIT_NIR")
reference_response_uvb_class = classification_rule(
    "RRESPONSE_MERGE1D_SLIT_UVB",
    {kwd.instrume: "XSHOOTER", kwd.pro_catg: "RRESPONSE_MERGE1D_SLIT_UVB"},
)
reference_response_vis_class = classification_rule(
    "RRESPONSE_MERGE1D_SLIT_VIS",
    {kwd.instrume: "XSHOOTER", kwd.pro_catg: "RRESPONSE_MERGE1D_SLIT_VIS"},
)
reference_response_nir_class = classification_rule(
    "RRESPONSE_MERGE1D_SLIT_NIR",
    {kwd.instrume: "XSHOOTER", kwd.pro_catg: "RRESPONSE_MERGE1D_SLIT_NIR"},
)

sky_map_nir_class = classification_rule("SKY_MAP_NIR")

ifu_cfg_tab_uvb_class = classification_rule("IFU_CFG_TAB_UVB")
ifu_cfg_tab_vis_class = classification_rule("IFU_CFG_TAB_VIS")
ifu_cfg_tab_nir_class = classification_rule("IFU_CFG_TAB_NIR")

arc_line_list_intmon_uvb_class = classification_rule("ARC_LINE_LIST_INTMON_UVB")
arc_line_list_intmon_vis_class = classification_rule("ARC_LINE_LIST_INTMON_VIS")
arc_line_list_intmon_nir_class = classification_rule("ARC_LINE_LIST_INTMON_NIR")

sky_sub_bkpts_uvb_class = classification_rule("SKY_SUB_BKPTS_UVB")
sky_sub_bkpts_vis_class = classification_rule("SKY_SUB_BKPTS_VIS")
sky_sub_bkpts_nir_class = classification_rule("SKY_SUB_BKPTS_NIR")

# ---------------- PIPELINE DATA PRODUCTS --------------------------------
bp_map_nl_uvb = classification_rule("BP_MAP_NL_UVB")
bp_map_nl_vis = classification_rule("BP_MAP_NL_VIS")
bp_map_nl_nir = classification_rule("BP_MAP_NL_NIR")

master_bias_uvb = classification_rule("MASTER_BIAS_UVB")
master_bias_vis = classification_rule("MASTER_BIAS_VIS")
master_bias_agc = classification_rule("MASTER_BIAS_AGC")

master_dark_uvb = classification_rule("MASTER_DARK_UVB")
master_dark_vis = classification_rule("MASTER_DARK_VIS")
master_dark_nir = classification_rule("MASTER_DARK_NIR")

order_tab_guess_uvb = classification_rule("ORDER_TAB_GUESS_UVB")
order_tab_guess_vis = classification_rule("ORDER_TAB_GUESS_VIS")
order_tab_guess_nir = classification_rule("ORDER_TAB_GUESS_NIR")

xsh_mod_cfg_opt_fmt_uvb = classification_rule("XSH_MOD_CFG_OPT_FMT_UVB")
xsh_mod_cfg_opt_fmt_vis = classification_rule("XSH_MOD_CFG_OPT_FMT_VIS")
xsh_mod_cfg_opt_fmt_nir = classification_rule("XSH_MOD_CFG_OPT_FMT_NIR")
wave_tab_2d_uvb_class = classification_rule("WAVE_TAB_2D_UVB")
wave_tab_2d_vis_class = classification_rule("WAVE_TAB_2D_VIS")
wave_tab_2d_nir_class = classification_rule("WAVE_TAB_2D_NIR")

WAVE_TAB_ARC_IFU_DOWN_UVB = classification_rule("WAVE_TAB_ARC_IFU_DOWN_UVB")
WAVE_TAB_ARC_IFU_CEN_UVB = classification_rule("WAVE_TAB_ARC_IFU_CEN_UVB")
WAVE_TAB_ARC_IFU_UP_UVB = classification_rule("WAVE_TAB_ARC_IFU_UP_UVB")
WAVE_TAB_ARC_IFU_DOWN_VIS = classification_rule("WAVE_TAB_ARC_IFU_DOWN_VIS")
WAVE_TAB_ARC_IFU_CEN_VIS = classification_rule("WAVE_TAB_ARC_IFU_CEN_VIS")
WAVE_TAB_ARC_IFU_UP_VIS = classification_rule("WAVE_TAB_ARC_IFU_UP_VIS")
WAVE_TAB_ARC_IFU_DOWN_NIR = classification_rule("WAVE_TAB_ARC_IFU_DOWN_NIR")
WAVE_TAB_ARC_IFU_CEN_NIR = classification_rule("WAVE_TAB_ARC_IFU_CEN_NIR")
WAVE_TAB_ARC_IFU_UP_NIR = classification_rule("WAVE_TAB_ARC_IFU_UP_NIR")

WAVE_TAB_AFC_UVB = classification_rule("WAVE_TAB_AFC_UVB")
WAVE_TAB_AFC_VIS = classification_rule("WAVE_TAB_AFC_VIS")
WAVE_TAB_AFC_NIR = classification_rule("WAVE_TAB_AFC_NIR")

WAVE_MAP_UVB = classification_rule("WAVE_MAP_UVB")
WAVE_MAP_VIS = classification_rule("WAVE_MAP_VIS")
WAVE_MAP_NIR = classification_rule("WAVE_MAP_NIR")

order_tab_centr_uvb = classification_rule("ORDER_TAB_CENTR_UVB")
order_tab_centr_vis = classification_rule("ORDER_TAB_CENTR_VIS")
order_tab_centr_nir = classification_rule("ORDER_TAB_CENTR_NIR")

order_tab_edges_slit_uvb = classification_rule("ORDER_TAB_EDGES_SLIT_UVB")
order_tab_edges_slit_vis = classification_rule("ORDER_TAB_EDGES_SLIT_VIS")
order_tab_edges_slit_nir = classification_rule("ORDER_TAB_EDGES_SLIT_NIR")
order_tab_edges_ifu_uvb = classification_rule("ORDER_TAB_EDGES_IFU_UVB")
order_tab_edges_ifu_vis = classification_rule("ORDER_TAB_EDGES_IFU_VIS")
order_tab_edges_ifu_nir = classification_rule("ORDER_TAB_EDGES_IFU_NIR")

master_flat_slit_uvb = classification_rule("MASTER_FLAT_SLIT_UVB")
master_flat_slit_vis = classification_rule("MASTER_FLAT_SLIT_VIS")
master_flat_slit_nir = classification_rule("MASTER_FLAT_SLIT_NIR")
master_flat_ifu_uvb = classification_rule("MASTER_FLAT_IFU_UVB")
master_flat_ifu_vis = classification_rule("MASTER_FLAT_IFU_VIS")
master_flat_ifu_nir = classification_rule("MASTER_FLAT_IFU_NIR")

xsh_mod_cfg_opt_2d_uvb = classification_rule("XSH_MOD_CFG_OPT_2D_UVB")
xsh_mod_cfg_opt_2d_vis = classification_rule("XSH_MOD_CFG_OPT_2D_VIS")
xsh_mod_cfg_opt_2d_nir = classification_rule("XSH_MOD_CFG_OPT_2D_NIR")

DISP_TAB_UVB = classification_rule("DISP_TAB_UVB")
DISP_TAB_VIS = classification_rule("DISP_TAB_VIS")
DISP_TAB_NIR = classification_rule("DISP_TAB_NIR")

order_tab_afc_slit_uvb = classification_rule("ORDER_TAB_AFC_SLIT_UVB")
order_tab_afc_slit_vis = classification_rule("ORDER_TAB_AFC_SLIT_VIS")
order_tab_afc_slit_nir = classification_rule("ORDER_TAB_AFC_SLIT_NIR")
order_tab_afc_ifu_uvb = classification_rule("ORDER_TAB_AFC_IFU_UVB")
order_tab_afc_ifu_vis = classification_rule("ORDER_TAB_AFC_IFU_VIS")
order_tab_afc_ifu_nir = classification_rule("ORDER_TAB_AFC_IFU_NIR")

xsh_mod_cfg_opt_afc_uvb = classification_rule("XSH_MOD_CFG_OPT_AFC_UVB")
xsh_mod_cfg_opt_afc_vis = classification_rule("XSH_MOD_CFG_OPT_AFC_VIS")
xsh_mod_cfg_opt_afc_nir = classification_rule("XSH_MOD_CFG_OPT_AFC_NIR")

disp_tab_afc_uvb = classification_rule("DISP_TAB_AFC_UVB")
disp_tab_afc_vis = classification_rule("DISP_TAB_AFC_VIS")
disp_tab_afc_nir = classification_rule("DISP_TAB_AFC_NIR")

response_merge1d_slit_uvb = classification_rule("RESPONSE_MERGE1D_SLIT_UVB")
response_merge1d_slit_vis = classification_rule("RESPONSE_MERGE1D_SLIT_VIS")
response_merge1d_slit_nir = classification_rule("RESPONSE_MERGE1D_SLIT_NIR")

# molecfit classifications
SCI_IDP_NIR = classification_rule("SCI_SLIT_FLUX_IDP_NIR")
SCI_IDP_VIS = classification_rule("SCI_SLIT_FLUX_IDP_VIS")
SCI_IDP_UVB = classification_rule("SCI_SLIT_FLUX_IDP_UVB")
TELL_IDP_NIR = classification_rule("TELL_SLIT_FLUX_IDP_NIR")
TELL_IDP_VIS = classification_rule("TELL_SLIT_FLUX_IDP_VIS")
TELL_IDP_UVB = classification_rule("TELL_SLIT_FLUX_IDP_UVB")
STD_IDP_NIR = classification_rule("STD_SLIT_FLUX_IDP_NIR")
STD_IDP_VIS = classification_rule("STD_SLIT_FLUX_IDP_VIS")
STD_IDP_UVB = classification_rule("STD_SLIT_FLUX_IDP_UVB")
BEST_FIT_PARAMETERS_STD_VIS = classification_rule("BEST_FIT_PARAMETERS_STD_VIS")
BEST_FIT_PARAMETERS_STD_NIR = classification_rule("BEST_FIT_PARAMETERS_STD_NIR")
BEST_FIT_MODEL_STD_VIS = classification_rule("BEST_FIT_MODEL_STD_VIS")
BEST_FIT_MODEL_STD_NIR = classification_rule("BEST_FIT_MODEL_STD_NIR")
BEST_FIT_PARAMETERS_SCI_VIS = classification_rule("BEST_FIT_PARAMETERS_SCI_VIS")
BEST_FIT_PARAMETERS_SCI_NIR = classification_rule("BEST_FIT_PARAMETERS_SCI_NIR")
BEST_FIT_MODEL_SCI_VIS = classification_rule("BEST_FIT_MODEL_SCI_VIS")
BEST_FIT_MODEL_SCI_NIR = classification_rule("BEST_FIT_MODEL_SCI_NIR")
GDAS_VIS = classification_rule("GDAS_VIS ")
GDAS_NIR = classification_rule("GDAS_NIR")
ATM_PARAMETERS_STD_VIS = classification_rule("ATM_PARAMETERS_STD_VIS")
ATM_PARAMETERS_STD_NIR = classification_rule("ATM_PARAMETERS_STD_NIR")
ATM_PARAMETERS_SCI_VIS = classification_rule("ATM_PARAMETERS_SCI_VIS")
ATM_PARAMETERS_SCI_NIR = classification_rule("ATM_PARAMETERS_SCI_NIR")
MOLECULES_NIR = classification_rule(
    "MODEL_MOLECULES_NIR", {kwd.pro_catg: "MODEL_MOLECULES_NIR", kwd.seq_arm: "NIR"}
)
MOLECULES_VIS = classification_rule(
    "MODEL_MOLECULES_VIS", {kwd.pro_catg: "MODEL_MOLECULES_VIS", kwd.seq_arm: "VIS"}
)
WAVE_INCLUDE_NIR = classification_rule(
    "WAVE_INCLUDE_NIR",
    {kwd.pro_catg: ["WAVE_INCLUDE", "WAVE_INCLUDE_NIR"], kwd.seq_arm: "NIR"},
)
WAVE_INCLUDE_VIS = classification_rule(
    "WAVE_INCLUDE_VIS",
    {kwd.pro_catg: ["WAVE_INCLUDE", "WAVE_INCLUDE_VIS"], kwd.seq_arm: "VIS"},
)
TELLURIC_CORR_VIS = classification_rule("TELLURIC_CORR_VIS")
TELLURIC_CORR_NIR = classification_rule("TELLURIC_CORR_NIR")
SPECTRUM_1D = classification_rule("SPECTRUM_1D")
