from edps import data_source, alternative_associated_inputs
from edps.generator.time_range import *

from .xshooter_classification import *
from .xshooter_task_functions import *

# Data sources
# Convention for Data sources Association rule levels:
# Each data source can have several match function which correspond to different
# quality levels for the selected data. The level is specified as a number that
# follows this convention:
#   level < 0: more restrictive than the calibration plan
#   level = 0 follows the calibration plan
#   level = 1 quality sufficient for QC1 certification
#   level = 2 probably still acceptable quality
#   level = 3 significant risk of bad quality results

keys_chip = [kwd.det_chip_id, kwd.det_chip1_id]
# UVB and VIS use DET.CHIP1.ID, but do not have DET.CHIP.ID. NIR uses DET.CHIP.ID, but does not have DET.CHIP1.ID

keys_bin = [kwd.det_binx, kwd.det_biny]
keys_detector = keys_chip + keys_bin + [kwd.det_read_clock]
keys_arm_read_clock = [kwd.seq_arm] + [kwd.det_read_clock]
keys_chip_arm = keys_chip + [kwd.seq_arm]
keys_opti2_name = [kwd.ins_opti2_name]
keys_chip_arm_opti2 = keys_chip_arm + [kwd.ins_opti2_id]
keys_opti3_name = [kwd.ins_opti3_name]
keys_opti4_name = [kwd.ins_opti4_name]
keys_opti5_name = [kwd.ins_opti5_name]
keys_optis_name = keys_opti2_name + keys_opti3_name + keys_opti4_name + keys_opti5_name
keys_dits = [kwd.det_dit, kwd.det_ndit]
setup = [kwd.det_read_clock] + keys_dits + keys_optis_name + [kwd.seq_arm]
setup_star_nod = [kwd.det_read_clock] + keys_optis_name + [kwd.seq_arm]

group_0 = keys_bin + [kwd.tpl_start] + keys_arm_read_clock
arm_tplstart = [kwd.tpl_start, kwd.seq_arm]

keys_exptime = [kwd.det_win1_uit1]
instrument = [kwd.instrume]
match_linearity = keys_bin + keys_chip
match_dark_opt = keys_bin + keys_chip + keys_arm_read_clock
match_dark_nir = [kwd.seq_arm, kwd.det_dit]

match_flat = instrument + keys_chip + keys_arm_read_clock + keys_bin + keys_opti2_name
match_flat_uvb = match_flat + keys_opti3_name
match_flat_vis = match_flat + keys_opti4_name
match_flat2 = instrument + keys_chip + keys_bin + [kwd.seq_arm] + keys_opti2_name
match_flat2_uvb = match_flat2 + keys_opti3_name
match_flat2_vis = match_flat2 + keys_opti4_name
match_flat_nir = instrument + keys_chip + [kwd.seq_arm] + keys_opti2_name + keys_opti5_name  # match_flat

instrument_and_arm = [kwd.instrume, kwd.seq_arm]

# Note: I cannot put raw_linearity UVB/VIS/NIR into a single data_soource because the have different constrain on
#      min_group_size.


# --- Datasources for raw types --------------------------------------------------------------------


raw_linearity_detmon_nir_on = (data_source("DETMON_NIR_ON")
                               .with_classification_rule(linearity_detmon_on_nir_class)
                               .with_grouping_keywords(group_0)
                               .with_min_group_size(8)
                               .with_setup_keywords([kwd.seq_arm])
                               .with_match_keywords([kwd.seq_arm], time_range=ONE_MONTH, level=0)
                               .with_match_keywords([kwd.seq_arm], time_range=UNLIMITED, level=3)
                               .build())

raw_linearity_detmon_nir_off = (data_source("DETMON_NIR_OFF")
                                .with_classification_rule(linearity_detmon_off_nir_class)
                                .with_grouping_keywords([kwd.tpl_start])
                                .with_min_group_size(8)
                                .with_match_keywords([kwd.tpl_start], level=0)
                                .build())

raw_linearity_detmon_uvbvis_on = (data_source("DETMON_OPTICAL_ON")
                                  .with_classification_rule(linearity_detmon_on_uvb_class)
                                  .with_classification_rule(linearity_detmon_on_vis_class)
                                  .with_classification_rule(linearity_detmon_on_agc_class)
                                  .with_grouping_keywords(group_0)
                                  .with_min_group_size(8)
                                  .with_setup_keywords(keys_arm_read_clock)
                                  .with_match_keywords(keys_arm_read_clock, time_range=ONE_MONTH,
                                                       level=0)
                                  .with_match_keywords(keys_arm_read_clock, time_range=UNLIMITED,
                                                       level=3)
                                  .build())

raw_linearity_detmon_uvbvis_off = (data_source("DETMON_OPTICAL_OFF")
                                   .with_classification_rule(linearity_detmon_off_uvb_class)
                                   .with_classification_rule(linearity_detmon_off_vis_class)
                                   .with_classification_rule(linearity_detmon_off_agc_class)
                                   .with_grouping_keywords(arm_tplstart)
                                   .with_match_keywords(arm_tplstart, level=0)
                                   .build())
# --

# --- Datasource for bias
raw_bias = (data_source("BIAS")
            .with_classification_rule(bias_uvb_class)
            .with_classification_rule(bias_vis_class)
            .with_classification_rule(bias_agc_class)
            .with_min_group_size(5)
            .with_grouping_keywords(keys_arm_read_clock + [kwd.tpl_start])
            .with_setup_keywords(keys_arm_read_clock)
            .build())

# --- Datasource for dark
raw_dark = (data_source("DARK")
            .with_classification_rule(dark_uvb_class)
            .with_classification_rule(dark_vis_class)
            .with_classification_rule(dark_nir_class)
            .with_classification_rule(dark_agc_class)
            .with_grouping_keywords(
    [kwd.tpl_start, kwd.ins_filt1_name] + keys_dits + keys_arm_read_clock + keys_exptime)
            .with_setup_keywords(keys_dits + keys_arm_read_clock + [kwd.exptime])
            .with_match_keywords(match_dark_nir, time_range=THREE_DAYS, level=0)
            .with_match_keywords(match_dark_nir, time_range=UNLIMITED, level=3)
            .build())

# --- Datasources for format check and order definition
raw_fmtchk = (data_source("FORMAT_CHECK")
              .with_classification_rule(fmtchk_uvb_class)
              .with_classification_rule(fmtchk_vis_class)
              .with_classification_rule(fmtchk_nir_on_class)
              .with_grouping_keywords([kwd.arcfile])
              .with_setup_keywords(keys_dits + keys_arm_read_clock)
              # Note: UVB/VIS use SEQ.ARM and DET.READ.CLOCK for setup. NIR uses SEQ.ARM, DET.NDIT and DET.DIT.
              .with_match_keywords(keys_chip_arm, time_range=TWO_DAYS, level=0)
              .with_match_keywords(keys_chip_arm, time_range=ONE_MONTH, level=2)
              .with_match_keywords(keys_chip_arm, time_range=UNLIMITED, level=3)
              .build())

raw_fmtchk_nir_off = (data_source()
                      .with_classification_rule(fmtchk_nir_off_class)
                      .with_grouping_keywords([kwd.arcfile])
                      .with_match_keywords(arm_tplstart, time_range=UNLIMITED, level=0)
                      .build())

raw_orderdef = (data_source("ORDER_DEFINITION")
                .with_classification_rule(orderdef_uvb_class)
                .with_classification_rule(orderdef_vis_class)
                .with_classification_rule(orderdef_nir_class)
                .with_grouping_keywords([kwd.arcfile])
                # UVB/VIS uses seq.arm and read.clock; nir uses seq.arm, ndit, and dit
                .with_setup_keywords(keys_dits + keys_arm_read_clock)
                .with_match_keywords(keys_chip_arm, time_range=TWO_DAYS, level=0)
                .with_match_keywords(keys_chip_arm, time_range=UNLIMITED, level=3)
                .build())

raw_orderdef_qth_uvb = (data_source()
                        .with_classification_rule(orderdef_uvb_qth_class)
                        .with_match_keywords([kwd.tpl_start], level=0)
                        .build())

raw_orderdef_nir_off = (data_source()
                        .with_classification_rule(orderdef_nir_off_class)
                        .with_grouping_keywords([kwd.arcfile])
                        .with_match_keywords(arm_tplstart, level=0)
                        .build())

# ---

# --- Datasources for flats
raw_skyflat_agc = (data_source()
                   .with_classification_rule(skyflat_agc_class)
                   .with_grouping_keywords([kwd.ins_filt1_name, kwd.seq_arm, kwd.tpl_start])
                   .with_setup_keywords([kwd.ins_filt1_name, kwd.seq_arm])
                   .build())

raw_flat = (data_source("FLAT_ON")
            .with_classification_rule(flat_d2_slit_uvb_class)
            .with_classification_rule(flat_d2_ifu_uvb_class)
            .with_classification_rule(flat_qth_slit_uvb_class)
            .with_classification_rule(flat_qth_ifu_uvb_class)
            .with_classification_rule(flat_slit_vis_class)
            .with_classification_rule(flat_ifu_vis_class)
            .with_classification_rule(flat_slit_nir_on_class)
            .with_classification_rule(flat_ifu_nir_on_class)
            .with_grouping_keywords(arm_tplstart)
            .with_min_group_size(2)
            .with_setup_keywords(setup)
            .build())

raw_flat_nir_off = (data_source("FLAT_OFF")
                    .with_classification_rule(flat_slit_nir_off_class)
                    .with_classification_rule(flat_ifu_nir_off_class)
                    .with_grouping_keywords(arm_tplstart)
                    .with_match_keywords(arm_tplstart, time_range=ONE_DAY, level=0)
                    .build())

raw_flat_hc = (data_source("FLAT_ON_HC")
               .with_classification_rule(flat_d2_ifu_uvb_hc_class)
               .with_classification_rule(flat_ifu_vis_hc_class)
               .with_classification_rule(flat_ifu_nir_on_hc_class)
               .with_grouping_keywords(arm_tplstart)
               .with_setup_keywords(setup)
               .build())

raw_flat_qth_hc = (data_source("FLAT_QTH_IFU_UVB")
                   .with_classification_rule(flat_qth_ifu_uvb_hc_class)
                   .with_grouping_keywords(arm_tplstart)
                   .with_match_keywords(arm_tplstart, time_range=ONE_DAY, level=0)
                   .build())

raw_flat_nir_off_hc = (data_source("FLAT_OFF_HC")
                       .with_classification_rule(flat_ifu_nir_off_hc_class)
                       .with_grouping_keywords(arm_tplstart)
                       .with_match_keywords(arm_tplstart, time_range=ONE_DAY, level=0)
                       .build())
# ---

# --- Datasources for tasks related to wavelength calibration
raw_wave = (data_source("WAVE")
            .with_classification_rule(wave_uvb_class)
            .with_classification_rule(wave_vis_class)
            .with_classification_rule(wave_nir_on_class)
            .with_grouping_keywords([kwd.arcfile])
            .with_setup_keywords(setup)
            .with_match_keywords(instrument_and_arm, time_range=TWO_DAYS, level=0)
            .with_match_keywords(instrument_and_arm, time_range=ONE_WEEK, level=1)
            .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
            .build())

raw_wave_nir_off = (data_source()
                    .with_classification_rule(wave_nir_off_class)
                    .with_grouping_keywords([kwd.arcfile])
                    .with_match_keywords([kwd.tpl_start, kwd.seq_arm, kwd.ins_opti2_name], time_range=UNLIMITED,
                                         level=0)
                    .build())

raw_arc = (data_source("ARC")
           .with_classification_rule(arc_slit_uvb_class)
           .with_classification_rule(arc_slit_vis_class)
           .with_classification_rule(arc_slit_nir_on_class)
           .with_classification_rule(arc_ifu_uvb_class)
           .with_classification_rule(arc_ifu_vis_class)
           .with_classification_rule(arc_ifu_nir_on_class)
           .with_grouping_keywords([kwd.arcfile])
           .with_match_keywords(instrument_and_arm, time_range=TWO_DAYS, level=0)
           .with_setup_keywords(setup)
           .build())

raw_arc_slit_nir_off = (data_source()
                        .with_classification_rule(arc_slit_nir_off_class)
                        .with_match_keywords([kwd.tpl_start], time_range=ONE_DAY, level=0)
                        .with_match_keywords([kwd.instrume], time_range=UNLIMITED, level=3)
                        .build())

raw_arc_ifu_nir_off = (data_source()
                       .with_classification_rule(arc_ifu_nir_off_class)
                       .with_grouping_keywords([kwd.tpl_start])
                       .with_match_keywords([kwd.instrume, kwd.seq_arm, kwd.tpl_start], time_range=ONE_DAY, level=0)
                       .with_match_keywords([kwd.instrume, kwd.seq_arm], time_range=ONE_DAY, level=3)
                       .build())
# ---

# --- Datasource for flexure correction
raw_flex = (data_source("FLEXURE")
            .with_classification_rule(flex_slit_uvb_class)
            .with_classification_rule(flex_slit_vis_class)
            .with_classification_rule(flex_slit_nir_class)
            .with_classification_rule(flex_ifu_uvb_class)
            .with_classification_rule(flex_ifu_vis_class)
            .with_classification_rule(flex_ifu_nir_class)
            .with_grouping_keywords([kwd.arcfile])
            .with_setup_keywords([kwd.obs_name, kwd.seq_arm])
            .build())

# --- Datasources for flux standard stars
raw_flux_sky_slit_off_uvb = (data_source()
                             .with_classification_rule(sky_flux_slit_off_uvb_class)
                             .with_grouping_keywords(arm_tplstart)
                             .with_match_keywords([kwd.instrume, kwd.night] + keys_chip, time_range=SAME_NIGHT,
                                                  level=-1)
                             .with_match_keywords([kwd.instrume] + keys_chip, time_range=ONE_WEEK, level=0)
                             .with_match_keywords([kwd.instrume] + keys_chip, time_range=TWO_WEEKS, level=1)
                             .with_match_keywords([kwd.instrume] + keys_chip, time_range=UNLIMITED, level=3)
                             .build())

raw_flux_sky_slit_off_vis = (data_source()
                             .with_classification_rule(sky_flux_slit_off_vis_class)
                             .with_grouping_keywords(arm_tplstart)
                             .with_match_keywords([kwd.instrume, kwd.night] + keys_chip, time_range=SAME_NIGHT,
                                                  level=-1)
                             .with_match_keywords([kwd.instrume] + keys_chip, time_range=ONE_WEEK, level=0)
                             .with_match_keywords([kwd.instrume] + keys_chip, time_range=TWO_WEEKS, level=1)
                             .with_match_keywords([kwd.instrume] + keys_chip, time_range=UNLIMITED, level=3)
                             .build())

raw_flux_sky_slit_off_nir = (data_source()
                             .with_classification_rule(sky_flux_slit_off_nir_class)
                             .with_grouping_keywords(arm_tplstart)
                             .with_match_keywords([kwd.instrume, kwd.night] + keys_chip, time_range=SAME_NIGHT,
                                                  level=-1)
                             .with_match_keywords([kwd.instrume] + keys_chip, time_range=ONE_WEEK, level=0)
                             .with_match_keywords([kwd.instrume] + keys_chip, time_range=TWO_WEEKS, level=1)
                             .with_match_keywords([kwd.instrume] + keys_chip, time_range=UNLIMITED, level=3)
                             .build())

raw_standard = (data_source("STANDARD_STAR")
                .with_classification_rule(raw_std_slit_off_uvb_class)
                .with_classification_rule(raw_std_slit_off_vis_class)
                .with_classification_rule(raw_std_slit_off_nir_class)
                .with_classification_rule(raw_std_slit_nod_uvb_class)
                .with_classification_rule(raw_std_slit_nod_vis_class)
                .with_classification_rule(raw_std_slit_nod_nir_class)
                .with_classification_rule(raw_std_slit_stare_uvb_class)
                .with_classification_rule(raw_std_slit_stare_vis_class)
                .with_classification_rule(raw_std_slit_stare_nir_class)
                .with_setup_keywords(setup)
                .with_grouping_keywords(arm_tplstart)
                .with_match_keywords(instrument_and_arm + [kwd.night], time_range=SAME_NIGHT, level=-1)
                .with_match_keywords(instrument_and_arm, time_range=ONE_WEEK, level=0)
                .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                .build())

raw_standard_slit_offset = (data_source("STANDARD_STAR_OFFSET")
                            .with_classification_rule(raw_std_slit_off_uvb_class)
                            .with_classification_rule(raw_std_slit_off_vis_class)
                            .with_classification_rule(raw_std_slit_off_nir_class)
                            .with_grouping_keywords(arm_tplstart)
                            .with_setup_keywords(setup)
                            .with_match_keywords(instrument_and_arm + [kwd.night], time_range=ONE_DAY, level=-1)
                            .with_match_keywords(instrument_and_arm, time_range=ONE_WEEK, level=0)
                            .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                            .build())

raw_standard_slit_nod = (data_source("STANDARD_STAR_NODDING")
                         .with_classification_rule(raw_std_slit_nod_uvb_class)
                         .with_classification_rule(raw_std_slit_nod_vis_class)
                         .with_classification_rule(raw_std_slit_nod_nir_class)
                         .with_grouping_keywords(arm_tplstart)
                         .with_min_group_size(2)
                         .with_setup_keywords(setup_star_nod)
                         .with_match_keywords(instrument_and_arm + [kwd.night], time_range=ONE_DAY, level=-1)
                         .with_match_keywords(instrument_and_arm, time_range=ONE_WEEK, level=0)
                         .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                         .build())

raw_standard_slit_stare = (data_source("STANDARD_STAR_STARE")
                           .with_classification_rule(raw_std_slit_stare_uvb_class)
                           .with_classification_rule(raw_std_slit_stare_vis_class)
                           .with_classification_rule(raw_std_slit_stare_nir_class)
                           .with_grouping_keywords(arm_tplstart)
                           .with_setup_keywords(setup)
                           .with_match_keywords(instrument_and_arm + [kwd.night], time_range=ONE_DAY, level=-1)
                           .with_match_keywords(instrument_and_arm, time_range=ONE_WEEK, level=0)
                           .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                           .build())

# Define a datasource where the raw flux  standards data are grouped in the way that the spectra should be combined
# later on. In other words, the spectra of each group defined here will later be co-added.
raw_standard_to_combine = (data_source("STANDARD_TO_COMBINE")
                           .with_classification_rule(raw_std_slit_stare_uvb_class)
                           .with_classification_rule(raw_std_slit_stare_vis_class)
                           .with_classification_rule(raw_std_slit_stare_nir_class)
                           .with_classification_rule(raw_std_slit_nod_uvb_class)
                           .with_classification_rule(raw_std_slit_nod_vis_class)
                           .with_classification_rule(raw_std_slit_nod_nir_class)
                           .with_classification_rule(raw_std_slit_off_uvb_class)
                           .with_classification_rule(raw_std_slit_off_vis_class)
                           .with_classification_rule(raw_std_slit_off_nir_class)
                           .with_grouping_keywords([kwd.seq_arm, kwd.obs_targ_name])
                           .with_setup_keywords(setup)
                           .build())
# ---

# --- Data sources of science exposures. Telluric standards are considered as science observations. -
raw_science_slit_nod = (data_source("SCIENCE_SLIT_NODDING")
                        .with_classification_rule(raw_sci_slit_nod_uvb_class)
                        .with_classification_rule(raw_sci_slit_nod_vis_class)
                        .with_classification_rule(raw_sci_slit_nod_nir_class)
                        .with_grouping_keywords(arm_tplstart)
                        .with_setup_keywords(setup)
                        .with_match_keywords([kwd.arcfile])
                        .build())

raw_science_slit_stare = (data_source("SCIENCE_SLIT_STARE")
                          .with_classification_rule(raw_sci_slit_stare_uvb_class)
                          .with_classification_rule(raw_sci_slit_stare_vis_class)
                          .with_classification_rule(raw_sci_slit_stare_nir_class)
                          .with_grouping_keywords(arm_tplstart)
                          .with_setup_keywords(setup)
                          .with_match_keywords([kwd.arcfile])
                          .build())

raw_science_slit_offset = (data_source("SCIENCE_SLIT_OFFSET")
                           .with_classification_rule(raw_sci_slit_off_uvb_class)
                           .with_classification_rule(raw_sci_slit_off_vis_class)
                           .with_classification_rule(raw_sci_slit_off_nir_class)
                           .with_grouping_keywords(arm_tplstart)
                           .with_match_keywords([kwd.arcfile])
                           .with_setup_keywords(setup)
                           .build())

raw_science_ifu_stare = (data_source("SCIENCE_IFU_STARE")
                         .with_classification_rule(raw_sci_ifu_stare_uvb_class)
                         .with_classification_rule(raw_sci_ifu_stare_vis_class)
                         .with_classification_rule(raw_sci_ifu_stare_nir_class)
                         .with_classification_rule(raw_std_ifu_stare_uvb_class)
                         .with_classification_rule(raw_std_ifu_stare_vis_class)
                         .with_classification_rule(raw_std_ifu_stare_nir_class)
                         .with_setup_keywords(setup)
                         .with_grouping_keywords(arm_tplstart)
                         .build())

raw_science_ifu_nod = (data_source("SCIENCE_IFU_NODDING")
                       .with_classification_rule(raw_sci_ifu_nod_uvb_class)
                       .with_classification_rule(raw_sci_ifu_nod_vis_class)
                       .with_classification_rule(raw_sci_ifu_nod_nir_class)
                       .with_classification_rule(raw_std_ifu_nod_uvb_class)
                       .with_classification_rule(raw_std_ifu_nod_vis_class)
                       .with_classification_rule(raw_std_ifu_nod_nir_class)
                       .with_setup_keywords(setup)
                       .with_grouping_keywords(arm_tplstart)
                       .build())

raw_science_ifu_offset = (data_source("SCIENCE_IFU_OFFSET")
                          .with_classification_rule(raw_sci_ifu_offset_uvb_class)
                          .with_classification_rule(raw_sci_ifu_offset_vis_class)
                          .with_classification_rule(raw_sci_ifu_offset_nir_class)
                          .with_classification_rule(raw_std_ifu_off_uvb_class)
                          .with_classification_rule(raw_std_ifu_off_vis_class)
                          .with_classification_rule(raw_std_ifu_off_nir_class)
                          .with_setup_keywords(setup)
                          .with_grouping_keywords(arm_tplstart)
                          .build())

raw_science = (data_source("SCIENCE_GENERAL")
               .with_classification_rule(raw_sci_slit_stare_uvb_class)
               .with_classification_rule(raw_sci_slit_nod_uvb_class)
               .with_classification_rule(raw_sci_slit_off_uvb_class)
               .with_classification_rule(raw_sci_slit_stare_vis_class)
               .with_classification_rule(raw_sci_slit_nod_vis_class)
               .with_classification_rule(raw_sci_slit_off_vis_class)
               .with_classification_rule(raw_sci_slit_stare_nir_class)
               .with_classification_rule(raw_sci_slit_nod_nir_class)
               .with_classification_rule(raw_sci_slit_off_nir_class)
               .with_classification_rule(raw_tell_slit_stare_uvb_class)
               .with_classification_rule(raw_tell_slit_stare_vis_class)
               .with_classification_rule(raw_tell_slit_stare_nir_class)
               .with_classification_rule(raw_tell_slit_nod_uvb_class)
               .with_classification_rule(raw_tell_slit_nod_vis_class)
               .with_classification_rule(raw_tell_slit_nod_nir_class)
               .with_classification_rule(raw_sci_ifu_offset_uvb_class)
               .with_classification_rule(raw_sci_ifu_offset_vis_class)
               .with_classification_rule(raw_sci_ifu_offset_nir_class)
               .with_classification_rule(raw_std_ifu_off_uvb_class)
               .with_classification_rule(raw_std_ifu_off_vis_class)
               .with_classification_rule(raw_std_ifu_off_nir_class)
               .with_classification_rule(raw_sci_ifu_stare_uvb_class)
               .with_classification_rule(raw_sci_ifu_stare_vis_class)
               .with_classification_rule(raw_sci_ifu_stare_nir_class)
               .with_classification_rule(raw_std_ifu_stare_uvb_class)
               .with_classification_rule(raw_std_ifu_stare_vis_class)
               .with_classification_rule(raw_std_ifu_stare_nir_class)
               .with_classification_rule(raw_tell_ifu_stare_uvb_class)
               .with_classification_rule(raw_tell_ifu_stare_vis_class)
               .with_classification_rule(raw_tell_ifu_stare_nir_class)
               .with_classification_rule(raw_tell_ifu_offset_uvb_class)
               .with_classification_rule(raw_tell_ifu_offset_vis_class)
               .with_classification_rule(raw_tell_ifu_offset_nir_class)
               .with_classification_rule(raw_tell_ifu_nod_uvb_class)
               .with_classification_rule(raw_tell_ifu_nod_vis_class)
               .with_classification_rule(raw_tell_ifu_nod_nir_class)
               .with_setup_keywords(setup)
               .with_grouping_keywords(arm_tplstart)
               .with_match_keywords([kwd.instrume, kwd.seq_arm, kwd.night], time_range=ONE_DAY, level=-1)
               .with_match_keywords([kwd.instrume, kwd.seq_arm], time_range=ONE_WEEK, level=0)
               .with_match_keywords([kwd.instrume, kwd.seq_arm], time_range=UNLIMITED, level=3)
               .build())

# Define a datasource where the raw science data are grouped in the way that the spectra should be combined
# later on. In other words, the spectra of each group defined here will later be co-added.
raw_science_to_combine = (data_source("SCIENCE_TO_COMBINE")
                          .with_classification_rule(raw_sci_slit_stare_uvb_class)
                          .with_classification_rule(raw_sci_slit_nod_uvb_class)
                          .with_classification_rule(raw_sci_slit_off_uvb_class)
                          .with_classification_rule(raw_sci_slit_stare_vis_class)
                          .with_classification_rule(raw_sci_slit_nod_vis_class)
                          .with_classification_rule(raw_sci_slit_off_vis_class)
                          .with_classification_rule(raw_sci_slit_stare_nir_class)
                          .with_classification_rule(raw_sci_slit_nod_nir_class)
                          .with_classification_rule(raw_sci_slit_off_nir_class)
                          .with_grouping_keywords([kwd.seq_arm, kwd.obs_targ_name])
                          .with_setup_keywords(setup)
                          .build())

raw_telluric_slit_nod = (data_source("TELLURIC_SLIT_NODDING")
                         .with_classification_rule(raw_tell_slit_nod_uvb_class)
                         .with_classification_rule(raw_tell_slit_nod_vis_class)
                         .with_classification_rule(raw_tell_slit_nod_nir_class)
                         .with_grouping_keywords(arm_tplstart)
                         .with_match_keywords([kwd.arcfile])
                         .with_setup_keywords(setup_star_nod)
                         .build())

raw_telluric_slit_stare = (data_source("TELLURIC_SLIT_STARE")
                           .with_classification_rule(raw_tell_slit_stare_uvb_class)
                           .with_classification_rule(raw_tell_slit_stare_vis_class)
                           .with_classification_rule(raw_tell_slit_stare_nir_class)
                           .with_grouping_keywords(arm_tplstart)
                           .with_setup_keywords(setup)
                           .with_match_keywords([kwd.arcfile])
                           .build())

raw_telluric_slit_offset = (data_source("TELLURIC_SLIT_OFFSET")
                            .with_classification_rule(raw_tell_slit_offset_uvb_class)
                            .with_classification_rule(raw_tell_slit_offset_vis_class)
                            .with_classification_rule(raw_tell_slit_offset_nir_class)
                            .with_grouping_keywords(arm_tplstart)
                            .with_match_keywords([kwd.arcfile])
                            .with_setup_keywords(setup)
                            .build())

# Define a datasource where the raw telluric standards data are grouped in the way that the spectra should be combined
# later on. In other words, the spectra of each group defined here will later be co-added.
raw_telluric_to_combine = (data_source("TELLURIC_TO_COMBINE")
                           .with_classification_rule(raw_tell_slit_stare_uvb_class)
                           .with_classification_rule(raw_tell_slit_stare_vis_class)
                           .with_classification_rule(raw_tell_slit_stare_nir_class)
                           .with_classification_rule(raw_tell_slit_nod_uvb_class)
                           .with_classification_rule(raw_tell_slit_nod_vis_class)
                           .with_classification_rule(raw_tell_slit_nod_nir_class)
                           .with_classification_rule(raw_tell_slit_offset_uvb_class)
                           .with_classification_rule(raw_tell_slit_offset_vis_class)
                           .with_classification_rule(raw_tell_slit_offset_nir_class)
                           .with_grouping_keywords([kwd.seq_arm, kwd.obs_targ_name])
                           .with_setup_keywords(setup)
                           .build())

raw_sky = (data_source("RAW_SKY")
           .with_classification_rule(raw_sky_slit_off_uvb_class)
           .with_classification_rule(raw_sky_slit_off_vis_class)
           .with_classification_rule(raw_sky_slit_off_nir_class)
           .with_classification_rule(raw_sky_ifu_offset_uvb_class)
           .with_classification_rule(raw_sky_ifu_offset_vis_class)
           .with_classification_rule(raw_sky_ifu_offset_nir_class)
           .with_classification_rule(sky_flux_ifu_off_uvb_class)
           .with_classification_rule(sky_flux_ifu_off_vis_class)
           .with_classification_rule(sky_flux_ifu_off_nir_class)
           .with_setup_keywords(setup)
           .with_grouping_keywords(arm_tplstart + [kwd.dpr_tech])
           .with_match_keywords([kwd.seq_arm, kwd.dpr_tech], time_range=SAME_NIGHT, level=0)
           .build())

# Acquisition images are associated to science tasks to be delivered by CalSelector, but not used in the process.
raw_acquisition = (data_source("ACQUISITION_IMAGE")
                   .with_classification_rule(acquisition_class)
                   .with_grouping_keywords([kwd.seq_arm, kwd.obs_id])
                   .with_match_keywords([kwd.obs_id], time_range=IN_THE_PAST, level=0)
                   .build())
# --------------------------------------------------------------------------------------------------

# --- Datasources for static calibrations

# --- Master and reference response function
mresponse_merge1d_slit_vis = (data_source()
                              .with_classification_rule(mresponse_merge1d_slit_vis_class)
                              .with_match_keywords(keys_chip, time_range=ONE_DAY, level=0)
                              .with_match_keywords(keys_chip, time_range=TWO_WEEKS, level=1)
                              .with_match_keywords(keys_chip, time_range=ONE_MONTH, level=2)
                              .with_match_keywords(keys_chip, time_range=UNLIMITED, level=3)
                              .build())

reference_response = (data_source("REFERENCE_RESPONSE")
                      .with_classification_rule(reference_response_uvb_class)
                      .with_classification_rule(reference_response_vis_class)
                      .with_classification_rule(reference_response_nir_class)
                      .with_grouping_keywords([kwd.arcfile])
                      .with_match_keywords(instrument_and_arm, level=0)
                      .build())
resp_fit_points_cat = (data_source("RESPONSE_FIT_POINTS")
                       .with_classification_rule(resp_fit_points_cat_uvb_class)
                       .with_classification_rule(resp_fit_points_cat_vis_class)
                       .with_classification_rule(resp_fit_points_cat_nir_class)
                       .with_grouping_keywords([kwd.arcfile])
                       .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                       .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                       .build())

mresponse_merge1d_slit = (data_source("MASTER_RESPONSE")
                          .with_classification_rule(mresponse_merge1d_slit_uvb_class)
                          .with_classification_rule(mresponse_merge1d_slit_vis_class)
                          .with_classification_rule(mresponse_merge1d_slit_nir_class)
                          .with_grouping_keywords([kwd.arcfile])
                          .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                          .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                          .build())
# ---

# --- Flux standard star catalogs
flux_std_catalog_uvb = (data_source()
                        .with_classification_rule(flux_std_catalog_uvb_class)
                        .with_grouping_keywords([kwd.arcfile])
                        .with_match_keywords(instrument, time_range=IN_THE_PAST, level=0)
                        .with_match_keywords(instrument, time_range=UNLIMITED, level=3)
                        .build())

flux_std_catalog_vis = (data_source()
                        .with_classification_rule(flux_std_catalog_vis_class)
                        .with_grouping_keywords([kwd.arcfile])
                        .with_match_keywords(instrument, time_range=IN_THE_PAST, level=0)
                        .with_match_keywords(instrument, time_range=UNLIMITED, level=3)
                        .build())

flux_std_catalog_nir = (data_source()
                        .with_classification_rule(flux_std_catalog_nir_class)
                        .with_grouping_keywords([kwd.arcfile])
                        .with_match_keywords(instrument, time_range=IN_THE_PAST, level=0)
                        .with_match_keywords(instrument, time_range=UNLIMITED, level=3)
                        .build())
# ---
# --- Bad pixel maps

bp_map_rp_uvb = (data_source()
                 .with_classification_rule(bp_map_rp_uvb_class)
                 .with_match_keywords(keys_bin + keys_chip + [kwd.seq_arm], time_range=IN_THE_PAST, level=0)
                 .with_match_keywords(keys_bin + keys_chip + [kwd.seq_arm], time_range=UNLIMITED, level=3)
                 .build())

bp_map_rp_vis = (data_source()
                 .with_classification_rule(bp_map_rp_vis_class)
                 .with_match_keywords(keys_bin + keys_chip + [kwd.seq_arm], time_range=IN_THE_PAST, level=0)
                 .with_match_keywords(keys_bin + keys_chip + [kwd.seq_arm], time_range=UNLIMITED, level=3)
                 .build())

bp_map_rp_nir = (data_source()
                 .with_classification_rule(bp_map_rp_nir_class)
                 .with_match_keywords([kwd.instrume], time_range=IN_THE_PAST, level=0)
                 .with_match_keywords([kwd.instrume], time_range=UNLIMITED, level=3)
                 .build())

bp_map_rp_agc = (data_source()
                 .with_classification_rule(bp_map_rp_agc_class)
                 .with_match_keywords([kwd.instrume], time_range=IN_THE_PAST, level=0)
                 .with_match_keywords([kwd.instrume], time_range=UNLIMITED, level=3)
                 .build())

bad_pixel_map_rp_static_calibrations = (alternative_associated_inputs()
                                        .with_associated_input(bp_map_rp_uvb, min_ret=0, condition=is_UVB)
                                        .with_associated_input(bp_map_rp_vis, min_ret=0, condition=is_VIS)
                                        .with_associated_input(bp_map_rp_nir, min_ret=0, condition=is_NIR)
                                        .with_associated_input(bp_map_rp_agc, min_ret=0, condition=is_AGC))
# ---

# --- Configuration tables
xsh_mod_cfg_tab = (data_source("MODEL_CONFIGURATION_TABLE")
                   .with_classification_rule(xsh_mod_cfg_tab_uvb_class)
                   .with_classification_rule(xsh_mod_cfg_tab_vis_class)
                   .with_classification_rule(xsh_mod_cfg_tab_nir_class)
                   .with_grouping_keywords([kwd.arcfile])
                   .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                   .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                   .build())

theo_tab_sing = (data_source("THEO_TABLE")
                 .with_classification_rule(theo_tab_uvb_class)
                 .with_classification_rule(theo_tab_vis_class)
                 .with_classification_rule(theo_tab_nir_class)
                 .with_grouping_keywords([kwd.arcfile])
                 .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                 .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                 .build())

ifu_cfg_tab = (data_source("IFU_CFG")
               .with_classification_rule(ifu_cfg_tab_uvb_class)
               .with_classification_rule(ifu_cfg_tab_vis_class)
               .with_classification_rule(ifu_cfg_tab_nir_class)
               .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
               .build())

# --- Arc line catalogs
arc_line_list_afc = (data_source("ARC_LINES_AFC")
                     .with_classification_rule(arc_line_list_afc_uvb_class)
                     .with_classification_rule(arc_line_list_afc_vis_class)
                     .with_classification_rule(arc_line_list_afc_nir_class)
                     .with_grouping_keywords([kwd.arcfile])
                     .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                     .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                     .build())

arc_line_list = (data_source("ARC_LINES")
                 .with_classification_rule(arc_line_list_uvb_class)
                 .with_classification_rule(arc_line_list_vis_class)
                 .with_classification_rule(arc_line_list_nir_class)
                 .with_grouping_keywords([kwd.arcfile])
                 .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                 .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                 .build())

arc_line_list_intmon = (data_source("ARC_LINES_INTMON")
                        .with_classification_rule(arc_line_list_intmon_uvb_class)
                        .with_classification_rule(arc_line_list_intmon_vis_class)
                        .with_classification_rule(arc_line_list_intmon_nir_class)
                        .with_grouping_keywords([kwd.arcfile])
                        .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                        .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                        .build())

sky_line_list = (data_source("SKY_LINES")
                 .with_classification_rule(sky_line_list_uvb_class)
                 .with_classification_rule(sky_line_list_vis_class)
                 .with_classification_rule(sky_line_list_nir_class)
                 .with_grouping_keywords([kwd.arcfile])
                 .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                 .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                 .build())
# ---
# --- Telluric-correction related static calibrations
atmos_ext = (data_source("ATMOSPHERIC_EXTINCTION")
             .with_classification_rule(atmos_ext_uvb_class)
             .with_classification_rule(atmos_ext_vis_class)
             .with_classification_rule(atmos_ext_nir_class)
             .with_grouping_keywords([kwd.arcfile])
             .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
             .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
             .build())

tell_mod_cat = (data_source("TELLURIC_MODEL")
                .with_classification_rule(tell_mod_cat_vis_class)
                .with_classification_rule(tell_mod_cat_nir_class)
                .with_grouping_keywords([kwd.arcfile])
                .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                .build())

tell_mask = (data_source("TELLURIC_MASK")
             .with_classification_rule(tell_mask_uvb_class)
             .with_classification_rule(tell_mask_vis_class)
             .with_classification_rule(tell_mask_nir_class)
             .with_grouping_keywords([kwd.arcfile])
             .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
             .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
             .build())

molecules = (data_source()
             .with_classification_rule(MOLECULES_VIS)
             .with_classification_rule(MOLECULES_NIR)
             .with_match_keywords(instrument_and_arm, time_range=UNLIMITED)
             .build())

wave_include = (data_source()
                .with_classification_rule(WAVE_INCLUDE_VIS)
                .with_classification_rule(WAVE_INCLUDE_NIR)
                .with_match_keywords(instrument_and_arm, time_range=UNLIMITED)
                .build())

# ---


sky_slit = (data_source("SKY_SLIT")
            .with_classification_rule(sky_slit_uvb_class)
            .with_classification_rule(sky_slit_vis_class)
            .with_classification_rule(sky_slit_nir_class)
            .with_grouping_keywords([kwd.arcfile])
            .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
            .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
            .build())

sky_map_nir = (data_source()
               .with_classification_rule(sky_map_nir_class)
               .with_grouping_keywords([kwd.arcfile])
               .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
               .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
               .build())

# --- Spectra format static calibrations ---
spectral_format_tab_uvb = (data_source()
                           .with_classification_rule(spectral_format_tab_uvb_class)
                           .with_grouping_keywords([kwd.arcfile])
                           .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                           .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                           .build())

spectral_format_tab_vis = (data_source()
                           .with_classification_rule(spectral_format_tab_vis_class)
                           .with_grouping_keywords([kwd.arcfile])
                           .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                           .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                           .build())

spectral_format_tab_nir = (data_source()
                           .with_classification_rule(spectral_format_tab_nir_class)
                           .with_grouping_keywords([kwd.arcfile])
                           .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                           .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                           .build())

spectral_format_tab_nir_jh = (data_source()
                              .with_classification_rule(spectral_format_tab_nir_jh_class)
                              .with_grouping_keywords([kwd.arcfile])
                              .with_match_keywords(instrument_and_arm, time_range=IN_THE_PAST, level=0)
                              .with_match_keywords(instrument_and_arm, time_range=UNLIMITED, level=3)
                              .build())

alternatives_spectral_format = (alternative_associated_inputs()
                                .with_associated_input(spectral_format_tab_uvb, condition=is_UVB)
                                .with_associated_input(spectral_format_tab_vis, condition=is_VIS)
                                .with_associated_input(spectral_format_tab_nir, condition=is_NIR_normal)
                                .with_associated_input(spectral_format_tab_nir_jh, condition=is_NIR_JH))
# ---
