from edps import subworkflow, task, match_rules

from .xshooter_datasources import *
from .xshooter_task_functions import *


@subworkflow("atmosphere_modelling_using_telluric_standard", "")
def model_on_standard(telluric_spectrum, input_mode=""):
    atmospheric_model = (task("atmospheric_model_on_standard_" + input_mode)
                         .with_recipe("xsh_molecfit_model")
                         .with_main_input(telluric_spectrum)
                         .with_associated_input(molecules)
                         .with_associated_input(wave_include)
                         .with_condition(attempt_telluric_correction)
                         .with_dynamic_parameter("arm", which_arm)
                         .with_input_filter(TELL_IDP_VIS, TELL_IDP_NIR, MOLECULES_VIS, MOLECULES_NIR,
                                            WAVE_INCLUDE_VIS, WAVE_INCLUDE_NIR)
                         .build())

    return atmospheric_model


@subworkflow("atmosphere_modelling_using_science_spectra", "")
def model_on_science(science_spectrum, input_type="", input_mode=""):
    atmospheric_model_on_science = (task("atmospheric_model_on_science_" + input_type + "_" + input_mode)
                                    .with_recipe("xsh_molecfit_model")
                                    .with_main_input(science_spectrum)
                                    .with_associated_input(molecules)
                                    .with_associated_input(wave_include)
                                    .with_dynamic_parameter("arm", which_arm)
                                    .with_condition(attempt_telluric_correction)
                                    .with_input_filter(SCI_IDP_NIR, SCI_IDP_VIS, TELL_IDP_NIR, TELL_IDP_VIS,
                                                       STD_IDP_NIR, STD_IDP_VIS, MOLECULES_VIS, MOLECULES_NIR,
                                                       WAVE_INCLUDE_VIS, WAVE_INCLUDE_NIR)
                                    .build())

    return atmospheric_model_on_science


def create_telluric_correction_task(spectrum_to_correct, atmospheric_transmission, input_type="", metatarget=""):
    telluric_correction = (task("telluric_corrected_" + input_type)
                           .with_recipe("xsh_molecfit_correct")
                           .with_main_input(spectrum_to_correct)
                           .with_associated_input(atmospheric_transmission)
                           .with_dynamic_parameter("arm", which_arm)
                           .with_condition(attempt_telluric_correction)
                           .with_input_filter(SCI_IDP_NIR, SCI_IDP_VIS, TELL_IDP_NIR, TELL_IDP_VIS,
                                              STD_IDP_NIR, STD_IDP_VIS, TELLURIC_CORR_NIR, TELLURIC_CORR_VIS)
                           .with_meta_targets([metatarget])
                           .build())

    return telluric_correction


@subworkflow("telluric_correction", "")
def xshoo_telluric_correction(spectrum_to_correct, atmospheric_model_nod, atmospheric_model_stare,
                              atmospheric_model_offset, atmospheric_model_on_science, input_type="", input_mode=""):
    associate_telluric = (match_rules()
                          .with_match_keywords([kwd.instrume, kwd.seq_arm, kwd.ins_opti5_name, kwd.night], level=-1)
                          .with_match_keywords([kwd.instrume, kwd.seq_arm, kwd.ins_opti5_name], time_range=ONE_WEEK,
                                               level=0))

    associate_very_same_science = (match_rules()
                                   .with_match_keywords([kwd.tpl_start]))

    # The atmospheric model (either from telluric standard or the scientific file itself) is selected with the
    # workflow parameter "telluric_correction_mode". Possible values are:
    # "standard": a telluric standard within the same night matching the instrument setup of the file to correct is used.
    #             If not found, the search is extended to 1 week. If also not found, the correction is not performed.
    # "science":  the very same science file to correct is used also to determine the atmospheric transmission.
    # "none":     no telluric correction
    atmospheric_model = (alternative_associated_inputs(sort_keys=[kwd.mjd_obs])
                         .with_associated_input(atmospheric_model_nod, condition=telluric_on_standard,
                                                match_rules=associate_telluric)
                         .with_associated_input(atmospheric_model_stare, condition=telluric_on_standard,
                                                match_rules=associate_telluric)
                         .with_associated_input(atmospheric_model_offset, condition=telluric_on_standard,
                                                match_rules=associate_telluric, min_ret=0)
                         .with_associated_input(atmospheric_model_on_science, condition=telluric_on_science,
                                                match_rules=associate_very_same_science, min_ret=0))

    atmospheric_transmission = (task("atmospheric_transmission_for_" + input_type + "_" + input_mode)
                                .with_recipe("xsh_molecfit_calctrans")
                                .with_main_input(spectrum_to_correct)
                                .with_alternative_associated_inputs(atmospheric_model)
                                .with_condition(attempt_telluric_correction)
                                .with_dynamic_parameter("arm", which_arm)
                                .with_input_filter(SCI_IDP_NIR, SCI_IDP_VIS, TELL_IDP_NIR, TELL_IDP_VIS,
                                                   STD_IDP_NIR, STD_IDP_VIS, GDAS_VIS, GDAS_NIR,
                                                   BEST_FIT_PARAMETERS_STD_VIS, BEST_FIT_PARAMETERS_STD_NIR,
                                                   ATM_PARAMETERS_STD_VIS, ATM_PARAMETERS_STD_NIR,
                                                   BEST_FIT_PARAMETERS_SCI_VIS, BEST_FIT_PARAMETERS_SCI_NIR,
                                                   BEST_FIT_MODEL_SCI_VIS, BEST_FIT_MODEL_SCI_NIR,
                                                   BEST_FIT_MODEL_STD_VIS, BEST_FIT_MODEL_STD_NIR,
                                                   ATM_PARAMETERS_SCI_VIS, ATM_PARAMETERS_SCI_NIR,
                                                   MOLECULES_NIR, MOLECULES_VIS, WAVE_INCLUDE_NIR,
                                                   WAVE_INCLUDE_VIS)
                                .build())

    telluric_correction = create_telluric_correction_task(spectrum_to_correct, atmospheric_transmission,
                                                          input_type=input_type + "_" + input_mode,
                                                          metatarget=input_type)
    return telluric_correction
