;*******************************************************************************
; File: myambergui.pro
; Front-end for amdlib and ammyorick AMBER data reduction
; Part of OYSTER (Christian Hummel)
;
; Modified amdlibUtils.c (look for CAH).
;
; Block directory:
; ----------------
;
; Block 1: alloc_amber_options,
;	   badpixelmap,flatfieldmap,
;	   cleanamberindex,cleanamberrawdir,
;	   analyzeamberconfigids,amberconfigids,
;	   amberrawdirchecksum,selectamberfiles,
;	   amberupdateconfig,ambergenconfig,
;	   amberstarids,amberscan
; Block 2: amdlibp2vm,amdlibappendoidata,amdlibcomputeoidata,
;	   amdlibperformframeselection
; Block 3: ambergui_event,ambergui_configid,ambergui_options,files_event,
;	   myambergui,
;	   amber2p2vgui_event,amber2p2vgui,amber2p2vgui2_event,
;	   amberp2vmfiles_event,amberp2vmgui_event,amberp2vmgui,
;	   amberrecipes_event,amberrecipesgui_event,amberrecipesgui,
;	   amberobjectgui_event,amberobjectgui,
;	   amberdefaultrecipes,amberinitializepipe,myamberpipe,
;	   myamberpipe_merge,myambervsop,checkp2vm,
;	   calpiston,calamber
;
; Change log for AMBER_DRS_2.0_Beta_1 release (2007-09-07):
; 1,$s/fringe_contrast_snr/base_fringe_snr/g
; 1,$s/piston_opd/base_opd/g
; 1,$s/flux_pi_more_pj_err/base_flux_sum_correction/g
; 1,$s/flux_pi_more_pj/base_flux_sum/g
;
; Change log for AMBER_DRS_2.0_Beta_2 release (2007-10-17):
; base_opd -> opd
; base_fringe_snr -> fringe_snr
; 
;
;************************************************************************Block 1
function alloc_amber_options
;
; Old versions:
; amber_options={piston:'PHASOR',errors:'THEORIC',frcrit:'FRG',frvalu:'2.0'}
;
nsel=3
return,{version:2.0, $
	short_config:0, $
	piston:'PHASE', $
	errors:'STATISTIC', $
	merges:'>', $
	selcrit:strarr(nsel)+'None', $
	seltype:strarr(nsel)+'Threshold', $
	selvalu:fltarr(nsel)+1.0}
;
end
;-------------------------------------------------------------------------------
function badpixelmap,obht
;
; Obsolete!
; Data from http://www.eso.org/observing/dfo/quality/AMBER/qc/qc1.html
;
date=strmid(obht(0).filename,6,10)
parsedate,date,y,m,d
if julian(y,m,d) lt julian(2007,9,15) then begin
return,!oyster_dir+'source/amber/BadPixelMap_oldDetector.fits'
endif else if julian(y,m,d) lt julian(2008,2,6) then begin
return,!oyster_dir+'source/amber/bpm/BPM_070915.fits'
endif else if julian(y,m,d) lt julian(2008,9,14) then begin
return,!oyster_dir+'source/amber/bpm/BPM_080206.fits'
endif else if julian(y,m,d) lt julian(2009,11,4) then begin
return,!oyster_dir+'source/amber/bpm/BPM_080914.fits'
endif else if julian(y,m,d) lt julian(2010,5,22) then begin
return,!oyster_dir+'source/amber/bpm/AM_GBPM_091104A_ENGINEERING_1.fits'
endif else if julian(y,m,d) lt julian(2013,1,14) then begin
return,!oyster_dir+'source/amber/bpm/AM_GBPM_100522A_ENGINEERING_1.fits'
endif else begin
return,!oyster_dir+'source/amber/bpm/AM_GBPM_130222A_ENGINEERING_1.fits'
endelse
;
end
;-------------------------------------------------------------------------------
function flatfieldmap,obht
;
; Obsolete!
; Data from http://www.eso.org/observing/dfo/quality/AMBER/qc/qc1.html
;
date=strmid(obht(0).filename,6,10)
parsedate,date,y,m,d
if julian(y,m,d) lt julian(2007,9,15) then begin
return,!oyster_dir+'source/amber/FlatFieldMap_oldDetector.fits'
endif else if julian(y,m,d) lt julian(2008,2,6) then begin
return,!oyster_dir+'source/amber/ffm/FFM_070915.fits'
endif else if julian(y,m,d) lt julian(2008,9,14) then begin
return,!oyster_dir+'source/amber/ffm/FFM_080206.fits'
endif else if julian(y,m,d) lt julian(2009,11,4) then begin
return,!oyster_dir+'source/amber/ffm/FFM_080914.fits'
endif else if julian(y,m,d) lt julian(2010,5,22) then begin
return,!oyster_dir+'source/amber/ffm/AM_GFFM_091104A_ENGINEERING_8.fits'
endif else if julian(y,m,d) lt julian(2013,1,14) then begin
return,!oyster_dir+'source/amber/ffm/AM_GFFM_100522A_ENGINEERING_8.fits'
endif else begin
return,!oyster_dir+'source/amber/ffm/AM_GFFM_130222A_ENGINEERING_8.fits'
endelse
;
end
;-------------------------------------------------------------------------------
function badpixelmap,obht
;
; Data from http://www.eso.org/observing/dfo/quality/AMBER/qc/qc1.html
;
jd_date=date2jd(strmid(obht(0).filename,6,10))
bpm_dir=!oyster_dir+'source/amber/bpm/'
;
bpm_files=specname(file_search(bpm_dir+'/AM_GBPM_*ENGINEERING_?.fits'))
bpm_jds=date2jd('20'+strmid(bpm_files,8,2)+'-' $
		    +strmid(bpm_files,10,2)+'-' $
	   	    +strmid(bpm_files,12,2))
for i=0,n_elements(bpm_files)-1 do $
	if jd_date le bpm_jds(i) then return,bpm_dir+bpm_files(i)
;
return,bpm_dir+bpm_files(n_elements(bpm_files)-1)
;
end
;-------------------------------------------------------------------------------
function flatfieldmap,obht
;
; Data from http://www.eso.org/observing/dfo/quality/AMBER/qc/qc1.html
;
jd_date=date2jd(strmid(obht(0).filename,6,10))
ffm_dir=!oyster_dir+'source/amber/ffm/'
;
ffm_files=specname(file_search(ffm_dir+'/AM_GFFM_*ENGINEERING_?.fits'))
ffm_jds=date2jd('20'+strmid(ffm_files,8,2)+'-' $
		    +strmid(ffm_files,10,2)+'-' $
	   	    +strmid(ffm_files,12,2))
for i=0,n_elements(ffm_files)-1 do $
	if jd_date le ffm_jds(i) then return,ffm_dir+ffm_files(i)
;
return,ffm_dir+ffm_files(n_elements(ffm_files)-1)
;
end
;-------------------------------------------------------------------------------
function obsdark,obht
;
; Returns the file name of the DARK observation right before the OBJECT
; dateobs=strmid(strtrim(prihead->getpar('DATE-OBS')),0,23)
;
jd_date=date2jd(strmid(obht(0).filename,6,10))
;
obt=obj_new('fitsfilelist',searchString='AMBER*.fits')
obt_hks=obt->getHeaderKeys(['MJD-OBS','INS MODE','DPR TYPE'])
obt_hks=obt_hks(where(strcompress(obt_hks.dprtype,/remove_all) eq 'DARK'))
parsedate,strmid(obht(0).filename,6,10),y,m,d
mjdobs=julian(y(0),m(0),d(0))-2400000.d0
index=where(abs(obt_hks.mjdobs-mjdobs) eq min(abs(obt_hks.mjdobs-mjdobs)))
obt_hks=obt_hks(index)
jndex=where(abs(obt_hks.mjdobs-mjdobs) eq min(abs(obt_hks.mjdobs-mjdobs)) $
	and strcompress(obt_hks.dprtype,/remove_all) eq 'DARK')
return,obt_hks.filename

; parsedate,strmid(obht(objindex(i)).filename,6,10),y,m,d
; jd_obj(i)=julian(y,m,d)+hms2h(strmid(obht(objindex(i)).filename,17,12))/24.
;
end
;-------------------------------------------------------------------------------
function cleanamberindex,obs
;
; This function will return the indices of files corresponding
; to data of the first full observation in the observation 
; array given to it.
;
; Identify the first block containing a fringe track record
;
n=n_elements(obs)
;
i=where(obs.m eq '2P2V' $
     or obs.m eq '3P2V' $
     or obs.m eq 'DARK' $
     or obs.m eq 'SKY' $
     or obs.m eq 'OBJECT',c)
if c eq 0 then return,-2	; No more data
; This is the target we will consider
target=obs(i(0)).t
insopt=obs(i(0)).o
i0=0
i1=n_elements(obs)-1
n=n_elements(obs)
obs0=obs(i0:i1)				; This part contains our observation
; To be processed in the next call
n0=n_elements(obs0)
i=0
while obs0(n0-1-i).m eq 'FLUX' $
   or obs0(n0-1-i).m eq 'COLPOS' $
   or obs0(n0-1-i).m eq 'SPECPOS' $
   or obs0(n0-1-i).m eq 'ZOFF,3TEL' $
   or obs0(n0-1-i).m eq 'FRNSRC' $
   or obs0(n0-1-i).m eq 'COHERENC' $
   or obs0(n0-1-i).m eq 'CPTPIST' do i=i+1
i1=i1+1-i
if n-1 ge i1 then obs=obs(i1:n-1) $
	     else begin
		  obs=obs(0)
		  obs.m=''
	     endelse
; Only select observations of this target (the ACQ at the end is of the next OB)
obs0=obs0(0:n0-1-i)
; Now determine the locations of the templates we need
index=where(obs0.m eq '2P2V' $
         or obs0.m eq '3P2V' $
         or obs0.m eq 'DARK' $
         or obs0.m eq 'SKY' $
         or obs0.m eq 'OBJECT')
index=index(0)
return,obs0(index).i
;
end
;-------------------------------------------------------------------------------
pro cleanamberrawdir,files
;
; Prepare and execute repeated calls of cleanamberindex until all
; files not needed for data reduction are moved to a directory 
; "AMBER.auxiliary_engineering_files".
;
if n_elements(files) eq 0 then files=''
if strlen(files(0)) gt 0 then $
ob=obj_new('fitsfilelist',files) else $
ob=obj_new('fitsfilelist',searchString='AMBER*.fits')
hk=ob->getHeaderKeys(['INS MODE','TARG NAME','DPR TYPE'])
n=n_elements(hk)
i=indgen(n)
mindex=intarr(n)
f=ob->files()
obj_destroy,ob
m=hk.insmode
t=strcompress(hk.targname,/remove_all)
o=strcompress(hk.dprtype,/remove_all)
obs={i:0,f:'',m:'',t:'',o:''}
obs=replicate(obs,n)
obs.i=i
obs.f=f
obs.m=m
obs.t=t
obs.o=o
obs0=obs
;
index=where(strpos(obs.m,'ENGINEERING') eq -1 $
       and  strpos(obs.o,'FLUX') eq -1 $
       and  strpos(obs.o,'COLPOS') eq -1 $
       and  strpos(obs.o,'SPECPOS') eq -1 $
       and  strpos(obs.o,'ZOFF,3TEL') eq -1 $
       and  strpos(obs.o,'FRNSRC') eq -1 $
       and  strpos(obs.o,'COHERENC') eq -1 $
       and  strpos(obs.o,'CPTPIST') eq -1)
obs=obs(index)
;
repeat begin
	r=cleanamberindex(obs)
	if r(0) ge 0 then begin
		if n_elements(index) eq 0 then index=r $
					  else index=[index,r]
	endif
endrep until r(0) eq -2
;
mindex(index)=1
spawn,'mkdir -p AMBER.auxiliary_engineering_files'
for i=0,n-1 do begin
	if mindex(i) eq 0 then begin
		spawn,'mv '+obs0(i).f+' AMBER.auxiliary_engineering_files'
	endif
endfor
;
end
;-------------------------------------------------------------------------------
function analyzeamberconfigids
;
; Determine how many different configurations are present in the raw data
; of the working directory. Configurations are different for different
; arrays, instrument modes, P2VM IDs, and read-out windows. However, an
; option exists where only baseline and spectral setup ("short config") are
; used. This allows one to put all corresponding data into one output file.
;
files=findfile('AMBER*.fits')
;
if strlen(files(0)) eq 0 then return,'No AMBER raw files'
;
n=n_elements(files)
configurations=strarr(n)
shortconfigids=strarr(n)
;
for i=0,n-1 do begin
;
	fitsfile=obj_new('fitsfile',files(i))
	prihead=fitsfile->prihead()
;
	insmode=strtrim(prihead->getpar('INS MODE'))
	p2vmid=strtrim(prihead->getpar('OCS P2VM ID'),2)
	starty=strtrim(prihead->getpar('DET WIN STARTY'),2)
	dprtype=strtrim(prihead->getpar('DPR TYPE'))
;
	configid=p2vmid+'+'+starty
;
	station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
	station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
	station_3=prihead->getpar('ISS CONF STATION3')
	if isnumeric(station_3) $
	then station_3='' $
	else $	; Next statement from backup 231007
;	station_3='-'+vlti_stationid(strtrim(station_3))
;	The following of 231026 causes a problem when initializing myambergui
	station_3=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION3')))
;
	if strpos(dprtype,'OBJECT') ge 0 then $
;	A new P2VM will change the closure phase offset!
	configurations(i)=insmode+'+'+strtrim(string(configid),2) $
				 +'+'+station_1+'-'+station_2+'-'+station_3
	shortconfigids(i)=insmode $
				 +'+'+station_1+'-'+station_2+'-'+station_3
;
	obj_destroy,fitsfile
;
endfor
;
; Extended list
index=where(strlen(configurations) ne 0,count)
if count gt 0 $
then configurations=unique([configurations(index),shortconfigids(index)]) $
else configurations=''
;
return,configurations
;
end
;-------------------------------------------------------------------------------
function amberconfigids,obht
;
; Return a set of configids currently loaded.
;
insmode=strtrim(obht.insmode)
starty=strcompress(string(obht.detwinstarty),/remove_all)
p2vmid=strtrim(string(obht.p2vmid),2)
configid=p2vmid+'+'+starty
;
station_1=vlti_stationid(strtrim(obht.ISSCONFSTATION1,2))
station_2=vlti_stationid(strtrim(obht.ISSCONFSTATION2,2))
station_3=vlti_stationid(strtrim(obht.ISSCONFSTATION3,2))
;
for i=0,n_elements(station_3)-1 do $
if isnumeric(station_3(i)) then station_3(i)='' $
			   else station_3(i)='-'+station_3(i)
;
configurations=insmode+'+'+configid+'+' $
 	+station_1+'-'+station_2+station_3
;
; Extended list
configurations=[insmode+'+'+configid+'+' $
 	+station_1+'-'+station_2+station_3, $
	insmode+'+' $
	+station_1+'-'+station_2+station_3]
;
return,configurations
return,configurations(where(strpos(obht.dprtype,'OBJECT') ge 0))
;
end
;-------------------------------------------------------------------------------
function amberrawdirchecksum
;
; Returns a checksum of the MIDI files in the current directory.
;
f=findfile('AMBER*.fits')
n=n_elements(f)
if n eq 0 then return,0
r=file_info(f(0))
r=replicate(r,n)
for i=0,n-1 do r(i)=file_info(f(i))
;
return,strjoin(string(r.name)+string(r.ctime)+string(r.mtime)+string(r.size))
;
end
;-------------------------------------------------------------------------------
pro selectamberfiles,all=all
;
; Displays the gorgonzola gui to select files. Stores the results in
; common block. If all are requested, return only those corresponding
; to the currently selected configuration.
;
common AmberGuiWids,amber_wid,file_wid,window_slide_wid,pipeline_wid,opmenuc_wid
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberOptions,amber_options,p2vmfile
;
keys=['TARG NAME','INS MODE','DET WIN STARTY','P2VM ID','FT SENSOR','DPR TYPE',$
      'ISS CONF STATION1','ISS CONF STATION2','ISS CONF STATION3']
print,'Please wait, reading files...'
print,'Note: only select files with station information!'
if keyword_set(all) then begin
	ob=obj_new('fitsfilelist',searchString='AMBER*.fits')
	obht=ob->getHeaderKeys(keys)
	nobht=n_elements(obht)
	ob->setHeader,obht,keys
	obsfiles=ob->files()
	obht.filename=obsfiles
endif else begin
	ob=obj_new('fitsfilelist',dir=dir, $
		guikey=keys,guisearch="'AMBER*.fits'")
;	The header table initially contains all files
	obht=ob->headertable()
	nobht=n_elements(obht)
	obsfiles=ob->files()
;	Remove "./" in front of the file names
	prefix=0
	prefix=strpos((ob->files())(0),'AMBER')
	files=obht.filename
	n=n_elements(obsfiles)
	if strlen(obsfiles(0)) eq 0 then n=0
	if n gt 0 then begin
		index=intarr(n)
		for i=0,n-1 do begin
			words=specname(nameparse(obsfiles(i)))
			index(i)=where(files eq words(0))
		endfor
		obht=obht(index)
		obht.filename=strmid(obsfiles,prefix,34)
	endif
endelse
obj_destroy,ob
obht.targname=strcompress(obht.targname,/remove_all)
;
index=where(strpos(obht.dprtype,'P2V') ge 0 $
	 or strpos(obht.dprtype,'WAVE') ge 0,count)
if count gt 0 then obht(index).p2vmid=-1
;
; The P2VM files have no stations, but we need to assign them a configuration
index=where(obht.p2vmid lt 0,count)
jndex=where(obht.p2vmid gt 0)
while count gt 0 do begin
	kndex=where(jndex gt index(0),count)
	if count gt 0 then begin
		j=jndex(kndex(0))
		kndex=where(index lt j)
		obht(index(kndex)).issconfstation1=obht(j).issconfstation1
		obht(index(kndex)).issconfstation2=obht(j).issconfstation2
		if not isnumeric(obht(0).issconfstation3) then $
		obht(index(kndex)).issconfstation3=obht(j).issconfstation3
		obht(index(kndex)).p2vmid=obht(j).p2vmid
		obht(index(kndex)).detwinstarty=obht(j).detwinstarty
		index=where(obht.p2vmid lt 0,count)
	endif else count=0
endwhile
if nobht eq n_elements(obsfiles) then save,obht,filename='obht.sav'
;
; For option all, we do however select according to the current configid
if keyword_set(all) then begin
	configurations=amberconfigids(obht)
	if n_elements(configurations) eq n_elements(obht)*2 $
		then incl_short_configs=1 $
		else incl_short_configs=0
	words=nameparse(configid,'+')
	if n_elements(words) gt 2 then p2vmid=long(words(1)) else p2vmid=0
;	r=obht.p2vmid eq p2vmid and obht.detwinstarty eq 1
;	if incl_short_configs then r=[r,r]
;	index=where(configurations eq configid or r,count)
	index=where(configurations eq configid,count)
	if p2vmid eq 0 then index=index-n_elements(obht)
	if count gt 0 then obht=obht(index)
endif
;
; The obsindex shall refer exactly to those files needed for a visibility
objindex=where(strtrim(obht.dprtype,2) eq 'OBJECT')
skyindex=where(strtrim(obht.dprtype,2) eq 'SKY')
darkindex=where(strtrim(obht.dprtype,2) eq 'DARK')
;
; Display selected files in list widget
if n_elements(file_wid) eq 0 then file_wid=0l
if widget_info(file_wid,/valid) then $
widget_control,file_wid,set_value=obht.filename,set_uvalue=obht.filename
;
end
;-------------------------------------------------------------------------------
pro amberupdateconfig,genconfig
;
; The wavelength section of GenConfig is not known until la_bins have been
; defined, either by default, or using lambda_gui. This procedure is called
; once this info is available to finish the full definition of GenConfig.
;
common AmberOptions,amber_options,p2vmfile
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
;
dispersion=strtrim(unique(obht.insgris),2)
;
win1nx=detwin(0)
win1strx=detwin(1)
wavelengths=xmdvavg(pix2lambda(dindgen(win1nx)+1+win1strx,dispersion), $
			la_bins)*1e-6
genconfig.numspecchan(*)=n_elements(wavelengths)
genconfig.wavelength(0:genconfig.numspecchan(0)-1,0)=wavelengths
genconfig.wavelength(*,1)=genconfig.wavelength(*,0)
genconfig.wavelengtherr=genconfig.wavelength*0+0.1e-6
;
end
;-------------------------------------------------------------------------------
pro ambergenconfig,filename,genconfig
;
; Open specified file, and extract information into given genconfig.
; This does not include channel wavelengths and widths.
;
; Note on FINITO: there is a fixed mapping of IPs and FINITO channels:
; IP1 -> F_CH2
; IP3 -> F_CH0
; IP5 -> F_CH1
; The reference channel of FINITO is 0, i.e. the tracking baselines are 
; IP1-IP3 and IP5-IP3.
;
; Open file
fitsfile=obj_new('fitsfile',filename)
prihead=fitsfile->prihead()
;
; Read configuration
insmode=strtrim(prihead->getpar('INS MODE'))
p2vmid=strtrim(prihead->getpar('OCS P2VM ID'),2)
starty=strtrim(prihead->getpar('DET WIN STARTY'),2)
detdit=strtrim(prihead->getpar('DET DIT'),2)
configid=p2vmid+'+'+starty
;
station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
station_3=prihead->getpar('ISS CONF STATION3')
if isnumeric(station_3) then station_3='' $
			else station_3=vlti_stationid(strtrim(station_3))
stations=[station_1,station_2,station_3]
telescope_1=strtrim(prihead->getpar('ISS CONF T1NAME'))
telescope_2=strtrim(prihead->getpar('ISS CONF T2NAME'))
telescope_3=strtrim(prihead->getpar('ISS CONF T3NAME'))
;
delayline_1=strtrim(prihead->getpar('ISS CONF DL1'))
delayline_2=strtrim(prihead->getpar('ISS CONF DL2'))
delayline_3=strtrim(prihead->getpar('ISS CONF DL3'))
if isnumeric(delayline_3) then delayline_3='DL0'
delaylines=[delayline_1,delayline_2,delayline_3]
delaylineids=strmid(delaylines,2,1)
refname=prihead->getpar('DEL REF NAME')
bcinput_1=long(strtrim(prihead->getpar('ISS CONF INPUT1'))) > 1
bcinput_2=long(strtrim(prihead->getpar('ISS CONF INPUT2'))) > 3
bcinput_3=long(strtrim(prihead->getpar('ISS CONF INPUT3'))) > 5
bcinputs=[bcinput_1,bcinput_2,bcinput_3]
;
; Close file
obj_destroy,prihead
obj_destroy,fitsfile
;
; Store configuration in genconfig
index=where(strlen(stations) gt 0,numsid)
genconfig.numsid=numsid
genconfig.stationid(0)=stations(0)
genconfig.stationid(1)=stations(1)
genconfig.stationid(2)=stations(2)
genconfig.baselineid(0,*)=genconfig.stationid(0)+'-'+genconfig.stationid(1)
if numsid eq 3 then begin
genconfig.baselineid(1,*)=genconfig.stationid(1)+'-'+genconfig.stationid(2)
genconfig.baselineid(2,*)=genconfig.stationid(2)+'-'+genconfig.stationid(0)
endif
genconfig.delaylineid=delaylineids
ref_count=0
if not isnumeric(refname) then $
ref_index=where(delaylineids eq long(strmid(refname,2,1)),ref_count)
if ref_count eq 0 then ref_index=0
genconfig.refstation=ref_index+1
genconfig.bcinputid=bcinputs
genconfig.numoutbeam=3
genconfig.numbaseline=((numsid-1)*numsid)/2
genconfig.instrcohint=detdit*1000	; ms
;
genconfig.configid=insmode+'+'+configid+'+' $
	+genconfig.stationid(0)+'-'+genconfig.stationid(1)
if numsid eq 3 then genconfig.configid=genconfig.configid+'-'+genconfig.stationid(2)
;
if strpos(telescope_1,'UT') ge 0 then genconfig.diameter=8 $
				 else genconfig.diameter=1.8
;
end
;-------------------------------------------------------------------------------
function amberstarids,scans
;
; Adopt a single starid (the first) for stars of the same name.
; This is important for the OBJ designation in case it was derived
; from apparent coordinates.
;
stars=unique(scans.star)
for i=0,n_elements(stars)-1 do begin
	index=where(scans.star eq stars(i))
	scans(index).starid=scans(index(0)).starid
endfor
;
return,scans
;
end
;-------------------------------------------------------------------------------
pro amberscan,filename,scan_i
;
; Open specified file, and extract information into given OYSTER scan
;
; Open file
fitsfile=obj_new('fitsfile',filename)
prihead=fitsfile->prihead()
;
; Read data
dateobs=strtrim(prihead->getpar('DATE-OBS'))
time=hms2h(strmid(dateobs,11,12))*3600	; [s]
airmass=float(strtrim(prihead->getpar('ISS AIRM END')))
seeing=float(strtrim(prihead->getpar('ISS AMBI FWHM')))
coherence=float(strtrim(prihead->getpar('ISS AMBI TAU0')))
detdit=strtrim(prihead->getpar('DET DIT'),2)
obstarg=strcompress(prihead->getpar('OBS TARG NAME'),/remove_all)
obsra=prihead->getpar('RA')/15.
obsdec=prihead->getpar('DEC')
dlt1s=prihead->getpar('DEL DLT1 OPL START')
dlt2s=prihead->getpar('DEL DLT2 OPL START')
dlt3s=prihead->getpar('DEL DLT3 OPL START')
dlt1e=prihead->getpar('DEL DLT1 OPL END')
dlt2e=prihead->getpar('DEL DLT2 OPL END')
dlt3e=prihead->getpar('DEL DLT3 OPL END')
refopl=prihead->getpar('DEL REF OPL')
refname=prihead->getpar('DEL REF NAME')
categ=prihead->getpar('DPR CATG')
;
; Store FDLPOS
scan_i.fdlpos(0)=(dlt1s+dlt1e)/2
scan_i.fdlpos(1)=(dlt2s+dlt2e)/2
scan_i.fdlpos(2)=(dlt3s+dlt3e)/2
scan_i.fdlposerr=10e-6
;
; FINITO phase RMS. FINITO tracks on baseline 0-1 and 0-2, where
; 0=IP3, 1=IP5, 2=IP1
phaserms1=float(strtrim(prihead->getpar('DEL FNT PHA_RMS_CH1'))) ;0-1
phaserms2=float(strtrim(prihead->getpar('DEL FNT PHA_RMS_CH2'))) ;0-2
;
; IRIS error signal RMS
iris_rmsx1=float(strtrim(prihead->getpar('ISS IAS IRIS_ERRSTDX1')))
iris_rmsy1=float(strtrim(prihead->getpar('ISS IAS IRIS_ERRSTDY1')))
iris_rmsx2=float(strtrim(prihead->getpar('ISS IAS IRIS_ERRSTDX2')))
iris_rmsy2=float(strtrim(prihead->getpar('ISS IAS IRIS_ERRSTDY2')))
iris_rmsx3=float(strtrim(prihead->getpar('ISS IAS IRIS_ERRSTDX3')))
iris_rmsy3=float(strtrim(prihead->getpar('ISS IAS IRIS_ERRSTDY3')))
;
; Close file
obj_destroy,prihead
obj_destroy,fitsfile
;
; Get starid, coords are passed in case of OBJ classification
starid=cri_vlti(obstarg,obsra,obsdec)
; if strpos(starid,'OBJ') ge 0 then print,cri_simbad(starid),obstarg
;
; Store data in scan
scan_i.time=time
scan_i.star=obstarg
scan_i.starid=starid
scan_i.ra=obsra
scan_i.dec=obsdec
scan_i.r0=seeing
scan_i.int_time=detdit*1000	; s, 1000 frames per file
;
; Fill the scans.t0 array, which has some redundant indices here
r=size(scan_i.t0)
for i=0,r(1)-1 do begin
for j=0,r(2)-1 do begin
for k=0,r(3)-1 do begin
	scan_i.t0(i,j,k)=coherence*1000	; [ms]
endfor
endfor
endfor
; Store phase RMS in OB=2 (H-band), assuming same order as IPs
scan_i.phaserms(1,1)=phaserms1
scan_i.phaserms(1,0)=phaserms2
scan_i.phaserms(1,2)=sqrt(phaserms1^2+phaserms2^2)
;
; Store IRIS data in NATJitter2
if n_elements(scan_i.natjitter2(*,0)) eq 3 then begin
scan_i.natjitter2(0,0)=iris_rmsx2
scan_i.natjitter2(0,1)=iris_rmsy2
scan_i.natjitter2(1,0)=iris_rmsx1
scan_i.natjitter2(1,1)=iris_rmsy1
scan_i.natjitter2(2,0)=iris_rmsx3
scan_i.natjitter2(2,1)=iris_rmsy3
endif
;
if airmass gt 0 then $
scan_i.za=acos(1/airmass)*(180/!pi)
;
end
;************************************************************************Block 2
pro amdlibp2vm
;
; Compute P2VM. Spectral calibration (shifts) are computed from WAVE files
; if provided, otherwise this procedure will look for file speccal.txt.
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberOptions,amber_options,p2vmfile
;
p2vm_file='p2vm.fits'
spawn,'rm -f '+p2vm_file
;
bpix=badpixelmap(obht)
flat=flatfieldmap(obht)
dark=obsdark(obht)
;
index=where(strpos(obht.dprtype,'WAVE') ge 0,wave_count)
if wave_count gt 0 then begin
        print,'Computing spectral calibration...'
        wave_file='offsety.txt'
        infiles=bpix+' '+flat+' '+strjoin(obht(index).filename,' ')
;       infiles=bpix+' '+flat+' '+dark+' '+strjoin(obht(index(0:2)).filename,' ')
	if fix(amber_options.version) ne 3 then begin
		print,'Sorry, amdlib version must be 3.0.9!'
		retall
	endif
	if fix(amber_options.version) eq 3 then $
        spawn,'amdlibComputeSpectralCalibration '+infiles+' -o '+wave_file $
	else $
        spawn,'amdlibComputeSpectralCalibration '+infiles+' '+wave_file
endif else begin
        wavefile=findfile('speccal.txt')
        if strlen(wavefile) ne 0 then begin
                print,'Using spectral calibration from speccal.txt'
                wave_file=wavefile
        endif 
endelse
if n_elements(wave_file) ne 0 then begin
        offsety=''
        status=dc_read_fixed(wave_file,offsety,/col,ignore=['!'],format='(a40)')
	noff=n_elements(offsety)
	if noff gt 1 then begin
		pc0=fltarr(noff)
		pc1=fltarr(noff)
		pc2=fltarr(noff)
		pc3=fltarr(noff)
		for i=0,noff-1 do begin
			if fix(amber_options.version) eq 3 then begin
				words=nameparse(offsety(i))
				offsety(i)=words(1)+','+words(3)
			endif
			words=nameparse(offsety(i),',')
			pc0(i)=float(words(0))
			pc1(i)=float(words(1))
			pc2(i)=float(words(2))
			if fix(amber_options.version) eq 3 then $
			pc3(i)=float(words(3))
		endfor
		index=where(pc0 ne 0 and pc1 ne 0 and pc2 ne 0,noff)
		if noff gt 0 then begin
			pc0=pc0(index)
			pc1=pc1(index)
			pc2=pc2(index)
			pc3=pc3(index)
		endif
		pc0=mean(pc0)
		pc1=mean(pc1)
		pc2=mean(pc2)
		pc3=mean(pc3)
		pcm=[pc0,pc1,pc2]
		if fix(amber_options.version) eq 3 then begin
		offsety='-s ' $
		       +strcompress(strjoin(string(pcm,format='(f5.2)'),','), $
							/remove_all) $
		       +' -S '+string(pc3,format='(f5.2)')
		endif else begin
		offsety=strcompress(strjoin(string(pcm,format='(f5.2)'),',') $
			,/remove_all)
		endelse
		print,'Offsets = '+offsety
	endif
	if fix(amber_options.version) ne 3 then offsety=' -a -s '+offsety
endif else begin
	offsety=''
	print,'No wave file, setting offsets to zero!'
endelse
;
index=where(strpos(obht.dprtype,'P2V') ge 0)
infiles=bpix+' '+flat+' '+strjoin(obht(index).filename,' ')
spawn,'amdlibComputeP2vm '+offsety+' '+infiles+' '+p2vm_file
;
fitsfile=obj_new('fitsfile',p2vm_file)
prihead=fitsfile->prihead()
insmode=strtrim(prihead->getpar('INS MODE'))
p2vmid=strtrim(prihead->getpar('OCS P2VM ID'))
obj_destroy,prihead
obj_destroy,fitsfile
;
new_file=insmode+'.'+strtrim(string(p2vmid),2)+'.'+p2vm_file
print,'Renaming to: '+new_file
spawn,'mv -f '+p2vm_file+' '+new_file
;
if strlen(offsety) ne 0 and wave_count gt 0 then begin
        new_file=insmode+'.'+strtrim(string(p2vmid),2)+'.'+wave_file
        print,'Renaming to: '+new_file
        spawn,'mv -f '+wave_file+' '+new_file
endif
;
end
;-------------------------------------------------------------------------------
pro amdlibcomputespectralcalibration
;
; Compute spectral calibration from WAVE files.
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
;
bpix=badpixelmap(obht)
flat=flatfieldmap(obht)
;
index=where(strpos(obht.dprtype,'WAVE') ge 0,count)
types=unique(obht.dprtype)
if n_elements(types) ne 1 then begin
	print,'***Error: 2- and 3-telescope data mixed!'
	return
endif
types=types(0)
words=nameparse(obht(index(0)).dprtype,',')
ntel=fix(words(1))
nwav=ntel+1
for i=0,count/nwav-1 do begin
        if i eq 0 then wave_file='speccal.txt' else wave_file='speccal1.txt'
	infiles=bpix+' '+flat+' '+ $
		strjoin(obht(index(i*nwav:(i+1)*nwav-1)).filename,' ')
	spawn,'amdlibComputeSpectralCalibration '+infiles+' '+wave_file
	if i gt 0 then spawn,'cat speccal1.txt >> speccal.txt'
endfor
;
end
;-------------------------------------------------------------------------------
pro amdlibappendoidata,infiles,outfile
;
command='amdlibAppendOiData '+strjoin(infiles,' ')+' '+outfile
spawn,nameparse(command),/noshell
;
end
;-------------------------------------------------------------------------------
pro amdlibcomputeoidata,objfile,destfile,darkfile,skyfile,scan=scan
;
; amdlib 1.0
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberOptions,amber_options,p2vmfile
;
if n_elements(scan) eq 0 then scan=0 else scan=scan gt 0
;
bpix=badpixelmap(obht)
flat=flatfieldmap(obht)
if n_elements(darkfile) eq 0 then darkfile=''
if n_elements(skyfile) eq 0 then skyfile=''
if strlen(darkfile) ne 0 then dark=darkfile else dark=obht(darkindex(0)).filename
if strlen(skyfile) ne 0 then sky=skyfile else sky=dark
;
calfiles=bpix+' '+flat+' '+p2vmfile+' '+dark+' '+sky
if scan then binning=' ' else binning=' -b 1 '
piston=' -p '+amber_options.piston
errors=' -e '+amber_options.errors
frcrit=' -c '+amber_options.frcrit
frvalu=' -r '+amber_options.frvalu
options=piston+errors+frcrit+frvalu+' '
if not scan then options=' '
;
command='amdlibExtractVis -s'+binning+options+calfiles+' '+objfile+' '+destfile
spawn,command
;
end
;-------------------------------------------------------------------------------
pro amdlibcomputeoidata,objfile,destfile,darkfile,skyfile,scan=scan
;
; amdlib 2.1
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberOptions,amber_options,p2vmfile
;
if n_elements(scan) eq 0 then scan=0 else scan=scan gt 0
;
bpix=badpixelmap(obht)
flat=flatfieldmap(obht)
if n_elements(darkfile) eq 0 then darkfile=''
if n_elements(skyfile) eq 0 then skyfile=''
if strlen(darkfile) ne 0 then dark=darkfile else begin
	if darkindex(0) ne -1 then dark=obht(darkindex(0)).filename $
			      else dark='""'
endelse
if strlen(skyfile) ne 0 then sky=skyfile else begin
	if skyindex(0) ne -1 then sky=obht(skyindex(0)).filename $
			     else sky='""'
endelse
;
calfiles=bpix+' '+flat+' '+p2vmfile+' '+dark
if fix(amber_options.version) le 2 then calfiles=calfiles+' '+sky
if scan then binning=' ' else binning=' -b 1 '
piston=' -p '+amber_options.piston
errors=' -e '+amber_options.errors
; frcrit=' -c '+amber_options.frcrit
frcrit=''
; frvalu=' -r '+amber_options.frvalu
frvalu=''
options=piston+errors+frcrit+frvalu+' '
if not scan then options=' '
;
; Option -f forces the use of the P2VM
command='amdlibComputeOiData -f -s'+binning+options $
	+calfiles+' '+objfile+' '+destfile
print,command
spawn,command
;
end
;-------------------------------------------------------------------------------
pro amdlibperformframeselection,inputfile,outputfile
;
; amdlib 2.1
;
common AmberOptions,amber_options,p2vmfile
;
selcrit='SNR'
seltype='Threshold'
selvalu=0.01
nsel=1
index=where(amber_options.selcrit ne 'None',count)
if count gt 0 then begin
	selcrit=amber_options.selcrit(index)
	seltype=amber_options.seltype(index)
	selvalu=amber_options.selvalu(index)
	nosel=0
endif else begin
	amber_options.selcrit(0)=selcrit
	amber_options.seltype(0)=seltype
	amber_options.selvalu(0)=selvalu
	nosel=1
endelse
nsel=n_elements(selcrit)
;
inselfile='frame_selection_in.fits'
outselfile='frame_selection_out.fits'
outselcommand=' -o '+outselfile
;
spawn,'rm -f '+outselfile
;
for i=0,nsel-1 do begin
	case selcrit(i) of
	'Flux':		selection=' -c FLUX_'
	'Piston':	selection=' -c OPD_'
	'SNR':		selection=' -c FRG_'
	endcase
	case seltype(i) of
	'Threshold':	selection=selection+'THR'
	'Percent':	selection=selection+'PCG'
	endcase
	value=selvalu(i)
	if seltype(i) eq 'Percent' then value=value/100
	if fix(amber_options.version) le 2 then $
	if selcrit(i) eq 'Piston' then value=value*1000	; mu -> nm
	ratio=' -r '+string(value)
	if i eq 0 then inselcommand=' ' else inselcommand= ' -i '+inselfile
	command='amdlibPerformFrameSelection -v 2 ' $
		+selection $
		+ratio $
		+inselcommand $
		+outselcommand+' ' $
		+inputfile+' ' $
		+outputfile
;	spawn,nameparse(command),/noshell
	spawn,command
	f=findfile(outselfile)
	if strlen(f(0)) ne 0 then spawn,'mv -f '+outselfile+' '+inselfile
endfor
;
if nosel then amber_options.selcrit(0)='None'
;
end
;************************************************************************Block 3
function ambergui_event,event
;
; This is the call back for buttons on the main MyAmberGui.
;
common AmberGuiWids,amber_wid,file_wid,window_slide_wid,pipeline_wid,opmenuc_wid
common AmberTaskWid,amber2p2v_wid,amberobject_wid,amberp2vm_wid,amberrecipes_wid
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberOptions,amber_options,p2vmfile
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
widget_control,event.id,get_uvalue=dir
;
case event.value of
;
'Observation':	begin
;		Clean desktop a little
		wdall
		if widget_info(amber2p2v_wid,/valid) then $
			widget_control,amber2p2v_wid,/destroy
		if widget_info(amberobject_wid,/valid) then $
			widget_control,amberobject_wid,/destroy
;		Clear p2vm selection
		if widget_info(amberp2vm_wid,/valid) then amberp2vmgui,/update
;		Clear p2vm file
		p2vmfile=''
		set_screen
		!x.range=0
		!y.range=0
		!p.psym=0
;		Store current OYSTER observation
		if n_elements(geninfo) ne 0 and n_elements(bufferinfo) ne 0 $
			then begin
			storenight,11
			save,recipes,filename= $
				'Recipes_'+genconfig.configid+'.xdr'
		endif
;
		selectamberfiles
;
;		If a complete observation was selected, get additional info
		if objindex(0) ge 1 and darkindex(0) eq 1 and skyindex(0) eq 1 $
			then begin 
			fitsfiles=obht(objindex(0)).filename
			fitsfile=obj_new('fitsfile',fitsfiles(0))
			prihead=fitsfile->prihead()
			airmass=strtrim(prihead->getpar('ISS AIRM END'))
			seeing=strtrim(prihead->getpar('ISS AMBI FWHM'))
			dispersion=strtrim(prihead->getpar('INS MODE'))
			p2vmid=strtrim(prihead->getpar('OCS P2VM ID'))
			obj_destroy,prihead
			obj_destroy,fitsfile
			print,'Target: ',obht(objindex(0)).targname
			print,'Airmass:',airmass,', seeing:',seeing
			print,'Setup:',dispersion,', P2VM ID: ',p2vmid
;			Load corresponding OYSTER scans structure
			if n_elements(geninfo) ne 0 then begin
				currentconfig= $
					unique(amberconfigids(obht(objindex)))
				index=where(geninfo.configid eq  $
					currentconfig,count)
;				This is currently not strictly needed
				if count eq 1 then begin
					datum=geninfo(index).date
					loadnight,datum,currentconfig
					restore,'Recipes_'+currentconfig+'.xdr'
				endif
			endif
		endif
		end
'AMBER':	begin
		spawn,'dfits AMBER*.fits | fitsort '+ $
			'OBS.TARG.NAME OBJECT '+ $
			'DET.DIT '+ $
			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=r,width=142
		end
'FINITO':	begin
		spawn,'dfits AMBER*.fits | fitsort '+ $
			'OBS.TARG.NAME OBJECT '+ $
			'ISS.OPDC.ALGOTYPE DEL.FNT.TRK_MODE ISS.FNT.DIT_CH2  '+ $
			'DEL.FNT.PHA_RMS_CH1 DEL.FNT.LOCKR_CH1 '+ $
			'DEL.FNT.PHA_RMS_CH2 DEL.FNT.LOCKR_CH2 '+ $
			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=r,width=142
		end
'COUDE':	begin
		spawn,'dfits AMBER*.fits | fitsort '+ $
			'OBS.TARG.NAME OBJECT '+ $
			'COU.AO2.T0_MEAN COU.AO2.FWHM_MEAN '+ $
			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=r,width=142
		end
'AMBIENT':	begin
		spawn,'dfits AMBER*.fits | fitsort '+ $
			'OBS.TARG.NAME OBJECT '+ $
			'ISS.AMBI.FWHM.START ISS.AMBI.TAU0.START ISS.AMBI.WINDSP'+ $
			'| sort -n -k 1',r
		xdisplayfile,'',title='dfits',text=r,width=142
		end
'P2VM Matrix': 	begin
		print,'Please select five/ten P2V files, optionally with 4 WAVE files!'
		selectamberfiles
		count=n_elements(obht)
		if count ne 5 and count ne 10 and count ne 14 then return,-1
		index=where(strpos(obht.dprtype,'P2V') ge 0,count)
		if count ne 5 and count ne 10 then return,-1
		amdlibp2vm
		end
'P2VM Tracker': begin
		amberp2vmgui
		end
'Pipeline':	begin
		!p.multi=[0,2,2]
		myamberpipe
		print,'Note: J-band data are not processed!'
		end
'Recipes': 	begin
		amberrecipesgui
		end
'Cleanup': 	begin
		selectamberfiles,/all
		targets=unique(obht.targname)
		for i=0,n_elements(targets)-1 do begin
		spawn,'rm -f '+strcompress(targets(i),/remove_all)+'.AVG*.fits'
		spawn,'rm -f '+strcompress(targets(i),/remove_all)+'.PHAS*.fits'
		endfor
		print,'Cleanup done.'
		end
'OYSTER': 	begin
		amberinitializepipe
		print,'OYSTER data and recipes initialized.'
		print,'J-band data will not be processed!'
		widget_control,pipeline_wid,/sensitive
		oyster
		end
'HELP':		begin
		print,'-----------------------------------------'
		print,'Observation: View and select data files'
		print,'P2VM Matrix: select files to compute P2VM'
		print,'P2VM Tracker: list file names of P2VMs'
		print,''
		print,'To run the pipeline, initialize w/OYSTER'
		print,''
		print,'Pipeline: processes all AMBER files' 
		print,'selected with "Observation" and stores'
	       	print,'results in OYSTER scans structure.'
	        print,'Data should be of same configuration.'
		print,''
		print,'Cleanup: delete all .AVG and .PHAS files'
		print,'----------------------------------------'
		end
'SmartMove':    begin
		print,'Reading files...'
                if n_elements(obht) gt 0 then begin
			index=where(strlen(strcompress(obht.issconfstation1, $
				/remove_all)) ne 2,count)
			if count gt 0 then begin
			spawn,'mkdir -p AMBER.auxiliary_engineering_files'
			for i=0,count-1 do spawn, $
				'mv '+obht(index(i)).filename $
					+' AMBER.auxiliary_engineering_files'
			endif
			index=where(strlen(strcompress(obht.issconfstation1, $
				/remove_all)) eq 2,count)
			if count gt 0 then obht=obht(index)
                        cleanamberrawdir,obht.filename
                endif else cleanamberrawdir
		print,'Moved files to AMBER.auxiliary_engineering_files.'
                end
'Move':		begin
		for i=0,n_elements(obht.filename)-1 do $
			print,obht(i).filename
		answer=' '
		read,prompt= $
		'Move these files to "AMBER.auxiliary_engineering_files"? '+ $
			'(y/n): ',answer
		if answer eq 'y' then begin
			spawn,'mkdir -p AMBER.auxiliary_engineering_files'
			for i=0,n_elements(obht.filename)-1 do begin
				spawn,'mv '+obht(i).filename+ $
					' AMBER.auxiliary_engineering_files'
			endfor
			configids=analyzeamberconfigids()
			widget_control,opmenuc_wid,set_value=configids, $
						  set_uvalue=configids
			configid=configids(0)
		endif
		end
'Delete':	begin
		index=where(strpos(obht.filename, $
			'AMBER.auxiliary_engineering_files') ne 0,count)
		if count gt 0 then begin
			obht=obht(index)
			for i=0,n_elements(obht.filename)-1 do $
				print,obht(i).filename
			answer=' '
			read,prompt='Delete these files? (yes/no): ',answer
			if answer eq 'yes' then begin
				spawn,'rm -f '+strjoin(obht.filename,' ')
				configids=analyzeamberconfigids()
				widget_control,opmenuc_wid, $
					set_value=configids, $
					set_uvalue=configids
				configid=configids(0)
			endif
			if answer ne 'yes' then print,'File(s) not removed.'
		endif
		end
'Rename':	begin
		for i=0,n_elements(obht.filename)-1 do begin
			fitsfile=obj_new('fitsfile',obht(i).filename)
			prihead=fitsfile->prihead()
			dateobs=strmid(strtrim(prihead->getpar('DATE-OBS')),0,23)
			obj_destroy,fitsfile
			outfile='AMBER.'+dateobs+'.fits'
			if obht(i).filename ne outfile then begin
				command='mv '+obht(i).filename+' '+outfile
				print,command
				spawn,command
			endif
		endfor
		end
'Reset':	begin
		spawn,'rm -f gorgonzola.sav'
		spawn,'rm -f obht.sav'
		spawn,'rm -f rawdirchecksum.sav'
		RawDirCheckSum=''
		if n_elements(obht) ne 0 then begin
			obht=obht(0)
			obht.filename=''
			widget_control,file_wid, $
				set_value=obht.filename,set_uvalue=obht.filename
		endif
		print,'Done.'
		end
'2P2V':		begin
		if n_elements(obht) eq 0 then begin
			print,'Please select files first!'
			return,-1
		endif
		p2vmindex=where(strtrim(obht.dprtype,2) eq '2P2V',count)
		if count le 5 then amber2p2vgui
		end
'3P2V':		begin
		if n_elements(obht) eq 0 then begin
			print,'Please select files first!'
			return,-1
		endif
		p2vmindex=where(strtrim(obht.dprtype,2) eq '3P2V',count)
		if count le 10 then begin
			checkp2vm,obht(p2vmindex).filename
			amber2p2vgui
		endif
		end
'OBJECT': 	begin
		if n_elements(obht) eq 0 then begin
			print,'Please select files first!'
			return,-1
		endif
		if objindex(0) ne -1 then amberobjectgui $
				     else print,'***Error: no objects selected!'
		end
'WAVE':		begin
		amdlibcomputespectralcalibration
		print,'Done.'
		end
'DARK':		begin
		n=n_elements(obht.filename)
		date=strmid(obht(n-1).filename,6,10)
		ins=obht.insmode
		p2v=obht.p2vmid
		dit=strarr(n)
		for i=0,n-1 do begin
			fitsfile=obj_new('fitsfile',obht(i).filename)
			prihead=fitsfile->prihead()
			dit(i)=strtrim(string(prihead->getpar('DET DIT'),format='(f6.3)'),2)
			obj_destroy,prihead
			obj_destroy,fitsfile
		endfor
		config=ins+'+' $
		      +string(p2v,format='(i8.8)')+'+' $
		      +dit
		configs=unique(config)
		m=n_elements(configs)
		dits=fltarr(m)
		for i=0,m-1 do begin
			print,'Analyzing '+configs(i)+'...'
			words=nameparse(configs(i),'+')
			filestub=date+'_'+configs(i)
			openw,unit,filestub+'.txt',/get_lun
			dits(i)=float(words(2))
			index=where(config eq configs(i),count)
			for j=0,count-1 do begin
				d=mrdfits(obht(index(j)).filename,1,header,/silent)
				d1=total(d.data1,3)/n_elements(d)
				d2=total(d.data2,3)/n_elements(d)
				d3=total(d.data3,3)/n_elements(d)
				d4=total(d.data4,3)/n_elements(d)
				d5=total(d.data5,3)/n_elements(d)
				dn=[d1,d2,d3,d4,d5]
				r=size(dn)
				if j eq 0 then dt=fltarr(r(1),r(2),count)
				dt(*,*,j)=dn
				if j eq 0 then begin
					d1m=d1
					d2m=d2
					d3m=d3
					d4m=d4
					d5m=d5
					d1s=d1^2
					d2s=d2^2
					d3s=d3^2
					d4s=d4^2
					d5s=d5^2
				endif else begin
					d1m=d1m+d1
					d2m=d2m+d2
					d3m=d3m+d3
					d4m=d4m+d4
					d5m=d5m+d5
					d1s=d1s+d1^2
					d2s=d2s+d2^2
					d3s=d3s+d3^2
					d4s=d4s+d4^2
					d5s=d5s+d5^2
				endelse
			endfor	
			d1m=d1m/count
			d2m=d2m/count
			d3m=d3m/count
			d4m=d4m/count
			d5m=d5m/count
			d1v=d1s/count-d1m^2
			d2v=d2s/count-d2m^2
			d3v=d3s/count-d3m^2
			d4v=d4s/count-d4m^2
			d5v=d5s/count-d5m^2
;
			print,'Results for DIT[ms]=',dits(i),', ',count,' files'
			mv=medianve(d1m,me)
			k=where(abs(d1m-mv) lt me)
			mvs=string(mv,format='(f6.2)')
			mes=string(me,format='(f6.2)')
			vmax=string(max(d1v(k)),format='(f7.2)')
			vmed=string(median(d1v(k)),format='(f7.3)')
			print,'Window 1: median/error='+mvs+'/'+mes $
				+', variance max/median='+vmax+'/'+vmed
			d1med=median(d1,3) & d1med=d1med-mean(d1med)
			psd_d1med=abs(fft(d1med))
			results=[mvs,mes,vmax,vmed,max(psd_d1med)]
			printf,unit,results
			mv=medianve(d2m,me)
			k=where(abs(d2m-mv) lt me)
			mvs=string(mv,format='(f6.2)')
			mes=string(me,format='(f6.2)')
			vmax=string(max(d2v(k)),format='(f7.2)')
			vmed=string(median(d2v(k)),format='(f7.3)')
			print,'Window 2: median/error='+mvs+'/'+mes $
				+', variance max/median='+vmax+'/'+vmed
			d2med=median(d2,3) & d2med=d2med-mean(d2med)
			psd_d2med=abs(fft(d2med))
			results=[mvs,mes,vmax,vmed,max(psd_d2med)]
			printf,unit,results
			mv=medianve(d3m,me)
			k=where(abs(d3m-mv) lt me)
			mvs=string(mv,format='(f6.2)')
			mes=string(me,format='(f6.2)')
			vmax=string(max(d3v(k)),format='(f7.2)')
			vmed=string(median(d3v(k)),format='(f7.3)')
			print,'Window 3: median/error='+mvs+'/'+mes $
				+', variance max/median='+vmax+'/'+vmed
			d3med=median(d3,3) & d3med=d3med-mean(d3med)
			psd_d3med=abs(fft(d3med))
			results=[mvs,mes,vmax,vmed,max(psd_d3med)]
			printf,unit,results
			mv=medianve(d4m,me)
			k=where(abs(d4m-mv) lt me)
			mvs=string(mv,format='(f6.2)')
			mes=string(me,format='(f6.2)')
			vmax=string(max(d4v(k)),format='(f7.2)')
			vmed=string(median(d4v(k)),format='(f7.3)')
			print,'Window 4: median/error='+mvs+'/'+mes $
				+', variance max/median='+vmax+'/'+vmed
			d4med=median(d4,3) & d4med=d4med-mean(d4med)
			psd_d4med=abs(fft(d4med))
			results=[mvs,mes,vmax,vmed,max(psd_d4med)]
			printf,unit,results
			mv=medianve(d5m,me)
			k=where(abs(d5m-mv) lt me)
			mvs=string(mv,format='(f6.2)')
			mes=string(me,format='(f6.2)')
			vmax=string(max(d5v(k)),format='(f7.2)')
			vmed=string(median(d5v(k)),format='(f7.3)')
			print,'Window 5: median/error='+mvs+'/'+mes $
				+', variance max/median='+vmax+'/'+vmed
			d5med=median(d5,3) & d5med=d5med-mean(d5med)
			psd_d5med=abs(fft(d5med))
			results=[mvs,mes,vmax,vmed,max(psd_d5med)]
			printf,unit,results
;
			free_lun,unit
;
			save,dt,filename=filestub+'.xdr'
;
			dm=[d1m,d2m,d3m,d4m,d5m]
			atv,dm,/histeq
		endfor
		end
;
endcase
;
end
;-------------------------------------------------------------------------------
function ambergui_configid,event
;
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Set configid
r=size(event)
if r(n_elements(r)-2) eq 8 then begin
	widget_control,event.id,get_uvalue=configids
	configid=configids(event.index)
endif else configid=event
;
; Load buffered OYSTER observation
if n_elements(geninfo) eq 0 or n_elements(recipes) eq 0 then begin
	print,'Warning: OYSTER not initialized'
	return,-1
endif
index=where(geninfo.configid eq configid)
datum=geninfo(index).date
array=geoinfo(index).systemid
loadnight,datum,array,configid
recipes_file='Recipes_'+configid+'.xdr'
restore,recipes_file
print,'Restored recipes ',recipes_file
selectamberfiles,/all
;
end
;-------------------------------------------------------------------------------
pro ambergui_frvalu,event
;
common AmberOptions,amber_options,p2vmfile
;
widget_control,event.id,get_value=frvalu
amber_options.frvalu=frvalu
print,'Fringe criterion set to: ',amber_options.frcrit+' '+amber_options.frvalu
;
end
;-------------------------------------------------------------------------------
pro ambergui_selvalu,event
;
; New for amdlib 2.0 beta-2
;
common AmberOptions,amber_options,p2vmfile
;
widget_control,event.id,get_uvalue=command,get_value=selvalu
if command eq 'selValue1' then amber_options.selvalu(0)=selvalu
if command eq 'selValue2' then amber_options.selvalu(1)=selvalu
if command eq 'selValue3' then amber_options.selvalu(2)=selvalu
print,command+' set to: '+selvalu
;
end
;-------------------------------------------------------------------------------
function ambergui_options,event
;
common AmberOptions,amber_options,p2vmfile
;
widget_control,event.id,get_uvalue=command,get_value=values
if command eq 'piston' then amber_options.piston=values(event.index)
if command eq 'errors' then amber_options.errors=values(event.index)
if command eq 'merges' then amber_options.merges=values(event.index)
if command eq 'frsel'  then amber_options.frcrit=values(event.index)
;
; amdlib 2.0 beta-2
if command eq 'selcrit1' then amber_options.selcrit(0)=values(event.index)
if command eq 'selcrit2' then amber_options.selcrit(1)=values(event.index)
if command eq 'selcrit3' then amber_options.selcrit(2)=values(event.index)
if command eq 'seltype1' then amber_options.seltype(0)=values(event.index)
if command eq 'seltype2' then amber_options.seltype(1)=values(event.index)
if command eq 'seltype3' then amber_options.seltype(2)=values(event.index)
if command eq 'selvalu1' then amber_options.selvalu(0)=values(event.index)
if command eq 'selvalu2' then amber_options.selvalu(1)=values(event.index)
if command eq 'selvalu3' then amber_options.selvalu(2)=values(event.index)
;
if values(event.index) eq 'Flux' then print,'For Flux, threshold is in units of noise.'
if values(event.index) eq 'Piston' then print,'For Piston, threshold is in units of microns.'
;
end
;-------------------------------------------------------------------------------
function files_event,event
;
; When clicking on a file, start fv on it.
;
widget_control,event.id,get_uvalue=files
if n_elements(files) ne 0 then $
spawn,'fv -cmap 2 '+files(event.index)+' &'
;
end
;-------------------------------------------------------------------------------
pro myambergui,dir
;
; Creates the MyAmberiGui main widget.
;
common AmberGuiWids,amber_wid,file_wid,window_slide_wid,pipeline_wid,opmenuc_wid
common AmberTaskWid,amber2p2v_wid,amberobject_wid,amberp2vm_wid,amberrecipes_wid
common AmberOptions,amber_options,p2vmfile
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
;
; Initialization and cleanup
if n_elements(amber_wid) eq 0 then amber_wid=0L
if n_elements(amber2p2v_wid) eq 0 then amber2p2v_wid=0L
if n_elements(amberobject_wid) eq 0 then amberobject_wid=0L
if n_elements(amberp2vm_wid) eq 0 then amberp2vm_wid=0L
if n_elements(amberrecipes_wid) eq 0 then amberrecipes_wid=0L
if widget_info(amber_wid,/valid) then widget_control,amber_wid,/destroy
if widget_info(amber2p2v_wid,/valid) then widget_control,amber2p2v_wid,/destroy
if widget_info(amberobject_wid,/valid) then widget_control,amberobject_wid,/destroy
if widget_info(amberp2vm_wid,/valid) then widget_control,amberp2vm_wid,/destroy
if widget_info(amberrecipes_wid,/valid) then widget_control,amberrecipes_wid,/destroy
p2vmfile=''
;
if total(strlen(findfile('rawdirchecksum'))) ne 0 then restore,'rawdirchecksum'
;
RawDir=''
if n_elements(dir) eq 0 then spawn,'pwd',RawDir else RawDir=dir
RawDir=RawDir(0)+'/'
if total(strlen(findfile(RawDir))) eq 0 then begin
	print,'***Error(MYAMBERGUI): raw file directory non-existent!'
	return
endif
;
; This is used to detect changes to the raw directory
if n_elements(RawDirCheckSum) eq 0 then RawDirCheckSum=''
;
; Allocate options
amber_options=alloc_amber_options()
spawn,'amdlibComputeOiData -h',result
ipos=where(strpos(result,'amdlib') ge 0 and strpos(result,'version') ge 0)
words=nameparse(result(ipos))
if not isnumeric(words(1)) then words(1)=words(2)
print,'Installed version of amdlib: '+words(1)
amber_options.version=float(words(1))
if fix(amber_options.version) ne 3 then begin
	print,'Sorry, amdlib version must be 3.0.9!'
	print,'Please get amdlib from Jean-Marie Mariotti Center.'
	retall
endif
;
amber_wid=widget_base(/column,title=RawDir(0))
;
templates=['Observation', $
	   'P2VM Matrix', $
	   'P2VM Tracker']
button_wid=cw_bgroup(amber_wid,/row,templates, $
	event_func='ambergui_event',/return_name)
;
templates=['2P2V', $
	   '3P2V', $
	   'DARK', $
	   'OBJECT', $
	   'WAVE']
button_wid=cw_bgroup(amber_wid,/row,templates, $
 	event_func='ambergui_event',/return_name)
;
templates=['AMBER','FINITO','COUDE','AMBIENT']
button_wid=cw_bgroup(amber_wid,/row,templates, $
	event_func='ambergui_event',/return_name)
;
file_wid=widget_list(amber_wid,value=strarr(10)+' ', $
	event_func='files_event', $
	xsize=15,ysize=10)
;
templates=['Reset','SmartMove','Move','Delete','Rename']
button_wid=cw_bgroup(amber_wid,/row,templates, $
	event_func='ambergui_event',/return_name)
;
col_wid=widget_base(amber_wid,/col,/frame)
;
configids=analyzeamberconfigids()
opmenuc_wid=widget_droplist(col_wid,event_func='ambergui_configid', $
	uvalue=configids,value=configids)
configid=configids(0)
;
row_wid=widget_base(col_wid,/row)
;
templates=['Pipeline','Recipes','Cleanup']
pipeline_wid=cw_bgroup(row_wid,/row,templates, $
	event_func='ambergui_event',/return_name)
widget_control,pipeline_wid,sensitive=0
;
if strlen(configid) gt 0 then begin
templates=['OYSTER','HELP']
pipeline2_wid=cw_bgroup(row_wid,/row,templates, $
	event_func='ambergui_event',/return_name)
endif else print,'Warning: no science data found!'
;
row2_wid=widget_base(col_wid,/row,/frame)
;
piston=['PHASE','PHASOR']
opmenue_wid=widget_droplist(row2_wid,event_func='ambergui_options', $
	uvalue='piston',value=piston)
;
errors=['STATISTIC','THEORIC']
opmenup_wid=widget_droplist(row2_wid,event_func='ambergui_options', $
	uvalue='errors',value=errors)
;
merges=['>','>>']
opmenup_wid=widget_droplist(row2_wid,event_func='ambergui_options', $
	uvalue='merges',value=merges)
;
; amdlib 2.0 beta-1
if 0 then begin
frcrit=['FRG','FLUX']
opmenuf_wid=widget_droplist(row2_wid,event_func='ambergui_options', $
	uvalue='frsel',value=frcrit)
;
textfr_wid=widget_text(row2_wid,event_pro='ambergui_frvalu', $
	/editable,xsize=3,value=amber_options.frvalu)
endif
;
; amdlib 2.0 beta-2
selcrit=['None','Flux','Piston','SNR']
seltype=['Threshold','Percent']
;
row3_wid=widget_base(col_wid,/row)
opmenu1crit_wid=widget_droplist(row3_wid,event_func='ambergui_options', $
	uvalue='selcrit1',value=selcrit)
opmenu1type_wid=widget_droplist(row3_wid,event_func='ambergui_options', $
	uvalue='seltype1',value=seltype)
txt1selvalu_wid=widget_text(row3_wid,event_pro='ambergui_selvalu', $
	/editable,xsize=4,uvalue='selValue1', $
	value=string(amber_options.selvalu(0),format='(f4.1)'))
;
row4_wid=widget_base(col_wid,/row)
opmenu2crit_wid=widget_droplist(row4_wid,event_func='ambergui_options', $
	uvalue='selcrit2',value=selcrit)
opmenu2type_wid=widget_droplist(row4_wid,event_func='ambergui_options', $
	uvalue='seltype2',value=seltype)
txt2selvalu_wid=widget_text(row4_wid,event_pro='ambergui_selvalu', $
	/editable,xsize=4,uvalue='selValue2', $
	value=string(amber_options.selvalu(1),format='(f4.1)'))
;
row5_wid=widget_base(col_wid,/row)
opmenu3crit_wid=widget_droplist(row5_wid,event_func='ambergui_options', $
	uvalue='selcrit3',value=selcrit)
opmenu3type_wid=widget_droplist(row5_wid,event_func='ambergui_options', $
	uvalue='seltype3',value=seltype)
txt3selvalu_wid=widget_text(row5_wid,event_pro='ambergui_selvalu', $
	/editable,xsize=4,uvalue='selValue3', $
	value=string(amber_options.selvalu(2),format='(f4.1)'))
;
widget_control,amber_wid,/realize
xmanager,'myambergui',amber_wid,/no_block
;
end
;-------------------------------------------------------------------------------
function amber2p2vgui_options,event
;
common AmberP2vmOptions,amberp2vm_options
;
case event.value of
	0: amberp2vm_options.c=event.select
endcase
;
end
;-------------------------------------------------------------------------------
function amber2p2vgui_event,event
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common Amber2p2vGuiWids,draw_wid
common AmberP2vmOptions,amberp2vm_options
;
; OYSTER common blocks
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
widget_control,event.id,get_uvalue=shutters
shutter=shutters(event.index)
;
widget_control,draw_wid,get_value=id
wset,id
erase
;
index=where(shutters eq shutter)
if amberp2vm_options.c then begin
	bpix=badpixelmap(obht)
	flat=flatfieldmap(obht)
	dark=obht(p2vmindex(0)).filename
	command='amdlibCalibrateRawData '+bpix+' '+flat+' '+dark+' 1 ' $
		+obht(p2vmindex(index)).filename+' cosmetic.p2vm.fits'
	command='amdlibCalibrateRawData '+bpix+' '+flat+' '+dark+' ' $
		+obht(p2vmindex(index)).filename+' cosmetic.p2vm.fits'
	spawn,command
	filename='cosmetic.p2vm.fits'
endif else filename=obht(p2vmindex(index)).filename
data=oirgetdata(filename)
;
image1=bytscl(total(data.data1,3))
image2=bytscl(total(data.data2,3))
image3=bytscl(total(data.data3,3))
image4=bytscl(total(data.data4,3))
image=[image1,image2,image3,image4]
;
; Display in GUI
loadct,0
tvsclm,image,2
;
end
;-------------------------------------------------------------------------------
pro amber2p2vgui
;
; GUI for P2VM analysis.
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberTaskWid,amber2p2v_wid,amberobject_wid,amberp2vm_wid,amberrecipes_wid
common Amber2p2vGuiWids,draw_wid
common AmberOptions,amber_options,p2vmfile
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberP2vmOptions,amberp2vm_options
;
; Remove any existing widget
if widget_info(amber2p2v_wid,/valid) then widget_control,amber2p2v_wid,/destroy
;
amberp2vm_options={c:0}
;
; Obtain shutter information
p2vmimages=obht.filename
n=n_elements(p2vmimages)
shutters=strarr(n)
for i=0,n-1 do begin
	fitsfile=obj_new('fitsfile',p2vmimages(i))
	prihead=fitsfile->prihead()
	shutter1=string(prihead->getpar('INS SHUT1 ST'))
	shutter2=string(prihead->getpar('INS SHUT2 ST'))
	shutter3=string(prihead->getpar('INS SHUT3 ST'))
	shutters(i)='Shutters '+strcompress(shutter1+shutter2+shutter3,/remove_all)
	obj_destroy,prihead
	obj_destroy,fitsfile
endfor
;
; Now create GUI
amber2p2v_wid=widget_base(/col, $
	title='P2VM for '+configid)
row_wid=widget_base(amber2p2v_wid,/row)
;
opmenu_wid=widget_droplist(row_wid,event_func='amber2p2vgui_event', $
	uvalue=shutters,value=shutters)
;
options=['Cosmetics']
options_wid=cw_bgroup(row_wid,options,/row,/nonexclusive, $
	event_func='amber2p2vgui_options')
;
draw_wid=widget_draw(amber2p2v_wid,event_pro='acqdraw_event', $
	scr_xsize=62*4,scr_ysize=69*2)
;
widget_control,amber2p2v_wid,/realize
xmanager,'amber2p2vgui2',amber2p2v_wid,/no_block
;
end
;-------------------------------------------------------------------------------
function amberp2vmfiles_event,event
;
; Call-back for clicks on mask files in the amberp2vmgui.
;
common AmberOptions,amber_options,p2vmfile
common AmberP2vmGuiWids,p2vmfile_wid,draw_wid
;
widget_control,event.id,get_uvalue=files
p2vmfile=files(event.index) & p2vmfile=p2vmfile(0)
;
print,'Selected p2vm file: ',p2vmfile
;
widget_control,draw_wid,get_value=id
wset,id
;
end
;-------------------------------------------------------------------------------
function amberp2vmgui_event,event
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberP2vmGuiWids,p2vmfile_wid,draw_wid
common AmberOptions,amber_options,p2vmfile
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
;
widget_control,draw_wid,get_value=id
wset,id
;
case event.value of
;
'Update/Clear':	amberp2vmgui,/update
'Check shift' : begin
		if strlen(p2vmfile) eq 0 then begin
			print,'Error: please select P2VM file!'
			return,-1
		endif
		spawn,'rm -f batch.i'
		openw,unit,'batch.i',/get_lun
		printf,unit,'#include "amdlib.i"'
		bpix=badpixelmap(obht)
		flat=flatfieldmap(obht)
		printf,unit,'amdlibLoadFlatFieldMap, inputFlatFieldFile="'+flat+'"'
		printf,unit,'amdlibLoadBadPixelMap, inputBadPixelFile="'+bpix+'"'
		command='amdlibCheckSpectralShift,' $
		       +'inputRawFile="'+obht(objindex(0)).filename+'",' $
		       +'inputBiasFile="'+obht(darkindex(0)).filename+'",' $
		       +'inputP2vmFile="'+p2vmfile+'"'
		printf,unit,command
		printf,unit,'pause,50000;'
		printf,unit,'quit;'
		free_lun,unit
		print,'Calling yorick...(in background)'
		spawn,'yorick -batch batch.i &'
		end
'Store':	begin
		print,'Not yet implemented.'
		end
;
endcase
;
end
;-------------------------------------------------------------------------------
pro amberp2vmgui,update=update
;
; GUI to select and manage p2vm files.
;
common AmberOptions,amber_options,p2vmfile
common AmberP2vmGuiWids,p2vmfile_wid,draw_wid
common AmberTaskWid,amber2p2v_wid,amberobject_wid,amberp2vm_wid,amberrecipes_wid
;
if n_elements(update) eq 0 then update=0
;
p2vmfile=''
spawn,'pwd',currentdir
;
; First, find all p2vm files in current directory
p2vmfiles=findfile('*.p2vm.fits')
n=n_elements(p2vmfiles)
if strlen(p2vmfiles(0)) eq 0 then begin
	print,'No P2VM files found in '+currentdir+'!'
endif
;
if update and widget_info(amberp2vm_wid,/valid) then begin
	widget_control,p2vmfile_wid,set_value=p2vmfiles,set_uvalue=p2vmfiles
	p2vmfile=''
	print,'P2VM file selection cleared.'
endif else begin
;
; Remove any existing widget
if widget_info(amberp2vm_wid,/valid) then widget_control,amberp2vm_wid,/destroy
;
amberp2vm_wid=widget_base(/col,title='P2VM Tracker '+currentdir)
;
draw_wid=widget_draw(amberp2vm_wid,event_pro='p2vmdraw_event', $
	scr_xsize=300,scr_ysize=200)
;
p2vmfile_wid=widget_list(amberp2vm_wid,value=p2vmfiles,uvalue=p2vmfiles, $
        event_func='amberp2vmfiles_event',xsize=25,ysize=10)
;
tools=['Update/Clear','Check shift','Store']
button_wid=cw_bgroup(amberp2vm_wid,/row,tools, $
	event_funct='amberp2vmgui_event',/return_name)
;
widget_control,amberp2vm_wid,/realize
xmanager,'amberp2vmgui',amberp2vm_wid,/no_block
;
endelse
;
end
;-------------------------------------------------------------------------------
function amberrecipes_event,event
;
; Call-back for clicks on recipes files in the amberrecipesgui.
;
common AmberOptions,amber_options,p2vmfile
common AmberRecipesGuiWids,recipes_wid,recipetext_wid
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
;
widget_control,event.id,get_uvalue=tags
tag=tags(event.index) & tag=tag(0)
;
index=where(recipes.tag eq tag)
fields=tag_names(recipes)
for i=0,n_elements(fields)-1 do fields(i)=fields(i)+'='+recipes(index).(i)
widget_control,recipetext_wid,set_value=fields
;
end
;-------------------------------------------------------------------------------
function amberrecipesgui_event,event
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberOptions,amber_options,p2vmfile
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberRecipesGuiWids,recipes_wid,recipetext_wid
common AmberP2vmGuiWids,p2vmfile_wid,draw_wid
;
case event.value of
;
'Update':	begin
		tags=recipes.tag
		widget_control,recipes_wid,set_value=tags
		end
'Store':	begin
		filename='Recipes_'+configid+'.xdr'
		save,recipes,filename=filename
		print,'Saved recipes to: ',filename
		end
'Get P2VM':	begin
		if n_elements(p2vmfile_wid) eq 0 then p2vmfile_wid=0
		if widget_info(p2vmfile_wid,/valid) eq 0 then begin
			print,'***Error: Please start the P2VM tracker first!'
			return,-1
		endif
		widget_control,p2vmfile_wid,get_uvalue=p2vmfiles
		recipes.p2vm_file=p2vmfiles( $
		      widget_info(p2vmfile_wid,/list_select))
		print,'Stored '+p2vmfiles(0)+' in all recipes.'
		end
;
endcase
;
end
;-------------------------------------------------------------------------------
pro amberrecipesgui
;
; GUI to select and manage recipes.
;
common AmberOptions,amber_options,p2vmfile
common AmberP2vmGuiWids,p2vmfile_wid,draw_wid
common AmberTaskWid,amber2p2v_wid,amberobject_wid,amberp2vm_wid,amberrecipes_wid
common AmberRecipesGuiWids,recipes_wid,recipetext_wid
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
;
; Remove any existing widget
if widget_info(amberrecipes_wid,/valid) then widget_control,amberrecipes_wid,/destroy
;
amberrecipes_wid=widget_base(/col,title='Recipes')
;
amberrow_wid=widget_base(amberrecipes_wid,/row)
;
tags=recipes.tag
;
recipes_wid=widget_list(amberrow_wid,value=tags,uvalue=tags, $
        event_func='amberrecipes_event',xsize=25,ysize=10)
;
tools=['Update','Get P2VM','Store']
button_wid=cw_bgroup(amberrow_wid,/col,tools, $
	event_funct='amberrecipesgui_event',/return_name)
;
recipetext_wid=widget_text(amberrecipes_wid,value='Click to see recipe', $
	/scroll,ysize=8)
;
widget_control,amberrecipes_wid,/realize
xmanager,'amberrecipesgui',amberrecipes_wid,/no_block
;
end
;-------------------------------------------------------------------------------
function vis_row_files,vis_file
;
; Both with amdlib pre-2.0 and the new amdlib with the -s option,
; different rows (i.e. bands) are written into separate files.
;
n=strlen(vis_file)
vis_files=findfile(strmid(vis_file,0,n-5)+'*')
;
; The file format has changed with DRS_2.0_Beta_1, and we now use ComputeOiData.
; if strlen(vis_files(0)) ne 0 then begin
; 	fitsfile=obj_new('fitsfile',vis_files(0))
; 	prihead=fitsfile->prihead()
; 	program=strtrim(prihead->getpar('PRO REC1 ID'))
; 	obj_destroy,prihead
; 	obj_destroy,fitsfile
; 	if program ne 'amdlibComputeOiData' then vis_files=''
; endif
;
if n_elements(vis_files) eq 3 then begin
	print,'Only using bands H and K!'
	vis_files=vis_files([0,2])
endif
;
return,vis_files
;
end
;-------------------------------------------------------------------------------
function alt_dest_file,dest_file
;
idot=strpos(dest_file,'.')
target=strmid(dest_file,0,idot)
objectfile=strmid(dest_file,idot+1,strlen(dest_file)-idot-1)
;
spawn,'dfits *.fits | fitsort PIPEFILE DATE-OBS OBS.TARG.NAME',r
i1=0
i2=strpos(r(0),'PIPEFILE')
i3=strpos(r(0),'DATE-OBS')
i4=strpos(r(0),'OBS.TARG.NAME')
if 0 then begin
for j=0,n_elements(r)-1 do begin
	seconds=strcompress(strmid(r(j),i3+17,7),/remove_all)
	if strlen(seconds) ne 0 then begin
	rj=r(j)
	seconds=float(seconds)+0.00001
	strput,rj,string(seconds,format='(f6.3)')+' ',i3+17
	if seconds lt 10 then strput,rj,'0',i3+17
	r(j)=rj
	endif
endfor
endif
t_obs=hms2h(strmid(r,i3+11,13))
index=where(strlen(strcompress(strmid(r,i2,i3-i2),/remove)) ne 0 $
	and abs(t_obs-hms2h(strmid(objectfile,17,12))) lt 0.0000005 $
	and strpos(r,target) ge 0,count)
if count eq 1 then file_dest=strmid(r(index),i1,i2-i1) $
	      else file_dest=''
;
return,strcompress(file_dest,/remove_all)
;
end
;-------------------------------------------------------------------------------
function amberobjectgui_event,event
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberOptions,amber_options,p2vmfile
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberObjectGuiWids,file_wid,draw_wid,c_wid,qc_wid,fp_wid
common AmberObjectGui,objectfiles,row,channel,baseline,threshold
common AmberData,amber_data,oi_array,oi_wavelength,oi_vis2,oi_t3
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(p2vmfile) eq 0 then p2vmfile=''
;
widget_control,event.id,get_uvalue=values
if n_elements(values) eq 1 then value=event.value else value=values(event.index)
;
if objindex(0) lt 0 then return,-1
target=strtrim(obht(objindex(0)).targname,2)
;
widget_control,draw_wid,get_value=id
wset,id
set_screen
!p.charsize=1
if total(abs(!y.range)) ne 0 or total(abs(!x.range)) ne 0 $
	then user_range=1 $
	else user_range=0
;
case value of
'All selected files': begin
		objectfiles=obht(objindex).filename
;		widget_control,fp_wid,sensitive=0
		end
'J':		begin
		row=0
		if n_elements(oi_wavelength) ne 0 then begin
		r=size(oi_wavelength.j)
		if r(n_elements(r)-2) ne 8 then begin
			print,'No J data present!'
			widget_control,fp_wid,sensitive=0
			return,-1
		endif
		max_channel=n_elements(oi_wavelength.(row)(0).d(*).eff_wave)
		widget_control,c_wid,set_value=[1,1,max_channel]
		widget_control,fp_wid,/sensitive
		endif
		end
'H':		begin
		row=1
		if n_elements(oi_wavelength) ne 0 then begin
		r=size(oi_wavelength.h)
		if r(n_elements(r)-2) ne 8 then begin
			print,'No H data present!'
			widget_control,fp_wid,sensitive=0
			return,-1
		endif
		max_channel=n_elements(oi_wavelength.(row)(0).d(*).eff_wave)
		widget_control,c_wid,set_value=[1,1,max_channel]
		widget_control,fp_wid,/sensitive
		endif
		end
'K':		begin
		row=2
		if n_elements(oi_wavelength) ne 0 then begin
		r=size(oi_wavelength.k)
		if r(n_elements(r)-2) ne 8 then begin
			print,'No K data present!'
			widget_control,fp_wid,sensitive=0
			return,-1
		endif
		max_channel=n_elements(oi_wavelength.(row)(0).d(*).eff_wave)
		widget_control,c_wid,set_value=[1,1,max_channel]
		widget_control,fp_wid,/sensitive
		endif
		end
'Cleanup':	begin
		for i=0,n_elements(objectfiles)-1 do begin
			dest_file=target+'.'+ $
				  amber_options.piston+'.'+ $
				  amber_options.errors+'.'+ $
				  objectfiles(i)
			vis_files=vis_row_files(dest_file)
			for j=0,n_elements(vis_files)-1 do spawn,'rm -f '+vis_files(j)
		endfor
		print,'Cleanup done.'
		end
'ExtractVis':	begin
		if strlen(p2vmfile) eq 0 then begin
			print,'***Error: no P2VM file selected!'
			return,-1
		endif
		n=n_elements(objectfiles)
		for i=0,n-1 do begin
			dest_file=target+'.'+ $
				  amber_options.piston+'.'+ $
				  amber_options.errors+'.'+ $
				  objectfiles(i)
			vis_files=vis_row_files(dest_file)
;			Check for gasgano/reflex product files, else extract vis
;			if strlen(vis_files(0)) eq 0 then file_dest=alt_dest_file(dest_file)
			if strlen(vis_files(0)) eq 0 then begin
				amdlibcomputeoidata,objectfiles(i),dest_file
				vis_files=vis_row_files(dest_file)
			endif
			m=n_elements(vis_files)
			row_index=intarr(m)
			for j=0,m-1 do begin
				d=mrdfits(vis_files(j),3,header)
				j_obs=total(johnson_j(d.eff_wave*1e9))
				h_obs=total(johnson_h(d.eff_wave*1e9))
				k_obs=total(johnson_k(d.eff_wave*1e9))
				index=where([j_obs,h_obs,k_obs] eq max([j_obs,h_obs,k_obs]))
				row_index(j)=index(0)
			endfor
			if i eq 0 then begin
				oi_array=mrdfits(vis_files(0),1,header)
				if n_elements(oi_array) eq 2 then extns=[3,5,6]
				if n_elements(oi_array) eq 3 then extns=[3,5,7,6]
;				Wavelength
				j=0 & h=0 & k=0
				for l=0,m-1 do begin
					d={d:mrdfits(vis_files(l),extns(0))}
					if row_index(l) eq 0 then j=d
					if row_index(l) eq 1 then h=d
					if row_index(l) eq 2 then k=d
				endfor
				oi_wavelength={j:replicate(j,n),h:replicate(h,n),k:replicate(k,n)}
;				Squared visibility
				j=0 & h=0 & k=0
				for l=0,m-1 do begin
					d={d:mrdfits(vis_files(l),extns(1))}
					if row_index(l) eq 0 then j=d
					if row_index(l) eq 1 then h=d
					if row_index(l) eq 2 then k=d
				endfor
				oi_vis2={j:replicate(j,n),h:replicate(h,n),k:replicate(k,n)}
;				Amber data
				j=0 & h=0 & k=0
				for l=0,m-1 do begin
					d={d:mrdfits(vis_files(l),extns(2))}
					d.d.fringe_snr=0
					if row_index(l) eq 0 then j=d
					if row_index(l) eq 1 then h=d
					if row_index(l) eq 2 then k=d
				endfor
				amber_data={j:replicate(j,n),h:replicate(h,n),k:replicate(k,n)}
;				Triple products
				if n_elements(extns) eq 4 then begin
				j=0 & h=0 & k=0
				for l=0,m-1 do begin
					d={d:mrdfits(vis_files(l),extns(3))}
					if row_index(l) eq 0 then j=d
					if row_index(l) eq 1 then h=d
					if row_index(l) eq 2 then k=d
				endfor
				oi_t3={j:replicate(j,n),h:replicate(h,n),k:replicate(k,n)}
				endif
			endif
			for j=0,m-1 do begin
				k=row_index(j)
				d={d:mrdfits(vis_files(j),extns(0))}
				oi_wavelength.(k)(i)=d
				d={d:mrdfits(vis_files(j),extns(1))}
				ds=oi_vis2.(k)(i)
				struct_assign,d,ds
				oi_vis2.(k)(i)=ds
				d={d:mrdfits(vis_files(j),extns(2))}
				ds=amber_data.(k)(i)
				struct_assign,d,ds
				amber_data.(k)(i)=ds
				if n_elements(extns) eq 4 then begin
				d={d:mrdfits(vis_files(j),extns(3))}
				ds=oi_t3.(k)(i)
				struct_assign,d,ds
				oi_t3.(k)(i)=ds
				endif
			endfor
		endfor
		max_channel=n_elements(oi_wavelength.(row)(0).d(*).eff_wave)
		widget_control,c_wid,/sensitive
		widget_control,c_wid,set_value=[0,0,max_channel]
;		help,oi_wavelength.(0)(0).d(0).eff_wave
		widget_control,qc_wid,/sensitive
		widget_control,fp_wid,/sensitive
		widget_control,file_wid,/sensitive
		end
'Merge':	begin
		words=nameparse(objectfile(0),'.')
		outfile=words(0)+'.'+words(1)+'.All.fits'
		amdlibappendoidata,objectfiles,outfile
		end
'V2 vs SNR':	begin
		!x.title='SNR'
		!y.title='V2'
		if not user_range then begin
		case strmid(oi_array(0).tel_name,0,1) of
		'U': !y.range=[0,0.5]
		'A': !y.range=[0,1.0]
		endcase
		case strmid(oi_array(0).tel_name,0,1) of
		'U': !x.range=[0,20.]
		'A': !y.range=[0,20.]
		endcase
		endif
		objectfiles0=obht(objindex).filename
		first=1
		for i0=0,n_elements(objectfiles)-1 do begin
		i=where(objectfiles0 eq objectfiles(i0))
		sta_index0=amber_data.k(i).d.sta_index(0)
		sta_index1=amber_data.k(i).d.sta_index(1)
		for j=0,n_elements(sta_index0)-1 do $
			sta_index0(j)=where(oi_array.sta_index eq sta_index0(j))
		for j=0,n_elements(sta_index1)-1 do $
			sta_index1(j)=where(oi_array.sta_index eq sta_index1(j))
		index=where(vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
		if index(0) eq -1 then $
		index=where(vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
;		x=amber_data.(row)(i).d(index).base_fringe_snr_dispersed
;		x=amber_data.(row)(i).d(index).base_fringe_snr
		x=amber_data.(row)(i).d(index).fringe_snr
		y=oi_vis2.(row)(i).d(index).vis2data
;		snr=amber_data.(row)(i).d(index).base_fringe_snr_dispersed
;		snr=amber_data.(row)(i).d(index).base_fringe_snr
		snr=amber_data.(row)(i).d(index).fringe_snr
		if channel lt 0 then begin
;			x=total(x,1)/n_elements(x(*,0))
			y=total(y,1)/n_elements(y(*,0))
;			snr=amber_data.(row)(i).d(index).base_fringe_snr
		endif else begin
;			x=x(channel,*)
			y=y(channel,*)
		endelse
		index=where(snr gt threshold and finite(snr) eq 1,count)
		if count gt 0 then begin
		if first then plot,x(index),y(index),psym=3 $
			 else oplot,x(index),y(index),psym=3,color=getcolor(i)
		first=0
		endif else print,'No data to plot!'
		endfor
		if not user_range then begin
			!x.range=0
			!y.range=0
		endif
		end
'V2 vs Piston':	begin
		!y.title='V2'
		!x.title='Piston [mm]'
		if not user_range then begin
		case strmid(oi_array(0).tel_name,0,1) of
		'U': !y.range=[0,0.5]
		'A': !y.range=[0,1.0]
		endcase
		!x.range=[-0.04,0.04]
		endif
		objectfiles0=obht(objindex).filename
		first=1
		for i0=0,n_elements(objectfiles)-1 do begin
		i=where(objectfiles0 eq objectfiles(i0))
		sta_index0=amber_data.k(i).d.sta_index(0)
		sta_index1=amber_data.k(i).d.sta_index(1)
		for j=0,n_elements(sta_index0)-1 do $
			sta_index0(j)=where(oi_array.sta_index eq sta_index0(j))
		for j=0,n_elements(sta_index1)-1 do $
			sta_index1(j)=where(oi_array.sta_index eq sta_index1(j))
		index=where(vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
		if index(0) eq -1 then $
		index=where(vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
;		x=amber_data.(row)(i).d(index).base_opd_dispersed
;		x=amber_data.(row)(i).d(index).base_opd*1e3
		x=amber_data.(row)(i).d(index).opd*1e3
		y=oi_vis2.(row)(i).d(index).vis2data
;		snr=amber_data.(row)(i).d(index).base_fringe_snr_dispersed
;		snr=amber_data.(row)(i).d(index).base_fringe_snr
		snr=amber_data.(row)(i).d(index).fringe_snr
		if channel lt 0 then begin
;			x=amber_data.(row)(i).d(index).base_opd*1e3
			y=total(y,1)/n_elements(y(*,0))
;			snr=amber_data.(row)(i).d(index).base_fringe_snr
		endif else begin
;			x=x(channel,*)*1e3
			y=y(channel,*)
		endelse
		index=where(snr gt threshold and finite(snr) eq 1,count)
		if count gt 0 then begin
		if first then plot,x(index),y(index),psym=3 $
			 else oplot,x(index),y(index),psym=3,color=getcolor(i)
		first=0
		endif else print,'No data to plot!'
		endfor
		if not user_range then begin
			!x.range=0
			!y.range=0
		endif
		end
'Piston vs SNR':begin
		!x.title='SNR'
		!y.title='Piston [mm]'
		if not user_range then begin
		!y.range=[-0.6,0.6]
		!x.range=0
		endif
		objectfiles0=obht(objindex).filename
		first=1
		for i0=0,n_elements(objectfiles)-1 do begin
		i=where(objectfiles0 eq objectfiles(i0))
		sta_index0=amber_data.k(i).d.sta_index(0)
		sta_index1=amber_data.k(i).d.sta_index(1)
		for j=0,n_elements(sta_index0)-1 do $
			sta_index0(j)=where(oi_array.sta_index eq sta_index0(j))
		for j=0,n_elements(sta_index1)-1 do $
			sta_index1(j)=where(oi_array.sta_index eq sta_index1(j))
		index=where(vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
		if index(0) eq -1 then $
		index=where(vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
;		x=amber_data.(row)(i).d(index).base_fringe_snr_dispersed
;		x=amber_data.(row)(i).d(index).base_fringe_snr
		x=amber_data.(row)(i).d(index).fringe_snr
;		y=amber_data.(row)(i).d(index).base_opd_dispersed
;		y=amber_data.(row)(i).d(index).base_opd*1e3
		y=amber_data.(row)(i).d(index).opd*1e3
;		snr=amber_data.(row)(i).d(index).base_fringe_snr_dispersed
;		snr=amber_data.(row)(i).d(index).base_fringe_snr
		snr=amber_data.(row)(i).d(index).fringe_snr
		if channel lt 0 then begin
;			x=amber_data.(row)(i).d(index).base_fringe_snr
;			y=amber_data.(row)(i).d(index).base_opd*1e3
;			snr=amber_data.(row)(i).d(index).base_fringe_snr
		endif else begin
;			x=x(channel,*)
;			y=y(channel,*)*1e3
		endelse
		index=where(snr gt threshold and finite(snr) eq 1,count)
		if count gt 0 then begin
		if first then plot,x(index),y(index),psym=3 $
			 else oplot,x(index),y(index),psym=3,color=getcolor(i)
		first=0
		endif else print,'No data to plot!'
		endfor
		if not user_range then begin
			!x.range=0
			!y.range=0
		endif
		end
'Piston vs Time':begin
		!x.title='Time [s]'
		!y.title='Piston [mm]'
		if not user_range then begin
		!y.range=[-0.02,0.02]
		!x.range=0
		endif
		objectfiles0=obht(objindex).filename
		first=1
		for i0=0,n_elements(objectfiles)-1 do begin
		i=where(objectfiles0 eq objectfiles(i0))
		sta_index0=amber_data.k(i).d.sta_index(0)
		sta_index1=amber_data.k(i).d.sta_index(1)
		for j=0,n_elements(sta_index0)-1 do $
			sta_index0(j)=where(oi_array.sta_index eq sta_index0(j))
		for j=0,n_elements(sta_index1)-1 do $
			sta_index1(j)=where(oi_array.sta_index eq sta_index1(j))
		index=where(vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
		if index(0) eq -1 then $
		index=where(vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
		x=amber_data.(row)(i).d(index).time & x=x-x(0)
;		y=amber_data.(row)(i).d(index).base_opd_dispersed
;		y=amber_data.(row)(i).d(index).base_opd*1e3
		y=amber_data.(row)(i).d(index).opd*1e3
;		snr=amber_data.(row)(i).d(index).base_fringe_snr_dispersed
;		snr=amber_data.(row)(i).d(index).base_fringe_snr
		snr=amber_data.(row)(i).d(index).fringe_snr
		if channel lt 0 then begin
;			y=amber_data.(row)(i).d(index).base_opd*1e3
;			snr=amber_data.(row)(i).d(index).base_fringe_snr
		endif else begin
;			y=y(channel,*)*1e3
		endelse
		ti=where(amber_data.(row)(i).d(index).time gt 0)
		index=where(snr(ti) gt threshold and finite(snr(ti)) eq 1,count)
		if count gt 0 then begin
		if first then plot,x(ti(index)),y(ti(index)),psym=3 $
			 else oplot,x(ti(index)),y(ti(index)),psym=3,color=getcolor(i)
		first=0
		endif else print,'No data to plot!'
		endfor
		if not user_range then begin
			!x.range=0
			!y.range=0
		endif
		end
'Piston Closure':begin
		!x.title='Time [s]'
		!y.title='Closure Piston [mm]'
		if not user_range then begin
		!y.range=0
		!x.range=0
		endif
		objectfiles0=obht(objindex).filename
		first=1
		for i0=0,n_elements(objectfiles)-1 do begin
		i=where(objectfiles0 eq objectfiles(i0))
		d=fltarr(n_elements(amber_data.(row)(i).d)/3)
		snr=fix(d)+1
		for j=0,2 do begin
			index=indgen(n_elements(d))*3+j
			x=amber_data.(row)(i).d(index).time & x=x-x(0)
;			y=amber_data.(row)(i).d(index).base_opd_dispersed
;			y=amber_data.(row)(i).d(index).base_opd*1e3
			y=amber_data.(row)(i).d(index).opd*1e3
;			s=amber_data.(row)(i).d(index).base_fringe_snr_dispersed
;			s=amber_data.(row)(i).d(index).base_fringe_snr
			s=amber_data.(row)(i).d(index).fringe_snr
			if channel lt 0 then begin
				x=amber_data.(row)(i).d(index).time & x=x-x(0)
;				y=amber_data.(row)(i).d(index).base_opd*1e3
;				s=amber_data.(row)(i).d(index).base_fringe_snr
			endif else begin
;				y=y(channel,*)*1e3
			endelse
			sndex=where(s lt threshold and finite(s) eq 1,count)
			if count gt 0 then snr(sndex)=0
			if j eq 0 then ti=where(amber_data.(row)(i).d(index).time gt 0)
			if j eq 2 then y=-y	; this could be a bug in amdlib
			d(ti)=d(ti)+y(ti)
		endfor
		index=where(snr(ti) eq 1,count)
		if count gt 0 then begin
		if first then plot,x(ti(index)),d(ti(index)),psym=3 $
			 else oplot,x(ti(index)),d(ti(index)),psym=3,color=getcolor(i)
		first=0
		endif else print,'No data to plot!'
		endfor
		if not user_range then begin
			!x.range=0
			!y.range=0
		endif
		end
'V2 Histogram':	begin
		if not user_range then begin
		!y.range=0
		endif
		objectfiles0=obht(objindex).filename
		for i0=0,n_elements(objectfiles)-1 do begin
		i=where(objectfiles0 eq objectfiles(i0))
		sta_index0=amber_data.k(i).d.sta_index(0)
		sta_index1=amber_data.k(i).d.sta_index(1)
		for j=0,n_elements(sta_index0)-1 do $
			sta_index0(j)=where(oi_array.sta_index eq sta_index0(j))
		for j=0,n_elements(sta_index1)-1 do $
			sta_index1(j)=where(oi_array.sta_index eq sta_index1(j))
		index=where(vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
		if index(0) eq -1 then $
		index=where(vlti_stationid(oi_array(sta_index1).sta_name) $
			eq strmid(baseline,0,3) $
			and vlti_stationid(oi_array(sta_index0).sta_name) $
			eq strmid(baseline,4,3) $
			and sta_index0 ge 0)
		y=oi_vis2.(row)(i).d(index).vis2data
		if channel lt 0 then begin
			y=total(y,1)/n_elements(y(*,0))
		endif else begin
			y=y(channel,*)
		endelse
		if i0 eq 0 then $
		histograph,y,min=0,max=1.2,binsize=0.1 else $
		histograph,y,min=0,max=1.2,binsize=0.1,/oplot,color=getcolor(i)
		endfor
		if not user_range then begin
			!x.range=0
			!y.range=0
		endif
		end
'DataQC':	begin
		if n_elements(amber_data) eq 0 then begin
			print,'Please compute visibilities first!'
			return,-1
		endif
		row=2	; We do QC with just the K band results
		n=n_elements(objectfiles)
		staindex0=amber_data.(row)(0).d.sta_index
		sta_index=amber_data.(row)(0).d.sta_index-1
;		The next line does not work anymore
;		if min(sta_index) ne 0 then sta_index=sta_index mod min(sta_index)
		for i=0,n_elements(oi_array.sta_index)-1 do $
		sta_index(where(staindex0 eq oi_array(i).sta_index))=i
		bls=strjoin(vlti_stationid(oi_array(sta_index).sta_name),'-')
		ubls=unique(bls)
		!p.multi=[0,n,n_elements(oi_array)]
		!x.range=[2.1,2.4]
		!x.range=0
		xsize=n*300
		if n_elements(oi_array) eq 3 then begin
			window,/free,xsize=xsize,ysize=700
			!p.charsize=2
		endif else begin
			window,/free,xsize=xsize
			!p.charsize=2
		endelse
		objectfiles0=obht(objindex).filename
;		Flux
		!y.range=0
		for i0=0,n-1 do begin
			i=where(objectfiles0 eq objectfiles(i0))
			range_max=0
			for j=0,n_elements(ubls)-1 do begin
				index=where(bls eq ubls(j))
				snr=amber_data.(row)(i).d(index).fringe_snr
				flux=amber_data.(row)(i).d(index).flux_sum
				k=where(snr gt threshold,count)
				if count gt 10 then flux=flux(*,k) else count=n_elements(snr)
				flux=total(flux,2)/count
				if max(flux) gt range_max then range_max=max(flux)
			endfor
			range=[0,range_max]
			if total(!y.range) eq 0 then crange=range else crange=!y.range
			for j=0,n_elements(ubls)-1 do begin
				index=where(bls eq ubls(j))
				snr=amber_data.(row)(i).d(index).fringe_snr
				flux=amber_data.(row)(i).d(index).flux_sum
				lambda=oi_wavelength.(row)(i).d.eff_wave*1e6
				k=where(snr gt threshold,count)
				if count gt 10 then flux=flux(*,k) else count=n_elements(snr)
				flux=total(flux,2)/count
				if j eq 0 then begin
					plot,lambda,flux,title=objectfiles(i),yrange=crange, $
						xtitle='Wavelength',ytitle='Flux'
				endif else begin
					oplot,lambda,flux,color=tci(j+1)
				endelse
				xyouts,min(lambda)+j*(max(lambda)-min(lambda))/3,mean(flux)/3,ubls(j), $
					charsize=1,color=tci(j+1)
			endfor
		endfor
;		Squared visibility
		!y.range=[0,1.2]
		for i0=0,n-1 do begin
			i=where(objectfiles0 eq objectfiles(i0))
			range_max=0
			for j=0,n_elements(ubls)-1 do begin
				index=where(bls eq ubls(j))
				snr=amber_data.(row)(i).d(index).fringe_snr
				v2=oi_vis2.(row)(i).d(index).vis2data
				index=where(v2 lt -1,count)
				if count gt 0 then begin
					index=whereindex(index,v2)
					snr(index(1,*))=0
				endif
				k=where(snr gt threshold,count)
				if count gt 10 then v2=v2(*,k) else count=n_elements(snr)
				v2=total(v2,2)/count
				if avg(v2) gt range_max then range_max=(avg(v2)*1.2)<1
			endfor
			range=[0,range_max]
			if total(!y.range) eq 0 then crange=range else crange=!y.range
			for j=0,n_elements(ubls)-1 do begin
				index=where(bls eq ubls(j))
				snr=amber_data.(row)(i).d(index).fringe_snr
				v2=oi_vis2.(row)(i).d(index).vis2data
				index=where(v2 lt -1,count)
				if count gt 0 then begin
					index=whereindex(index,v2)
					snr(index(1,*))=0
				endif
				k=where(snr gt threshold,count)
				if count gt 10 then v2=v2(*,k) else count=n_elements(snr)
				v2=total(v2,2)/count
				lambda=oi_wavelength.(row)(i).d.eff_wave*1e6
				if j eq 0 then begin
					plot,lambda,v2,title=objectfiles(i),yrange=crange, $
						xtitle='Wavelength',ytitle='Squared visibility'
				endif else begin
					oplot,lambda,v2,color=tci(j+1)
				endelse
;				xyouts,min(lambda),mean(v2)/3,ubls(j),color=tci(j+1)
				xyouts,min(lambda)+j*(max(lambda)-min(lambda))/3,mean(v2)/3,ubls(j), $
					charsize=1,color=tci(j+1)
			endfor
		endfor
;		Closure phase
		!y.range=[-90,90]
		rad=180/!pi
		if n_elements(oi_array) eq 3 then begin
		for i0=0,n-1 do begin
			i=where(objectfiles0 eq objectfiles(i0))
			phase=oi_t3.(row)(i).d.t3phi
			amp=oi_t3.(row)(i).d.t3amp
			ndata=n_elements(phase(0,*))
			snr=intarr(ndata)+1
			for j=0,2 do begin
				index=indgen(ndata)*3+j
				s=amber_data.(row)(i).d(index).fringe_snr
				sndex=where(s lt threshold and finite(s) eq 1,count)
				if count gt 0 then snr(sndex)=0
			endfor
			lambda=oi_wavelength.(row)(i).d.eff_wave*1e6
			k=where(snr gt 0,count)
			if count gt 10 then begin
				phase=phase(*,k) 
				amp=amp(*,k)
			endif else count=n_elements(snr)
;			complex_i=amp*sin(phase/rad)
;			complex_r=amp*cos(phase/rad)
;			phase=atan(total(complex_i,2),total(complex_r,2))*rad
			tphase=total(phase,2)	; but there are NaNs in the data!!
			for j=0,n_elements(phase(*,0))-1 do begin
				index=where(finite(phase(j,*)) eq 1)
				tphase(j)=mean(phase(j,index))
			endfor
			plot,lambda,tphase,title=objectfiles(i), $
					xtitle='Wavelength',ytitle='Closure phase'
		endfor
		endif
		!x.range=0
		!y.range=0
		end
	else:	if strlen(value) eq 7 then baseline=value $
				      else begin
					   objectfiles=value
;					   widget_control,fp_wid,sensitive=0
					   endelse
else:
endcase
;
end
;-------------------------------------------------------------------------------
pro amberobjectgui
;
; Display GUI for software which computes visibilities.
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberOptions,amber_options,p2vmfile
common AmberTaskWid,amber2p2v_wid,amberobject_wid,amberp2vm_wid,amberrecipes_wid
common AmberObjectGuiWids,file_wid,draw_wid,c_wid,qc_wid,fp_wid
common AmberObjectGui,objectfiles,row,channel,baseline,threshold
common AmberObjectGui2Local,c_value,s_value
;
; Remove any existing widget
if widget_info(amberobject_wid,/valid) then widget_control,amberobject_wid,/destroy
;
amberobject_wid=widget_base(/col, $
	title=obht(objindex(0)).targname+' ' $
	     +obht(objindex(0)).insmode $
	     +'('+strtrim(string(obht(objindex(0)).p2vmid),2)+')')
;
reduction_wid=widget_base(amberobject_wid,/col,/frame)
;
; Compute visibilities
reductionrow_wid=widget_base(reduction_wid,/col)
objectfiles=obht(objindex).filename
files=['All selected files',objectfiles]
file_wid=widget_droplist(reductionrow_wid,value=files,uvalue=files, $
	event_func='amberobjectgui_event')
widget_control,file_wid,sensitive=0
row_wid=widget_base(reductionrow_wid,/row)
button_wid=cw_bgroup(row_wid,/row,['ExtractVis','Cleanup'], $
	event_funct='amberobjectgui_event',/return_name)
qc_guis=['DataQC']
qc_wid=cw_bgroup(row_wid,/row,qc_guis, $
	event_funct='amberobjectgui_event',/return_name)
widget_control,qc_wid,sensitive=0
;
amberobjectrow_wid=widget_base(amberobject_wid,/row)
plots=['V2 vs SNR','V2 vs Piston','Piston vs SNR','Piston vs Time','Piston Closure','V2 Histogram']
fp_wid=widget_droplist(amberobjectrow_wid,event_func='amberobjectgui_event', $
	uvalue=plots,value=plots)
widget_control,fp_wid,sensitive=0
rows=['J','H','K'] &  row=2
rowmenu_wid=widget_droplist(amberobjectrow_wid,event_func='amberobjectgui_event', $
	uvalue=rows,value=rows)
widget_control,rowmenu_wid,set_droplist_select=2
baselines=[vlti_stationid(obht(objindex(0)).issconfstation1) $
      +'-'+vlti_stationid(obht(objindex(0)).issconfstation2)]
if not isnumeric(obht(objindex(0)).issconfstation3) then $
baselines=[baselines, $
	vlti_stationid(obht(objindex(0)).issconfstation2)+'-' $
       +vlti_stationid(obht(objindex(0)).issconfstation3), $
	vlti_stationid(obht(objindex(0)).issconfstation3)+'-' $
       +vlti_stationid(obht(objindex(0)).issconfstation1)]
baseline=baselines(0)
basemenu_wid=widget_droplist(amberobjectrow_wid,event_func='amberobjectgui_event', $
	uvalue=[baselines,'dummy'],value=baselines)
;
draw_wid=widget_draw(amberobject_wid,event_pro='trackdraw_event', $
	scr_xsize=300,scr_ysize=200)
;
max_channel=500
if strpos(obht(objindex(0)).insmode,'Low') ge 0 then max_channel=15
c_value=0
c_wid=cw_fslider(amberobject_wid,minimum=0,maximum=max_channel,value=0,/drag, $
	title='Channel (0=white light)',xsize=300,format='(i4)',uvalue='c')
widget_control,c_wid,sensitive=0
channel=-1
;
s_value=0
s_wid=cw_fslider(amberobject_wid,minimum=0,maximum=5,value=s_value,/drag, $
        title='SNR threshold for plotting',xsize=300,format='(i1)',uvalue='s')
threshold=0
;
widget_control,amberobject_wid,/realize
xmanager,'amberobjectgui2',amberobject_wid,/no_block
;
end
;-------------------------------------------------------------------------------
pro amberobjectgui2_event,event
;
common AmberObjectGui,objectfiles,row,channel,baseline,threshold
common AmberObjectGui2Local,c_value,s_value
;
widget_control,event.id,get_uvalue=slider,get_value=value
case slider of
	'c':c_value=long(value)
	's':s_value=long(value)
endcase
;
channel=c_value-1
threshold=s_value
;
end
;-------------------------------------------------------------------------------
function amberdefaultrecipes
;
; For the currently selected AMBER files, establish default recipes.
; Ideally, this function establishes the file associations and the 
; pipeline exexcutes each recipe, rather than deciding itself what
; to reduce and looking if there is a recipe. The cleanamberrawdir
; function should still be used before to assist this routine.
;
; Rules:
; We assume, for SM at least, that every observing sequence starts with
; a dark, and ends with a sky exposure.
;
; We do not enforce that every observing sequence ends with a sky exposure
;
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberOptions,amber_options,p2vmfile
;
count_obs=n_elements(objindex)
if count_obs ne n_elements(recipes) then return,-1
;
jd_obj=dblarr(count_obs)
dit_obj=fltarr(count_obs)
for i=0,count_obs-1 do begin
	parsedate,strmid(obht(objindex(i)).filename,6,10),y,m,d
	jd_obj(i)=julian(y,m,d)+hms2h(strmid(obht(objindex(i)).filename,17,12))/24.
	fitsfile=obj_new('fitsfile',obht(objindex(i)).filename)
	prihead=fitsfile->prihead()
	dit_obj(i)=prihead->getpar('DET DIT')
	obj_destroy,prihead
	obj_destroy,fitsfile
endfor
;
count_sky=n_elements(skyindex)
jd_sky=dblarr(count_sky)
if skyindex(0) lt 0 then count_sky=0
for i=0,count_sky-1 do begin
	parsedate,strmid(obht(skyindex(i)).filename,6,10),y,m,d
	jd_sky(i)=julian(y,m,d) $
		 +hms2h(strmid(obht(skyindex(i)).filename,17,12))/24.
endfor
;
count_dark=n_elements(darkindex)
jd_dark=dblarr(count_dark)
dit_dark=fltarr(count_dark)
if darkindex(0) lt 0 then count_dark=0
for i=0,count_dark-1 do begin
	parsedate,strmid(obht(darkindex(i)).filename,6,10),y,m,d
	jd_dark(i)=julian(y,m,d) $
		  +hms2h(strmid(obht(darkindex(i)).filename,17,12))/24.
	fitsfile=obj_new('fitsfile',obht(darkindex(i)).filename)
	prihead=fitsfile->prihead()
	dit_dark(i)=prihead->getpar('DET DIT')
	obj_destroy,prihead
	obj_destroy,fitsfile
endfor
;
; Check existence of P2VM files, compute if necessary
p2vmids=unique(obht.p2vmid)
for i=0,n_elements(p2vmids)-1 do begin
	f=findfile('*.'+strcompress(string(p2vmids(i)),/remove_all)+'.p2vm.fits')
	if strlen(f(0)) eq 0 then begin
		obht_bck=obht
                index=where(obht.p2vmid eq p2vmids(i) $
                        and (strpos(obht.dprtype,'P2V') ge 0),count)
		index=where(obht.p2vmid eq p2vmids(i) $
			and (strpos(obht.dprtype,'P2V') ge 0 $
			  or strpos(obht.dprtype,'WAVE') ge 0),count)
		if count gt 0 then begin
			obht=obht(index)
			amdlibp2vm
			obht=obht_bck
		endif
	endif
endfor
;
FOR i=0,count_obs-1 DO BEGIN
;
recipes(i).object_file=obht(objindex(i)).filename
;
; Of all DARKs on this target, choose the one immediately preceding it
if darkindex(0) ge 0 then begin
index=where(darkindex lt objindex(i) $
   and obht(darkindex).targname eq obht(objindex(i)).targname,count)
if count gt 0 then begin
	recipes(i).dark_file= $
		obht(darkindex(index(n_elements(index)-1))).filename
endif else begin
	recipes(i).dark_file='""'
; Take another DARK, but must be the same DIT!
	index=where(dit_dark eq dit_obj(i) $
	    and obht(darkindex).detwinstarty eq obht(objindex(i)).detwinstarty $
	    and obht(darkindex).insmode eq obht(objindex(i)).insmode,count)
	if count gt 0 then $
	recipes(i).dark_file=obht(darkindex(index(count-1))).filename
endelse
endif
;
; Of all SKYs on this target, choose the one immediately following it
if skyindex(0) ge 0 then begin
	index=where(skyindex gt objindex(i) $
		and obht(skyindex).p2vmid eq obht(objindex(i)).p2vmid $
		and obht(skyindex).targname eq obht(objindex(i)).targname,count)
	if count gt 0 then begin
		recipes(i).sky_file=obht(skyindex(index(0))).filename
	endif else begin
; 	If none was found, choose the one immediately preceding it
	index=where(skyindex lt objindex(i) $
		and obht(skyindex).p2vmid eq obht(objindex(i)).p2vmid $
		and obht(skyindex).targname eq obht(objindex(i)).targname,count)
	if count gt 0 then $
		recipes(i).sky_file=obht(skyindex(index(count-1))).filename $
	else $
		recipes(i).sky_file='""'
	endelse
endif
;
; P2VM file
recipes(i).p2vm_file=findfile('*.' $
	+strcompress(string(obht(objindex(i)).p2vmid),/remove_all)+'.p2vm.fits')
recipes(i).method=amber_options.errors
;
ENDFOR
;
save,recipes,filename='Recipes_'+configid+'.xdr'
;
return,0
;
end
;-------------------------------------------------------------------------------
pro amberinitializepipe
;
; This procedure will initialize the OYSTER scans structure for each fringe
; track file of all configurations found in the current directory.
; It will also remove intermediate products from previous runs.
;
; OYSTER/AMBER configurations are defined by the specific selection of baseline,
; spectrometer, and P2VM ID.
;
; Ideally, the following files should be in the current folder:
; For P2VM: 4 WAVE,3TEL, 10 3P2V 
; For  OBS: 1 DARK, 5 OBJECT, 1 SKY
;
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberOptions,amber_options,p2vmfile
;
; OYSTER common blocks
common Starbase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; Cleanup
spawn,'rm -f ????-??-??_*+*+*+*-*-*.xdr'
spawn,'rm -f ????-??-??_*+*-*-*.xdr'
spawn,'rm -f *.PHASE.STATISTIC.AMBER.*.fits'
spawn,'rm -f Recipes_*+*.xdr'
spawn,'rm -f *.p2vm.fits'
;
; Save configid
configid_save=configid
;
initialized=0
configids=analyzeamberconfigids()
num_config=n_elements(configids)
;
FOR iconfig=0,num_config-1 DO BEGIN
;
configid=configids(iconfig)
selectamberfiles,/all
fitsfile=obj_new('fitsfile',obht(objindex(0)).filename)
prihead=fitsfile->prihead()
dateobs=prihead->getpar('date-obs')
obj_destroy,prihead
obj_destroy,fitsfile
Date=strmid(dateobs,0,10)
time=hms2h(strmid(dateobs,11,12))
if time gt 12 then Date=nextdate(Date); This date is stored in geninfo.date
GenConfig=0	; otherwise get_genconfig will take stations from GenConfig
if strpos(configid,'Low') ge 0 then get_sysconfig,sysid='VLTI/AMBER-LR'
if strpos(configid,'Med') ge 0 then get_sysconfig,sysid='VLTI/AMBER-MR'
if strpos(configid,'Hig') ge 0 then get_sysconfig,sysid='VLTI/AMBER-HR'
;
; We count each "coherenced" OBJECT file as a separate observation
count_obs=n_elements(objindex)
;
; The recipe table will hold all reduction info for the pipeline
recipe={tag:'',dark_file:'',object_file:'',sky_file:'', $
	p2vm_file:'',method:''}
recipes=replicate(recipe,count_obs)
status=amberdefaultrecipes()
;
; Open first file and read configuration information
ambergenconfig,obht(objindex(0)).filename,genconfig
genconfig.configid=configid
; Allocate scans structure
scans=replicate(scan(),count_obs)
; Read data for each file
for i=0,count_obs-1 do begin
	s=scans(i)
	amberscan,recipes(i).object_file,s
	if genconfig.refstation eq 1 then begin
		s.fdlpos(1)=s.fdlpos(1)-s.fdlpos(0)
		s.fdlpos(2)=s.fdlpos(2)-s.fdlpos(0)
	endif
	if genconfig.refstation eq 2 then begin
		s.fdlpos(0)=s.fdlpos(0)-s.fdlpos(1)
		s.fdlpos(2)=s.fdlpos(2)-s.fdlpos(1)
	endif
	if genconfig.refstation eq 3 then begin
		s.fdlpos(0)=s.fdlpos(0)-s.fdlpos(2)
		s.fdlpos(1)=s.fdlpos(1)-s.fdlpos(2)
	endif
	s.iscan=i+1
	scans(i)=s
	recipes(i).tag=obht(objindex(i)).targname+' '+hms(scans(i).time/3600)
endfor
scans=amberstarids(scans)
get_scantable
get_startable,/names
;
; Initialization for AMOEBA
if not initialized then begin
	freememory
	bufferinfo=replicate(nightinfo(),num_config)
	geoinfo=replicate(geoparms,num_config)
	geninfo=replicate(allocgenconfig(/geninfo),num_config)
	geninfo.configid=configids
	table=startable
	initialized=1
endif else begin
	table=merge_startable(table,startable)
endelse
; confignum=where(geninfo.configid eq configid)
g=geninfo(iconfig)
struct_assign,genconfig,g
geninfo(iconfig)=g
geoinfo(iconfig)=geoparms
storenight
save,recipes,filename='Recipes_'+configid+'.xdr'
;
ENDFOR
;
; Make sure startable has entries for all stars observed this night
startable=table
index=where(startable.hdn eq 0,count)
if count gt 0 then print,'Warning: these stars not found in catalog: ', $
	startable(index).starid
;
; Restore selection of configid
configid=configid_save
r=ambergui_configid(configid)
;
end
;-------------------------------------------------------------------------------
pro myamberpipe,dir
;
; Processes all AMBER files returned by selectamberfiles and stores results 
; in OYSTER scans structure. Data should be of same configuration.
;
; Myamberpipe uses the instructions contained in the recipes structure for 
; the selection of the reduction method. The default recipes are created
; and stored when initializing OYSTER.
;
common AmberObservation,obht,skyindex,darkindex,objindex,p2vmindex
common AmberGuiWids,amber_wid,file_wid,window_slide_wid,pipeline_wid,opmenuc_wid
common AmberOptions,amber_options,p2vmfile
common AmberPipeline,rawdir,rawdirchecksum,configid,recipes
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Starbase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common ModelFit,parameters,ds_options
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
common Hds,path,hds_file_stub
;
RAD=180/!pi
;
; Obtained from 2006-08-05
offsetcal=[ $
[     2.63538,     32.0744], $
[     2.38552,     32.0878], $
[     2.04100,     30.8738], $
[     1.82366,     29.5868], $
[     1.61406,     29.4839], $
[     1.19371,     30.7547], $
[    0.810011,     30.9114], $
[    0.481977,     30.1389], $
[   0.0595671,     29.0591], $
[   -0.337672,     28.6404], $
[   -0.892803,     28.7946], $
[    -1.12233,     31.8707], $
[    -1.56444,     39.4966], $
[    -1.80804,     45.7032]]
x=[ $
       2.4150488,      2.3819618,      2.3485879,      2.3153266, $
       2.2818417,      2.2484926,      2.2149809,      2.1816336, $
       2.1481815,      2.1149240,      2.0816160,      2.0485343, $
       2.0154580,      1.9826393]
r0=poly_fit(x,offsetcal(0,*),2)
r1=poly_fit(x,offsetcal(1,*),4)
;
selectamberfiles,/all
;
; Determine how many observations there are
count_obs=n_elements(objindex)
;
; We process using the recipes, make sure they correspond
count_rec=n_elements(recipes)
if count_rec ne count_obs then begin
        print,'***Error(MYAMBERPIPE): Recipes inconsistent with selected files!'
        return
endif
;
; Obtain some general information on this night from the first file
;
fitsfile=obj_new('fitsfile',obht(objindex(0)).filename)
prihead=fitsfile->prihead()
;
; UT Date of the first observation
Date=strmid(prihead->getpar('date-obs'),0,10)
parsedate,Date,y,m,d
jd0=julian(y,m,d)
;
; Cleanup
obj_destroy,prihead
obj_destroy,fitsfile
;
; Star names and categories
stars=strarr(count_obs)
starids=strarr(count_obs)
categrs=strarr(count_obs)
;
; Make sure no valid data from previous run remains
scans.vissqerr=-1
scans.vissqcerr=-1
scans.diffphaseerr=-1
scans.diffphasecerr=-1
scans.tripleamperr=-1
scans.tripleampcerr=-1
scans.triplephaseerr=-1
scans.triplephasecerr=-1
;
; Loop over all observations
;
FOR iobs=0,count_obs-1 DO BEGIN
;
; Get some general info on this observation from the fringe track file
;
; Open first fringe track file and get some general information
; print,'Opening '+recipes(iobs).object_file
fitsfile=obj_new('fitsfile',recipes(iobs).object_file)
prihead=fitsfile->prihead()
categ=prihead->getpar('DPR CATG')
obj_destroy,fitsfile
;
; obstarg=strcompress(obht(objindex(iobs)).targname,/remove_all)
obstarg=scans(iobs).star
;
p2vmfile=recipes(iobs).p2vm_file
;
; Reduce data...
IF strlen(p2vmfile) ne 0 THEN BEGIN
;
; Run amdlibExtractVis first without binning for diagnostics
dest_file=obstarg+'.'+ $
	  amber_options.piston+'.'+ $
	  amber_options.errors+'.'+ $
	  recipes(iobs).object_file
vis_files=vis_row_files(dest_file)
if strlen(vis_files(0)) eq 0 then begin
	print,'myamberpipe: calling amdlibcomputeoidata...'
	amdlibcomputeoidata,recipes(iobs).object_file,dest_file, $
		recipes(iobs).dark_file,recipes(iobs).sky_file
	vis_files=vis_row_files(dest_file)
endif
m=n_elements(vis_files)
oi_array=mrdfits(vis_files(0),1,header)
if n_elements(oi_array) eq 2 then extns=[3,5,6]
if n_elements(oi_array) eq 3 then extns=[3,5,7,6]
fits_info,vis_files(m-1),extname=extname,/silent
index=where(extname eq 'AMBER_DATA',count)
if count eq 0 then continue	; continue next iteration in enclosed for-loop
d=mrdfits(vis_files(m-1),extns(2))
sta_index0=d.sta_index(0)
sta_index1=d.sta_index(1)
sta_index=sta_index0
for j=0,n_elements(sta_index0)-1 do $
	sta_index0(j)=where(oi_array.sta_index eq sta_index(j))
sta_index=sta_index1
for j=0,n_elements(sta_index1)-1 do $
	sta_index1(j)=where(oi_array.sta_index eq sta_index(j))
for l=0,genconfig.numbaseline(2)-1 do begin
	index=where(vlti_stationid(oi_array(sta_index0).sta_name) $
		eq strmid(genconfig.baselineid(l,2),0,3) $
		and vlti_stationid(oi_array(sta_index1).sta_name) $
		eq strmid(genconfig.baselineid(l,2),4,3) $
		and sta_index0 ge 0,count)
if count eq 0 then $
	index=where(vlti_stationid(oi_array(sta_index1).sta_name) $
		eq strmid(genconfig.baselineid(l,2),0,3) $
		and vlti_stationid(oi_array(sta_index0).sta_name) $
		eq strmid(genconfig.baselineid(l,2),4,3) $
		and sta_index0 ge 0,count)
	opd=d(index).opd*1e6
	fringe_snr=d(index).fringe_snr
	index=where(fringe_snr gt 1.0,count)
	if count gt 3 then begin
	scans(iobs).trackoffset(*,l)=medianve(opd(index))
	scans(iobs).delayrms(*,l)=stddev(opd(index))
	if n_elements(index) gt 30 then begin
		opd_median=median(opd(index),21)
		scans(iobs).delayrms(*,l)=stddev(opd_median-shift(opd_median,20))
	endif
	endif
endfor
d=0	; Release some memory
;
; Run amdlibExtractVis now with binning for the final results
; if 0 then begin		; Only amdlib 2.0 beta-1
; dest_file=obstarg+'.'+ $
; 	  amber_options.piston+'.'+ $
; 	  amber_options.errors+'.'+ $
; 	  amber_options.frcrit+amber_options.frvalu+'.'+ $
; 	  recipes(iobs).object_file
; p2vmfile=recipes(iobs).p2vm_file
; vis_files=vis_row_files(dest_file)
; Check for gasgano/reflex product files, else extract vis
; if strlen(vis_files(0)) eq 0 then file_dest=alt_dest_file(dest_file)
; if strlen(vis_files(0)) eq 0 then begin
; 	amdlibcomputeoidata,recipes(iobs).object_file,dest_file, $
; 		recipes(iobs).dark_file,recipes(iobs).sky_file,/scan
; 	vis_files=vis_row_files(dest_file)
; endif
; endif
;
; If merging was requested...
if amber_options.merges eq '>>' then begin
	index=where(recipes.sky_file eq recipes(iobs).sky_file,count)
	objectfile=recipes(index(count-1)).object_file
	if recipes(iobs).object_file eq objectfile then begin
		for n=0,count-1 do begin
			dest_file=obstarg+'.'+ $
				  amber_options.piston+'.'+ $
				  amber_options.errors+'.'+ $
				  recipes(index(n)).object_file
			vis_files=vis_row_files(dest_file)
			if n eq 0 then $
			vis_files_all=strarr(n_elements(vis_files),count)
			vis_files_all(*,n)=vis_files
		endfor
		for j=0,n_elements(vis_files)-1 do begin
		words=nameparse(vis_files_all(j,count-1),'.')
		outfile=strjoin(words(0:4),'.')+'.All_'+strmid(words(5),4,1) $
					       +'.fits'
;		out_file=findfile(outfile)
;		if strlen(out_file(0)) eq 0 then $
		files_all=reform(vis_files_all(j,*)) 
		n_fil_all=n_elements(files_all)
		index_all=indgen(n_fil_all)
		for k=0,n_fil_all-1 do begin
			d=mrdfits(files_all(k),7,h)
			if fitshparse(h,'EXTNAME') ne 'AMBER_DATA' $
				then index_all(k)=-1
		endfor
		files_all=files_all(where(index_all ge 0))
		amdlibappendoidata,files_all,outfile
		vis_files(j)=outfile
		endfor
	endif else goto,SKIPSELECTION
endif
;
; Run amdlibPerformFrameSelection, which also averages the data
ipos=strpos(vis_files(0),'.')
for i=0,n_elements(vis_files)-1 do begin
	dest_file=obstarg+'.AVG.'+ $
	  strmid(vis_files(i),ipos+1,strlen(vis_files(0))-ipos)
	amdlibperformframeselection,vis_files(i),dest_file
	vis_files(i)=dest_file
endfor
;
ENDIF ELSE BEGIN
;
; Just read previously reduced data
;
vis_files=recipes(iobs).object_file
;
ENDELSE
;
; Only one OBJECT file at a time
n=1
m=n_elements(vis_files)
row_index=intarr(m)
for j=0,m-1 do begin
help,vis_files(j)
	d=mrdfits(vis_files(j),3,header)
	print,'====================================================='
	help,d,/struc
	print,'====================================================='
	r=size(d)
	if r(n_elements(r)-2) eq 8 then begin
		j_obs=total(johnson_j(d.eff_wave*1e9))
		h_obs=total(johnson_h(d.eff_wave*1e9))
		k_obs=total(johnson_k(d.eff_wave*1e9))
		index=where([j_obs,h_obs,k_obs] eq max([j_obs,h_obs,k_obs]))
		row_index(j)=index(0)+1
	endif
endfor
i=where(row_index eq 1,c) & if c eq 1 then j_obs=1 else j_obs=0
i=where(row_index eq 2,c) & if c eq 1 then h_obs=1 else h_obs=0
i=where(row_index eq 3,c) & if c eq 1 then k_obs=1 else k_obs=0
row_index=row_index-1
;
oi_array=mrdfits(vis_files(0),1,header)
if n_elements(oi_array) eq 2 then extns=[3,4,5,6]
if n_elements(oi_array) eq 3 then extns=[3,4,5,7,6]
;	Wavelength
j=0 & h=0 & k=0
for l=0,m-1 do begin
	d={d:mrdfits(vis_files(l),extns(0))}
	if row_index(l) eq 0 then j=d
	if row_index(l) eq 1 then h=d
	if row_index(l) eq 2 then k=d
endfor
oi_wavelength={j:replicate(j,n),h:replicate(h,n),k:replicate(k,n)}
;	Differential phase
j=0 & h=0 & k=0
for l=0,m-1 do begin
	d={d:mrdfits(vis_files(l),extns(1))}
	if row_index(l) eq 0 then j=d
	if row_index(l) eq 1 then h=d
	if row_index(l) eq 2 then k=d
endfor
oi_vis={j:replicate(j,n),h:replicate(h,n),k:replicate(k,n)}
;	Squared visibility
j=0 & h=0 & k=0
for l=0,m-1 do begin
	d={d:mrdfits(vis_files(l),extns(2))}
	if row_index(l) eq 0 then j=d
	if row_index(l) eq 1 then h=d
	if row_index(l) eq 2 then k=d
endfor
oi_vis2={j:replicate(j,n),h:replicate(h,n),k:replicate(k,n)}
;	Amber spectrum
j=0 & h=0 & k=0
for l=0,m-1 do begin
	d={d:mrdfits(vis_files(l),extns(3))}
	if row_index(l) eq 0 then j=d
	if row_index(l) eq 1 then h=d
	if row_index(l) eq 2 then k=d
endfor
amber_spectrum={j:replicate(j,n),h:replicate(h,n),k:replicate(k,n)}
;	Triple products
if n_elements(extns) eq 5 then begin
j=0 & h=0 & k=0
for l=0,m-1 do begin
	d={d:mrdfits(vis_files(l),extns(4))}
	if row_index(l) eq 0 then j=d
	if row_index(l) eq 1 then h=d
	if row_index(l) eq 2 then k=d
endfor
oi_t3={j:replicate(j,n),h:replicate(h,n),k:replicate(k,n)}
endif else genconfig.numtriple=0
;
; Fill in wavelength info assuming all others are like the first obs.
; if iobs eq 0 then begin	; Not all bands might be present
	if j_obs then begin
	row=0
	wave=oi_wavelength.j.d.eff_wave
	band=oi_wavelength.j.d.eff_band
	nwave=n_elements(wave)
	genconfig.numspecchan(row)=nwave
	genconfig.wavelength(0:genconfig.numspecchan(row)-1,row)=wave
	genconfig.wavelengtherr=genconfig.wavelength*0+0.1e-6
	genconfig.chanwidth(0:genconfig.numspecchan(row)-1,row)=band
	genconfig.chanwidtherr=genconfig.wavelength*0+0.1e-6
	if genconfig.numtriple ge 1 then genconfig.triplenumchan(row)=nwave
	endif
	if h_obs then begin
	row=1
	wave=oi_wavelength.h.d.eff_wave
	band=oi_wavelength.h.d.eff_band
	nwave=n_elements(wave)
	genconfig.numspecchan(row)=nwave
	genconfig.wavelength(0:genconfig.numspecchan(row)-1,row)=wave
	genconfig.wavelengtherr=genconfig.wavelength*0+0.1e-6
	genconfig.chanwidth(0:genconfig.numspecchan(row)-1,row)=band
	genconfig.chanwidtherr=genconfig.wavelength*0+0.1e-6
	if genconfig.numtriple ge 1 then genconfig.triplenumchan(row)=nwave
	endif
	if k_obs then begin
	row=2
	wave=oi_wavelength.k.d.eff_wave
	band=oi_wavelength.k.d.eff_band
	nwave=n_elements(wave)
	genconfig.numspecchan(row)=nwave
	genconfig.wavelength(0:genconfig.numspecchan(row)-1,row)=wave
	genconfig.wavelengtherr=genconfig.wavelength*0+0.1e-6
	genconfig.chanwidth(0:genconfig.numspecchan(row)-1,row)=band
	genconfig.chanwidtherr=genconfig.wavelength*0+0.1e-6
	if genconfig.numtriple ge 1 then genconfig.triplenumchan(row)=nwave
	endif
; endif
;
; Store visibility
rad=180/!pi
; Between amdlib version 2 and 3 the sign of the closure phase changed!
if fix(amber_options.version) eq 3 then cps=-1.0 else cps=1.0
if j_obs then begin
	row=0
	for l=0,genconfig.numbaseline(row)-1 do begin
	scans(iobs).uvw(row,*,l,0) $
		=oi_vis2.j.d(l).ucoord/genconfig.wavelength(*,row)
	scans(iobs).uvw(row,*,l,1) $
		=oi_vis2.j.d(l).vcoord/genconfig.wavelength(*,row)
	scans(iobs).vissq(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis2.j.d(l).vis2data
	scans(iobs).vissqerr(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis2.j.d(l).vis2err
	scans(iobs).diffphase(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis.j.d(l).visphi/RAD
	scans(iobs).diffphaseerr(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis.j.d(l).visphierr/RAD
	endfor
	if genconfig.numtriple ge 1 then begin
	scans(iobs).tripleamp(row,0:genconfig.triplenumchan(row)-1) $
		=oi_t3.j.d(0).t3amp
	scans(iobs).tripleamperr(row,0:genconfig.triplenumchan(row)-1) $
		=oi_t3.j.d(0).t3amperr
	scans(iobs).triplephase(row,0:genconfig.triplenumchan(row)-1) $
		=cps*oi_t3.j.d(0).t3phi/RAD
	scans(iobs).triplephaseerr(row,0:genconfig.triplenumchan(row)-1) $
		=oi_t3.j.d(0).t3phierr/RAD
	endif
endif
if h_obs then begin
	row=1
	for l=0,genconfig.numbaseline(row)-1 do begin
	scans(iobs).uvw(row,*,l,0) $
		=oi_vis2.h.d(l).ucoord/genconfig.wavelength(*,row)
	scans(iobs).uvw(row,*,l,1) $
		=oi_vis2.h.d(l).vcoord/genconfig.wavelength(*,row)
	scans(iobs).vissq(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis2.h.d(l).vis2data
	scans(iobs).vissqerr(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis2.h.d(l).vis2err
	scans(iobs).diffphase(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis.h.d(l).visphi/RAD
	scans(iobs).diffphaseerr(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis.h.d(l).visphierr/RAD
	endfor
	if genconfig.numtriple ge 1 then begin
	scans(iobs).tripleamp(row,0:genconfig.triplenumchan(row)-1) $
		=oi_t3.h.d(0).t3amp
	scans(iobs).tripleamperr(row,0:genconfig.triplenumchan(row)-1) $
		=oi_t3.h.d(0).t3amperr
	scans(iobs).triplephase(row,0:genconfig.triplenumchan(row)-1) $
		=cps*oi_t3.h.d(0).t3phi/RAD
	scans(iobs).triplephaseerr(row,0:genconfig.triplenumchan(row)-1) $
		=oi_t3.h.d(0).t3phierr/RAD
	endif
endif
if k_obs then begin
	row=2
	f=fltarr(genconfig.numspecchan(row))+1
	for l=0,genconfig.numbaseline(row)-1 do begin
;	Apply LR piston correction
	if instrument_id(systemid) eq 'AMBER-LR_OBSOLETE' then begin
		for j=0,genconfig.numspecchan(row)-1 do begin
			a=fltarr(3)
			a(0)=1.0
			a(1)=poly(genconfig.wavelength(j,row)*1e6,r0)
			a(2)=poly(genconfig.wavelength(j,row)*1e6,r1)
			genconfig.offsetcal(*,l,j,row)=a(1:2)
			funct_gauss,scans(iobs).trackoffset(row,l),a,v
			f(j)=v
		endfor
	endif
	scans(iobs).uvw(row,*,l,0) $
		=oi_vis2.k.d(l).ucoord/genconfig.wavelength(*,row)
	scans(iobs).uvw(row,*,l,1) $
		=oi_vis2.k.d(l).vcoord/genconfig.wavelength(*,row)
	scans(iobs).vissq(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis2.k.d(l).vis2data/f
	scans(iobs).vissqerr(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis2.k.d(l).vis2err/f
	scans(iobs).diffphase(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis.k.d(l).visphi/RAD
	scans(iobs).diffphaseerr(row,0:genconfig.numspecchan(row)-1,l) $
		=oi_vis.k.d(l).visphierr/RAD
	endfor
	if genconfig.numtriple ge 1 then begin
	scans(iobs).tripleamp(row,0:genconfig.triplenumchan(row)-1) $
		=oi_t3.k.d(0).t3amp
	scans(iobs).tripleamperr(row,0:genconfig.triplenumchan(row)-1) $
		=abs(oi_t3.k.d(0).t3amperr)
	scans(iobs).triplephase(row,0:genconfig.triplenumchan(row)-1) $
		=cps*oi_t3.k.d(0).t3phi/RAD
	scans(iobs).triplephaseerr(row,0:genconfig.triplenumchan(row)-1) $
		=abs(oi_t3.k.d(0).t3phierr/RAD)
	endif
endif
scans(iobs).vissqc=scans(iobs).vissq
scans(iobs).vissqcerr=scans(iobs).vissqerr
scans(iobs).diffphasec=scans(iobs).diffphase
scans(iobs).diffphasecerr=scans(iobs).diffphaseerr
scans(iobs).tripleampc=scans(iobs).tripleamp
scans(iobs).tripleampcerr=scans(iobs).tripleamperr
scans(iobs).triplephasec=scans(iobs).triplephase
scans(iobs).triplephasecerr=scans(iobs).triplephaseerr
;
; Store normalizing spectrum in Photonrate
if j_obs then begin
	row=0
	photonrate=total(amber_spectrum.j.d.spectrum,1)
	scans(iobs).photonrate(row,0:genconfig.numspecchan(row)-1)=photonrate
	photonrateerr=sqrt(total(amber_spectrum.j.d.spectrum_error^2,1))
	scans(iobs).photonrateerr(row,0:genconfig.numspecchan(row)-1)=photonrateerr
endif
if h_obs then begin
	row=1
	photonrate=total(amber_spectrum.h.d.spectrum,1)
	scans(iobs).photonrate(row,0:genconfig.numspecchan(row)-1)=photonrate
	photonrateerr=sqrt(total(amber_spectrum.h.d.spectrum_error^2,1))
	scans(iobs).photonrateerr(row,0:genconfig.numspecchan(row)-1)=photonrateerr
endif
if k_obs then begin
	row=2
	nw=genconfig.numspecchan(row)
	if strpos(genconfig.configid,'High') ge 0 then index=390+indgen(21)
	if strpos(genconfig.configid,'High') ge 0 then index=indgen(nw)
	if strpos(genconfig.configid,'Medium') ge 0 then index=indgen(nw)
	if strpos(genconfig.configid,'Low') ge 0 then index=indgen(nw)
; 	AMBER gain: 4.7 e-/ADU, but assume e-<=> photons are output by amdlib
	gain=4.7
	scans(iobs).natcounts=total(amber_spectrum.k.d(index).spectrum,2) $
			     /mean(oi_vis2.k.d.int_time)
	scans(iobs).natcountserr $
		=sqrt(total(amber_spectrum.k.d(index).spectrum_error^2,2)) $
		/mean(oi_vis2.k.d.int_time)
	photonrate=total(amber_spectrum.k.d.spectrum,1)
;	photonrate=total(amber_spectrum.k.d.flux_pi_more_pj,2)
	scans(iobs).photonrate(row,0:genconfig.numspecchan(row)-1)=photonrate
	photonrateerr=sqrt(total(amber_spectrum.k.d.spectrum_error^2,1))
;	photonrateerr=sqrt(total(amber_spectrum.k.d.flux_pi_more_pj_err^2,2))
	scans(iobs).photonrateerr(row,0:genconfig.numspecchan(row)-1)=photonrateerr
endif
;
; Store time
; obsdate=strmid(dateobs,0,10)
; parsedate,obsdate,y,m,d
; jdobs=julian(y,m,d)
; time0=hms2h(strmid(dateobs,11,12))*3600	; [s]
; scans(iobs).time=time0+(jdobs-jd0)*86400
;
; Information for StarTable, see if we can find HD or HIP numbers
categrs(iobs)=strtrim(categ,2)
stars(iobs)=obht(objindex(iobs)).targname
; starid=cri_vlti(stars(iobs),obsra,obsdec)
; starids(iobs)=starid
; scans(iobs).starid=starid
; scans(iobs).iscan=iobs+1
; scans(iobs).star=stars(iobs)
; scans(iobs).ra=obsra
; scans(iobs).dec=obsdec
; scans(iobs).r0=seeing
; scans(iobs).za=acos(1/airmass)*RAD
;
; print,'Done storing data'
SKIPSELECTION:
ENDFOR
;
if max(abs(scans.time/3600)) gt 12 then Date=nextdate(Date)
genconfig.date=Date
confignum=where(geninfo.configid eq configid)
g=geninfo(confignum)
struct_assign,genconfig,g
geninfo(confignum)=g
geoinfo(confignum)=geoparms
;
; Make sure we have only one star ID per star
scans=amberstarids(scans)
get_scantable
;
; Now allocate and fill the startable
; get_startable,unique(starids)
for i=0,n_elements(startable)-1 do begin
	j=where(scans.starid eq startable(i).starid,count)
	if count gt 0 then begin
		k=where(strpos(categrs(j),'CALIB') ge 0,count)
		if count gt 0 then startable(i).bflag='C' $
			      else startable(i).bflag='.'
	endif
endfor
get_stationtable
storenight,11
;
; Save data as XDR file
index=where(amber_options.selcrit ne 'None',count)
if count eq 0 then selection='' $
	      else selection='.'+strjoin(amber_options.selcrit(index),'.')
hds_file_stub= $
	genconfig.date+'_'+strjoin(nameparse(genconfig.configid,'.'))+selection
put_xdr,hds_file_stub
;
; Also save as OIFITS file
put_oifits,hds_file_stub
;
if n_elements(errorlog) ne 0 then ConstrictorLog=errorlog
r=ambergui_configid(configid)
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
ds_options.i=1	; True: interferometry loaded
ds_options.v2=1
ds_options.ta=0
ds_options.tp=1
;
end
;-------------------------------------------------------------------------------
pro myamberpipe_merge,amp_file,phase_file
;
; Normally, we use 20% and 80% of the best frames for amplitudes and phases,
; respectively. These two files (with 'SNR' in their name) have to be merged,
; and the output is written to a file with just the core name and a '.MRG.fits'
; and '.MRG.cha.xdr' extension.
;
; Input files are '.cha.xdr' files.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
; get_oifits,phase_file
get_xdr,phase_file
phase_scans=scans
; get_oifits,amp_file
get_xdr,amp_file
;
scans.tripleamp=phase_scans.tripleamp
scans.tripleampc=phase_scans.tripleampc
scans.triplephase=phase_scans.triplephase
scans.triplephasec=phase_scans.triplephasec
;
words=nameparse(amp_file,'.')
words(0)=words(0)+'.MRG'
;
oifits_file=words(0)+'.fits'
f=findfile(oifits_file)
if strlen(f(0)) gt 0 then print,'***Error: file exists: '+oifits_file $
		  else put_oifits,oifits_file
xdr_file=words(0)+'.cha.xdr'
f=findfile(xdr_file)
if strlen(f(0)) gt 0 then print,'***Error: file exists: '+xdr_file $
		  else put_xdr,xdr_file
;
end
;-------------------------------------------------------------------------------
pro myambervsop,xdrfile
;
common Starbase,StarTable,Notes
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
; Derive OIFITS file name
filestub=strmid(xdrfile,0,strlen(xdrfile)-7)
fitsfile=filestub+'fits'
;
; Read OIFITS file
; get_oifits,fitsfile
;
; Read XDR file
get_xdr,xdrfile
startable.name=startable.starid
medium_h=0
medium_k=0
if strpos(genconfig.configid,'Medium_H') ge 0 then medium_h=1
if strpos(genconfig.configid,'Medium_K') ge 0 then medium_k=1
band=2
if medium_h eq 1 then band=1
if medium_k eq 1 then band=2
low_lim=[1.0,1.5,2.02]*1e-6
;
get_jhk
diameter_vk,/force
calcviscal
;
vsop_dir='/home/chummel/public_html/amber/vsop/'
dates_dir=vsop_dir+'Dates/'
stars_dir=vsop_dir+'Stars/'
;
configuration=strjoin(genconfig.stationid,'-')
;
; Update dbDates
dates_file=vsop_dir+'dbDates.html'
status=dc_read_fixed(dates_file,dates_lines,/col,format='(a300)')
index=where(strpos(dates_lines,genconfig.date+' '+configuration) ge 0,count)
if count eq 0 then dates_lines=addline(dates_lines,'</ul>',/before, $
	'<li><a href="Dates/'+genconfig.date+'/index.html" target="display">' $
	+genconfig.date+' '+configuration+'</a></li>')
openw,unit,dates_file,/get_lun
for j=0,n_elements(dates_lines)-1 do printf,unit,dates_lines(j)
free_lun,unit
;
; Directory by date
date_dir=dates_dir+genconfig.date
date_file=date_dir+'/index.html'
f=findfile(date_dir)
if strlen(f(0)) eq 0 then begin
	spawn,'mkdir '+date_dir
	spawn,'cp -f '+fitsfile+' '+date_dir
	openw,unit,date_file,/get_lun
	printf,unit,'<html>'
	printf,unit,'<head>'
	printf,unit,'<title>VSOP AMBER calibrator results</title>'
	printf,unit,'</head>'
	printf,unit,'<body bgcolor="white" text="black">'
	printf,unit,'<h2>'+genconfig.date+'</h2>'
	printf,unit,'<table>'
	printf,unit,'</table>'
	printf,unit,'</body>'
	printf,unit,'</html>'
	free_lun,unit
endif
status=dc_read_fixed(date_file,date_lines,/col,format='(a300)')
;
; Plot transfer function and closure phase for this date
;
; Make a copy of scans
scans_bck=scans
;
; Compute (median) white light values
scans.triplephase=scans.triplephasec
scans.triplephaseerr=scans.triplephasecerr
medianob
;
if init_plot('seeing','pt') ne 0 then return
!xsize=500
!ysize=400
set_ps
s=set_datasel('x')
ps_options.l=1
ps_options.p=0
ps_options.a=0
ps_options.c=1
ps_options.e=0
slice='pt'
st_dir='All'
ds_x.item=22	; time
ds_y.item=81	; R0
!y.range=[0,2]
!x.range=0
s=set_datasel('y')
stubr=filestub+'r0'
device,filename=stubr+'.ps'
plotdata
device,/close
spawn,'ps2gif '+stubr+'.ps'
ds_x.item=22	; time
ds_y.item=82	; T0
!y.range=[0,8]
!x.range=0
s=set_datasel('y')
stubt=filestub+'t0'
device,filename=stubt+'.ps'
plotdata
device,/close
spawn,'ps2gif '+stubt+'.ps'
;
if init_plot('scan','pt') ne 0 then return
!xsize=500
!ysize=400
set_ps
s=set_datasel('x')
ps_options.l=1
ps_options.p=0
ps_options.a=0
ps_options.c=1
ps_options.e=1
slice='pt'
ds_y.tr=band
ds_y.ch=-1
ds_y.ch(0)=4
st_dir='All'
ds_x.item=22	; time
ds_y.item=33	; triplephase c
!y.range=[-40,40]
!x.range=0
s=set_datasel('y')
stubp=filestub+'clphase'
device,filename=stubp+'.ps'
plotdata
device,/close
spawn,'ps2gif '+stubp+'.ps'
;
ds_y.ob=band
ds_y.bl=-1
ds_y.bl(0)=0
ds_x.item=22	; time
ds_y.item=27	; vissqec
!y.range=[0,1]
if strpos(fitsfile,'UT') ge 0 then !y.range=[0,0.6]
!x.range=0
s=set_datasel('y')
stubv1=filestub+'TF1'
device,filename=stubv1+'.ps'
plotdata
device,/close
spawn,'ps2gif '+stubv1+'.ps'
ds_y.bl(0)=1
s=set_datasel('y')
stubv2=filestub+'TF2'
device,filename=stubv2+'.ps'
plotdata
device,/close
spawn,'ps2gif '+stubv2+'.ps'
ds_y.bl(0)=2
s=set_datasel('y')
stubv3=filestub+'TF3'
device,filename=stubv3+'.ps'
plotdata
device,/close
spawn,'ps2gif '+stubv3+'.ps'
;
; Copy all plot files for this date to directory
spawn,'cp -f '+stubr+'.gif '+date_dir
spawn,'cp -f '+stubt+'.gif '+date_dir
spawn,'cp -f '+stubv1+'.gif '+date_dir
spawn,'cp -f '+stubv2+'.gif '+date_dir
spawn,'cp -f '+stubv3+'.gif '+date_dir
spawn,'cp -f '+stubp+'.gif '+date_dir
;
index=where(strpos(date_lines,fitsfile) ge 0,count)
if count eq 0 then begin
	date_lines=addline(date_lines,'</table>',/before, $
	'<tr><td colspan="2"><a href="'+fitsfile+'">'+fitsfile+'</a></td></tr>')
	date_lines=addline(date_lines,'</table>',/before, $
	'<tr><td><img src="'+stubr+'.gif"></td><td><img src="' $
			    +stubt+'.gif"></td></tr>')	
	date_lines=addline(date_lines,'</table>',/before, $
	'<tr><td><img src="'+stubv1+'.gif"></td><td><img src="' $
			    +stubv2+'.gif"></td></tr>')	
	date_lines=addline(date_lines,'</table>',/before, $
	'<tr><td><img src="'+stubv3+'.gif"></td><td><img src="' $
			    +stubp+'.gif"></td></tr>')	
endif
;
; Restore scans
scans=scans_bck
;
; Loop over all stars of this night
starids=unique(scans.starid)
for i=0,n_elements(starids)-1 do begin
;
ds_star=starids(i)
;
; Directory by star
star_dir=stars_dir+ds_star
star_file=star_dir+'/index.html'
f=findfile(star_dir)
if strlen(f(0)) eq 0 then begin
	spawn,'mkdir '+star_dir
	openw,unit,star_file,/get_lun
	printf,unit,'<html>'
	printf,unit,'<head>'
	printf,unit,'<title>VSOP AMBER calibrator results</title>'
	printf,unit,'</head>'
	printf,unit,'<body bgcolor="white" text="black">'
	printf,unit,'<h2>'+ds_star+'</h2>'
	printf,unit,'<table>'
	printf,unit,'</table>'
	printf,unit,'</body>'
	printf,unit,'</html>'
	free_lun,unit
endif
status=dc_read_fixed(star_file,star_lines,/col,format='(a300)')
;
; Update dbStars
stars_file=vsop_dir+'dbStars.html'
status=dc_read_fixed(stars_file,stars_lines,/col,format='(a300)')
index=where(strpos(stars_lines,ds_star) ge 0,count)
if count eq 0 then begin
	stars_lines=addline(stars_lines,'</ul>',/before, $
	'<li><a href="Stars/'+ds_star+'/" target="display">'+ds_star+'</a></li>')
	openw,unit,stars_file,/get_lun
	for j=0,n_elements(stars_lines)-1 do printf,unit,stars_lines(j)
	free_lun,unit
endif
;
if init_plot('uv','pt') ne 0 then return
!xsize=500
!ysize=400
set_ps
s=set_datasel('x')
ps_options.l=0
ps_options.p=1
ps_options.a=0
ps_options.c=1
ps_options.e=1
slice='pt'
ds_y.ob=band
ds_y.tr=band
ds_y.bl=indgen(genconfig.numbaseline(band))
ds_y.ch=indgen(genconfig.numspecchan(band))
if medium_h or medium_k then begin
	medianob,/all
	ds_y.ch=100
endif
index=where(genconfig.wavelength(*,band) lt low_lim(band),count)
if count gt 0 then scans(*).vissqcerr(band,index,*)=-1
st_dir='Sel'
st_sel=starids(i)
ds_y.item=25	; VisSq c
ds_x.item=41	; uv-radius
!y.range=[0,1.0]
!x.range=0
s=set_datasel('y')
stubv=filestub+starids(i)+'.vissq'
device,filename=stubv+'.ps'
plotuv
device,/close
spawn,'ps2gif '+stubv+'.ps'
;
if init_plot('scan','pt') ne 0 then return
!xsize=500
!ysize=400
set_ps
s=set_datasel('x')
ps_options.l=1
ps_options.p=1
ps_options.a=1
ps_options.c=1
ps_options.e=1
slice='ch'
ds_y.ob=band
ds_y.tr=band
ds_y.ch=indgen(genconfig.numspecchan(band))
if band eq 1 then begin
	ds_y.ch=100
endif
index=where(genconfig.wavelength(*,band) lt low_lim(band),count)
if count gt 0 then scans(*).triplephasecerr(band,index)=-1
st_dir='Sel'
st_sel=starids(i)
ds_y.item=32	; Triple phase
ds_x.item=45	; wavelength
!y.range=[-40,40]
!x.range=0
s=set_datasel('y')
stubp=filestub+starids(i)+'.phase'
device,filename=stubp+'.ps'
plotdata
device,/close
spawn,'ps2gif '+stubp+'.ps'
;
; Copy all plot files for this date to directory
spawn,'cp -f '+stubv+'.gif '+date_dir
spawn,'cp -f '+stubp+'.gif '+date_dir
;
; Update Dates/Date/index.html
index=where(strpos(date_lines,ds_star) ge 0,count)
index=where(strpos(date_lines,filestub+starids(i)) ge 0,count)
if count eq 0 then begin
date_lines=addline(date_lines,'</table>',/before,'<tr><td>'+ds_star+'</td></tr>')
date_lines=addline(date_lines,ds_star,/after, $
	'<tr><td><img src="'+stubv+'.gif"></td><td><img src="' $
			    +stubp+'.gif"></td></tr>')	
endif
;
; Update Stars/Star/index.html
index=where(strpos(star_lines,Date) ge 0,count)
if count eq 0 then begin
star_lines=addline(star_lines,'</table>',/before, $
	'<tr><td>'+Date+' '+configuration+'</td></tr>')
star_lines=addline(star_lines,Date,/after, $
	  '<tr><td><img src="../../Dates/'+Date+'/'+stubv+'.gif">' $
	+'</td><td><img src="../../Dates/'+Date+'/'+stubp+'.gif"></td></tr>')	
openw,unit_star,star_file,/get_lun
for j=0,n_elements(star_lines)-1 do printf,unit_star,star_lines(j)
free_lun,unit_star
endif
;
endfor
;
openw,unit_date,date_file,/get_lun
for j=0,n_elements(date_lines)-1 do printf,unit_date,date_lines(j)
free_lun,unit_date
;
set_plot,'x'
;
end
;-------------------------------------------------------------------------------
pro checkp2vm,files
;
for i=0,n_elements(files)-1 do begin
;
d=oirgetdata(files(i))
n=n_elements(d)
;
index=where(total(total(abs(d.data1),1),1) eq 0,n1)
index=where(total(total(abs(d.data2),1),1) eq 0,n2)
index=where(total(total(abs(d.data3),1),1) eq 0,n3)
index=where(total(total(abs(d.data4),1),1) eq 0,n4)
index=where(total(total(abs(d.data5),1),1) eq 0,n5)
;
print,'Zero frames in file '+files(i)+' (total frames ',n,'):', $
	format='(a,i3,a)'
print,'Data1: ',n1,', data2: ',n2,', data3: ',n3,', data4: ',n4, $
    ', data5: ',n5,format='(5(a7,2x,i3))'
;
endfor
;
end
;-------------------------------------------------------------------------------
pro calpiston,reset=reset
;
; Use the genconfig.offsetcal coefficients to remove the bias in LR mode
; due to piston.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if not keyword_set(reset) then reset=0 else reset=1
if reset then !offsetcal=0 else !offsetcal=1
;
for ob=0,genconfig.numoutbeam-1 do begin
for bl=0,genconfig.numbaseline(ob)-1 do begin
for ch=0,genconfig.numspecchan(ob)-1 do begin
;
if reset then f=1.0 else begin
	a=[1.0,genconfig.offsetcal(*,bl,ch,ob)]
	funct_gauss,scans.trackoffset(ob,bl),a,f
endelse
scans.vissqc(ob,ch,bl)=scans.vissq(ob,ch,bl)/f
scans.vissqec(ob,ch,bl)=scans.vissqe(ob,ch,bl)/f
;
endfor
endfor
endfor
;
end
;-------------------------------------------------------------------------------
pro calamber
;
; Estimate VLTI+AMBER throughput, and parameterize with seeing.
;
common LocalNatCounts,nc,nce
common Starbase,StarTable,Notes
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
index=where(startable.mk eq 100,count)
if count gt 0 then begin
	get_jhk
	index=where(startable.mk eq 100,count)
	if count gt 0 then begin
		print,'Warning: some K magnitudes are missing!'
	endif
endif
;
if n_elements(nc) eq 0 then begin
	nc=scans.natcounts
	nce=scans.natcountserr
endif
; scans.natcounts=nc
; scans.natcountserr=nce
;
f_vega=4.53e9	; K band photons per m^2 s micron from Vega
f_vega_c=f_vega*!pi*(genconfig.diameter(0)/2)^2 $
		   *total(genconfig.chanwidth(*,2))*1e6
gain=4.7
;
scans_bck=scans
;
for i=0,n_elements(startable)-1 do begin
	index=where(scans.starid eq startable(i).starid,count)
	if count ne 0 then begin
	if startable(i).mk ne 100 then begin
	f_star_c=10^(-startable(i).mk/2.5)*f_vega_c
	for k=0,genconfig.numsid-1 do $
	scans(index).natcounts(k)=100*(scans(index).natcounts(k)/f_star_c)
	for k=0,genconfig.numsid-1 do $
	scans(index).natcountserr(k)=100*scans(index).natcountserr(k)/f_star_c
	endif else scans(index).natcounts=0
	endif
endfor
;
natcounts=scans.natcounts
index=where(total(natcounts,1) gt 0)
natcounts=natcounts(*,index)
natjitter=scans.natjitter2
natrms=reform(sqrt(natjitter(*,0,index)*natjitter(*,1,index)))
!x.range=[min(natrms),max(natrms)]
!x.range=[0.2,max(natrms)]
!p.multi=[0,1,3]
!p.charsize=2
openw,unit,'Throughput_'+Date+'_'+genconfig.configid+'.txt',/get_lun
for i=0,genconfig.numsid-1 do begin
x=reform(natrms(i,*))
y=reform(natcounts(i,*))
si=sort(x)
x=x(si)
y=y(si)
r=poly_fit(x,y,2,yfit)
; printf,unit,'Median throughput station '+genconfig.stationid(i)+': ', $
; 	median(natcounts(i,*))
printf,unit,'Throughput at NATRMS=0.3 station '+genconfig.stationid(i)+': ', $
	poly(0.3,r)
title=genconfig.stationid(i)
if i eq 0 then title=title+' ('+date+')'
plot,x,y,title=title,ytitle='Throughput [%]',xtitle='IRIS seeing',psym=1
oplot,x,yfit,psym=0
endfor
free_lun,unit
;
scans=scans_bck
;
end
;-------------------------------------------------------------------------------
