;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;filterData
;#description#
;real or complex filters as function of wavelength
;#end_class#
;*******************************************************

PRO filterData__DEFINE
;********************************************************
;#structure#
; filterData
;#inheritance#
; None
;#description#
; Internal variables filter data object
;#structure_text#
;
   filter1={filterData, $
   type    :0,    $ 0=real, 1=complex; 2=ampphase
   Frequency:PTR_NEW(),   $  spatial freq in radians/meter
   Transmission:PTR_NEW() $ ; filter transmission
}
;#end_structure#
;********************************************************
END

FUNCTION filterData::INIT, input, freq=freq, trans=trans, wave=wave,$
   iErr=iErr
;construct an filterData object
;input is polymorphic:
; (1)  if a number assume that this is nFreq and create a
;      default, mostly empty filterData table with this many
;      rows.  The user can then:
;        Extract a pointer to this table using the ::table method
;        Fill in the values in the rows of this table
;
;
; (2)  a string, assumed to be a file name.  Open the file and
;      get the filter information from there.
; (3)  none, but the arrays trans and either freq or wave are
;      specified.  Then these arrays will be used to initialize
;      the table.  Freq is frequencies in THz, wave is
;      wavelengths in microns.  If both are given, wave will
;      be used
   iErr = 0
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
;
;determine input type
   new = 0
;only keyword inputs
   if (n_params() LT 1) then begin
;are the transmissions specified
      if (NOT KEYWORD_SET(trans)) then midiSetError, $
         'No transmissions set at input'
      nFreq = N_ELEMENTS(trans)
;are the frequencies specified; and the right number
      if (KEYWORD_SET(freq)) then begin
         if (N_ELEMENTS(freq) NE nFreq) then midiSetError,$
       '#elements of freq and trans not equal'
      endif else begin
;are the wavelengths specified
         if (KEYWORD_SET(wave)) then begin
       if (N_ELEMENTS(wave) NE nFreq) then midiSetError,$
          '#elements of freq and wave not equal'
       ;freq = 299.7925/wave  ; convert microns to THz
       freq = 2.*!pi * 1.e6/wave; convert microns to radians/m
    endif else midiSetError, 'neither freq or wave specified'
      endelse         ; freq not specified
      transSet = 1
      new = 1
   endif else begin
      transSet = 0; no input
      sInput = size(input)
      sInput = sInput[sInput[0]+1]
      if (sInput GE 1 AND (sInput LE 3)) then begin
         nFreq = input
    new = 1
      endif
   endelse
;input is either empty or specified, but not from file
   if (new EQ 1) then begin
      if (transSet EQ 1) then begin
         self.frequency    = PTR_NEW(freq)
         self.transmission = PTR_NEW(FLTARR(2, nFreq))
         (*self.transmission)[0,*] = trans
         (*self.transmission)[1,*] = 0.
         self.type = 0
      endif
;finished with initialization from input parameters
   endif else if (sInput eq 7 or (sInput eq 11)) then begin
;input is string or object assume it to be a file name of an existing FITS file
;
;      assume filter file format with 1st REAL/COMPLEX/AMPPHASE
;      2nd row row # of points followed
;      by wavelength(micron) transmission pairs
      OPENR,UNIT,input,/GET_LUN
      nFreq = 0
      transType =''
      READF, UNIT, transType
      if (transType EQ 'COMPLEX') then self.type = 1 $
      else if (transType EQ 'AMPPHASE') then self.type = 2 $
      else self.type = 0
      READF, UNIT, nFreq
      wave = fltarr(nFreq)
      trans= fltarr(2,nFreq)
      x = 1.
      y = 1.
      z = 0.
      for i=0, nFreq-1 do begin 
         if (self.type GE 1) then begin
            READF,UNIT,x,y,z 
            trans[*,i] = [y,z]
         endif else begin
            READF,UNIT,x,y
            trans[*,i] = [y,0.]
         endelse
         wave[i]  = x
      endfor
;make a bunch of them and point table to them
      self.transmission = PTR_NEW(trans)
      self.frequency  = PTR_NEW(2*!pi*1.e6/wave)
   endif else midiSetError,'Incorrect Input Type'
RETURN,1
END

PRO filterData::CLEANUP
PTR_FREE,self.frequency
PTR_FREE,self.transmission
RETURN
END

FUNCTION filterData::interpolate, freq, wave=wave, iErr=iErr
;return stored filter data interpolated to given frequencies
;using cubic splines
;INPUTS
;   freq  array of numeric   desired frequencies in THz
;   /wave if freq is not specified you can ask for
;   wavelengths in microns
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
;
;check if freq is given
   if (N_PARAMS() LT 1) then begin
      if (NOT KEYWORD_SET(wave)) then midiSetError,$
         'no valid inputs specified'
      ;freq = 299.7925/wave  ; convert microns to THz
      freq = 2*!pi*1.e6/wave  ; convert microns radians/m
   endif
;check if there is adequate internal data
   inFreq =  *self.frequency
   inTrans1 = (*self.transmission)[0,*]
   inTrans2 = (*self.transmission)[1,*]
   if (N_ELEMENTS(inFreq) LT 3) then midiSetError,errMsg
   sortFreq = sort(inFreq)
   iErr = 0
   smallFreq = (freq < max(inFreq)) > min(inFreq)
   ss = sort(smallFreq) ; spline like sorted input
   t1= SPLINE(inFreq(sortFreq), inTrans1(sortFreq), smallFreq[ss], 10)
   t2= SPLINE(inFreq(sortFreq), inTrans2(sortFreq), smallFreq[ss], 10)
   if (self.type EQ 0) then begin
      trans= FLTARR(N_ELEMENTS(ss))
      trans[ss] = t1 ; unsort
   endif else if (self.type EQ 1) then begin
      trans = COMPLEXARR(N_ELEMENTS(ss))
      trans[ss] = COMPLEX(t1,t2)
   endif else if (self.type EQ 2) then begin
      trans = COMPLEXARR(N_ELEMENTS(ss))
      trans[ss] = t1 * EXP(COMPLEX(0.,1.)*t2)
   endif
   if (total(freq LT min(infreq)) GT 0) then $
      trans(WHERE(freq LT MIN(inFreq))) = 0.
   if (total(freq GT  MAX(infreq)) GT 0) then $
      trans(WHERE(freq GT MAX(inFreq))) = 0.
; get rid of non-finite and underflow values
   ft = FINITE(trans)
   if (TOTAL(ft NE 1) GT 0) then trans(WHERE(ft NE 1)) = 0.
   ma = MACHAR()
   ft = ABS(trans) LT ma.XMIN AND (trans NE 0.)
   if (TOTAL(ft) GT 0) then trans(WHERE(ft)) = 0.
RETURN, trans
END

FUNCTION filterData::spFreq & RETURN,*self.frequency & END
FUNCTION filterData::trans  
if (self.type EQ 1) then RETURN,COMPLEX((*self.transmission)[0,*], $
   (*self.transmission)[1,*])
if (self.type EQ 2) then RETURN,(*self.transmission)[0,*]*EXP(COMPLEX(0,1.)*$
   (*self.transmission)[1,*])
RETURN,(*self.transmission)[0,*]
END
