;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
FUNCTION coordCompression::init, weight, dX=dX, dY=dY, maxDegree=maxDegree, $
   buffSize, iErr=iErr
;construct CoordCompression object.  This involves reading in and storing
;an input detector table and compression weighting arrays from "weight"
;then constructing the output detector object, and recomputing the
;distortion objects for the output detector
;
;This is a subclass of Data Filters
;INPUTS:
;  weight  poly   something pointing to a FITS file containing an
;                 imageData table.  The input may be either
;                 a filename, or a ImageData object.
;                 The imageData table should contain arrays
;                 of weights.  The actual detector data arrays will
;                 be multiplyied by these weights and summed perpendicular
;                 to the dispersion (dispersed data) or in both directions
;                 (undispersed data)
;        
;  dX, dY         shift the given array these amounts (in pixels) before
;                 multiplying and adding
;                 NOT IMPLEMENTED YET
;  maxDegree      maximum degree of polynomials used to represent
;                 output distortions: 2 if not specified
;  buffSize       suggested internal buffersize.  If not specified a
;                 default will be taken
   bottom = 0
   errMsg = ''
;analyze input type
   sInput = SIZE(weight)
;string input; 
  inputOK =  ((sInput[0] EQ 0) AND (sInput[1] EQ 7)) 
;get the imagedata object
   if (inputOK) then begin
      weightObj = obj_new('imagedata',weight,iErr=iErr)
      if (iErr NE 0) then begin 
         errMsg = 'Couldnt open input weight data'
         GOTO, ERR_RET
      endif 
   endif else begin   ;perhaps an object as input
      if (NOT inputOK) then if (sInput[1+sInput[0]] eq 11) then $
         inputOK = OBJ_CLASS(weight) EQ 'IMAGEDATA'
      if (NOT inputOK) then begin
         errMsg = 'incorrect input type '
         GOTO, ERR_RET_1
      endif
      weightObj = weight
   endelse
;input looks OK, initialize superClass dataFilter
   output = self->dataFilter::init(/float)
;check if offsets are wanted
;  if (KEYWORD_SET(dY)) then self.dY = FIX(dY) else self.dY = 0
;  if (KEYWORD_SET(dX)) then self.dX = FIX(dX) else self.dX = 0
   if (NOT KEYWORD_SET(maxDegree)) then self.maxDegree = 2 $
      else self.maxDegree = maxDegree
   if (NOT KEYWORD_SET(buffSize)) then self.transferSize=1024l*1024l else $
      self.transferSize = buffSize
;read one (and only one) row of imaging data for weighting
   weightData = weightObj->readrows(1, iErr=iErr)
   if (iErr NE 0) then begin
      errMsg = 'Failure reading input weighting images'
      GOTO, ERR_RET
   endif
   self.weightImageTable = PTR_NEW(weightData)
   iErr = 0
RETURN,1
ERR_RET_1:
   bottom = 1
ERR_RET:
   if (bottom eq 1) then midiAddErrMsg, ErrMsg, /trace else $
      midiAddErrMsg, ErrMsg
      midiPrintErrMain
RETURN,0
END

PRO coordCompression::cleanup
OBJ_DESTROY, self.outputDetector
PTR_FREE, self.weightImageTable
PTR_FREE, self.outputTemplate
self->dataFilter::cleanup
RETURN
END

FUNCTION coordCompression::detector, inputDetector, iErr=iErr
;create output detector table, given input detector object
;INPUTS
;     inputDetector  
;
;number of data regions
   nData = inputDetector->nRegion()
;construct output Detector object
   self.outputDetector = OBJ_NEW('imageDetector', nData, $
      maxCoeff=self.maxDegree, iErr=iErr)
   if (iErr NE 0) then begin
      errMsg = 'creation of output detector failed'
      GOTO, ERR_RET
   endif
;input and output table pointers
   inTable = inputDetector->table()
   outTable = self.outputDetector->table()
;check whether the x direction is Frequency or space
   if (strTrim((*inTable)[0].cType[0],2) EQ 'FREQ') then $
      self.spectralMode = 1 else self.spectralMode = 0
;
;dumb copy of table. dont copy distortion information because format
;may have changed
   detTags = TAG_NAMES(*inTable)
   for iTag = 0, N_ELEMENTS(detTags)-1 do begin
      tag3 = STRMID(detTags(iTag),0,3)
      if (tag3 NE 'DMP' AND (tag3 NE 'DMC')) then $
         (*outTable).(iTag) = (*inTable).(iTag)
   endfor
;begin to fill in correct values in output table
   (*outTable).nAxes[1] = 1
;is this spectral mode?
   if (self.spectralMode EQ 1) then begin
;loop over regions
      nPix = 10
      for iRegion = 0, nData -1 do begin
;get data; remember that stored column numbers are 1-rel
	 data = (*self.weightImageTable).((*self.dataCol)[iRegion]-1)
;define a set of points to compute average Y position as a function of x
         nAxes = (*inTable)[iRegion].nAxes
         xPix = [1+(nAxes[0]/float(nPix))*findgen(nPix), nAxes[0]]
         yPix = 1+ findgen(nAxes[1])
	 oo = replicate(1.,nAxes[1])
	 yAve = data[xPix,*]#yPix/(data[xPix,*]#oo)
;make a distortion object for input data
         inDistort = OBJ_NEW('coordDistortion', inputDetector, $
	    region=iRegion+1, iErr=iErr)
         if (iErr NE 0) then BEGIN
            errMsg = 'Error creating coordinate object '
            GOTO, ERR_RET
         endif
;make a distortion object for output data
         outDistort = OBJ_NEW('coordDistortion', 2, nAxes=nAxes)
;extend the area being considered 1 pixel in y, to prevent signularities
         xPix = [xPix, xPix]
         yAve = [yAve, yAve+1]
         yy = [replicate(1.,nPix+1), replicate(2., nPix+1)]
;compute coordinate at average Y position
	 inDistort->pixToCoord,xPix, xCoord, yAve, yCoord
;output coordinate distortion information
         outDistort->fitPoints, xPix, xCoord, yy, yCoord, $
	    maxDegree=[self.maxDegree,0]
;put information in detector object
         self.outputDetector->setDistortion, outDistort, iRegion+1, iErr=iErr
	 if (iErr NE 0) then begin
	    errMsg = 'couldnt put compressed distortion data into detector'
	    GOTO, ERR_RET
	 endif
	 obj_destroy, inDistort
	 obj_destroy, outDistort
      endfor             ; loop over regions
   endif else begin         ;spectral mode begin 2-d spatial mode
      (*outTable).nAxes[0] = 1
   endelse
   iErr = 0
RETURN, self.outputDetector
ERR_RET:
   midiAddErrMsg, ErrMsg
   midiPrintErrMain
RETURN,0
END

FUNCTION coordCompression::weights
;return weight data table
RETURN, *self.weightImageTable
END

FUNCTION coordCompression::filterRows, inData
;convert a set of input data into compressed form
;this is most efficiently done with a large array of data
;
;create an output structure from input structure
;how many fields in input data?
   tags = TAG_NAMES(inData)
   nTags = N_ELEMENTS(tags)
   outTable = self.outputDetector->table()
;create output template with same structure as input except
;DATA fields are compressed and floating format
   if (NOT PTR_VALID(self.outputTemplate)) then $
      self.outputTemplate=PTR_NEW(self.outData->template())
;create output data array
   nData = N_ELEMENTS(inData)
   outData = REPLICATE(*self.outputTemplate, nData)
;copy fields, compressing data
   for iTag = 0, nTags - 1 do begin
;is this a data column?
      if (total((iTag+1) EQ *self.dataCol) NE 0) then begin
;dispersed or undispersed data
         if(self.spectralMode EQ 1) then $ dispersed, sum in y-dim.
         for iData = 0, nData-1 do  begin 
	    outData[iData].(iTag) = total((inData[iData].(iTag) $
	    *(*self.tScale)[iTag] + (*self.tZero)[iTag])  $
	    *(*self.weightImageTable).(iTag),2)
	 endfor else $   ; not dispersed, just sum
         for iData = 0, nData-1 do  begin 
	    outData[iData].(iTag) = total((inData[iData].(iTag) $
	    *(*self.tScale)[iTag] + (*self.tZero)[iTag])  $
	    *(*self.weightImageTable).(iTag))
	 endfor 
      endif else outData.(iTag) = inData.(iTag)
      endfor  ;tags
RETURN, outData
END

