;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
FUNCTION CoordDistortion::init, input, naxes=naxes, crVal=crVal, crPix=crPix, $
   cType=cType, cd=cd, dmp=dmp, dmc=dmc, region=region, iErr=iErr
;Initialize CoordDistortion object
;
;INPUT:
;   input    poly      if a number, then input is nAxis, and
;                      just fill in input data from the calling sequence
;                      if an IMAGEDETECTOR object, then use Region 
;                      to select region and get data from table
;
  sInput = size(input)
  sInput = sInput[1+sInput[0]]
;a number
  if ((sInput GT 0) AND (sInput LT 4)) then begin
     nAxis = fix(input)
     self.nAxis = nAxis
     if (KEYWORD_SET(naxes)) then self.naxes[0:nAxis-1] = naxes[0:nAxis-1]
     if (KEYWORD_SET(crVal)) then begin
        self.crVal[0:nAxis-1] = crVal[0:nAxis-1]
        self.valSet = 1
      endif
      if (KEYWORD_SET(crPix)) then begin
         self.crPix[0:nAxis-1] = crPix[0:nAxis-1]
         self.pixSet = 1
      endif
      if (KEYWORD_SET(cd)) then begin 
         self.cd[0:nAxis-1,0:nAxis-1] = cd[0:nAxrs-1,0:nAxis-1]
         self.cdSet = 1
      endif
      if (KEYWORD_SET(dmp)) then self.dmp = PTR_NEW(dmp)
      if (KEYWORD_SET(dmc)) then self.dmc = PTR_NEW(DOUBLE(dmc))
      if (PTR_VALID(self.dmp)) then begin 
         dSize = size(self.dmp)
         self.maxCoeff = dSize(3)
      endif
;force cType strings to be 8 characters long
      if (KEYWORD_SET(cType)) then for i = 0,1 do $
         self.cType[i]=STRMID(STRTRIM(cType[i],2)+string(replicate(32b,8)),0,8)
;end of integer input (nAxis)  begin of Object input
   endif else if (sInput eq 11)  then begin
      if(OBJ_CLASS(input) NE 'IMAGEDETECTOR') then begin
         errMsg = 'input object class is not IMAGEDETECTOR'
         GOTO, ERR_RET_1
      endif
;right type of object, let's get the data
      if (NOT KEYWORD_SET(region)) then begin
         errMsg = 'Region not set'
         GOTO, ERR_RET_1
      endif
      if ((region lt 1) OR (region GT input->nRegion())) then begin
         errMsg = 'Region number '+strtrim(region,2)+' invalid'
         GOTO, ERR_RET_1
      endif
      if (NOT PTR_VALID(input->table())) then begin
         errMsg = 'Detector Table not initialized'
         GOTO, ERR_RET_1
      endif
      data = (*(input->table()))[region-1]
      self.nAxis = 2
      self.naxes = data.naxes
      self.crVal = data.crVal
      self.crPix = data.crPix
      self.cd = data.cd
      self.dmp = PTR_NEW(data.dmp)
      self.dmc = PTR_NEW(data.dmc)
      self.valSet = 1
      self.pixSet = 1
      self.cdSet = 1
      self.maxCoeff = n_elements(data.dmc)/self.nAxis
      self.cType = data.cType
  endif else begin
      errMsg = 'input of illegal type'
      GOTO, ERR_RET_1
   endelse
RETURN,1
ERR_RET_1:
   bottom = 1
ERR_RET:
   if (bottom EQ 1) then midiAddErrMsg, ErrMsg, /trace $
      else midiAddErrMsg, ErrMsg
   midiPrintErrMain
RETURN,0
END

PRO CoordDistortion::cleanup
;release internal storage in structure
PTR_FREE,self.dmp
PTR_FREE,self.dmc
PTR_FREE,self.rDmc
RETURN
END


PRO CoordDistortion::set,naxes=naxes, crVal=crVal, crPix=crPix, $
   cType=cType, cd=cd, dmp=dmp, dmc=dmc
;Update CoordDistortion object
;
   nAxis = self.nAxis
   if (KEYWORD_SET(crVal)) then begin
      self.crVal = crVal[0:nAxis-1]
      self.valSet = 1
   endif
   if (KEYWORD_SET(crPix)) then begin
      self.crPix = crPix[0:nAxis-1]
      self.pixSet = 1
   endif
   if (KEYWORD_SET(cd)) then begin 
      self.cd = cd[0:nAxis-1,0:nAxis-1]
      self.cdSet = 1
   endif
   if (KEYWORD_SET(naxes)) then self.naxes = naxes[0:nAxis-1]
   if (KEYWORD_SET(dmp)) then self.dmp = PTR_NEW(dmp)
   if (KEYWORD_SET(dmc)) then self.dmc = PTR_NEW(DOUBLE(dmc))
   if (PTR_VALID(self.dmp)) then begin 
      dSize = size(self.dmp)
      self.maxCoeff = dSize(3)
   endif
;force cType strings to be 8 characters long
   if (KEYWORD_SET(cType)) then for i = 0,1 do $
      self.cType[i] = STRMID(STRTRIM(cType[i],2)+string(replicate(32b,8)),0,8)
RETURN
END

PRO CoordDistortion::get,naxes=naxes, crVal=crVal, crPix=crPix, $
   cType=cType, cd=cd, dmp=dmp, dmc=dmc, rDmc=rDmc, maxCoeff=maxCoeff
;Return CoordDistortion object values
;
   if (KEYWORD_SET(naxes)) then naxes = self.naxes
   if (KEYWORD_SET(crVal)) then crVal = self.crVal
   if (KEYWORD_SET(crPix)) then crPix = self.crPix
   if (KEYWORD_SET(maxCoeff)) then maxCoeff = self.maxCoeff
   if (KEYWORD_SET(cd)) then cd = self.cd
   if (KEYWORD_SET(dmp)) then dmp = *self.dmp
   if (KEYWORD_SET(dmc)) then dmc = *self.dmc
   if (KEYWORD_SET(rDmc)) then rDmc = *self.rDmc
   if (KEYWORD_SET(cType)) then cType = self.cType
RETURN
END

PRO CoordDistortion::resetCd
;set the internal cd array to invalid
   self.cdSet = 0
RETURN
END

PRO CoordDistortion::resetVal
;set the internal crVal array to invalid
   self.ValSet = 0
RETURN
END

PRO CoordDistortion::resetPix
;set the internal crPix array to invalid
   self.PixSet = 0
RETURN
END

PRO CoordDistortion::resetDm
;set internal dmp,dmc arrays to invalid
   PTR_FREE,self.dmp
   PTR_FREE,self.dmc
RETURN
END

PRO CoordDistortion::fitPoints, xPixel, xCoord, yPixel, yCoord,$
   maxDegree=maxDegree, weights=weights, cd=cd, $
   crPix=crPix, noDistort=noDistort, sigma=sigma, $
   reverse=reverse, both=both, iErr=iErr
;calculate the dmp, dmc and optionally cD matrices given
;a set of measured pixel and coordinate values for a number
;of points. 
;
;INPUTS:
;  xPixel         float       an array of x pixel values
;  xCoord         float/doub  an array of coordinates 
;  yPixel         float       an array of y pixel values if nAxis=2
;  yCoord         float/doub  a 2nd array of coordinates 
;  maxDegree int               a 1 or 2 element array giving the
;                              maximum degree in x,y of the
;                              fitting polynomial.  If only
;                              1 elements applies to both x and y
;  weights                     [nPixel] array of relative weights for
;                              input points.  These are all 1 if ommitted
; crPix, cd                    arrays to set reference pix,cdArray, if given.
;                              If not given use current internal
;                              values.  If not internal values, 
;                              choose own defaults.
; noDistort                    If true, fit only (1st order) cD matrix
;
   nPixel = N_ELEMENTS(xPixel)
   nDim = self.nAxis
   maxD = intarr(nDim)
   if ((NOT KEYWORD_SET(maxDegree)) AND (NOT KEYWORD_SET(NoDistort))) $
      then begin
      errMsg = 'Must Specifiy maximum degree'
      GOTO, ERR_RET
   endif else if (KEYWORD_SET(maxDegree)) then maxD[*] = maxDegree $
      else maxD[*] = 1
   if (KEYWORD_SET(weights)) then pixWt = weights else $
      pixWt = replicate(1., nPixel)
;how many polynomials?
; the +1 includes the 0th order term
   nFit = maxD[0] + 1
;check for second dimension
   if (nDim EQ 2) then begin
      if (n_PARAMS() NE 4) then begin
         errMsg = 'insufficient input parameters'
         GOTO, ERR_RET
      endif
      nFit = nFit * (maxD[1]+1)
   endif
   if (nPixel LT nFit) then begin
      errMsg = 'not enough points for polynomial fit'
      GOTO, ERR_RET
   endif
;total fit coefficients
   if (KEYWORD_SET(noDistort)) then self.maxCoeff = 0 $
      else self.maxCoeff = nFit 
;local version of input variables
   pix = reform(xPixel, 1, nPixel)
   coord = reform(xCoord, 1, nPixel)
;set defaults
   if (self.pixSet EQ 0) then self.crPix[0] =  mean(Pix)
;modify input values for 0-point
   pix = pix - self.crPix[0]
;2nd dimension
   if (nDim EQ 2) then begin
      Pix = [pix, reform(yPixel,1, nPixel)]
      coord = [coord, reform(yCoord, 1, nPixel)]
      if (self.pixSet EQ 0) then self.crPix[1] =  mean(Pix[1,*])
      Pix[1,*] = Pix[1,*] - self.crPix[1]
   endif
   self.pixSet = 1
;if no current CD matrix, calculate 1st order fits
   if (self.cdSet EQ 0) then begin
;use IDL REGRESS fit function
;input array
      for oDim = 0, 1 do begin 
         cFit = REGRESS(Pix, reform(coord[oDim,*],nPixel), pixWt, values, Zero)
         self.cd[*,oDim ]  = cFit
         self.crVal[oDim] = Zero
      endfor
      self.cdSet  = 1
      self.valSet = 1
;whats left?
      for oDim = 0, nDim -1 do  coord[oDim,*] = coord[oDim,*] - self.crval[oDim]
      if (nDim EQ 1) then residual = coord - self.cd[0,0]*pix $
      else residual = coord - self.cd # pix
      sigma = sqrt(mean(residual^2))
   endif             ; calculate CD matrix
;is that all?
   iErr = 0
   if (KEYWORD_SET(noDistort)) then RETURN
;remove cD matrix from output coordinates
   unCd = invert(self.cd[0:nDim-1,0:nDim-1])
;theoretical pixel coords after reversal of cD
   unPix = transpose(unCd) # coord  
;now fit the distortion matrix
;first dimension it
   dmp = intarr(nDim, nDim, nFit)
   dmc = dblarr(nDim, nFit)
;stick powers of x into dmp
   px = indgen(nFit) mod (1+maxD[0])
   dmp[0, 0, *] = px
   if (nDim EQ 2) then begin
      dmp[1, 0, *] = px
;stick powers of y into dmp
      px = indgen(nFit)/(1+maxD[0])
      dmp[*, 1, *] = [px,px]
   endif
;input array to IDL REGRESS.  Drop one
;fitting order because REGRESS fits
;constant term separately
   inFit = fltarr(nFit-1, nPixel)
;forward solution
   if ((NOT KEYWORD_SET(reverse)) OR (KEYWORD_SET(both))) then begin
      for outDim = 0, nDim -1 do begin
         for iFit = 1, nFit -1 do inFit(iFit-1, *) = pix[0,*]^dmp[outDim, 0, iFit]
         if (nDim EQ 2) then for iFit = 1, nFit -1 do inFit(iFit-1, *) = $
            inFit(iFit-1, *) * (pix[1,*]^dmp[outDim, 1, iFit])
         dmc[outDim, 1:*] = REGRESS(inFit, reform(unPix[outDim, *],nPixel), $
            pixWt, values, Zero)
         dmc[outDim, 0] = Zero
      endfor            ; output dimension
      PTR_FREE, self.dmc
      PTR_FREE, self.dmp
      self.dmc = PTR_NEW(dmc)
      self.dmp = PTR_NEW(dmp)
   endif 
   if ((KEYWORD_SET(reverse)) OR (KEYWORD_SET(both))) then begin
      for outDim = 0, nDim -1 do begin
         for iFit = 1, nFit -1 do inFit(iFit-1, *) = unPix[0,*]^dmp[outDim, 0, iFit]
         if (nDim EQ 2) then for iFit = 1, nFit -1 do inFit(iFit-1, *) = $
            inFit(iFit-1, *) * (unPix[1,*]^dmp[outDim, 1, iFit])
         dmc[outDim, 1:*] = REGRESS(inFit, reform(Pix[outDim, *],nPixel), $
            pixWt, values, Zero)
         dmc[outDim, 0] = Zero
      endfor            ; output dimension
      self.rDmc = PTR_NEW(dmc)
   endif                ; reverse solution
RETURN
ERR_RET:
   midiAddErrMsg, ErrMsg, /trace
   midiPrintErrMain
RETURN
END

PRO CoordDistortion::pixToCoord, xPixel, xCoord, yPixel, yCoord, iErr=iErr
; Return coordinates corresponding to a set of pixels
; 
;
;INPUTS:
;  xPixel  float   set of 1st(x) pixel positions
;  yPixel  float   set of 2nd(y) pixel positions, if 2-D system
;
;RETURNS
;  xCoord        double  1st WCS coordinate
;  yCoord        double  2nd WCS coordinate
;
;check inputs
   iErr = 1
   nDim = self.nAxis
   nPix = N_ELEMENTS(xPixel)
   if ((self.valSet EQ 0) OR (self.pixSet EQ 0) OR (self.cdSet EQ 0))$
      then begin
      errMsg = 'Coordinate data not initialized '
      GOTO, ERR_RET
   endif
;if distortion matrices not inialized, apply cd matrix only
   if ((NOT PTR_VALID(self.dmp)) OR (NOT PTR_VALID(self.dmc)))$
      then begin
         coords = reform(xPixel, 1, nPix)
	 if (nDim EQ 2) then coords= [coords, reform(yPixel, 1, nPix)]
         GOTO, CD_MATRIX
   endif
;apply distortion matrices
   dmp = *self.dmp
   dmc = *self.dmc
   nCoeff = n_elements(dmc)/nDim
;internal versions of inputs
   pix = reform(xPixel,1,nPix) - self.crPix[0]
   if (nDim EQ 2) then pix = [pix,reform(yPixel,1,nPix) - self.crPix[1]]
   coords = dblarr(nDim, nPix)
;loop over output dimension
   for oDim = 0, nDim -1 do begin
;sort dmp by x-coord and make a local version
;     sDmp = sort(dmp[oDim, 0, *])
;     lDmp = reform(dmp[oDim, *, sDmp], nDim, nCoeff)
;     lDmc = reform(dmc[oDim, sDmp], nCoeff)
lDmp = reform(dmp[oDim, *, *], nDim, nCoeff)
lDmc = reform(dmc[oDim, *], nCoeff)
;initialize current x-exponent,y-exp arrays
;current x-power
      xPold = lDmp[0, 0]
;pixels to this power
      xArray = pix[0,*]^xPold
;times coefficient
      pArray = xArray * lDmc[0]
      if (nDim EQ 2) then begin
;current y-power
         yPold = lDmp[1, 0]
;pixels to this power
	 yArray = pix[1,*]^yPold
;total coefficient times x and y powers
	 pArray = pArray * yArray
      endif
;first term in polynomial
      coords[oDim, *] = pArray
;rest of the coefficients
      for iCoeff = 1, nCoeff - 1 do begin
;desired xPower
         xP = lDmp[0, iCoeff]
; if (xP gt xPold) then xArray = xArray * pix[0,*]^[xP-xPold]
xArray = pix[0,*]^xP
	 xPold = xP
	 pArray = xArray * lDmc[iCoeff]
	 if (nDim EQ 2) then begin 
            yP = lDmp[1, iCoeff]
;    if (yP gt yPold) then yArray = yArray * pix[1,*]^[yP-yPold] $
;    else if (yP lt yPold) then yArray = pix[1,*] ^ yP
yArray = pix[1,*]^yP
	    yPold = yP
            pArray = pArray * yArray
	 endif
	 coords[oDim, *] = coords[oDim, *] + pArray
      endfor        ; loop over coefficients
   endfor           ; output dimension oDim
;now apply CD matrix and offsets
CD_MATRIX:
;calculate dimensionality of input pixel array
   sPix = size(xPixel)
   sPix = sPix[1:sPix[0]]
   if (nDim EQ 1) then xCoord = reform(self.crVal[0] + $
      self.cd[0,0]*coords, sPix) $
      else begin
         coords = transpose(self.cd) # coords
         xCoord = reform(coords[0,*], sPix) + self.crVal[0]
         yCoord = reform(coords[1,*], sPix) + self.crVal[1]
      endelse
   iErr = 0
RETURN
ERR_RET:
   midiAddErrMsg, ErrMsg, /trace
   midiPrintErrMain
RETURN
END

PRO CoordDistortion::coordToPix, xCoord, xPixel, yCoord, yPixel, iErr=iErr
; Return pixels corresponding to a set of coordinates
;
;INPUTS:
;  xCoord  double   set of 1st(x) WCS coord positions
;  yCoord  double   set of 2nd(y) WCS coord positions, if 2-D system
;
;RETURNS
;  xPixel        double  1st pixel position
;  yPixel        double  2nd pixel position
;
;check inputs
   iErr = 1
   nDim = self.nAxis
   nPix = N_ELEMENTS(xCoord)
   if ((self.valSet EQ 0) OR (self.pixSet EQ 0) OR (self.cdSet EQ 0))$
      then begin
      errMsg = 'Coordinate data not initialized '
      GOTO, ERR_RET
   endif
;internal versions of inputs
   coord = reform(xCoord,1,nPix) - self.crVal[0]
;remove linear CD transform
   if (nDim EQ 1) then coord = coord/cd[0,0] $
   else begin
      coord = [coord,reform(yCoord,1,nPix) - self.crVal[1]]
      unCd = invert(self.cd[0:1,0:1])
      coord = transpose(unCd) # coord  
   endelse
;only linear part?
   if ((NOT PTR_VALID(self.rDmc)) OR (NOT PTR_VALID(self.dmp))) then begin
      xPixel = coord[0,*] + self.crPix[0]
      if (nDim EQ 2) then yPixel = coord[1,*] + self.crPix[1]
      RETURN
   endif
;apply reverse non-linear distortion correction 
;distortion matrices
   dmp = *self.dmp
   rDmc = *self.rDmc
   nCoeff = n_elements(rDmc)/nDim
   pix = fltarr(nDim, nPix)
;loop over output dimension
   for oDim = 0, nDim -1 do begin
;sort dmp by x-coord and make a local version
      lDmp = reform(dmp[oDim, *, *], nDim, nCoeff)
      lDmc = reform(rDmc[oDim, *], nCoeff)
;current x-power
      xPold = lDmp[0, 0]
;pixels to this power
      xArray = coord[0,*]^xPold
;times coefficient
      pArray = xArray * lDmc[0]
      if (nDim EQ 2) then begin
;current y-power
         yPold = lDmp[1, 0]
;pixels to this power
	 yArray = coord[1,*]^yPold
;total coefficient times x and y powers
	 pArray = pArray * yArray
      endif
;first term in polynomial
      pix[oDim, *] = pArray
;rest of the coefficients
      for iCoeff = 1, nCoeff - 1 do begin
;desired xPower
         xP = lDmp[0, iCoeff]
         xArray = coord[0,*]^xP
	 pArray = xArray * lDmc[iCoeff]
	 if (nDim EQ 2) then begin 
            yP = lDmp[1, iCoeff]
            yArray = coord[1,*]^yP
            pArray = pArray * yArray
	 endif
	 pix[oDim, *] = pix[oDim, *] + pArray
      endfor        ; loop over coefficients
      pix[oDim, *] = pix[oDim, *] + self.crPix[oDim]
   endfor           ; output dimension oDim
;reformat outputs to match inputs
   sPix = size(xCoord)
   sPix = sPix[1:sPix[0]]
   xPixel = reform(pix[0,*], sPix) 
   if (nDim EQ 2) then yPixel = reform(pix[1,*], sPix)
   iErr = 0
RETURN
ERR_RET:
   midiAddErrMsg, ErrMsg, /trace
   midiPrintErrMain
RETURN
END

PRO CoordDistortion::CoordGrid, xCoord, yCoord, iErr=iErr
; return coordinates at all pixels specified by naxes
; are naxes filled in?
   if (total(self.naxes[0:self.nAxis-1] EQ 0) NE 0) then begin
      midiAddErrMsg, 'nAxes not initialized', /trace
      GOTO, ERR_RET
   endif
   nDim = self.nAxis
   if (nDim EQ 1) then begin
      xPixel = self.crPix[0] + findgen(self.nAxes[0]) 
      self->pixToCoord, xPixel, xCoord, iErr=iErr
RETURN
   endif else begin
      xPixel = replicate(self.crPix[0]+findgen(self.nAxes[0]),self.nAxes[1])
      yPixel = transpose(replicate(self.crPix[1]+findgen(self.nAxes[1]),$
         self.nAxes[0]))
      self->pixToCoord, xPixel, xCoord, yPixel, yCoord, iErr=iErr
RETURN
   endelse
ERR_RET:
   midiPrintErrMain
RETURN
END

PRO coordDistortion::fixCoord,xPixel,xCoord,yPixel,yCoord,iErr=iErr
; calculate yPixel locations corresponding to a set of xPixels and 
;single fixed YCoord
;INPUTS:
;   xPixel:   array of xpixel positions
;   yCoord;   fixed valued of y-coordinate; should be scalar
;OUTPUTS:
;   yPixel:   array of ypixels, corresponding to xPixel,which
;             all have coordinate value yCoord
;   xCoord:   xcoordinate at each of the pixels:[xpixel,ypixel]
;
;start with guess that yPixel = reference pixel
   nPix = n_elements(xPixel)
   yPixel = replicate(self.crPix[1], nPix)
   yOut   = replicate(yCoord[0], nPix)
   dY     = 2
;calculate xyCoords corresponding to this
   maxIter = 10
   for iter = 1, maxIter do begin
      self->pixToCoord, xPixel, xCoord, yPixel, y1
      self->pixToCoord, xPixel, xCoord, yPixel+dY, y2
      dPix =  (yOut-y1)*dY/(y2-y1)
      yPixel = yPixel + dPix
      if (max(abs(dPix)) lt .01) then GOTO, Done
   endfor
   iErr = 1
RETURN
Done:
   iErr = 0
RETURN
END
