;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;oirImageData
;#description#
;methods of OIR FITS imaging_data table
;#end_class#
;*******************************************************

PRO imageData__DEFINE
;********************************************************
;#structure#
; imageData
;#inheritance#
; fitsTable
;#description#
;define structures for imageData class
;#structure_text#
oir3={ImageData, $   ;#class members#
INHERITS FITSTABLE, $;#class inheritance#
detector:OBJ_NEW(), $;object & detector object associated with this data
revision:1, $        ;int   & revision number of this format
Origin:'ESO PARNAL',      $;
instrume:'       ', $;
nRegion:0,          $;int   & number of regions on detector
dateObs:'',         $;str   & date of observation
mjdObs:1.0,         $;float   & date of observation
date  :'',          $;str   & date data creation
detDic:' ',         $; ESO dictionary version
detId:' ',          $; ESO software version
maxTel:2,           $;int   & max telescopes imaged per region
maxIns:2,           $;int   & max entires in INS_TRAIN table
maxStep:10,         $;int   & max steps in stepping_phase
dataCols:PTR_NEW(),  $ptr->int array & which columns (1-rel) have pixels
sourceCols:PTR_NEW(),$ptr->int array & which cols. have source numbers
targetCols:PTR_NEW() $ptr->int array & which cols. have target types
}
;#end_structure#
;********************************************************
RETURN
END

PRO imageDataStruct__DEFINE
;********************************************************
;#structure#
; imageDataStruct
;#inheritance#
; None
;#description#
; template row in image data table
;#structure_text#
oir4={ImageDataStruct,    $ ;#FITS TABLE structure#
frame:0L,        $          ;long & frame number
time:0.0D,       $          ;dbl  & time of exposure
expTime:0.0    $          ;float & length of exposure
;reference:0,     $          ;int & phase reference source number
;opd:DBLARR(2),   $          ;dbl array & main delay line OPD offsets (fsec)
;localOpd:DBLARR(2),   $     ;dbl array & instrument OPD offset (fsec)
;offset:fltarr(2),      $    ;flt array & pointing offset rel. source position
;rotation:0.0,    $          ;flt & rotation of field rel. north (degrees)
;stepping_Phase:0 $          ;long & sequence no. in OPD stepping cycle
;opt_train         eso optical train designator (maxTel not known)
;ins_train         internal optial train designator (maxTel not known)
;DataN             data to be added later (N not yet known)
;SourceN           source number to be added later (N not yet known)
;TarTypN           Target Type to be added later (N not yet known)
}
;#end_structure#
;********************************************************
RETURN
END

PRO imageData::cleanup
  OBJ_DESTROY,self.Detector    
  PTR_FREE,self.DataCols       
  PTR_FREE,self.SourceCols   
  PTR_FREE,self.TargetCols  
  self->FitsTable::cleanup      ; cleanup inherited object too
END

FUNCTION imageData::init, input, iErr=iErr, buffSize = buffSize, $
   float=float, byte=byte, long=long, double=double, int=int,    $
   complex=complex
;construct IMAGEDATA object
;polymorphic input can be of the following types:
;(1)  string
;     This is assumed to be a bin table file name.  The file is opened,
;     an IMAGEDETECTOR object is created from the image_detector table
;     SELF is then created as a special case of FITSTABLE, ready to read.
;(2)  IMAGEDETECTOR object.  Same as case one.  For READ files one
;     can now begin to read rows.  For WRITE files, one can "appendToFile"
;     For WRITE one should specify one of the data types, otherwise
;     float will be assumed
;
;     float,byte...  used to specify image data type for "new" files
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
   if (NOT KEYWORD_SET(buffSize)) then buffSize = 100
   iErr = 0
   sInput = SIZE(input)
;string input; treat as file name
   if ((sInput[0] EQ 0) AND (sInput[1] EQ 7)) then begin
      self.Detector = OBJ_NEW('IMAGEDETECTOR', input, iErr=iErr)
      if(iErr NE 0) then midiSetError, $
         'reading image detector table from input file:'+input $
        +' failed', /notInitial
;incarnate myself as a FITSTABLE, using file object stored in imagedetector
      if(0 EQ self->FITSTABLE::init(self.Detector->file(), $
         extName='IMAGING_DATA', iErr=iErr)) then midiSetError,$
         'reading image data header from input file:'+input $
        +'failed' ,/notInitial
      self.nRegion = (self.head)->getPar('NREGION', matches)
      if (matches LE 0) then $
         midiSetError, 'NREGION not specified in data header'
      self.MaxTel = (self.head)->getPar('MAXTEL', matches)
      if (matches LE 0) then $
         midiSetError,  'maxTel not specified in data header'
      self.maxIns = (self.head)->getPar('MAXINS', matches)
      if (matches LE 0) then  begin
         self.maxIns = 2 
         (self.head)->addPar,'MAXINS', self.maxIns
      endif
      self.maxStep = (self.head)->getPar('MAXSTEP', matches)
      if (matches LE 0) then  begin
         self.maxStep = 100
         (self.head)->addPar,'MAXSTEP', self.maxStep
      endif
      self.Revision = self.head->getPar('REVISION', matches)
      self.dateObs = (self.head)->getPar('DATE-OBS', matches)
      if (matches LE 0) then  begin
         self.dateObs = '1947-08-23T14:00:00:00.000'
         (self.head)->addPar,'DATE-OBS', self.dateObs
      endif
      self.date = (self.head)->getPar('DATE', matches)
      if (matches LE 0) then  begin
         self.date= '1957-08-23T14:00:00:00.000'
         (self.head)->addPar,'DATE', self.date
      endif
      self.mjdObs = (self.head)->getPar('MJD-OBS', matches)
      if (matches LE 0) then  begin
         self.mjdObs= 12345.6789
         (self.head)->addPar,'MJD-OBS', self.mjdObs
      endif
      self.Origin = (self.head)->getPar('ORIGIN', matches)
      if (matches LE 0) then  begin
         self.origin= 'ESO PARANAL'
         (self.head)->addPar,'ORIGIN', self.origin
      endif
      self.instrume = (self.head)->getPar('INSTRUME', matches)
      if (matches LE 0) then begin
         self.instrume='MIDI'
         (self.head)->addPar,'INSTRUME', self.instrume
      endif
      self.detDic = (self.head)->getPar('HIERARCH ESO DET DID', matches)
      if (matches LE 0) then begin
         self.detDic='WaWas dictionary''
         (self.head)->addPar,'HIERARCH ESO DET DID', self.detDic
      endif
      self.detid = (self.head)->getPar('HIERARCH ESO DET ID', matches)
      if (matches LE 0) then begin
         self.detid ='WaWas detector''
         (self.head)->addPar,'HIERARCH ESO DET ID', self.detid
      endif
RETURN,1
   endif     $         ; string input
;object input
   else if (sInput[1+sInput[0]] eq 11)  then begin
      if(OBJ_CLASS(input) NE 'IMAGEDETECTOR') then $
         midiSetError, 'input object class is not IMAGEDETECTOR'
;incarnate myself using template from imagedetector
      self.detector = input
      dataTemplate = input->dataTemplate(iErr=iErr, byte=byte, long=long, $
         float=float, double=double, int=int, complex=complex)
      if (iErr NE 0) then midiSetError, $
         'data template from IMAGEDATA input unavailable ',$
     /notInitial
      if(0 EQ self->FITSTABLE::init(dataTemplate, iErr=iErr, $
         extName='IMAGING_DATA' ))then $
     midiSetError, 'construction from IMAGEDETECTOR object failed',$
     /notInitial
;copy header info from imagedetector
      detHead = input->head()
      self.head->addPar,'ORIGIN',detHead->getPar('ORIGIN', matches)
      self.head->addPar,'INSTRUME',detHead->getPar('INSTRUME', matches)
      self.head->addPar,'MJD-OBS',detHead->getPar('MJD-OBS', matches)
      self.head->addPar,'DATE-OBS',detHead->getPar('DATE-OBS', matches)
      self.head->addPar,'DATE',detHead->getPar('DATE', matches)
      self.head->addESOPar,'DET DID',detHead->getPar('HIERARCH ESO DET DID', matches)
      self.head->addESOPar,'DET ID',detHead->getPar('HIERARCH ESO DET ID', matches)
      self.head->addPar,'NREGION',detHead->getPar('NREGION', matches)
      self.head->addPar,'MAXTEL', detHead->getPar('MAXTEL', matches)
      self.head->addPar,'MAXINS', 2L
      self.head->addPar,'MAXSTEP', 100L
      self.head->addPar,'REVISION', 1L
   RETURN,1
   endif              ; object input
;not recognized input
   midiSetError,'input of unrecognized type'
RETURN,0
END

FUNCTION imageData::detector
;return internally stored pointer to imageDetector object
RETURN, self.Detector
END

FUNCTION imageData::dataColumns, iErr=iErr
;return column numbers of template structure that contain data
;do we know the answer?
   iErr = 0
   if (PTR_VALID(self.dataCols)) then RETURN,*(self.dataCols)
   columns = self->columnNumbers('DATA*', iErr=iErr)
   if (iErr EQ 0) then self.dataCols = PTR_NEW(columns)
RETURN, columns
END

FUNCTION imageData::sourceColumns, iErr=iErr
;return column numbers of template structure that contain source numbers
;do we know the answer?
   iErr = 0
   if (PTR_VALID(self.sourceCols)) then RETURN,*(self.sourceCols)
   columns = self->columnNumbers('TARGET*', iErr=iErr)
   if (iErr EQ 0) then self.sourceCols = PTR_NEW(columns)
RETURN, columns
END

FUNCTION imageData::targetColumns, iErr=iErr
;return column numbers of template structure that contain target types
;do we know the answer?
   iErr = 0
   if (PTR_VALID(self.targetCols)) then RETURN,*(self.targetCols)
   columns = self->columnNumbers('TARTYP*', iErr=iErr)
   if (iErr EQ 0) then self.targetCols = PTR_NEW(columns)
RETURN, columns
END

FUNCTION imageData::readRowsF, rows, columns=columns, iErr=iErr
;specialization of fitsTable::readRows that returns requested
;rows and columns, but converts data columns to floating point
;if necessary
;INPUTS
;   rows     array of numbers specifying rows to be read
;   columns  array of column numbers or names to read
;RETURNS
;   an array of structures representing the image data table rows
;   from disk, but data has been converted to float, if necessary,
;   with application of tzero and tscale
;data column numbers 1 rel
   dCol = self->dataColumns()
   if(N_PARAMS() GT 0) then RETURN, self->fitsTable::readRows(rows, $
      columns=columns, floatCol=dCol, iErr = iErr) else $
      RETURN, self->fitsTable::readRows(columns=columns, floatCol=dCol, $
         iErr=iErr)
END

FUNCTION imageData::makeScans, rows, iErr=iErr
;combine 0 or 1 dimensional input time series into "scans"
;INPUTS
;   rows     long(2) first and last rows to be included in scan
;OUTPUTS
;   outScan  a structure similar to inData structure but
;            with concatenation of all parts into a time sequence
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
;read data
   inStruct = self->readrows(rows[0]+lindgen(rows[1]-rows[0]+1),iErr=iErr)
   if (iErr NE 0) then midiSetError, 'couldnt read input data',/notInitial
   outData = {frame:inStruct.frame, $
      time:inStruct.time, $
      expTime:inStruct.expTime, $
      reference:inStruct[0].reference, $
      opd:inStruct.opd, $
      localOpd:inStruct.localOpd, $
      offset:inStruct[0].offset, $
      rotation:inStruct.rotation, $
      stepping_Phase:inStruct.stepping_Phase, $
      opt_Train:inStruct[0].opt_Train, $
      ins_Train:inStruct[0].ins_Train $
      }
;data, source, tartype
   dataCols = self->dataColumns() - 1
   sourceCols = self->sourceColumns() - 1
   tarCols = self->targetColumns() - 1
   nData = N_ELEMENTS(dataCols)
   for iData = 0, nData-1 do begin
      sData = STRTRIM(STRING(iData+1),2)
      outData = CREATE_STRUCT(outData, 'DATA'+sData, $
         inStruct.(dataCols[iData]), $
     'TARGET'+sData, inStruct[0].(sourceCols[iData]), $
     'TARTYP'+sData, inStruct[0].(tarCols[iData]))
   endfor
   iErr = 0
RETURN, outData
END

