;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;oirImageDetector
;#description#
;class to support OIR IMAGING_DETECTOR table
;#end_class#
;*******************************************************

PRO imageDetStruct__DEFINE
;********************************************************
;#structure#
; imageDetStruct
;#inheritance#
; NONE
;#description#
;template for 1 row of IMAGING_DETECTOR table
;#structure_text#
oir1={ImageDetStruct,   $ ;#FITS TABLE structure#
region:0,           $  ; int & which region
detector:1           $  ; int & which detector
;correlation:0,      $  ; int & type of correlation (self/cross)
;regName:STRING(REPLICATE(32B,16)),$ string & name for this region
;corner:intarr(2),    $ ; long array & pixel coordinates of b.l.c.
;gain:double(1.),     $ ; double & electrons per adu
;nAxis:intarr(2),     $ ; long array &size of region in pixels
;crVal:dblarr(2),     $ ; dbl array &reference pixel coordinates
;crPix:fltarr(2),     $ ; float arr&position of reference pixel
;cType:REPLICATE(STRING(REPLICATE(32B,8)),2), $ str array &dimension type
;cD:dblarr(2,2)      $ ;  dbl array & linear coordinate transf. to WCS
; ports              size not known yet which input ports are correlated
; dmp                size not known yet
; dmc                size not known yet
}
;#end_structure#
;********************************************************
END

PRO imageDetector__DEFINE
;********************************************************
;#structure#
; ImageDetector
;#inheritance#
; fitsTable
;#description#
; Internal variables for ImagingDetector Object
;#structure_text#
oir2={ImageDetector,    $ ;#class members#
INHERITS FITSTABLE,  $    ;#class inheritance#
revision:1,          $    ;int & revision number
nDetect:1,           $    ;int & number of physical detectors
origin:'ESO PARANAL',   $    ;str & source of this file
instrument:'MIDI',   $    ;str & instrument name
mjdObs:1.0,          $    ;float & julian date of obs
dateObs:' ',         $    ;str & civil observation date
date:' ',            $    ;str & date of file creation
DetDic:' ',          $    : eso dictionary name
detid:' ',           $    : eso software ID
nRegion:0,           $    ;int & number of specified regions on detector
maxCoeff:0,          $    ;int & max no. of coefficients in coord distortions
numDim:2,            $    ;int & dimensionality of detector regions
maxTel:2,            $    ;int & max number of telescopes per region
dataType:-1,         $    ;int & idl data type of information
dataTempl:PTR_NEW()  $    ;ptr->struct &representation of 1 row of table
}
;#end_structure#
;********************************************************
END

FUNCTION imageDetector::init, input, iErr=iErr, nDetect=nDetect,   $
   instrument=instrument, maxTel=maxTel, dateObs=dateObs, numDim=numDim,$
   maxCoeff=maxCoeff,origin=origen,mjd=mjd,date=date,dic=dic,id=id
;construct an ImageDetector object
;input is polymorphic:
; (1)  if a number assume that this is nRegion and create a
;      default, mostly empty ImageDetector table with this many
;      rows.  The user should then:
;        Extract a pointer to this table using the ::table method
;        Fill in the values in the rows of this table
;        E.G. the following lines indicate a detector with 4 regions
;        containing the auto and cross correlations between 2 input ports
;         ID = obj_new('imagedetector', 4)
;         IDtable = ID->table()
;         (*IDtable).region = 1+indgen(4)
;         (*IDtable).ports[0] = [1,1,2,2]
;         (*IDtable).ports[1] = [1,2,1,2]
;         (*IDtable).correlation   = [1,2,2,1]
;              and so on
;         Use nDetect, maxtel... to fill in these keywords in header
;
; (2)  a string, assumed to be a file name.  Open the file and
;      get the detector information from there.
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
;determine input type
   sInput = size(input)
   sInput = sInput[sInput[0]+1]
;integer, thus input is the size of the ImageDet table
   if (sInput ge 1 and (sInput le 3)) then begin
      nRegion = input
      self.nRegion = nRegion
;see if inputs set, otherwise use defaults
      if (KEYWORD_SET(nDetect)) then self.nDetect=nDetect else self.nDetect = 1
      self.nRegion = nRegion
      if (KEYWORD_SET(instrument)) then self.instrument = instrument else $
         self.instrument = 'MIDI'
      if (KEYWORD_SET(maxTel)) then self.maxTel = maxTel else self.maxTel = 2
      if (KEYWORD_SET(dateObs)) then self.dateObs = dateObs else $
         self.dateObs = ''
      if (KEYWORD_SET(numDim)) then self.numDim = numDim else self.numDim=2
      numDim = self.numDim
      if (KEYWORD_SET(maxCoeff)) then self.maxCoeff = maxCoeff else $
         self.maxCoeff = 0
      if (KEYWORD_SET(origin)) then self.origin = origin else $
         self.origin = 'ESO PARANAL' 
      if (KEYWORD_SET(mjd)) then self.mjdObs = mjd else $
         self.mjdObs = 52331.0 
      if (KEYWORD_SET(date)) then self.date = date else $
         self.date = 'Today' 
      if (KEYWORD_SET(dic)) then self.detdic = dic else $
         self.detdic = 'UNKNOWN' 
      if (KEYWORD_SET(id)) then self.detid = id else $
         self.detid = 'UNKNOWN' 
      maxC = self.MaxCoeff
;create a single template row, without ports, dmp or dmc
      template = {ImageDetStruct}
;append PORTS to structure
      if (self.maxTel GT 0) then template = CREATE_STRUCT(template, $
         'PORTS', intarr(self.maxTel))
; append other things to structure
      template = CREATE_STRUCT(template, 'CORRELATION', 0, $
         'REGNAME', STRING(REPLICATE(32B,16)), $
         'CORNER', INTARR(2), $
         'GAIN', 1.0d, $
         'NAXIS', intarr(2), $
         'CRVAL', dblarr(2), $
         'CRPIX', fltarr(2), $
         'CTYPE', REPLICATE(STRING(REPLICATE(32B,8)),2), $
         'CD', dblarr(2,2))
;append DMP and DMC to structure
      if (maxC GT 0) then  template = CREATE_STRUCT(template, 'DMP', $
        intarr(numDim, numDim, maxC), 'DMC', dblarr(numDim, maxC))
;fill in character strings with blanks to establish lengths
      template.regName=STRING(REPLICATE(32B,16))
      template.cType[*]=STRING(REPLICATE(32B,8))
;make a bunch of them and point table to them; set region
      self.table = PTR_NEW(REPLICATE(template, nRegion))
      (*self.table).region = 1 + indgen(nRegion)
;set me up as a fitstable with header etc. and formats given by template
      if (0 eq self->FITSTABLE::init(template, $
         extname='IMAGING_DETECTOR', iErr=iErr)) then midiSetError, $
         'Image Detector Creation Failed', /notInitial
;fill in values in internal header
      (self.head)->addPar,'ORIGIN', self.origin
      (self.head)->addPar,'INSTRUME', self.instrument
      (self.head)->addPar,'MJD-OBS', self.mjdObs
      (self.head)->addPar,'DATE-OBS', self.dateObs
      (self.head)->addPar,'DATE',     self.date
      (self.head)->addESOPar,'DET DID', self.detdic
      (self.head)->addESOPar,'DET ID', self.detid
      (self.head)->addPar,'NDETECT',  self.nDetect
      (self.head)->addPar,'NREGION',  self.nRegion
      (self.head)->addPar,'MAX_COEF', self.maxCoeff
      (self.head)->addPar,'NUM_DIM',  self.numDim
      (self.head)->addPar,'MAXTEL',   self.maxTel

;finished with initialization from input parameters
   endif else if (sInput eq 7 or (sInput eq 11)) then begin
;input is string, assume it to be a file name of an existing FITS file
      if(0 eq self->FITSTABLE::init(input,extName='IMAGING_DETECTOR',/table, $
     iErr=iErr))  then midiSetError, $
         'Image Detector Creation from Disk Failed',/notInitial
      if (iErr NE 0) then midiSetError,'',/notInitial
      self.nDetect =    (self.head)->getPar('NDETECT', matches)
      if (matches EQ 0) then self.nDetect = 1
      self.nRegion =    (self.head)->getPar('NREGION', matches)
      if (matches EQ 0) then self.nRegion = self.naxis1
      self.instrument = (self.head)->getPar('INSTRUME', matches)
      if (matches EQ 0) then self.instrument = 'MIDI'
      self.maxTel     = (self.head)->getPar('MAXTEL', matches)
      if (matches EQ 0) then self.maxTel = 2
      self.dateObs    = (self.head)->getPar('DATE-OBS', matches)
      if (matches EQ 0) then self.dateObs = 'TODAY'
      self.revision    = (self.head)->getPar('REVISION', matches)
      if (matches EQ 0) then self.revision = 1
      self.numDim      = (self.head)->getPar('NUM_DIM', matches)
      if (matches EQ 0) then self.numDim = 2
      self.maxCoeff     = (self.head)->getPar('MAX_COEF', matches)
      if (matches EQ 0) then self.maxCoeff = 0
      self.origin     = (self.head)->getPar('ORIGIN', matches)
      if (matches EQ 0) then self.origin = 'ESO PARANAL'
      self.mjdObs     = (self.head)->getPar('MJD-OBS', matches)
      if (matches EQ 0) then self.mjdObs = 52331.0
      self.date        = (self.head)->getPar('DATE', matches)
      if (matches EQ 0) then self.date = 'TODAY'
      self.detDic        = (self.head)->getPar('HIERARCH ESO DET DID', matches)
      if (matches EQ 0) then self.detDic = 'UNKNOWN'
      self.detid        = (self.head)->getPar('HIERARCH ESO DET ID', matches)
      if (matches EQ 0) then self.detid = 'UNKNOWN'
   endif else midiSetError, 'Incorrect Input Type'
RETURN,1
END

PRO imageDetector::cleanup
  PTR_FREE, self.DataTempl  
  self->fitsTable::cleanup 
END

FUNCTION imageDetector::nRegion
RETURN,self.nRegion
END

PRO imageDetector::set, region, iErr=iErr, ports=ports, $
   correlation=correlation, regName=regName, corner=corner, $
   naxis=naxis, crVal=crVal, crPix=crPix, cType=cType, $
   cD=cD, dmp = dmp, dmc = dmc
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;
   if (region le 0 or (region gt self.nRegion)) then $
      midiSetError, 'Region '+STRTRIM(STRING(region),2)+ $
         ' out of range; maxregion='+ STRTRIM(STRING(fix(self.nRegion)),2)
   rr = region - 1
   if (KEYWORD_SET(ports)) then (*self.table)[rr].ports=ports
   if (KEYWORD_SET(correlation)) then (*self.table)[rr].correlation=correlation
   if (KEYWORD_SET(corner)) then (*self.table)[rr].corner=corner
   if (KEYWORD_SET(nAxis)) then (*self.table)[rr].nAxis=nAxis
   if (KEYWORD_SET(crVal)) then (*self.table)[rr].crVal=crVal
   if (KEYWORD_SET(crPix)) then (*self.table)[rr].crPix=crPix
   if (KEYWORD_SET(cD)) then (*self.table)[rr].cD=cD
   if (KEYWORD_SET(dmp)) then begin
      sD = size(dmp)
      if (TOTAL(sD[0:3] NE [3, self.numDim, self.numDim, self.maxCoeff]) NE 0) then $
         midiSetError,' Input DMP array incorrectly dimensioned; should be '+ $
        strTrim(string(self.numDim),2)+','+strTrim(string(self.numDim),2)+ $
        ','+strTrim(string(self.maxCoeff),2)
      (*self.table)[rr].dmp=dmp
   endif
   if (KEYWORD_SET(dmc)) then begin
      sD = size(dmc)
      if (TOTAL(sD[0:2] NE [2, self.numDim, self.maxCoeff]) NE 0) then midiSetError, $
         ' Input DMC array incorrectly dimensioned; should be '+ $
     strTrim(string(self.numDim),2)+','+ strTrim(string(self.maxCoeff),2)
      (*self.table)[rr].dmc=dmc
   endif
   if (KEYWORD_SET(cType)) then (*self.table)[rr].cType=cType
   iErr = 0
RETURN
END

FUNCTION imageDetector::maxCoeff
;return internally stored valued of maxCoeff
RETURN, self.maxCoeff
END

PRO imageDetector::SetTable, inTable, iErr=iErr
;insert an external table in imageDetector body
;this is intrinsically dangerous because format is not checked
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
   sinput = size(InTable)
   sinput = sinput[sinput[0]+1]
   errMsg = 'Input is not ImageDetStruct'
   if (sinput ne 8) then midiSetError, $
      errMsg = 'Input is not ImageDetStruct'
   sinput = TAG_NAMES(inTable,/STRUCTURE_NAME)
   PTR_FREE,self.table
   self.table = PTR_NEW(inTable)
   self.nRegion = n_elements(inTable)
   iErr = 0
RETURN
END

FUNCTION imageDetector::dataTemplate, iErr=iErr, idlType=idlType, $
   byte=byte, long=long, float=float, double=double, int=int,     $
   complex=complex
;
;   Create a template "IMAGING_DATA" structure corresponding
;   to the detector description in this imageDet object
;
;   data structures will be R*4 unless idlType, byte,long,int, double,or complex
;   is specified
;   If more than one is specified, the highest precision specified will be used
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
   if (NOT PTR_VALID(self.table)) then $
      midiSetError, 'Imaging Detector Table not Defined'
   temp = {ImageDataStruct}
;create OPT_TRAIN
   temp = CREATE_STRUCT(temp, 'OPT_TRAIN', intarr(self.maxTel), $
          'INS_TRAIN', intarr(2))
;create lots of other tags
   temp = CREATE_STRUCT(temp, 'REFERENCE', 0, $
      'OPD', dblarr(2), $
      'LOCALOPD', dblarr(2), $
      'OFFSET', fltarr(2), $
      'ROTATION', 0.0, $
      'STEPPING_PHASE', 0)
;create target and tartype areas
   for region = 1, self.nRegion do begin
      stRegion =STRTRIM(STRING(region),2)
      temp = CREATE_STRUCT(temp, 'TARGET'+stRegion, 0, $
         'TARTYP'+stRegion, 'T ')
   endfor
;create DATAn areas with correct type
   for region = 1, self.nRegion do begin
      nAxis = (*self.table)[region-1].NAXIS
      stRegion =STRTRIM(STRING(region),2)
      if (KEYWORD_SET(idlType)) then temp = CREATE_STRUCT(temp, $
         'DATA'+stRegion, MAKE_ARRAY(nAxis[0],nAxis[1],TYPE=idlType)) $
      else if (KEYWORD_SET(double)) then temp = CREATE_STRUCT(temp, $
        'DATA'+ stRegion,  MAKE_ARRAY(nAxis[0],nAxis[1], /DOUBLE)) $
      else if (KEYWORD_SET(long)) then temp = CREATE_STRUCT(temp, $
     'DATA'+ stRegion,  MAKE_ARRAY(nAxis[0],nAxis[1], /LONG)) $
      else if (KEYWORD_SET(int)) then temp = CREATE_STRUCT(temp, $
        'DATA'+ stRegion,  MAKE_ARRAY(nAxis[0],nAxis[1], /INT)) $
      else if (KEYWORD_SET(byte)) then temp = CREATE_STRUCT(temp, $
        'DATA'+ stRegion,  MAKE_ARRAY(nAxis[0],nAxis[1], /BYTE)) $
      else if (KEYWORD_SET(complex)) then temp = CREATE_STRUCT(temp, $
        'DATA'+ stRegion,  MAKE_ARRAY(nAxis[0],nAxis[1], /COMPLEX)) $
      else temp = CREATE_STRUCT(temp, $
        'DATA'+ stRegion,  MAKE_ARRAY(nAxis[0],nAxis[1], /FLOAT))
   endfor
   iErr = 0
RETURN, temp
END

FUNCTION imageDetector::window, data, iErr=iErr
;Extract portions of a detector frame
;INPUTS
;   window   numerical array   input data, can be single plane or cube
;
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
   sData = SIZE(data)
   nRegion = self.nRegion
;type of data
   tData = sData[1+sData[0]]
;output data ; do we already know which data type and do we have a template?
   if((tData NE self.dataType)  OR $
      (NOT PTR_VALID(self.dataTempl))) then begin
      self.dataType = tData
      PTR_FREE, self.dataTempl
      self.dataTempl = PTR_NEW(self->dataTemplate(idlType = tData))
   endif
;is this plane or cube
   if ((sData[0] LT 2) OR (sData[0] GT 3)) then midiSetError, $
      ' input data must be 2 or 3-D'
   if (sData[0] EQ 2) then winData = *(self.dataTempl) else $
      winData = REPLICATE(*(self.dataTempl), sData[3])
;which columns correspond with which DATA and detector areas
   dataTagNames = TAG_NAMES(winData)
   dataCol   = INTARR(self.nRegion)
   for iRegion = 1, nRegion do dataCol[iRegion-1] = $
         WHERE(dataTagNames EQ 'DATA'+strTrim(string(iRegion),2))
;loop over regions
   for iRegion = 0, nRegion - 1 do begin
;beginning pixels and lengths or each region; remember idl is 0-relative
      xx = (*self.table)[iRegion].corner - 1
      ll = (*self.table)[iRegion].naxis  - 1
;extract data
;if some of naxis are 1, idl degenerate axes suppression causes problems
      sWin = size(winData.(dataCol[iRegion]))
      if (sWin[0] EQ 3) then winData.(dataCol[iRegion]) = data[xx[0]:xx[0]+ll[0], $
         xx[1]:xx[1]+ll[1], *] $
      else if (sWin[0] EQ 2) then winData.(dataCol[iRegion]) = $
         REFORM(data[xx[0]:xx[0]+ll[0], xx[1]:xx[1]+ll[1], *], sWin[1], sWin[2])$
      else if (sWin[0] EQ 1) then winData.(dataCol[iRegion]) = $
         REFORM(data[xx[0]:xx[0]+ll[0], xx[1]:xx[1]+ll[1], *], sWin[1])
   endfor        ; nRegion
   iErr = 0
RETURN, winData
END

PRO imageDetector::setDistortion, distort, region, iErr = iErr
;given a coordDistortion object, set the internal coordinate values
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;
   if (region GT self.nRegion OR (region LT 1)) then midiSetError, $
      'specified region '+strtrim(region,0)+' out of range'
;set dummy variables
   cd=1&dmc=1&crVal=1&dmp=1&crPix=1
;get values from distortion object
   distort->get,cd=cd, dmp=dmp, dmc=dmc, crVal=crVal, crPix=crPix
;will data fit?
   if ((N_ELEMENTS(dmc) GT N_ELEMENTS((*self.table)[region-1].dmc)) OR $
       (N_ELEMENTS(dmp) GT N_ELEMENTS((*self.table)[region-1].dmp))) $
      then midiSetError, 'Distortion matrix too large to store'
;copy into table
   (*self.table)[region-1].cd =cd
   (*self.table)[region-1].crVal = crVal
   (*self.table)[region-1].crPix = crPix
   (*self.table)[region-1].dmp = dmp
   (*self.table)[region-1].dmc = dmc
RETURN
END

FUNCTION imageDetector::clone, iErr=iErr
;create a new imageDetector object that is identical to myself
   detector = obj_new('imageDetector', self.nRegion, $
   nDetect=self.nDetect, instrument=self.instrument, $
   maxTel=self.maxTel, numDim=self.numDim, $
   maxCoeff=self.maxCoeff, iErr=iErr)
   if (iErr NE 0) then midiSetError,'Detector cloning failed',/notInitial
;copy header
   detector->replaceHeader,self.head->clone()
;copy table
   outTable = detector->table()
;copy tag by tag.  This is safer than a blind copy
;because it will noticably crash if the output structure
;is different from input
;copy by name because they might be in the wrong order
   intagNames = TAG_NAMES(*self.table)
   outtagNames= TAG_NAMES(*outtable)
   nTags = N_ELEMENTS(intagNames) - 1
   for iTag = 0, nTags do begin
      outTag = MIN(WHERE(outtagNames EQ intagNames[iTag]))
      (*outTable).(outTag)=REFORM((*self.table).(iTag))
   endfor
RETURN, detector
END

PRO ImageDetector::windowWCS, region, C1, C2, type1, type2, iErr=iErr
;return the World Coordinate System coordinate values for all
;pixels in a specified region.
;INPUTS
;  region  int  which region are you interested in
;OUTPUTS
;   C1  dbl arr  The values of the 1st WC at each pixel
;   C2  dbl arr  The values of the 2nd WC at each pixel
;   type1 str    The type of the 1st WC
;   type2 str    The type of the 2nd WC
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN
   endif         ; handle actual error
;
   if (region LT 1 OR (region GT self.nRegion)) then midiSetError, $
      'region specified outside permitted range'
   if (NOT PTR_VALID(self.table)) then midiSetError, $
      'Detector Table Not Initialized'
   coordDist = obj_new('coordDistortion', self, region=region, $
      iErr=iErr)
   if (iErr NE 0) then midiSetError, $
      'failed to intialize coordDistortion object for region ' $
        + string(Region), /notInitial
   coordDist->coordGrid, C1, C2, iErr=iErr
   if (iErr NE 0) then midiSetError, 'Coordinate calculation failed', $
      /notInitial
   type1 = (*self.table)[region-1].cType[0]
   type2 = (*self.table)[region-1].cType[1]
RETURN
END


PRO ImageDetector::newFile, fileName, iErr=iErr, priHead=priHead
   self->fitsTable::newFile, fileName, /table, iErr=iErr, priHead=priHead
RETURN
END

PRO ImageDetector::appendToFile, input, iErr=iErr
   self->fitsTable::appendToFile, input, /table, iErr=iErr
RETURN
END
