;
; Copyright 2005, 2006 University of Leiden.
;
; This file is part of MIA+EWS.
;
; MIA+EWS is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.
;
; MIA+EWS is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.
;
; You should have received a copy of the GNU General Public License
; along with MIA; if not, write to the Free Software
; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;
;********************************************************
;#class#
;oirVis
;#description#
;methods of OIR FITS VIS
;#end_class#
;*******************************************************

PRO oi_VIS__DEFINE
;********************************************************
;#structure#
; oi_vis
;#inheritance#
; fitsTable
;#description#
;define structures for oi_Vis class
;#structure_text#
oir3={oi_vis, $   ;#class members#
INHERITS FITSTABLE, $;#class inheritance#
revision:1, $        ;int   & revision number of this format
insname:'       ', $; str   & instrument/disperser name
arrname:'       ', $; str   & array (configuration) name
nWave:0,          $;int   & number of wavelengths
dateObs:'',         $;str   & date of observation
dataTemplate:PTR_NEW()  $; array $ what one row looks like
}
;#end_structure#
;********************************************************
RETURN
END

PRO oiVisStruct__DEFINE
;********************************************************
;#structure#
; oiVisStruct
;#inheritance#
; None
;#description#
; template row in image data table
;#structure_text#
oir4={oiVisStruct, $ ;#FITS TABLE structure#
target_id:0,       $ ;int & pointer to target table
time:0.0D,         $ ;dbl  & time of exposure
mjd:0.0D,          $ ;dbl  & mjd of exposure
int_time:0.0,      $ ;float & length of exposure
ucoord:0.0D,       $ ;dbl  & U-coordinate(meters)
vcoord:0.0D,       $ ;dbl  & V-coordinate(meters)
sta_index:intarr(2) $     ; station indices
;visamp              ;visibility amp; (nWave not yet known)
;visamperr           ;visibility amperr; (nWave not yet known)
;visphi              ;visibility phase; (nWave not yet known)
;visphierr           ;visibility phase error; (nWave not yet known)
;flag                ;flags; 0 = OK; (nWave not yet known)
}
;#end_structure#
;********************************************************
RETURN
END

PRO oi_vis::cleanup
  PTR_FREE,self.dataTemplate
  self->FitsTable::cleanup      ; cleanup inherited object too
END

FUNCTION oi_vis::init, input, insname=insname, arrname=arrname, $
   version=version, dateObs=dateObs, iErr=iErr
;construct OI_VIS object
;polymorphic input can be of the following types:
;(1)  string
;     This is assumed to be a bin table file name.  The file is opened,
;     SELF is then created as a special case of FITSTABLE, ready to read.
;(2)  numeric
;     This is assumed to be NWAVE, the number of entries per row
;     A template with this size is created and stored
;establish error handler
   cErr = 0
   catch, cErr
   if (cErr NE 0) then begin
;supress further handling at this level
      catch, /cancel
      midiCatchError
      iErr = 1
RETURN,0
   endif         ; handle actual error
   iErr = 0
   sInput = SIZE(input)
;string input; treat as file name
   if ((sInput[0] EQ 0) AND (sInput[1] EQ 7)) then begin
;incarnate myself as a FITSTABLE,
      if(0 EQ self->FITSTABLE::init(input, extName='OI_VIS', $
         extNumber=version, iErr=iErr)) $
         then midiSetError, 'reading vis from input file:'+input $
        +'failed' ,/notInitial
      self.insname  = (self.head)->getPar('INSNAME', matches)
      self.arrname  = (self.head)->getPar('ARRNAME', matches)
      self.Revision = self.head->getPar('OI_REVN', matches)
      self.dateObs  = (self.head)->getPar('DATE-OBS', matches)
      if (matches LE 0) then  begin
         self.dateObs = '1947-08-23T14:00:00:00.000'
         (self.head)->addPar,'DATE-OBS', self.dateObs
      endif
RETURN,1
   endif     $         ; string input
;numeric input
   else if (sInput[1+sInput[0]] ge 2 AND (sInput[1+sInput[0]] le 4))  then begin
;incarnate myself using template 
      self.nWave   = input
      dataTemplate = CREATE_STRUCT({oiVisStruct}, $
         'visamp',dblarr(self.nWave), $
         'visamperr',dblarr(self.nWave), $
         'visphi',dblarr(self.nWave),$
         'visphierr',dblarr(self.nWave),$
         'flag',bytarr(self.nWave))
         self.dataTemplate = PTR_NEW(dataTemplate)
      if (KEYWORD_SET(insname)) then self.insname = insname $
         else self.insname = 'UNKNOWN'
      if (KEYWORD_SET(arrname)) then self.insname = arrname $
         else self.arrname = 'UNKNOWN'
      if (KEYWORD_SET(dateObs)) then self.dateObs = dateObs $
         else self.dateObs = '1947-08-23T14:00:00:00.000'
      if(0 EQ self->FITSTABLE::init(dataTemplate, logCol=["FLAG"],$
         iErr=iErr, extName='OI_VIS' ))then $
     midiSetError, 'construction from OI_VIS object failed',$
     /notInitial
;set header info 
      self.head->addPar,'OI_REVN', 1L
      self.head->addPar,'INSNAME', self.insname
      self.head->addPar,'ARRNAME', self.arrname
      self.head->addPar,'DATE-OBS', self.dateObs
   RETURN,1
   endif              ; object input
;not recognized input
   midiSetError,'input of unrecognized type'
RETURN,0
END

FUNCTION oi_vis::dataTemplate
RETURN, *self.dataTemplate
END
