/*******************************************************************************
amoeba.c

USNO/NRL Optical Interferometer
3450 Massachusetts Avenue NW
Washington DC 20392-5420

System: Data Reduction
Subsystem: CHAMELEON

Description:
Package of C-routines for AMOEBA callable from IDL via CALL_EXTERNAL.

Modification history:
24-Feb-1995  C.A.Hummel, file created
*******************************************************************************/

/* Includes */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define NRANSI
#include "nrutil.h"
#include "export.h"
#include "novas.h"

/* Defines */

#define PI 3.141592653589793
#define NEG(x) ((x) > 0 ? -(x) : (x))
#define MIN(x,y) ((x) > (y) ? (y) : (x))
#define MAX(x,y) ((x) > (y) ? (x) : (y))
#define RAD(x) ((x)/360.*2.*PI)

/* Function prototypes */

/* ...callable from IDL */

int nightbuffer(int argc, char *argp[]);
int svdwave(int argc, char *argp[]);
int svbwave(int argc, char *argp[]);
int gaussjwave(int argc, char *argp[]);
int bessjyidl(int argc, char *argp[]);
int marquardt(int argc, char *argp[]);
int marquardt_init(int argc, char *argp[]);
int marquardt_free(int argc, char *argp[]);
int store_ell(int argc, char *argp[]);
int free_ell(int argc, char *argp[]);
int splineint(int argc, char *argp[]);
int gsmooth8(int argc, char *argp[]);

/* ...called from within C code */

void ellfuncs(double x[], double a[], double ymod[], double **dyda, int ma,
		int ndata, int ia[]);
void orbfuncs(double jd[], double a[], double rmod[], double **drda, int ma,
		int ndata, int ia[]);
int fixell(double p[], double r[], int ndata, double a[], double rfit[]);
int true2app(double jd[], int ndata, double o_parms[], 
				double rho[], double theta[]);

/* Global variables; storage for data used in fit routines */

static double **covar,**alpha;
static double *pfEllipse_x,*pfEllipse_y;
double *pfXdata,*pfYdata;

/*----------------------------------------------------------------------------*/
int nightbuffer(int argc, char *argp[]){
	/*
	Store or retrieve one night's data into/from a temporary buffer.
	Also free memory, if requested through piMode parameter.
	Mode=0		Free memory
	Mode=10		Store new night
	Mode=11		Overwrite existing night
	Mode=12		Internal mode; store data, do not allocate space
	Mode=20		Locate memory and load data
	Mode=21		Internal Mode, load data, do not locate memory
	*/

	long *piDims,*piMode;
	char **pcBuffer;
	IDL_STRING *pcDate,*pzBuffer;
	int i,j,iType,iLen,iNumElements,iFactor,iNumBytes;
	static int iNumPointer=-1,iPointer;
	static char **pcPointerArray=NULL,**pcDates=NULL;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piDims=   ((long**)argp)[0];
	pcBuffer=((char***)argp)[1];
	pzBuffer=((IDL_STRING**)argp)[1];
	pcDate=  ((IDL_STRING**)argp)[2];iLen=strlen(pcDate[0].s);
	piMode=   ((long**)argp)[3];

/*	Free allocated memory for mode=0 */
	if(piMode[0]==0){
		for(i=0;i<iNumPointer;i++){
			free(pcPointerArray[i]);
			free(pcDates[i]);
			}
		free(pcPointerArray);
		pcPointerArray=NULL;
		free(pcDates);
		pcDates=NULL;
		iNumPointer=-1;
		return(0);
		}

	/* Determine size of data block */
	iType=piDims[piDims[0]+1];
	iNumElements=piDims[piDims[0]+2];
	iFactor=1;
	switch(iType){
		case 1:iFactor=1;break;
		case 2:iFactor=2;break;
		case 3:iFactor=4;break;
		case 4:iFactor=4;break;
		case 5:iFactor=8;break;
		case 6:iFactor=8;break;
		case 7:{
			for(i=0;i<iNumElements;i++)
				iFactor=MAX(iFactor,strlen(pzBuffer[i].s));
			iFactor+=1;
			break;
			}
		}
	iNumBytes=iNumElements*iFactor;

/*	Store night */
	if(piMode[0]/10==1){

	/* Allocate/reallocate memory for pointer array */
	if((piMode[0]-10)==0){
	if((pcPointerArray=(char**)realloc(pcPointerArray,
				(++iNumPointer+1)*sizeof(char**)))==NULL){
		fprintf(stderr,"Error allocating PointerArray!\n");
		return(1);
		}

	/* Allocate memory of required amount */
	if((pcPointerArray[iNumPointer]=(char*)malloc(iNumBytes))==NULL){
		fprintf(stderr,"Error allocating memory!\n");
		return(1);
		}

	/* Allocate/reallocate memory for pcDates array */
	if((pcDates=(char**)realloc(pcDates,
				(iNumPointer+1)*sizeof(char**)))==NULL){
		fprintf(stderr,"Error allocating pcDates array!\n");
		return(1);
		}

	/* Allocate memory for copy of pcDate */
	if((pcDates[iNumPointer]=(char*)malloc(iLen+1))==NULL){
		fprintf(stderr,"Error allocating memory!\n");
		return(1);
		}

	iPointer=iNumPointer;
	}

/*	If overwrite mode requested, locate specific night */
	else if((piMode[0]-10)==1){
		for(iPointer=0;iPointer<iNumPointer;iPointer++)
			if(strncmp(pcDate[0].s,pcDates[iPointer],iLen)==0)break;
		if(iPointer==iNumPointer){
		 fprintf(stderr,"Could not find requested date:%s\n",pcDate[0].s);
		 return(1);
		 }
		piMode[0]=12;
		}
	else iPointer++;

	/* Copy pcDate into newly allocated block of memory */
	memset(pcDates[iPointer],'\0',iLen+1);
	memcpy(pcDates[iPointer],pcDate[0].s,iLen);

	/* Store data */
	if(iType==7){
		for(i=0;i<iNumElements;i++){
			memcpy(pcPointerArray[iPointer]+i*iFactor,
				pzBuffer[i].s,strlen(pzBuffer[i].s));
			memset(pcPointerArray[iPointer]+i*iFactor
				+strlen(pzBuffer[i].s),'\0',1);
                	}
		}
	else{
		memcpy(pcPointerArray[iPointer],pcBuffer,iNumBytes);
		}

	return(0);
	}

/* 	Load night modes */
	else if(piMode[0]/10==2){

	/* First call has to locate beginning of memory block for req. night */
	if((piMode[0]-20)==0){
		for(iPointer=0;iPointer<iNumPointer;iPointer++)
			if(strncmp(pcDate[0].s,pcDates[iPointer],iLen)==0)break;
		if(iPointer==iNumPointer){
		 fprintf(stderr,"Could not find requested date:%s\n",pcDate[0].s);
		 return(1);
		 }
		piMode[0]=21;
		}
	else iPointer++;
	
	if(iType==7){
		for(j=0;j<iNumElements;j++){
			memcpy(pzBuffer[j].s,
				pcPointerArray[iPointer]+j*iFactor,
				 strlen(pcPointerArray[iPointer]+j*iFactor));
                	}
		}
	else{
		memcpy(pcBuffer,pcPointerArray[iPointer],iNumBytes);
		}
		
	return(0);
	}

	return(0);
	}
/*----------------------------------------------------------------------------*/
int svdwave(int argc, char *argp[]){
	/*
	Wrapper for svdecomp, callable from IDL
	*/

	double *pdfA,*pdfW,*pdfV;
	int *piM,*piN;
	int i,j;
	double **a,**v;
	double *w;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	pdfA=((double**)argp)[0];
	piM=    ((int**)argp)[1];
	piN=    ((int**)argp)[2];
	pdfW=((double**)argp)[3];
	pdfV=((double**)argp)[4];

	a=dmatrix(1,piM[0],1,piN[0]);
	v=dmatrix(1,piN[0],1,piN[0]);
	w=dvector(1,piN[0]);

	for(i=0;i<piM[0];i++){
		for(j=0;j<piN[0];j++){
			a[i+1][j+1]=pdfA[i+piM[0]*j];
			}
		}
	for(i=0;i<piN[0];i++){
		for(j=0;j<piN[0];j++){
			v[i+1][j+1]=pdfV[i+piN[0]*j];
			}
		}

	svdecomp(a,piM[0],piN[0],w,v);

	for(i=0;i<piM[0];i++){
		for(j=0;j<piN[0];j++){
			pdfA[i+piM[0]*j]=a[i+1][j+1];
			}
		}
	for(i=0;i<piN[0];i++){
		pdfW[i]=w[i+1];
		for(j=0;j<piN[0];j++){
			pdfV[i+piN[0]*j]=v[i+1][j+1];
			}
		}

	free_dmatrix(a,1,piM[0],1,piN[0]);
	free_dmatrix(v,1,piN[0],1,piN[0]);
	free_dvector(w,1,piN[0]);
	
	return(0);

	}
/*----------------------------------------------------------------------------*/
int svbwave(int argc, char *argp[]){
	/*
	Wrapper for svdbacks, callable from IDL
	*/

	double *pdfU,*pdfW,*pdfV,*pdfB,*pdfX;
	int *piM,*piN;
	int i,j;
	double **u,**v;
	double *w,*b,*x;

        if (argc != 7) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	pdfU=((double**)argp)[0];
	pdfW=((double**)argp)[1];
	pdfV=((double**)argp)[2];
	piM=    ((int**)argp)[3];
	piN=    ((int**)argp)[4];
	pdfB=((double**)argp)[5];
	pdfX=((double**)argp)[6];

	u=dmatrix(1,piM[0],1,piN[0]);
	v=dmatrix(1,piN[0],1,piN[0]);
	w=dvector(1,piN[0]);
	b=dvector(1,piM[0]);
	x=dvector(1,piN[0]);

	for(i=0;i<piM[0];i++){
		b[i+1]=pdfB[i];
		for(j=0;j<piN[0];j++){
			u[i+1][j+1]=pdfU[i+piM[0]*j];
			}
		}
	for(i=0;i<piN[0];i++){
		w[i+1]=pdfW[i];
		for(j=0;j<piN[0];j++){
			v[i+1][j+1]=pdfV[i+piN[0]*j];
			}
		}
	svdbacks(u,w,v,piM[0],piN[0],b,x);

	for(i=0;i<piN[0];i++)pdfX[i]=x[i+1];

	free_dmatrix(u,1,piM[0],1,piN[0]);
	free_dmatrix(v,1,piN[0],1,piN[0]);
	free_dvector(w,1,piN[0]);
	free_dvector(b,1,piM[0]);
	free_dvector(x,1,piN[0]);
	
	return(0);

	}
/*----------------------------------------------------------------------------*/
int gaussjwave(int argc, char *argp[]){
	/*
	Wrapper for gaussjordan, callable from IDL
	*/

	double *pdfA,*pdfB;
	int *piM,*piN;
	int i,j;
	double **a,**b;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	pdfA=((double**)argp)[0];
	piN=    ((int**)argp)[1];
	pdfB=((double**)argp)[2];
	piM=    ((int**)argp)[3];

	a=dmatrix(1,piN[0],1,piN[0]);
	b=dmatrix(1,piN[0],1,piM[0]);

	for(i=0;i<piN[0];i++){
		for(j=0;j<piN[0];j++){
			a[i+1][j+1]=pdfA[i+piN[0]*j];
			}
		}
	for(i=0;i<piN[0];i++){
		for(j=0;j<piM[0];j++){
			b[i+1][j+1]=pdfB[i+piN[0]*j];
			}
		}

	gaussjordan(a,piN[0],b,piM[0]);

	for(i=0;i<piN[0];i++){
		for(j=0;j<piN[0];j++){
			pdfA[i+piN[0]*j]=a[i+1][j+1];
			}
		}
	for(i=0;i<piN[0];i++){
		for(j=0;j<piM[0];j++){
			pdfB[i+piN[0]*j]=b[i+1][j+1];
			}
		}

	free_dmatrix(a,1,piN[0],1,piN[0]);
	free_dmatrix(b,1,piN[0],1,piM[0]);
	
	return(0);

	}
/*----------------------------------------------------------------------------*/
int bessjyidl(int argc, char *argp[]){
 
	long *piN;
        double *pdfArg,*pdfArgNu,*pdfResult;
        double x,y,z;
	long i;
 
        if (argc != 4) {
                fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
                return(1);
                }
 
	piN=	       ((long**)argp)[0];
        pdfArg=      ((double**)argp)[1];
        pdfArgNu=    ((double**)argp)[2];
        pdfResult=   ((double**)argp)[3];
 
	for (i=0;i<piN[0];i++)
        besseljy(pdfArg[i],pdfArgNu[0],pdfResult+i,&x,&y,&z);

        }
/*----------------------------------------------------------------------------*/
int marquardt(int argc, char *argp[]){
	/*
	Wrapper for Marquardt-Levenberg routine.
	*/

	double *pfSigma,*pfParms,*pfCovar,*pfAlpha,*pfAlamda,*pfTol,*pfChisq;
	int *piNdata,*piIA,*piMA,*piMode;
	IDL_STRING *azFunction;
	void (*pFunction)(double x[],double a[],double ymod[],double **dyda,
		int ma,int ndata,int ia[]);

	int i,j;

        if (argc != 14) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	pfXdata=        ((double**)argp)[0];
	pfYdata=        ((double**)argp)[1];
	pfSigma=        ((double**)argp)[2];
	piNdata=           ((int**)argp)[3];
	pfParms=        ((double**)argp)[4];
	piIA=              ((int**)argp)[5];
	piMA=              ((int**)argp)[6];
	pfCovar=        ((double**)argp)[7];
	pfAlpha=        ((double**)argp)[8];
	pfChisq=        ((double**)argp)[9];
	azFunction=((IDL_STRING**)argp)[10];
	pfAlamda=      ((double**)argp)[11];
	pfTol=         ((double**)argp)[12];
	piMode=           ((int**)argp)[13];

	for(i=0;i<piMA[0];i++){
		for(j=0;j<piMA[0];j++){
			covar[i+1][j+1]=pfCovar[i+piMA[0]*j];
			alpha[i+1][j+1]=pfAlpha[i+piMA[0]*j];
			}
		}

	pFunction=NULL;
	if(strncmp(azFunction[0].s,"ellfuncs",8)==0)pFunction=&ellfuncs;
	if(strncmp(azFunction[0].s,"orbfuncs",8)==0)pFunction=&orbfuncs;
	if(pFunction==NULL){
		fprintf(stderr,"Function not defined: %s",azFunction[0].s);
		return(1);
		}
	mrqlmin(pfXdata,pfYdata,pfSigma,piNdata[0],pfParms,piIA,piMA[0],
		covar,alpha,pfChisq,pFunction,pfAlamda,pfTol,piMode[0]);

	for(i=0;i<piMA[0];i++){
		for(j=0;j<piMA[0];j++){
			pfCovar[i+piMA[0]*j]=covar[i+1][j+1];
			pfAlpha[i+piMA[0]*j]=alpha[i+1][j+1];
			}
		}
	
	return(0);

	}
/*----------------------------------------------------------------------------*/
int marquardt_init(int argc, char *argp[]){

	int *piMA;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piMA=((int**)argp)[0];

	covar=dmatrix(1,piMA[0],1,piMA[0]);
	alpha=dmatrix(1,piMA[0],1,piMA[0]);

	}
/*----------------------------------------------------------------------------*/
int marquardt_free(int argc, char *argp[]){

	int *piMA;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piMA=((int**)argp)[0];

	free_dmatrix(covar,1,piMA[0],1,piMA[0]);
	free_dmatrix(alpha,1,piMA[0],1,piMA[0]);

	}
/*----------------------------------------------------------------------------*/
int store_ell(int argc, char *argp[]){

	double *pfEx,*pfEy;
	int *piNumPoints;
	
        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	pfEx=    ((double**)argp)[0];
	pfEy=    ((double**)argp)[1];
	piNumPoints=((int**)argp)[2];

	pfEllipse_x=dvector(1,piNumPoints[0]);
	pfEllipse_y=dvector(1,piNumPoints[0]);

	memcpy(pfEllipse_x+1,pfEx+1,piNumPoints[0]*sizeof(double));
	memcpy(pfEllipse_y+1,pfEy+1,piNumPoints[0]*sizeof(double));

	return(0);
	}	
/*----------------------------------------------------------------------------*/
int free_ell(int argc, char *argp[]){

	int *piNumPoints;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNumPoints=((int**)argp)[0];

	free_dvector(pfEllipse_x,1,piNumPoints[0]);
	free_dvector(pfEllipse_y,1,piNumPoints[0]);

	return(0);
	}
/*----------------------------------------------------------------------------*/
void ellfuncs(double p[], double a[], double rmod[], double **drda, int ma,
		int ndata, int ia[]){

	int i,j;
	double deltaa[6];
	double *b,*r_high,*r_low;

	deltaa[1]=0.1;			/* Center x */
	deltaa[2]=0.1;			/* Center y */
	deltaa[3]=0.1;			/* Major axis */
	deltaa[4]=0.1;			/* Minor axis */
	deltaa[5]=0.02;			/* Position angle */

	b=dvector(1,ma);
	for(j=1;j<=ma;j++)b[j]=a[j];
	r_high=dvector(1,ndata);
	r_low=dvector(1,ndata);

	for(j=1;j<=ma;j++){
		if(ia[j]){
			b[j]=a[j]-deltaa[j];
			fixell(p,pfYdata,ndata,b,r_low);
			for(i=1;i<=ndata;i++)r_low[i]-=pfYdata[i];
			b[j]=a[j]+deltaa[j];
			fixell(p,pfYdata,ndata,b,r_high);
			for(i=1;i<=ndata;i++)r_high[i]-=pfYdata[i];
			b[j]=a[j];
			for(i=1;i<=ndata;i++){
				drda[i][j]=(r_high[i]-r_low[i])/(2*deltaa[j]);
				}
			}
		}

	free_dvector(b,1,ma);
	free_dvector(r_high,1,ndata);
	free_dvector(r_low,1,ndata);

	fixell(pfXdata,pfYdata,ndata,a,rmod);

	}
/*----------------------------------------------------------------------------*/
void orbfuncs(double jd[], double a[], double rmod[], double **drda, int ma,
		int ndata, int ia[]){

	int i,j;
	double deltaa[8];
	double *b,*r_high,*r_low,*rho,*theta,x,y;

	deltaa[1]=0.1;				/* Major axis increment */
	deltaa[2]=0.02;				/* Eccentricity */
	deltaa[3]=0.02;				/* Inclination */
	deltaa[4]=0.005/(pow(a[2],2)+0.005);	/* Periastron */
	deltaa[5]=0.02/(sin(a[3])+0.02);	/* Node */
	deltaa[6]=a[6]/1000;			/* Period */
	deltaa[7]=a[6]/1000;			/* Epoch */

	b=dvector(1,ma);
	for(j=1;j<=ma;j++)b[j]=a[j];

	r_high=dvector(1,ndata);
	r_low=dvector(1,ndata);
	rho=dvector(1,ndata);
	theta=dvector(1,ndata);

	for(j=1;j<=ma;j++){
		if(ia[j]){
			b[j]=a[j]-deltaa[j];
			true2app(jd,ndata,b,rho,theta);
			for(i=1;i<=ndata;i++){
				x=rho[i]*sin(theta[i]);
				y=rho[i]*cos(theta[i]);
				r_low[i]=sqrt(pow(pfEllipse_x[i]-x,2)
				             +pow(pfEllipse_y[i]-y,2));
				}
			b[j]=a[j]+deltaa[j];
			true2app(jd,ndata,b,rho,theta);
			for(i=1;i<=ndata;i++){
				x=rho[i]*sin(theta[i]);
				y=rho[i]*cos(theta[i]);
				r_high[i]=sqrt(pow(pfEllipse_x[i]-x,2)
				              +pow(pfEllipse_y[i]-y,2));
				}
			b[j]=a[j];
			for(i=1;i<=ndata;i++){
				drda[i][j]=(r_high[i]-r_low[i])/(2*deltaa[j]);
				}
			}
		}

	true2app(jd,ndata,a,rho,theta);
	for(i=1;i<=ndata;i++){
		x=rho[i]*sin(theta[i]);
		y=rho[i]*cos(theta[i]);
		rmod[i]=sqrt(pow(pfEllipse_x[i]-x,2)
			    +pow(pfEllipse_y[i]-y,2));
		}

	free_dvector(b,1,ma);
	free_dvector(r_high,1,ndata);
	free_dvector(r_low,1,ndata);
	free_dvector(rho,1,ndata);
	free_dvector(theta,1,ndata);

	}
/*----------------------------------------------------------------------------*/
int fixell(double p[], double r[], int ndata, double a[], double rfit[]){

	int i;
	double dx,dy,esqr;

	for(i=1;i<=ndata;i++){
		dx=pfEllipse_x[i]-a[1];
		dy=pfEllipse_y[i]-a[2];
		r[i]=sqrt(dx*dx+dy*dy);
		p[i]=atan2(dx,dy)-a[5];
		esqr=1-a[4]*a[4]/(a[3]*a[3]);
		rfit[i]=sqrt(a[4]*a[4]/(1-esqr*pow(cos(p[i]),2)));
		}

	return(0);
	}
/*----------------------------------------------------------------------------*/
int true2app(double jd[], int ndata, double o_parms[], 
				double rho[], double theta[]){

	int j,k;
	double M_anom,E_anom,T_anom;
	double a,e,i,w,n,p,t;

	a=o_parms[1];
	e=o_parms[2];
	i=o_parms[3];
	w=o_parms[4];
	n=o_parms[5];
	p=o_parms[6];
	t=o_parms[7];

	for(j=1;j<=ndata;j++){
		M_anom=2.0*PI*fmod(jd[j]-t,p)/p;
		E_anom=M_anom+e*sin(M_anom)+pow(e,2)/2*sin(2*M_anom);
		for(k=0;k<=4;k++){
			E_anom+=(M_anom-E_anom+e*sin(E_anom))/(1-e*cos(E_anom));
			}
		T_anom=2*atan(sqrt((1+e)/(1-e))*tan(E_anom/2));
		theta[j]=atan(tan(T_anom+w)*cos(i))+n;
		rho[j]=a*(1-pow(e,2))/(1+e*cos(T_anom))
			*cos(T_anom+w)/cos(theta[j]-n);
		if(rho[j]<0){
			theta[j]+=PI;
			rho[j]=fabs(rho[j]);
			}
		}

	return(0);
	}
/*----------------------------------------------------------------------------*/
int splineint(int argc, char *argp[]){
 
	long *piN,*piM;
        double *pdfX,*pdfY,*pdfY2,*pdfT,*pdfR;
	double yp1,ypn;
	long i;
 
        if (argc != 7) {
                fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
                return(1);
                }
 
        pdfX= (((double**)argp)[0])-1;
        pdfY= (((double**)argp)[1])-1;
        pdfY2=(((double**)argp)[2])-1;
        pdfT= (((double**)argp)[3]);
        pdfR= ((double**)argp)[4];
	piN=    ((long**)argp)[5];
	piM=    ((long**)argp)[6];
 
	yp1=1e30;
	ypn=1e30;

	spline0(pdfX,pdfY,piN[0],yp1,ypn,pdfY2);

	for (i=0;i<piM[0];i++){
		splint8(pdfX,pdfY,pdfY2,piN[0],pdfT[i],pdfR+i);
		}

	return(0);
        }
/*----------------------------------------------------------------------------*/
int gsmooth8(int argc, char *argp[]){
 
	long *piN;
        double *pdfT,*pdfY,*pdfS,*pdfR;
	long i,j,j0,j1;
	double w,W;
 
        if (argc != 5) {
                fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
                return(1);
                }
 
        pdfT= ((double**)argp)[0];
        pdfY= ((double**)argp)[1];
        pdfS= ((double**)argp)[2];
	piN=    ((long**)argp)[3];
        pdfR= ((double**)argp)[4];

	j1=0;
	for (i=0;i<piN[0];i++){
		W=0;
		j0=j1;
		for (j=j0;j<piN[0];j++) {
			w=exp(-pow((pdfT[j]-pdfT[i])/pdfS[0],2));
			if (w<4.5e-5 && W==0)j1=j+1;
			else{
				pdfR[i]+=(w*pdfY[j]);
				W+=w;
				}
			if (w<4.5e-5 && W!=0)break;
			}
		pdfR[i]/=W;
		}

	return(0);
        }
/*----------------------------------------------------------------------------*/
