/*******************************************************************************
cobra.c

Source:
-------
Dr. Christian A. Hummel
NPOI Division, Flagstaff Station
c/o US Naval Observatory
3450 Massachusetts Avenue NW
Washington DC 20392-5420

Description:
------------
NPOI raw data access for the OYSTER software.

Modification history:
27-Sep-2001  C.A.Hummel, file created
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "keyinc.h"
#include "export.h"
#include "cobra.h"
#include "packettype.h"
#include "swap.h"
#include "sys_config.h"

#define PI 3.141592653589793
#define RAD(x) ((x)/360.*2.*PI)
#define NEG(x) ((x) > 0 ? -(x) : (x))
#define MIN(x,y) ((x) > (y) ? (y) : (x))
#define MAX(x,y) ((x) > (y) ? (x) : (y))
#define STREQ(a,b) ( strcmp((a),(b)) == 0 )

#define LAMBDA_HENE (632.9914e-9)	/* HeNe wavelength in vacuum */
#define FDL_MET_UNITS (LAMBDA_HENE/512)
#define FDLLO_MET_UNITS (LAMBDA_HENE/512)
#define TWO223 (8388608)
#define FDLHI_MET_UNITS (LAMBDA_HENE*TWO223)

/* Define fixed-width integers to give machine-independence where needed.
   The definitions below hold for a typical, i.e. say Pentium PC, 32 bit
   architecture. But if we go to 64 bit, int might be 8 byte, so int32 will
   no longer be int, but a different type to be specified in the compiler
   for that machine.  */
 
#define uint32 unsigned int
#define uint16 unsigned short
#define uint8  unsigned char
#define int32  int
#define int16  short

char azTransferString[KSTRING_LEN];
#define IDL_StringStore(dest,source,numchar) \
 (memset(azTransferString,'\0',KSTRING_LEN), \
  strncpy(azTransferString,source,numchar), \
  IDL_StrStore(&(dest),azTransferString))

/* Global variables */

IDL_STRING *azRawFile;
long *piOffset,*piLength;
FILE *fp;
char *pBody,*pRec;
long i,j,k,l;

/* Function prototypes */

int scandisk(int argc, char *argp[]);
int readfileheader(int argc, char *argp[]);
int readnatcounts(int argc, char *argp[]);
int readfdlposition(int argc, char *argp[]);
int readfringedataversion1(int argc, char *argp[]);
int readfringebg(int argc, char *argp[]);
int readfringedark(int argc, char *argp[]);
int randpoisson(int argc, char *argp[]);

/*----------------------------------------------------------------------------*/
int scandisk(int argc, char *argp[]){

	long *piDay,*piTime,*piType,*piMaxDirEnt;
	DATAconHeader CurrentHeader,TempHeader;
	long iEntry,iOffset;

        if (argc != 7) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	azRawFile=((IDL_STRING**)argp)[0];
	piDay=          ((long**)argp)[1];
	piTime=         ((long**)argp)[2];
	piType=         ((long**)argp)[3];
	piLength=       ((long**)argp)[4];
	piOffset=       ((long**)argp)[5];
	piMaxDirEnt=    ((long**)argp)[6];

	if ((fp=fopen(azRawFile[0].s,"rb")) == NULL) return(-1);
	iOffset=0;
	for (iEntry=0; iEntry < piMaxDirEnt[0]; iEntry++){
    		if (fread(&TempHeader, sizeof(DATAconHeader), 1, fp) != 1) break;
    		piDay[iEntry] = swaplong(TempHeader.dchDay);
    		piTime[iEntry] = swaplong(TempHeader.dchTime);
    		piType[iEntry] = swaplong(TempHeader.dchType);
    		piLength[iEntry] = swaplong(TempHeader.dchLength)
				 - sizeof(DATAconHeader);
		piOffset[iEntry] = iOffset;
		iOffset += (piLength[iEntry] + sizeof(DATAconHeader));

    		if (piType[iEntry] == pktEND_OF_DISK) break;
    		if (fseek(fp, piLength[iEntry], SEEK_CUR)) {
		    piLength[iEntry] = 0;
		    break;
		    }
		}
	fclose(fp);
	return(0);
	}
/*----------------------------------------------------------------------------*/
int readfileheader(int argc, char *argp[]){

	IDL_STRING *azFileDate;

/* Keyinc parameters */

  	static char azDATAconVersion[KSTRING_LEN], azDATAconFileID[KSTRING_LEN], 
              azSiteName[KSTRING_LEN], azDate[KSTRING_LEN], azTime[KSTRING_LEN];
  	static int iDiskSeqID, iHdrSeqID, iTapeRecSize, iTapeBlockSize;

  	static keyrec KeyTable[]={
    	{ "DATACON_VERSION", azDATAconVersion, kstring, 1, NULL, "", "" },
    	{ "FILE_ID", azDATAconFileID, kstring, 1, NULL, "", "" },
    	{ "DISK_SEQUENCE_ID", &iDiskSeqID, kinteger, 1, NULL, "", "" },
    	{ "HEADER_SEQUENCE_ID", &iHdrSeqID, kinteger, 1, NULL, "", "" },
    	{ "SITE", azSiteName, kstring, 1, NULL, "", "" },
    	{ "FILE_DATE", azDate, kstring, 1, NULL, "", "" },
    	{ "FILE_START_TIME", azTime, kstring, 1, NULL, "", "" },
    	{ "TAPE_LOGICAL_RECORD_SIZE", &iTapeRecSize, kinteger, 1, NULL, "", "" },
    	{ "TAPE_FILE_MARK_INTERVAL", &iTapeBlockSize, kinteger, 1, NULL, "", "" }
    	};
  	kioStruc IOStream;
  	int iNpar=sizeof(KeyTable)/sizeof(keyrec);
  	char azReturnString[KSTRING_LEN];
  	char azErrBuff[1000];

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	azRawFile=((IDL_STRING**)argp)[0];
	piOffset=	((long**)argp)[1];
	piLength=	((long**)argp)[2];

	azFileDate=((IDL_STRING**)argp)[3];

/*	Locate and read packet */

	if ((fp=fopen(azRawFile[0].s,"rb")) == NULL) return(-1);
	if (fseek(fp,piOffset[0]+sizeof(DATAconHeader),SEEK_SET)) return(-1);
	pBody=malloc(piLength[0]);
	fread(pBody,piLength[0],1,fp);	
	fclose(fp);

/* 	Decode the packet */

  	IOStream.Type = kioBuffer;
  	IOStream.iOutputLineLen = 80;
  	IOStream.iMaxOutputChars = sizeof(azErrBuff);
  	IOStream.In.Buff = (char *)pBody;
  	IOStream.Out.Buff = azErrBuff;
  	azErrBuff[0] = '\0';
  	keyinc(KeyTable, iNpar, "no_prog", IOStream, azReturnString);
  	if (azErrBuff[0] != '\0' || !STREQ(azReturnString, "EOF")){
		printf("%s\n",azErrBuff);
    		return(-1);
    		}
	IDL_StringStore(azFileDate[0],azDate,strlen(azDate));

	free(pBody);
	}
/*----------------------------------------------------------------------------*/
int readsysconfig(int argc, char *argp[]){

	static SYS_CONFIG_C_STRUCT SysConfig;

	static double *pdfLatitude,*pdfLongitude,*pdfAltitude;
	static double *pdfEarthRadius,*pdfJ2,*pdfTDTUTC,*pdfUT1UTC,*pdfInstrCohInt;
	static long *piNumBin,*piBeamCID;

	static long *piNumSid,*piFDLTankID;
	static double *pdfStationCoord;
	static float *pfStroke;
	static IDL_STRING *azStation;

	static long *piMaxBas,*piMaxWav;
	static long *piNumOut,*piNumBas,*piNumBias,*piNumWav,*piFringeMod,*piBiasMod,*piInputPair;
	static double *pdfWavelength,*pdfWavelengthErr,*pdfChanWidth,*pdfChanWidthErr;

	long *piCommand;

/* Keyinc parameters */

	static int MaxWav, MaxFrFreq, MaxSid;
	static struct {
		char StationID[VAR_NUM_SID][KSTRING_LEN];
		} TransferString;

  	static keyrec KeyTable[]={
    	{ "MAX_WAV", &MaxWav, kinteger, 1, NULL, "", "" },
    	{ "MAX_FRFREQ", &MaxFrFreq, kinteger, 1, NULL, "", "" },
	{ "MAX_SID", &MaxSid, kinteger, 1, NULL, "", "" },

    	{ "Latitude", &(SysConfig.Latitude), kdouble, 1, NULL, "", "" }, 
    	{ "Longitude", &(SysConfig.Longitude), kdouble, 1, NULL, "", "" },
    	{ "Altitude", &(SysConfig.Altitude), kdouble, 1, NULL, "", "" },
    	{ "EarthRadius", &(SysConfig.EarthRadius), kdouble, 1, NULL, "", "" },
    	{ "J2", &(SysConfig.J2), kdouble, 1, NULL, "", "" },
    	{ "TDTminusUTC", &(SysConfig.TDTminusUTC), kdouble, 1, NULL, "", "" },
    	{ "UT1minusUTC", &(SysConfig.UT1minusUTC), kdouble, 1, NULL, "", "" },
    	{ "InstrCohInt", &(SysConfig.InstrCohInt), kdouble, 1, NULL, "", "" },
    	{ "NumBin", &(SysConfig.NumBin), kinteger, 1, NULL, "", "" },
    	{ "BeamCombinerID", &(SysConfig.BeamCombinerID), kinteger, 1, NULL, "", ""},

    	{ "InputBeam.NumSid", &(SysConfig.InputBeam.NumSid), kinteger,
		1, NULL, "", "" },
    	{ "InputBeam.StationCoord", &(SysConfig.InputBeam.StationCoord[0][0]), kdouble,
		VAR_NUM_SID * 4, NULL, "", "" },
    	{ "InputBeam.StationID", &(TransferString.StationID[0][0]), kstring,
		VAR_NUM_SID, NULL, "", "" },
    	{ "InputBeam.SiderostatID", &(SysConfig.InputBeam.SiderostatID[0]), kinteger,
		VAR_NUM_SID, NULL, "", "" },
    	{ "InputBeam.SidConID", &(SysConfig.InputBeam.SidConID[0]), kinteger,
		VAR_NUM_SID, NULL, "", "" },
    	{ "InputBeam.FDLTankID", &(SysConfig.InputBeam.FDLTankID[0]), kinteger,
		VAR_NUM_SID, NULL, "", "" },
    	{ "InputBeam.Stroke", &(SysConfig.InputBeam.Stroke[0]),
		kdouble, VAR_NUM_SID, NULL, "", "" },
    	{ "InputBeam.QuadCellID", &(SysConfig.InputBeam.QuadCellID[0]), kinteger,
		VAR_NUM_SID, NULL, "", "" },
    	{ "InputBeam.BCInputID", &(SysConfig.InputBeam.BCInputID[0]), kinteger,
		VAR_NUM_SID, NULL, "", "" },

    	{ "OutputBeam.NumOutBeam", &(SysConfig.OutputBeam.NumOutBeam), kinteger,
		1, NULL, "", "" },
    	{ "OutputBeam.BCOutputID", &(SysConfig.OutputBeam.BCOutputID[0]), kinteger,
		VAR_NUM_SPEC, NULL, "", "" },
        { "OutputBeam.NumSid", &(SysConfig.OutputBeam.NumSid), kinteger,
	        VAR_NUM_SPEC, NULL, "", "" },
        { "OutputBeam.StationID", &(SysConfig.OutputBeam.StationID), kinteger,
	        VAR_NUM_SPEC * VAR_NUM_SID, NULL, "", "" },
    	{ "OutputBeam.APDArrayID", &(SysConfig.OutputBeam.APDArrayID[0]), kinteger,
		VAR_NUM_SPEC, NULL, "", "" },
    	{ "OutputBeam.NumFringeFreq", &(SysConfig.OutputBeam.NumFringeFreq[0]), 
		kinteger, VAR_NUM_SPEC, NULL, "", "" },
    	{ "OutputBeam.NumBiasFreq", &(SysConfig.OutputBeam.NumBiasFreq[0]), 
		kinteger, VAR_NUM_SPEC, NULL, "", "" },
    	{ "OutputBeam.NumSpecChan", &(SysConfig.OutputBeam.NumSpecChan[0]), kinteger,
		VAR_NUM_SPEC, NULL, "", "" },
    	{ "OutputBeam.FringeMod", &(SysConfig.OutputBeam.FringeMod[0][0]), kinteger,
		VAR_NUM_SPEC * VAR_NUM_BAS, NULL, "", "" },
    	{ "OutputBeam.BiasMod", &(SysConfig.OutputBeam.BiasMod[0][0]), kinteger,
		VAR_NUM_SPEC * VAR_NUM_BAS, NULL, "", "" },
    	{ "OutputBeam.InputPair", &(SysConfig.OutputBeam.InputPair[0][0][0]), kinteger,
        	VAR_NUM_SPEC * VAR_NUM_BAS * 2, NULL, "", "" },
    	{ "OutputBeam.Wavelength", &(SysConfig.OutputBeam.Wavelength[0][0]),
		kdouble, VAR_NUM_SPEC * VAR_NUM_CHAN, NULL, "", "" },
    	{ "OutputBeam.WavelengthErr", &(SysConfig.OutputBeam.WavelengthErr[0][0]),
		kdouble, VAR_NUM_SPEC * VAR_NUM_CHAN, NULL, "", "" },
    	{ "OutputBeam.ChanWidth", &(SysConfig.OutputBeam.ChanWidth[0][0]),
		kdouble, VAR_NUM_SPEC * VAR_NUM_CHAN, NULL, "", "" },
    	{ "OutputBeam.ChanWidthErr", &(SysConfig.OutputBeam.ChanWidthErr[0][0]),
		kdouble, VAR_NUM_SPEC * VAR_NUM_CHAN, NULL, "", "" }
    	};

  	kioStruc IOStream;
  	int iNpar=sizeof(KeyTable)/sizeof(keyrec);
  	char azReturnString[KSTRING_LEN];
  	char azErrBuff[1000];

	int iSid,iBas,iOut,iWav,iSid1,iSid2,i,j;
	float fFringeMod;
        if (argc > 32) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	azRawFile=((IDL_STRING**)argp)[0];
	piOffset=	((long**)argp)[1];
	piLength=	((long**)argp)[2];

	piCommand=          ((long**)argp)[3];
	piNumSid=           ((long**)argp)[4];
	piNumOut=           ((long**)argp)[5];
	piMaxWav=           ((long**)argp)[6];
	piMaxBas=           ((long**)argp)[7];

	if (piCommand[0]==1){
	pdfLatitude=      ((double**)argp)[8];
	pdfLongitude=     ((double**)argp)[9];
	pdfAltitude=     ((double**)argp)[10];
	pdfEarthRadius=  ((double**)argp)[11];
	pdfJ2=           ((double**)argp)[12];
	pdfTDTUTC=       ((double**)argp)[13];
	pdfUT1UTC=       ((double**)argp)[14];
	pdfInstrCohInt=  ((double**)argp)[15];
	piNumBin=          ((long**)argp)[16];
	piBeamCID=         ((long**)argp)[17];

	pdfStationCoord= ((double**)argp)[18];
	azStation=   ((IDL_STRING**)argp)[19];
	pfStroke=         ((float**)argp)[20];
	piFDLTankID=       ((long**)argp)[21];

	piFringeMod=       ((long**)argp)[22];
	piBiasMod=         ((long**)argp)[23];
	piInputPair=       ((long**)argp)[24];
	pdfWavelength=   ((double**)argp)[25];
	pdfWavelengthErr=((double**)argp)[26];
	pdfChanWidth=    ((double**)argp)[27];
	pdfChanWidthErr= ((double**)argp)[28];
	piNumBas=          ((long**)argp)[29];
	piNumBias=         ((long**)argp)[30];
	piNumWav=          ((long**)argp)[31];
	} else {

/*	Locate and read packet */

	if ((fp=fopen(azRawFile[0].s,"rb")) == NULL) return(-1);
	if (fseek(fp,piOffset[0]+sizeof(DATAconHeader),SEEK_SET)) return(-1);
	pBody=malloc(piLength[0]+2);
	fread(pBody,piLength[0],1,fp);	
	*(pBody+piLength[0])='/';
	*(pBody+piLength[0]+1)='\0';
	fclose(fp);

/* 	Decode the packet */

	memset(&TransferString, 0, sizeof(TransferString));
  	IOStream.Type = kioBuffer;
  	IOStream.iOutputLineLen = 80;
  	IOStream.iMaxOutputChars = sizeof(azErrBuff);
  	IOStream.In.Buff = (char *)pBody;
  	IOStream.Out.Buff = azErrBuff;
  	azErrBuff[0] = '\0';
  	keyinc(KeyTable, iNpar, "no_prog", IOStream, azReturnString);
  	if (azErrBuff[0] != '\0' || !STREQ(azReturnString, "/")){
		printf("%s\n",azErrBuff);
    		return(-1);
    		}
	free(pBody);

/* 	Check the basic numbers to make sure nothing overflowed */
	
  	if (SysConfig.InputBeam.NumSid > VAR_NUM_SID
         || SysConfig.OutputBeam.NumOutBeam > VAR_NUM_SPEC
         || MaxWav > VAR_NUM_CHAN
	 || MaxSid > VAR_NUM_SID
         || MaxFrFreq > VAR_NUM_BAS ){
    		return(-1);
    		}

/* 	Transfer the strings from keyinc-style strings */

  	for (iSid = 0; iSid < SysConfig.InputBeam.NumSid; iSid++){
    		strncpy(
	    		SysConfig.InputBeam.StationID[iSid],
	    		TransferString.StationID[iSid],
	    		MAX_IDSTR);
    		SysConfig.InputBeam.StationID[iSid][MAX_IDSTR-1] = '\0';
    		}

/* Shift array contents of SysConfig due to mismatch with keyin declarations */

  	for (iOut = SysConfig.OutputBeam.NumOutBeam-1; iOut > 0; iOut--) {
    		memmove(SysConfig.OutputBeam.Wavelength[0]+VAR_NUM_CHAN*iOut,
            		SysConfig.OutputBeam.Wavelength[0]+MaxWav*iOut,MaxWav*sizeof(double));
    		memmove(SysConfig.OutputBeam.WavelengthErr[0]+VAR_NUM_CHAN*iOut,
            		SysConfig.OutputBeam.WavelengthErr[0]+MaxWav*iOut,MaxWav*sizeof(double));
    		memmove(SysConfig.OutputBeam.ChanWidth[0]+VAR_NUM_CHAN*iOut,
            		SysConfig.OutputBeam.ChanWidth[0]+MaxWav*iOut,MaxWav*sizeof(double));
    		memmove(SysConfig.OutputBeam.ChanWidthErr[0]+VAR_NUM_CHAN*iOut,
            		SysConfig.OutputBeam.ChanWidthErr[0]+MaxWav*iOut,MaxWav*sizeof(double));
    		memmove(SysConfig.OutputBeam.FringeMod[0]+VAR_NUM_BAS*iOut,
            		SysConfig.OutputBeam.FringeMod[0]+MaxFrFreq*iOut,MaxFrFreq*sizeof(int));
    		memmove(SysConfig.OutputBeam.BiasMod[0]+VAR_NUM_BAS*iOut,
            		SysConfig.OutputBeam.BiasMod[0]+MaxFrFreq*iOut,MaxFrFreq*sizeof(int));
    		memmove(SysConfig.OutputBeam.InputPair[0][0]+VAR_NUM_BAS*2*iOut,
            		SysConfig.OutputBeam.InputPair[0][0]+MaxFrFreq*2*iOut,MaxFrFreq*2*sizeof(int));
		if (MaxSid != 0)
		memmove(SysConfig.OutputBeam.StationID[0]+VAR_NUM_SID*iOut,
			SysConfig.OutputBeam.StationID[0]+MaxSid*iOut,MaxSid*sizeof(int));
    		}
	piNumSid[0]=SysConfig.InputBeam.NumSid;
	piNumOut[0]=SysConfig.OutputBeam.NumOutBeam;
	piMaxBas[0]=MaxFrFreq;
	piMaxWav[0]=MaxWav;
	return(0);
	}

/* If stations are defined for the ouput beams, re-calculate input pairs */

        for (iOut = 0; iOut < piNumOut[0]; iOut++) {
          iBas = 0;
          for (iSid1 = 0; iSid1 < (SysConfig.OutputBeam.NumSid[iOut]-1); iSid1++) {
          for (iSid2 = iSid1+1; iSid2 < SysConfig.OutputBeam.NumSid[iOut]; iSid2++) {
            fFringeMod = SysConfig.InputBeam.Stroke[SysConfig.OutputBeam.StationID[iOut][iSid2]-1]
	               - SysConfig.InputBeam.Stroke[SysConfig.OutputBeam.StationID[iOut][iSid1]-1];
            i = iSid1;
            j = iSid2;
            if (fFringeMod < 0) {
	        i = iSid2;
	        j = iSid1;
	        fFringeMod = -fFringeMod;
      	        }
            SysConfig.OutputBeam.InputPair[iOut][iBas][0] 
	        = SysConfig.OutputBeam.StationID[iOut][i]-1;
            SysConfig.OutputBeam.InputPair[iOut][iBas][1] 
	        = SysConfig.OutputBeam.StationID[iOut][j]-1;
            SysConfig.OutputBeam.FringeMod[iOut][iBas] = (fFringeMod/1e-6)+0.1;
            iBas++;
            }
            }
          SysConfig.OutputBeam.NumFringeFreq[iOut] = iBas;	 
          }

/*	Now, on the second call, we store the data */

	for (iOut=0;iOut<piNumOut[0];iOut++){
		piNumWav[iOut]=SysConfig.OutputBeam.NumSpecChan[iOut];
		piNumBas[iOut]=SysConfig.OutputBeam.NumFringeFreq[iOut];
		piNumBias[iOut]=SysConfig.OutputBeam.NumBiasFreq[iOut];
		}

	pdfLatitude[0]=SysConfig.Latitude;
	pdfLongitude[0]=SysConfig.Longitude;
	pdfAltitude[0]=SysConfig.Altitude;
	pdfEarthRadius[0]=SysConfig.EarthRadius;
	pdfJ2[0]=SysConfig.J2;
	pdfTDTUTC[0]=SysConfig.TDTminusUTC;
	pdfUT1UTC[0]=SysConfig.UT1minusUTC;

	pdfInstrCohInt[0]=SysConfig.InstrCohInt;
	piNumBin[0]=SysConfig.NumBin;
	piBeamCID[0]=SysConfig.BeamCombinerID;

	for (iSid=0;iSid<piNumSid[0];iSid++){
		IDL_StringStore(azStation[iSid],
			        SysConfig.InputBeam.StationID[iSid],
			        strlen(SysConfig.InputBeam.StationID[iSid]));
		for (i=0;i<4;i++){
			pdfStationCoord[i+4*iSid]=SysConfig.InputBeam.StationCoord[iSid][i];
			}
		pfStroke[iSid]=SysConfig.InputBeam.Stroke[iSid];
		piFDLTankID[iSid]=SysConfig.InputBeam.FDLTankID[iSid];
		}
	for (iOut=0;iOut<piNumOut[0];iOut++){
		for (iBas=0;iBas<piNumBas[iOut];iBas++){
			piFringeMod[iBas+MaxFrFreq*iOut]=SysConfig.OutputBeam.FringeMod[iOut][iBas];
			piBiasMod[iBas+MaxFrFreq*iOut]=SysConfig.OutputBeam.BiasMod[iOut][iBas];
			piInputPair[0+2*(iBas+MaxFrFreq*iOut)]=SysConfig.OutputBeam.InputPair[iOut][iBas][0];
			piInputPair[1+2*(iBas+MaxFrFreq*iOut)]=SysConfig.OutputBeam.InputPair[iOut][iBas][1];
			}
		for (iWav=0;iWav<piNumWav[iOut];iWav++){
			pdfWavelength[iWav+MaxWav*iOut]=SysConfig.OutputBeam.Wavelength[iOut][iWav];
			pdfWavelengthErr[iWav+MaxWav*iOut]=SysConfig.OutputBeam.WavelengthErr[iOut][iWav];
			pdfChanWidth[iWav+MaxWav*iOut]=SysConfig.OutputBeam.ChanWidth[iOut][iWav];
			pdfChanWidthErr[iWav+MaxWav*iOut]=SysConfig.OutputBeam.ChanWidthErr[iOut][iWav];
			}
		}

	}
/*----------------------------------------------------------------------------*/
int readscanstartversion1(int argc, char *argp[]){

	long *piScanId;
	IDL_STRING *azStarId;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	azRawFile=((IDL_STRING**)argp)[0];
	piOffset=	((long**)argp)[1];
	piLength=	((long**)argp)[2];

	piScanId=      ((long**)argp)[3];
	azStarId=((IDL_STRING**)argp)[4];

/*	Locate and read packet */

	if ((fp=fopen(azRawFile[0].s,"rb")) == NULL) return(-1);
	if (fseek(fp,piOffset[0]+sizeof(DATAconHeader),SEEK_SET)) return(-1);
	pBody=malloc(piLength[0]);
	fread(pBody,piLength[0],1,fp);	
	fclose(fp);

/* 	Decode the packet */

	pRec=pBody;
	piScanId[0]=swaplong(*((int32*)pRec)); pRec+=sizeof(int32);
	IDL_StringStore(azStarId[0],pRec,strlen(pRec));

	free(pBody);
	}
/*----------------------------------------------------------------------------*/
int readnatcounts(int argc, char *argp[]){

	short *piCounts;
	long *piStamps;
	long *piNIB,*piNRC;

        if (argc != 7) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	azRawFile=((IDL_STRING**)argp)[0];
	piOffset=	((long**)argp)[1];
	piLength=	((long**)argp)[2];

	piCounts=((short**)argp)[3];
	piStamps=((long**)argp)[4];
	piNIB=   ((long**)argp)[5];
	piNRC=   ((long**)argp)[6];

/*	Locate and read packet */

	if ((fp=fopen(azRawFile[0].s,"rb")) == NULL) return(-1);
	if (fseek(fp,piOffset[0]+sizeof(DATAconHeader),SEEK_SET)) return(-1);
	pBody=malloc(piLength[0]);
	fread(pBody,piLength[0],1,fp);
	fclose(fp);

/* 	Decode the packet */

	pRec=pBody;
	for (i=0;i<piNRC[0];i++){
	for (j=0;j<piNIB[0];j++){
		for (k=0;k<4;k++) {
			piCounts[j+piNIB[0]*(k+4*i)]=swapshort(*((short*)pRec));
			pRec+=2;
			}
		}
		piStamps[i]=swaplong(*((unsigned long*)pRec));
		pRec+=4;
		}

	free(pBody);
	}
/*----------------------------------------------------------------------------*/
int readfdlposition(int argc, char *argp[]){

	double *pdfLaser;
	long *piTime;
	long *piNIB,*piNRC;
	int iJitter;

	unsigned int uLaserLo,uLaserHi,uTime;

        if (argc != 7) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	azRawFile=((IDL_STRING**)argp)[0];
	piOffset=       ((long**)argp)[1];
	piLength=       ((long**)argp)[2];

	pdfLaser=((double**)argp)[3];
	piTime=    ((long**)argp)[4];
	piNIB=     ((long**)argp)[5];
	piNRC=     ((long**)argp)[6];

/*	Locate and read packet */

	if ((fp=fopen(azRawFile[0].s,"rb")) == NULL) return(-1);
	if (fseek(fp,piOffset[0]+sizeof(DATAconHeader),SEEK_SET)) return(-1);
	pBody=malloc(piLength[0]);
	fread(pBody,piLength[0],1,fp);	
	fclose(fp);

	pRec=pBody;
	for (i=0;i<piNRC[0];i++){
	for (j=0;j<piNIB[0];j++){
		uLaserLo= swaplong(*((unsigned int*)pRec)); 
		pRec+=sizeof(int);
		uLaserHi= swaplong(*((unsigned int*)pRec)) & 0x000000FF; 
		pRec+=sizeof(int);
		if (j==0)
		piTime[i]= (swaplong(*((unsigned int*)pRec))+10)/20; 
		pRec+=sizeof(int);
		iJitter= *((unsigned int*)pRec); pRec+=sizeof(int);
      		pdfLaser[j+piNIB[0]*i] = 
        		(0x000000FF - uLaserHi) * FDLHI_MET_UNITS +
        		(0xFFFFFFFF - uLaserLo) * FDLLO_MET_UNITS;
		}
		}

	free(pBody);
	}
/*----------------------------------------------------------------------------*/
int readfringedataversion1(int argc, char *argp[]){

	char *piCounts;
	long *piNOB,*piNCH,*piNBN,*piNRC;

	int eChannels[32]={3,2,1,0,7,6,5,4,11,10,9,8,15,14,13,12,19,18,17,16,
                    23,22,21,20,27,26,25,24,31,30,29,28};

        if (argc != 8) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	azRawFile=((IDL_STRING**)argp)[0];
	piOffset=	((long**)argp)[1];
	piLength=	((long**)argp)[2];

	piCounts=((char**)argp)[3];
	piNOB=   ((long**)argp)[4];
	piNBN=   ((long**)argp)[5];
	piNCH=   ((long**)argp)[6];
	piNRC=   ((long**)argp)[7];

/*	Locate and read packet */

	if ((fp=fopen(azRawFile[0].s,"rb")) == NULL) return(-1);
	if (fseek(fp,piOffset[0]+sizeof(DATAconHeader),SEEK_SET)) return(-1);
	pBody=malloc(piLength[0]);
	fread(pBody,piLength[0],1,fp);	
	fclose(fp);

	pRec=pBody;
	for (i=0;i<piNRC[0];i++){
	for (j=0;j<piNOB[0];j++){
	for (l=0;l<piNCH[0];l++){
		pRec=pBody + (256 * j) + (piNBN[0] * 4) - 1 +
	   	((eChannels[l] % piNBN[0]) << 5) - (eChannels[l] / piNBN[0]);
		pRec += (i*(piNOB[0]*piNCH[0]*piNBN[0]+4*piNOB[0]));
		for (k=0;k<piNBN[0];k++) {
			piCounts[j+piNOB[0]*(k+piNBN[0]*(l+piNCH[0]*i))]=*pRec;
			pRec-=4;
			}
		}
		}
		}

	free(pBody);
	}
/*----------------------------------------------------------------------------*/
int readfringebg(int argc, char *argp[]){

	char *piCounts;
	long *piStamps;
	long *piNOB,*piNCH,*piNBN,*piNRC;

        if (argc != 9) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	azRawFile=((IDL_STRING**)argp)[0];
	piOffset=	((long**)argp)[1];
	piLength=	((long**)argp)[2];

	piCounts=((char**)argp)[3];
	piStamps=((long**)argp)[4];
	piNOB=   ((long**)argp)[5];
	piNBN=   ((long**)argp)[6];
	piNCH=   ((long**)argp)[7];
	piNRC=   ((long**)argp)[8];

/*	Locate and read packet */

	if ((fp=fopen(azRawFile[0].s,"rb")) == NULL) return(-1);
	if (fseek(fp,piOffset[0]+sizeof(DATAconHeader),SEEK_SET)) return(-1);
	pBody=malloc(piLength[0]);
	fread(pBody,piLength[0],1,fp);	
	fclose(fp);

	pRec=pBody;
	for (i=0;i<piNRC[0];i++){
	for (j=0;j<piNOB[0];j++){
	for (l=0;l<piNCH[0];l++){
		for (k=0;k<piNBN[0];k++) {
			piCounts[j+piNOB[0]*(k+piNBN[0]*(l+piNCH[0]*i))]=*(pRec++);
			}
		}
		}
		piStamps[i]=swaplong(*((unsigned long*)pRec));
		pRec+=4;
		}

	free(pBody);
	}
/*----------------------------------------------------------------------------*/
int readfringedark(int argc, char *argp[]){

	float *pfCounts;
	long *piStamp;
	long *piNOB,*piNCH;
	int iNumInt;

        if (argc != 7) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	azRawFile=((IDL_STRING**)argp)[0];
	piOffset=	((long**)argp)[1];
	piLength=	((long**)argp)[2];

	pfCounts=((float**)argp)[3];
	piStamp=  ((long**)argp)[4];
	piNOB=    ((long**)argp)[5];
	piNCH=    ((long**)argp)[6];

/*	Locate and read packet */

	if ((fp=fopen(azRawFile[0].s,"rb")) == NULL) return(-1);
	if (fseek(fp,piOffset[0]+sizeof(DATAconHeader),SEEK_SET)) return(-1);
	pBody=malloc(piLength[0]);
	fread(pBody,piLength[0],1,fp);
	fclose(fp);

	pRec=pBody;
	piStamp[0]=swaplong(*((unsigned long*)pRec));
	pRec+=sizeof(long);
	iNumInt=swaplong(*((int*)pRec));
	pRec+=sizeof(int);
	for (j=0;j<piNOB[0];j++){
	for (l=0;l<piNCH[0];l++){
		pfCounts[j+piNOB[0]*l]=(float)swaplong(*((long*)pRec))/(2*iNumInt);
		pRec+=sizeof(long);
		}
		pRec+=((VAR_NUM_CHAN-piNCH[0])*sizeof(long));
		}

	free(pBody);
	}
/*----------------------------------------------------------------------------*/
int randpoisson(int argc, char *argp[]){

  unsigned char *pcCounts;
  float *pfCounts;
  long *piNum;

  float poidev(float xm, long *idum);
  float fResult,xm;
  static long idum=-1;
  int i;

  pcCounts=((unsigned char**)argp)[0];
  pfCounts=((float**)argp)[1];
  piNum=((long**)argp)[2];

  for(i=0;i<piNum[0];i++){
	xm=(float) pfCounts[i];
  	fResult=poidev(xm,&idum);
	pcCounts[i]=(unsigned char)fResult;
	}

  }

/*----------------------------------------------------------------------------*/
