/*******************************************************************************
hds_wrap.c

USNO/NRL Optical Interferometer
3450 Massachusetts Avenue NW
Washington DC 20392-5400

System: Data Reduction
Subsystem: CHAMELEON

Description:
Package of C-routines for HDS access callable from PV-WAVE via LINKNLOAD.

Modification history:
05-Feb-1994  C.A.Hummel, file created
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <sae_par.h>
#include <dat_par.h>

#include "export.h"

#define MAXDIM 7
#define MAXLEVELS 10

int status;
int len0,len1,len2,len3;

int LocatorID = -1;
char LocatorStack[MAXLEVELS][DAT__SZLOC];

/*----------------------------------------------------------------------------*/
int clearstatus(void){
	/*
	Clears status variable (= set to SAI__OK).
	*/

	status=SAI__OK;

	return(status);
	}
/*----------------------------------------------------------------------------*/
int toplevel(void){
	/* 
	Move to the top of the stack, releasing lower level locators.
	*/

	while(LocatorID > 0) {
		dat_annul_(LocatorStack[LocatorID],&status,len0);
		LocatorID -= 1;
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int hds_state(int argc, char *argp[]){
	/*
	Check HDS state.
	*/
	int *piState;

	if (argc !=1) {
                fprintf(stderr,"\nWrong # of parameters(%d)",argc);
                return(1);
                }

	piState=((int**)argp)[0];

	hds_state_(piState,&status);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int hds_new(int argc, char *argp[]){
	/*
	Create a new container file and return locator.
	*/
	IDL_STRING *file,*name,*type;
	int *piDims,*piNdim;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	file=((IDL_STRING**)argp)[0];len1=strlen(file[0].s);
	name=((IDL_STRING**)argp)[1];len2=strlen(name[0].s);
	type=((IDL_STRING**)argp)[2];len3=strlen(type[0].s);
	piNdim=((int**)argp)[3];
	piDims=((int**)argp)[4];

	len0=DAT__SZLOC;
	status=SAI__OK; 

	LocatorID += 1;
	hds_new_(file[0].s,name[0].s,type[0].s,piNdim,piDims,LocatorStack[LocatorID],
			&status,len1,len2,len3,len0);
	return(status);
	}
/*----------------------------------------------------------------------------*/
int hds_open(int argc, char *argp[]){
	/* 
	Open HDS file and return locator.
	*/
	IDL_STRING *file,*mode;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	file=((IDL_STRING**)argp)[0];
	mode=((IDL_STRING**)argp)[1];

	/* Start HDS */
	status=SAI__OK; 
	hds_start_(&status);

	/* Open existing container file */
	len0=DAT__SZLOC;
	len1=strlen(file[0].s);
	len2=strlen(mode[0].s);
	LocatorID += 1;
	hds_open_(file[0].s,mode[0].s,LocatorStack[LocatorID],&status,len1,len2,len0); 

	if (status != 0) {
		hds_stop_(&status);
		LocatorID -= 1;
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_shape(int argc, char *argp[]){
	/*
	Enquire the shape of an object.
	*/
	IDL_STRING *nam;
	int *piDims,*piNdim;
	int iMaxDim;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piDims=((int**)argp)[1];
	piNdim=((int**)argp)[2];

	iMaxDim=7;
	cmp_shape_(LocatorStack[LocatorID],nam[0].s,&iMaxDim,piDims,piNdim,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get0c(int argc, char *argp[]){
	/* 
	Read character scalar component.
	*/
	IDL_STRING *nam,*data;
	int *piLen;
	char *pos;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	data=((IDL_STRING**)argp)[1];
	piLen= ((int**)argp)[2];

	cmp_get0c_(LocatorStack[LocatorID],nam[0].s,data[0].s,&status,len0,len1,*piLen);
	if((pos=(char*)memchr(data[0].s,'\0',*piLen))!=NULL){
		memset(pos,' ',data[0].s+(*piLen)-pos);
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get0i(int argc, char *argp[]){
	/*
	Read integer scalar component.
	*/
	IDL_STRING *nam;
	int *value;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	value=((int**)argp)[1];

	cmp_get0i_(LocatorStack[LocatorID],nam[0].s,value,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get0r(int argc, char *argp[]){
	/*
	Read real scalar component.
	*/
	IDL_STRING *nam;
	float *value;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=  ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	value=((float**)argp)[1];

	cmp_get0r_(LocatorStack[LocatorID],nam[0].s,value,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get0d(int argc, char *argp[]){
	/*
	Read double scalar component.
	*/
	IDL_STRING *nam;
	double *value;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	value=((double**)argp)[1];

	cmp_get0d_(LocatorStack[LocatorID],nam[0].s,value,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get1c(int argc, char *argp[]){
	/* 
	Read character vector component.
	*/
	IDL_STRING *nam,*data;
	int *piNum,*piLen;
	char *Data,*pos;
	int i;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum= ((int**)argp)[1];
	data=((IDL_STRING**)argp)[2];
	piLen= ((int**)argp)[3];

	if((Data=(char*)malloc(*piLen*(*piNum)))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_get1c!");
		return(1);
		}

	cmp_get1c_(LocatorStack[LocatorID],nam[0].s,piNum,Data,piNum,&status,len0,len1,*piLen);
	for(i=0;i<*piNum;i++){
		memcpy(data[i].s,Data+i*(*piLen),*piLen);
		while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
			memset(pos,' ',*piLen-(pos-data[i].s));
			}
		}
	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get1i(int argc, char *argp[]){
	/*
	Read integer vector component.
	*/
	IDL_STRING *nam;
	int *piNum,*piData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum= ((int**)argp)[1];
	piData=((int**)argp)[2];

	cmp_get1i_(LocatorStack[LocatorID],nam[0].s,piNum,piData,piNum,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get1r(int argc, char *argp[]){
	/*
	Read real vector component.
	*/
	IDL_STRING *nam;
	int *piNum;
	float *pfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=   ((int**)argp)[1];
	pfData=((float**)argp)[2];

	cmp_get1r_(LocatorStack[LocatorID],nam[0].s,piNum,pfData,piNum,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get1d(int argc, char *argp[]){
	/*
	Read double vector component.
	*/
	IDL_STRING *nam;
	int *piNum;
	double *pdfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=     ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=     ((int**)argp)[1];
	pdfData=((double**)argp)[2];

	cmp_get1d_(LocatorStack[LocatorID],nam[0].s,piNum,pdfData,piNum,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_getnc(int argc, char *argp[]){
	/* 
	Read character array component.
	*/
	IDL_STRING *nam,*data;
	int *piNdim,*piDims,*piLen;
	char *Data,*pos;
	int i,n;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=((int**)argp)[1];
	piDims=((int**)argp)[2];
	data=((IDL_STRING**)argp)[3];
	piLen= ((int**)argp)[4];

	n=1;
	for(i=0;i<*piNdim;i++)n*=piDims[i];

	if((Data=(char*)malloc(*piLen*n))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_getnc!");
		return(1);
		}

	cmp_getnc_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,Data,piDims,
			&status,len0,len1,*piLen);
	for(i=0;i<n;i++){
		memcpy(data[i].s,Data+i*(*piLen),*piLen);
		while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
			memset(pos,' ',*piLen-(pos-data[i].s));
			}
		}
	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_getni(int argc, char *argp[]){
	/*
	Read a named integer array primitive.
	*/
	IDL_STRING *nam;
	int *piNdim,*piDims;
	int *piData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=((int**)argp)[1];
	piDims=((int**)argp)[2];
	piData=((int**)argp)[3];

	cmp_getni_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,(int*)piData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_getnr(int argc, char *argp[]){
	/*
	Read a named real array primitive.
	*/
	IDL_STRING *nam;
	int *piNdim,*piDims;
	float *pfData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=  ((int**)argp)[1];
	piDims=  ((int**)argp)[2];
	pfData=((float**)argp)[3];

	cmp_getnr_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,(float*)pfData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_getnd(int argc, char *argp[]){
	/*
	Read a named double array primitive.
	*/
	IDL_STRING *nam;
	int *piNdim,*piDims;
	double *pdfData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=     ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=    ((int**)argp)[1];
	piDims=    ((int**)argp)[2];
	pdfData=((double**)argp)[3];

	cmp_getnd_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,(double*)pdfData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put0c(int argc, char *argp[]){
	/*
	Write a scalar string component.
	*/
	IDL_STRING *name,*data;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	name=((IDL_STRING**)argp)[0];len1=strlen(name[0].s);
	data=((IDL_STRING**)argp)[1];len2=strlen(data[0].s);

	cmp_put0c_(LocatorStack[LocatorID],name[0].s,data[0].s,&status,len0,len1,len2);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put0i(int argc, char *argp[]){
	/*
	Write a scalar integer component.
	*/
	IDL_STRING *nam;
	int *piValue;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=    ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piValue=  ((int**)argp)[1];

	cmp_put0i_(LocatorStack[LocatorID],nam[0].s,piValue,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put0r(int argc, char *argp[]){
	/*
	Write a scalar float component.
	*/
	IDL_STRING *nam;
	float *pfValue;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=    ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	pfValue=((float**)argp)[1];

	cmp_put0r_(LocatorStack[LocatorID],nam[0].s,pfValue,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put0d(int argc, char *argp[]){
	/*
	Write a scalar double component.
	*/
	IDL_STRING *nam;
	double *pdfValue;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=      ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	pdfValue=((double**)argp)[1];

	cmp_put0d_(LocatorStack[LocatorID],nam[0].s,pdfValue,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put1c(int argc, char *argp[]){
	/*
	Write a vector string component.
	*/
	IDL_STRING *nam,*data;
	int *piNum,*piLen;
	char *Data;
	int i;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=  ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=  ((int**)argp)[1];
	data= ((IDL_STRING**)argp)[2];
	piLen=  ((int**)argp)[3];

	if((Data=(char*)malloc(*piLen*(*piNum)))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_put1c!");
		return(1);
		}

	for(i=0;i<*piNum;i++){
		memcpy(Data+i*(*piLen),data[i].s,*piLen);
		}

	cmp_put1c_(LocatorStack[LocatorID],nam[0].s,piNum,Data,&status,len0,len1,*piLen);

	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put1i(int argc, char *argp[]){
	/*
	Write integer vector component.
	*/
	IDL_STRING *nam;
	int *piNum,*piData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum= ((int**)argp)[1];
	piData=((int**)argp)[2];

	cmp_put1i_(LocatorStack[LocatorID],nam[0].s,piNum,piData,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put1r(int argc, char *argp[]){
	/*
	Write float vector component.
	*/
	IDL_STRING *nam;
	float *pfData;
	int *piNum;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=   ((int**)argp)[1];
	pfData=((float**)argp)[2];

	cmp_put1r_(LocatorStack[LocatorID],nam[0].s,piNum,pfData,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put1d(int argc, char *argp[]){
	/*
	Write double vector component.
	*/
	IDL_STRING *nam;
	double *pdfData;
	int *piNum;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=     ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=     ((int**)argp)[1];
	pdfData=((double**)argp)[2];

	cmp_put1d_(LocatorStack[LocatorID],nam[0].s,piNum,pdfData,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_putnc(int argc, char *argp[]){
	/* 
	Write character array component.
	*/
	IDL_STRING *nam,*data;
	int *piNdim,*piDims,*piLen;
	char *Data;
	int i,j,n;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=((int**)argp)[1];
	piDims=((int**)argp)[2];
	data=((IDL_STRING**)argp)[3];
	piLen= ((int**)argp)[4];

	n=1;
	for(i=0;i<*piNdim;i++)n*=piDims[i];

	if((Data=(char*)malloc(*piLen*n))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_putnc!");
		return(1);
		}

	for(i=0;i<n;i++){
		memcpy(Data+i*(*piLen),data[i].s,*piLen);
		}

	cmp_putnc_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,Data,piDims,
			&status,len0,len1,*piLen);

	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_putni(int argc, char *argp[]){
	/*
	Write an integer array primitive.
	*/
	IDL_STRING *nam;
	int *piNdim,*piDims;
	int *piData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=((int**)argp)[1];
	piDims=((int**)argp)[2];
	piData=((int**)argp)[3];

	cmp_putni_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,piData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_putnr(int argc, char *argp[]){
	/*
	Write a float array primitive.
	*/
	IDL_STRING *nam;
	int *piNdim,*piDims;
	float *pfData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=  ((int**)argp)[1];
	piDims=  ((int**)argp)[2];
	pfData=((float**)argp)[3];

	cmp_putnr_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,pfData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_putnd(int argc, char *argp[]){
	/*
	Write a double array primitive.
	*/
	IDL_STRING *nam;
	int *piNdim,*piDims;
	double *pdfData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=     ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=    ((int**)argp)[1];
	piDims=    ((int**)argp)[2];
	pdfData=((double**)argp)[3];

	cmp_putnd_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,pdfData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_there(int argc, char *argp[]){
	/*
	Enquire component existence.
	*/
	IDL_STRING *nam;
	int *piReply;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=  ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piReply=((int**)argp)[1];

	dat_there_(LocatorStack[LocatorID],nam[0].s,piReply,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_find(int argc, char *argp[]){
	/*
	Return locator to named component.
	*/
	IDL_STRING *nam;
	char loc[DAT__SZLOC];

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);

	dat_find_(LocatorStack[LocatorID],nam[0].s,loc,&status,len0,len1,len0);
	LocatorID += 1;
	memcpy(LocatorStack[LocatorID],loc,DAT__SZLOC);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_name(int argc, char *argp[]){
	/*
	Get name of object.
	*/
	IDL_STRING *nam;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=((IDL_STRING**)argp)[0];

	dat_name_(LocatorStack[LocatorID],nam[0].s,&status,len0,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_type(int argc, char *argp[]){
	/*
	Enquire object type.
	*/
	IDL_STRING *typ;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	typ=((IDL_STRING**)argp)[0];

	dat_type_(LocatorStack[LocatorID],typ[0].s,&status,len0,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_ncomp(int argc, char *argp[]){
	/*
	Get number of components.
	*/
	int *ncomp;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	ncomp=((int**)argp)[0];

	dat_ncomp_(LocatorStack[LocatorID],ncomp,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_prim(int argc, char *argp[]){
	/*
	Enquire object primitive.
	*/
	int *piReply;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piReply=((int**)argp)[0];

	dat_prim_(LocatorStack[LocatorID],piReply,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_size(int argc, char *argp[]){
	/*
	Enquire object size.
	*/
	int *piSize;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piSize=((int**)argp)[0];

	dat_size_(LocatorStack[LocatorID],piSize,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_shape(int argc, char *argp[]){
	/*
	Enquire the shape of an object.
	*/
	int *piDims,*piNdim;
	int iMaxDim;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piDims=((int**)argp)[0];
	piNdim=((int**)argp)[1];

	iMaxDim=MAXDIM;
	dat_shape_(LocatorStack[LocatorID],&iMaxDim,piDims,piNdim,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_clen(int argc, char *argp[]){
	/*
	Obtain character string length.
	*/
	int *piLen;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piLen=((int**)argp)[0];

	dat_clen_(LocatorStack[LocatorID],piLen,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_cell(int argc, char *argp[]){
	/*
	Return locator to a cell of an array object.
	*/
	char loc[DAT__SZLOC];
	int *piNdim,*piCell;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((int**)argp)[0];
	piCell=((int**)argp)[1];

	dat_cell_(LocatorStack[LocatorID],piNdim,piCell,loc,&status,len0,len0);
	LocatorID += 1;
	memcpy(LocatorStack[LocatorID],loc,DAT__SZLOC);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_index(int argc, char *argp[]){
	/*
	Return locator to item in component list.
	*/
	char loc[DAT__SZLOC];
	int *index;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	index= ((int**)argp)[0];

	dat_index_(LocatorStack[LocatorID],index,loc,&status,len0,len0);
	LocatorID += 1;
	memcpy(LocatorStack[LocatorID],loc,DAT__SZLOC);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_new(int argc, char *argp[]){
	/*
	Create a new component in a structure.
	*/
	IDL_STRING *nam,*typ;
	int *piNdim,*piDims;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	typ= ((IDL_STRING**)argp)[1];len2=strlen(typ[0].s);
	piNdim=((int**)argp)[2];
	piDims=((int**)argp)[3];

	dat_new_(LocatorStack[LocatorID],nam[0].s,typ[0].s,piNdim,piDims,&status,len0,len1,len2);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_new0c(int argc, char *argp[]){
	/*
	Create scalar string component.
	*/
	IDL_STRING *nam;
	int *piLen;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piLen=((int**)argp)[1];

	dat_new0c_(LocatorStack[LocatorID],nam[0].s,piLen,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_slice(int argc, char *argp[]){
	/*
	Return locator to a slice of an array object.
	*/
	char loc[DAT__SZLOC];
	int *piDiml,*piDimu,*piNdim;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((int**)argp)[0];
	piDiml=((int**)argp)[1];
	piDimu=((int**)argp)[2];

	dat_slice_(LocatorStack[LocatorID],piNdim,piDiml,piDimu,loc,&status,len0,len0);
	LocatorID += 1;
	memcpy(LocatorStack[LocatorID],loc,DAT__SZLOC);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_alter(int argc, char *argp[]){
	/*
	Alter object size.
	*/
	int *piDims,*piNdim;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((int**)argp)[0];
	piDims=((int**)argp)[1];

	dat_alter_(LocatorStack[LocatorID],piNdim,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_erase(int argc, char *argp[]){
	/* 
	Erase component.
	*/
	IDL_STRING *nam;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);

	dat_erase_(LocatorStack[LocatorID],nam[0].s,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_annul(int argc, char *argp[]){
	/*
	Annuls HDS locator.
	*/

        if (argc != 0) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	dat_annul_(LocatorStack[LocatorID],&status,len0);
	LocatorID -= 1;

	if (LocatorID == -1){
/*		printf("HDS file closed.\n"); */
		hds_stop_(&status);
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get(int argc, char *argp[]){
	/*
	Read primitive.
	*/
	IDL_STRING *typ;
	int *piNdim,*piDims,*piLen;
	unsigned char *pData;

	/* These are for character data */
	IDL_STRING *data;
	char *Data,*pos;
	int i,n;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	typ=          ((IDL_STRING**)argp)[0];len1=strlen(typ[0].s);
	piNdim=         ((int**)argp)[1];
	piDims=         ((int**)argp)[2];
	pData=((unsigned char**)argp)[3];
	data=         ((IDL_STRING**)argp)[3];
	piLen=          ((int**)argp)[4];

	if(strstr(typ[0].s,"_CHAR")!=NULL){
		n=1;
		for(i=0;i<*piNdim;i++)n*=piDims[i];
		if((Data=(char*)malloc(*piLen*n))==NULL){
			fprintf(stderr,"Error allocating memory in dat_get!");
			return(1);
			}
		dat_get_(LocatorStack[LocatorID],typ[0].s,piNdim,piDims,(unsigned char*)Data,
			&status,len0,len1,piLen[0]);
		for(i=0;i<n;i++){
			memcpy(data[i].s,Data+i*(*piLen),*piLen);
			while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
				memset(pos,' ',*piLen-(pos-data[i].s));
				}
			}
		free(Data);
		}
	else{
		dat_get_(LocatorStack[LocatorID],typ[0].s,piNdim,piDims,(unsigned char*)pData,
			&status,len0,len1,piLen[0]);
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get0c(int argc, char *argp[]){
	/*
	Read character scalar primitive.
	*/
	IDL_STRING *string;
	int *piLen;
	char *pos;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	string=((IDL_STRING**)argp)[0];
	piLen=   ((int**)argp)[1];

	dat_get0c_(LocatorStack[LocatorID],string[0].s,&status,len0,*piLen);
	if((pos=(char*)memchr(string[0].s,'\0',*piLen))!=NULL){
		memset(pos,' ',string[0].s+(*piLen)-pos);
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get0i(int argc, char *argp[]){
	/*
	Read integer scalar primitive.
	*/
	int *value;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	value=((int**)argp)[0];

	dat_get0i_(LocatorStack[LocatorID],value,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get0r(int argc, char *argp[]){
	/*
	Read integer scalar primitive.
	*/
	float *value;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	value=((float**)argp)[0];

	dat_get0r_(LocatorStack[LocatorID],value,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get0d(int argc, char *argp[]){
	/*
	Read integer scalar primitive.
	*/
	double *value;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	value=((double**)argp)[0];

	dat_get0d_(LocatorStack[LocatorID],value,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get1c(int argc, char *argp[]){
	/* 
	Read character vector component.
	*/
	IDL_STRING *data;
	int *piNum,*piLen;
	char *Data,*pos;
	int i;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum= ((int**)argp)[0];
	data=((IDL_STRING**)argp)[1];
	piLen= ((int**)argp)[2];

	if((Data=(char*)malloc(*piLen*(*piNum)))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_get1c!");
		return(1);
		}

	dat_get1c_(LocatorStack[LocatorID],piNum,Data,piNum,&status,len0,*piLen);
	for(i=0;i<*piNum;i++){
		memcpy(data[i].s,Data+i*(*piLen),*piLen);
		while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
			memset(pos,' ',*piLen-(pos-data[i].s));
			}
		}
	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get1i(int argc, char *argp[]){
	/*
	Read integer vector component.
	*/
	int *piNum,*piData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum= ((int**)argp)[0];
	piData=((int**)argp)[1];

	dat_get1i_(LocatorStack[LocatorID],piNum,piData,piNum,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get1r(int argc, char *argp[]){
	/*
	Read real vector component.
	*/
	int *piNum;
	float *pfData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=   ((int**)argp)[0];
	pfData=((float**)argp)[1];

	dat_get1r_(LocatorStack[LocatorID],piNum,pfData,piNum,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get1d(int argc, char *argp[]){
	/*
	Read double vector component.
	*/
	int *piNum;
	double *pdfData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=     ((int**)argp)[0];
	pdfData=((double**)argp)[1];

	dat_get1d_(LocatorStack[LocatorID],piNum,pdfData,piNum,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_getnc(int argc, char *argp[]){
	/* 
	Read character array component.
	*/
	IDL_STRING *data;
	int *piNdim,*piDims,*piLen;
	char *Data,*pos;
	int i,n;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((int**)argp)[0];
	piDims=((int**)argp)[1];
	data=((IDL_STRING**)argp)[2];
	piLen= ((int**)argp)[3];

	n=1;
	for(i=0;i<*piNdim;i++)n*=piDims[i];

	if((Data=(char*)malloc(*piLen*n))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_getnc!");
		return(1);
		}

	dat_getnc_(LocatorStack[LocatorID],piNdim,piDims,Data,piDims,
			&status,len0,*piLen);
	for(i=0;i<n;i++){
		memcpy(data[i].s,Data+i*(*piLen),*piLen);
		while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
			memset(pos,' ',*piLen-(pos-data[i].s));
			}
		}
	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_getni(int argc, char *argp[]){
	/*
	Read an array primitive.
	*/
	int *piNdim,*piDims;
	int *piData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=  ((int**)argp)[0];
	piDims=  ((int**)argp)[1];
	piData=  ((int**)argp)[2];

	dat_getni_(LocatorStack[LocatorID],piNdim,piDims,(int*)piData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_getnr(int argc, char *argp[]){
	/*
	Read an array primitive.
	*/
	int *piNdim,*piDims;
	float *pfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=  ((int**)argp)[0];
	piDims=  ((int**)argp)[1];
	pfData=((float**)argp)[2];

	dat_getnr_(LocatorStack[LocatorID],piNdim,piDims,(float*)pfData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_getnd(int argc, char *argp[]){
	/*
	Read a double array primitive.
	*/
	int *piNdim,*piDims;
	double *pdfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=    ((int**)argp)[0];
	piDims=    ((int**)argp)[1];
	pdfData=((double**)argp)[2];

	dat_getnd_(LocatorStack[LocatorID],piNdim,piDims,(double*)pdfData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put0c(int argc, char *argp[]){
	/*
	Write a scalar string component.
	*/
	IDL_STRING *data;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	data=((IDL_STRING**)argp)[0];len1=strlen(data[0].s);

	dat_put0c_(LocatorStack[LocatorID],data[0].s,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put1c(int argc, char *argp[]){
	/*
	Write a vector string component.
	*/
	IDL_STRING *data;
	int *piNum,*piLen;
	char *Data;
	int i;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=  ((int**)argp)[0];
	data= ((IDL_STRING**)argp)[1];
	piLen=  ((int**)argp)[2];

	if((Data=(char*)malloc(*piLen*(*piNum)))==NULL){
		fprintf(stderr,"Error allocating memory in dat_put1c!");
		return(1);
		}

	for(i=0;i<*piNum;i++){
		memcpy(Data+i*(*piLen),data[i].s,*piLen);
		}

	dat_put1c_(LocatorStack[LocatorID],piNum,Data,&status,len0,*piLen);

	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put1i(int argc, char *argp[]){
	/*
	Write integer vector component.
	*/
	int *piNum,*piData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum= ((int**)argp)[0];
	piData=((int**)argp)[1];

	dat_put1i_(LocatorStack[LocatorID],piNum,piData,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put1r(int argc, char *argp[]){
	/*
	Write real vector component.
	*/
	int *piNum;
	float *pfData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=   ((int**)argp)[0];
	pfData=((float**)argp)[1];

	dat_put1r_(LocatorStack[LocatorID],piNum,pfData,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put1d(int argc, char *argp[]){
	/*
	Write double vector component.
	*/
	int *piNum;
	double *pdfData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=     ((int**)argp)[0];
	pdfData=((double**)argp)[1];

	dat_put1d_(LocatorStack[LocatorID],piNum,pdfData,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_putnc(int argc, char *argp[]){
	/* 
	Write character array component.
	*/
	IDL_STRING *data;
	int *piNdim,*piDims,*piLen;
	char *Data;
	int i,j,n;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((int**)argp)[0];
	piDims=((int**)argp)[1];
	data=((IDL_STRING**)argp)[2];
	piLen= ((int**)argp)[3];

	n=1;
	for(i=0;i<*piNdim;i++)n*=piDims[i];

	if((Data=(char*)malloc(*piLen*n))==NULL){
		fprintf(stderr,"Error allocating memory in dat_putnc!");
		return(1);
		}

	for(i=0;i<n;i++){
		memcpy(Data+i*(*piLen),data[i].s,*piLen);
		}

	dat_putnc_(LocatorStack[LocatorID],piNdim,piDims,Data,piDims,&status,len0,*piLen);

	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_putni(int argc, char *argp[]){
	/*
	Write an array primitive.
	*/
	int *piNdim,*piDims;
	int *piData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=  ((int**)argp)[0];
	piDims=  ((int**)argp)[1];
	piData=  ((int**)argp)[2];

	dat_putni_(LocatorStack[LocatorID],piNdim,piDims,piData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_putnr(int argc, char *argp[]){
	/*
	Write an array primitive.
	*/
	int *piNdim,*piDims;
	float *pfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=  ((int**)argp)[0];
	piDims=  ((int**)argp)[1];
	pfData=((float**)argp)[2];

	dat_putnr_(LocatorStack[LocatorID],piNdim,piDims,pfData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_putnd(int argc, char *argp[]){
	/*
	Write a double array primitive.
	*/
	int *piNdim,*piDims;
	double *pdfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=    ((int**)argp)[0];
	piDims=    ((int**)argp)[1];
	pdfData=((double**)argp)[2];

	dat_putnd_(LocatorStack[LocatorID],piNdim,piDims,pdfData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
