/*******************************************************************************
hichol.c

USNO/NRL Optical Interferometer
3450 Massachusetts Avenue NW
Washington DC 20392-5420

Description:
Package for the Cholesky decomposition of sparse matrices in row-wise
format.

Modification history:
27-Oct-1993  C.A.Hummel, functions created
08-Dec-2000  C.A.Hummel, wrappers created
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* Function prototypes */

int hi_chol(double v[],unsigned long iv[],unsigned short jv[],
           unsigned short ni[],unsigned short ns,unsigned short nm, 
           double t[]);
int hi_prep(double vd[],unsigned long iv[],unsigned short jv[],double rd[],
            unsigned short ni[],unsigned short ns,unsigned short nm, 
            float v[],float t[],float r[],float s[]);
int hi_proc(double v[],unsigned long iv[],unsigned short jv[],
            unsigned short ir,unsigned short nr,unsigned short mb,
            double t[],unsigned short nm,unsigned short nb);
int cholesky(double t[],unsigned short n);
int hi_solve(double v[],unsigned long iv[],unsigned short jv[],
             unsigned short ni[],unsigned short ns,unsigned short nm,
             double t[],double r[],double s[]);
int hi_res(double vd[],unsigned long iv[],unsigned short jv[],double rd[],
	   double rn[],double s[],unsigned short nm);

/*----------------------------------------------------------------------------*/
int hi_chol_wrapper(int argc, char *argp[]){

	double *v,*t;
	unsigned short *jv,*ni,*ns,*nm;
	unsigned long *iv;

	unsigned short i,nb;

	v=((double**)argp)[0];
	iv=((unsigned long**)argp)[1];
	jv=((unsigned short**)argp)[2];
	ni=((unsigned short**)argp)[3];
	ns=((unsigned short**)argp)[4];
	nm=((unsigned short**)argp)[5];
/*	t=((double**)argp)[6];*/

        nb=0;
        for(i=0;i<ns[0];i++){nb+=ni[i];}

	hi_chol(v-1,iv-1,jv-1,ni-1,ns[0],nm[0],v-1+iv[nb]-1);

	}
/*----------------------------------------------------------------------------*/
int hi_solve_wrapper(int argc, char *argp[]){

	double *v,*t,*r,*s;
	unsigned short *jv,*ni,*ns,*nm;
	unsigned long *iv;

	unsigned short i,nb;

	v=((double**)argp)[0];
	iv=((unsigned long**)argp)[1];
	jv=((unsigned short**)argp)[2];
	ni=((unsigned short**)argp)[3];
	ns=((unsigned short**)argp)[4];
	nm=((unsigned short**)argp)[5];
/*	t=((double**)argp)[6];*/
	r=((double**)argp)[6];
	s=((double**)argp)[7];

        nb=0;
        for(i=0;i<ns[0];i++){nb+=ni[i];}

	hi_solve(v-1,iv-1,jv-1,ni-1,ns[0],nm[0],v-1+iv[nb]-1,r-1,s-1);

	}
/*----------------------------------------------------------------------------*/
int hi_res_wrapper(int argc, char *argp[]){

	double *vd,*rd,*rn;
	double *s;
	unsigned short *jv,*nm;
	unsigned long *iv;

	vd=((double**)argp)[0];
	iv=((unsigned long**)argp)[1];
	jv=((unsigned short**)argp)[2];
	rd=((double**)argp)[3];
	rn=((double**)argp)[4];
	s=((double**)argp)[5];
	nm=((unsigned short**)argp)[6];

	hi_res(vd-1,iv-1,jv-1,rd-1,rn-1,s-1,nm[0]);

	}
/*----------------------------------------------------------------------------*/

int hi_chol(double v[],unsigned long iv[],unsigned short jv[],
           unsigned short ni[],unsigned short ns,unsigned short nm, 
           double t[])
        {

        unsigned short i;
        unsigned short nb,ib,nr,mb,np;

        /*
        Performs Cholesky-decomposition of sparse matrices.
        Matrix must be positive definite, symmetric doubly-bordered
        block diagonal. Diagonal blocks can have different sizes.
        Border band may be sparse too, but must consist of blocks with
        vertical extent matching the corresponding diagonal block. 
        The upper triangle of the matrix is stored in row-wise format 
        in v[], except for the upper triangle of the lower right block
        formed by the overlap of both borders, which is stored row-wise
        in t[]. The storage of the former must account for zeros in 
        lower right part; the latter must be filled, i.e. non-sparse.
        The row-wise storage is as follows: iv[] indicates position in v[],
        where row i begins, jv[] gives column number of element v[i].
        Dimension of the matrix is nm. There are ns diagonal blocks,
        each with dimension ni[i]. (All arrays [0..N-1].) */


        /* Calculate number of columns in border band of matrix.
        This is the dimension of matrix t[] at the same time. */
        nb=0;
        for(i=1;i<=ns;i++){nb+=ni[i];}
        nb=nm-nb;

        /* Calculate decomposition down to lower right triangle.*/
        ib=1;              /* We start with first block */
        nr=ni[ib];         /* Number of rows in block to operate on */
        mb=iv[2]-iv[1]-nr; /* Number of non-zero columns in border block */
        np=nm-nb;
        for(i=1;i<=np;i++){
                hi_proc(v,iv,jv,i,nr,mb,t,nm,nb); 
                nr-=1;
                if(nr==0){
                        ib+=1;
                        nr=ni[ib];
                        mb=iv[i+2]-iv[i+1]-nr;
                        }
                }

        /* Calculate decomposition of lower right triangle */
        cholesky(t,nb);          

        return 0;
        }

int hi_proc(double v[],unsigned long iv[],unsigned short jv[],
            unsigned short ir,unsigned short nr,unsigned short mb,
            double t[],unsigned short nm,unsigned short nb)
        {

        double v1;
        unsigned long i,j,ii,jj,jjmax,i1,i2;

        /* Work on current row */
        v[iv[ir]]=sqrt(v[iv[ir]]);
	v1=1.0/v[iv[ir]];
        for(i=iv[ir]+1;i<=iv[ir+1]-1;i++){v[i]*=v1;}

        /* Work on remaining rows of current block */
        jj=ii=iv[ir]+1;
        jjmax=iv[ir+1];
        for(i=iv[ir+1];i<=iv[ir+nr]-1;i++){
                v[i]-=v[ii]*v[jj];
                if((jj+=1)==jjmax){jj=++ii;}
                }

        /* Work on rows in lower right triangle */
        i2=iv[ir+1]-1;
        i1=i2-mb+1;
        for(i=i1;i<=i2;i++){
                ii=jv[i]-(nm-nb);         
                for(j=i;j<=i2;j++){
                        jj=jv[j]-(nm-nb); 
                        t[(ii-1)*nb-(ii-1)*(ii-2)/2+1+jj-ii]-=v[i]*v[j];
                        }
                }   
        return 0;
        }

int cholesky(double t[],unsigned short n)
        {
        double t1;
        unsigned long ii,jj,i,j,k,l;

        ii=1;
        jj=(unsigned long)n*(n+1)/2;
        for(i=n;i>0;i--){
                t[ii]=sqrt(t[ii]);
		t1=1.0/t[ii];
                for(j=ii+1;j<ii+i;j++){t[j]*=t1;}
                k=l=ii+1;
                for(j=ii+i;j<=jj;j++){
                        t[j]-=t[k]*t[l];
                        if(++l==ii+i){l=++k;}
                        }
                ii+=i;
                }
        return 0;
        }

int hi_solve(double v[],unsigned long iv[],unsigned short jv[],
             unsigned short ni[],unsigned short ns,unsigned short nm,
             double t[],double r[],double s[])
        {

        /* Given the Cholesky decomposition of a matrix as returned
        by HICHOL, solve for right hand side r[] by forward substitution
        and backsubstitution. Right hand side is overwritten with
	solution, which is subtracted from s for improved solution.*/

        double sum;
	unsigned short nb,np;
        unsigned long ii,jj,i,j,k,l;

        nb=0;
        for(i=1;i<=ns;i++){nb+=ni[i];}
        nb=nm-nb;
	np=nm-nb;

        /* Skip leading zeros in right hand side */
        for(ii=1;r[ii]==0;ii++);

        /* Forward substitution in sparse part...*/
        if(ii<=np){
                for(i=ii;i<=np;i++){
                        r[i]/=v[iv[i]];
                        for(j=iv[i]+1;j<iv[i+1];j++){
                                r[jv[j]]-=v[j]*r[i];
                                }
                        }
                }
        /* ...and in triangle */
        if(ii<np+1){ii=np+1;}
        ii-=np;
        for(i=ii;i<=nb;i++){
                r[i+np]/=t[(i-1)*nb-(i-1)*(i-2)/2+1];
                for(j=i+1;j<=nb;j++){
                        r[j+np]-=t[(i-1)*nb-(i-1)*(i-2)/2+1+j-i]*r[i+np];
                        }
                }

        /* Back substitution in triangle */
        for(i=nm;i>np;i--){
                sum=r[i];
                k=i-np;
                for(j=i+1;j<=nm;j++){
                        l=j-np;
                        sum-=t[(k-1)*nb-(k-1)*(k-2)/2+1+l-k]*r[j];
                        }
                r[i]=sum/t[(k-1)*nb-(k-1)*(k-2)/2+1];
                }
        /* ...and in sparse part...*/
        for(i=np;i>0;i--){
                sum=r[i];
                for(j=iv[i]+1;j<=iv[i+1]-1;j++){
                        sum-=v[j]*r[jv[j]];
                        }
                r[i]=sum/v[iv[i]];
                }

        for(i=1;i<=nm;i++){s[i]-=r[i];} 

        return 0;
        }

int hi_res(double vd[],unsigned long iv[],unsigned short jv[],
           double rd[],double rn[],double s[],unsigned short nm)
	{

	double sum;
	unsigned short i;
	unsigned long j;

	/*
	Calculate residual of vd*s from right hand side rd and
	prepare new right hand side rn for calculation of improved
	solution with hi_solve. */ 

	/* upper triangle */
        for(i=1;i<=nm;i++){
                sum=-rd[i];
                for(j=iv[i];j<iv[i+1];j++){sum+=vd[j]*s[jv[j]];}
                rn[i]=sum;
                }
	/* lower triangle */
        for(i=1;i<=nm;i++){
                for(j=iv[i]+1;j<iv[i+1];j++){rn[jv[j]]+=vd[j]*s[i];}
                }
	return 0;
	}
