/**************************************************************************
keyinc.c


USNO/NRL Optical Interferometer
3450 Massachusetts Avenue, NW
Washintgon, DC 20392-5420

Subsystem:  Data Reduction


description:
------------

Implements keyinc, the c replacement for keyin


functions: 
----------
keyinc()


modification history:
--------------------
22-Oct-1993 DFB Created.
14-Nov-1993 DFB First usable version.
21-Apr-1994 DFB Added semicolon as a comment delimiter
22-Jul-1994 DFB Fixed the behaviour of "SHOW"
***********************************************************************/
/* includes */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "keyinc.h"

/* defines */

#define MAX_TOKEN_LEN KSTRING_LEN
#define INBUFFSIZE   1000
#define OUTBUFFSIZE 10000
#define MIN(a,b) ( a < b ? a : b )

/* typedefs, structures */

/* Enumerated type to define the possible key matches */

enum MinMatchType {MatchNoMatch, MatchAmbiguous, MatchExact, MatchMinimal};

/* Structure of character buffer used like an internal file */

typedef struct { char *pzString; int iLen; int iPos; } SBUFF;

/* external function declarations */


/* external variable declarations */


/* forward function declarations */

int keyinc(const keyrec KeyTable[], int iNpar, char *pzProgName,
	   kioStruc IOStream, char *pzReturnString);
int sputs(const char *string, SBUFF *buff);
static int CheckTable(const keyrec KeyTable[], int iNpar, char *pzReturnString);
static int GetLogicalLine(kioType StreamType, FILE *In, FILE *Out,
			  char *pzInBuff, int iBuffSize);
static int RequiresContinuation(char *pzLine);
static int keyinc1(const keyrec KeyTable[], int iNpar, char *pzProgName,
		   int iOutputLineLen, int iMaxOutputChars,
		   char *pzInBuff, char *pzOutBuff, char *pzReturnString);
static int ShowValue(const keyrec *KeyRec, SBUFF *OutBuff, int iOutputLineLen);
static const char *GenParse(const char *pzCurrentIn, char *pzToken,
			    int iMaxTokenLen);
static const char *MoveToNextLine(const char *psLine);
static int LookUp(const char *pzToken, const char **ppzTable, int iStride,
		  int iNpar, enum MinMatchType *piMatchType);
static int MatchEnum(const char *pzToken, const char *pzOpts, int *piElement);
static const char *DoBuiltIn(const char *pzCommand, const keyrec KeyTable[],
		       int iNpar, 
		       char *pzProgName, int iOutputLineLen,
		       const char *pzCurrentIn, SBUFF *OutBuff,
		       char *pzReturnString);
static int
MinMatch(const char *azString1, const char *azString2, int *piExact,
	 int *piStatus);

/* global variable declarions */




/**************************************************************************
keyinc()

description:
------------
 * Takes a specification of the input parameters and their
 * values in KeyTable, and modifies the parameters according to keyed input,
 * i.e. text input of the form "key=value".
 *
 * Input comes from either the user, a file, or a list of strings, as
 * specified in IOStream. The format of a list-of-strings is a sequence
 * of lines terminated by newline chars, the final newline being followed
 * by a '\0' char, signifying "end-of-file".
 *
 * Input is terminated with "END" "QUIT", an EOF condition, or a
 * string corresponding to a parameter of type "kaction". This
 * terminating string (or one generated by keyinc to fill its place,
 * e.g. "EOF" or "ERR") is returned in pzReturnString. The first
 * token is uppercased and has trailing space removed if there are
 * no further tokens, making simple cases easy to recognize.
 *
 * All strings are assumed to be dimensioned KSTRING_LEN, unless otherwise
 * specified, e.g. in the opts field.
 *
inputs:
-------



outputs:
--------
 * Return Value:
 * -------------
 * Set to the number of parameter alterations made (every successful
 * parameter-element changed counts as one, thus "Var=el1,el2"
 * counts as two). Negative if the table is not self-consistent
 * (duplicate keynames etc) - an error msg is returned in pzReturnString.
 * Incorrect keyed input generates error messages
 * on the output channel, but do not affect the return values, except in
 * the case of excessive errors.


modification history:
--------------------
22-Oct-1993 DFB function created


***********************************************************************/
int
keyinc(const keyrec KeyTable[], int iNpar, char * pzProgName,
		kioStruc IOStream,
		char * pzReturnString)
{
/*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   The way this master routine works is to convert all the possible types
   of IOStreams to a series of buffers and then call the main function
   to do the work on each resulting buffer. Any resulting output is
   printed to output streams where necessary.
  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */
    char pzInBuff[INBUFFSIZE], pzOutBuff[OUTBUFFSIZE];
    int iParsChanged=0, eof;

/* Default return */
    strcpy(pzReturnString,"EOF");
  
/* Do sanity checking on the table */
  if (! CheckTable(KeyTable, iNpar, pzReturnString) ) return (-1);

  switch ( IOStream.Type ) {

  case kioBuffer : /* Input and output to/from memory buffers */
    iParsChanged = keyinc1(KeyTable, iNpar, pzProgName,
			   IOStream.iOutputLineLen, IOStream.iMaxOutputChars,
			   IOStream.In.Buff, IOStream.Out.Buff,
			   pzReturnString);
    break;
    
  case kioTty : /* Interactive session - input and output from tty */
    do {
      eof=GetLogicalLine(IOStream.Type, IOStream.In.File,
			 IOStream.Out.File,
			 pzInBuff, INBUFFSIZE);
      if ( ! eof ){
	int rval=keyinc1(KeyTable, iNpar, pzProgName,
			 IOStream.iOutputLineLen, OUTBUFFSIZE,
			 pzInBuff, pzOutBuff,
			 pzReturnString);
	iParsChanged += rval;
	fputs(pzOutBuff, IOStream.Out.File);
	if(strcmp(pzReturnString,"EOF") != 0) eof=TRUE;
      }
    } while (eof == FALSE);
    break;

  case kioLoad : /* Load saved parameters from file: any output to buffer */
    do {
      eof=GetLogicalLine(IOStream.Type, IOStream.In.File,
			 NULL,
			 pzInBuff, INBUFFSIZE);
      if ( ! eof ){
	int rval=keyinc1(KeyTable, iNpar, pzProgName,
			 IOStream.iOutputLineLen, OUTBUFFSIZE,
			 pzInBuff, IOStream.Out.Buff,
			 pzReturnString);
	iParsChanged += rval;
	if(strcmp(pzReturnString,"EOF") != 0) eof=TRUE;
      }
    } while (eof == FALSE);
    break;

  default :
    printf("**Real dumb error in keyinc\n");
    iParsChanged = -1;
  }
    return (iParsChanged);
}

/*----------------------------------------------------------------------
  Check a KeyTable for consistency
 ----------------------------------------------------------------------*/
static int
CheckTable(const keyrec KeyTable[], int iNpar, char *pzReturnString)
{
  return(1);
}


/*-----------------------------------------------------------------------
  Get the next line of input, plus any continuation lines.
  Regularizes input so that last line ends with "\n\0".
  Return TRUE if EOF occurs.
 -----------------------------------------------------------------------*/
static int
GetLogicalLine(kioType StreamType, FILE *In, FILE *Out,
	       char *pzInBuff, int iBuffSize)
{
  char *pzIn=pzInBuff;
  int iInLeft=iBuffSize-1, iInLen=0;
  int cont,eof=FALSE;
  
  do {
    
/* Prompt user if interactive */
    
    if ( StreamType == kioTty ){
      if(iInLen == 0)fprintf(Out, "* ");
      else fprintf(Out, "> ");
    }
    
    /* Get line from user or file and exit at EOF */
    
    if (fgets(pzIn,iInLeft,In) == NULL){
      if (iInLen == 0){
	eof=TRUE;
	break;
      }
    }
    
/* Regularise lines with no newline char */
    
    iInLen=strlen(pzIn);
    if (pzIn[iInLen-1] != '\n' ){
      pzIn[iInLen]='\n'; pzIn[iInLen+1]='\0'; iInLen +=1;
    }
    
/* Check need for a continuation */
    
    cont = RequiresContinuation(pzIn);	
    iInLeft -= iInLen; pzIn += iInLen;
  } while ( cont && iInLeft > 1 );

  return(eof);
}


/*------------------------------------------------------------------------
   Subroutine to check if a line requires a continuation on the next line
   - at the moment, just checks for a trailing comma. Returns TRUE if
   a continuation is indicated. Assumes standard keyinc line format,
   i.e. a line terminated by "\n\0".
------------------------------------------------------------------------- */

static int
RequiresContinuation(char *pzLine)
{
  int i, cont=FALSE;
  for (i=strlen(pzLine)-2; i >= 0; i--){
    char ch=pzLine[i];
    if (ch == ','){ cont = TRUE; break; }
    if (ch != ' ' && ch != '\t') break;
  }
  return(cont);
}

/*-------------------------------------------------------------------------
  Main routine to parse a buffer and alter parameter values
  -------------------------------------------------------------------------*/
static int
keyinc1(const keyrec KeyTable[], int iNpar, char *pzProgName,
	int iOutputLineLen, int iMaxOutputChars,
	char *pzInBuff, char *pzOutBuff,
	char *pzReturnString)
{
  static const char *BuiltInTable[]={ "SHOW", "@", "SAVE", "HELP", 
					"QUIT", "/" };
  static const int iNBuiltIn=sizeof(BuiltInTable)/sizeof(BuiltInTable[0]);
  int iBuiltIn;
  enum MinMatchType iMatchTypeBuiltIn, iMatchTypeVar;
  int iVar, iEl, iElSize, iElsChanged, iChanged=0, success;

/* Different types of the possible parameter element, for alignment purposes */
  double pdfElement[MAX_TOKEN_LEN/sizeof(double)];
  int *piElement = (int *) pdfElement;
  char *pzElement = (char *) pdfElement;

  SBUFF OutBuff;
  const char *pzCurrentIn=pzInBuff;
  char pzToken[MAX_TOKEN_LEN];
  char *pzType;
  char *pzEnd;

/* Default return */
  strcpy(pzReturnString,"EOF");

/* setup output buffer */
  OutBuff.pzString=pzOutBuff;
  OutBuff.iLen=iMaxOutputChars;
  OutBuff.iPos=0;
  pzOutBuff[0]='\0';
    
/* Loop over input lines */
  while (*pzCurrentIn != '\0'){

/* Get first/next token */
    pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
    
/* Check for blank line */
    if (pzToken[0] == '\n')
       goto NextLine;

/* Check keyword against builtin commands and variable names */
    iBuiltIn=LookUp(pzToken, &(BuiltInTable[0]),
		    sizeof(BuiltInTable[0]), iNBuiltIn, &iMatchTypeBuiltIn);
    iVar=LookUp(pzToken, (const char **)&(KeyTable[0].key),
		    sizeof(KeyTable[0]), iNpar, &iMatchTypeVar);
    if (iMatchTypeBuiltIn == MatchExact 
	|| ( iMatchTypeBuiltIn == MatchMinimal && iMatchTypeVar != MatchExact)) {
	pzCurrentIn=DoBuiltIn(BuiltInTable[iBuiltIn], KeyTable, iNpar, 
			      pzProgName, iOutputLineLen,
			      pzCurrentIn, &OutBuff,
			      pzReturnString);
	if(pzCurrentIn == NULL)goto EndOfCommandProcessing;
	goto NextLine;
    }

/* Else */

/* Check we have a good variable name */
    if (iVar < 0 || iMatchTypeVar == MatchAmbiguous ){
      sputs("**Unknown or ambiguous variable : '", &OutBuff);
      sputs(pzToken, &OutBuff);
      sputs("'\n", &OutBuff);
      goto NextLine;
    }
    pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);

/* Handle user action */

/* REMOVE FOR NOW - DFB
    if ( KeyTable[iVar].type == kaction ){
      currpos=strncpy(pzReturnString,pzToken,KSTRING_LEN);
      to_upper(pzReturnString);
      pos =index(pzCurrentIn,'\n');
*/
/* Check up on this one!! */
/*
      strncat(pzReturnString[currpos],pzCurrentIn); 
      return(iChanged);
    }
*/  
/* Handle array subscript */
    iEl=1;
    if (pzToken[0] == '('){
      pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
      if ( sscanf(pzToken,"%d",&iEl) != 1
	  || iEl < 1
	  || iEl > KeyTable[iVar].nval ){
	sputs("**Invalid subscript : '",&OutBuff);
	sputs(pzToken, &OutBuff);
	sputs("'\n",&OutBuff);
	goto NextLine;
      }
      pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
      if (pzToken[0] != ')' ){
	sputs("**Syntax error: closing parenthesis expected\n",&OutBuff);
	goto NextLine;
      }
      pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
    }
    
/* Discard optional equals sign */
    if (pzToken[0] == '=')
      pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
    
/*************************************/
/* Loop to handle optional arguments */
/*************************************/
    iElsChanged = 0;
    while (pzToken[0] != '\0' && pzToken[0] != '\n') {

/* A comma with nothing preceding it indicates do nothing to this element */
      if (pzToken[0] == ',') { iEl++; goto NextToken; }
      
      switch(KeyTable[iVar].type){

      case kinteger :
	pzType = "Integer";
	*piElement = strtol(pzToken,&pzEnd,0);
	success = (*pzEnd == 0);
	iElSize = sizeof(int);
	break;

      case kdouble :
	pzType = "Double";
	*pdfElement = strtod(pzToken,&pzEnd);
	success = (*pzEnd == 0);
	iElSize = sizeof(double);
	break;

      case kstring :
	pzType = "String";
/* Strip quotes */
	if (pzToken[0] == '"' || pzToken[0] == '\'' ){
	  static char pzQuote[]=" \n";

	  pzQuote[0] = pzToken[0];
	  iElSize=strcspn(pzCurrentIn,pzQuote);
	  strncpy(pzElement,pzCurrentIn, MIN(iElSize,KSTRING_LEN-1));
	  pzElement[MIN(iElSize,KSTRING_LEN-1)]='\0';
	  pzCurrentIn += iElSize;
	  if (pzCurrentIn[0] == pzToken[0]) pzCurrentIn++;

/* Unquoted String */
	} else {
	  strncpy(pzElement,pzToken,KSTRING_LEN-1);
	  pzElement[KSTRING_LEN-1]='\0';
	}
	iElSize=KSTRING_LEN;
	success = 1;
	break;

      case kenum :
	pzType = "Enum";
	success = MatchEnum(pzToken, KeyTable[iVar].opts, piElement);
	iElSize = sizeof(int);
	break;

/* Treat booleans as special cases of enums */

      case kboolean :
	pzType = "Boolean";
	success = MatchEnum(pzToken, "enum=true:false:on:off:yes:no",
			    piElement);
	*piElement = (*piElement + 1) & 1;
	iElSize = sizeof(int);
	break;

      default :
	printf("**Not implemented!!\n");
	success = FALSE;
	iElSize = 0;
	pzType = "Unknown";
      }
      if (! success){
	sputs("**Unsuitable value for type ",&OutBuff);
	sputs(pzType,&OutBuff);
	sputs(": '", &OutBuff);
	sputs(pzToken, &OutBuff);
	sputs("'\n", &OutBuff);
	goto NextLine;
      }
      if (iEl > KeyTable[iVar].nval){
	sputs("**Too many values for variable ",&OutBuff);
	sputs(KeyTable[iVar].key, &OutBuff);
	sputs("'\n", &OutBuff);
	goto NextLine;
      }

/* If we get here we have a valid entry in Element: Copy it into storage
   and increment counters/pointers */
      memcpy((char *)KeyTable[iVar].val+(iEl - 1)*iElSize, pzElement, iElSize);
      iElsChanged++;
      iEl++;

/* A comma now indicates more elements to go */
      pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
      if (pzToken[0] != ',' ){
	if (pzToken[0] != '\n'){
	  sputs("**Extra tokens at end of line ignored\n",&OutBuff);
	}
	break;
      }
      NextToken :
	pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);

/* Check if we need to read a new line for the next element */
      while (pzToken[0] == '\n'){
	pzCurrentIn=MoveToNextLine(pzCurrentIn);
	pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
	if ( pzToken[0] == '\0' ){
	  sputs("**Continuation line expected\n",&OutBuff);
	  goto EndOfCommandProcessing;
	}
      }
    }
    
    
/* Trap no elements changed - make exception for booleans */
    if (iElsChanged < 1 ){
      if (KeyTable[iVar].type == kboolean){
	( (int *) KeyTable[iVar].val )[iEl-1] = TRUE;
	iElsChanged++;
      } else {
	sputs("**No values given for variable ",&OutBuff);
	sputs(KeyTable[iVar].key, &OutBuff);
	sputs("'\n", &OutBuff);
	goto NextLine;
      }
    }

/* Accumulate total number of elements altered */
    iChanged +=iElsChanged;
    
    NextLine :
      pzCurrentIn=MoveToNextLine(pzCurrentIn);
  }

  EndOfCommandProcessing :
  return(iChanged);
}


/*-------------------------------------------------------------------------
  Print key and value for a given parameter. Appends output to OutBuff.
  Returns number of chars printed.
  -------------------------------------------------------------------------*/
static int
ShowValue(const keyrec *KeyRec, SBUFF *OutBuff, int iOutputLineLen)
{
#define MAXLINESIZE 1000
  int iEl, iElStrSize, iOutChars, i;
  int *pInteger; double *pDouble;
  char *pFmt=NULL, pzLineBuff[MAXLINESIZE], *pzOpt;

/* Print out key part */
  sprintf(pzLineBuff,"%s = ",KeyRec->key);
  iOutChars=strlen(pzLineBuff);
  sputs(pzLineBuff,OutBuff);

/*--------------------------*/
/* Loop to print out values */
/*--------------------------*/
  for (iEl=0; iEl < KeyRec->nval; iEl++){
    switch (KeyRec->type) {

    case kinteger :
      pInteger=KeyRec->val;
      if(pFmt == NULL) pFmt="%d";
      sprintf(pzLineBuff,pFmt,pInteger[iEl]);
      break;

    case kdouble :
      pDouble=KeyRec->val;
      if(pFmt == NULL) pFmt="%.15g";
      sprintf(pzLineBuff,pFmt,pDouble[iEl]);
      break;

    case kstring :
      strcpy(pzLineBuff,"\"");
      strcat(pzLineBuff, (char *)KeyRec->val + iEl*KSTRING_LEN);
      strcat(pzLineBuff,"\"");
      break;

    case kenum :
/* Find relevant option */
      pzOpt=KeyRec->opts;
      while (pzOpt[0] != 0) {
	pzOpt += strspn(pzOpt,", \t");
	if(! strncmp(pzOpt,"enum=",5) ) break;
        pzOpt += strcspn(pzOpt,", \t") + 1;
      }

/* Skip to enum value within option */
      for(i=0, pzOpt += 5; i < ( (int *)KeyRec->val )[iEl] && pzOpt[0]; i++)
	pzOpt += strcspn(pzOpt,":") + 1;

/* Copy option */
      iElStrSize=strcspn(pzOpt,":");
      strncpy(pzLineBuff, pzOpt, iElStrSize);
      pzLineBuff[iElStrSize]='\0';
      break;

    case kboolean :
      if ( ((int *)KeyRec->val )[iEl] ){
	strcpy(pzLineBuff,"On");
      } else {
	strcpy(pzLineBuff,"Off");
      }
      break;

    default :
      printf("I can't do this type!");
      return(-1);
    }

/* Append a comma where necessary */
    if (iEl != KeyRec->nval - 1) strcat(pzLineBuff,", ");

/* Start a new line where neccessary */
    iElStrSize = strlen(pzLineBuff);
    if (iOutChars + iElStrSize > iOutputLineLen){
      sputs("\n    ",OutBuff);
      iOutChars = 4;
    }

/* Print this element */
    sputs(pzLineBuff,OutBuff);
    iOutChars += iElStrSize;
  }

/* Terminate with a newline */  
  sputs("\n",OutBuff);
  return(0);
}

  
/*-----------------------------------------------------------------------
  Routine to make a character buffer look like an output file, i.e.
  writing appends information to the buffer. Also prevents buffer overflow.

  Appends string to buff, returning number of characters written (excluding
  the terminating '\0', which is always written).
  ----------------------------------------------------------------------*/
int
sputs(const char *string, SBUFF *buff){
  int i, iLeft = buff->iLen - buff->iPos - 1;
  char *pzCurrBuff = &(buff->pzString[buff->iPos]);

  for(i=0; i<iLeft; i++){
    if(string[i] == '\0')break;
    pzCurrBuff[i]=string[i];
  }
  pzCurrBuff[i]='\0';
  buff->iPos += i;
  return(i);
}

/**********************************************************************
GenParse

Parses input line and returns the next token. Advances the input pointer to
the char following the returned token (in most cases - see below).

Has several classes of token delimiters:

   (1) WHITESPACE delimits other tokens but is not a token itself.

   (2) SELF-DELIMIT is itself a single-character token.

   (3) LINE-TERMINATOR is like self-delimit, except that the input pointer
       is never advanced beyond the token (MoveToNextLine must be called
       to do this). Also, all line terminators are returned as a newline
       token.

   (4) FILE-TERMINATOR like line-terminator, but null is returned, nor does
       the function MoveToNextLine advance the input pointer over it.


NB iMaxTokenLen must be >= 2
***********************************************************************/

static const char*
GenParse(const char *pzCurrentIn, char *pzToken, int iMaxTokenLen)
{
  static const char *pzWhiteSpace=" \t", 
  *pzAll=" \t\"',=@()\n!;";
  const char *pzIn=pzCurrentIn;
  int pos;

/* Skip whitespace */

  pos=strspn(pzIn,pzWhiteSpace);
  pzIn=pzCurrentIn+pos;

/* Now see what we hit to determine how much to make into a token */

  switch (*pzIn){

/* One-character tokens: grab and advance input pointer */

  case ',' :
  case '=' :
  case '@' :
  case '(' :
  case ')' :
  case '"' :
  case '\'' :
    pzToken[0] = *pzIn;
    pzToken[1] = '\0';
    pzIn++;
    break;

/* Line-terminator tokens: represent as newline and do not advance pointer */

  case '!' :
  case ';' :
  case '\n' :
    pzToken[0] = '\n';
    pzToken[1] = '\0';
    break;

/* End-of-file: represent as null and do not advance pointer */

  case '\0' :
    pzToken[0] = '\0';
    break;


/* Everything else: copy up to next delimiter and advance pointer */

  default :
    pos=strcspn(pzIn,pzAll);
    if(pos > 0)strncpy(pzToken,pzIn,MIN(pos,iMaxTokenLen-1));
    pzToken[MIN(pos,iMaxTokenLen-1)]='\0';
    pzIn +=pos;
  }
  return(pzIn);
}


/********************************************************************
MoveToNextLine

Step the input pointer forward past the next newline character, which can
include the character currently pointed to. However, don't go past a null.

*********************************************************************/
static const char*
MoveToNextLine(const char *psLine)
{
  const char *psIn=psLine;

  while (*psIn != '\0' && *psIn != '\n')psIn++;
  if(*psIn !='\0')psIn++;
  return(psIn);
}
/********************************************************************
LookUp

Table lookup function. Will have globbing one day.
*********************************************************************/

static int 
LookUp(const char *pzToken, const char **ppzTable, int iStride, int iNpar,
       enum MinMatchType *piMatchType)
{
  const char *pzKey;
  int iMatch, iRec, iMatchRec, iExact, iStatus = 0;

  *piMatchType=MatchNoMatch;
  iMatchRec=-1;
  for (iRec = 0; iRec < iNpar; iRec++){

/* Get next Key to match with */

    pzKey=*( (char **) ( (char *)ppzTable + iRec*iStride ) );

/* Case-insensitive partial match */

    iMatch = MinMatch(pzToken, pzKey, &iExact, &iStatus);

/* Do minimal-match logic */

    if (iMatch) {
      if (iExact){
	iMatchRec=iRec;
	*piMatchType=MatchExact;
	break;
      } else if (iMatchRec != -1) {
	*piMatchType=MatchAmbiguous;
	break;
      } else {
        iMatchRec=iRec;
	*piMatchType=MatchMinimal;
      }
    }
  }
  return(iMatchRec);
}

/*************************************************************************
MatchEnum

Match an input string with a list of possible values given in an enumeration
options string, specified as "enum=enum0:enum1:enum2:...:enumN"

Returns option number in piElement. Enums begin at 0, as in C.
If no minimal-match is found, return FALSE, otherwise TRUE.
**************************************************************************/
static int
MatchEnum(const char *pzToken, const char *pzOpts, int *piElement)
{
  const char *pzCurrOpt = pzOpts;
  int i, j, bMatch, iMatch, chTok, chKey;
  enum MinMatchType iMatchType;

/* Find enum option */
  while (pzCurrOpt[0] != 0) {
    pzCurrOpt += strspn(pzCurrOpt,", \t");
    if(! strncmp(pzCurrOpt,"enum=",5) ) break;
    pzCurrOpt += strcspn(pzCurrOpt,", \t") + 1;
  }

/* Look for matching option */
  iMatchType = MatchNoMatch;
  iMatch = -1;
  pzCurrOpt = pzCurrOpt+5;
  for (i=0; pzCurrOpt[0] != 0; i++){
    
/* Case-insensitive partial match, i.e. a Token shorter than Key can match */
    j=0; bMatch = TRUE;
    while((chTok = pzToken[j]) != 0){
      if ((  chKey = pzCurrOpt[j]) == ':' 
	  || chKey == ',' 
	  || chKey == 0 ){
	bMatch=FALSE;
	break;
      }
      chKey=tolower(chKey);
      chTok=tolower(chTok);
      if (chKey != chTok ){
	bMatch=FALSE;
	break;
      }
      j++;
    }

/* Do minimal-match logic */
    if (bMatch) {
      if ((  chKey = pzCurrOpt[j]) == ':' 
	  || chKey == ',' 
	  || chKey == 0 ){
	iMatch=i;
	iMatchType=MatchExact;
	break;
      }
      if (iMatchType != MatchNoMatch){
	iMatchType=MatchAmbiguous;
      } else {
        iMatch=i;
	iMatchType=MatchMinimal;
      }
    }
    
/* Fetch Next Option */
    pzCurrOpt += strcspn(pzCurrOpt, ":,");
    if ( pzCurrOpt[0] == ',')break;
    pzCurrOpt++;
  }

/* Figure out if we were successful */
  if (iMatchType == MatchMinimal || iMatchType == MatchExact ){
    *piElement=iMatch;
    return(TRUE);
  } else {
    *piElement=-1;
    return(FALSE);
  }
}


/****************************************************************************
DoBuiltIn

Executes a builtin function
*****************************************************************************/
static const char *
DoBuiltIn(const char *pzCommand, const keyrec KeyTable[], int iNpar, 
	  char *pzProgName, int iOutputLineLen,
	  const char *pzCurrentIn, SBUFF *OutBuff,
	  char *pzReturnString)
{
  char pzToken[MAX_TOKEN_LEN], pzLine[OUTBUFFSIZE];
  FILE *OutFile, *InFile;
  int iPar;
  SBUFF LineBuff;
  kioStruc IOStream;
  enum MinMatchType iMatchTypeVar;
  int iStatus = 0, iExact;

  if(!strcmp(pzCommand,"SHOW")){
    pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
    if ( strchr("\n!;", *pzToken) ){
      for (iPar=0; iPar < iNpar; iPar++){
	ShowValue(&KeyTable[iPar], OutBuff, iOutputLineLen);
      }
    } else {
      iPar=LookUp(pzToken, (const char **)&(KeyTable[0].key),
		  sizeof(KeyTable[0]), iNpar, &iMatchTypeVar);
      switch (iMatchTypeVar){

      case MatchAmbiguous :
	for (iPar=0; iPar < iNpar; iPar++){
	  if (MinMatch(pzToken, KeyTable[iPar].key, &iExact, &iStatus)){
	    ShowValue(&KeyTable[iPar], OutBuff, iOutputLineLen);
	  }
	}
	break;

      case MatchExact :
      case MatchMinimal :
	ShowValue(&KeyTable[iPar], OutBuff, iOutputLineLen);
	break;

      case MatchNoMatch :
	sputs("**Unknown variable : '", OutBuff);
	sputs(pzToken, OutBuff);
	sputs("'\n", OutBuff);
	break;
      }
    }
  }
  else if (!strcmp(pzCommand,"SAVE")){
    pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
    if ( strchr("\n!;", *pzToken) ){
      strcpy(pzToken, pzProgName);
      strcat(pzToken, ".par");
    }
    OutFile=fopen(pzToken, "wt");
    if (OutFile == NULL){
      sputs("**Unable to open file '", OutBuff);
      sputs(pzToken, OutBuff);
      sputs("'\n",OutBuff);
    } else {
      LineBuff.pzString = pzLine;
      LineBuff.iLen = sizeof(pzLine);
      for (iPar=0; iPar < iNpar; iPar++){
	LineBuff.iPos=0;
	ShowValue(&KeyTable[iPar], &LineBuff, iOutputLineLen);
	fputs(pzLine, OutFile);
      }
      fclose(OutFile);
      sputs("Output in file '", OutBuff);
      sputs(pzToken, OutBuff);
      sputs("'\n",OutBuff);
    }
  }
  else if (!strcmp(pzCommand,"@") || !strcmp(pzCommand, "LOAD")){
    pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
    if ( strchr("\n!", *pzToken) ){
      strcpy(pzToken, pzProgName);
      strcat(pzToken, ".par");
    }
    InFile=fopen(pzToken, "rt");
    if (InFile == NULL){
      sputs("**Unable to open file '", OutBuff);
      sputs(pzToken, OutBuff);
      sputs("'\n",OutBuff);
    } else {
      sputs("Loading file '", OutBuff);
      sputs(pzToken, OutBuff);
      sputs("'\n",OutBuff);
      IOStream.Type = kioLoad;
      IOStream.iOutputLineLen = iOutputLineLen;
      IOStream.iMaxOutputChars = OutBuff->iLen - OutBuff->iPos;
      IOStream.In.File = InFile;
      IOStream.Out.Buff = OutBuff->pzString + OutBuff->iPos;
      keyinc(KeyTable, iNpar, pzProgName,
	     IOStream,
	     pzReturnString);
      fclose(InFile);
      OutBuff->iPos=strlen(OutBuff->pzString);
    }
  }
  else if (! strcmp(pzCommand,"QUIT") || !strcmp(pzCommand, "/") ){
    strcpy(pzReturnString, pzCommand);
  }
  else if (!strcmp(pzCommand,"HELP")){
    pzCurrentIn=GenParse(pzCurrentIn, pzToken, MAX_TOKEN_LEN);
    if ( strchr("\n!", *pzToken) ){
      for (iPar=0; iPar < iNpar; iPar++){
	sputs(KeyTable[iPar].key, OutBuff);
	sputs(" : ", OutBuff);
	sputs(KeyTable[iPar].helpstring, OutBuff);
	sputs("\n", OutBuff);
      }
    } else {
      iPar=LookUp(pzToken, (const char **)&(KeyTable[0].key),
		  sizeof(KeyTable[0]), iNpar, &iMatchTypeVar);
      if (iPar < 0) {
	sputs("**Unknown variable: '", OutBuff);
	sputs(pzToken, OutBuff);
	sputs("'\n",OutBuff);
      } else {
	sputs(KeyTable[iPar].key, OutBuff);
	sputs(" : ", OutBuff);
	sputs(KeyTable[iPar].helpstring, OutBuff);
	sputs("\n", OutBuff);
      }
    }
  }
  else {
    sputs("**Unimplemented builtin: '",OutBuff);
    sputs(pzCommand, OutBuff);
    sputs("'\n", OutBuff);
  }
  return(pzCurrentIn);
}

/***********************************************************************

 ****MinMatch

 Purpose: do case-insensitive minimal matching, i.e. S1 shorter than S2
          can match, providing all the characters of S1 match the
	  corresponding characters of S2
 Return:  TRUE if there is a match. *piExact is true if it is an exact match,
          i.e. both strings the same length.
 ************************************************************************/
static int
MinMatch(const char *azString1, const char *azString2, int *piExact,
	 int *piStatus)
{
  int iChar, iMatch;
  int iS1, iS2;

  if (*piStatus) return FALSE;

  iMatch = TRUE;
  for (iChar=0; (iS1 = azString1[iChar]) != 0; iChar++){
    if ((iS2 = azString2[iChar]) == 0){
      iMatch=FALSE;
      break;
    }
    iS1=tolower(iS1);
    iS2=tolower(iS2);
    if (iS2 != iS1 ){
      iMatch=FALSE;
      break;
    }
  }
  if (iMatch && azString2[iChar] == 0) {
    *piExact = TRUE;
  } else {
    *piExact = FALSE;
  }
  return(iMatch);
}
