/*******************************************************************************
vis2con.c

USNO/NRL Optical Interferometer
3450 Massachusetts Avenue NW
Washington DC 20392-5400

System: Data Reduction
Subsystem: Data transfer Mark III --> HDS

Description:

This program reads Mark III vis-files and
writes the data into a constrictor-format
HDS-file.

Modification history:

28-Jan-1994  C.A.Hummel, file created
17-Sep-1999  C.A.Hummel, added missing objects
*******************************************************************************/

/* includes */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include </home/cah/starlink/include/sae_par.h>
#include </home/cah/starlink/include/dat_par.h>

/* defines */

#define RECL 76
#define MAXREC 50000
#define MAXSCANS 300
#define MAXPTS  600
#define PISQ 9.8696

/* function declarations */
int parse(void *ptr,char *ia,short iaf,short ial);

/*----------------------------------------------------------------------------*/
main(){
	/* Binary .vis file has 76 byte record length with following data: */
	char ia[RECL];
	double time0,dlas;
	float anum[6],aden[4],ha0,dec0,avar[4],zd0,tfirst,tlast,phirms,phi12;
	long idark[4],ifilt[4],djtr;
	short icode,istar,ibase,tau0,dtau0,tsample,imon,iday,iyear,jittr,ilock;

	/* Additional variables from vplot: */
	long ifmt;
	long rec0[MAXSCANS],rec1[MAXSCANS],drec[MAXSCANS],base[MAXSCANS];
	short ifirst,ilast,inscan;
	int NumCoh[MAXSCANS],NumIncoh[MAXSCANS],NumPoint[MAXSCANS],ScanId[MAXSCANS];
	int NumSid,NumOutBeam;
	int NumBaseline[2],NumTriple,NumSpecChan[2],FringeMod[2][1];
	int SiderostatId[2],DelayLineId[2],StarTrackerId[2];
	int BeamCombinerId,BCInputId[2];
	int i,iscan,iBeam;
	double delay[2][MAXSCANS],CohAvgInt[MAXSCANS],BackgndTime[MAXSCANS];
	double PointTime[MAXPTS],StartTime[MAXSCANS],StopTime[MAXSCANS];
	double Wavelength[2][2],WavelengthErr[2][2],ChanWidth[2][2],ChanWidthErr[2][2];
	double waves[]={700.0,399.7,400.0,400.0,451.8,450.2,450.0,501.2,500.0, \
		      500.0,550.1,550.0,550.0,600.9,599.5,600.0,650.6,650.0, \
		      650.0,699.0,700.0,700.0,750.1,750.0,750.0,799.2,800.0, \
		      800.0,687.0,397.0,410.0,434.0,486.0,656.0,650.0,750.0, \
		      656.0,  0.0,  0.0,  0.0,  0.0,623.0,672.0,701.0,712.0, \
		      754.0};
	double widths[46];
	double Latitude,Longitude,Altitude,Cohint;
	double FDLPos[2][MAXPTS];
	float BackgndRate[2][MAXSCANS][2],BackgndErr[2][MAXSCANS][2];
	float zd[MAXSCANS],ha[MAXSCANS],dec[MAXSCANS];
	float VisSq[2][MAXPTS][1][2],VisSqErr[2][MAXPTS][1][2];
	float ComplexVis[2][MAXPTS][1][2][2],ComplexVisErr[2][MAXPTS][1][2][2];
	float PhotonRate[2][MAXPTS][2],PhotonRateErr[2][MAXPTS][2];
	float FDLJitter[2][MAXPTS],FDLJitterErr[2][MAXPTS],cohint;
	double BGDec[MAXSCANS],BGRa[MAXSCANS];
	char SpectrometerId[2][14]={"SPECTROMETER1","SPECTROMETER2"};
	char StarId[MAXSCANS][8],StationId[2][4],BaselineIds[2][1][8];
	char BaselineId[38][8]={"NAS-SAS","NAS-EAS","NA_-SA_","NA_-SB_","NB_-SA_",
	"NB_-SB_","NC_-SA_","NA_-SC_","NC_-SB_","NB_-SC_","ND_-SA_","NC_-SC_",
	"NA_-SD_","ND_-SB_","NE_-SA_","NB_-SD_","NE_-SB_","ND_-SC_","NF_-SA_",
	"NC_-SD_","NA_-SE_","NF_-SB_","NE_-SC_","NB_-SE_","ND_-SD_","NC_-SE_",
	"NA_-SF_","NF_-SC_","NE_-SD_","NB_-SF_","ND_-SE_","NC_-SF_","NF_-SD_",
	"NE_-SE_","ND_-SF_","NF_-SE_","NE_-SF_","NF_-SF_"};
	double StationCoords[15][4]={
		{  0.000000,   0.000000,   0.000000,   0.000000},
		{  0.032600, -12.005001,   0.003622,   0.005900},
		{-10.061400,  -6.526640,  -0.001800,  -4.588900},
		{  0.000000,   0.000000,   0.000000,   0.000000},
		{ -0.012200,   2.302114,  -0.001167,   2.302100},
		{ -0.026900,   3.912477,  -0.001232,   3.910000},
		{ -0.050500,   7.636018,  -0.002759,   7.634200},
		{ -0.028300,   9.750615,  -0.001683,   9.746700},
		{ -0.083800,  12.140564,  -0.002376,  12.138000},
		{  0.018800,  -2.991314,   0.002432,   0.707100},
		{  0.031700,  -4.252417,   0.001769,   0.936200},
		{  0.053200,  -7.514050,   0.001411,   4.199000},
		{  0.071000, -11.446727,   0.000516,   8.124000},
		{  0.107500, -15.444887,  -0.000648,  12.122700},
		{  0.108000, -19.370790,  -0.005280,  16.053300}};
	double StationCoord[2][4];
	char StationIds[15][4]={"NAS","SAS","EAS",
                                "NA_","NB_","NC_","ND_","NE_","NF_",
				"SA_","SB_","SC_","SD_","SE_","SF_"};

	/* Other variables this program needs: */
	long irec,j,k;
	short iaf,ial,iftr,eof;
	FILE *fp;

	/* HDS variables declarations: */
	char loc1[DAT__SZLOC],loc2[DAT__SZLOC],loc3[DAT__SZLOC],loc4[DAT__SZLOC];
	char loc5[DAT__SZLOC],loc6[DAT__SZLOC],loc7[DAT__SZLOC];
	int len0,len1,len2,len3,len4;
	char ext[]="con",*loc,*systemid,*date,*format;
	char *file,*name,*type;
	int status,ndim,ch;
	int dims[7]; /* The FORTRAN convention has to be used here! */

        /* Allocate memory for strings */
	if((file=(char*)malloc(80))==NULL){
		fprintf(stderr,"\nNot enough memory to allocate buffer");
		exit(1);
		}
	if((name=(char*)malloc(40))==NULL){
		fprintf(stderr,"\nNot enough memory to allocate buffer");
		exit(1);
		}
	if((type=(char*)malloc(40))==NULL){
		fprintf(stderr,"\nNot enough memory to allocate buffer");
		exit(1);
		}
	if((systemid=(char*)malloc(40))==NULL){
		fprintf(stderr,"\nNot enough memory to allocate buffer");
		exit(1);
		}
	if((format=(char*)malloc(40))==NULL){
		fprintf(stderr,"\nNot enough memory to allocate buffer");
		exit(1);
		}
	if((date=(char*)malloc(11))==NULL){
		fprintf(stderr,"\nNot enough memory to allocate buffer");
		exit(1);
		}

	/* Open .vis file */
	printf("%s","\nPlease enter .vis filename: ");
	scanf("%s",file);
	if((fp=fopen(file,"rb"))==NULL){
		fprintf(stderr,"\nError opening file!");
		exit(1);
		}
	else printf("\nFile opened.");

	/* Open HDS file */
	status=SAI__OK; 
	hds_start_(&status);
	printf("\nHDS started; status = %d",status);

	/* Create container file */
	ch='.';
	loc=strchr(file,ch);
	strcpy(loc+1,ext)           ;len1=strlen(file);
	strcpy(name,"Session")      ;len2=strlen(name);
	strcpy(type,"CONSTRICTOR")  ;len3=strlen(type);
	len0=DAT__SZLOC;
	ndim=0;
	hds_new_(file,name,type,&ndim,dims,loc1,&status,len1,len2,len3,len0); 

	/* Create component 'SystemID' */ 
	strcpy(name,"SystemID")       ;len1=strlen(name);
	strcpy(type,"_char*8")        ;len2=strlen(type);
	strcpy(systemid,"Mark_III")   ;len3=strlen(systemid);
	dat_new_(loc1,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc1,name,loc2,&status,len0,len1,len0);
	dat_putc_(loc2,&ndim,dims,(int*)systemid,&status,len0,len3);
	dat_annul_(loc2,&status,len0);

	/* Create component 'Format' */
	strcpy(name,"Format")       ;len1=strlen(name);
	strcpy(type,"_char*11")     ;len2=strlen(type);
	strcpy(format,"CONSTRICTOR");len3=strlen(format);
	dat_new_(loc1,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc1,name,loc2,&status,len0,len1,len0);
	dat_putc_(loc2,&ndim,dims,(int*)format,&status,len0,len3);
	dat_annul_(loc2,&status,len0);

	/* Create component 'GeoParms' */
	strcpy(name,"GeoParms");len1=strlen(name);
	strcpy(type,"GeoParms");len2=strlen(type);
	ndim=0;
	dat_new_(loc1,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc1,name,loc2,&status,len0,len1,len0);

	strcpy(name,"Latitude");len1=strlen(name);
	strcpy(type,"_double") ;len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	Latitude=34.2166944;
	cmp_put0d_(loc2,name,&Latitude,&status,len0,len1);

	strcpy(name,"Longitude");len1=strlen(name);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	Longitude=-118.059167;
	cmp_put0d_(loc2,name,&Longitude,&status,len0,len1);

	strcpy(name,"Altitude");len1=strlen(name);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	Altitude=1742.;
	cmp_put0d_(loc2,name,&Altitude,&status,len0,len1);
	dat_annul_(loc2,&status,len0);

	/* Create component 'ScanData' */
	strcpy(name,"ScanData");len1=strlen(name);
	strcpy(type,"TABLE")   ;len2=strlen(type);
	ndim=0;
	dat_new_(loc1,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc1,name,loc2,&status,len0,len1,len0);

		/* Create sub-component 'PointData' */
		ndim=1;dims[0]=MAXSCANS;
		strcpy(name,"PointData");len1=strlen(name);
		strcpy(type,"EXTTABLE") ;len2=strlen(type);
		dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
		dat_annul_(loc2,&status,len0);

	/* Read data and write to HDS whenever a scan is finished */
	iscan=0;
	iftr=1;
	ilast=0;
	inscan=0;
	eof=0;
	for(i=0;i<MAXSCANS;i++)drec[i]=-1;
	for(irec=1;irec<MAXREC;irec++){
		if(fread(ia,1,RECL,fp)!=RECL){
			printf("\nEND-OF-FILE encountered!");
			if(inscan && (ilast==0))iscan-=1;
			eof=1;
			}
		iaf=0;ial=7;
		parse(&time0,ia,iaf,ial);
		if(irec==0 && time0!=0){
				fprintf(stderr,"\nFirst record of file is not header!");
				exit(1);
				}
		iaf=8;ial=9;
		parse(&icode,ia,iaf,ial);
		if(time0==0){ 			/* Header record */
			printf("\nRecord %d is header",irec);
			iaf=8;ial=11;
			parse(&ifmt,ia,iaf,ial);
			if(ifmt==2){
				iaf=14;ial=15;
				parse(&tau0,ia,iaf,ial);
				cohint=(float)tau0;}
			else
				cohint=4.0;
			printf(",\t Format=%ld, cohint=%f",ifmt,cohint);
			if(inscan && (ilast==0)){iscan-=1;inscan=0;}
			}
		else if(time0>0){		 /* Data record */
			if(inscan){
				ilast=irec;
				for(j=0;j<4;j++){
					iaf=8+j*4;ial=11+j*4;
					parse(anum+j,ia,iaf,ial);
					iaf=32+j*4;ial=35+j*4;
					parse(aden+j,ia,iaf,ial);
					PhotonRate[(int)fmod((double)j,2.0)][ilast-ifirst-1][j/2]=aden[j];
					PhotonRateErr[(int)fmod((double)j,2.0)][ilast-ifirst-1][j/2]=sqrt(aden[j]);
					VisSq[(int)fmod((double)j,2.0)][ilast-ifirst-1][0][j/2]=anum[j];
					VisSqErr[(int)fmod((double)j,2.0)][ilast-ifirst-1][0][j/2]=1.0;
					ComplexVis[(int)fmod((double)j,2.0)][ilast-ifirst-1][0][j/2][0]=sqrt(fabs(anum[j]))/1.414213562;
					ComplexVis[(int)fmod((double)j,2.0)][ilast-ifirst-1][0][j/2][1]=0.0;
					ComplexVisErr[(int)fmod((double)j,2.0)][ilast-ifirst-1][0][j/2][0]=sqrt(fabs(anum[j]))/10.0;
					ComplexVisErr[(int)fmod((double)j,2.0)][ilast-ifirst-1][0][j/2][1]=sqrt(fabs(anum[j]))/10.0;
					}
				PointTime[ilast-ifirst-1]=(double)(time0*24.*60.*60.*1000.);
				iaf=64;ial=65;
				parse(&jittr,ia,iaf,ial);
				if(jittr<0)jittr+=256;
				iaf=68;ial=71;
				parse(&phirms,ia,iaf,ial);
				FDLJitter[1][ilast-ifirst-1]=phirms;
				FDLJitter[0][ilast-ifirst-1]=0.1;
				FDLJitterErr[1][ilast-ifirst-1]=0.1;
				FDLJitterErr[0][ilast-ifirst-1]=0.0;
				}
			} 
		else if(icode==0){		/* Time/new scan record */
			if(inscan && (ilast==0))iscan-=1;
			/* Get baseline */
			iaf=12;ial=13;
			parse(&ibase,ia,iaf,ial);
			/* Get date */
			iaf=18;ial=19;
			parse(&imon,ia,iaf,ial);
			iaf=20;ial=21;
			parse(&iday,ia,iaf,ial);
			iaf=22;ial=23;
			parse(&iyear,ia,iaf,ial);
			/* Get filter information */
			for(j=0;j<4;j++){
				iaf=28+j*4;ial=31+j*4;
				parse(ifilt+j,ia,iaf,ial);
				}
			if(((iyear+imon+iday)!=0)&&((ifilt[0]+ifilt[1]+ifilt[2]+ifilt[3])!=0)&&iftr){
				sprintf(date,"%4d-%2.2d-%2.2d",iyear,imon,iday);
							len3=strlen(date);
				strcpy(name,"Date")    ;len1=strlen(name);
				strcpy(type,"_char*10");len2=strlen(type);
				ndim=0;
				dat_new_(loc1,name,type,&ndim,dims,&status,len0,len1,len2);
				dat_find_(loc1,name,loc2,&status,len0,len1,len0);
				dat_putc_(loc2,&ndim,dims,(int*)date,&status,len0,len3);
				dat_annul_(loc2,&status,len0);
				for(j=0;j<4;j++){
					Wavelength[(int)fmod((double)j,2.0)][j/2]=waves[ifilt[j]]*1.e-9;
					WavelengthErr[(int)fmod((double)j,2.0)][j/2]=(double)0.5*1.e-9;
					ChanWidth[(int)fmod((double)j,2.0)][j/2]=(double)25.0*1.e-9;
					ChanWidthErr[(int)fmod((double)j,2.0)][j/2]=(double)0.5*1.e-9;
					}
				iftr=0;
				}
			inscan=1;
			iscan+=1;
			ifirst=irec;
			ilast=0;
			/* Get star number */
			iaf=10;ial=11;
			parse(&istar,ia,iaf,ial);
			sprintf(StarId[iscan-1],"FKV%4.4d",istar);
			}
		else if(icode==1){		 /* Dark data record */
			if(inscan && (ilast==0)){iscan-=1;inscan=0;}
			else if(inscan){
				drec[iscan-1]=irec;
				for(j=0;j<4;j++){
					iaf=24+j*4;ial=27+j*4;
					parse(idark+j,ia,iaf,ial);
					}
				iaf=14;ial=15;
				parse(&dtau0,ia,iaf,ial);
				for(j=0;j<4;j++){
					BackgndRate[(int)fmod((double)j,2.0)][iscan-1][j/2]=(float)cohint*idark[j]/(float)dtau0;
					BackgndErr[(int)fmod((double)j,2.0)][iscan-1][j/2]=0.05;
					}

				iaf=16;ial=23;
				parse(&dlas,ia,iaf,ial);
				dlas=dlas/1000000.0;
				delay[0][iscan-1]=0.0;
				delay[1][iscan-1]=-dlas;
				iaf=48;ial=51;
				parse(&zd0,ia,iaf,ial);
				zd[iscan-1]=zd0;
				iaf=40;ial=43;
				parse(&ha0,ia,iaf,ial);
				ha[iscan-1]=ha0;
				iaf=44;ial=47;
				parse(&dec0,ia,iaf,ial);
				dec[iscan-1]=dec0;
				}
			}
		else{
			fprintf(stderr,"\nRecord %d illegal format!",irec);
			exit(1);
			} 
		/* Write data of a scan to HDS */
		if(time0<=0 && inscan && (ilast!=0)){
			ScanId[iscan-1]=iscan;
			rec0[iscan-1]=ifirst;
			rec1[iscan-1]=ilast;
			inscan=0;
			NumPoint[iscan-1]=ilast-ifirst;
			StartTime[iscan-1]=PointTime[0];
			StopTime[iscan-1]=PointTime[ilast-ifirst-1];
			for(i=0;i<ilast-ifirst;i++){
				for(j=0;j<2;j++){
					FDLPos[j][i]=-delay[j][iscan-1];
					}
				}
			CohAvgInt[iscan-1]=(double)cohint;
			NumIncoh[iscan-1]=(int)(500.0/cohint);
			NumCoh[iscan-1]=1;
			BackgndTime[iscan-1]=StopTime[iscan-1];
			BGDec[iscan-1]=0.0;
			BGRa[iscan-1]=0.0;
			if(drec[iscan-1]==(-1)){
				printf("\nWarning: no dark record found; scan%d",iscan);
				for(j=0;j<4;j++){
					BackgndRate[(int)fmod((double)j,2.0)][iscan-1][j/2]=0.;
					BackgndErr[(int)fmod((double)j,2.0)][iscan-1][j/2]=0.0;
					}
				}

			/* Fill in cell of array PointData in component 'ScanData' */
			strcpy(name,"ScanData");len1=strlen(name);
			dat_find_(loc1,name,loc2,&status,len0,len1,len0);
			strcpy(name,"PointData");len1=strlen(name); 
			dat_find_(loc2,name,loc3,&status,len0,len1,len0);
			ndim=1;
			dat_cell_(loc3,&ndim,&iscan,loc4,&status,len0,len0);

			strcpy(name,"OutputBeam");len1=strlen(name);
			strcpy(type,"Beam")      ;len2=strlen(type);
			ndim=1;dims[0]=2;
			dat_new_(loc4,name,type,&ndim,dims,&status,len0,len1,len2);
			dat_find_(loc4,name,loc5,&status,len0,len1,len0);
			iBeam=1;
			dat_cell_(loc5,&ndim,&iBeam,loc6,&status,len0,len0);

			strcpy(name,"VisSq");len1=strlen(name);
			strcpy(type,"_real");len2=strlen(type);
			ndim=3;dims[2]=ilast-ifirst;dims[1]=1;dims[0]=2;
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)VisSq[0],dims,&status,len0,len1);
			strcpy(name,"VisSqErr");len1=strlen(name);
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)VisSqErr[0],dims,&status,len0,len1);

			strcpy(name,"ComplexVis");len1=strlen(name);
			ndim=4;dims[3]=ilast-ifirst;dims[2]=1;dims[1]=2;dims[0]=2;
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)ComplexVis[0],dims,&status,len0,len1);
			strcpy(name,"ComplexVisErr");len1=strlen(name);
			ndim=4;dims[3]=ilast-ifirst;dims[2]=1;dims[1]=2;dims[0]=2;
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)ComplexVisErr[0],dims,&status,len0,len1);

			strcpy(name,"PhotonRate");len1=strlen(name);
			ndim=2;dims[1]=ilast-ifirst;dims[0]=2;
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)PhotonRate[0],dims,&status,len0,len1);
			strcpy(name,"PhotonRateErr");len1=strlen(name);
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)PhotonRateErr[0],dims,&status,len0,len1);
			dat_annul_(loc6,&status,len0);

			iBeam=2;
			ndim=1;
			dat_cell_(loc5,&ndim,&iBeam,loc6,&status,len0,len0);
			ndim=3;dims[2]=ilast-ifirst;dims[1]=1;dims[0]=2;
			strcpy(name,"VisSq");len1=strlen(name);
			strcpy(type,"_real");len2=strlen(type);
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)VisSq[1],dims,&status,len0,len1);
			strcpy(name,"VisSqErr");len1=strlen(name);
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)VisSqErr[1],dims,&status,len0,len1);

			strcpy(name,"ComplexVis");len1=strlen(name);
			ndim=4;dims[3]=ilast-ifirst;dims[2]=1;dims[1]=2;dims[0]=2;
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)ComplexVis[1],dims,&status,len0,len1);
			strcpy(name,"ComplexVisErr");len1=strlen(name);
			ndim=4;dims[3]=ilast-ifirst;dims[2]=1;dims[1]=2;dims[0]=2;
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)ComplexVisErr[1],dims,&status,len0,len1);

			strcpy(name,"PhotonRate");len1=strlen(name);
			ndim=2;dims[1]=ilast-ifirst;dims[0]=2;
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)PhotonRate[1],dims,&status,len0,len1);
			strcpy(name,"PhotonRateErr");len1=strlen(name);
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)PhotonRateErr[1],dims,&status,len0,len1);
			dat_annul_(loc6,&status,len0);
			dat_annul_(loc5,&status,len0);

			strcpy(name,"InputBeam");len1=strlen(name);
			strcpy(type,"InputBeam");len2=strlen(type);
			ndim=1;dims[0]=2;
			dat_new_(loc4,name,type,&ndim,dims,&status,len0,len1,len2);
			dat_find_(loc4,name,loc5,&status,len0,len1,len0);
			iBeam=1;
			dat_cell_(loc5,&ndim,&iBeam,loc6,&status,len0,len0);
			strcpy(name,"FDLPosErr")  ;len1=strlen(name);
			strcpy(type,"_real")      ;len2=strlen(type);
			ndim=1;dims[0]=ilast-ifirst;
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)FDLJitter[0],dims,&status,len0,len1);
/*
			strcpy(name,"FDLJitterErr");len1=strlen(name);
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)FDLJitterErr[0],dims,&status,len0,len1);
*/
			strcpy(name,"FDLPos") ;len1=strlen(name);
			strcpy(type,"_double");len2=strlen(type);
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnd_(loc6,name,&ndim,dims,(double*)FDLPos[0],dims,&status,len0,len1);
			dat_annul_(loc6,&status,len0);
			iBeam=2;
			dat_cell_(loc5,&ndim,&iBeam,loc6,&status,len0,len0);
			strcpy(name,"FDLPosErr")  ;len1=strlen(name);
			strcpy(type,"_real")      ;len2=strlen(type);
			ndim=1;dims[0]=ilast-ifirst;
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)FDLJitter[1],dims,&status,len0,len1);
/*
			strcpy(name,"FDLJitterErr");len1=strlen(name);
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc6,name,&ndim,dims,(float*)FDLJitterErr[1],dims,&status,len0,len1);
*/
			strcpy(name,"FDLPos") ;len1=strlen(name);
			strcpy(type,"_double");len2=strlen(type);
			dat_new_(loc6,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnd_(loc6,name,&ndim,dims,(double*)FDLPos[1],dims,&status,len0,len1);
			dat_annul_(loc6,&status,len0);
			dat_annul_(loc5,&status,len0);

			strcpy(name,"Time")   ;len1=strlen(name);
			strcpy(type,"_double");len2=strlen(type);
			ndim=1;dims[0]=ilast-ifirst;
			dat_new_(loc4,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_put1d_(loc4,name,dims,PointTime,&status,len0,len1);

			dat_annul_(loc4,&status,len0);
			dat_annul_(loc3,&status,len0);
			dat_annul_(loc2,&status,len0);
			}
		if(eof)break;
		}			
	fclose(fp);

	/* Fill in rest of 'ScanData' items */
	strcpy(name,"ScanData");len1=strlen(name);
	dat_find_(loc1,name,loc2,&status,len0,len1,len0);

	ndim=1;
	dims[0]=iscan;
	strcpy(name,"ScanId")  ;len1=strlen(name);
	strcpy(type,"_integer");len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1i_(loc2,name,dims,ScanId,&status,len0,len1);

	strcpy(name,"StarID") ;len1=strlen(name);
	strcpy(type,"_char*8");len2=strlen(type);
	len3=8;
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1c_(loc2,name,dims,StarId[0],&status,len0,len1,len3);

	strcpy(name,"StartTime");len1=strlen(name);
	strcpy(type,"_double")  ;len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1d_(loc2,name,dims,StartTime,&status,len0,len1);

	strcpy(name,"StopTime");len1=strlen(name);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1d_(loc2,name,dims,StopTime,&status,len0,len1);

	strcpy(name,"NumCoh")  ;len1=strlen(name);
	strcpy(type,"_integer");len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1i_(loc2,name,dims,NumCoh,&status,len0,len1);

	strcpy(name,"NumIncoh");len1=strlen(name);
	strcpy(type,"_integer");len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1i_(loc2,name,dims,NumIncoh,&status,len0,len1);

	strcpy(name,"NumPoint");len1=strlen(name);
	strcpy(type,"_integer");len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1i_(loc2,name,dims,NumPoint,&status,len0,len1);

	strcpy(name,"NumScan");len1=strlen(name);
	strcpy(type,"_integer");len2=strlen(type);
	ndim=0;
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put0i_(loc2,name,&iscan,&status,len0,len1);

	dat_annul_(loc2,&status,len0);

	/* Create component 'BGScanData' */
	strcpy(name,"BGScanData");len1=strlen(name);
	strcpy(type,"EXTTABLE")  ;len2=strlen(type);
	ndim=0;
	dat_new_(loc1,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc1,name,loc2,&status,len0,len1,len0);

	strcpy(name,"NumBGScan");len1=strlen(name);
	strcpy(type,"_integer") ;len2=strlen(type);
	ndim=0;
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put0i_(loc2,name,&iscan,&status,len0,len1);

	ndim=1;
	dims[0]=iscan;
	strcpy(name,"ScanId")  ;len1=strlen(name);
	strcpy(type,"_integer");len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1i_(loc2,name,dims,ScanId,&status,len0,len1);

	strcpy(name,"Time")       ;len1=strlen(name);
	strcpy(type,"_double")    ;len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1d_(loc2,name,dims,BackgndTime,&status,len0,len1);

	strcpy(name,"Dec")        ;len1=strlen(name);
	strcpy(type,"_double")    ;len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1d_(loc2,name,dims,BGDec,&status,len0,len1);

	strcpy(name,"Ra")        ;len1=strlen(name);
	strcpy(type,"_double")   ;len2=strlen(type);
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	cmp_put1d_(loc2,name,dims,BGRa,&status,len0,len1);

	strcpy(name,"OutputBeam");len1=strlen(name);
	strcpy(type,"Beam")      ;len2=strlen(type);
	ndim=1;
	dims[0]=2;
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc2,name,loc3,&status,len0,len1,len0);
			iBeam=1;
			ndim=1;
			dat_cell_(loc3,&ndim,&iBeam,loc4,&status,len0,len0);
			strcpy(name,"Rate") ;len1=strlen(name);
			strcpy(type,"_real");len2=strlen(type);
			ndim=2;dims[0]=2;dims[1]=iscan;
			dat_new_(loc4,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc4,name,&ndim,dims,(float*)BackgndRate[0],dims,&status,len0,len1);
			strcpy(name,"RateErr");len1=strlen(name);
			dat_new_(loc4,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc4,name,&ndim,dims,(float*)BackgndErr[0],dims,&status,len0,len1);
			dat_annul_(loc4,&status,len0);

			iBeam=2;
			ndim=1;
			dat_cell_(loc3,&ndim,&iBeam,loc4,&status,len0,len0);
			strcpy(name,"Rate");len1=strlen(name);
			ndim=2;dims[0]=2;dims[1]=iscan;
			dat_new_(loc4,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc4,name,&ndim,dims,(float*)BackgndRate[1],dims,&status,len0,len1);
			strcpy(name,"RateErr");len1=strlen(name);
			dat_new_(loc4,name,type,&ndim,dims,&status,len0,len1,len2);
			cmp_putnr_(loc4,name,&ndim,dims,(float*)BackgndErr[1],dims,&status,len0,len1);
			dat_annul_(loc4,&status,len0);
			dat_annul_(loc3,&status,len0);
			dat_annul_(loc2,&status,len0);

	/* Create component 'GenConfig' */
	strcpy(name,"GenConfig");len1=strlen(name);
	strcpy(type,"GenConfig");len2=strlen(type);
	ndim=0;
	dat_new_(loc1,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc1,name,loc2,&status,len0,len1,len0);

	strcpy(name,"InstrCohInt");len1=strlen(name);
	strcpy(type,"_double")    ;len2=strlen(type);
	ndim=0;
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	Cohint=(double)cohint;
	cmp_put0d_(loc2,name,&Cohint,&status,len0,len1);

	strcpy(name,"BeamCombinerId");len1=strlen(name);
	strcpy(type,"_integer")      ;len2=strlen(type);
	ndim=0;
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	BeamCombinerId=1;
	cmp_put0i_(loc2,name,&BeamCombinerId,&status,len0,len1);

	strcpy(name,"InputBeam");len1=strlen(name);
	strcpy(type,"TABLE")    ;len2=strlen(type);
	ndim=0;
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc2,name,loc3,&status,len0,len1,len0);

		strcpy(name,"NumSid")  ;len1=strlen(name);
		strcpy(type,"_integer");len2=strlen(type);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		NumSid=2;
		cmp_put0i_(loc3,name,&NumSid,&status,len0,len1);

		ndim=1;dims[0]=2;
		strcpy(name,"StationID");len1=strlen(name);
		strcpy(type,"_char*4")  ;len2=strlen(type);
		strncpy(StationId[0],BaselineId[ibase-1],3);
		StationId[0][3]='\0';
		strncpy(StationId[1],BaselineId[ibase-1]+4,3);
		StationId[1][3]='\0';
		len3=4;
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		cmp_put1c_(loc3,name,dims,StationId[0],&status,len0,len1,len3);

		for(i=0;i<15;i++){
			if(strncmp(StationId[0],StationIds[i],3)==0){
				for(j=0;j<4;j++)StationCoord[0][j]=StationCoords[i][j];
				}
			if(strncmp(StationId[1],StationIds[i],3)==0){
				for(j=0;j<4;j++)StationCoord[1][j]=StationCoords[i][j];
				}
			}
		ndim=2;dims[1]=2;dims[0]=4;
		strcpy(name,"StationCoord");len1=strlen(name);
		strcpy(type,"_double")     ;len2=strlen(type);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		cmp_putnd_(loc3,name,&ndim,dims,StationCoord[0],dims,&status,len0,len1);

		ndim=1;dims[0]=2;
		strcpy(name,"SiderostatId");len1=strlen(name);
		strcpy(type,"_integer")    ;len2=strlen(type);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		SiderostatId[0]=1;SiderostatId[1]=2;
		cmp_put1i_(loc3,name,dims,SiderostatId,&status,len0,len1);

		strcpy(name,"DelayLineId");len1=strlen(name);
		strcpy(type,"_integer")   ;len2=strlen(type);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		DelayLineId[0]=1;DelayLineId[1]=2;
		cmp_put1i_(loc3,name,dims,DelayLineId,&status,len0,len1);

		strcpy(name,"StarTrackerId");len1=strlen(name);
		strcpy(type,"_integer")     ;len2=strlen(type);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		StarTrackerId[0]=1;StarTrackerId[1]=2;
		cmp_put1i_(loc3,name,dims,StarTrackerId,&status,len0,len1);

		strcpy(name,"BCInputId");len1=strlen(name);
		strcpy(type,"_integer") ;len2=strlen(type);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		BCInputId[0]=1;BCInputId[1]=2;
		cmp_put1i_(loc3,name,dims,BCInputId,&status,len0,len1);

		dat_annul_(loc3,&status,len0);

	strcpy(name,"OutputBeam");len1=strlen(name);
	strcpy(type,"TABLE")     ;len2=strlen(type);
	ndim=0;
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc2,name,loc3,&status,len0,len1,len0);

		strcpy(name,"NumOutBeam");len1=strlen(name);
		strcpy(type,"_integer")  ;len2=strlen(type);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		NumOutBeam=2;
		cmp_put0i_(loc3,name,&NumOutBeam,&status,len0,len1);

		strcpy(name,"NumBaseline");len1=strlen(name);
		ndim=1;
		dims[0]=2;
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		NumBaseline[0]=1;NumBaseline[1]=1;
		cmp_put1i_(loc3,name,dims,NumBaseline,&status,len0,len1);

		strcpy(name,"NumSpecChan");len1=strlen(name);
		dims[0]=2;
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		NumSpecChan[0]=2;NumSpecChan[1]=2;
		cmp_put1i_(loc3,name,dims,NumSpecChan,&status,len0,len1);

		strcpy(name,"SpectrometerId");len1=strlen(name);
		strcpy(type,"_char*14")      ;len2=strlen(type);
		ndim=1;dims[0]=2;
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		len3=14;
		cmp_put1c_(loc3,name,dims,SpectrometerId[0],&status,len0,len1,len3);

		strcpy(BaselineIds[0][0],BaselineId[ibase-1]);
		strcpy(BaselineIds[1][0],BaselineId[ibase-1]);
		strcpy(name,"BaselineId");len1=strlen(name);
		strcpy(type,"_char*8")   ;len2=strlen(type);
		ndim=2;dims[0]=1;dims[1]=2;
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		len3=8;
		cmp_putnc_(loc3,name,&ndim,dims,(char*)BaselineIds,dims,&status,len0,len1,len3);

		strcpy(name,"Wavelength");len1=strlen(name);
		strcpy(type,"_double")   ;len2=strlen(type);
		ndim=2;dims[0]=2;dims[1]=2;
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		cmp_putnd_(loc3,name,&ndim,dims,Wavelength[0],dims,&status,len0,len1);
		strcpy(name,"WavelengthErr");len1=strlen(name);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		cmp_putnd_(loc3,name,&ndim,dims,WavelengthErr[0],dims,&status,len0,len1);
		strcpy(name,"ChanWidth");len1=strlen(name);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		cmp_putnd_(loc3,name,&ndim,dims,ChanWidth[0],dims,&status,len0,len1);
		strcpy(name,"ChanWidthErr");len1=strlen(name);
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		cmp_putnd_(loc3,name,&ndim,dims,ChanWidthErr[0],dims,&status,len0,len1);

		strcpy(name,"FringeMod");len1=strlen(name);
		strcpy(type,"_integer") ;len2=strlen(type);
		ndim=2;dims[1]=2;dims[0]=1;
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		cmp_putni_(loc3,name,&ndim,dims,FringeMod[0],dims,&status,len0,len1);
		dat_annul_(loc3,&status,len0);

	strcpy(name,"Triple");len1=strlen(name);
	strcpy(type,"TABLE") ;len2=strlen(type);
	ndim=0;
	dat_new_(loc2,name,type,&ndim,dims,&status,len0,len1,len2);
	dat_find_(loc2,name,loc3,&status,len0,len1,len0);
		strcpy(name,"NumTriple");len1=strlen(name);
		strcpy(type,"_integer") ;len2=strlen(type);
		ndim=0;
		dat_new_(loc3,name,type,&ndim,dims,&status,len0,len1,len2);
		NumTriple=0;
		cmp_put0i_(loc3,name,&NumTriple,&status,len0,len1);
		dat_annul_(loc3,&status,len0);
		dat_annul_(loc2,&status,len0);

	/* Finish up */
	printf("\n");
	printf("\n%d records read, %d scans.",irec,iscan);
	printf("\nDate is (day month year): %d %d %d",iday,imon,iyear);
	printf("\nFilter numbers are:");
	for(j=0;j<4;j++)printf(" %d ",ifilt[j]);
	printf("\nBaseline is: %d\n",ibase);
		
	hds_close_(loc1,&status,len0);
	hds_stop_(&status);
	printf("\nHDS stopped; status = %d\n",status);

	return 0;
	}

/*----------------------------------------------------------------------------*/
int parse(void *ptr,char *ia,short iaf,short ial)
	{
	short j;
	char c;
	/* Swap bytes using XOR function */
/*
	for(j=iaf;j<=iaf+(ial-iaf)/2;j++){
		c=ia[j]^ia[ial-j+iaf];
		ia[j]=ia[j]^c;
		ia[ial-j+iaf]=ia[ial-j+iaf]^c;
		}
*/
	/* Copy memory location into requested variable */
	memcpy(ptr,ia+iaf,ial-iaf+1);
	/* Swap bytes to original order in case memory is parsed again */ 
/*
	for(j=iaf;j<=iaf+(ial-iaf)/2;j++){
		c=ia[j]^ia[ial-j+iaf];
		ia[j]=ia[j]^c;
		ia[ial-j+iaf]=ia[ial-j+iaf]^c;
		}
*/
	return 0;
	}
