
/*   This program tests the siderostat model fitting routines. */

/*   It generates a bunch of data with one siderosta model, changes */
/*   the model and tries to restore the original model with a least */
/*   squares fit. */

/*   Model parameters are as described in Matt Kelly's thesis */

/*   Written and debugged September 1, 1990.  dm */
/*   ported to C  March 2, 1997 dm */

/*     No plotting */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <math.h>

#define LATITUDE  35.0967
#define PI    3.14159365358979323
#define RAD   (PI/180.)
#define NUM_SIDS 3
/* #define R0  0.0127 */

#define R0  0.01675

/*   Forward function definitions */

void doFit (double *x, double *y, double *sig, int ndata, double *q, double *chisq  ) ;
int getData ( char *pcFile, char **pcStar, double *x, double *y, double *sig, int iTargetSid ) ;

int funcs( double *x, double *q, double *y, double *dyda, int npar) ;
int get_motor(double *thetaa, double *thetae, double *hastar, double *decstar, double *fazi,
	      double *fele, double *sazi, double *sele, double *delta_a, double *azi0,
	      double *delta_m, double *ele0) ;
int get_angles(double *azimuth, double *elevation, double *wp, double *sazi,
	       double *sele, double *delta_a, double *azi0, double *delta_m, double *ele0) ;
int get_star(double *az, double *el, double *hastar, double *decstar, double *fazi,
	     double *fele, double *sazi, double *sele, double *delta_a, double *azi0,
	     double *delta_m, double *ele0) ;
int get_norm (double *az, double *el, double *wp, double *sazi, double *sele, double *delta_a, 
	      double *azi0, double *delta_m, double *ele0 ) ;
int rotate (char *axis, double *theta, double *phi, double *in, double *out ) ;
int angle_to_vector(double *azi, double *ele, double *vec) ;
int vector_to_angle (double *azi, double *ele, double *vec) ;
int refrac (double *s, double *sr) ;
int un_refrac(double *s, double *sr) ;
int mrqmin (double *x, double *y, double *sig, int nmax, int ndata, double *a,
	    int npar, int *lista, int *mfit, double *covar, double *alpha, int nca,
	    double *chisq, double *alamda) ;
int mrqcof (double *x, double *y, double *sig, int *nmax, int *ndata, double *a,
	    int npar, int *lista, int *mfit, double *alpha, double *beta, int *nalp,
	    double *chisq ) ;
int gaussj(double *a, int *n, int *np, double *b, int m, int mp) ;
int covsrt (double *covar, int *ncvm, int *ma, int *lista, int *mfit) ;
double gaussdev (int *idum) ;
double ran1(int *idum) ;
int StringParse( char *cBuffer, int iLen, char cType, void *pvValue ) ;


/* Table of constant values */

static double c_b123 = 180.;

/****** Global variables ************/

int iVerbose = 0 ;    /* true => more output */



/* Main program */ 
int main()
{
  /*                FAZI, FELE,  SAZI,  SELE, DELTA_A,  AZI0, DELTA_M, ELE0 */
  /*   ideal model.  Start the fit using this one. */ 
  double q0[8] = { 87., 19.81, 20., 90., 0., -.44, 0., 48.628 };
  /*   Actual model.  Generate the data for the fit with this one. */
  double q[8] =  { 87., 19.81, 20., 90., 0., -.44, 0., 48.628 };
  double qBest[8][NUM_SIDS] ;
  double fBestChisq[NUM_SIDS] ;
  double dyda[16]	/* was [2][8] */;
  double ymod[2];
  double x[200] ;	/* was [2][100] */
  double y[200] ;	/* was [2][100] */
  double sig[200]	/* was [2][100] */;
  double delta[2];
  double chisq ;
  double fRetroAzi, fRetroEle, pfFeed[3] ;
  double fThetaAzi, fThetaEle ;

  float fNewAzi ;

  int ierr, i, j, ndata ;
  int iFirst, iSid ;

  char pcFile[80];
  char *pcStar[100] ;

/* ----------------------------------------------------------------------- */

  /* ask the user for the data set to fit */

  printf ( "Enter data file name: ") ;
  scanf ( "%s",pcFile ) ;

  /* Allocate memory for file names */

  for ( i = 0 ; i < 100 ; i++ )
    {
      pcStar[i] = (char *)malloc ( 8 ) ;
    }

  /* Display the initial model */

  printf ( "Initial model" ) ;
  for ( i = 0 ; i < 8 ; i++ )
    printf ( "%7.3f", q[i] ) ;
  printf ( "\n" ) ;

  /* Loop over the siderostats fitting the data */
  
  for ( iSid = 1 ; iSid < NUM_SIDS+1 ; iSid++ )
    {

      /* Get the data for this siderostat */

      ndata = getData ( pcFile, pcStar, x, y, sig, iSid ) ;
      printf ( "Data set for Sid #%2d has been read, %d data points\n", iSid, ndata ) ;

      /* Display the residuals relative to the default model */

      if ( iVerbose )
	{
	  printf ( "Residuals using default model (all angles in degrees)\n" ) ;
	  printf ( "sid  star        ha      dec     actAZI   actELE     modAZI  modELEe deltaAZI deltaELE\n" );
      
	  for (i = 0 ; i < ndata ; ++i)
	    {
	      ierr = funcs( &x[2*i], q, ymod, dyda, 8);
	      if (ierr != 0)
		printf ( "model calculation failed for scan %d!\n", i ) ;

	      delta[0] = y[2*i]   - ymod[0];
	      delta[1] = y[2*i+1] - ymod[1];
		  
	      printf ( "%3d %8s%8.2f%8.2f  %9.3f%9.3f  %9.3f%9.3f  %8.3f%8.3f\n", 
		       iSid, pcStar[i], x[2*i], x[2*i+1], y[2*i], y[2*i+1],
		       ymod[0], ymod[1], delta[0], delta[1] ) ;
	    }
	}

      /* Don't fit the data if there is not enough data points */

      if ( ndata < 5 )
	{
	  printf ( "Not enough data to fit model for siderostat # %d\n\n", iSid ) ;
	  continue ;
	}

      /*     Determine the retro-reflection angles.
             These are needed for adjusting the siderostat zero points. */

      angle_to_vector ( &q0[0], &q0[1], pfFeed ) ;
      get_angles( &fRetroAzi, &fRetroEle, pfFeed, &q[2], &q[3], &q[4], &q[5], &q[6], &q[7] ) ;
      if ( iVerbose) 
	printf ( "Initial Retro direction is azimuth = %f    ele = %f\n", fRetroAzi, fRetroEle ) ;
      
      iFirst = 1 ;
      for ( fNewAzi = 0. ; fNewAzi < 360. ; fNewAzi+=20. )
	{

	  /*  Set the trial model equal to the ideal model,
	      with the exception of the siderostat azimuth
	      which is adjusted with a grid search. */

	  for (i = 0; i < 8; ++i)
	    q[i] = q0[i];

	  q0[2] = fNewAzi ;
	  q[2]  = fNewAzi ;

	  /*     After changing the siderostat azimuth, we also have to change the
		 azimuth zero point, othersize, the starting model is too far off
		 to provide a good fit to the data. */

	  get_angles( &fThetaAzi, &fThetaEle, pfFeed, &q[2], &q[3], &q[4], &q[5], &q[6], &q[7] ) ;
	  q[5] += fThetaAzi - fRetroAzi ;
	  q[7] += fThetaEle - fRetroEle ;

	  /*  Fit the data. */

	  doFit ( x, y, sig, ndata, q, &chisq  ) ;

	  /* Display the chi square and new model parameters */

	  printf ( "Init Azi=%4d, chisq =%6.1f, Q = ", (int)fNewAzi, chisq ) ;
	  for ( j = 0 ; j < 8 ; j++ )
	    printf ( "%8.2f", q[j] ) ;
	  printf ( "\n" ) ;

	  /* If this is the best fit, save it */

	  if ( iFirst || ( chisq < fBestChisq[iSid-1] ) )
	    {
	      iFirst = 0 ;
	      fBestChisq[iSid-1] = chisq ;
	      for ( i = 0 ; i < 8 ; i++ )
		{
		  qBest[i][iSid-1] = q[i] ;
		}
	    }
	}

      /*   Using the "best" model, re-evaluate and display the residuals. */

      for (i = 0; i < 8; ++i)
	q[i] = qBest[i][iSid-1];

      doFit ( x, y, sig, ndata, q, &chisq  ) ;

      /* Display the chi square and new model parameters */

      printf ( "Init Azi=%4d, chisq =%6.1f, Q = ", (int)fNewAzi, chisq ) ;
      for ( j = 0 ; j < 8 ; j++ )
	printf ( "%8.2f", q[j] ) ;
      printf ( "\n" ) ;

      printf ( "sid  star        ha      dec     actAZI   actELE     modAZI  modELEe deltaAZI deltaELE\n" );

      for (i = 0; i < ndata; ++i)
	{
	  ierr = funcs(&x[2*i], q, ymod, dyda, 8);
	  if (ierr != 0)
	    printf ( "model evaluation failed\n" ) ;

	  delta[0] = y[2*i]   - ymod[0];
	  delta[1] = y[2*i+1] - ymod[1];
	  printf ( "%3d %8s%8.2f%8.2f  %9.3f%9.3f  %9.3f%9.3f  %8.3f%8.3f\n", 
		   iSid, pcStar[i], x[2*i], x[2*i+1], y[2*i], y[2*i+1],
		   ymod[0], ymod[1], delta[0], delta[1] ) ;
	}

      printf ( "Residual pointing error(axis angles) = %f arc seconds\n", sqrt(fBestChisq[iSid-1]) ) ;
      printf ( "Best model parameters are\n" ) ;
      for ( j = 0 ; j < 8 ; j++ )
	printf ( "%9.3f", q[j] ) ;
      printf ( "\n" ) ;
    }

  /***************************************************************/
  /* Display the fitted model parameters for all the siderostats */

  printf ( "Best model parameters are\n" ) ;
  printf ( "Latitude = %f\n", LATITUDE ) ;
  printf ( "Refractive constant = %f\n", R0 ) ;

  printf ( "sid FeedAzi  FeedEle  SidAzi   SidEle  AxisErr    AZI0   MirrorErr  ELE0  rms(arcsec)\n" ) ;
  for ( iSid = 1 ; iSid < 4 ; iSid++ )
    {
      printf ( "%d", iSid ) ;
      for ( j = 0 ; j < 8 ; j++ )
	printf ( "%9.3f", qBest[j][iSid-1] ) ;
      printf ( "%8.2f\n", sqrt(fBestChisq[iSid-1]) ) ;
    }
  return (1) ;
}

void doFit (double *x, double *y, double *sig, int ndata, double *q, double *chisq  ) 
{
  static int lista[8] = { 1, 2, 3, 4, 6, 8, 5, 7 } ;
  int ndof, nfit ;
  int npar = 8 ;
  int niter = 10;
  int i, j ;
  double alamda ;
  double covar[64], alpha[64] ;

  if ( ndata < 4 )
    {
      printf ( "doFit: not enough data to fit\n" );
      return ;
    }
  else if ( ndata < 60 )
    {
      printf ( "doFit: Not enough data, doing a 6 parameter fit\n" );
      nfit = 6 ;
    }
  else
    {
      nfit = 8 ;
    }

  ndof = 2*ndata - nfit ;
	
  alamda = -1.;
  mrqmin (x, y, sig, 100, ndata, q, npar, lista, &nfit, covar, alpha, 
		8, chisq, &alamda);

  if ( iVerbose )
    {
      printf( "iter chisq/ndf log(lambda)  Q\n" ) ;
      printf( "%3d  %6.2f  %5.1f ", 0, (*chisq)/ndof, log10(alamda) ) ;
      for ( j = 0 ; j < 8 ; j++ )
	printf ( "%9.3f", q[j] ) ;
      printf ( "\n" ) ;
    }

  for (i = 1; i <= niter; ++i)
    {
      mrqmin (x, y, sig, 100, ndata, q, npar, lista, &nfit, covar, 
	      alpha, 8, chisq, &alamda);
      if ( iVerbose )
	{
	  printf( "%3d  %6.2f  %5.1f ", i, (*chisq)/ndof, log10(alamda) ) ;
	  for ( j = 0 ; j < 8 ; j++ )
	    printf ( "%9.3f", q[j] ) ;
	  printf ( "\n" ) ;
	}
    }
  *chisq /= (float)ndof ;
}



/************************************
  getData

  open data file and read data.

  estimates the error on each 
  data point of 1 arc second

  This way the square root of the
  chi square is equal to the 
  error in arc seconds.
***************************/
int getData ( char *pcFile, char **pcStar, double *x, double *y, double *sig, int iTargetSid )
{
  FILE *pFILE ;
  char pcLine[256] ;
  double jDate ;
  double sidtime, ra, dec, azimuth, elevation, ha ;
  int ndata, i, iSid ;
  float sdev ;

  sdev = 1./3600. ;

  pFILE = fopen( pcFile, "r" ) ;
  if ( pFILE == NULL )
    {
      printf ( "Error datafile \"%s\" does not exist\n", pcFile );
      return (0) ;
    }

  ndata = 0 ;
  while( fgets ( pcLine, 256, pFILE ) != NULL )
    {
      if ( strlen ( pcLine ) == 1 )
	continue ;
      else if ( strlen ( pcLine ) < 86 )
	{
	  printf ( "input line is too short: length = %d should be 87\n", strlen(pcLine) ) ;
	  continue ;
	}

      i = 0 ;
      i += StringParse( &pcLine[ 0],  2, 'i', &iSid ) ;
      i += StringParse( &pcLine[ 3],  8, 's', pcStar[ndata] ) ;
      i += StringParse( &pcLine[12], 14, 'd', &jDate ) ;
      i += StringParse( &pcLine[27], 11, 'd', &sidtime ) ;
      i += StringParse( &pcLine[39], 11, 'd', &ra ) ;
      i += StringParse( &pcLine[51], 11, 'd', &dec ) ;
      i += StringParse( &pcLine[63], 11, 'd', &azimuth ) ;
      i += StringParse( &pcLine[75], 11, 'd', &elevation ) ;
      /*
      printf ( " row = %3d  sid = %3d  JD = %14.6f\n", ndata, iSid, jDate ) ;
      */
      if ( i )
	{
	  printf ( "read error in input data file\n" ) ;
	}
      else if ( ( ( jDate > 1. ) || ( jDate < -1.0 ) ) && ( iSid == iTargetSid ) )
	{
	  ha = sidtime - ra;
	  if (ha < -12.)
	    ha += 24 ;
	  if (ha > 12.)
	    ha += -24 ;

	  x[2*ndata  ] = ha;
	  x[2*ndata+1] = dec;
	  y[2*ndata  ] = azimuth;
	  y[2*ndata+1] = elevation;
	  sig[2*ndata] = sdev;
	  sig[2*ndata+1] = sdev;
	  ndata++ ;
	}
    }

  fclose ( pFILE ) ;
  return ndata ;
}




/***************************************************************
  funcs
   For star positions, X, and model Q, calculate
   the motor positions, Y, and derivatives with respect to the 
   model parameters, DYDA.  There are NPAR model parameters. 
***************************************************************/
int funcs( double *x, double *q, double *y, double *dyda, int npar)
{
    static double delta = .01;
    int ret_val, i__1, i__2;
    static double qtry[8];
    static int i, j;
    static double y0[2], y1[2];

    /* Parameter adjustments */
    dyda -= 3;
    --y;
    --q;
    --x;

    /* Function Body */
    if (npar != 8)
      {
	printf ( "funcs: incorrect initialization\n" );
	exit ( 1 ) ;
      }
    i__1 = npar;
    for (i = 1; i <= i__1; ++i)
      {
	qtry[i - 1] = q[i];
      }
    get_motor(&y[1], &y[2], &x[1], &x[2], qtry, &qtry[1], &qtry[2], &qtry[3]
	      , &qtry[4], &qtry[5], &qtry[6], &qtry[7]);
    i__1 = npar;
    for (i = 1; i <= i__1; ++i)
      {
	i__2 = npar;
	for (j = 1; j <= i__2; ++j)
	  {
	    qtry[j - 1] = q[j];
	  }
	qtry[i - 1] = q[i] - delta;
	get_motor(y0, &y0[1], &x[1], &x[2], qtry, &qtry[1], &qtry[2], &qtry[3],
		  &qtry[4], &qtry[5], &qtry[6], &qtry[7]);
	qtry[i - 1] = q[i] + delta;
	get_motor(y1, &y1[1], &x[1], &x[2], qtry, &qtry[1], &qtry[2], &qtry[3],
		  &qtry[4], &qtry[5], &qtry[6], &qtry[7]);
	for (j = 1; j <= 2; ++j)
	  {
	    dyda[j + (i << 1)] = (y1[j - 1] - y0[j - 1]) / (delta * 2.);
	  }
      }
    ret_val = 0;
    return ret_val;
}

/***************************************************************
  get_motor

  Calculates the rotation of the siderostat axes. 

  HASTAR    hour angle of star (in hours) 
  DECSTAR   declination of star 

  THETAA   angle of  azimuth  axis (output) 
  THETAE   angle of elevation axis (output) 

  Fazi      feed beam azimuth    (0=north,   90 = west) 
  Fele      feed beam elevation  (0=horizon, 90 = vertical)

  See comments in GET_ANGLES for definitions of other model parameters. 

***************************************************************/
int get_motor(double *thetaa, double *thetae, double *hastar, double *decstar, double *fazi,
	      double *fele, double *sazi, double *sele, double *delta_a, double *azi0,
	      double *delta_m, double *ele0)
{
    double d__1;
    static double norm, f[3];
    static int i;
    static double s[3], ap[3];

    /*  Calculate the vector toward the star, including atmospheric
	refraction. */

    angle_to_vector(fazi, fele, f);
    d__1 = *hastar * -15.;
    angle_to_vector(&d__1, decstar, s);
    d__1 = 90. - LATITUDE;
    rotate ("Y ", &c_b123, &d__1, s, s);
    refrac(s, s);

    /*  Calculate mirror normal, Ap, in local coordinates. */

    norm = 0.;
    for (i = 1; i <= 3; ++i)
      {
	ap[i - 1] = s[i - 1] + f[i - 1];
	norm += ap[i - 1] * ap[i - 1];
      }
    norm = sqrt(norm);
    for (i = 1; i <= 3; ++i)
      {
	ap[i - 1] /= norm;
      }

    get_angles(thetaa, thetae, ap, sazi, sele, delta_a, azi0, delta_m, ele0);
    return 0;
}

/***************************************************************
  get_angles
   Calculates the rotation of the siderostat axes.
   All angles are in degrees, except star hour angle.

  AZIMUTH   angle of  azimuth  axis (output)
  ELEVATION angle of elevation axis (output)/

  AP is the mirror normal in the local coordinate system.

  Siderostat model parameters.

  SAZI      siderostat azimuth   (0=north,   90 = west)
  SELE      siderostat elevation (0=horizon, 90 = vertical)
  DELTA_A   error in angle between axes (0=> axes are orthogonal)
  DELTA_M   tilt of mirror in its cell (0=> mirror normal perp to axis) 

  AZI0      zero point on azimuth axis
  ELE0      zero point on elevation axis 

  With the siderostat axis roughly vertical, The motors are at
  azi=180, ele=90 when the normal to the mirror is 
  pointing north and is normal to the azimuth axis.

   See Matt Kelly's thesis, appendix A for a description of this
   algorithm.
***************************************************************/
int get_angles(double *azimuth, double *elevation, double *wp, double *sazi,
	       double *sele, double *delta_a, double *azi0, double *delta_m, double *ele0)
{
  double d__1, d__2;
  static double a, b, costa, coste, sinta, sinte, ap[3] ;

  /* Parameter adjustments */
  --wp;

  /* Function Body */

  d__1 = *sele - 90.;
  d__2 = *sazi - 90.;
  rotate ("ZT", &d__1, &d__2, &wp[1], ap );
  coste = (ap[2] + sin(*delta_m * RAD) * sin(*delta_a * RAD)) /
    (cos(*delta_m * RAD) * cos(*delta_a * RAD));
  sinte = sqrt(1. - coste * coste);
  a = cos(*delta_a * RAD) * sin(*delta_m * RAD) + sin(*delta_a * RAD) 
    * cos(*delta_m * RAD) * coste;
  b = -cos(*delta_m * RAD) * sinte;
  sinta = a * ap[0] + b * ap[1];
  costa = b * ap[0] - a * ap[1];
  *elevation = atan2(sinte, coste)/RAD - *ele0;
  *azimuth =   atan2(sinta, costa)/RAD - *azi0;
  if (*elevation < -180.)
    {
      *elevation += 360.;
    }
  if (*elevation > 180.)
    {
      *elevation += -360.;
    }
    if (*azimuth < -180.)
      {
	*azimuth += 360.;
      }
    if (*azimuth > 180.)
      {
	*azimuth += -360.;
      }
    return 0;
}

/* ----------------------------------------------------------------------- */
/*  Calculate star position, HASTAR, DECSTAR, given the model */
/*  and motor angles. */
int get_star(double *az, double *el, double *hastar, double *decstar, double *fazi,
	     double *fele, double *sazi, double *sele, double *delta_a, double *azi0,
	     double *delta_m, double *ele0)
{
    double d__1;
    static double norm;
    static int i;
    static double s[3], fdotm;
    static double wf[3], wp[3];

    /*   Get the mirror normal */

    get_norm(az, el, wp, sazi, sele, delta_a, azi0, delta_m, ele0);

    /*   Convert mirror normal star direction. */

    angle_to_vector(fazi, fele, wf);
    fdotm = wp[0] * wf[0] + wp[1] * wf[1] + wp[2] * wf[2];
    norm = 0.;
    for (i = 1; i <= 3; ++i)
      {
	s[i - 1] = fdotm * 2. * wp[i - 1] - wf[i - 1];
	norm += s[i - 1] * s[i - 1];
      }
    if ((d__1 = norm - 1., abs(d__1)) > 1e-5)
      {
	printf ( "get_star: unit vector has length %f\n", norm ) ;
      }

    /*  Remove atmospheric refraction. */

    un_refrac(s, s);

    /*  Convert alt and azi of star to hour angle and declination. */

    d__1 = 90. - LATITUDE ;
    rotate ("YT", &c_b123, &d__1, s, s );
    vector_to_angle(hastar, decstar, s);
    *hastar = -(*hastar) / 15.;
    return 0;
}

/************************************************************
  get star position, given motor angles.
************************************************************/
int get_norm (double *az, double *el, double *wp, double *sazi, double *sele, double *delta_a, 
	      double *azi0, double *delta_m, double *ele0 )
{
  double d__1, d__2;
  static double ap[3], ep[3], mp[3];
  static double cel, caz, sel, saz;

  /* Parameter adjustments */
  --wp;

  /* Function Body */

  caz = cos((*az + *azi0) * RAD);
  saz = sin((*az + *azi0) * RAD);
  cel = cos((*el + *ele0) * RAD);
  sel = sin((*el + *ele0) * RAD);

  mp[0] = 0.;
  mp[1] = 1.;
  mp[2] = 0.;
  d__1 = *el + *ele0;
  rotate ("Z ", delta_m, &d__1, mp, ep );
  d__1 = *delta_a + 90.;
  d__2 = *az + *azi0;
  rotate ("Z ", &d__1, &d__2, ep, ap );
  d__1 = *sele - 90.;
  d__2 = *sazi - 90.;
  rotate ("Z ", &d__1, &d__2, ap, &wp[1] );
  return 0;
}

/************************************************************
  Performs a coordinate transformation between two reference frames.

  initial frame unit vectors  x0, y0, z0 
  final   frame unit vectors  x1, y1, z1 

  Description of transformation.

         if        rotate by PHI             rotate by THETA
      AXIS is          about        then          about 
       X                 X                          Y 
       Y                 Y                          Z 
       Z                 Z                          X 

  If the axis designation is followed by an I or a T, then the
  transformstion is the inverse (=transpose).  That is performed
  in the other order and with the opposite sign. 

  The angles THETA and PHI are in degrees.
  Vectors IN and OUT can be the same.
****************************************/
int rotate (char *axis, double *theta, double *phi, double *in, double *out )
{
  static double temp[3];
  static int i, j, i1, i2, i3;
  static double  matrix[9] ; /* was [3][3] */

  /* Parameter adjustments */
  --out;
  --in;

  /* Function Body */

  if (*axis == 'Z')
    {
      i1 = 1;
      i2 = 2;
      i3 = 3;
    }
  else if (*axis == 'X')
    {
      i1 = 2;
      i2 = 3;
      i3 = 1;
    }
  else if (*axis == 'Y')
    {
      i1 = 3;
      i2 = 1;
      i3 = 2;
    }
  else
    {
      printf ( "rotate: bad value of axis, \"%c\"\n", *axis ) ;
      exit(1) ;
    }

  /*           ROW COL */
  matrix[i1 + i1 * 3 - 4] =  cos(*phi * RAD);
  matrix[i1 + i2 * 3 - 4] = -sin(*phi * RAD) * cos(*theta * RAD);
  matrix[i1 + i3 * 3 - 4] =  sin(*phi * RAD) * sin(*theta * RAD);
  matrix[i2 + i1 * 3 - 4] =  sin(*phi * RAD);
  matrix[i2 + i2 * 3 - 4] =  cos(*phi * RAD) * cos(*theta * RAD);
  matrix[i2 + i3 * 3 - 4] = -cos(*phi * RAD) * sin(*theta * RAD);
  matrix[i3 + i1 * 3 - 4] =  0.;
  matrix[i3 + i2 * 3 - 4] =  sin(*theta * RAD);
  matrix[i3 + i3 * 3 - 4] =  cos(*theta * RAD);

  for (i = 1; i <= 3; ++i)
    {
      temp[i - 1] = in[i];
      out[i] = 0.;
    }
    if (axis[1] == 'I' || axis[1] == 'T')
      {
	for (i = 1; i <= 3; ++i)
	  {
	    for (j = 1; j <= 3; ++j)
	      {
		out[i] += matrix[j + i * 3 - 4] * temp[j - 1];
	      }
	  }
      }
    else
      {
	for (i = 1; i <= 3; ++i)
	  {
	    for (j = 1; j <= 3; ++j)
	      {
		out[i] += matrix[i + j * 3 - 4] * temp[j - 1];
	      }
	  }
      }
    return 0;
}


/********************
  angle_to_vector

  converts an elevation and azimuth into a unit
  vector pointing in that direction

 inputs:
  ELE  = elevation.  Angle 'above' the X-Y plane.
  AZI  = azimuth.    Angle in the X-Y plane, measured from the X-axis, 
                     counterclockwise toward the Y-axis.
 output
  VEC(1..3) = output vector
  ********************/
int angle_to_vector(double *azi, double *ele, double *vec)
{

  /* Parameter adjustments */
  --vec;

  vec[1] = cos(RAD * *ele) * cos(RAD * *azi);
  vec[2] = cos(RAD * *ele) * sin(RAD * *azi);
  vec[3] = sin(RAD * *ele);
  return 0;
}


/*  This function is the inverse of ANGLE_TO_VECTOR.  Given VEC, it
  returns ELE and AZI. */

int vector_to_angle (double *azi, double *ele, double *vec)
{
  static double cel ;

  /* Parameter adjustments */
  --vec;

  /* Function Body */

  cel = sqrt(vec[1] * vec[1] + vec[2] * vec[2]);
  *ele = atan2(vec[3], cel) / RAD ;
  *azi = atan2(vec[2], vec[1]) / RAD ;
  return 0;
}

/***********************************************
  refrac
   Correct a star vector for atmospheric refraction.  The input 
   unrefracted vector is S, the output refracted vector is SR.
   The vector must be in 'local' coordinates.
   (X, Y, Z  = north, west, vertical).
   **********************************************/
int refrac (double *s, double *sr)
{
  static double ele, azi;

  /* Parameter adjustments */
  --sr;
  --s;

  /* Function Body */

  vector_to_angle(&azi, &ele, &s[1]);
  ele += R0 / tan(ele * RAD);
  angle_to_vector(&azi, &ele, &sr[1]);
  return 0;
}
/***********************************************
   unrefrac
   Remove atmospheric refraction from a star vector.  The input
   refracted vector is Sr, the output unrefracted vector is S. 
   The vector must be in 'local' coordinates; 
   (X, Y, Z  = north, west, vertical). 
   **********************************************/
int un_refrac(double *s, double *sr)
{
  static double ele, azi;

  /* Parameter adjustments */
  --sr;
  --s;

  /* Function Body */

  vector_to_angle(&azi, &ele, &sr[1]);
  ele -= R0 / tan(ele * RAD);
  angle_to_vector(&azi, &ele, &s[1]);
  return 0;
}


/***********************************************************************
  mrqmin

  Levenberg-Marquardt method, attempting to reduce the value of chi
  square of a fit between a set of NDATA vectors X(i), Y(i) with
  individual standard deviations SIG(i) and a non-linear function
  dependent on NPAR coefficients A.  There are NX independent variables 
  and NY dependent variables in each data point.

  The array LISTA is a permutation
  of the parameters A such that the first MFIT elements correspond 
  to values actually being adjusted; the remaining parameters are held 
  fixed at their input values.

  The program returns current best fit
  values for the parameters and chi square CHISQ.  The arrays
  COVAR(NCA,NCA), ALPHA(NCA,NCA) with dimensions NCA>MFIT are used as
  working space during most itterations.  Supply a subroutine
  FUNCS(X,A,YFIT,DYDA,NPAR), that evaluates the fitting function YFIT, 
  and its derivatives DYDA with respect to the fitting parameters A
  at X.  On the first call provide an initial guess for the parameters 
  A, and set ALAMDA<0 for initialization.  If a step fails, ALAMDA 
  grows by a factor of ten.

  MRQMIN must be called repeatedly until 
  convergence is achieved.  A final call with ALAMDA=0 will calculate 
  COVAR and ALPHA, the covariance and curvature matrices. 

  Code taken from Numerical Recepies and modified to allow X and Y 
  to be vectors. 

  perhaps, the largest number of fit parameters = 20.
***********************************************************************/

int mrqmin (double *x, double *y, double *sig, int nmax, int ndata, double *a,
	    int npar, int *lista, int *mfit, double *covar, double *alpha, int nca,
	    double *chisq, double *alamda)
{
  static double beta[20];
  static int ihit;
  static double atry[20];
  static int j, k;
  static double da[20];
  static int kk;
  static double ochisq;

  /* System generated locals */
  int covar_dim1, covar_offset, alpha_dim1, alpha_offset, i__1, i__2;

  /* Parameter adjustments */
  alpha_dim1 = nca;
  alpha_offset = alpha_dim1 + 1;
  alpha -= alpha_offset;
  covar_dim1 = nca;
  covar_offset = covar_dim1 + 1;
  covar -= covar_offset;
  --lista;
  --a;
  sig -= 3;
  y -= 3;
  x -= 3;

  /* Function Body */


  /*   Initialize */

    if (*alamda < 0.)
      {

	/*   Check that LISTA is a valid permutation. */

	kk = *mfit + 1;
	i__1 = npar;
	for (j = 1; j <= i__1; ++j)
	  {
	    ihit = 0;
	    i__2 = *mfit;
	    for (k = 1; k <= i__2; ++k)
	      {
		if (lista[k] == j)
		  ++ihit;
	      }
	    if (ihit == 0)
	      {
		lista[kk] = j;
		++kk;
	      }
	    else if (ihit > 1)
	      {
		printf ( "mrqmin: Improper permutation in LISTA" );
		exit (1) ;
	      }
	  }
	if (kk != npar + 1)
	  {
	    printf ( "mrqmin: Improper permutation in LISTA" );
	    exit (1) ;
	  }

	*alamda = .001 ;
	mrqcof (&x[3], &y[3], &sig[3], &nmax, &ndata, &a[1], npar, &lista[1], 
		mfit, &alpha[alpha_offset], beta, &nca, chisq );
	ochisq = *chisq;
	i__1 = npar;
	for (j = 1; j <= i__1; ++j)
	  {
	    atry[j - 1] = a[j];
	  }
    }

    /*   Alter linearized fitting matrix by augmenting diagonal elements. */

    i__1 = *mfit;
    for (j = 1; j <= i__1; ++j)
      {
	i__2 = *mfit;
	for (k = 1; k <= i__2; ++k)
	  {
	    covar[j + k * covar_dim1] = alpha[j + k * alpha_dim1];
	  }
	covar[j + j * covar_dim1] = alpha[j + j * alpha_dim1] * (*alamda + 1.);
	da[j - 1] = beta[j - 1];
      }

    /*   Matrix solution. */

    gaussj (&covar[covar_offset], mfit, &nca, da, 1, 1);

    /*   Evaluate the covariance matrix */

    if (*alamda == 0.)
      {
	covsrt(&covar[covar_offset], &nca, &npar, &lista[1], mfit);
	return 0;
      }
    else
      {

	/*     Did this step decrease the chi square? */

	i__1 = *mfit;
	for (j = 1; j <= i__1; ++j)
	  {
	    atry[lista[j] - 1] += da[j - 1];
	  }
	mrqcof (&x[3], &y[3], &sig[3], &nmax, &ndata, atry, npar, &lista[1], 
		mfit, &covar[covar_offset], da, &nca, chisq );
	if (*chisq < ochisq)
	  {

	    /*  The step was in the right direction, use this solution and run with
		a smaller value of ALAMDA next time. */

	    *alamda *= .1f;
	    ochisq = *chisq;
	    i__1 = *mfit;
	    for (j = 1; j <= i__1; ++j)
	      {
		i__2 = *mfit;
		for (k = 1; k <= i__2; ++k)
		  {
		    alpha[j + k * alpha_dim1] = covar[j + k * covar_dim1];
		  }
		beta[j - 1] = da[j - 1];
		a[lista[j]] = atry[lista[j] - 1];
	      }
	  }
	else
	  {

	    /*   The step was in the wrong direction, do not use this solution and
		 run with a larger value of ALAMDA next time. */

	    *alamda *= 10.;
	    *chisq = ochisq;
	  }
      }
    return 0;
}


/**************************
  mrqcof
  Evaluates the linearized fitting matrix, ALPHA and vector BETA.
  *************************/
int mrqcof (double *x, double *y, double *sig, int *nmax, int *ndata, double *a,
	    int npar, int *lista, int *mfit, double *alpha, double *beta, int *nalp,
	    double *chisq )
{
    int alpha_dim1, alpha_offset, i__1, i__2, i__3;

    static double dyda[40]	/* was [2][20] */;
    static int ierr;
    static double ymod[2], sig2i;
    static int i, j, k, l;
    static double dy, wt;

    /* Parameter adjustments */
    --beta;
    alpha_dim1 = *nalp;
    alpha_offset = alpha_dim1 + 1;
    alpha -= alpha_offset;
    --lista;
    --a;
    sig -= 3;
    y -= 3;
    x -= 3;

    /* Function Body */

    i__1 = *mfit;
    for (j = 1; j <= i__1; ++j)
      {
	i__2 = j;
	for (k = 1; k <= i__2; ++k)
	  {
	    alpha[j + k * alpha_dim1] = 0.;
	  }
	beta[j] = 0.;
      }
    *chisq = 0.;

/*   Loop over the data. */

    i__1 = *ndata;
    for (i = 1; i <= i__1; ++i)
      {
	ierr = (*funcs)(&x[(i << 1) + 1], &a[1], ymod, dyda, npar);
	if (ierr != 0)
	  {
	    printf ("mrqcof: function call failed.  Error code %d\n", ierr );
	  }
	for (l = 1; l <= 2; ++l)
	  {
	    dy = y[l + (i << 1)] - ymod[l - 1];
	    i__2 = *mfit;
	    for (j = 1; j <= i__2; ++j)
	      {
		sig2i = 1. / (sig[l + (i << 1)] * sig[l + (i << 1)]);
		wt = dyda[l + (lista[j] << 1) - 3] * sig2i;
		i__3 = j;
		for (k = 1; k <= i__3; ++k)
		  {
		    alpha[j + k * alpha_dim1] += wt * dyda[l + (lista[k] << 1) - 3];
		  }
		beta[j] += dy * wt;
	      }
	    *chisq += dy * dy * sig2i;
	  }
      }

    /*   Fill in the symmetric side. */

    i__1 = *mfit;
    for (j = 2; j <= i__1; ++j)
      {
	i__2 = j - 1;
	for (k = 1; k <= i__2; ++k)
	  {
	    alpha[k + j * alpha_dim1] = alpha[j + k * alpha_dim1];
	  }
      }
    return 0;
}



/*******************************************************************
  solve a linear set of equations using gauss-jordan elimination 
  ******************************************************************/
int gaussj(double *a, int *n, int *np, double *b, int m, int mp)
{
    static int icol, ipiv[50], irow, i, j, k, l, indxc[50], indxr[50], ll;
    static double pivinv, big, dum;
    /* System generated locals */
    int a_dim1, a_offset, b_dim1, b_offset, i__1, i__2, i__3;
    double d__1;

/*  Solve the matrix equation */

/*        A(i,j) * x(j,k) = B(i,k) */

/*  and replace B with x */

    /* Parameter adjustments */
    b_dim1 = *np;
    b_offset = b_dim1 + 1;
    b -= b_offset;
    a_dim1 = *np;
    a_offset = a_dim1 + 1;
    a -= a_offset;

    /* Function Body */
    i__1 = *n;
    for (j = 1; j <= i__1; ++j)
	ipiv[j - 1] = 0;
    i__1 = *n;
    for (i = 1; i <= i__1; ++i)
      {
	big = 0.;
	i__2 = *n;
	for (j = 1; j <= i__2; ++j)
	  {
	    if (ipiv[j - 1] != 1)
	      {
		i__3 = *n;
		for (k = 1; k <= i__3; ++k)
		  {
		    if (ipiv[k - 1] == 0)
		      {
			if ((d__1 = a[j + k * a_dim1], abs(d__1)) >= big)
			  {
			    big = (d__1 = a[j + k * a_dim1], abs(d__1));
			    irow = j;
			    icol = k;
			  }
		      }
		    else if (ipiv[k - 1] > 1)
		      {
			printf ( "gaussj:  Singular matrix:1\n" ) ;
			exit (1) ;
		      }
		  }
	      }
	  }
	++ipiv[icol - 1];
	if (irow != icol)
	  {
	    i__2 = *n;
	    for (l = 1; l <= i__2; ++l)
	      {
		dum = a[irow + l * a_dim1];
		a[irow + l * a_dim1] = a[icol + l * a_dim1];
		a[icol + l * a_dim1] = dum;
	      }
	    i__2 = m;
	    for (l = 1; l <= i__2; ++l)
	      {
		dum = b[irow + l * b_dim1];
		b[irow + l * b_dim1] = b[icol + l * b_dim1];
		b[icol + l * b_dim1] = dum;
	      }
	  }
	indxr[i - 1] = irow;
	indxc[i - 1] = icol;
	if (a[icol + icol * a_dim1] == 0.)
	  {
	    printf ( "gaussj: Singular matrix:2\n" ) ;
	    exit (1) ;
	  }
	pivinv = 1. / a[icol + icol * a_dim1];
	a[icol + icol * a_dim1] = 1.;
	i__2 = *n;
	for (l = 1; l <= i__2; ++l)
	  {
	    a[icol + l * a_dim1] *= pivinv;
	  }
	i__2 = m;
	for (l = 1; l <= i__2; ++l)
	  {
	    b[icol + l * b_dim1] *= pivinv;
	  }
	i__2 = *n;
	for (ll = 1; ll <= i__2; ++ll)
	  {
	    if (ll != icol)
	      {
		dum = a[ll + icol * a_dim1];
		a[ll + icol * a_dim1] = 0.;
		i__3 = *n;
		for (l = 1; l <= i__3; ++l)
		  {
		    a[ll + l * a_dim1] -= a[icol + l * a_dim1] * dum;
		  }
		i__3 = m;
		for (l = 1; l <= i__3; ++l)
		  {
		    b[ll + l * b_dim1] -= b[icol + l * b_dim1] * dum;
		  }
	      }
	  }
      }
    for (l = *n; l >= 1; --l)
      {
	if (indxr[l - 1] != indxc[l - 1])
	  {
	    i__1 = *n;
	    for (k = 1; k <= i__1; ++k)
	      {
		dum = a[k + indxr[l - 1] * a_dim1];
		a[k + indxr[l - 1] * a_dim1] = a[k + indxc[l - 1] * a_dim1];
		a[k + indxc[l - 1] * a_dim1] = dum;
	      }
	  }
      }
    return 0;
}


/********************************************
  Sort the covariance matrix 
 *******************************************/
int covsrt (double *covar, int *ncvm, int *ma, int *lista, int *mfit)
{
    int covar_dim1, covar_offset, i__1, i__2;
    static double swap;
    static int i, j;

    /* Parameter adjustments */

    lista-- ;
    covar_dim1 = *ncvm;
    covar_offset = covar_dim1 + 1;
    covar -= covar_offset;

    /* Function Body */

    i__1 = *ma - 1;
    for (j = 1; j <= i__1; ++j)
      {
	i__2 = *ma;
	for (i = j + 1; i <= i__2; ++i) 
	  {
	    covar[i + j * covar_dim1] = 0. ;
	  }
      }
    i__1 = *mfit - 1;
    for (i = 1; i <= i__1; ++i) 
      {
	i__2 = *mfit;
	for (j = i + 1; j <= i__2; ++j) 
	  {
	    if (lista[j] > lista[i]) 
	      {
		covar[lista[j] + lista[i] * covar_dim1] = covar[i + j * 
							       covar_dim1];
	      }
	    else 
	      {
		covar[lista[i] + lista[j] * covar_dim1] = covar[i + j * 
							       covar_dim1];
	      }
	  }
      }

    swap = covar[covar_dim1 + 1];
    i__1 = *ma;
    for (j = 1; j <= i__1; ++j) 
      {
	covar[j * covar_dim1 + 1] = covar[j + j * covar_dim1];
	covar[j + j * covar_dim1] = 0.;
      }
    covar[lista[1] + lista[1] * covar_dim1] = swap;
    i__1 = *mfit;
    for (j = 2; j <= i__1; ++j) 
      {
	covar[lista[j] + lista[j] * covar_dim1] = covar[j * covar_dim1 + 1];
      }
    i__1 = *ma;
    for (j = 2; j <= i__1; ++j) 
      {
	i__2 = j - 1;
	for (i = 1; i <= i__2; ++i) 
	  {
	    covar[i + j * covar_dim1] = covar[j + i * covar_dim1];
	  }
      }
    return 0;
}

/***********************
  returns a gaussian deviate
******************************/

double gaussdev (int *idum)
{
  static int iset = 0;
  static double gset, r, v1, v2, fac;
  double ret_val;

  if (iset == 0)
    {
      r = 2. ;
      while ( ( r > 1.) || ( r == 0. )  )
	{
	  v1 = 2. * ran1(idum) - 1. ;
	  v2 = 2. * ran1(idum) - 1. ;
	  r = v1 * v1 + v2 * v2 ;
	}

      fac = sqrt( - 2. * log(r) / r ) ;
      gset = v1 * fac;
      ret_val = v2 * fac;
      iset = 1;
    }
  else
    {
      ret_val = gset;
      iset = 0;
    }
  return ret_val;
}

/*************************************************
  returns a random number on the interval 0 - 1
**************************************************/
double ran1(int *idum)
{
    double ret_val;
    static int j;
    static double r[97];
    static int ix1, ix2, ix3;
    static int iFirst = 1;

/*  Return a uniform random number. */

    if (*idum < 0 || iFirst)
      {
	iFirst = 0 ;
	ix1 = (54773 - *idum) % 259200;
	ix1 = (ix1 * 7141 + 54773) % 259200;
	ix2 = ix1 % 134456;
	ix1 = (ix1 * 7141 + 54773) % 259200;
	ix3 = ix1 % 243000;
	for (j = 1; j <= 97; ++j)
	  {
	    ix1 = (ix1 * 7141 + 54773) % 259200;
	    ix2 = (ix2 * 8121 + 28411) % 134456;
	    r[j - 1] = ( (float)ix1 + (float)ix2 * 7.4373773e-6) * 3.8580247e-6;
	  }
	*idum = 1;
      }
    ix1 = (ix1 * 7141 + 54773) % 259200;
    ix2 = (ix2 * 8121 + 28411) % 134456;
    ix3 = (ix3 * 4561 + 51349) % 243000;
    j = ix3 * 97 / 243000 + 1;
    if (j > 97 || j < 1)
      {
	printf ( "ran1:  error generating random number!\n" );
	exit (1) ;
      }
    ret_val = r[j - 1];
    r[j - 1] = ( (float)ix1 + (float)ix2 * 7.4373773e-6) * 3.8580247e-6;
    return ret_val;
}

/********************************************************************
  This function takes a number, entered as a string, converts 
  it into a numerical value and returns it as either a float or integer
  value.


inputs:

    char *cBuffer     pointer to the start of the character string
    int iLen          length of input string
    char cType        type of value to return.  valid values are
                       i for integer output
                       f for  float   output
                       d for  double  output
		       s for string  output
    void *pvValue     pointer to variable of type cType for output data

outputs:

    *pvValue          The value to be returned

The function evaluates to 1 if there is an error, 0 otherwise.

 ********************************************************************/
int StringParse( char *cBuffer, int iLen, char cType, void *pvValue )
{
  int iCount, i ;
  int iDot ;
  int iSign ;
  long int iValue ;
  long int iDivisor ;
  double dDivisor, dValue ;
  char *pChar ;

  /* Take care of character strings up front */

  if ( cType == 's' )
    {
      pChar = (char *)pvValue ;
      for ( i = 0 ; i < iLen ; i++ )
	*(pChar++) = cBuffer[i] ;

      *(pChar) = '\000' ;
      return (0) ;
    }

  /* Check for bad data type */

  if ( (cType != 'i') && (cType != 'f') && (cType != 'd') )
    {
      printf ( "invalid data type \"%d\"\n", cType ) ;
      return (1) ;
    }

  /*  Strip off trailing blanks */

  while ( ( cBuffer[iLen-1] == ' ' ) && ( iLen > 0 ) )
      iLen-- ;

  /*  Strip off leading blanks */

  iCount = 0 ;
  while ( (cBuffer[iCount]==' ') && (iCount < iLen) )
      iCount++ ;

  /*  a blank string has a value of zero */

  if ( iLen == 0 )
  {
      iValue = 0 ;
      if ( cType=='i' )
	  *(int *)pvValue = iValue ;
      else if ( cType=='f' )
	  *(float *)pvValue = (float)iValue ;
      else if ( cType=='d' )
	  *(double *)pvValue = (double)iValue ;
  }

  /* check for a leading sign */

  iSign = 1 ;
  if ( cBuffer[iCount]=='-' )
    {
      iSign = -1 ;
      iCount++;
    }
  else if ( cBuffer[iCount]=='+')
    {
      iCount++ ;
    }
      
  /* parse the remaining string */

  iDot = 0 ;
  iValue = 0 ;
  iDivisor = 1 ;
  dValue = 0. ;
  dDivisor = 1. ;
  for (  ; iCount < iLen; iCount++ )
    {
      i = (int)cBuffer[iCount] ;
      if ( (i>47) && (i<58) )
	{
	  if ( cType == 'i' )
	    {
	      iValue = 10*iValue + i-48 ;
	    }
	  else if ( iDot )
	    {
	      dDivisor *= 0.1 ;
	      dValue += (i-48)*dDivisor ;
	    }
	  else
	    {
	      dValue = 10.*dValue + (i-48) ;
	    }
	}
      else if ( (i==46) && (iDot==0) ) 
	{
	  iDot = 1 ;

	  if ( cType == 'i' )
	    {
	      printf ( "Decimal not allowed in integer\n" ) ;
	      for (i=0; i<iLen; i++ )
		{
		  printf("%c", cBuffer[i]);
		}
	      printf ("\n                           ");
	      for (i=0; i<iCount ; i++ )
		{
		  printf (" ");
		}
	      printf("^\n");
	      return 1 ;
	    }
	}
      else
	{
	  printf ("\nInvalid character in number");
	  for (i=0; i<iLen; i++ )
	    {
	      printf("%c", cBuffer[i]);
	    }
	  printf ("\n                           ");
	  for (i=0; i<iCount ; i++ )
	    {
	      printf (" ");
	    }
	  printf("^\n");
	  return 1 ;
	}
    }
  /*
  printf ( "parseString:  value = %d,  divisor = %d  dot = %d\n", iValue, iDivisor, iDot ) ;
  */

  if ( cType=='i' )
    {
      *(int *)pvValue = iSign*iValue ;
    }
  else if ( cType=='f' )
    {
      *(float *)pvValue = (float)(iSign*dValue) ;
    }
  else if ( cType=='d' )
    {
      *(double *)pvValue = (double)iSign * dValue ;
    }

  return 0 ;
}
