/*******************************************************************************
chameleon.c

Source:
-------
Dr. Christian A. Hummel
European Southern Observatory
Karl-Schwarzschild-Str. 2
85748 Garching
Germany

Description:
------------
Data reduction and astrometry functions.

Modification history:
---------------------
18-Mar-1994  C.A.Hummel, file created
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "export.h"
#include "novas.h"
#include "pos_prop.h"

#define PI 3.141592653589793
#define RAD(x) ((x)/360.*2.*PI)
#define NEG(x) ((x) > 0 ? -(x) : (x))
#define MIN(x,y) ((x) > (y) ? (y) : (x))
#define MAX(x,y) ((x) > (y) ? (x) : (y))

/* Global variables */

IDL_STRING *azCatalogPath,*azAtmosphPath,*azLibraryPath;
char azHIC[250],azHIP[250],azSAO[250],azFKV[250],azBSC[250];
char azBAT[250],azSKY[250],azFLN[250],azSPL[250],azJPL[250];

/* Function prototypes */

int init_oyster(int argc, char *argp[]);
int flagpointdata(int argc, char *argp[]);
int flaginchdata(int argc, char *argp[]);
int unwrapphase(int argc, char *argp[]);
int unwraptriple(int argc, char *argp[]);
int catalog(int argc, char *argp[]);
int crossindex(int argc, char *argp[]);
int jacoby(int argc, char *argp[]);
int specparse(int argc, char *argp[]);
int tpstar(int argc, char *argp[]);
int apstar(int argc, char *argp[]);
int tpplanet(int argc, char *argp[]);
int applanet(int argc, char *argp[]);
int ut12gst(int argc, char *argp[]);
int polarmotion(int argc, char *argp[]);
int siderostat(int argc, char *argp[]);
int sid_funcs( double *x, double *q, double *y, double *dyda, int npar);
int get_star(double *az, double *el, double *hastar, double *decstar, 
	     double *fazi, double *fele, double *sazi, double *sele, 
	     double *delta_a, double *azi0, double *delta_m, double *ele0);

int spec_parse(char *azSpectrum,
	float *pfType1,float *pfType2,float *pfClass1,float *pfClass2);
int countflag(float fValue, float fFlag);
int IDL_ChartrStore(unsigned char *dest,unsigned char *source,int numchar);

/*----------------------------------------------------------------------------*/
int init_oyster(int argc, char *argp[]){
	/*
	Make known to the C functions the location of the directories
	for catalogs and ephemerides.
	*/

        int iLen;
	double jdbeg,jdend;

	azCatalogPath=((IDL_STRING**)argp)[0];
	azAtmosphPath=((IDL_STRING**)argp)[1];
	azLibraryPath=((IDL_STRING**)argp)[2];

	strcat(strcpy(azHIC,azCatalogPath[0].s),"hipparcos/input/hipp.dat");
	strcat(strcpy(azHIP,azCatalogPath[0].s),"hipparcos/hip_main.dat");
	strcat(strcpy(azSAO,azCatalogPath[0].s),"sao/sao.dat");
	strcat(strcpy(azFKV,azCatalogPath[0].s),"fk5/fk5.dat");
	strcat(strcpy(azBSC,azCatalogPath[0].s),"bsc4/bsc4.dat");
	strcat(strcpy(azBAT,azCatalogPath[0].s),"batten/sb9.dat");
	strcat(strcpy(azSKY,azCatalogPath[0].s),"skycat/skycat.dat");
	strcat(strcpy(azFLN,azCatalogPath[0].s),"findlist/findlist.dat");
	strcat(strcpy(azSPL,azAtmosphPath[0].s),"jacoby/jacoby.dat");
	strcat(strcpy(azJPL,azLibraryPath[0].s),"ephemeris.200");

        iLen=strlen(azJPL);

/* 	Ephemeris access using jpleph.f */
/*      initnovas_(azJPL,&iLen); */

/* 	Ephemeris access using jpleph_dfb.c */
/*	ephopn(azJPL,&jdbeg,&jdend); */

/*	Do nothing if no ephemeris access is required */

	return(0);
	}
/*----------------------------------------------------------------------------*/
int flagpointdata(int argc, char *argp[]){
	/* 
	Apply or remove flags in PointData variables listed in CHAMELEON flag 
	table.
	*/

	IDL_LONG *piNFlag,*piFlag;
	unsigned char *pucItem,*pucIBeam,*pucOBeam,*pucTr,*pucBl;
	short int *piCh; /* Now needed for high-res spectrometers */
	double *pdfTime;
	double *pdfPointTime;
	float *pfFDLPosErr,*pfFDLJitterErr,*pfNATJitterErr,*pfNATCountsErr;
	float *pfVisSqErr,*pfVisAmpErr,*pfVisPhaseErr;
	float *pfTripleAmpErr,*pfTriplePhaseErr,*pfPhotonRateErr;
	float *pfGrpDelayErr,*pfDryDelayErr,*pfWetDelayErr;
	IDL_STRING *azStationId,*azBaselineId;
	IDL_LONG *piTripleOB,*piTripleBl,*piTripleCh,*piTripleNCh;
	IDL_LONG *piNIBeam,*piNOBeam,*piNTr,*piNBl,*piNCh,*piNPt;
	IDL_LONG *piMBl,*piMTr,*piMTCh,*piMCh;
	int iOBeam,iIBeam,iBl,iBl1,iBl2,iTr,iCh;

	long i,j,k,k1,k2,l;
	int iNFlagged=0;
	float fFlag;

        if (argc != 39) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	piNFlag=             ((IDL_LONG**)argp)[0];
	piFlag=              ((IDL_LONG**)argp)[1];

	pucItem=    ((unsigned char**)argp)[2];
	pucIBeam=   ((unsigned char**)argp)[3];
	pucOBeam=   ((unsigned char**)argp)[4];
	pucTr=      ((unsigned char**)argp)[5];
	pucBl=      ((unsigned char**)argp)[6];
	piCh=		((short int**)argp)[7];
	pdfTime=           ((double**)argp)[8];

	pdfPointTime=      ((double**)argp)[9];
	pfFDLPosErr=       ((float**)argp)[10];
	pfFDLJitterErr=    ((float**)argp)[11];
        pfNATJitterErr=    ((float**)argp)[12];
        pfNATCountsErr=    ((float**)argp)[13];
	pfPhotonRateErr=   ((float**)argp)[14];
	pfVisSqErr=        ((float**)argp)[15];
	pfVisAmpErr=       ((float**)argp)[16];
	pfVisPhaseErr=     ((float**)argp)[17];
	pfTripleAmpErr=    ((float**)argp)[18];
	pfTriplePhaseErr=  ((float**)argp)[19];
	pfGrpDelayErr=     ((float**)argp)[20];
	pfDryDelayErr=     ((float**)argp)[21];
	pfWetDelayErr=     ((float**)argp)[22];

	azStationId=  ((IDL_STRING**)argp)[23];
	azBaselineId= ((IDL_STRING**)argp)[24];

	piTripleOB=         ((IDL_LONG**)argp)[25];
	piTripleBl=         ((IDL_LONG**)argp)[26];
	piTripleCh=         ((IDL_LONG**)argp)[27];
	piTripleNCh=        ((IDL_LONG**)argp)[28];

	piNIBeam=           ((IDL_LONG**)argp)[29];
	piNOBeam=           ((IDL_LONG**)argp)[30];
	piNTr=              ((IDL_LONG**)argp)[31];
	piNBl=              ((IDL_LONG**)argp)[32];
	piNCh=              ((IDL_LONG**)argp)[33];
	piNPt=              ((IDL_LONG**)argp)[34];

	piMTr=              ((IDL_LONG**)argp)[35];
	piMTCh=             ((IDL_LONG**)argp)[36];
	piMBl=              ((IDL_LONG**)argp)[37];
	piMCh=              ((IDL_LONG**)argp)[38];

	fFlag=(float)piFlag[0];	/* 1 for flag; -1 for unflag */
	
	/* Loop over all table entries */
	for(i=0;i<piNFlag[0];i++){

	 /* Find the index of the data point corresponding to piTime */
	 j=0;
	 while((j<piNPt[0])&&(pdfTime[i]!=fabs(pdfPointTime[j])))j++; 
	 if(j==piNPt[0])continue;   /* Data not found; goto next flag entry */

	 /* First, we edit the primary (selected) data */
	 switch(pucItem[i]){
	  case 1:
	  case 2:{
	   /* Time and PointNo */
	   iNFlagged+=countflag(pdfPointTime[j],fFlag);
	   pdfPointTime[j]=NEG(pdfPointTime[j])*fFlag;
	   break;
 	   }
	  case 3:
	  case 4:{
	   /* VisSq or FASN will edit VisSq and VisAmp */
	   /* Note that arrays in PV-Wave/IDL are stored in FORTRAN convention,
	      i.e. the fastest varying index to the left. */
	   k=pucOBeam[i]+piNOBeam[0]*(piCh[i]+piMCh[0]*(pucBl[i]+piMBl[0]*j));
	   iNFlagged+=countflag(pfVisSqErr[k],fFlag);
	   pfVisSqErr[k]=NEG(pfVisSqErr[k])*fFlag;
	   pfVisAmpErr[k]=NEG(pfVisAmpErr[k])*fFlag;
	   break;
	   }
	  case 5:{
	   /* VisPhase */
	   k=pucOBeam[i]+piNOBeam[0]*(piCh[i]+piMCh[0]*(pucBl[i]+piMBl[0]*j));
	   iNFlagged+=countflag(pfVisPhaseErr[k],fFlag);
	   pfVisPhaseErr[k]=NEG(pfVisPhaseErr[k])*fFlag;
	   break;
	   }
	  case 6:
	  case 7:{
	   /* TripleAmp or TriplePhase will edit TripleAmp and TriplePhase */
	   k=pucTr[i]+piNTr[0]*(piCh[i]+piMTCh[0]*j);
	   iNFlagged+=countflag(pfTripleAmpErr[k],fFlag);
	   pfTripleAmpErr[k]=NEG(pfTripleAmpErr[k])*fFlag;
	   pfTriplePhaseErr[k]=NEG(pfTriplePhaseErr[k])*fFlag;
	   break;
	   }
	  case 8:{
	   /* PhotonRate */
	   k=pucOBeam[i]+piNOBeam[0]*(piCh[i]+piMCh[0]*j);
	   iNFlagged+=countflag(pfPhotonRateErr[k],fFlag);
	   pfPhotonRateErr[k]=NEG(pfPhotonRateErr[k])*fFlag;
	   break;
	   }
	  case  9:
	  case 17:{
	   /* FDLPos */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   iNFlagged+=countflag(pfFDLPosErr[k],fFlag);
	   pfFDLPosErr[k]=NEG(pfFDLPosErr[k])*fFlag;
           break;
	   }
	  case 13:
	  case 14:
	  case 15:{
	   /* GrpDelay,DryDelay,WetDelay */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   iNFlagged+=countflag(pfGrpDelayErr[k],fFlag);
	   pfGrpDelayErr[k]=NEG(pfGrpDelayErr[k])*fFlag;
	   pfDryDelayErr[k]=NEG(pfDryDelayErr[k])*fFlag;
	   pfWetDelayErr[k]=NEG(pfWetDelayErr[k])*fFlag;
           break;
	   }
	  case 10:{
	   /* FDLJitter */
	   k=pucOBeam[i]+piNOBeam[0]*(pucBl[i]+piMBl[0]*j);
	   iNFlagged+=countflag(pfFDLJitterErr[k],fFlag);
	   pfFDLJitterErr[k]=NEG(pfFDLJitterErr[k])*fFlag;
           break;
	   }
	  case 11:
	  case 12:{
	   /* NATJitter, NATCounts */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   iNFlagged+=countflag(pfNATJitterErr[k],fFlag);
	   pfNATJitterErr[k]=NEG(pfNATJitterErr[k])*fFlag;
	   pfNATCountsErr[k]=NEG(pfNATCountsErr[k])*fFlag;
           break;
	   }
	  }

	 /* Second, we edit logically dependent data */
	 switch(pucItem[i]){
	  case 1:
	  case 2:{
	   /* Time or PointNo: edit all data */
	   for(iOBeam=0;iOBeam<piNOBeam[0];iOBeam++){
	    for(iCh=0;iCh<piNCh[iOBeam];iCh++){
	     for(iBl=0;iBl<piNBl[iOBeam];iBl++){
	      k=iOBeam+piNOBeam[0]*(iCh+piMCh[0]*(iBl+piMBl[0]*j));
	      pfVisSqErr[k]=NEG(pfVisSqErr[k])*fFlag;
	      pfVisAmpErr[k]=NEG(pfVisAmpErr[k])*fFlag;
	      pfVisPhaseErr[k]=NEG(pfVisPhaseErr[k])*fFlag;
	      }
	     k=iOBeam+piNOBeam[0]*(iCh+piMCh[0]*j);
	     pfPhotonRateErr[k]=NEG(pfPhotonRateErr[k])*fFlag;
	     }
	    }
	   for(iTr=0;iTr<piNTr[0];iTr++){
	    for(iCh=0;iCh<piTripleNCh[iTr];iCh++){
	     k=iTr+piNTr[0]*(iCh+piMTCh[0]*j);
	     pfTripleAmpErr[k]=NEG(pfTripleAmpErr[k])*fFlag;
	     pfTriplePhaseErr[k]=NEG(pfTriplePhaseErr[k])*fFlag;
	     }
	    }
	   for(iOBeam=0;iOBeam<piNOBeam[0];iOBeam++){
            for(iBl=0;iBl<piNBl[iOBeam];iBl++){
	     k=iOBeam+piNOBeam[0]*(iBl+piMBl[0]*j);
	     pfFDLJitterErr[k]=NEG(pfFDLJitterErr[k])*fFlag;
	     }
	    }
	   for(iIBeam=0;iIBeam<piNIBeam[0];iIBeam++){
	    k=iIBeam+piNIBeam[0]*j;
	    pfFDLPosErr[k]=NEG(pfFDLPosErr[k])*fFlag;
            pfNATJitterErr[k]=NEG(pfNATJitterErr[k])*fFlag;
            pfNATCountsErr[k]=NEG(pfNATCountsErr[k])*fFlag;
	    }
	   break;
	   }
	  case 3:
	  case 4:
	  case 5:{
	   /* VisSq, VisAmp or VisPhase edits TripleAmp+Phase containing b.l. */
	   for(iTr=0;iTr<piNTr[0];iTr++){
	    for(iBl=0;iBl<3;iBl++){
	     k=iBl+3*iTr;
	     if((piTripleOB[k]==(int)pucOBeam[i])&&
	        (piTripleBl[k]==(int)pucBl[i])){
	      /* We found a triple containing the current baseline */
	      /* We now have to find the correct triple channel */
	      for(iCh=0;iCh<piTripleNCh[iTr];iCh++){
	       k=iCh+piMTCh[0]*(iBl+3*iTr);
	       if(piTripleCh[k]==(int)piCh[i]){
		if(iBl==0){iBl1=1;iBl2=2;}
		if(iBl==1){iBl1=0;iBl2=2;}
		if(iBl==2){iBl1=0;iBl2=1;}
	        k1=piTripleOB[iBl1+3*iTr]
                  +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl1+3*iTr)]
                                +piMCh[0]*(piTripleBl[iBl1+3*iTr]+piMBl[0]*j));
	        k2=piTripleOB[iBl2+3*iTr]
                  +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl2+3*iTr)]
                                +piMCh[0]*(piTripleBl[iBl2+3*iTr]+piMBl[0]*j));
	 	if(fFlag>0 || (fFlag<0 && pfVisSqErr[k1]>0 && pfVisSqErr[k2]>0)){
	         k=iTr+piNTr[0]*(iCh+piMTCh[0]*j);
	         pfTripleAmpErr[k]=NEG(pfTripleAmpErr[k])*fFlag;
	         pfTriplePhaseErr[k]=NEG(pfTriplePhaseErr[k])*fFlag;
		 }
	        break;
		}
	       }
	      }
	     }
	    }
	   break;
	   }
	  case 6:
	  case 7:break;
	  case 8:{
	   /* PhotonRate will edit VisSq + VisAmp + VisPhase and all triples */
	   for(iBl=0;iBl<piNBl[pucOBeam[i]];iBl++){
	    k=pucOBeam[i]+piNOBeam[0]*(piCh[i]+piMCh[0]*(iBl+piMBl[0]*j));
	    pfVisSqErr[k]=NEG(pfVisSqErr[k])*fFlag;
	    pfVisAmpErr[k]=NEG(pfVisAmpErr[k])*fFlag;
	    pfVisPhaseErr[k]=NEG(pfVisPhaseErr[k])*fFlag;
	    }
	   for(iTr=0;iTr<piNTr[0];iTr++){
	    for(iBl=0;iBl<3;iBl++){
	     k=iBl+3*iTr;
	     if((piTripleOB[k]==(int)pucOBeam[i])){
	      /* We found a triple containing the current outputbeam */
	      /* We now have to find the correct triple channel */
	      for(iCh=0;iCh<piTripleNCh[iTr];iCh++){
	       k=iCh+piMTCh[0]*(iBl+3*iTr);
	       if(piTripleCh[k]==(int)piCh[i]){
		if(iBl==0){iBl1=1;iBl2=2;}
		if(iBl==1){iBl1=0;iBl2=2;}
		if(iBl==2){iBl1=0;iBl2=1;}
	        k1=piTripleOB[iBl1+3*iTr]
                  +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl1+3*iTr)]
                                +piMCh[0]*(piTripleBl[iBl1+3*iTr]+piMBl[0]*j));
	        k2=piTripleOB[iBl2+3*iTr]
                  +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl2+3*iTr)]
                                +piMCh[0]*(piTripleBl[iBl2+3*iTr]+piMBl[0]*j));
	 	if(fFlag>0 || (fFlag<0 && pfVisSqErr[k1]>0 && pfVisSqErr[k2]>0)){
	         k=iTr+piNTr[0]*(iCh+piMTCh[0]*j);
	         pfTripleAmpErr[k]=NEG(pfTripleAmpErr[k])*fFlag;
	         pfTriplePhaseErr[k]=NEG(pfTriplePhaseErr[k])*fFlag;
		 }
	        break;
		}
	       }
	      }
	     }
	    }
	   break;
	   }
	  case  9:
	  case 17:{
	   /* Will edit Vis,ComplexVis,ComplTriple,FDLJitter in any OB 
	      containing IB */
	   for(iOBeam=0;iOBeam<piNOBeam[0];iOBeam++){
	    for(iBl=0;iBl<piNBl[iOBeam];iBl++){
	     k=iBl+piMBl[0]*iOBeam;
	     if(strstr(azBaselineId[k].s,azStationId[pucIBeam[i]].s)!=NULL){
	      k=iOBeam+piNOBeam[0]*(iBl+piMBl[0]*j);
	      pfFDLJitterErr[k]=NEG(pfFDLJitterErr[k])*fFlag;
	      for(iCh=0;iCh<piNCh[iOBeam];iCh++){
	       k=iOBeam+piNOBeam[0]*(iCh+piMCh[0]*(iBl+piMBl[0]*j));
	       pfVisSqErr[k]=NEG(pfVisSqErr[k])*fFlag;
	       pfVisAmpErr[k]=NEG(pfVisAmpErr[k])*fFlag;
	       pfVisPhaseErr[k]=NEG(pfVisPhaseErr[k])*fFlag;
	       }
	      }
	     }
	    }
	   for(iTr=0;iTr<piNTr[0];iTr++){
	    for(iBl=0;iBl<3;iBl++){
	     k=iBl+3*iTr;
	     k=piTripleBl[k]+piNBl[0]*piTripleOB[k];
	     if(strstr(azBaselineId[k].s,azStationId[pucIBeam[i]].s)!=NULL){
              if(iBl==0){iBl1=1;iBl2=2;}
              if(iBl==1){iBl1=0;iBl2=2;}
              if(iBl==2){iBl1=0;iBl2=1;}
	      for(iCh=0;iCh<piTripleNCh[iTr];iCh++){
	       k1=piTripleOB[iBl1+3*iTr]
                 +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl1+3*iTr)]
                               +piMCh[0]*(piTripleBl[iBl1+3*iTr]+piMBl[0]*j));
	       k2=piTripleOB[iBl2+3*iTr]
                 +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl2+3*iTr)]
                               +piMCh[0]*(piTripleBl[iBl2+3*iTr]+piMBl[0]*j));
               if(fFlag>0 || (fFlag<0 && pfVisSqErr[k1]>0 && pfVisSqErr[k2]>0)){
	        k=iTr+piNTr[0]*(iCh+piMTCh[0]*j);
	        pfTripleAmpErr[k]=NEG(pfTripleAmpErr[k])*fFlag;
	        pfTriplePhaseErr[k]=NEG(pfTriplePhaseErr[k])*fFlag;
		}
  	       }
	      }
	     }
	    }
	   break;
	   }
	  case 13:
	  case 14:
	  case 15:{
	   break;
	   }
	  case 16:{
	   /* Will edit Grp-, Dry, and WetDelay */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   pfGrpDelayErr[k]=NEG(pfGrpDelayErr[k])*fFlag;
	   pfDryDelayErr[k]=NEG(pfDryDelayErr[k])*fFlag;
	   pfWetDelayErr[k]=NEG(pfWetDelayErr[k])*fFlag;
           break;
	   }
	  case 10:{
	   /* Will edit VisSq,VisAmp,VisPhase */
	   for(iCh=0;iCh<piNCh[pucOBeam[i]];iCh++){
	    k=pucOBeam[i]+piNOBeam[0]*(iCh+piMCh[0]*(pucBl[i]+piMBl[0]*j));
	    pfVisSqErr[k]=NEG(pfVisSqErr[k])*fFlag;
	    pfVisAmpErr[k]=NEG(pfVisAmpErr[k])*fFlag;
	    pfVisPhaseErr[k]=NEG(pfVisPhaseErr[k])*fFlag;
	    }
	   /* Will edit TripleAmp+Phase containing b.l. */
	   for(iTr=0;iTr<piNTr[0];iTr++){
	    for(iBl=0;iBl<3;iBl++){
	     k=iBl+3*iTr;
	     if((piTripleOB[k]==(int)pucOBeam[i])&&
	        (piTripleBl[k]==(int)pucBl[i])){
	      /* We found a triple containing the current baseline */
              if(iBl==0){iBl1=1;iBl2=2;}
              if(iBl==1){iBl1=0;iBl2=2;}
              if(iBl==2){iBl1=0;iBl2=1;}
	      for(iCh=0;iCh<piTripleNCh[iTr];iCh++){
	       k1=piTripleOB[iBl1+3*iTr]
                 +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl1+3*iTr)]
                               +piMCh[0]*(piTripleBl[iBl1+3*iTr]+piMBl[0]*j));
	       k2=piTripleOB[iBl2+3*iTr]
                 +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl2+3*iTr)]
                               +piMCh[0]*(piTripleBl[iBl2+3*iTr]+piMBl[0]*j));
               if(fFlag>0 || (fFlag<0 && pfVisSqErr[k1]>0 && pfVisSqErr[k2]>0)){
	        k=iTr+piNTr[0]*(iCh+piMTCh[0]*j);
	        pfTripleAmpErr[k]=NEG(pfTripleAmpErr[k])*fFlag;
	        pfTriplePhaseErr[k]=NEG(pfTriplePhaseErr[k])*fFlag;
		}
	       }
	      break;
	      }
	     }
	    }
	   break;
	   }
	  case 11:{
	   /* Will edit Vis,ComplexVis,ComplTriple,FDLJitter in any OB 
	      containing IB, as well as FDLPos of IB */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   pfFDLPosErr[k]=NEG(pfFDLPosErr[k])*fFlag;
	   for(iOBeam=0;iOBeam<piNOBeam[0];iOBeam++){
	    for(iBl=0;iBl<piNBl[iOBeam];iBl++){
	     k=iBl+piMBl[0]*iOBeam;
	     if(strstr(azBaselineId[k].s,azStationId[pucIBeam[i]].s)!=NULL){
	      k=iOBeam+piNOBeam[0]*(iBl+piMBl[0]*j);
	      pfFDLJitterErr[k]=NEG(pfFDLJitterErr[k])*fFlag;
	      for(iCh=0;iCh<piNCh[iOBeam];iCh++){
	       k=iOBeam+piNOBeam[0]*(iCh+piMCh[0]*(iBl+piMBl[0]*j));
	       pfVisSqErr[k]=NEG(pfVisSqErr[k])*fFlag;
	       pfVisAmpErr[k]=NEG(pfVisAmpErr[k])*fFlag;
	       pfVisPhaseErr[k]=NEG(pfVisPhaseErr[k])*fFlag;
	       }
	      }
	     }
	    }
	   for(iTr=0;iTr<piNTr[0];iTr++){
	    for(iBl=0;iBl<3;iBl++){
	     k=iBl+3*iTr;
	     k=piTripleBl[k]+piNBl[0]*piTripleOB[k];
	     if(strstr(azBaselineId[k].s,azStationId[pucIBeam[i]].s)!=NULL){
              if(iBl==0){iBl1=1;iBl2=2;}
              if(iBl==1){iBl1=0;iBl2=2;}
              if(iBl==2){iBl1=0;iBl2=1;}
	      for(iCh=0;iCh<piTripleNCh[iTr];iCh++){
	       k1=piTripleOB[iBl1+3*iTr]
                 +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl1+3*iTr)]
                               +piMCh[0]*(piTripleBl[iBl1+3*iTr]+piMBl[0]*j));
	       k2=piTripleOB[iBl2+3*iTr]
                 +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl2+3*iTr)]
                               +piMCh[0]*(piTripleBl[iBl2+3*iTr]+piMBl[0]*j));
               if(fFlag>0 || (fFlag<0 && pfVisSqErr[k1]>0 && pfVisSqErr[k2]>0)){
	        k=iTr+piNTr[0]*(iCh+piMTCh[0]*j);
	        pfTripleAmpErr[k]=NEG(pfTripleAmpErr[k])*fFlag;
	        pfTriplePhaseErr[k]=NEG(pfTriplePhaseErr[k])*fFlag;
		}
  	       }
	      }
	     }
	    }
	   break;
	   }
	  case 12:{
	   /* Will edit Vis,ComplexVis,ComplTriple,FDLJitter in any OB 
	      containing IB, as well as PhotonRate in OB containing IB 
	      and FDLPos of IB */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   pfFDLPosErr[k]=NEG(pfFDLPosErr[k])*fFlag;
	   for(iOBeam=0;iOBeam<piNOBeam[0];iOBeam++){
	    for(iBl=0;iBl<piNBl[iOBeam];iBl++){
	     k=iBl+piMBl[0]*iOBeam;
	     if(strstr(azBaselineId[k].s,azStationId[pucIBeam[i]].s)!=NULL){
	      k=iOBeam+piNOBeam[0]*(iBl+piMBl[0]*j);
	      pfFDLJitterErr[k]=NEG(pfFDLJitterErr[k])*fFlag;
	      for(iCh=0;iCh<piNCh[iOBeam];iCh++){
	       k=iOBeam+piNOBeam[0]*(iCh+piMCh[0]*j);
	       pfPhotonRateErr[k]=NEG(pfPhotonRateErr[k])*fFlag;
	       k=iOBeam+piNOBeam[0]*(iCh+piMCh[0]*(iBl+piMBl[0]*j));
	       pfVisSqErr[k]=NEG(pfVisSqErr[k])*fFlag;
	       pfVisAmpErr[k]=NEG(pfVisAmpErr[k])*fFlag;
	       pfVisPhaseErr[k]=NEG(pfVisPhaseErr[k])*fFlag;
	       }
	      }
	     }
	    }
	   for(iTr=0;iTr<piNTr[0];iTr++){
	    for(iBl=0;iBl<3;iBl++){
	     k=iBl+3*iTr;
	     k=piTripleBl[k]+piNBl[0]*piTripleOB[k];
	     if(strstr(azBaselineId[k].s,azStationId[pucIBeam[i]].s)!=NULL){
              if(iBl==0){iBl1=1;iBl2=2;}
              if(iBl==1){iBl1=0;iBl2=2;}
              if(iBl==2){iBl1=0;iBl2=1;}
	      for(iCh=0;iCh<piTripleNCh[iTr];iCh++){
	       k1=piTripleOB[iBl1+3*iTr]
                 +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl1+3*iTr)]
                               +piMCh[0]*(piTripleBl[iBl1+3*iTr]+piMBl[0]*j));
	       k2=piTripleOB[iBl2+3*iTr]
                 +piNOBeam[0]*(piTripleCh[iCh+piMTCh[0]*(iBl2+3*iTr)]
                               +piMCh[0]*(piTripleBl[iBl2+3*iTr]+piMBl[0]*j));
               if(fFlag>0 || (fFlag<0 && pfVisSqErr[k1]>0 && pfVisSqErr[k2]>0)){
	        k=iTr+piNTr[0]*(iCh+piMTCh[0]*j);
	        pfTripleAmpErr[k]=NEG(pfTripleAmpErr[k])*fFlag;
	        pfTriplePhaseErr[k]=NEG(pfTriplePhaseErr[k])*fFlag;
		}
  	       }
	      }
	     }
	    }
	   break;
	   }
	  default:return(-1);
	  }
	 }
	return(iNFlagged);
	}
/*----------------------------------------------------------------------------*/
int flaginchdata(int argc, char *argp[]){
	/* 
	Apply or remove flags in InchData variables listed in CHAMELEON flag 
	table.
	*/

	IDL_LONG *piNFlag,*piFlag;
	unsigned char *pucItem,*pucIBeam;
	double *pdfTime;
	double *pdfMetroTime;
	float *pfParXErr, *pfParYErr, *pfParZErr, *pfDelayErr;
	double *pdfMotorAngleErr;
	IDL_LONG *piNIBeam,*piNPt;
	int iIBeam;

	int i,j,k;
	int iNFlagged=0;
	float fFlag;

        if (argc != 13) {
      		fprintf(stderr,"\nWrong # of parameters(%d)!\n",argc);
      		return(1);
      		}

	piNFlag=             ((IDL_LONG**)argp)[0];
	piFlag=              ((IDL_LONG**)argp)[1];

	pucItem=    ((unsigned char**)argp)[2];
	pucIBeam=   ((unsigned char**)argp)[3];
	pdfTime=           ((double**)argp)[4];

	pdfMetroTime=      ((double**)argp)[5];
	pfParXErr=	    ((float**)argp)[6];
	pfParYErr=	    ((float**)argp)[7];
	pfParZErr=	    ((float**)argp)[8];
	pfDelayErr=         ((float**)argp)[9];
	pdfMotorAngleErr=  ((double**)argp)[10];

	piNIBeam=           ((IDL_LONG**)argp)[11];
	piNPt=              ((IDL_LONG**)argp)[12];

	fFlag=(float)piFlag[0];	/* 1 for flag; -1 for unflag */
	
	/* Loop over all table entries */
	for(i=0;i<piNFlag[0];i++){

	 /* Find the index of the data point corresponding to piTime */
	 j=0;
	 while((j<piNPt[0])&&(pdfTime[i]!=fabs(pdfMetroTime[j])))j++; 
	 if(j==piNPt[0])continue;   /* Data not found; goto next flag entry */

	 /* First, we edit the primary (selected) data */
	 switch(pucItem[i]){
	  case 60:{
	   /* MetroTime */
	   iNFlagged+=countflag(pdfMetroTime[j],fFlag);
	   pdfMetroTime[j]=NEG(pdfMetroTime[j])*fFlag;
	   break;
 	   }
	  case 61:{
	   /* ParX */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   iNFlagged+=countflag(pfParXErr[k],fFlag);
	   pfParXErr[k]=NEG(pfParXErr[k])*fFlag;
           break;
	   }
	  case 62:{
	   /* ParY */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   iNFlagged+=countflag(pfParYErr[k],fFlag);
	   pfParYErr[k]=NEG(pfParYErr[k])*fFlag;
           break;
	   }
	  case 63:{
	   /* ParZ */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   iNFlagged+=countflag(pfParZErr[k],fFlag);
	   pfParZErr[k]=NEG(pfParZErr[k])*fFlag;
           break;
	   }
	  case 64:
	  case 65:{
	   /* MetroPath */
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   iNFlagged+=countflag(pfDelayErr[k],fFlag);
	   pfDelayErr[k]=NEG(pfDelayErr[k])*fFlag;
           break;
	   }
	  case 68:{
	   /* Hour angle */
	   k=0+2*(pucIBeam[i]+piNIBeam[0]*j);
	   iNFlagged+=countflag(pdfMotorAngleErr[k],fFlag);
	   pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	   k=1+2*(pucIBeam[i]+piNIBeam[0]*j);
	   iNFlagged+=countflag(pdfMotorAngleErr[k],fFlag);
	   pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	   break;
 	   }
	  case 69:{
	   /* Declination */
	   k=0+2*(pucIBeam[i]+piNIBeam[0]*j);
	   iNFlagged+=countflag(pdfMotorAngleErr[k],fFlag);
	   pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	   k=1+2*(pucIBeam[i]+piNIBeam[0]*j);
	   iNFlagged+=countflag(pdfMotorAngleErr[k],fFlag);
	   pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	   break;
 	   }
	  case 70:{
	   /* Azimuth */
	   k=0+2*(pucIBeam[i]+piNIBeam[0]*j);
	   iNFlagged+=countflag(pdfMotorAngleErr[k],fFlag);
	   pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	   k=1+2*(pucIBeam[i]+piNIBeam[0]*j);
	   iNFlagged+=countflag(pdfMotorAngleErr[k],fFlag);
	   pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	   break;
 	   }
	  case 71:{
	   /* Elevation */
	   k=0+2*(pucIBeam[i]+piNIBeam[0]*j);
	   iNFlagged+=countflag(pdfMotorAngleErr[k],fFlag);
	   pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	   k=1+2*(pucIBeam[i]+piNIBeam[0]*j);
	   iNFlagged+=countflag(pdfMotorAngleErr[k],fFlag);
	   pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	   break;
 	   }
	   break;
	  default:return(-1);
	  }

	 /* Second, we edit logically dependent data */
	 switch(pucItem[i]){
	  case 60:{
	   /* Time */
	   for(iIBeam=0;iIBeam<piNIBeam[0];iIBeam++){
	    k=iIBeam+piNIBeam[0]*j;
	    pfParXErr[k]=NEG(pfParXErr[k])*fFlag;
	    pfParYErr[k]=NEG(pfParYErr[k])*fFlag;
	    pfParZErr[k]=NEG(pfParZErr[k])*fFlag;
	    pfDelayErr[k]=NEG(pfDelayErr[k])*fFlag;
	    k=0+2*(iIBeam+piNIBeam[0]*j);
	    pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	    k=1+2*(iIBeam+piNIBeam[0]*j);
	    pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	    }
	   break;
	   }
	  case 68:{
	   /* Hourangle */
	   for(iIBeam=0;iIBeam<piNIBeam[0];iIBeam++){
	    k=iIBeam+piNIBeam[0]*j;
	    pfParXErr[k]=NEG(pfParXErr[k])*fFlag;
	    pfParYErr[k]=NEG(pfParYErr[k])*fFlag;
	    pfParZErr[k]=NEG(pfParZErr[k])*fFlag;
	    pfDelayErr[k]=NEG(pfDelayErr[k])*fFlag;
	    k=0+2*(iIBeam+piNIBeam[0]*j);
	    pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	    k=1+2*(iIBeam+piNIBeam[0]*j);
	    pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	    }
	   break;
	   }
	  case 69:{
	   /* Declination */
	   for(iIBeam=0;iIBeam<piNIBeam[0];iIBeam++){
	    k=iIBeam+piNIBeam[0]*j;
	    pfParXErr[k]=NEG(pfParXErr[k])*fFlag;
	    pfParYErr[k]=NEG(pfParYErr[k])*fFlag;
	    pfParZErr[k]=NEG(pfParZErr[k])*fFlag;
	    pfDelayErr[k]=NEG(pfDelayErr[k])*fFlag;
	    k=0+2*(iIBeam+piNIBeam[0]*j);
	    pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	    k=1+2*(iIBeam+piNIBeam[0]*j);
	    pdfMotorAngleErr[k]=NEG(pdfMotorAngleErr[k])*fFlag;
	    }
	   break;
	   }
	  case 70:{
	   /* Azimuth */
	   for(iIBeam=0;iIBeam<piNIBeam[0];iIBeam++){
	    k=iIBeam+piNIBeam[0]*j;
	    pfParXErr[k]=NEG(pfParXErr[k])*fFlag;
	    pfParYErr[k]=NEG(pfParYErr[k])*fFlag;
	    pfParZErr[k]=NEG(pfParZErr[k])*fFlag;
	    pfDelayErr[k]=NEG(pfDelayErr[k])*fFlag;
	    }
	   break;
	   }
	  case 71:{
	   /* Elevation */
	   for(iIBeam=0;iIBeam<piNIBeam[0];iIBeam++){
	    k=iIBeam+piNIBeam[0]*j;
	    pfParXErr[k]=NEG(pfParXErr[k])*fFlag;
	    pfParYErr[k]=NEG(pfParYErr[k])*fFlag;
	    pfParZErr[k]=NEG(pfParZErr[k])*fFlag;
	    pfDelayErr[k]=NEG(pfDelayErr[k])*fFlag;
	    }
	   break;
	   }
	  case 61:{
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   pfParYErr[k]=NEG(pfParYErr[k])*fFlag;
	   pfParZErr[k]=NEG(pfParZErr[k])*fFlag;
	   pfDelayErr[k]=NEG(pfDelayErr[k])*fFlag;
           break;
	   }
	  case 62:{
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   pfParXErr[k]=NEG(pfParXErr[k])*fFlag;
	   pfParZErr[k]=NEG(pfParZErr[k])*fFlag;
	   pfDelayErr[k]=NEG(pfDelayErr[k])*fFlag;
           break;
	   }
	  case 63: {
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   pfParXErr[k]=NEG(pfParXErr[k])*fFlag;
	   pfParYErr[k]=NEG(pfParYErr[k])*fFlag;
	   pfDelayErr[k]=NEG(pfDelayErr[k])*fFlag;
           break;
	   }
	  case 64: 
	  case 65: {
	   k=pucIBeam[i]+piNIBeam[0]*j;
	   pfParXErr[k]=NEG(pfParXErr[k])*fFlag;
	   pfParYErr[k]=NEG(pfParYErr[k])*fFlag;
	   pfParZErr[k]=NEG(pfParZErr[k])*fFlag;
           break;
	   }
	  default:return(-1);
	  }
	 }
	return(iNFlagged);
	}
/*----------------------------------------------------------------------------*/
int unwrapphase(int argc, char *argp[]){
	/* 
	Unwrap the phase in ComplexVis.
	*/

	float *pfVisPhase,*pfVisPhaseErr;
	IDL_LONG *piNOB,*piNBl,*piNCH,*piNumBaseline,*piNumChannel,*piNumPoint;
	int iOBeam,iPt,iBl,iCh;
	int k,iNcycle;
	float curr,last,diff,twopi;

        if (argc != 8) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	pfVisPhase=   ((float**)argp)[0];
	pfVisPhaseErr=((float**)argp)[1];
	piNOB=         ((IDL_LONG**)argp)[2];
	piNBl=         ((IDL_LONG**)argp)[3];
	piNCH=         ((IDL_LONG**)argp)[4];
	piNumBaseline= ((IDL_LONG**)argp)[5];
	piNumChannel=  ((IDL_LONG**)argp)[6];
	piNumPoint=    ((IDL_LONG**)argp)[7];

	twopi=2.*PI;

	for(iOBeam=0;iOBeam<piNOB[0];iOBeam++){
	 for(iPt=0;iPt<piNumPoint[0];iPt++){
	  for(iBl=0;iBl<piNumBaseline[iOBeam];iBl++){
	   last=0;
	   iNcycle=0;
	   for(iCh=0;iCh<piNumChannel[iOBeam];iCh++){
	    curr=last;
	    k=iOBeam+piNOB[0]*(iCh+piNCH[0]*(iBl+piNBl[0]*iPt));
	    if(pfVisPhaseErr[k]>0)curr=pfVisPhase[k];
	    diff=curr-last;
	    last=curr;
	    if(diff<-PI)iNcycle+=1;
	    else if(diff>PI)iNcycle-=1;
	    pfVisPhase[k]+=(twopi*iNcycle);
	    }
	   }
	  }
	 }
	return(0);
	}
/*----------------------------------------------------------------------------*/
int unwraptriple(int argc, char *argp[]){
	/* 
	Unwrap the TriplePhase.
	*/

	float *pfTriplePhase,*pfTriplePhaseErr;
	IDL_LONG *piNTr,*piNCh,*piMTr,*piMCh,*piNumPoint;
	int iTr,iPt,iCh;
	int k,iNcycle;
	float curr,last,diff,twopi;

        if (argc != 7) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	pfTriplePhase=   ((float**)argp)[0];
	pfTriplePhaseErr=((float**)argp)[1];
	piNTr=            ((IDL_LONG**)argp)[2];
	piNCh=            ((IDL_LONG**)argp)[3];
	piMTr=            ((IDL_LONG**)argp)[4];
	piMCh=            ((IDL_LONG**)argp)[5];
	piNumPoint=       ((IDL_LONG**)argp)[6];

	twopi=2.*PI;

	for(iTr=0;iTr<piNTr[0];iTr++){
	 for(iCh=0;iCh<piNCh[iTr];iCh++){
	  last=0;
	  iNcycle=0;
	  for(iPt=0;iPt<piNumPoint[0];iPt++){
	   curr=last;
	   k=iTr+piMTr[0]*(iCh+piMCh[0]*iPt);
	   if(pfTriplePhaseErr[k]>0)curr=pfTriplePhase[k];
	   diff=curr-last;
	   last=curr;
	   if(diff<-PI)iNcycle+=1;
	   else if(diff>PI)iNcycle-=1;
	   pfTriplePhase[k]+=(twopi*iNcycle);
	   }
	  }
	 }
	return(0);
	}
/*----------------------------------------------------------------------------*/
int catalog(int argc, char *argp[]){
	/* 
	Get information on list of stars from various catalogs.
	*/

	/* Hipparcos input catalogue */
	#define HIC_RECLEN 431
	#define HIC_NLINES 118209
  	char azHICRecord[HIC_RECLEN+1];
	FILE *fpHIC;

	/* Hipparcos output catalogue */
	#define HIP_RECLEN 451
	#define HIP_NLINES 118218
  	char azHIPRecord[HIP_RECLEN+1];
	FILE *fpHIP;

	/* SAO catalogue */
	#define SAO_RECLEN 205
	#define SAO_NLINES 258944
	char azSAORecord[SAO_RECLEN+1];
	FILE *fpSAO;

	/* FK5 catalogue */
	#define FKV_RECLEN 204
	#define FKV_NLINES 4652
	char azFKVRecord[FKV_RECLEN+1];
	FILE *fpFKV;

	/* Bright star catalogue */
	#define BSC_RECLEN 213
	#define BSC_NLINES 9096
	char azBSCRecord[BSC_RECLEN+1];
	FILE *fpBSC;

	/* Batten's Catalogue (extended version) */
	#define BAT_RECLEN 241
	#define BAT_NLINES 1513
	char azBATRecord[BAT_RECLEN+1];
	FILE *fpBAT;

	/* Sky Catalogue 2000.0 */
	#define SKY_RECLEN 108
	#define SKY_NLINES 49418
	char azSKYRecord[SKY_RECLEN+1];
	FILE *fpSKY;

	/* A Finding List for Observers of Interacting Binary Stars, 5th Ed. */
	#define FLN_RECLEN 176
	#define FLN_NLINES 3564
	char azFLNRecord[FLN_RECLEN+1];
	FILE *fpFLN;

	/* Define string large enough to hold copy of any catalog record */
	#define MAX_RECLEN 500	
	char azRecord[MAX_RECLEN];	
	char azBlank[MAX_RECLEN];
	char azTransferString[MAX_RECLEN];

	#define GET_INT(record,firstcol,width) \
 	 (memcpy(azRecord,record+firstcol,width),azRecord[width]='\0', \
						atol(azRecord))
	#define GET_DBL(record,firstcol,width) \
 	 (memcpy(azRecord,record+firstcol,width),azRecord[width]='\0', \
						atof(azRecord))
	#define IDL_StringStore(dest,source,numchar) \
	 (memset(azTransferString,'\0',MAX_RECLEN), \
	  strncpy(azTransferString,source,numchar), \
	  IDL_StrStore(&(dest),azTransferString))
/*	Disable this macro for use in GDL */
	#define IDL_StringStore(dest,source,numchar) 

	unsigned char *acStarName,*acVarName,*acSpectrum,*acHFlag,*acVFlag;
	IDL_LONG *pibStarName,*pibVarName,*pibSpectrum,*pibHFlag,*pibVFlag;

	IDL_STRING *azStarID,*azStarName,*azVarName,*azSpectrum,*azHFlag,*azVFlag;
	float *pfType1,*pfType2,*pfClass1,*pfClass2;
	float *pfPMRa,*pfPMDec,*pfRV,*pfVsini,*pfPX,*pfPXE;
	float *pfmV,*pfBV,*pfUB,*pfRI,*pfDmV;
	float *pfPeriod,*pfEcc,*pfomega,*pfK1,*pfK2,*pfMF;
	float *pfM1sin3i,*pfM2sin3i,*pfA1sini,*pfA2sini,*pfAmajor;
	double *pdfRa,*pdfDec,*pdfEpoch;
	IDL_LONG *piNumStar,*piHDN,*piHIC,*piSAO;
	short *piBAT,*piBSC,*piFKV,*piFLN,*piADS;
	long  iHICNumber,iSAONumber,iHDNNumber;
	short iFKVNumber,iBSCNumber,iBATNumber,iFLNNumber;
  	int iRaHour,iRaMin,iDecDeg,iDecMin;
	int i,j,k,iEntry,iCount,iAll;
	float fM1sin3i,fM2sin3i;
 	double dfRaSec,dfDecSec,dfRIC;
 	char sSign;

	REALNUMBER t0,a0[6],c0[6][6],t,a[6],c[6][6];

	/* Initialize blank record */
	memset(azBlank,' ',MAX_RECLEN-1);
	azBlank[MAX_RECLEN-1]='\0';

        if (argc != 54) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNumStar=    ((IDL_LONG**)argp)[0];
	azStarID=   ((IDL_STRING**)argp)[1];
	azStarName= ((IDL_STRING**)argp)[2];
	azVarName=  ((IDL_STRING**)argp)[3]; 
	pdfRa=          ((double**)argp)[4];
	pdfDec=         ((double**)argp)[5];
	piBAT=           ((short**)argp)[6];
	piHDN=        ((IDL_LONG**)argp)[7];
	piBSC=           ((short**)argp)[8];
	piFKV=           ((short**)argp)[9];
	piHIC=       ((IDL_LONG**)argp)[10];
	piSAO=       ((IDL_LONG**)argp)[11];
	piFLN=          ((short**)argp)[12];
	piADS=          ((short**)argp)[13];
	pfmV=           ((float**)argp)[14];
	pfBV=           ((float**)argp)[15];
	pfUB=           ((float**)argp)[16];
	pfRI=           ((float**)argp)[17];
	pfDmV=          ((float**)argp)[18];
	pfPMRa=         ((float**)argp)[19];
	pfPMDec=        ((float**)argp)[20];
	pfRV=           ((float**)argp)[21];
	pfVsini=        ((float**)argp)[22];
	pfPX=           ((float**)argp)[23];
	pfPXE=          ((float**)argp)[24];
	azSpectrum=((IDL_STRING**)argp)[25];
	pfType1=        ((float**)argp)[26];
	pfType2=        ((float**)argp)[27];
	pfClass1=       ((float**)argp)[28];
	pfClass2=       ((float**)argp)[29];
	pfPeriod=       ((float**)argp)[30];
	pdfEpoch=      ((double**)argp)[31];
	pfomega=        ((float**)argp)[32];
	pfEcc=          ((float**)argp)[33];
	pfK1=           ((float**)argp)[34];
	pfK2=           ((float**)argp)[35];
	pfMF=           ((float**)argp)[36];
	pfM1sin3i=      ((float**)argp)[37];
	pfM2sin3i=      ((float**)argp)[38];
	pfA1sini=       ((float**)argp)[39];
	pfA2sini=       ((float**)argp)[40];
	pfAmajor=       ((float**)argp)[41];
	azHFlag=   ((IDL_STRING**)argp)[42];
	azVFlag=   ((IDL_STRING**)argp)[43];
	acStarName=      ((char**)argp)[44];
	acVarName=       ((char**)argp)[45]; 
	acSpectrum=      ((char**)argp)[46];
	acHFlag=         ((char**)argp)[47];
	acVFlag=         ((char**)argp)[48];
	pibStarName= ((IDL_LONG**)argp)[49];
	pibVarName=  ((IDL_LONG**)argp)[50]; 
	pibSpectrum= ((IDL_LONG**)argp)[51];
	pibHFlag=    ((IDL_LONG**)argp)[52];
	pibVFlag=    ((IDL_LONG**)argp)[53];

	iAll=0;
	switch(piNumStar[0]){
		case HIC_NLINES:{iAll=HIC_NLINES;break;}
		case HIP_NLINES:{iAll=HIP_NLINES;break;}
		case BAT_NLINES:{iAll=BAT_NLINES;break;}
		case SAO_NLINES:{iAll=SAO_NLINES;break;}
		case SKY_NLINES:{iAll=SKY_NLINES;break;}
		case FKV_NLINES:{iAll=FKV_NLINES;break;}
		case BSC_NLINES:{iAll=BSC_NLINES;break;}
		case FLN_NLINES:{iAll=FLN_NLINES;break;}
		}
	for(i=0;i<piNumStar[0];i++){
		if(strncmp(azStarID[i].s,azStarID[0].s,3)!=0)break;
		}
	if((iAll!=0)&&(iAll==i)){iAll=1;}else{iAll=0;}

	fpHIC=fpHIP=fpSAO=fpSKY=fpFKV=fpBSC=fpBAT=fpFLN=NULL;

	for(i=0;i<piNumStar[0];i++){
/* 		Ignore entry number 0 */
		if((iEntry=atol(azStarID[i].s+3))<1){
			continue;
			}

		/* Hipparcos Input Catalogue */
		/* sort -n +1 -2 hipp.dat.orig > hipp.dat */
		if(strncmp(azStarID[i].s,"HIC",3)==0){
			if(fpHIC==NULL){
				if((fpHIC=fopen(azHIC,"r"))==NULL){
				fprintf(stderr,"Error opening %s!\n",azHIC);
				return(1);
				}
				}
			iCount=MIN(iEntry,HIC_NLINES);
			fseek(fpHIC,(iCount-1)*HIC_RECLEN,SEEK_SET);
			fread(azHICRecord,1,HIC_RECLEN,fpHIC);
			iHICNumber=GET_INT(azHICRecord,0,6);
			/* Some entries in HIC are missing, some out of order */
			while(iHICNumber>iEntry && iHICNumber!=iEntry){
				if(--iCount==0)break;
				fseek(fpHIC,-(HIC_RECLEN*2),SEEK_CUR);
				fread(azHICRecord,1,HIC_RECLEN,fpHIC);
				iHICNumber=GET_INT(azHICRecord,0,6);
				}
			if(iHICNumber!=iEntry){
				fprintf(stderr,"Entry not found: %s!\n",
						azStarID[i].s);
				piHIC[i]=0;
				IDL_TTYReset();
				continue;
				}
  			azHICRecord[HIC_RECLEN] = '\0';
			piHIC[i]=iHICNumber;

			/* HD number */
			piHDN[i]=GET_INT(azHICRecord,359,6);

			/* RA [h], DEC[deg] (J2000.0) */
 			if(sscanf(azHICRecord+14,"%d %d %lf",
			     &iRaHour,&iRaMin,&dfRaSec)!=3
     			 ||sscanf(azHICRecord+27,"%c%d %d %lf",
			     &sSign,&iDecDeg,&iDecMin,&dfDecSec)!=4){
				fprintf(stderr,"Error: bad entry!");
				continue;
				}
			pdfRa[i]=(double)iRaHour+(double)iRaMin/60.
						+dfRaSec/3600.;
  			pdfDec[i]=(double)iDecDeg+(double)iDecMin/60.
						+dfDecSec/3600.;
			if(sSign=='-')pdfDec[i]=-pdfDec[i];

			/* Centennial proper motion in RA [s] */
			pfPMRa[i]=GET_DBL(azHICRecord,155,6);
			pfPMRa[i]/=(cos(RAD(pdfDec[i]))*0.15);

			/* Centennial proper motion in DEC ["] */
			pfPMDec[i]=GET_DBL(azHICRecord,162,6)*100;

			/* Radial velocity [km/s] */
			pfRV[i]=GET_DBL(azHICRecord,241,6);

			/* Parallax ["] */
			pfPX[i]=(double)GET_INT(azHICRecord,230,5)/1000;

			/* Visual magnitude */
			pfmV[i]=GET_DBL(azHICRecord,190,6);

			/* (B-V) */
			pfBV[i]=GET_DBL(azHICRecord,202,6);

			/* Spectrum */
			IDL_StringStore(azSpectrum[i],azHICRecord+216,11);
			IDL_ChartrStore(acSpectrum+i*pibSpectrum[0],azHICRecord+216,11);
			spec_parse(azSpectrum[i].s,pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			spec_parse((char*)acSpectrum+i*pibSpectrum[0],pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			}

		/* Hipparcos Catalogue */
		/* sort -n +1 -2 hip_main.dat.orig > hip_main.dat */
                else if(strncmp(azStarID[i].s,"HIP",3)==0){
			if(fpHIP==NULL){
				if((fpHIP=fopen(azHIP,"r"))==NULL){
				fprintf(stderr,"Error opening %s!\n",azHIP);
				return(1);
				}
				}
			iCount=MIN(iEntry+10,HIP_NLINES);
			fseek(fpHIP,(iCount-1)*HIP_RECLEN,SEEK_SET);
			fread(azHIPRecord,1,HIP_RECLEN,fpHIP);
			iHICNumber=GET_INT(azHIPRecord,2,12);
			/* Some entries in HIP are missing, some out of order */
			while(iHICNumber>iEntry && iHICNumber!=iEntry){
				if(--iCount==0)break;
				fseek(fpHIP,-(HIP_RECLEN*2),SEEK_CUR);
				fread(azHIPRecord,1,HIP_RECLEN,fpHIP);
				iHICNumber=GET_INT(azHIPRecord,2,12);
				}
			if(iHICNumber!=iEntry){
				fprintf(stderr,"Entry not found: %s!\n",
						azStarID[i].s);
				piHIC[i]=0;
				IDL_TTYReset();
				continue;
				}
  			azHIPRecord[HIP_RECLEN] = '\0';
			piHIC[i]=iHICNumber;

			/* HD number */
			piHDN[i]=GET_INT(azHIPRecord,390,6);

			/* RA [deg], DEC[deg] (J1991.25) */
			if(strncmp(azHIPRecord+51,azBlank,12)!=0){
				pdfRa[i]=GET_DBL(azHIPRecord,51,12);
				pdfDec[i]=GET_DBL(azHIPRecord,64,12);
				}

			/* Proper motion in direction of RA [mas/yr] */
			pfPMRa[i]=GET_DBL(azHIPRecord,87,8);

			/* Proper motion in DEC [mas/yr] */
			pfPMDec[i]=GET_DBL(azHIPRecord,96,8);

			/* Parallax [mas] */
			pfPX[i]=GET_DBL(azHIPRecord,79,7);
			pfPXE[i]=GET_DBL(azHIPRecord,119,6);

			/* Visual magnitude */
			pfmV[i]=GET_DBL(azHIPRecord,41,5);

			/* Multiple star flag */
			IDL_StringStore(azHFlag[i]," ",1);
			IDL_ChartrStore(acHFlag+i*pibHFlag[0]," ",1);
			/* Solution quality */
			if(strncmp(azHIPRecord+350,"S",1)==0)
			IDL_StringStore(azHFlag[i],"S",1);
			if(strncmp(azHIPRecord+350,"S",1)==0)
			IDL_ChartrStore(acHFlag+i*pibHFlag[0],"S",1);
			if(strncmp(azHIPRecord+346,azBlank,1)!=0)
			IDL_StringStore(azHFlag[i],azHIPRecord+346,1);
			if(strncmp(azHIPRecord+346,azBlank,1)!=0)
			IDL_ChartrStore(acHFlag+i*pibHFlag[0],azHIPRecord+346,1);
/*
			if(strncmp(azHIPRecord+346,azBlank,1)!=0)
			IDL_StringStore(azHFlag[i],"B",1);
*/

			/* Variability type flag */
			IDL_StringStore(azVFlag[i]," ",1);
			if(strncmp(azHIPRecord+321,azBlank,1)!=0)
			IDL_StringStore(azVFlag[i],azHIPRecord+321,1);
			if(strncmp(azHIPRecord+321,azBlank,1)!=0)
			IDL_ChartrStore(acVFlag+i*pibVFlag[0],azHIPRecord+321,1);

			/* Separation of components */
			if(strncmp(azHIPRecord+359,azBlank,7)!=0)
			pfAmajor[i]=GET_DBL(azHIPRecord,359,7);

			/* Magnitude difference between components */
			if(strncmp(azHIPRecord+373,azBlank,5)!=0)
			pfDmV[i]=GET_DBL(azHIPRecord,373,5);

			/* (B-V) */
			pfBV[i]=GET_DBL(azHIPRecord,245,6);

			/* Spectrum */
			IDL_StringStore(azSpectrum[i],azHIPRecord+435,12);
			IDL_ChartrStore(acSpectrum+i*pibSpectrum[0],azHIPRecord+435,12);
			spec_parse(azSpectrum[i].s,pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			spec_parse((char*)acSpectrum+i*pibSpectrum[0],pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);

			/* Change to epoch J2000.0 */
			t0 = 1991.25;
			a0[0] = pdfRa[i];
			a0[1] = pdfDec[i];
			a0[2] = pfPX[i];
			a0[3] = pfPMRa[i];
			a0[4] = pfPMDec[i];
			a0[5] = pfRV[i];
			a0[5] = pfRV[i]*a0[2]/4.740470446;
			for (j=0; j<6; j++) {
				for (k=0; k<6; k++) c[j][k]=0;
				}
			t = 2000.0;
			pos_prop(t0,a0,c0,t,a,c);
			if (a[2] != 0) {
				pdfRa[i] = a[0]/15;		/* [h] */
				pdfDec[i] = a[1];
				pfPX[i] = a[2]/1000;		/* ["] */
				pfPXE[i] /= 1000;
				pfPMRa[i] = a[3]/(cos(RAD(pdfDec[i]))*150);
				pfPMDec[i] = a[4]*0.1;
				pfRV[i] = a[5]/a[2]*4.740470446;
				}
			}

		/* SAO Catalogue */
		else if(strncmp(azStarID[i].s,"SAO",3)==0){
			if(fpSAO==NULL){
				if((fpSAO=fopen(azSAO,"r"))==NULL){
				fprintf(stderr,"Error opening %s!\n",azSAO);
				return(1);
				}
				}
			iCount=MIN(iEntry,SAO_NLINES);
			fseek(fpSAO,(iCount-1)*SAO_RECLEN,SEEK_SET);
			fread(azSAORecord,1,SAO_RECLEN,fpSAO);
			iSAONumber=GET_INT(azSAORecord,0,6);
			/* Some entries in SAO are missing */
			while(iSAONumber>iEntry){
				if(--iCount==0)break;
				fseek(fpSAO,-(SAO_RECLEN*2),SEEK_CUR);
				fread(azSAORecord,1,SAO_RECLEN,fpSAO);
				iSAONumber=GET_INT(azSAORecord,0,6);
				}
			if(iSAONumber!=iEntry){
				fprintf(stderr,"Entry not found: %s!\n",
						azStarID[i].s);
				piSAO[i]=0;
				IDL_TTYReset();
				continue;
				}
  			azSAORecord[SAO_RECLEN] = '\0';
			piSAO[i]=iSAONumber;

			/* HD number */
			piHDN[i]=GET_INT(azSAORecord,117,6);

			/* RA, DEC (J2000.0) */
			iRaHour=GET_INT(azSAORecord,150,2);
			iRaMin=GET_INT(azSAORecord,152,2);
			dfRaSec=GET_DBL(azSAORecord,154,6);
			iDecDeg=GET_INT(azSAORecord,168,2);
			iDecMin=GET_INT(azSAORecord,170,2);
			dfDecSec=GET_DBL(azSAORecord,172,5);
			pdfRa[i]=(double)iRaHour+(double)iRaMin/60.
						+dfRaSec/3600.;
  			pdfDec[i]=(double)iDecDeg+(double)iDecMin/60.
						+dfDecSec/3600.;
			if(azSAORecord[167]=='-')pdfDec[i]=-pdfDec[i];

			/* Proper motions */
			pfPMRa[i]=GET_DBL(azSAORecord,160,7);
			pfPMRa[i]*=100;
			pfPMDec[i]=GET_DBL(azSAORecord,177,6);
			pfPMDec[i]*=100;

			/* Visual magnitude */
			pfmV[i]=GET_DBL(azSAORecord,80,4);
			if(pfmV[i]==99.9)pfmV[i]=+100.0;

			/* Spectrum */
			IDL_StringStore(azSpectrum[i],azSAORecord+84,3);
			IDL_ChartrStore(acSpectrum+i*pibSpectrum[0],azSAORecord+84,3);
			spec_parse(azSpectrum[i].s,pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			spec_parse((char*)acSpectrum+i*pibSpectrum[0],pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			}

		/* Sky Catalogue 2000.0 */
		else if(strncmp(azStarID[i].s,"HDN",3)==0){
			if(fpSKY==NULL){
				if((fpSKY=fopen(azSKY,"r"))==NULL){
				fprintf(stderr,"Error opening %s!\n",azSKY);
				return(1);
				}
				}
			if(iAll==0){
				if(iEntry>226320){
					iCount=(int)(48923+(double)iEntry*0.00145);
					}
				else{
					iCount=iEntry/4.6;
					}
				iCount=MIN(iCount,SKY_NLINES);
				fseek(fpSKY,(iCount-1)*SKY_RECLEN,SEEK_SET);
				}
			fread(azSKYRecord,1,SKY_RECLEN,fpSKY);
			iHDNNumber=GET_INT(azSKYRecord,0,6);
			if(iAll==0){
				/* Some entries in SKY are missing */
				while(iHDNNumber>iEntry){
					if(--iCount==0)break;
					fseek(fpSKY,-(SKY_RECLEN*2),SEEK_CUR);
					fread(azSKYRecord,1,SKY_RECLEN,fpSKY);
					iHDNNumber=GET_INT(azSKYRecord,0,6);
					}
				while(iHDNNumber<iEntry){
					if(++iCount>SKY_NLINES)break;
					fread(azSKYRecord,1,SKY_RECLEN,fpSKY);
					iHDNNumber=GET_INT(azSKYRecord,0,6);
					}
				if(iHDNNumber!=iEntry){
					fprintf(stderr,"Entry not found: %s!\n",
						azStarID[i].s);
					piHDN[i]=0;
					IDL_TTYReset();
					continue;
					}
				}
  			azSKYRecord[SKY_RECLEN] = '\0';
			piHDN[i]=iHDNNumber;

			/* SAO number */
			piSAO[i]=GET_INT(azSKYRecord,7,6);

			/* ADS number */
			piADS[i]=GET_INT(azSKYRecord,87,5);

			/* RA, DEC (J2000.0) */
			iRaHour=GET_INT(azSKYRecord,25,2);
			iRaMin=GET_INT(azSKYRecord,27,2);
			dfRaSec=GET_DBL(azSKYRecord,29,3)/10;
			iDecDeg=GET_INT(azSKYRecord,33,2);
			iDecMin=GET_INT(azSKYRecord,35,2);
			dfDecSec=GET_DBL(azSKYRecord,37,2);
			pdfRa[i]=(double)iRaHour+(double)iRaMin/60.
						+dfRaSec/3600.;
  			pdfDec[i]=(double)iDecDeg+(double)iDecMin/60.
						+dfDecSec/3600.;
			if(azSKYRecord[32]=='-')pdfDec[i]=-pdfDec[i];

			/* Proper motions */
			pfPMRa[i]=GET_DBL(azSKYRecord,40,4);
			pfPMRa[i]/=10;
			if(azSKYRecord[39]=='-')pfPMRa[i]=-pfPMRa[i];
			pfPMDec[i]=GET_DBL(azSKYRecord,45,3);
			if(azSKYRecord[44]=='-')pfPMDec[i]=-pfPMDec[i];

			/* Visual magnitude */
			if(azSKYRecord[51]==' ')azSKYRecord[51]='0';
			pfmV[i]=GET_DBL(azSKYRecord,48,4);
			pfmV[i]/=100;

			/* (B-V) */
			if(azSKYRecord[58]==' ')azSKYRecord[58]='0';
			pfBV[i]=GET_DBL(azSKYRecord,55,4);
			pfBV[i]/=100;

			/* Radial velocity */
			pfRV[i]=GET_DBL(azSKYRecord,78,4);

			/* Parallax */
			pfPX[i]=GET_DBL(azSKYRecord,82,4);
			if(pfPX[i]!=0)pfPX[i]=1/pfPX[i];

			/* Spectrum */
			IDL_StringStore(azSpectrum[i],azSKYRecord+62,16);
			IDL_ChartrStore(acSpectrum+i*pibSpectrum[0],azSKYRecord+62,16);
			spec_parse(azSpectrum[i].s,pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			spec_parse((char*)acSpectrum+i*pibSpectrum[0],pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);

			/* Name */
			IDL_StringStore(azStarName[i],azSKYRecord+13,12);
			}

		/* FK5 Catalogue */
		else if(strncmp(azStarID[i].s,"FKV",3)==0){
			if(fpFKV==NULL){
				if((fpFKV=fopen(azFKV,"r"))==NULL){
				fprintf(stderr,"Error opening %s!\n",azFKV);
				return(1);
				}
				}
			if(iAll==0){
				iCount=iEntry;
				if(iCount>1670)iCount-=332;
				iCount=MIN(iCount,FKV_NLINES);
				fseek(fpFKV,(iCount-1)*FKV_RECLEN,SEEK_SET);
				}
			fread(azFKVRecord,1,FKV_RECLEN,fpFKV);
			iFKVNumber=GET_INT(azFKVRecord,0,4);
			if(iAll==0){
				/* Some entries in FKV are missing */
				while(iFKVNumber>iEntry){
					if(--iCount==0)break;
					fseek(fpFKV,-(FKV_RECLEN*2),SEEK_CUR);
					fread(azFKVRecord,1,FKV_RECLEN,fpFKV);
					iFKVNumber=GET_INT(azFKVRecord,0,4);
					}
				if(iFKVNumber!=iEntry){
					fprintf(stderr,"Entry not found: %s!\n",
						azStarID[i].s);
					piFKV[i]=0;
					IDL_TTYReset();
					continue;
					}
				}
  			azFKVRecord[FKV_RECLEN] = '\0';
			piFKV[i]=iFKVNumber;

			/* HD number */
			piHDN[i]=GET_INT(azFKVRecord,180,6);

			/* RA, DEC (J2000.0) */
 			if(sscanf(azFKVRecord+6,"%d %d %lf",
			     &iRaHour,&iRaMin,&dfRaSec)!=3
     			 ||sscanf(azFKVRecord+29,"%c%d %d %lf",
			     &sSign,&iDecDeg,&iDecMin,&dfDecSec)!=4){
				fprintf(stderr,"Error: bad entry!");
				continue;
				}
			pdfRa[i]=(double)iRaHour+(double)iRaMin/60.
						+dfRaSec/3600.;
  			pdfDec[i]=(double)iDecDeg+(double)iDecMin/60.
						+dfDecSec/3600.;
			if(sSign=='-')pdfDec[i]= -pdfDec[i];

			/* Proper motions */
			pfPMRa[i]=GET_DBL(azFKVRecord,21,6);
		 	if(strncmp(azFKVRecord+20,"-",1)==0)
						pfPMRa[i]=-pfPMRa[i];
			pfPMDec[i]=GET_DBL(azFKVRecord,44,6);
		 	if(strncmp(azFKVRecord+43,"-",1)==0)
						pfPMDec[i]=-pfPMDec[i];

			/* Radial velocity (field might be blank) */
			if(strncmp(azFKVRecord+159,azBlank,6)==0)pfRV[i]=0.0;
			else{
				pfRV[i]=GET_DBL(azFKVRecord,159,6);
		 		if(strncmp(azFKVRecord+158,"-",1)==0)
						pfRV[i]=-pfRV[i];
				}

			/* Parallax (field might be blank) */
			if(strncmp(azFKVRecord+150,azBlank,6)==0)pfPX[i]=0.0;
			else{
				pfPX[i]=GET_DBL(azFKVRecord,150,6);
				}

			/* Visual magnitude */
			pfmV[i]=GET_DBL(azFKVRecord,136,4);
		 	if(strncmp(azFKVRecord+135,"-",1)==0)
						pfmV[i]=-pfmV[i];

			/* Spectrum */
			IDL_StringStore(azSpectrum[i],azFKVRecord+142,6);
			IDL_ChartrStore(acSpectrum+i*pibSpectrum[0],azFKVRecord+142,6);
			spec_parse(azSpectrum[i].s,pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			spec_parse((char*)acSpectrum+i*pibSpectrum[0],pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			}

		/* Bright Star Catalogue */
		else if(strncmp(azStarID[i].s,"BSC",3)==0){
			if(fpBSC==NULL){
				if((fpBSC=fopen(azBSC,"r"))==NULL){
				fprintf(stderr,"Error opening %s!\n",azBSC);
				return(1);
				}
				}
			if(iAll==0){
				iCount=MIN(iEntry,BSC_NLINES);
				fseek(fpBSC,(iCount-1)*BSC_RECLEN,SEEK_SET);
				}
			fread(azBSCRecord,1,BSC_RECLEN,fpBSC);
			iBSCNumber=GET_INT(azBSCRecord,0,4);
			if(iAll==0){
				/* Some entries in BSC are missing */
				while(iBSCNumber>iEntry){
					if(--iCount==0)break;
					fseek(fpBSC,-(BSC_RECLEN*2),SEEK_CUR);
					fread(azBSCRecord,1,BSC_RECLEN,fpBSC);
					iBSCNumber=GET_INT(azBSCRecord,0,4);
					}
				if(iBSCNumber!=iEntry){
					fprintf(stderr,"Entry not found: %s!\n",
						azStarID[i].s);
					piBSC[i]=0;
					IDL_TTYReset();
					continue;
					}
				}
  			azBSCRecord[BSC_RECLEN] = '\0';
			piBSC[i]=iBSCNumber;

			/* HD Number */
			piHDN[i]=GET_INT(azBSCRecord,25,6);

			/* RA, DEC (J2000.0) */
 			if(sscanf(azBSCRecord+73,"%d %d %lf",
			     &iRaHour,&iRaMin,&dfRaSec)!=3
     			 ||sscanf(azBSCRecord+84,"%c%d %d %lf",
			     &sSign,&iDecDeg,&iDecMin,&dfDecSec)!=4){
				fprintf(stderr,"Error: bad entry!");
				continue;
				}
			pdfRa[i]=(double)iRaHour+(double)iRaMin/60.
						+dfRaSec/3600.;
  			pdfDec[i]=(double)iDecDeg+(double)iDecMin/60.
						+dfDecSec/3600.;
			if(sSign=='-')pdfDec[i]= -pdfDec[i];

			/* Proper motions */
			pfPMRa[i]=GET_DBL(azBSCRecord,156,5);
			pfPMRa[i]/=(cos(RAD(pdfDec[i]))*0.15);
		 	if(strncmp(azBSCRecord+155,"-",1)==0)
						pfPMRa[i]=-pfPMRa[i];
			pfPMDec[i]=GET_DBL(azBSCRecord,163,5);
			pfPMDec[i]*=100;
		 	if(strncmp(azBSCRecord+162,"-",1)==0)
						pfPMDec[i]=-pfPMDec[i];

			/* Radial velocity (field might be blank) */
			IDL_StringStore(azHFlag[i]," ",1);
			IDL_ChartrStore(acHFlag+i*pibHFlag[0]," ",1);
			if(strncmp(azBSCRecord+175,azBlank,4)==0)pfRV[i]=0.0;
			else{
				pfRV[i]=(double)GET_INT(azBSCRecord,176,3);
				if(strncmp(azBSCRecord+175,"-",1)==0)
						pfRV[i]=-pfRV[i];
				if(strncmp(azBSCRecord+179,"SB",2)==0)
						IDL_StringStore(azHFlag[i],"R",1);
				if(strncmp(azBSCRecord+179,"V",2)==0)
						IDL_StringStore(azHFlag[i],"R",1);
				}

			/* Parallax (field might be blank) */
			if(strncmp(azBSCRecord+169,azBlank,5)==0)pfPX[i]=0.0;
			else{
				pfPX[i]=GET_DBL(azBSCRecord,170,4);
				if(strncmp(azBSCRecord+169,"-",1)==0)
						pfPX[i]=-pfPX[i];
				}

			/* V sin(i) (field might be blank) */
			if(strncmp(azBSCRecord+185,azBlank,3)==0)pfVsini[i]=0.0;
			else{
				pfVsini[i]=GET_DBL(azBSCRecord,185,3);
				}

			/* Photometry */
			pfmV[i]=GET_DBL(azBSCRecord,108,5);
			if(strncmp(azBSCRecord+115,azBlank,5)==0)
						pfBV[i]=+100.0;
			else{
				pfBV[i]=GET_DBL(azBSCRecord,115,5);
				}
			if(strncmp(azBSCRecord+121,azBlank,5)==0)
						pfUB[i]=+100.0;
			else{
				pfUB[i]=GET_DBL(azBSCRecord,121,5);
				}
			if(strncmp(azBSCRecord+127,azBlank,5)==0)
						pfRI[i]=+100.0;
			else{
				pfRI[i]=GET_DBL(azBSCRecord,127,5);
				if(strncmp(azBSCRecord+132,"E",1)==0){
					if(pfRI[i]<0.35){
						dfRIC=0.118+1.030*pfRI[i];
						}
					else {
						dfRIC=0.033+1.246*pfRI[i];
						}
					pfRI[i]=(dfRIC-0.025)/0.856;
					}
				if(strncmp(azBSCRecord+132,"C",1)==0){
					pfRI[i]=(pfRI[i]-0.025)/0.856;
					}
				}	

			/* Spectrum */
			IDL_StringStore(azSpectrum[i],azBSCRecord+134,20);
			IDL_ChartrStore(acSpectrum+i*pibSpectrum[0],azBSCRecord+134,20);
			spec_parse(azSpectrum[i].s,pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			spec_parse((char*)acSpectrum+i*pibSpectrum[0],pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);

			/* Name */
			IDL_StringStore(azStarName[i],azBSCRecord+4,10);
			IDL_ChartrStore(acStarName+i*pibStarName[0],azBSCRecord+4,10);

			/* Variable name */
			IDL_StringStore(azVarName[i],azBSCRecord+42,9);
			IDL_ChartrStore(acVarName+i*pibVarName[0],azBSCRecord+42,9);
			}

		/* Batten's Catalogue */
		else if(strncmp(azStarID[i].s,"BAT",3)==0){
			if(fpBAT==NULL){
				if((fpBAT=fopen(azBAT,"r"))==NULL){
				fprintf(stderr,"Error opening %s!\n",azBAT);
				return(1);
				}
				}
			if(iAll==0){
				iCount=MIN(iEntry,BAT_NLINES);
				fseek(fpBAT,(iCount-1)*BAT_RECLEN,SEEK_SET);
				}
			fread(azBATRecord,1,BAT_RECLEN,fpBAT);
			iBATNumber=GET_INT(azBATRecord,0,4);
			if(iAll==0){
				/* Some entries in BAT are missing */
				while(iBATNumber>iEntry){
					if(--iCount==0)break;
					fseek(fpBAT,-(BAT_RECLEN*2),SEEK_CUR);
					fread(azBATRecord,1,BAT_RECLEN,fpBAT);
					iBATNumber=GET_INT(azBATRecord,0,4);
					}
				if(iBATNumber!=iEntry){
					fprintf(stderr,"Entry not found: %s!\n",
						azStarID[i].s);
					piBAT[i]=0;
					IDL_TTYReset();
					continue;
					}
				}
  			azBATRecord[BAT_RECLEN] = '\0';
			piBAT[i]=iBATNumber;

			/* HD Number */
			piHDN[i]=GET_INT(azBATRecord,19,7);

			/* RA, DEC (B1900.0) */
			iRaHour=GET_INT(azBATRecord,5,2);
			iRaMin=GET_INT(azBATRecord,7,2);
			dfRaSec=GET_DBL(azBATRecord,9,1)*60;
			pdfRa[i]=(double)iRaHour+(double)iRaMin/60.
						+dfRaSec/3600.;
			sscanf(azBATRecord+10,"%c",&sSign);
			iDecDeg=GET_INT(azBATRecord,11,2);
			iDecMin=GET_INT(azBATRecord,13,2);
  			pdfDec[i]=(double)iDecDeg+(double)iDecMin/60.
						+dfDecSec/3600.;
			if(sSign=='-')pdfDec[i]= -pdfDec[i];

			/* Photometry */
			pfmV[i]=GET_DBL(azBATRecord,77,5);
			if(strncmp(azBATRecord+84,azBlank,5)==0)
						pfDmV[i]=+100.0;
			else{
				pfDmV[i]=GET_DBL(azBATRecord,84,5);
				pfDmV[i]-=pfmV[i];
				}

			/* Spectrum */
			if(strncmp(azBATRecord+109,azBlank,9)!=0){
				memset(azBATRecord+107,'+',1);
				}
			IDL_StringStore(azSpectrum[i],azBATRecord+91,26);
			IDL_ChartrStore(acSpectrum+i*pibSpectrum[0],azBATRecord+91,26);
			spec_parse(azSpectrum[i].s,pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			spec_parse((char*)acSpectrum+i*pibSpectrum[0],pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);

			/* Period [d] */
			pfPeriod[i]=GET_DBL(azBATRecord,120,10);
			if(strncmp(azBATRecord+130,"y",1)==0){
				pfPeriod[i]*=365.25;
				}

			/* Epoch [JD] */
			pdfEpoch[i]=GET_DBL(azBATRecord,133,10);
			pdfEpoch[i]+=2400000;

			/* omega [deg] */
			pfomega[i]=GET_DBL(azBATRecord,145,5);

			/* Eccentricity */
			pfEcc[i]=GET_DBL(azBATRecord,157,4);

			/* K1, K2 [km/s] */
			pfK1[i]=GET_DBL(azBATRecord,168,5);
			pfK2[i]=GET_DBL(azBATRecord,175,5);

			/* RV [km/s] */
			pfRV[i]=GET_DBL(azBATRecord,183,5);
			sscanf(azBATRecord+182,"%c",&sSign);
			if(sSign=='-')pfRV[i]= -pfRV[i];

			/* f(m) */
			pfMF[i]=GET_DBL(azBATRecord,197,7);

			/* M1sin3i, M2sin3i [M_sun] */
			pfM1sin3i[i]=GET_DBL(azBATRecord,205,7);
			pfM2sin3i[i]=GET_DBL(azBATRecord,213,7);
			/* Check these values with K1, K2, P, and e */
			if((pfK1[i]!=0)&&(pfK2[i]!=0)&&(pfPeriod[i]!=0)){
				fM1sin3i=1.0385e-7
					*pow(1.0-pow(pfEcc[i],2),3./2.)
					*pow(pfK1[i]+pfK2[i],2)
					*pfK2[i]*pfPeriod[i];
				fM2sin3i=1.0385e-7
					*pow(1.0-pow(pfEcc[i],2),3./2.)
					*pow(pfK1[i]+pfK2[i],2)
					*pfK1[i]*pfPeriod[i];
				if((fabs((fM1sin3i/pfM1sin3i[i])-1)>0.1)||
			  	   (fabs((fM2sin3i/pfM2sin3i[i])-1)>0.1)){
					pfM1sin3i[i]=fM1sin3i;
					pfM2sin3i[i]=fM2sin3i;
					}
				}

			/* A1sini, A2sini [km] */
			pfA1sini[i]=GET_DBL(azBATRecord,221,8);
			pfA2sini[i]=GET_DBL(azBATRecord,230,8);

			/* Name */
			IDL_StringStore(azStarName[i],azBATRecord+58,17);
			IDL_ChartrStore(acStarName+i*pibStarName[0],azBATRecord+58,17);
			}

		/* Interacting binary star finding list */
		else if(strncmp(azStarID[i].s,"FLN",3)==0){
			if(fpFLN==NULL){
				if((fpFLN=fopen(azFLN,"r"))==NULL){
				fprintf(stderr,"Error opening %s!\n",azFLN);
				return(1);
				}
				}
			if(iAll==0){
				iCount=MIN(iEntry,FLN_NLINES);
				fseek(fpFLN,(iCount-1)*FLN_RECLEN,SEEK_SET);
				}
			fread(azFLNRecord,1,FLN_RECLEN,fpFLN);
			iFLNNumber=GET_INT(azFLNRecord,0,4);
			if(iAll==0){
				/* Some entries in FLN are missing */
				while(iFLNNumber>iEntry){
					if(--iCount==0)break;
					fseek(fpFLN,-(FLN_RECLEN*2),SEEK_CUR);
					fread(azFLNRecord,1,FLN_RECLEN,fpFLN);
					iFLNNumber=GET_INT(azFLNRecord,0,4);
					}
				if(iFLNNumber!=iEntry){
					fprintf(stderr,"Entry not found: %s!\n",
						azStarID[i].s);
					piFLN[i]=0;
					IDL_TTYReset();
					continue;
					}
				}
  			azFLNRecord[FLN_RECLEN] = '\0';
			piFLN[i]=iFLNNumber;

			/* HD Number */
			piHDN[i]=GET_INT(azFLNRecord,123,6);

			/* Photometry */
			pfmV[i]=GET_DBL(azFLNRecord,19,4);
			if(pfmV[i]==99.0)pfmV[i]=+100.0;
			if(strncmp(azFLNRecord+24,azBlank,3)==0)
						pfDmV[i]=+100.0;
			else{
				pfDmV[i]=GET_DBL(azFLNRecord,24,3);
				}

			/* Spectrum */
			if(strncmp(azFLNRecord+43,azBlank,10)!=0){
				memset(azFLNRecord+42,'+',1);
				}
			else{
				memset(azFLNRecord+42,' ',1);
				}
			IDL_StringStore(azSpectrum[i],azFLNRecord+32,21);
			IDL_ChartrStore(acSpectrum+i*pibSpectrum[0],azFLNRecord+32,21);
			spec_parse(azSpectrum[i].s,pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);
			spec_parse((char*)acSpectrum+i*pibSpectrum[0],pfType1+i,pfType2+i,
						pfClass1+i,pfClass2+i);

			/* Period [d] */
			pfPeriod[i]=GET_DBL(azFLNRecord,66,16);
			if(strncmp(azFLNRecord+82,"Y",1)==0){
				pfPeriod[i]*=365.25;
				}

			/* Epoch [JD] */
			pdfEpoch[i]=GET_DBL(azFLNRecord,54,11);
			pdfEpoch[i]+=2400000;

			}
		else{
			if(strncmp(azStarID[i].s,"OBJ",3)!=0)
			IDL_TTYReset();
			printf("Star not found in standard catalogs: %s\n",azStarID[i].s);
			IDL_TTYReset();
			}
		}

	if(fpHIC!=NULL)fclose(fpHIC);
	if(fpHIP!=NULL)fclose(fpHIP);
	if(fpSAO!=NULL)fclose(fpSAO);
	if(fpSKY!=NULL)fclose(fpSKY);
	if(fpFKV!=NULL)fclose(fpFKV);
	if(fpBSC!=NULL)fclose(fpBSC);
	if(fpBAT!=NULL)fclose(fpBAT);
	if(fpFLN!=NULL)fclose(fpFLN);

	return(0);
	}
/*----------------------------------------------------------------------------*/
int crossindex(int argc, char *argp[]){
	/* 
	Read Hipparcos, FKV, and BSC catalogs and write out entry numbers
	to file crossindex.dat.
	*/

	/* Hipparcos catalogue */
/*
	#define HIC_RECLEN 431
	#define HIC_NLINES 118209
*/
  	char azHICRecord[HIC_RECLEN+1];
	FILE *fpHIC;

	/* FK5 catalogue */
/*
	#define FKV_RECLEN 204
	#define FKV_NLINES 4652
*/
	char azFKVRecord[FKV_RECLEN+1];
	FILE *fpFKV;

	/* Bright star catalogue */
/*
	#define BSC_RECLEN 213
	#define BSC_NLINES 9096
*/
	char azBSCRecord[BSC_RECLEN+1];
	FILE *fpBSC;

	/* Define string large enough to hold copy of any catalog record */
/*
	#define MAX_RECLEN 500	
*/
	char azRecord[MAX_RECLEN];	
	char azBlank[MAX_RECLEN];
	char azFKVName[8];
/*
	#define GET_INT(record,firstcol,width) \
 	 (memcpy(azRecord,record+firstcol,width),azRecord[width]='\0',atol(azRecord))
	#define GET_DBL(record,firstcol,width) \
 	 (memcpy(azRecord,record+firstcol,width),azRecord[width]='\0',atof(azRecord))
*/
	FILE *fpCrossIndex;
	int i,iCount,iHICNumber,iFKVNumber,iHRNumber,iHDNumber,iSAONumber;

	/* Initialize blank record */
	memset(azBlank,' ',MAX_RECLEN-1);
	azBlank[MAX_RECLEN]='\0';

	if((fpCrossIndex=fopen("crossindex.dat","w"))==NULL){
		fprintf(stderr,"Error opening crossindex.dat!");
		exit(-1);
		}

	/* HIPPARCOS catalog */
	if((fpHIC=fopen(azHIC,"r"))==NULL){
		fprintf(stderr,"Error opening %s!\n",azHIC);
		return(-1);
		}
	iHRNumber=0;
	for(iCount=0;iCount<HIC_NLINES;iCount++){
		fread(azHICRecord,1,HIC_RECLEN,fpHIC);

		iHICNumber=GET_INT(azHICRecord,0,6);
		iHDNumber=GET_INT(azHICRecord,359,6);
		iFKVNumber=GET_INT(azHICRecord,366,7);
		memcpy(azFKVName,azHICRecord+366,7);
		azFKVName[7]='\0';
		iFKVNumber=0;
		if(strncmp(azFKVName,"F",1)==0)iFKVNumber=GET_INT(azHICRecord,367,6);
		iSAONumber=GET_INT(azHICRecord,385,6);

		fprintf(fpCrossIndex,"%6d %4d %4d %6d %6d\n", \
			iHICNumber,iFKVNumber,iHRNumber,iHDNumber,iSAONumber);
		}
	fclose(fpHIC);

	/* FK5 catalog */
	if((fpFKV=fopen(azFKV,"r"))==NULL){
		fprintf(stderr,"Error opening %s!\n",azFKV);
		return(-1);
		}
	iHICNumber=0;
	iSAONumber=0;
	iHRNumber=0;
	for(iCount=0;iCount<FKV_NLINES;iCount++){
		fread(azFKVRecord,1,FKV_RECLEN,fpFKV);

		iFKVNumber=GET_INT(azFKVRecord,0,4);
		iHDNumber=GET_INT(azFKVRecord,180,6);

		fprintf(fpCrossIndex,"%6d %4d %4d %6d %6d\n", \
			iHICNumber,iFKVNumber,iHRNumber,iHDNumber,iSAONumber);
		}
	fclose(fpFKV);

	/* Bright star catalog */
	if((fpBSC=fopen(azBSC,"r"))==NULL){
		fprintf(stderr,"Error opening %s!\n",azBSC);
		return(-1);
		}
	iHICNumber=0;
	iFKVNumber=0;
	iSAONumber=0;
	for(iCount=0;iCount<BSC_NLINES;iCount++){
		fread(azBSCRecord,1,BSC_RECLEN,fpBSC);

		iHRNumber=GET_INT(azBSCRecord,0,4);
		iHDNumber=GET_INT(azBSCRecord,25,6);
		fprintf(fpCrossIndex,"%6d %4d %4d %6d %6d\n", \
			iHICNumber,iFKVNumber,iHRNumber,iHDNumber,iSAONumber);
		}
	fclose(fpBSC);

	fclose(fpCrossIndex);

	return(0);
	}
/*----------------------------------------------------------------------------*/
int jacoby(int argc, char *argp[]){
	/*
	Read a spectrum from the spectrum library data file.
	*/

	#define SPL_RECLEN 82
	#define SPL_NLINES 56672
	#define NUMHREC 2
	#define NUMDREC 350
	#define NUMFLUXES 8

	IDL_LONG *piSpectrum;
	float *pfFluxes;

	FILE *fpSPL;
	long iRec,i;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piSpectrum=((IDL_LONG**)argp)[0];
	pfFluxes= ((float**)argp)[1];

	/* Open jacoby.dat */
	if((fpSPL=fopen(azSPL,"r"))==NULL){
		fprintf(stderr,"Error opening %s!\n",azSPL);
		return(1);
		}
	iRec=piSpectrum[0]*(NUMHREC+NUMDREC);
	if(iRec>=SPL_NLINES){
		fprintf(stderr,"Requested spectrum ID non-existent!\n");
		return(1);
		}
	fseek(fpSPL,(iRec+NUMHREC)*SPL_RECLEN,SEEK_SET);
	for(i=0;i<NUMDREC*NUMFLUXES-1;i++){
		if(fscanf(fpSPL,"%f",pfFluxes+i)!=1){
			fprintf(stderr,"Error reading flux value!\n");
			return(1);
			}
		}
	fclose(fpSPL);
	return(0);
	}
/*----------------------------------------------------------------------------*/
int specparse(int argc, char *argp[]){
	/* 
	Parse a spectral classification into integer types and classes.
	Called from PV-WAVE; calls function spec_parse to do the work.
	*/

	IDL_STRING *azSpectrum;
	IDL_LONG *piNumSpectrum;
	float *pfType1,*pfType2,*pfClass1,*pfClass2;
	int i;

        if (argc != 6) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNumSpectrum=((IDL_LONG**)argp)[0];
	azSpectrum=  ((IDL_STRING**)argp)[1];
	pfType1=     ((float**)argp)[2];
	pfType2=     ((float**)argp)[3];
	pfClass1=    ((float**)argp)[4];
	pfClass2=    ((float**)argp)[5];

	for(i=0;i<piNumSpectrum[0];i++){
		spec_parse(azSpectrum[i].s,pfType1+i,pfType2+i,
					pfClass1+i,pfClass2+i);
		}

	return(0);
	}
/*----------------------------------------------------------------------------*/
int tpstar(int argc, char *argp[]){
	/* 
	Call topo_star in NOVAS C-library of astrometry functions.
	*/

	IDL_LONG *piNumElements;
	double *pdfTJD,*pdfTDT_UT1,*pdfLocation;
	IDL_STRING *azStarName;
	IDL_LONG *piStarNumber;
	double *pdfRa,*pdfDec,*pdfRa_app,*pdfDec_app;
	double *pdfPMRa,*pdfPMDec,*pdfPX,*pdfRV,*pdfmV;
	cat_entry star;
	site_info locale;
	body earth;
	int i;
	short status;
	double jd1,jd2;

        if (argc != 14) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNumElements=    ((IDL_LONG**)argp)[0];
	pdfTJD=         ((double**)argp)[1];
	pdfTDT_UT1=     ((double**)argp)[2];
	pdfLocation=    ((double**)argp)[3];
	azStarName= ((IDL_STRING**)argp)[4];
	piStarNumber= ((IDL_LONG**)argp)[5];
	pdfRa=          ((double**)argp)[6];
	pdfDec=         ((double**)argp)[7];
	pdfPMRa=        ((double**)argp)[8];
	pdfPMDec=       ((double**)argp)[9];
	pdfPX=         ((double**)argp)[10];
	pdfRV=         ((double**)argp)[11];
	pdfRa_app=     ((double**)argp)[12];
	pdfDec_app=    ((double**)argp)[13];

	locale.longitude=pdfLocation[0];
	locale.latitude=pdfLocation[1];
	locale.height=pdfLocation[2];
	locale.temperature=pdfLocation[3];
	locale.pressure=pdfLocation[4];

	earth.type=0;
	earth.number=3;
	
	for(i=0;i<piNumElements[0];i++){
		strncpy(star.catalog,azStarName[i].s,3);
		strncpy(star.starname,azStarName[i].s,8);
		star.starnumber=piStarNumber[i];
		star.ra=pdfRa[i];
		star.dec=pdfDec[i];
		star.promora=pdfPMRa[i];
		star.promodec=pdfPMDec[i];
		star.parallax=pdfPX[i];
		star.radialvelocity=pdfRV[i];

		status=topo_star(pdfTJD[i],&earth,pdfTDT_UT1[i],&star,
			&locale,pdfRa_app+i,pdfDec_app+i);

		}

	return((int)status);
	}
/*----------------------------------------------------------------------------*/
int apstar(int argc, char *argp[]){
	/* 
	Call app_star in NOVAS C-library of astrometry functions.
	*/

	IDL_LONG *piNumElements;
	double *pdfTJD;
	IDL_STRING *azStarName;
	IDL_LONG *piStarNumber;
	double *pdfRa,*pdfDec,*pdfRa_app,*pdfDec_app;
	double *pdfPMRa,*pdfPMDec,*pdfPX,*pdfRV,*pdfmV;
	cat_entry star;
	body earth;
	int i;
	short status;
	double jd1,jd2;

        if (argc != 12) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNumElements=    ((IDL_LONG**)argp)[0];
	pdfTJD=         ((double**)argp)[1];
	azStarName= ((IDL_STRING**)argp)[2];
	piStarNumber=     ((IDL_LONG**)argp)[3];
	pdfRa=          ((double**)argp)[4];
	pdfDec=         ((double**)argp)[5];
	pdfPMRa=        ((double**)argp)[6];
	pdfPMDec=       ((double**)argp)[7];
	pdfPX=          ((double**)argp)[8];
	pdfRV=          ((double**)argp)[9];
	pdfRa_app=     ((double**)argp)[10];
	pdfDec_app=    ((double**)argp)[11];

	for(i=0;i<piNumElements[0];i++){
		strncpy(star.catalog,azStarName[i].s,3);
		strncpy(star.starname,azStarName[i].s,8);
		star.starnumber=piStarNumber[i];
		star.ra=pdfRa[i];
		star.dec=pdfDec[i];
		star.promora=pdfPMRa[i];
		star.promodec=pdfPMDec[i];
		star.parallax=pdfPX[i];
		star.radialvelocity=pdfRV[i];

		earth.type=0;
		earth.number=3;

		status=app_star(pdfTJD[i],&earth,&star,
			pdfRa_app+i,pdfDec_app+i);

		}

	return((int)status);
	}
/*----------------------------------------------------------------------------*/
int tpplanet(int argc, char *argp[]){
	/* 
	Call topo_planet in NOVAS C-library of astrometry functions.
	*/

	IDL_LONG *piNumElements,*piPlanet;
	double *pdfTJD,*pdfTDT_UT1,*pdfLocation;
	double *pdfRa_app,*pdfDec_app,*pdfDis;
	site_info locale;
	body planet,earth;
	int i;
	short status;

        if (argc != 8) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNumElements=   ((IDL_LONG**)argp)[0];
	piPlanet=	 ((IDL_LONG**)argp)[1];
	pdfTJD=        ((double**)argp)[2];
	pdfTDT_UT1=    ((double**)argp)[3];
	pdfLocation=   ((double**)argp)[4];
	pdfRa_app=     ((double**)argp)[5];
	pdfDec_app=    ((double**)argp)[6];
	pdfDis=	       ((double**)argp)[7];

	locale.longitude=pdfLocation[0];
	locale.latitude=pdfLocation[1];
	locale.height=pdfLocation[2];

	earth.type=0;
	earth.number=3;
	
	for(i=0;i<piNumElements[0];i++){

		planet.type=0;
		planet.number=piPlanet[i];

		status=topo_planet(pdfTJD[i],&planet,&earth,pdfTDT_UT1[i],
			&locale,pdfRa_app+i,pdfDec_app+i,pdfDis+i);

		}

	return((int)status);
	}
/*----------------------------------------------------------------------------*/
int applanet(int argc, char *argp[]){
	/* 
	Call app_planet in NOVAS C-library of astrometry functions.
	*/

	IDL_LONG *piNumElements,*piPlanet;
	double *pdfTJD;
	double *pdfRa_app,*pdfDec_app,*pdfDis;
	body planet,earth;
	int i;
	short status;

        if (argc != 6) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNumElements=   ((IDL_LONG**)argp)[0];
	piPlanet=	 ((IDL_LONG**)argp)[1];
	pdfTJD=        ((double**)argp)[2];
	pdfRa_app=     ((double**)argp)[3];
	pdfDec_app=    ((double**)argp)[4];
	pdfDis=	       ((double**)argp)[5];

	earth.type=0;
	earth.number=3;

	for(i=0;i<piNumElements[0];i++){

		planet.type=0;
		planet.number=piPlanet[i];

		status=app_planet(pdfTJD[i],&planet,&earth,
			pdfRa_app+i,pdfDec_app+i,pdfDis+i);

		}

	return((int)status);
	}
/*----------------------------------------------------------------------------*/
int ut12gst(int argc, char *argp[]){
	/* 
	Calls sidereal_time in  NOVAS C-library of astrometry functions.
	*/

	IDL_LONG *piNumElements,*piMean;
	double *pdfJulian_high,*pdfJulian_low,*pdfGST,*pdfDELTAT;
	int i;

	double tjd,ujd,tdb,peb[3],veb[3],pes[3],ves[3],oblm,oblt,eqeq,psi,eps;
	short int error;
	body earth;

        if (argc != 6) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNumElements=    ((IDL_LONG**)argp)[0];
	pdfJulian_high= ((double**)argp)[1];
	pdfJulian_low=  ((double**)argp)[2];
	pdfDELTAT=      ((double**)argp)[3];
	pdfGST=         ((double**)argp)[4];
	piMean=           ((IDL_LONG**)argp)[5];

	earth.type=0;
	earth.number=3;

	for(i=0;i<piNumElements[0];i++){
	tjd=pdfJulian_high[i]+pdfJulian_low[i];
	ujd=tjd-pdfDELTAT[0]/86400.0;
	error=get_earth(tjd,&earth,&tdb,peb,veb,pes,ves);
	earthtilt(tdb,&oblm,&oblt,&eqeq,&psi,&eps);
	if (piMean[0] != 0) eqeq=0;
	sidereal_time(ujd,0.0,eqeq,pdfGST+i);
	}

	return(0);
	}
/*----------------------------------------------------------------------------*/
int polarmotion(int argc, char *argp[]){
	/*
	Calls wobble in NOVAS C-library of astromtric functions.
	*/

	IDL_LONG *piNumElements;
	double *pdfPole_x,*pdfPole_y,*pdfPos1,*pdfPos2;
	int i;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNumElements=((IDL_LONG**)argp)[0];
	pdfPole_x=  ((double**)argp)[1];
	pdfPole_y=  ((double**)argp)[2];
	pdfPos1=    ((double**)argp)[3];
	pdfPos2=    ((double**)argp)[4];

	for(i=0;i<piNumElements[0];i++)
		wobble(pdfPole_x[i],pdfPole_y[i],pdfPos1+i*3,pdfPos2+i*3);

	return(0);
	}
/*----------------------------------------------------------------------------*/
int siderostat(int argc, char *argp[]){
	/*
	Computes motor elevation and azimuth relative to home index position
	for a NPOI siderostat specified through the model parameters.
	*/

	IDL_LONG *piN,*piM,*piDir;
	double *pdfHA,*pdfDec,*pdfAz,*pdfEl,*pdfModel;
	double x[2],y[2],dyda[16];
	int npar;
	int i;
	short status;

        if (argc != 8) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piN=          ((IDL_LONG**)argp)[0];
	pdfHA=      ((double**)argp)[1];
	pdfDec=     ((double**)argp)[2];
	pdfAz=      ((double**)argp)[3];
	pdfEl=      ((double**)argp)[4];
	pdfModel=   ((double**)argp)[5];
	piM=	      ((IDL_LONG**)argp)[6];
	piDir=        ((IDL_LONG**)argp)[7];

/*
	Christian,

	Here's the code.  on input,

	x[0] = hour angle in hours
	x[1] = declination in degrees
	q    = array of model parameters.  a reasonable set of values is
		{ 88.570, 18.125, 267.145., 90.214, 0., 91.856, 0., 48.840 }
	npar = 8

	on output

	y[0] = siderostat azimuth (in degrees)
	y[1] = siderostat elevation (in degrees)

	This file is ~dm/alg.c on fornax

	Dave
*/
	if (piDir[0] == 1) {
		for(i=0;i<piN[0];i++){
			status=get_star(pdfAz+i,pdfEl+i,pdfHA+i,pdfDec+i,
			      pdfModel+0+i*piM[0],pdfModel+1+i*piM[0],
			      pdfModel+2+i*piM[0],pdfModel+3+i*piM[0],
			      pdfModel+4+i*piM[0],pdfModel+5+i*piM[0],
			      pdfModel+6+i*piM[0],pdfModel+7+i*piM[0]);
			}
		}
	else {
		npar=8;
		for(i=0;i<piN[0];i++){
			x[0]=pdfHA[i];
			x[1]=pdfDec[i];
			status=sid_funcs(x,pdfModel+i*piM[0],y,dyda,npar);
			pdfAz[i]=y[0];
			pdfEl[i]=y[1];
			}
		} 
	return((int)status);
	}

/*----------------------------------------------------------------------------*/
int spec_parse(char *azSpectrum,
	float *pfType1,float *pfType2,float *pfClass1,float *pfClass2){
	/* 
	Parse a spectral classification into integer types and classes.
	*/

	long i;
	char *loc;
	long iLen,iPos,iPos1,iPos2,iPosBck;
	float fType,fSubclass,fLclass;

	iLen=strlen(azSpectrum);

	iPos=0;
	while(iPos<iLen){
		if((azSpectrum[iPos]>=65)&&(azSpectrum[iPos]<=90)){
			*pfType1=+100;
			switch (azSpectrum[iPos]){
				case 'W':{*pfType1= 0;iPos+=1;break;}
				case 'O':{*pfType1= 0;break;}
				case 'B':{*pfType1=10;break;}
				case 'A':{*pfType1=20;break;}
				case 'F':{*pfType1=30;break;}
				case 'G':{*pfType1=40;break;}
				case 'K':{*pfType1=50;break;}
				case 'M':{*pfType1=60;break;}
				case 'S':{*pfType1=60;break;}
				case 'C':{*pfType1=70;break;}
				case 'N':{*pfType1=70;break;}
				}
			fSubclass=(float)azSpectrum[iPos+1]-48;
			if((fSubclass>9)||(fSubclass<0)){
				fSubclass=5;
				iPos-=1;
				}
			*pfType1+=fSubclass;
			if(*pfType1<0 || *pfType1>100)*pfType1=0;
			iPos+=2;
			if(azSpectrum[iPos]==58)iPos+=1;
			if(azSpectrum[iPos]==46){
				*pfType1+=(float)(azSpectrum[iPos+1]-48)/10;
				iPos+=2;
				}
			else if(((azSpectrum[iPos]==45)||(azSpectrum[iPos]==47))
				&&(azSpectrum[iPos+1]<65)){
				*pfType1-=fSubclass;
				fSubclass=(fSubclass+(float)(azSpectrum[iPos+1]-48))/2;
				*pfType1+=fSubclass;
				iPos+=2;
				}
			else if((azSpectrum[iPos]==45)||(azSpectrum[iPos]==47)){
				fType=+100;
				iPosBck=iPos;
				switch (azSpectrum[iPos+1]){
					case 'W':{fType= 0;iPos+=1;break;}
					case 'O':{fType= 0;break;}
					case 'B':{fType=10;break;}
					case 'A':{fType=20;break;}
					case 'F':{fType=30;break;}
					case 'G':{fType=40;break;}
					case 'K':{fType=50;break;}
					case 'M':{fType=60;break;}
					case 'S':{fType=60;break;}
					case 'C':{fType=70;break;}
					case 'N':{fType=70;break;}
					}
				fSubclass=(float)(azSpectrum[iPos+2]-48);
				fType+=fSubclass;
				iPos+=3;
				if(azSpectrum[iPos]==46){
					fType+=(float)(azSpectrum[iPos+1]-48)/10;
					iPos+=2;
					}
				if((fType<0)||(fType>100)){
					fType=*pfType1;
					iPos=iPosBck;
					while(iPos<iLen){
						if((azSpectrum[iPos]==73)||(azSpectrum[iPos]==86))break;
						iPos+=1;
						}
					}
				*pfType1=(*pfType1+fType)/2;
				}
			i=0;
			if(strncmp(azSpectrum+iPos,"I",1)==0){*pfClass1=1.0;i=1;}
			if(strncmp(azSpectrum+iPos,"Ia",2)==0){*pfClass1=1.0;i=2;}
			if(strncmp(azSpectrum+iPos,"Iab",3)==0){*pfClass1=1.25;i=3;}
			if(strncmp(azSpectrum+iPos,"Ib",2)==0){*pfClass1=1.5;i=2;}
			if(strncmp(azSpectrum+iPos,"II",2)==0){*pfClass1=2.0;i=2;}
			if(strncmp(azSpectrum+iPos,"II-III",6)==0){*pfClass1=2.5;i=6;}
			if(strncmp(azSpectrum+iPos,"III",3)==0){*pfClass1=3.0;i=3;}
			if(strncmp(azSpectrum+iPos,"IV",2)==0){*pfClass1=4.0;i=2;}
			if(strncmp(azSpectrum+iPos,"IV-V",4)==0){*pfClass1=4.5;i=4;}
			if(strncmp(azSpectrum+iPos,"V",1)==0){*pfClass1=5.0;i=1;}
			fLclass=0.0;
			if((azSpectrum[iPos+i]==45)||(azSpectrum[iPos+i]==47)){
				if(strncmp(azSpectrum+iPos+i+1,"I",1)==0)fLclass=1.0;
				if(strncmp(azSpectrum+iPos+i+1,"Ia",2)==0)fLclass=1.0;
				if(strncmp(azSpectrum+iPos+i+1,"Iab",3)==0)fLclass=1.25;
				if(strncmp(azSpectrum+iPos+i+1,"Ib",2)==0)fLclass=1.5;
				if(strncmp(azSpectrum+iPos+i+1,"II",2)==0)fLclass=2.0;
				if(strncmp(azSpectrum+iPos+i+1,"II-III",6)==0)fLclass=2.5;
				if(strncmp(azSpectrum+iPos+i+1,"III",3)==0)fLclass=3.0;
				if(strncmp(azSpectrum+iPos+i+1,"IV",2)==0)fLclass=4.0;
				if(strncmp(azSpectrum+iPos+i+1,"IV-V",4)==0)fLclass=4.5;
				if(strncmp(azSpectrum+iPos+i+1,"V",1)==0)fLclass=5.0;
				*pfClass1=(*pfClass1+fLclass)/2;
				}
			break;
			}
		iPos++;
		}
	loc=strchr(azSpectrum,'+');
	iPos=loc-azSpectrum;
	while((loc!=NULL)&&(iPos<iLen)){
		if((azSpectrum[iPos]>=65)&&(azSpectrum[iPos]<=90)){
			*pfType2=+100;
			switch (azSpectrum[iPos]){
				case 'W':{*pfType2= 0;iPos+=1;;break;}
				case 'O':{*pfType2= 0;break;}
				case 'B':{*pfType2=10;break;}
				case 'A':{*pfType2=20;break;}
				case 'F':{*pfType2=30;break;}
				case 'G':{*pfType2=40;break;}
				case 'K':{*pfType2=50;break;}
				case 'M':{*pfType2=60;break;}
				case 'S':{*pfType2=60;break;}
				case 'C':{*pfType2=70;break;}
				case 'N':{*pfType2=70;break;}
				}
			if((azSpectrum[iPos+1]==45)||(azSpectrum[iPos+1]==47)){
				iPos+=1;
				fSubclass=0;
				fType=+100;
				switch (azSpectrum[iPos+1]){
					case 'W':{fType= 0;iPos+=1;break;}
					case 'O':{fType= 0;break;}
					case 'B':{fType=10;break;}
					case 'A':{fType=20;break;}
					case 'F':{fType=30;break;}
					case 'G':{fType=40;break;}
					case 'K':{fType=50;break;}
					case 'M':{fType=60;break;}
					case 'S':{fType=60;break;}
					case 'C':{fType=70;break;}
					case 'N':{fType=70;break;}
					}
				*pfType2=(*pfType2+fType)/2;
				}
			else {
				fSubclass=(float)azSpectrum[iPos+1]-48;
				}
			if((fSubclass>9)||(fSubclass<0)){
				fSubclass=5;
				iPos-=1;
				}
			*pfType2+=fSubclass;
			if(*pfType2<0 || *pfType2>100)*pfType2=0;
			iPos+=2;
			if(azSpectrum[iPos]==58)iPos+=1;
			if(azSpectrum[iPos]==46){
				*pfType2+=(float)(azSpectrum[iPos+1]-48)/10;
				iPos+=2;
				}
			else if(((azSpectrum[iPos]==45)||(azSpectrum[iPos]==47))
				&&(azSpectrum[iPos+1]<65)){
				*pfType2-=fSubclass;
				fSubclass=(fSubclass+(float)(azSpectrum[iPos+1]-48))/2;
				*pfType2+=fSubclass;
				iPos+=2;
				}
			else if((azSpectrum[iPos]==45)||(azSpectrum[iPos]==47)){
				fType=+100;
				switch (azSpectrum[iPos+1]){
					case 'W':{fType= 0;iPos+=1;break;}
					case 'O':{fType= 0;break;}
					case 'B':{fType=10;break;}
					case 'A':{fType=20;break;}
					case 'F':{fType=30;break;}
					case 'G':{fType=40;break;}
					case 'K':{fType=50;break;}
					case 'M':{fType=60;break;}
					case 'S':{fType=60;break;}
					case 'C':{fType=70;break;}
					case 'N':{fType=70;break;}
					}
				fSubclass=(float)(azSpectrum[iPos+2]-48);
				fType+=fSubclass;
				iPos+=3;
				if(azSpectrum[iPos]==46){
					fType+=(float)(azSpectrum[iPos+1]-48)/10;
					iPos+=2;
					}
				if((fType<0)||(fType>100)){
					fType=*pfType2;
					iPos=iPosBck;
					while(iPos<iLen){
						if((azSpectrum[iPos]==73)||(azSpectrum[iPos]==86))break;
						iPos+=1;
						}
					}
				*pfType2=(*pfType2+fType)/2;
				}
			i=0;
			if(strncmp(azSpectrum+iPos,"I",1)==0){*pfClass2=1.0;i=1;}
			if(strncmp(azSpectrum+iPos,"Ia",2)==0){*pfClass2=1.0;i=2;}
			if(strncmp(azSpectrum+iPos,"Iab",3)==0){*pfClass2=1.25;i=3;}
			if(strncmp(azSpectrum+iPos,"Ib",2)==0) {*pfClass2=1.5;i=2;}
			if(strncmp(azSpectrum+iPos,"II",2)==0){*pfClass2=2.0;i=2;}
			if(strncmp(azSpectrum+iPos,"II-III",6)==0){*pfClass1=2.5;i=6;}
			if(strncmp(azSpectrum+iPos,"III",3)==0){*pfClass2=3.0;i=3;}
			if(strncmp(azSpectrum+iPos,"IV",2)==0){*pfClass2=4.0;i=2;}
			if(strncmp(azSpectrum+iPos,"IV-V",2)==0){*pfClass2=4.5;i=4;}
			if(strncmp(azSpectrum+iPos,"V",1)==0){*pfClass2=5.0;i=1;}
			fLclass=0.0;
			if((azSpectrum[iPos+i]==45)||(azSpectrum[iPos+i]==47)){
				if(strncmp(azSpectrum+iPos+i+1,"I",1)==0)fLclass=1.0;
				if(strncmp(azSpectrum+iPos+i+1,"Ia",2)==0)fLclass=1.0;
				if(strncmp(azSpectrum+iPos+i+1,"Iab",3)==0)fLclass=1.25;
				if(strncmp(azSpectrum+iPos+i+1,"Ib",2)==0)fLclass=1.5;
				if(strncmp(azSpectrum+iPos+i+1,"II",2)==0)fLclass=2.0;
				if(strncmp(azSpectrum+iPos+i+1,"II-III",6)==0)fLclass=2.5;
				if(strncmp(azSpectrum+iPos+i+1,"III",3)==0)fLclass=3.0;
				if(strncmp(azSpectrum+iPos+i+1,"IV",2)==0)fLclass=4.0;
				if(strncmp(azSpectrum+iPos+i+1,"IV-V",4)==0)fLclass=4.5;
				if(strncmp(azSpectrum+iPos+i+1,"V",1)==0)fLclass=5.0;
				*pfClass2=(*pfClass2+fLclass)/2;
				}
			break;
			}
		iPos++;
		}
	return(0);
	}
/*----------------------------------------------------------------------------*/
int countflag(float fValue, float fFlag){
	/*
	Tell the calling function whether to increment the flag counter.
	*/

	if(fFlag==1){if(fValue>0)return(1); else return(0);}
	else        {if(fValue<0)return(1); else return(0);}
	}
/*----------------------------------------------------------------------------*/
int IDL_ChartrStore(unsigned char *dest, unsigned char *source, int numchar){

	int i;
	
	for(i=0;i<numchar;i++){
		dest[i]=source[i];
		}
	}
/*----------------------------------------------------------------------------*/
