/*******************************************************************************
hds_wrap.c

USNO/NRL Optical Interferometer
3450 Massachusetts Avenue NW
Washington DC 20392-5400

System: Data Reduction
Subsystem: CHAMELEON

Description:
Package of C-routines for HDS access callable from PV-WAVE via LINKNLOAD.

Modification history:
05-Feb-1994  C.A.Hummel, file created
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <sae_par.h>
#include <dat_par.h>

#include "export.h"

#define MAXDIM 7
#define MAXLEVELS 10

int status;
int len0,len1,len2,len3;

int LocatorID = -1;
char LocatorStack[MAXLEVELS][DAT__SZLOC];

/* Function prototypes */

int hds_state_(IDL_LONG *piState,int *status);
int hds_new_(  char *name,char *name2,char *type,IDL_LONG *piNdim,IDL_LONG *piDims,char *LS,int *status,int len1,int len2,int len3,int len4);
int hds_open_( char *name,char *mode,char *LS,int *status,int len1,int len2,int len3); 
int hds_start_(int *status);
int hds_stop_( int *status);
int dat_annul_(char *LS,int *status,int len0);
int cmp_shape_(char *LS,char *name,int *iMaxDim,IDL_LONG *piDims,IDL_LONG *piNdim,int *status,int len0,int len1);
int cmp_get0c_(char *LS,char *name,char *DATA,int *status,int len0,int len1,IDL_LONG piLen);
int cmp_get0i_(char *LS,char *name,IDL_LONG *value,int *status,int len0,int len1);
int cmp_get0r_(char *LS,char *name,float *value,int *status,int len0,int len1);
int cmp_get0d_(char *LS,char *name,double *value,int *status,int len0,int len1);
int cmp_get1c_(char *LS,char *name,IDL_LONG *piNum,char *Data,IDL_LONG *piNum2,int *status,int len0,int len1,IDL_LONG piLen);
int cmp_get1i_(char *LS,char *name,IDL_LONG *piNum,IDL_LONG *piData,IDL_LONG *piNum2,int *status,int len0,int len1);
int cmp_get1r_(char *LS,char *name,IDL_LONG *piNum,float *pfData,IDL_LONG *piNum2,int *status,int len0,int len1);
int cmp_get1d_(char *LS,char *name,IDL_LONG *piNum,double *pdfData,IDL_LONG *piNum2,int *status,int len0,int len1);
int cmp_getnc_(char *LS,char *name,IDL_LONG *piNdim,IDL_LONG *piDims,char *Data,IDL_LONG *piDims2,int *status,int len0,int len1,IDL_LONG piLen);
int cmp_getni_(char *LS,char *name,IDL_LONG *piNdim,IDL_LONG *piDims,IDL_LONG *piData,IDL_LONG *piDims2,int *status,int len0,int len1);
int cmp_getnr_(char *LS,char *name,IDL_LONG *piNdim,IDL_LONG *piDims,float *pfData,IDL_LONG *piDims2,int *status,int len0,int len1);
int cmp_getnd_(char *LS,char *name,IDL_LONG *piNdim,IDL_LONG *piDims,double *pdfData,IDL_LONG *piDims2,int *status,int len0,int len1);
int cmp_put0c_(char *LS,char *name,char *DATA,int *status,int len0,int len1,int len2);
int cmp_put0i_(char *LS,char *name,IDL_LONG *piValue,int *status,int len0,int len1);
int cmp_put0r_(char *LS,char *name,float *pfValue,int *status,int len0,int len1);
int cmp_put0d_(char *LS,char *name,double *pdfValue,int *status,int len0,int len1);
int cmp_put1c_(char *LS,char *name,IDL_LONG *piNum,char *Data,int *status,int len0,int len1,IDL_LONG piLen);
int cmp_put1i_(char *LS,char *name,IDL_LONG *piNum,IDL_LONG *piData,int *status,int len0,int len1);
int cmp_put1r_(char *LS,char *name,IDL_LONG *piNum,float *pfData,int *status,int len0,int len1);
int cmp_put1d_(char *LS,char *name,IDL_LONG *piNum,double* pdfData,int *status,int len0,int len1);
int cmp_putnc_(char *LS,char *name,IDL_LONG *piNdim,IDL_LONG *piDims,char *Data,IDL_LONG *piDims2,int *status,int len0,int len1, IDL_LONG piLen);
int cmp_putni_(char *LS,char *name,IDL_LONG *piNdim,IDL_LONG *piDims,IDL_LONG *piData,IDL_LONG *piDims2,int *status,int len0,int len1);
int cmp_putnr_(char *LS,char *name,IDL_LONG *piNdim,IDL_LONG *piDims,float *pfData,IDL_LONG *piDims2,int *status,int len0,int len1);
int cmp_putnd_(char *LS,char *name,IDL_LONG *piNdim,IDL_LONG *piDims,double *pdfData,IDL_LONG *piDims2,int *status,int len0,int len1);
int dat_there_(char *LS,char *name,IDL_LONG *piReply,int *status,int len0,int len1);
int dat_find_( char *LS,char *name,char *loc,int *status,int len0,int len1,int len2);
int dat_name_( char *LS,char *name,int *status,int len0,int len1);
int dat_type_( char *LS,char *type,int *status,int len0,int len1);
int dat_ncomp_(char *LS,IDL_LONG *ncomp,int *status,int len0);
int dat_prim_( char *LS,IDL_LONG *piReply,int *status,int len0);
int dat_size_( char *LS,IDL_LONG *piSize,int *status,int len0);
int dat_shape_(char *LS,int *iMaxDim,IDL_LONG *piDims,IDL_LONG *piNdim,int *status,int len0);
int dat_clen_( char *LS,IDL_LONG *piLen,int *status,int len0);
int dat_cell_( char *LS,IDL_LONG *piNdim,IDL_LONG *piCell,char *loc,int *status,int len0,int len1);
int dat_index_(char *LS,IDL_LONG *index,char *loc,int *status,int len0,int len1);
int dat_new_(  char *LS,char *name,char *type,IDL_LONG *piNdim,IDL_LONG *piDims,int *status,int len0,int len1,int len2);
int dat_new0c_(char *LS,char *name,IDL_LONG *piLen,int *status,int len0,int len1);
int dat_slice_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDiml,IDL_LONG *piDimu,char *loc,int *status,int len0,int len1);
int dat_alter_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDims,int *status,int len0);
int dat_erase_(char *LS,char *name,int *status,int len0,int len1);
int dat_get_(  char *LS,char *type,IDL_LONG *piNdim,IDL_LONG *piDims,unsigned char* Data,int *status,int len0,int len1,IDL_LONG piLen);
int dat_get0c_(char *LS,char *value,int *status,int len0,IDL_LONG piLen);
int dat_get0i_(char *LS,IDL_LONG *value,int *status,int len0);
int dat_get0r_(char *LS,float *value,int *status,int len0);
int dat_get0d_(char *LS,double *value,int *status,int len0);
int dat_get1c_(char *LS,IDL_LONG *piNum,char *Data,IDL_LONG *piNum2,int *status,int len0,IDL_LONG piLen);
int dat_get1i_(char *LS,IDL_LONG *piNum,IDL_LONG *piData,IDL_LONG *piNum2,int *status,int len0);
int dat_get1r_(char *LS,IDL_LONG *piNum,float *pfData,IDL_LONG *piNum2,int *status,int len0);
int dat_get1d_(char *LS,IDL_LONG *piNum,double *pdfData,IDL_LONG *piNum2,int *status,int len0);
int dat_getnc_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDims,char *Data,IDL_LONG *piDims2,int *status,int len0,IDL_LONG piLen);
int dat_getni_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDims,IDL_LONG *piData,IDL_LONG *piDims2,int *status,int len0);
int dat_getnr_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDims,float *pfData,IDL_LONG *piDims2,int *status,int len0);
int dat_getnd_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDims,double *pdfData,IDL_LONG *piDims2,int *status,int len0);
int dat_put0c_(char *LS,char *DATA,int *status,int len0,int len1);
int dat_put1c_(char *LS,IDL_LONG *piNum,char *Data,int *status,int len0,IDL_LONG piLen);
int dat_put1i_(char *LS,IDL_LONG *piNum,IDL_LONG *piData,int *status,int len0);
int dat_put1r_(char *LS,IDL_LONG *piNum,float *pfData,int *status,int len0);
int dat_put1d_(char *LS,IDL_LONG *piNum,double *pdfData,int *status,int len0);
int dat_putnc_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDims,char *Data,IDL_LONG *piDims2,int *status,int len0,IDL_LONG piLen);
int dat_putni_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDims,IDL_LONG *piData,IDL_LONG *piDims2,int *status,int len0);
int dat_putnr_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDims,float *pfData,IDL_LONG *piDims2,int *status,int len0);
int dat_putnd_(char *LS,IDL_LONG *piNdim,IDL_LONG *piDims,double *pdfData,IDL_LONG *piDims2,int *status,int len0);

/*----------------------------------------------------------------------------*/
int clearstatus(void){
	/*
	Clears status variable (= set to SAI__OK).
	*/

	status=SAI__OK;

	return(status);
	}
/*----------------------------------------------------------------------------*/
int toplevel(void){
	/* 
	Move to the top of the stack, releasing lower level locators.
	*/

	while(LocatorID > 0) {
		dat_annul_(LocatorStack[LocatorID],&status,len0);
		LocatorID -= 1;
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int hds_state(int argc, char *argp[]){
	/*
	Check HDS state.
	*/
	IDL_LONG *piState;

	if (argc !=1) {
                fprintf(stderr,"\nWrong # of parameters(%d)",argc);
                return(1);
                }

	piState=((IDL_LONG**)argp)[0];

	hds_state_(piState,&status);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int hds_new(int argc, char *argp[]){
	/*
	Create a new container file and return locator.
	*/
	IDL_STRING *file,*name,*type;
	IDL_LONG *piDims,*piNdim;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	file=((IDL_STRING**)argp)[0];len1=strlen(file[0].s);
	name=((IDL_STRING**)argp)[1];len2=strlen(name[0].s);
	type=((IDL_STRING**)argp)[2];len3=strlen(type[0].s);
	piNdim=((IDL_LONG**)argp)[3];
	piDims=((IDL_LONG**)argp)[4];

	len0=DAT__SZLOC;
	status=SAI__OK; 

	LocatorID += 1;
	hds_new_(file[0].s,name[0].s,type[0].s,piNdim,piDims,LocatorStack[LocatorID],
			&status,len1,len2,len3,len0);
	return(status);
	}
/*----------------------------------------------------------------------------*/
int hds_open(int argc, char *argp[]){
	/* 
	Open HDS file and return locator.
	*/
	IDL_STRING *file,*mode;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	file=((IDL_STRING**)argp)[0];
	mode=((IDL_STRING**)argp)[1];

	/* Start HDS */
	status=SAI__OK; 
	hds_start_(&status);

	/* Open existing container file */
	len0=DAT__SZLOC;
	len1=strlen(file[0].s);
	len2=strlen(mode[0].s);
	LocatorID += 1;
	hds_open_(file[0].s,mode[0].s,LocatorStack[LocatorID],&status,len1,len2,len0); 

	if (status != 0) {
		hds_stop_(&status);
		LocatorID -= 1;
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_shape(int argc, char *argp[]){
	/*
	Enquire the shape of an object.
	*/
	IDL_STRING *nam;
	IDL_LONG *piDims,*piNdim;
	int iMaxDim;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piDims=((IDL_LONG**)argp)[1];
	piNdim=((IDL_LONG**)argp)[2];

	iMaxDim=7;
	cmp_shape_(LocatorStack[LocatorID],nam[0].s,&iMaxDim,piDims,piNdim,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get0c(int argc, char *argp[]){
	/* 
	Read character scalar component.
	*/
	IDL_STRING *nam,*data;
	IDL_LONG *piLen;
	char *pos;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	data=((IDL_STRING**)argp)[1];
	piLen= ((IDL_LONG**)argp)[2];

	cmp_get0c_(LocatorStack[LocatorID],nam[0].s,data[0].s,&status,len0,len1,*piLen);
	if((pos=(char*)memchr(data[0].s,'\0',*piLen))!=NULL){
		memset(pos,' ',data[0].s+(*piLen)-pos);
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get0i(int argc, char *argp[]){
	/*
	Read integer scalar component.
	*/
	IDL_STRING *nam;
	IDL_LONG *value;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	value=((IDL_LONG**)argp)[1];

	cmp_get0i_(LocatorStack[LocatorID],nam[0].s,value,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get0r(int argc, char *argp[]){
	/*
	Read real scalar component.
	*/
	IDL_STRING *nam;
	float *value;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=  ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	value=((float**)argp)[1];

	cmp_get0r_(LocatorStack[LocatorID],nam[0].s,value,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get0d(int argc, char *argp[]){
	/*
	Read double scalar component.
	*/
	IDL_STRING *nam;
	double *value;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	value=((double**)argp)[1];

	cmp_get0d_(LocatorStack[LocatorID],nam[0].s,value,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get1c(int argc, char *argp[]){
	/* 
	Read character vector component.
	*/
	IDL_STRING *nam,*data;
	IDL_LONG *piNum,*piLen;
	char *Data,*pos;
	int i;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum= ((IDL_LONG**)argp)[1];
	data=((IDL_STRING**)argp)[2];
	piLen= ((IDL_LONG**)argp)[3];

	if((Data=(char*)malloc(*piLen*(*piNum)))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_get1c!");
		return(1);
		}

	cmp_get1c_(LocatorStack[LocatorID],nam[0].s,piNum,Data,piNum,&status,len0,len1,*piLen);
	for(i=0;i<*piNum;i++){
		memcpy(data[i].s,Data+i*(*piLen),*piLen);
		while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
			memset(pos,' ',*piLen-(pos-data[i].s));
			}
		}
	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get1i(int argc, char *argp[]){
	/*
	Read integer vector component.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNum,*piData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum= ((IDL_LONG**)argp)[1];
	piData=((IDL_LONG**)argp)[2];

	cmp_get1i_(LocatorStack[LocatorID],nam[0].s,piNum,piData,piNum,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get1r(int argc, char *argp[]){
	/*
	Read real vector component.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNum;
	float *pfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=   ((IDL_LONG**)argp)[1];
	pfData=((float**)argp)[2];

	cmp_get1r_(LocatorStack[LocatorID],nam[0].s,piNum,pfData,piNum,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_get1d(int argc, char *argp[]){
	/*
	Read double vector component.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNum;
	double *pdfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=     ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=     ((IDL_LONG**)argp)[1];
	pdfData=((double**)argp)[2];

	cmp_get1d_(LocatorStack[LocatorID],nam[0].s,piNum,pdfData,piNum,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_getnc(int argc, char *argp[]){
	/* 
	Read character array component.
	*/
	IDL_STRING *nam,*data;
	IDL_LONG *piNdim,*piDims,*piLen;
	char *Data,*pos;
	int i,n;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=((IDL_LONG**)argp)[1];
	piDims=((IDL_LONG**)argp)[2];
	data=((IDL_STRING**)argp)[3];
	piLen= ((IDL_LONG**)argp)[4];

	n=1;
	for(i=0;i<*piNdim;i++)n*=piDims[i];

	if((Data=(char*)malloc(*piLen*n))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_getnc!");
		return(1);
		}

	cmp_getnc_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,Data,piDims,
			&status,len0,len1,*piLen);
	for(i=0;i<n;i++){
		memcpy(data[i].s,Data+i*(*piLen),*piLen);
		while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
			memset(pos,' ',*piLen-(pos-data[i].s));
			}
		}
	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_getni(int argc, char *argp[]){
	/*
	Read a named integer array primitive.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNdim,*piDims;
	IDL_LONG *piData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=((IDL_LONG**)argp)[1];
	piDims=((IDL_LONG**)argp)[2];
	piData=((IDL_LONG**)argp)[3];

	cmp_getni_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,(IDL_LONG*)piData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_getnr(int argc, char *argp[]){
	/*
	Read a named real array primitive.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNdim,*piDims;
	float *pfData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=  ((IDL_LONG**)argp)[1];
	piDims=  ((IDL_LONG**)argp)[2];
	pfData=((float**)argp)[3];

	cmp_getnr_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,(float*)pfData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_getnd(int argc, char *argp[]){
	/*
	Read a named double array primitive.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNdim,*piDims;
	double *pdfData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=     ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=    ((IDL_LONG**)argp)[1];
	piDims=    ((IDL_LONG**)argp)[2];
	pdfData=((double**)argp)[3];

	cmp_getnd_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,(double*)pdfData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put0c(int argc, char *argp[]){
	/*
	Write a scalar string component.
	*/
	IDL_STRING *name,*data;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	name=((IDL_STRING**)argp)[0];len1=strlen(name[0].s);
	data=((IDL_STRING**)argp)[1];len2=strlen(data[0].s);

	cmp_put0c_(LocatorStack[LocatorID],name[0].s,data[0].s,&status,len0,len1,len2);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put0i(int argc, char *argp[]){
	/*
	Write a scalar integer component.
	*/
	IDL_STRING *nam;
	IDL_LONG *piValue;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=    ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piValue=  ((IDL_LONG**)argp)[1];

	cmp_put0i_(LocatorStack[LocatorID],nam[0].s,piValue,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put0r(int argc, char *argp[]){
	/*
	Write a scalar float component.
	*/
	IDL_STRING *nam;
	float *pfValue;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=    ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	pfValue=((float**)argp)[1];

	cmp_put0r_(LocatorStack[LocatorID],nam[0].s,pfValue,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put0d(int argc, char *argp[]){
	/*
	Write a scalar double component.
	*/
	IDL_STRING *nam;
	double *pdfValue;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=      ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	pdfValue=((double**)argp)[1];

	cmp_put0d_(LocatorStack[LocatorID],nam[0].s,pdfValue,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put1c(int argc, char *argp[]){
	/*
	Write a vector string component.
	*/
	IDL_STRING *nam,*data;
	IDL_LONG *piNum,*piLen;
	char *Data;
	int i;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=  ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=  ((IDL_LONG**)argp)[1];
	data= ((IDL_STRING**)argp)[2];
	piLen=  ((IDL_LONG**)argp)[3];

	if((Data=(char*)malloc(*piLen*(*piNum)))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_put1c!");
		return(1);
		}

	for(i=0;i<*piNum;i++){
		memcpy(Data+i*(*piLen),data[i].s,*piLen);
		}

	cmp_put1c_(LocatorStack[LocatorID],nam[0].s,piNum,Data,&status,len0,len1,*piLen);

	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put1i(int argc, char *argp[]){
	/*
	Write integer vector component.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNum,*piData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum= ((IDL_LONG**)argp)[1];
	piData=((IDL_LONG**)argp)[2];

	cmp_put1i_(LocatorStack[LocatorID],nam[0].s,piNum,piData,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put1r(int argc, char *argp[]){
	/*
	Write float vector component.
	*/
	IDL_STRING *nam;
	float *pfData;
	IDL_LONG *piNum;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=   ((IDL_LONG**)argp)[1];
	pfData=((float**)argp)[2];

	cmp_put1r_(LocatorStack[LocatorID],nam[0].s,piNum,pfData,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_put1d(int argc, char *argp[]){
	/*
	Write double vector component.
	*/
	IDL_STRING *nam;
	double *pdfData;
	IDL_LONG *piNum;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=     ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNum=     ((IDL_LONG**)argp)[1];
	pdfData=((double**)argp)[2];

	cmp_put1d_(LocatorStack[LocatorID],nam[0].s,piNum,pdfData,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_putnc(int argc, char *argp[]){
	/* 
	Write character array component.
	*/
	IDL_STRING *nam,*data;
	IDL_LONG *piNdim,*piDims,*piLen;
	char *Data;
	int i,j,n;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=((IDL_LONG**)argp)[1];
	piDims=((IDL_LONG**)argp)[2];
	data=((IDL_STRING**)argp)[3];
	piLen= ((IDL_LONG**)argp)[4];

	n=1;
	for(i=0;i<*piNdim;i++)n*=piDims[i];

	if((Data=(char*)malloc(*piLen*n))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_putnc!");
		return(1);
		}

	for(i=0;i<n;i++){
		memcpy(Data+i*(*piLen),data[i].s,*piLen);
		}

	cmp_putnc_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,Data,piDims,
			&status,len0,len1,*piLen);

	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_putni(int argc, char *argp[]){
	/*
	Write an integer array primitive.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNdim,*piDims;
	IDL_LONG *piData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=((IDL_LONG**)argp)[1];
	piDims=((IDL_LONG**)argp)[2];
	piData=((IDL_LONG**)argp)[3];

	cmp_putni_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,piData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_putnr(int argc, char *argp[]){
	/*
	Write a float array primitive.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNdim,*piDims;
	float *pfData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=   ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=  ((IDL_LONG**)argp)[1];
	piDims=  ((IDL_LONG**)argp)[2];
	pfData=((float**)argp)[3];

	cmp_putnr_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,pfData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int cmp_putnd(int argc, char *argp[]){
	/*
	Write a double array primitive.
	*/
	IDL_STRING *nam;
	IDL_LONG *piNdim,*piDims;
	double *pdfData;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=     ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piNdim=    ((IDL_LONG**)argp)[1];
	piDims=    ((IDL_LONG**)argp)[2];
	pdfData=((double**)argp)[3];

	cmp_putnd_(LocatorStack[LocatorID],nam[0].s,piNdim,piDims,pdfData,piDims,
			&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_there(int argc, char *argp[]){
	/*
	Enquire component existence.
	*/
	IDL_STRING *nam;
	IDL_LONG *piReply;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=  ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piReply=((IDL_LONG**)argp)[1];

	dat_there_(LocatorStack[LocatorID],nam[0].s,piReply,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_find(int argc, char *argp[]){
	/*
	Return locator to named component.
	*/
	IDL_STRING *nam;
	char loc[DAT__SZLOC];

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);

	dat_find_(LocatorStack[LocatorID],nam[0].s,loc,&status,len0,len1,len0);
	LocatorID += 1;
	memcpy(LocatorStack[LocatorID],loc,DAT__SZLOC);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_name(int argc, char *argp[]){
	/*
	Get name of object.
	*/
	IDL_STRING *nam;
	unsigned char *nam_b;
	unsigned char word[15];
	int i;

	len0=DAT__SZLOC;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
/* 	Swap commenting to activate old version */
	nam=((IDL_STRING**)argp)[0];
/*	nam_b=((char**)argp)[0];*/

/* 	Swap commenting to activate old version */
	dat_name_(LocatorStack[LocatorID],nam[0].s,&status,len0,len0);
/*	dat_name_(LocatorStack[LocatorID],word,&status,len0,len0);*/

/*	Uncomment for old version */
	/*
	for (i=0;i<len0-1;i++) {
		nam_b[i]=word[i];
		}
	*/

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_type(int argc, char *argp[]){
	/*
	Enquire object type.
	*/
	IDL_STRING *typ;
	unsigned char *typ_b;
	unsigned char word[15];
	int i;

	len0=DAT__SZLOC;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
/* 	Swap commenting to activate old version */
	typ=((IDL_STRING**)argp)[0];
/*	typ_b=((char**)argp)[0];*/

/* 	Swap commenting to activate old version */
	dat_type_(LocatorStack[LocatorID],typ[0].s,&status,len0,len0);
/*	dat_type_(LocatorStack[LocatorID],word,&status,len0,len0);*/

/*	Uncomment for old version */
	/*
	for (i=0;i<len0-1;i++) {
		typ_b[i]=word[i];
		}
	*/

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_ncomp(int argc, char *argp[]){
	/*
	Get number of components.
	*/
	IDL_LONG *ncomp;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	ncomp=((IDL_LONG**)argp)[0];

	dat_ncomp_(LocatorStack[LocatorID],ncomp,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_prim(int argc, char *argp[]){
	/*
	Enquire object primitive.
	*/
	IDL_LONG *piReply;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piReply=((IDL_LONG**)argp)[0];

	dat_prim_(LocatorStack[LocatorID],piReply,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_size(int argc, char *argp[]){
	/*
	Enquire object size.
	*/
	IDL_LONG *piSize;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piSize=((IDL_LONG**)argp)[0];

	dat_size_(LocatorStack[LocatorID],piSize,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_shape(int argc, char *argp[]){
	/*
	Enquire the shape of an object.
	*/
	IDL_LONG *piDims,*piNdim;
	int iMaxDim;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piDims=((IDL_LONG**)argp)[0];
	piNdim=((IDL_LONG**)argp)[1];

	iMaxDim=MAXDIM;
	dat_shape_(LocatorStack[LocatorID],&iMaxDim,piDims,piNdim,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_clen(int argc, char *argp[]){
	/*
	Obtain character string length.
	*/
	IDL_LONG *piLen;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piLen=((IDL_LONG**)argp)[0];

	dat_clen_(LocatorStack[LocatorID],piLen,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_cell(int argc, char *argp[]){
	/*
	Return locator to a cell of an array object.
	*/
	char loc[DAT__SZLOC];
	IDL_LONG *piNdim,*piCell;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((IDL_LONG**)argp)[0];
	piCell=((IDL_LONG**)argp)[1];

	dat_cell_(LocatorStack[LocatorID],piNdim,piCell,loc,&status,len0,len0);
	LocatorID += 1;
	memcpy(LocatorStack[LocatorID],loc,DAT__SZLOC);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_index(int argc, char *argp[]){
	/*
	Return locator to item in component list.
	*/
	char loc[DAT__SZLOC];
	IDL_LONG *index;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	index= ((IDL_LONG**)argp)[0];

	dat_index_(LocatorStack[LocatorID],index,loc,&status,len0,len0);
	LocatorID += 1;
	memcpy(LocatorStack[LocatorID],loc,DAT__SZLOC);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_new(int argc, char *argp[]){
	/*
	Create a new component in a structure.
	*/
	IDL_STRING *nam,*typ;
	IDL_LONG *piNdim,*piDims;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam= ((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	typ= ((IDL_STRING**)argp)[1];len2=strlen(typ[0].s);
	piNdim=((IDL_LONG**)argp)[2];
	piDims=((IDL_LONG**)argp)[3];

	dat_new_(LocatorStack[LocatorID],nam[0].s,typ[0].s,piNdim,piDims,&status,len0,len1,len2);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_new0c(int argc, char *argp[]){
	/*
	Create scalar string component.
	*/
	IDL_STRING *nam;
	IDL_LONG *piLen;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	nam=((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);
	piLen=((IDL_LONG**)argp)[1];

	dat_new0c_(LocatorStack[LocatorID],nam[0].s,piLen,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_slice(int argc, char *argp[]){
	/*
	Return locator to a slice of an array object.
	*/
	char loc[DAT__SZLOC];
	IDL_LONG *piDiml,*piDimu,*piNdim;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((IDL_LONG**)argp)[0];
	piDiml=((IDL_LONG**)argp)[1];
	piDimu=((IDL_LONG**)argp)[2];

	dat_slice_(LocatorStack[LocatorID],piNdim,piDiml,piDimu,loc,&status,len0,len0);
	LocatorID += 1;
	memcpy(LocatorStack[LocatorID],loc,DAT__SZLOC);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_alter(int argc, char *argp[]){
	/*
	Alter object size.
	*/
	IDL_LONG *piDims,*piNdim;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((IDL_LONG**)argp)[0];
	piDims=((IDL_LONG**)argp)[1];

	dat_alter_(LocatorStack[LocatorID],piNdim,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_erase(int argc, char *argp[]){
	/* 
	Erase component.
	*/
	IDL_STRING *nam;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	nam=((IDL_STRING**)argp)[0];len1=strlen(nam[0].s);

	dat_erase_(LocatorStack[LocatorID],nam[0].s,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_annul(int argc, char *argp[]){
	/*
	Annuls HDS locator.
	*/

        if (argc != 0) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	dat_annul_(LocatorStack[LocatorID],&status,len0);
	LocatorID -= 1;

	if (LocatorID == -1){
/*		printf("HDS file closed.\n"); */
		hds_stop_(&status);
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get(int argc, char *argp[]){
	/*
	Read primitive.
	*/
	IDL_STRING *typ;
	IDL_LONG *piNdim,*piDims,*piLen;
	unsigned char *pData;

	/* These are for character data */
	IDL_STRING *data;
	char *Data,*pos;
	int i,n;

        if (argc != 5) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	typ=          ((IDL_STRING**)argp)[0];len1=strlen(typ[0].s);
	piNdim=         ((IDL_LONG**)argp)[1];
	piDims=         ((IDL_LONG**)argp)[2];
	pData=((unsigned char**)argp)[3];
	data=         ((IDL_STRING**)argp)[3];
	piLen=          ((IDL_LONG**)argp)[4];

	if(strstr(typ[0].s,"_CHAR")!=NULL){
		n=1;
		for(i=0;i<*piNdim;i++)n*=piDims[i];
		if((Data=(char*)malloc(*piLen*n))==NULL){
			fprintf(stderr,"Error allocating memory in dat_get!");
			return(1);
			}
		dat_get_(LocatorStack[LocatorID],typ[0].s,piNdim,piDims,(unsigned char*)Data,
			&status,len0,len1,piLen[0]);
		for(i=0;i<n;i++){
			memcpy(data[i].s,Data+i*(*piLen),*piLen);
			while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
				memset(pos,' ',*piLen-(pos-data[i].s));
				}
			}
		free(Data);
		}
	else{
		dat_get_(LocatorStack[LocatorID],typ[0].s,piNdim,piDims,(unsigned char*)pData,
			&status,len0,len1,piLen[0]);
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get0c(int argc, char *argp[]){
	/*
	Read character scalar primitive.
	*/
	IDL_STRING *string;
	IDL_LONG *piLen;
	char *pos;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	string=((IDL_STRING**)argp)[0];
	piLen=   ((IDL_LONG**)argp)[1];

	dat_get0c_(LocatorStack[LocatorID],string[0].s,&status,len0,*piLen);
	if((pos=(char*)memchr(string[0].s,'\0',*piLen))!=NULL){
		memset(pos,' ',string[0].s+(*piLen)-pos);
		}

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get0i(int argc, char *argp[]){
	/*
	Read integer scalar primitive.
	*/
	IDL_LONG *value;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	value=((IDL_LONG**)argp)[0];

	dat_get0i_(LocatorStack[LocatorID],value,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get0r(int argc, char *argp[]){
	/*
	Read integer scalar primitive.
	*/
	float *value;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	value=((float**)argp)[0];

	dat_get0r_(LocatorStack[LocatorID],value,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get0d(int argc, char *argp[]){
	/*
	Read integer scalar primitive.
	*/
	double *value;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	value=((double**)argp)[0];

	dat_get0d_(LocatorStack[LocatorID],value,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get1c(int argc, char *argp[]){
	/* 
	Read character vector component.
	*/
	IDL_STRING *data;
	IDL_LONG *piNum,*piLen;
	char *Data,*pos;
	int i;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum= ((IDL_LONG**)argp)[0];
	data=((IDL_STRING**)argp)[1];
	piLen= ((IDL_LONG**)argp)[2];

	if((Data=(char*)malloc(*piLen*(*piNum)))==NULL){
		fprintf(stderr,"Error allocating memory in cmp_get1c!");
		return(1);
		}

	dat_get1c_(LocatorStack[LocatorID],piNum,Data,piNum,&status,len0,*piLen);
	for(i=0;i<*piNum;i++){
		memcpy(data[i].s,Data+i*(*piLen),*piLen);
		while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
			memset(pos,' ',*piLen-(pos-data[i].s));
			}
		}
	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get1i(int argc, char *argp[]){
	/*
	Read integer vector component.
	*/
	IDL_LONG *piNum,*piData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum= ((IDL_LONG**)argp)[0];
	piData=((IDL_LONG**)argp)[1];

	dat_get1i_(LocatorStack[LocatorID],piNum,piData,piNum,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get1r(int argc, char *argp[]){
	/*
	Read real vector component.
	*/
	IDL_LONG *piNum;
	float *pfData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=   ((IDL_LONG**)argp)[0];
	pfData=((float**)argp)[1];

	dat_get1r_(LocatorStack[LocatorID],piNum,pfData,piNum,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_get1d(int argc, char *argp[]){
	/*
	Read double vector component.
	*/
	IDL_LONG *piNum;
	double *pdfData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=     ((IDL_LONG**)argp)[0];
	pdfData=((double**)argp)[1];

	dat_get1d_(LocatorStack[LocatorID],piNum,pdfData,piNum,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_getnc(int argc, char *argp[]){
	/* 
	Read character array component.
	*/
	IDL_STRING *data;
	IDL_LONG *piNdim,*piDims,*piLen;
	char *Data,*pos;
	int i,n;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((IDL_LONG**)argp)[0];
	piDims=((IDL_LONG**)argp)[1];
	data=((IDL_STRING**)argp)[2];
	piLen= ((IDL_LONG**)argp)[3];

	n=1;
	for(i=0;i<*piNdim;i++)n*=piDims[i];

	if((Data=(char*)malloc(*piLen*n))==NULL){
		fprintf(stderr,"Error allocating memory in dat_getnc!");
		return(1);
		}

	dat_getnc_(LocatorStack[LocatorID],piNdim,piDims,Data,piDims,
			&status,len0,*piLen);
	for(i=0;i<n;i++){
		memcpy(data[i].s,Data+i*(*piLen),*piLen);
		while((pos=(char*)memchr(data[i].s,'\0',*piLen))!=NULL){
			memset(pos,' ',*piLen-(pos-data[i].s));
			}
		}
	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_getni(int argc, char *argp[]){
	/*
	Read an array primitive.
	*/
	IDL_LONG *piNdim,*piDims;
	IDL_LONG *piData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=  ((IDL_LONG**)argp)[0];
	piDims=  ((IDL_LONG**)argp)[1];
	piData=  ((IDL_LONG**)argp)[2];

	dat_getni_(LocatorStack[LocatorID],piNdim,piDims,(IDL_LONG*)piData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_getnr(int argc, char *argp[]){
	/*
	Read an array primitive.
	*/
	IDL_LONG *piNdim,*piDims;
	float *pfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=  ((IDL_LONG**)argp)[0];
	piDims=  ((IDL_LONG**)argp)[1];
	pfData=((float**)argp)[2];

	dat_getnr_(LocatorStack[LocatorID],piNdim,piDims,(float*)pfData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_getnd(int argc, char *argp[]){
	/*
	Read a double array primitive.
	*/
	IDL_LONG *piNdim,*piDims;
	double *pdfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=    ((IDL_LONG**)argp)[0];
	piDims=    ((IDL_LONG**)argp)[1];
	pdfData=((double**)argp)[2];

	dat_getnd_(LocatorStack[LocatorID],piNdim,piDims,(double*)pdfData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put0c(int argc, char *argp[]){
	/*
	Write a scalar string component.
	*/
	IDL_STRING *data;

        if (argc != 1) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	data=((IDL_STRING**)argp)[0];len1=strlen(data[0].s);

	dat_put0c_(LocatorStack[LocatorID],data[0].s,&status,len0,len1);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put1c(int argc, char *argp[]){
	/*
	Write a vector string component.
	*/
	IDL_STRING *data;
	IDL_LONG *piNum,*piLen;
	char *Data;
	int i;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=  ((IDL_LONG**)argp)[0];
	data= ((IDL_STRING**)argp)[1];
	piLen=  ((IDL_LONG**)argp)[2];

	if((Data=(char*)malloc(*piLen*(*piNum)))==NULL){
		fprintf(stderr,"Error allocating memory in dat_put1c!");
		return(1);
		}

	for(i=0;i<*piNum;i++){
		memcpy(Data+i*(*piLen),data[i].s,*piLen);
		}

	dat_put1c_(LocatorStack[LocatorID],piNum,Data,&status,len0,*piLen);

	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put1i(int argc, char *argp[]){
	/*
	Write integer vector component.
	*/
	IDL_LONG *piNum,*piData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum= ((IDL_LONG**)argp)[0];
	piData=((IDL_LONG**)argp)[1];

	dat_put1i_(LocatorStack[LocatorID],piNum,piData,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put1r(int argc, char *argp[]){
	/*
	Write real vector component.
	*/
	IDL_LONG *piNum;
	float *pfData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=   ((IDL_LONG**)argp)[0];
	pfData=((float**)argp)[1];

	dat_put1r_(LocatorStack[LocatorID],piNum,pfData,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_put1d(int argc, char *argp[]){
	/*
	Write double vector component.
	*/
	IDL_LONG *piNum;
	double *pdfData;

        if (argc != 2) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNum=     ((IDL_LONG**)argp)[0];
	pdfData=((double**)argp)[1];

	dat_put1d_(LocatorStack[LocatorID],piNum,pdfData,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_putnc(int argc, char *argp[]){
	/* 
	Write character array component.
	*/
	IDL_STRING *data;
	IDL_LONG *piNdim,*piDims,*piLen;
	char *Data;
	int i,j,n;

        if (argc != 4) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	piNdim=((IDL_LONG**)argp)[0];
	piDims=((IDL_LONG**)argp)[1];
	data=((IDL_STRING**)argp)[2];
	piLen= ((IDL_LONG**)argp)[3];

	n=1;
	for(i=0;i<*piNdim;i++)n*=piDims[i];

	if((Data=(char*)malloc(*piLen*n))==NULL){
		fprintf(stderr,"Error allocating memory in dat_putnc!");
		return(1);
		}

	for(i=0;i<n;i++){
		memcpy(Data+i*(*piLen),data[i].s,*piLen);
		}

	dat_putnc_(LocatorStack[LocatorID],piNdim,piDims,Data,piDims,&status,len0,*piLen);

	free(Data);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_putni(int argc, char *argp[]){
	/*
	Write an array primitive.
	*/
	IDL_LONG *piNdim,*piDims;
	IDL_LONG *piData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=  ((IDL_LONG**)argp)[0];
	piDims=  ((IDL_LONG**)argp)[1];
	piData=  ((IDL_LONG**)argp)[2];

	dat_putni_(LocatorStack[LocatorID],piNdim,piDims,piData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_putnr(int argc, char *argp[]){
	/*
	Write an array primitive.
	*/
	IDL_LONG *piNdim,*piDims;
	float *pfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=  ((IDL_LONG**)argp)[0];
	piDims=  ((IDL_LONG**)argp)[1];
	pfData=((float**)argp)[2];

	dat_putnr_(LocatorStack[LocatorID],piNdim,piDims,pfData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
int dat_putnd(int argc, char *argp[]){
	/*
	Write a double array primitive.
	*/
	IDL_LONG *piNdim,*piDims;
	double *pdfData;

        if (argc != 3) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}
	
	piNdim=    ((IDL_LONG**)argp)[0];
	piDims=    ((IDL_LONG**)argp)[1];
	pdfData=((double**)argp)[2];

	dat_putnd_(LocatorStack[LocatorID],piNdim,piDims,pdfData,piDims,&status,len0);

	return(status);
	}
/*----------------------------------------------------------------------------*/
