/*----------------------------------------------------------------------
*
* COMPONENT: PROPAGATION OF ASTROMETRIC PARAMETERS
*            AND COVARIANCE MATRICES
*                                                     SOURCE: POS_PROP.C
*
* TITLE: PROPAGATION OF A VECTOR OF ASTROMETRIC PARAMETERS 'ASTROPARAM'
*        AND ITS ASSOCIATED COVARIANCE MATRIX 'COVARMATRIX'
*        FROM ONE EPOCH TO ANOTHER
*
*------------------------------------------------------------------------*
*              PURPOSE OF THE FUNCTION
*
*  The following subroutine performs the transformation of a vector of
*  astrometric parameters 'AstroParam', and its associated covariance
*  matrix 'CovarMatrix', from one epoch to another.
*
*  The positional components in AstroParam are alpha and delta expressed
*  in [deg], although the corresponding elements of the covariance matrix
*  refer to xi and eta, expressed in [mas] relative to the fixed normal
*  triad at the position.
*
*  FORTRAN CODE WRITTEN BY:
*       L Lindegren, Lund   Observatory, 15 March 1995
*  TRANSLATED FROM FORTRAN INTO C BY:
*       D Priou    , Meudon Observatory, 28 March 1995
*
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!  WARNING : IN THE FORTRAN CODE, THE INDICES OF ALL THE ARRAYS AND       !
!            MATRICES RANGE FROM 1 (!!!!!ONE!!!!!) TO 3 OR 6.             !
!            IN THE C CODE, THE INDICES OF ALL THE ARRAYS AND MATRICES    !
!            RANGE FROM 0 (!!!!!ZERO!!!!!) TO 2 OR 5.                     !
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*------------------------------------------------------------------------*/


/*------------------------------------------------------------------------*
 The type REALNUMBER is defined. It allows the use of any real-valued type
 just by modifying this definition.

 Examples :
 typedef      float  REALNUMBER ;
 typedef      double REALNUMBER ;
 typedef long double REALNUMBER ;

 The name of the mathematical functions can change according to the
 selected real type and to the computer.
 The user should change the names of the sin, cos, atan2, and sqrt
 functions according to his choice and to the computer.

-------------------------------------------------------------------------*/

typedef double REALNUMBER ;

#define COSINE(x)   (cos(x))
#define SINE(x)     (sin(x))
#define ATAN2(x,y)  (atan2(x,y))
#define SQRT(x)     (sqrt(x))


/*------------------------------------------------------------------------*
*                           PROTOTYPES
-------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------
                              scalar
     Computes the scalar product of two 3-dimensional vectors x and y
------------------------------------------------------------------------*/

REALNUMBER scalar ( REALNUMBER x[3] ,
                           REALNUMBER y[3] ) ;

/*-----------------------------------------------------------------------
                             pos_prop

  Propagates the 6-dimensional vector of barycentric astrometric 
  parameters and the associated covariance matrix from epoch t0 to t
  assuming uniform space motion

  This is an implementation of the equations in The Hipparcos  
  and Tycho Catalogues (ESA SP-1200), Volume 1, Section 1.5.5,
  'Epoch Transformation: Rigorous Treatment'.

  INPUT (all real variables are REALNUMBER):

  t0       = original epoch [yr] - see Note 1
  a0[0]    = right ascension at t0 [deg]
  a0[1]    = declination at t0 [deg]
  a0[2]    = parallax at t0 [mas]
  a0[3]    = proper motion in R.A., mult by cos(Dec), at t0 [mas/yr]
  a0[4]    = proper motion in Dec at t0 [mas/yr]
  a0[5]    = normalised radial velocity at t0 [mas/yr] - see Note 2
  c0[i][j] = cov(a0[i],a0[j]) in [mas^2, mas^2/yr, mas^2/yr^2]
             (i, j = 0, 1, .., 5) - see Note 3
  t        = new epoch [yr] - see Note 1


  OUTPUT (all real variables are REALNUMBER):

  a[0]    = right ascension at t [deg]
  a[1]    = declination at t [deg]
  a[2]    = parallax at t [mas]
  a[3]    = proper motion in R.A., mult by cos(Dec), at t [mas/yr]
  a[4]    = proper motion in Dec at t [mas/yr]
  a[5]    = normalised radial velocity at t [mas/yr]
  c[i][j] = cov(a[i],a[j]) in [mas^2, mas^2/yr, mas^2/yr^2]
            (i, j = 0, 1, .., 5) - see Note 3

  FUNCTIONS/SUBROUTINES CALLED:

  REALNUMBER FUNCTION scalar = scalar product of two vectors

  NOTES:

  1. Only t-t0 is used; the origin of the time scale is
     therefore irrelevant but must be the same for t0 and t.
  
  2. The normalised radial velocity at epoch t0 is given by
        a0[5] = vr0*a0[2]/4.740470446
     where vr0 is the barycentric radial velocity in [km/s] at
     epoch t0; similarly a[5] = vr*a[2]/4.740470446 at epoch t.
  
  3. The matrices c0 and c are dimensioned as c0[6][6], c[6][6] 
     in this subroutine and must have compatible dimensions in the
     calling program unit.  Although c0 and c are symmetric 
     matrices, all elements in c0 must be defined on entry, and 
     all elements in c are defined on return.
  
  4. This routine adheres to ANSI C.

------------------------------------------------------------------------*/

void pos_prop ( REALNUMBER t0       ,
                       REALNUMBER a0[6]    ,
                       REALNUMBER c0[6][6] ,
                       REALNUMBER t        ,
                       REALNUMBER a[6]     ,
                       REALNUMBER c[6][6]  ) ;

/*-----------------------------------------------------------------------
                           END OF THE COMPONENT
------------------------------------------------------------------------*/

