/*******************************************************************************
writefits.c

USNO/NRL Optical Interferometer
3450 Massachusetts Avenue NW
Washington DC 20392-5420

System: Data Reduction
Subsystem: CHAMELEON/STARBASE/AMOEBA

Description:
Library of C-routines callable from PV-WAVE via LINKNLOAD.

Modification history:
17-Jun-1998  C.A.Hummel, file created
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "export.h"
#include "fitsio.h"

/* Function prototypes */

int writefits(int argc, char *argp[]);
/*----------------------------------------------------------------------------*/
int writefits(int argc, char *argp[]){
	/* 
	Write ScanData to FITS
	*/

#define FITS_CLEN_ERRMSG 31

/* Variables passed from IDL */

	IDL_STRING *azDateNow,*azDateObs,*azStar,*azStation;
	IDL_STRING *azOrigin,*azTelescope,*azInstrument,*azObserver,*azAuthor;

	double *pdfJD,*pdfRA,*pdfDec;
	double *pdfTime,*pdfUVW,*pdfVisReal,*pdfVisImag,*pdfWeight,*pdfBase;
	double *pdfGSTIA0,*pdfUT1UTC,*pdfDATUTC,*pdfCoord,*pdfArrayC;
        double *pdfIFFreq0,*pdfIFFreqs;
	float *pfChWidth;
	IDL_LONG *piNScan,*piNVis,*piSourceID,*piNStars;
	IDL_LONG *piNIBeam,*piNOBeam,*piNCh,*piNBl;
	IDL_LONG *piMCh,*piMBl;

/* Miscellaneous variables */

	int iScan;
	long iSample;

	int i,j,k,l;
	double dfValue;
	long iValue;

/* Variables used by the FITSIO library functions */

	char azKeyWord[]="123456";
	char azParameter[80];
	char azComment[80];
	char azOutFile[15],azErrText[FITS_CLEN_ERRMSG];
	int status=0, iRecLen=2880;
	fitsfile *fptr;

	int simple, bitpix, naxis, groups, extend;
	long naxes[7], pcount, gcount;
	int decimals = 11;
	long pixel,group,fparm,nparm;
	long extvers=1;

	double DataBuffer[76800];

/* Binary table extension keywords */

#define MAX_FIELDS 100
/* #define MAX_IF 32 */
#define MAX_IF 1628
#define MAX_STARS 20
#define MAX_STATIONS 27

	int nrows,tfields;
	char *ttype[MAX_FIELDS],*tform[MAX_FIELDS],*tunit[MAX_FIELDS];
	double pdfValues[MAX_FIELDS];
	long piValues[MAX_FIELDS];

	char *azStars[MAX_STARS];
	long aiSideBand[MAX_IF];
	float afChanWidth[MAX_IF],afTotalBand[MAX_IF];
	char *azStations[MAX_STATIONS];
	char *azStokesA[MAX_STATIONS],*azStokesB[MAX_STATIONS];

/* Allocate space for extension table keywords */

	for (i=0;i<MAX_FIELDS;i++){
		ttype[i]=(char*)malloc(20);
		tform[i]=(char*)malloc(20);
		tunit[i]=(char*)malloc(20);
		}
	for (i=0;i<MAX_STATIONS;i++){
		azStations[i]=(char*)malloc(20);
		azStokesA[i]=(char*)malloc(10);
		azStokesB[i]=(char*)malloc(10);
		}
	for (i=0;i<MAX_STARS;i++){
		azStars[i]=(char*)malloc(20);
		}

/* Check number of parameters and initialize the pointers */

        if (argc != 36) {
      		fprintf(stderr,"\nWrong # of parameters(%d)",argc);
      		return(1);
      		}

	pdfJD=            ((double**)argp)[0];
	azDateNow=    ((IDL_STRING**)argp)[1];
	azDateObs=    ((IDL_STRING**)argp)[2];
	azOrigin=     ((IDL_STRING**)argp)[3];
	azTelescope=  ((IDL_STRING**)argp)[4];
	azInstrument= ((IDL_STRING**)argp)[5];
	azObserver=   ((IDL_STRING**)argp)[6];
	azAuthor=     ((IDL_STRING**)argp)[7];
	azStar=       ((IDL_STRING**)argp)[8];
	pdfRA=            ((double**)argp)[9];
	pdfDec=          ((double**)argp)[10];
	pdfGSTIA0=	 ((double**)argp)[11];
	pdfUT1UTC=	 ((double**)argp)[12];
	pdfDATUTC=	 ((double**)argp)[13];
	pdfIFFreq0=      ((double**)argp)[14];
	pdfIFFreqs=	 ((double**)argp)[15];
	pfChWidth=        ((float**)argp)[16];

	piNVis=            ((IDL_LONG**)argp)[17];
	piNScan=           ((IDL_LONG**)argp)[18];
	piNIBeam=          ((IDL_LONG**)argp)[19];
	piNOBeam=          ((IDL_LONG**)argp)[20];
        piNCh=             ((IDL_LONG**)argp)[21];
        piNBl=             ((IDL_LONG**)argp)[22];

	piMCh=             ((IDL_LONG**)argp)[23];
	piMBl=             ((IDL_LONG**)argp)[24];

	pdfTime=         ((double**)argp)[25];
	pdfUVW=          ((double**)argp)[26];
	pdfVisReal=      ((double**)argp)[27];
	pdfVisImag=      ((double**)argp)[28];
	pdfWeight=       ((double**)argp)[29];
 
	pdfBase=         ((double**)argp)[30];
	azStation=   ((IDL_STRING**)argp)[31];
	pdfCoord= 	 ((double**)argp)[32];
	pdfArrayC=       ((double**)argp)[33];
	piSourceID=        ((IDL_LONG**)argp)[34];
	piNStars=	   ((IDL_LONG**)argp)[35];

/* Compose output file name */
	
	strcpy(azOutFile,azStar[0].s);
	strcat(azOutFile,".FITS");

/* Open the new FITS file*/

	ffinit(&fptr,azOutFile,&status);
	ffgerr(status,azErrText);
	printf ("ffinit status = %d : %s\n", status, azErrText);
  
	simple = 1;
	bitpix = -32;		/* bits per pixel */
	naxis = 7;		/* Number of axes */
	naxes[0] = 0;		/* No normal image in groups */
	naxes[1] = 3;		/* Real, imaginary, weight */
	naxes[2] = 1; 		/* Stokes parms, here only I */
	naxes[3] = 1; 		/* Freq */
	naxes[4] = piNCh[0];	/* Number of channels (IFs) */
	naxes[5] = 1;		/* RA */
	naxes[6] = 1;		/* Dec */
	extend = 1;		/* no extension tables present */
	groups  = 1;		/* random groups present */
	if (piNStars[0] == 1) 
	pcount = 6; else 	/* 6 parameters per random group */
	pcount = 8;		/* 8 for Multi-source files */
	gcount = piNVis[0]; 	/* Number of groups, i.e. baselines */

/* Write the required primary array keywords */

	ffphpr(fptr,simple,bitpix,naxis,naxes,pcount,gcount,extend,&status);
	ffgerr(status,azErrText);
	printf ("ffphpr status = %d : %s\n", status, azErrText);

/* Define the coordinate axes */

	strcpy(azKeyWord,"CTYPE2");
	strcpy(azParameter,"COMPLEX");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"CRPIX2");
	dfValue=1.0;
	decimals=11;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CRVAL2");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CDELT2");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CROTA2");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"CTYPE3");
	strcpy(azParameter,"STOKES");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"CRPIX3");
	dfValue=1.0;
	decimals=11;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CRVAL3");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CDELT3");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CROTA3");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"CTYPE4");
	strcpy(azParameter,"FREQ");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"CRPIX4");
	dfValue=1.0;
	decimals=11;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CRVAL4");
	dfValue=pdfIFFreq0[0];
	decimals=1;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CDELT4");
	dfValue=1.0;
	decimals=11;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CROTA4");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"CTYPE5");
	strcpy(azParameter,"IF");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"CRPIX5");
	dfValue=1.0;
	decimals=11;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CRVAL5");
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CDELT5");
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CROTA5");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"CTYPE6");
	strcpy(azParameter,"RA");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"CRPIX6");
	dfValue=1.0;
	decimals=11;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CRVAL6");
	dfValue=pdfRA[0];
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CDELT6");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CROTA6");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"CTYPE7");
	strcpy(azParameter,"DEC");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"CRPIX7");
	dfValue=1.0;
	decimals=11;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CRVAL7");
	dfValue=pdfDec[0];
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CDELT7");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"CROTA7");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

/* Define the random group parameters */

	strcpy(azKeyWord,"PTYPE1");
	strcpy(azParameter,"UU");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"PSCAL1");
	dfValue=1.0;
	decimals=11;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"PZERO1");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"PTYPE2");
	strcpy(azParameter,"VV");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"PSCAL2");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"PZERO2");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"PTYPE3");
	strcpy(azParameter,"WW");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"PSCAL3");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"PZERO3");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"PTYPE4");
	strcpy(azParameter,"BASELINE");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"PSCAL4");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"PZERO4");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"PTYPE5");
	strcpy(azParameter,"DATE");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"PSCAL5");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"PZERO5");
	dfValue=pdfJD[0];
	decimals=7;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	strcpy(azKeyWord,"PTYPE6");
	strcpy(azParameter,"DATE");
	ffpkys(fptr,azKeyWord,azParameter,"",&status);
	strcpy(azKeyWord,"PSCAL6");
	dfValue=1.0;
	decimals=11;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
	strcpy(azKeyWord,"PZERO6");
	dfValue=0.5;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

	if (piNStars[0] > 1) {
		strcpy(azKeyWord,"PTYPE7");
		strcpy(azParameter,"SOURCE");
		ffpkys(fptr,azKeyWord,azParameter,"",&status);
		strcpy(azKeyWord,"PSCAL7");
		dfValue=1.0;
		ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
		strcpy(azKeyWord,"PZERO7");
		dfValue=0.0;
		ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);

		strcpy(azKeyWord,"PTYPE8");
		strcpy(azParameter,"FREQSEL");
		ffpkys(fptr,azKeyWord,azParameter,"",&status);
		strcpy(azKeyWord,"PSCAL8");
		dfValue=1.0;
		ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
		strcpy(azKeyWord,"PZERO8");
		dfValue=0.0;
		ffpkyg(fptr,azKeyWord,dfValue,decimals,"",&status);
		}

/* Add key words for our application */

	decimals=11;

	strcpy(azKeyWord,"BUNIT");
	strcpy(azParameter,"UNCALIB");
	strcpy(azComment,"Arbitrary units");
	ffpkys(fptr,azKeyWord,azParameter,azComment,&status);

	strcpy(azKeyWord,"BSCALE");
	strcpy(azComment,"Scale factor");
	dfValue=1.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,azComment,&status);

	strcpy(azKeyWord,"BZERO");
	strcpy(azComment,"Data offset");
	dfValue=0.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,azComment,&status);

	strcpy(azKeyWord,"DATE");
	strcpy(azComment,"Creation date");
	ffpkys(fptr,azKeyWord,azDateNow[0].s,azComment,&status);

	strcpy(azKeyWord,"DATE-OBS");
	strcpy(azComment,"Observation date");
	ffpkys(fptr,azKeyWord,azDateObs[0].s,azComment,&status);

	strcpy(azKeyWord,"ORIGIN");
	strcpy(azComment,"Host computer");
	ffpkys(fptr,azKeyWord,azOrigin[0].s,azComment,&status);

	strcpy(azKeyWord,"TELESCOP");
	strcpy(azComment,"Telescope name");
	ffpkys(fptr,azKeyWord,azTelescope[0].s,azComment,&status);

	strcpy(azKeyWord,"INSTRUME");
	strcpy(azComment,"Instrument name");
	ffpkys(fptr,azKeyWord,azInstrument[0].s,azComment,&status);

	strcpy(azKeyWord,"OBSERVER");
	strcpy(azComment,"Institution");
	ffpkys(fptr,azKeyWord,azObserver[0].s,azComment,&status);

	strcpy(azKeyWord,"OBJECT");
	strcpy(azComment,"Source name");
	ffpkys(fptr,azKeyWord,azStar[0].s,azComment,&status);

	strcpy(azKeyWord,"OBSRA");
	strcpy(azComment,"Siderostat pointing");
	dfValue=pdfRA[0];
	ffpkyg(fptr,azKeyWord,dfValue,decimals,azComment,&status);

	strcpy(azKeyWord,"OBSDEC");
	strcpy(azComment,"Siderostat pointing");
	dfValue=pdfDec[0];
	ffpkyg(fptr,azKeyWord,dfValue,decimals,azComment,&status);

	strcpy(azKeyWord,"AUTHOR");
	strcpy(azComment,"User");
	ffpkys(fptr,azKeyWord,azAuthor[0].s,azComment,&status);

	strcpy(azKeyWord,"EPOCH");
	strcpy(azComment,"Should be EQUINOX");
	dfValue=2000.0;
	ffpkyg(fptr,azKeyWord,dfValue,decimals,azComment,&status);

/*
	strcpy(azKeyWord,"OBSRA";
	strcpy(azComment,"Pointing RA";
	ffpkyg(fptr,azKeyWord,pdfRA[0],decimals,azComment,&status);

	strcpy(azKeyWord,"OBSDEC";
	strcpy(azComment,"Pointing DEC";
	ffpkyg(fptr,azKeyWord,pdfDec[0],decimals,azComment,&status);
*/

/* Add history cards */
	strcpy(azComment,"HISTORY FITS file created by OYSTER (C.A. Hummel USNO)                         ");	
    	ffprec(fptr,azComment,&status);

/* Write data */

	group=1;
	fparm=1;
	nparm=pcount;
	pixel=1;

	for(iScan=0;iScan<piNScan[0];iScan++){

	for(i=0;i<piNOBeam[0];i++){
	for(l=0;l<piNBl[i];l++){

	j=0;
	iSample=0;

	/* Random parameter No 1: U */
	k=i+piNOBeam[0]*(j+piMCh[0]*(l+piMBl[0]*(0+3*iScan)));
	DataBuffer[iSample++]=pdfUVW[k]/pdfIFFreq0[0];

	/* Random parameter No 2: V */
	k=i+piNOBeam[0]*(j+piMCh[0]*(l+piMBl[0]*(1+3*iScan)));
	DataBuffer[iSample++]=pdfUVW[k]/pdfIFFreq0[0];

	/* Random parameter No 3: W */
	k=i+piNOBeam[0]*(j+piMCh[0]*(l+piMBl[0]*(2+3*iScan)));
	DataBuffer[iSample++]=pdfUVW[k]/pdfIFFreq0[0];

	/* Random parameter No 4: baseline index */
	k=l+piMBl[0]*i;
	DataBuffer[iSample++]=pdfBase[k];

	/* Random parameter No 5: Date */
	DataBuffer[iSample++]=0.0;

	/* Random parameter No 6: Date offset */
	DataBuffer[iSample++]=pdfTime[iScan];

	if (piNStars[0] > 1) {
		/* Random parameter No 7: Source ID */
		DataBuffer[iSample++]=piSourceID[iScan];

		/* Random parameter No 8: FreqSel */
		DataBuffer[iSample++]=1.0;
		}

	ffpgpd(fptr,group,fparm,nparm,DataBuffer,&status);

	/* Collect data for random group array */
	iSample=0;
	for(j=0;j<piNCh[i];j++){
		k=i+piNOBeam[0]*(j+piMCh[0]*(l+piMBl[0]*iScan));
		DataBuffer[iSample++]=pdfVisReal[k];
		DataBuffer[iSample++]=pdfVisImag[k];
		DataBuffer[iSample++]=pdfWeight[k];
		}

	ffpprd(fptr,group,pixel,iSample,DataBuffer,&status);
	group+=1;

	}
	}

	}

/* Write binary FQ table */

	nrows=1;
	tfields=5;

	strcpy(ttype[0],"FRQSEL");
	strcpy(ttype[1],"IF FREQ");
	strcpy(ttype[2],"CH WIDTH");
	strcpy(ttype[3],"TOTAL BANDWIDTH");
	strcpy(ttype[4],"SIDEBAND");

	strcpy(tunit[0]," ");
	strcpy(tunit[1],"HZ");
	strcpy(tunit[2],"HZ");
	strcpy(tunit[3],"HZ");
	strcpy(tunit[4]," ");

	strcpy(tform[0],"1J");
	sprintf(tform[1],"%dD",piNCh[0]);
	sprintf(tform[2],"%dE",piNCh[0]);
	sprintf(tform[3],"%dE",piNCh[0]);
	sprintf(tform[4],"%dJ",piNCh[0]);

	ffibin(fptr,nrows,tfields,ttype,tform,tunit,"AIPS FQ",0L,&status);

/*	Write keyword-value pairs */

	ffpkyj(fptr,"EXTVER",extvers,"Extension version number", &status);
	ffpkyj(fptr,"NO_IF",piNCh[0],"Number of channels",&status);

/*	Write table data */

	iValue=1;
	ffpclj(fptr,1,1,1,1,&iValue,&status);
	ffpcld(fptr,2,1,1,piNCh[0],pdfIFFreqs,&status);
	ffpcle(fptr,3,1,1,piNCh[0],pfChWidth,&status);
	ffpcle(fptr,4,1,1,piNCh[0],pfChWidth,&status);
	for (i=0;i<piNCh[0];i++)aiSideBand[i]=1;
	ffpclj(fptr,5,1,1,piNCh[0],aiSideBand,&status);

/* Write binary AN table */

	nrows=piNIBeam[0];
	tfields=12;

	strcpy(ttype[0],"ANNAME");
	strcpy(ttype[1],"STABXYZ");
	strcpy(ttype[2],"ORBPARM");
	strcpy(ttype[3],"NOSTA");
	strcpy(ttype[4],"MNTSTA");
	strcpy(ttype[5],"STAXOF");
	strcpy(ttype[6],"POLTYA");
	strcpy(ttype[7],"POLAA");
	strcpy(ttype[8],"POLCALA");
	strcpy(ttype[9],"POLTYB");
	strcpy(ttype[10],"POLAB");
	strcpy(ttype[11],"POLCALB");

	strcpy(tunit[0]," ");
	strcpy(tunit[1],"METERS");
	strcpy(tunit[2]," ");
	strcpy(tunit[3]," ");
	strcpy(tunit[4]," ");
	strcpy(tunit[5],"METERS");
	strcpy(tunit[6]," ");
	strcpy(tunit[7],"DEGREES");
	strcpy(tunit[8]," ");
	strcpy(tunit[9]," ");
	strcpy(tunit[10],"DEGREES");
	strcpy(tunit[11]," ");

	strcpy(tform[0],"8A");
	strcpy(tform[1],"3D");
	strcpy(tform[2],"0D");
	strcpy(tform[3],"1J");
	strcpy(tform[4],"1J");
	strcpy(tform[5],"1E");
	strcpy(tform[6],"1A");
	strcpy(tform[7],"1E");
	strcpy(tform[8],"0E");
	strcpy(tform[9],"1A");
	strcpy(tform[10],"1E");
	strcpy(tform[11],"0E");

	ffibin(fptr,nrows,tfields,ttype,tform,tunit,"AIPS AN",0L,&status);

/*	Write keyword-value pairs */

	ffpkyj(fptr,"EXTVER",extvers,"Extension version number", &status);
	decimals=13;
	ffpkyg(fptr,"ARRAYX",pdfArrayC[0],decimals,"",&status);
	ffpkyg(fptr,"ARRAYY",pdfArrayC[1],decimals,"",&status);
	ffpkyg(fptr,"ARRAYZ",pdfArrayC[2],decimals,"",&status);
	decimals=11;
	ffpkyg(fptr,"GSTIA0",pdfGSTIA0[0],decimals,"",&status);
	dfValue=360.98564497330;
	ffpkyg(fptr,"DEGPDY",dfValue,decimals,"",&status);
	decimals=1;
	ffpkyg(fptr,"FREQ",pdfIFFreq0[0],decimals,"",&status);
	ffpkys(fptr,"RDATE",azDateObs[0].s,"",&status);
	dfValue=0.0;
	decimals=14;
	ffpkyg(fptr,"POLARX",dfValue,decimals,"",&status);
	ffpkyg(fptr,"POLARY",dfValue,decimals,"",&status);
	decimals=15;
	ffpkyg(fptr,"UT1UTC",pdfUT1UTC[0],decimals,"",&status);
	decimals=13;
	ffpkyg(fptr,"IATUTC",pdfDATUTC[0],decimals,"",&status);
	ffpkys(fptr,"TIMSYS","IAT","",&status);
	ffpkys(fptr,"ARRNAM",azTelescope[0].s,"",&status);
	iValue=0;
	ffpkyj(fptr,"NUMORB",iValue,"",&status);
	iValue=0;
	ffpkyj(fptr,"NOPCAL",iValue,"",&status);

/*	Write table data */

	for(i=0;i<piNIBeam[0];i++){
		strcpy(azStations[i],azStation[i].s);
		strcpy(azStokesA[i],"R");
		strcpy(azStokesB[i],"L");
		}
	ffpcls(fptr,1,1,1,piNIBeam[0],azStations,&status);
	iValue=piNIBeam[0]*3;
	ffpcld(fptr,2,1,1,iValue,pdfCoord,&status);
	for(i=0;i<piNIBeam[0];i++)piValues[i]=i+1;;
	ffpclj(fptr,4,1,1,piNIBeam[0],piValues,&status);
	ffpcls(fptr,7,1,1,piNIBeam[0],azStokesA,&status);
	ffpcls(fptr,10,1,1,piNIBeam[0],azStokesB,&status);

/* Write binary SU table */

	if (piNStars[0] > 1){

	nrows=piNStars[0];
	tfields=19;

	strcpy(ttype[0],"ID. NO.");
	strcpy(ttype[1],"SOURCE");
	strcpy(ttype[2],"QUAL");
	strcpy(ttype[3],"CALCODE");
	strcpy(ttype[4],"IFLUX");
	strcpy(ttype[5],"QFLUX");
	strcpy(ttype[6],"UFLUX");
	strcpy(ttype[7],"VFLUX");
	strcpy(ttype[8],"FREQOFF");
	strcpy(ttype[9],"BANDWIDTH");
	strcpy(ttype[10],"RAEPO");
	strcpy(ttype[11],"DECEPO");
	strcpy(ttype[12],"EPOCH");
	strcpy(ttype[13],"RAAPP");
	strcpy(ttype[14],"DECAPP");
	strcpy(ttype[15],"LSRVEL");
	strcpy(ttype[16],"RESTFREQ");
	strcpy(ttype[17],"PMRA");
	strcpy(ttype[18],"PMDEC");

	strcpy(tunit[0]," ");
	strcpy(tunit[1]," ");
	strcpy(tunit[2]," ");
	strcpy(tunit[3]," ");
	strcpy(tunit[4],"JY");
	strcpy(tunit[5],"JY");
	strcpy(tunit[6],"JY ");
	strcpy(tunit[7],"JY");
	strcpy(tunit[8],"HZ");
	strcpy(tunit[9],"HZ");
	strcpy(tunit[10],"DEGREES");
	strcpy(tunit[11],"DEGREES");
	strcpy(tunit[12],"YEARS");
	strcpy(tunit[13],"DEGREES");
	strcpy(tunit[14],"DEGREES");
	strcpy(tunit[15],"M/SEC");
	strcpy(tunit[16],"HZ");
	strcpy(tunit[17],"DEG/DAY");
	strcpy(tunit[18],"DEG/DAY");

	strcpy(tform[0],"1J");
	strcpy(tform[1],"16A");
	strcpy(tform[2],"1J");
	strcpy(tform[3],"4A");
	strcpy(tform[4],"20E");
	strcpy(tform[5],"20E");
	strcpy(tform[6],"20E");
	strcpy(tform[7],"20E");
	strcpy(tform[8],"20D");
	strcpy(tform[9],"1D");
	strcpy(tform[10],"1D");
	strcpy(tform[11],"1D");
	strcpy(tform[12],"1D");
	strcpy(tform[13],"1D");
	strcpy(tform[14],"1D");
	strcpy(tform[15],"20D");
	strcpy(tform[16],"20D");
	strcpy(tform[17],"1D");
	strcpy(tform[18],"1D");

	ffibin(fptr,nrows,tfields,ttype,tform,tunit,"AIPS SU",0L,&status);

/*	Write keyword-value pairs */

	ffpkyj(fptr,"EXTVER",extvers,"Extension version number", &status);
	ffpkyj(fptr,"NO_IF",piNCh[0],"",&status);
	ffpkys(fptr,"VELTYP","","",&status);
	ffpkys(fptr,"VELDEF","OPTICAL","",&status);
	decimals=13;
	iValue=-1;
	ffpkyg(fptr,"FREQID",iValue,decimals,"",&status);

/*	Write table data */

	for(i=0;i<piNStars[0];i++){
		strcpy(azStars[i],azStar[i].s);
		}
	for(i=0;i<piNStars[0];i++)piValues[i]=i+1;
	ffpclj(fptr,1,1,1,piNStars[0],piValues,&status);
	ffpcls(fptr,2,1,1,piNStars[0],azStars,&status);
	ffpcld(fptr,11,1,1,piNStars[0],pdfRA,&status);
	ffpcld(fptr,12,1,1,piNStars[0],pdfDec,&status);
	for(i=0;i<piNStars[0];i++)pdfValues[i]=2000.0;
	ffpcld(fptr,13,1,1,piNStars[0],pdfValues,&status);
	ffpcld(fptr,14,1,1,piNStars[0],pdfRA,&status);
	ffpcld(fptr,15,1,1,piNStars[0],pdfDec,&status);
	}

/* Free space previously allocated */

	for (i=0;i<MAX_FIELDS;i++){
		free(ttype[i]);
		free(tform[i]);
		free(tunit[i]);
		}
	for(i=0;i<MAX_STATIONS;i++){
		free(azStations[i]);
		free(azStokesA[i]);
		free(azStokesB[i]);
		}
	for(i=0;i<MAX_STARS;i++){
		free(azStars[i]);
		}

/* Close output file and quit */

	ffclos(fptr, &status);
    	ffgerr(status,azErrText);
	printf ("fcclos status = %d : %s\n", status, azErrText);

	}
/*----------------------------------------------------------------------------*/
