;*******************************************************************************
; File: ameeba.pro
;
; Description:
; ------------
; Main container of IDL scripts for AMOEBA software. AMOEBA is closely
; related to CHAMELEON wrt shared common blocks, procedures, and structure.
; AMOEBA will fit models of stellar systems to data from interferometry,
; spectroscopy, photometry, etc.
;
; Block directory:
; ----------------
; Block 1: list_starcomps,list_bincomps,list_modcomp,list_modcomps
;          list_poscomps,list_velcomps,list_magcomps,
;	   list_wavelengths,list_filters,
;	   list_summary_amoeba
;
; Block 2: load_interferometry,load_astrometry,load_spectroscopy,load_photometry
;	   flag_visibilities,flag_channels
;	   set_parallax,set_k2,add_component
;
; Block 3: storenight,loadnight,freememory
;
; Block 4: fitwaveparms,adjustfluxes,adjustmasses
;
; Block 5: readmodel,writemodel,calcmodel,calfactorplot
;	   readshell,displayshell,readimage,displayimage,
;	   readimages,checkimages,sedimages,displayimages,plotimagefluxes,
;	   marquardtdata
;
; Block 6: mark32cha,mark32oifits,mark32psn,
;	   calibrators2tex,obslog2tex,psn2tex,pti2tex,vel2tex
;
;************************************************************************Block 1
pro list_starcomps,components
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
components=''
if n_elements(star_model) eq 0 then begin
	if n_params() eq 0 then $
		print,'***Error(LIST_COMPONENTS): star_model not defined!'
	return
endif
if num_star() gt 0 then components=star_model.component
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_bincomps,components
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
components=''
if n_elements(binary_model) eq 0 then begin
	if n_params() eq 0 then $
		print,'***Error(LIST_COMPONENTS): binary_model not defined!'
	return
endif
if num_binary() gt 0 then components=binary_model.component
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_modcomp,component,method
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then return
if n_elements (component) eq 0 then return
;
; Always list all parameters
; method=0
;
print,'_______________________________________________________________'
index=where(binary_model.component eq component,count)
if count gt 0 then begin
	tags=tag_names(binary_model) & ntags=n_elements(tags)
	if method eq 0 then it=indgen(ntags)
	if method eq 1 then it=indgen(ntags-2)
	if method eq 2 then it=[0,1,2,3,13,14]
	for k=0,n_elements(it)-1 do begin
		i=it(k)
		if n_elements(binary_model(index).(i)) eq 1 then begin
			result=size(binary_model(index).(i))
			if result(n_elements(result)-2) eq 5 then $
			print,tags(i),'=',binary_model(index).(i), $
				  ' +/- ',binary_error(index).(i) else $
			print,tags(i),'=',binary_model(index).(i)
		endif else begin
			print,tags(i),'=',binary_model(index).(i)
			blank=' '
			for j=0,strlen(tags(i))-6 do blank=blank+' '
			print,blank+' +/- ',binary_error(index).(i)
		endelse
	endfor
endif
tcount=count
index=where(star_model.component eq component,count)
if count gt 0 then begin
	tags=tag_names(star_model)
	for i=0,n_elements(tags)-1 do begin
		if tags(i) eq 'SPOT' then begin
			if total(star_model(index).spot) eq 0 then continue
		endif
		if tags(i) eq 'SPOTPARMS' then begin
			if total(star_model(index).spotparms) eq 0 then continue
		endif
		if n_elements(star_model(index).(i)) eq 1 then begin
			result=size(star_model(index).(i))
			if result(n_elements(result)-2) eq 5 then $
			print,tags(i),'=',star_model(index).(i), $
		          	  ' +/- ',star_error(index).(i) else $
			print,tags(i),'=',star_model(index).(i)
		endif else begin
			print,tags(i),'=',star_model(index).(i)
			blank=' '
                        for j=0,strlen(tags(i))-6 do blank=blank+' '
			print,blank+' +/- ',star_error(index).(i)
		endelse
	endfor
endif
tcount=tcount+count
if tcount eq 0 then begin
	tags=tag_names(gen_model)
	for i=0,n_elements(tags)-1 do begin
		print,tags(i),'=',gen_model.(i)
		blank=' '
                for j=0,strlen(tags(i))-6 do blank=blank+' '
		print,blank+' +/- ',gen_error.(i)
	endfor
endif
print,'______________________________***______________________________'
;
end
;-------------------------------------------------------------------------------
pro list_modcomps,error=error
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then return
;
if n_elements(error) eq 0 then error=0
;
if error then begin
;
print,'Parameter errors: '
help,gen_error,/structure
for i=0,num_binary()-1 do begin
        help,binary_error(i),/structure
        hak,mesg='Hit any key to continue...'
endfor
for i=0,num_star()-1 do begin
        help,star_error(i),/structure
        if i lt num_star()-1 then hak,mesg='Hit any key to continue...' $
                else print,'Info complete.'
endfor
;
endif else begin
;
print,'Parameter values: '
help,gen_model,/structure
for i=0,num_binary()-1 do begin
        help,binary_model(i),/structure
        hak,mesg='Hit any key to continue...'
endfor
for i=0,num_star()-1 do begin
        help,star_model(i),/structure
        if i lt num_star()-1 then hak,mesg='Hit any key to continue...' $
                else print,'Info complete.'
endfor
;
endelse
;
end
;-------------------------------------------------------------------------------
pro list_poscomps,components
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
components=''
if n_elements(positions) eq 0 then begin
	print,'***Error(LIST_POSCOMPS): no data!'
	return
endif
components=[positions.component]
components=components(uniq(components,sort(components)))
;
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_velcomps,components
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
components=''
if n_elements(velocities) eq 0 then begin
	print,'***Error(LIST_VELCOMPS): no data!'
	return
endif
components=[velocities.component]
components=components(uniq(components,sort(components)))
;
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_magcomps,components
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
components=''
if n_elements(magnitudes) eq 0 then begin
	print,'***Error(LIST_MAGCOMPS): no data!'
	return
endif
components=[magnitudes.component]
components=components(uniq(components,sort(components)))
;
if n_params() eq 0 then print,components
;
end
;-------------------------------------------------------------------------------
pro list_filters,filters
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
filters=''
if n_elements(magnitudes) eq 0 then begin
        print,'***Error(LIST_WAVELENGTHS): no data!'
        return
endif
filters=[magnitudes.filter]
filters=filters(uniq(filters,sort(filters)))
;
if n_params() eq 0 then print,filters
;
end
;-------------------------------------------------------------------------------
pro list_nights,nights
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
nights=GenInfo.Date+' '+GeoInfo.SystemId+' '+GenInfo.ConfigId
;
if n_params() eq 0 then print,nights
;
end
;-------------------------------------------------------------------------------
pro list_summary_amoeba
;
common ModelFit,parameters,ds_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common AuxData,parallaxes,k1,k2,vsini
;
print,''
print,'_________________________________________________________'
if n_elements(scans) ne 0 then begin
	print,'Interferometry loaded, weight=',ds_options.i
	print,'    Number of nights:',n_elements(GeoInfo.date)
	print,'  V2',string(9B),'  VP',string(9B), $
	      '  TA',string(9B),'  TP',string(9B),'  FL'
	print,ds_options.v2 ge 1,string(9B),ds_options.vp ge 1,string(9B), $
	      ds_options.ta ge 1,string(9B),ds_options.tp ge 1,string(9B), $
	      ds_options.fl ge 1
endif
if n_elements(positions) ne 0 then begin
	print,'Astrometry loaded, weight=',ds_options.a
	print,'    Number of positions:',n_elements(positions)
	list_poscomps,poscomps
	print,'    Components: ',poscomps
endif
if n_elements(velocities) ne 0 then begin
	print,'Spectroscopy loaded, weight=',ds_options.s
	print,'    Number of velocities:',n_elements(velocities)
	list_velcomps,velcomps
	print,'    Components: ',velcomps
endif
if n_elements(magnitudes) ne 0 then begin
	print,'Photometry loaded, weight=',ds_options.p
	print,'    Number of magnitudes:',n_elements(magnitudes)
	if ds_options.pm eq 0 then print,'Photometry type: time series.'
	if ds_options.pm eq 1 then print,'Photometry type: model components.'
endif
if n_elements(parallaxes) ne 0 then begin
	if ds_options.px gt 0 then print,'Parallax set, weight=',ds_options.px
	if ds_options.px lt 0 then print,'Parallax set, value= -1'
	if ds_options.px eq 0 then print,'Parallax not set'
endif
if n_elements(k2) ne 0 then begin
	print,'K2 set, weight=',ds_options.k2
endif
if n_elements(vsini) ne 0 then begin
	print,'Vsin(i) set, weight=',ds_options.vsini
endif
print,'___________________________***___________________________'
;
end
;************************************************************************Block 2
pro load_interferometry,files_in
;
; Procedure to load multiple .cha/.xdr/.*fits scan averaged data files and store
; each in memory using storenight. Expand GenConfig and GeoParms into arrays of
; structures, geninfo and geoinfo, respectively, to have some configuration data
; readily available for all stored data.
;
; Modified version for reading pndrs *calibrated* data files.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,StarTable,Notes
common LocalAmoebaBuffer,BUFFERSIZE,BUFFERLIMIT
common ModelFit,parameters,ds_options
;
common LocalChaSel,chafiles
common LocalFitSel,oiffiles
;
; Next 4 lines commented 2016-02-24
; Clear buffer
; freememory
; GenInfo=replicate(GenConfig,1)
; GeoInfo=replicate(GeoParms,1)
;
files=file_search(strtrim(files_in,2))
if strlen(files(0)) eq 0 then begin
	print,'***Error(LOAD_INTERFEROMETRY): files not found!'
	return
endif
num_files=n_elements(files)
max_scans=num_files*10	; total number of observations
ds_options.i=0
; Next two variables for associating stars and calibrators (VLTI only)
stars=strarr(num_files)
bflag=strarr(num_files)+'.'
;
; Combine if all files are FITS for same night, array, and configuration
temp_do_not_keep=0
f_ext=unique(extname(files))
if num_files gt 1 and n_elements(f_ext) eq 1 and f_ext(0) EQ 'fits' then begin
	print,'Please wait, reading OIFITS files...'
	date_obs=strarr(num_files)
	arr_config=strarr(num_files)
	spec_config=strarr(num_files)
;	Environmental data: T, pressure, %hum., seeing, coherence time, PWV
;	Read here (one per file!) and pass to combined file in next step.
;	Note that a single file may however contain more than one observation!
	emjd=dblarr(num_files)
	temp=fltarr(num_files)
	pres=fltarr(num_files)
	rhum=fltarr(num_files)
	s_start=fltarr(num_files)	; seeing
	s_end=fltarr(num_files)
	t_start=fltarr(num_files)	; T0
	t_end=fltarr(num_files)
	p_start=fltarr(num_files)	; PVW
	p_end=fltarr(num_files)
;
	for i=0,num_files-1 do begin
		d=mrdfits(files(i),0,h,/silent)
		if i eq 0 then first_header=h	; Header for the merged file
		obsname=fitshparse(h,'OBS.NAME',/quiet)	; OB name
		dateobs=fitshparse(h,'DATE-OBS',/quiet)
;		Ignore these files (e.g., PIONIER calibrations)
		if strpos(obsname,'SpectralCalib') ge 0 then files(i)=''
		if strpos(obsname,'Calibrations') ge 0 then files(i)=''
		if strlen(files(i)) eq 0 then continue
;		Extract target name
		stars(i)=strjoin( $
			nameparse(fitshparse(h,'OBJECT',/quiet),'_'), $
			'')
;		VLTI: if CAL, OBJECT is "STD"
		if strpos(stars(i),'STD') eq 0 then $ ;stars(i)=obsname
		stars(i)=strjoin( $
			nameparse(fitshparse(h,'OBS.TARG.NAME',/quiet),'_'), $
			'')
;		VLTI: is this a calibrator observation?
		if strpos(obsname,'CAL_') eq 0 then begin
			bflag(i)='C'
		endif
;		Extract valid OYSTER star ID
		stars(i)=cri_vlti(fitshparse(h,'OBS.TARG.NAME',/quiet))
;		Read header information
		temp(i)=fitshparse(h,'ISS.AMBI.TEMP',/quiet)
		pres(i)=fitshparse(h,'ISS.AMBI.PRES',/quiet)
		rhum(i)=fitshparse(h,'ISS.AMBI.RHUM',/quiet)
		s_start(i)=float(fitshparse(h,'ISS.AMBI.FWHM.START',/quiet))
		s_end(i)=float(fitshparse(h,'ISS.AMBI.FWHM.END',/quiet))
		t_start(i)=float(fitshparse(h,'ISS.AMBI.TAU0.START',/quiet))
		t_end(i)=float(fitshparse(h,'ISS.AMBI.TAU0.END',/quiet))
		p_start(i)=float(fitshparse(h,'ISS.AMBI.IWV.START',/quiet))
		p_end(i)=float(fitshparse(h,'ISS.AMBI.IWV.END',/quiet))
		fits_info,files(i),n_ext=n_ext,extname=ext,/silent
		index=where(strpos(ext,'OI_VIS2') ge 0,count)
		d=mrdfits(files(i),index(0),h,/silent)
		emjd(i)=mean(d.mjd)
;		
;		Get info on array stations (e.g., A0)/telescopes (e.g., UT1)
		exten=find_oiextn(files(i),'OI_ARRAY')
		oidata=mrdfits(files(i),exten,h,/silent)
		sta_names=strcompress(oidata.sta_name,/remove_all)
		sta_index=oidata.sta_index
		tel_names=strarr(max(sta_index)+1)
		tel_names(sta_index)=strcompress(oidata.tel_name,/remove_all)
;		Get info on UT of local midnight
		systemid=strtrim(fitshparse(h,'ARRNAME',/quiet))
		night_start=24-system_config(systemid(0),'MIDNIGHT')
;		Read file header for time of observation
		d=mrdfits(files(i),0,h,/silent)
		dateobs=fitshparse(h,'DATE-OBS',/quiet)
		if strlen(dateobs) eq 0 then begin
			fits_info,files(i),n_ext=n_ext,extname=ext,/silent
			index=where(strpos(ext,'OI_VIS2') ge 0,count)
			d=mrdfits(files(i),index(0),h,/silent)
			date_obs(i)=jd2date(d(0).mjd,/mjd)
		endif else date_obs(i)=strmid(dateobs,0,10)
		time_start=0
		if strlen(dateobs) gt 10 then $
			time_start=hms2h(strmid(dateobs,11,8))
;		Check on date changes if observations started before UT=0
		if time_start gt night_start then begin
			!date_change=1
			date_obs(i)=nextdate(date_obs(i))
		endif else !date_change=0
;		Extract info on configurations (stations)
		exten=find_oiextn(files(i),'OI_VIS2')
		oidata=mrdfits(files(i),exten,h,/silent)
		arr_config(i)=strjoin( $
			unique(strjoin(tel_names(oidata.sta_index),'-')),'_')
;		Extract info on spectral setups (spectrometers)
		exten=find_oiextn(files(i),'OI_WAVELENGTH')
		oidata=mrdfits(files(i),exten,h,/silent)
		spec_config(i)=strjoin(string(oidata.eff_wave),' ')
	endfor
;
	index=where(strlen(files) gt 0,num_files)
	if num_files eq 0 then begin
		print,'ERROR: no valid observations selected!'
		files_in=''
		return
	endif
	date_obs=date_obs(index)
	arr_config=arr_config(index)
	spec_config=spec_config(index)
	emjd=emjd(index)
	temp=temp(index)
	rhum=rhum(index)
	pres=pres(index)
	s_start=s_start(index)
	s_end=s_end(index)
	t_start=t_start(index)
	t_end=t_end(index)
	p_start=p_start(index)
	p_end=p_end(index)
	files=files(index)
	stars=stars(index)
	bflag=bflag(index)
	if n_elements(unique(date_obs)) eq 1 and $
	   n_elements(unique(arr_config)) eq 1 and $
	   n_elements(unique(spec_config)) eq 1 then temp_do_not_keep=1
endif
;
IF temp_do_not_keep THEN BEGIN
;
;	Create and read combined OIFITS file
;
	chafiles=''
	oiffiles=files
;
;	Open temporary file to combine the matching FITS files
;	Number of columns increases (added by OYSTER) in OI_TARGET extensions
;	MNTSTA column is removed from OI_ARRAY table
;	No changes in OI_WAVELENGTH
;	
	outfile='temp_do_not_keep.fits'
	print,'Please wait, merging OIFITS files...'
;	Sort files by time
	files=files(sort(emjd))
	merge_oidata,outfile=outfile,infiles=files
;	Add header from first file
	modfits,outfile,0,first_header
	get_oifits,outfile,date_obs=date_obs(0)
	spawn,'rm -f '+outfile
	sttbl=stationtable
;
;	Sort environmental data
	si=sort(emjd)
	etim=(emjd(si)-date2jd(date_obs,/mjd))*86400
	temp=temp(si)
	rhum=rhum(si)
	pres=pres(si)
	fwhm=(s_start(si)+s_end(si))/2
	tau0=(t_start(si)+t_end(si))/2
	pwvc=(p_start(si)+p_end(si))/2
;
	n=n_elements(scans)
	if n_elements(temp) eq n then begin
;		Copy environmental data
		scans.r0=(s_start+s_end)/2
		for i=0,n-1 do scans(i).t0=(t_start(i)+t_end(i))/2
		scans.pwv=(p_start+p_end)/2
		scans.temp=temp
		scans.rhum=rhum
		scans.pres=pres
	endif else begin
;		Interpolate if some files had more then one observation
		if n_elements(unique([scans.time,etim])) eq n then begin
		index=where(scans.time lt etim(0),count)
		if count gt 0 then begin
			etim=[scans(index).time,etim]
			fwhm=[scans(index).r0,fwhm]
			tau0=[scans(index).t0(0,0),tau0] ; [NOB,MC,MB]
			pwvc=[scans(index).pwv,pwvc]
			temp=[scans(index).temp,temp]
			rhum=[scans(index).rhum,rhum]
			pres=[scans(index).pres,pres]
		endif
		index=where(scans.time gt etim(0),count)
		if count gt 0 then begin
			etim=[etim,scans(index).time]
			fwhm=[etim,scans(index).r0]
			tau0=[etim,scans(index).t0(0,0)] ; [NOB,MC,MB]
			pwvc=[etim,scans(index).pwv]
			temp=[etim,scans(index).temp]
			rhum=[etim,scans(index).rhum]
			pres=[etim,scans(index).pres]
		endif
		scans.r0=fwhm
		scans.t0=tau0
		scans.pwv=pwvc
		scans.temp=temp
		scans.rhum=rhum
		scans.pres=pres
		endif
	endelse
;
;	Calibrator info from SCI_ and CAL_ files if produced by pndrsReduce
;	(headers include VLTI header with OB names, "CAL" for VLTI calibrator)
	for i=0,n_elements(startable)-1 do begin
		j=where(stars eq startable(i).name,count)
		if count eq 0 then $
		j=where(stars eq startable(i).starid,count)
		if count ge 1 then startable(i).bflag=bflag(j(0))
	endfor
;
;	Calibrator info from SCI_ and CAL_ files if produced by pndrsCalibrate
;	(headers do not include the VLTI file header, must parse name instead)
	f_pndrs_cal=file_search('????-??-??_CAL_*_oidataCalibrated.fits')
	n_cal=n_elements(f_pndrs_cal)
	for i=0,n_cal-1 do begin
		words=nameparse(f_pndrs_cal(i),'_')
		k=n_elements(words)-2
		j=where(startable.starid eq words(k),count)
		if count eq 1 then startable(j).bflag='C'
	endfor
;
;	Special feature: if reading files in PNDRS tree, read oiDiam file
	dir=pwd()
	words=nameparse(dir,'/')
	index=where(strpos(words,'_abcd') gt 0,count)
	civil_date=strmid(words(index(0)),0,10)
	oidiam_file=strjoin(words(0:index(0)),'/')+'/'+civil_date+'_oiDiam.fits'
	if count eq 1 and file_test(oidiam_file) then begin
		d=mrdfits(oidiam_file,1,h)
		for i=0,n_elements(startable)-1 do begin
			j=where(strtrim(d.target) eq startable(i).name,count)
			if count eq 1 then $
			startable(i).diameter=d(j).diam
		endfor
	endif
;
ENDIF ELSE BEGIN ; temp_do_not_keep=0
;
;	Read and store files separately
;
;	Save lists of files to restore after reading each one of them
	chafiles_to_load=''
	index_cha=where(extname(files) eq 'cha',num_cha)
	if num_cha gt 0 then chafiles_to_load=files(index_cha)
	oiffiles_to_load=''
	index_oif=where(extname(files) eq 'fits',num_oif)
	if num_oif gt 0 then oiffiles_to_load=files(index_oif)
;
; if num_files gt 30 then BUFFERLIMIT=0 else BUFFERLIMIT=200000000L
;
	files=strcompress(files,/remove_all)
	first=1
	j=0
;
	hds_close
;
	for i=0,num_files-1 do begin
	result=file_search(files(i),count=fcount)
	if fcount ne 0 then begin
		if strpos(files(i),'xdr') ge 0 then begin
			get_xdr,files(i),freememory=first
		endif else if strpos(files(i),'fits') ge 0 then begin
			get_oifits,files(i),freememory=first
			print,unique(genconfig.baselineid)
		endif else begin
			get_scandata,files(i),freememory=first
			hds_close
		endelse
		startable.bflag=bflag(i)
		get_stationtable,update=0 ; Do not update station coordinates!
		if first or num_files eq 1 then begin
			first=0
			table=startable
			sttbl=stationtable
		endif else begin
			table=merge_startable(table,startable)
			sttbl=[sttbl,stationtable]
		endelse
		calcastrom,/skipuv	; Do not re-compute uv-coordinates!
		storenight,file=files(i)
		j=j+1
	endif else print,'***Error(LOAD_INTERFEROMETRY): file not found: ', $
			files(i),'!'
	endfor
	if n_elements(table) eq 0 then return else StarTable=table
;
;	Restore filelists
	if num_cha gt 0 then chafiles=files(index_cha)
	if num_oif gt 0 then oiffiles=files(index_oif)
;
; Get names and diameters for FKV and BSC stars in the startable (no OIFITS!)
	starids=startable.starid
	stars=startable.name
	rename_starids,'fkv-bsc'
	index=where(startable.bsc ne 0,count)
	get_diameter
	t=startable
	read_catalogs
	rename_bsc
	if count gt 0 then stars(index)=startable(index).name
	startable=t
	startable.starid=starids
	startable.name=stars
;
ENDELSE	
;
; Compute all estimated visibilities
calcviscal
;
index=uniq(sttbl.stationid,sort(sttbl.stationid))
stationtable=sttbl(index)
;
if num_files eq 1 or temp_do_not_keep then plotinit=init_plot('scan','pt') $
		  		      else plotinit=init_plot('amoeba','pt')
ds_options.i=1
;
; Special section to flag interferometry
if !owner eq 'chummel' and pwd() eq '/science/73leo' then begin
	flag_channels,[11,12,13,14,15,16],/all
	print,'For data in /science73leo, channels 11 to 16 were flagged!'
endif
;
end
;-------------------------------------------------------------------------------
pro load_astrometry,files
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ModelFit,parameters,ds_options
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error(LOAD_ASTROMETRY): no files specified!'
	return
endif
ds_options.a=0
;
for i=0,n_elements(files)-1 do begin
	c=''
	y=0.d0
	r=0.d0
	t=0.d0
	a=0.d0
	b=0.d0
	p=0.d0
	status=dc_read_free(files(i),c,y,r,t,a,b,p, $
		/col,ignore=['!'],resize=[1,2,3,4,5,6,7])
	if status ne 0 then return
	if i eq 0 then begin
		component=c
		rho=r
		theta=t
		emajor=a
		eminor=b
		pa=p
		jy=y
	endif else begin
		component=[component,c]
		rho=[rho,r]
		theta=[theta,t]
		emajor=[emajor,a]
		eminor=[eminor,b]
		pa=[pa,p]
		jy=[jy,y]
	endelse
endfor
;
index=where(emajor lt eminor,count)
if count gt 0 then begin
	print,'***Error: major axes must be larger than minor axes!'
	return
endif
index=where(emajor eq 0 or eminor eq 0,count)
if count gt 0 then begin
	print,'***Error: one or more error ellipse axes zero!'
	return
endif
;
num_scan=n_elements(jy)
positions=replicate(location(),num_scan)
;
RAD=180/pi_circle
;
positions.component=component
positions.rho=rho
positions.theta=theta/RAD	; Note internal units of radians!
positions.emajor=emajor		; Semi-major axis of uncertainty ellipse [mas]
positions.eminor=eminor		; Semi-minor axis of uncertainty ellipse [mas]
positions.pa=pa/RAD		; Note internal units of radians!
positions.jy=jy
;
positions.jd=jy2jd(jy)-2440000d0
positions.date=jd2date(positions.jd+2440000d0)
;
; Code to fake positions
; RAD=180/3.141592653d0
; a=109.d0
; e=0.2d0
; i=32.d0/RAD
; w=99.d0/RAD
; n=30.d0/RAD
; p=974.d0
; t=5675.d0
; o_parms=dblarr(8)
; o_parms(0)=a
; o_parms(1)=e
; o_parms(2)=i
; o_parms(3)=w
; o_parms(4)=n
; o_parms(5)=p
; o_parms(6)=t
; o_parms(7)=1
; jd=positions.jd
; xy=true2app(jd,o_parms,rho,theta)
; positions.theta=theta
; positions.rho=rho
;
print,strtrim(string(n_elements(positions)),2)+' positions read.'
ds_options.a=1
end
;-------------------------------------------------------------------------------
pro load_spectroscopy,files
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ModelFit,parameters,ds_options
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error(LOAD_SPECTROSCOPY): no files specified!'
	return
endif
ds_options.s=0
;
for i=0,n_elements(files)-1 do begin
	c=''
	hjd=0.d0
	v=0.d0
	e=0.d0
	s=0
	f=0
	status=dc_read_free(files(i),c,hjd,v,e,s,f, $
		/col,ignore=['!'],resize=[1,2,3,4,5,6])
	if status ne 0 then return
	if i eq 0 then begin
		component=c
		jd=hjd
		value=v
		error=e
		symbol=s
		color=f>1
	endif else begin
		component=[component,c]
		jd=[jd,hjd]
		value=[value,v]
		error=[error,e]
		symbol=[symbol,s]
		color=[color,f>1]
	endelse
endfor
;
num_scan=n_elements(jd)
velocities=replicate(velocity(),num_scan)
;
RAD=180/pi_circle
;
velocities.component=component
velocities.value=value
velocities.error=error
velocities.jd=jd-2440000d0
velocities.symbol=symbol
velocities.color=color
;
velocities.date=jd2date(velocities.jd+2440000d0)
;
print,strtrim(string(n_elements(velocities)),2)+' velocities read.'
ds_options.s=1
end
;-------------------------------------------------------------------------------
pro load_photometry,files
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common ModelFit,parameters,ds_options
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error(LOAD_PHOTOMETRY): no files specified!'
	return
endif
ds_options.p=0
;
for i=0,n_elements(files)-1 do begin
	c=''
	f=''
	hjd=0.d0
	m=0.d0
	e=0.d0
	status=dc_read_free(files(i),c,f,hjd,m,e, $
		/col,ignore=['!'],resize=[1,2,3,4,5])
	if status ne 0 then return
	if i eq 0 then begin
		component=c
		filter=f
		jd=hjd
		value=m
		error=e
	endif else begin
		component=[component,c]
		filter=[filter,f]
		jd=[jd,hjd]
		value=[value,m]
		error=[error,e]
	endelse
endfor
;
num_scan=n_elements(jd)
magnitudes=replicate(magnitude(),num_scan)
;
RAD=180/pi_circle
;
magnitudes.component=component
magnitudes.filter=filter
for i=0,num_scan-1 do magnitudes(i).wavelength=filter_cw(filter(i)) ; [nm]
magnitudes.value=value
magnitudes.error=error
magnitudes.jd=jd-2440000d0
;
magnitudes.date=jd2date(magnitudes.jd+2440000d0)
;
print,'Magnitudes read.'
if ds_options.pm eq 0 then ds_options.p=1
;
end
;-------------------------------------------------------------------------------
pro flag_visibilities
;
; Flag squared visibility data of the star defined in gen_model where
; they are inconsistent with the model values at the level of 3 sigma 
; or more. Only applies to data loaded in the buffer!
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common AuxData,parallaxes,k1,k2,vsini
;
y=0.d0
ysig=0.d0
ymod=0.d0
;
; Interferometry
if ds_options.i ne 0 then begin
	nights=geninfo.date
	arrays=geoinfo.systemid
	confgs=geninfo.configid
	wf=1/sqrt(ds_options.i)
	for n=0,n_elements(nights)-1 do begin
		night=where(GenInfo.date eq nights(n) $
			and GeoInfo.systemid eq arrays(n) $
			and GenInfo.configid eq confgs(n))
		if n_elements(bufferinfo) gt 1 then $
		loadnight,GenInfo(night(0)).date, $
			  GeoInfo(night(0)).systemid, $
			  GenInfo(night(0)).configid
;
; 		Scan selection
		index=where(scans.starid eq gen_model.starid,NS)
		if ns gt 0 and ds_options.v2 eq 1 then begin
			r=size(scans(index).vissqc,/dim)
			ndata=n_elements(scans(index).vissqc)
			y=reform(scans(index).vissqc,ndata)
			ysig=reform(scans(index).vissqcerr,ndata)
			ymod=reform(scans(index).vissqm,ndata)
			bad=where(abs(y-ymod) gt 2*ysig and ysig gt 0,count)
			if count gt 0 then ysig(bad)=-ysig(bad)
			scans(index).vissqcerr=reform(ysig,r)
			storenight,11l
		endif else print,'No V2 data found for '+nights(n)
	endfor
endif
;
end
;-------------------------------------------------------------------------------
pro flag_channels,channels,all=all
;
; Flag squared visibility data of the star defined in gen_model 
; in the given channels. Only applies to data loaded in the buffer!
;
; If keyword all is set, edit data for all stars.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common AuxData,parallaxes,k1,k2,vsini
;
; Check dimensions of scans.vissq
if n_elements(size(scans.vissqc)) ne 7 then begin
	print,'***Error(FLAG_CHANNELS): procedure can only handle 4 dimensions!'
	return
endif
;
; Check if model file is loaded
if not keyword_set(all) and n_elements(gen_model) eq 0 then begin
	print,'***Error(FLAG_CHANNELS): procedure needs a model file!'
	return
endif
;
; Interferometry
if ds_options.i ne 0 then begin
;
nights=geninfo.date
arrays=geoinfo.systemid
confgs=geninfo.configid
wf=1/sqrt(ds_options.i)
for n=0,n_elements(nights)-1 do begin
	night=where(GenInfo.date eq nights(n) $
		and GeoInfo.systemid eq arrays(n) $
		and GenInfo.configid eq confgs(n))
	if n_elements(bufferinfo) gt 1 then $
	loadnight,GenInfo(night(0)).date, $
		  GeoInfo(night(0)).systemid, $
		  GenInfo(night(0)).configid
;
; 	Channel selection
	ns=n_elements(scans)
	if not keyword_set(all) then $
		index=where(scans.starid eq gen_model.starid,ns) $
		else $
		index=indgen(ns)
;
	if ns gt 0 then begin
;
	if ds_options.v2 eq 1 then begin
		r=size(scans(index).vissqc,/dim)
		if n_elements(r) ne 4 then begin
		   print,'Error: can only handle VisSq w/4 dimensions!'
		   return
		endif
		for ob=0,r(0)-1 do scans(*).vissqcerr(ob,channels-1,*) $
			       =-1*abs(scans(*).vissqcerr(ob,channels-1,*))
		if n_elements(geninfo) gt 1 then storenight,11l
	endif else print,'No V2 data found for '+nights(n)
;
	if ds_options.tp eq 1 then begin
		r=size(scans(index).triplephasec,/dim)
		if n_elements(r) ne 3 then begin
		   print,'Error: can only handle TriplePhaseC w/3 dimensions!'
		   return
		endif
		for tr=0,r(0)-1 do scans(*).triplephasecerr(tr,channels-1,*) $
			=-1*abs(scans(*).triplephasecerr(tr,channels-1,*))
		if n_elements(geninfo) gt 1 then storenight,11l
	endif else print,'No TP data found for '+nights(n)
;
	if ds_options.ta eq 1 then begin
		r=size(scans(index).tripleampc,/dim)
		if n_elements(r) ne 3 then begin
		   print,'Error: can only handle TripleAmpC w/3 dimensions!'
		   return
		endif
		for tr=0,r(0)-1 do scans(*).tripleampcerr(tr,channels-1,*) $
			=-1*abs(scans(*).tripleampcerr(tr,channels-1,*))
		if n_elements(geninfo) gt 1 then storenight,11l
	endif else print,'No TA data found for '+nights(n)
;
	endif else print,'No data found for this star in '+nights(n)+'!'
endfor
;
endif
;
end
;-------------------------------------------------------------------------------
pro set_parallax,value,error,weight=weight
;
; Units of parallax are mas. Set weight=0 to have parallax ignored.
; Set the value to a negative number to only enforce equality of parallaxes
; for all binary components.
;
common AuxData,parallaxes,k1,k2,vsini
common ModelFit,parameters,ds_options
;
if n_elements(value) eq 0 then begin
	print,'Usage: set_parallax,value,error,weight=weight'
	print,'Set value to -1 to enforce equality of all parallaxes'
	print,'To remove parallax, set value or weight to zero'
	return
endif
;
if value eq 0 then begin
	parallaxes.value=0
	ds_options.px=0
	return
endif
;
if n_elements(weight) eq 0 then weight=1 
if n_elements(parallaxes) eq 0 then parallaxes=alloc_parallax()
;
if weight eq 0 then begin
	ds_options.px=0
	print,'Weight of parallax set to zero.'
	return
endif
if value lt 0 then begin
	ds_options.px=-1
	if n_elements(error) eq 0 then error=1
	parallaxes.error=error
	return
endif
if n_params() ne 2 then begin
	print,'***Error(SET_PARALLAX): please provide value AND error!'
	return
endif
parallaxes.value=value
parallaxes.error=error
ds_options.px=weight
;
end
;-------------------------------------------------------------------------------
pro set_k1,value,error,weight
;
common AuxData,parallaxes,k1,k2,vsini
common ModelFit,parameters,ds_options
;
if n_params() eq 1 then begin
	print,'***Error(SET_K2): please provide value AND error!'
	return
endif
if n_params() eq 0 then ds_options.k1=0 else begin
	k1=alloc_k1()
	k1.value=value
	k1.error=error
	if n_elements(weight) eq 0 then weight=1 
	ds_options.k1=weight
endelse
;
end
;-------------------------------------------------------------------------------
pro set_k2,value,error,weight
;
common AuxData,parallaxes,k1,k2,vsini
common ModelFit,parameters,ds_options
;
if n_params() eq 1 then begin
	print,'***Error(SET_K2): please provide value AND error!'
	return
endif
if n_params() eq 0 then ds_options.k2=0 else begin
	k2=alloc_k2()
	k2.value=value
	k2.error=error
	if n_elements(weight) eq 0 then weight=1 
	ds_options.k2=weight
endelse
;
end
;-------------------------------------------------------------------------------
pro set_vsini,value,error,weight
;
common AuxData,parallaxes,k1,k2,vsini
common ModelFit,parameters,ds_options
;
if n_params() eq 1 then begin
	print,'***Error(SET_VSINI): please provide value AND error!'
	return
endif
if n_params() eq 0 then ds_options.vsini=0 else begin
	vsini=alloc_vsini()
	vsini.value=value
	vsini.error=error
	if n_elements(weight) eq 0 then weight=1 
	ds_options.vsini=weight
endelse
;
end
;-------------------------------------------------------------------------------
pro add_component,component,origin,reference
;
; e.g. add_component,'B-C','A-C','A-B' does B-C=A-C-(A-B)
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
n_old=n_elements(positions)
index=where(positions.component eq origin,n_new)
new_positions=replicate(location(),n_old+n_new)
new_positions(0:n_old-1).component=positions.component
new_positions(0:n_old-1).date=positions.date
new_positions(0:n_old-1).jy=positions.jy
new_positions(0:n_old-1).jd=positions.jd
new_positions(0:n_old-1).rho=positions.rho
new_positions(0:n_old-1).theta=positions.theta
new_positions(0:n_old-1).rhom=positions.rhom
new_positions(0:n_old-1).thetam=positions.thetam
new_positions(0:n_old-1).emajor=positions.emajor
new_positions(0:n_old-1).eminor=positions.eminor
new_positions(0:n_old-1).pa=positions.pa
;
RAD=180/!pi
jd0=2440000.d0
;
for i=0,n_new-1 do begin
	p=reform(binarypos(jd0+positions(index(i)).jd,reference))
	rar=p(0)*sin(p(1)/RAD)
	dcr=p(0)*cos(p(1)/RAD)
	rao=positions(index(i)).rho*sin(positions(index(i)).theta)
	dco=positions(index(i)).rho*cos(positions(index(i)).theta)
	dra=rao-rar
	ddc=dco-dcr
	new_positions(n_old+i).component=component
	new_positions(n_old+i).rho=sqrt(dra^2+ddc^2)
	new_positions(n_old+i).theta=atan(dra,ddc)
	new_positions(n_old+i).date=positions(index(i)).date
	new_positions(n_old+i).jy=positions(index(i)).jy
	new_positions(n_old+i).jd=positions(index(i)).jd
	new_positions(n_old+i).emajor=positions(index(i)).emajor
	new_positions(n_old+i).eminor=positions(index(i)).eminor
	new_positions(n_old+i).pa=positions(index(i)).pa
endfor
;
positions=new_positions
;
end	
;************************************************************************Block 3
pro storenight,mode,file=file
;
; Store one night's data (scans) in memory (or on disk if too big). 
; These data can be retrieved later with loadnight.
; mode=10:	Store new night (default)
; mode=11:	Overwrite existing night
;
; Note that GenInfo and bufferinfo already have to have an entry for
; this currently loaded night before this function can be called!
;
; Also note: as a general rule, OYSTER does not buffer a night if it is the
; only one. The buffer should only be used in case there are 2 or more nights
; to be stored. 
;
common LocalAmoebaBuffer,BUFFERSIZE,BUFFERLIMIT
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(file) eq 0 then file='' else file=string(file)
;
; Determine which index of bufferinfo is in use by unique identifier
; composed of date, systemid, and configid
cdate=checkdate()
arrayid=GeoParms.systemid
configid=GenConfig.configid

j=where(GenInfo.date eq cdate $
    and GeoInfo.SystemId eq arrayid $
    and GenInfo.ConfigId eq configid)
j=j(0)
;
if n_elements(mode) eq 0 then mode=10L
mode=long(mode)
;
if mode eq 10 or j lt 0 then begin
	mode=10L
	r=size(bufferinfo)
	if r(2) ne 8 then begin	; Initialize if bufferinfo is not a structure
		j=0
		freememory
		bufferinfo=replicate(nightinfo(),1)
		GeoInfo=replicate(GeoParms,1)
		GenInfo=replicate(allocgenconfig(/geninfo),1)
		GeoInfo(j)=GeoParms
		G=GenInfo(j)
		struct_assign,GenConfig,G
		GenInfo(j)=G
	endif
	if j lt 0 then begin
		print,'Night had not been allocated yet...'
		bufferinfo=[bufferinfo,nightinfo()]
		GeoInfo=[GeoInfo,GeoParms]
		GenInfo0=GenInfo
		GenInfo=replicate(allocgenconfig(/geninfo),1+n_elements(GenInfo))
		for k=0,n_elements(GenInfo0)-1 do begin
			G=GenInfo(k)
			struct_assign,GenInfo0(k),G
			GenInfo(k)=G
		endfor
		j=k
		struct_assign,GenConfig,G
		GenInfo(j)=G
	endif
endif
;
bufferinfo(j).file=file
;
if n_elements(BUFFERLIMIT) eq 0 then BUFFERLIMIT=500000000L
if n_elements(BUFFERSIZE) eq 0 then BUFFERSIZE=0L
;
if sizeof(scans)+BUFFERSIZE gt BUFFERLIMIT then begin
;
	bufferinfo(j).xdr=1
	save,scans,genconfig,filename=cdate+arrayid+configid+'.xdr'
;
endif else begin
;
	bufferinfo(j).xdr=0
;
	if mode ne 11 then $
	BUFFERSIZE=BUFFERSIZE+sizeof(scans)+sizeof(genconfig)
;
	bufferinfo(j).nigc.numsid=genconfig.numsid
	bufferinfo(j).nigc.numoutbeam=genconfig.numoutbeam
;
	bufferinfo(j).nigc.maxtriple=n_elements(genconfig.triplenumchan)
	bufferinfo(j).nigc.maxbaseline=n_elements(genconfig.baselineid(*,0))
	bufferinfo(j).nigc.maxchannel=n_elements(genconfig.wavelength(*,0))
	bufferinfo(j).nigc.maxconfig=n_elements(genconfig.config)
;
	n=0
	fields=tag_names(genconfig)
	for i=0,n_elements(fields)-1 do begin
		data=genconfig.(i)
		dims=size(data)
		if dims(dims(0)+1) eq 7 then begin
                        sl=strlen(data)
                        index=where(sl eq 0,count)
                        if count gt 0 then data(index)=' '
			bufferinfo(j).nigc.maxlen(n)=max(strlen(data))
			n=n+1
		endif
		bufferinfo(j).nigc.size(i,0:n_elements(dims)-1)=dims
		status=linknload(!external_lib,'nightbuffer', $
			dims,data,cdate+arrayid+configid,mode)
		if status then begin
			print,'***Error(STORENIGHT): error storing GenConfig!'
			return
		endif
	endfor
;
	bufferinfo(j).nisc.numscan=n_elements(scans)
;
	n=0
	fields=tag_names(scans)
	for i=0,n_elements(fields)-1 do begin
		data=scans.(i)
		dims=size(data)
		if dims(dims(0)+1) eq 7 then begin
                        sl=strlen(data)
                        index=where(sl eq 0,count)
                        if count gt 0 then data(index)=' '
			bufferinfo(j).nisc.maxlen(n)=max(strlen(data))
			n=n+1
		endif
		bufferinfo(j).nisc.size(i,0:n_elements(dims)-1)=dims
		status=linknload(!external_lib,'nightbuffer', $
			dims,data,cdate+arrayid+configid,mode)
		if status then begin
			print,'***Error(STORENIGHT): error storing scans!'
			return
		endif
	endfor
;
endelse
;
end
;-------------------------------------------------------------------------------
pro loadnight,datum,arrayid,configid,force=force
;
; Load specified night from memory into internal data structure scans.
; Datum can be a combination of date (YYYY-MM-DD) and config string, but
; the latter must not contain any blanks.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
datum=datum(0)
words=nameparse(datum)
datum=words(0)
if n_elements(words) eq 3 then begin
	arrayid=words(1)
	configid=words(2)
endif
if n_elements(arrayid) eq 1 then arrayid=arrayid(0)
if n_elements(configid) eq 1 then configid=configid(0)
;
; Force=1 to force reading from buffer
if n_elements(force) eq 0 then force=1 else force=force gt 0
if not force and datum eq Date then return
;
; ConfigId enables storing different configurations in the same night
if n_elements(configid) eq 0 then configid='DEFAULT'
;
Date=datum
;
j=where(GenInfo.date eq Date $ 
    and GeoInfo.SystemId eq arrayid $
    and GenInfo.configid eq configid,count)
if count eq 0 then begin
        set=Date+' '+arrayid+' '+configid
	print,'***Error(LOADNIGHT): this night was not stored before: '+set+'!'
	return
endif
j=j(0)
;
; This is now buffered
; GenConfig=GenInfo(j)
GeoParms=GeoInfo(j)
;
mode=20L
;
if bufferinfo(j).xdr then begin
;
; This restores both scans and GenConfig
restore,date+arrayid+configid+'.xdr'
;
endif else begin
;
GenConfig=allocGenConfig(bufferinfo(j).nigc.numsid, $
			 bufferinfo(j).nigc.numoutbeam, $
			 bufferinfo(j).nigc.maxtriple, $
			 bufferinfo(j).nigc.maxbaseline, $
			 bufferinfo(j).nigc.maxchannel, $
			 bufferinfo(j).nigc.maxconfig)
;
n=0
fields=tag_names(GenConfig)
for i=0,n_elements(fields)-1 do begin
	data=GenConfig.(i)
	dims=size(data)
;	dims=bufferinfo(j).nigc.size(i,*)
;	data=make_array(size=dims(where(dims ne 0)))
	if dims(dims(0)+1) eq 7 then begin
		blank=''
		for k=0,bufferinfo(j).nigc.maxlen(n)-1 do blank=blank+' '
		data(*)=blank
		n=n+1
	endif
	status=linknload(!external_lib,'nightbuffer', $
		dims,data,Date+arrayid+configid,mode)
	if status then begin
		print,'***Error(LOADNIGHT): error loading data!'
		print,Date+arrayid+configid
		return
	endif
	if dims(dims(0)+1) eq 7 then data=strtrim(data,2)
	GenConfig.(i)=data
endfor
;
; Note: scan() must return dimensions compatible with GenConfig!
num_scans=bufferinfo(j).nisc.numscan
scans=replicate(scan(),num_scans)
;
n=0
fields=tag_names(scans)
for i=0,n_elements(fields)-1 do begin
	data=scans.(i)
	dims=size(data)
;	dims=bufferinfo(j).nisc.size(i,*)
;	data=make_array(size=dims(where(dims ne 0)))
	if dims(dims(0)+1) eq 7 then begin
		blank=''
		for k=0,bufferinfo(j).nisc.maxlen(n)-1 do blank=blank+' '
		data(*)=blank
		n=n+1
	endif
	status=linknload(!external_lib,'nightbuffer', $
		dims,data,Date+arrayid+configid,mode)
	if status then begin
		print,'***Error(LOADNIGHT): error loading data!'
		return
	endif
	if dims(dims(0)+1) eq 7 then data=strtrim(data,2)
	scans.(i)=data
endfor
;
endelse
;
!qiet=1 & get_scantable & !qiet=0
;
Date=GeoParms.Date
SystemId=GeoParms.SystemId
;
end
;-------------------------------------------------------------------------------
pro freememory
;
; Clear the data buffer memory.
;
common LocalAmoebaBuffer,BUFFERSIZE,BUFFERLIMIT
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
ddims=0L
ddata=' '
ddate=' '
mode=0L
;
status=linknload(!external_lib,'nightbuffer',ddims,ddata,ddate,mode)
BUFFERSIZE=0
;
; Remove disk buffer files
if n_elements(bufferinfo) ne 0 then begin
	r=size(bufferinfo)
	if r(2) eq 8 then begin	; is a structure and not zero
	index=where(bufferinfo.xdr eq 1,count)
	if count gt 0 then begin
		print,'Please wait while temporary files are removed...'
		files=geninfo(index).date $
		     +geoinfo(index).systemid $
		     +geninfo(index).configid+'.xdr'
		for i=0,n_elements(files)-1 do spawn,'rm -f '+files(i),r
	endif
	endif
endif
bufferinfo=0
;
end
;************************************************************************Block 4
pro fitwaveparms
;
; Obsolete!
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
num_wave=n_elements(gen_model.wavelengths)
;
; Stars:
degree=num_wave-1
ld_fitcoeffs=dblarr(degree+1,num_star())
flux_fitcoeffs=dblarr(degree+1,num_star())
for i=0,num_star()-1 do begin
	ld_fitcoeffs(*,i)= $
		poly_fit(gen_model.wavelengths,star_model(i).ld_coeffs,degree)
	flux_fitcoeffs(*,i)= $
		poly_fit(gen_model.wavelengths,star_model(i).fluxes,degree)
endfor
star_model.ld_fitcoeffs=ld_fitcoeffs
star_model.flux_fitcoeffs=flux_fitcoeffs
;
; Binaries:
; if num_binary() gt 0 then begin
; 	dm_fitcoeffs=dblarr(degree+1,num_binary())
; 	for i=0,num_binary()-1 do begin
; 		dm_fitcoeffs(*,i)= $
; 			poly_fit(gen_model.wavelengths, $
; 				binary_model(i).magdifference,degree)
; 	endfor
; 	binary_model.dm_fitcoeffs=dm_fitcoeffs
; endif
;
end
;-------------------------------------------------------------------------------
pro adjustfluxes,component,flux
;
; Obsolete!
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if componentparse(component,comp_1,comp_2) eq -1 then return
if strlen(comp_1) eq 1 then s1=where(star_model.component eq comp_1) else s1=-1
if strlen(comp_2) eq 1 then s2=where(star_model.component eq comp_2) else s2=-1
s1=s1(0) & s2=s2(0)
;
i=where(binary_model.component eq component)
magdifference=binary_model(i).magdifference
num_wave=n_elements(gen_model.wavelengths)
f1=dblarr(num_wave)
f2=dblarr(num_wave)
;
for j=0,num_wave-1 do begin
	f2(j)=flux(j)/(1+10^(magdifference(j)/2.5))
	f1(j)=flux(j)-f2(j)
endfor
if s1 ne -1 then star_model(s1).flux=f1 else adjustfluxes,comp_1,f1
if s2 ne -1 then star_model(s2).flux=f2 else adjustfluxes,comp_1,f2
;
end
;-------------------------------------------------------------------------------
pro adjustmasses,component,masstotal
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(component) eq 0 then begin
	component=topbincomp()
	masstotal=gen_model.sm
endif
;
i=where(binary_model.component eq component)
massratio=binary_model(i).massratio
;
if modelparse(component,comp_1,comp_2) eq -1 then return
if strlen(comp_1) eq 1 then s1=where(star_model.component eq comp_1) else s1=-1
if strlen(comp_2) eq 1 then s2=where(star_model.component eq comp_2) else s2=-1
s1=s1(0) & s2=s2(0)
;
m1=masstotal/(1+massratio)
m2=masstotal-m1
if s1 ne -1 then star_model(s1).mass=m1 else adjustmasses,comp_1,m1
if s2 ne -1 then star_model(s2).mass=m2 else adjustmasses,comp_1,m2
;
end
;************************************************************************Block 5
pro readmodel,modelfile
;
; This procedure sets up a hierarchical model format and reads a file
; with the model information ("commands"). It also initializes the two
; structures identical to the model structures, which hold the
; parameter error information. A model may include images (or image cubes),
; these require the definition of the cellsize [mas], unless the image
; has been loaded before calling this procedure.
;
; The model file may also contain commands which analyze the model parameters
; and print the derived information (e.g., parallax).
;
common StarBase,StarTable,Notes
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common AstroConstants,m_sun,r_sun,a_unit,s_year
common ReadImagesLocal,message_counter
;
message_counter=0
;
f=file_search(modelfile)
if strlen(f(0)) eq 0 then begin
	print,'Error(READMODEL): file not found!'
	return
endif else modelfile=f(0)
;
; Reset models to state 'undefined'
;undefine,gen_model
;undefine,star_model
;undefine,binary_model
;undefine,gen_error
;undefine,star_error
;undefine,binary_error
;
; Disabled 12.12.2023, function now in ww_modelfit
; Initialize modelfit
; if init_modelfit() ne 0 then return
;
; If in XDR format, just restore all components
if strpos(modelfile,'xdr') eq strlen(modelfile)-3 then begin
	restore,modelfile
	return
endif
;
; Read command file into string array
commands=''
status=dc_read_fixed(modelfile,commands,resize=[1],format='(a80)',/col)
if status ne 0 then begin
	print,'***Error(READMODEL): error reading file or file not found!'
	return
endif
;
; Remove comment lines (begin with ';') 
commands=commands[where(strmid(strcompress(commands,/remove_all),0,1) ne ';')]
;
; Remove comments at end of commands (after ';')
for i=0,n_elements(commands)-1 do begin
	command_i=strsplit(commands(i),';',/extract)
	commands(i)=command_i(0)
endfor
;
; Get info on number of stars (single, double)
; Previous implementation:
num_binaries=10
num_stars=2*num_binaries
; Implementation up to 4 stars
; index=where(strpos(commands,'name') ge 0,num_stars)
; if num_stars gt 4 then begin
; 	print,'Multiple systems with more than 4 stars are not supported!'
; 	retall
; endif
; num_binaries=num_stars-1	; OK at least up to 4 stars
;
; Wavelengths of spectral channels
;index=where(strpos(strupcase(commands),'WAVELENGTHS') ne -1,count)
;if count ge 1 then for i=0,count-1 do r=execute(commands(index(i))) else begin
;	print,'***Error(READMODEL): Check definition of WAVELENGTHS!'
;	return
;endelse
;
; Get info on number of spectral channels
; index=where(strpos(strupcase(commands),'NUMSPECCHAN') ne -1,count)
; if count ge 1 then for i=0,count-1 do r=execute(commands(index(i)))
;
; Check for info on the wavelength grid
index=where(strpos(strupcase(commands),'WAVELENGTHS') ne -1,count)
if count ge 1 then begin
	for i=0,count-1 do r=execute(commands(index(i)))
	numspecchan=n_elements(wavelengths)
endif else begin
	print,'***Error(READMODEL): Add definition of WAVELENGTHS!'
	return
endelse
;
; Assign StarID using the last entry
; index=where(strpos(strupcase(commands),'STARID') ne -1,count)
; r=execute(commands(index(count-1)))
; get_startable,starid
;
; General parameters
num_wave=n_elements(wavelengths)
if num_wave eq 0 then begin
	wavelengths=1
	num_wave=1
endif
num_spot=2
starid=''
px=0.d0					; Parallax [mas]
rv=0.d0					; radial velocity [km/s]
sm=1.d0					; System mass
;
; Star parameters
name    	=strarr(num_stars)	; A, B, C, D,...from model file
wmc		=strarr(num_stars)	; WMC designation, e.g. 'Aa'
type		=intarr(num_stars)+1	; Component model type
sed		=strarr(num_stars)	; SED data file
model		=strarr(num_stars)	; Model atmosphere file
cellsize	=dblarr(num_stars)	; Cell size for images w/out header
mass		=dblarr(num_stars)+1	; solar masses
diameter	=dblarr(num_stars)	; Diameter of disk or inner rim [mas]
width    	=dblarr(num_stars)	; Disk width / inner radius [0-1]
ratio		=dblarr(num_stars)+1	; Minor axis/major axis
pa		=dblarr(num_stars)	; Major axis position angle, 0=N
omega      	=dblarr(num_stars)	; omega (rotation) / omega breakup  
omega      	=dblarr(num_stars)	; ratio of axial to orbital rate
tilt       	=dblarr(num_stars)	; inclination of rot. axis, 90=eq.on
gr		=dblarr(num_stars)	; exponent in gravity darkening law
albedo		=dblarr(num_stars)	; bolometric albedo
teff		=dblarr(num_stars)	; effective temperature
alpha		=dblarr(num_stars)	; accretion disk: T ~ r^alpha
logg		=dblarr(num_stars)	; log(g)
xoff		=dblarr(num_stars)	; RA offset in mas (positive to East)
yoff		=dblarr(num_stars)	; Dec offset in mas (positive to North)
spot		=dblarr(4,num_stars)	; Teff-factor,radius,angle,diameter
spotparms	=dblarr(4,num_spot,num_stars)
magnitudes	=dblarr(num_wave,num_stars)
;
; Binary parameters
if num_binaries ge 1 then begin
component	=strarr(num_binaries)	; A-B, AB-C, AB-CD,...
method		=intarr(num_binaries)
wdtype		=intarr(num_binaries)
wdband		=intarr(num_binaries)
massratio	=dblarr(num_binaries)	; 
semimajoraxis	=dblarr(num_binaries)	; mas
inclination	=dblarr(num_binaries)	; degrees
ascendingnode	=dblarr(num_binaries)	; degrees, component RV positive
eccentricity	=dblarr(num_binaries)
periastron	=dblarr(num_binaries)	; degrees, of primary!
apsidalmotion	=dblarr(num_binaries)	; degrees/year
epoch		=dblarr(num_binaries)	; secondary at periastron
period		=dblarr(num_binaries)	; days
rho             =dblarr(num_binaries)	; mas
theta           =dblarr(num_binaries)	; degrees
endif
;
; The model may contain analytic commands which can only be executed after 
; the model was updated! Here we save handling of modelpx() for later.
; The line after can be used for other commands with the same reason.
;
command_i=''	; blank entry
command_n=0
; if n_elements(startable) ne 0 then begin
	for i=0,n_elements(commands)-1 do begin
		if strpos(commands(i),'modelpx') ge 0 then begin
			command_i=[command_i,commands(i)]
			command_n=command_n+1
		endif else if strpos(commands(i),'some_other_parameter') ge 0 $
		then begin
			command_i=[command_i,commands(i)]
			command_n=command_n+1
		endif else begin
;			print,commands(i)
			r=execute(commands(i))	; prints results to screen
		endelse
	endfor
; 	Remove the first blank entry
	if n_elements(command_i) gt 1 then $
		command_i=command_i(1:n_elements(command_i)-1)
; endif
;
; Allocate structures, arrays, store information, and remove zero entries
;
; General model parameters
gen_model=alloc_gen_model(num_wave)
wavelengths=double(wavelengths)
si=sort(wavelengths)
gen_model.wavelengths=wavelengths(si)
gen_error=gen_model
if total(strlen(starid)) gt 0 then begin
	gen_model.starid=starid
	gen_model.rv=rv
	gen_model.px=px
	gen_model.sm=sm
	if n_elements(StarTable) eq 0 then get_startable,starid
;
; 	If StarTable has only one entry, use that name for gen_model.starid
	if n_elements(startable) eq 1 then gen_model.starid=startable.starid
	index=where(StarTable.starid eq gen_model.starid,count)
	if count eq 0 then begin
		addstar,starid
		index=where(StarTable.starid eq starid)
	endif
	gen_model.ra=StarTable(index(0)).ra
	gen_model.dec=StarTable(index(0)).dec
;	set_parallax,StarTable(index(0)).px*1000,StarTable(index(0)).pxe,0
endif
;
; Star model parameters
star_struct=alloc_star_struct(num_wave,num_spot)
star_model=replicate(star_struct,num_stars)
star_error=star_model
;alphabet=['A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q']
;for i=0,num_stars-1 do star_model(i).component=alphabet(i)
;
if total(strlen(starid)) gt 0 then begin
	star_model.component		=name
	star_model.wmc			=wmc
	star_model.type			=type
	star_model.sed			=sed
	star_model.model		=model
	star_model.cellsize		=cellsize
	star_model.mass			=mass
	star_model.diameter		=diameter
	star_model.width     		=width
	star_model.ratio		=ratio
	star_model.pa			=pa
	star_model.omega  		=omega  
	star_model.tilt   		=tilt
	star_model.gr			=gr
	star_model.albedo		=albedo
	star_model.teff			=teff
	star_model.alpha		=alpha
	star_model.logg			=logg
	star_model.xoff			=xoff
	star_model.yoff			=yoff
	star_model.spot			=spot
	star_model.spotparms		=spotparms
	star_model.magnitudes		=reform(magnitudes(si,*))
;
	star_error.component		=name
;
	star_model=star_model(where(strlen(star_model.component) ne 0))
	star_error=star_error(where(strlen(star_model.component) ne 0))
endif
;
; Binary model parameters
if num_binaries ge 1 then begin
binary_struct=alloc_binary_struct()
binary_model=replicate(binary_struct,num_binaries)
binary_error=binary_model
index=where(epoch gt 0,count)
if count gt 0 then epoch(index)=epoch(index)-2440000.d0	; check this!
if total(strlen(component)) gt 0 then begin
	binary_model.component		=component
	binary_model.method		=method
	binary_model.wdtype		=wdtype
	binary_model.wdband		=wdband
	binary_model.massratio		=massratio
	binary_model.semimajoraxis	=semimajoraxis
	binary_model.inclination	=inclination
	binary_model.ascendingnode	=ascendingnode
	binary_model.eccentricity	=eccentricity
	binary_model.periastron		=periastron
	binary_model.apsidalmotion	=apsidalmotion
	binary_model.epoch		=epoch
	binary_model.period		=period
	binary_model.rho		=rho
	binary_model.theta		=theta
;
	binary_error.component		=component
;
;	The following was found not to work anymore on May 6, 2024
;	c=fltarr(num_binaries)
;	for i=0,num_binaries-1 do $
;	 for j=1,n_tags(binary_model)-1 do c(i)=c(i)+total(binary_model(i).(j))
;	index=where(c ne 0,count)
;	if count eq num_binaries then $
;		print,'Warning: maximum of ' $
;		      +string(num_binaries,format='(i2)') $
;		      +' binary model parameters found, please check model!'
;	binary_model=binary_model(where(c ne 0))
;	binary_error=binary_error(where(c ne 0))
;
;	Replaced on May 6, 2024
	binary_model=binary_model(where(strlen(binary_model.component) ne 0))
	binary_error=binary_error(where(strlen(binary_model.component) ne 0))
endif	; num_binaries
endif
;
; Execute embedded analytic model commands now that the model was created
for j=0,command_n-1 do r=execute(command_i(j))
command_n=0
;
if checkmodel() eq -1 then begin
	print,'Problem with model definition!'
	return
endif
;
if num_binaries ge 1 then begin
	if total(strlen(binary_model.component)) ne 0 then begin
; 		adjustfluxes,topbincomp(),dblarr(num_wave)+1
; 		adjustmasses,topbincomp()
	endif
endif
if n_elements(star_model) gt 1 and total(abs(star_model.teff)) eq 0 then begin
; 	For non-physical models, compute all as blackbody (req. by Pearl)
	star_model.teff=-5555
	print,'Note: setting all Teff to -5555 K!'
endif
; fitwaveparms
;
; Check availability of images for code=12
index=where(star_model.type eq 12,count)
;
for i=0,count-1 do begin
;
if strlen(star_model(index(i)).model) gt 0 then begin
	image=file_search(star_model(index(i)).model)
	if strlen(image(0)) eq 0 then begin
		print,'Error: image file not found: '+star_model(index(i)).model
		return
	endif else begin
		readimages,image,cellsize=cellsize_return
		cellsize(i)=cellsize_return
	endelse
endif
;
endfor
;
; Check availability of images for code=13
index=where(star_model.type eq 13,count)
for i=0,count-1 do begin
	if strlen(star_model(index(i)).model) gt 0 then begin
	image=file_search(star_model(index(i)).model)
	if strlen(image(0)) eq 0 then begin
		print,'Error: image file not found: '+star_model(index(i)).model
		return
	endif else readimage,image
	endif else begin
		print,'Warning: image not specified, assume it is loaded!'
	endelse
endfor
; Check availability of parallax and Teff for code=14
index=where(star_model.type eq 14,count)
if count gt 0 then print,'Note: you must set a value for the parallax!'
for i=0,count-1 do begin
	if star_model(i).teff eq 0 then print,'***Error: Teff must be non-zero!'
endfor
;
print,'Model file read and checked successfully.'
;
end
;-------------------------------------------------------------------------------
pro writemodel,modelfile
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
; If in XDR format, just save all components
if strpos(modelfile,'xdr') eq strlen(modelfile)-3 then begin
	save,gen_model,star_model,binary_model, $
	     gen_error,star_error,binary_error,filename=modelfile
	return
endif
;
; Write model using OYSTER model format
; Not yet implemented
;
end
;-------------------------------------------------------------------------------
pro calcmodel,pearlmv=pearlmv,pearlcv=pearlcv,quiet=quiet,noload=noload
;
; Main model data computation function for interferometric, spectroscopic,
; astrometric, and photometric data. The common block variable ds_options
; specifies by weight which data sets are requested.
;
; Integration over time to account for fringe smearing has been included
; as of version 8.
;
; Option pearlmv: return complex visibility (mv) for self-calibration in Pearl.
; Option pearlcv: return mv and model flux for Pearl to subtract from cv.
; The option pearlcv is useful for the analysis of triple stars.
; These two options cannot be used together!
;
; Keyword noload to just compute model for currently loaded data.
;
common MarquardtFit,fit_options
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,p_magnitudes
common Tables,scantable,bgtable,stationtable
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AuxData,parallaxes,k1,k2,vsini
common RandomModel,RandomSeed
common LocalRocheVis,LDinit
;
RAD=180/pi_circle
RandomSeed=0
LDinit=1
;
; http://www.idlcoyote.com/code_tips/underflow.html
; Disable reporting of underflow math errors
currentExcept = !Except
!Except = 0
void = Check_Math()
;
if n_elements(ds_options) eq 0 then begin
	print,'***Error(CALCMODEL): no data weights specified!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(CALCMODEL): no model!'
	return
endif
if n_elements(fit_options) eq 0 then fit_options=alloc_fit_options()
if n_elements(quiet) eq 0 then quiet=0
if n_elements(noload) eq 0 then noload=0
;
; Section for visibility data...................................................
;
if ds_options.i ne 0 then begin
;
if n_elements(GenInfo) eq 0 then begin
	print,'***Error(CALCMODEL): no data (GENINFO undefined)!'
	return
endif
;
if n_elements(GeoInfo) eq 0 then begin
	print,'***Error(CALCMODEL): no data (GEOINFO undefined)!'
	return
endif
;
; The model is always computed for data of all nights
nights=GenInfo.date
arrays=GeoInfo.systemid
confgs=GenInfo.configid
;
; Save statistics of floating calibration factors
if fit_options.f then begin
	if not quiet then begin
	openw,unit,'calfactors.txt',/get_lun
	print,'Warning: floating calibration option selected!'
	print,'Use calfactorplot to see histograms of factors.'
	endif
;
	floating_notified=0
endif
;
; Do one night at a time
;
FOR n=0,n_elements(nights)-1 DO BEGIN
;
; Load a night
night=where(GenInfo.date eq nights(n) $
        and GeoInfo.systemid eq arrays(n) $
        and GenInfo.configid eq confgs(n),count)
if count eq 0 and n_elements(GenInfo.date) eq 1 then night=0
if strupcase(!version.os) ne 'WIN32' $
	and not noload $
	and n_elements(bufferinfo) gt 1 then $
	loadnight,GenInfo(night(0)).date, $
		GeoInfo(night(0)).systemid, $
		GenInfo(night(0)).configid
; Save copy of uv-coordinates and compute offsets
uvw=scans.uvw
calcastrom,/quiet
uvw_off=uvw-scans.uvw
;
; Select data for model star
index0=where(scans.starid eq gen_model.starid and scantable.code eq 1,NS0)
if NS0 eq 0 then begin
; print,'***Error(CALCMODEL): no data selected for ',gen_model.starid,' on ', $
; 	nights(n)+' '+confgs(n),'!'
goto,NEXT
endif
;
time=abs(scans(index0).time)
r0=median(reform(scans(index0).r0,NS0))
am=median(1./cos(reform(scans(index0).za,NS0)/RAD))
int_time=median(reform(scans(index0).int_time,NS0))
; int_time=median(scantable(index0).stoptime-scantable(index0).starttime)
if not quiet and !int_time ne 0 then $
	print,'Median integration time [s] for '+Date+': ',int_time
;
; Baseline visibilities
;
for ob=0,GenConfig.NumOutBeam-1 do begin
;
; For multiple stellar systems, compute positions and apodize fluxes
if num_binary()*0 ge 1 then begin	; Obsolete
	magnitudes=star_model.magnitudes
	parsedate,date,y,m,d
	xy=reform(modelpos(julian(y,m,d)-2440000.d0))
	radius=sqrt(total(xy^2,2))
	rad2mas=180l*3600l*1000l/!pi
; 	Compute Airy's disk
	lamda=mean(genconfig.wavelength(0:genconfig.numspecchan(ob)-1,ob))
	airy=(lamda/genconfig.diameter(0))*rad2mas
	dm=-2.5*alog10(exp(-(2*radius/airy)^2))
	for i=0,n_elements(star_model)-1 do $
	star_model(i).magnitudes(*)=star_model(i).magnitudes(*)+dm(i)
endif
if num_binary() ge 1 then begin
;	Check for information on variable component magnitudes
;	It is assumed that the filter sequence matches wavelengths!
	if ds_options.pm then begin
		parsedate,date,y,m,d
		jd=julian(y,m,d)-2440000.d0
		wl=gen_model.wavelengths*1000	; [nm]
		index=where(abs(jd-p_magnitudes.jd) eq 0)
		if index(0) ne -1 then begin
		p_mags=p_magnitudes(index).value
		p_comp=p_magnitudes(index).component
		p_filt=p_magnitudes(index).filter
		for k=0,n_elements(star_model)-1 do begin
			index=where(p_comp eq star_model(k).component)
			if index(0) ge 0 then star_model(k).magnitudes=p_mags
		endfor
		endif
	endif
	magnitudes=star_model.magnitudes
	dm=apodize_m(r0,am)
;	Do not change the absolute magnitude of a cached image component
	index=where(star_model.type eq 12,count) & index=index(0)
	if count eq 1 then dm=dm-dm(index)
	index=where(star_model.type ne 0)
	star_model(index).magnitudes=star_model(index).magnitudes+dm(*,index)
endif
;
; Set up grid of wavelengths for band pass integrations
; Note that all functions in filter.pro use [nm] as units for lambda
lambda_grid=system_config(SystemId,'GRID')
if string(lambda_grid(0)) eq 'SPECTROMETER' then begin
	grid_type='SPECTROMETER'
	lambda_grid=genconfig.wavelength(0:genconfig.numspecchan(ob)-1,ob)*1e9
endif else begin
	grid_type='GRID'
	ch=indgen(genconfig.numspecchan(ob))
	jndex=where(lambda_grid*1e-9 le max(genconfig.wavelength(ch,ob) $
					   +genconfig.chanwidth(ch,ob)/2) $
		and lambda_grid*1e-9 ge min(genconfig.wavelength(ch,ob) $
					   -genconfig.chanwidth(ch,ob)/2),count)
;	The next statement was commented between 2012 and 2013, why?
	if count gt 0 then lambda_grid=lambda_grid(jndex)
endelse
lamda=lambda_grid/1d9	; convert to SI units
n_lam=n_elements(lamda)
nu=1/lamda
;
nc=GenConfig.NumSpecChan(ob)
vis=reform(complexarr(nc,NS0),nc,NS0)
pwr=reform(complexarr(nc,NS0),nc,NS0)
;
for bl=0,GenConfig.NumBaseline(ob)-1 do begin
;	Only compute model for scans which have data in at least one channel
;	Example: ups-Sgr-MIRC-2016Jul16.oifits
;	scans = Array[1, 8, 15, 48]: 48 scans, 15 baselines, 8 channels, 1 OB
	e0=reform(scans(*).vissqcerr(ob,0:GenConfig.NumSpecChan(ob)-1,bl) > 0)
	if GenConfig.NumSpecChan(ob) gt 1 then e=total(e0,1) else e=e0
	index=where(scans.starid eq gen_model.starid $
		and scantable.code eq 1 $
		and e gt 0,NS)
	if NS gt 0 then begin
	time=abs(scans(index).time)
	int_time=scans(index).int_time
	vis=reform(complexarr(nc,NS),nc,NS)
	pwr=reform(complexarr(nc,NS),nc,NS)
;
;	Set up integral over int_time
	time=scans(index).time	; Make backup of time stamps
	if !int_time ne 0 then num_int=max(nint(int_time/!int_time)) $
			  else num_int=0
	if num_int mod 2 eq 0 then num_int=num_int+1
	time_step=0
	if num_int gt 1 then $
	time_step=int_time/(num_int-1)	; array, [s]
	vis_grid=dcomplexarr(n_lam,NS)
;	Only compute model for wavelengths with data for SPECTROMETER
	if grid_type eq 'SPECTROMETER' then begin
		if n_lam gt 1 then begin
		if n_elements(scans) eq 1 then ee=e0 $ ; option added 2020
					  else ee=total(reform(e0),2)
		endif else ee=e0
		ei=where(ee gt 0,n_lam_ei)
	endif
	for i=-num_int/2,num_int/2 do begin
		scans(index).time=time+i*time_step(index)
		calcastrom,/quiet
		scans.uvw=scans.uvw+uvw_off
		um=[scans(index).uvw(ob,0,bl,0)*GenConfig.Wavelength(0,ob)]
		vm=[scans(index).uvw(ob,0,bl,1)*GenConfig.Wavelength(0,ob)]
		t=abs(scans(index).time)
		if grid_type eq 'SPECTROMETER' then begin
			u=nu(ei)#um
			v=nu(ei)#vm
			vis_grid(ei,*)=vis_grid(ei,*) $
			+reform(modelvis(t,lamda(ei),fluxes,u,v),n_lam_ei,NS)
		endif else begin
			u=nu#um
			v=nu#vm
			vis_grid=vis_grid $
				+reform(modelvis(t,lamda,fluxes,u,v),n_lam,NS)
		endelse
	endfor
	vis_grid=vis_grid/num_int
	scans(index).time=time
;	calcastrom,/quiet; commented Jan 4, 2020, added the next line
	scans.uvw=uvw
;
;	Bandpass integration
	if grid_type eq 'SPECTROMETER' then begin
;		High R spectrometer
		for sc=0,NS-1 do begin
;			vis(*,sc)=vis_grid(*,sc)/fluxes
;			scans(index(sc)).photometrym(0,ob,0:nc-1)=fluxes
			vis(ei,sc)=vis_grid(ei,sc)/fluxes
			scans(index(sc)).photometrym(0,ob,ei)=fluxes
		endfor
	endif else begin
		for ch=0,nc-1 do begin
			f=system_config(SystemId,'FILTERS')
			if n_elements(f) gt 1 then filter=f(ch,ob) $
					      else filter=f(0)
			if filter(0) eq 'generic_c' then $
				tm=generic_c(lambda_grid,ch,ob) $
						 else $
				tm=call_function(filter,lambda_grid)
			if NS eq 1 then begin
				tm=reform(tm,1,n_lam)
				lam=reform(lamda,1,n_lam)
			endif else lam=lamda
			if total(tm) eq 0 then begin
				print,'***Error(CALCMODEL): wl grid too coarse!'
				return
			endif
;			ABCD fringe detection
			vis(ch,*)=(tm#vis_grid)/total(tm*fluxes)
;			Scanning interferometers
			pwr_grid=reform(abs(vis_grid)^2,n_lam,NS)
;			pwr(ch,*)=(tm^2#pwr_grid)/total(tm^2*fluxes^2)
			pwr(ch,*)=((lam^2*tm^2)#pwr_grid) $
				 /total(tm^2*fluxes^2*lam^2)
;			Store model SED
			scans(index).photometrym(0,ob,ch)=total(tm*fluxes) $
							 /total(tm)
		endfor
	endelse
	dimensions= $
	size(scans(index).complexvis(ob,0:GenConfig.NumSpecChan(ob)-1,bl))
	scans(index).complexvis(ob,0:GenConfig.NumSpecChan(ob)-1,bl)= $
		reform(vis,dimensions(1:dimensions(0)))
;	Added 2016-06-06 for testing 
	scans(index).complexweight(ob,0:GenConfig.NumSpecChan(ob)-1,bl)=1
;
	dimensions= $
	size(scans(index).vissqm(ob,0:GenConfig.NumSpecChan(ob)-1,bl))
;	Use amplitude (default) for ABCD method (e.g. NPOI)
	scans(index).vissqm(ob,0:GenConfig.NumSpecChan(ob)-1,bl)= $
		reform(abs(vis)^2,dimensions(1:dimensions(0)))
	scans(index).visphasem(ob,0:GenConfig.NumSpecChan(ob)-1,bl)= $
		reform(cphase(vis),dimensions(1:dimensions(0)))
	scans(index).diffphasem(ob,0:GenConfig.NumSpecChan(ob)-1,bl)= $
		reform(cphase(vis),dimensions(1:dimensions(0))) $
			-median(cphase(vis))
;	Use power for scanning interferometers (e.g. VINCI, PIONIER)
	if strpos(instrument_id(systemid),'VINCI') ge 0 or $ 
	   strpos(instrument_id(systemid),'PIONIER') ge 0 then $
	scans(index).vissqm(ob,0:GenConfig.NumSpecChan(ob)-1,bl)= $
		reform(pwr,dimensions(1:dimensions(0)))
;	Custom option for floating calibration
	if fit_options.f then begin
	for sc=0,n_elements(index)-1 do begin
		v=scans(index(sc)).vissqc(ob,0:nc-1,bl)
		e=scans(index(sc)).vissqcerr(ob,0:nc-1,bl)
		m=scans(index(sc)).vissqm(ob,0:nc-1,bl)
		eindex=where(e gt 0,ecount)
		if ecount gt 0 then begin
;			nf=160	; Should be same as in triple section
;			f=findgen(nf+1)/100+(1.0-nf/200.) & r=f
; 			Define range 1/2 to 2 in steps of 0.01 for cal. factor
			nf=150	; Should be same as in triple section
			f=findgen(nf+1)/100+0.5 & r=f
			nf=125	; Should be same as in triple section
			f=findgen(nf+1)*0.03+0.25 & r=f
			if not floating_notified and not quiet then begin
				print,'Floating cal. limits: ',min(f),max(f)
				floating_notified=1
			endif
			for i=0,nf do $
			r(i)=total(((v(eindex)*f(i)-m(eindex))/(e(eindex)))^2)
			i=where(r eq min(r)) & i=i(0)
			b=genconfig.baselineid(bl,ob)
			if i gt 0 and i lt nf-1 then begin ; Don't hit limits
				f(i)=sqrt(f(i))	; don't apply full scaling
				scans(index(sc)).vissqc(ob,0:nc-1,bl)= $
				scans(index(sc)).vissqc(ob,0:nc-1,bl)*f(i) 
;				scans(index(sc)).vissqcerr(ob,0:nc-1,bl)= $
;				scans(index(sc)).vissqcerr(ob,0:nc-1,bl)*f(i)
;				print,date+' '+b,f(i)
				if not quiet then printf,unit,'2 ',f(i)
			endif else begin
				print,date+' '+b+' scan '+string(index(sc)+1, $
					format='(i3)') $
					+': limit hit for floating V2!'
				scans(index(sc)).vissqc(ob,0:nc-1,bl)= $
				scans(index(sc)).vissqc(ob,0:nc-1,bl)*f(i) 
			endelse
		endif
	endfor
	endif
	endif	; NS > 0
endfor
; Reset magnitudes outside BL loop
if num_binary() ge 1 then star_model.magnitudes=magnitudes
endfor
;
if keyword_set(pearlmv) or keyword_set(pearlcv) then begin
; 	The following (modified) code is taken from pearldata (pearl.pro)
; 	Pass complex visibilities and model SED to Pearl. Reminder next line:
;	index0=where(scans.starid eq gen_model.starid and scantable.code eq 1)
	mv=scans(index0).complexvis
	ow=scans(index0).complexweight
; 	Not all scans have photometry computed
	ti=3
	if n_elements(size(scans(index0).photometrym(0,0,*),/dim)) eq 2 $
	then ti=2
	fi=where(total(scans(index0).photometrym(0,0,*),ti) gt 0)
; 	Derive weight array from ow
	vw=total(ow,1)
	r=size(vw,/dim) & r=[1,r]
	if n_elements(index0) eq 1 then r=[r,1]
	vw=reform(vw,r)
;
	spectrometers=strarr(genconfig.numoutbeam)
	w=genconfig.wavelength	; save wavelength info
	if system_id(systemid) eq 'NPOI' then begin
; 		Assume same channel layout for NPOI!
		mean_w=total(w,2)/genconfig.numoutbeam
		for i=0,genconfig.numoutbeam-1 do $
			genconfig.wavelength(*,i)=mean_w
	endif
	for i=0,genconfig.numoutbeam-1 do $
		spectrometers(i)=strjoin(string( $
		genconfig.wavelength(0:genconfig.numspecchan(i)-1,i)*1e6))
	uspms=unique(spectrometers)
	for i=0L,n_elements(uspms)-1 do begin
	  k=where(spectrometers eq uspms(i))
	  fl0=reform( $
	  scans(index0(fi(0))).photometrym(0,k(0),0:genconfig.numspecchan(i)-1))
;	  Code for removal of bad channels as in pearldata
	  e=reform(vw(k(0),0:GenConfig.NumSpecChan(k(0))-1, $
			   0:max(GenConfig.NumBaseline)-1,*))
	  r=size(e,/dim)
	  if GenConfig.NumSpecChan(k(0)) eq 1 then begin
		r=[1,r]
		e=reform(e,r)
	  endif
;	  if n_elements(r) eq 1 then r=[r,1]
	  if NS0 eq 1 then r=[r,1]
	  jndex=where(total(reform(e, $
		GenConfig.NumSpecChan(k(0)), $
		produkt(r(1:n_elements(r)-1))),2) ne 0)
	  if jndex(0) ne -1 then begin
		if i eq 0 then fl=fl0(jndex) else fl=[fl,fl0(jndex)]
	  endif
	endfor
	genconfig.wavelength=w
;
	index=where(ow gt 0,count)
	if count gt 0 then begin
	   mv=mv(index)	; mv = scans.complexvis
	   if n eq 0 then mv_all=mv else mv_all=[mv_all,mv]
	   if n eq 0 then fl_all=fl else fl_all=[fl_all,fl]
	   if n eq n_elements(nights)-1 then begin
		if keyword_set(pearlmv) then pearlmodel,mv_all,/mv
		if keyword_set(pearlcv) then pearlmodel,mv_all,fl_all,/cv
	   endif
	endif
endif
;
; Triple visibilities
;
time=abs(scans(index0).time)
;
for tr=0,GenConfig.NumTriple-1 do begin
	nc=GenConfig.TripleNumChan(tr)
	triple=strjoin(unique(nameparse(strjoin(genconfig.baselineid( $
		genconfig.triplebase(*,tr), $
		genconfig.triplebeam(*,tr)),'-'),'-')),'-')

;
; 	For multiple stellar systems, compute positions and apodize fluxes
	if num_binary()*0 ge 1 then begin	; obsolete
		parsedate,date,y,m,d
		xy=reform(modelpos(julian(y,m,d)-2440000.d0))
		radius=sqrt(total(xy^2,2))
		rad2mas=180l*3600l*1000l/!pi
; 		Compute Airy's disk (assume same wavelength for all baselines)
		lamda=mean(genconfig.wavelength(genconfig.triplechan(*,0,tr), $
						 genconfig.triplebeam(0,tr)))
		airy=(lamda/genconfig.diameter(0))*rad2mas
		dm=-2.5*alog10(exp(-(2*radius/airy)^2))
		magnitudes=star_model.magnitudes
		for i=0,n_elements(star_model)-1 do $
		star_model(i).magnitudes(*)=star_model(i).magnitudes(*)+dm(i)
	endif
	if num_binary() ge 1 then begin
		magnitudes=star_model.magnitudes
		dm=apodize_m(r0,am)
		index=where(star_model.type ne 0)
		star_model(index).magnitudes=star_model(index).magnitudes $
					    +dm(*,index)
	endif
;
;	Do not compute model for channels without data
;	Example: ups-Sgr-MIRC-2016Jul16.oifits
;	triplephase = Array[20, 8, 48]: 48 scans, 20 triples, 8 channels
	e0=reform(scans(*).triplephasecerr(tr,0:GenConfig.TripleNumChan(tr)-1) > 0)
	if GenConfig.TripleNumChan(tr) gt 1 then e=total(e0,1) else e=e0
	index=where(scans.starid eq gen_model.starid $
		and scantable.code eq 1 $
		and e gt 0,NS)
	if NS gt 0 then begin
	time=abs(scans(index).time)
	int_time=scans(index).int_time
 	vis=reform(complexarr(3,nc,NS),3,nc,NS)
;	Compute the BaseFactor, which is -1 if a baseline is flipped
;	The sum of the atmospheric phase over all baselines must be zero
	fBaseMatrix=intarr(GenConfig.NumSid,3)
	fBaseFactor=fltarr(3)+1
	for l=0,2 do begin
		ob=GenConfig.TripleBeam(l,tr)
		bl=GenConfig.TripleBase(l,tr)
		j1=where(GenConfig.StationId eq $
		  strmid(GenConfig.BaselineId(bl,ob),0,3))
		j2=where(GenConfig.StationId eq $
		  strmid(GenConfig.BaselineId(bl,ob),4,3))
		fBaseMatrix(j1,l)=+1
		fBaseMatrix(j2,l)=-1
		if l gt 0 then $
			fBaseFactor(l)=-total(fBaseMatrix(*,0)*fBaseMatrix(*,l))
	endfor
	for l=0,2 do begin
		ob=GenConfig.TripleBeam(l,tr)
		bl=GenConfig.TripleBase(l,tr)
		ch=GenConfig.TripleChan(0:GenConfig.TripleNumChan(tr)-1,l,tr)
		lambda_grid=system_config(SystemId,'GRID')
		if string(lambda_grid(0)) eq 'SPECTROMETER' then begin
			grid_type='SPECTROMETER'
			lambda_grid=genconfig.wavelength(ch,ob)*1e9
		endif else begin
			grid_type='GRID'
			jndex=where(lambda_grid*1e-9 le $
				max(genconfig.wavelength(ch,ob) $
				   +genconfig.chanwidth(ch,ob)/2) $
				and lambda_grid*1e-9 ge $
				min(genconfig.wavelength(ch,ob) $
				   -genconfig.chanwidth(ch,ob)/2),count)
			if count gt 0 then lambda_grid=lambda_grid(jndex)
		endelse
		lamda=lambda_grid/1d9
		n_lam=n_elements(lamda)
		nu=1/lamda
;
;		Set up integral over int_time
		time=scans(index).time  ; Make backup of time stamps
		if !int_time ne 0 then num_int=max(nint(int_time/!int_time)) $
				  else num_int=0
		if num_int mod 2 eq 0 then num_int=num_int+1
		time_step=0
		if num_int gt 1 then $
		time_step=int_time/(num_int-1)	; array, [s]
		vis_grid=dcomplexarr(n_lam,NS)
;		Only compute model for wavelengths with data for SPECTROMETER
		if grid_type eq 'SPECTROMETER' then begin
			if n_lam gt 1 then begin
			if n_elements(scans) eq 1 then ee=e0 $	; added 2020
						  else ee=total(e0,2)
			endif else ee=e0
			ei=where(ee gt 0,n_lam_ei)
		endif
		for i=-num_int/2,num_int/2 do begin
		  scans(index).time=time+i*time_step(index)
		  calcastrom,/quiet
		  scans.uvw=scans.uvw+uvw_off
		  um=[scans(index).uvw(ob,0,bl,0)*GenConfig.Wavelength(0,ob)]
		  vm=[scans(index).uvw(ob,0,bl,1)*GenConfig.Wavelength(0,ob)]
		  t=abs(scans(index).time)
		  if grid_type eq 'SPECTROMETER' then begin
			u=nu(ei)#um
			v=nu(ei)#vm
			vis_grid(ei,*)=vis_grid(ei,*) $
		 	  +reform(modelvis(t,lamda(ei),fluxes,u,v),n_lam_ei,NS)
		  endif else begin
			u=nu#um
			v=nu#vm
			vis_grid=vis_grid $
		 	  +reform(modelvis(t,lamda,fluxes,u,v),n_lam,NS)
		  endelse
		endfor
		vis_grid=vis_grid/num_int
		scans(index).time=time
;		calcastrom,/quiet; commented Jan 4, 2020, added the next line
		scans.uvw=uvw

;		Bandpass integration
		if grid_type eq 'SPECTROMETER' then begin
;			High R spectrometer
			for sc=0,NS-1 do begin
;				vis(l,*,sc)=vis_grid(*,sc)/fluxes
				vis(l,ei,sc)=vis_grid(ei,sc)/fluxes
			endfor
		endif else begin
		for j=0,nc-1 do begin
			f=system_config(SystemId,'FILTERS')
			if n_elements(f) gt 1 then filter=f(ch(j),ob) $
					      else filter=f(0)
			if filter eq 'generic_c' then $
				tm=generic_c(lambda_grid,ch(j),ob) $
						 else $
				tm=call_function(filter,lambda_grid)
			if NS eq 1 then tm=reform(tm,1,n_lam)
			if total(tm) eq 0 then begin
				print,'***Error(CALCMODEL): wl grid too coarse!'
				return
			endif
			vis(l,j,*)=(tm#vis_grid)/total(tm*fluxes)
		endfor
		endelse
		if fBaseFactor(l) lt 0 then $
			vis(l,*,*)=conj(vis(l,*,*))
	endfor
;
;	Form complex triple products
	if grid_type eq 'SPECTROMETER' then begin
;
	result=reform(float(vis(0,ei,*)*vis(1,ei,*)*vis(2,ei,*)),1,1,n_lam_ei,NS)
	if n_lam_ei eq 1 then result=reform(result,1,NS)
	if n_lam_ei eq 1 and NS eq 1 then result=reform(result)
	scans(index).compltriplem(tr,0,ei)=result
	result=reform(imaginary(vis(0,ei,*)*vis(1,ei,*)*vis(2,ei,*)),1,1,n_lam_ei,NS)
	if n_lam_ei eq 1 then result=reform(result,1,NS)
	if n_lam_ei eq 1 and NS eq 1 then result=reform(result)
	scans(index).compltriplem(tr,1,ei)=result
;
;	Compute amp and phase
	result=reform(sqrt(scans(index).compltriplem(tr,0,ei)^2 $
		     +scans(index).compltriplem(tr,1,ei)^2),1,n_lam_ei,NS)
	if n_lam_ei eq 1 then result=reform(result,1,NS)
	if n_lam_ei eq 1 and NS eq 1 then result=reform(result)
	scans(index).tripleampm(tr,ei)=result
	result=reform(atan(scans(index).compltriplem(tr,1,ei), $
		      scans(index).compltriplem(tr,0,ei)),1,n_lam_ei,NS)
	if n_lam_ei eq 1 then result=reform(result,1,NS)
	if n_lam_ei eq 1 and NS eq 1 then result=reform(result)
	scans(index).triplephasem(tr,ei)= result
;
	endif else begin
;
	result=reform(float(vis(0,*,*)*vis(1,*,*)*vis(2,*,*)),1,1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans(index).compltriplem(tr,0,0:nc-1)=result
	result=reform(imaginary(vis(0,*,*)*vis(1,*,*)*vis(2,*,*)),1,1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans(index).compltriplem(tr,1,0:nc-1)=result
;
;	Compute amp and phase
	result=reform(sqrt(scans(index).compltriplem(tr,0,0:nc-1)^2 $
		     +scans(index).compltriplem(tr,1,0:nc-1)^2),1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans(index).tripleampm(tr,0:nc-1)=result
	result=reform(atan(scans(index).compltriplem(tr,1,0:nc-1), $
		      scans(index).compltriplem(tr,0,0:nc-1)),1,nc,NS)
	if nc eq 1 then result=reform(result,1,NS)
	if nc eq 1 and NS eq 1 then result=reform(result)
	scans(index).triplephasem(tr,0:nc-1)= result
;
	endelse
;
; 	If data set does not have valid triple amplitudes, use model values
	kndex=where(scans.tripleampcerr gt 0,count)
;	Was...:
;	if count eq 0 or not ds_options.ta then $
;	Now...:
	if count eq 0 then $
		scans(kndex).tripleampc=scans(kndex).tripleampm
;
;	Custom option for floating calibration
	if fit_options.f and ds_options.ta then begin
	for sc=0,n_elements(index)-1 do begin
		v=scans(index(sc)).tripleampc(tr,0:nc-1)
		k=where(v lt 0,n_neg)
		e=scans(index(sc)).tripleampcerr(tr,0:nc-1)
		m=scans(index(sc)).tripleampm(tr,0:nc-1)
		eindex=where(e gt 0,ecount)
		if ecount gt 0 and n_neg eq 0 then begin
;			nf=160
;			f=findgen(nf+1)/100+(1.0-nf/200.) & r=f
			nf=150	; Should be same as in baseline section
			f=findgen(nf+1)/100+0.5 & r=f
			nf=125	; Should be same as in triple section
			f=findgen(nf+1)*0.03+0.25 & r=f
			for i=0,nf do $
			r(i)=total(((v(eindex)*f(i)-m(eindex))/(e(eindex)))^2)
			i=where(r eq min(r)) & i=i(0)
			if i gt 0 and i lt nf-1 then begin ; Don't hit limits
				scans(index(sc)).tripleampc(tr,0:nc-1)= $
				scans(index(sc)).tripleampc(tr,0:nc-1)*f(i)
;				scans(index(sc)).tripleampcerr(tr,0:nc-1)= $
;				scans(index(sc)).tripleampcerr(tr,0:nc-1)*f(i)
				if not quiet then printf,unit,'3 ',f(i)
			endif else begin
				print,date+' '+triple+' scan ' $
					+string(index(sc)+1,format='(i3)') $
					+' : limit hit for floating TA!'
				scans(index(sc)).tripleampc(tr,0:nc-1)= $
				scans(index(sc)).tripleampc(tr,0:nc-1)*f(i)
			endelse
		endif
	endfor
	endif
;
;	Custom option for triple phases (disabled)
	if fit_options.f lt 0 then begin
	for sc=0,n_elements(index)-1 do begin
		p=scans(index(sc)).triplephasec(tr,0:nc-1)
		e=scans(index(sc)).triplephasecerr(tr,0:nc-1)
		m=scans(index(sc)).triplephasem(tr,0:nc-1)
		eindex=where(e gt 0,ecount)
		if ecount gt 0 then begin
			nf=100
			f=findgen(nf+1)/2-nf/4 & f=f/RAD & r=f
			for i=0,nf do $
			r(i)=total(((p(eindex)+f(i)-m(eindex))/(e(eindex)))^2)
			i=where(r eq min(r)) & i=i(0)
			if i gt 0 and i lt nf-1 then begin ; Don't hit limits
				scans(index(sc)).triplephasec(tr,0:nc-1)= $
				scans(index(sc)).triplephasec(tr,0:nc-1)+f(i)
;				scans(index(sc)).triplephasecerr(tr,0:nc-1)= $
;				scans(index(sc)).triplephasecerr(tr,0:nc-1)*f(i)
			endif else begin
				print,date+' '+triple+' scan ' $
					+string(index(sc)+1,format='(i3)') $
					+' : limit hit for floating CP!'
				scans(index(sc)).triplephasec(tr,0:nc-1)= $
				scans(index(sc)).triplephasec(tr,0:nc-1)+f(i)
			endelse
		endif
	endfor
	endif
;
;	Unwrap the data phase by comparing it to the model
	for iscan=0,n_elements(index)-1 do begin
		repeat begin
		wrap=where((scans(index(iscan)).triplephasec(tr,*)- $
			    scans(index(iscan)).triplephasem(tr,*) $
					gt pi_circle) $
		       and (scans(index(iscan)).triplephasecerr(tr,*) gt 0), $
		       count)
		tcount=count
		if count gt 0 then scans(index(iscan)).triplephasec(tr,wrap)=$
				   scans(index(iscan)).triplephasec(tr,wrap) $
				  -2*pi_circle
		wrap=where((scans(index(iscan)).triplephasec(tr,*)- $
			    scans(index(iscan)).triplephasem(tr,*) $
					lt -pi_circle) $
		       and (scans(index(iscan)).triplephasecerr(tr,*) gt 0), $
			      count)
		tcount=tcount+count
		if count gt 0 then scans(index(iscan)).triplephasec(tr,wrap)=$
				   scans(index(iscan)).triplephasec(tr,wrap) $
				  +2*pi_circle
		endrep until tcount eq 0
	endfor
;
	endif ; NS > 1
; Reset magnitudes inside TR loop
if num_binary() ge 1 then star_model.magnitudes=magnitudes
endfor
;
; Update night stored with new model data
if strupcase(!version.os) ne 'WIN32' and n_elements(bufferinfo) gt 1 $
				     and not noload then storenight,11
;
NEXT:
ENDFOR
;
; Save statistics of floating calibration factors
if fit_options.f and not quiet then free_lun,unit
;
endif
;
; Section for positions.........................................................
;
if ds_options.a ne 0 then begin
;
if n_elements(positions) eq 0 then begin
	print,'***Error(CALCMODEL): no position data!'
	return
endif
;
; Use shortest wavelength from model
lambda=fltarr(1)+gen_model.wavelengths(0)*1e-6
;
p=modelpos(positions.jd)
list_poscomps,components
for i=0,n_elements(components)-1 do begin
	index=where(positions.component eq components(i),count)
	pos1=dblarr(count,2)
	if componentparse(components(i),comp_1,comp_2) ne 0 then return
	tflux=0.d0
	for k=0,strlen(comp_1)-1 do begin
		j=where(strmid(comp_1,k,1) eq star_model.component) & j=j(0)
		flux=stellarfluxes(star_model(j),lambda) $
		    *modelfluxes(star_model(j),lambda) & flux=total(flux)
		
		pos1=pos1+reform(p(index,j,*)*flux,count,2)
		tflux=tflux+flux
	endfor
	pos1=pos1/tflux
	pos2=dblarr(count,2)
	tflux=0.d0
	for k=0,strlen(comp_2)-1 do begin
		j=where(strmid(comp_2,k,1) eq star_model.component) & j=j(0)
		flux=stellarfluxes(star_model(j),lambda) $
		    *modelfluxes(star_model(j),lambda) & flux=total(flux)
		pos2=pos2+reform(p(index,j,*)*flux,count,2)
		tflux=tflux+flux
	endfor
	pos2=pos2/tflux
	dx=pos2(*,0)-pos1(*,0)
	dy=pos2(*,1)-pos1(*,1)
	rhom=sqrt(dx^2+dy^2)
	thetam=atan(dx,dy)
	if count eq 1 then begin
		positions(index).rhom=rhom(0)
		positions(index).thetam=thetam(0)
	endif else begin
		positions(index).rhom=rhom
		positions(index).thetam=thetam
	endelse
endfor
;
endif
;
; Section for velocities........................................................
if ds_options.s ne 0 then begin
;
if n_elements(velocities) eq 0 then begin
        print,'***Error(CALCMODEL): no velocity data!'
        return
endif
;
v=modelvel(velocities.jd)
list_velcomps,components
; Allow composite component of 2 or more stars, e.g. 'AB'
nv=n_elements(velocities)
nc=n_elements(components)
r=dblarr(nv,nc)
for i=0,n_elements(components)-1 do begin
	index=whereequal(star_model.component,components(i))
	if n_elements(index) ge 2 then begin
	r(*,i)=total(v(*,index)*((dblarr(nv)+1)#star_model(index).mass),2) $
	      /total(star_model(index).mass)
	endif else $
	r(*,i)=v(*,index)
endfor
for i=0,n_elements(components)-1 do begin
	index=where(velocities.component eq components(i))
	velocities(index).valuem=r(index,i)
endfor
;
endif
;
; Section for magnitudes........................................................
;
if ds_options.p gt 0 and not ds_options.pm then begin
;
if n_elements(p_magnitudes) eq 0 then begin
        print,'***Error(CALCMODEL): no magnitude data!'
        return
endif
;
filters=unique(p_magnitudes.filter)
wdbands=binary_model.wdband
;
for i=0,n_elements(filters)-1 do begin
;
case filters(i) of
	'U': 	begin
		lambda=findgen(23)*5+305
		tm=johnson_u(lambda)
		wdband=5
		end
	'B': 	begin
		lambda=findgen(40)*5+360
		tm=johnson_b(lambda)
		wdband=6
		end
	'V': 	begin
		lambda=findgen(54)*5+475
		tm=johnson_v(lambda)
		wdband=7
		end
	'R': 	begin
		lambda=findgen(23)*20+520
		tm=johnson_r(lambda)
		wdband=8
		end
	'I': 	begin
		lambda=findgen(26)*20+680
		tm=johnson_i(lambda)
		wdband=9
		end
	'J': 	begin
		lambda=findgen(31)*20+960
		tm=johnson_i(lambda)
		wdband=10
		end
	'H': 	begin
		lambda=findgen(17)*23+1460
		tm=johnson_i(lambda)
		wdband=45
		end
	'K': 	begin
		lambda=findgen(17)*50+1800
		tm=johnson_i(lambda)
		wdband=11
		end
	'Hp': 	begin
		lambda=findgen(113)*5+335
		tm=hipparcos_hp(lambda)
		wdband=25
		end
endcase
lambda=lambda*1d-9
;
index=where(p_magnitudes.filter eq filters(i))
jd=p_magnitudes(index).jd
;
binary_model.wdband=wdband
valuem=-2.5*alog10(reform(tm#modelflux(jd,lambda))/total(tm))
;
; Minimize median difference between model and data values
dm=median(p_magnitudes(index).value-valuem)
p_magnitudes(index).valuem=valuem+dm
;
endfor
;
binary_model.wdband=wdbands
;
endif
;
; Section for auxilliary data...................................................
;
; Parallax
if ds_options.px ne 0 then begin
if n_elements(parallaxes) eq 0 then begin
        print,'***Error(CALCMODEL): no parallax data!'
        return
endif
j=0
for i=0,n_elements(binary_model.component)-1 do begin
	if binary_model(i).method eq 1 then begin
		parallaxes.valuem(j)=modelpx(binary_model(i).component)
		j=j+1
	endif
endfor
endif
;
; K1
if ds_options.k1 ne 0 then begin
if n_elements(k1) eq 0 then begin
        print,'***Error(CALCMODEL): no K1 data!'
        return
endif
k1.valuem=modelk('A')
endif
;
; K2
if ds_options.k2 ne 0 then begin
if n_elements(k2) eq 0 then begin
        print,'***Error(CALCMODEL): no K2 data!'
        return
endif
k2.valuem=modelk('B')
endif
;
; V sin(i)
; 
; Restore handling of floating point exceptions and math errors
; http://www.idlcoyote.com/code_tips/underflow.html
if 0 then begin
floating_point_underflow = 32 
status = Check_Math() ; Get status and reset accumulated math error register.
IF(status AND NOT floating_point_underflow) NE 0 THEN $
	Message, 'IDL Check_Math() error: ' + StrTrim(status, 2)
!Except = currentExcept
endif
;
end
;-------------------------------------------------------------------------------
pro calfactorplot,triple=triple
;
status=dc_read_free('calfactors.txt',i,v,/col)
;
if n_elements(triple) eq 0 then triple=1
;
if triple then !p.multi=[0,1,2] else !p.multi=0
!p.charsize=1.5
index=where(i eq 2,count)
if count gt 0 then begin
	v2=v(index)
	!x.title='Visibility (squared) calibration factor'
	histograph,v2,min=0.25-0.05/2,max=2.00+0.05/2,binsize=0.05
	f=0.25 & fs=string(f*100,format='(i2)')+'% off: '
	index=where(abs(v2-1) le 0.25,n)
	print,'Percentage of factors less than '+fs,100*float(n)/count
	
endif
;
if triple then begin
index=where(i eq 3,count)
if count gt 0 then begin
	ta=v(index)
	!x.title='Triple amplitude calibration factor'
	histograph,ta,min=0.25-0.05/2,max=2.00+0.05/2,binsize=0.05
endif
endif
;
end
;-------------------------------------------------------------------------------
pro readshell,modelfile,l,r,f
;
; Read radial flux density profiles provided by various spherical shell
; radiative transfer simulations, such as DUSTY or Ralf's.
;
common ShellImage,l_shell,r_shell,f_shell
;
; Read and decode file, last # line before data must contain wavelengths
status=dc_read_fixed(modelfile,lines,/col,format='(a300)')
index=where(strpos(lines,'#') ge 0,count)
nr=n_elements(lines)-count              ; number of values of b
;
; Determine format: DUSTY, Ralf's raytrace, etc...
v=float(lines(count:count+nr-1))
if n_elements(unique(v)) lt nr/2 then begin
;	Ralf's...
	l_shell=unique(v)*1e-6
	nl=n_elements(l_shell)
	nr=nr/nl
	status=dc_read_free(modelfile,l,p,i,/col,ignore=['#'])
	r_shell=p(0:nr-1)/p(1)	; normalize to inner shell radius
	f_shell=transpose(reform(i,nr,nl))
endif else begin
       	words=nameparse(lines(index(count-1)))
       	nl=n_elements(words)-3                  ; do not count #,b,t(b)
       	l_shell=float(words(3:3+nl-1))
	nr=nr-1
       	f_shell=fltarr(nl,nr)
       	r_shell=fltarr(nr)
       	for i=0,nr-1 do begin
               	words=nameparse(lines(count+i))
               	r_shell(i)=float(words(0))
               	for j=0,nl-1 do f_shell(j,i)=float(words(2+j))
       	endfor
	l_shell=l_shell*1e-6 
endelse
;
l=l_shell
;
; Some radii occur more than once
uindex=uniq(r_shell)
r=r_shell(uindex)
f=f_shell(*,uindex)
;
for i=0,n_elements(r)-1 do begin
	index=where(r_shell eq r(i),count)
	if count gt 1 then f(*,i)=total(f_shell(*,index),2)/count
endfor
;
end
;-------------------------------------------------------------------------------
pro displayshell,filename,multiplier=multiplier
;
common ShellImage,l_shell,r_shell,f_shell
;
; status=dc_read_free(filename,r,tau,f,/col,ignore=['#'])
;
if n_elements(multiplier) eq 0 then multiplier=1
;
atv,interpol(f_shell,r_shell*multiplier,shift(dist(104,104),50,50))
;
end
;-------------------------------------------------------------------------------
pro readimage,fitsfile,cellsize=cellsize
;
; Read a FITS image and convert into effective temperature Pearl image.
; Effective temperature and log(g) maps should be found in the image extensions,
; if not, the procedure will ask them to be defined using XROI.
;
; Cellsize in arcseconds.
;
forward_function mrdfits,pearlimage
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common StarBase,StarTable,Notes
;
cmmap=mrdfits(fitsfile,0,header,/silent)
;
nx=n_elements(cmmap(*,0))
ny=n_elements(cmmap(0,*))
imsze=[nx,ny]
;
index=where(strpos(header,'RA---') ge 0,exists)
if exists then begin
	words=nameparse(header(index(0)+2))
	cdelt1=abs(float(words(2)))*3600000
	index=where(strpos(words,'radians') ge 0,count)
	if count gt 0 then cdelt1=cdelt1*180/!pi
endif else cdelt1=25.0
index=where(strpos(header,'DEC--') ge 0,exists)
if exists then begin
	words=nameparse(header(index(0)+2))
	cdelt2=abs(float(words(2)))*3600000
	index=where(strpos(words,'radians') ge 0,count)
	if count gt 0 then cdelt2=cdelt2*180/!pi
endif else cdelt2=25.0
if cdelt1 ne cdelt2 then begin
	print,'***Error(READIMAGE): cell size in RA and Dec different!'
	return
endif
if n_elements(cellsize) eq 0 then cellsze=cdelt1 else cellsze=cellsize
;
cm=pearlimage(imsze,cellsze,/relax)
cc=cm
cm.maps(0).map=cmmap
cc.maps(0).map=0
;
; Check wavelength axis
index=where(strpos(header,'WAVE') ge 0,exists)
if not exists then begin
;	print,'***Error(READIMAGE): wavelength axis undefined in header!'
;	read,wavelength,prompt='Please enter min and max wavelengths: '
	wavelength=1.0	; For this type of image wavelength is not used
endif else begin
	words=nameparse(header(index(0)+1))
	crval3=float(words(2))
	words=nameparse(header(index(0)+2))
	cdelt3=float(words(2))
	words=nameparse(header(index(0)+3))
	crpix3=float(words(2))
	wavelength=crval3
endelse
;
; Initialize default model for this type of image
flag=0
if n_elements(gen_model) eq 0 then begin
	flag=1
endif else begin
	if n_elements(gen_model.wavelengths) $
	ne n_elements(wavelength) then flag=1
endelse
if flag then begin
	if n_elements(gen_model) ne 0 then starid=gen_model.starid
	num_wave=1
	gen_model=alloc_gen_model(num_wave)
	gen_error=gen_model
	gen_model=alloc_gen_model(num_wave)
	gen_error=gen_model
	if n_elements(startable) eq 1 then gen_model.starid=startable.starid $
				      else gen_model.starid='CCCNNNN'
	if n_elements(starid) ne 0 then gen_model.starid=starid
	gen_model.wavelengths=wavelength	; model file wavelengths [mu]
	num_spot=2
	star_model=alloc_star_struct(num_wave,num_spot)
	star_error=star_model
	star_model(0).component='A'
	star_model(0).type=13
	binary_struct=alloc_binary_struct()
	binary_model=replicate(binary_struct,1)
endif
;
; Check if map is AIPS/MX ICL map
index=where(strpos(header,"IMCLASS='ICL") ge 0,exists)
if exists then begin
	rad=180/!pi
	mas2rad=rad*3600000.0
	cclist=mrdfits(fitsfile,1,header,/silent)
	cm.maps(0).map=0
	for i=0,n_elements(cclist)-1 do begin
		j=where(abs(cm.x-cclist(i).deltax/rad) lt cellsze/mas2rad $
		    and abs(cm.y-cclist(i).deltay/rad) lt cellsze/mas2rad,count)
		if count eq 0 then begin
			print,'***Error (READIMAGE): could not assign CC!'
			return
		endif
		cm.maps(0).map(j)=cm.maps(0).map(j)+cclist(i).flux
	endfor
endif else begin
; 	Read FITS extensions for eff. temp. and log(g) region maps
	etmap=mrdfits(fitsfile,1,header,/silent)
	lgmap=mrdfits(fitsfile,2,header,/silent)
endelse
;
if n_elements(etmap) le 1 then begin
	print,'------------------------------------------------------------------'
	print,'This type of image requires the definition of Teff/log(g) regions.'
	print,'Please use XROI to define regions of interest, close (X) when done.'
	print,'The largest regions will be processed first, smaller ones override.'
	print,'------------------------------------------------------------------'
	scale_factor=4
	xroi,bytscl(rebin(cm.maps(0).map,imsze*scale_factor),top=200), $
		roi_geometry=roigeo,regions_out=regout,/block
	if n_elements(roigeo) eq 0 then return
;
	print,'Now enter Teff/log(g) for each region in order of decreasing size.'
	si=reverse(sort(roigeo.area))
	roigeo=roigeo(si)
	regout=regout(si)
	etmap=fltarr(nx,ny)
	lgmap=fltarr(nx,ny)
	nr=n_elements(regout)
	for ireg=0,nr-1 do begin
		read,teff,logg,prompt='Please enter Teff,log(g) for region '+string(ireg,format='(i1)')+': '
		regout(ireg)->scale,fltarr(2)+1./float(scale_factor)
; The following command is not compiled by GDL
;		mask=regout(ireg)->computemask(dimension=[nx,ny])
		etmap(where(mask eq 255))=teff
		lgmap(where(mask eq 255))=logg
;		Make sure there are no pixels without initialization
		if ireg eq 0 then begin
			etmap(*)=teff
			lgmap(*)=logg
		endif
	endfor
	ext_header=["XTENSION= 'IMAGE' / IMAGE extension", $
		    "BITPIX  = -32 /  Number of bits per data pixel", $
		    "NAXIS   = 2 / Number of data axes", $
		    "NAXIS1  = "+string(nx)+" /", $
		    "NAXIS2  = "+string(ny)+" /", $
		    "PCOUNT  = 0 / No Group Parameters", $
		    "GCOUNT  = 1 / One Data Group", $
		    "EXTNAME = 'Teff' /", $
		    "END     "]
		
	writefits,fitsfile,etmap,ext_header,/append
	ext_header(where(strpos(ext_header,'EXTNAME') ge 0))="EXTNAME = 'log(g)' /"
	writefits,fitsfile,lgmap,ext_header,/append
;
endif
;
; Here we have a valid eff. temp. map, now we use it to initialize the rest
etmap=fix(etmap)
index=where(etmap gt 0)
etlg=string(etmap(index),format='(i5)')+' '+string(lgmap(index),format='(f3.1)')
uetlg=unique(etlg)
nuetlg=n_elements(uetlg)
rt=fltarr(nuetlg)
rg=rt
for i=0,nuetlg-1 do begin
	words=nameparse(uetlg(i))
	rt(i)=float(words(0))
	rg(i)=float(words(1))
endfor
cm.teff=etmap
cm.logg=lgmap
cc.teff=etmap
cc.logg=lgmap
;
; Read convolving beam, if any
; a=0.
; read,a,b,c,prompt='Please enter major, minor axis [mas] and PA of restoring beam: '
; if a ne 0 then cb=cleankernel(cm,a) else cb=0
;
end
;-------------------------------------------------------------------------------
pro displayimage,compress=compress,magnify=magnify,crop=crop,invert=invert
;
; Display a GUI to display PEARL images. Compress (integer > 1)
; will scale (^(1/compress)) the dynamic range, magnify will enlarge the images.
; With crop (>1), a smaller centered area will be displayed.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
channel_wavelengths=(findgen(20)*0.1+0.4)*1e-6
;
labels=string(channel_wavelengths*1e6,format='(f6.3)')
;
nx=n_elements(cm.teff(*,0))
ny=n_elements(cm.teff(0,*))
nc=n_elements(channel_wavelengths)
channel_images=fltarr(nx,ny,nc)
for j=0,nc-1 do channel_images(*,*,j)=pearlmap(channel_wavelengths(j))
;
; Find center
nx=n_elements(channel_images(*,0,0))
ny=n_elements(channel_images(0,*,0))
nz=n_elements(channel_images(0,0,*))
index=where(channel_images(*,*,0) eq max(channel_images(*,*,0)))
index=index(0)
i=index mod nx
j=index/nx
i=nx/2	; Let's not try to center on the peak for now...
j=ny/2	; Let's not try to center on the peak for now...
if n_elements(compress) eq 0 then compress=1
if n_elements(magnify) eq 0 then magnify=1 else magnify=fix(magnify > 1)
if n_elements(crop) eq 0 then crop=1 else crop=fix(crop > 1)
if magnify gt crop then crop=magnify
i0=i-(nx/2)/crop > 0
i1=i+(nx/2)/crop < nx
j0=j-(ny/2)/crop > 0
j1=j+(ny/2)/crop < ny
nx=i1-i0+1
ny=j1-j0+1
images=(channel_images(i0:i1,j0:j1,*)>0)^(1./compress)
images=rebin(images,nx*magnify,ny*magnify,nz)
if n_elements(invert) eq 0 then invert=0 else invert=invert gt 0
if invert then images=max(images)-images
display_images,images,labels
;
end
;-------------------------------------------------------------------------------
pro readimages,fitsfile,cellsize=cellsize,wavelengths=wavelengths,quiet=quiet
;
; Read a FITS file containing an image cube to be used for modeling.
; Input: cellsize [mas], wavelengths [microns]. Wavelengths should be 
; specified if they are not in the header of the FITS file (WAVE keyword). 
; Instead of a cube, the images can be specified using wildcards,
; and in that case they will be assembled into a cube. Here one also has the
; option to load a model file first, and the wavelengths specified there
; take the place of the wavelengths required by this routine.
;
; Recognizes BSMEM images by PIXELATION keyword, reverses RA.
;
; The images and related information are stored in common ChannelImage.
; Function chmap (model.pro) is used to compute image at given wavelength.
;
; Wavelength [m]
;
; A few words on passing keyword parameters using this example:
; The call has to use this syntax: readimages,file,cellsize=cellsize_return,...
; The value of cellsize is then returned in cellsize_return.
;
; General example with this code inserted below after the common blocks:
; help,cellsize
; cellsize=4
; return
;
; Execute:
; cellsize=3
; cellsize_inout=6
; readimages,cellsize=cellsize_inout
; Inside function:
; CELLSIZE        INT =  6 	; the value "inout" is copied to cellsize
; On return:
; print,cellsize,cellsize_inout
;        3       4		; cellsize not changed, cellsize_inout changed 
; If called with cellsize=cellsize, cellsize = 4 on output!
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common ReadImagesLocal,message_counter
;
if n_elements(message_counter) eq 0 then message_counter=0
;
; Check if multiple files were given, one image per wavelength
files=file_search(fitsfile)
if strlen(files(0)) eq 0 then begin
	print,'***Error: file does not exist!'
	channel_images=0
	return
endif
nz=n_elements(files)
;
; Read first file and get dimensions and header
channel_image=mrdfits(files(0),0,header,/silent)
index=where(strpos(header,'NAXIS1') ge 0)
words=nameparse(header(index(0)),'=')
naxis1=fix(words(1))
index=where(strpos(header,'NAXIS2') ge 0)
words=nameparse(header(index(0)),'=')
naxis2=fix(words(1))
index=where(strpos(header,'NAXIS3') ge 0,count)
if count eq 1 then begin
	words=nameparse(header(index(0)),'=')
	naxis3=fix(words(1))
endif else naxis3=1
nx=n_elements(channel_image(*,0))
ny=n_elements(channel_image(0,*))
imsze=[nx,ny]
;
; Check if image was written by HOCHUNK3D and select the "All photons" cube
if n_elements(size(channel_image,/dim)) eq 4 then begin
	if message_counter eq 0 then $
	print,'Detected image computed by HOCHUNK3D!'
	message_counter=1
	channel_image=channel_image(*,*,*,0)
	wavelengths=mrdfits(files(0),1,header_ext,/silent)
	wavelengths=wavelengths.wavelength ; microns, converted to m below
	index=where(strpos(header,'RMAXI') ge 0)
	words=nameparse(header(index(0)),'=')
	dmaxi=double(words(1))*2	; full size of image [cm]
	pixel=dmaxi/nx
	if gen_model.px eq 0 then begin
		print,'Error: please specify parallax in model file!'
		return
	endif
	cellsize=gen_model.px*pixel/a_unit
endif
;
; Check if image was written by BSMEM
index=where(strpos(header,'PIXELATION') ge 0,count)
if count eq 1 then begin
	flipra=1	; BSMEM images have W to the left
	words=nameparse(header(index(0)),'=')
	cellsize=float(words(1))
	print,'BSMEM image: cellsize = ',cellsize
endif else flipra=0
;
; Check for wavelength or frequency axis
index=where(strpos(header,'WAVE') ge 0,wave_exists)
index=where(strpos(header,'FREQ') ge 0 and $
	    strpos(header,'RESTFREQ') lt 0,freq_exists)
if not wave_exists and not freq_exists then begin
	if n_elements(wavelengths) eq 0 then begin
		if not keyword_set(quiet) then $
		print,'Wavelength axis undefined in header of image!'
		if n_elements(gen_model) eq 0 then begin
			print,'***Error: no wavelengths defined in model!'
			return
		endif
		channel_wavelengths=gen_model.wavelengths*1e-6
		if not keyword_set(quiet) then $
		print,'Wavelengths for images taken from model.'
	endif else begin
		channel_wavelengths=wavelengths*1e-6
	endelse
endif else begin
	index=where(strpos(header,'WAVE') ge 0,exists)
	if exists and n_elements(files) eq 1 then begin
		index=where(strpos(header,'CRVAL3') ge 0)
		words=nameparse(header(index(0)),'=')
		crval3=float(words(1))
		index=where(strpos(header,'CDELT3') ge 0)
		words=nameparse(header(index(0)),'=')
		cdelt3=float(words(1))
		index=where(strpos(header,'CRPIX3') ge 0)
		words=nameparse(header(index(0)),'=')
		crpix3=float(words(1))
		channel_wavelengths=findgen(naxis3)*cdelt3+crval3 $
				   -(crpix3-1)*cdelt3
	endif
	index=where(strpos(header,'FREQ') ge 0 and $
		    strpos(header,'RESTFREQ') lt 0,exists)
	if exists and n_elements(files) eq 1 then begin
		index=where(strpos(header,'CRVAL3') ge 0)
		words=nameparse(header(index(0)),'=')
		crval3=float(words(1))
		index=where(strpos(header,'CDELT3') ge 0)
		words=nameparse(header(index(0)),'=')
		cdelt3=float(words(1))
		index=where(strpos(header,'CRPIX3') ge 0)
		words=nameparse(header(index(0)),'=')
		crpix3=float(words(1))
		channel_wavelengths=c_light/(findgen(naxis3)*cdelt3+crval3 $
				   -(crpix3-1)*cdelt3)
	endif
endelse
; If more than one image specified, assemble into cube
if nz gt 1 then begin
	if nz ne n_elements(channel_wavelengths) then begin
		print,'***Error: different number of files and wavelengths!'
		return
	endif
	channel_images=fltarr(nx,ny,nz)
	for i=0,nz-1 do begin
		channel_images(*,*,i)=mrdfits(files(i),0,header,/silent) > 0
		if keyword_set(flipra) then $
		for j=0,ny-1 do channel_images(*,j,i) $
		       =reverse(channel_images(*,j,i))
		index=where(strpos(header,'WAVE') ge 0,exists)
		if exists then begin
			words=nameparse(header(index(0)+1))
			crval3=float(words(2))
			channel_wavelengths(i)=crval3
		endif
	endfor
;	Sort by wavelength
	si=sort(channel_wavelengths)
	channel_wavelengths=channel_wavelengths(si)
	channel_images(*,*,*)=channel_images(*,*,si)
endif else begin
;	channel_images=mrdfits(fitsfile(0),0,header,/silent) > 0
	channel_images=channel_image
;	Replace image pixels with zero flux by minimum non-zero flux
	index=where(channel_images eq 0,count)
	if count gt 0 then channel_images(index)= $
				min(channel_images(where(channel_images gt 0)))
	if keyword_set(flipra) then begin
		for i=0,naxis3-1 do $
		for j=0,ny-1 do $
		channel_images(*,j,i)=reverse(channel_images(*,j,i))
	endif
endelse
; Get pixel scales
flag=0
index=where(strpos(header,'RA---') ge 0,exists)
if not exists then begin
	index=where(strpos(header,'PXSCAL1') ge 0 $
		 or strpos(header,'SCALE') ge 0,exists)
	if not exists then begin
		flag=1
	endif else begin
		words=nameparse(header(index(0)))
		if n_elements(words) eq 5 then words(4)=strupcase(words(4))
		index=where(strpos(words,'ARCSEC') ge 0,count)
		if count ne 0 then f=1000.0 else f=1.0
		cdelt1=abs(float(words(2)))*f	; [mas]
	endelse
	if flag eq 1 then begin
		index=where(strpos(header,'CDELT1') ge 0,exists)
		if not exists then begin
			flag=2
			if not keyword_set(cellsize) then begin
			print,'Warning(READIMAGES): pixel scale undefined!'
			print,'Please supply scale in keyword cellsize [mas]!'
			endif else cdelt1=cellsize
		endif else begin
			words=nameparse(header(index(0)))
			cdelt1=abs(float(words(2)))*3600000; convert deg to mas
			index=where(strpos(words,'radians') ge 0,count)
			if count gt 0 then cdelt1=cdelt1*180/!pi
		endelse
	endif
endif else begin
	words=nameparse(header(index(0)))
	index=where(strpos(header,'CDELT'+strmid(words(0),5,1)) ge 0)
	words=nameparse(header(index(0)))
	cdelt1=abs(float(words(2)))*3600000	; convert deg to mas
	index=where(strpos(words,'radians') ge 0,count)
	if count gt 0 then cdelt1=cdelt1*180/!pi
	cellsize=cdelt1
endelse
;
flag=0
index=where(strpos(header,'DEC--') ge 0,exists)
if not exists then begin
	index=where(strpos(header,'PXSCAL2') ge 0 $
		 or strpos(header,'SCALE') ge 0,exists)
	if not exists then begin
		flag=1
	endif else begin
		words=nameparse(header(index(0)))
		if n_elements(words) eq 5 then words(4)=strupcase(words(4))
		index=where(strpos(words,'ARCSEC') ge 0,count)
		if count ne 0 then f=1000.0 else f=1.0
		cdelt2=abs(float(words(2)))*f	; [mas]
	endelse
	if flag eq 1 then begin
		index=where(strpos(header,'CDELT2') ge 0,exists)
		if not exists then begin
			flag=2
			if not keyword_set(cellsize) then begin
			print,'Warning(READIMAGES): pixel scale undefined!'
			print,'Please supply scale in keyword cellsize [mas]!'
			endif else cdelt2=cellsize
		endif else begin
			words=nameparse(header(index(0)))
			cdelt2=abs(float(words(2)))*3600000; convert deg to mas
			index=where(strpos(words,'radians') ge 0,count)
			if count gt 0 then cdelt2=cdelt2*180/!pi
		endelse
	endif
endif else begin
	words=nameparse(header(index(0)))
	index=where(strpos(header,'CDELT'+strmid(words(0),5,1)) ge 0)
	words=nameparse(header(index(0)))
	cdelt2=abs(float(words(2)))*3600000	; convert deg to mas
	index=where(strpos(words,'radians') ge 0,count)
	if count gt 0 then cdelt2=cdelt2*180/!pi
	cellsize=cdelt2
endelse
if n_elements(cdelt1) ne 0 and n_elements(cdelt2) ne 0 then begin 
	if cdelt1 ne cdelt2 then begin
		print,'***Error(READIMAGES): cell size in RA and Dec different!'
		return
	endif
endif
;
; If cellsize was specified on input, use it to set cdelt1 and cdelt2
if n_elements(cellsize) ne 0 then images_cellsize=cellsize
;
; Initialize default model for this type of image
flag=0
if n_elements(gen_model) eq 0 then begin
	flag=1
endif else begin
;	Causes problems, therefore commented out Feb 2017
;	if n_elements(gen_model.wavelengths) $
;	ne n_elements(channel_wavelengths) then flag=1
endelse
; Protect procedure argument
if n_elements(wavelengths) ne 0 then wavelengths_in=wavelengths	
if flag then begin
	if n_elements(gen_model) ne 0 then starid=gen_model.starid
	wavelengths=[min(channel_wavelengths),max(channel_wavelengths)]*1e6
	if wavelengths(0) eq wavelengths(1) then wavelengths=wavelengths(0)
	num_wave=n_elements(wavelengths)
	gen_model=alloc_gen_model(num_wave)
	gen_error=gen_model
	if n_elements(startable) eq 1 then gen_model.starid=startable.starid $
				      else gen_model.starid='CCCNNNN'
	if n_elements(starid) ne 0 then gen_model.starid=starid
	gen_model.wavelengths=wavelengths	; model file wavelengths [mu]
	num_spot=2
	star_model=alloc_star_struct(num_wave,num_spot)
	star_error=star_model
	star_model(0).component='A'
	star_model(0).type=12
	binary_struct=alloc_binary_struct()
	binary_model=replicate(binary_struct,1)
endif
; restore procedure argument
if n_elements(wavelengths_in) ne 0 then wavelengths=wavelengths_in	
;
end
;-------------------------------------------------------------------------------
pro checkimages
;
; Check that images exist for the entire wavelength range specified in genconfig
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(genconfig) eq 0 then begin
	print,'No check possible: GenConfig not loaded.'
	return
endif
;
nx=n_elements(channel_images(*,0,0))
ny=n_elements(channel_images(0,*,0))
;
for i=0,genconfig.numoutbeam-1 do begin
index=where(scans.vissqerr(i,*,*) gt 0,count)
if count gt 0 then begin
;
minw=min(channel_wavelengths)
maxw=max(channel_wavelengths)
minl=min(genconfig.wavelength(0:genconfig.numspecchan(i)-1,i))
maxl=max(genconfig.wavelength(0:genconfig.numspecchan(i)-1,i))
;
nz=n_elements(channel_images(0,0,*))
if minl lt minw then begin
	print,'Warning: min.lambda cube > min.lambda genconfig): ', $
		minw*1e6,minl*1e6
	print,'Adding image(min.lambda cube) for min.lambda...'
	channel_wavelengths=[minl,channel_wavelengths]
	images=channel_images
	channel_images=fltarr(nx,ny,nz+1)
	channel_images(*,*,0)=images(*,*,0)
	channel_images(*,*,1:nz)=images
endif
nz=n_elements(channel_images(0,0,*))
if maxl gt maxw then begin
	print,'Warning: max.lambda cube < max.lambda genconfig): ', $
		minw*1e6,minl*1e6
	print,'Adding image(max.lambda cube) for max.lambda...'
	channel_wavelengths=[channel_wavelengths,maxl]
	images=channel_images
	channel_images=fltarr(nx,ny,nz+1)
	channel_images(*,*,0:nz-1)=images
	channel_images(*,*,nz)=images(*,*,nz-1)
endif
;
endif
endfor
;
end
;-------------------------------------------------------------------------------
pro sedimages,sedfile,split=split
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
if n_elements(sedfile) eq 0 then begin
	print,'***Error: please specify filename!'
	return
endif
;
flux=total(total(channel_images,1),1)
l=channel_wavelengths
;
if keyword_set(split) then begin
	r=poly_fit(l*1e6,flux,1,yfit)
	v=medianve(flux-yfit,e)
	index=where(abs(flux-yfit) lt 2*e)
	r=poly_fit(l(index)*1e6,flux(index),2)
	yfit=poly(l*1e6,r)
	plot,l,flux,psym=0
	oplot,l,yfit,psym=0
	f=yfit
	save,l,f,filename=sedfile+'.A.xdr'
	f=abs(flux-yfit)
	save,l,f,filename=sedfile+'.B.xdr'
	return
endif
;
save,l,f,filename=sedfile
;
end
;-------------------------------------------------------------------------------
pro displayimages,compress=compress,magnify=magnify,crop=crop,invert=invert
;
; Display a GUI to browse through images of a cube. Compress (integer > 1)
; will scale (^(1/compress)) the dynamic range, magnify will enlarge the images.
; With crop (>1), a smaller centered area will be displayed.
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
labels=string(channel_wavelengths*1e6,format='(f8.3)')
;
; Find center
nx=n_elements(channel_images(*,0,0))
ny=n_elements(channel_images(0,*,0))
nz=n_elements(channel_images(0,0,*))
index=where(channel_images(*,*,0) eq max(channel_images(*,*,0)))
index=index(0)
i=index mod nx
j=index/nx
i=nx/2	; Let's not try to center on the peak for now...
j=ny/2	; Let's not try to center on the peak for now...
if n_elements(compress) eq 0 then compress=1
if n_elements(magnify) eq 0 then magnify=1 else magnify=fix(magnify > 1)
if n_elements(crop) eq 0 then crop=1 else crop=fix(crop > 1)
if magnify gt crop then crop=magnify
i0=i-(nx/2)/crop > 0
i1=i+(nx/2)/crop < nx-1
j0=j-(ny/2)/crop > 0
j1=j+(ny/2)/crop < ny-1
nx=i1-i0+1
ny=j1-j0+1
images=(channel_images(i0:i1,j0:j1,*)>0)^(1./compress)
images=rebin(images,nx*magnify,ny*magnify,nz)
if n_elements(invert) eq 0 then invert=0 else invert=invert gt 0
if invert then images=max(images)-images
display_frames,images,labels
;
end
;-------------------------------------------------------------------------------
pro plotimagefluxes,lamda,flux
;
; Plot spectrum of image cube. Return lambda(microns),fluxes
;
common ChannelImage,channel_images,channel_wavelengths,images_cellsize
;
nl=n_elements(channel_wavelengths)
lamda=channel_wavelengths*1e6	; convert m to microns
flux=fltarr(nl)
for j=0,nl-1 do flux(j)=total(channel_images(*,*,j))
;
if total(!x.range) eq 0 then !x.range=[min(lamda),max(lamda)]
index=where(lamda gt !x.range(0) and lamda lt !x.range(1))
if total(!y.range) eq 0 then !y.range=[0,max(flux(index))]
plot,lamda,flux, $
	xtitle='Wavelength [microns]',ytitle='Flux',charsize=1.5
!y.range=0
;
end
;-------------------------------------------------------------------------------
pro marquardtdata,y,ysig,ymod,noload=noload
;
; Return a `flat' array of the combined observed or model data sets, as
; requested per ds_options. Please note that this procedures relies
; on error bars of unobserved quantities to be negative since it does
; not check boundary information in GenConfig.
;
; Uses all loaded interferometric data, if any, unless noload=1.
; If triple amplitudes are not measured, set them equal to their model values.
; If triple phases are not measured, set them equal to their model values.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common AuxData,parallaxes,k1,k2,vsini
;
if n_elements(noload) eq 0 then noload=0
;
y=0.d0
ysig=0.d0
ymod=0.d0
;
; Interferometry
if ds_options.i ne 0 then begin
	nights=geninfo.date
	arrays=geoinfo.systemid
	confgs=geninfo.configid
	wf=1/sqrt(ds_options.i)
	num_nights=n_elements(nights)
	if noload then num_nights=1
	for n=0,num_nights-1 do begin
		night=where(GenInfo.date eq nights(n) $
			and GeoInfo.systemid eq arrays(n) $
			and GenInfo.configid eq confgs(n))
		if n_elements(bufferinfo) gt 1 and not noload then $
		loadnight,GenInfo(night(0)).date, $
			  GeoInfo(night(0)).systemid, $
			  GenInfo(night(0)).configid
;
; 		Scan selection
		NS=n_elements(scans)
		index=indgen(NS)
		if n_elements(gen_model) ne 0 then $
		index=where(scans.starid eq gen_model.starid,NS)
		if ns gt 0 and ds_options.v2 then begin
			ndata=n_elements(scans(index).vissqc)
			y=[y,reform(scans(index).vissqc,ndata)]
			ysig=[ysig,reform(scans(index).vissqcerr,ndata)*wf]
			ymod=[ymod,reform(scans(index).vissqm,ndata)]
		endif
		if ns gt 0 and ds_options.vp then begin
			ndata=n_elements(scans(index).visphase)
			y=[y,reform(scans(index).visphase,ndata)]
			ysig=[ysig,reform(scans(index).visphaseerr,ndata)*wf]
			ymod=[ymod,reform(scans(index).visphasem,ndata)]
		endif
		if ns gt 0 and (ds_options.ta or ds_options.tp) then begin
			set_compltriple	; Fill complex triple data
			ndata=n_elements(scans(index).compltriplem)
;			New algorithm appropriate for Buscher's Theorem.
;			Rotate complex triple by closure phase
;			to match error in real part to amplitude
;			and error in imaginary part to phase.
			ct=reform(scans(index).compltriple)
			cm=reform(scans(index).compltriplem)
			cp=reform(scans(index).triplephasec)
			ce=reform(scans(index).compltripleerr)
;			Catch uninitialized bad data (option nozero used)
;			index=where(ce lt 0,count)
;			if count ge 1 then begin
;				ct(index)=0
;				cp(index)=0
;				cm(index)=0
;			endif
			if genconfig.numtriple ge 2 then begin
				ctr=reform(ct(*,0,*,*))
				cti=reform(ct(*,1,*,*))
				ct(*,0,*,*)=ctr*cos(-cp)-cti*sin(-cp)
				ct(*,1,*,*)=cti*cos(-cp)+ctr*sin(-cp)
				cmr=reform(cm(*,0,*,*))
				cmi=reform(cm(*,1,*,*))
				cm(*,0,*,*)=cmr*cos(-cp)-cmi*sin(-cp)
				cm(*,1,*,*)=cmi*cos(-cp)+cmr*sin(-cp)
				if not ds_options.ta then begin
					ct(*,0,*,*)=cm(*,0,*,*)
					ce(*,0,*,*)=-1
				endif
				if not ds_options.tp then begin
					ct(*,1,*,*)=cm(*,1,*,*)
					ce(*,1,*,*)=-1
				endif
			endif else begin
				ctr=reform(ct(0,*,*))
				cti=reform(ct(1,*,*))
				ct(0,*,*)=ctr*cos(-cp)-cti*sin(-cp)
				ct(1,*,*)=cti*cos(-cp)+ctr*sin(-cp)
				cmr=cm(0,*,*)
				cmi=cm(1,*,*)
				cm(0,*,*)=cmr*cos(-cp)-cmi*sin(-cp)
				cm(1,*,*)=cmi*cos(-cp)+cmr*sin(-cp)
				if not ds_options.ta then begin
					ct(0,*,*)=cm(0,*,*)
					ce(0,*,*)=-1
				endif
				if not ds_options.tp then begin
					ct(1,*,*)=cm(1,*,*)
					ce(1,*,*)=-1
				endif
			endelse
			y=[y,reform(ct,ndata)]
			ysig=[ysig,reform(ce,ndata)*wf]
			ymod=[ymod,reform(cm,ndata)]
		endif
	endfor
endif
;
; Astrometry
if ds_options.a ne 0 then begin
	wf=1/sqrt(ds_options.a)
	y_ra=positions.rho*sin(positions.theta)
	y_dc=positions.rho*cos(positions.theta)
	ymod_ra=positions.rhom*sin(positions.thetam)
	ymod_dc=positions.rhom*cos(positions.thetam)
	y=[y,dblarr(n_elements(positions))]
	ddc=ymod_dc-y_dc
	dra=ymod_ra-y_ra
	phi=atan(dra,ddc)-positions.pa
	if ds_options.c ge 1 then phi(*)=0
	eminor=positions.eminor
	emajor=positions.emajor
	if ds_options.c eq 2 then begin
		eminor(*)=1
		emajor(*)=1
	endif
	e=sqrt(emajor^2-eminor^2)/emajor
	ysig=[ysig,sqrt(eminor^2/(1-(e*cos(phi))^2))*wf]
	ymod=[ymod,sqrt(dra^2+ddc^2)]
endif
;
; Spectroscopy
if ds_options.s ne 0 then begin
	wf=1/sqrt(ds_options.s)
	y=[y,velocities.value]
	ysig=[ysig,velocities.error*wf]
	ymod=[ymod,velocities.valuem]
endif
;
; Photometry
if ds_options.p ne 0 then begin
	wf=1/sqrt(ds_options.p)
	y=[y,magnitudes.value]
	ysig=[ysig,magnitudes.error*wf]
	ymod=[ymod,magnitudes.valuem]
endif
;
; Auxilliary data
; Parallax, and optional enforcement of same parallax for all binary components
if ds_options.px ne 0 then begin
	wf=1/sqrt(abs(ds_options.px))
;	Scale the weight of the parallax if ds_options.px < 0
	if ds_options.px lt 0 then begin
                opt_px=ds_options.px
                ds_options.px=0
                index=where(ysig gt 0,count)
		chisq=modelchisq()
		if chisq eq 0 then chisq=1
                wf=1/sqrt(count*chisq)
                ds_options.px=opt_px
        endif
	index=where(binary_model.method eq 1,count)
	if count gt 0 then begin
		pxs=fltarr(count)
		if ds_options.px lt 0 then pxs(*)=mean(parallaxes.valuem(index))
		if ds_options.px gt 0 then pxs=parallaxes.value(index)
		y=[y,pxs]
		ysig=[ysig,parallaxes.error(index)*wf]
		ymod=[ymod,parallaxes.valuem(index)]
	endif
endif
; K1
if ds_options.k1 gt 0 then begin
	wf=1/sqrt(ds_options.k1)
	y=[y,k1.value]
	ysig=[ysig,k1.error*wf]
	ymod=[ymod,k1.valuem]
endif
; K2
if ds_options.k2 gt 0 then begin
	wf=1/sqrt(ds_options.k2)
	y=[y,k2.value]
	ysig=[ysig,k2.error*wf]
	ymod=[ymod,k2.valuem]
endif
; V sin(i)
 ds_options.vsini=0
if ds_options.vsini gt 0 then begin
	wf=1/sqrt(ds_options.vsini)
	y=[y,vsini.value]
	ysig=[ysig,vsini.error*wf]
	ymod=[ymod,vsini.valuem]
endif
;
; Edit the data
index=where(ysig gt 0,count)
if count gt 0 then begin
	y=y(index)
	ysig=ysig(index)
	ymod=ymod(index)
endif else begin
	print,'***Error(MARQUARDTDATA): no valid data found!'
	y=0
	ysig=0
	ymod=0
endelse
;
end
;************************************************************************Block 6
pro mark32cha,files
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error(MARK32CHA): no files specified!'
	return
endif
;
for i=0,n_elements(files)-1 do begin
	get_mark3data,files(i),status
	if status eq 0 then begin
	dotpos=strpos(files(i),'.')
	filename=strmid(files(i),0,dotpos)+'.cha'
	result=file_search(filename,count=fcount)
	if fcount ne 0 then hds_open,filename,'UPDATE',status else $
		hds_new,filename,'DataSet','CHAMELEON',status
	if status ne 0 then begin
		print,'***Error(MARK32CHA): file problem!'
		clearstatus
		return
	endif
	put_SysConfig
	put_scans
	toplevel
	dat_annul
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro mark32oifits,files
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error(MARK32OIFITS): no files specified!'
	return
endif
;
for i=0,n_elements(files)-1 do begin
	get_mark3data,files(i),status
	if status eq 0 then begin
	dotpos=strpos(files(i),'.')
	filename=strmid(files(i),0,dotpos)+'.oifits'
	result=file_search(filename,count=fcount)
	if fcount eq 1 then begin
		command='rm -f '+filename
		if safe(command) then spawn,command
	endif
	put_oifits,filename
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro cha2oifits,files
;
num_files=n_elements(files)
if num_files eq 0 then begin
	print,'***Error(CHA2OIFITS): no files specified!'
	return
endif
;
for i=0,n_elements(files)-1 do begin
	get_data,files(i)
	dotpos=strpos(files(i),'.')
	filename=strmid(files(i),0,dotpos)+'.oifits'
	result=file_search(filename,count=fcount)
	if fcount eq 1 then begin
		command='rm -f '+filename
		if safe(command) then spawn,command
	endif
	put_oifits,filename
endfor
;
end
;-------------------------------------------------------------------------------
pro mark32psn,file
;
status=dc_read_free(file,r,t,a,b,p,d,s,x,y,/col,ignore=['!','$TEXT_IN_NUMERIC'])
openw,unit,'star.psn',/get_lun
;
parsemark3,d,y,m,d
jy=jd2jy(julian(y,m,d))
for i=0,n_elements(d)-1 do begin
	printf,unit,'A-B',jy(i),r(i),t(i),a(i),b(i),p(i), $
		format='(a,2x,f9.4,2x,f6.2,1x,f7.2,2x,f5.2,1x,f4.2,2x,f7.2)'
endfor
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro calibrators2tex,target,aas=aas,calstars=calstars
;
; Based on currently loaded interferometry reduced and calibrated by npoipipe, 
; write information on all calibrators with valid data to LaTex. Include
; number of nights a calibrator was used, and the minimum estimated visibility.
; The latter is computed by simulation, i.e. by calling mockdata and calcviscal.
; Prepare in A&A LaTex format (or AAS if aas=1).
;
; Specify calstars to force procedure to use just these as calibrators.
;
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(target) eq 0 then begin
	target=''
	read,target,prompt='Please enter name of target: '
endif
if n_elements(aas) eq 0 then aas=0
;
; Save current startable
startable_bck=startable 
;
; Replace all visibility amplitude by unity
mockdata,/init
;
; Compute estimated visibilities given diameter information
calcviscal,/multiple
;
nights=GenInfo.date
arrays=GeoInfo.systemid
confgs=GenInfo.configid
;
nstars=n_elements(startable)
minvis=fltarr(nstars)+1
exist=intarr(nstars)
stars=startable.starid
;
if n_elements(calstars) ne 0 then begin
	startable.bflag='.'
	for i=0,n_elements(calstars)-1 do begin
		j=where(startable.starid eq calstars(i),count)
		if count eq 1 then startable(j).bflag='C'
	endfor
endif
calibrators=startable(where(startable.bflag eq 'C')).starid
used_calibs=strarr(n_elements(calibrators))
;
FOR n=0,n_elements(nights)-1 DO BEGIN
;
night=where(GenInfo.date eq nights(n) $
        and GeoInfo.systemid eq arrays(n) $
        and GenInfo.configid eq confgs(n),count)
if count eq 0 and n_elements(GenInfo.date) eq 1 then night=0
if strupcase(!version.os) ne 'WIN32' $
	and n_elements(bufferinfo) gt 1 then $
	loadnight,GenInfo(night(0)).date, $
		GeoInfo(night(0)).systemid, $
		GenInfo(night(0)).configid
;
; Set bflag to '.' for calibrators always outside of 80 min smoothing interval
t_index=where(scans.starid eq target)
for i=0,n_elements(calibrators)-1 do begin
	c_index=where(scans.starid eq calibrators(i),c_count)
	if c_count gt 0 then begin
	dt=fltarr(c_count)
	for j=0,c_count-1 do $
		dt(j)=min(abs(scans(t_index).time-scans(c_index(j)).time))
	if min(dt) lt 80.0*60.0 then $
		used_calibs(i)='u'
	endif
endfor
;
; Consider only channels 1-8
for i=0,genconfig.numoutbeam-1 do begin
	if genconfig.numspecchan(i) gt 8 then $
	scans(*).vissqcerr(*,8:genconfig.numspecchan(i)-1,*)=-1
endfor
;
if strlen(target) ne 0 then begin
	index=where(scans.starid eq target and scantable.code eq 1,target_count)
	if target_count gt 1 then $
		target_mask=total(scans(index).vissqcerr gt 0,4) $
	else if target_count eq 1 then $
		target_mask=scans(index).vissqcerr gt 0
endif else target_count=0
	
for i=0,nstars-1 do begin
	index=where(scans.starid eq stars(i) and scantable.code eq 1,count)
	if count gt 0 then begin
		if count gt 1 then $
			calib_mask=total(scans(index).vissqcerr gt 0,4) $
		else 	calib_mask=scans(index).vissqcerr gt 0
		if target_count gt 0 then $
			common_mask=(calib_mask*target_mask) gt 0 else $
			common_mask=calib_mask gt 0
		vissqcerr=scans(index).vissqcerr
		vissqe=scans(index).vissqe
		for j=0,count-1 do vissqcerr(*,*,*,j)= $
				   vissqcerr(*,*,*,j)*common_mask
		index=where(vissqcerr gt 0,count)
		if count gt 0 then begin
			exist(i)=exist(i)+1
			minvis(i)=min([minvis(i),1./max(vissqe(index))])
		endif
	endif
endfor
;
ENDFOR
;
; Set bflag to '.' for calibrators always outside of 80 min smoothing interval
index=where(used_calibs ne 'u',count)
for i=0,count-1 do $
	startable(where(startable.starid eq calibrators(index(i)))).bflag='.'
;
; Reduce startable to observed calibrators
index=where(exist eq 0 or startable.bflag ne 'C')
minvis(index)=0
index=where(minvis ne 0)
exist=exist(index)
minvis=minvis(index)
startable=startable(index)
;
; Get JHK data
starids=startable.starid
rename_starids,'bsc-hdn'
read_catalogs
get_jhk_bsc
;
si=sort(startable.hdn)
startable=startable(si)
starids=starids(si)
minvis=minvis(si)
exist=exist(si)
;
mv=startable.mv
bv=startable.bv
bv_stars
;
s=startable
bv0=s.bv
ebmv=bv-bv0
vmk=s.mv-s.mk
;
openw,unit,target+'_cal.tex',/get_lun
;
IF aas THEN BEGIN
;
printf,unit,'\documentclass[preprint]{/home/chummel/latex/aastex/aastex61}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{llcrrccc}'
printf,unit,'\tabletypesize{\scriptsize}'
printf,unit,'\tablecaption{List of NPOI calibrators used.'+ $
	    ' $\theta_{V-K}$ was not corrected for extinction (see text).'+ $
	    ' $V^2_{\rm min}$ is the minimum estimated calibrator visibility'+ $
	    ' based on the diameter $\theta_{V-K}$}'
printf,unit,'\tablehead{'
printf,unit,'\colhead{HD}&'
printf,unit,'\colhead{Type}&'
printf,unit,'\colhead{$V$}&'
printf,unit,'\colhead{$V-K$}&'
printf,unit,'\colhead{$E(B-V)$}&'
printf,unit,'\colhead{$\theta_{V-K}$}&'
printf,unit,'\colhead{$V^2_{\rm min}$}&'
printf,unit,'\colhead{Nights}'
printf,unit,'}'
printf,unit,'\startdata'
;
ENDIF ELSE BEGIN
;
printf,unit,'\documentclass[twocolumn]{/home/chummel/latex/aa-package/aa}'
printf,unit,'\usepackage{graphicx}'
printf,unit,'\usepackage{txfonts}'
printf,unit,'\usepackage{natbib}'
printf,unit,'\usepackage{upgreek}'
printf,unit,'\usepackage{url}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{NPOI observations of '+target+' }'
printf,unit,'\label{npoilog}'
printf,unit,'\begin{tabular}{llcrrccc}'
printf,unit,'\hline'
printf,unit,'\hline'
printf,unit,'List of NPOI calibrators used for '+target+'\\'
printf,unit,'HD&Type&$V$&$V-K$&$E(B-V)$&$\theta_{V-K}$&$V^2_{\rm min}$&Nights\\'
printf,unit,'  &    &   &     &        &         [mas]&               &      \\'
printf,unit,'\hline'
;
ENDELSE
;
for i=0,n_elements(s)-1 do begin
;	printf,unit,'HD ',s(i).hdn,s(i).spectrum, $
;	format='(a3,i6,"&",a,"&",f5.2,"&",f5.2,"&",f5.2,"&",f4.2,"&",f4.2,"&",i2,"\\")'
	printf,unit,starids(i),s(i).spectrum, $
		mv(i),vmk(i),ebmv(i),s(i).diameter,minvis(i),exist(i), $
	format='(a,"&",a,"&",f5.2,"&",f5.2,"&",f5.2,"&",f4.2,"&",f4.2,"&",i2,"\\")'
endfor
;
IF aas THEN BEGIN
	printf,unit,'\enddata'
	printf,unit,'\end{deluxetable}'
ENDIF ELSE BEGIN
	printf,unit,'\hline'
	printf,unit,'\end{tabular}'
	printf,unit,'\end{table}'
ENDELSE
printf,unit,'\end{document}'
;
free_lun,unit
;
; Restore startable
startable=startable_bck
;
end
;-------------------------------------------------------------------------------
pro obslog2tex_73leo,aas=aas
;
; Version for 73 Leo paper
;
; Based on currently loaded interferometry reduced and calibrated by npoipipe, 
; a log of the observations including calibrators is prepared for the specified 
; target in A&A LaTex format (or AAS if aas=1). Calibrator ID are from BSC (HR).
;
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(scantable) eq 0 then begin
	print,'Please load data before calling this procedure!'
	retall
endif
;
target='BSC4365'
aas=1
if n_elements(target) eq 0 then begin
	target=''
	read,target,prompt='Please enter name of target: '
endif
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
obslog=target+'_log.tex'
openw,unit,obslog,/get_lun
;
IF aas THEN BEGIN
;
; Use AASTEX
printf,unit,'\documentclass{/home/chummel/latex/aastex/aastex61}'
; printf,unit,'\usepackage{lscape}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{lccccc}'
; printf,unit,'\rotate'
printf,unit,'\tabletypesize{\scriptsize}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +target+'\label{table1}}'
printf,unit,'\tablehead{'
printf,unit,'\colhead{UT Date}&'
printf,unit,'\colhead{\# of obs.}&'
printf,unit,'\colhead{Triangles and baselines}&'
printf,unit,'\colhead{Max. length}&'
printf,unit,'\colhead{\# of vis.}&'
printf,unit,'\colhead{\# of C. Ph.}'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}&'
printf,unit,'\colhead{(6)}'
printf,unit,'}'
printf,unit,'\startdata'
;
ENDIF ELSE BEGIN
;
printf,unit,'\documentclass[twocolumn]{/home/chummel/latex/aa-package/aa}'
printf,unit,'\usepackage{graphicx}'
printf,unit,'\usepackage{txfonts}'
printf,unit,'\usepackage{natbib}'
printf,unit,'\usepackage{upgreek}'
printf,unit,'\usepackage{url}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{NPOI observations of '+target+' }'
printf,unit,'\label{npoilog}'
; printf,unit,'\begin{tabular}{llll}'
printf,unit,'\begin{tabular}{llllll}'
printf,unit,'\hline'
printf,unit,'\hline'
; printf,unit,'UT Date&Julian Year&Triangles and baselines&Calibrators (HR)\\'
; printf,unit,' (1) & (2) & (3) & (4) \\'
printf,unit,'UT Date&Julian Year&Triangles and baselines&'+ $
	    '$B_{\rm min} [m]$&$B_{\rm max} [m]$&Calibrators (HR)\\'
printf,unit,' (1) & (2) & (3) & (4) & (5) & (6) \\'
printf,unit,'\hline'
;
ENDELSE
;
nights=GenInfo.date
arrays=GeoInfo.systemid
confgs=GenInfo.configid
;
FOR n=0,n_elements(nights)-1 DO BEGIN
;
; Load data
night=where(GenInfo.date eq nights(n) $
        and GeoInfo.systemid eq arrays(n) $
        and GenInfo.configid eq confgs(n),count)
if count eq 0 and n_elements(GenInfo.date) eq 1 then night=0
if strupcase(!version.os) ne 'WIN32' $
	and n_elements(bufferinfo) gt 1 then begin
	loadnight,GenInfo(night(0)).date, $
		GeoInfo(night(0)).systemid, $
		GenInfo(night(0)).configid
endif
;
; Master index for target scans
index=where(scans.starid eq target and scantable.code eq 1,num_scans)
vndex=where(scans(index).vissqcerr gt 0,num_vis)
tndex=where(scans(index).triplephasecerr gt 0,num_triple)
;
; Prepare for baseline length computation
uvw=scans(index).uvw
bl_max=intarr(max(genconfig.numbaseline),genconfig.numoutbeam)
bl_min=bl_max
;
; Date and Julian Year
parsedate,geninfo(n).date,year,month,day
if num_scans gt 1 then day=day+median(scans(index).time)/86400 $
		  else day=day+       scans(index).time /86400
jy=jd2jy(julian(year,month,day))
utdate=strmid(geninfo(n).date,0,4) $
	+' '+months(month-1) $
	+' '+strmid(geninfo(n).date,8,2)
;
; Maximum baseline lengths
if genconfig.numtriple gt 0 then begin
baseids=genconfig.baselineid( $
	genconfig.triplebase(*,0:genconfig.numtriple-1), $
	genconfig.triplebeam(*,0:genconfig.numtriple-1))
endif else baseids=genconfig.baselineid
for i=0,genconfig.numoutbeam-1 do begin
for j=0,genconfig.numbaseline(i)-1 do begin
	for m=0,1 do uvw(i,*,j,m,*)=uvw(i,*,j,m,*) $
		   *(genconfig.wavelength(*,i)#(fltarr(n_elements(index))+1))
	uv=reform(sqrt(uvw(i,*,j,0,*)^2+uvw(i,*,j,1,*)^2))
	uv=reform(uv,n_elements(uv))
	ve=scans(index).vissqcerr(i,0:genconfig.numspecchan(i)-1,j)
	k=where(reform(ve) gt 0,count)
;	Do not include baselines without data
	if count eq 0 then continue
	bl_max(j,i)=nint(max(uv(k)))
	bl_min(j,i)=nint(min(uv(k)))
endfor
endfor
;
; Determine baseline/triple configurations
configs=strarr(genconfig.numoutbeam)
for i=0,genconfig.numoutbeam-1 do begin
	baselines=genconfig.baselineid(*,i)
	j=where(baselines ne 'ST1-ST2',nj)
	baselines=baselines(j)
	if nj eq 3 then $
	configs(i)=strjoin(unique(nameparse(strjoin(baselines,'-'),'-')),'-') $
	else $
	configs(i)=baselines(j)
endfor
b_index=where(strlen(configs) eq  7,b_count)	; index into configs
t_index=where(strlen(configs) eq 11,t_count)	; index into configs
triangles=configs(t_index)
baselines=configs(b_index)
;
; Remove baselines which are part of triples
for i=0,n_elements(baselines)-1 do begin
	for j=0,n_elements(triangles)-1 do begin
		if strpos(triangles(j),baselines(i)) ge 0 or $
		   strpos(triangles(j),breve(baselines(i))) ge 0 $
		then baselines(i)=''
	endfor	
endfor
;
; Check for triples on target without data
gc=genconfig
for i=0,t_count-1 do begin
	j=t_index(i)
	jndex=where(scans(index).triplephasecerr(j) gt 0,jcount)
	if jcount eq 0 then begin
;		triangles(t_index(i))=''
		triangles(i)=''
;		Check which baselines in this triple do have data
;		scans.vissq = array[#combiner,#channel,#baseline,[#scans]]
		baseids=genconfig.baselineid(*,j)
		for k=0,2 do begin
		k_index=where(scans(index).vissqcerr(gc.triplebeam(j),*,k) $
			gt 0,num_k)
		if num_k ge 1 then baselines=[baselines,baseids(k)]
		endfor
	endif
endfor
config=strtrim(strjoin(triangles,' ')+' '+strjoin(baselines,' '),2)
;
; Visibility errors for the requested target and its calibrators
index=where(scans.starid eq target and scantable.code eq 1,target_count)
if target_count gt 1 then $
	target_mask=total(scans(index).vissqcerr gt 0,4) $
else if target_count eq 1 then $
	target_mask=scans(index).vissqcerr gt 0
;
stars=startable.starid
nstars=n_elements(stars)	
exist=intarr(nstars)
for i=0,nstars-1 do begin
	index=where(scans.starid eq stars(i) and scantable.code eq 1,count)
	if count gt 0 then begin
		if count gt 1 then $
			calib_mask=total(scans(index).vissqcerr gt 0,4) $
		else 	calib_mask=scans(index).vissqcerr gt 0
		if target_count gt 0 then $
			common_mask=(calib_mask*target_mask) gt 0 else $
			common_mask=calib_mask gt 0
		vissqcerr=scans(index).vissqcerr
		vissqe=scans(index).vissqe
		kndex=where(vissqe eq 0,kount)
		if kount gt 0 then vissqe(kndex)=1
		for j=0,count-1 do vissqcerr(*,*,*,j)= $
				   vissqcerr(*,*,*,j)*common_mask
		index=where(vissqcerr gt 0,count)
		if count gt 0 then exist(i)=exist(i)+1
	endif
endfor
;
; Print table
if strlen(config) ne 0 then printf,unit,utdate,num_scans,config,max(bl_max), $
	num_vis,num_triple, $
	format='(a,"&",i3,"&",a,"&",i3,"&",i4,"&",i3,"\\")'
;
ENDFOR
;
IF aas THEN BEGIN
	printf,unit,'\enddata'
	printf,unit,'\end{deluxetable}'
ENDIF ELSE BEGIN
	printf,unit,'\hline'
	printf,unit,'\end{tabular}'
	printf,unit,'\end{table}'
ENDELSE
printf,unit,'\end{document}'
free_lun,unit
;
spawn,'sed -i {s/AC0/AC/g} '+obslog
spawn,'sed -i {s/AE0/AE/g} '+obslog
spawn,'sed -i {s/AW0/AW/g} '+obslog
spawn,'sed -i {s/W07/W7/g} '+obslog
spawn,'sed -i {s/E06/E6/g} '+obslog
;
end
;-------------------------------------------------------------------------------
pro obslog2tex_36tau,target,aas=aas
;
; Version for 36 Tau paper (HR 1252)
;
; Based on currently loaded interferometry reduced and calibrated by npoipipe, 
; a log of the observations including calibrators is prepared for the specified 
; target in A&A LaTex format (or AAS if aas=1). Calibrator ID are from BSC (HR).
;
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(scantable) eq 0 then begin
	print,'Please load data before calling this procedure!'
	retall
endif
;
target='BSC1252'
calibr='FKV0147'
aas=1
if n_elements(target) eq 0 then begin
	target=''
	read,target,prompt='Please enter name of target: '
endif
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
obslog=target+'_log.tex'
openw,unit,obslog,/get_lun
;
IF aas THEN BEGIN
;
; Use AASTEX
printf,unit,'\documentclass{/home/chummel/latex/aastex/aastex61}'
; printf,unit,'\usepackage{lscape}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{lcccc}'
; printf,unit,'\rotate'
printf,unit,'\tabletypesize{\scriptsize}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +target+'\label{table1}}'
printf,unit,'\tablehead{'
printf,unit,'\colhead{UT Date}&'
printf,unit,'\colhead{Triangles and baselines}&'
printf,unit,'\colhead{Min. length}&'
printf,unit,'\colhead{Max. length}&'
printf,unit,'\colhead{\# of vis.}&'
printf,unit,'\colhead{\# of C. Ph.}'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}&'
printf,unit,'\colhead{(6)}'
printf,unit,'}'
printf,unit,'\startdata'
;
ENDIF ELSE BEGIN
;
printf,unit,'\documentclass[twocolumn]{/home/chummel/latex/aa-package/aa}'
printf,unit,'\usepackage{graphicx}'
printf,unit,'\usepackage{txfonts}'
printf,unit,'\usepackage{natbib}'
printf,unit,'\usepackage{upgreek}'
printf,unit,'\usepackage{url}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{NPOI observations of '+target+' }'
printf,unit,'\label{npoilog}'
printf,unit,'\begin{tabular}{llllll}'
printf,unit,'\hline'
printf,unit,'\hline'
; printf,unit,'UT Date&Julian Year&Triangles and baselines&Calibrators (HR)\\'
; printf,unit,' (1) & (2) & (3) & (4) \\'
printf,unit,'UT Date&Triangles and baselines&'+ $
	    'B_{\rm min} [m]&B_{\rm max} [m]&\# of vis.&\#C.Ph.\\'
printf,unit,' (1) & (2) & (3) & (4) & (5) & (6) \\'
printf,unit,'\hline'
;
ENDELSE
;
nights=GenInfo.date
arrays=GeoInfo.systemid
confgs=GenInfo.configid
;
FOR n=0,n_elements(nights)-1 DO BEGIN
;
; Load data
night=where(GenInfo.date eq nights(n) $
        and GeoInfo.systemid eq arrays(n) $
        and GenInfo.configid eq confgs(n),count)
if count eq 0 and n_elements(GenInfo.date) eq 1 then night=0
if strupcase(!version.os) ne 'WIN32' $
	and n_elements(bufferinfo) gt 1 then begin
	loadnight,GenInfo(night(0)).date, $
		GeoInfo(night(0)).systemid, $
		GenInfo(night(0)).configid
endif
;
; Master index for target scans
index=where(scans.starid eq target and scantable.code eq 1,num_scans)
vndex=where(scans(index).vissqcerr gt 0,num_vis)
tndex=where(scans(index).triplephasecerr gt 0,num_triple)
;
; Prepare for baseline length computation
uvw=scans(index).uvw
bl_max=intarr(max(genconfig.numbaseline),genconfig.numoutbeam)
bl_min=bl_max
;
; Date and Julian Year
parsedate,geninfo(n).date,year,month,day
if num_scans gt 1 then day=day+median(scans(index).time)/86400 $
		  else day=day+	      scans(index).time/86400
jy=jd2jy(julian(year,month,day))
utdate=strmid(geninfo(n).date,0,4) $
	+' '+months(month-1) $
	+' '+strmid(geninfo(n).date,8,2)
;
; Maximum baseline lengths
if genconfig.numtriple gt 0 then begin
baseids=genconfig.baselineid( $
	genconfig.triplebase(*,0:genconfig.numtriple-1), $
	genconfig.triplebeam(*,0:genconfig.numtriple-1))
endif else baseids=genconfig.baselineid
for i=0,genconfig.numoutbeam-1 do begin
for j=0,genconfig.numbaseline(i)-1 do begin
	for m=0,1 do uvw(i,*,j,m,*)=uvw(i,*,j,m,*) $
		   *(genconfig.wavelength(*,i)#(fltarr(n_elements(index))+1))
	uv=reform(sqrt(uvw(i,*,j,0,*)^2+uvw(i,*,j,1,*)^2))
	uv=reform(uv,n_elements(uv))
	ve=scans(index).vissqcerr(i,0:genconfig.numspecchan(i)-1,j)
	k=where(reform(ve) gt 0,count)
;	Do not include baselines without data
	if count eq 0 then continue
	bl_max(j,i)=nint(max(uv(k)))
	bl_min(j,i)=nint(min(uv(k)))
endfor
endfor
;
; Determine baseline/triple configurations
configs=strarr(genconfig.numoutbeam)
for i=0,genconfig.numoutbeam-1 do begin
	baselines=genconfig.baselineid(*,i)
	j=where(baselines ne 'ST1-ST2',nj)
	baselines=baselines(j)
	if nj ge 3 then $
	configs(i)=strjoin(unique(nameparse(strjoin(baselines,'-'),'-')),'-') $
	else $
	configs(i)=baselines(j)
endfor
b_index=where(strlen(configs) eq  7,b_count)	; index into configs
t_index=where(strlen(configs) eq 11,t_count)	; index into configs
triangles=configs(t_index)
baselines=configs(b_index)
;
; Remove baselines which are part of triples
for i=0,n_elements(baselines)-1 do begin
	for j=0,n_elements(triangles)-1 do begin
		if strpos(triangles(j),baselines(i)) ge 0 or $
		   strpos(triangles(j),breve(baselines(i))) ge 0 $
		then baselines(i)=''
	endfor	
endfor
;
; Check for triples on target without data
gc=genconfig
for i=0,t_count-1 do begin
	j=t_index(i)
	jndex=where(scans(index).triplephasecerr(j) gt 0,jcount)
	if jcount eq 0 then begin
;		triangles(t_index(i))=''
		triangles(i)=''
;		Check which baselines in this triple do have data
;		scans.vissq = array[#combiner,#channel,#baseline,[#scans]]
		baseids=genconfig.baselineid(*,j)
		for k=0,2 do begin
		k_index=where(scans(index).vissqcerr(gc.triplebeam(j),*,k) $
			gt 0,num_k)
		if num_k ge 1 then baselines=[baselines,baseids(k)]
		endfor
	endif
endfor
config=strtrim(strjoin(triangles,' ')+' '+strjoin(baselines,' '),2)
;
; Visibility errors for the requested target and its calibrators
index=where(scans.starid eq target and scantable.code eq 1,target_count)
if target_count gt 1 then $
	target_mask=total(scans(index).vissqcerr gt 0,4) $
else if target_count eq 1 then $
	target_mask=scans(index).vissqcerr gt 0
;
stars=startable.starid
nstars=n_elements(stars)	
exist=intarr(nstars)
for i=0,nstars-1 do begin
	index=where(scans.starid eq stars(i) and scantable.code eq 1,count)
	if count gt 0 then begin
		if count gt 1 then $
			calib_mask=total(scans(index).vissqcerr gt 0,4) $
		else 	calib_mask=scans(index).vissqcerr gt 0
		if target_count gt 0 then $
			common_mask=(calib_mask*target_mask) gt 0 else $
			common_mask=calib_mask gt 0
		vissqcerr=scans(index).vissqcerr
		vissqe=scans(index).vissqe
		kndex=where(vissqe eq 0,kount)
		if kount gt 0 then vissqe(kndex)=1
		for j=0,count-1 do vissqcerr(*,*,*,j)= $
				   vissqcerr(*,*,*,j)*common_mask
		index=where(vissqcerr gt 0,count)
		if count gt 0 then exist(i)=exist(i)+1
	endif
endfor
;
; Print table
if strlen(config) ne 0 then printf,unit,utdate,config,min(bl_min),max(bl_max), $
	num_vis,num_triple, $
	format='(a,"&",a,"&",i3,"&",i4,"&",i4,"&",i3,"\\")'
;if strlen(config) ne 0 then printf,unit,utdate,num_scans,config,max(bl_max), $
;	num_vis,num_triple, $
;	format='(a,"&",i3,"&",a,"&",i3,"&",i4,"&",i3,"\\")'
;
ENDFOR
;
IF aas THEN BEGIN
	printf,unit,'\enddata'
	printf,unit,'\end{deluxetable}'
ENDIF ELSE BEGIN
	printf,unit,'\hline'
	printf,unit,'\end{tabular}'
	printf,unit,'\end{table}'
ENDELSE
printf,unit,'\end{document}'
free_lun,unit
;
spawn,'sed -i {s/AC0/AC/g} '+obslog
spawn,'sed -i {s/AE0/AE/g} '+obslog
spawn,'sed -i {s/AW0/AW/g} '+obslog
spawn,'sed -i {s/W07/W7/g} '+obslog
spawn,'sed -i {s/E06/E6/g} '+obslog
;
end
;-------------------------------------------------------------------------------
pro obslog2tex_betari,target,aas=aas
;
; Version for Beta Arietis (FK5 0066)
;
; Based on currently loaded interferometry reduced and calibrated by npoipipe, 
; a log of the observations including calibrators is prepared for the specified 
; target in A&A LaTex format (or AAS if aas=1). Calibrator ID are from BSC (HR).
;
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(scantable) eq 0 then begin
	print,'Please load data before calling this procedure!'
	retall
endif
;
target='FKV0066'
aas=1
if n_elements(target) eq 0 then begin
	target=''
	read,target,prompt='Please enter name of target: '
endif
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
obslog=target+'_log.tex'
openw,unit,obslog,/get_lun
;
IF aas THEN BEGIN
;
; Use AASTEX
printf,unit,'\documentclass{/home/chummel/latex/aastex/aastex61}'
; printf,unit,'\usepackage{lscape}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{lcccc}'
; printf,unit,'\rotate'
printf,unit,'\tabletypesize{\scriptsize}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +target+'\label{table1}}'
printf,unit,'\tablehead{'
printf,unit,'\colhead{UT Date}&'
printf,unit,'\colhead{Triangles and baselines}&'
printf,unit,'\colhead{Max. length}&'
printf,unit,'\colhead{\# of vis.}&'
printf,unit,'\colhead{\# of C. Ph.}'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}'
printf,unit,'}'
printf,unit,'\startdata'
;
ENDIF ELSE BEGIN
;
printf,unit,'\documentclass[twocolumn]{/home/chummel/latex/aa-package/aa}'
printf,unit,'\usepackage{graphicx}'
printf,unit,'\usepackage{txfonts}'
printf,unit,'\usepackage{natbib}'
printf,unit,'\usepackage{upgreek}'
printf,unit,'\usepackage{url}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{NPOI observations of '+target+' }'
printf,unit,'\label{npoilog}'
; printf,unit,'\begin{tabular}{llll}'
printf,unit,'\begin{tabular}{lllll}'
printf,unit,'\hline'
printf,unit,'\hline'
; printf,unit,'UT Date&Julian Year&Triangles and baselines&Calibrators (HR)\\'
; printf,unit,' (1) & (2) & (3) & (4) \\'
printf,unit,'UT Date&Triangles and baselines&'+ $
	    'B_{\rm max} [m]&\# of vis.&\#C.Ph.\\'
printf,unit,' (1) & (2) & (3) & (4) & (5) \\'
printf,unit,'\hline'
;
ENDELSE
;
nights=GenInfo.date
arrays=GeoInfo.systemid
confgs=GenInfo.configid
;
FOR n=0,n_elements(nights)-1 DO BEGIN
;
; Load data
night=where(GenInfo.date eq nights(n) $
        and GeoInfo.systemid eq arrays(n) $
        and GenInfo.configid eq confgs(n),count)
if count eq 0 and n_elements(GenInfo.date) eq 1 then night=0
if strupcase(!version.os) ne 'WIN32' $
	and n_elements(bufferinfo) gt 1 then begin
	loadnight,GenInfo(night(0)).date, $
		GeoInfo(night(0)).systemid, $
		GenInfo(night(0)).configid
endif
;
; Master index for target scans
index=where(scans.starid eq target and scantable.code eq 1,num_scans)
vndex=where(scans(index).vissqcerr gt 0,num_vis)
tndex=where(scans(index).triplephasecerr gt 0,num_triple)
;
; Prepare for baseline length computation
uvw=scans(index).uvw
bl_max=intarr(max(genconfig.numbaseline),genconfig.numoutbeam)
bl_min=bl_max
;
; Date and Julian Year
parsedate,geninfo(n).date,year,month,day
if num_scans gt 1 then day=day+median(scans(index).time)/86400 $
		  else day=day+	      scans(index).time/86400
jy=jd2jy(julian(year,month,day))
utdate=strmid(geninfo(n).date,0,4) $
	+' '+months(month-1) $
	+' '+strmid(geninfo(n).date,8,2)
;
; Maximum baseline lengths
if genconfig.numtriple gt 0 then begin
baseids=genconfig.baselineid( $
	genconfig.triplebase(*,0:genconfig.numtriple-1), $
	genconfig.triplebeam(*,0:genconfig.numtriple-1))
endif else baseids=genconfig.baselineid
for i=0,genconfig.numoutbeam-1 do begin
for j=0,genconfig.numbaseline(i)-1 do begin
	for m=0,1 do uvw(i,*,j,m,*)=uvw(i,*,j,m,*) $
		   *(genconfig.wavelength(*,i)#(fltarr(n_elements(index))+1))
	uv=reform(sqrt(uvw(i,*,j,0,*)^2+uvw(i,*,j,1,*)^2))
	uv=reform(uv,n_elements(uv))
	ve=scans(index).vissqcerr(i,0:genconfig.numspecchan(i)-1,j)
	k=where(reform(ve) gt 0,count)
;	Do not include baselines without data
	if count eq 0 then continue
	bl_max(j,i)=nint(max(uv(k)))
	bl_min(j,i)=nint(min(uv(k)))
endfor
endfor
;
; Determine baseline/triple configurations
configs=strarr(genconfig.numoutbeam)
for i=0,genconfig.numoutbeam-1 do begin
	baselines=genconfig.baselineid(*,i)
	j=where(baselines ne 'ST1-ST2',nj)
	baselines=baselines(j)
	if nj ge 3 then $
	configs(i)=strjoin(unique(nameparse(strjoin(baselines,'-'),'-')),'-') $
	else $
	configs(i)=baselines(j)
endfor
b_index=where(strlen(configs) eq  7,b_count)	; index into configs
t_index=where(strlen(configs) eq 11,t_count)	; index into configs
triangles=configs(t_index)
baselines=configs(b_index)
;
; Remove baselines which are part of triples
for i=0,n_elements(baselines)-1 do begin
	for j=0,n_elements(triangles)-1 do begin
		if strpos(triangles(j),baselines(i)) ge 0 or $
		   strpos(triangles(j),breve(baselines(i))) ge 0 $
		then baselines(i)=''
	endfor	
endfor
;
; Check for triples on target without data
gc=genconfig
for i=0,t_count-1 do begin
	j=t_index(i)
	jndex=where(scans(index).triplephasecerr(j) gt 0,jcount)
	if jcount eq 0 then begin
;		triangles(t_index(i))=''
		triangles(i)=''
;		Check which baselines in this triple do have data
;		scans.vissq = array[#combiner,#channel,#baseline,[#scans]]
		baseids=genconfig.baselineid(*,j)
		for k=0,2 do begin
		k_index=where(scans(index).vissqcerr(gc.triplebeam(j),*,k) $
			gt 0,num_k)
		if num_k ge 1 then baselines=[baselines,baseids(k)]
		endfor
	endif
endfor
config=strtrim(strjoin(triangles,' ')+' '+strjoin(baselines,' '),2)
;
; Visibility errors for the requested target and its calibrators
index=where(scans.starid eq target and scantable.code eq 1,target_count)
if target_count gt 1 then $
	target_mask=total(scans(index).vissqcerr gt 0,4) $
else if target_count eq 1 then $
	target_mask=scans(index).vissqcerr gt 0
;
stars=startable.starid
nstars=n_elements(stars)	
exist=intarr(nstars)
for i=0,nstars-1 do begin
	index=where(scans.starid eq stars(i) and scantable.code eq 1,count)
	if count gt 0 then begin
		if count gt 1 then $
			calib_mask=total(scans(index).vissqcerr gt 0,4) $
		else 	calib_mask=scans(index).vissqcerr gt 0
		if target_count gt 0 then $
			common_mask=(calib_mask*target_mask) gt 0 else $
			common_mask=calib_mask gt 0
		vissqcerr=scans(index).vissqcerr
		vissqe=scans(index).vissqe
		kndex=where(vissqe eq 0,kount)
		if kount gt 0 then vissqe(kndex)=1
		for j=0,count-1 do vissqcerr(*,*,*,j)= $
				   vissqcerr(*,*,*,j)*common_mask
		index=where(vissqcerr gt 0,count)
		if count gt 0 then exist(i)=exist(i)+1
	endif
endfor
;
; Print table
if strlen(config) ne 0 then printf,unit,utdate,config,max(bl_max), $
	num_vis,num_triple, $
	format='(a,"&",a,"&",i3,"&",i4,"&",i3,"\\")'
;if strlen(config) ne 0 then printf,unit,utdate,num_scans,config,max(bl_max), $
;	num_vis,num_triple, $
;	format='(a,"&",i3,"&",a,"&",i3,"&",i4,"&",i3,"\\")'
;
ENDFOR
;
IF aas THEN BEGIN
	printf,unit,'\enddata'
	printf,unit,'\end{deluxetable}'
ENDIF ELSE BEGIN
	printf,unit,'\hline'
	printf,unit,'\end{tabular}'
	printf,unit,'\end{table}'
ENDELSE
printf,unit,'\end{document}'
free_lun,unit
;
spawn,'sed -i {s/AC0/AC/g} '+obslog
spawn,'sed -i {s/AE0/AE/g} '+obslog
spawn,'sed -i {s/AW0/AW/g} '+obslog
spawn,'sed -i {s/W07/W7/g} '+obslog
spawn,'sed -i {s/E06/E6/g} '+obslog
;
end
;-------------------------------------------------------------------------------
pro obslog2tex,target,aas=aas,calstars=calstars
;
; Original version, following the target-optimized versions above.
;
; Based on currently loaded interferometry reduced and calibrated by npoipipe, 
; a log of the observations including calibrators is prepared for the specified 
; target in A&A LaTex format (or AAS if aas=1). Calibrator ID are from BSC (HR).
;
common StarBase,StarTable,Notes
common Tables,scantable,bgtable,stationtable
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Make copy of startable as it will be modified in this procedure
startable_bck=startable
;
if n_elements(target) eq 0 then begin
	target=''
	read,target,prompt='Please enter name of target: '
endif
if n_elements(aas) eq 0 then aas=0
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
obslog=target+'_log.tex'
openw,unit,obslog,/get_lun
;
IF aas THEN BEGIN
;
; Use AASTEX
printf,unit,'\documentclass{/home/chummel/latex/aastex/aastex61}'
; printf,unit,'\usepackage{lscape}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{lllccl}'
; printf,unit,'\rotate'
printf,unit,'\tabletypesize{\scriptsize}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +target+'\label{table1}}'
printf,unit,'\tablehead{'
printf,unit,'\colhead{UT Date}&'
printf,unit,'\colhead{Julian Year}&'
printf,unit,'\colhead{Triangles and baselines}&'
printf,unit,'\colhead{Min. length}&'
printf,unit,'\colhead{Max. length}&'
printf,unit,'\colhead{Calibrators (HR)}'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}&'
printf,unit,'\colhead{(6)}'
printf,unit,'}'
printf,unit,'\startdata'
;
ENDIF ELSE BEGIN
;
printf,unit,'\documentclass[twocolumn]{/home/chummel/latex/aa-package/aa}'
printf,unit,'\usepackage{graphicx}'
printf,unit,'\usepackage{txfonts}'
printf,unit,'\usepackage{natbib}'
printf,unit,'\usepackage{upgreek}'
printf,unit,'\usepackage{url}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{NPOI observations of '+target+' }'
printf,unit,'\label{npoilog}'
; printf,unit,'\begin{tabular}{llll}'
printf,unit,'\begin{tabular}{llllll}'
printf,unit,'\hline'
printf,unit,'\hline'
; printf,unit,'UT Date&Julian Year&Triangles and baselines&Calibrators (HR)\\'
; printf,unit,' (1) & (2) & (3) & (4) \\'
printf,unit,'UT Date&Julian Year&Triangles and baselines&'+ $
	    '$B_{\rm min} [m]$&$B_{\rm max} [m]$&Calibrators (HR)\\'
printf,unit,' (1) & (2) & (3) & (4) & (5) & (6) \\'
printf,unit,'\hline'
;
ENDELSE
;
nights=GenInfo.date
arrays=GeoInfo.systemid
confgs=GenInfo.configid
;
FOR n=0,n_elements(nights)-1 DO BEGIN
;
; Load data
night=where(GenInfo.date eq nights(n) $
        and GeoInfo.systemid eq arrays(n) $
        and GenInfo.configid eq confgs(n),count)
if count eq 0 and n_elements(GenInfo.date) eq 1 then night=0
if strupcase(!version.os) ne 'WIN32' $
	and n_elements(bufferinfo) gt 1 then $
	loadnight,GenInfo(night(0)).date, $
		GeoInfo(night(0)).systemid, $
		GenInfo(night(0)).configid
;
; Master index for target scans
index=where(scans.starid eq target and scantable.code eq 1,num_scans)
;
; Prepare for baseline length computation
uvw=scans(index).uvw
bl_max=intarr(max(genconfig.numbaseline),genconfig.numoutbeam)
bl_min=bl_max
;
; Date and Julian Year
parsedate,geninfo(n).date,year,month,day
; day=day+median(scans(index).time)/86400
if num_scans gt 1 then day=day+median(scans(index).time)/86400 $
		  else day=day+       scans(index).time /86400
jy=jd2jy(julian(year,month,day))
utdate=strmid(geninfo(n).date,0,4) $
	+' '+months(month-1) $
	+' '+strmid(geninfo(n).date,8,2)
;
mask=intarr(size(genconfig.baselineid,/dim))+1
;
; "Triangles"
if GenConfig.NumTriple gt 0 then begin
configs=genconfig.baselineid( $
	genconfig.triplebase(*,0:GenConfig.NumTriple-1), $
	genconfig.triplebeam(*,0:GenConfig.NumTriple-1))
endif else configs=genconfig.baselineid
for i=0,genconfig.numoutbeam-1 do begin
for j=0,genconfig.numbaseline(i)-1 do begin
for m=0,1 do uvw(i,*,j,m,*)=uvw(i,*,j,m,*) $
		   *(genconfig.wavelength(*,i)#(fltarr(n_elements(index))+1))
uv=sqrt(uvw(i,*,j,0,*)^2+uvw(i,*,j,1,*)^2)
k=where(scans(index).vissqcerr(i,0:genconfig.numspecchan(i)-1,j) gt 0,count)
bl_max(j,i)=nint(max(uv(k)))
bl_min(j,i)=nint(min(uv(k)))
if count eq 0 then begin
	mask(j,i)=0
	if GenConfig.NumTriple gt 0 then begin
		k=where(configs eq genconfig.baselineid(j,i),count)
		if count gt 0 then configs(k)=''
	endif
endif
endfor
endfor
;
if genconfig.numtriple gt 0 then begin
if genconfig.numtriple gt 0 then triangles=strarr(genconfig.numtriple)
for i=0,genconfig.numtriple-1 do begin
	slen=strlen(configs(*,i))
	if total(slen) eq 21 then begin
		triple=strtrim(strjoin(configs(*,i),' '),2)
		stations=strarr(6)
		for j=0,5 do stations(j)=strmid(triple,j*4,3)
		triangles(i)=strjoin(unique(stations),'-')
	endif
endfor
index=where(strlen(triangles) ne 0,count)
if count ne 0 then triangles=unique(triangles(index))
count_t=n_elements(triangles)
for i=0,count_t-1 do begin
	if strlen(triangles(i)) ne 0 then begin
	stations=nameparse(triangles(i),'-')
	for j=0,genconfig.numtriple-1 do begin
	for k=0,2 do begin
		if strlen(configs(k,j)) ne 0 then begin
			words=nameparse(configs(k,j),'-')
			index=where(stations eq words(0) $
				 or stations eq words(1),count)
			if count eq 2 then configs(k,j)=''
		endif
	endfor
	endfor
	endif
endfor
endif else triangles=''
index=where(strlen(configs) ne 0,count)
if count ne 0 then configs=unique(configs(index))
config=strtrim(strjoin(triangles,' ')+' '+strjoin(configs(*),' '),2)
;
; Find out which calibrators were observed
if n_elements(calstars) ne 0 then begin
	startable.bflag='.'
	for i=0,n_elements(calstars)-1 do begin
		j=where(startable.starid eq calstars(i),count)
		if count eq 1 then startable(j).bflag='C'
	endfor
endif
calibrators=startable(where(startable.bflag eq 'C')).starid
;
index=where(scans.starid eq target and scantable.code eq 1,target_count)
if target_count gt 1 then $
	target_mask=total(scans(index).vissqcerr gt 0,4) $
else if target_count eq 1 then $
	target_mask=scans(index).vissqcerr gt 0
;
stars=startable.starid
nstars=n_elements(stars)	
exist=intarr(nstars)
for i=0,nstars-1 do begin
	index=where(scans.starid eq stars(i) and scantable.code eq 1,count)
	if count gt 0 then begin
		if count gt 1 then $
			calib_mask=total(scans(index).vissqcerr gt 0,4) $
		else 	calib_mask=scans(index).vissqcerr gt 0
		if target_count gt 0 then $
			common_mask=(calib_mask*target_mask) gt 0 else $
			common_mask=calib_mask gt 0
		vissqcerr=scans(index).vissqcerr
		vissqe=scans(index).vissqe
		kndex=where(vissqe eq 0,kount)
		if kount gt 0 then vissqe(kndex)=1
		for j=0,count-1 do vissqcerr(*,*,*,j)= $
				   vissqcerr(*,*,*,j)*common_mask
		index=where(vissqcerr gt 0,count)
		if count gt 0 then begin
			exist(i)=exist(i)+1
		endif
	endif
endfor
;
; Set bflag to '.' for calibrators always outside of 80 min smoothing interval
t_index=where(scans.starid eq target)
sci_ra=startable(where(startable.starid eq target)).ra
sci_dec=startable(where(startable.starid eq target)).dec
cal_ra=startable(where(startable.bflag eq 'C')).ra
cal_dec=startable(where(startable.bflag eq 'C')).dec
used_calibs=strarr(n_elements(calibrators))
for i=0,n_elements(calibrators)-1 do begin
	c_index=where(scans.starid eq calibrators(i),c_count)
	if c_count gt 0 then begin
	dt=fltarr(c_count)
	for j=0,c_count-1 do $
		dt(j)=min(abs(scans(t_index).time-scans(c_index(j)).time))
		angle=winkel(cal_ra(i),cal_dec(i),sci_ra,sci_dec)
;	if min(dt) lt 80.0*60.0 and angle lt 30 then $
	if min(dt) lt 80.0*60.0 then $
		used_calibs(i)='u'
	endif
endfor
index=where(used_calibs ne 'u',count)
for i=0,count-1 do $
	startable(where(startable.starid eq calibrators(index(i)))).bflag='.'
;
; If data set contains science target only
index=where(startable.bflag eq 'C',count)
if count eq 0 and n_elements(startable) eq 1 then startable.bflag='C'
;
index=where(exist eq 1 and startable.bflag eq 'C',count)
IF count GT 0 THEN BEGIN
max_n_cal=5
if count le max_n_cal then begin
calibrators1=strjoin(startable(index).starid,' ')
; calibrators1=strjoin(strtrim(string(startable(index).bsc),2),' ')
calibrators2=''
endif else begin
calibrators1=strjoin(startable(index(0:max_n_cal-1)).starid,' ')
; calibrators1=strjoin(strtrim(string(startable(index(0:max_n_cal-1)).bsc),2),' ')
calibrators2=strjoin(startable(index(max_n_cal:count-1)).starid,' ')
; calibrators2=strjoin(strtrim(string(startable(index(max_n_cal:count-1)).bsc),2),' ')
endelse
;
if strlen(config) ne 0 then begin
; printf,unit,utdate,jy,config,calibrators1, $
; 	format='(a,"&",f9.4,"&",a,"&",a,"\\")'
configs=nameparse(config)
for j=0,n_elements(configs)-1 do begin
if j eq 0 then begin
	printf,unit,utdate,jy,configs(j), $
		min(bl_min(where(bl_min ne 0))),max(bl_max),calibrators1, $
		format='(a,"&",f9.4,"&",a,"&",i3,"&",i3,"&",a,"\\")'
	if strlen(calibrators2) ne 0 then $
		printf,unit,calibrators2, $
			format='("& & & & &",a,"\\")'
endif else begin
	printf,unit,blanks(11),blanks(9),configs(j), $
	format='(a,"&",a,"&",a,"\\")'
endelse
endfor
endif
;
ENDIF
;
startable=startable_bck
;
ENDFOR
;
IF aas THEN BEGIN
	printf,unit,'\enddata'
	printf,unit,'\end{deluxetable}'
ENDIF ELSE BEGIN
	printf,unit,'\hline'
	printf,unit,'\end{tabular}'
	printf,unit,'\end{table}'
ENDELSE
printf,unit,'\end{document}'
free_lun,unit
;
spawn,'sed -i {s/AC0/AC/g} '+obslog
spawn,'sed -i {s/AE0/AE/g} '+obslog
spawn,'sed -i {s/AW0/AW/g} '+obslog
spawn,'sed -i {s/W07/W7/g} '+obslog
spawn,'sed -i {s/E06/E6/g} '+obslog
;
end
;-------------------------------------------------------------------------------
pro psn2tex,component,system,utd=utd,hjd=hjd,aas=aas,load=load
;
; Prepare LaTeX file from astrometric measurements of a binary.
; Must load astrometry and read model first. Use HJD if hjd=1 instead of JY. 
; Load interferometry first if load=1 to compute number of visibilities used.
; Prepare in A&A LaTex format (or AAS if aas=1).
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
if n_params() ne 2 then begin
	print,'Error: call must include comp. (e.g. A-B) and system (e.g. NPOI)'
	return
endif
;
print,'Note: all observations must be of the same interferometer!'
;
if n_elements(utd) eq 0 then utd=0
if n_elements(hjd) eq 0 then hjd=0
if n_elements(aas) eq 0 then aas=0
if n_elements(load) eq 0 then load=0
if n_elements(geninfo) eq 0 then load=0
;
if n_elements(component) eq 0 then begin
	component=''
	read,component,prompt='Please enter component: '
endif
if n_elements(positions) eq 0 then begin
	print,'***Error(PSN2TEX): no astrometry!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(PSN2TEX): no model!'
	return
endif
index=where(positions.component eq component,count)
if count eq 0 then begin
	print,'***Error(PSN2TEX): no data for this component!'
	return
endif
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
; Compute data
RAD=180/!pi
ds_options_i=ds_options.i
ds_options.i=0
calcmodel
ds_options.i=ds_options_i
x=positions.rho*sin(positions.theta)
y=positions.rho*cos(positions.theta)
xm=positions.rhom*sin(positions.thetam)
ym=positions.rhom*cos(positions.thetam)
oc=sqrt((x-xm)^2+(y-ym)^2)
oca=atan(x-xm,y-ym)*RAD
ocr=positions.rho-positions.rhom
oct=((positions.theta-positions.thetam)*RAD) mod 360
index=where(oct gt 180,count)
if count gt 0 then oct(index)=oct(index)-360
;
; Open output file
file_out=gen_model.starid+'_psn.tex'
if file_test(file_out) then begin
	answer=''
	read,answer,prompt='File '+file_out+' exists, overwrite? (y/n): '
	if strupcase(answer) eq N then return
endif
openw,unit,file_out,/get_lun
;
IF aas THEN BEGIN
;
; AJ and ApJ
printf,unit,'\documentclass{/home/chummel/latex/aastex/aastex61}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{lcccrrrrrrr}'
printf,unit,'\tabletypesize{\scriptsize}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +gen_model.starid+'}\label{table1}'
printf,unit,'\tablehead{'
printf,unit,'\colhead{}&'
if hjd then printf,unit,'\colhead{HJD $-$}&' $
       else printf,unit,'\colhead{}&'
if load then $
printf,unit,'\colhead{Number of} &'
printf,unit,'\colhead{$\rho$}&'
printf,unit,'\colhead{$\theta$}&'
printf,unit,'\colhead{$\sigma_{\rm maj}$}&'
printf,unit,'\colhead{$\sigma_{\rm min}$}&'
printf,unit,'\colhead{$\varphi$}&'
printf,unit,'\colhead{O--C$_\rho$}&'
printf,unit,'\colhead{O--C$_\theta$}'
printf,unit,'\\'
printf,unit,'\colhead{UT Date}&'
if hjd then printf,unit,'\colhead{$2,400,000$}&' $
       else printf,unit,'\colhead{Julian Year}&'
if load then $
printf,unit,'\colhead{visibilities}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}&'
printf,unit,'\colhead{(6)}&'
printf,unit,'\colhead{(7)}&'
printf,unit,'\colhead{(8)}&'
printf,unit,'\colhead{(9)}&'
if load then $
printf,unit,'\colhead{(10)}'
printf,unit,'}'
printf,unit,'\startdata'
;
ENDIF ELSE BEGIN
;
; A&A
printf,unit,'\documentclass[twocolumn]{/home/chummel/latex/aa-package/aa}'
printf,unit,'\usepackage{graphicx}'
printf,unit,'\usepackage{txfonts}'
printf,unit,'\usepackage{natbib}'
printf,unit,'\usepackage{upgreek}'
printf,unit,'\usepackage{url}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{NPOI results for '+gen_model.starid+'}'
printf,unit,'\label{npoi_res}'
if load then printf,unit,'\begin{tabular}{lccrrrrrrrr}' $
	else printf,unit,'\begin{tabular}{lccrrrrrrr}'
printf,unit,'\hline'
printf,unit,'\hline'
if load then begin
	if hjd then begin
	printf,unit,'UT Date&HJD $-$&Number of&$\rho$&$\theta$&$\sigma_{\rm maj}$&$\sigma_{\rm min}$&$\phi$&$O-C_\rho$&$O-C_\theta$\\' 
	printf,unit,'&$2,400,000$&visibilities&(mas)&(deg)&(mas)&(mas)&(deg)&(mas)&(deg)\\'
	endif else begin
	printf,unit,'UT Date&Julian year&Number of&$\rho$&$\theta$&$\sigma_{\rm maj}$&$\sigma_{\rm min}$&$\phi$&$O-C_\rho$&$O-C_\theta$\\'
	printf,unit,'&$2,400,000$&visibilities&(mas)&(deg)&(mas)&(mas)&(deg)&(mas)&(deg)\\'
	endelse
endif else begin
	printf,unit,'UT Date&Julian year&Number of&$\rho$&$\theta$&$\sigma_{\rm maj}$&$\sigma_{\rm min}$&$\phi$&$O-C_\rho$&$O-C_\theta$\\'
	printf,unit,'UT Date&HJD $-$&Number of&$\rho$&$\theta$&$\sigma_{\rm maj}$&$\sigma_{\rm min}$&$\phi$&$O-C_\rho$&$O-C_\theta$\\'
	printf,unit,'&$2,400,000$&visibilities&(mas)&(deg)&(mas)&(mas)&(deg)&(mas)&(deg)\\'
endelse
printf,unit,' (1) & (2) & (3) & (4) & (5) & (6) & (7) & (8) & (9) & (10) \\'
printf,unit,'\hline'
;
ENDELSE
;
for i=0,n_elements(positions)-1 do begin
	if positions(i).component eq component then begin
	date=positions(i).date	; Can be off by 1 day if obs. started early!
	date=jd2date(2440000.d0+positions(i).jd $
;		+double(system_config('NPOI','MIDNIGHT'))/24)
		+double(system_config(system,'MIDNIGHT'))/24)
	parsedate,date,y,m,d
	year=string(y,format='(i4.4)')
	month=months(long(strmid(date,5,2))-1)
	day=strmid(date,8,2)
	if positions(i).jy lt 1996 then filename=mark3date(y,m,d)+'.cha' else $
					filename=constrictordate(y,m,d)+'.cha'
	if load then begin
	night=where(geninfo.date eq date,count)
	if count eq 1 then begin
		loadnight,GenInfo(night(0)).date, $
			  GeoInfo(night(0)).systemid, $
			  GenInfo(night(0)).configid
		system=GeoInfo(night(0)).systemid
		ds_options.a=0
		marquardtdata,y,ysig,ymod,/noload
		ds_options.a=1
		num_vis=n_elements(y)
	endif else num_vis=0
	if hjd then $
	printf,unit,year+' '+month+' '+day, $
		jy2jd(positions(i).jy)-2400000.d0,num_vis, $
		positions(i).rho,positions(i).theta*RAD, $
		positions(i).emajor,positions(i).eminor,positions(i).pa*RAD, $
;		oc(i),oca(i), $
		ocr(i),oct(i), $
		format='(a,"&",f9.3,"&",i4,"&",f7.3,"&",f7.2,"&",f7.3,"&",f7.3,"&",f5.1,"&",f7.3,"&",f6.2,"\\")' $
	       else $
	printf,unit,month+' '+day+'\dotfill', $
		positions(i).jy,num_vis, $
		positions(i).rho,positions(i).theta*RAD, $
		positions(i).emajor,positions(i).eminor,positions(i).pa*RAD, $
;		oc(i),oca(i), $
		ocr(i),oct(i), $
		format='(a,"&",f9.4,"&",i4,"&",f6.2,"&",f7.2,"&",f7.3,"&",f7.3,"&",f5.1,"&",f6.2,"&",f6.1,"\\")'
	endif else begin
		num_vis=0
	printf,unit,month+' '+day+'\dotfill',positions(i).jy, $
		positions(i).rho,positions(i).theta*RAD, $
		positions(i).emajor,positions(i).eminor,positions(i).pa*RAD, $
		ocr(i),oct(i), $
		format='(a,"&",f9.3,"&",f6.2,"&",f7.2,"&",f6.2,"&",f6.2,"&",f5.1,"&",f6.2,"&",f6.1,"\\")'
	endelse
	endif
endfor
;
IF aas THEN BEGIN
	printf,unit,'\enddata'
	printf,unit,'\end{deluxetable}'
ENDIF ELSE BEGIN
	printf,unit,'\hline'
	printf,unit,'\end{tabular}'
	printf,unit,'\end{table}'
ENDELSE
printf,unit,'\end{document}'
free_lun,unit
;
print,'Table saved to: '+file_out
;
return
;
; Extra code
; A&A
printf,unit,'\documentclass{/home/chummel/latex/aa-package/aa}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption[]{}'
printf,unit,'\caption[]{Observation and result log for ' $
	    +gen_model.starid+'}'
printf,unit,'\label{table}'
printf,unit,'\centering'
printf,unit,'\begin{tabular}{ccc}'
printf,unit,'\hline\hline'
for i=0,n_elements(positions)-1 do begin
	if positions(i).component eq component then begin
	date=positions(i).date
	parsedate,date,y,m,d
	month=months(long(strmid(date,5,2))-1)
	day=strmid(date,8,2)
	printf,unit,month+' '+day+'\dotfill',positions(i).jy, $
		positions(i).rho,positions(i).theta*RAD, $
		positions(i).emajor,positions(i).eminor,positions(i).pa*RAD, $
		format='(a,"&",f9.3,"&",f6.2,"&",f7.2,"&",f6.2,"&",f6.2,"&",f5.1,"\\")'
	endif
endfor
printf,unit,'\hline'
printf,unit,'\end{tabular}'
printf,unit,'\end{table}'
printf,unit,'\end{document}'
;
; Plain LaTex
printf,unit,'\documentstyle[12pt]{article}'
printf,unit,'\renewcommand{\baselinestretch}{1.0}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{Observation and result log for '+gen_model.starid+'}'
printf,unit,'\begin{tabular}{lcccrrrrrr} \tableline \tableline'
printf,unit,'&&&'+ $
	    '\multicolumn{1}{c}{Number of}&'+ $
	    '\multicolumn{1}{c}{$\rho$}&'+ $
	    '\multicolumn{1}{c}{$\theta$}&'+ $
            '\multicolumn{1}{c}{$\sigma_{\rm maj}$}&'+ $
	    '\multicolumn{1}{c}{$\sigma_{\rm min}$}&'+ $
            '\multicolumn{1}{c}{$\varphi$}&'+ $
	    '\multicolumn{1}{c}{O-C}\\'
printf,unit,'\multicolumn{1}{c}{UT Date}&&\multicolumn{1}{c}{Julian Year}&'+ $
	    '\multicolumn{1}{c}{visibilities}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(deg)}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(deg)}&'+ $
	    '\multicolumn{1}{c}{(mas)}\\'
printf,unit,'\multicolumn{1}{c}{(1)}&&'+ $
	    '\multicolumn{1}{c}{(2)}&'+ $
	    '\multicolumn{1}{c}{(3)}&'+ $
	    '\multicolumn{1}{c}{(4)}&'+ $
	    '\multicolumn{1}{c}{(5)}&'+ $
	    '\multicolumn{1}{c}{(6)}&'+ $
	    '\multicolumn{1}{c}{(7)}&'+ $
	    '\multicolumn{1}{c}{(8)}&'+ $
	    '\multicolumn{1}{c}{(9)}\\ \tableline'
printf,unit,'\multicolumn{2}{l}{'+month+' '+day+'\dotfill}', $
		positions(i).jy,num_vis, $
		positions(i).rho,positions(i).theta*RAD, $
		positions(i).emajor,positions(i).eminor,positions(i).pa*RAD, $
		oc(i), $
		format='(a,"&",f9.4,"&",i4,"&",f5.2,"&",f7.2,"&",f6.3,"&",f6.3,"&",f5.1,"&",f5.3,"\\")'
printf,unit,'\tableline \tableline'
printf,unit,'\end{tabular}'
printf,unit,'\end{table}'
printf,unit,'\end{document}'
;
end
;-------------------------------------------------------------------------------
pro psn2tex_73leo,component,hjd=hjd,aas=aas
;
; Version for 73 Leo paper
;
; Prepare LaTeX file from astrometric measurements of a binary.
; Must load astrometry and read model first. Use HJD if hjd=1 instead of JY. 
; Prepare in A&A LaTex format (or AAS if aas=1).
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
if n_elements(hjd) eq 0 then hjd=0
if n_elements(aas) eq 0 then aas=0
;
component='A-B'
if n_elements(component) eq 0 then begin
	component=''
	read,component,prompt='Please enter component: '
endif
if n_elements(positions) eq 0 then begin
	print,'***Error(PSN2TEX): no astrometry!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(PSN2TEX): no model!'
	return
endif
index=where(positions.component eq component,count)
if count eq 0 then begin
	print,'***Error(PSN2TEX): no data for this component!'
	return
endif
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
; Compute data
RAD=180/!pi
ds_options_i=ds_options.i
ds_options.i=0
calcmodel
ds_options.i=ds_options_i
x=positions.rho*sin(positions.theta)
y=positions.rho*cos(positions.theta)
xm=positions.rhom*sin(positions.thetam)
ym=positions.rhom*cos(positions.thetam)
oc=sqrt((x-xm)^2+(y-ym)^2)
oca=atan(x-xm,y-ym)*RAD
ocr=positions.rho-positions.rhom
oct=((positions.theta-positions.thetam)*RAD) mod 360
index=where(oct gt 180,count)
if count gt 0 then oct(index)=oct(index)-360
;
; Add the delta_m_700 data included in Table 2
status=dc_read_free('dm.txt',jy,rho,theta,dm,edm,/col)
;
; Open output file
file_out=gen_model.starid+'_psn.tex'
if file_test(file_out) then begin
	answer=''
	read,answer,prompt='File '+file_out+' exists, overwrite? (y/n): '
	if strupcase(answer) eq 'N' then return
endif
openw,unit,file_out,/get_lun
;
IF aas THEN BEGIN
;
; AJ and ApJ
printf,unit,'\documentclass{/home/chummel/latex/aastex/aastex61}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{cccccrrr}'
printf,unit,'\tabletypesize{\scriptsize}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +gen_model.starid+'}\label{table1}'
printf,unit,'\tablehead{'
if hjd then printf,unit,'\colhead{HJD $-$}&' $
       else printf,unit,'\colhead{Julian Year}&'
printf,unit,'\colhead{$\theta$}&'
printf,unit,'\colhead{$\rho$}&'
printf,unit,'\colhead{$\sigma_{\rm maj}$}&'
printf,unit,'\colhead{$\sigma_{\rm min}$}&'
printf,unit,'\colhead{$\varphi$}&'
printf,unit,'\colhead{O--C$_\rho$}&'
printf,unit,'\colhead{O--C$_\theta$}'
printf,unit,'\\'
if hjd then printf,unit,'\colhead{$2,400,000$}&' $
       else printf,unit,'\colhead{}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}&'
printf,unit,'\colhead{(6)}&'
printf,unit,'\colhead{(7)}&'
printf,unit,'\colhead{(8)}'
printf,unit,'}'
printf,unit,'\startdata'
;
ENDIF ELSE BEGIN
;
printf,unit,'\documentclass[twocolumn]{/home/chummel/latex/aa-package/aa}'
printf,unit,'\usepackage{graphicx}'
printf,unit,'\usepackage{txfonts}'
printf,unit,'\usepackage{natbib}'
printf,unit,'\usepackage{upgreek}'
printf,unit,'\usepackage{url}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{NPOI results for '+gen_model.starid+'}'
printf,unit,'\label{npoi_res}'
printf,unit,'\begin{tabular}{ccrrrrrrr}'
printf,unit,'\hline'
printf,unit,'\hline'
printf,unit,'Julian year&Number of&$\rho$&$\theta$&$\sigma_{\rm maj}$&$\sigma_{\rm min}$&$\phi$&$O-C_\rho$&$O-C_\theta$\\'
printf,unit,'HJD $-$&Number of&$\rho$&$\theta$&$\sigma_{\rm maj}$&$\sigma_{\rm min}$&$\phi$&$O-C_\rho$&$O-C_\theta$\\'
printf,unit,'$2,400,000$&visibilities&(mas)&(deg)&(mas)&(mas)&(deg)&(mas)&(deg)\\'
printf,unit,' (1) & (2) & (3) & (4) & (5) & (6) & (7) & (8) & (9) \\'
printf,unit,'\hline'
;
ENDELSE
;
; Use shift_j to join consecutive lines
dm700=[ $
	'4.26±0.15', '4.24±0.14', '3.59±0.06', '3.47±0.05', '3.38±0.03', $
	'4.06±0.04', '4.32±0.07', '5.01±0.01', '3.38±0.12', '4.98±0.23', $
	'5.33±0.42', '3.58±0.05', '3.84±0.09', '3.93±0.10', '3.34±0.10', $
	'3.74±0.05', '4.39±0.27', '3.23±0.15', '3.69±0.07', '3.72±0.05', $
	'2.86±0.09', '3.71±0.07', '4.10±0.08', '3.70±0.07', '3.39±0.09', $
	'4.59±0.20', '4.24±0.22', '3.54±0.05', '3.56±0.05', '3.71±0.07', $
	'3.12±0.05', '3.55±0.05', '3.47±0.04', '3.85±0.07', '3.67±0.08', $
	'3.93±0.06', '3.85±0.06', '3.65±0.06', '3.35±0.03']

for i=0,n_elements(positions)-1 do begin
	if positions(i).component eq component then begin
	date=positions(i).date	; Can be off by 1 day if obs. started early!
	date=jd2date(2440000.d0+positions(i).jd $
		+double(system_config('NPOI','MIDNIGHT'))/24)
	parsedate,date,y,m,d
	year=string(y,format='(i4.4)')
	month=months(long(strmid(date,5,2))-1)
	day=strmid(date,8,2)
;
	if hjd then $
	printf,unit, $
		jy2jd(positions(i).jy)-2400000.d0, $
		positions(i).theta*RAD,positions(i).rho, $
		positions(i).emajor,positions(i).eminor,positions(i).pa*RAD, $
		ocr(i),oct(i), $
		format='(f9.3,"&",f7.2,"&",f6.2,"&",f6.2,"&",f6.2,"&",f5.1,"&",f6.2,"&",f6.1,"\\")' $
	       else $
	printf,unit, $
		positions(i).jy, $
		positions(i).theta*RAD,positions(i).rho, $
		dm700(i), $
		positions(i).emajor,positions(i).eminor,positions(i).pa*RAD, $
		ocr(i),oct(i), $
		format='(f9.3,"&",f7.2,"&",f6.2,"&",f6.2,"&",f6.2,"&",f5.1,"&",f6.2,"&",f6.1,"\\")'
	endif
endfor
;
IF aas THEN BEGIN
	printf,unit,'\enddata'
	printf,unit,'\end{deluxetable}'
ENDIF ELSE BEGIN
	printf,unit,'\hline'
	printf,unit,'\end{tabular}'
	printf,unit,'\end{table}'
ENDELSE
printf,unit,'\end{document}'
free_lun,unit
;
print,'Table saved to: '+file_out
;
end
;-------------------------------------------------------------------------------
pro psn2tex_36tau,component,hjd=hjd,aas=aas
;
; Version for 36 Tau paper
;
; Prepare LaTeX file from astrometric measurements of a binary.
; Must load astrometry and read model first. Use HJD if hjd=1 instead of JY. 
; Prepare in A&A LaTex format (or AAS if aas=1).
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
if n_elements(hjd) eq 0 then hjd=0
if n_elements(aas) eq 0 then aas=0
;
component='A-B'
if n_elements(component) eq 0 then begin
	component=''
	read,component,prompt='Please enter component: '
endif
if n_elements(positions) eq 0 then begin
	print,'***Error(PSN2TEX): no astrometry!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(PSN2TEX): no model!'
	return
endif
index=where(positions.component eq component,count)
if count eq 0 then begin
	print,'***Error(PSN2TEX): no data for this component!'
	return
endif
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
; Compute data
RAD=180/!pi
ds_options_i=ds_options.i
ds_options.i=0
calcmodel
ds_options.i=ds_options_i
x=positions.rho*sin(positions.theta)
y=positions.rho*cos(positions.theta)
xm=positions.rhom*sin(positions.thetam)
ym=positions.rhom*cos(positions.thetam)
oc=sqrt((x-xm)^2+(y-ym)^2)
oca=atan(x-xm,y-ym)*RAD
ocr=positions.rho-positions.rhom
oct=((positions.theta-positions.thetam)*RAD) mod 360
index=where(oct gt 180,count)
if count gt 0 then oct(index)=oct(index)-360
;
; Open output file
file_out=gen_model.starid+'_psn.tex'
if file_test(file_out) then begin
	answer=''
	read,answer,prompt='File '+file_out+' exists, overwrite? (y/n): '
	if strupcase(answer) eq 'N' then return
endif
openw,unit,file_out,/get_lun
;
IF aas THEN BEGIN
;
; AJ and ApJ
printf,unit,'\documentclass{/home/chummel/latex/aastex/aastex61}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{cccccrrr}'
printf,unit,'\tabletypesize{\scriptsize}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +gen_model.starid+'}\label{table1}'
printf,unit,'\tablehead{'
if hjd then printf,unit,'\colhead{HJD $-$}&' $
       else printf,unit,'\colhead{Julian Year}&'
printf,unit,'\colhead{$\theta$}&'
printf,unit,'\colhead{$\rho$}&'
printf,unit,'\colhead{$\sigma_{\rm maj}$}&'
printf,unit,'\colhead{$\sigma_{\rm min}$}&'
printf,unit,'\colhead{$\varphi$}&'
printf,unit,'\colhead{O--C$_\rho$}&'
printf,unit,'\colhead{O--C$_\theta$}'
printf,unit,'\\'
if hjd then printf,unit,'\colhead{$2,400,000$}&' $
       else printf,unit,'\colhead{}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}&'
printf,unit,'\colhead{(6)}&'
printf,unit,'\colhead{(7)}&'
printf,unit,'\colhead{(8)}'
printf,unit,'}'
printf,unit,'\startdata'
;
ENDIF ELSE BEGIN
;
printf,unit,'\documentclass[twocolumn]{/home/chummel/latex/aa-package/aa}'
printf,unit,'\usepackage{graphicx}'
printf,unit,'\usepackage{txfonts}'
printf,unit,'\usepackage{natbib}'
printf,unit,'\usepackage{upgreek}'
printf,unit,'\usepackage{url}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{NPOI results for '+gen_model.starid+'}'
printf,unit,'\label{npoi_res}'
printf,unit,'\begin{tabular}{ccrrrrrrr}'
printf,unit,'\hline'
printf,unit,'\hline'
printf,unit,'Julian year&Number of&$\rho$&$\theta$&$\sigma_{\rm maj}$&$\sigma_{\rm min}$&$\phi$&$O-C_\rho$&$O-C_\theta$\\'
printf,unit,'HJD $-$&Number of&$\rho$&$\theta$&$\sigma_{\rm maj}$&$\sigma_{\rm min}$&$\phi$&$O-C_\rho$&$O-C_\theta$\\'
printf,unit,'$2,400,000$&visibilities&(mas)&(deg)&(mas)&(mas)&(deg)&(mas)&(deg)\\'
printf,unit,' (1) & (2) & (3) & (4) & (5) & (6) & (7) & (8) & (9) \\'
printf,unit,'\hline'
;
ENDELSE
;
for i=0,n_elements(positions)-1 do begin
	if positions(i).component eq component then begin
	date=positions(i).date	; Can be off by 1 day if obs. started early!
	date=jd2date(2440000.d0+positions(i).jd $
		+double(system_config('NPOI','MIDNIGHT'))/24)
	parsedate,date,y,m,d
	year=string(y,format='(i4.4)')
	month=months(long(strmid(date,5,2))-1)
	day=strmid(date,8,2)
;
	if hjd then $
	printf,unit, $
		jy2jd(positions(i).jy)-2400000.d0, $
		positions(i).theta*RAD,positions(i).rho, $
		positions(i).emajor,positions(i).eminor,positions(i).pa*RAD, $
		ocr(i),oct(i), $
		format='(f9.3,"&",f7.2,"&",f6.2,"&",f6.2,"&",f6.2,"&",f5.1,"&",f6.2,"&",f6.1,"\\")' $
	       else $
	printf,unit, $
		positions(i).jy, $
		positions(i).theta*RAD,positions(i).rho, $
		positions(i).emajor,positions(i).eminor,positions(i).pa*RAD, $
		ocr(i),oct(i), $
		format='(f9.3,"&",f7.2,"&",f6.2,"&",f6.2,"&",f6.2,"&",f5.1,"&",f6.2,"&",f6.1,"\\")'
	endif
endfor
;
IF aas THEN BEGIN
	printf,unit,'\enddata'
	printf,unit,'\end{deluxetable}'
ENDIF ELSE BEGIN
	printf,unit,'\hline'
	printf,unit,'\end{tabular}'
	printf,unit,'\end{table}'
ENDELSE
printf,unit,'\end{document}'
free_lun,unit
;
print,'Table saved to: '+file_out
;
end
;-------------------------------------------------------------------------------
pro psn3tex
;
; Prepare LaTeX file from astrometric measurements of a triple star
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
if n_elements(positions) eq 0 then begin
	print,'***Error(PSN3TEX): no astrometry!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(PSN3TEX): no model!'
	return
endif
;
dates=unique(positions.jy)
comps=unique(positions.component)
num=n_elements(dates)
;
openw,unit,'psn.tex',/get_lun
;
printf,unit,'\documentclass{/home/chummel/latex/aastex502/aastex}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{lcccccrrrrrr}'
printf,unit,'\tablecaption{Observation and result log for ' $
	    +gen_model.starid+'\label{table1}}'
printf,unit,'\tablehead{'
printf,unit,'\colhead{}&'
printf,unit,'\colhead{}&'
printf,unit,'\colhead{} &'
printf,unit,'\colhead{$\sigma_{\rm maj}$}&'
printf,unit,'\colhead{$\sigma_{\rm min}$}&'
printf,unit,'\colhead{$\varphi$}&'
printf,unit,'\colhead{$\rho_{\rm '+comps(0)+'}$}&'
printf,unit,'\colhead{$\theta_{\rm '+comps(0)+'}$}&'
printf,unit,'\colhead{O--C}&'
printf,unit,'\colhead{$\rho_{\rm '+comps(1)+'}$}&'
printf,unit,'\colhead{$\theta_{\rm '+comps(1)+'}$}&'
printf,unit,'\colhead{O--C}'
printf,unit,'\\'
printf,unit,'\colhead{UT Date}&'
printf,unit,'\colhead{Julian Year}&'
printf,unit,'\colhead{$N_{\rm vis.}$}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(mas)}&'
printf,unit,'\colhead{(deg)}&'
printf,unit,'\colhead{(mas)}'
printf,unit,'\\'
printf,unit,'\colhead{(1)}&'
printf,unit,'\colhead{(2)}&'
printf,unit,'\colhead{(3)}&'
printf,unit,'\colhead{(4)}&'
printf,unit,'\colhead{(5)}&'
printf,unit,'\colhead{(6)}&'
printf,unit,'\colhead{(7)}&'
printf,unit,'\colhead{(8)}&'
printf,unit,'\colhead{(9)}&'
printf,unit,'\colhead{(19)}&'
printf,unit,'\colhead{(11)}&'
printf,unit,'\colhead{(12)}'
printf,unit,'}'
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
RAD=180/!pi
;
printf,unit,'\startdata'
;
for i=0,n_elements(dates)-1 do begin
	i0=where(positions.jy eq dates(i))
	i1=where(positions.jy eq dates(i) and positions.component eq comps(0),i1c)
	i2=where(positions.jy eq dates(i) and positions.component eq comps(1),i2c)
	i0=i0(0)
	i1=i1(0)
	i2=i2(0)
;
	if i1c gt 0 then begin
		rho1=string(positions(i1).rho,format='(f3.1)')
		theta1=string(positions(i1).theta*RAD,format='(f5.1)')
		x=positions(i1).rho*sin(positions(i1).theta)
		y=positions(i1).rho*cos(positions(i1).theta)
		xm=positions(i1).rhom*sin(positions(i1).thetam)
		ym=positions(i1).rhom*cos(positions(i1).thetam)
		oc1=string(sqrt((x-xm)^2+(y-ym)^2),format='(f4.2)')
	endif else begin
		rho1=''
		theta1=''
		oc1=''
	endelse
;
	if i2c gt 0 then begin
		rho2=string(positions(i2).rho,format='(f5.1)')
		theta2=string(positions(i2).theta*RAD,format='(f5.1)')
		x=positions(i2).rho*sin(positions(i2).theta)
		y=positions(i2).rho*cos(positions(i2).theta)
		xm=positions(i2).rhom*sin(positions(i2).thetam)
		ym=positions(i2).rhom*cos(positions(i2).thetam)
		oc2=string(sqrt((x-xm)^2+(y-ym)^2),format='(f4.2)')
	endif else begin
		rho2=''
		theta2=''
		oc2=''
	endelse
;
	parsedate,positions(i0).date,y,m,d
	filename=constrictordate(y,m,d)+'.cha'
	load_interferometry,filename
	ds_options.i=1
	ds_options.a=0
	marquardtdata,y,ysig,ymod
	num_vis=n_elements(y)
	month=months(long(strmid(date,5,2))-1)
	day=strmid(date,8,2)
	printf,unit,month+' '+day+'\dotfill', $
		positions(i0).jy,num_vis, $
		positions(i0).emajor,positions(i0).eminor,positions(i0).pa*RAD, $
		rho1,theta1,oc1,rho2,theta2,oc2, $
		format='(a,"&",f8.3,"&",i4,"&",f4.2,"&",f4.2,"&",f5.1,"&",a,"&",a,"&",a,"&",a,"&",a,"&",a,"\\")'
endfor
;
printf,unit,'\enddata'
;
printf,unit,'\end{deluxetable}'
printf,unit,'\end{document}'
;
free_lun,unit
;
end
;-------------------------------------------------------------------------------
pro pti2tex
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ModelFit,parameters,ds_options
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SelDirs,nt_dir,nt_sel,st_dir,st_sel,x_dir,y_dir,z_dir,x_sel,y_sel,z_sel
;
if n_elements(scans) eq 0 then begin
	print,'***Error(PTI2TEX): no interferometry!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(PTI2TEX): no model!
	return
endif
;
openw,unit,'pti.tex',/get_lun
;
printf,unit,'\documentclass[12pt]{article}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{PTI Observation log for '+gen_model.starid+'}'
printf,unit,'\begin{tabular}{lcccrr} \tableline \tableline'
printf,unit,'&&&'+ $
	    '\multicolumn{1}{c}{Number of}&'+ $
	    '\multicolumn{1}{c}{$\rho_\mathrm{eph.}$}&'+ $
	    '\multicolumn{1}{c}{$\theta_\mathrm{eph.}$}\\'
printf,unit,'\multicolumn{1}{c}{UT Date}&&\multicolumn{1}{c}{Julian Year}&'+ $
	    '\multicolumn{1}{c}{visibilities}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(deg)}\\'
printf,unit,'\multicolumn{1}{c}{(1)}&&'+ $
	    '\multicolumn{1}{c}{(2)}&'+ $
	    '\multicolumn{1}{c}{(3)}&'+ $
	    '\multicolumn{1}{c}{(4)}&'+ $
	    '\multicolumn{1}{c}{(5)}\\ \tableline'
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
parsedate,date,y,m,d
jd=julian(y,m,d)+scans.time/86400
dates=jd2date(jd)
nites=unique(dates)
;
for i=0,n_elements(nites)-1 do begin
	index=where(dates eq nites(i),num_vis)
	jy=jd2jy(avg(jd(index)))
	pos=binarypos(jy2jd(jy))
	month=months(long(strmid(nites(i),5,2))-1)
	day=strmid(nites(i),8,2)
	printf,unit,'\multicolumn{2}{l}{'+month+' '+day+'\dotfill}', $
		jy,num_vis, $
		pos(0),pos(1), $
		format='(a,"&",f9.4,"&",i4,"&",f5.2,"&",f7.2,"\\")'
endfor
;
printf,unit,'\tableline \tableline'
printf,unit,'\end{tabular}'
printf,unit,'\label{table}'
printf,unit,'\end{table}'
printf,unit,'\end{document}'
;
free_lun,unit
return
;
; Version not using aastex
printf,unit,'\documentstyle[12pt]{article}'
printf,unit,'\renewcommand{\baselinestretch}{1.0}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{PTI Observation log for '+gen_model.starid+'}'
printf,unit,'\begin{tabular}{lcccrr} \tableline \tableline'
printf,unit,'&&&'+ $
	    '\multicolumn{1}{c}{Number of}&'+ $
	    '\multicolumn{1}{c}{$\rho_\mathrm{eph.}$}&'+ $
	    '\multicolumn{1}{c}{$\theta_\mathrm{eph.}$}\\'
printf,unit,'\multicolumn{1}{c}{UT Date}&&\multicolumn{1}{c}{Julian Year}&'+ $
	    '\multicolumn{1}{c}{visibilities}&'+ $
	    '\multicolumn{1}{c}{(mas)}&'+ $
	    '\multicolumn{1}{c}{(deg)}\\'
printf,unit,'\multicolumn{1}{c}{(1)}&&'+ $
	    '\multicolumn{1}{c}{(2)}&'+ $
	    '\multicolumn{1}{c}{(3)}&'+ $
	    '\multicolumn{1}{c}{(4)}&'+ $
	    '\multicolumn{1}{c}{(5)}\\ \tableline'
;
printf,unit,'\multicolumn{2}{l}{'+month+' '+day+'\dotfill}', $
		jy,num_vis, $
		pos(0),pos(1), $
		format='(a,"&",f9.4,"&",i4,"&",f5.2,"&",f7.2,"\\")'
;
printf,unit,'\tableline \tableline'
printf,unit,'\end{tabular}'
printf,unit,'\label{table}'
printf,unit,'\end{table}'
printf,unit,'\end{document}'
;
end
;-------------------------------------------------------------------------------
pro vel2tex,aas=aas
;
; Prepare LaTeX file from radial velocity data of a binary.
; Must load data and read model first (for computing (O-C) values).
; Prepare in A&A LaTex format (or AAS if aas=1).
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(aas) eq 0 then aas=0
;
if n_elements(velocities) eq 0 then begin
	print,'***Error(VEL2TEX): no RVs!'
	return
endif
if n_elements(gen_model) eq 0 then begin
	print,'***Error(VEL2TEX): no model!'
	return
endif
;
openw,unit,gen_model.starid+'_vel.tex',/get_lun
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
IF aas THEN BEGIN
;
; AJ and ApJ
;
printf,unit,'\documentclass{/home/chummel/latex/aastex6/aastex}'
printf,unit,'\begin{document}'
printf,unit,'\begin{deluxetable}{rllrrrrrr}'
printf,unit,'\tablecaption{Radial velocity result log for '+gen_model.starid+'\label{table1}}'
;
printf,unit,'\tablehead{'
;
printf,unit,'\colhead{}&'
printf,unit,'\colhead{Julian}&'
printf,unit,'\colhead{}&'
printf,unit,'\multicolumn{3}{c}{Primary RVs [km/s]}&'
printf,unit,'\multicolumn{3}{c}{Secondary RVs [km/s]}'
printf,unit,'\\'
printf,unit,'\colhead{Date}&'
printf,unit,'\colhead{Year}&'
printf,unit,'\colhead{MJD}&'
printf,unit,'\colhead{RV}&'
printf,unit,'\colhead{$\sigma$}&'
printf,unit,'\colhead{$(O-C)$}&'
printf,unit,'\colhead{RV}&'
printf,unit,'\colhead{$\sigma$}&'
printf,unit,'\colhead{$(O-C)$}'
printf,unit,'}'
;
jds=unique(velocities.jd)
;
printf,unit,'\startdata'
;
for i=0,n_elements(jds)-1 do begin
	i_a=where(velocities.jd eq jds(i) and velocities.component eq 'A',a_c)
	i_b=where(velocities.jd eq jds(i) and velocities.component eq 'B',b_c)
	if a_c eq 1 then index=i_a else index=i_b
	parsedate,velocities(index).date,y,m,d
	month=months(long(strmid(velocities(index).date,5,2))-1)
	day=strmid(velocities(index).date,8,2)
	if a_c+b_c eq 2 then begin
	printf,unit,month+' '+day+'\dotfill', $
		jd2jy(velocities(index).jd+2440000.d0), $
		velocities(index).jd+39999.5d0, $
		velocities(i_a).value, $
		velocities(i_a).error, $
		velocities(i_a).value-velocities(i_a).valuem, $
		velocities(i_b).value, $
		velocities(i_b).error, $
		velocities(i_b).value-velocities(i_b).valuem, $
		format='(a,"&",f9.4,"&",f9.2,"&$",f5.1,"$&$",f4.1,"$&$",f4.1,"$&$",f5.1,"$&$",f4.1,"$&$",f4.1,"$","\\")'
	endif else if a_c eq 1 then begin
	printf,unit,month+' '+day+'\dotfill', $
		jd2jy(velocities(index).jd+2440000.d0), $
		velocities(i_a).jd+39999.5d0, $
		velocities(i_a).value, $
		velocities(i_a).error, $
		velocities(i_a).value-velocities(i_a).valuem, $
		format='(a,"&",f9.4,"&",f9.2,"&$",f5.1,"$&$",f4.1,"$&$",f5.2,"$&","&","&","\\")'
	endif else begin
	printf,unit,month+' '+day+'\dotfill', $
		jd2jy(velocities(index).jd+2440000.d0), $
		velocities(i_b).jd+39999.5d0, $
		velocities(i_b).value, $
		velocities(i_b).error, $
		velocities(i_b).value-velocities(i_b).valuem, $
		format='(a,"&",f9.4,"&",f9.2,"&","&","&","&$",f5.1,"$&$",f4.1,"$&$",f4.1,"$","\\")'
	endelse
endfor
;
printf,unit,'\enddata'
printf,unit,'\end{deluxetable}'
;
ENDIF ELSE BEGIN
;
; Version not using aastex
printf,unit,'\documentstyle[12pt]{article}'
printf,unit,'\renewcommand{\baselinestretch}{1.0}'
printf,unit,'\begin{document}'
printf,unit,'\begin{table}'
printf,unit,'\caption{Observation and result log for '+gen_model.starid+'}'
printf,unit,'\begin{tabular}{lllrrrrrrr}'
;
printf,unit,'&'+ $
	    '\multicolumn{1}{c}{Julian}&'+ $
	    '&'+ $
	    '\multicolumn{3}{c}{Primary}&'+ $
	    '\multicolumn{3}{c}{Secondary}\\'
printf,unit,'\multicolumn{1}{c}{Date}&'+ $
	    '\multicolumn{1}{c}{Year}&'+ $
	    '\multicolumn{1}{c}{MJD}&'+ $
	    '\multicolumn{1}{c}{[km/s]}&'+ $
	    '\multicolumn{1}{c}{$\pm$}&'+ $
	    '\multicolumn{1}{c}{$(O-C)$}&'+ $
	    '\multicolumn{1}{c}{[km/s]}&'+ $
	    '\multicolumn{1}{c}{$\pm$}&'+ $
	    '\multicolumn{1}{c}{$(O-C)$}&'+ $
            '\multicolumn{1}{c}{Observatory}\\ \tableline'
;
printf,unit,month+' '+day+'\dotfill', $
		jd2jy(velocities(index(i)).jd+2440000.d0), $
		velocities(index(i)).jd+39999.5d0, $
		velocities(index(i)).value,velocities(index(i)).error, $
		velocities(index(i)).value-velocities(index(i)).valuem, $
		velocities(index).value,velocities(index).error, $
		velocities(index).value-velocities(index).valuem, $
		velocities(index(i)).comment, $
		format='(a,"&",f9.4,"&",f9.2,"&",f5.1,"&",f5.2,"&",f5.2,"&",f5.1,"&",f5.2,"&",f5.2,"&",a,"\\")'
;
printf,unit,'\tableline \tableline'
printf,unit,'\end{tabular}'
printf,unit,'\end{table}'
;
ENDELSE
;
printf,unit,'\end{document}'
free_lun,unit
;
end
;-------------------------------------------------------------------------------
