;*******************************************************************************
; File: beam.pro
;
; Description:
; ------------
;
; Block directory:
; ----------------
; Block 1: beampatch,dirtybeam,cleanbeam,displaybeam,contourbeam
;
;************************************************************************Block 1
function beampatch,beam,level=level,radius=radius
;
; Superceded by function of same name after this!
;
; Return a map with the beam patch, i.e. all pixel values greater than 1/5
; of the maximum set to 1, all others set to 0. Use radius [pixels] to blank
; all pixels outside the specified radius.
;
if n_elements(level) eq 0 then level=0.2
if n_elements(radius) eq 0 then radius=0
;
imsize=n_elements(beam(*,0))
jmsize=n_elements(beam(0,*))
;
if radius ne 0 then begin
	b=shift(dist(imsize,jmsize),imsize/2,jmsize/2)
	b=-b+max(b)
	index=where(b lt sqrt(2)*((imsize+jmsize)/4-radius))
	beam(index)=0
endif
;
b=fltarr(imsize+2,jmsize+2)
patch=b
b(1:imsize,1:jmsize)=beam
bmax=max(b)
blim=bmax*level
;
i0=imsize/2+1
j0=jmsize/2+1
;
; Don't assume peak is at center
index=where(beam eq max(beam))
i0=index(0) mod imsize
j0=index(0) / jmsize
;
i=0l
repeat begin
	j=0l
	while b(i0+i,j0-j) gt blim do begin
		patch(i0+i,j0-j)=1
		j=j+1
	endwhile
	j1=j
	j=1l
	while b(i0+i,j0+j) gt blim do begin
		patch(i0+i,j0+j)=1
		j=j+1
	endwhile
	j2=j
	j0=j0+(j2-j1)/2
	i=i+1
endrep until b(i0+i,j0) le blim
;
i=1l
j0=jmsize/2+1
repeat begin
	j=0l
	while b(i0-i,j0-j) gt blim do begin
		patch(i0-i,j0-j)=1
		j=j+1
	endwhile
	j1=j
	j=0l
	while b(i0-i,j0+j) gt blim do begin
		patch(i0-i,j0+j)=1
		j=j+1
	endwhile
	j2=j
	j0=j0+(j2-j1)/2
	i=i+1
endrep until b(i0-i,j0) le blim
;
return,patch(1:imsize,1:jmsize)
;
end
;-------------------------------------------------------------------------------
function beampatch2,beam,level=level,radius=radius
;
; Return a map with the beam patch, i.e. all pixel values greater than 1/5
; of the maximum set to 1, all others set to 0. Use radius [pixels] to blank
; all pixels outside the specified radius.
;
; Supercedes version above with better results.
;
if n_elements(level) eq 0 then level=0.1
if n_elements(radius) eq 0 then radius=0
;
imsize=n_elements(beam(*,0))
jmsize=n_elements(beam(0,*))
;
; Rotate and add
beam_in=beam
beam=(beam_in+rotate(beam_in,45)+rotate(beam_in,90)+rotate(beam_in,135))/4
; tvscl,beam
;
if radius ne 0 then begin
	b=shift(dist(imsize,jmsize),imsize/2,jmsize/2)
	b=-b+max(b)
	index=where(b lt sqrt(2)*((imsize+jmsize)/4-radius))
	beam(index)=0
endif
;
b=fltarr(imsize+2,jmsize+2)
patch=b
b(1:imsize,1:jmsize)=beam
bmax=max(b)
blim=bmax*level
;
i0=imsize/2+1
j0=jmsize/2+1
;
; Assume peak is at center
index=where(beam eq max(beam))
i0=index(0) mod imsize
j0=index(0) / jmsize
if 2*(i0+1) ne imsize or 2*(j0+1) ne jmsize then begin
	print,'Error(beampatch): beam is not at center!'
	return,0
endif
;
i=0l
repeat begin
	j=0l
	while b(i0+i,j0-j) gt blim do begin
		patch(i0+i,j0-j)=1
		j=j+1
	endwhile
	j1=j
	j=1l
	while b(i0+i,j0+j) gt blim do begin
		patch(i0+i,j0+j)=1
		j=j+1
	endwhile
	j2=j
	j0=j0+(j2-j1)/2
	i=i+1
endrep until b(i0+i,j0) le blim
;
i=1l
j0=jmsize/2+1
repeat begin
	j=0l
	while b(i0-i,j0-j) gt blim do begin
		patch(i0-i,j0-j)=1
		j=j+1
	endwhile
	j1=j
	j=0l
	while b(i0-i,j0+j) gt blim do begin
		patch(i0-i,j0+j)=1
		j=j+1
	endwhile
	j2=j
	j0=j0+(j2-j1)/2
	i=i+1
endrep until b(i0-i,j0) le blim
;
; Restore beam to input
beam=beam_in
;
return,patch(1:imsize,1:jmsize)
;
end
;-------------------------------------------------------------------------------
function dirtybeam,star
;
; Return the synthesized beam for a given star and observation.
;
common BeamInfo,reduce
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,scantable,bgtable,stationtable
;
if n_elements(star) eq 0 then begin
	print,'***Error(DIRTYBEAM): no star specified!'
	return,-1
endif
;
imsize=512
;
for night=0,n_elements(bufferinfo)-1 > 0 do begin
if n_elements(bufferinfo) gt 1 then $
loadnight,GenInfo(night(0)).date, $
          GeoInfo(night(0)).systemid, $
          GenInfo(night(0)).configid
index=where(scans.starid eq star and scantable.code eq 1,count)
if night eq 0 then reduce=2e6*max(abs(scans(index).uvw))/70e6 $
	      else reduce=max([2e6*max(abs(scans(index).uvw))/70e6,reduce])
endfor
map=fltarr(imsize,imsize)
;
FOR night=0,n_elements(bufferinfo)-1 > 0 DO BEGIN
;
if n_elements(bufferinfo) gt 1 then $
loadnight,GenInfo(night(0)).date, $
          GeoInfo(night(0)).systemid, $
          GenInfo(night(0)).configid
;
index=where(scans.starid eq star and scantable.code eq 1,count)
if count eq 0 then begin
	print,'***Error(DIRTYBEAM): no scans on this star for this date!'
	return,-1
endif
;
e=reform(scans(index).vissqcerr,n_elements(scans(index).vissqerr))
v=reform(scans(index).uvw(*,*,*,1),n_elements(scans(index).uvw(*,*,*,1)))
u=reform(scans(index).uvw(*,*,*,0),n_elements(scans(index).uvw(*,*,*,0)))
; If you want to compare the beam to the one in AIPS, flip the sign of u
u=-u
;
; Edit the uv-coverage
index=where(e gt 0,count)
if count eq 0 then begin
	print,'***Error(DIRTYBEAM): no data for this star!'
	return,-1
endif
u=u(index) & v=v(index)
;
; Place the data on a regular grid and add coordinates symmetric to origin
index=where(u lt 0,count) & if count gt 0 then u(index)=u(index)-reduce/2
index=where(u gt 0,count) & if count gt 0 then u(index)=u(index)+reduce/2
index=where(v lt 0,count) & if count gt 0 then v(index)=v(index)-reduce/2
index=where(v gt 0,count) & if count gt 0 then v(index)=v(index)+reduce/2
u=fix(u/reduce) & u=[u,-u]
v=fix(v/reduce) & v=[v,-v]
;
; Flip negative spatial frequencies
index=where(u lt 0,count) & if count gt 0 then u(index)=imsize+u(index)
index=where(v lt 0,count) & if count gt 0 then v(index)=imsize+v(index)
;
; Copy the grid into the map
map(u,v)=1
;
endfor
;
; Compute and return the beam
return,shift(float(fft(map,1)),imsize/2,imsize/2)
;
end
;-------------------------------------------------------------------------------
function cleanbeam,beam,scale,level=level,blank=radius
;
; Return major and minor axis, and position angle [deg] of FWHM clean beam
; ellipse. Iterate to find optimum level to avoid sidelobe contamination.
;
common BeamInfo,reduce
common FitEllipse,ex,ey
;
if not keyword_set(scale) then scale=1.
;
; These are passed to function beampatch
if n_elements(level) eq 0 then level=0.2
if n_elements(radius) eq 0 then radius=0
;
RAD=180/!pi
;
m=max(beam)
;
; Skipped, see iterative loop below
if 1 then begin
;
index=where(beam gt m/2-m/15 and beam lt m/2+m/15 $
	and beampatch(beam,level=level,radius=radius) eq 1,ndata)
if ndata lt 8 then begin
	print,'***Error(CLEANBEAM): too few half-maximum points!'
	return,-1
endif
imsize=n_elements(beam(0,*))
; Set data.
x=(index mod imsize)*(180/!pi)/(imsize*reduce)*3600e3 & x=x-avg(x)
y=(index/imsize)*(180/!pi)/(imsize*reduce)*3600e3 & y=y-avg(y)
;
endif else begin
;
; Loop to avoid sidelobes starting at input level
repeat begin
	index=where(beam gt m/2-m/15 and beam lt m/2+m/15 $
		and beampatch(beam,level=level,radius=radius) eq 1,ndata)
	if ndata lt 8 then begin
		print,'***Error(CLEANBEAM): too few half-maximum points!'
		return,-1
	endif
	imsize=n_elements(beam(0,*))
; 	Set data.
	x=(index mod imsize)*(180/!pi)/(imsize*reduce)*3600e3 & x=x-avg(x)
	y=(index/imsize)*(180/!pi)/(imsize*reduce)*3600e3 & y=y-avg(y)
;	Plot beam patch at current level
;	plot,x,y & wait,1
; 	Check for "sidelobes"
	r=poly_fit(x,y,1,yfit)
	xs=x(sort(x))
	nel=n_elements(xs)
	uxs=unique(xs(1:nel-1)-xs(0:nel-2))
	level=level+0.1
endrep until max(uxs) lt 0.6	; The value for level needs to be justified...
;
endelse
print,'Optimum level to suppress sidelobes: ',level-0.1
;
r=sqrt(x^2+y^2)
p=!pi-atan(x,y)
;
; Set parameters to fit
ma=5
ia=lonarr(ma)+1 & ia(0)=0 & ia(1)=0
index=where(ia ne 0,mfit)
e_parms=fltarr(ma)
e_parms(0)=0
e_parms(1)=0
e_parms(2)=max(r)
e_parms(3)=min(r)
i=where(r eq max(r)) & i=i(0)
e_parms(4)=p(i)
if e_parms(4) gt !pi then e_parms(4)=e_parms(4)-!pi
a=e_parms
;
; Store ellipse data
ex=r*sin(p)
ey=r*cos(p)
;
; Call marquardt
sig=fltarr(ndata)+0.1
pa=p	; ellfuncs changes p!
marquardt,'ellfuncs',pa,r,sig,ndata,a,ia,ma,chisq,covar
;
; Plot the half-power points and fit
; plot,-x,y,psym=1
; r=ellipse(a(2),a(3),p-a(4))
; oplot,r*sin(p),r*cos(p),psym=3
;
; Set new parameters
e_parms=a
e_parms(2)=a(2)*2*scale
e_parms(3)=a(3)*2*scale
e_parms(4)=a(4)*RAD
;
return,e_parms(2:4)
;
; Print inital estimates to screen
print,'__________________________________'
print,'Center x = ',e_parms(0)
print,'Center y = ',e_parms(1)
print,'Semi-major axis [mas] = ',e_parms(2)
print,'Semi-minor axis [mas] = ',e_parms(3)
print,'Position  angle [deg] = ',e_parms(4)
print,'_______________***________________'
;
end
;-------------------------------------------------------------------------------
pro displaybeam,star
;
; Display TV image of dirty beam.
;
beam=dirtybeam(star)
if beam(0) eq -1 then return
imsize=n_elements(beam(0,*))
window,/free,xsize=imsize,ysize=imsize,title=star
tvscl,beam
;
end
;-------------------------------------------------------------------------------
pro contourbeam,star
;
; Plot contours of dirty beam.
;
common BeamInfo,reduce
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
beam=dirtybeam(star)
if beam(0) eq -1 then return
;
imsize=n_elements(beam(0,*))
x=(findgen(imsize)-imsize/2)*(180/!pi)/(imsize*reduce)*3600e3
y=x
plevels=[5,10,20,50,80]
nlevels=[-80,-50,-20,-10,-5]
levels=[nlevels,plevels]
levels_label=''
for i=0,n_elements(levels)-1 do $
	levels_label=levels_label+string(levels(i),format='(i4)')
contour,beam,x,y,levels=plevels*max(beam)/100,xstyle=1,ystyle=1, $
	xtitle='Contours: '+levels_label+' % of peak',ytitle='[mas]', $
	title='Dirty beam for '+star+' on '+date
contour,beam,x,y,levels=nlevels*max(beam)/100,c_linestyle=1,/overplot
oplot,[0,0],[y(0),y(imsize-1)],psym=0,linestyle=1
oplot,[x(0),x(imsize-1)],[0,0],psym=0,linestyle=1
;
end
;-------------------------------------------------------------------------------
