;*******************************************************************************
; File: access.pro
;
; Description:
; ------------
; This container contains all procedures for data access, HDS, UV-FITS,
; OI-FITS. High-level IDL VLTI FITS data access is also included, but the
; lower level procedures are in the VINCI sub-directory.
; Some routines accessing interferometer data in ASCII format can also
; be found here.
;
; Block directory:
; ----------------
; Block 1: list_tree,list_names,list_days
;
;	   Access all HDS files
; Block 2: get_format,get_datum,get_systemid,get_sysconfig,get_bias,
;	   get_geoparms,get_genconfig,get_metroconfig,
;	   get_scantable,
;          get_scanlog,get_observerlog,
;	   get_constrictorlog,get_oysterlog,get_inchwormlog,get_logs,
;	   read_scanlog,read_observerlog,
;          get_numscan,get_numbgscan,get_nummotiongroup,get_numastromcorgrp,
;	   get_data,get_object
;
;	   Access CONSTRICTOR files
; Block 3: get_points,get_star,get_pointdata,
;	   get_bgtable,get_bgscans,get_bgdata,
;
;	   Access CHAMELEON files
; Block 4: get_scans,get_scandata,get_seeingdata,
;
;	   Access HDS raw files
; Block 5: get_records,
;	   get_rawfile,get_rawdata,get_bincounts,get_natcounts,
;	   quack
;
; Block 6: Access INCHWORM files
;	   get_avegroup,get_motiongroup,get_astromcorrgroup,get_inchgroup,
;	   get_inchdata,
;	   get_metrodata
;
;	   Write CHAMELEON data
; Block 7: put_sysconfig,put_geoparms,put_genconfig,put_metroconfig,
;	   put_format,put_date,put_systemid,put_userid,
;          put_scanlog,put_observerlog,put_constrictorlog,
;	   put_scandata,put_scans
;
;	   These read and write other formats
; Block 8: get_oifits,find_oiextn,check_oiextns,check_oifits,get_oitarget
;	   get_mark3data,get_ptidata,get_vincidata,
;	   get_amberdata,get_pionierdata,set_pionierfree,
;	   get_xdr,get_wdlinedata,get_opddata,get_sed,
; 	   put_oifits,put_oitarget,put_uvfits,put_xdr,
;	   put_mark3data
;
; Block 9: get_stationtable,put_stationtable,get_opdmodel
;
;************************************************************************Block 1
pro list_tree,indent,line
;
; Starting at the current level, list the HDS file structure. In case of
; array objects, this procedure will only descend the first element.
;
common ListTree,lines,do_return
;
if hds_state() eq 0 then begin
	print,'***Error(LIST_TREE): no HDS file open!'
	return
endif
;
if n_elements(indent) eq 0 then indent=-1
if n_elements(line) eq 0 then begin
	line=0
	lines=getenv('LINES')-2
	if lines le 0 then lines=22
	do_return=0
endif
;
dat_name,name
dat_type,type
dat_prim,prim
dat_shape,ndim,dims
if indent ge 0 then print,string(13b),'     ',format='(a,a,$)'
for i=1,indent do print,'     ',format='(a,$)'
if ndim gt 0 then begin
   print,name+'[',format='(a,$)'
   for i=1,ndim-1 do print,dims(i-1),'][',format='(i6,a,$)'
   print,dims(ndim-1),'] <'+strcompress(type,/remove_all)+'>',format='(i6,a,$)'
endif else print,name,' <'+strcompress(type,/remove_all)+'>',format='(a,a,$)'
for i=1,5-indent-ndim do print,'     ',format='(a,$)' & print,'          '
line=line+1
if line mod lines eq 0 then begin
	again:
	case myhak() of
		'q':	begin
			do_return=1
			return
			end
		' ':	lines=22
	string(10b):	lines=1
	       else:	goto,again
	endcase
endif
if prim eq 1 then return		; We return when object is primitive
dims=lonarr(7)+1
if ndim gt 0 then dat_cell,ndim,dims 	; We only look at the first cell!
list_names,comp_names
for i=1,n_elements(comp_names) do begin
	dat_find,comp_names(i-1)
	list_tree,indent+1,line
	dat_annul
	if do_return then begin
		if ndim gt 0 then dat_annul
		dat_name,name
		if indent lt 0 then print,string(13b), $
			'Current level = ',name,'.                             '
		return
	endif
endfor
if ndim gt 0 then dat_annul
dat_name,name
if indent lt 0 then print,string(13b), $
	'Current level = ',name,'.                             '
;
end
;-------------------------------------------------------------------------------
pro list_names,comp_names
;
if hds_state() eq 0 then begin
	print,'***Error(LIST_NAMES): no HDS file open!'
	return
endif
;
dat_shape,ndim,dims
;
; Check if object is primitive
dat_prim,reply
if reply eq 1 then begin
	print,'Object is primitive!'
	dat_type,type
	if ndim gt 0 then begin
	   	print,'Type=',type,', No. of dimensions=',ndim
		print,'Lengths=',dims(0:ndim-1) 
	endif else print,'Type=',type,', Scalar'
	return
endif
;
if ndim eq 0 then begin	
	dat_ncomp,n
	if n gt 0 then comp_names=strarr(n)
	for i=1,n do begin
		dat_index,i
		dat_name,name
		comp_names(i-1)=name
		dat_annul
	endfor
endif else if ndim eq 1 then begin
	comp_names=strarr(dims(0))
	for i=0,dims(0)-1 do comp_names(i)=string(format='(i4)',i+1)
endif else begin
	comp_names=elements(ndim,dims)
endelse
n=n_elements(comp_names)
if n_params() eq 0 then begin
	case ndim of
	0: for i=1,n do print,comp_names(i-1)
	1: print,'Array object: ',n,' elements!'
     else: print,'Array object: ',ndim,' dimensions!'
	endcase
endif
;
end
;-------------------------------------------------------------------------------
pro list_days,days
;
; List dates contained in HDS database file. This file must be open. 
; Now obsolete!
;
if hds_state() eq 0 then begin
	print,'***Error(LIST_DAYS): no HDS file open!'
	return
endif
;
get_format,format
if format ne 'CALBASE' then begin
	print,'***Error(LIST_DAYS): file must be CALBASE format!'
	return
end
;
toplevel
list_names,names
days=names(1:n_elements(names)-1)
if n_params() eq 0 then print,days
;
end
;************************************************************************Block 2
pro get_format,format
;
format=''
if hds_state() eq 0 then begin
	print,'***Error(GET_FORMAT): no HDS file open!'
	return
endif
toplevel
;
cmp_get0c,'FORMAT',format
;
end
;-------------------------------------------------------------------------------
pro get_datum,datum
;
; Read date. If no HDS file open, set to datum. Default is <today>.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if hds_state() eq 0 then begin
	if n_elements(datum) eq 0 then begin
		print,'Warning(GET_DATUM): Date undefined!'
		idldate=systime()
        	parseidldate,idldate,y,m,d
        	datum=nextdate(constrictordate(y,m,d))
        	print,'Set date to: ',datum
	endif
	Date=datum
	return
endif else begin
	toplevel
	cmp_get0c,'DATE',datum
	if n_params() eq 0 then Date=datum
endelse
;
end
;-------------------------------------------------------------------------------
pro get_systemid,sysid
;
; Read SystemId. If no HDS file open, set to sysid. Default is NPOI.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if hds_state() eq 0 then begin
	if n_elements(sysid) eq 0 then begin
		print,'Warning(GET_SYSTEMID): SystemId undefined!'
		sysid='NPOI/6way'
		print,'Set SystemId to: ',sysid
	endif
	SystemId=sysid
	return
endif else begin
	toplevel
	cmp_get0c,'SYSTEMID',sysid
	dat_find,'GENCONFIG'
	cmp_get0i,'BEAMCOMBINERID',BeamCombinerId
	sysid=system_id(sysid)+'/' $
	     +instrument_id(sysid,beamcombiner=BeamCombinerId)
	if n_params() eq 0 then SystemId=sysid
	dat_annul
	return
endelse
;
end
;-------------------------------------------------------------------------------
pro get_sysconfig,sysid=sysid,datum=datum,stations=stations
;
; Read all items in the common block SysConfig, i.e. the Date, SystemID,
; GeoParms, and GenConfig from the HDS data file. Compound procedure.
;
; If any of the SysConfig keywords are defined, close HDS file and
; create SysConfig objects from scratch.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; if keyword_set(sysid) then SystemId=sysid
; if keyword_set(datum) then Date=datum
;
; If sysid and/or datum and stations are supplied, close file and initialize
if keyword_set(sysid) or keyword_set(datum) $
		      or keyword_set(stations) gt 0 then hds_close
;
get_geoparms,sysid,datum
get_genconfig,stations
if checkdata([7,8]) ne 0 then return
;
; Metrology configuration only for NPOI observations (disabled in 2012)
; if SystemId eq 'NPOI' then get_metroconfig,no_alloc=no_alloc
;
; GenConfig.Date is a unique identifier for these observations
genconfig.date=geoparms.date
;
end
;-------------------------------------------------------------------------------
pro get_bias,bias
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([8]) ne 0 then return
maxBaseline=max(GenConfig.NumBaseline(0:GenConfig.NumOutBeam-1))
;
Bias=0
if hds_state() eq 1 then begin
	toplevel
	dat_find,'GenConfig'
	dat_find,'OutputBeam'
	name='BIAS'
	dat_there,name,reply
	if reply eq 1 then begin
		cmp_shape,name,ndim,dims
		cmp_getnd,name,ndim,dims,Bias
		print,'Bias read from HDS file.'
	endif
	dat_annul
endif
if total(abs(Bias)) eq 0 and SystemId eq 'NPOI' then begin
	parsedate,checkdate(),y,m,d
	biasfile=!oyster_dir+'npoi/biasbase.xdr'
	if total(strlen(file_search(biasfile))) ne 0 then begin
		restore,!oyster_dir+'npoi/biasbase.xdr'
		config=database.config
		parsedate,database.date,y0,m0,d0
		jd0=julian(y0,m0,d0) & si=sort(jd0) & jd0=jd0(si)
		index=where(jd0 le julian(y,m,d),count)
		if count gt 0 then begin
			index=where(jd0(count-1) eq julian(y0,m0,d0))
			Bias=database(index).bias
			GenConfig.config=config(index)
			print,'Bias read from ',database(index).date,' in data base.'
		endif
	endif else count=0
	if count eq 0 then begin
;	Based on 1998-07-14
	if julian(y,m,d) le julian(1999,12,31) then begin
	bdate='1998-07-14'
	Bias=transpose(reform([ $
	  [0.0899023,    0.167974,   0.0988566], $
	  [ 0.138122,    0.120250,    0.105369], $
	  [0.0853347,    0.128038,    0.108213], $
	  [0.0624924,    0.211609,   0.0632165], $
	  [ 0.111021,    0.110428,   0.0874976], $
	  [ 0.142510,    0.206276,   0.0679928], $
	  [0.0194781,   0.0885834,   0.0524271], $
	  [0.0583806,    0.110569,   0.0439823], $
	  [0.0667493,   0.0356532,   0.0646192], $
	  [0.0495556,   0.0542092,   0.0713222], $
	  [0.0757267,   0.0280108,   0.0711093], $
	  [0.0711259,   0.0437314,   0.0641880], $
	  [0.0651558,   0.0576786,   0.0598149], $
	  [0.0758625,   0.0294509,   0.0632960], $
	  [ 0.118095,   0.0377117,   0.0878602], $
	  [0.0985272,   0.0670980,    0.125428], $
	  [0.0592213,    0.121473,    0.162831], $
	  [0.0699054,    0.277655,   0.0521088], $
	  [0.0806393,    0.143614,   0.0768278], $
	  [0.0682241,    0.135369,    0.113044], $
	  [0.0818196,    0.243253,    0.289125], $
	  [0.0938335,    0.154664,     0.00000], $
	  [0.0718071,    0.130604,     0.00000], $
	  [0.0705534,    0.144733,     0      ], $
	  [0.0768314,    0.209294,    0.111030], $
	  [ 0.269136,    0.269348,   0.0987174], $
	  [  0.00000,    0.142651,   0.0929011], $
	  [0.0723583,    0.162545,   0.0498033], $
	  [0.0737517,     0.00000,     0.00000], $
	  [0.0647651,     0.00000,     0.00000], $
	  [0.0616645,     0.00000,     0.00000], $
	  [0.0605535,     0.00000,     0.00000],[fltarr(3,32)-1]],3,32,1,2))
	endif
;	Based on 2000-07-22
	if julian(y,m,d) ge julian(2000,7,21) then begin
	bdate='2000-07-22'
	Bias=reform(transpose( $
	      [[[   1.69177,    0.478276,   0.0796059], $
  		[  0.919373,    0.481476,    0.347879], $
  		[  0.302004,    0.370480,    0.158449], $
  		[  0.412788,    0.529370,    0.316756], $
  		[  0.561398,    0.274930,    0.750652], $
  		[  0.382302,    0.271530,    0.720391], $
  		[  0.428420,    0.180083,    0.127880], $
  		[   1.63636,    0.120278,    0.178796], $
  		[  0.269416,    0.529102,    0.341602], $
  		[  0.604920,    0.285504,    0.635365], $
  		[  0.415821,    0.240923,    0.499662], $
  		[  0.256408,    0.217809,     0.00000], $
  		[  0.289473,    0.312165,    0.230929], $
  		[  0.338644,    0.405591,    0.648344], $
  		[  0.199341,    0.233833,     2.45829], $
  		[  0.560465,    0.254822,     1.77624], $
  		[  0.213516,    0.629395,   0.0509382], $
  		[  0.182308,    0.411363,    0.260507], $
  		[  0.193218,    0.640461,    0.243130], $
  		[  0.164621,     1.24305,    0.220178], $
  		[  0.511859,     1.48610,    0.415244], $
  		[   1.26748,    0.137225,    0.342397], $
  		[  0.338130,    0.433200,   0.0746368], $
  		[  0.329144,    0.560588,   0.0979971], $
  		[  0.835394,    0.132254,    0.268300], $
  		[  0.355850,    0.469768,     6.89633], $
  		[  0.322371,    0.396236,   0.0951167], $
  		[  0.240567,    0.883198,    0.301117], $
  		[   0.00000,    0.489537,    0.130460], $
  		[   0.00000,     0.00000,    0.124764], $
  		[   0.00000,    0.453756,    0.106889], $
  		[   0.00000,    0.398829,    0.403420]], $
 	       [[  -1.25808,    -1.05920,   -0.869705], $
  		[  -1.09746,   -0.996786,   -0.981200], $
  		[  -1.07957,   -0.931071,   -0.878819], $
  		[  -1.15658,    -1.00325,    -1.01269], $
  		[  -1.06493,    -1.10437,    -1.09944], $
  		[  -1.02319,    -1.02622,    -1.06814], $
  		[  -1.04508,   -0.940143,    -1.15401], $
  		[  -1.10566,   -0.916836,   -0.914486], $
  		[  -1.00148,    -1.02493,   -0.985655], $
  		[  -1.03988,   -0.952320,    -1.12783], $
  		[  -1.20093,   -0.817902,    -1.10659], $
  		[ -0.974683,   -0.909846,     0.00000], $
  		[  -1.13678,   -0.938783,    -1.01431], $
  		[  -1.02520,   -0.935511,    -1.08890], $
  		[  -1.02507,   -0.833425,    -1.21337], $
  		[  -1.20829,    -1.00098,    -1.12522], $
  		[ -0.859674,    -1.00461,   -0.757941], $
  		[  -1.21178,   -0.866884,    -1.01431], $
  		[  -1.21244,    -1.01319,    -1.00573], $
  		[ -0.732162,   -0.962564,   -0.974857], $
  		[  -1.26526,   -0.932072,    -1.21793], $
  		[  -1.01042,   -0.818386,   -0.948013], $
  		[  -1.00903,    -1.03310,   -0.732898], $
  		[  -1.06772,   -0.967852,    -1.01297], $
  		[  -1.07422,   -0.872275,   -0.888332], $
  		[  -1.01493,   -0.945318,    -1.05196], $
  		[ -0.961374,   -0.961688,   -0.784427], $
  		[ -0.916645,   -0.831189,   -0.861407], $
  		[   0.00000,   -0.901701,    -1.71984], $
  		[   0.00000,     0.00000,   -0.823252], $
  		[   0.00000,   -0.972456,   -0.589181], $
  		[   0.00000,   -0.507537,   -0.830297]]]),2,1,32,3)
	endif
;	Based on 2000-09-20
	if julian(y,m,d) ge julian(2000,7,26) then begin
	bdate='2000-09-20'
	Bias=reform(transpose( $
	 [[[ 2.34974,    0.463135,    0.549775], $
	   [ 1.98768,    0.702093,    0.534707], $
	   [0.376229,    0.708216,    0.286134], $
	   [0.380634,    0.804916,    0.501519], $
	   [ 1.13563,    0.299085,    0.994746], $
	   [0.660806,    0.355672,    0.923398], $
	   [0.921255,    0.154945,   0.0534789], $
	   [ 2.47268,    0.230520,    0.275792], $
	   [0.375058,    0.689234,    0.538494], $
	   [ 1.06977,    0.453575,    0.590598], $
	   [0.201151,    0.481969,    0.820368], $
	   [0.550956,    0.262498,     1.87802], $
	   [0.109338,    0.444762,    0.275550], $
	   [0.229670,    0.604727,    0.676104], $
	   [0.306275,    0.394604,     2.55712], $
	   [0.391842,    0.365425,     1.92808], $
	   [0.252809,    0.709528,   0.0555068], $
	   [0.241256,    0.443538,    0.329732], $
	   [0.335814,    0.700353,    0.407276], $
	   [0.291203,     1.65236,    0.312362], $
	   [0.295144,     1.93693,    0.321580], $
	   [ 1.50629,    0.189681,    0.473475], $
	   [0.310947,    0.489772,    0.123695], $
	   [0.260669,    0.602627,   0.0825925], $
	   [0.834961,    0.140548,    0.278869], $
	   [0.364395,    0.491624,     9.71347], $
	   [0.335162,    0.411833,    0.124889], $
	   [0.254994,     1.04750,    0.347032], $
	   [ 0.00000,    0.531638,    0.142471], $
	   [ 0.00000,    0.535319,    0.112205], $
	   [ 0.00000,    0.480229,   0.0833645], $
	   [ 0.00000,    0.262069,    0.458110]], $
	  [[-1.34646,    -1.05240,    -1.54923], $
	   [-1.31814,    -1.12604,    -1.12693], $
	   [-1.17054,    -1.10487,    -1.10097], $
	   [-1.12462,    -1.15640,    -1.17742], $
	   [-1.23921,    -1.21548,    -1.17914], $
	   [-1.17260,    -1.09474,    -1.14134], $
	   [-1.29162,   -0.882525,   -0.912641], $
	   [-1.22885,    -1.14704,    -1.08366], $
	   [-1.11320,    -1.09484,    -1.10998], $
	   [-1.19563,    -1.11445,    -1.10278], $
	  [-0.979332,    -1.03008,    -1.19085], $
	  [ -1.21475,    -1.07988,   -0.486431], $
	  [-0.895925,    -1.06897,    -1.12273], $
	  [-0.964488,    -1.11071,    -1.11081], $
	  [ -1.24854,    -1.05966,    -1.21917], $
	  [ -1.14140,    -1.25711,    -1.16141], $
	  [ -1.00234,    -1.01273,   -0.845195], $
	  [ -1.42969,   -0.960407,    -1.12477], $
	  [ -1.41212,    -1.02264,    -1.36548], $
	  [-0.995011,    -1.06182,    -1.24190], $
	  [ -1.04056,    -1.02183,    -1.12542], $
	  [ -1.10609,    -1.02245,    -1.14857], $
	  [ -1.04787,    -1.05178,    -1.25820], $
	  [-0.940433,    -1.00779,    -1.05952], $
	  [ -1.08760,    -1.03038,   -0.908170], $
	  [ -1.00389,   -0.966287,    -1.20345], $
	  [-0.977306,   -0.966761,    -1.31777], $
	  [ -1.03555,    -1.02206,    -1.00028], $
	  [  0.00000,    -1.02635,    -1.27341], $
	  [  0.00000,   -0.741966,   -0.915020], $
	  [  0.00000,    -1.03946,   -0.916895], $
	  [  0.00000,    -1.01027,    -1.05683]]]),2,1,32,3)
	endif
	print,'Bias coefficients used are from '+bdate+'.'
	endif
endif
;
if n_params() eq 0 then GenConfig.V2Bias(*,0:maxBaseline-1,*,*) $
		                   =Bias(*,0:maxBaseline-1,*,*)
;
end
;-------------------------------------------------------------------------------
pro get_geoparms,sysid,datum
;
; Read GeoParms object in HDS file. If no file is open, set defaults
; based on SystemID. If sysid or datum are defined, close files and
; create object from inputs.
;
; Read earth orientation data in file mark3.dat.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
format=''
if hds_state() eq 1 then get_format,format
;
; Default Earth parameters
EarthRadius=6378140.d0
J2=0.00335281d0
;
; If sysid and/or datum are supplied, close file and initialize
if n_elements(sysid) ne 0 or n_elements(datum) ne 0 then hds_close
if n_elements(sysid) ne 0 then systemid=sysid
if n_elements(datum) ne 0 then date=datum
;
get_systemid,systemid
get_datum,date
;
; Create object from input parameters if no file open, or wrong format
IF hds_state() eq 0 or format eq 'INCHWORM' THEN BEGIN
	Longitude=system_config(systemid,'longitude')
	Latitude=system_config(systemid,'latitude')
	Altitude=system_config(systemid,'altitude')
ENDIF
;
; Read from HDS file if one is open
IF hds_state() eq 1 THEN BEGIN
;
toplevel
;
; Get site coordinates. Compare NPOI coordinates to coordinates of
; Perkins 72" telescope: Lon=111:32:09.3, Lat=35:05:48.6, Alt=2198 m
;
if format ne 'INCHWORM' then begin
	dat_find,'GEOPARMS'
	cmp_get0d,'LATITUDE',Latitude
	cmp_get0d,'LONGITUDE',Longitude	; East Longitude!
	cmp_get0d,'ALTITUDE',Altitude
	dat_there,'EARTHRADIUS',reply
	if reply eq 1 then begin
		cmp_get0d,'EARTHRADIUS',EarthRadius
		cmp_get0d,'J2',value
		if value ne 0 then J2=value
	endif
	dat_annul
endif
;
ENDIF
;
; Set TAI-UTC. See file ser7 in directory ser7 on maia.usno.navy.mil.
; If a new leap second is introduced, copy the first line with the proper 
; date and TAI-UTC entered to the end of the list below. Note that if, for
; example, a leap second is to be introduced June 30th, the seconds in the
; evening of that day will go 23:59:58, 23:59:59, 23:59:60, 0:00:00, i.e.
; the leap second will be the last second of that day.
;
TAI_UTC=28
parsedate,Date,y,m,d
if julian(y,m,d) ge julian(1994L,7L,1L) then TAI_UTC=29
if julian(y,m,d) ge julian(1996L,1L,1L) then TAI_UTC=30
if julian(y,m,d) ge julian(1997L,7L,1L) then TAI_UTC=31
if julian(y,m,d) ge julian(1999L,1L,1L) then TAI_UTC=32
if julian(y,m,d) ge julian(2006L,1L,1L) then TAI_UTC=33
if julian(y,m,d) ge julian(2009L,1L,1L) then TAI_UTC=34
if julian(y,m,d) ge julian(2012L,7L,1L) then TAI_UTC=35
if julian(y,m,d) ge julian(2015L,7L,1L) then TAI_UTC=36
if julian(y,m,d) ge julian(2017L,1L,1L) then TAI_UTC=37
; Last updated/checked 26 Oct 2023
;
; As of May, 1994, TDT=TAI+32.184s
TDT_TAI=32.184
;
; Get UT1-UTC data from VLBI polar motion measurements provided through 
; maia.usno.navy.mil:ser7/mark3.out. 
;
; Read one line, see what date it is...
status=dc_read_free(!oyster_dir+'usno/mark3.dat',year,month,day,mjd, $
	pole_x,pole_x_e,pole_y,pole_y_e,ut1_utc,ut1_utc_e,/col, $
	filters=['P'],ignore=['$TEXT_IN_NUMERIC'],nskip=3,nrecs=1)
if status ne 0 then begin
	print,'Warning(GET_GEOPARMS): failed to read UT1-UTC data!'
	mjd=0
endif
; ...and skip ahead to make reading faster
nskip=long(julian(y,m,d)-(mjd+2400000.5d0)) & nskip=nskip(0)
neighborhood=10
nskip=nskip-neighborhood
if nskip lt 0 then nskip=0
spawn,'wc -l '+!oyster_dir+'usno/mark3.dat',r
words=nameparse(r(0))
nlines=long(words(0))
if nskip gt nlines-neighborhood then nskip=0
status=dc_read_free(!oyster_dir+'usno/mark3.dat',year,month,day,mjd, $
        pole_x,pole_x_e,pole_y,pole_y_e,ut1_utc,ut1_utc_e,/col, $
        filters=['P'],ignore=['$TEXT_IN_NUMERIC'],nskip=nskip, $
	nrecs=2*neighborhood,resize=[1,2,3,4,5,6,7,8,9,10])
if status ne 0 then begin
	print,'Warning(GET_GEOPARMS): failed to read UT1-UTC data!'
	mjd=0
endif
;
; Believe it or not, but on 2002-03-12 the predicted ut1_utc_e was zero
index=where(ut1_utc_e eq 0,count)
if count gt 0 then ut1_utc_e(index)=median(ut1_utc_e)
;
; Fit 2nd order polynomials through data.
index=where((mjd+2400000.5d0) eq julian(y,m,d),count)
index=index(0)
if count eq 0 then begin
	print,'Warning(GET_GEOPARMS): date of observation not found '+ $
		'in VLBI data base...'
	ut1utc_coeffs=dblarr(3)
	pole_x_coeffs=dblarr(3)
	pole_y_coeffs=dblarr(3)
endif else if (index eq 0) or (index ge n_elements(mjd)-2) then begin
	print,'Warning(GET_GEOPARMS): date too near to limits '+ $
		'in VLBI data base...'
	ut1utc_coeffs=dblarr(3)
	pole_x_coeffs=dblarr(3)
	pole_y_coeffs=dblarr(3)
endif else begin
;
; 	Do a quadratic fit through the UT1-UTC data for 4 days starting 
;	with the night before the date of observation.
	t=fltarr(4)
	t(0)=mjd(index-1)-mjd(index)
	t(1)=0
	t(2)=mjd(index+1)-mjd(index)
	t(3)=mjd(index+2)-mjd(index)
	v=ut1_utc(index-1:index+2)
	e=ut1_utc_e(index-1:index+2)*10000
	ut1utc_coeffs=poly_fit(t,v,2,measure_errors=e,yfit=vfit)
        sig=stddev(v-vfit)
	if sig gt 0.001 then begin
		print,'Warning(GET_GEOPARMS): STDV (',sig,'[s]) of fit '+ $
			'to UT1-UTC large!'
	endif
;
;	Do a quadratic fit through pole_x data
	v=pole_x(index-1:index+2)
	e=pole_x_e(index-1:index+2)*1000
	pole_x_coeffs=poly_fit(t,v,2,measure_errors=e,yfit=vfit)
        sig=stddev(v-vfit)
	if sig gt 0.001 then begin
		print,'Warning(GET_GEOPARMS): STDV (',sig,'[m]) of fit '+ $
			'to pole_x large!'
	endif
;
;	Do a quadratic fit through pole_y data
	v=pole_y(index-1:index+2)
	e=pole_y_e(index-1:index+2)*1000
	pole_y_coeffs=poly_fit(t,v,2,measure_errors=e,yfit=vfit)
        sig=stddev(v-vfit)
	if sig gt 0.001 then begin
		print,'Warning(GET_GEOPARMS): STDV (',sig,'[m]) of fit ', $
			'to pole_y large!'
	endif
endelse
;
; Now allocate GeoParms structure
if n_elements(GeoParms) eq 0 then GeoParms=allocGeoParms()
;
; Fill in values
GeoParms.Date=Date
GeoParms.SystemId=system_id(SystemId)
GeoParms.Latitude=Latitude
GeoParms.Longitude=Longitude
GeoParms.Altitude=Altitude
GeoParms.EarthRadius=EarthRadius
GeoParms.J2=J2
GeoParms.pole_x_coeffs=pole_x_coeffs
GeoParms.pole_y_coeffs=pole_y_coeffs
GeoParms.ut1utc_coeffs=ut1utc_coeffs
GeoParms.TAI_UTC=TAI_UTC
GeoParms.TDT_TAI=TDT_TAI
;
; Store a copy in GeoInfo after clearing it, for use in AMOEBA
; if n_elements(GeoInfo) le 1 then GeoInfo=replicate(GeoParms,1)
;
print,'GeoParms loaded for '+date+'.'
;
end
;-------------------------------------------------------------------------------
pro get_genconfig,stations
;
; Read GenConfig object in HDS data file. Store a copy in GenInfo.
; If file not open, or stations are defined, close file and create
; object from inputs.
;
; GenConfig fields and their meaning for NPOI:
;
; Date			Not the "data date", but UT date of the night.
;			This date field is set when reading data.
; BeamCombinerID	1: 3-way, 2: 6-way, 3: 4-way astrometry.
; InstrCohInt		Instrumental coherent integration time in ms, i.e. 2 ms.
; RefStation		Reference station, first = 1.
; NumBin		Number of phase bins., e.g. 8 (3-way) or 64 (6-way).
; 
; INPUT BEAM
;
; NumSid	Number of siderostats used.
; StationID	AC0,AE0,AN0,AW0,N00,..,W01,..,E02,...,: Name of pier station.
; SiderostatID	1,2,...: unique IDs of the siderostats/NATs; NAT data is stored
;		by this ID, not in the order of StationID!
; SidConId	This is the ID of the sidcon rack;
; DelaylineID	1-6: 1 is closest to south wall, furthest from E-room.
; Stroke	Stroke amplitude in m for each FDL.
; LDL ID        1-6: from E to W.
; StarTrackerID	1-6: Quadcell ID used for angle tracking.
; BCInputID	1-6: 1 is western most (FDL side), 6 is east (Manifold side).
;		With 6-way, there is a 1-to-1 correspondence to DelaylineID.
; StationCoord	x,y,z,d: d is delay from station to array center.
;
; OUTPUT BEAM
;
; NumOutBeam		Number of spectrometers.
; SpectrometerID	Hardware ID of spectrometer.
; NumBaseline		Number of baselines in a spectrometer.
; FringeMod		Baseline Modulation, e.g. 1,2,3...
; NumBiasFreq		Number of frequencies used for bias computation
; BiasMod		Bias frequencies
; NumSpecChan		Number of spectral channels in a spectrometer.
; BaselineID		N09-W03,...: baseline names. Orientation sensitive.
; Wavelength		in m, for each channel and spectrometer.
; WavelengthErr
; ChanWidth		Width of spectral channel, in m.
; ChanWidthErr
;
; TRIPLE
;
; NumTriple	Number of computed triple combinations.
; TripleBeam	For each baseline, which spectrometer.
; TripleBase    For each baseline, which baseline.
; TripleChan	For each baseline, channel numbers involved.
; TripleNumChan	For each triple, how many channels.
;
; NPOI array diagram
;
;                                                          E-arm AE
;                                                           /\    
;                                                          /  \AE
;                                              AN         /   
;                                     N-arm----------- NF/\    
;                                             	      3 /6 \AC
;                                         6-pipe      2/ 5
;                                          stack      1 /4
;                                             	     / /
;               S6  S5  S4  S3  S2  S1              / WF
;        LDL 1                                   P1/ /  \AW
;                                                 /|/    \
;        LDL 2                                 P2/ |     W-arm   
;                                               /|/|
;        LDL 3  <------------E-------------- P3/ | |
;                ------------W---------------> |/| | 
;                                              | | | 
;                                             /| | | 
;        LDL 4                             P4/ | | |   <- Manifold room area
;                                           /| | | |
;        LDL 5                           P5/ | | | |
;                                         /| | | | |
;        LDL 6                         P6/ | | | | |
;                                        | | | | | |
;                                        6 5 4 3 2 1   -> south
;                                        | | | | | |
;
;
;                Beam combiner            --------6-way--------------   3-way
;                                                                     (before 2000)
;                                                       
;                   /--------------  6     N
;              M3b /-\-------------  5     W07
;                 /-\-\------------  4     W
;                  \ \ \
;                   \ \ \
;                    \ \ \ S1
;              /------\-\-/--------  3     E                              W(4)
;          M1 /-\------\-/-\-------  2     C                              E(3)
;            /-\-\------/-\-\------  1     E02                            C(2)
;             \ \ \      \ \ \
;              \ \ \ S2   \ \ \           
;         /-----\-\-/------\-/ \         
;    /-----\-----\-/--------/M2 \        
;     \  /--\-----/--------------/M3a    
;      \  \  \     \
;       \  \  \    (unused)                     
; Sp. ID 3  2  1                       |
; 3-way CE CW EW (before 2000)         |
; 3-way CW EW CE (2000 and after)      V
; 6-way 12 13 23 (2002 and after)     west
;       45 46 56
;
; All spectrometers have 50 micron pinholes (but see note below)
;
; Note: Sp3 has had 25 micron pinhole in place between August 1, 2000,
;		and September 23, 2002. 
;       Sp1 channel 12 is dead (broken fiber; I think beginning in 2000)
;
; Some notes on the array configuration:
; Light is intercepted by siderostats, and dropped vertically by the
; Elevator cans (inside astrometric huts, and on piers close to imaging
; stations). Feedpipes transfer light to array center, where the array
; center cans transfer light into the manifold room. Here they are dropped
; again by the periscopes, and fed into the long delay lines. There are 3 
; feed pipes per arm, 9 total. There are six delay pipes, always connected 
; to the same delay lines in the laboratory.
;
; In order to compute the path lengths, one file (station file) contains
; the paths from the siderostats to the array center. Another file gives
; for each combination of delayline ID and feedpipe ID the path through
; the array center to the beamcombiner. Another file will give for each
; combination of delayline ID and BeamCombinerInput ID the final pieces
; of path lengths for a specific BeamCombiner.
;
; Since the advent of the 6-way beam combiner and the Huggins fringe engine,
; observers have the capability to switch in and out stations during the night.
; Strictly speaking, these actions correspond to configuration changes which
; are not allowed in a single data container file. However, we have introduced
; a configuration ID index into some GenConfig fields, for now just Bias and
; Response, since the results for these calibration data depend on the 
; particular configuration.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; Create object from input parameters if defined, or no file open
if n_elements(stations) gt 0 or hds_state() eq 0 then begin
	hds_close
;	GenConfig valid
	if n_elements(stations) eq 0 and n_elements(size(GenConfig)) eq 4 $
		then stations=GenConfig.StationId(0:GenConfig.NumSid-1)
;	GenConfig invalid
	if n_elements(stations) eq 0 and n_elements(size(GenConfig)) eq 3 $
		then stations=system_config(systemid,'STATIONS')
;
	GenConfig=system_config(systemid,'CONFIG',stations=stations)
	result=size(GenConfig)
	if result(n_elements(result)-2) eq 8 then get_stationtable
	genconfig.diameter=system_config(system_id(systemid),'DIAMETER', $
		stations=genconfig.stationid(0))
	genconfig.configid=instrument_id(systemid)
	return
endif
;
; Change to toplevel in HDS file
toplevel
;
; Get format
get_format,format
;
; Get Date and SystemId
get_datum
get_systemid
;
; Extract configid
words=nameparse(systemid,'/')
if n_elements(words) eq 2 then configid=words(1)
;
; If refstation is defined, do not change it (see exceptions below)
if n_elements(size(GenConfig)) eq 4 then refstation=GenConfig.refstation $
			            else refstation=99
;
; If reading a CONSTRICTOR (pointdata) file, do read the reference station info
if format eq 'CONSTRICTOR' then refstation=99
;
; If reading a COBRA (raw data) file, do read the reference station info
if format eq 'COBRA' then refstation=99
;
; As of Version 8, we always read the refstation
refstation=99
;
; If date field is defined, keep it until changed when data is read
if n_elements(size(GenConfig)) eq 4 then datum=GenConfig.Date $
			            else datum=date
;
; Find dimensioning information and allocate GenConfig
dat_find,'GENCONFIG'
dat_find,'INPUTBEAM'
cmp_get0i,'NUMSID',NumSid
MaxSid=NumSid
dat_annul
dat_find,'OUTPUTBEAM'
cmp_get0i,'NUMOUTBEAM',NumOutBeam
cmp_get1i,'NUMBASELINE',NumOutBeam,NumBaseline
MaxBaseline=max(NumBaseline) 
cmp_get1i,'NUMSPECCHAN',NumOutBeam,NumSpecChan
MaxSpecChan=max(NumSpecChan)
dat_annul
name='TRIPLE'
dat_there,name,reply
if reply then begin
	dat_find,name
	cmp_get0i,'NUMTRIPLE',NumTriple
	dat_annul
endif else NumTriple=0	
MaxNumTriple=max([1,NumTriple])
dat_annul
;
MaxConfig=system_config(SystemId,'MAXCONFIG')
case string(maxConfig) of
	'scans':MaxConfig=scanconfig(/maxconfig)
	'stars':MaxConfig=n_elements(scanconfig(/starlist))
	   else:MaxConfig=fix(MaxConfig)
endcase
if MaxConfig le 0 then begin
	print,'***Error(GET_GENCONFIG): unable to get maxConfig!'
	return
endif
GenConfig=allocgenconfig(MaxSid,NumOutBeam,MaxNumTriple, $
			 MaxBaseline,MaxSpecChan,MaxConfig)
;
; Now load data into temporary variables
;
dat_find,'GENCONFIG'
;
IF format EQ 'INCHWORM' THEN BEGIN
;
cmp_get0i,'NUMPLATE',NumPlate
dat_find,'INPUTBEAM'
cmp_get0i,'NUMSID',NumSid
cmp_get1i,'SIDEROSTATID',NumSid,SiderostatId
SidConId=SiderostatId
dat_there,'STATIONID',reply
; This object was only implemented just before Nick left in 2002
if reply eq 1 then begin
	cmp_get1c,'STATIONID',NumSid,StationId
	GenConfig.StationId=StationId
endif
if refstation eq 99 then refstation=1
GenConfig.RefStation=refstation
GenConfig.NumSid=NumSid
GenConfig.SiderostatId=SiderostatId
GenConfig.SidConId=SidConId
GenConfig.NumPlate=NumPlate
print,'GenConfig (INCHWORM) loaded.'
;
ENDIF ELSE BEGIN
;
dat_there,'NUMPLATE',reply
if reply then cmp_get0i,'NUMPLATE',NumPlate
cmp_get0d,'INSTRCOHINT',InstrCohint 
cmp_get0i,'BEAMCOMBINERID',BeamCombinerId
dat_there,'NUMBIN',reply
if reply eq 1 then cmp_get0i,'NUMBIN',NumBin else NumBin=8
dat_there,'REFSTATION',reply
if reply eq 1 and refstation eq 99 then cmp_get0i,'REFSTATION',refstation
if refstation eq 99 then refstation=1	; in case reply=0
dat_there,'CONFIGID',reply
if reply eq 1 and n_elements(configid) eq 0 then $
	cmp_get0c,'CONFIGID',configid ; else configid='UNKNOWN'
;
; Get components in OutputBeam table............................................
dat_find,'OUTPUTBEAM'
cmp_get0i,'NUMOUTBEAM',NumOutBeam
cmp_get1i,'NUMBASELINE',NumOutBeam,NumBaseline
cmp_get1i,'NUMSPECCHAN',NumOutBeam,NumSpecChan
cmp_get1c,'SPECTROMETERID',NumOutBeam,SpectrometerId
;
MaxBaseline=max(NumBaseline) 
cmp_shape,'BASELINEID',ndim,dims
icheck=0
if dims(0) ne MaxBaseline then icheck=1
if dims(1) ne NumOutBeam then icheck=1
if icheck eq 1 then begin
	print,'***Error(GET_GENCONFIG): found dimensions of BaselineId '+ $
		'inconsistent with declarations!'
	return
endif
cmp_getnc,'BASELINEID',ndim,dims,BaselineId
index=where(strlen(BaselineId) eq 0,count)
if count gt 0 then BaselineId(index)='ST1-ST2'
;
MaxSpecChan=Max(NumSpecChan)
cmp_shape,'WAVELENGTH',ndim,dims
icheck=0
if dims(0) ne MaxSpecChan then icheck=1
if dims(1) ne NumOutBeam then icheck=1
if icheck eq 1 then begin
	print,'***Error(GET_GENCONFIG): found dimensions of Wavelength '+ $
		'inconsistent with declarations!'
	return
endif
cmp_getnd,'WAVELENGTH',ndim,dims,Wavelength
cmp_getnd,'WAVELENGTHERR',ndim,dims,WavelengthErr
cmp_getnd,'CHANWIDTH',ndim,dims,ChanWidth
cmp_getnd,'CHANWIDTHERR',ndim,dims,ChanWidthErr
;
V2Bias=0
name='BIAS'
dat_there,name,reply
if reply eq 1 then begin
	cmp_shape,name,ndim,dims
	cmp_getnd,name,ndim,dims,V2Bias
	if ndim eq 5 then numConfig=dims(ndim-1) $
		     else numConfig=1
	if numConfig gt MaxConfig then begin
		V2Bias=V2Bias(*,*,*,*,0:MaxConfig-1)
		numConfig=MaxConfig
	endif
endif else numConfig=MaxConfig
;
MaxNumBaseline=max(NumBaseline)
cmp_shape,'FRINGEMOD',ndim,dims
icheck=0
if dims(0) ne MaxNumBaseline then icheck=1
if dims(1) ne NumOutBeam then icheck=1
if icheck eq 1 then begin
	print,'***Error(GET_GENCONFIG): found dimensions of FringeMod '+ $
		'inconsistent with declarations!'
	return
endif
cmp_getni,'FRINGEMOD',ndim,dims,FringeMod
;
name='NumBiasFreq'
dat_there,name,reply
if reply then begin
	cmp_get1i,'NUMBIASFREQ',NumOutBeam,NumBiasFreq
	cmp_getni,'BIASMOD',ndim,dims,BiasMod
endif else begin
	NumBiasFreq=0
	BiasMod=0
endelse
dat_annul
;
; Get components in InputBeam table.............................................
dat_find,'INPUTBEAM'
cmp_get0i,'NUMSID',NumSid
if n_elements(NumPlate) eq 0 then NumPlate=NumSid
cmp_get1c,'STATIONID',NumSid,StationId
cmp_get1i,'SIDEROSTATID',NumSid,SiderostatId
dat_there,'SIDCONID',reply
if reply eq 1 then cmp_get1i,'SIDCONID',NumSid,SidConId $
	      else SidConId=SiderostatId
if total(abs(SidConId)) eq 0 then SidConId=SiderostatId
cmp_get1i,'DELAYLINEID',NumSid,DelayLineId
cmp_get1r,'STROKE',NumSid,Stroke 
if n_elements(Stroke) eq 0 then begin
	Stroke=fltarr(NumSid)
	if SystemId eq 'NPOI' and NumSid eq 3 then Stroke=[0,900d-9,1800d-9]
endif
cmp_get1i,'STARTRACKERID',NumSid,StarTrackerId
cmp_get1i,'BCINPUTID',NumSid,BCInputId
;
; Get station coordinates
StationCoord=dblarr(4,NumSid)
dat_find,'STATIONCOORD'
dat_shape,ndim,dims
icheck=0
if dims(0) ne 4 then icheck=1
if dims(1) ne NumSid then icheck=1
if icheck eq 1 then begin
	print,'***Error(GET_GENCONFIG): found dimensions of STATIONCOORD'+ $
		'inconsistent with declarations!'
	return
endif
dat_getnd,ndim,dims,StationCoord
dat_annul
dat_annul
;
; Get components in Triple table................................................
NumTriple=0
TripleBeam=0
TripleBase=0
TripleChan=0
TripleNumChan=0
TABias=0
name='TRIPLE'
dat_there,name,reply
if reply eq 1 then begin
	dat_find,name
	cmp_get0i,'NUMTRIPLE',NumTriple
	if NumTriple gt 0 then begin
;
		name='BIAS'
		dat_there,name,reply
		if reply eq 1 then begin
			cmp_shape,name,ndim,dims
			cmp_getnd,name,ndim,dims,TABias
			if ndim eq 4 then numConfig=dims(ndim-1) $
		     		else numConfig=1
			if numConfig gt MaxConfig then begin
				TABias=TABias(*,*,*,0:MaxConfig-1)
				numConfig=MaxConfig
			endif
		endif ; else numConfig=MaxConfig
;
		cmp_shape,'OUTPUTBEAM',ndim,dims
		icheck=0
		if ndim ne 2 then icheck=1
		if dims(0) ne 3 then icheck=1
		if dims(1) ne NumTriple then icheck=1
		if icheck eq 1 then begin
		 print,'***Error(GET_GENCONFIG): found dimensions of '+name+ $
				'inconsistent with declarations!'
		 return
		endif
		cmp_getni,'OUTPUTBEAM',ndim,dims,TripleBeam
		cmp_getni,'BASELINE',ndim,dims,TripleBase
;
		cmp_get1i,'NUMSPECCHAN',NumTriple,TripleNumChan
;
		cmp_shape,'SPECCHAN',ndim,dims
		icheck=0
		if ndim ne 3 then icheck=1
		if dims(0) lt Max(TripleNumChan) then icheck=1
	 	if dims(1) ne 3 then icheck=1
		if dims(2) ne NumTriple then icheck=1
		if icheck eq 1 then begin
		 print,'***Error(GET_GENCONFIG): found dimensions of '+name+ $
				'inconsistent with declarations!'
		 return
		endif
		cmp_getni,'SPECCHAN',ndim,dims,TripleChan
		TripleChan=TripleChan(0:max(TripleNumChan)-1,*,*)
	endif
endif 
;
; Now store temporary variables into previously allocated structure
;
GenConfig.Date=datum
GenConfig.RefStation=refstation
GenConfig.NumSid=NumSid
GenConfig.NumPlate=NumPlate
GenConfig.InstrCohint=InstrCohint
GenConfig.BeamCombinerId=BeamCombinerId > 1
GenConfig.ConfigId=strcompress(string(GenConfig.BeamCombinerId),/remove_all)
GenConfig.ConfigId=configid
GenConfig.NumBin=NumBin
GenConfig.StationId(0:NumSid-1)=StationId
GenConfig.SiderostatId(0:NumSid-1)=SiderostatId
GenConfig.SidConId(0:NumSid-1)=SidConId
GenConfig.DelayLineId(0:NumSid-1)=DelayLineId
GenConfig.Stroke(0:NumSid-1)=Stroke
GenConfig.BCInputId(0:NumSid-1)=BCInputId
GenConfig.StarTrackerId(0:NumSid-1)=StarTrackerId
GenConfig.StationCoord(*,0:NumSid-1)=StationCoord
GenConfig.NumOutBeam=NumOutBeam
GenConfig.NumBaseline(0:NumOutBeam-1)=NumBaseline
GenConfig.NumBiasFreq(0:NumOutBeam-1)=NumBiasFreq
GenConfig.NumSpecChan(0:NumOutBeam-1)=NumSpecChan
GenConfig.SpectrometerId(0:NumOutBeam-1)=SpectrometerId
GenConfig.BaselineId(0:MaxBaseline-1,0:NumOutBeam-1)=BaselineId
GenConfig.Wavelength(0:MaxSpecChan-1,0:NumOutBeam-1)=Wavelength
GenConfig.WavelengthErr(0:MaxSpecChan-1,0:NumOutBeam-1)=WavelengthErr
GenConfig.ChanWidth(0:MaxSpecChan-1,0:NumOutBeam-1)=ChanWidth
GenConfig.ChanWidthErr(0:MaxSpecChan-1,0:NumOutBeam-1)=ChanWidthErr
GenConfig.V2Bias(*,0:MaxBaseline-1,0:MaxSpecChan-1, $
		   0:NumOutBeam-1,0:numConfig-1)=V2Bias
GenConfig.TABias(*,0:MaxSpecChan-1,0:max([1,NumTriple])-1,0:numConfig-1)=TABias
GenConfig.FringeMod(0:MaxNumBaseline-1,0:NumOutBeam-1)=FringeMod
GenConfig.BiasMod(0:MaxNumBaseline-1,0:NumOutBeam-1)=BiasMod
GenConfig.NumTriple=NumTriple
if NumTriple gt 0 then begin
	GenConfig.TripleBeam(*,0:NumTriple-1)=TripleBeam
	GenConfig.TripleBase(*,0:NumTriple-1)=TripleBase
	GenConfig.TripleNumChan=TripleNumChan
	GenConfig.TripleChan(0:max(TripleNumChan)-1,*,0:NumTriple-1)=TripleChan
endif
genconfig.diameter=system_config(system_id(systemid),'DIAMETER', $
	stations=genconfig.stationid(0))
print,'GenConfig loaded.'
;
; Make sure configid is set to something useful
if strlen(configid) eq 0 then configid=strjoin(StationId,'-')
configid=strict_configid(configid)
;
ENDELSE
;
; Check fringe modulation against strokes
if system_id(systemid) eq 'NPOI' then begin 
for i=0,genconfig.numoutbeam-1 do begin
for j=0,genconfig.numbaseline(i)-1 do begin
        i1=where(genconfig.stationid eq strmid(genconfig.baselineid(j,i),0,3))
        i2=where(genconfig.stationid eq strmid(genconfig.baselineid(j,i),4,3))
        if nint((genconfig.stroke(i2)-genconfig.stroke(i1))/1e-6) ne $
           genconfig.fringemod(j,i) and genconfig.beamcombinerid ne 1 then $
        print,'Warning(GET_GENCONFIG): inconsistent modulation on beam ' $
                +string(i+1,format='(i1)') $
                +', baseline '+genconfig.baselineid(j,i)+'!'
endfor
endfor
endif
;
; Return to first level
toplevel
;
; Store a copy in GenInfo after clearing it, for use in AMOEBA
; if n_elements(GenInfo) le 1 then GenInfo=replicate(GenConfig,1)
;
end
;-------------------------------------------------------------------------------
pro fixWaves
  ;+
  ; :Description:
  ;    Used to replace incorrect wavelengths data with correct numbers. 
  ; Helpful for use when an update is needed because the entries in 
  ;     
  ;    genconfig.wavelength[*,*]
  ;    
  ; do not match those measured by FTS scans. 
  ; 
  ; :How to use this procedure, you ask?
  ; 
  ;   1) Start OYSTER
  ;   2) Data -> INTERFEROMETRY 
  ;              load file that needs corrected wavelengths
  ;   3) OYSTER> .r /Path/to/This/pro/file/fixWaves
  ;   4) OYSTER> fixWaves
  ;   5) Access -> Write -> HDS
  ;              writes the corrected chameleon file to disk
  ;              the changes in genconfig.wavelength
  ;              
  ; :If you want to see that the changes took hold, print the 
  ;  wavelengths to the screen:
  ;      
  ;      OYSTER> for i=0,15 DO print,genconfig.wavelength[i,0],' ',genconfig.wavelength[i,1]
  ; 
  ; :Author: bob.zavala
  ;-

  print,'**********'
  print,' Beginnning procedure fixWaves '
  print,'**********'
  

  ; Some OYSTER common statements
  COMMON SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
  COMMON ScanData,scans

  ; I may only need to fix one spectrometer. Or two. Thus I will makde to 
  ; arrays, one for each possible hybrid combiner output beam. Then I will fix 
  ; the required the number of spectrometers
  
  new_waves_ob1 = DBLARR(16)
  new_waves_ob2 = DBLARR(16)
  
  ; Make default entries for each array. For this I will use the wavelengths 
  ; needed to change the 2013-May NPOI observations of Beta Herculis from H-alpha 
  ; to Imaging and use measured results from 2013-05-19-025803Z.S1S2.30 FTS
  
  new_waves_ob1 = [8.454e-07,8.174e-07,7.907e-07,7.659e-07,7.427e-09,7.207e-09,7.017e-09, $ 
                   6.826e-07,6.649e-07,6.482e-07,6.179e-07,6.041e-07,5.910e-07,5.790e-07, $
                   5.679e-07,5.570e-07]

  new_waves_ob2 = [8.461e-07,8.180e-07,7.914e-07,7.669e-07,7.439e-09,7.221e-09,7.030e-09, $
                   6.837e-07,6.659e-07,6.489e-07,6.187e-07,6.048e-07,5.917e-07,5.796e-07, $
                   5.686e-07,5.577e-07]
                   
  new_waves = [[new_waves_ob1],[new_waves_ob2]]

  ; set the output beams to be repaired using the index numbers NOT the output 
  ; beam numbers
  fix_output_beams = [0,1]
  
  ; Select the chameleon files to fix
  ; Save this for later
  ; files_to_fix_waves = DIALOG_PICKFILE(/READ, FILTER='*.cha')
  
  ; Tell the user we are ready to fix those wavelegths
  
  print,' '
  print,' I will now fix those wavelegths.'
  print,' '
  
  ; Loop through these files
  ;FOR i=0,n_elements(files_to_fix_waves) - 1 DO BEGIN
   ;   get_data,files_to_fix_waves[i]
  
      ; Loop over the output beams for which we need to fix the wavelengths 
      FOR i=0,1 DO BEGIN
        genconfig.wavelength[*,i] = new_waves[*,i]
      ENDFOR  
  
  ;ENDFOR
  
  print,' '
  print,' Wavelengths should be fixed, for ',genconfig.date,'. Here, take a look.'
  print,' '
  print,'     Spec ',genconfig.spectrometerid[0],'           Spec ',genconfig.spectrometerid[1]
  for i=0,15 DO print,genconfig.wavelength[i,0],' ',genconfig.wavelength[i,1]
  print,' '
  
  ; Halfway there :)
  
end
;-------------------------------------------------------------------------------
pro get_metroconfig,no_alloc=no_alloc
;
; Get metrology configuration.
;
; Note the conversion factor for motor angles in the sid model:
; angle [deg] = motor counts * 360e-8.
;
; Note that mudroom hydra channel 6 measures the air temperature, whereas
; the remaining channels 1-5,7-10 correspond to the probes measuring the
; temperatures of various solid parts, and have the corresponding labels
; as listed below.
;
; Siderostat model parameters are sent in the SidModel packet for each
; sidData.N file. The N extension specifies the SidCon ID. The models
; are stored in this order so that OYSTER has to use the SidConId
; to re-order the data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; If no_alloc=1, this procedure will not re-allocate MetroConfig. For multiple 
; night loading, re-allocation would cause a structure conflict with MetroInfo.
if n_elements(no_alloc) eq 0 then no_alloc=0 else no_alloc=no_alloc gt 0
;
if hds_state() eq 0 then begin
	print,'***Error(GET_METROCONFIG): no HDS file open!'
	return
endif
toplevel
;
; Check whether this object is available
dat_there,'METROCONFIG',reply
if not reply then begin
;	print,'***Error(GET_METROCONFIG): MetroConfig object not found!'
	return
endif
;
; Get number of siderostats and their SidCon IDs
dat_find,'GENCONFIG'
dat_find,'INPUTBEAM'
cmp_get0i,'NUMSID',NumSid
cmp_get1i,'SIDEROSTATID',NumSid,SiderostatId
dat_there,'SIDCONID',reply
if reply eq 1 then cmp_get1i,'SIDCONID',NumSid,SidConId $
	      else SidConId=SiderostatId
dat_annul
dat_annul
;
; Read format
get_format,format
if format eq 'CHAMELEON' or format eq 'INCHWORM' then max_alloc=1 $
						 else max_alloc=0
;
; Determine number of siderostats for which metrology was configured and rec.
dat_find,'GENCONFIG'
dat_there,'NUMPLATE',reply
if reply then cmp_get0i,'NUMPLATE',NumPlate else NumPlate=NumSid
dat_annul
;
; Allocate MetroConfig structure
if max_alloc then begin
	MaxSid=6 
	MaxPlate=6
endif else begin
	MaxSid=NumSid
	MaxPlate=NumPlate
endelse
if n_elements(MetroConfig) eq 0 or no_alloc eq 0 or max_alloc eq 0 then $
MetroConfig=allocmetroconfig(MaxPlate,MaxSid,9)
;
; If number of plates is not equal to number of siderostats, assume 
; standard 4-way metroconfig and determine index into allocated objects
if NumPlate ne NumSid then begin
	AStationIDs=['AC0','AE0','AW0','AN0']
	if NumPlate ne n_elements(AStationIDs) then begin
		print,'***Error(GET_METROCONFIG): unknown metro configuration!'
		return
	endif
	for i=0,NumPlate-1 do $
		MetroConfig.PlateIdx(i)=where(GenConfig.StationId eq AStationIDs(i))
endif else MetroConfig.PlateIdx=indgen(NumPlate)
pidx=MetroConfig.PlateIdx
;
; Locate main container
dat_find,'METROCONFIG',status & if status ne 0 then return
;
; Read feed beam air sensor configuration
dat_find,'FBAIRTEMPCONF',status 
IF status eq 0 THEN BEGIN
NumSensor=intarr(NumSid)
for i=0,NumPlate-1 do begin
	dat_cell,1,i+1,status
	if status eq 0 then begin
		cmp_get0i,'NUMSENSOR',v
		if pidx(i) ne -1 then NumSensor(pidx(i))=v
		dat_annul
	endif
endfor
dat_annul
MetroConfig.FBAirTemp.NumSensor(0:NumSid-1)=NumSensor
Metroconfig.FBAirTemp.Label(*)='Sid room air, near NAT (below stellar beam)'
ENDIF
;
; Read feed beam air sensor configuration
dat_find,'FBSOLIDTMPCONF',status 
IF status eq 0 THEN BEGIN
NumSensor=intarr(NumSid)
for i=0,NumPlate-1 do begin
	dat_cell,1,i+1,status
	if status eq 0 then begin
		cmp_get0i,'NUMSENSOR',v
		if pidx(i) ne -1 then NumSensor(pidx(i))=v
		dat_annul
	endif
endfor
dat_annul
MetroConfig.FBSolidTmp.NumSensor(0:NumSid-1)=NumSensor
labels=['Feed can, level B (top mirror level plate, INSIDE can)', $
        'Feed can, level C (middle mirror level plate, INSIDE can)', $
        'Feed can, level D (bottom mirror level plate, INSIDE can)', $
        'Feed can exterior, south side', $
        'NAT mount base', $
        'Elevator can top plate, INSIDE can', $
        'Elevator can middle (movable) plate, INSIDE can', $
        'Elevator can  bottom plate, INSIDE can', $
        'Elevator can exterior']
for i=0,NumSid -1 do MetroConfig.FBSolidTmp.Label(*,i)=labels
;
; For a few months, the center hut probes were at different locations
labels=['Feed can, level B (top mirror level plate, INSIDE can)', $
        'Feed can, level C (middle mirror level plate, INSIDE can)', $
        'Feed can, level D (bottom mirror level plate, INSIDE can)', $
        'Feed can exterior, south side', $
        'Center feed can concrete pier', $
        'Elevator can top plate, INSIDE can', $
        'Elevator can middle (movable) plate, INSIDE can', $
        'Elevator can  bottom plate, INSIDE can', $
        'Exterior, bottom seal plate center feed can']
parsedate,date,y,m,d
if julian(y,m,d) ge julian(2000,10,1) and julian(y,m,d) le julian(2001,4,2) then $
	MetroConfig.FBSolidTmp.Label(*,0)=labels
; Here is the N hut for astrometry with the 6-way combiner
labels=['North array center, level B (top    mirror support plate, inside can)', $
	'North array center, level C (middle mirror support plate, inside can)', $
        'North array center, level D (bottom mirror support plate, inside can)', $
        'North array center can EXTERIOR, north side', $
	'North NAT mirror mount base, north side', $
	'North elevator skeleton top plate (inside can)', $
        'North elevator skeleton elevator (movable) plate (inside can)', $
        'North elevator skeleton bottom plate (inside can)', $
        'North elevator can EXTERIOR']
if julian(y,m,d) ge julian(2002,3,10) and NumSid gt 2 then $
MetroConfig.FBSolidTmp.Label(*,3)=labels

ENDIF
;
; Read siderostat model parameters
dat_find,'SIDMODEL',status 
IF status eq 0 THEN BEGIN
FeedBeamAng=dblarr(2,NumSid)
FeedBeamAngErr=dblarr(2,NumSid)
SidAng=dblarr(2,NumSid)
SidAngErr=dblarr(2,NumSid)
ZeroAng=dblarr(2,NumSid)
ZeroAngErr=dblarr(2,NumSid)
AxisOffAng=dblarr(NumSid)
AxisOffAngErr=dblarr(NumSid)
MirrorOffAng=dblarr(NumSid)
MirrorOffAngErr=dblarr(NumSid)
CatsEyeOff=dblarr(NumSid)
CatsEyeOffErr=dblarr(NumSid)
for i=0,NumSid-1 do begin
	j=where(SidConId eq (i+1)) & j=j(0)
	if j eq -1 then j=i
	dat_cell,1,i+1
	dat_there,'FEEDBEAMANG',reply
	if reply then begin
		cmp_get1d,'FEEDBEAMANG',2,v
		FeedBeamAng(*,j)=v
		cmp_get1d,'FEEDBEAMANGERR',2,v
		FeedBeamAngErr(*,j)=v
		cmp_get1d,'SIDANG',2,v
		SidAng(*,j)=v
		cmp_get1d,'SIDANGERR',2,v
		SidAngErr(*,j)=v
		cmp_get1d,'ZEROANG',2,v
		ZeroAng(*,j)=v
		cmp_get1d,'ZEROANGERR',2,v
		ZeroAngErr(*,j)=v
		cmp_get0d,'AXISOFFANG',v
		AxisOffAng(j)=v
		cmp_get0d,'AXISOFFANGERR',v
		AxisOffAngErr(j)=v
		cmp_get0d,'MIRROROFFANG',v
		MirrorOffAng(j)=v
		cmp_get0d,'MIRROROFFANGERR',v
		MirrorOffAngErr(j)=v
		if SidAng(0,j) gt 180 then begin
			SidAng(0,j)=SidAng(0,j)-180
			SidAng(1,j)=180-SidAng(1,j)
			ZeroAng(0,j)=ZeroAng(0,j)+180
		endif
	endif
	dat_annul
endfor
dat_annul
MetroConfig.SidModel.FeedBeamAng(*,0:NumSid-1)=feedbeamang
MetroConfig.SidModel.FeedBeamAngErr(*,0:NumSid-1)=feedbeamangErr
MetroConfig.SidModel.SidAng(*,0:NumSid-1)=sidang
MetroConfig.SidModel.SidAngErr(*,0:NumSid-1)=sidangErr
MetroConfig.SidModel.ZeroAng(*,0:NumSid-1)=zeroang
MetroConfig.SidModel.ZeroAngErr(*,0:NumSid-1)=zeroangErr
MetroConfig.SidModel.AxisOffAng(0:NumSid-1)=axisoffang
MetroConfig.SidModel.AxisOffAngErr(0:NumSid-1)=AxisOffangErr
MetroConfig.SidModel.MirrorOffAng(0:NumSid-1)=Mirroroffang
MetroConfig.SidModel.MirrorOffAngErr(0:NumSid-1)=MirrorOffangErr
MetroConfig.SidModel.CatsEyeOff(0:NumSid-1)=CatsEyeOff
MetroConfig.SidModel.CatsEyeOffErr(0:NumSid-1)=CatsEyeOffErr
ENDIF
;
; Return to first level
toplevel
;
print,'MetroConfig loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_scantable
;
; Create scan table. If no HDS file open, create from scan data.
; Note that since the advent of the 6-way configuration, the number
; of participating (i.e. light contributing) stations may change
; from scan to scan. Therefore, we store in the scantable a logical
; index for each station specifying whether or not this station
; was putting light onto the detector. This information is currently
; (2001/2) not available in the data files, and therefore is added manually
; using the fixconfiguration procedure.
;
common Tables,ScanTable,BGTable,StationTable
common NPOI,npoi_cat,npoi_hip
common Mark3,mark3_baselines,mark3_configs,mark3_cat,mark3_bsc
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if hds_state() eq 0 then begin
	format='AMOEBA'
	NumScan=n_elements(scans)
endif else begin
	toplevel
	get_format,format
	get_numscan,NumScan
endelse
;
if NumScan eq 0 then begin
	print,'***Error(GET_SCANTABLE): no scans!'
	return
endif
MaxSid=9
;
Iscan=lindgen(NumScan)+1	; First scan has ID 1!
NumPoint=lonarr(NumScan)	; Number of data points in a scan
NumCoh=lonarr(NumScan)+1	; Number of 2ms samples in coherent supersample
NumIncoh=lonarr(NumScan)	; Number of supersamples in incoherent int.
Code=lonarr(NumScan)+1		; 1=coherent, 0=incoherent
Station=lonarr(MaxSid,NumScan)+1; 1=station in, 0=station out
;
if format eq 'AMOEBA' then begin
	ScanId=scans.iscan
	StarId=scans.starid
endif else begin
	dat_find,'SCANDATA'
	cmp_get1i,'SCANID',NumScan,ScanId	; ScanId from embedded system
	cmp_get1c,'STARID',NumScan,StarId
	dat_there,'STATION',reply
	if reply eq 1 then begin
		cmp_shape,'STATION',ndim,dims
		cmp_getni,'STATION',ndim,dims,Station
	endif
endelse
;
case format of
'CONSTRICTOR':	begin
	cmp_get1d,'STARTTIME',NumScan,StartTime & StartTime=StartTime/1000
	cmp_get1d,'STOPTIME',NumScan,StopTime & StopTime=StopTime/1000
	cmp_get1i,'NUMPOINT',NumScan,NumPoint
	cmp_get1i,'NUMCOH',NumScan,Numcoh
	cmp_get1i,'NUMINCOH',NumScan,NumIncoh
	cmp_get1i,'CODE',NumScan,Code
	end
'COBRA':	begin
	cmp_get1d,'STARTTIME',NumScan,StartTime & StartTime=StartTime/1000
	cmp_get1d,'STOPTIME',NumScan,StopTime & StopTime=StopTime/1000
	cmp_get1i,'NUMREC',NumScan,NumPoint
	cmp_get1i,'CODE',NumScan,Code
	end
'INCHWORM':	begin
	cmp_get1d,'STARTTIME',NumScan,StartTime & StartTime=StartTime/1000
	cmp_get1d,'STOPTIME',NumScan,StopTime & StopTime=StopTime/1000
	end
'CHAMELEON':	begin
	dat_there,'STARTTIME',reply
	if reply eq 1 then begin
		cmp_get1d,'STARTTIME',NumScan,StartTime
		cmp_get1d,'STOPTIME',NumScan,StopTime
	endif else begin
		cmp_get1d,'SCANTIME',NumScan,ScanTime 
		StartTime=abs(ScanTime)
		StopTime=abs(ScanTime)+30
	endelse
;	index=where(StopTime-StartTime eq 0,count)
;	if count gt 0 then StopTime(index)=StartTime(index)+30
	dat_there,'NUMCOH',reply
	if reply eq 1 then cmp_get1i,'NUMCOH',NumScan,NumCoh
	dat_there,'NUMINCOH',reply
	if reply eq 1 then cmp_get1i,'NUMINCOH',NumScan,NumInCoh
	dat_there,'NUMPOINT',reply
	if reply eq 1 then cmp_get1i,'NUMPOINT',NumScan,NumPoint
	dat_there,'CODE',reply
	if reply eq 1 then cmp_get1i,'CODE',NumScan,Code
	end
'AMOEBA':	begin
	StartTime=scans.time
	StopTime=scans.time
	end
endcase
;
if format ne 'AMOEBA' then dat_annul
;
StarId=fixstarids(StarId)
;
ScanTable=build_scantable(Iscan,ScanId,StarId,Code,Station, $
			StartTime,StopTime,NumPoint, $
			NumCoh,NumIncoh)
;
index=where(scantable.numpoint ne 0,count)
if format eq 'CONSTRICTOR' and count lt NumScan then begin
	print,'***Warning(GET_SCANTABLE): some scans without data (fixed)!'
	scantable=scantable(index)
endif
;
if not !qiet then print,'Scantable created.'
;
end
;-------------------------------------------------------------------------------
pro get_scanlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if hds_state() eq 0 then begin
	print,'***Error(GET_OBSERVERLOG): no HDS file open!'
	return
endif
toplevel
;
name='SCANLOG'
dat_there,name,reply
ScanLog=' '
if reply eq 1 then begin
	cmp_get0c,name,ScanLog
	print,'Scan log read.'
endif else print,'Warning(GET_SCANLOG): ',name,' not found!'
;
end
;-------------------------------------------------------------------------------
pro get_observerlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if hds_state() eq 0 then begin
	print,'***Error(GET_OBSERVERLOG): no HDS file open!'
	return
endif
toplevel
;
name='OBSERVERLOG'
dat_there,name,reply
ObserverLog=' '
if reply eq 1 then begin
	cmp_get0c,name,ObserverLog
	print,'Observer log read.'
endif else print,'Warning(GET_OBSERVERLOG): ',name,' not found!'
;
end
;-------------------------------------------------------------------------------
pro get_constrictorlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if hds_state() eq 0 then begin
	print,'***Error(GET_CONSTRICTORLOG): no HDS file open!'
	return
endif
toplevel
;
name='CONSTRICTORLOG'
dat_there,name,reply
ConstrictorLog=' '
if reply eq 1 then begin
	cmp_get0c,name,ConstrictorLog 
	print,'CONSTRICTOR log read.'
endif else print,'Warning(GET_CONSTRICTORLOG): ',name,' not found!'
;
end
;-------------------------------------------------------------------------------
pro get_oysterlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if hds_state() eq 0 then begin
        print,'***Error(GET_OYSTERLOG): no HDS file open!'
        return
endif
toplevel
;
name='OYSTERLOG'
dat_there,name,reply
InchwormLog=' '
if reply eq 1 then begin
        cmp_get0c,name,OysterLog
        print,'OYSTER log read.'
endif else print,'Warning(GET_OYSTERLOG): ',name,' not found!'
;
end
;-------------------------------------------------------------------------------
pro get_inchwormlog
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if hds_state() eq 0 then begin
	print,'***Error(GET_INCHWORMLOG): no HDS file open!'
	return
endif
toplevel
;
name='INCHWORMLOG'
dat_there,name,reply
InchwormLog=' '
if reply eq 1 then begin
	cmp_get0c,name,InchwormLog
	print,'INCHWORM log read.'
endif else print,'Warning(GET_INCHWORMLOG): ',name,' not found!'
;
end
;-------------------------------------------------------------------------------
pro get_logs
;
get_scanlog
get_observerlog
get_constrictorlog
get_oysterlog
get_inchwormlog
;
end
;-------------------------------------------------------------------------------
pro read_scanlog,file
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
status=dc_read_free(file,log,/col,delim='10b')
if status ne 0 then return
ScanLog=strjoin(log,string(10b))
;
end
;-------------------------------------------------------------------------------
pro read_observerlog,file
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
status=dc_read_free(file,log,/col,delim='10b')
if status ne 0 then return
ObserverLog=strjoin(log,string(10b))
;
end
;************************************************************************Block 3
pro get_points,Iscan_ids
;
; Load PointData for a list of scans. Some data integrity checking is done,
; in particular whether error/rms/uncertainty data is finite or not. The 
; raw data averaging software can produce NaNs if the raw signal is constant
; but non-zero, in which case the square-root formula could get a small but
; negative argument.
;
; Also note that beginning with the 6-way observations at NPOI scan
; configuration (or subbarray as listed in scantable.station) is
; used to flag data involving unused stations.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
	FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
	DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
	NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
	GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
	PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
	ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
	VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
	TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
if hds_state() eq 0 then begin
	print,'***Error(GET_POINTS): no HDS file open!'
	return
endif
toplevel
if checkdata([1,8]) ne 0 then return
;
; Find out about reference station of data
dat_find,'GENCONFIG'
dat_there,'REFSTATION',reply
if reply eq 1 then cmp_get0i,'REFSTATION',refstation else refstation=1
dat_annul
;
; If SidConId exists, NAT data are in the right place
dat_find,'GENCONFIG'
dat_find,'INPUTBEAM'
dat_there,'SIDCONID',reply
if reply eq 1 then use_sidcon=0 else use_sidcon=1
dat_annul
dat_annul
;
; If no scans are specified, load all data
if n_elements(Iscan_ids) eq 0 then Iscan_ids=indgen(n_elements(ScanTable))+1
scan_ids=Iscan_ids-1	; scan_ids used to address arrays, which start at 0!
num_scans=n_elements(scan_ids)	; This is the number of scans to be loaded
;
; Find out about the total number of points: MP
MP=long(total(ScanTable(scan_ids).NumPoint))
if MP eq 0 then begin
	print,'Warning(GET_POINTS): no valid data points found.'
	return
endif
;
; Find out about the maximum number of baselines/outputbeam: MB
MB=max(GenConfig.NumBaseline)
;
; Find out about the maximum number of spectral channels/outputbeam: MC
MC=max(GenConfig.NumSpecChan)
;
; Allocate arrays
; Note the /nozero keyword, which accelerates memory allocation.
; However, this should only be done for variables which are
; always present in the data!
;
PhotonRate=fltarr(GenConfig.NumOutBeam,MC,MP,/nozero)
PhotonRateErr=fltarr(GenConfig.NumOutBeam,MC,MP)-1
VisSq=fltarr(GenConfig.NumOutBeam,MC,MB,MP,/nozero)
VisSqErr=fltarr(GenConfig.NumOutBeam,MC,MB,MP)-1
;
; These arrays exist only for coherently averaged data
ComplexVis=fltarr(GenConfig.NumOutBeam,2,MC,MB,MP)
ComplexVisErr=fltarr(GenConfig.NumOutBeam,2,MC,MB,MP)-1
VisAmp=fltarr(GenConfig.NumOutBeam,MC,MB,MP)
VisAmpErr=fltarr(GenConfig.NumOutBeam,MC,MB,MP)-1
VisPhase=fltarr(GenConfig.NumOutBeam,MC,MB,MP)		
VisPhaseErr=fltarr(GenConfig.NumOutBeam,MC,MB,MP)-1
;
; This array holds the group delay
SoftDelay=fltarr(GenConfig.NumOutBeam,MB,MP)
SoftDelayErr=fltarr(GenConfig.NumOutBeam,MB,MP)-1
;
; Jitter in FDL delay (error not measured)
DelayJitter=fltarr(GenConfig.NumOutBeam,MB,MP)
DelayJitterErr=fltarr(GenConfig.NumOutBeam,MB,MP)+1e-7
; 
FDLPos=dblarr(GenConfig.NumSid,MP)
FDLPosErr=fltarr(GenConfig.NumSid,MP)+1e-6
NATJitter=fltarr(GenConfig.NumSid,MP)
NATJitterErr=fltarr(GenConfig.NumSid,MP)+1e-2
NATCounts=fltarr(GenConfig.NumSid,MP)
NATCountsErr=fltarr(GenConfig.NumSid,MP)+1
;
; Allocate these arrays full-size only if needed
if GenConfig.NumTriple gt 0 then begin
	ComplTriple=fltarr(GenConfig.NumTriple,2,MC,MP,/nozero)
	ComplTripleErr=fltarr(GenConfig.NumTriple,2,MC,MP,/nozero)
	TripleAmp=fltarr(GenConfig.NumTriple,MC,MP,/nozero)
	TripleAmpErr=fltarr(GenConfig.NumTriple,MC,MP,/nozero)
	TriplePhase=fltarr(GenConfig.NumTriple,MC,MP,/nozero)
	TriplePhaseErr=fltarr(GenConfig.NumTriple,MC,MP,/nozero)
endif else begin
;	ComplTriple=fltarr(1,2,MC,MP)
;	ComplTripleErr=fltarr(1,2,MC,MP)
;	TripleAmp=fltarr(1,MC,MP)
;	TripleAmpErr=fltarr(1,MC,MP)
;	TriplePhase=fltarr(1,MC,MP)
;	TriplePhaseErr=fltarr(1,MC,MP)
	ComplTriple=0
	ComplTripleErr=0
	TripleAmp=0
	TripleAmpErr=0
	TriplePhase=0
	TriplePhaseErr=0
endelse
PointTime=dblarr(MP,/nozero)
Iscan=ScanTable(scan_ids).Iscan
StarId=ScanTable(scan_ids).StarId
Rec0=lonarr(num_scans)
RecN=lonarr(num_scans)-1
print,'Allocated arrays for ',num_scans,' scans in ',GenConfig.NumOutBeam, $
	" ob's; MB =",MB,', MC =',MC,', MP =',MP, $
	format='(a,i3,a,i2,a,i2,a,i3,a,i5)'
;
; Get data
vdim=1L
max_items=100L	
num_items=max_items
items=sindgen(num_items)	; This is for a listing of items loaded
for i=0,num_items-1 do items(i)='123456789012345'
complex_read=0
triples_read=0
;
dat_find,'SCANDATA'
dat_find,'POINTDATA'
;
; Before 2000-01-01, CONSTRICTOR would output some illegal values
parsedate,date,y,m,d
if julian(y,m,d) lt julian(2000L,1L,1L) $
	then !except=0 $ ; do not report math errors
	else !except=2   ; report math error immediately
r=machar() & tiny=r.xmin
;
FOR i=0,num_scans-1 DO BEGIN
	vdim=1L
	dat_cell,vdim,scan_ids(i)+1
;
	NumPoint=ScanTable(scan_ids(i)).NumPoint
	RecN(i)=max(RecN)+NumPoint
	Rec0(i)=RecN(i)-NumPoint+1
;
	name='TIME'
	dat_there,name,reply
	if reply eq 1 then begin
	dat_find,name
	size=0L
	dat_size,size
	if size ne NumPoint then begin
		print,'***Error(GET_POINTS): dimensions of '+ $
			name+' inconsistent with declarations!'
		return
	endif
	dat_get1d,NumPoint,v
	PointTime(Rec0(i):RecN(i))=v/1000 	; Convert to seconds
	dat_annul
	endif else print,'Warning(GET_POINTS(',i+1,'): Not found: ',name,'!'
;
        name='INPUTBEAM'
        dat_there,name,reply
        if reply eq 1 then begin
	dat_find,name
	for j=0L,GenConfig.NumSid-1 do begin
	 dat_cell,vdim,j+1
 	 list_names,comp_names
 	 for k=0,n_elements(comp_names)-1 do begin
 	  dat_find,comp_names(k)
	  if strtrim(comp_names(k)) eq 'FDLPOS' then begin
	   dat_size,size
	   icheck=0
	   if size ne NumPoint then begin
		print,'***Error(GET_POINTS): size of FDLPos not = NumPoint!'
		return
	   endif
	   dat_get1d,NumPoint,v
	   FDLPos(j,Rec0(i):RecN(i))=v
	   if scantable(scan_ids(i)).station(j) eq 0 $
	   or scantable(scan_ids(i)).code eq 0 then FDLPosErr(j,Rec0(i):RecN(i))=-1
 	  endif else if strtrim(comp_names(k)) eq 'FDLPOSERR' then begin
	   dat_size,size
	   icheck=0
	   if size ne NumPoint then begin
		print,'***Error(GET_POINTS): size of FDLPosErr not = NumPoint!'
		return
	   endif
	   dat_get1r,NumPoint,v
	   index=where(finite(v) eq 0,count)
	   if count gt 0 then v(index)=-1e-6
	   FDLPosErr(j,Rec0(i):RecN(i))=v
	   if scantable(scan_ids(i)).station(j) eq 0 $
	   or scantable(scan_ids(i)).code eq 0 then FDLPosErr(j,Rec0(i):RecN(i))= $
					       -abs(FDLPosErr(j,Rec0(i):RecN(i)))
 	  endif else if strtrim(comp_names(k)) eq 'NATJITTER' then begin
 	   dat_size,size
 	   if size ne NumPoint then begin
  	    print,'***Error(GET_POINTS): size of NATJitter not = NumPoint!'
  	    return
 	   endif
 	   dat_get1r,NumPoint,v
	   jj=where(j+1 eq genconfig.sidconid(0:genconfig.numsid-1)) & jj=jj(0)
;	   Beginning in 2003, CONSTRICTOR puts the data into the right places
;	   if long(strmid(date,0,4)) ge 2003 then jj=j
	   if not use_sidcon then jj=j
 	   NATJitter(jj,Rec0(i):RecN(i))=v
	   if scantable(scan_ids(i)).station(jj) eq 0 then $
	   NATJitterErr(jj,Rec0(i):RecN(i))=-1e-2
 	  endif else if strtrim(comp_names(k)) eq 'NATCOUNTS' then begin
 	   dat_size,size
 	   if size ne NumPoint then begin
  	    print,'***Error(GET_POINTS): size of NATCounts not = NumPoint!'
  	    return
 	   endif
 	   dat_get1r,NumPoint,v
	   jj=where(j+1 eq genconfig.sidconid(0:genconfig.numsid-1)) & jj=jj(0)
;	   Beginning in 2003, CONSTRICTOR puts the data into the right places
;	   if long(strmid(date,0,4)) ge 2003 then jj=j
	   if not use_sidcon then jj=j
 	   NATCounts(jj,Rec0(i):RecN(i))=v
	   NATCountsErr(jj,Rec0(i):RecN(i))=sqrt(v)
	   if scantable(scan_ids(i)).station(jj) eq 0 then $
	   NATCountsErr(jj,Rec0(i):RecN(i))=-1e-2
	  endif else begin
;	   print,'Warning(GET_POINTS): data type unknown: ',comp_names(k)
	  endelse
	  dat_annul
	 endfor
  	 dat_annul
	endfor
	dat_annul
	endif else print,'Warning(GET_POINTS(',i+1,'): Not found: ',name,'!'
;
        name='OUTPUTBEAM'
        dat_there,name,reply
        if reply eq 1 then begin
	dat_find,name
	for j=0L,GenConfig.NumOutBeam-1 do begin 
;	 Check to make sure at least one baseline is in this beam
	 if GenConfig.NumBaseline(j) eq 0 then continue
	 bindex=intarr(GenConfig.NumBaseline(j))
	 for l=0,n_elements(bindex)-1 do $
	  if scantable(scan_ids(i)).station(where(GenConfig.StationId $
					eq strmid(GenConfig.BaselineId(l,j),0,3))) eq 0 $
	  or scantable(scan_ids(i)).station(where(GenConfig.StationId $
					eq strmid(GenConfig.BaselineId(l,j),4,3))) eq 0 $
	  then bindex(l)=1
         if total(bindex) gt 0 then bindex=where(bindex eq 1) else bindex=-1
	 pflag=0
	 s1=intarr(genconfig.numsid)
	 s2=scantable(scan_ids(i)).station
	 for k=0,genconfig.numsid-1 do $
		s1(k)=strpos(strjoin(genconfig.baselineid(*,j)), $
				genconfig.stationid(k))
	 index=where(s1 ne -1 and s2 ne 0,count)
	 if count eq 0 then pflag=1
	 dat_cell,vdim,j+1
	 list_names,comp_names
	 for k=0,n_elements(comp_names)-1 do begin
  	  dat_find,comp_names(k)
	  if strtrim(comp_names(k)) eq 'VISSQ' then begin
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 3 then icheck=1
	   if dims(0) ne max(GenConfig.NumSpecChan(0:GenConfig.NumOutBeam-1)) then icheck=1
	   if dims(1) ne max(GenConfig.NumBaseline(0:GenConfig.NumOutBeam-1)) then icheck=1
	   if dims(2) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
	    print,'***Error(GET_POINTS): VisSq dimensions '+ $
	  	  'inconsistent with declarations!'
	    return
	   endif
	   dat_getnr,ndim,dims,v
	   VisSq(j,0:dims(0)-1,0:dims(1)-1,Rec0(i):RecN(i))=v
	  endif else if strtrim(comp_names(k)) eq 'VISSQERR' then begin
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 3 then icheck=1
	   if dims(0) ne max(GenConfig.NumSpecChan(0:GenConfig.NumOutBeam-1)) then icheck=1
      	   if dims(1) ne max(GenConfig.NumBaseline(0:GenConfig.NumOutBeam-1)) then icheck=1
	   if dims(2) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
		print,'***Error(GET_POINTS): VisSqErr dimensions '+ $
			'inconsistent with declarations!'
		return
	   endif
	   dat_getnr,ndim,dims,v
	   index=where(finite(v) eq 0,count)
	   if count gt 0 then v(index)=-1
	   index=where(abs(v) lt sqrt(tiny),count)
	   if count gt 0 then v(index)=-1
	   VisSqErr(j,0:dims(0)-1,0:dims(1)-1,Rec0(i):RecN(i))=v
	   if bindex(0) ne -1 then $
	   VisSqErr(j,0:dims(0)-1,bindex     ,Rec0(i):RecN(i))=-1
 	  endif else if strtrim(comp_names(k)) eq 'SOFTDELAY' then begin
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 2 then icheck=1
      	   if dims(0) ne max(GenConfig.NumBaseline($
			   0:GenConfig.NumOutBeam-1)) then icheck=1
	   if dims(1) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
		print,'Warning(GET_POINTS): SoftDelay dimensions '+ $
			'inconsistent with declarations!'
	   endif else begin
	   	dat_getnr,ndim,dims,v
	   	SoftDelay(j,0:dims(0)-1,Rec0(i):RecN(i))=v
	   endelse
 	  endif else if strtrim(comp_names(k)) eq 'SOFTDELAYERR' then begin
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 2 then icheck=1
      	   if dims(0) ne max(GenConfig.NumBaseline($
			   0:GenConfig.NumOutBeam-1)) then icheck=1
	   if dims(1) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
		print,'Warning(GET_POINTS): SoftDelayErr dimensions '+ $
			'inconsistent with declarations!'
	   endif else begin
	   	dat_getnr,ndim,dims,v
	   	index=where(finite(v) eq 0,count)
	   	if count gt 0 then v(index)=-1e-6
	   	index=where(abs(v) lt sqrt(tiny),count)
	   	if count gt 0 then v(index)=-1e-6
	   	SoftDelayErr(j,0:dims(0)-1,Rec0(i):RecN(i))=v
	        if bindex(0) ne -1 then $
	   	SoftDelayErr(j,bindex     ,Rec0(i):RecN(i))=-1e-6
	   endelse
	  endif else if strtrim(comp_names(k)) eq 'PHOTONRATE' then begin
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 2 then icheck=1
	   if dims(0) ne max(GenConfig.NumSpecChan(0:GenConfig.NumOutBeam-1)) then icheck=1
	   if dims(1) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
		print,'***Error(GET_POINTS): PhotonRate dimensions '+ $
			'inconsistent with declarations!'
		return
	   endif
	   dat_getnr,ndim,dims,v
	   PhotonRate(j,0:dims(0)-1,Rec0(i):RecN(i))=v
	  endif else if strtrim(comp_names(k)) eq 'PHOTONRATEERR' then begin
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 2 then icheck=1
	   if dims(0) ne max(GenConfig.NumSpecChan(0:GenConfig.NumOutBeam-1)) then icheck=1
	   if dims(1) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
		print,'***Error(GET_POINTS): PhotonRateErr dimensions '+ $
			'inconsistent with declarations!'
		return
	   endif
	   dat_getnr,ndim,dims,v
	   if pflag then v(*)=-1
	   index=where(finite(v) eq 0,count)
	   if count gt 0 then v(index)=-1
	   index=where(abs(v) lt sqrt(tiny),count)
	   if count gt 0 then v(index)=-1
	   PhotonRateErr(j,0:dims(0)-1,Rec0(i):RecN(i))=v
	  endif else if strtrim(comp_names(k)) eq 'COMPLEXVIS' then begin
	   complex_read=complex_read+1
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 4 then icheck=1
	   if dims(0) ne 2 then icheck=1
	   if dims(1) ne max(GenConfig.NumSpecChan(0:GenConfig.NumOutBeam-1)) then icheck=1
      	   if dims(2) ne max(GenConfig.NumBaseline(0:GenConfig.NumOutBeam-1)) then icheck=1
	   if dims(3) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
		print,'***Error(GET_POINTS): ComplexVis dimensions '+ $
			'inconsistent with declarations!'
		return
	   endif
	   dat_getnr,ndim,dims,v
	   ComplexVis(j,0:dims(0)-1,0:dims(1)-1,0:dims(2)-1,Rec0(i):RecN(i))=v
	  endif else if strtrim(comp_names(k)) eq 'COMPLEXVISERR' then begin
	   complex_read=complex_read+1
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 4 then icheck=1
	   if dims(0) ne 2 then icheck=1
	   if dims(1) ne max(GenConfig.NumSpecChan) then icheck=1
      	   if dims(2) ne max(GenConfig.NumBaseline) then icheck=1
	   if dims(3) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
		print,'***Error(GET_POINTS): ComplexVisErr dimensions '+ $
			'inconsistent with declarations!'
		return
	   endif
	   dat_getnr,ndim,dims,v
	   index=where(finite(v) eq 0,count)
	   if count gt 0 then v(index)=-1
	   index=where(abs(v) lt sqrt(tiny),count)
	   if count gt 0 then v(index)=-1
	   ComplexVisErr(j,0:dims(0)-1,0:dims(1)-1,0:dims(2)-1, $
							Rec0(i):RecN(i))=v
           if bindex(0) ne -1 then $
	   ComplexVisErr(j,0:dims(0)-1,0:dims(1)-1,bindex     , $
							Rec0(i):RecN(i))=-1
 	  endif else if strtrim(comp_names(k)) eq 'DELAYJITTER' then begin
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 2 then icheck=1
      	   if dims(0) ne max(GenConfig.NumBaseline(0:GenConfig.NumOutBeam-1)) then icheck=1
	   if dims(1) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
		print,'***Error(GET_POINTS): DelayJitter dimensions '+ $
			'inconsistent with declarations!'
		return
	   endif
	   dat_getnr,ndim,dims,v
	   DelayJitter(j,0:dims(0)-1,Rec0(i):RecN(i))=v
	   if bindex(0) ne -1 then $
	   DelayJitterErr(j,bindex     ,Rec0(i):RecN(i))=-1e-6
	   if scantable(scan_ids(i)).code eq 0 then $
	   DelayJitterErr(j,*          ,Rec0(i):RecN(i))=-1e-6
	  endif else begin
;	   print,'Warning(GET_POINTS): data type unknown: ',comp_names(k),'!'
	  endelse
	  dat_annul
	 endfor
	 dat_annul
	endfor
	dat_annul
	endif else print,'Warning(GET_POINTS(',i+1,'): Not found: ',name,'!'
;
        if GenConfig.NumTriple gt 0 then begin
        name='TRIPLE'
        dat_there,name,reply
        if reply eq 1 then begin
	dat_find,name
	for j=0L,GenConfig.NumTriple-1 do begin 
	 tflag=0
	 for l=0,2 do $
	  if scantable(scan_ids(i)).station( $
		where(GenConfig.StationId $
	    eq strmid(GenConfig.BaselineId(GenConfig.TripleBase(l,j), $
					   GenConfig.TripleBeam(l,j)),0,3))) eq 0 $
	  or scantable(scan_ids(i)).station( $
		where(GenConfig.StationId $
	    eq strmid(GenConfig.BaselineId(GenConfig.TripleBase(l,j), $
					   GenConfig.TripleBeam(l,j)),4,3))) eq 0 $
	  then tflag=1
	 dat_cell,vdim,j+1
	 list_names,comp_names
	 for k=0,n_elements(comp_names)-1 do begin
  	  dat_find,comp_names(k)
	  if strtrim(comp_names(k)) eq 'COMPLTRIPLE' then begin
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 3 then icheck=1
	   if dims(0) ne 2 then icheck=1
	   if dims(1) ne max(GenConfig.TripleNumChan(0:GenConfig.NumTriple-1)) then icheck=1
	   if dims(2) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
	    print,'***Error(GET_POINTS): ComplTriple dimensions '+ $
	  	  'inconsistent with declarations!'
	    return
	   endif
	   dat_getnr,ndim,dims,v
	   ComplTriple(j,0:dims(0)-1,0:dims(1)-1,Rec0(i):RecN(i))=v
	   triples_read=triples_read+1
	  endif else if strtrim(comp_names(k)) eq 'COMPLTRIPLEERR' then begin
	   dat_shape,ndim,dims
	   icheck=0
	   if ndim ne 3 then icheck=1
	   if dims(0) ne 2 then icheck=1
	   if dims(1) ne max(GenConfig.TripleNumChan(0:GenConfig.NumTriple-1)) then icheck=1
	   if dims(2) ne NumPoint then icheck=1
	   if icheck eq 1 then begin
	    print,'***Error(GET_POINTS): ComplTripleErr dimensions '+ $
	  	  'inconsistent with declarations!'
	    return
	   endif
	   dat_getnr,ndim,dims,v
	   if tflag then v(*)=-1
	   index=where(finite(v) eq 0,count)
	   if count gt 0 then v(index)=-1
	   index=where(abs(v) lt sqrt(tiny),count)
	   if count gt 0 then v(index)=-1
	   ComplTripleErr(j,0:dims(0)-1,0:dims(1)-1,Rec0(i):RecN(i))=v
	   triples_read=triples_read+1
	  endif else begin
;	   print,'***Error(GET_POINTS): data type unknown: ',comp_names(k)
	  endelse
	  dat_annul
	 endfor
	 dat_annul
	endfor
	dat_annul
	endif else print,'Warning(GET_POINTS(',i+1,'): Not found: ',name,'!'
	endif
;
	dat_annul
ENDFOR
;
; Check some variables for finiteness if before 2000-01-01
if !except eq 0 then begin
	index=where(finite(photonrate) eq 0,count)
	if count gt 0 then begin
		photonrate(index)=0
		photonrateerr(index)=-1
	endif
endif
r=check_math()
!except=1
;
; Check uniqueness of time stamps
if n_elements(PointTime) ne n_elements(unique(PointTime)) then begin
	print,'***Error(GET_POINTS): duplicate time stamps!'
	return
endif
;
if max(abs(PointTime)/3600) gt 24 then genconfig.date=nextdate(geoparms.date) $
			          else genconfig.date=geoparms.date
;
; Initialize calibrated delays
dispinit,'point'
;
; Re-reference "pos" data, if necessary
if GenConfig.refstation ne refstation then begin
	NewStation=GenConfig.refstation
	GenConfig.refstation=refstation
	referencestation,GenConfig.StationId(NewStation-1),'point'
endif
;
; Set amplitudes and phases
if complex_read eq num_scans*2*GenConfig.NumOutBeam $
    and complex_read gt 0 then set_visampphase $
else if complex_read gt 0 then $
	print,'***Error(GET_POINTS): fewer complex vis. found than expected!'
;
if triples_read eq num_scans*2*GenConfig.NumTriple $
    and GenConfig.NumTriple gt 0 then set_tripleampphase $
else if GenConfig.NumTriple gt 0 then $
	print,'***Error(GET_POINTS): fewer triple vis. found that expected!'
;
toplevel
calcgeo
;
print,'Data loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_star,star
;
common Tables,ScanTable,BGTable,StationTable
;
if checkdata([1]) ne 0 then return
;
index=where(ScanTable.StarId eq star,count)
if count eq 0 then begin
	print,'***Error(GET_STAR): no match! Check StarId!'
	return
endif
get_points,ScanTable(index).Iscan
print,'Data for star ',star,' loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_pointdata,filename
;
; Read .con file and get auxilliary data. Calls procedures fixconfiguration 
; and flagstations for fixes and updates of tables and configuration before 
; proceeding.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(filename) ne 0 then begin
	hds_open,filename,'READ',status
	if status ne 0 then return
endif
;
get_format,format
if format ne 'CONSTRICTOR' then begin
	print,'***Error(GET_POINTDATA): wrong file format!'
	return
endif
;
get_scantable
get_sysconfig
get_bgdata
fixconfiguration
;
get_startable
get_numscan,n
if n gt 0 then begin
	get_points;,indgen(n)+1
	flagstations
	fixrefstation
endif
get_scanlog
get_observerlog
get_constrictorlog
;
; Default is to clear AMOEBA's data buffer and initialize Info arrays
if n_elements(freememory) eq 0 then freememory=1
if freememory then begin
	freememory
	GenInfo=replicate(GenConfig,1)
	GeoInfo=replicate(GeoParms,1)
endif
;
; Special check for output beams without baselines
index=where(genconfig.numbaseline ne 0,numoutbeam)
genconfig.numoutbeam=numoutbeam
;
end
;-------------------------------------------------------------------------------
pro get_bgtable
;
; Create background scan table.
;
common Tables,ScanTable,BGTable,StationTable
;
if hds_state() eq 0 then begin
	print,'***Error(GET_BGTABLE): no HDS file open!'
	return
endif
toplevel
;
get_numbgscan,numscan
if numscan gt 0 then begin
	Iscan=lindgen(numscan)+1
;
	dat_find,'BGSCANDATA'
	cmp_get1i,'SCANID',numscan,ScanId
	cmp_get1d,'TIME',numscan,Time & Time=Time/1000
	cmp_get1d,'RA',numscan,Ra
	cmp_get1d,'DEC',numscan,Dec
	dat_annul
;
	BGTable=build_bgtable(Iscan,ScanId,Time,Ra,Dec)
;
	print,'Backgroundtable created.'
endif else print,'Warning(GET_BGTABLE): no BG data!'
;
end
;-------------------------------------------------------------------------------
pro get_bgscans
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if hds_state() eq 0 then begin
	print,'***Error(GET_BGSCANS): no HDS file open!'
	return
endif
;
if checkdata([8]) ne 0 then return
;
get_numbgscan,numscan
if numscan gt 0 then begin
; 	Allocate scan items
	MS=max(GenConfig.NumSpecChan)
	s=allocbgscan(GenConfig.NumOutBeam,MS)
	bgscans=replicate(s,numscan)
	time=dblarr(numscan,/nozero)
;
	vdim=1L
	ndim=0L
	maxdim=7L
	dims=lonarr(maxdim)
;
	dat_find,'BGSCANDATA'
	cmp_get1d,'TIME',numscan,time
; 	Convert to seconds
	if numscan gt 1 then bgscans.time=time/1000 $
			else bgscans.time=time(0)/1000
;
	dat_find,'OUTPUTBEAM'
	for j=0L,GenConfig.NumOutBeam-1 do begin
		dat_cell,vdim,j+1
		dat_find,'RATE'
   		dat_shape,ndim,dims
		icheck=0
   		if ndim ne 2 then icheck=1
   		if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
   		if dims(1) ne numscan  then icheck=1
 		if icheck eq 1 then begin
 			print,'***Error(GET_BGSCANS): found Rate dimensions ', $
 		  		'inconsistent with declarations!'
 		   	return
 		endif
		a=fltarr(1,dims(0),dims(1),/nozero)
		dat_getnr,ndim,dims,aa
		a(0,*,*)=aa
		bgscans.Rate(j,0:dims(0)-1)=a
		dat_annul
;		dat_find,'RATEERR'
;		dat_getnr,ndim,dims,aa
;		a(0,*,*)=aa
;		bgscans.RateErr(j,0:dims(0)-1)=a
;		dat_annul
		e=sqrt(abs(a)/1000)
		index=where(a le 0,count)
		if count gt 0 then e(index)=-1
		bgscans.RateErr(j,0:dims(0)-1)=e
		dat_annul
	endfor
	dat_annul
	dat_annul
	print,'Background data loaded.'
endif else print,'Warning(GET_BGSCANS): no BG data!'
;
end
;-------------------------------------------------------------------------------
pro get_bgdata
;
get_bgtable
get_bgscans
;
end
;************************************************************************Block 4
pro get_scans
;
; Read .cha output file of CHAMELEON. The procedure checks the reference 
; station for the data, and re-references the delays if the current reference
; (as listed in GenConfig) is different.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if checkdata([7,8]) ne 0 then return
;
if hds_state() eq 0 then begin
	print,'***Error(GET_SCANS): no HDS file open!'
	return
endif
toplevel
;
get_format,format
if format ne 'CHAMELEON' then begin
	print,'***Error(GET_SCANS): format of file is ',format,'!'
	return
endif
;
; Find out about reference station of data
dat_find,'GENCONFIG'
dat_there,'REFSTATION',reply
if reply eq 1 then cmp_get0i,'REFSTATION',refstation else refstation=1
dat_annul
;
get_numscan,NS
if NS lt 1 then begin
	print,'***Error(GET_SCANS): file must have more than one scan!'
	return
endif
;
; Allocate scans
scans=replicate(scan(),NS)
scans.iscan=indgen(NS)+1
;
dat_find,'SCANDATA'
;
; Get miscellaneous one dimensional vectors
cmp_get1c,'STARID',NS,StarId
scans.starid=fixstarids(StarId)
;
dat_there,'STAR',reply
if reply eq 1 then begin
	cmp_get1c,'STAR',NS,stars
	scans.star=stars
endif
;
cmp_get1d,'SCANTIME',NS,times
scans.time=times
if max(abs(times)/3600) gt 24 and max(abs(times)/3600) lt 48 then begin
	genconfig.date=nextdate(geoparms.date)
	!date_change=1	; To be tested as of 2020-07-27
endif else begin
	genconfig.date=geoparms.date
	!date_change=0	; To be tested as of 2020-07-27
endelse
;
dat_there,'INT_TIME',reply
if reply eq 1 then begin
	cmp_get1d,'INT_TIME',NS,int_times
	scans.int_time=int_times
endif
;
name='R0'
dat_there,name,reply
if reply then begin
	cmp_get1r,name,NS,r0
	scans.r0=r0
endif
if total(scans.r0) eq 0 then scans.r0=1	; Default 1" seeing
;
name='PWV'
dat_there,name,reply
if reply then begin
	cmp_get1r,name,NS,pwv
	scans.pwv=pwv
endif
if total(scans.pwv) eq 0 then scans.pwv=10	; Default 10 mm PW column
;
; Now load OUTPUTBEAM, TRIPLE, INPUTBEAM objects
vdim=1L
max_items=100L	
num_items=max_items
items=sindgen(num_items)	; This is for a listing of items loaded
for i=0,num_items-1 do items(i)='123456789012345'
for i=0,num_items-1 do items(i)='               '
;
dat_find,'OUTPUTBEAM'
for j=0L,GenConfig.NumOutBeam-1 do begin 
	dat_cell,vdim,j+1
	list_names,comp_names
	for k=0,n_elements(comp_names)-1 do begin
		dat_find,comp_names(k)
		if strtrim(comp_names(k)) eq 'VISSQ' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'VisSq dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
				for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).VisSq(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'VISSQERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'VisSqErr dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
				for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).VisSqErr(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'VISSQC' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'VisSqC dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
				for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).VisSqC(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'VISSQCERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'VisSqCErr dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
			    for ic=0,GenConfig.NumSpecChan(j)-1 do $
			    scans(*).VisSqCErr(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'VISPHASE' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'VisPhase dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
				for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).VisPhase(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'VISPHASEERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'VisPhaseErr dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
				for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).VisPhaseErr(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'DIFFPHASE' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'DiffPhase dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
				for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).DiffPhase(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'DIFFPHASEERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'DiffPhaseErr dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
				for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).DiffPhaseErr(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'DIFFPHASEC' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'DiffPhaseC dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
				for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).DiffPhaseC(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'DIFFPHASECERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'DiffPhaseCErr dimensions inconsistent '+ $
					'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do begin
				for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).DiffPhaseCErr(j,ic,ib)=reform(v(ic,ib,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'DELAYJITTER' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				   'DelayJitter dimensions inconsistent '+ $
				   'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do $
			    scans(*).DelayJitter(j,ib)=reform(v(ib,*),NS)
		endif else if strtrim(comp_names(k)) eq 'DELAYJITTERERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				   'DelayJitterErr dimensions inconsistent '+ $
				   'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do $
			    scans(*).DelayJitterErr(j,ib)=reform(v(ib,*),NS)
		endif else if strtrim(comp_names(k)) eq 'TrackRMS' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				   'TrackRMS dimensions inconsistent '+ $
				   'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do $
			    scans(*).TrackRMS(j,ib)=reform(v(ib,*),NS)
		endif else if strtrim(comp_names(k)) eq 'TrackJitter' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne GenConfig.NumBaseline(j) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				   'TrackJitter dimensions inconsistent '+ $
				   'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,GenConfig.NumBaseline(j)-1 do $
			    scans(*).TrackJitter(j,ib)=reform(v(ib,*),NS)
		endif else if strtrim(comp_names(k)) eq 'PHOTONRATE' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'PhotonRate dimensions inconsistent '+ $
					'with declarations!'
					return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).PhotonRate(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'PHOTONRATEERR' then $
									begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'PhotonRateErr dimensions '+ $
					'inconsistent with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).PhotonRateErr(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'BACKGNDRATE' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'BackGndRate dimensions inconsistent '+ $
					'with declarations!'
					return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).BackGndRate(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'BACKGNDERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne GenConfig.NumSpecChan(j) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'BackGndErr dimensions inconsistent '+ $
					'with declarations!'
					return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.NumSpecChan(j)-1 do $
				scans(*).BackGndErr(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'UVW' then begin
			dat_shape,ndim,dims
			dat_getnd,ndim,dims,v
			if ndim eq 4 then begin
				for ib=0,GenConfig.NumBaseline(j)-1 do begin
			    	for ic=0,GenConfig.NumSpecChan(j)-1 do begin
					for l=0,2 do scans(*).UVW(j,ic,ib,l)= $
						reform(v(ic,ib,l,*),NS)
			    	endfor
				endfor
			endif else if ndim eq 3 then begin
				for ib=0,GenConfig.NumBaseline(j)-1 do begin
			    	for ic=0,GenConfig.NumSpecChan(j)-1 do begin
					for l=0,2 do scans(*).UVW(j,ic,ib,l)= $
						reform(v(ib,l,*)/GenConfig.Wavelength(ic,j),NS)
			    	endfor
				endfor
			endif else begin
				print,'***Error(GET_SCANS): found '+ $
					'UVW dimensions '+ $
					'inconsistent with declarations!'
				return
			endelse
		endif else begin
;			print,'Warning(GET_SCANS): data type unknown: ', $
;				comp_names(k),'!'
		endelse
		dat_annul
	endfor
	dat_annul
endfor
dat_annul
;
name='TRIPLE'
dat_there,name,reply
if reply then begin
dat_find,name
for j=0L,GenConfig.NumTriple-1 do begin 
	dat_cell,vdim,j+1
	list_names,comp_names
	for k=0,n_elements(comp_names)-1 do begin
		dat_find,comp_names(k)
		if strtrim(comp_names(k)) eq 'COMPLTRIPLE' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne 2 then icheck=1
			if dims(1) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'ComplTriple dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,1 do begin
			  for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).ComplTriple(j,ib,ic)=reform(v(ib,ic,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'COMPLTRIPLEERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 3 then icheck=1
			if dims(0) ne 2 then icheck=1
			if dims(1) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(2) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'ComplTripleErr dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ib=0,1 do begin
			  for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).ComplTripleErr(j,ib,ic)=reform(v(ib,ic,*),NS)
			endfor
		endif else if strtrim(comp_names(k)) eq 'TRIPLEAMP' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'TripleAmp dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).TripleAmp(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'TRIPLEAMPERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'TripleAmpErr dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).TripleAmpErr(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'TRIPLEPHASE' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'TriplePhase dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).TriplePhase(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'TRIPLEPHASEERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'TriplePhaseErr dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).TriplePhaseErr(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'TRIPLEAMPC' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'TripleAmpC dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).TripleAmpC(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'TRIPLEAMPCERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'TripleAmpCErr dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).TripleAmpCErr(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'TRIPLEPHASEC' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'TriplePhaseC dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).TriplePhaseC(j,ic)=reform(v(ic,*),NS)
		endif else if strtrim(comp_names(k)) eq 'TRIPLEPHASECERR' then begin
			dat_shape,ndim,dims
			icheck=0
			if ndim ne 2 then icheck=1
			if dims(0) ne max(GenConfig.NumSpecChan) then icheck=1
			if dims(1) ne NS then icheck=1
			if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
				    'TriplePhaseCErr dimensions inconsistent '+ $
				    'with declarations!'
				return
			endif
			dat_getnr,ndim,dims,v
			index=where(finite(v) eq 0,count)
			if count gt 0 then begin
				print,'Warning(GETR_SCANS): TriplePhaseCErr not finite! (fixed)'
				v(index)=-1
			endif
			s=1
;			if date eq '2014-04-17' or date eq '2018-08-26' then s=4
			for ic=0,GenConfig.TripleNumChan(j)-1 do $
			  scans(*).TriplePhaseCErr(j,ic)=reform(v(ic,*),NS)/s
		endif else begin
;			print,'Warning(GET_SCANS): data type unknown: ', $
;				comp_names(k),'!'
		endelse
		dat_annul
	endfor
	dat_annul
endfor
dat_annul
; In some cases, phases flip between -180 and +180 degrees, here we correct this
for j=0L,GenConfig.NumTriple-1 do begin 
for k=0,n_elements(scans)-1 do begin
	compltriple=scans(k).compltriple(j,*,*)
	index=where(compltriple(0,0,*) lt 0 and compltriple(0,1,*) lt 0,count)
	if count gt 0 then scans(k).triplephase(j,index)= $
			   scans(k).triplephase(j,index)+2*!pi
endfor
endfor
endif
;
dat_find,'INPUTBEAM'
for j=0L,GenConfig.NumSid-1 do begin 
	dat_cell,vdim,j+1
	list_names,comp_names
	for k=0,n_elements(comp_names)-1 do begin
		dat_find,comp_names(k)
		dat_shape,ndim,dims
		icheck=0
		if dims(ndim-1) ne NS then icheck=1
		if icheck eq 1 then begin
			print,'***Error(GET_SCANS): found '+ $
				comp_names(k)+' dimensions inconsistent '+ $
				'with declarations!'
			return
		endif else begin
		dat_type,type & type=strcompress(type,/remove_all)
		dat_get,type,ndim,dims,v
;		case type of
;			'_REAL'  :begin
;				  dat_get1r,NS,v
;				  end
;			'_DOUBLE':begin
;				  dat_get1d,NS,v
;				  end
;			     else:begin
;				  print,'***Error(GET_SCANS): unknown type!'
;				  return
;				  end
;		endcase
		case strcompress(comp_names(k),/remove_all) of
		'FDLPOS'       :scans(*).FDLPos(j)=v
		'FDLPOSERR'    :scans(*).FDLPosErr(j)=v
		'GRPDELAY'     :scans(*).GrpDelay(j)=v
		'GRPDELAYERR'  :scans(*).GrpDelayErr(j)=v
		'DRYDELAY'     :scans(*).DryDelay(j)=v
		'DRYDELAYERR'  :scans(*).DryDelayErr(j)=v
		'WETDELAY'     :scans(*).WetDelay(j)=v
		'WETDELAYERR'  :scans(*).WetDelayErr(j)=v
		'PARX'	       :scans(*).ParX(j)=v
		'PARXERR'      :scans(*).ParXErr(j)=v
		'PARY'	       :scans(*).ParY(j)=v
		'PARYERR'      :scans(*).ParYErr(j)=v
		'PARZ'	       :scans(*).ParZ(j)=v
		'PARZERR'      :scans(*).ParZErr(j)=v
		'METRODELAY'   :scans(*).MetroDelay(j)=v
		'METRODELAYERR':scans(*).MetroDelayErr(j)=v
		'NATJITTER'    :scans(*).NATJitter(j)=v
		'NATJITTERERR' :scans(*).NATJitterErr(j)=v
		'NATCOUNTS'    :scans(*).NATCounts(j)=v
		'NATCOUNTSERR' :scans(*).NATCountsErr(j)=v
		'NATJITTER2'   :scans(*).NATJitter2(j,*)=reform(v,1,2,NS)
		'PHOTOMETRY'   :begin
				icheck=0
				if ndim ne 3 then icheck=1
				if dims(0) ne GenConfig.NumOutBeam then icheck=1
				if dims(2) ne NS then icheck=1
				if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'Photometry dimensions '+ $
					'inconsistent with declarations!'
				return
				endif
				if n_elements(scans) gt 1 and $
					genconfig.numspecchan(0) gt 1 then begin
				scans(*).Photometry(j,*,*) $
					=reform(v,[1,dims(0:2)])
				endif else if n_elements(scans) eq 1 and $
					genconfig.numspecchan(0) gt 1 then begin
				scans.Photometry(j,*,*) $
					=reform(v,[1,dims(0:1)])
				endif else scans.Photometry(j)=reform(v)
				end
		'PHOTOMETRYERR':begin
				icheck=0
				if ndim ne 3 then icheck=1
				if dims(0) ne GenConfig.NumOutBeam then icheck=1
				if dims(2) ne NS then icheck=1
				if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'PhotometryErr dimensions '+ $
					'inconsistent with declarations!'
				return
				endif
				if n_elements(scans) gt 1 and $
					genconfig.numspecchan(0) gt 1 then begin
				scans(*).PhotometryErr(j,*,*) $
					=reform(v,[1,dims(0:2)])
				endif else if n_elements(scans) eq 1 and $
					genconfig.numspecchan(0) gt 1 then begin
				scans.PhotometryErr(j,*,*) $
					=reform(v,[1,dims(0:1)])
				endif else scans.PhotometryErr(j)=reform(v)
				end
		'PHOTOMETRYC'  :begin
				icheck=0
				if ndim ne 3 then icheck=1
				if dims(0) ne GenConfig.NumOutBeam then icheck=1
				if dims(2) ne NS then icheck=1
				if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'PhotometryC dimensions '+ $
					'inconsistent with declarations!'
				return
				endif
				if n_elements(scans) gt 1 and $
					genconfig.numspecchan(0) gt 1 then begin
				scans(*).PhotometryC(j,*,*) $
					=reform(v,[1,dims(0:2)])
				endif else if n_elements(scans) eq 1 and $
					genconfig.numspecchan(0) gt 1 then begin
				scans.PhotometryC(j,*,*) $
					=reform(v,[1,dims(0:1)])
				endif else scans.PhotometryC(j)=reform(v)
				end
		'PHOTOMETRYCERR':begin
				icheck=0
				if ndim ne 3 then icheck=1
				if dims(0) ne GenConfig.NumOutBeam then icheck=1
				if dims(2) ne NS then icheck=1
				if icheck eq 1 then begin
				print,'***Error(GET_SCANS): found '+ $
					'PhotometryCErr dimensions '+ $
					'inconsistent with declarations!'
				return
				endif
				if n_elements(scans) gt 1 and $
					genconfig.numspecchan(0) gt 1 then begin
				scans(*).PhotometryCErr(j,*,*) $
					=reform(v,[1,dims(0:2)])
				endif else if n_elements(scans) eq 1 and $
					genconfig.numspecchan(0) gt 1 then begin
				scans.PhotometryCErr(j,*,*) $
					=reform(v,[1,dims(0:1)])
				endif else scans.PhotometryCErr(j)=reform(v)
				end
			else   :begin
;			        print,'Warning(GET_SCANS): '+ $
;				'data type unknown: ',comp_names(k),'!'
			        end
		endcase
		endelse
		dat_annul
	endfor
	dat_annul
endfor
toplevel
;
; Re-reference "pos" data, if necessary (should not happen w/Version 8)
if GenConfig.refstation gt GenConfig.NumSid then begin
	print,'Warning(GET_SCANS): reference station changed to ' $
		+string(refstation,format='(i1)')+'!'
	GenConfig.RefStation=refstation
endif
if GenConfig.refstation ne refstation then begin
	NewStation=GenConfig.refstation
	GenConfig.refstation=refstation
	referencestation,GenConfig.StationId(NewStation-1),'scan'
endif
;
; Apply deadtime correction
; deadcorr
;
print,'Scans loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_scandata,filename,freememory=freememory
;
; Read entire .cha file and get auxilliary data. Also clear AMOEBA's buffer.
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common ModelFit,parameters,ds_options
;
if n_elements(filename) ne 0 then begin
	hds_open,filename,'READ',status
	if status ne 0 then return
endif
;
get_format,format
if format ne 'CHAMELEON' then begin
	print,'***Error(GET_SCANDATA): wrong format!'
	return
endif
;
get_scantable
get_sysconfig
get_startable
get_scans
calcastrom
calcviscal
get_scanlog
get_observerlog
get_constrictorlog
;
; We need to transfer this info from scantable to scans for older files
if total(scans.int_time) eq 0 then $
scans.int_time=scantable.stoptime-scantable.starttime
;
; Star names are not stored with StarTable, so get them from scans
index1=where(strlen(startable.name) eq 0,count1)
for i=0,count1-1 do begin
	index2=where(scans.starid eq startable(index1(i)).starid,count2)
	if count2 ge 1 then startable(index1(i)).name=scans(index2(0)).star
endfor
;
; Default is to clear AMOEBA's data buffer and initialize Info arrays
if n_elements(freememory) eq 0 then freememory=1
if freememory then begin
	freememory
	GenInfo=replicate(GenConfig,1)
	GeoInfo=replicate(GeoParms,1)
endif
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
;
; Set the available data types
ds_options.i=1	; True: interferometry loaded
ds_options.v2=1	; True: squared visibilities
ds_options.ta=1	; True: triple amplitudes
ds_options.tp=1	; True: triple phases
;
; If CHA file is not from NPOI, ignore triple amplitudes
if strpos(systemid,'NPOI') lt 0 then ds_options.ta=0
;
end
;-------------------------------------------------------------------------------
pro get_seeingdata,filename
;
; Read scan averaged seeing data from either .con or .cha file.
; Note that delayjitter and natjitter are point data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common SeeingData,scintillation,r0,t0,delayrms,phaserms,delayjitter2,natjitter2
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(filename) ne 0 then begin
	hds_open,filename,'READ',status
	if status ne 0 then return
endif
;
for i=0,genconfig.numoutbeam-1 do begin
	get_object,'SCANDATA.OUTPUTBEAM('+string(i+1)+').SCINTILLATION',si
	get_object,'SCANDATA.OUTPUTBEAM('+string(i+1)+').T0',tn
	get_object,'SCANDATA.OUTPUTBEAM('+string(i+1)+').DELAYRMS',dr
;	get_object,'SCANDATA.OUTPUTBEAM('+string(i+1)+').PHASERMS',pr
	get_object,'SCANDATA.OUTPUTBEAM('+string(i+1)+').DELAYJITTER',dj
	if i eq 0 then begin
		r=size(si)
		if r(0) gt 0 then begin
			r(r(0)+2)=r(r(0)+2)*genconfig.numoutbeam
			r=[r(0),genconfig.numoutbeam,r(1:r(0)+2)]
			r(0)=r(0)+1
			scintillation=make_array(size=r,/float)
		endif

		r=size(tn)
		if r(0) gt 0 then begin
			r(r(0)+2)=r(r(0)+2)*genconfig.numoutbeam
			r=[r(0),genconfig.numoutbeam,r(1:r(0)+2)]
			r(0)=r(0)+1
			t0=make_array(size=r,/float)
		endif
;
		r=size(dr)
		if r(0) gt 0 then begin
			r(r(0)+2)=r(r(0)+2)*genconfig.numoutbeam
			r=[r(0),genconfig.numoutbeam,r(1:r(0)+2)]
			r(0)=r(0)+1
			delayrms=make_array(size=r,/float)
		endif
;
		r=size(pr)
		if r(0) gt 0 then begin
			r(r(0)+2)=r(r(0)+2)*genconfig.numoutbeam
			r=[r(0),genconfig.numoutbeam,r(1:r(0)+2)]
			r(0)=r(0)+1
			phaserms=make_array(size=r,/float)
		endif
;
		r=size(dj)
		if r(0) gt 0 then begin
			r(r(0)+2)=r(r(0)+2)*genconfig.numoutbeam
			r=[r(0),genconfig.numoutbeam,r(1:r(0)+2)]
			r(0)=r(0)+1
			delayjitter2=make_array(size=r,/float)
		endif
	endif
	if n_elements(si) ne 0 then scintillation(i,*,*)=si
	if n_elements(tn) ne 0 then t0(i,*,*,*)=tn
	if n_elements(dr) ne 0 then delayrms(i,*,*)=dr
	if n_elements(pr) ne 0 then phaserms(i,*,*)=pr
	if n_elements(dj) ne 0 then delayjitter2(i,*,*)=dj
endfor
;
for i=0,genconfig.numsid-1 do begin
	get_object,'SCANDATA.INPUTBEAM('+string(i+1)+').NATJITTER',aj
	if i eq 0 then begin
                r=size(aj)
		if r(0) gt 0 then begin
                	r(r(0)+2)=r(r(0)+2)*genconfig.numsid
                	r=[r(0),genconfig.numsid,r(1:r(0)+2)]
                	r(0)=r(0)+1
                	natjitter2=make_array(size=r,/float)
		endif
	endif
	if n_elements(aj) ne 0 then natjitter2(i,*,*)=aj
endfor
;
; These four variables are also part of the scans structure
if n_elements(si) ne 0 then scans.si=scintillation
if n_elements(tn) ne 0 then scans.t0=t0
if n_elements(dr) ne 0 then scans.delayrms=delayrms
if n_elements(pr) ne 0 then scans.phaserms=phaserms
if n_elements(dj) ne 0 then scans.delayjitter2=delayjitter2
if n_elements(aj) ne 0 then scans.natjitter2=natjitter2
;
end
;-------------------------------------------------------------------------------
pro get_numscan,numscan
;
numscan=0L
if hds_state() eq 0 then begin
	print,'***Error(GET_NUMSCAN): no HDS file open!'
	return
endif
toplevel
;
dat_find,'SCANDATA'
cmp_get0i,'NUMSCAN',numscan
dat_annul
;
end
;-------------------------------------------------------------------------------
pro get_numbgscan,numscan
;
numscan=0L
if hds_state() eq 0 then begin
	print,'***Error(GET_NUMBGSCAN): no HDS file open!'
	return
endif
toplevel
;
name='BGSCANDATA'
dat_there,name,reply
if reply eq 1 then begin
	dat_find,name
	cmp_get0i,'NUMBGSCAN',numscan
	dat_annul
endif
;
end
;-------------------------------------------------------------------------------
pro get_nummotiongroup,nummotion
;
nummotion=0L
if hds_state() eq 0 then begin
        print,'***Error(GET_NUMMOTIONGROUP): no HDS file open!'
        return
endif
toplevel
;
name='NUMMOTIONGROUP'
dat_there,name,reply
if reply eq 0 then return
cmp_get0i,name,nummotion
;
end
;-------------------------------------------------------------------------------
pro get_numastromcorgrp,numastrom
;
numastrom=0L
if hds_state() eq 0 then begin
        print,'***Error(GET_NUMASTROMCORGRP): no HDS file open!'
        return
endif
toplevel
;
name='NUMASTROMCORGRP'
dat_there,name,reply
if reply eq 0 then return
cmp_get0i,name,numastrom
;
end
;-------------------------------------------------------------------------------
pro get_data,filename,status
;
; Opens any HDS file and calls the appropriate function to read this file.
; Closes file afterwards.
;
common LocalChaSel,chafiles
;
status=0
if n_elements(filename) ne 0 then begin
	hds_open,filename,'READ',status
	if status ne 0 then return
; 	Make sure amoeba knows about this file
	chafiles=filename
endif
;
get_format,format
;
case format of
'CONSTRICTOR':	get_pointdata
'CHAMELEON':	get_scandata
'INCHWORM':	get_inchdata
'COBRA':	get_rawfile
else:		print,'***Error(GET_DATA): unknown format'
endcase
;
hds_close
;
end
;-------------------------------------------------------------------------------
pro get_object,path,v
;
; Direct loading of primitive object with given path, e.g.
; path='SCANDATA.POINTDATA(1).INPUTBEAM(2).FDLPOS'. 
; Allocation of v is automatic.
;
path=strupcase(path)
;
il=0
i1=0
i2=strpos(path,'.',i1)
while i2 ne -1 do begin
	name=strmid(path,i1,i2-i1)
	j1=strpos(name,'(',0)
	if j1 ne -1 then begin
		dat_find,strmid(name,0,j1),status
		if status ne 0 then goto,cleanup
		il=il+1
		j2=strpos(name,')',0)
		cell=stringparse(strmid(name,j1+1,j2-j1-1))
		vdim=n_elements(cell)
		dat_cell,vdim,cell,status
		if status ne 0 then goto,cleanup
		il=il+1
	endif else begin
		dat_find,name,status
		if status ne 0 then goto,cleanup
		il=il+1
	endelse
	i1=i2+1
	i2=strpos(path,'.',i1)
endwhile
dat_find,strmid(path,i1,strlen(path)-i1),status
if status ne 0 then goto,cleanup
il=il+1
;
dat_prim,reply
if reply ne 1 then begin
	print,'***Error(GET_OBJECT): object not primitive!'
	goto,cleanup
endif
dat_type,type & type=strcompress(type,/remove_all)
dat_shape,ndim,dims
dat_get,type,ndim,dims,v
;
cleanup:
for i=1,il do dat_annul
;
end
;************************************************************************Block 4
pro get_records
;
; Obsolete!
; Read the raw data records from a CONSTRICTOR HDS file.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if hds_state() eq 0 then begin
	print,'***Error(GET_RECORDS): no HDS file open!'
	return
endif
toplevel
;
if checkdata([8]) ne 0 then return
;
numbin=GenConfig.NumBin
numsid=GenConfig.NumSid
numout=GenConfig.NumOutBeam
maxchn=max(GenConfig.NumSpecChan)
maxbas=max(GenConfig.NumBaseline)
numrec=long(total(ScanTable.NumPoint))
;
num_scans=1
dat_find,'SCANDATA'
dat_find,'RAWDATA'
FOR i=0,num_scans-1 DO BEGIN
	vdim=1L
	dat_cell,vdim,i+1
;
	get_object,'Time',TimeStamp
;
	dat_find,'OutputBeam'
	dat_cell,1,1
	dat_there,'SoftDelay',reply_soft
	dat_there,'BinCounts',reply_bins
	dat_annul
	if reply_soft then SoftDelay=fltarr(numout,maxbas,numrec)
	if reply_bins then BinCounts=bytarr(numout,numbin,maxchn,numrec)
	for i=0,numout-1 do begin
		dat_cell,1,i+1
		if reply_bins then begin
			get_object,'BinCounts',c
			BinCounts(i,*,*,*)=c
		endif
		if reply_soft then begin
			get_object,'SoftDelay',d
			SoftDelay(i,*,*)=d
		endif
		dat_annul
	endfor
	dat_annul
;
	LaserPos=dblarr(numsid,numrec)
	dat_find,'InputBeam'
	dat_cell,1,1
	dat_there,'Quad',reply_quad
	dat_there,'GeoDelay',reply_geo
	dat_annul
	if reply_quad then QuadCounts=fltarr(numsid,4,numrec)
	if reply_geo then GeoPos=dblarr(numsid,numrec)
	for i=0,numsid-1 do begin
		dat_cell,1,i+1
		get_object,'FDLDelay',d
		LaserPos(i,*)=d
		if reply_geo then begin
			get_object,'GeoDelay',d
			GeoPos(i,*)=d
		endif
		if reply_quad then begin
			get_object,'Quad',c
			QuadCounts(i,*,*)=c
		endif
		dat_annul
	endfor
ENDFOR
;
toplevel
;
print,'Records loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_rawfile,filename
;
; Obsolete!
; Open a CONSTRICTOR HDS raw data file and read all data.
; These files have names of format YYYY-MM-DD.NNN.NN, where NNN is the scan
; number (starting with 1) and NN is the subscan number (starting with 1). 
;
if n_elements(filename) ne 0 then begin
        hds_open,filename,'READ',status
        if status ne 0 then return
endif
;
get_format,format
if format ne 'COBRA' then begin
	print,'***Error(GET_RAWFILE): wrong file format!'
	return
endif
;
get_scantable
get_sysconfig
get_bgdata
;
get_startable
get_records
;
end
;-------------------------------------------------------------------------------
pro get_rawdata,filename
;
; New standard procedure which will read and concatenate TimeStamp, LaserPos, 
; and Geopos for all sub-scan raw files. These are items with
; time and siderostat indices only.
;
; Big ticket items like BinCounts are read from individual sub-scan files
; using different procedures. This item has a spectrometer (output beam) index.
;
; If filename is defined, it must be of the form YYYY-MM-DD.raw.NNN where NNN
; is the scan number. This will be expanded into all sub-scan file of the
; form YYYY-MM-DD.raw.NNN.NN.
;
; If no filename is provided, assume a sub-scan raw file is open and read
; it as usual. Leave the file open. Also leave the file open if the filename
; equates to just one file.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(filename) eq 0 then begin
	if hds_state() eq 0 then begin
		print,'***Error(GET_RAWDATA): no HDS file open!'
		return
	endif
	toplevel
	nf=1
endif else begin
	files=file_search(filename+'.??')
	nf=n_elements(files)
	if nf eq 0 then begin
		print,'Error(GET_RAWDATA): no files found!'
		return
	endif
	hds_open,files(nf-1),'READ',status
	if status ne 0 then return
endelse
;
get_format,format
if format ne 'COBRA' then begin
	hds_close
	print,'***Error(GET_RAWDATA): wrong file format! (File closed)'
	return
endif
get_scantable
get_sysconfig
fixconfiguration
scantable=replicate(scantable(0),nf)
get_bgdata
;
get_startable
if n_elements(filename) ne 0 then hds_close
;
numrec=0
for i=0,nf-1 do begin
	if n_elements(filename) ne 0 then hds_open,files(i)
	get_object,'scandata.numrec',n
	numrec=numrec+n(0)
	if n_elements(filename) ne 0 then hds_close
endfor
timestamp=dblarr(numrec)
geopos=dblarr(genconfig.numsid,numrec)
laserpos=dblarr(genconfig.numsid,numrec)
quadcounts=intarr(genconfig.numsid,4,numrec)
;
rec0=0
for i=0,nf-1 do begin
	if n_elements(filename) ne 0 then hds_open,files(i)
;
; 	If SidConId exists, NAT data are in the right place
	dat_find,'GENCONFIG'
	dat_find,'INPUTBEAM'
	dat_there,'SIDCONID',reply
	if reply eq 1 then use_sidcon=0 else use_sidcon=1
	dat_annul
	dat_annul
;
	dat_find,'scandata'
	cmp_get1i,'numrec',1,nrec & nrec=nrec(0)
	scantable(i).numpoint=nrec
	dat_find,'rawdata'
	dat_cell,1,1
	cmp_get1d,'time',nrec,time
	timestamp(rec0:rec0+nrec-1)=time
	dat_find,'inputbeam'
	for k=0,genconfig.numsid-1 do begin
	   	kk=where(k+1 eq genconfig.sidconid(0:genconfig.numsid-1)) 
		kk=kk(0)
;	   	Beginning in 2003, CONSTRICTOR puts the data into the right places
;	        if long(strmid(date,0,4)) ge 2003 then kk=k
	        if not use_sidcon then kk=k
		dat_cell,1,k+1
		dat_there,'QUAD',reply
		if reply then begin
			get_object,'Quad',c
			quadcounts(kk,*,rec0:rec0+nrec-1)=c
		endif
		dat_there,'GEOPOS',reply
		if reply then begin
			cmp_get1d,'GEOPOS',nrec,delay
			geopos(k,rec0:rec0+nrec-1)=delay
		endif
		cmp_get1d,'FDLDELAY',nrec,delay
		laserpos(k,rec0:rec0+nrec-1)=delay
		dat_annul
	endfor
	rec0=rec0+nrec
	if n_elements(filename) ne 0 and nf gt 1 then hds_close
endfor
;
; Reset the raw data pointers assuming bincounts are out of sync now
Raw0=0
RawN=0
;
print,'Raw data for star ',scantable.starid,' loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_bincounts,beam
;
; To be used to read BinCounts for a specified output beam from a raw
; data file. Assume time and delay data have already been read using 
; get_rawdata. Therefore, the RawData common block variables hold the
; sub-scan data of one interferometer (i.e. output beam/spectrometer), plus
; time and laser data for the whole night.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
	Raw0,RawN,OutputBeam,BinCounts
;
if n_elements(beam) eq 0 then begin
	print,'Error(GET_BINCOUNTS): no beam specified!'
	return
endif
ob=beam-1
outputbeam=beam
;
if hds_state() eq 0 then begin
	print,'***Error(GET_BINCOUNTS): no HDS file open!'
	return
endif
toplevel
;
if checkdata([8]) ne 0 then return
parsedate,date,y,m,d
jd=julian(y,m,d)
;
numsid=GenConfig.NumSid
numout=GenConfig.NumOutBeam
maxchn=max(GenConfig.NumSpecChan)
maxbas=max(GenConfig.NumBaseline)
numrec=long(total(ScanTable.NumPoint))
;
num_scans=1
dat_find,'SCANDATA'
cmp_get1i,'numrec',1,nrec
dat_find,'RAWDATA'
FOR i=0,num_scans-1 DO BEGIN
	vdim=1L
	dat_cell,vdim,i+1
;
	cmp_get1d,'time',nrec(i),time
	Raw0=where(TimeStamp eq time(0)) & Raw0=Raw0(0) > 0
	RawN=where(TimeStamp eq time(nrec(i)-1)) & RawN=RawN(0)
;
	dat_find,'OutputBeam'
	dat_cell,1,beam
	dat_there,'BinCounts',reply
	if not reply then return
	get_object,'BinCounts',BinCounts
	NumBin=n_elements(BinCounts(*,0))
	if NumBin ne GenConfig.NumBin then $
		BinCounts=add(BinCounts,NumBin/GenConfig.NumBin)
;	Correct channel offsets between 1998-01-29 and 2001-09-24
	if NumBin eq 8 and jd lt julian(2001,9,25) and beam eq 2 then begin
		bincounts(*,8:13,*)=bincounts(*,8:13,*)-4
		bincounts(*,14:15,*)=bincounts(*,14:15,*)-6
	endif
	dat_annul
;
ENDFOR
;
toplevel
; quack
;
print,'BinCounts loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_natcounts,natcounts,Iscan_ids,rec0=rec0,recn=recn
;
; This procedure may serve as a template
; for procedures loading individual HDS object from NPOI data files.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([1,8]) ne 0 then return
;
if hds_state() eq 0 then begin
	print,'***Error(GET_NATCOUNTS): no HDS file open!'
	return
endif
toplevel
;
get_format,format
if format eq 'CHAMELEON' then begin
	print,'***Error(GET_NATCOUNTS): wrong file format!'
	return
endif
;
if n_elements(Iscan_ids) eq 0 then Iscan_ids=indgen(n_elements(ScanTable))+1
scan_ids=Iscan_ids-1	; scan_ids used to address arrays, which start at 0!
num_scans=n_elements(scan_ids)	; This is the number of scans to be loaded
;
; Find out about the total number of points: MP
MP=long(total(ScanTable(scan_ids).NumPoint))
if MP eq 0 then begin
	print,'Warning(GET_NATCOUNTS): no valid data points found.'
	return
endif
;
; Allocate arrays
natcounts=fltarr(genconfig.numsid,MP,/nozero)
;
rec0=intarr(num_scans)
recn=intarr(num_scans)-1
;
; Get data
for i=0,num_scans-1 do begin
	numpoint=scantable(scan_ids(i)).numpoint
	recn(i)=max(recn)+numpoint
	rec0(i)=recn(i)-numpoint+1
for j=0,GenConfig.NumSid-1 do begin
	get_object,'SCANDATA.POINTDATA('+string(scan_ids(i)+1)+').INPUTBEAM('+string(genconfig.sidconid(j))+').NATCOUNTS',c
	NatCounts(j,rec0(i):recn(i))=c
endfor
endfor
toplevel
;
end
;-------------------------------------------------------------------------------
pro quack,point=point
;
; For raw data (default), detect (first) gap larger than 50 s and flag data 
; before or beyond gap depending on which section is smaller.
; For loaded pointdata, flag the last point in each scan.
;
common LocalQuack,i1,i2
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common RawData,TimeStamp,LaserPos,GeoPos,QuadCounts, $
        Raw0,RawN,OutputBeam,BinCounts
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
; if checkdata([1,8,11]) ne 0 then return
;
if n_elements(point) eq 0 then point=0
;
IF not point THEN BEGIN
;
	t=timestamp/1000. & t=t-t(0)
	n=n_elements(t)
	i=where(t(1:n-1)-t(0:n-2) gt 50,count)
	if count gt 0 then begin
		i=i(0) ; Just do first gap
		if i gt n/2 then begin
			i1=0
			i2=i
		endif else begin
			i1=i+1
			i2=n-1
		endelse
		timestamp=timestamp(i1:i2)
		laserpos=laserpos(*,i1:i2)
		geopos=geopos(*,i1:i2)
		quadcounts=quadcounts(*,*,i1:i2)
		if n_elements(bincounts) ne 0 and RawN gt 0 then begin
			if rawn+1 eq n_elements(bincounts(0,0,*)) then begin
				bincounts=bincounts(*,*,i1:i2)
				rawn=i2-i1
			endif
		endif
	endif
	i=n_elements(timestamp)
	if n_elements(bincounts) ne 0 then begin
		if n_elements(bincounts(0,0,*)) gt i then begin
			bincounts=bincounts(*,*,i1:i2)
			rawn=i2-i1
		endif
	endif
;
ENDIF ELSE BEGIN
;
	if n_elements(Rec0) eq 0 then return
;
	NS=n_elements(RecN)
;
	if n_elements(pointflagtable) eq 0 then create_pointflagtable
	index=where(pointflagtable.time ge 0,count)
	id0=count
;
	timestamp=systime()
	reason=timestamp+' '+'QUACK'
;
	item=lonarr(NS)+1
	inbeam=lonarr(NS)
	outbeam=lonarr(NS)
	triple=lonarr(NS)
	chan=lonarr(NS)
	base=lonarr(NS)
	time=abs(PointTime(RecN))
	set_pointflagtable,reason,item,inbeam,outbeam,triple,chan,base,time
	flagpointdata,pointflagtable(id0:id0+NS-1),flag=1
;
ENDELSE
;
print,'Quack completed.'
;
end
;************************************************************************Block 6
pro get_avegroup,avegroup
;
; From INCHWORM output file (.inch), read avegroup. Such a group is 
; characterized by a specific integration time.
;
; Note the conversion factor for motor angles:
; angle [deg] = motor counts * 360e-8.
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if hds_state() eq 0 then begin
        print,'***Error(GET_AVEGROUP): no HDS file open!'
        return
endif
toplevel
;
if n_elements(avegroup) eq 0 then avegroup=1
;
get_format,format
if format ne 'INCHWORM' then begin
	print,'***Error(GET_AVEGROUP): not an INCHWORM file!'
	return
endif
;
if checkdata([1,8]) ne 0 then return
get_datum,datum
;
; If SidConId exists, NAT data are in the right place
dat_find,'GENCONFIG'
dat_find,'INPUTBEAM'
dat_there,'SIDCONID',reply
if reply eq 1 then use_sidcon=0 else use_sidcon=1
dat_annul
dat_annul
;
; Get info on which siderostat corresponds to which plate
dat_find,'GENCONFIG'
cmp_get0i,'NUMPLATE',numplate
vdim=1L
plateid=intarr(numplate)
dat_find,'PLATE'
for i=1,numplate do begin
	dat_cell,vdim,i
	cmp_get0i,'PLATEEMBEDDED',n
	plateid(i-1)=n
	dat_annul
endfor
dat_annul
dat_annul
nsd=max(plateid)
if nsd lt GenConfig.NumSid then begin
	print,'Warning(GET_AVEGROUP): fewer plates than siderostats!'
endif
;
dat_find,'AVEGROUP'
dat_cell,vdim,avegroup
cmp_get0i,'NUMTIME',numtime
cmp_get1d,'TIME',numtime,MetroTime
MetroTime=MetroTime/1000
if nextdate(datum) eq date then MetroTime=MetroTime-86400
if nextdate(date) eq datum then MetroTime=MetroTime+86400
;
MotorAngle=dblarr(2,GenConfig.NumSid,numtime)
MotorAngleErr=dblarr(2,GenConfig.NumSid,numtime)
;
dat_find,'MOTORDATA'
dat_shape,ndim,dims
;
for i=1L,GenConfig.NumSid do begin
k=where(GenConfig.SidConId eq i)
; Beginning in 2003, CONSTRICTOR puts the data into the right places
; if long(strmid(date,0,4)) ge 2003 then k=i-1
if not use_sidcon then k=i-1
for j=1L,2 do begin
	dat_cell,ndim,[i,j]
	dat_there,'DATA',reply
	if reply then begin
		cmp_get1d,'DATA',numtime,v
		MotorAngle(j-1,k,*)=v*360e-8
		cmp_get1d,'DATAERR',numtime,v
		index=where(v eq 0,count)
		if count gt 0 then v(index)=360e8
		MotorAngleErr(j-1,k,*)=v*360e-8
	endif
	index=where(MotorAngle(0,k,*) eq 0 and MotorAngle(1,k,*) eq 0,count)
	if count gt 0 then begin
		MotorAngleErr(0,k,index)=-360e-8
		MotorAngleErr(1,k,index)=-360e-8
	endif
	dat_annul
endfor
endfor
;
numscan=n_elements(ScanTable)
Sol0=lonarr(numscan)
SolN=lonarr(numscan)
Jscan=ScanTable.Iscan
for i=0,numscan-1 do begin
	index=where(MetroTime ge ScanTable(i).StartTime $
		and MetroTime le ScanTable(i).StopTime,count)
	if count gt 0 then begin
		Sol0(i)=index(0)
		SolN(i)=index(count-1)
	endif else begin
		Sol0(i)=-1
		SolN(i)=-1
	endelse
endfor
;
; Initialize other variables for this avegroup
ParX=fltarr(GenConfig.NumSid,numtime)
ParXErr=fltarr(GenConfig.NumSid,numtime)-1
ParY=fltarr(GenConfig.NumSid,numtime)
ParYErr=fltarr(GenConfig.NumSid,numtime)-1
ParZ=fltarr(GenConfig.NumSid,numtime)
ParZErr=fltarr(GenConfig.NumSid,numtime)-1
;
MetroPath=fltarr(GenConfig.NumSid,numtime)
MetroPathErr=fltarr(GenConfig.NumSid,numtime)-1
;
toplevel
print,'Ave group loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_motiongroup,motiongroup
;
; From INCHWORM output file (.inch), read solution #motiongroup.
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if hds_state() eq 0 then begin
        print,'***Error(GET_MOTIONGROUP): no HDS file open!'
        return
endif
toplevel
;
; Check if MetroConfig is present
if checkdata([21]) ne 0 then return
;
get_format,format
if format ne 'INCHWORM' then begin
	print,'***Error(GET_MOTIONGROUP): not an INCHWORM file!'
	return
endif
;
if n_elements(motiongroup) eq 0 then motiongroup=1
;
get_nummotiongroup,n
if n lt motiongroup then begin
	print,'***Error(GET_MOTIONGROUP): only ',n,' groups!'
	return
endif
;
; Load corresponding avegroup first
dat_find,'MOTIONGROUP'
vdim=1L
dat_cell,vdim,motiongroup
cmp_get0i,'AVEGROUP',avegroup
dat_annul
dat_annul
get_avegroup,avegroup
numtime=n_elements(MetroTime)
;
; Get info on which siderostat corresponds to which plate
dat_find,'GENCONFIG'
cmp_get0i,'NUMPLATE',numplate
plateid=intarr(numplate)
dat_find,'PLATE'
vdim=1L
for i=1,numplate do begin
	dat_cell,vdim,i
	cmp_get0i,'PLATEEMBEDDED',n
	plateid(i-1)=n
	dat_annul
endfor
dat_annul
dat_annul
nsd=max(plateid)
if nsd lt GenConfig.NumSid then begin
	print,'Warning(GET_MOTIONGROUP): fewer plates than siderostats!'
endif
;
dat_find,'MOTIONGROUP'
dat_cell,vdim,motiongroup
dat_find,'SIDEROSTAT'
dat_shape,ndim,dims
NumPlate=dims(0)
flag=0
if ndim ne 1 then flag=1
if flag then begin
    print,'***Error(GET_MOTIONGROUP): Object SIDEROSTAT with wrong dimensions'
    toplevel
    return
endif
;
pidx=MetroConfig.PlateIdx
for i=1L,NumPlate do begin
	dat_cell,vdim,i
	dat_there,'PAR',reply
	if reply eq 1 and pidx(plateid(i-1)-1) ne -1 then begin
		cmp_get0i,'NUMPAR',numpar
		ndim=2
		dims(0)=numpar
		dims(1)=numtime
		cmp_getnd,'PAR',ndim,dims,v
		ParX(pidx(plateid(i-1)-1),*)=v(0,*)
		ParY(pidx(plateid(i-1)-1),*)=v(1,*)
		ParZ(pidx(plateid(i-1)-1),*)=v(2,*)
		cmp_getnd,'PARERR',ndim,dims,v
		index=where(v(0,*) eq 0,count)
		if count gt 0 then v(0,index)=-1
		ParXErr(pidx(plateid(i-1)-1),*)=v(0,*)
		index=where(v(1,*) eq 0,count)
		if count gt 0 then v(1,index)=-1
		ParYErr(pidx(plateid(i-1)-1),*)=v(1,*)
		index=where(v(2,*) eq 0,count)
		if count gt 0 then v(2,index)=-1
		ParZErr(pidx(plateid(i-1)-1),*)=v(2,*)
		index=where(MotorAngleErr(0,pidx(plateid(i-1)-1),*) lt 0,count)
		if count gt 0 then begin
			ParXErr(pidx(plateid(i-1)-1),index)=-1
			ParYErr(pidx(plateid(i-1)-1),index)=-1
			ParZErr(pidx(plateid(i-1)-1),index)=-1
		endif
	endif
	dat_annul
endfor
;
toplevel
print,'Motion group loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_astromcorrgroup,motiongroup,astromgroup
;
; From INCHWORM output file (.inch), read delay solution #astromgroup
; computed for #motiongroup. User has to load corresponding motion
; group before!
;
common InchData,Sol0,SolN,Jscan,MetroTime,FeedVector, $
	ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if hds_state() eq 0 then begin
        print,'***Error(GET_ASTROMCORRGROUP): no HDS file open!'
        return
endif
toplevel
;
get_format,format
if format ne 'INCHWORM' then begin
	print,'***Error(GET_ASTROMCORRGROUP): not an INCHWORM file!'
	return
endif
;
if n_elements(motiongroup) eq 0 then motiongroup=1
if n_elements(astromgroup) eq 0 then astromgroup=1
;
get_numastromcorgrp,n
if n lt astromgroup then begin
	print,'***Error(GET_ASTROMCORRGROUP): only ',n,' groups!'
	return
endif
dat_find,'ASTROMCORRGROUP'
astromgroups=intarr(n)
for i=0,n-1 do begin
	dat_cell,1,i+1
	cmp_get0i,'MOTIONGROUP',group
	dat_annul
	if group eq motiongroup then astromgroups(i)=i+1
endfor
index=where(astromgroups ne 0,count)
if count eq 0 then begin
	print,'***Error(GET_ASTROMCORRGROUP): motiongroup not found!'
	return
endif
astromgroups=astromgroups(index)
index=where(astromgroups eq astromgroup,count)
if count eq 0 then begin
	print,'***Error(GET_ASTROMCORRGROUP): astromgroup not found!'
	return
endif
astromgroup0=astromgroups(index(0))
toplevel
;
; Get info on which siderostat corresponds to which plate
dat_find,'GENCONFIG'
cmp_get0i,'NUMPLATE',numplate
vdim=1L
plateid=intarr(numplate)
dat_find,'PLATE'
for i=1,numplate do begin
	dat_cell,vdim,i
	cmp_get0i,'PLATEEMBEDDED',n
	plateid(i-1)=n
	dat_annul
endfor
dat_annul
dat_annul
nsd=max(plateid)
if nsd lt GenConfig.NumSid then begin
	print,'Warning(GET_ASTROMCORRGROUP): fewer plates than siderostats!'
endif
;
dat_find,'ASTROMCORRGROUP'
dat_cell,vdim,astromgroup0
cmp_get0i,'AVEGROUP',avegroup
dat_annul
dat_annul
;
dat_find,'AVEGROUP'
dat_cell,vdim,avegroup
cmp_get0i,'NUMTIME',numtime
cmp_get1d,'TIME',numtime,MetroTime
MetroTime=MetroTime/1000
dat_annul
dat_annul
;
dat_find,'ASTROMCORRGROUP'
dat_cell,vdim,astromgroup
;
dat_find,'DELAY'
dat_shape,ndim,dims
NumPlate=dims(0)
flag=0
if ndim ne 1 then flag=1
if flag then begin
    print,'***Error(GET_ASTROMCORRGRP): Object DELAY with wrong dimensions'
    toplevel
    return
endif
;
Delay=fltarr(GenConfig.NumSid,numtime)
DelayErr=fltarr(GenConfig.NumSid,numtime)-1
;
pidx=MetroConfig.PlateIdx
for i=1L,NumPlate do begin
	dat_cell,vdim,i
	name1='DATA'
	name2='DATAERR'
	dat_there,name1,reply
	if reply eq 1 and pidx(plateid(i-1)-1) ne -1 then begin
		cmp_get1d,name1,numtime,v
		Delay(pidx(plateid(i-1)-1),*)=v
		cmp_get1d,name2,numtime,v
		index=where(v eq 0,count)
		if count gt 0 then v(index)=-1
        	v(*)=1 ; Until Nick has added error bars
		DelayErr(pidx(plateid(i-1)-1),*)=v
	endif
	dat_annul
endfor
dat_annul
;
dat_find,'SIDCTERM'
dat_shape,ndim,dims
NumPlate=dims(0)
flag=0
if ndim ne 1 then flag=1
if flag then begin
    print,'***Error(GET_ASTROMCORGRP): Object SIDCTERM with wrong dimensions'
    toplevel
    return
endif
;
SidCTerm=fltarr(GenConfig.NumSid,numtime)
SidCTermErr=fltarr(GenConfig.NumSid,numtime)-1
;
for i=1L,nsd do begin
	dat_cell,vdim,i
	name1='DATA'
	name2='DATAERR'
	dat_there,name1,reply
	if reply eq 1 and pidx(plateid(i-1)-1) ne -1 then begin
		cmp_get1d,name1,numtime,v
		SidCTerm(pidx(plateid(i-1)-1),*)=v
		cmp_get1d,name2,numtime,v
		index=where(v eq 0,count)
		if count gt 0 then v(index)=-1
        	v(*)=1 ; Until Nick has added error bars
		SidCTermErr(pidx(plateid(i-1)-1),*)=v
	endif
	dat_annul
endfor
;
MetroPath=Delay+SidCTerm
sign=fltarr(nsd,numtime)+1
index=where(DelayErr lt 0 or SidCTermErr lt 0,count)
if count gt 0 then sign(index)=-1
MetroPathErr=sqrt(DelayErr^2+SidCTermErr^2)*sign
;
toplevel
print,'AstromCorr group loaded.'
;
end
;-------------------------------------------------------------------------------
pro get_inchgroup,motiongroup,astromgroup
;
; This procedure loads a motion group and the corresponding astromcorrgroup
; if it has been created with INCHWORM. The group number refers to the
; motiongroup simply because this group represents the primary metrology
; solution for the pixot (x,y,z) coordinates, whereas the astromcorrgroup 
; represents the delay correction based on a specific motiongroup. Both groups
; contain a pointer to a corresponding avegroup which contains the timestamps
; and other data relevant for this integration.
;
if hds_state() eq 0 then begin
        print,'***Error(GET_INCHGROUP): no HDS file open!'
        return
endif
;
if checkdata([1,21]) ne 0 then return
;
if n_elements(motiongroup) eq 0 then motiongroup=1
if n_elements(astromgroup) eq 0 then astromgroup=1
;
get_motiongroup,motiongroup
get_astromcorrgroup,motiongroup,astromgroup
;
end
;-------------------------------------------------------------------------------
pro get_inchdata,filename
;
; This compound procedure opens a .inch file and reads all objects necessary
; for a standalone analysis of these data. Note that the main data that are
; read are the so-called INCHWORM groups, AVE, MOTION, and ASTROMCORR group
; which contain averages either by time (usually 15 s) or scan. More than one
; group can exist and result from different reductions.
;
if n_elements(filename) ne 0 then begin
	hds_open,filename,'READ',status
	if status ne 0 then return
endif
;
get_scantable
get_sysconfig
get_startable
get_inchgroup,1
;
end
;-------------------------------------------------------------------------------
pro get_metrodata
;
; Load metrology data from CONSTRICTOR output. These data are found in the
; METRODATA object both in .inch and .con files. These are raw data.
;
common MetroData,fbairtemp,fbsolidtmp
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([8,21]) ne 0 then return
numsid=genconfig.numsid
;
if hds_state() eq 0 then begin
        print,'***Error(GET_METRODATA): no HDS file open!'
        return
endif
toplevel
;
name='metrodata'
dat_there,name,reply
if not reply then return
dat_find,name
;
; Get FB (feed beam) data
dat_find,'fbairtempdata',status
if status eq 0 then begin
num=intarr(genconfig.numsid)
for i=0,numsid-1 do begin
	dat_cell,1,i+1
	get_object,'numdata',n
	num(i)=n
	dat_annul
endfor
fbairtemp={numdata:num,timestamp:fltarr(max(num),numsid), $
	data:fltarr(max(num),max(metroconfig.fbairtemp.numsensor),numsid), $
	error:fltarr(max(num),max(metroconfig.fbairtemp.numsensor),numsid)+0.1}
for i=0,numsid-1 do begin
        dat_cell,1,i+1
        get_object,'data',d
        get_object,'timestamp',t
        dat_annul
        fbairtemp.data(0:fbairtemp.numdata(i)-1,*,i)=float(d)/100
        fbairtemp.timestamp(0:fbairtemp.numdata(i)-1,i)=float(t)/3600000
endfor
dat_annul
endif
;
dat_find,'fbsolidtmpdata',status
if status eq 0 then begin
num=intarr(genconfig.numsid)
for i=0,numsid-1 do begin
	dat_cell,1,i+1
	get_object,'numdata',n
	num(i)=n
	dat_annul
endfor
fbsolidtmp={numdata:num,timestamp:fltarr(max(num),numsid), $
	data:fltarr(max(num),max(metroconfig.fbsolidtmp.numsensor),numsid), $
	error:fltarr(max(num),max(metroconfig.fbsolidtmp.numsensor),numsid)+0.1}
for i=0,numsid-1 do begin
        dat_cell,1,i+1
        get_object,'data',d
        get_object,'timestamp',t
        dat_annul
        fbsolidtmp.data(0:fbsolidtmp.numdata(i)-1,*,i)=float(transpose(d))/100
        fbsolidtmp.timestamp(0:fbsolidtmp.numdata(i)-1,i)=float(t)/3600000l
endfor
dat_annul
endif
;
toplevel
;
end
;************************************************************************Block 7
pro put_sysconfig
;
; Write all items in the common block SysConfig, i.e. the Date, SystemID,
; GeoParms, and GenConfig into the HDS data file. Compound procedure.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
put_geoparms
put_genconfig
if n_elements(MetroConfig) ne 0 then put_metroconfig
;
end
;-------------------------------------------------------------------------------
pro put_geoparms
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if checkdata([7]) ne 0 then return
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_GEOPARMS): no HDS file currently open!'
	return
endif
toplevel
;
name='GeoParms'
dat_there,name,reply
if reply eq 1 then dat_erase,name
;
type='GEOPARMS'
ndim=0L
dims=lonarr(7)
dat_new,name,type,ndim,dims
dat_find,name
;
	name='Latitude'
	type='_double'
	dat_new,name,type,ndim,dims
	cmp_put0d,name,GeoParms.Latitude
;
	name='Longitude'
	dat_new,name,type,ndim,dims
	cmp_put0d,name,GeoParms.Longitude
;
	name='Altitude'
	dat_new,name,type,ndim,dims
	cmp_put0d,name,GeoParms.Altitude
;
	name='EarthRadius'
	dat_new,name,type,ndim,dims
	cmp_put0d,name,GeoParms.EarthRadius
;
	name='J2'
	dat_new,name,type,ndim,dims
	cmp_put0d,name,GeoParms.J2
;
	name='TAI-UTC'
	dat_new,name,type,ndim,dims
	cmp_put0d,name,GeoParms.TAI_UTC
;
	name='TDT-TAI'
	dat_new,name,type,ndim,dims
	cmp_put0d,name,GeoParms.TDT_TAI
;
	dat_annul
;
end
;-------------------------------------------------------------------------------
pro put_genconfig
;
; Write GenConfig to HDS file. Erase any existing object of same name.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
;
if checkdata([8]) ne 0 then return
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_SYSCONFIG): no HDS file currently open!'
	return
endif
toplevel
;
put_format,'CHAMELEON'
put_date
put_systemid
put_userid
;
name='GenConfig'
dat_there,name,reply
if reply eq 1 then dat_erase,name
;
ndim=0L
dims=lonarr(7)
;
type='GENCONFIG'
dat_new,name,type,ndim,dims
dat_find,name
;
name='InstrCohInt'
type='_double'
dat_new,name,type,ndim,dims
value=GenConfig.InstrCohInt
cmp_put0d,name,value
;
name='BeamCombinerId'
type='_integer'
dat_new,name,type,ndim,dims
value=GenConfig.BeamCombinerId
cmp_put0i,name,value
;
; Added 2006-11-14
name='ConfigId'
len=strlen(GenConfig.ConfigId)
dat_new0c,name,len
cmp_put0c,name,GenConfig.ConfigId
;
name='NumBin'
type='_integer'
dat_new,name,type,ndim,dims
value=GenConfig.NumBin
cmp_put0i,name,value
;
name='Refstation'
type='_integer'
dat_new,name,type,ndim,dims
value=GenConfig.refstation
cmp_put0i,name,value
;
name='InputBeam'
type='TABLE'
dat_new,name,type,ndim,dims
dat_find,name
;
	name='NumSid'
	ndim=0L
	type='_integer'
	value=GenConfig.NumSid
	dat_new,name,type,ndim,dims
	cmp_put0i,name,value
;
	name='SiderostatId'
	ndim=1L
	dims(0)=GenConfig.NumSid
	values=GenConfig.SiderostatId(0:dims(0)-1)
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='SidConId'
	ndim=1L
	dims(0)=GenConfig.NumSid
	values=GenConfig.SidConId(0:dims(0)-1)
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='BCInputId'
	ndim=1L
	dims(0)=GenConfig.NumSid
	values=GenConfig.BCInputId(0:dims(0)-1)
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='StartrackerId'
	ndim=1L
	dims(0)=GenConfig.NumSid
	values=GenConfig.StartrackerId(0:dims(0)-1)
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='StationId'
	ndim=1L
	dims(0)=GenConfig.NumSid
	data=GenConfig.StationId(0:dims(0)-1)
	type='_char*'+strcompress(string(strlen(data(0))),/remove_all)
	dat_new,name,type,ndim,dims
	cmp_put1c,name,data
;
	name='DelayLineId'
	type='_integer'
	ndim=1L
	dims(0)=GenConfig.NumSid
	values=GenConfig.DelayLineId(0:dims(0)-1)
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='StationCoord'
	ndim=2L
	dims(0)=4
	dims(1)=GenConfig.NumSid
	values=GenConfig.StationCoord(0:dims(0)-1,0:dims(1)-1)
	type='_double'
	dat_new,name,type,ndim,dims
	cmp_putnd,name,ndim,dims,values
;
	name='Stroke'
	type='_real'
	ndim=1L
	dims(0)=GenConfig.NumSid
	values=GenConfig.Stroke(0:dims(0)-1)
	dat_new,name,type,ndim,dims
	cmp_put1r,name,values
;
	dat_annul
;
name='OutputBeam'
type='TABLE'
ndim=0L
dat_new,name,type,ndim,dims
dat_find,name
;
	name='NumOutBeam'
	ndim=0L
	value=GenConfig.NumOutBeam
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_put0i,name,value
;
	name='NumBaseline'
	ndim=1L
	dims(0)=GenConfig.NumOutBeam
	values=GenConfig.NumBaseline(0:dims(0)-1)
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='NumBiasFreq'
	ndim=1L
	dims(0)=GenConfig.NumOutBeam
	values=GenConfig.NumBiasFreq(0:dims(0)-1)
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='NumSpecChan'
	ndim=1L
	dims(0)=GenConfig.NumOutBeam
	values=GenConfig.NumSpecChan(0:dims(0)-1)
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='SpectrometerId'
	ndim=1L
	dims(0)=GenConfig.NumOutBeam
	data=GenConfig.SpectrometerId(0:dims(0)-1)
	type='_char*'+strcompress(string(strlen(data(0))),/remove_all)
	dat_new,name,type,ndim,dims
	cmp_put1c,name,data
;
	name='BaselineId'
	ndim=2L
	dims(0)=max(GenConfig.NumBaseline)
	dims(1)=GenConfig.NumOutBeam
	data=GenConfig.BaselineId(0:dims(0)-1,0:dims(1)-1)
	type='_char*'+strcompress(string(strlen(data(0))),/remove_all)
	dat_new,name,type,ndim,dims
	cmp_putnc,name,ndim,dims,data
;
	name='Wavelength'
	ndim=2L
	dims(0)=max(GenConfig.NumSpecChan)
	dims(1)=GenConfig.NumOutBeam
	values=GenConfig.Wavelength(0:dims(0)-1,0:dims(1)-1)
	type='_double'
	dat_new,name,type,ndim,dims
	cmp_putnd,name,ndim,dims,values
	name='WavelengthErr'
	values=GenConfig.WavelengthErr(0:dims(0)-1,0:dims(1)-1)
	dat_new,name,type,ndim,dims
	cmp_putnd,name,ndim,dims,values
	name='ChanWidth'
	values=GenConfig.ChanWidth(0:dims(0)-1,0:dims(1)-1)
	dat_new,name,type,ndim,dims
	cmp_putnd,name,ndim,dims,values
	name='ChanWidthErr'
	values=GenConfig.ChanWidthErr(0:dims(0)-1,0:dims(1)-1)
	dat_new,name,type,ndim,dims
	cmp_putnd,name,ndim,dims,values
;
	name='Bias'
	ndim=5L
	dims(0)=2
	dims(1)=max(GenConfig.NumBaseline)
	dims(2)=max(GenConfig.NumSpecChan)
	dims(3)=GenConfig.NumOutBeam
	maxConfig=system_config(SystemId,'MAXCONFIG')
	case string(maxConfig) of
		'scans':maxConfig=scanconfig(/maxconfig)
		'stars':maxConfig=n_elements(scanconfig(/starlist))
		   else:maxConfig=fix(maxConfig)
	endcase
	dims(4)=maxConfig
	values=double(GenConfig.V2Bias(0:dims(0)-1, $
				       0:dims(1)-1, $
				       0:dims(2)-1, $
				       0:dims(3)-1, $
				       0:dims(4)-1))
	type='_double'
	dat_new,name,type,ndim,dims
	cmp_putnd,name,ndim,dims,values
;
  	name='FringeMod'
	ndim=2L
	dims(0)=max(GenConfig.NumBaseline)
	dims(1)=GenConfig.NumOutBeam
	values=GenConfig.FringeMod(0:dims(0)-1,0:dims(1)-1)
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_putni,name,ndim,dims,values
;
  	name='BiasMod'
	ndim=2L
	dims(0)=max(GenConfig.NumBaseline)
	dims(1)=GenConfig.NumOutBeam
	values=GenConfig.BiasMod(0:dims(0)-1,0:dims(1)-1)
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_putni,name,ndim,dims,values
;
	dat_annul
;
if GenConfig.NumTriple gt 0 then begin
name='Triple'
type='TABLE'
ndim=0L
dat_new,name,type,ndim,dims
dat_find,name
;
	name='NumTriple'
	ndim=0L
	value=GenConfig.NumTriple
	type='_integer'
	dat_new,name,type,ndim,dims
	cmp_put0i,name,value
;
	name='OutputBeam'
	ndim=2L
	dims(0)=3
	dims(1)=GenConfig.NumTriple
	values=GenConfig.TripleBeam(0:dims(0)-1,0:dims(1)-1)
	dat_new,name,type,ndim,dims
	cmp_putni,name,ndim,dims,values
;
	name='Baseline'
	values=GenConfig.TripleBase(0:dims(0)-1,0:dims(1)-1)
	dat_new,name,type,ndim,dims
	cmp_putni,name,ndim,dims,values
;
	name='SpecChan'
	ndim=3L
	dims(0)=max(GenConfig.NumSpecChan)
	dims(1)=3
	dims(2)=GenConfig.NumTriple
	values=GenConfig.TripleChan(0:dims(0)-1,0:dims(1)-1,0:dims(2)-1)
	dat_new,name,type,ndim,dims
	cmp_putni,name,ndim,dims,values
;
	name='NumSpecChan'
	ndim=1L
	dims(0)=GenConfig.NumTriple
	values=GenConfig.TripleNumChan(0:dims(0)-1)
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='Bias'
	ndim=4L
	dims(0)=2
	dims(1)=max(GenConfig.NumSpecChan)
	dims(2)=GenConfig.NumTriple
	maxConfig=system_config(SystemId,'MAXCONFIG')
	case string(maxConfig) of
		'scans':maxConfig=scanconfig(/maxconfig)
		'stars':maxConfig=n_elements(scanconfig(/starlist))
		   else:maxConfig=fix(maxConfig)
	endcase
	dims(3)=maxConfig
	values=double(GenConfig.TABias(0:dims(0)-1, $
				       0:dims(1)-1, $
				       0:dims(2)-1, $
				       0:dims(3)-1))
	type='_double'
	dat_new,name,type,ndim,dims
	cmp_putnd,name,ndim,dims,values
;
	dat_annul
;
endif
;
dat_annul
;
end
;-------------------------------------------------------------------------------
pro put_metroconfig
;
; Write MetroConfig to HDS file. Erase any existing object of same name.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
;
if checkdata([21]) ne 0 then return
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_METROCONFIG): no HDS file currently open!'
	return
endif
;
toplevel
;
ndim=0L
dims=lonarr(7)
;
name='MetroConfig'
dat_there,name,reply
if reply eq 1 then dat_erase,name
dat_new,name,name,ndim,dims
dat_find,name
;
name='SidModel'
ndim=1L
dims(0)=GenConfig.NumSid
type='TABLE'
dat_new,name,type,ndim,dims
dat_find,name
;
for i=0,GenConfig.NumSid-1 do begin
;
	ndim=1L
	dat_cell,ndim,i+1
;
	ndim=1L
	dims(0)=2
	type='_double'
;
	name='FeedBeamAng'
	dat_new,name,type,ndim,dims
	values=MetroConfig.SidModel.FeedBeamAng(*,i)
	cmp_put1d,name,values
;
	name='FeedBeamAngErr'
	dat_new,name,type,ndim,dims
	values=MetroConfig.SidModel.FeedBeamAngErr(*,i)
	cmp_put1d,name,values
;
	name='SidAng'
	dat_new,name,type,ndim,dims
	values=MetroConfig.SidModel.SidAng(*,i)
	cmp_put1d,name,values
;
	name='SidAngErr'
	dat_new,name,type,ndim,dims
	values=MetroConfig.SidModel.SidAngErr(*,i)
	cmp_put1d,name,values
;
	name='ZeroAng'
	dat_new,name,type,ndim,dims
	values=MetroConfig.SidModel.ZeroAng(*,i)
	cmp_put1d,name,values
;
	name='ZeroAngErr'
	dat_new,name,type,ndim,dims
	values=MetroConfig.SidModel.ZeroAngErr(*,i)
	cmp_put1d,name,values
;
	ndim=0L
;
	name='AxisOffAng'
	dat_new,name,type,ndim,dims
	value=MetroConfig.SidModel.AxisOffAng(i)
	cmp_put0d,name,value
;
	name='AxisOffAngErr'
	dat_new,name,type,ndim,dims
	value=MetroConfig.SidModel.AxisOffAngErr(i)
	cmp_put0d,name,value
;
	name='MirrorOffAng'
	dat_new,name,type,ndim,dims
	value=MetroConfig.SidModel.MirrorOffAng(i)
	cmp_put0d,name,value
;
	name='MirrorOffAngErr'
	dat_new,name,type,ndim,dims
	value=MetroConfig.SidModel.MirrorOffAngErr(i)
	cmp_put0d,name,value
;
	name='CatsEyeOff'
	dat_new,name,type,ndim,dims
	value=MetroConfig.SidModel.CatsEyeOff(i)
	cmp_put0d,name,value
;
	name='CatsEyeOffErr'
	dat_new,name,type,ndim,dims
	value=MetroConfig.SidModel.CatsEyeOffErr(i)
	cmp_put0d,name,value
;
	dat_annul
;
endfor
;
dat_annul
dat_annul
;
end
;-------------------------------------------------------------------------------
pro put_format,format
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_FORMAT): no HDS file currently open!'
	return
endif
if n_elements(format) eq 0 then begin
	print,'***Error(PUT_FORMAT): no format specified!'
	return
endif
toplevel
name='FORMAT'
dat_there,name,reply
if reply eq 1 then dat_erase,name
len=strlen(format)
dat_new0c,name,len
cmp_put0c,name,format
;
end
;-------------------------------------------------------------------------------
pro put_date,datum
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_DATE): no HDS file currently open!'
	return
endif
if n_elements(datum) eq 0 then datum=Date
toplevel
name='DATE'
dat_there,name,reply
if reply eq 1 then dat_erase,name
len=strlen(datum)
dat_new0c,name,len
cmp_put0c,name,datum
;
end
;-------------------------------------------------------------------------------
pro put_systemid,sys_id
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_SYSTEMID): no HDS file currently open!'
	return
endif
if n_elements(sys_id) eq 0 then sys_id=SystemId
toplevel
name='SYSTEMID'
dat_there,name,reply
if reply eq 1 then dat_erase,name
len=strlen(sys_id)
dat_new0c,name,len
cmp_put0c,name,sys_id
;
end
;-------------------------------------------------------------------------------
pro put_userid,userid
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_USERID): no HDS file open!'
	return
endif
if n_elements(userid) eq 0 then spawn,'whoami',userid
toplevel
name='USERID'
dat_there,name,reply
if reply eq 1 then dat_erase,name
len=strlen(userid)
dat_new0c,name,len
cmp_put0c,name,userid
;
end
;-------------------------------------------------------------------------------
pro put_scanlog,scan_log
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_SCANLOG): no HDS file currently open!'
	return
endif
if n_elements(scan_log) eq 0 then begin
	if n_elements(ScanLog) eq 0 then begin
		print,'Warning(PUT_SCANLOG): no ScanLog present!'
		return
	endif else if strlen(ScanLog) eq 0 then begin
		print,'Warning(PUT_SCANLOG): ScanLog empty!'
		return
	endif
	scan_log=ScanLog
endif
toplevel
name='SCANLOG'
dat_there,name,reply
if reply eq 1 then dat_erase,name
len=strlen(scan_log)
dat_new0c,name,len
cmp_put0c,name,scan_log
;
end
;-------------------------------------------------------------------------------
pro put_observerlog,observer_log
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_OBSERVERLOG): no HDS file currently open!'
	return
endif
if n_elements(observer_log) eq 0 then begin
	if n_elements(ObserverLog) eq 0 then begin
		print,'Warning(PUT_OBSERVERLOG): no ObserverLog present!'
		return
	endif else if strlen(ObserverLog) eq 0 then begin
		print,'Warning(PUT_OBSERVERLOG): ObserverLog empty!'
		return
	endif
	observer_log=ObserverLog
endif
toplevel
name='OBSERVERLOG'
dat_there,name,reply
if reply eq 1 then dat_erase,name
len=strlen(observer_log)
dat_new0c,name,len
cmp_put0c,name,observer_log
;
end
;-------------------------------------------------------------------------------
pro put_constrictorlog,constrictor_log
;
common Logs,ScanLog,ObserverLog,ConstrictorLog,OysterLog,InchwormLog
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_CONSTRICTORLOG): no HDS file currently open!'
	return
endif
if n_elements(constrictor_log) eq 0 then begin
	if n_elements(ConstrictorLog) eq 0 then begin
		print,'Warning(PUT_CONSTRICTORLOG): no Log present!'
		return
	endif else if strlen(ConstrictorLog) eq 0 then begin
		print,'Warning(PUT_CONSTRICTORLOG): ConstrictorLog empty!'
		return
	endif
	constrictor_log=ConstrictorLog
endif
toplevel
name='CONSTRICTORLOG'
dat_there,name,reply
if reply eq 1 then dat_erase,name
len=strlen(constrictor_log)
dat_new0c,name,len
cmp_put0c,name,constrictor_log
;
end
;-------------------------------------------------------------------------------
pro put_scandata,file
;
; Write scan data and related tables to file. File is updated if it already
; exists, otherwise it will be created.
;
if hds_state() eq 0 then begin
	if n_elements(file) eq 0 then begin
		print,'***Error(PUT_SCANDATA): file not specified!'
		return
	endif
	result=file_search(file,count=fcount)
	if fcount ne 0 then hds_open,file,'UPDATE',status else $
        	hds_new,file,'DataSet','CHAMELEON',status
	if status ne 0 then begin
        	print,'***Error(PUT_SCANDATA): file problem!'
        	clearstatus
        	return
	endif
endif else begin
	print,'***Error(PUT_SCANDATA): please close the current file first!'
	return
endelse
;
put_scanlog
put_observerlog
put_constrictorlog
put_sysconfig
put_scans
;
hds_close
;
end
;-------------------------------------------------------------------------------
pro put_scans
;
; Write averaged scandata to HDS file. Erases any object with same name.
; HDS file must have been opened before calling this procedure.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if hds_state() eq 0 then begin
	print,'***Error(PUT_SCANDATA): no HDS file currently open!'
	return
endif
toplevel
;
; For astrometry, remove corrections from delays first
whitecorr,/remove
pivotcorr,/remove
;
; We need to remove the dead time correction, but leave the current data intact
scans_bck=scans
; deadcorr,/remove
;
name='SCANDATA'
dat_there,name,reply
if reply eq 1 then dat_erase,name
type='TABLE'
ndim=0L
dims=lonarr(7)
dat_new,name,type,ndim,dims
dat_find,name
	name='NumScan'
	type='_integer'
	ndim=0L
	dat_new,name,type,ndim,dims
	value=n_elements(scans)
	cmp_put0i,name,value
;
	name='ScanId'
	type='_integer'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	dat_new,name,type,ndim,dims
	values=ScanTable.ScanId
	cmp_put1i,name,values
;
	name='Code'
	type='_integer'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	dat_new,name,type,ndim,dims
	values=ScanTable.Code
	cmp_put1i,name,values
;
	name='Station'
	type='_integer'
	ndim=2L
	dims(0)=GenConfig.NumSid
	dims(1)=n_elements(ScanTable)
	dat_new,name,type,ndim,dims
	values=ScanTable.Station(0:GenConfig.NumSid-1)
	cmp_putni,name,ndim,dims,values
;
	name='StarId'
	data=ScanTable.StarId
	type='_char*'+strcompress(string(max(strlen(data))),/remove_all)
	ndim=1L
	dims(0)=n_elements(ScanTable)
	dat_new,name,type,ndim,dims
	cmp_put1c,name,data
;
	name='Star'
	data=scans.Star
	type='_char*'+strcompress(string((max(strlen(data))>1)),/remove_all)
	ndim=1L
	dims(0)=n_elements(ScanTable)
	dat_new,name,type,ndim,dims
	cmp_put1c,name,data
;
	name='RA'
	type='_double'
	ndim=1L
	dims(0)=n_elements(scans)
	values=scans.RA
	dat_new,name,type,ndim,dims
	cmp_put1d,name,values
;
	name='Dec'
	type='_double'
	ndim=1L
	dims(0)=n_elements(scans)
	values=scans.Dec
	dat_new,name,type,ndim,dims
	cmp_put1d,name,values
;
	name='ScanTime'
	type='_double'
	ndim=1L
	dims(0)=n_elements(scans)
	values=scans.Time
	dat_new,name,type,ndim,dims
	cmp_put1d,name,values
;
	name='StartTime'
	type='_double'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=ScanTable.StartTime
	dat_new,name,type,ndim,dims
	cmp_put1d,name,values
;
	name='StopTime'
	type='_double'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=ScanTable.StopTime
	dat_new,name,type,ndim,dims
	cmp_put1d,name,values
;
	name='Int_Time'
	type='_double'
	ndim=1L
	dims(0)=n_elements(scans)
	values=scans.Int_Time
	dat_new,name,type,ndim,dims
	cmp_put1d,name,values
;
	name='NumPoint'
	type='_integer'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=ScanTable.NumPoint
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='NumCoh'
	type='_integer'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=ScanTable.NumCoh
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='NumInCoh'
	type='_integer'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=ScanTable.NumInCoh
	dat_new,name,type,ndim,dims
	cmp_put1i,name,values
;
	name='R0'
	type='_real'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=scans.r0
	dat_new,name,type,ndim,dims
	cmp_put1r,name,values
;
	name='PRES'
	type='_real'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=scans.pres
	dat_new,name,type,ndim,dims
	cmp_put1r,name,values
;
	name='RHUM'
	type='_real'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=scans.rhum
	dat_new,name,type,ndim,dims
	cmp_put1r,name,values
;
	name='TEMP'
	type='_real'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=scans.temp
	dat_new,name,type,ndim,dims
	cmp_put1r,name,values
;
	name='PWV'
	type='_real'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=scans.pwv
	dat_new,name,type,ndim,dims
	cmp_put1r,name,values
;
	name='ZA'
	type='_double'
	ndim=1L
	dims(0)=n_elements(ScanTable)
	values=scans.za
	dat_new,name,type,ndim,dims
	cmp_put1d,name,values
;
	name='OutputBeam'
	type='EXTCOLUMN'
	ndim=1L
	dims(0)=GenConfig.NumOutBeam
	dat_new,name,type,ndim,dims
	dat_find,name
		vdim=1L
		for i=0L,GenConfig.NumOutBeam-1 do begin
			dat_cell,vdim,i+1
			type='_real'
			ndim=3L
			dims(0)=GenConfig.NumSpecChan(i)
			dims(1)=GenConfig.NumBaseline(i)
			dims(2)=n_elements(scans)
			name='VisSq'
			data=scans.VisSq(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='VisSqErr'
			data=scans.VisSqErr(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='VisSqC'
			data=scans.VisSqC(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='VisSqCErr'
			data=scans.VisSqCErr(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='VisPhase'
			data=scans.VisPhase(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='VisPhaseErr'
			data=scans.VisPhaseErr(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='DiffPhase'
			data=scans.DiffPhase(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='DiffPhaseErr'
			data=scans.DiffPhaseErr(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='DiffPhaseC'
			data=scans.DiffPhaseC(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='DiffPhaseCErr'
			data=scans.DiffPhaseCErr(i,0:dims(0)-1,0:dims(1)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='DelayJitter'
			ndim=2L
			dims(0)=GenConfig.NumBaseline(i)
			dims(1)=n_elements(scans)
			data=scans.DelayJitter(i,0:dims(0)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='DelayJitterErr'
			ndim=2L
			dims(0)=GenConfig.NumBaseline(i)
			dims(1)=n_elements(scans)
			data=scans.DelayJitterErr(i,0:dims(0)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='DelayRMS'
			ndim=2L
			dims(0)=GenConfig.NumBaseline(i)
			dims(1)=n_elements(scans)
			data=scans.DelayRMS(i,0:dims(0)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='DelayJitter2'
			ndim=2L
			dims(0)=GenConfig.NumBaseline(i)
			dims(1)=n_elements(scans)
			data=scans.DelayJitter2(i,0:dims(0)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='PhotonRate'
			ndim=2L
			dims(0)=GenConfig.NumSpecChan(i)
			dims(1)=n_elements(scans)
			data=scans.PhotonRate(i,0:dims(0)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='PhotonRateErr'
			data=scans.PhotonRateErr(i,0:dims(0)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='BackGndRate'
			data=scans.BackGndRate(i,0:dims(0)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='BackGndErr'
			data=scans.BackGndErr(i,0:dims(0)-1)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,data
;
			name='UVW'
			type='_double'
			ndim=3L
			dims(0)=GenConfig.NumBaseline(i)
			dims(1)=3
			dims(2)=n_elements(scans)
			data=scans.UVW(i,0,0:dims(0)-1,0:dims(1)-1) $
			    *GenConfig.Wavelength(0,i)
			dat_new,name,type,ndim,dims
			cmp_putnd,name,ndim,dims,data
;
			dat_annul
;
		endfor
		dat_annul
;
	if GenConfig.NumTriple gt 0 then begin
	name='Triple'
	type='EXTCOLUMN'
	ndim=1L
	dims(0)=GenConfig.NumTriple
	dat_new,name,type,ndim,dims
	dat_find,name
	  vdim=1L
	  for i=0L,GenConfig.NumTriple-1 do begin
	    dat_cell,vdim,i+1
	    type='_real'
	    ndim=3L
	    dims(0)=2
	    dims(1)=max(GenConfig.NumSpecChan)
	    dims(2)=n_elements(scans)
	    name='ComplTriple'
	    data=scans(0:dims(2)-1).ComplTriple(i,0:dims(0)-1,0:dims(1)-1)
	    dat_new,name,type,ndim,dims
	    cmp_putnr,name,ndim,dims,data
;
	    name='ComplTripleErr'
	    data=scans(0:dims(2)-1).ComplTripleErr(i,0:dims(0)-1,0:dims(1)-1)
	    dat_new,name,type,ndim,dims
	    cmp_putnr,name,ndim,dims,data
;
	    name='TripleAmp'
	    ndim=2L
	    dims(0)=max(GenConfig.NumSpecChan)
	    dims(1)=n_elements(scans)
	    data=scans(0:dims(1)-1).TripleAmp(i,0:dims(0)-1)
	    dat_new,name,type,ndim,dims
	    cmp_putnr,name,ndim,dims,data
;
	    name='TripleAmpErr'
	    data=scans(0:dims(1)-1).TripleAmpErr(i,0:dims(0)-1)
	    dat_new,name,type,ndim,dims
            cmp_putnr,name,ndim,dims,data
;
	    name='TriplePhase'
	    data=scans(0:dims(1)-1).TriplePhase(i,0:dims(0)-1)
	    dat_new,name,type,ndim,dims
            cmp_putnr,name,ndim,dims,data
;
	    name='TriplePhaseErr'
	    data=scans(0:dims(1)-1).TriplePhaseErr(i,0:dims(0)-1)
	    dat_new,name,type,ndim,dims
            cmp_putnr,name,ndim,dims,data
;
	    name='TripleAmpC'
	    ndim=2L
	    dims(0)=max(GenConfig.NumSpecChan)
	    dims(1)=n_elements(scans)
	    data=scans(0:dims(1)-1).TripleAmpC(i,0:dims(0)-1)
	    dat_new,name,type,ndim,dims
	    cmp_putnr,name,ndim,dims,data
;
	    name='TripleAmpCErr'
	    data=scans(0:dims(1)-1).TripleAmpCErr(i,0:dims(0)-1)
	    dat_new,name,type,ndim,dims
            cmp_putnr,name,ndim,dims,data
;
	    name='TriplePhaseC'
	    data=scans(0:dims(1)-1).TriplePhaseC(i,0:dims(0)-1)
	    dat_new,name,type,ndim,dims
            cmp_putnr,name,ndim,dims,data
;
	    name='TriplePhaseCErr'
	    data=scans(0:dims(1)-1).TriplePhaseCErr(i,0:dims(0)-1)
	    dat_new,name,type,ndim,dims
            cmp_putnr,name,ndim,dims,data
;
	    dat_annul
          endfor
	  dat_annul
	endif
;
	name='InputBeam'
	type='EXTCOLUMN'
	ndim=1L
	dims(0)=GenConfig.NumSid
	dat_new,name,type,ndim,dims
	dat_find,name
		vdim=1L
		for i=0L,GenConfig.NumSid-1 do begin
			dat_cell,vdim,i+1
			name='FDLPos'
			type='_double'
			ndim=1L
			dims(0)=n_elements(scans)
			values=scans.FDLPos(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1d,name,values
;
			name='FDLPosErr'
			type='_real'
			values=scans.FDLPosErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='GrpDelay'
			type='_double'
			dat_new,name,type,ndim,dims
			values=scans.GrpDelay(i,*)
			cmp_put1d,name,values
;
			name='GrpDelayErr'
			type='_real'
			values=scans.GrpDelayErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='DryDelay'
			type='_double'
			values=scans.DryDelay(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1d,name,values
;
			name='DryDelayErr'
			type='_real'
			values=scans.DryDelayErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='WetDelay'
			type='_double'
			values=scans.WetDelay(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1d,name,values
;
			name='WetDelayErr'
			type='_real'
			values=scans.WetDelayErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='ParX'
			type='_real'
			values=scans.ParX(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='ParXErr'
			type='_real'
			values=scans.ParXErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='ParY'
			type='_real'
			values=scans.ParY(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='ParYErr'
			type='_real'
			values=scans.ParYErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='ParZ'
			type='_real'
			values=scans.ParZ(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='ParZErr'
			type='_real'
			values=scans.ParZErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='MetroDelay'
			type='_real'
			values=scans.MetroDelay(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='MetroDelayErr'
			type='_real'
			values=scans.MetroDelayErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='NATJitter'
			type='_real'
			values=scans.NATJitter(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='NATJitterErr'
			type='_real'
			values=scans.NATJitterErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='NATCounts'
			type='_real'
			values=scans.NATCounts(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			name='NATCountsErr'
			type='_real'
			values=scans.NATCountsErr(i,*)
			dat_new,name,type,ndim,dims
			cmp_put1r,name,values
;
			ndim=2L
			dims(0)=2
			dims(1)=n_elements(scans)
			name='NATJitter2'
			type='_real'
			values=scans.NATJitter2(i,*)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,values
;
;			VLTI/MIDI
			ndim=2L
			dims(0)=5
			dims(1)=n_elements(scans)
			name='AcqPSF'
			type='_real'
			values=scans.AcqPSF(i,*)
			dat_new,name,type,ndim,dims
			cmp_putnr,name,ndim,dims,values
;
			ndim=1L
			dims(0)=n_elements(scans)
			name='AcqFlux'
			values=scans.AcqFlux(i)
                        dat_new,name,type,ndim,dims
                        cmp_put1r,name,values
;
			ndim=3L
			dims(0)=genconfig.numoutbeam
			dims(1)=max(genconfig.numspecchan)
			dims(2)=n_elements(scans)
			name='Photometry'
			values=scans.Photometry(i,*,*)
                        dat_new,name,type,ndim,dims
                        cmp_putnr,name,ndim,dims,values
;
			name='PhotometryErr'
			values=scans.PhotometryErr(i,*,*)
                        dat_new,name,type,ndim,dims
                        cmp_putnr,name,ndim,dims,values
;
			name='PhotometryC'
			values=scans.PhotometryC(i,*,*)
                        dat_new,name,type,ndim,dims
                        cmp_putnr,name,ndim,dims,values
;
			name='PhotometryCErr'
			values=scans.PhotometryCErr(i,*,*)
                        dat_new,name,type,ndim,dims
                        cmp_putnr,name,ndim,dims,values

;
			dat_annul
		endfor
		dat_annul
;
toplevel
;
; Reinstate the deadtime corrected data
scans=scans_bck
;
end
;************************************************************************Block 7
pro get_oifits,fitsfile,freememory=freememory,pionierfree=pionierfree, $
	stationfile=station_file,photfile=sed_file, $
	uvcheck=uvcheck,date_obs=date_obs_in
;
forward_function tag_exist
forward_function matisse_zerocounts
;
; Procedure to read interferometric data from a single OI-FITS file.
; Stores data in AMOEBA buffer. Note that strings read from OIFITS have
; trailing blanks for an as yet unknown reason.
;
; Checks presence of file "????-??-??_oiDiam.fits" and reads it to find out
; which stars are calibrators (this file is produced by pndrsCalibrate).
;
; Checks the presence of file seeing.txt and reads it to fill scans.r0.
;
; A spectrum [lambda in microns, flux] stored in sed_file can be supplied.
;
; If freememory is true (1), then free memory before storing new data (default)
; If pionierfree, assign each BL to a new OB for calibration (default=no)
;
; If specified, date_obs sets the date of observation for all data sets.
;
common Hds,path,hds_file_stub
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common StarBase,StarTable,Notes
common ModelFit,parameters,ds_options
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common WwDataSel,opmenul_wid,opmenuw_wid,opmenua_wid,opmenup_wid,opmenuf_wid
common LocalChaSel,chafiles
common LocalFitSel,oiffiles
;
RAD=180/pi_circle
;
; Initialize spectral_cal for PIONEER
common Pionier,pionier_spectral_cal
pionier_spectral_cal='generic_c'
;
; The IDs UU? are inserted to handle the 2-letter U1, U2,... designations.
; They are translated to the designations (UT1,...) by left-shifting indices.
stationids=[ $
	'UT1','UT2','UT3','UT4','UU1','UU2','UU3','UU4', $ $
	'AA0','AA1','AB0','AB1','AB2','AB3','AB4','AB5','AC0','AC1','AC2', $
	'AC3','AD0','AD1','AD2','AE0','AG0','AG1','AG2','AH0','AI1','AI2', $
	'AJ1','AJ2','AJ3','AJ4','AJ5','AJ6','AK0','AL0','AM0']
;
if n_elements(fitsfile) eq 0 then fitsfile='oidata.fits'
if strlen(fitsfile) eq 0 then return
files=file_search(fitsfile)
if strlen(files(0)) eq 0 then begin
	print,'***Error(GET_OIFITS): file not found!'
	return
endif
print,'Reading file: '+fitsfile
;
; Close any open data file
hds_close
;
; Read OI data
read_oidata,fitsfile,oiarray,oitarget,oiwavelength,oivis,oivis2,oit3,oiflux, $
	/inventory
if fitsfile eq 'temp_do_not_keep.fits' then begin
	merged=1 
	if n_elements(oivis) gt 0 then oivis=oivis(sort(oivis.mjd))
	if n_elements(oivis2) gt 0 then oivis2=oivis2(sort(oivis2.mjd))
	if n_elements(oit3) gt 0 then oit3=oit3(sort(oit3.mjd))
	if n_elements(oiflux) gt 0 then oiflux=oiflux(sort(oiflux.mjd))
endif else merged=0
check_oiextns,fitsfile,n_oivis,n_oivis2,n_oit3,n_oiflux
if n_oivis eq 0 and n_oivis2 eq 0 and n_oit3 eq 0 and n_oiflux eq 0 then begin
	print,'No data to load.'
	return
endif
;
; Check OIFITS arrays
if 1 then $
check_oifits,oiarray,oitarget,oiwavelength, $
	oivis1=oivis,oivis2=oivis2,oit3=oit3,oiflux=oiflux
;
; Merge split polarisations (WIP) 
if 0 then begin
n_vis=0
if n_oivis2 gt 0 then n_vis=n_oivis2 else if n_oivis gt 0 then n_vis=n_oivis
if n_vis gt 0 then begin
	if n_oivis2 gt 0 then begin
		if n_elements(oivis2.time)/unique(oivis2.time) eq 2 then begin
		endif

	endif
	if n_oivis gt 0 then begin
	endif
endif
endif
;
; Get some information from header (ESO headers only)
d=mrdfits(fitsfile,0,h)
temp=fitshparse(h,'ISS.AMBI.TEMP',/quiet)
pres=fitshparse(h,'ISS.AMBI.PRES',/quiet)
rhum=fitshparse(h,'ISS.AMBI.RHUM',/quiet)
seeing=(fitshparse(h,'ISS.AMBI.FWHM.START',/quiet) $
       +fitshparse(h,'ISS.AMBI.FWHM.END',/quiet))/2.
taunot=(fitshparse(h,'ISS.AMBI.TAU0.START',/quiet) $
       +fitshparse(h,'ISS.AMBI.TAU0.END',/quiet))/2.
pwv=(fitshparse(h,'ISS.AMBI.IWV.START',/quiet) $
    +fitshparse(h,'ISS.AMBI.IWV.END',/quiet))/2.
;
; OYSTER's UT Date has only a single value, hence times are wrt to UT=0
; Check night with observations before UT=0 => date+1d and times-1d (neg.)
systemid=strtrim(oiarray(0).arrname,2)
night_start=24-system_config(systemid(0),'MIDNIGHT'); UT of when night begins
!date_change=0
n2=0
if n_elements(oivis2) ne 0 then begin
	oivis2.date_obs=strmid(oivis2.date_obs,0,10)
	oivis2.time=(oivis2.mjd mod 1)*86400
	idx2=where((oivis2.mjd-long(oivis2.mjd))*24 gt night_start,n2)
	if n2 gt 0 then begin
		!date_change=1
		date_obs=strmid(oivis2(idx2).date_obs,0,10)
		oivis2(idx2).date_obs=nextdate(date_obs)
		oivis2(idx2).time=oivis2(idx2).time-86400
	endif
	if keyword_set(date_obs_in) then oivis2.date_obs=date_obs_in
endif
if n_elements(oivis) ne 0 then begin
	oivis.date_obs=strmid(oivis.date_obs,0,10)
	oivis.time=(oivis.mjd mod 1)*86400
	idxv=where((oivis.mjd-long(oivis.mjd))*24 gt night_start,nv)
	if nv gt 0 then begin
		!date_change=1
		date_obs=strmid(oivis(idxv).date_obs,0,10)
		oivis(idxv).date_obs=nextdate(date_obs)
		oivis(idxv).time=oivis(idxv).time-86400
	endif
	if keyword_set(date_obs_in) then oivis.date_obs=date_obs_in
endif
if n_elements(oit3) ne 0 then begin 
	oit3.date_obs=strmid(oit3.date_obs,0,10)
	oit3.time=(oit3.mjd mod 1)*86400
	idx3=where((oit3.mjd-long(oit3.mjd))*24 gt night_start,n3)
	if n3 gt 0 then begin
		!date_change=1
		date_obs=strmid(oit3(idx3).date_obs,0,10)
		oit3(idx3).date_obs=nextdate(date_obs)
		oit3(idx3).time=oit3(idx3).time-86400
	endif
	if keyword_set(date_obs_in) then oit3.date_obs=date_obs_in
endif
if n_elements(oiflux) ne 0 then begin 
	oiflux.date_obs=strmid(oiflux.date_obs,0,10)
	oiflux.time=(oiflux.mjd mod 1)*86400
	idxf=where((oiflux.mjd-long(oiflux.mjd))*24 gt night_start,nf)
	if nf gt 0 then begin
		!date_change=1
		date_obs=strmid(oiflux(idxf).date_obs,0,10)
		oiflux(idxf).date_obs=nextdate(date_obs)
		oiflux(idxf).time=oiflux(idxf).time-86400
	endif
	if keyword_set(date_obs_in) then oiflux.date_obs=date_obs_in
endif
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
ds_options.i=1  ; True: interferometry loaded
;
; Assemble startable
if n_elements(oitarget) eq 0 then begin
	starid=''
	read,starid, $
	prompt='Target table not present, please enter star ID [HDN123456]: ', $
	get_startable,starid
endif else begin
	targets=unique(strcompress(oitarget.target,/remove_all),si)
;	Make sure order of targets and oitarget is the same
	oitargets=oitarget(si)	; Note that oitarget is preserved!
;	Create and fill the StarTable
	create_startable,targets
	startable.name=targets
	startable.ra=oitargets.raep0/15
	startable.dec=oitargets.decep0
	startable.rae=oitargets.ra_err
	startable.dece=oitargets.dec_err
	startable.rv=oitargets.sysvel
	startable.pmra=oitargets.pmra
	startable.pmdec=oitargets.pmdec
	startable.pmrae=oitargets.pmra_err
	startable.pmdece=oitargets.pmdec_err
	startable.px=oitargets.parallax
	startable.pxe=oitargets.para_err
	startable.spectrum=oitargets.spectyp
;	startable_id will be an index into startable given a OI target_id
	if n_elements(oivis2) gt 0 then begin
	startable_id=intarr(n_elements(unique(oivis2.target_id))+1)
	startable_st=strarr(n_elements(unique(oivis2.target_id))+1) ; temporary
	endif else begin
	startable_id=intarr(n_elements(unique(oivis.target_id))+1)
	startable_st=strarr(n_elements(unique(oivis.target_id))+1) ; temporary
	endelse
;	Establish and assign the OYSTER StarId, to be used by this command:
;	scans(k).starid=startable(startable_id(v2(index(0)).target_id)).starid
	for i=0,n_elements(oitarget)-1 do begin
		j=where(startable.name eq $
			strcompress(oitarget(i).target,/remove_all))
;		startable_id is an index into startable given a target_id
		startable_id(oitarget(i).target_id)=j
		if valid_cat(targets(j)) eq 1 then starid=targets(j) $
					      else starid=cri_vlti(targets(j))
		if (strlen(starid) eq 0 or valid_cat(starid) eq 0) $
			and !owner eq 'chummel' then begin
			print,'Warning: '+targets(j)+' is not a recognized ID!'
			print,'Please add object to '+ $
				!oyster_dir+'starbase/vlti.usr'
		endif
		if strlen(starid) eq 0 then begin
			iras_id=irasid(startable(j).ra,startable(j).dec)
			if strlen(iras_id) gt 0 then begin
				startable(j).starid='IRAS'+iras_id 
			endif else $
			startable(j).starid='OBJ'+esoid(startable(j).ra, $
							startable(j).dec)
		endif else begin
			startable(j).starid=starid
		endelse
;		Temporarily store the target name for this oi_target
		startable_st(oitarget(i).target_id)=starid
	endfor
;	Remove identical StarIDs with different names
	starids=unique(startable.starid,si)
	startable=startable(si)
	targets=targets(si)
	for i=0,n_elements(oitarget)-1 do begin
		j=where(startable.starid eq startable_st(oitarget(i).target_id))
		startable_id(oitarget(i).target_id)=j
	endfor
;	Read catalog positions if target has an HD number
	index=where(strpos(startable.starid,'HDN') eq 0,count)
	if count gt 0 then begin
		startable_bck=startable
		startable=startable(index)
		startable.hdn=long(strmid(startable.starid,3,6))
		read_catalogs
		startable_bck(index).ra=startable.ra
		startable_bck(index).dec=startable.dec
		startable=startable_bck
	endif
;
endelse
;
; Determine number of nights
if n_elements(oivis) ne 0 then begin
	nights_vc=unique(oivis.date_obs)
	num_nights_vc=n_elements(nights_vc)
endif else begin
	nights_vc=''
	num_nights_vc=0
endelse
print,'Number of nights with complex vis data:   ',num_nights_vc
if n_elements(oivis2) ne 0 then begin
	ds_options.v2=1
	nights_v2=unique(oivis2.date_obs)
	num_nights_v2=n_elements(nights_v2)
endif else begin
	nights_v2=''
	num_nights_v2=0
endelse
print,'Number of nights with squared vis data:   ',num_nights_v2
if n_elements(oit3) ne 0 then begin
	ds_options.tp=1
	ds_options.ta=0
	nights_t3=unique(oit3.date_obs)
	num_nights_t3=n_elements(nights_t3)
endif else begin
	nights_t3=''
	num_nights_t3=0
endelse
print,'Number of nights with complex triple data:',num_nights_t3
if n_elements(oiflux) ne 0 then begin
	ds_options.fl=1
	nights_fl=unique(oiflux.date_obs)
	num_nights_fl=n_elements(nights_fl)
endif else begin
	nights_fl=''
	num_nights_fl=0
endelse
print,'Number of nights with flux data:          ',num_nights_fl
nights=unique([nights_vc,nights_v2,nights_t3,nights_fl])
index=where(strlen(nights) gt 0,count)
if count eq 0 then begin
	print,'***Error(GET_OIFITS): no data!'
	return
endif
nights=nights(index)
num_nights=n_elements(nights)
print,'Total number of nights: ',num_nights
;
; Detect missing time stamps
if n_elements(oivis) ne 0 then begin
	if total(oivis.time) eq 0 then oivis.time=(oivis.mjd mod 1)*86400
	uinsname=unique(oivis.insname)
	for i=0,n_elements(oivis.insname)-1 do $
	if oivis(i).time eq 0 then $
	   oivis(i).time=where(uinsname eq oivis(i).insname)*3600
endif
if n_elements(oivis2) ne 0 then begin
	if total(oivis2.time) eq 0 then oivis2.time=(oivis2.mjd mod 1)*86400
	uinsname=unique(oivis2.insname)
	for i=0,n_elements(oivis2.insname)-1 do $
	if oivis2(i).time eq 0 then $
	   oivis2(i).time=where(uinsname eq oivis2(i).insname)*3600
endif
if n_elements(oit3) ne 0 then begin
	if total(oit3.time) eq 0 then oit3.time=(oit3.mjd mod 1)*86400
	uinsname=unique(oit3.insname)
	for i=0,n_elements(oit3.insname)-1 do $
	if oit3(i).time eq 0 then $
	   oit3(i).time=where(uinsname eq oit3(i).insname)*3600
endif
if n_elements(oiflux) ne 0 then begin
	if total(oiflux.time) eq 0 then oiflux.time=(oiflux.mjd mod 1)*86400
	uinsname=unique(oiflux.insname)
	for i=0,n_elements(oiflux.insname)-1 do $
	if oiflux(i).time eq 0 then $
	   oiflux(i).time=where(uinsname eq oiflux(i).insname)*3600
endif
;
; Detect identical time stamps for different observations
if n_elements(oivis2) ne 0 then begin
	epochs=strcompress(oivis2.date_obs $
		+startable(startable_id(oivis2.target_id)).starid $
		+string(oivis2.time),/remove_all)
	times=strcompress(oivis2.date_obs $
		+string(oivis2.time),/remove_all)
	if n_elements(unique(times)) lt n_elements(unique(epochs)) then begin
;		Initialize time stamps
		epoch=epochs(0)
		time=hms2h(strmid(times(0),11,12))*3600
		for i=1,n_elements(epochs)-1 do begin
			if epochs(i) ne epoch then begin
				time=time+600
				epoch=epochs(i)
			endif
			oivis2(i).time=time
		endfor
	endif
endif
;
; Check whether the different instrument names really are different
nins=n_elements(oiwavelength)
insn=strarr(nins)
for i=0,nins-1 do begin
	p=oiwavelength(i).eff_wave
	for j=0,oiwavelength(i).nwave-1 do insn(i)=insn(i)+string((*p)(j))
endfor
if n_elements(unique(insn)) eq 1 then begin
	oiwavelength=oiwavelength(0)
	if n_elements(oivis) ne 0 then oivis(*).insname=oiwavelength.insname
	if n_elements(oivis2) ne 0 then oivis2(*).insname=oiwavelength.insname
	if n_elements(oit3) ne 0 then oit3(*).insname=oiwavelength.insname
	if n_elements(oiflux) ne 0 then oiflux(*).insname=oiwavelength.insname
endif
;
; Reduce array names to standard
if n_elements(oivis) ne 0 then begin
	for i=0,n_elements(oivis)-1 do $
		oivis(i).arrname=system_id(oivis(i).arrname)
endif
if n_elements(oivis2) ne 0 then begin
	for i=0,n_elements(oivis2)-1 do $
		oivis2(i).arrname=system_id(oivis2(i).arrname)
endif
if n_elements(oit3) ne 0 then begin
	for i=0,n_elements(oit3)-1 do $
		oit3(i).arrname=system_id(oit3(i).arrname)
endif
if n_elements(oiflux) ne 0 then begin
	for i=0,n_elements(oiflux)-1 do $
		oiflux(i).arrname=system_id(oiflux(i).arrname)
endif
;
; Reduce oiarray if duplicate entries exist
if n_elements(oiarray) ne 0 then begin
	oiarrays=oiarray
	oiarray.arrname=''
	oiarray.extver=1
	s=unique(stringof(oiarray),index)
	oiarray=oiarrays(index)
	for i=0,n_elements(oiarray)-1 do $
		oiarray(i).arrname=system_id(oiarray(i).arrname)
endif
;
; Determine number of data sets (differing in date and/or array)
; 2nd lines allow GRAVITY SC and FT in 2 OBs, ConfigID='GRAVITY_SCFT'
if n_elements(oivis) ne 0 $
;	then oivis_sets=oivis.date_obs+' '+oivis.arrname+' '+oivis.insname $
	then oivis_sets=oivis.date_obs+' '+oivis.arrname $
	else oivis_sets=''
if n_elements(oivis2) ne 0 $
;	then oivis2_sets=oivis2.date_obs+' '+oivis2.arrname+' '+oivis2.insname $
	then oivis2_sets=oivis2.date_obs+' '+oivis2.arrname $
	else oivis2_sets=''
if n_elements(oit3) ne 0 $
;	then oit3_sets=oit3.date_obs+' '+oit3.arrname+' '+oit3.insname $
	then oit3_sets=oit3.date_obs+' '+oit3.arrname $
	else oit3_sets=''
if n_elements(oiflux) ne 0 $
;	then oiflux_sets=oiflux.date_obs+' '+oiflux.arrname+' '+oiflux.insname $
	then oiflux_sets=oiflux.date_obs+' '+oiflux.arrname $
	else oiflux_sets=''
all_sets=[oivis_sets,oivis2_sets,oit3_sets,oiflux_sets]
index=where(strlen(all_sets) gt 0,count)
if count gt 0 then all_sets=all_sets(index)
sets=unique(all_sets)
num_sets=n_elements(sets)
index=where(sets(0) eq oiflux_sets,count)
if count ge 1 then sets=reverse(sets)	; The first set must have config info
print,'Number of datasets:',num_sets
for i=0,num_sets-1 do print,sets(i)
;
; If table oiarray missing (John, Sylvestre) or incomplete (Peter), 
; create default table. Also initialize the station indices in the data.
arrnames=strarr(num_sets)
for i=0,num_sets-1 do begin
	words=nameparse(sets(i),' ')
	arrnames(i)=words(0)	; Does not work if array name has blanks
;	3/2 words in a data set name (arrname, insname, date_obs), see above
;	arrnames(i)=strjoin(words(0:n_elements(words)-3),' ')
	arrnames(i)=strjoin(words(0:n_elements(words)-2),' ')
endfor
arrnames=unique(arrnames)
if n_elements(oiarray) eq 0 then begin
	if n_elements(station_file) ne 0 then begin
		l=''
		status=dc_read_free(station_file,l,/col)
		stations=strarr(3,n_elements(l))
		for i=0,n_elements(l)-1 do begin
			words=nameparse(l(i))
			stations(0:n_elements(words)-1,i)=words
		endfor
	endif
	def_index=unique(oivis2.sta_index)
	def_stations='A'+string(def_index,format='(i2.2)')
	num_tel=n_elements(def_stations)
	oiarrays=replicate({ $
		arrname:'', $
		sta_name:'', $
		sta_index:0, $
		arrayx:0.d0, $
		arrayy:0.d0, $
		arrayz:0.d0, $
		frame:'UNKNOWN', $
		staxyz:fltarr(3) $
		},num_tel*n_elements(arrnames))
	for i=0,n_elements(arrnames)-1 do begin
;		Group all telescopes for same array together
		oiarrays(i*num_tel:(i+1)*num_tel-1).arrname=arrnames(i)
;		Use info from station file, if available
		if n_elements(stations) gt 0 $
		then oiarrays(i*num_tel:(i+1)*num_tel-1).sta_name=stations(*,i) $
		else oiarrays(i*num_tel:(i+1)*num_tel-1).sta_name=def_stations
		if n_elements(stations) gt 0 $
		then oiarrays(i*num_tel:(i+1)*num_tel-1).sta_index=def_index $
		else oiarrays(i*num_tel:(i+1)*num_tel-1).sta_index=def_index
	endfor
	oiarray=oiarrays
endif
;
; Obtain info for allocation of geninfo and geoinfo
;
MaxSid=n_elements(unique(oiarray.sta_name))
MS=0
MB=0
MC=max(oiwavelength.nwave)
maxConfig=10
for i=0,num_nights_vc-1 do begin
	j=where(oivis.date_obs eq nights_vc(i),jc)
	if jc gt 0 then begin
		specs=unique(oivis(j).insname)
                num_spec=n_elements(specs)
                MS=max([MS,num_spec])
                vc=oivis(j)
                for j=0,num_spec-1 do begin
                k=where(vc.insname eq specs(j))
                MB=max([MB,n_elements(unique(strjoin(vc(k).sta_index)))])
                endfor
        endif
endfor
for i=0,num_nights_v2-1 do begin
	j=where(oivis2.date_obs eq nights_v2(i),jc)
	if jc gt 0 then begin
		specs=unique(oivis2(j).insname)
		num_spec=n_elements(specs)
		MS=max([MS,num_spec])
		v2=oivis2(j)
		for j=0,num_spec-1 do begin
		k=where(v2.insname eq specs(j))
		MB=max([MB,n_elements(unique(strjoin(v2(k).sta_index)))])
		endfor
	endif
endfor
MT=1
for i=0,num_nights_t3-1 do begin
	j=where(oit3.date_obs eq nights_t3(i),jc)
	if jc gt 0 then begin
		MT=max([MT,n_elements(unique(oit3(j).insname))]) $
		  *max([MT,n_elements(unique(strjoin(oit3.sta_index)))])
	endif
endfor
; Check for problems due to missing vc, v2, or t3
if MB eq 0 then begin
	MS=1
	MB=3
endif
; Starting in Version 6, we now allocate a reduced size version of geninfo
if num_sets gt 1 then begin
   geninfo=replicate(allocgenconfig(MaxSid,MS,MT,MB,MC,maxConfig,/geninfo), $
		  num_sets)
   geoinfo=replicate(allocgeoparms(),num_sets)
endif
baselineij=strarr(MB,MS)
;
; Now loop over all nights
;
for i=0,num_sets-1 do begin
;
; 	Starting in Version 6, we no longer allocate geninfo with the full
;	capacity. Therefore, we repeat the allocation for this particular night.
	if total(strlen(strcompress(oiarray.sta_name,/remove_all))) eq 0 then $
		oiarray.sta_name=oiarray.tel_name
	MaxSid=n_elements(unique(oiarray.sta_name))
	MS=0				; spectrometers
	MT=1				; triples
	MB=0				; baselines
	MC=max(oiwavelength.nwave)	; channels
	if n_elements(unique(oiarray.arrname)) eq 1 then begin
		systemid=strtrim(oiarray(0).arrname,2)
		maxConfig=system_config(system_id(systemid),'MAXCONFIG')
		case string(maxConfig) of
			'scans':maxConfig=scanconfig(/maxconfig)
			'stars':maxConfig=n_elements(scanconfig(/starlist))
			   else:maxConfig=fix(maxConfig)
		endcase
	endif else begin
		maxConfig=10
	endelse
	count_vc=0
	if n_elements(oivis) ne 0 then $
	index_vc=where(oivis_sets eq sets(i),count_vc)
	if count_vc gt 0 then begin
		specs=unique(oivis(index_vc).insname)
                num_spec=n_elements(specs)
                MS=max([MS,num_spec])
                vc=oivis(index_vc)
                for j=0,num_spec-1 do begin
                k=where(vc.insname eq specs(j))
                MB=max([MB,n_elements(unique(strjoin(vc(k).sta_index)))])
                endfor
		date=strmid(oivis(index_vc(0)).date_obs,0,10)
        endif
	count_v2=0
	if n_elements(oivis2) ne 0 then $
	index_v2=where(oivis2_sets eq sets(i),count_v2)
	if count_v2 gt 0 then begin
		specs=unique(oivis2(index_v2).insname)
		num_spec=n_elements(specs)
		MS=max([MS,num_spec])
		v2=oivis2(index_v2)
		for j=0,num_spec-1 do begin
		k=where(v2.insname eq specs(j))
		MB=max([MB,n_elements(unique(strjoin(v2(k).sta_index)))])
		endfor
		date=strmid(oivis2(index_v2(0)).date_obs,0,10)
	endif
	count_t3=0
	if n_elements(oit3) ne 0 then $
	index_t3=where(oit3_sets eq sets(i),count_t3)
	if count_t3 gt 0 then begin
;		MT=max([MT,n_elements(unique(strjoin(oit3.sta_index)))])
		MT=max([MT,n_elements(unique(oit3.insname))]) $
		  *max([MT,n_elements(unique(strjoin(oit3.sta_index)))])
		date=strmid(oit3(index_t3(0)).date_obs,0,10)
	endif
	count_fl=0
	if n_elements(oiflux) ne 0 then $
	index_fl=where(oiflux_sets eq sets(i),count_fl)
	if count_fl gt 0 then begin
		specs=unique(oiflux(index_fl).insname)
		num_spec=n_elements(specs)
		MS=max([MS,num_spec])
		fl=oiflux(index_fl)
		date=strmid(oiflux(index_fl(0)).date_obs,0,10)
		oiflux_max_time_offset=0; time offsets [s] between VIS2 and FLUX
	endif
; 	Check for problems due to missing vc, v2, or t3
	if MB eq 0 then begin
		MS=1
		MB=3
	endif
;	Set MB to maximum possible (only pndrs: may have CLP while missing BL!)
	MaxBase=(MaxSid*(MaxSid-1))/2
	if n_elements(oiwavelength) eq 1 then k=0 else k=i
	if strpos(oiwavelength(k).insname,'PIONIER') ge 0 and $
			count_v2 gt 0 and count_t3 gt 0 then $
		if MB lt MaxBase then MB=(MaxSid*(MaxSid-1))/2
;
;	Allocate genconfig, in the following, we fill GenConfig and geoinfo
	genconfig=allocgenconfig(MaxSid,MS,MT,MB,MC,maxConfig)
	geoparms=allocgeoparms()
;
;	Determine what kind of data were recorded
;
	num_spec=0
	num_triple=0
	int_time_vc=0
	if count_vc gt 0 then begin
;	Here we have a night with complex visibility data
		vc=oivis(index_vc)
		time_vc=vc.time
		int_time_vc=vc.int_time
;		int_time must be in seconds!
		if medianve(int_time_vc/60) gt 60 $
			then int_time_vc=int_time_vc/1000   
		vb=vc
	endif else time_vc=0
	int_time_v2=0
	if count_v2 gt 0 then begin
;	Here we have a night with squared visibility data
		v2=oivis2(index_v2)
		time_v2=v2.time
		int_time_v2=v2.int_time
;		int_time must be in seconds!
		if medianve(int_time_v2/60) gt 60 $
			then int_time_v2=int_time_v2/1000   
		vb=v2
	endif else time_v2=0
	int_time_t3=0
	if count_t3 gt 0 then begin
;	Here we have a night with complex triple data
		t3=oit3(index_t3)
		time_t3=t3.time
		int_time_t3=t3.int_time
;		int_time must be in seconds!
		if medianve(int_time_t3/60) gt 60 $
			then int_time_t3=int_time_t3/1000   
	endif else time_t3=0
	int_time_fl=0
	if count_fl gt 0 then begin
;	Here we have a night with flux data
		fl=oiflux(index_fl)
		time_fl=fl.time
		int_time_fl=fl.int_time
;		int_time must be in seconds!
		if medianve(int_time_fl/60) gt 60 $
			then int_time_fl=int_time_fl/1000   
	endif else time_fl=0
;
;	If we we have either v2 or vc, we need to extract geoparms and genconfig
	if count_vc gt 0 or count_v2 gt 0 then begin
;
;		Assemble info for GeoInfo
		r_earth=6378.136d3
		k=where(strtrim(oiarray.arrname,2) $
			eq strtrim(vb(0).arrname,2)) & k=k(0)
		systemid=strtrim(oiarray(k).arrname,2)
		geoparms.systemid=systemid
		geoparms.earthradius=r_earth
		if oiarray(k).arrayx+oiarray(k).arrayy+oiarray(k).arrayz eq 0 $
		then oiarray(k).frame='Unknown'
		case oiarray(k).frame of
		'GEOCENTRIC': begin
			      RAD=180/!pi
			      altitude=sqrt(oiarray(k).arrayx^2 $
					   +oiarray(k).arrayy^2 $
					   +oiarray(k).arrayz^2)-r_earth
			      latitude=asin(oiarray(k).arrayz $
					       /(altitude+r_earth))*RAD
			      longitude=atan(oiarray(k).arrayy, $
					     oiarray(k).arrayx)*RAD
			      end
		else:	      begin
			      latitude= $
			      system_config(system_id(systemid),'LATITUDE')
			      longitude= $
			      system_config(system_id(systemid),'LONGITUDE')
			      altitude= $
			      system_config(system_id(systemid),'ALTITUDE')
			      oiarray(k).frame='GEOCENTRIC'
			      end
		endcase
		geoparms.altitude=altitude
		geoparms.latitude=latitude
		geoparms.longitude=longitude
;		
;		Assemble info for GenConfig
;
;		Stations
		sta_index=oiarray.sta_index		; 2020-06-26
		sta_index=unique(vb.sta_index)		; original (no diff.)
		num_sid=n_elements(sta_index)
		stations=strarr(num_sid)+'___'
		siderostats=intarr(num_sid)
		siderostat_ids=intarr(num_sid)
		for j=0,num_sid-1 do begin
			siderostats(j)=sta_index(j)
			index=where(oiarray.sta_index eq sta_index(j))
			if strmid(oiarray(index(0)).tel_name,0,3) eq 'SID' $
				then k=3 else k=2
;			tel_name=strmid(oiarray(index(0)).tel_name,k,1)
;			if strlen(strcompress(tel_name,/remove_all)) eq 0 then $
;				siderostat_ids(j)=j else $
;			siderostat_ids(j)=fix(strmid( $
;				oiarray(index(0)).tel_name,k,1))
			siderostat_ids(j)=fix(strmid( $
			   strcompress(oiarray(index(0)).tel_name,/remove_all),k,1))
			station=stations(j)
			sta_name=strtrim(oiarray(index).sta_name,2)
			strput,station,sta_name(0)
			stations(j)=station
		endfor
		genconfig.numsid=num_sid
		genconfig.stationid(0:num_sid-1)=stations
;		genconfig.siderostatid(0:num_sid-1)=siderostats
		if systemid eq 'VLTI' then $
			genconfig.siderostatid(0:num_sid-1)=siderostat_ids
		genconfig.diameter=system_config(systemid,'DIAMETER', $
						stations=genconfig.stationid)
		genconfig.refstation=1
		cl=cos(latitude/RAD)
		sl=sin(latitude/RAD)
		for j=0,num_sid-1 do begin
		  k=where(oiarray.sta_index eq sta_index(j)) & k=k(0)
		  staxyz=oiarray(k).staxyz
		  if total(staxyz) ne 0 then begin
;		  Geocentric system (the OIFITS standard)
		  genconfig.stationcoord(0:2,j)=equatorial2horizon( $
					      geocentric2equatorial(staxyz(*)))
;		  Issue discovered by Regis Lachaume for GRAVITY pipeline,
;		  but true for all VLTI pipelines (PNDRS does not write coord.):
;		  OIFITS files have horizon coordinates instead of geocentric
		  if systemid eq 'VLTI' and stddev(oiarray.staxyz(2)) lt 0.1 $
			  then genconfig.stationcoord(0:2,j)=staxyz*[-1,-1,1]
		  endif
		endfor
;
;		Spectrometers (a.k.a. instruments)
		specs=unique(vb.insname)
		if strpos(specs(0),'GRAVITY') ge 0 then begin
			reverse_specs=1
			si=sort(specs)
			specs=specs(reverse(si))	; SC 1st
		endif else reverse_specs=0
		num_spec=n_elements(specs)
		genconfig.numoutbeam=num_spec
;
;		Baselines
		for j=0,num_spec-1 do begin
			k=where(vb.insname eq specs(j))
;			b=unique(strjoin(vb(k).sta_index))
			b=remove_dup(strjoin(vb(k).sta_index))
			genconfig.numbaseline(j)=n_elements(b)
			for l=0,n_elements(b)-1 do begin
				baselineij(l,j)=b(l)
				ij=long(nameparse(b(l)))
				st1=stations(where(sta_index eq ij(0)))
				st2=stations(where(sta_index eq ij(1)))
				genconfig.baselineid(l,j)=st1+'-'+st2
			endfor
			k=where(oiwavelength.insname eq specs(j))
			nw=oiwavelength(k).nwave
			genconfig.numspecchan(j)=nw
			genconfig.wavelength(0:nw-1,j)= $
				*oiwavelength(k).eff_wave
			genconfig.wavelengtherr(0:nw-1,j)= $
				*oiwavelength(k).eff_wave*0.01
			genconfig.chanwidth(0:nw-1,j)= $
				*oiwavelength(k).eff_band
		endfor
;
;		Prepare ConfigID (must follow OYSTER standards)
		configid=shortest(strtrim(vb.insname,2))
;
;		Instrument-specific adjustments of ConfigID
		if configid eq 'GRAVITY_' then configid='GRAVITY_SCFT'
;
		if strpos(configid,'MATISSE') ge 0 then begin
			if genconfig.numoutbeam eq 1 then begin
			if median(genconfig.wavelength) le 6e-6 then b='L'
			if median(genconfig.wavelength) gt 6e-6 then b='N'
			configid=configid+'-'+b
			case b of ; Time [s] offsets FLUX relative to VIS2
				'L':oiflux_max_time_offset=200
				'N':oiflux_max_time_offset=200
			endcase
			endif else begin	; numoutbeam=2
				configid=configid+'-'+'LN'
				oiflux_max_time_offset=200
			endelse
		endif
;
		if strpos(configid,'PIONIER_Pnat') ge 0 then begin
			words=nameparse(configid,['(',')'])
			words=nameparse(words(1),'/')
			nchan=string(genconfig.numspecchan,format='(i1)')
			if n_elements(unique(words)) eq 1 $
				then configid='PIONIER_FREE' $
				else configid='PIONIER_GRS'+nchan
		endif else begin
			if strpos(configid,'PIONIER') eq 0 then begin
				nchan=string(genconfig.numspecchan,format='(i1)')
				if nchan eq 1 then configid='PIONIER_FREE' $
					      else configid='PIONIER_GRISM'
			endif
		endelse
;
;		Replace unwanted characters in ConfigID
		genconfig.configid=stredit(configid,['(','/',')'],['_','-',' '])
;		
		words=strtrim(specs(0),2)
		genconfig.beamcombinerid=beamcombiner_id(system_id(systemid) $
					+'/'+words(0))
	endif
;	If we we have t3, we need to extract geoparms and genconfig
	if count_t3 gt 0 then begin
;
;		Assemble info for GenConfig
		r_earth=6378.136d3
		k=where(strtrim(oiarray.arrname,2) eq strtrim(t3(0).arrname,2)) 
		k=k(0)
		systemid=strtrim(oiarray(k).arrname,2)
		geoparms.systemid=systemid
		geoparms.earthradius=r_earth
		geoparms.latitude=system_config(system_id(systemid),'LATITUDE')
		geoparms.longitude=system_config(system_id(systemid),'LONGITUDE')
		geoparms.altitude=system_config(system_id(systemid),'ALTITUDE')
;
;		Only retrieve specs if there are no v2 or vc data
		if n_elements(vb) eq 0 then begin
			specs=unique(t3.insname)
			if strpos(specs(0),'GRAVITY') ge 0 then begin
				reverse_specs=1
				si=sort(specs)
				specs=specs(reverse(si))
			endif else reverse_specs=0
			num_spec=n_elements(specs)
			genconfig.numoutbeam=num_spec
		endif
;
;		triples=unique(strjoin(t3.sta_index)+' '+t3.insname)	; alternative
;		triples=unique(t3.insname+' '+strjoin(t3.sta_index))
		triples=remove_dup(t3.insname+' '+strjoin(t3.sta_index))
		if reverse_specs then begin
			si=sort(triples)
			triples=triples(reverse(si))
		endif
		num_triple=n_elements(triples)
		genconfig.numtriple=num_triple
;
		sta_index=oiarray.sta_index		; 2020-06-26
		sta_index=unique(t3.sta_index)		; original (no diff.)
		num_sid=n_elements(sta_index)
		stations=strarr(num_sid)+'___'
		siderostats=intarr(num_sid)
		siderostat_ids=intarr(num_sid)
		for j=0,num_sid-1 do begin
			siderostats(j)=sta_index(j)
			index=where(oiarray.sta_index eq sta_index(j))
			if strmid(oiarray(index(0)).tel_name,0,3) eq 'SID' $
				then k=3 else k=2
;			tel_name=strmid(oiarray(index(0)).tel_name,k,1)
;			if strlen(strcompress(tel_name,/remove_all)) eq 0 then $
;				siderostat_ids(j)=j else $
;			siderostat_ids(j)=fix(strmid( $
;				oiarray(index(0)).tel_name,k,1))
			siderostat_ids(j)=fix(strmid( $
			   strcompress(oiarray(index(0)).tel_name,/remove_all),k,1))
			station=stations(j)
			sta_name=strtrim(oiarray(index(0)).sta_name,2)
			strput,station,sta_name(0)
			stations(j)=station
		endfor
		genconfig.numsid=num_sid
		genconfig.stationid(0:num_sid-1)=stations
;		genconfig.siderostatid(0:num_sid-1)=siderostats
		if systemid eq 'VLTI' then $
			genconfig.siderostatid(0:num_sid-1)=siderostat_ids
		genconfig.diameter=system_config(systemid,'DIAMETER', $
						stations=genconfig.stationid)
		genconfig.refstation=1
		configid=shortest(strtrim(t3.insname,2))
;
;		Instrument-specific adjustments of ConfigID
		if configid eq 'GRAVITY_' then configid='GRAVITY_SCFT'
		if strpos(configid,'GRAVITY') ge 0 then begin
			oiflux_max_time_offset=0.001
		endif
;
		if strpos(configid,'MATISSE') ge 0 then begin
			if genconfig.numoutbeam eq 1 then begin
			if median(genconfig.wavelength) le 6e-6 then b='L'
			if median(genconfig.wavelength) gt 6e-6 then b='N'
			configid=configid+'-'+b
			case b of ; Time [s] offsets FLUX relative to VIS2
				'L':oiflux_max_time_offset=200
				'N':oiflux_max_time_offset=200
			endcase
			endif else begin	; numoutbeam=2
				configid=configid+'-'+'LN'
				oiflux_max_time_offset=200
			endelse
		endif
		if strpos(configid,'PIONIER_Pnat') ge 0 then begin
			words=nameparse(configid,['(',')'])
			words=nameparse(words(1),'/')
			nchan=string(genconfig.numspecchan,format='(i1)')
			if n_elements(unique(words)) eq 1 $
				then configid='PIONIER_FREE' $
				else configid='PIONIER_GRS'+nchan
		endif else begin
			if strpos(configid,'PIONIER') eq 0 then begin
				nchan=string(genconfig.numspecchan,format='(i1)')
				if nchan eq 1 then configid='PIONIER_FREE' $
					      else configid='PIONIER_GRISM'
			endif
		endelse
;
;		Replace unwanted characters in ConfigID
		genconfig.configid=stredit(configid,['(','/',')'],['_','-',' '])
;
		words=strtrim(specs(0),2)
		genconfig.beamcombinerid=beamcombiner_id(system_id(systemid) $
					+'/'+words(0))
;
;		For data sets with only triple products initialize baselines
		if total( $
		   strlen( $
		   strcompress( $
		   strjoin( $
		   genconfig.baselineid),/remove_all))) eq 0 $
		then begin
			genconfig.numoutbeam=1
			genconfig.numbaseline=3
			genconfig.baselineid(0)='ST1-ST2'
			genconfig.baselineid(1)='ST2-ST3'
			genconfig.baselineid(2)='ST3-ST1'
			genconfig.baselineid(0)=stations(0)+'-'+stations(1)
			genconfig.baselineid(1)=stations(1)+'-'+stations(2)
			genconfig.baselineid(2)=stations(2)+'-'+stations(0)
			genconfig.triplebase(0)=0
			genconfig.triplebase(1)=0
			genconfig.triplebase(2)=0
		endif
;		Identify baselines
		for j=0,num_triple-1 do begin
			t=triples(j)
;			index=where(strjoin(t3.sta_index)+' '+t3.insname eq t)	; alternative
			index=where(t3.insname+' '+strjoin(t3.sta_index) eq t)
			words=nameparse(t)
;			ijk=long(words(0:2))	; alternative
			ijk=long(words(1:3))
			st0=strtrim(oiarray(where(strtrim(oiarray.arrname,2) $
						      eq systemid $
				and oiarray.sta_index eq ijk(0))).sta_name,2)
			st1=strtrim(oiarray(where(strtrim(oiarray.arrname,2) $
						      eq systemid $
				and oiarray.sta_index eq ijk(1))).sta_name,2)
			st2=strtrim(oiarray(where(strtrim(oiarray.arrname,2) $
						      eq systemid $
				and oiarray.sta_index eq ijk(2))).sta_name,2)
;			If oiarray has duplicate entries
			st0=st0(0) & st1=st1(0) & st2=st2(0)
			st0=stations(where(strpos(stations,st0) ge 0))
			st0=st0(0)
			st1=stations(where(strpos(stations,st1) ge 0))
			st1=st1(0)
			st2=stations(where(strpos(stations,st2) ge 0))
			st2=st2(0)
			b1=st0+'-'+st1
			b2=st1+'-'+st2
			b3=st2+'-'+st0
			for k=0,genconfig.numoutbeam-1 do begin
			i1=where(genconfig.baselineid $
				(0:genconfig.numbaseline(k)-1,k) eq b1 $
			      or genconfig.baselineid $
				(0:genconfig.numbaseline(k)-1,k) eq breve(b1))
			i2=where(genconfig.baselineid $
				(0:genconfig.numbaseline(k)-1,k) eq b2 $
			      or genconfig.baselineid $
				(0:genconfig.numbaseline(k)-1,k) eq breve(b2))
			i3=where(genconfig.baselineid $
				(0:genconfig.numbaseline(k)-1,k) eq b3 $
			      or genconfig.baselineid $
				(0:genconfig.numbaseline(k)-1,k) eq breve(b3))
			if i1(0) ge 0 then begin
				genconfig.triplebase(0,j)=i1(0)
;				if specs(k) eq words(3) then $	; alternative
				if specs(k) eq words(0) then $
				genconfig.triplebeam(0,j)=k
			endif
			if i2(0) ge 0 then begin
				genconfig.triplebase(1,j)=i2(0)
;				if specs(k) eq words(3) then $	; alternative
				if specs(k) eq words(0) then $
				genconfig.triplebeam(1,j)=k
			endif
			if i3(0) ge 0 then begin
				genconfig.triplebase(2,j)=i3(0)
;				if specs(k) eq words(3) then $	; alternative
				if specs(k) eq words(0) then $
				genconfig.triplebeam(2,j)=k
			endif else begin
				if 0 then begin
;				pndrs may have CLP while missing BL!
				i3=where(strlen(strcompress( $
					genconfig.baselineid,/remove_all)) eq 0)
				i3=i3(0)
				genconfig.baselineid(i3(0))=b3
				genconfig.triplebase(2,j)=i3(0)
				st1=stations(where(stations eq strmid(b3,0,3)))
				st2=stations(where(stations eq strmid(b3,3,3)))
				genconfig.baselineid(i3(0))=st1+'-'+st2
				endif
			endelse
			endfor
;			k=where(strtrim(oiwavelength.insname) eq specs(j)) ; alternative
			k=where(strtrim(oiwavelength.insname) eq words(0))
			nw=oiwavelength(k).nwave
			genconfig.triplenumchan(j)=nw
			genconfig.wavelength(0:nw-1,k)= $
				*oiwavelength(k).eff_wave
			genconfig.wavelengtherr(0:nw-1,k)= $
				*oiwavelength(k).eff_wave*0.01
			genconfig.chanwidth(0:nw-1,k)= $
				*oiwavelength(k).eff_band
			lambda=*oiwavelength(k).eff_wave
			for l=0,nw-1 do begin
			for k=0,2 do begin
			index=where(genconfig.wavelength $
				(0:nw-1,genconfig.triplebeam(k,j)) eq lambda[l])
			genconfig.triplechan(l,k,j)=index
			genconfig.numspecchan(genconfig.triplebeam(k,j)) $
				=genconfig.triplenumchan(j)
			endfor
			endfor
		endfor
	endif
;	Establish a common vector of time stamps
	time=[time_vc,time_v2,time_t3]
	int_time=[int_time_vc,int_time_v2,int_time_t3]
	index=where(time ne 0,count)
	if count gt 0 then begin
		time=unique(time(index),ui)
		if n_elements(ui) gt 1 then int_time=int_time(ui)	; 
		num_scan=n_elements(time)
		scans=replicate(scan(),num_scan)
		scans.iscan=indgen(num_scan)+1
		scans.time=time
		scans.temp=temp
		scans.pres=pres
		scans.rhum=rhum
		scans.r0=seeing
		scans.t0=taunot
		scans.pwv=pwv
;
		true=byte('T') & true=true(0)
;		Work on every scan
		FOR k=0,num_scan-1 do BEGIN
;
		FOR j=0,num_spec-1 do BEGIN
			nw=genconfig.numspecchan(j)
		FOR l=0,genconfig.numbaseline(j)-1 do BEGIN
;
option=1	; Testing purpose, option 2 has bugs
if option eq 1 then begin
;		Complex visibility data (both amp/phase and complex vis/err)
		if count_vc gt 0 then begin
		oivis_ok=0
		index=where(vc.time eq time(k) $
			and vc.insname eq specs(j) $
			and strjoin(vc.sta_index) eq baselineij(l,j),count)
		if count gt 0 then begin
		index=index(0)
;		Look for OIFITS2-compliant VISAMP and VISPHI
		if tag_exists('VISAMP',vc) and tag_exists('VISPHI',vc) then begin
		if vc(0).visamp ne !NULL and vc(0).visphi ne !NULL then begin
			scans(k).diffphasec(j,0:nw-1,l) $
				=(*vc(index).visphi)(0:nw-1)
			scans(k).diffphasecerr(j,0:nw-1,l) $
				=((*vc(index).visphierr)(0:nw-1)) < 3.4e38
			vc_real=((*vc(index).visamp)(0:nw-1)) $
				*cos(((*vc(index).visphi)(0:nw-1))/RAD)
			vc_imag=((*vc(index).visamp)(0:nw-1)) $
				*sin(((*vc(index).visphi)(0:nw-1))/RAD)
			scans(k).complexvis(j,0:nw-1,l)=complex(vc_real,vc_imag)
			scans(k).complexweight(j,0:nw-1,l)= $
				byte(*vc(index).flag) ne true
			oivis_ok=1
		endif
		endif
		endif
		if not oivis_ok then begin
		if tag_exists('RVIS',vc) and tag_exists('IVIS',vc) then begin
		if vc(0).rvis ne !NULL and vc(0).ivis ne !NULL then begin
;			RVIS and IVIS are the real and imaginary part of the 
;			complex coherent flux
			scans(k).complexvis(j,0:nw-1,l)= $
				complex(*vc(index).rvis,*vc(index).ivis)
			scans(k).complexweight(j,0:nw-1,l)= $
				byte(*vc(index).flag) ne true
			oivis_ok=1
		endif
		endif
		endif
		if not oivis_ok then begin
;		Look for non-compliant GRAVITY VISDATA and VISERR columns
		if tag_exists('VISDATA',vc) and tag_exists('VISERR',vc) then begin
;			print,'Warning: this code not yet validated...'
		if vc(0).visdata ne !NULL and vc(0).viserr ne !NULL then begin
			if k+j+l eq 0 then $
			print,'Warning: non-compliant tags VISDATA and VISERR!'
			scans(k).diffphasec(j,0:nw-1,l) $
				=cphase((*vc(index).visdata)(0:nw-1))
			idx=where(abs((*vc(index).visdata)(0:nw-1)) gt 0,nidx)
			if nidx gt 0 then $
				scans(k).diffphasecerr(j,idx,l) $
					=abs((*vc(index).viserr)(idx)) $
					/abs((*vc(index).visdata)(idx))
			idx=where(byte(*vc(index).flag) eq true,nidx)
			if nidx ge 1 then scans(k).diffphasecerr(j,idx,l)=-1
;			VISDATA is the complex flux, not mormalized
			scans(k).complexvis(j,0:nw-1,l)= $
				(*vc(index).visdata)(0:nw-1)
			scans(k).complexweight(j,0:nw-1,l)= $
				byte((*vc(index).flag)(0:nw-1)) ne true
			oivis_ok=1
		endif
		endif
		endif
		endif	; count_vc
;
endif else begin
;	This part is an update of the code above, but has issues
;		Complex visibility data (both amp/phase and complex vis/err)
		if count_vc gt 0 then begin
		index=where(vc.time eq time(k) $
			and vc.insname eq specs(j) $
			and strjoin(vc.sta_index) eq baselineij(l,j),count)
		if count gt 0 then begin
;		Look for OIFITS2-compliant VISAMP and VISPHI
		oivis_ok=0
		if tag_exists('VISAMP',vc) and tag_exists('VISPHI',vc) then begin
		if vc(0).visamp ne !NULL and vc(0).visphi ne !NULL then begin
		for i=0,count-1 do begin
			scans(k).diffphasec(j,index,l)=*(vc(index(i)).visphi)
			scans(k).diffphasecerr(j,index,l)=*(vc(index(i)).visphierr)
			vc_real=(*vc(index(i)).visamp)*cos(*vc(index(i)).visphi/RAD)
			vc_imag=(*vc(index(i)).visamp)*sin(*vc(index(i)).visphi/RAD)
			scans(k).complexvis(j,index,l)=complex(vc_real,vc_imag)
			scans(k).complexweight(j,index,l)=*(vc(index(i)).flag) ne true
		endfor
		oivis_ok=1
		endif
		endif
		if not oivis_ok then begin
;		Look for RVIS and IVIS, the real and imaginary parts of the 
;		complex coherent flux
		if tag_exists('RVIS',vc) and tag_exists('IVIS',vc) then begin
		if vc(0).rvis ne !NULL and vc(0).ivis ne !NULL then begin
		for i=0,count-1 do begin
			scans(k).complexvis(j,index,l)= $
				complex(*vc(index(i)).rvis,*vc(index(i)).ivis)
			scans(k).complexweight(j,index,l)= $
				*(vc(index(i)).flag) ne true
		endfor
		oivis_ok=1
		endif
		endif
		endif
		if not oivis_ok then begin
;		Look for non-compliant GRAVITY VISDATA and VISERR columns
		if tag_exists('VISDATA',vc) and tag_exists('VISERR',vc) then begin
;			print,'Warning: this code not yet validated...'
		if vc(0).visdata ne !NULL and vc(0).viserr ne !NULL then begin
			if k+j+l eq 0 then $
			print,'Warning: non-compliant tags VISDATA and VISERR!'
		for i=0,count-1 do begin
			scans(k).diffphasec(j,index,l)=cphase(*vc(index(i)).visdata)
			scans(k).diffphase(j,index,l)=scans(k).diffphasec(j,*,l)
			idx=where(abs((*vc(index(i)).visdata)) gt 0,nidx)
			if nidx gt 0 then $
				scans(k).diffphasecerr(j,i(idx),l) $
					=abs(((*vc(index(i)).viserr)(idx))) $
					/abs(((*vc(index(i)).visdata)(idx)))
			idx=where(byte(*vc(index(i)).flag) eq true,nidx)
			if nidx ge 1 then scans(k).diffphasecerr(j,i(idx),l)=-1
			scans(k).diffphaseerr(j,index,l)=scans(k).diffphasecerr(j,*,l)
;			VISDATA is the complex flux, not mormalized
			scans(k).complexvis(j,index,l)=(*vc(index(i)).visdata)
			scans(k).complexweight(j,index,l)=(*vc(index(i)).flag) ne true
		endfor
		oivis_ok=1
		endif
		endif
		endif
;
		scans(k).starid=startable( $
			startable_id(vc(index(0)).target_id)).starid
		scans(k).star=targets(startable_id(vc(index(0)).target_id))
		scans(k).uvw(j,0:nw-1,l,0)=vc(index).ucoord $
					  /genconfig.wavelength(0:nw-1,j)
		scans(k).uvw(j,0:nw-1,l,1)=vc(index).vcoord $
					  /genconfig.wavelength(0:nw-1,j)
;		Read additional columns if file is output of GRAVITY astrometry
		if strpos(fitsfile,'ASTROREDUCED') ge 0 then begin
			scans.vissqcerr=1 ; used as error for UVW values
		endif
;
		endif	; count
		endif	; count_vc
endelse
;
;		Squared visibility data, min(err) set to 0.001 unless flag=true
		if count_v2 gt 0 then begin
		index=where(v2.time eq time(k) $
			and v2.insname eq specs(j) $
			and strjoin(v2.sta_index) eq baselineij(l,j),count)
		if count gt 0 then begin
		index=index(0)
		scans(k).vissqc(j,0:nw-1,l)=(*v2(index).vis2data)(0:nw-1)
;		There have been NaNs in these data
		nan=where(finite((*v2(index).vis2err)(0:nw-1)) eq 0,n_nan)
		if n_nan gt 0 then (*v2(index).vis2err)(nan)=-1
		scans(k).vissqcerr(j,0:nw-1,l)=(*v2(index).vis2err)(0:nw-1) > 0.001
		eindex=where(byte(*(v2(index(0)).flag)) eq true,ecount)
		if ecount ge 1 then scans(k).vissqcerr(j,eindex,l)=-1
;		The MATISSE pipeline flags more FL data than VIS data!
;		Optionally use FL flags here (experimental!)
		if strpos(configid,'MATISSE') ge 0 $
			and num_nights_fl gt 0 then begin
			index=where(fl.time eq time(k) $
				and fl.insname eq specs(j),count)
; 			Comment next line if VIS2 flags are to be used for FL
;			count=0
			if count gt 0 then begin
			   index=index(0)
			   eindex=where(byte(*fl(index).flag) eq true,ecount)
			   if ecount ge 1 then scans(k).vissqcerr(j,eindex,l)=-1
			endif
		endif
		jndex=where(finite(scans(k).vissqc(j,0:nw-1,l)) ne 1,count)
		if count gt 0 then begin
			scans(k).vissqc(j,jndex,l)=1.0
			scans(k).vissqcerr(j,jndex,l)=-1.0
		endif
		scans(k).starid=startable( $
			startable_id(v2(index(0)).target_id)).starid
		scans(k).star=targets(startable_id(v2(index).target_id))
		scans(k).uvw(j,0:nw-1,l,0)=v2(index).ucoord $
					  /genconfig.wavelength(0:nw-1,j)
		scans(k).uvw(j,0:nw-1,l,1)=v2(index).vcoord $
					  /genconfig.wavelength(0:nw-1,j)
		endif
		endif
;
		ENDFOR	; baselines
;
;		Flux data, min(err) set to 0.001 unless flag=true
		for l=0,num_sid-1 do begin
		if count_fl gt 0 then begin
		oiflux_time_offset=median(fl.time-time(k))
		if oiflux_max_time_offset ne 0 and not merged and l eq 0 and j eq 0 then $
		print,'Median FLUX-VIS2 time [s]: ',oiflux_time_offset,', scan:',fix(k)
		if oiflux_max_time_offset ne 0 then $
		index=where(abs(fl.time-time(k)) le oiflux_max_time_offset $
			and fl.insname eq specs(j) $
			and fl.sta_index eq siderostats(l),count) $
		else $
		index=where(fl.time eq time(k) $
			and fl.insname eq specs(j) $
			and fl.sta_index eq siderostats(l),count)
		if count gt 0 then begin
		 index=index(0)
		 scans(k).photometry(l,j,0:nw-1) $
			 =(*fl(index).fluxdata)(0:nw-1)
		 scans(k).photometryerr(l,j,0:nw-1) $
			 =(*fl(index).fluxerr)(0:nw-1) > 0.001
		 eindex=where(byte(*fl(index).flag) eq true,ecount)
;		 The MATISSE pipeline flags more FL data than VIS data!
;		 Optionally use VIS2 flags here (experimental!)
		 if strpos(configid,'MATISSE') ge 0 then $
		 eindex=where(byte(*v2(index).flag) eq true,ecount)
;		 Comment next line if instead the FL flags are used for VIS2
;		 if ecount ge 1 then scans(k).photometryerr(l,j,eindex)=-1
		 jndex=where(finite(scans(k).photometry(l,j,0:nw-1)) ne 1,count)
		 if count gt 0 then begin
			scans(k).photometry(l,j,jndex)=1.0
			scans(k).photometryerr(l,j,jndex)=-1.0
		 endif
		 scans(k).starid=startable( $
			 startable_id(fl(index(0)).target_id)).starid
		 scans(k).star=targets(startable_id(fl(index(0)).target_id))
		endif
		endif
		endfor	; stations
;
		ENDFOR	; spectrometers
;
		FOR j=0,num_triple-1 DO BEGIN
			nw=genconfig.triplenumchan(j)
;
;		Triple product data, min(err) set to 0.001 unless flag=true
		index=where(t3.time eq time(k) $
		  and t3.insname+' '+strjoin(t3.sta_index) eq triples(j),count)
		if count gt 0 then begin
		index=index(0)
		scans(k).tripleampc(j,0:nw-1)=(*t3(index).t3amp)(0:nw-1)
		if total(finite((*t3(index).t3amperr)(0:nw-1))) ne 0 then $ 
			scans(k).tripleampcerr(j,0:nw-1) $
			=(*t3(index).t3amperr)(0:nw-1) > 0.001
; 		Flip sign of phase (based on Iota Peg observations)
		scans(k).triplephasec(j,0:nw-1) $
			=((-(*t3(index).t3phi)(0:nw-1)) mod 360)/RAD	
		if min(scans(k).triplephasec(j,0:nw-1)) lt -!pi $
			then scans(k).triplephasec(j,0:nw-1) $
				=scans(k).triplephasec(j,0:nw-1)+2*!pi
		scans(k).triplephasecerr(j,0:nw-1) $
			=(*t3(index).t3phierr)(0:nw-1)/RAD > 0.001
		eindex=where(byte(*t3(index).flag) eq true,ecount)
		if ecount ge 1 then begin
			scans(k).tripleampcerr(j,eindex)=-1
			scans(k).triplephasecerr(j,eindex)=-1
		endif
		jndex=where(finite(scans(k).tripleampc(j,0:nw-1)) ne 1,count)
		if count gt 0 then begin
			scans(k).tripleampc(j,jndex)=1.0
			scans(k).tripleampcerr(j,jndex)=-1.0
			scans(k).triplephasec(j,jndex)=1.0
			scans(k).triplephasecerr(j,jndex)=-1.0
		endif
		jndex=where(finite(scans(k).triplephasec(j,0:nw-1)) ne 1,count)
		if count gt 0 then begin
			scans(k).triplephasec(j,jndex)=1.0
			scans(k).triplephasecerr(j,jndex)=-1.0
			scans(k).tripleampc(j,jndex)=1.0
			scans(k).tripleampcerr(j,jndex)=-1.0
		endif
		scans(k).starid=startable( $
			startable_id(t3(index(0)).target_id)).starid
		scans(k).star=targets(startable_id(t3(index(0)).target_id))
;		Make sure we follow the rules...
		fBaseMatrix=intarr(GenConfig.NumSid,3)
		fBaseFactor=fltarr(3)+1
	        for l=0,2 do begin
                	ob=GenConfig.TripleBeam(l,j)
                	bl=GenConfig.TripleBase(l,j)
                	j1=where(GenConfig.StationId eq $
				strmid(GenConfig.BaselineId(bl,ob),0,3))
                	j2=where(GenConfig.StationId eq $
				strmid(GenConfig.BaselineId(bl,ob),4,3))
                	fBaseMatrix(j1,l)=+1
                	fBaseMatrix(j2,l)=-1
                	if l gt 0 then fBaseFactor(l)= $
				-total(fBaseMatrix(*,0)*fBaseMatrix(*,l))
        	endfor
		l=0
		scans(k).uvw(genconfig.triplebeam(l,j), $
			     0:genconfig.triplenumchan(j)-1, $
			     genconfig.triplebase(l,j),0) $
			=t3(index).u1coord*fBaseFactor(l) $
			/genconfig.wavelength(0:genconfig.triplenumchan(j)-1, $
					      genconfig.triplebeam(l,j))
		scans(k).uvw(genconfig.triplebeam(l,j), $
			     0:genconfig.triplenumchan(j)-1, $
			     genconfig.triplebase(l,j),1) $
			=t3(index).v1coord*fBaseFactor(l) $
			/genconfig.wavelength(0:genconfig.triplenumchan(j)-1, $
					      genconfig.triplebeam(l,j))
		l=1
		scans(k).uvw(genconfig.triplebeam(l,j), $
			     0:genconfig.triplenumchan(j)-1, $
			     genconfig.triplebase(l,j),0) $
			=t3(index).u2coord*fBaseFactor(l) $
			/genconfig.wavelength(0:genconfig.triplenumchan(j)-1, $
					      genconfig.triplebeam(l,j))
		scans(k).uvw(genconfig.triplebeam(l,j), $
			     0:genconfig.triplenumchan(j)-1, $
			     genconfig.triplebase(l,j),1) $
			=t3(index).v2coord*fBaseFactor(l) $
			/genconfig.wavelength(0:genconfig.triplenumchan(j)-1, $
					      genconfig.triplebeam(l,j))
		l=2
		scans(k).uvw(genconfig.triplebeam(l,j), $
			     0:genconfig.triplenumchan(j)-1, $
			     genconfig.triplebase(l,j),0) $
			=(-t3(index).u1coord-t3(index).u2coord)*fBaseFactor(l) $
			/genconfig.wavelength(0:genconfig.triplenumchan(j)-1, $
					      genconfig.triplebeam(l,j))
		scans(k).uvw(genconfig.triplebeam(l,j), $
			     0:genconfig.triplenumchan(j)-1, $
			     genconfig.triplebase(l,j),1) $
			=(-t3(index).v1coord-t3(index).v2coord)*fBaseFactor(l) $
			/genconfig.wavelength(0:genconfig.triplenumchan(j)-1, $
					      genconfig.triplebeam(l,j))
		endif
;
		ENDFOR	; triples
;
		ENDFOR	; scans
;
;		Here we have a good night
		genconfig.date=checkdate()
		geoparms.date=date
; 		Default is to clear AMOEBA's buffer and initialize Info arrays
		if n_elements(freememory) eq 0 then freememory=1
		if freememory then begin
			freememory
			GenInfo=replicate(GenConfig,1)
			GeoInfo=replicate(GeoParms,1)
			if num_sets gt 1 then $
			freememory=0	; OIFITS file can have more than one set
		endif
;		Next 4 lines commented for OIFITS in load_interferometry
;		if i eq 0 then freememory
;		if i eq 0 and num_sets gt 1 then begin
;			bufferinfo=replicate(nightinfo(),num_sets)
;		endif
		get_scantable
;		Store INT_TIME information indirectly in scantable
		scantable.stoptime=scantable.starttime+median(int_time)
; 		Store photometric INT_TIME needed for flux calibration
		if int_time_fl(0) gt 0 then $
		scans.int_time=median(int_time_fl) else $
		scans.int_time=scantable.stoptime-scantable.starttime
;
;		Use the uv-coordinates from the data file
		calcastrom,/skipuv
		scans.vissq=scans.vissqc
		scans.vissqerr=scans.vissqcerr
		scans.vissqe=scans.vissq
		scans.vissqeerr=scans.vissqerr
		scans.vissqec=scans.vissqc
		scans.vissqecerr=scans.vissqcerr
		scans.tripleamp=scans.tripleampc
		scans.tripleamperr=scans.tripleampcerr
		scans.tripleampe=scans.tripleamp
		scans.tripleampeerr=scans.tripleamperr
		scans.tripleampec=scans.tripleampc
		scans.tripleampecerr=scans.tripleampcerr
		scans.triplephase=scans.triplephasec
		scans.triplephaseerr=scans.triplephasecerr
;
; 		When reading VLTI data produced by, e.g. pndrs, 
;		convert station names to standard OYSTER format.
		if systemid eq 'VLTI' then begin
		vlt_stn=strmid(stationids,1,2)+'_'
		for n=0,genconfig.numsid-1 do begin
		  j=where(strpos(vlt_stn,genconfig.stationid(n)) eq 0,count)
		  if j(0) gt 7 then begin
			  genconfig.stationid(n)=stationids(j)
		  endif else if j(0) gt 3 then begin
;			  Shift index to replace proper UT designation
			  genconfig.stationid(n)=stationids(j-4)
		  endif
		endfor
		if count gt 0 then begin
		  print,'Station names converted.'
		  get_stationtable,update=0
		endif
		for ob=0,genconfig.numoutbeam-1 do begin
;		for n=0,genconfig.numbaseline(ob)-1 do begin
		for n=0,MB-1 do begin	; pndrs: n(BL) < n(n-1) with all tr!
		j=where(strpos(vlt_stn,strmid(genconfig.baselineid(n,ob),0,3)) $
		    eq 0)
		k=where(strpos(vlt_stn,strmid(genconfig.baselineid(n,ob),4,3)) $
		    eq 0)
	    	j=j(0) & k=k(0)
		if j ne -1 and k ne -1 then begin
		  if j le 7 then j=j-4; shift index to replace label for UTs
		  if k le 7 then k=k-4; shift index to replace label for UTs
		  genconfig.baselineid(n,ob)=stationids(j)+'-'+stationids(k)
		endif
		endfor
		endfor
		bl_ids=genconfig.baselineid
		genconfig.configid=genconfig.configid+'_' $
;		Use vlt_stn string (could be ambiguous)
;		+strjoin(genconfig.stationid(0:num_sid-1),'-')
;		Use baseline string (unique), but could be very loooong...
;		+strjoin(genconfig.baselineid,'_')
;		Use unique station string (shorter than baseline string)
;		+unique_station_config(genconfig.baselineid)
;		Use unique station string (with blanks removed)
		+unique_station_config( $
			bl_ids(where(strlen(strtrim(bl_ids)) gt 0)))

;		Remove possible occurrences of double-underscores
		words=nameparse(genconfig.configid,'__')
		genconfig.configid=strjoin(words,'_')
		endif	; systemid='VLTI'
;
; 		If reading PIONIER files with mode FREE, 
;		allocate OB for each baseline (enable for special cases)
		if n_elements(pionierfree) eq 0 then pionierfree=0
		if pionierfree and strpos(genconfig.configid,'PIONIER') eq 0 $
		 and genconfig.numspecchan(0) eq 1 then set_pionierfree
;
;		Buffer scandata if more than one data set is in the OIFITS file
		if num_sets gt 1 then begin
;			Next 6 lines commented 2016-02-24
;			g=geninfo(i)
;			struct_assign,genconfig,g
;			geninfo(i)=g
;			g=geoinfo(i)
;			struct_assign,geoparms,g
;			geoinfo(i)=g
			storenight
		endif else begin
;			Next 2 lines commented 2016-02-24
;			geninfo=genconfig
;			geoinfo=geoparms
		endelse
	endif
endfor
;
; Complete some information in StarTable...
; PIONIER data reduction: which stars were used as calibrators?
oidiam_file=file_search('.','????-??-??_oiDiam.fits')
if strlen(oidiam_file(0)) gt 0 then begin
	i=where(strpos(oidiam_file,date) ge 0,count)
	if count gt 0 then begin
		t=mrdfits(oidiam_file(i(0)),1,header)
		index=where(strpos(t.target,'INTERNAL') lt 0)
		t=t(index)
		if n_elements(t) eq n_elements(startable) then begin
			index=where(t.iscal eq 1,count)
			if count gt 0 then startable(index).bflag='C'
		endif
	endif
endif
; If any star name is an OYSTER Star ID, fill HDN or HIC numbers
for i=0,n_elements(startable)-1 do begin
	if valid_cat(startable(i).starid) then begin
;		id=long(strmid(startable(i).starid,3,6))
;		There are cases such as HIP_5300
		id=long(strmid(strjoin(nameparse(startable(i).starid, $
			['_','-'])),3,6))
		case strmid(startable(i).starid,0,3) of
		'HDN':startable(i).hdn=id
		'HIP':startable(i).hic=id
		'BSC':startable(i).bsc=id
		'FKV':startable(i).fkv=id
		else:
		endcase
	endif
endfor
; Is this still needed?
startable=startable(uniq(startable.starid))
;
pipefile=fitshparse(h,'PIPEFILE',/quiet)+''
if strpos(pipefile,'RAW_INT') ge 0 or $ 
   strpos(pipefile,'CAL_INT') ge 0 then begin
;
; 	MATISSE flux calibration: counts -> Jansky
	if strpos(genconfig.configid,'MATISSE_disabled') ge 0 and $
		  genconfig.numoutbeam eq 1 then begin
;	Remove instrumental response first: photometry -> photometry!
	matisse_response; Comment this for matisse_cal_response
;
;	...then calibrate in Jy: photometry -> photometryc!
;	...unless we process data to compute the zerocounts
	if strpos(pwd(),'InputZeroCounts') lt 0 then begin
	restore,!oyster_dir+'source/matisse/zerocounts.xdr'
	if strpos(genconfig.configid,'-L_') ge 0 then begin
		band='L'
		ins_mode=fitshparse(h,'INS.DIL.ID')
	endif
	if strpos(genconfig.configid,'-N_') ge 0 then begin
		band='N'
		ins_mode=fitshparse(h,'INS.DIN.ID')
	endif
	if strpos(genconfig.stationid(0),'A') ge 0 then array='AT'
	if strpos(genconfig.stationid(0),'U') ge 0 then array='UT'
	det_mode=fitshparse(h,'DET.READ.CURNAME')
	det_dit=strmid( $
		strtrim(string(double(fitshparse(h,'DET.SEQ1.DIT'))),1),0,10)
	tpl_nexp=strtrim(string(fix(fitshparse(h,'TPL.NEXP'))),1)
	index=where(zerocounts.band eq band and $
		    zerocounts.array eq array and $
		    zerocounts.ins_mode eq ins_mode and $
		    zerocounts.det_mode eq det_mode and $
		    zerocounts.det_dit eq det_dit and $
		    zerocounts.tpl_nexp eq tpl_nexp,count)
	coeff_factor=1.
	if count eq 0 then begin
		if tpl_nexp eq '40' then begin
		index=where(zerocounts.band eq band and $
		    zerocounts.array eq array and $
		    zerocounts.ins_mode eq ins_mode and $
		    zerocounts.det_mode eq det_mode and $
		    zerocounts.det_dit eq det_dit and $
		    zerocounts.tpl_nexp eq '52',count)
	    	if count gt 0 and band eq 'L' then coeff_factor=1./2
	    	if count gt 0 and band eq 'N' then coeff_factor=2
		endif
		if tpl_nexp eq '52' then begin
		index=where(zerocounts.band eq band and $
		    zerocounts.array eq array and $
;		    zerocounts.ins_mode eq ins_mode and $
;		    zerocounts.det_mode eq det_mode and $
;		    zerocounts.det_dit eq det_dit and $
		    zerocounts.tpl_nexp eq '40',count)
	    	if count gt 0 and band eq 'L' then coeff_factor=2
	    	if count gt 0 and band eq 'N' then coeff_factor=1./2
		endif
	endif
; 	Compute median of coefficients for each telescope (siderostat)
	if count ne 0 then begin
;	Found zerocount coefficients
	zcs=fltarr(genconfig.numsid)
	if count eq 1 then zcs=zerocounts(index).zcs*coeff_factor $
	else for i=0,genconfig.numsid-1 do $
		zcs(i)=median(zerocounts(index).zcs(i)*coeff_factor)
; 	Compute estimated L or N band fluxes in Jy for each telescope
	fln=matisse_zerocounts(zcs=zcs,tm=tm)
; 	Calibrate scans.photometry
	i=0	; Single-star observation only
	for k=0,n_elements(scans)-1 do begin
	for j=0,genconfig.numsid-1 do begin
;	 Compute total flux and normalize to target flux in L or N band
	 f_band=total(median(reform(scans(k).photometry(j,0,*)),5)*tm)/total(tm)
	 if f_band gt 0 then begin
	 if fitshistory(h,'mat_mergeAllOiFits') then f_band=f_band/4
	 scans(k).photometryc(j,0,*)=scans(k).photometry(j,0,*)*fln(i,j)/f_band
	 scans(k).photometrycerr(j,0,*)=scans(k).photometryerr(j,0,*)/f_band
	 endif else scans(k).photometrycerr(j,0,*)=-1
	endfor
	endfor
	endif	; Data calibrated in Jy
;
	endif	; Not called to compile zero counts
	endif	; MATISSE
endif	; Reflex pipeline product
;
; Read spectrum if available, single night and target only
if n_elements(sed_file) eq 0 then sed_file='sed.dat'
sed_file=file_search(sed_file)
if strlen(sed_file) gt 0 and num_sets eq 1 $
			  and n_elements(startable) eq 1 then begin
	status=dc_read_free(sed_file,lambda,flux,/col)
	if status eq 0 then begin
		print,'Read spectrum (SED) from file '+sed_file
        	for k=0,genconfig.numoutbeam-1 do begin
        	nc=genconfig.numspecchan(k)
        	for i=0,n_elements(scans)-1 do scans(i).photometry(0,k,0:nc-1) $
	                =spline(lambda,flux,genconfig.wavelength(0:nc-1,k)*1e6)
       		endfor
	endif
	scans.photometryc=scans.photometry
endif else begin
;	scans.photometry=0
endelse
;
; Read seeing data from text files, if available (single night only)
; dfits PIONI*.fits | fitsort ISS.AMBI.FWHM.START > r0.txt
; dfits PIONI*.fits | fitsort ISS.AMBI.FWHM.END >> r0.txt
; sort r0.txt > r0.dat
; sed -i '1,2d' r0.dat
; Remove all lines without r0 values or other extra text
if n_elements(r0_file) eq 0 then r0_file='r0.dat'
r0_file=file_search(r0_file)
if strlen(r0_file) gt 0 and num_sets eq 1 then begin
	l=''
	status=dc_read_fixed(r0_file,l,/col,format='(a80)')
	if status eq 0 then begin
		print,'Read r0 data from file '+r0_file
		nl=n_elements(l)
		t=fltarr(nl)
		r0=fltarr(nl)
		ipos=strpos(l(0),'.')
		for k=0,nl-1 do begin
			words=nameparse(l(k))
			t(k)=hms2h(strmid(words(0),ipos+12,12))
			r0(k)=float(words(1))
		endfor
		index=where(r0 gt 0,count)
		if count gt 0 then begin
			t=t(index)
			r0=r0(index)
			index=where(t gt 24-system_config(systemid(0), $
				'MIDNIGHT'),count)
			if count gt 0 then t(index)=t(index)-24
			scans.r0=gsmooth(t,r0,0.02,scans.time/3600)
		endif
	endif
endif else scans.r0=1	; Default 1" r0
; dfits PIONI*.fits | fitsort ISS.AMBI.TAU0.START >> t0.txt
; dfits PIONI*.fits | fitsort ISS.AMBI.TAU0.END >> t0.txt
; sort t0.txt > t0.dat
; sed -i '1,2d' t0.dat
; Remove all lines without t0 values or other extra text
if n_elements(t0_file) eq 0 then t0_file='t0.dat'
t0_file=file_search(t0_file)
if strlen(t0_file) gt 0 and num_sets eq 1 then begin
	l=''
	status=dc_read_fixed(t0_file,l,/col,format='(a80)')
	if status eq 0 then begin
		print,'Read t0 data from file '+t0_file
		nl=n_elements(l)
		t=fltarr(nl)
		t0=fltarr(nl)
		ipos=strpos(l(0),'.')
		for k=0,nl-1 do begin
			words=nameparse(l(k))
			t(k)=hms2h(strmid(words(0),ipos+12,12))
			t0(k)=float(words(1))
		endfor
		index=where(t0 gt 0,count)
		if count gt 0 then begin
			t=t(index)
			t0=t0(index)
			index=where(t gt 24-system_config(systemid(0), $
				'MIDNIGHT'),count)
			if count gt 0 then t(index)=t(index)-24
;			t0=fltarr(NOB,MC,MB)
			r=gsmooth(t,t0,0.02,scans.time/3600)
			for i=0,n_elements(scans)-1 do scans(i).t0=r(i)
		endif
	endif
endif else scans.t0=0.001	; Default 0.001s t0
if strlen(r0_file) eq 0 and strlen(t0_file) eq 0 then begin
;	Alternatively, check if file has header with r0/t0/PWV values
	d=mrdfits(fitsfile,0,header)
	s_start=float(fitshparse(header,'ISS.AMBI.FWHM.START',/quiet))
	s_end=float(fitshparse(header,'ISS.AMBI.FWHM.END',/quiet))
	if s_start gt 0 and s_end gt 0 then scans.r0=(s_start+s_end)/2.
	t_start=float(fitshparse(header,'ISS.AMBI.TAU0.START',/quiet))
	t_end=float(fitshparse(header,'ISS.AMBI.TAU0.END',/quiet))
	if t_start gt 0 and t_end gt 0 then scans.t0(*)=(t_start+t_end)/2.
	p_start=float(fitshparse(header,'ISS.AMBI.IWV.START',/quiet))
	p_end=float(fitshparse(header,'ISS.AMBI.IWV.END',/quiet))
	if p_start gt 0 and p_end gt 0 then scans.pwv=(p_start+p_end)/2.
endif
;
; Test validity of station coordinates
uvw_bck=scans.uvw
uv=reform(scans(*).uvw(*,*,*,0:1))
calcastrom
uv_c=reform(scans(*).uvw(*,*,*,0:1))
;
n=size(uv,/dim)
print,'Checking uv-coordinates for case '+string(n_elements(n),format='(i1)')
case n_elements(n) of
	1:	begin
		u=uv(*)
		v=uv(*)
		u_c=uv_c(*)
		v_c=uv_c(*)
		end
	2:	begin
		u=uv(*,0)
		v=uv(*,1)
		u_c=uv_c(*,0)
		v_c=uv_c(*,1)
		end
	3:	begin
		u=uv(*,*,0)
		v=uv(*,*,1)
		u_c=uv_c(*,*,0)
		v_c=uv_c(*,*,1)
		end
	4:	begin
		u=uv(*,*,*,0)
		v=uv(*,*,*,1)
		u_c=uv_c(*,*,*,0)
		v_c=uv_c(*,*,*,1)
		end
	5:	begin
		u=uv(*,*,*,0,*)
		v=uv(*,*,*,1,*)
		u_c=uv_c(*,*,*,0,*)
		v_c=uv_c(*,*,*,1,*)
		end
endcase
index=where(u lt 0,count)
if count gt 0 then begin
	u(index)=-u(index)
	v(index)=-v(index)
endif
index=where(u_c lt 0,count)
if count gt 0 then begin
	u_c(index)=-u_c(index)
	v_c(index)=-v_c(index)
endif
index=where(v lt 0,count)
if count gt 0 then begin
	u(index)=-u(index)
	v(index)=-v(index)
endif
index=where(v_c lt 0,count)
if count gt 0 then begin
	u_c(index)=-u_c(index)
	v_c(index)=-v_c(index)
endif
uv=[[u],[v]]
uv_c=[[u_c],[v_c]]
index=where(uv ne 0)
;
if keyword_set(uvcheck) then begin
	uvcheck_file=basename(fitsfile)+'_uvcheck.txt'
	openw,unit,uvcheck_file,/get_lun
	printf,unit,fitsfile
endif
print,'Median absolute deviation in uv [m]:', $
	medianve(abs(uv(index)-uv_c(index)))*median(genconfig.wavelength)
if keyword_set(uvcheck) then $
printf,unit,'Median absolute deviation in uv [m]:', $
	medianve(abs(uv(index)-uv_c(index)))*median(genconfig.wavelength)
print,'Maximum absolute deviation in uv [m]:', $
	max(abs(uv(index)-uv_c(index)))*mean(genconfig.wavelength)
if keyword_set(uvcheck) then $
printf,unit,'Maximum absolute deviation in uv [m]:', $
	max(abs(uv(index)-uv_c(index)))*mean(genconfig.wavelength)
;
p=100*medianve((uv(index)-uv_c(index))/uv(index),e)
e=e*100
print,'Median percentage deviation in uv: '+ $
		string(p,format='(f6.3)')+'% +/- '+string(e,format='(f7.2)')+'%'
if keyword_set(uvcheck) then $
printf,unit,'Median percentage deviation in uv: '+ $
		string(p,format='(f6.3)')+'% +/- '+string(e,format='(f7.2)')+'%'
;
if p gt 1 or e gt 1 then begin
	print,'Warning: station coordinates inconsistent with uvw coordinates!'
	print,'The uv coordinates of the data file will be used.'
endif
scans.uvw=uvw_bck
if keyword_set(uvcheck) then free_lun,unit
;
; Update the menu options in the DataInterferometry GUI
if n_elements(opmenua_wid) eq 0 then opmenua_wid=0
if n_elements(opmenup_wid) eq 0 then opmenup_wid=0
if opmenua_wid ne 0 and opmenup_wid ne 0 then begin
	types=['Vsqr','TAmp','Both','None']
	sds=3
	if ds_options.v2 then sds=0
	if ds_options.ta then sds=1
	if ds_options.v2 and ds_options.ta then sds=2
	widget_control,opmenua_wid,set_droplist_select=sds
	types=['VPhs','TPhs','Both','None']
	sds=3
	if ds_options.vp then sds=0
	if ds_options.tp then sds=1
	if ds_options.vp and ds_options.tp then sds=2
	widget_control,opmenup_wid,set_droplist_select=sds
	print,'Interferometric data type selection updated.'
endif
;
; Make sure amoeba knows about this file, unless this procedure was called
; by load_interferometry with a temporary file.
chafiles=''
if fitsfile ne 'temp_do_not_keep.fits' then $
oiffiles=fitsfile
;
; Name of file is used by OYSTER for auxilliary files
if fitsfile ne 'temp_do_not_keep.fits' then $
	hds_file_stub=basename(fitsfile) else $
	hds_file_stub=genconfig.date+'_' $
		+system_id(systemid)+'_' $
		+genconfig.configid
;
; Enforce strict formatting of ConfigId (currently disabled)
genconfig.configid=strict_configid(genconfig.configid)
;
end
;-------------------------------------------------------------------------------
function find_oiextn,fitsfile,extname_in
;
; Find extension number for named binary table.
;
forward_function fxpar
;
exten=1
errmsg=''
fxbopen,unit,fitsfile,exten,header,errmsg=errmsg
;
while (errmsg eq '')  do begin
	extname = strtrim(fxpar(header,'EXTNAME'),2) 
	fxbclose,unit,errmsg=errmsg
	if extname eq extname_in then return,exten
	exten=exten+1
	fxbopen,unit,fitsfile,exten,header,errmsg=errmsg
endwhile
fxbclose,unit,errmsg=errmsg
;
end
;-------------------------------------------------------------------------------
pro check_oiextns,fitsfile,n_oivis,n_oivis2,n_oit3,n_oiflux
;
forward_function fxpar
;
n_oivis=0
n_oivis2=0
n_oit3=0
n_oiflux=0
;
exten=1
errmsg=''
fxbopen,unit,fitsfile,exten,header,errmsg=errmsg
print,errmsg
while (errmsg eq '')  do begin
	extname = strtrim(fxpar(header,'EXTNAME'),2 )
	if (extname eq 'OI_VIS') then n_oivis=n_oivis+1
	if (extname eq 'OI_VIS2') then n_oivis2=n_oivis2+1
	if (extname eq 'OI_T3') then n_oit3=n_oit3+1
	if (extname eq 'OI_FLUX') then n_oiflux=n_oiflux+1
	fxbclose,unit,errmsg=errmsg
	exten=exten+1
	fxbopen,unit,fitsfile,exten,header,errmsg=errmsg
endwhile
fxbclose,unit,errmsg=errmsg
;
end
;-------------------------------------------------------------------------------
pro check_oifits,oiarray,oitarget,oiwavelength, $
	oivis1=oivis,oivis2=oivis2,oit3=oit3,oiflux=oiflux
;
; Fix problems with OIFITS files written by various softwares.
; Currently, missing or broken oiarray tables are fixed in 
; get_oifits.
;
; Fix VLTI OB definition errors 
if n_elements(oivis2) gt 0 then begin
if oivis2(0).date_obs eq '2024-01-08T07:40:29'  $
	and strpos(oivis2(0).insname,'GRAVITY') ge 0 $
	and oitarget(0).target eq 'HD4660' then begin
		oitarget(0).target = 'HDN094660'
endif
if oivis2(0).date_obs eq '2024-03-10T03:05:19'  $
	and strpos(oivis2(0).insname,'GRAVITY') ge 0 $
	and oitarget(0).target eq 'HD4660' then begin
		oitarget(0).target = 'HDN094660'
endif
if oivis2(0).date_obs eq '2025-02-05T05:43:14'  $
	and strpos(oivis2(0).insname,'GRAVITY') ge 0 $
	and oitarget(0).target eq 'HD9466' then begin
		oitarget(0).target = 'HDN094660'
endif
endif
;
; Make sure StarTable only lists StarIDs for which data are available
flag=0
if n_elements(oivis2) gt 0 then begin
	oitarget_ids=intarr(n_elements(oitarget.target_id))+1
	for i=0,n_elements(oitarget.target_id)-1 do begin
		index=where(oivis2.target_id eq oitarget(i).target_id,n)
		if n eq 0 then oitarget_ids(i)=0
	endfor
	oitarget=oitarget(where(oitarget_ids eq 1))
endif else if n_elements(oivis) gt 0 then begin
	oitarget_ids=intarr(n_elements(oitarget.target_id))+1
	for i=0,n_elements(oitarget.target_id)-1 do begin
		index=where(oivis.target_id eq oitarget(i).target_id,n)
		if n eq 0 then oitarget_ids(i)=0
	endfor
	oitarget=oitarget(where(oitarget_ids eq 1))
endif else if n_elements(oit3) gt 0 then begin
	oitarget_ids=intarr(n_elements(oitarget.target_id))+1
	for i=0,n_elements(oitarget.target_id)-1 do begin
		index=where(oit3.target_id eq oitarget(i).target_id,n)
		if n eq 0 then oitarget_ids(i)=0
	endfor
	oitarget=oitarget(where(oitarget_ids eq 1))
endif else if n_elements(oiflux) gt 0 then begin
	oitarget_ids=intarr(n_elements(oitarget.target_id))+1
	for i=0,n_elements(oitarget.target_id)-1 do begin
		index=where(oiflux.target_id eq oitarget(i).target_id,n)
		if n eq 0 then oitarget_ids(i)=0
	endfor
	oitarget=oitarget(where(oitarget_ids eq 1))
endif
;
; Make sure target IDs start with 1 and are consecutive
if max(oitarget.target_id) gt n_elements(oitarget) then begin
	j=max(oitarget.target_id)
	num_target=n_elements(oitarget)
	old_ids=oitarget.target_id
	new_ids=indgen(num_target)+1
	for i=1,num_target do begin
		if n_elements(oivis2) ne 0 then begin
			index=where(oivis2.target_id eq oitarget(i-1).target_id)
			oivis2(index).target_id=i+j
		endif
		if n_elements(oivis) ne 0 then begin
			index=where(oivis.target_id eq oitarget(i-1).target_id)
			oivis(index).target_id=i+j
		endif
		if n_elements(oit3) ne 0 then begin
			index=where(oit3.target_id eq oitarget(i-1).target_id)
			oit3(index).target_id=i+j
		endif
		if n_elements(oiflux) ne 0 then begin
			index=where(oiflux.target_id eq oitarget(i-1).target_id)
			oiflux(index).target_id=i+j
		endif
	endfor
	if n_elements(oivis2) ne 0 then oivis2.target_id= $
					oivis2.target_id-j
	if n_elements(oivis) ne 0 then oivis.target_id= $
					oivis.target_id-j
	if n_elements(oit3) ne 0 then oit3.target_id= $
					oit3.target_id-j
	if n_elements(oiflux) ne 0 then oiflux.target_id= $
					oiflux.target_id-j
	oitarget.target_id=new_ids
endif
;
; Target IDs must conform to OYSTER scheme
for i=0,n_elements(oitarget)-1 do begin
	r=strsplit(oitarget(i).target,'HIP',/extract) & r=r(0)
	if isnumeric(r) then oitarget(i).target='HIP'+string(r,format='(i6.6)')
	if isnumeric(r) then continue
	r=strsplit(oitarget(i).target,'HIC',/extract) & r=r(0)
	if isnumeric(r) then oitarget(i).target='HIC'+string(r,format='(i6.6)')
	if isnumeric(r) then continue
	r=strsplit(oitarget(i).target,'HDN',/extract) & r=r(0)
	if isnumeric(r) then oitarget(i).target='HDN'+string(r,format='(i6.6)')
	if isnumeric(r) then continue
	r=strsplit(oitarget(i).target,'SAO',/extract) & r=r(0)
	if isnumeric(r) then oitarget(i).target='SAO'+string(r,format='(i6.6)')
	if isnumeric(r) then continue
	r=strsplit(oitarget(i).target,'FKV',/extract) & r=r(0)
	if isnumeric(r) then oitarget(i).target='FKV'+string(r,format='(i4.4)')
	if isnumeric(r) then continue
	r=strsplit(oitarget(i).target,'BSC',/extract) & r=r(0)
	if isnumeric(r) then oitarget(i).target='BSC'+string(r,format='(i4.4)')
	r=strsplit(oitarget(i).target,'FLN',/extract) & r=r(0)
	if isnumeric(r) then oitarget(i).target='FLN'+string(r,format='(i4.4)')
	if isnumeric(r) then continue
	r=cri_vlti(oitarget(i).target)
	if strlen(r) eq 0 then begin
		print,'Error (check_oifits): could not find ID for target ', $
			oitarget(i).target,' in catalogs!'
		print,'Please add HD ID in oyster/starbase/vlti.hdn.'
		return
	endif else oitarget(i).target=r
endfor
;
; NaNs happened in oivis2
if n_elements(oivis2) ne 0 then begin
;	index=where(finite(*oivis2.vis2err) eq 0,count)
;	index=where(finite((*oivis2.vis2err) ne 0)
;	if count gt 0 then oivis2.vis2err(index)=-1
endif

;
; Fix issues with GRAVITY pipeline products
fixed=0
if n_elements(oiflux) ne 0 then begin
;	Remove time stamp jitter within P1 and P2 series
;	P1/P2 are the two polarisations in SPLIT mode
	index=where(strpos(oiflux.insname,'GRAVITY_SC_P1') ge 0,count)
	if count gt 0 then oiflux(index).time=mean(oiflux(index).time)
	if count gt 0 then oiflux(index).mjd=mean(oiflux(index).mjd)
	index=where(strpos(oiflux.insname,'GRAVITY_SC_P2') ge 0,count)
	if count gt 0 then oiflux(index).time=mean(oiflux(index).time)
	if count gt 0 then oiflux(index).mjd=mean(oiflux(index).mjd)
	index=where(strpos(oiflux.insname,'GRAVITY_FT_P1') ge 0,count)
	if count gt 0 then oiflux(index).time=mean(oiflux(index).time)
	if count gt 0 then oiflux(index).mjd=mean(oiflux(index).mjd)
	index=where(strpos(oiflux.insname,'GRAVITY_FT_P2') ge 0,count)
	if count gt 0 then oiflux(index).time=mean(oiflux(index).time)
	if count gt 0 then oiflux(index).mjd=mean(oiflux(index).mjd)
;
; 	Pipeline adds _1,2,... for "extra" scans, are not new spectrometers
	index=where(strpos(oiflux.insname,'GRAVITY_SC_') ge 0,count)
	if count gt 0 then oiflux(index).insname='GRAVITY_SC'
	index=where(strpos(oiflux.insname,'GRAVITY_FT_') ge 0,count)
	if count ge 1 then begin
	if count gt 0 then oiflux(index).insname='GRAVITY_FT'
	endif
;
;	Remove time difference between SC and FT
	index_sc=where(strpos(oiflux.insname,'GRAVITY_SC') ge 0,count)
	index_ft=where(strpos(oiflux.insname,'GRAVITY_FT') ge 0,count)
	if count ge 1 then begin
		td=median(oiflux(index_sc).time-oiflux(index_ft).time)
		if td lt 0.1 then begin
			oiflux(index_ft)=oiflux(index_sc)
			fixed=1
		endif
	endif
endif
if n_elements(oivis2) ne 0 then begin
;	Remove time stamp jitter within P1 and P2 series
	index=where(strpos(oivis2.insname,'GRAVITY_SC_P1') ge 0,count)
	if count gt 0 then oivis2(index).time=mean(oivis2(index).time)
	if count gt 0 then oivis2(index).mjd=mean(oivis2(index).mjd)
	index=where(strpos(oivis2.insname,'GRAVITY_SC_P2') ge 0,count)
	if count gt 0 then oivis2(index).time=mean(oivis2(index).time)
	if count gt 0 then oivis2(index).mjd=mean(oivis2(index).mjd)
	index=where(strpos(oivis2.insname,'GRAVITY_FT_P1') ge 0,count)
	if count gt 0 then oivis2(index).time=mean(oivis2(index).time)
	if count gt 0 then oivis2(index).mjd=mean(oivis2(index).mjd)
	index=where(strpos(oivis2.insname,'GRAVITY_FT_P2') ge 0,count)
	if count gt 0 then oivis2(index).time=mean(oivis2(index).time)
	if count gt 0 then oivis2(index).mjd=mean(oivis2(index).mjd)
;
; 	Pipeline adds _1,2,... for "extra" scans, are not new spectrometers
	index=where(strpos(oivis2.insname,'GRAVITY_SC_') ge 0,count)
	if count gt 0 then oivis2(index).insname='GRAVITY_SC'
	index=where(strpos(oivis2.insname,'GRAVITY_FT_') ge 0,count)
	if count gt 0 then oivis2(index).insname='GRAVITY_FT'
;
;	Remove time difference between SC and FT
	index_sc=where(strpos(oivis2.insname,'GRAVITY_SC') ge 0,count)
	index_ft=where(strpos(oivis2.insname,'GRAVITY_FT') ge 0,count)
	if count ge 1 then begin
		td=median(oivis2(index_sc).time-oivis2(index_ft).time)
		if td lt 0.1 then begin
			oivis2(index_ft)=oivis2(index_sc)
			fixed=1
		endif
	endif
;
endif
if n_elements(oivis) ne 0 then begin
;	Remove time stamp jitter within P1 and P2 series
	index=where(strpos(oivis.insname,'GRAVITY_SC_P1') ge 0,count)
	if count gt 0 then oivis(index).time=mean(oivis(index).time)
	if count gt 0 then oivis(index).mjd=mean(oivis(index).mjd)
	index=where(strpos(oivis.insname,'GRAVITY_SC_P2') ge 0,count)
	if count gt 0 then oivis(index).time=mean(oivis(index).time)
	if count gt 0 then oivis(index).mjd=mean(oivis(index).mjd)
	index=where(strpos(oivis.insname,'GRAVITY_FT_P1') ge 0,count)
	if count gt 0 then oivis(index).time=mean(oivis(index).time)
	if count gt 0 then oivis(index).mjd=mean(oivis(index).mjd)
	index=where(strpos(oivis.insname,'GRAVITY_FT_P2') ge 0,count)
	if count gt 0 then oivis(index).time=mean(oivis(index).time)
	if count gt 0 then oivis(index).mjd=mean(oivis(index).mjd)
;
; 	Pipeline adds _1,2,... for "extra" scans, are not new spectrometers
	index=where(strpos(oivis.insname,'GRAVITY_SC_') ge 0,count)
	if count gt 0 then oivis(index).insname='GRAVITY_SC'
	index=where(strpos(oivis.insname,'GRAVITY_FT_') ge 0,count)
	if count gt 0 then oivis(index).insname='GRAVITY_FT'
;
;	Remove time difference between SC and FT
	index_sc=where(strpos(oivis.insname,'GRAVITY_SC') ge 0,count)
	index_ft=where(strpos(oivis.insname,'GRAVITY_FT') ge 0,count)
	if count ge 1 then begin
		td=median(oivis(index_sc).time-oivis(index_ft).time)
		if td lt 0.1 then begin
			oivis(index_ft)=oivis(index_sc)
			fixed=1
		endif
	endif
;
endif
if n_elements(oit3) ne 0 then begin
;	Remove time stamp jitter within P1 and P2 series
	index=where(strpos(oit3.insname,'GRAVITY_SC_P1') ge 0,count)
	if count gt 0 then oit3(index).time=mean(oit3(index).time)
	if count gt 0 then oit3(index).mjd=mean(oit3(index).mjd)
	index=where(strpos(oit3.insname,'GRAVITY_SC_P2') ge 0,count)
	if count gt 0 then oit3(index).time=mean(oit3(index).time)
	if count gt 0 then oit3(index).mjd=mean(oit3(index).mjd)
	index=where(strpos(oit3.insname,'GRAVITY_FT_P1') ge 0,count)
	if count gt 0 then oit3(index).time=mean(oit3(index).time)
	if count gt 0 then oit3(index).mjd=mean(oit3(index).mjd)
	index=where(strpos(oit3.insname,'GRAVITY_FT_P2') ge 0,count)
	if count gt 0 then oit3(index).time=mean(oit3(index).time)
	if count gt 0 then oit3(index).mjd=mean(oit3(index).mjd)
;
; 	Pipeline adds _1,2,... for "extra" scans, are not new spectrometers
	index=where(strpos(oit3.insname,'GRAVITY_SC_') ge 0,count)
	if count gt 0 then oit3(index).insname='GRAVITY_SC'
	index=where(strpos(oit3.insname,'GRAVITY_FT_') ge 0,count)
	if count gt 0 then oit3(index).insname='GRAVITY_FT'
;
;	Remove time difference between SC and FT
	index_sc=where(strpos(oit3.insname,'GRAVITY_SC') ge 0,count)
	index_ft=where(strpos(oit3.insname,'GRAVITY_FT') ge 0,count)
	if count ge 1 then begin
		td=median(oiflux(index_sc).time-oiflux(index_ft).time)
		if td lt 0.1 then begin
			oit3(index_ft)=oit3(index_sc)
			fixed=1
		endif
	endif
endif
if fixed then print,'Fixed GRAVITY time stamp jitter and scans data structure!'
;
; 	Update OIWAVELENGTH table
	index=where(strpos(oiwavelength.insname,'GRAVITY_SC_P1') ge 0,count1)
	if count gt 0 then oiwavelength(index).insname='GRAVITY_SC'
	index=where(strpos(oiwavelength.insname,'GRAVITY_SC_P2') ge 0,count2)
	if count gt 0 then oiwavelength(index).insname='GRAVITY_SC'
	index=where(strpos(oiwavelength.insname,'GRAVITY_FT_P1') ge 0,count3)
	if count gt 0 then oiwavelength(index).insname='GRAVITY_FT'
	index=where(strpos(oiwavelength.insname,'GRAVITY_FT_P2') ge 0,count4)
	if count gt 0 then oiwavelength(index).insname='GRAVITY_FT'
	oiwavelength=oiwavelength(uniq(oiwavelength.insname))
	if count1+count2+count3+count4 gt 0 then $
		print,'Ignored GRAVITY P1/P2 split mode assignments!'
; 	Remove FT data from GRAVITY dual observations
	index=where(strpos(oiwavelength.insname,'GRAVITY_SC') ge 0,count)
	if count gt 0 then oiwavelength=oiwavelength(index)
	index=where(strpos(oivis2.insname,'GRAVITY_SC') ge 0,count)
	if count gt 0 then begin
		target_id=unique(oivis2(index).target_id)
		oivis2=oivis2(index)
	endif
	if n_elements(oivis2) ne 0 then $
	index=where(strpos(oivis2.insname,'GRAVITY_SC') ge 0,count) else count=0
	if count gt 0 then oivis2=oivis2(index)
	if n_elements(oivis) ne 0 then $
	index=where(strpos(oivis.insname,'GRAVITY_SC') ge 0,count) else count=0
	if count gt 0 then oivis=oivis(index)
	if n_elements(oit3) ne 0 then $
	index=where(strpos(oit3.insname,'GRAVITY_SC') ge 0,count) else count=0
	if count gt 0 then oit3=oit3(index)
	if n_elements(oiflux) ne 0 then $
	index=where(strpos(oiflux.insname,'GRAVITY_SC') ge 0,count) else count=0
	if count gt 0 then oiflux=oiflux(index)
;
; Fix issue with merge_oidata: oiarray contains duplicates
oi=uniq(oiarray.sta_index,sort(oiarray.sta_index))
if n_elements(oi) lt n_elements(oiarray) then $
	oiarray=oiarray(oi)
;
; Fix issues with Beauty contest data
;
dates=''
arrnames=''
insnames=''
if n_elements(oivis2) ne 0 then begin
	dates=unique(strmid(oivis2.date_obs,0,10))
	arrnames=unique(strtrim(oivis2.arrname,2))
	insnames=unique(strtrim(oivis2.insname,2))
endif
;
; ...check that int_time > 0 if flag=FALSE (2014 HgMn PIONIER data)
false=byte('F') & false=false(0)
for i=0,n_elements(oivis)-1 do begin
	eindex=where(byte(*oivis(i).flag) eq false,ecount)
	if ecount gt 0 and oivis(i).int_time eq 0 then oivis(i).int_time=1
endfor
for i=0,n_elements(oivis2)-1 do begin
	eindex=where(byte(*oivis2(i).flag) eq false,ecount)
	if ecount gt 0 and oivis2(i).int_time eq 0 then oivis2(i).int_time=1
endfor
for i=0,n_elements(oit3)-1 do begin
	eindex=where(byte(*oit3(i).flag) eq false,ecount)
	if ecount gt 0 and oit3(i).int_time eq 0 then oit3(i).int_time=1
endfor
for i=0,n_elements(oiflux)-1 do begin
	eindex=where(byte(*oiflux(i).flag) eq false,ecount)
	if ecount gt 0 and oiflux(i).int_time eq 0 then oiflux(i).int_time=1
endfor
;
; ...Array name
fixed=0
if oiarray(0).arrname eq 'NPOI_2004-04-12' then begin
	oiarray.arrname='NPOI'
	fixed=1
endif
if oiarray(0).arrname eq 'NPOI_2004-07-19' then begin
	oiarray.arrname='NPOI'
	fixed=1
endif
if fixed then print,'Warning: Beauty contest data fixed (NPOI)!'
;
; ...CHARA stations need to have 3 letters
for i=0,n_elements(oiarray)-1 do begin
	if strpos(oiarray(i).arrname,'CHARA') eq 0 then begin
		tel_arm=strmid(oiarray(i).tel_name,0,1)
		sta_num=strmid(oiarray(i).tel_name,1,1)
		oiarray(i).tel_name=tel_arm+'0'+sta_num
		sta_arm=strmid(oiarray(i).sta_name,0,1)
		sta_num=strmid(oiarray(i).sta_name,1,1)
		oiarray(i).sta_name=sta_arm+'0'+sta_num
	endif
endfor
;
; ...missing keywords
fixed=0
tags=tag_names(oitarget)
for i=0,n_tags(oitarget)-1 do begin
	if tags(i) eq 'SYSVEL' then begin
		j=where(finite(oitarget.sysvel) eq 0,count)
		if count gt 0 then oitarget(j).sysvel=0
		if count gt 0 then fixed=1
	endif
	if tags(i) eq 'PMRA' then begin
		j=where(finite(oitarget.pmra) eq 0,count)
		if count gt 0 then oitarget(j).pmra=0
		if count gt 0 then fixed=1
	endif
	if tags(i) eq 'PMRA_ERR' then begin
		j=where(finite(oitarget.pmra_err) eq 0,count)
		if count gt 0 then oitarget(j).pmra_err=0
		if count gt 0 then fixed=1
	endif
	if tags(i) eq 'PMDEC' then begin
		j=where(finite(oitarget.pmdec) eq 0,count)
		if count gt 0 then oitarget(j).pmdec=0
		if count gt 0 then fixed=1
	endif
	if tags(i) eq 'PMDEC_ERR' then begin
		j=where(finite(oitarget.pmdec_err) eq 0,count)
		if count gt 0 then oitarget(j).pmdec_err=0
		if count gt 0 then fixed=1
	endif
	if tags(i) eq 'PARALLAX' then begin
		j=where(finite(oitarget.parallax) eq 0,count)
		if count gt 0 then oitarget(j).parallax=0
		if count gt 0 then fixed=1
	endif
	if tags(i) eq 'PARA_ERR' then begin
		j=where(finite(oitarget.para_err) eq 0,count)
		if count gt 0 then oitarget(j).para_err=0
		if count gt 0 then fixed=1
	endif
endfor
if fixed then print,'Warning: missing OI_TARGET header keywords added!'
;
; ...issue with JWST data (written by Joel):
; 	each epoch is in a new extension
fixed=0
if dates(0) eq '2021-10-20' and arrnames(0) eq 'JWST_SIM' then begin
	print,'Fixing JWST_SIM MASKING data...'
	arrname='JWST_SIM'
	apertures=system_config(arrname,'STATIONS')
	ipos=strlen(shortest(insnames))
	oivis2.arrname=arrname
	oiarray.tel_name=apertures
	oiarray.sta_name=apertures
	oivis2.insname='7Holes'+strmid(oivis2.insname,ipos,3)
	oit3.arrname=arrname
	oit3.insname='7Holes'+strmid(oit3.insname,ipos,3)
	if n_elements(oivis) ne 0 then begin
		oivis.arrname=arrname
		oivis.insname='7Holes'+strmid(oivis.insname,ipos,3)
	endif
	oiwavelength.insname='7Holes'
;	Merge identical wavelength tables
	oiwavelength=oiwavelength(0)
	*oiwavelength(0).eff_band=1e-6
	fixed=1
endif
if fixed then print,'Warning: SAM data from JWST_SIM fixed!'
;
; ...issue with SAM data from T-ReCS (written by John):
; 	each epoch is in a new extension
fixed=0
if dates(0) eq '2005-01-03' and arrnames(0) eq 'KECK-1_MASKING' then begin
	print,'Fixing KECK-1_MASKING data...'
	arrname='T-ReCS'
	apertures=system_config(arrname,'STATIONS')
	ipos=strlen(shortest(insnames))
	oivis2.arrname=arrname
	oivis2.insname='7Holes'+strmid(oivis2.insname,ipos,3)
	oit3.arrname=arrname
	oit3.insname='7Holes'+strmid(oit3.insname,ipos,3)
	if n_elements(oivis) ne 0 then begin
		oivis.arrname=arrname
		oivis.insname='7Holes'+strmid(oivis.insname,ipos,3)
	endif
	oiwavelength.insname=arrname
;	Merge identical wavelength tables
	oiwavelength=oiwavelength(0)
	*oiwavelength(0).eff_band=1e-6
	fixed=1
endif
if fixed then print,'Warning: SAM data from T-ReCS fixed!'
;
; ... issue with SAM data from NACO (written by Peter):
; 	arrname, insname, time not set
fixed=0
if dates(0) eq '2012-03-12' and arrnames(0) eq 'Fizeau Masking' then begin
	print,'Fixing NACO Fizeau masking data...'
	arrname='NACO'
	apertures=system_config(arrname,'STATIONS')
	num_tel=n_elements(apertures)
	oiarray=replicate(oiarray,num_tel)
	oiarray.arrname=arrname
	oiarray.tel_name=apertures
	oiarray.sta_name=apertures
	oiarray.sta_index=indgen(num_tel)
	case nint(*oiwavelength.eff_wave*1e8) of
		218:begin
		    band='_Ks'
		    *oiwavelength.eff_band=0.4e-6
		    end
		380:begin
		    band='_Lp'
		    *oiwavelength.eff_band=0.6e-6
		    end
	endcase
	insname='7Holes'+band
	oivis2.arrname=arrname
	oivis2.insname=insname
	oivis2.time=hms2h(strmid(oivis2.date_obs,11,13))*3600
	oivis2.date_obs=dates(0)
	oit3.arrname=arrname
	oit3.insname=insname
	oit3.time=hms2h(strmid(oit3.date_obs,11,13))*3600
	oit3.date_obs=dates(0)
	if n_elements(oivis) ne 0 then begin
		oivis.arrname=arrname
		oivis.insname=insname
		oivis.time=hms2h(strmid(oivis.date_obs,11,13))*3600
		oivis.date_obs=dates(0)
	endif
	oiwavelength.insname=insname
	if n_elements(oivis2) ne 0 then begin
	if total(oivis2.sta_index) eq 0 then begin
;		If the data were edited, we need to restore an original
;		full data set to match the (u,v) coordinates and identify IDs
		case nint(*oiwavelength.eff_wave*1e8) of
			218: n=168
			380: n=147 ; new file has 252, but 252/21=12 => OK!
		endcase
		index0=indgen(n)
		index=intarr(n_elements(oivis2))
		if n_elements(oivis2) lt n then begin
			case nint(*oiwavelength.eff_wave*1e8) of
				218:	begin
;					Restore ucoord0 and vcoord0
					restore,'2012-03-12_Ks.xdr'
					a=-0.0085
					end
				380:	begin
					restore,'2012-03-12_Lp.xdr'
					a=-0.0085
					a=0
					end
			endcase
;			Rotate coordinates to match
			ucoord=oivis2.ucoord*cos(a)-oivis2.vcoord*sin(a)
			vcoord=oivis2.vcoord*cos(a)+oivis2.ucoord*sin(a)
			for i=0,n_elements(index)-1 do begin
				k=where(abs(ucoord0-ucoord(i)) lt 0.001 and $
					abs(vcoord0-vcoord(i)) lt 0.001,count)
				if count eq 1 then begin
					index(i)=k
					oivis2(i).sta_index=sta_index0(*,k)
				endif
			endfor
		endif else begin
			nb=(num_tel-1)*num_tel/2
			epochs=strcompress(oivis2.date_obs $
				+oitarget(oivis2.target_id).target $
				+string(oivis2.time),/remove_all)
			ns=n_elements(unique(epochs))
			i=0
			j=1
			for n=0,ns-1 do begin
			for k=0,nb-1 do begin
			oivis2(k+n*nb).sta_index=[i,j]
			j=j+1
			if j eq num_tel then begin
				i=i+1
				j=i+1
			endif
			if i eq num_tel-1 then begin
				i=0
				j=1
			endif
			endfor
			endfor
			ucoord0=oivis2.ucoord
			vcoord0=oivis2.vcoord
			sta_index0=oivis2.sta_index
			case nint(*oiwavelength.eff_wave*1e8) of
			218:save,ucoord0,vcoord0,sta_index0, $
				filename='2012-03-12_Ks.xdr'
			380:save,ucoord0,vcoord0,sta_index0, $
				filename='2012-03-12_Lp.xdr'
			endcase
		endelse
	endif
	endif
	if n_elements(oit3) ne 0 then begin
	if total(oit3.sta_index) eq 0 then begin
		for k=0,n_elements(oit3)-1 do begin
			i=where(oivis2.ucoord eq oit3(k).u1coord $
			    and oivis2.vcoord eq oit3(k).v1coord)
			oit3(k).sta_index(0)=oivis2(i).sta_index(0)
			oit3(k).sta_index(1)=oivis2(i).sta_index(1)
			i=where(oivis2.ucoord eq oit3(k).u2coord $
			    and oivis2.vcoord eq oit3(k).v2coord)
			oit3(k).sta_index(2)=oivis2(i).sta_index(1)
		endfor
	endif
	endif
	fixed=1
endif
if fixed then print,'Warning: Fizeau masking data from NACO fixed!'
;
; ...issue with SAM data from NACO (written by Sylvestre):
; 	all time stamps identical to MJD
fixed=0
if arrnames(0) eq 'NACO' and insnames(0) eq 'SAM' then begin
	if total(oivis2.time-oivis2.mjd) eq 0 then $
		oivis2.time=(oivis2.mjd mod 1)*86400
	oivis2.date_obs=strmid(oivis2.date_obs,0,10)
	oit3.date_obs=strmid(oit3.date_obs,0,10)
	fixed=1
endif
if fixed then print,'Warning: SAM data from NACO fixed!'
;
;...unknown issue?
fixed=0
if arrnames(0) eq 'VLTI' and strpos(insnames(0),'SPECTRO') eq 0 then begin
	oivis2.time=oivis2.time-43200.0
	index=where(oivis2.time lt 0,count)
	if count gt 0 then oivis2.time(index)=oivis2.time(index)+86400.0
	if n_elements(oivis) ne 0 then begin
	oivis.time=oivis.time-43200.0
	index=where(oivis.time lt 0,count)
	if count gt 0 then oivis.time(index)=oivis.time(index)+86400.0
	endif
	if n_elements(oit3) ne 0 then begin
	oit3.time=oit3.time-43200.0
	index=where(oit3.time lt 0,count)
	if count gt 0 then oit3.time(index)=oit3.time(index)+86400.0
	endif
	fixed=1
endif
if fixed then print,'Warning: Beauty contest data fixed (SPECTRO)!'
;
;...Contest 2020, different arrays have same time stamps
fixed=0
if dates(0) eq '2019-06-02' and $
   n_elements(unique(oiarray.sta_index)) eq 34 and $
   n_elements(oivis2) eq 6*3*3 and $
   strpos(oiwavelength(0).insname,'MATISSE') eq 0 then begin
   	n=n_elements(oivis2)
	bl=strarr(n)
	sta_index=string(oivis2.sta_index)
   	for i=0,n-1 do bl(i)=strjoin(sta_index(*,i))
	oivis2.time=oivis2.time $
		+indgen(n_elements(oivis2.time))/n_elements(unique(bl))
	oivis2.mjd=oivis2.mjd $
		+(indgen(n_elements(oivis2.mjd))/n_elements(unique(bl)))/86400.
	oivis.time=oivis.time $
		+indgen(n_elements(oivis.time))/n_elements(unique(bl))
	oivis.mjd=oivis.mjd $
		+(indgen(n_elements(oivis.mjd))/n_elements(unique(bl)))/86400.
   	n=n_elements(oit3)
	tr=strarr(n)
	sta_index=string(oit3.sta_index)
   	for i=0,n-1 do tr(i)=strjoin(sta_index(*,i))
	oit3.time=oit3.time $
		+indgen(n_elements(oit3.time))/n_elements(unique(tr))
	oit3.mjd=oit3.mjd $
		+(indgen(n_elements(oit3.time))/n_elements(unique(tr)))/86400.
	fixed=1
endif
if fixed then print,'Warning: fixed Beauty contest data 2020 (time stamps)!'
;
;...Contest 2020, oiarray contains more stations than needed
fixed=0
reduce_oiarray=0
if dates(0) eq '2022-01-18' then begin
if n_elements(oivis2) gt 0 then begin
	sta_index=unique(oivis2.sta_index)
	oiv_index=oivis2.sta_index
	if n_elements(whereindex(sta_index,oiarray.sta_index)) lt $
	   n_elements(oiarray) then begin
		reduce_oiarray=1
		for i=1,n_elements(sta_index) do begin
			j=where(oiv_index(0,*) eq sta_index(i-1),n_j) 
			k=where(oiv_index(1,*) eq sta_index(i-1),n_k) 
			if n_j ge 1 then oivis2(j).sta_index(0)=i
			if n_k ge 1 then oivis2(k).sta_index(1)=i
		endfor
	endif
endif
if n_elements(oivis) gt 0 then begin
	sta_index=unique(oivis.sta_index)
	oiv_index=oivis.sta_index
	if n_elements(whereindex(sta_index,oiarray.sta_index)) lt $
	   n_elements(oiarray) then begin
		reduce_oiarray=1
		for i=1,n_elements(sta_index) do begin
			j=where(oiv_index(0,*) eq sta_index(i-1),n_j) 
			k=where(oiv_index(1,*) eq sta_index(i-1),n_k) 
			if n_j ge 1 then oivis(j).sta_index(0)=i
			if n_k ge 1 then oivis(k).sta_index(1)=i
		endfor
	endif
endif
if n_elements(oit3) gt 0 then begin
	sta_index=unique(oit3.sta_index)
	oit_index=oit3.sta_index
	if n_elements(whereindex(sta_index,oiarray.sta_index)) lt $
	   n_elements(oiarray) then begin
		reduce_oiarray=1
		for i=1,n_elements(sta_index) do begin
			j=where(oit_index(0,*) eq sta_index(i-1),n_j) 
			k=where(oit_index(1,*) eq sta_index(i-1),n_k) 
			l=where(oit_index(2,*) eq sta_index(i-1),n_l) 
			if n_j ge 1 then oit3(j).sta_index(0)=i
			if n_k ge 1 then oit3(k).sta_index(1)=i
			if n_l ge 1 then oit3(l).sta_index(2)=i
		endfor
	endif
endif
if reduce_oiarray then begin
	oiarray=oiarray(sta_index-1)
	oiarray.sta_index=indgen(n_elements(oiarray))+1
	fixed=1
endif
if fixed then print,'Warning: cleaned up Beauty contest data (OIARRAY)!'
endif
;
end
;-------------------------------------------------------------------------------
pro get_oitarget,fitsfile
;
; Read just revision 2 OI_TARGET binary table from OIFITS file.
;
common Tables,ScanTable,BGTable,StationTable
common StarBase,StarTable,Notes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if n_elements(fitsfile) eq 0 then fitsfile='oitarget.fits'
files=file_search(fitsfile)
if strlen(files(0)) eq 0 then begin
	print,'***Error(GET_OITARGET): file not found!'
	return
endif
;
read_oidata,fitsfile,oiarray,oitarget,oiwavelength,oivis,oivis2,oit3,/inventory
;
; Assemble startable
if n_elements(oitarget) eq 0 then begin
	starid=''
	read,starid, $
	prompt='Target table not present, please enter star ID [HDN123456]: ', $
	get_startable,starid
endif else begin
	targets=strcompress(oitarget.target,/remove_all)
	create_startable,targets
	startable.name=targets
	startable.ra=oitarget(*).raep0/15
	startable.dec=oitarget(*).decep0
	startable.rae=oitarget(*).ra_err
	startable.dece=oitarget(*).dec_err
	startable.rv=oitarget(*).sysvel
	startable.pmra=oitarget(*).pmra
	startable.pmdec=oitarget(*).pmdec
	startable.pmrae=oitarget(*).pmra_err
	startable.pmdece=oitarget(*).pmdec_err
	startable.px=oitarget(*).parallax
	startable.pxe=oitarget(*).para_err
	startable.spectrum=oitarget(*).spectyp
	startable_id=intarr(max(oitarget.target_id)+1)
	for i=0,n_elements(startable)-1 do begin
		startable_id(oitarget(i).target_id)=i
		if valid_cat(targets(i)) eq 1 then starid=targets(i) $
					      else starid=cri_vlti(targets(i))
		if strlen(starid) eq 0 then begin
			startable(i).starid='OBJ'+esoid(startable(i).ra, $
							startable(i).dec)
		endif else begin
			startable(i).starid=starid
		endelse
	endfor
; 	Update for OI_TARGET revision 2
	startable.diameter=oitarget(*).diameter
	startable.omega=oitarget(*).omega
	startable.vsini=oitarget(*).vsini
	startable.tilt=oitarget(*).tilt
	startable.rapa=oitarget(*).rapa
	startable.mass=oitarget(*).mass
	startable.teff=oitarget(*).teff
	startable.logg=oitarget(*).logg
	startable.zexp=oitarget(*).zexp
	for i=0,n_elements(startable)-1 do begin
		startable(i).sed=*oitarget(i).flux
		startable(i).wvl=*oitarget(i).wave
		startable(i).img=*oitarget(i).img
	endfor
;
	read_catalogs	; We had a case with bad coordinates in the OIFITS file
endelse
;
end
;-------------------------------------------------------------------------------
pro get_mark3data,file,status
;
; Read MarkIII .cal file.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Mark3,mark3_baselines,mark3_configs,mark3_cat,mark3_bsc
common ModelFit,parameters,ds_options
;
; Note: except for a header line containing date, baseline number, and the
; three channel wavelengths (all numbers), Mark III .cal files consist of
; eleven columns of numeric data. 
;
SystemId='Mark3'
;
; Mark III filter configurations
n_config=13
waves=[ $
[ 486.0,                486.0,                486.0], $
[ 656.0,                656.0,                550.0], $
[ 657.0,                656.0,                550.0], $
[ 700.0,                450.2,                550.0], $
[ 754.0,                623.0,                712.0], $
[ 799.2,                450.2,                550.1], $
[ 800.0,                450.2,                500.0], $
[ 800.0,                450.2,                550.0], $
[ 800.0,                451.8,                500.0], $
[ 800.0,                500.0,                550.0], $
[ 800.0,                500.0,                656.0], $
[ 800.0,                500.1,                550.0], $
[ 800.0,                656.0,                550.0]]
combinerids=indgen(n_config)
;
openr,unit,file,/get_lun,error=status
if status ne 0 then begin
	print,'***Error(GET_MARK3DATA): error opening file ',file,'!'
	return
endif
;
; Read header line with date, baseline, and wavelength information
mark3_ch1=700.0
readf,unit,mark3_date,mark3_baseline,mark3_ch2,mark3_ch3,mark3_ch4
;
max_scan=300
time=dblarr(max_scan)
mark3_stars=intarr(max_scan)
v2=fltarr(max_scan)
vc2=fltarr(max_scan)
vce2=fltarr(max_scan)
v3=fltarr(max_scan)
vc3=fltarr(max_scan)
vce3=fltarr(max_scan)
v4=fltarr(max_scan)
vc4=fltarr(max_scan)
vce4=fltarr(max_scan)
i=0
while not eof(unit) do begin
	readf,unit,t,id,d2,dc2,dce2,d3,dc3,dce3,d4,dc4,dce4
;	FK5 1817 is not a known star, and FK5 2001 is Nova Cygni
	if id ne 1817 and id ne 2001 then begin
	time(i)=t
	mark3_stars(i)=id
	v2(i)=d2
	vc2(i)=dc2
	vce2(i)=dce2
	v3(i)=d3
	vc3(i)=dc3
	vce3(i)=dce3
	v4(i)=d4
	vc4(i)=dc4
	vce4(i)=dce4
	i=i+1
	endif else print,'Warning: FK5 1817 and 2001 cannot be converted!'
endwhile
num_scan=i
free_lun,unit
;
parsemark3,mark3_date,y,m,d
Date=constrictordate(y,m,d)
;
NumSid=2
NumOutBeam=2
maxBaseline=1
maxSpecChan=2
maxNumTriple=10
maxConfig=1
GenConfig=allocgenconfig(NumSid,NumOutBeam,maxNumTriple, $
	maxBaseline,maxSpecChan,maxConfig)
;
; if abs(mark3_ch3-450) lt 2 then GenConfig.BeamCombinerId=1 $
;			   else GenConfig.BeamCombinerId=2
d=total(abs(waves-[mark3_ch2,mark3_ch3,mark3_ch4]#(fltarr(n_config)+1)),1)
; Zeta Aur data have slightly different w.l. so we don't look for d=0
index=where(d eq min(d) and d le 5,count)	
if count eq 0 then begin
	print,'***Error(GET_MARK3DATA): unknown channel configuration!'
	status=-1
	return
endif
GenConfig.BeamCombinerId=combinerids(index)
;
GenConfig.Date=Date
GenConfig.NumSid=NumSid
GenConfig.Diameter=0.05
GenConfig.NumOutBeam=NumOutBeam
GenConfig.NumBaseline=[1,1]
GenConfig.NumSpecChan=[2,2]
GenConfig.refstation=1
;
GenConfig.Wavelength(0,0)=mark3_ch1/1.d9
GenConfig.Wavelength(0,1)=mark3_ch2/1.d9
GenConfig.Wavelength(1,0)=mark3_ch3/1.d9
GenConfig.Wavelength(1,1)=mark3_ch4/1.d9
;
GenConfig.ChanWidth(0,0)=25.0/1.d9
GenConfig.ChanWidth(0,1)=25.0/1.d9
GenConfig.ChanWidth(1,0)=25.0/1.d9
GenConfig.ChanWidth(1,1)=25.0/1.d9
;
GenConfig.BaselineId(0,0)=mark3_baselines(mark3_baseline-1)
GenConfig.BaselineId(0,1)=mark3_baselines(mark3_baseline-1)
GenConfig.StationId(0)=strmid(GenConfig.BaselineId(0,0),0,3)
GenConfig.StationId(1)=strmid(GenConfig.BaselineId(0,1),4,3)
;
; Prepare star IDs
stars='FKV'+string(mark3_stars,format='(i4.4)')
; Replace Mark3 FKV numbers >= 1700 <= 2000 with corresponding BSC numbers 
for i=0,n_elements(mark3_cat)-1 do begin
	index=where(stars eq mark3_cat(i),count)
	if count gt 0 then stars(index)=mark3_bsc(i)
endfor
;
scans=replicate(scan(),num_scan)
scans.iscan=indgen(num_scan)+1
scans.starid=stars(0:num_scan-1)
scans.time=time(0:num_scan-1)*3600
scans.int_time=75	; standard scan length, not stored in CAL file
;
scans.VisSq(1,0,0)=v2(0:num_scan-1)
scans.VisSqC(1,0,0)=vc2(0:num_scan-1)
index=where(vc2 eq 2.0,count)
if count gt 0 then vce2(index)=-2
scans.VisSqErr(1,0,0)=vce2(0:num_scan-1)
scans.VisSqCErr(1,0,0)=vce2(0:num_scan-1)
;
scans.VisSq(0,1,0)=v3(0:num_scan-1)
scans.VisSqC(0,1,0)=vc3(0:num_scan-1)
index=where(vc3 eq 2.0,count)
if count gt 0 then vce3(index)=-2
scans.VisSqErr(0,1,0)=vce3(0:num_scan-1)
scans.VisSqCErr(0,1,0)=vce3(0:num_scan-1)
;
scans.VisSq(1,1,0)=v4(0:num_scan-1)
scans.VisSqC(1,1,0)=vc4(0:num_scan-1)
index=where(vc4 eq 2.0,count)
if count gt 0 then vce4(index)=-2
scans.VisSqErr(1,1,0)=vce4(0:num_scan-1)
scans.VisSqCErr(1,1,0)=vce4(0:num_scan-1)
;
print,'ScanData read.'
;
; Create ScanTable
Iscan=lindgen(num_scan)+1	; Attention: First scan has ID 1!
ScanId=lindgen(num_scan)	; ScanId of embedded system
StartTime=dblarr(num_scan)+scans.time-abs(scans.int_time)/2
StopTime=dblarr(num_scan)+scans.time+abs(scans.int_time)/2
NumCoh=lonarr(num_scan)
NumIncoh=lonarr(num_scan)
NumPoint=lonarr(num_scan)
Code=lonarr(num_scan)+1
Station=intarr(6,num_scan)+1
StarId=scans.starid
;
MaxSid=2
ScanTable=build_scantable(Iscan,ScanId,StarId,Code,Station, $
			StartTime,StopTime,NumPoint, $
			NumCoh,NumIncoh)
;
print,'Scantable created.'
;
get_startable
get_stationtable
get_geoparms
calcastrom
;
freememory
GenInfo=replicate(GenConfig,1)
GeoInfo=replicate(GeoParms,1)
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
ds_options.i=1	; True: interferometry loaded
ds_options.v2=1
ds_options.ta=0
ds_options.tp=0
;
end
;-------------------------------------------------------------------------------
pro get_ptidata,file,status
;
; Read standard PTI file, e.g., /science/omileo/omileo.?.pti.
; Note that some PTI data files deviate in format and must be read with
; the appropriate procedure in get_ptidata.pro in the same directory.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common ModelFit,parameters,ds_options
;
SystemId='PTI'
;
pti_stars=''
pti_date=''
status=dc_read_free(file,pti_stars,n,mjd,pti_date,ut,delay,wl, $
	vissqc,vissqcerr,vissq,vissqerr,sysvis,sysviserr,n_cal,u,v,ha, $
	/col,ignore=['#'])
if status ne 0 then begin
	print,'***Error(GET_PTIDATA): error reading file ',file,'!'
	return
endif
num_scan=n_elements(ut)
;
parsepti,pti_date,y,m,d
Date=constrictordate(y(0),m(0),d(0))
;
; OYSTER's UT Date has only a single value, hence times are wrt to UT=0
; Check for single night with observations before UT=0, times will be < 0
night_start=24-system_config(systemid(0),'MIDNIGHT')
index=where(ut gt night_start,n)
if n gt 0 then begin
	Date=nextdate(Date)
	ut(index)=ut(index)-24
endif
;
NumSid=2
NumOutBeam=1
maxBaseline=1
maxSpecChan=5
maxNumTriple=10
maxConfig=1
GenConfig=allocgenconfig(NumSid,NumOutBeam,maxNumTriple, $
	maxBaseline,maxSpecChan,maxConfig)
;
if n_elements(unique(ut)) eq n_elements(ut) then GenConfig.BeamCombinerId=1 $
					    else GenConfig.BeamCombinerId=5 
case GenConfig.BeamCombinerId of
	1:nc=1
	5:nc=5
endcase
;
GenConfig.Date=Date
GenConfig.NumSid=NumSid
GenConfig.Diameter=0.4
GenConfig.NumOutBeam=NumOutBeam
GenConfig.NumBaseline=[1]
GenConfig.NumSpecChan=[nc]
GenConfig.refstation=1
;
GenConfig.Wavelength(0,0)=2.2/1.d6
;
GenConfig.ChanWidth(0,0)=0.1/1.d6
;
GenConfig.BaselineId(0,0)='NAS-SAS'
GenConfig.StationId(0)=strmid(GenConfig.BaselineId(0,0),0,3)
GenConfig.StationId(1)=strmid(GenConfig.BaselineId(0,0),4,3)
;
; Prepare star IDs
stars='FKV'+string(cri(long(strmid(pti_stars,3,6)),'hdn-fkv'),format='(i4.4)')
;
scans=replicate(scan(),num_scan)
scans.iscan=indgen(num_scan)+1
scans.starid=stars
scans.time=ut*3600
scans.fdlpos(1)=delay
scans.fdlposerr(1)=1e-6
scans.fdlpos(0)=0
scans.fdlposerr(0)=1e-6
;
scans.VisSq(0,0,0)=vissq
scans.VisSqC(0,0,0)=vissqc
scans.VisSqErr(0,0,0)=vissqerr
scans.VisSqCErr(0,0,0)=vissqcerr
;
print,'ScanData read.'
;
; Create ScanTable
Iscan=lindgen(num_scan)+1	; Attention: First scan has ID 1!
ScanId=lindgen(num_scan)	; ScanId of embedded system
StartTime=dblarr(num_scan)
StopTime=dblarr(num_scan)
NumCoh=lonarr(num_scan)
NumIncoh=lonarr(num_scan)
NumPoint=lonarr(num_scan)
Code=lonarr(num_scan)+1
Station=lonarr(NumSid,Num_Scan)+1	; 1=station in, 0=station out
StarId=scans.starid
;
MaxSid=2
ScanTable=build_scantable(Iscan,ScanId,StarId,Code,Station, $
			StartTime,StopTime,NumPoint, $
			NumCoh,NumIncoh)
;
print,'Scantable created.'
;
get_startable
get_stationtable
get_geoparms
calcastrom
;
freememory
GenInfo=replicate(GenConfig,1)
GeoInfo=replicate(GeoParms,1)
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
ds_options.i=1	; True: interferometry loaded
ds_options.v2=1
ds_options.ta=0
ds_options.tp=0
;
end
;-------------------------------------------------------------------------------
pro get_vincidata,fitsfile
;
; In the VINCI pipeline, the FITS data get written in vndrs_FITS.c
; Currently obsolete.
;
; Notes on VLTI/VINCI configuration
;
;     W
;	   N				J6
;               E			|
;	      			UT3     J5
;     S          			|
;					|
;               UT2			J4
;					|
;					J3	UT4
;
;
;     UT1
;			Lab
;		       123456  Channel (BC Input ID)
; ------------------------|-----------------------------
;			 M16
;
;               M12->                       	DL1
;               M12->                       	DL2
; DL3				<-M12
; DL4				<-M12
;	M12->					DL5         
;	M12->					DL6           
; DL7				<-M12
; DL8				<-M12
; -------------------------|----------------------------
; A0 B0 C0  D0  E0   G0         H0          K0 L0 M0
; |  |  |   |
; A1 B1 C1  |
;    B2 C2  |                           J1
;    B3 C3  D1				|
;    B4     |                       I1  |
;    B5     D2                          J2
;
;                    G1
;
; VINCI light path: M1 (alt-az), M2 (fixed), M3-M4 (compressor), M5 (feed), M12 
; UT Coude path   : M1 (8m), M2 (tip-tilt, controlled by STRAP), M3 (folding to
;		    Nasmyth A), M4 (folding to Coude train), M5,6,7 (Coude),
;		    M8 (deformable, controlled by MACAO), M9 (dichroic), 
;		    M10 (movable), M12
; Guide star at Nasmyth focus has 2' FOV, 20 Hz servo loop;
; with MACAO 0.2 Hz loop for out-of-range error signal prevention
;
; Re-imaging of pupil with UTs: M2, M8, center of tunnel, beam compr., MIDI
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common ModelFit,parameters,ds_options
;
ob=vinci_of_read(fitsfile)
ffl=obj_new('fitsFileList',fitsfile)
;
; SystemId
kvt=ffl->getHeaderKeys('instrume')
SystemId=strcompress(kvt.instrume,/remove_all)
systemid='VLTI'
;
; Date
kvt=ffl->getHeaderKeys('date-obs')
date=strmid(kvt.dateobs,0,10)
;
; Get geoparms
get_geoparms
;
; Read file
ob=vinci_of_read(ffl->files())
;
; For reference
; cn=ob.source_table->columnnames()
;
; Allocate GenConfig
MaxSid=2
MaxOut=2
MaxTriple=10
MaxBase=6
Maxchan=16
MaxConfig=10
genconfig=allocgenconfig(MaxSid,MaxOut,MaxTriple,MaxBase,MaxChan,MaxConfig)
genconfig.date=date
;
; Station IDs
sids=ob.geom_data.sta_name
numsid=n_elements(sids)
stationids=strarr(numsid)+'V  '
for i=0,numsid-1 do begin
	name=stationids(i)
	if strmid(sids(i),0,1) eq 'U' $
		then strput,name,strcompress(sids(i),/remove),0 $
		else strput,name,strcompress(sids(i),/remove),1
	stationids(i)=name
endfor
genconfig.stationid(0:numsid-1)=stationids
genconfig.numsid=numsid
;
; Station coordinates
genconfig.stationcoord(0:2,0:numsid-1)=ob.geom_data.staxyz
;
; Other general configuration data
genconfig.refstation=1
genconfig.delaylineid(0:numsid-1)=ob.ot_data.dly_line
genconfig.bcinputid(0:numsid-1)=ob.ot_data.lab_beam
genconfig.wavelength(0,0)=2.2/1.d6
genconfig.chanwidth(0,0)=0.1/1.d6
genconfig.numspecchan=1
genconfig.numoutbeam=1
genconfig.numbaseline=1
genconfig.baselineid(0,0)=genconfig.stationid(1)+'-'+genconfig.stationid(0)
;
; Stars
stars=strtrim(reform(ob.source_data.source),2)
starids=stars
hic=cri(stars,'HIC')
for i=0,n_elements(stars)-1 do begin
	hic_id=0L
	update=0
	if hic(i) le 0 then begin
		read,hic_id, $
		prompt='Please enter Hipparcos number for '+stars(i)+': '
		update=1
	endif else hic_id=hic(i)
	hic(i)=hic_id
	star_id=string(hic(i),format='(i6.6)')
	if update then $
		spawn,'echo "'+star_id+' '+stars(i)+'" >> ' $
		+!oyster_dir+'starbase/vlti.hic'
	starids(i)='HIP'+star_id
endfor
get_startable,starids
startable.name=stars
;
; Number of scans (batches)
num_scan=ob.nbatch
scans=replicate(scan(),num_scan)
;
ut_days=reform(ob.batches.batch_data.time)
scans.time=(ut_days-long(ut_days))*86400
if scans(0).time gt scans(num_scan-1).time then begin
	index=where(scans.time lt scans(0).time)
	scans(index).time=scans(index).time+86400
	genconfig.date=nextdate(geoparms.date)
endif else genconfig.date=geoparms.date
scans.starid=startable(reform(ob.batches.batch_data.source)-1).starid
;
; Visibility squared first interferometric channel
scans.vissq=reform(ob.batches.batch_data.mu12,(size(scans.vissqc))(1:4))
scans.vissqerr=reform(ob.batches.batch_data.emu12,(size(scans.vissqc))(1:4))
;
; Visibility squared second interferometric channel
; scans.vissq=ob.batches.batch_data.mu22
; scans.vissqerr=ob.batches.batch_data.emu22
;
; Initialize calibrated visibility
scans.vissqc=scans.vissq
scans.vissqcerr=scans.vissqerr
;
; To access batch data
; batch=*ob.batches[0].scan_data
;
; Photometric channels A and B
scans.natcounts=reform(ob.batches.batch_data.photom)
scans.natcountserr=reform(ob.batches.batch_data.photomsd)
;
; Standard deviation of OPD
scans.delayjitter=reform(ob.batches.batch_data.sdopd,(size(scans.delayjitter))(1:3))
scans.delayjittererr=1e-6
;
; uvw
scans(*).uvw(*)=ob.batches.batch_data.uvw/genconfig.wavelength(0)
;
; OPD power spectrum moments
; ob.batches.batch_data.corpower(0) ; integrated power
; ob.batches.batch_data.corpower(1) ; position of the spectral peak (meters)
; ob.batches.batch_data.corpower(2) ; width of the spectral peak (meters)
;
; Create scantable
Iscan=lindgen(num_scan)+1       ; Attention: First scan has ID 1!
ScanId=lindgen(num_scan)        ; ScanId of embedded system
StartTime=scans.time
StopTime=scans.time
NumCoh=lonarr(num_scan)+1
NumIncoh=lonarr(num_scan)+1
NumPoint=reform(ob.batches.batch_data.nscan)
StarId=scans.starid
Code=lonarr(num_scan)+1
Station=lonarr(NumSid,Num_Scan)+1       ; 1=station in, 0=station out
;
ScanTable=build_scantable(Iscan,ScanId,StarId,Code,Station, $
                        StartTime,StopTime,NumPoint, $
                        NumCoh,NumIncoh)
;
print,'Scantable created.'
;
get_startable
get_stationtable
get_geoparms
calcastrom
;
obj_destroy,ob.prihead
obj_destroy,ob.source_table
obj_destroy,ob.ot_table
obj_destroy,ob.geom_table
obj_destroy,ob.batch_table
obj_destroy,ob.fringe_a_table
obj_destroy,ffl
;
freememory
GenInfo=replicate(GenConfig,1)
GeoInfo=replicate(GeoParms,1)
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
ds_options.i=1  ; True: interferometry loaded
ds_options.v2=1
ds_options.ta=0
ds_options.tp=0
;
end
;-------------------------------------------------------------------------------
pro get_amberdata,datatag,stations,starid
;
; The data tag has the following format: date and time of target obs., then cal.
; 20060218T093131.395_cal_20060218T083634.264
; The will be exanded by adding "_*.fits".
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
date=strmid(datatag,0,4)+'-'+strmid(datatag,4,2)+'-'+strmid(datatag,6,2)
time=hms2h(strmid(datatag,9,2)+':'+strmid(datatag,11,2)+':'+strmid(datatag,13,6))
;
systemid='VLTI/AMBER-MR'
fakedata,stations,starids=starid,times=time*3600
;
; Initialize the third output beam (normally K band)
genconfig.wavelength(*,2)=0
genconfig.wavelengtherr(*,2)=0
genconfig.chanwidth(*,2)=0
genconfig.chanwidtherr(*,2)=0
;
; Flag all data because it will be initialized later
scans.vissqerr=-1
scans.vissqcerr=-1
scans.tripleamperr=-1
scans.tripleampcerr=-1
scans.triplephaseerr=-1
scans.triplephasecerr=-1
;
datafiles=file_search(datatag+'_*.fits')
;
tag='wlen.'
w=readfits(datafiles(where(strpos(datafiles,tag) ge 0)))
nc=n_elements(w)
genconfig.numspecchan=nc
genconfig.triplenumchan(2)=nc
genconfig.wavelength(0:nc-1,2)=w
genconfig.wavelengtherr(0:nc-1,2)=1e-9
genconfig.chanwidth(0:nc-1,2)=(max(w)-min(w))/nc
genconfig.chanwidtherr(0:nc-1,2)=1e-9
genconfig.configid=instrument_id(systemid)
;
tag='vis.'
v=readfits(datafiles(where(strpos(datafiles,tag) ge 0)))
tag='visErr.'
e=readfits(datafiles(where(strpos(datafiles,tag) ge 0)))
for l=0,2 do begin
	scans.vissq(2,0:nc-1,l)=v(l,*)^2
	scans.vissqc(2,0:nc-1,l)=v(l,*)^2
	scans.vissqerr(2,0:nc-1,l)=2*v(l,*)*e(l,*)
	scans.vissqcerr(2,0:nc-1,l)=2*v(l,*)*e(l,*)
endfor
;
tag='clos.'
c=readfits(datafiles(where(strpos(datafiles,tag) ge 0)))
scans.triplephase(2,0:nc-1)=c
scans.triplephasec(2,0:nc-1)=c
tag='closErr.'
e=readfits(datafiles(where(strpos(datafiles,tag) ge 0)))
scans.triplephaseerr(2,0:nc-1)=e
scans.triplephasecerr(2,0:nc-1)=e
;
tag='flux1.'
f=readfits(datafiles(where(strpos(datafiles,tag) ge 0)))
scans.photonrate(2,0:nc-1)=f(0,*)
scans.photonrateerr(2,0:nc-1)=0.1
;
calcastrom
;
; Initialization for AMOEBA
freememory
num_config=1
bufferinfo=replicate(nightinfo(),num_config)
geoinfo=replicate(geoparms,num_config)
geninfo=replicate(allocgenconfig(/geninfo),num_config)
geninfo.configid=genconfig.configid
confignum=0
g=geninfo(confignum)
struct_assign,genconfig,g
geninfo(confignum)=g
geoinfo(confignum)=geoparms
storenight
;
end
;-------------------------------------------------------------------------------
pro get_pionierdata
;
; Read all PIONIER data files (OIFITS) in the current directory.
; Merge all data with the same spectral setup, also checking the OI_WAVELNGTH
; tables.
; Output data in OIFITS.
;
; Reduced (i.e., uncalibrated) data
files=findfile('PIONI.*_oidata.fits')
if strlen(files(0)) eq 0 then begin
	files=findfile('????-??-??_*_oidataCalibrated.fits')
	if strlen(files(0)) eq 0 then begin
		print,'No PIONIER files!'
		return
	endif else begin
		print,'Merging calibrated data.'
		prefix='DATE'
	endelse
endif else begin
	print,'Merging uncalibrated data.'
	prefix='PIONI'
endelse
;
n=n_elements(files)
configurations=strarr(n)
shortconfigids=strarr(n)
;
for i=0,n-1 do begin
;
if prefix eq 'PIONI' then begin
	fitsfile=obj_new('fitsfile',files(i))
	prihead=fitsfile->prihead()
;
	insmode1=strtrim(prihead->getpar('INS FILT1 NAME'))
	insmode2=strtrim(prihead->getpar('INS OPTI2 ID'))
	readout=strtrim(prihead->getpar('DET READOUT MODE'),2)
	nreads=strtrim(prihead->getpar('DET SCAN NREADS'),2)
	dprtype=strtrim(prihead->getpar('DPR TYPE'))
;
	insmode=insmode1+'-'+insmode2
	configid=readout+'+'+nreads
;
	station_1=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION1')))
	station_2=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION2')))
	station_3=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION3')))
	station_4=vlti_stationid(strtrim(prihead->getpar('ISS CONF STATION4')))
;
	if strpos(dprtype,'OBJECT') ge 0 then $
	configurations(i)=insmode+'+'+strtrim(string(configid),2) $
				 +'+'+station_1 $
				 +'-'+station_2 $
				 +'-'+station_3 $
				 +'-'+station_4
	shortconfigids(i)=insmode $
				 +'+'+station_1 $
				 +'-'+station_2 $
				 +'-'+station_3 $
				 +'-'+station_4
;
	obj_destroy,fitsfile
endif else begin
	array=mrdfits(files(i),3,h)
	stations=strjoin(array.sta_name,'-')
	wavelength=mrdfits(files(i),2,h)
	words=nameparse(h(where(strpos(h,'INSNAME') eq 0)),'=')
	insmode=strtrim(words(1),2)
	insmode=strmid(insmode,1,strlen(insmode)-2)
	words=nameparse(insmode,'/')
	insmode=strjoin(words,'-')
	strput,insmode,'_',strpos(insmode,'(')
	strput,insmode,'_',strpos(insmode,')')
;
	configurations(i)=strmid(files(i),0,10)+'_'+insmode+stations
endelse
;
endfor
;
; Unique configuration lists
index=where(strlen(configurations) ne 0,count)
; if count eq 0 then return
uconfs=unique([configurations(index)])
;
; Merge files with same configuration
for i=0,n_elements(uconfs)-1 do begin
	outfile=uconfs(i)+'.oifits'
	if prefix eq 'PIONI' then outfile='PIONI.'+uconfs(i)+'.oifits'
	index=where(configurations eq uconfs(i))
	merge_oidata,infiles=files(index),outfile=outfile, $
	 oiarray_out,oitarget_out,oiwavelength_out,oivis_out,oivis2_out,oit3_out
;	Merge wavelength tables as much as possible
	if 0 then begin
	nob=n_elements(oiwavelength_out)
	setups=strarr(nob)
	nch=n_elements(*oiwavelength_out(0).eff_wave)
	wavelengths=fltarr(nch,nob)
	for j=0,nob-1 do begin
		wavelengths(*,j)=*oiwavelength_out(j).eff_wave
		setups(j)=strjoin(string(*oiwavelength_out(j).eff_wave))
	endfor
	usetups=unique(setups,index)
	nuob=n_elements(usetups)
	uwaves=fltarr(nch,nuob)
	uwaves=wavelengths(*,index)
	uwavemeans=fltarr(nch)
	for j=0,nch-1 do uwavemeans(j)=mean(uwaves(j,*))
	for j=0,nch-1 do $
	    wavelengths(j,*)=1e9*abs(wavelengths(j,*)-mean(uwaves(j,*))) ge 1.0
;	Index of table with significantly different wavelength layouts
	index0=where(total(wavelengths,1) eq 0,count0)
	index1=where(total(wavelengths,1) ne 0,count1)
	if count0 eq 0 then index=indgen(nob)
	if count0 eq nob then index=0
	if count0 gt 0 and count1 lt nob then index=[index0(0),index1]
	oiwavelength_out=oiwavelength_out(index)
	put_oifits,outfile
	endif
	print,'Data saved to: '+outfile
endfor
;
end
;-------------------------------------------------------------------------------
pro set_pionierfree
;
; Converts configuration of PIONIER data taken with FREE setup from
; single-spec 6-baseline to 6-spec 1-baseline each. This allows to make
; use of the spectral bandpass calibration for each baseline separately.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Pionier,pionier_spectral_cal
;
systemid='VLTI/PIONIER-FREE'
genconfig.beamcombinerid=beamcombiner_id(systemid)
pionier_spectral_cal='generic_c'
;
; Obtain information on IP channels for each baseline
si=sort(genconfig.stationid(0:genconfig.numsid-1))
path=!oyster_dir+'vlti/pionier/'
file=path+strjoin(genconfig.stationid(si),'-')+'_*.txt'
files=file_search(file)
flag=0
if strlen(files(0)) eq 0 then flag=1
mjds=long(strmid(specname(files),16,5))
mjd=long(date2jd(date,/mjd))
index=where(mjds le mjd,count)
if count eq 0 or flag then begin
	print,'VLTI configuration not found for date '+date+': '+file
	print,'Using bandpass data from OIFITS file instead.'
	return
endif
file=files(index(count-1))
spawn,'cp -f '+path+strtrim(string(mjds(count-1)),1)+'/*.xdr '+path
;
stationid=''
status=dc_read_free(file,/col,stationid,ip,ignore=['!'])
genconfig.bcinputid(si)=ip
;
n=n_elements(scans)
s=scans
g=genconfig
;
r=size(scans.fdlpos,/dim)
numsid=r(0)	; 4
r=size(scans.vissqc,/dim)
numchan=1	; FREE
numbase=1
numout=r(2)	; 6, with one baseline each
r=size(scans.triplephasec,/dim)
numtriple=r(0)
maxconfig=1
;
genconfig=allocgenconfig(numsid,numout,numtriple,numbase,numchan,maxconfig)
genconfig.numsid=numsid
genconfig.numoutbeam=numout
genconfig.numtriple=numtriple
genconfig.numbaseline=numbase
genconfig.numspecchan=numchan
genconfig.stationid=g.stationid
genconfig.stationcoord=g.stationcoord
genconfig.baselineid=g.baselineid
genconfig.date=Date
genconfig.beamcombinerid=g.beamcombinerid
genconfig.diameter=g.diameter
genconfig.wavelength=g.wavelength(0)
genconfig.chanwidth=g.chanwidth(0)
genconfig.bcinputid=g.bcinputid
genconfig.triplebeam=g.triplebase
genconfig.triplenumchan=g.triplenumchan
;
; Copy information from 1-dimensional vectors
scans=replicate(scan(),n)
for i=0,n_tags(scans)-1 do if size(scans.(i),/n_dim) le 1 then scans.(i)=s.(i)
;
; Copy information from multi-dimensional vectors
v=s.vissqc
v=reform(v,numout,1,1,n)
scans.vissqc=v
v=s.vissqcerr
v=reform(v,numout,1,1,n)
scans.vissqcerr=v
uvw=s.uvw
uvw=reform(uvw,numout,1,1,3,n)
scans.uvw=uvw
scans.triplephasec=s.triplephasec
scans.triplephasecerr=s.triplephasecerr
scans.tripleampc=s.tripleampc
scans.tripleampcerr=s.tripleampcerr
;
; The following array indexed by (ob,ch) will contain the filter names
pionier_spectral_cal=strarr(1,genconfig.numoutbeam)
prefix='pionier_free_'
lambda_grid=findgen(150)*2+1500; [nm], H-band
for i=0,genconfig.numoutbeam-1 do begin
	j=where(genconfig.stationid eq strmid(genconfig.baselineid(i),0,3))
	k=where(genconfig.stationid eq strmid(genconfig.baselineid(i),4,3))
	bcid=[genconfig.bcinputid(j),genconfig.bcinputid(k)]
	si=sort(bcid) & bcid=bcid(si)
	filter=prefix+string(bcid(0)*10+bcid(1),format='(i2)')
	pionier_spectral_cal(0,i)=filter
	tm=call_function(filter,lambda_grid)
	genconfig.wavelength(0,i)=(total(lambda_grid*tm)/total(tm))*1e-9
endfor
;
; Set geninfo
geninfo=genconfig
;
; Must recalculate uv-coordinates as spectral bandpasses have changed
calcastrom
;
end
;-------------------------------------------------------------------------------
pro get_xdr,filename,freememory=freememory
;
; Workaround for systems where HDS cannot be installed. Write these data
; with put_xdr.
;
common Hds,path,hds_file_stub
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common ModelFit,parameters,ds_options
;
if n_elements(filename) eq 0 then begin
	print,'***Error(GET_XDR): no file specified!'
	return
endif 
;
restore,filename
;
; This variable is used for naming the HDS output if needed
hds_file_stub=basename(filename) 
;
; The genconfig structure might have changed
g=genconfig
GenConfig=allocGenConfig(genconfig.numsid, $
                         genconfig.numoutbeam, $
                         n_elements(genconfig.triplenumchan), $
                         n_elements(genconfig.baselineid(*,0)), $
                         n_elements(genconfig.wavelength(*,0)), $
                         n_elements(genconfig.config))
if genconfig.numsid ne n_elements(g.numsid) $
or genconfig.numoutbeam ne n_elements(g.numbaseline) then begin
	struct_assign,g,genconfig
endif else begin
	tagnames=tag_names(genconfig)
	for i=0,n_tags(genconfig)-1 do begin
		j=where(tag_names(g) eq tagnames(i),count)
		if count eq 1 then genconfig.(i)=g.(j)
	endfor
endelse
genconfig.diameter=system_config(system_id(systemid),'DIAMETER', $
	stations=genconfig.stationid(0))
;
; Store a copy of GenConfig in GenInfo after clearing it, for use in AMOEBA
; if n_elements(GenInfo) le 1 then GenInfo=replicate(GenConfig,1)
;
; Store a copy of GeoParms in GeoInfo after clearing it, for use in AMOEBA
; if n_elements(GeoInfo) le 1 then GeoInfo=replicate(GeoParms,1)
;
; The scans structure might have changed
s=scans
scans=replicate(scan(),n_elements(scans))
tagnames=tag_names(scans)
for i=0,n_tags(scans)-1 do begin
	j=where(tag_names(s) eq tagnames(i),count)
	if count eq 1 then scans.(i)=s.(j)
endfor
;
; The startable is also restored, but the format might have changed
t=startable
create_startable,t.starid
tagnames=tag_names(startable)
for i=0,n_tags(startable)-1 do begin
	j=where(tag_names(t) eq tagnames(i),count)
	if count eq 1 then startable.(i)=t.(j)
endfor
;
; Remove entries in startable which are not found in scans
ustarids=unique(scans.starid)
index=whereequal(startable.starid,ustarids)
startable=startable(index)
;
; Default is to clear AMOEBA's data buffer and store this night's data
if n_elements(freememory) eq 0 then freememory=1
if freememory then begin
	freememory
	GenInfo=replicate(GenConfig,1)
	GeoInfo=replicate(GeoParms,1)
endif
;
if n_elements(ds_options) eq 0 then ds_options=alloc_ds_options()
ds_options.i=1	; True: interferometry loaded
ds_options.v2=1
ds_options.ta=1
ds_options.tp=1
;
end
;-------------------------------------------------------------------------------
pro get_wdlinedata,opslog
;
; Read a file, e.g., wdline.2005-03-01.ops.log and create data with precise
; delays for OPD model fitting.
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
l=''
status=dc_read_fixed(opslog,l,/col,format='(a300)')
index=where(strpos(l,'POM MEASURE') ge 0 and strpos(l,'J 2000') ge 0,count)
if count eq 0 then begin
	print,'***Error(GET_WDLINEDATA): no POM data'
	return
endif
l=l(index)
n=count
;
ra=fltarr(n)
dec=fltarr(n)
configs=strarr(n)
dates=strarr(n)
times=fltarr(n)
delay=fltarr(n,2)
;
for k=0,n-1 do begin
	i0=strpos(l(k),'MEASURE')
	i1=strpos(l(k),'/')
	command=strmid(l(k),i0,i1-i0)
	r=execute(command)
	w=nameparse(measure)
	configs(k)=strmid(w(0),0,23)
	ra(k)=float(w(1))+float(w(2))/60+float(w(3))/3600
	dec(k)=abs(float(w(4)))+float(w(5))/60+float(w(6))/3600
	if strmid(w(4),0,1) eq '-' then dec(k)=-dec(k)
	dates(k)=w(13)+'-'+w(14)+'-'+w(15)
	times(k)=float(w(16))+float(w(17))/60+float(w(18))/3600
	delay(k,0)=float(w(19))	; Delay of dl1
	delay(k,1)=float(w(20))	; Delay of dl2
endfor
;
index=where(strpos(configs,'-dl') ge 0,count)
if count eq 0 then begin
	print,'***Error(GET_WDLINEDATA): no valid configurations found!'
	return
endif
configs=configs(index)
uconfigs=unique(configs)
print,'Found configurations: ',uconfigs
num_config=n_elements(uconfigs)
;
starids0='OBJ'+esoid(ra,dec)
hdn=cri(starids0,'hdn')
hdu=unique(hdn)
; Only if all targets have an HD number, read that catalog
index=where(hdu eq -1,count)
if count eq 0 then starids='HDN'+string(hdu,format='(i6.6)') $
	      else starids=unique(starids0)
get_startable,starids
index=where(hdu gt 0,count)
if count gt 0 then startable(index).hdn=hdu(index)
; Our HD catalog does not have good positions, so copy (ra,dec) from ops.log
for i=0,n_elements(startable)-1 do begin
	if startable(i).hdn gt 0 then begin
		index=where(hdn eq startable(i).hdn)
		starids0(index)=startable(i).starid
		startable(i).ra=ra(index(0))
		startable(i).dec=dec(index(0))
	endif
endfor
initialized=0
;
FOR iconf=0,num_config-1 DO BEGIN
;
config=uconfigs(iconf)
;
; UTs and ATs generate different config strings
dl1=fix(strmid(config,2,1))
dl2=fix(strmid(config,2+12,1))
ic1=fix(strmid(config,10,1))
ic2=fix(strmid(config,10+12,1))
if strmid(config,3,1) eq 'U' then begin
	s1=strmid(config,3,3)
	s2=strmid(config,15,3)
endif else begin
	s1=strmid(config,3,1)+strmid(config,6,2)
	s2=strmid(config,15,1)+strmid(config,18,2)
endelse
;
index=where(configs eq config,count_obs)
starids1=starids0(index)
dates1=dates(index)
times1=times(index)
delay1=delay(index,*)
date=unique(dates1)
if n_elements(date) gt 1 then begin
	date=date(0)
	index=where(times1 lt 12,count)
	if count gt 0 then times1(index)=times1(index)+24
endif
get_sysconfig,sysid='VLTI/VINCI-MONA'
scans=replicate(scan(),count_obs)
;
scans.starid=starids1
scans.star=starids1
scans.time=times1*3600
scans.fdlpos(0)=delay1(*,0)
scans.fdlpos(1)=delay1(*,1)
scans.fdlposerr=1e-6
;
genconfig.configid=config
genconfig.refstation=1
genconfig.numsid=2
genconfig.stationid(0)=s1
genconfig.stationid(1)=s2
genconfig.numbaseline=1
genconfig.baselineid=s1+'-'+s2
genconfig.delaylineid(0)=dl1
genconfig.delaylineid(1)=dl2
genconfig.bcinputid(0)=ic1
genconfig.bcinputid(1)=ic2
;
if max(abs(scans.time/3600)) gt 24 and max(abs(scans.time/3600)) lt 48 $
	      then genconfig.date=nextdate(geoparms.date) $
	      else genconfig.date=geoparms.date
;
if genconfig.refstation eq 1 $
	then scans.fdlpos(1)=scans.fdlpos(1)-scans.fdlpos(0) $
	else scans.fdlpos(0)=scans.fdlpos(0)-scans.fdlpos(1)
get_scantable
get_stationtable
calcastrom
;
; Initialization for AMOEBA
if not initialized then begin
        freememory
        bufferinfo=replicate(nightinfo(),num_config)
        geoinfo=replicate(geoparms,num_config)
        geninfo=replicate(allocgenconfig(/geninfo),num_config)
        geninfo.configid=uconfigs
        initialized=1
endif
confignum=where(geninfo.configid eq config)
g=geninfo(confignum)
struct_assign,genconfig,g
geninfo(confignum)=g
geoinfo(confignum)=geoparms
storenight
;
ENDFOR
;
end
;-------------------------------------------------------------------------------
pro get_opddata,opslog
;
; Open a VLTI/WDLINE ops.log file with OPD measurements and read it,
; storing the values in the existing scans structure.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
l=''
status=dc_read_fixed(opslog,l,/col,format='(a300)')
index=where(strpos(l,'POM MEASURE') ge 0 and strpos(l,'J 2000') ge 0,count)
if count eq 0 then begin
	print,'***Error(GET_WDLINEDATA): no POM data'
	return
endif
l=l(index)
n=count
;
ra=fltarr(n)
dec=fltarr(n)
config=strarr(n)
dates=strarr(n)
times=fltarr(n)
delay=fltarr(n,2)
;
for k=0,n-1 do begin
	i0=strpos(l(k),'MEASURE')
	i1=strpos(l(k),'/')
	command=strmid(l(k),i0,i1-i0)
	r=execute(command)
	w=nameparse(measure)
	config(k)=strmid(w(0),0,23)
	ra(k)=float(w(1))+float(w(2))/60+float(w(3))/3600
	dec(k)=abs(float(w(4)))+float(w(5))/60+float(w(6))/3600
	if strmid(w(4),0,1) eq '-' then dec(k)=-dec(k)
	dates(k)=w(13)+'-'+w(14)+'-'+w(15)
	times(k)=float(w(16))+float(w(17))/60+float(w(18))/3600
	delay(k,0)=float(w(19))	; Delay of dl1
	delay(k,1)=float(w(20))	; Delay of dl2
endfor
;
if checkdate() ne date then begin
	index=where(times lt 12)
	times(index)=times(index)+24
endif
;
index=where(strpos(config,'-dl') ge 0,count)
if count eq 0 then begin
	print,'***Error(GET_WDLINEDATA): no valid configurations found!'
	return
endif
config=config(index)
uconfig=unique(config)
print,'Found configurations: ',uconfig
index=where(times ge scans(0).time/3600 $
	and times le scans(n_elements(scans)-1).time/3600)
config=unique(config(index))
config=config(0)
print,'Using configuration: ',config
; UTs and ATs generate different config strings
dl1=fix(strmid(config,2,1))
dl2=fix(strmid(config,2+12,1))
ic1=fix(strmid(config,10,1))
ic2=fix(strmid(config,10+12,1))
if strmid(config,3,1) eq 'U' then begin
	s1=strmid(config,3,3)
	s2=strmid(config,15,3)
endif else begin
	s1=strmid(config,3,1)+strmid(config,6,2)
	s2=strmid(config,15,1)+strmid(config,18,2)
endelse
;
for i=0,n_elements(scans)-1 do begin
	dra=abs(ra-scans(i).ra)*15
	ddec=abs(dec-scans(i).dec)
	dt=abs(times-scans(i).time/3600)*15
	d=sqrt(dra^2+ddec^2+dt^2)
	index=where(dt eq min(dt))
	delay1=delay(index,0)
	delay2=delay(index,1)
;	if genconfig.refstation eq 1 then delay2=delay2-delay1 $
;				     else delay1=delay1-delay2
	j=where(genconfig.stationid eq s1)
	scans(i).fdlpos(j)=delay1
	j=where(genconfig.stationid eq s2)
	scans(i).fdlpos(j)=delay2
	scans(i).time=times(index)*3600
;
;	The position in scans is the apparent position
	scans(i).ra=ra(index)
	scans(i).dec=dec(index)
endfor
;
if genconfig.refstation eq 1 $
	then scans.fdlpos(1)=scans.fdlpos(1)-scans.fdlpos(0) $
	else scans.fdlpos(0)=scans.fdlpos(0)-scans.fdlpos(1)
;
end
;
;-------------------------------------------------------------------------------
pro get_sed,sed_file,starid=starid
;
; Read a spectrum file to obtain the spectral energy distribution (SED). 
; Units are not checked, Jy are recommended, but could also be erg/cm^2/s.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(scans) eq 0 then begin
	print,'Error: no scan data loaded!'
	return
endif
if n_elements(sed_file) eq 0 then sed_file='sed.dat'
sed_file=file_search(sed_file)
if strlen(sed_file) eq 0 then begin
	print,'Error: SED file not found and/or data not loaded!'
	return
endif else begin
	print,'Read spectrum (SED) from file '+sed_file
	status=dc_read_free(sed_file,lamda,flux,/col)
endelse
if status ne 0 then begin
	print,'Error: file reading error!'
	return
endif
if n_elements(startable) gt 1 and not keyword_set(starid) then begin
	print,'Error: please specify StarId!'
	return
endif
;
; Read spectrum if available, single night and target only
for k=0,genconfig.numoutbeam-1 do begin
	nc=genconfig.numspecchan(k)
	w=genconfig.wavelength(0:nc-1,k)
	si=sort(w)
      	for i=0,n_elements(scans)-1 do begin
		scans(i).photometry(0,k,si)=spline(lamda,flux,w(si))
		scans(i).photometryerr(0,k,si)=scans(i).photometry(0,k,si)/10
	endfor
endfor
scans.photometryc=scans.photometry
;
end
;-------------------------------------------------------------------------------
pro put_oifits,fitsfile,multiple=multiple,starid=starid
;
; Write averaged data from multiple nights and arrays into new OI-FITS format.
; Different beam combiner outputs are treated as different interferometers,
; i.e. instruments of the same array. They all should have their own wavelength
; tables, but in the case of them all being identical (simple_oiwave=1), this 
; procedure writes out only one and stores the different output beams as 
; different baselines.
;
; If there is more than one unique beam combiner output (simple_oiwave=0),
; and a triple is made up of baselines coming from more than one of them,
; the channel wavelengths are taken from the first baseline defined.
; Therefore, this procedure cannot handle the (in any case invalid) combination
; of data from different channels into one triple (simple_oitriple=0).
;
; If keyword Multiple not set, only save the current data.
; Select a specific target if the OIFITS reader only processes single target
; files.
;
forward_function concat_oitable
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if not keyword_set(multiple) then multiple=0
nndex=indgen(n_elements(geninfo))
count=n_elements(geninfo)
if not multiple then nndex=where(geninfo.date eq genconfig.date $
			     and geninfo.configid eq genconfig.configid $
			     and geoinfo.systemid eq geoparms.systemid,count)
;
arrays=unique(geoinfo(nndex).systemid)
num_arrays=n_elements(arrays)
print,'Number of arrays found: ',num_arrays
print,arrays
;
nights=unique(geninfo(nndex).date)
num_nights=n_elements(nights)
print,'Number of nights',num_nights
print,nights
;
num_sets=n_elements(geninfo(nndex))
print,'Number of data sets:',num_sets
;
; Make backup copy of data
startable_bck=startable
scantable_bck=scantable
scans_bck=scans
;
if n_elements(starid) ne 0 then begin
	index=where(startable.starid eq starid,count)
	if count eq 0 then begin
	index=where(startable.name eq starid,count)
	if count eq 0 then begin
		print,'Target not found!'
		return
	endif
	endif
	startable=startable(index)
endif
stars=startable.starid
; mask=intarr(n_elements(stars))
; for i=0,n_elements(stars) -1 do $
; 	mask(where(startable.starid eq stars(i)))=1
; startable=startable(where(mask eq 1))
;
; Assemble target information
define_oitarget,oitarget_unit
oitarget=replicate(oitarget_unit,n_elements(startable))
oitarget(*).target_id = indgen(n_elements(startable))
oitarget(*).target = startable.starid
oitarget(*).raep0 = startable.ra*15
oitarget(*).decep0 = startable.dec
oitarget(*).equinox = 2000.0
oitarget(*).ra_err = startable.rae
oitarget(*).dec_err = startable.dece
oitarget(*).sysvel = startable.rv
oitarget(*).veltyp = "LSR"
oitarget(*).veldef = "OPTICAL"
oitarget(*).pmra = startable.pmra
oitarget(*).pmdec = startable.pmdec
oitarget(*).pmra_err = startable.pmrae
oitarget(*).pmdec_err = startable.pmdece
oitarget(*).parallax = startable.px
oitarget(*).para_err = startable.pxe
oitarget(*).spectyp = startable.spectrum
; Update for OI_TARGET revision 2
; oitarget(*).nwave=n_elements(startable(0).wvl)
; oitarget(*).nx=n_elements(startable(0).img(*,0))
; oitarget(*).ny=n_elements(startable(0).img(0,*))
; for i=0,n_elements(startable)-1 do begin
; oitarget(i).flux=ptr_new(float(startable(i).sed))
; oitarget(i).wave=ptr_new(float(startable(i).wvl))
; oitarget(i).img =ptr_new(float(startable(i).img))
; endfor
;
; Allocate array data
define_oiarray,oiarray_unit
oiarray=replicate(oiarray_unit,long(total(geninfo(nndex).numsid)))
i0=0
;
oiwavelength_ver=1
oivis_ver=1
oivis2_ver=1
oit3_ver=1
;
FOR n=0,num_sets-1 do BEGIN
;
;	Make copy of genconfig for more readibility of code
	g=genconfig
;
	if num_sets gt 1 then $
	loadnight,geninfo(nndex(n)).date, $
		  geoinfo(nndex(n)).systemid, $
		  geninfo(nndex(n)).configid
;
; 	Assemble array information
	geoparms=geoinfo(nndex(n))
	index=indgen(g.numsid)+i0
	oiarray(index).extver=n+1
	array_name=system_id(systemid)
	oiarray(index).arrname=array_name
	oiarray(index).frame="GEOCENTRIC"
;
	RAD=180/pi_circle
	gstia0=ut12gst(0)*15.d0
	ut1utc=geoparms.ut1utc_coeffs(0)
	datutc=geoparms.tai_utc
	r_earth=6378.136d3
	arrayc=(r_earth+geoparms.altitude)* $
		[cos(geoparms.latitude/RAD)*cos(geoparms.longitude/RAD), $
	 	 cos(geoparms.latitude/RAD)*sin(geoparms.longitude/RAD), $
	 	 sin(geoparms.latitude/RAD)]
	oiarray(index).arrayx=arrayc(0)
	oiarray(index).arrayy=arrayc(1)
	oiarray(index).arrayz=arrayc(2)
;
	stations=strarr(g.numsid)
	telescopes=strarr(g.numsid)
	stncoord=dblarr(3,g.numsid)
	for i=0,g.numsid-1 do begin
;		Bring into geocentric system via equatorial
		npoi_x=GenConfig.StationCoord(0,i)
		npoi_y=GenConfig.StationCoord(1,i)
		npoi_z=GenConfig.StationCoord(2,i)
		x=cos(GeoParms.Latitude/RAD)*npoi_z $
		 -sin(GeoParms.Latitude/RAD)*npoi_y
		y=npoi_x
		z=cos(GeoParms.Latitude/RAD)*npoi_y $
		 +sin(GeoParms.Latitude/RAD)*npoi_z
		stncoord(*,i)=equatorial2geocentric([x,y,z])
		stations(i)=g.stationid(i)
		telescopes(i)='SID'+strtrim(string(g.siderostatid(i)),2)
	endfor
	oiarray(index).tel_name=telescopes
	oiarray(index).sta_name=stations
	oiarray(index).sta_index=indgen(g.numsid)
	oiarray(index).diameter=fltarr(g.numsid)+0.12
	for i=0,g.numsid-1 do oiarray(index(i)).staxyz=stncoord(*,i)
;	
	i0=i0+g.numsid
;
;	Assemble data for this night
;
	mask=intarr(n_elements(scans))
	for i=0,n_elements(stars)-1 do begin
		index=where(scans.starid eq stars(i),count)
		if count gt 0 then mask(index)=1
	endfor
	index=where(mask eq 1,count)
	IF count gt 0 then BEGIN
	scans=scans(index)
	scantable=scantable(index)
	target_ids=intarr(n_elements(scans))
	for j=0,n_elements(scans)-1 do $
		target_ids(j)=where(startable.starid eq scans(j).starid)
;	Test if all interferometers (i.e. OBs) have the same wavelengths
	simple_oiwave=1
	if total(g.numspecchan(0:g.numoutbeam-1) $
		-g.numspecchan(0)) ne 0 $
		then simple_oiwave=0
	for j=0,g.numspecchan(0)-1 do begin
		for k=1,g.numoutbeam-1 do $
		if abs(g.wavelength(j,k)-g.wavelength(j,0)) $
		gt 2e-9 then simple_oiwave=0
	endfor
	for i=0,g.numoutbeam-1 do begin
		nw=g.numspecchan(i)
; 		Assemble wavelength information
		define_oiwavelength,oiwavelength_unit,nw=nw
		instrument=instrument_id(system_id(systemid))
		if not simple_oiwave then $
			instrument=instrument+'_OB'+string(i+1,format='(i2.2)')
		if n_elements(oiwavelength) eq 0 then begin
			oiwavelength=oiwavelength_unit
			oiwavelength.extver=oiwavelength_ver
			oiwavelength.insname=instrument
			oiwavelength.eff_wave= $
				ptr_new(float(g.wavelength(0:nw-1,i)))
			oiwavelength.eff_band= $
				ptr_new(float(g.chanwidth(0:nw-1,i)))
			oiwavelength_ver=oiwavelength_ver+1
		endif
		j=where(oiwavelength.insname eq instrument,jn)
		if jn eq 0 then begin
			oiwavelength=concat_oitable(oiwavelength, $
						    oiwavelength_unit)
			k=oiwavelength_ver-1
			oiwavelength(k).extver=oiwavelength_ver
			oiwavelength(k).insname=instrument
			oiwavelength(k).eff_wave= $
				ptr_new(float(g.wavelength(0:nw-1,i)))
			oiwavelength(k).eff_band= $
				ptr_new(float(g.chanwidth(0:nw-1,i)))
			oiwavelength_ver=oiwavelength_ver+1
		endif
;		Loop over baselines in this spectrometer
		for l=0,g.numbaseline(i)-1 do begin
;		Assemble complex visibility data
		jndex=where(scans.complexweight(i,*,*) ne 0,count)
		if count gt 0 then begin
		define_oivis,oivis_unit,nw=nw
		if n_elements(oivis) ne 0 then oivis0=oivis
		oivis=replicate(oivis_unit,n_elements(scans))
		oivis(*).extver=oivis_ver
		oivis_ver=oivis_ver+1
		oivis(*).date_obs=geoinfo(nndex(n)).date
		oivis(*).arrname=array_name
		oivis(*).insname=instrument
		oivis(*).target_id=target_ids
		oivis(*).time=abs(scans.time)
		parsedate,geoparms.date,y,m,d
		oivis(*).mjd=julian(y,m,d)-2400000.5+abs(scans.time)/86400
		oivis(*).int_time=scans.int_time
		for j=0,n_elements(scans)-1 do begin
			amp=abs(scans(j).complexvis(i,0:nw-1,l))
			phi=cphase(scans(j).complexvis(i,0:nw-1,l))*RAD
			oivis(j).visamp=ptr_new(float(amp))
			oivis(j).visphi=ptr_new(float(phi))
			weight=scans(j).complexweight(i,0:nw-1,l)
			sig=weight*0
			jndex=where(weight ne 0,count)
			if count gt 0 then sig(jndex)=1/sqrt(weight(jndex))
                        jndex=where(weight eq 0,count)
                        if count gt 0 then amp(jndex)=1
			oivis(j).visamperr=ptr_new(sig)
			oivis(j).visphierr=ptr_new(sig/amp*RAD)
			oivis(j).ucoord=scans(j).uvw(i,0,l,0)*g.wavelength(0,i)
			oivis(j).vcoord=scans(j).uvw(i,0,l,1)*g.wavelength(0,i)
			si=where(g.stationid eq $
			  strmid(g.baselineid(l,i),0,3))
			sj=where(g.stationid eq $
			  strmid(g.baselineid(l,i),4,3))
			oivis(j).sta_index=[si,sj]
			flags=strarr(nw)+'F'
			fndex=where(scans(j).complexweight(i,0:nw-1,l) eq 0,count)
			if count gt 0 then flags(fndex)='T'
			oivis(j).flag=ptr_new(reform(byte(flags)))
		endfor
		if n ne 0 or i ne 0 or l ne 0 then $
			oivis=concat_oitable(oivis0,oivis)
		endif
		endfor
;		Assemble squared visibility data
		for l=0,g.numbaseline(i)-1 do begin
		define_oivis2,oivis2_unit,nw=nw
		if n_elements(oivis2) ne 0 then oivis0=oivis2
		oivis2=replicate(oivis2_unit,n_elements(scans))
		oivis2(*).extver=oivis2_ver
		oivis2_ver=oivis2_ver+1
		oivis2(*).date_obs=geoinfo(nndex(n)).date
		oivis2(*).arrname=array_name
		oivis2(*).insname=instrument
		oivis2(*).target_id=target_ids
		oivis2(*).time=abs(scans.time)
		parsedate,geoparms.date,y,m,d
		oivis2(*).mjd=julian(y,m,d)-2400000.5+abs(scans.time)/86400
		oivis2(*).int_time=scans.int_time
		for j=0,n_elements(scans)-1 do begin
			oivis2(j).vis2data= $
			ptr_new(scans(j).vissqc(i,0:nw-1,l))
			oivis2(j).vis2err= $
			ptr_new(scans(j).vissqcerr(i,0:nw-1,l))
			oivis2(j).ucoord=scans(j).uvw(i,0,l,0)*g.wavelength(0,i)
			oivis2(j).vcoord=scans(j).uvw(i,0,l,1)*g.wavelength(0,i)
			si=where(g.stationid eq strmid(g.baselineid(l,i),0,3))
			sj=where(g.stationid eq strmid(g.baselineid(l,i),4,3))
			oivis2(j).sta_index=[si,sj]
			flags=strarr(nw)+'F'
			fndex=where(scans(j).vissqcerr(i,0:nw-1,l) lt 0,count)
			if count gt 0 then flags(fndex)='T'
			oivis2(j).flag=ptr_new(reform(byte(flags)))
		endfor
		if n ne 0 or i ne 0 or l ne 0 then $
			oivis2=concat_oitable(oivis0,oivis2)
		endfor
	endfor
;	Assemble triple product data
	for i=0,g.numtriple-1 do begin
		nw=g.triplenumchan(i)
; 		Assemble wavelength information
		define_oiwavelength,oiwavelength_unit,nw=nw
		instrument=instrument_id(system_id(systemid))
		if not simple_oiwave then begin
			if total(g.triplebeam(*,i) $
				-g.triplebeam(0,i)) eq 0 then begin
			instrument=instrument $
				  +'_OB'+string(g.triplebeam(0,i)+1, $
				  		format='(i2.2)')
			simple_oitriple=1
			endif else begin
			instrument=instrument $
				  +'_TR'+string(i+1,format='(i2.2)')
			simple_oitriple=0
			endelse
		endif else simple_oitriple=1
		if n_elements(oiwavelength) eq 0 then begin
			oiwavelength=oiwavelength_unit
			oiwavelength.extver=oiwavelength_ver
			oiwavelength.insname=instrument
;			Assume identical wavelengths on each baseline
			oiwavelength.eff_wave= $
				ptr_new(float(g.wavelength( $
					g.triplechan(0:nw-1,0,i), $
					g.triplebeam(0,i))))
			oiwavelength.eff_band= $
				ptr_new(float(g.chanwidth( $
					g.triplechan(0:nw-1,0,i), $
					g.triplebeam(0,i))))
			oiwavelength_ver=oiwavelength_ver+1
;		Do not create a new wavelength table entry if not needed
		endif else if not simple_oiwave and not simple_oitriple then begin
			oiwavelength=concat_oitable(oiwavelength, $
						    oiwavelength_unit)
			k=oiwavelength_ver-1
			oiwavelength(k).extver=oiwavelength_ver
			oiwavelength(k).insname=instrument
			oiwavelength(k).eff_wave= $
				ptr_new(float(g.wavelength( $
					g.triplechan(0:nw-1,0,i), $
					g.triplebeam(0,i))))
			oiwavelength(k).eff_band= $
				ptr_new(float(g.chanwidth( $
					g.triplechan(0:nw-1,0,i), $
					g.triplebeam(0,i))))
			oiwavelength_ver=oiwavelength_ver+1
		endif
		define_oit3,oit3_unit,nwave=nw
		if n_elements(oit3) ne 0 then oit0=oit3
		oit3=replicate(oit3_unit,n_elements(scans))
		oit3(*).extver=oit3_ver
		oit3_ver=oit3_ver+1
		oit3(*).date_obs=geoinfo(nndex(n)).date
		oit3(*).arrname=array_name
		oit3(*).insname=instrument
		oit3(*).target_id=target_ids
		oit3(*).time=abs(scans.time)
		oit3(*).mjd=julian(y,m,d)-2400000.5+abs(scans.time)/86400
		oit3(*).int_time=scans.int_time
		for j=0,n_elements(scans)-1 do begin
			oit3(j).t3amp= $
				ptr_new(scans(j).tripleampc(i,0:nw-1))
			oit3(j).t3amperr= $
				ptr_new(scans(j).tripleampcerr(i,0:nw-1))
			oit3(j).t3phi= $
				ptr_new(-scans(j).triplephasec(i,0:nw-1)*RAD)	; 			Flip sign (Iota Peg)
			oit3(j).t3phierr= $
				ptr_new(scans(j).triplephasecerr(i,0:nw-1)*RAD)
			baselineids=g.baselineid( $
			   g.triplebase(*,i),g.triplebeam(*,i))
			si=where(g.stationid eq strmid(baselineids(0),0,3))
			sj=where(g.stationid eq strmid(baselineids(0),4,3))
			sk=where(g.stationid eq strmid(baselineids(1),0,3))
			if sk eq si or sk eq sj then $
			sk=where(g.stationid eq strmid(baselineids(1),4,3))
;			The rule requires a triple 12,23,31
			baselinereq=[baselineids(0), $
				g.stationid(sj)+'-'+g.stationid(sk), $
				g.stationid(sk)+'-'+g.stationid(si)]
			idx=intarr(3)	; To reorder the baseline
			fbf=fltarr(3)+1 ; To apply to a reversed baseline
			for l=0,2 do begin
				idx(l)=where(baselineids eq baselinereq(l) or $
				       breve(baselineids) eq baselinereq(l))
				if baselineids(idx(l)) ne baselinereq(l) then $
					fbf(l)=-1
			endfor
			oit3(j).sta_index=[si,sj,sk]
			oit3(j).u1coord=scans(j).uvw( $
					g.triplebeam(idx(0),i),0, $
					g.triplebase(idx(0),i),0) $
				*g.wavelength(0,g.triplebeam(idx(0),i))*fbf(0)
			oit3(j).v1coord=scans(j).uvw( $
					g.triplebeam(idx(0),i),0, $
					g.triplebase(idx(0),i),1) $
				*g.wavelength(0,g.triplebeam(idx(0),i))*fbf(0)
			oit3(j).u2coord=scans(j).uvw( $
					g.triplebeam(idx(1),i),0, $
					g.triplebase(idx(1),i),0) $
				*g.wavelength(0,g.triplebeam(idx(1),i))*fbf(1)
			oit3(j).v2coord=scans(j).uvw( $
					g.triplebeam(idx(1),i),0, $
					g.triplebase(idx(1),i),1) $
				*g.wavelength(0,g.triplebeam(idx(1),i))*fbf(1)
			flags=strarr(nw)+'F'
			fndex=where(scans(j).tripleampcerr(i,0:nw-1) lt 0 $
				 or scans(j).triplephasecerr(i,0:nw-1) lt 0, $
					count)
			if count gt 0 then flags(fndex)='T'
			oit3(j).flag=ptr_new(reform(byte(flags)))
		endfor
		if n ne 0 or i ne 0 then $
			oit3=concat_oitable(oit0,oit3)
		endfor
		ENDIF
;
ENDFOR
;
; Restore data
startable=startable_bck
scantable=scantable_bck
scans=scans_bck
;
; Write OI-FITS data
if n_elements(fitsfile) eq 0 then begin
	if keyword_set(starid) then fitsfile=starid+'_'+Date+'.fits' $
			       else fitsfile=Date+'.fits'
endif
if strlen(extname(fitsfile)) eq 0 then fitsfile=fitsfile+'.fits'
;
write_oidata,fitsfile,oiarray,oitarget,oiwavelength,oivis,oivis2,oit3
print,'OIFITS data saved to: '+fitsfile
;
end
;-------------------------------------------------------------------------------
pro put_oitarget,fitsfile
;
; Write StarTable to disk in OIFITS format, i.e. OI_TARGET table revision 2.
;
common Tables,ScanTable,BGTable,StationTable
common StarBase,StarTable,Notes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
stars=startable.starid
table=startable
mask=intarr(n_elements(startable))
for i=0,n_elements(stars) -1 do $
	mask(where(startable.starid eq stars(i)))=1
startable=startable(where(mask eq 1))
;
; Assemble target information
define_oitarget,oitarget_unit
oitarget=replicate(oitarget_unit,n_elements(startable))
oitarget(*).target_id = indgen(n_elements(startable))
oitarget(*).target = startable.starid
oitarget(*).raep0 = startable.ra*15
oitarget(*).decep0 = startable.dec
oitarget(*).equinox = 2000.0
oitarget(*).ra_err = startable.rae
oitarget(*).dec_err = startable.dece
oitarget(*).sysvel = startable.rv
oitarget(*).veltyp = "LSR"
oitarget(*).veldef = "OPTICAL"
oitarget(*).pmra = startable.pmra
oitarget(*).pmdec = startable.pmdec
oitarget(*).pmra_err = startable.pmrae
oitarget(*).pmdec_err = startable.pmdece
oitarget(*).parallax = startable.px
oitarget(*).para_err = startable.pxe
oitarget(*).spectyp = startable.spectrum
; Update for OI_TARGET revision 2
oitarget(*).diameter=startable.diameter
oitarget(*).omega=startable.omega
oitarget(*).vsini=startable.vsini
oitarget(*).tilt=startable.tilt
oitarget(*).rapa=startable.rapa
oitarget(*).mass=startable.mass
oitarget(*).teff=startable.teff
oitarget(*).logg=startable.logg
oitarget(*).zexp=startable.zexp
oitarget(*).nwave=n_elements(startable(0).wvl)
for i=0,n_elements(startable)-1 do begin
	oitarget(i).wave=ptr_new(float(startable(i).wvl))
	oitarget(i).flux=ptr_new(float(startable(i).sed))
	oitarget(i).lldc=ptr_new(float(startable(i).ldc))
endfor
;
; Write OI FITS data
if n_elements(fitsfile) eq 0 then fitsfile='oitarget.fits'
write_oidata,fitsfile,0,oitarget,0,0,0,0
print,'OIFITS data saved to: '+fitsfile
;
end
;-------------------------------------------------------------------------------
pro put_uvfits,stars
;
; Write averaged data from a single night into FITS multi-source file
; (single source file, i.e. no source table, if only one star is selected).
; New multi-IF version.
; Note that as in set_complexvis, it is assumed that the different
; spectrometers have identical channel layouts.
;
common Tables,ScanTable,BGTable,StationTable
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if checkdata([8,9]) ne 0 then return
;
; Determine star and scan indices
if n_elements(stars) eq 0 then stars=startable.starid
stars=strupcase(stars)
numstar=n_elements(stars)
star_mask=intarr(numstar)-1
scan_mask=intarr(n_elements(scans))-1
for k=0,numstar-1 do begin
	jndex=where(startable.starid eq stars(k),jcount)
	index=where(scans.starid eq stars(k),icount)
	if jcount gt 0 and icount gt 0 then begin
		star_mask(k)=jndex(0)
		scan_mask(index)=1
	endif else print,'Warning(PUT_UVFITS): no data for ',stars(k),'!'
endfor
scan_index=where(scan_mask ne -1,numscan)
if numscan eq 0 then begin
	print,'***Error(PUT_UVFITS): no valid scans found!'
	return
endif
star_index=star_mask(where(star_mask ge 0,numstar))
if numstar eq 1 then filename=stars(0)+'.FITS' $
		else filename=date+'.FITS'
;
r=file_search(filename,count=fcount)
if fcount ne 0 then begin
	print,'***Error(PUT_OIFITS): file ',filename,' exists!'
	return
endif
;
; Header keywords (p.26 of FITS User's Guide)
parsedate,jd2date(systime(/julian)),y,m,d
DATE_NOW=aipsdate(y,m,d)		; DATE
parsedate,Date,y,m,d
DATE_OBS=aipsdate(y,m,d)		; DATE-OBS
JD=julian(y,m,d)-0.5
ORIGIN=getenv('HOST')			; ORIGIN
TELESCOP=systemid			; TELESCOP
if system_id(systemid) eq 'NPOI' then begin
OBSERVER='USNO/NRL/Lowell'		; OBSERVER
case genconfig.beamcombinerid of
	1: INSTRUME='3-way'
	2: INSTRUME='6-way'
     else: INSTRUME='3-way'
endcase
endif else if system_id(systemid) eq 'VLTI' then begin
OBSERVER='ESO SciOps'			; OBSERVER
case genconfig.beamcombinerid mod 10 of
	1: INSTRUME='VINCI'
	2: INSTRUME='MIDI'
        3: INSTRUME='AMBER'
endcase
endif else begin
	OBSERVER='OBSERVER'
	INSTRUME='INSTRUMENT'
endelse
AUTHOR=getenv('USER')			; AUTHOR
;
; Dimensioning information
numsid=GenConfig.NumSid
numoutbeam=GenConfig.NumOutBeam
numchannel=GenConfig.NumSpecChan
numbaseline=GenConfig.NumBaseline
maxchannel=max(GenConfig.NumSpecChan)
maxbaseline=max(GenConfig.NumBaseline)
;
; Number of random groups (gcount)
numvis=long(numscan*summe(numbaseline(0:numoutbeam-1)))
;
; Visibility data
real_vis=double(scans(scan_index).complexvis)
imag_vis=double(imaginary(scans(scan_index).complexvis))
weight=double(scans(scan_index).complexweight)
;
; Random parameter data (pcount parameters per random group)
; U V W
uvw=scans(scan_index).uvw
; Baseline (= 256*i + j for antennas i < j)
fbase=dblarr(maxbaseline,numoutbeam)
for i=0,numoutbeam-1 do begin
for l=0,numbaseline(i)-1 do begin
	m=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(l,i),0,3))+1
	n=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(l,i),4,3))+1
	if m gt n then begin
		n0=n
		m0=m
		n=m0
		m=n0
		imag_vis(i,*,l,*)=-imag_vis(i,*,l,*)
		uvw(i,*,l,*,*)=-uvw(i,*,l,*,*)
	endif
	fbase(l,i)=256*m+n
endfor
endfor
; Date offset
time=abs(scans(scan_index).time/86400)
; Source ID
source_id=lonarr(numscan)
for i=0,numstar-1 do source_id(where(scans(scan_index).starid $
		                  eq startable(star_index(i)).starid))=i
;
; Data for Source FITS table
SOURCE=startable(star_index).starid
RAEPP=startable(star_index).ra*15
DECEPP=startable(star_index).dec
;
; Data for Frequency FITS table
iffreq0=c_light/genconfig.wavelength(0,0)
iffreqs=c_light/genconfig.wavelength(0:maxchannel-1,0:numoutbeam-1)-iffreq0
chwidth=c_light/genconfig.wavelength(0:maxchannel-1,0:numoutbeam-1)^2 $
               *genconfig.chanwidth(0:maxchannel-1,0:numoutbeam-1)
chwidth=float(chwidth)
;
; Keywords for Antenna table
RAD=180/pi_circle
gstia0=ut12gst(0)*15.d0
ut1utc=geoparms.ut1utc_coeffs(0)
datutc=geoparms.tai_utc
r_earth=6378.136d3
arrayc=(r_earth+geoparms.altitude)* $
	[cos(geoparms.latitude/RAD)*cos(geoparms.longitude/RAD), $
	 cos(geoparms.latitude/RAD)*sin(geoparms.longitude/RAD), $
	 sin(geoparms.latitude/RAD)]
;
; Data for Array Geometry FITS table (antenna table)
stations=strarr(numsid)
stncoord=dblarr(3,numsid)
for i=0,numsid-1 do begin
;	Bring into equatorial system
	npoi_x=GenConfig.StationCoord(0,i)
	npoi_y=GenConfig.StationCoord(1,i)
	npoi_z=GenConfig.StationCoord(2,i)
	x=cos(GeoParms.Latitude/RAD)*npoi_z-sin(GeoParms.Latitude/RAD)*npoi_y
	y=npoi_x
	z=cos(GeoParms.Latitude/RAD)*npoi_y+sin(GeoParms.Latitude/RAD)*npoi_z
	stncoord(*,i)=[x,y,z]
	stations(i)=genconfig.stationid(i)
endfor
;
; Call FITS writer
status=linknload(!external_lib,'writefits', $
	JD,DATE_NOW,DATE_OBS,ORIGIN,TELESCOP,INSTRUME,OBSERVER,AUTHOR,SOURCE, $
	RAEPP,DECEPP, $
	gstia0,ut1utc,datutc, $
	iffreq0,iffreqs,chwidth, $
	numvis,numscan,numsid,numoutbeam,numchannel,numbaseline, $
	maxchannel,maxbaseline, $
	time,uvw,real_vis,imag_vis,weight,fbase,stations,stncoord,arrayc, $
	source_id,numstar)
;
end
;-------------------------------------------------------------------------------
pro put_uvfits_bad,stars,beam=beam_in
;
; Write averaged data from a single night into FITS multi-source file
; (single source file, i.e. no source table, if only one star is selected).
; New multi-IF version.
; Note that as in set_complexvis, it is assumed that the different
; spectrometers have identical channel layouts.
;
common Tables,ScanTable,BGTable,StationTable
common StarBase,StarTable,Notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if checkdata([8,9]) ne 0 then return
;
if n_elements(beam_in) eq 0 then beam=0 else beam=beam_in-1
;
; Determine star and scan indices
if n_elements(stars) eq 0 then stars=startable.starid
stars=strupcase(stars)
numstar=n_elements(stars)
star_mask=intarr(numstar)-1
scan_mask=intarr(n_elements(scans))-1
for k=0,numstar-1 do begin
	jndex=where(startable.starid eq stars(k),jcount)
	index=where(scans.starid eq stars(k),icount)
	if jcount gt 0 and icount gt 0 then begin
		star_mask(k)=jndex(0)
		scan_mask(index)=1
	endif else print,'Warning(PUT_UVFITS): no data for ',stars(k),'!'
endfor
scan_index=where(scan_mask ne -1,numscan)
if numscan eq 0 then begin
	print,'***Error(PUT_UVFITS): no valid scans found!'
	return
endif
star_index=star_mask(where(star_mask ge 0,numstar))
if numstar eq 1 then filename=stars(0)+'.FITS' $
		else filename=date+'.FITS'
;
r=file_search(filename,count=fcount)
if fcount ne 0 then begin
	print,'***Error(PUT_OIFITS): file ',filename,' exists!'
	return
endif
;
; Header keywords (p.26 of FITS User's Guide)
parsedate,jd2date(systime(/julian)),y,m,d
DATE_NOW=aipsdate(y,m,d)		; DATE
parsedate,Date,y,m,d
DATE_OBS=aipsdate(y,m,d)		; DATE-OBS
JD=julian(y,m,d)-0.5
ORIGIN=getenv('HOST')			; ORIGIN
TELESCOP=systemid			; TELESCOP
if system_id(systemid) eq 'NPOI' then begin
OBSERVER='USNO/NRL/Lowell'		; OBSERVER
case genconfig.beamcombinerid of
	1: INSTRUME='3-way'
	2: INSTRUME='6-way'
     else: INSTRUME='3-way'
endcase
endif else if system_id(systemid) eq 'VLTI' then begin
OBSERVER='ESO SciOps'			; OBSERVER
case genconfig.beamcombinerid mod 10 of
	1: INSTRUME='VINCI'
	2: INSTRUME='MIDI'
        3: INSTRUME='AMBER'
       53: INSTRUME='GRAVITY'
endcase
endif else begin
	OBSERVER='OBSERVER'
	INSTRUME='INSTRUMENT'
endelse
AUTHOR=getenv('USER')			; AUTHOR
;
; Dimensioning information
numsid=GenConfig.NumSid
numoutbeam=GenConfig.NumOutBeam*0+1	; One beam at a time only
numchannel=GenConfig.NumSpecChan(beam)
numbaseline=GenConfig.NumBaseline(beam)
maxchannel=max(GenConfig.NumSpecChan(beam))
maxbaseline=max(GenConfig.NumBaseline(beam))
;
; Number of random groups (gcount)
; numvis=long(numscan*summe(numbaseline(0:numoutbeam-1)))
numvis=long(numscan*numbaseline)
;
; Visibility data
real_vis=reform(double(scans(scan_index).complexvis(beam,*)))
imag_vis=reform(double(imaginary(scans(scan_index).complexvis(beam,*))))
weight=reform(double(scans(scan_index).complexweight(beam,*)))
;
; Random parameter data (pcount parameters per random group)
; U V W
uvw=reform(scans(scan_index).uvw(beam,*))
; Baseline (= 256*i + j for antennas i < j)
fbase=dblarr(maxbaseline,numoutbeam)
i=beam
for l=0,numbaseline-1 do begin
	m=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(l,i),0,3))+1
	n=where(GenConfig.StationId eq strmid(GenConfig.BaselineId(l,i),4,3))+1
	if m gt n then begin
		n0=n
		m0=m
		n=m0
		m=n0
		imag_vis(*,l,*)=-imag_vis(*,l,*)
		uvw(*,l,*,*)=-uvw(*,l,*,*)
	endif
	fbase(l)=256*m+n
endfor
; Date offset
time=abs(scans(scan_index).time/86400)
; Source ID
source_id=lonarr(numscan)
for i=0,numstar-1 do source_id(where(scans(scan_index).starid $
		                  eq startable(star_index(i)).starid))=i
;
; Data for Source FITS table
SOURCE=startable(star_index).starid
RAEPP=startable(star_index).ra*15
DECEPP=startable(star_index).dec
;
; Data for Frequency FITS table
iffreq0=c_light/genconfig.wavelength(0,0)
iffreqs=c_light/genconfig.wavelength(0:maxchannel-1,beam)-iffreq0
chwidth=c_light/genconfig.wavelength(0:maxchannel-1,beam)^2 $
               *genconfig.chanwidth(0:maxchannel-1,beam)
chwidth=float(chwidth)
;
; Keywords for Antenna table
RAD=180/pi_circle
gstia0=ut12gst(0)*15.d0
ut1utc=geoparms.ut1utc_coeffs(0)
datutc=geoparms.tai_utc
r_earth=6378.136d3
arrayc=(r_earth+geoparms.altitude)* $
	[cos(geoparms.latitude/RAD)*cos(geoparms.longitude/RAD), $
	 cos(geoparms.latitude/RAD)*sin(geoparms.longitude/RAD), $
	 sin(geoparms.latitude/RAD)]
;
; Data for Array Geometry FITS table (antenna table)
stations=strarr(numsid)
stncoord=dblarr(3,numsid)
for i=0,numsid-1 do begin
;	Bring into equatorial system
	npoi_x=GenConfig.StationCoord(0,i)
	npoi_y=GenConfig.StationCoord(1,i)
	npoi_z=GenConfig.StationCoord(2,i)
	x=cos(GeoParms.Latitude/RAD)*npoi_z-sin(GeoParms.Latitude/RAD)*npoi_y
	y=npoi_x
	z=cos(GeoParms.Latitude/RAD)*npoi_y+sin(GeoParms.Latitude/RAD)*npoi_z
	stncoord(*,i)=[x,y,z]
	stations(i)=genconfig.stationid(i)
endfor
;
; Call FITS writer
status=linknload(!external_lib,'writefits', $
	JD,DATE_NOW,DATE_OBS,ORIGIN,TELESCOP,INSTRUME,OBSERVER,AUTHOR,SOURCE, $
	RAEPP,DECEPP, $
	gstia0,ut1utc,datutc, $
	iffreq0,iffreqs,chwidth, $
	numvis,numscan,numsid,numoutbeam*0+1,numchannel,numbaseline, $
	maxchannel,maxbaseline, $
	time,uvw,real_vis,imag_vis,weight,fbase,stations,stncoord,arrayc, $
	source_id,numstar)
;
end
;-------------------------------------------------------------------------------
pro put_xdr,filename
;
common StarBase,StarTable,Notes
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(filename) eq 0 then file='oyster.xdr' else file=filename+'.xdr'
save,startable,scantable,systemid,date,genconfig,geoparms,scans,file=file
print,'Saved scan data to: ',file
;
end
;-------------------------------------------------------------------------------
pro put_mark3data
;
; Write Mark III data in cal file format.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Mark3,mark3_baselines,mark3_configs,mark3_cat,mark3_bsc
;
if n_elements(SystemId) eq 0 then begin
	print,'***Error(PUT_MARK3DATA): SystemId undefined!'
	return
endif
if SystemId ne 'Mark3' then begin
	print,'***Error(PUT_MARK3DATA): SystemId is not Mark3!'
	return
endif
;
parsedate,Date,y,m,d
mark3_date=(y-1900)*10000L+m*100+d
;
mark3_ch1=GenConfig.Wavelength(0,0)*1.d9
mark3_ch2=GenConfig.Wavelength(0,1)*1.d9
mark3_ch3=GenConfig.Wavelength(1,0)*1.d9
mark3_ch4=GenConfig.Wavelength(1,1)*1.d9
;
i=where(mark3_baselines eq GenConfig.BaselineId(0),count)
if count eq 0 then begin
	print,'***Error(PUT_MARK3DATA): unknown baseline: ', $
		GenConfig.BaselineId(0),'!'
	return
endif
mark3_baseline=i(0)+1
;
index=where(scans.time gt 0,count)
if count eq 0 then begin
	print,'***Error(PUT_MARK3DATA): no valid scans to write!'
	return
endif
mark3_scans=scans(index)
;
; Prepare star numbers
stars=scans(index).starid
; Replace MARK_III BSC numbers with corresponding FKV numbers >= 1700 
for i=0,n_elements(mark3_bsc)-1 do begin
	index=where(stars eq mark3_bsc(i),count)
	if count gt 0 then stars(index)=mark3_cat(i)
endfor
mark3_stars=fix(strmid(stars,3,4))
;
filename=strcompress(string(mark3_date),/remove_all)+'m3.cal'
openw,unit,filename,/get_lun
;
dot='.'
printf,unit,mark3_date,dot,mark3_baseline,mark3_ch2,mark3_ch3,mark3_ch4, $
	format='(3x,i6,a1,3x,i2,2x,f5.1,17x,f5.1,17x,f5.1)'
for i=0,n_elements(mark3_scans)-1 do begin
	v2=scans(i).VisSq(1,0,0)
     	vc2=scans(i).VisSqC(1,0,0)
     	vce2=scans(i).VisSqCErr(1,0,0)
	if vce2 lt 0 then begin
		vc2=2.0
		vce2=0.0
	endif
     	v3=scans(i).VisSq(0,1,0)
     	vc3=scans(i).VisSqC(0,1,0)
     	vce3=scans(i).VisSqCErr(0,1,0)
	if vce3 lt 0 then begin
		vc3=2.0
		vce3=0.0
	endif
     	v4=scans(i).VisSq(1,1,0)
     	vc4=scans(i).VisSqC(1,1,0)
     	vce4=scans(i).VisSqCErr(1,1,0)
	if vce4 lt 0 then begin
		vc4=2.0
		vce4=0.0
	endif
     	printf,unit,scans(i).time/3600,mark3_stars(i), $
       		v2,vc2,vce2,v3,vc3,vce3,v4,vc4,vce4, $
       		format='(1x,f9.6,1x,i4,3(1x,f6.3),1x,3(1x,f6.3),1x,3(1x,f6.3))'
endfor
free_lun,unit
print,'Mark3 data written.'
;
end
;************************************************************************Block 7
pro get_stationtable,stnfile,update=update
;
; If specified, the input stationfile contains station coordinates and
; constant terms to update the coordinates in GenConfig. 
; Data need to be horizon coordinates. Update occurs only if update=1 (default).
;
; If no stationfile is defined, this routine will use files with path info
; and the array configuration to compute the coordinates and constant terms.
;
; Note that if no station file is defined and the system is NPOI or VLTI,
; constant term delay contributions are collected according to the
; specific feed system configuration in the GenConfig object.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(update) eq 0 then update=1 else update=update gt 0
if update then if checkdata([8]) ne 0 then return
;
; If no input defined, set information for config tables
if n_elements(stnfile) eq 0 then begin
   case system_id(SystemId) of
	'PTI'     : stnfile=!oyster_dir+'pti/stations.pti' 
	'CHARA'   : stnfile=!oyster_dir+'chara/stations.chara.old' 
	'IOTA'    : stnfile=!oyster_dir+'iota/stations.iota' 
	'VLTI'    : begin
		    stnfile=!oyster_dir+'vlti/stations.config' 
		    telfile=!oyster_dir+'vlti/tel.config'
		    uvwfile=!oyster_dir+'vlti/uvw.config'
		    fdlfile=!oyster_dir+'vlti/fdl.config'
		    m16file=!oyster_dir+'vlti/m16.config'
		    swyfile=!oyster_dir+'vlti/swy.config'
		    end
	'COAST'   : stnfile=!oyster_dir+'coast/stations.coast' 
	'Keck'    : stnfile=!oyster_dir+'keck/stations.keck' 
	'Mark3'   : stnfile=!oyster_dir+'mark3/stations.mark3' 
	'NPOI'    : begin
		    stnfile=!oyster_dir+'npoi/stations.config'
		    fdlfile=!oyster_dir+'npoi/fdl.config'
		    case genconfig.beamcombinerid of
		    	1: wayfile=!oyster_dir+'npoi/3way.config'
		    	2: wayfile=!oyster_dir+'npoi/6way.config'
		    	3: wayfile=!oyster_dir+'npoi/6way.config'
		    	9: wayfile=!oyster_dir+'npoi/6way.config'
		     else: begin
			   print,'***Error(GET_STATIONTABLE): ' $
                                +'unknown beam combiner ID: ', $
				genconfig.beamcombinerid,'!'
		           return
			   end
		    endcase
		    end
	      else: begin
		    print,'***Warning(GET_STATIONTABLE): unknown SystemId: ', $
						SystemId,'!'
		    S=genconfig.stationid
		    X=fltarr(genconfig.numsid)
		    Y=fltarr(genconfig.numsid)
		    Z=fltarr(genconfig.numsid)
		    D=fltarr(genconfig.numsid)
		    StationTable=build_stationtable(S,X,Y,Z,D)
		    return
		    end
   endcase
endif
;
; Check files exist
result=file_search(stnfile,count=fcount)
if fcount eq 0 then begin
	print,'***Error(GET_STATIONTABLE): ',stnfile,' not found!'
	return
endif
;
; NPOI needs additional configuration files
if SystemId eq 'NPOI' and n_params() eq 0 then begin
	result=file_search(fdlfile,count=fcount)
	if fcount eq 0 then begin
		print,'***Error(GET_STATIONTABLE): ',fdlfile,' not found!'
		return
	endif
	result=file_search(wayfile,count=fcount)
	if fcount eq 0 then begin
		print,'***Error(GET_STATIONTABLE): ',wayfile,' not found!'
		return
	endif
endif
; VLTI needs additional configuration files
if system_id(SystemId) eq 'VLTI' and n_params() eq 0 then begin
	result=file_search(telfile,count=fcount)
	if fcount eq 0 then begin
		print,'***Error(GET_STATIONTABLE): ',telfile,' not found!'
		return
	endif
	result=file_search(uvwfile,count=fcount)
	if fcount eq 0 then begin
		print,'***Error(GET_STATIONTABLE): ',uvwfile,' not found!'
		return
	endif
	result=file_search(fdlfile,count=fcount)
	if fcount eq 0 then begin
		print,'***Error(GET_STATIONTABLE): ',fdlfile,' not found!'
		return
	endif
	result=file_search(m16file,count=fcount)
	if fcount eq 0 then begin
		print,'***Error(GET_STATIONTABLE): ',m16file,' not found!'
		return
	endif
	result=file_search(swyfile,count=fcount)
	if fcount eq 0 then begin
		print,'***Error(GET_STATIONTABLE): ',swyfile,' not found!'
		return
	endif
endif
;
; Now read the station file
S=''
X=0.d0
Y=0.d0
Z=0.d0
D=0.d0
DL_ID=0
BC_IN=0
if dc_read_free(stnfile,S,X,Y,Z,D,DL_ID,BC_IN,/col,resize=[1,2,3,4,5],ignore=['!']) $
   ne 0 then begin
   print,'***Error(GET_STATIONTABLE): file, ',stnfile,' corrupt!'
   return
endif
;
; Read additional configuration data for some interferometers for computation
if system_id(SystemId) eq 'NPOI' and n_params() eq 0 then begin
	if dc_read_free(fdlfile,d_id,d_v,/col,ignore=['!']) ne 0 then begin
		print,'***Error(GET_STATIONTABLE): file, ',fdlfile,' corrupt!'
		return
	endif
	d_v=d_v(sort(d_id))
	if dc_read_free(wayfile,b_id,b_v,/col,ignore=['!']) ne 0 then begin
		print,'***Error(GET_STATIONTABLE): file, ',wayfile,' corrupt!'
		return
	endif
	b_v=b_v(sort(b_id))
endif
if system_id(SystemId) eq 'VLTI' and n_params() eq 0 then begin
	if dc_read_free(telfile,teltyp,telopl,teldu,teldv,/col,ignore=['!']) $
	ne 0 then begin
		print,'***Error(GET_STATIONTABLE): file, ',telfile,' corrupt!'
		return
	endif
	if dc_read_free(uvwfile,uvwtel,uvwu,uvwv,uvww,/col,ignore=['!']) $
	ne 0 then begin
		print,'***Error(GET_STATIONTABLE): file, ',uvwfile,' corrupt!'
		return
	endif
	if dc_read_free(fdlfile,fdlid,fdlopl,fdlu,fdlvin,fdlvout,/col,ignore=['!']) $
	ne 0 then begin
		print,'***Error(GET_STATIONTABLE): file, ',fdlfile,' corrupt!'
		return
	endif
	fdlopl=fdlopl(sort(fdlid))
	fdlu=fdlu(sort(fdlid))
	fdlvin=fdlvin(sort(fdlid))
	fdlvout=fdlvout(sort(fdlid))
	if dc_read_free(m16file,m16id,m16u,/col,ignore=['!']) ne 0 then begin
		print,'***Error(GET_STATIONTABLE): file, ',m16file,' corrupt!'
		return
	endif
	m16u=m16u(sort(m16id))
	if dc_read_free(swyfile,swyid,swyopl,/col,ignore=['!']) ne 0 then begin
		print,'***Error(GET_STATIONTABLE): file, ',swyfile,' corrupt!'
		return
	endif
	swyopl=swyopl(sort(swyid))
endif
;
; Create stationtable and  reduce table to stations defined in GenConfig.
; Update stationcoord if requested.
StationTable=build_stationtable(S,X,Y,Z,D)
SI=intarr(n_elements(stationtable))
if checkdata([8]) eq 0 then begin
for i=0,GenConfig.NumSid-1 do begin
	index=where(S eq GenConfig.StationId(i),count)
	if count eq 0 then begin
	   print,'***Error(GET_STATIONTABLE): '+ $
		 'station not found in StationTable!'
	   return
	endif
	if update then begin
		GenConfig.StationCoord(0,i)=x(index)
		GenConfig.StationCoord(1,i)=y(index)
		GenConfig.StationCoord(2,i)=z(index)
		GenConfig.StationCoord(3,i)=d(index)
	endif
	SI(index)=1
endfor
stationtable=stationtable(where(SI eq 1))
endif
;
; Apply additional configuration changes if computed and update if requested
if system_id(SystemId) eq 'NPOI' and n_params() eq 0 and update then begin
	GenConfig.StationCoord(3,0:GenConfig.NumSid-1)= $
	GenConfig.StationCoord(3,0:GenConfig.NumSid-1) $
			+d_v(GenConfig.DelaylineID(0:GenConfig.NumSid-1)-1) $
			+b_v(GenConfig.BCInputId(0:GenConfig.NumSid-1)-1)
endif
;
if system_id(SystemId) eq 'VLTI' and n_params() eq 0 and update then begin
	for i=0,genconfig.numsid-1 do begin
		j=where(teltyp eq strmid(genconfig.stationid(i),0,1)) & j=j(0)
;		OPL M3(pivot)-M11
		opd0=telopl(j)
		k=where(uvwtel eq strmid(genconfig.stationid(i),1,2)) & k=k(0)
;		Add telescope pivot W coordinate
		opd0=opd0+uvww(k)
;		M11 offset to station centre changes sign for a northern station
		if teltyp(j) eq 'A' and uvwv(k) gt -32 then begin
			teldv(j)=-teldv(j)
			teldu(j)=-teldu(j)
		endif
;		Compute M11 V-coordinate
		vm11=uvwv(k)+teldv(j)
;		Compute difference to M12 V-coordinate and add
		if teltyp(j) eq 'U' then $
		opd0=opd0+abs(vm11-fdlvin(genconfig.delaylineid(i)-1))
		if teltyp(j) eq 'A' then $
		opd0=opd0+abs(vm11-fdlvout(genconfig.delaylineid(i)-1))
;		Compute M12 U-coordinate
		um12=uvwu(k)+teldu(j)
;		Compute difference to DL mechanical zero position and add
		opd0=opd0+abs(fdlu(genconfig.delaylineid(i)-1)-um12)
;		Add internal delay of DL
		opd0=opd0+fdlopl(genconfig.delaylineid(i)-1)
;		Add difference in U-coord. to m16 U-coord.
		opd0=opd0+abs(fdlu(genconfig.delaylineid(i)-1) $
				-m16u(genconfig.bcinputid(i)-1))
;		Add difference in V-coord. to reference plane of switch yard
		if teltyp(j) eq 'U' then $
		opd0=opd0+abs(-36.0-fdlvout(genconfig.delaylineid(i)-1))
		if teltyp(j) eq 'A' then $
		opd0=opd0+abs(-36.0-fdlvin(genconfig.delaylineid(i)-1))
;		Add path from switch yard table to ZPD
		opd0=opd0+swyopl(genconfig.bcinputid(i)-1)
;		Store total OPD (i.e. OPD0)
		genconfig.stationcoord(3,i)=opd0
	endfor
endif
;
if update then $
print,'StationTable created; StationCoord updated.' else $
print,'StationTable created.'
;
end
;-------------------------------------------------------------------------------
pro put_stationtable,table=table
;
; Write StationCoord to file. Filename is derived from date of observation.
; Note that in the case of NPOI or VLTI, d is the total constant term value.
; In the case of VLTI, d is the OPL0. We also now write the delayline and
; beamcombiner input IDs, as these are important configuration information
; for the interpretation of the constant terms.
;
; If the table keyword is set, write the table, else write 
; genconfig.stationcoord.
;
common Hds,path,hds_file_stub
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Tables,ScanTable,BGTable,StationTable
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if checkdata([5,6,7]) ne 0 then return
;
if n_elements(table) eq 0 then table=1
;
; Set file name
if n_elements(hds_file_stub) eq 0 then file=''
if strlen(file) ne 0 then filename=hds_file_stub+'.stn' $
		     else filename=date+'.stn'
openw,unit,filename,/get_lun
;
RAD=180/pi_circle
stationid=genconfig.stationid
x=genconfig.stationcoord(0,*)
y=genconfig.stationcoord(1,*)
z=genconfig.stationcoord(2,*)
d=genconfig.stationcoord(3,*)
DL_ID=genconfig.delaylineid
BC_IN=genconfig.bcinputid
if table eq 1 then begin
	stationid=stationtable.stationid
	x=stationtable.x
	y=stationtable.y
	z=stationtable.z
	d=stationtable.d
	DL_ID=stationtable.DL_ID
	BC_IN=stationtable.BC_IN
endif
;
case system_id(systemid) of
	'Mark3':begin
		printf,unit,'! Mark3 station file. All units in m' 
		printf,unit,'! Station    X(east)    Y(north)   Z(up)      D'
		end
	'NPOI' :begin
		printf,unit,'! NPOI CONFIGURATION'
		printf,unit,'!'
		printf,unit,'! Array center on Anderson Mesa'
		printf,unit,'! Latitutde = ',dms(GeoParms.Latitude)
		printf,unit,'! Longitude = ',dms(GeoParms.Longitude) 
		printf,unit,'! Altitude = ',GeoParms.Altitude 
		printf,unit,'!'
		printf,unit,'! Latitude, longitude and height are for the', $
		   	       ' nominal "array center", from'
		printf,unit,'! which the other positions are measured. X,', $
			       ' Y, Z form a right handed coord'
		printf,unit,'! system such that X = East Y = North and', $
			       ' Z = Up. All distances are in meters.'
		printf,unit,'!'
		printf,unit,'!        X           Y           Z           D (actual)'
		end
	'VLTI' :begin
		printf,unit,'! VLTI OPD model'
		if n_elements(stationid) eq 2 then begin
			printf,unit,'! VLTI-DL pointing error file'
			printf,unit,'!    1    1.0'
			at=genconfig.stationcoord(*,1) $
			  -genconfig.stationcoord(*,0)
			at=at*1d6
			printf,unit,'! A1L '+string(-at(3))+' 0.0'
			printf,unit,'! T1X '+string(at(0))+' 0.0'
			printf,unit,'! T1Y '+string(at(1))+' 0.0'
			printf,unit,'! T1Z '+string(-at(2))+' 0.0'
			printf,unit,'! END'
			printf,unit,'!        X [m],      Y [m],      Z [m],      OPL0 [m], delayline, input channel'
		endif else begin
			printf,unit,'!        X [m],      Y [m],      Z [m],      Tel DL IC'
		endelse
		end
	   else:print,'***Error(PUT_STATIONTABLE): unknown system:', $
			systemid,'!'
endcase
for j=0,n_elements(stationid)-1 do printf,unit, $
	stationid(j),x(j),y(j),z(j),d(j),DL_ID(j),BC_IN(j), $
	format='(a,2x,4(1x,f11.6),2x,i1,2x,i1)'
free_lun,unit
print,'StationCoord written to: ',filename,'.'
;
end
;-------------------------------------------------------------------------------
pro get_opdmodel,opdmodel
;
; Read VLTI format OPD model
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
dl1=fix(strmid(opdmodel,2,1))
dl2=fix(strmid(opdmodel,2+12,1))
ic1=fix(strmid(opdmodel,10,1))
ic2=fix(strmid(opdmodel,10+12,1))
if strmid(opdmodel,3,1) eq 'U' then begin
	s1=strmid(opdmodel,3,3)
	s2=strmid(opdmodel,15,3)
endif else begin
	s1=strmid(opdmodel,3,1)+strmid(opdmodel,6,2)
	s2=strmid(opdmodel,15,1)+strmid(opdmodel,18,2)
endelse
;
l=''
status=dc_read_fixed(opdmodel,l,/col,format='(a300)')
;
i=strpos(l,'T1X') & j=where(i ge 0) & j=j(0)
words=nameparse(l(j))
t1x=double(words(1))
;
i=strpos(l,'T1Y') & j=where(i ge 0) & j=j(0)
words=nameparse(l(j))
t1y=double(words(1))
;
i=strpos(l,'T1Z') & j=where(i ge 0) & j=j(0)
words=nameparse(l(j))
t1z=double(words(1))
;
i=strpos(l,'A1L') & j=where(i ge 0) & j=j(0)
words=nameparse(l(j))
a1l=-double(words(1))
;
genconfig.stationcoord(*,0)=[t1x,t1y,t1z,a1l]/1d6
genconfig.stationcoord(*,1)=[t1x,t1y,t1z,a1l]/1d6
if s1 ne genconfig.stationid(genconfig.refstation-1) then $
	genconfig.stationcoord=-genconfig.stationcoord
genconfig.stationcoord(*,genconfig.refstation-1)=0
;
end
;-------------------------------------------------------------------------------
