;*******************************************************************************
; File: fitting.pro
;
; Description:
; ------------
; Container for procedures of the AMOEBA software related to fitting 
; of data.
;
; Block directory:
; ----------------
;
; Block 1: init_marquardtfit,init_modelfit,
;	   init_fitinterferometry,init_fitastrometry,
;	   init_candid
;
; Block 2: fitellipse1,fitorbit1,marquardt1,levenberg1
;
; Block 3: fitmodel,modelparmcopy,
;	   fitellipse,fitorbit
;
; Block 4: marquardt,
;	   fixell,ellfuncs,
;	   orbfuncs,
;	   modfuncs,
;	   funct_gauss,mask2pix,maskfuncs
;
; Block 5: set_h_steps,setorbit,setmodel
;
; Block 6: fitnights,fitnights_rescale,gridchisq,gridchisq_parallel,gridplot,
;	   fitchisq
;
; Renamed fit procedures Oct. 14th, 2024:
; gridfit -> gridchisq
; gridfit_parallel -> gridchisq_parallel
; gridplot -> gridplot (no change)
; gridpsn -> fitchisq (creates file fitchisq.psn)
; fitnights,plotselection=plotselection,sigma=sigma,cf=cf ->
; fitnights,plotselection=plotselection,chi2map=chi2map,sigma=sigma,cf=cf
;
;************************************************************************Block 1
function init_marquardtfit
;
; Initialize global Marquardt-Levenberg fit options.
;
common MarquardtFit,fit_options
;
fit_options=alloc_fit_options()
;
return,0
;
end
;-------------------------------------------------------------------------------
function init_modelfit,force=force
;
; Initialize parameters for modelfit. Clear only the parameters listed 
; for the global modelfit.
;
; Return non-zero value to indicate error occured.
;
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then begin
	print,'***Error(INIT_MODELFIT): no model!'
	return,-1
endif
;
; Check and report usage of FORCE keyword
if keyword_set(force) then print,'Note (init_modelfit): model will be reset!'
;
if n_elements(parameters) eq 0 or keyword_set(force) then begin
;	Reset model
	num_system=1
	component=alloc_component()
	parameters=replicate(component,num_star()+num_binary()+num_system)
endif else begin
	for i=0,n_elements(parameters.component)-1 do begin
		erase=0
		if strlen(parameters(i).component) eq 1 then erase=1
		index=where(parameters(i).names ne 'Rho' $
			and parameters(i).names ne 'Theta' $
			and strlen(parameters(i).names) ne 0,erase)
		if erase gt 0 then begin
			parameters(i).component=''
			parameters(i).names=''
		endif
	endfor
endelse
;
return,0
;
end
;-------------------------------------------------------------------------------
function init_fitinterferometry
;
; Initialize parameters for modelfit. Clear only the parameters listed 
; for the interferometric modelfit.
;
; Return non-zero value to indicate error occured.
;
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(gen_model) eq 0 then begin
	print,'***Error(INIT_FITINTERFEROMETRY): no model!'
	return,-1
endif
;
if n_elements(parameters) eq 0 then begin
	num_system=1
	component=alloc_component()
	parameters=replicate(component,num_star()+num_binary()+num_system)
endif else begin
	for i=0,n_elements(parameters.component)-1 do begin
;		index=where(parameters(i).names eq 'Rho' $
;			 or parameters(i).names eq 'Theta',count)
;		if count gt 0 then parameters(i).component=''
		erase=0
		if strlen(parameters(i).component) eq 1 then erase=1
		index=where((parameters(i).names eq 'Rho' $
			or parameters(i).names eq 'Theta') $
			and strlen(parameters(i).names) ne 0,erase)
		if erase gt 0 then begin
			parameters(i).component=''
			parameters(i).names=''
		endif
	endfor
endelse	
;
nights=''
;
return,0
;
end
;-------------------------------------------------------------------------------
function init_fitastrometry
;
; Initialize parameters for modelfit. Clear only the parameters listed 
; for the astrometric modelfit.
;
; Return non-zero value to indicate error occured.
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
;
; Ellipse
ellipse_options=alloc_ellipse_options()
if n_elements(e_parms) eq 0 then e_parms=dblarr(5)
;
; Orbit
orbit_options=alloc_orbit_options()
if n_elements(o_parms) eq 0 then o_parms=dblarr(8)
;
return,0
;
end
;-------------------------------------------------------------------------------
pro init_candid
;
; Quick/minimal setup of CANDID Python script for companion detection.
;
common Hds,path,hds_file_stub
common StarBase,StarTable,Notes
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(startable) gt 1 then begin
	print,'Only one dataset of a single target can be processed!'
	return
endif
file_stub=startable.starid+'_'+Date
;
ncpu=!cpu.hw_ncpu
;
file_name=hds_file_stub+'.inp'
openw,unit,file_name,/get_lun
;
printf,unit,"import candid"
printf,unit,"import time"
printf,unit,"import os"
printf,unit,"from matplotlib import pyplot as plt"
printf,unit,"candid.CONFIG['long exec warning'] = None"
printf,unit,"candid.CONFIG['Ncores'] = "+string(ncpu)
printf,unit,"global data"
printf,unit,"data=candid.Open('"+hds_file_stub+".fits')"
printf,unit,"data.observables=['v2','cp']"
printf,unit,"data.chi2Map(rmin=0,rmax=10,step=1)"
printf,unit,"data.fitMap(rmin=0,rmax=10,step=1)"
printf,unit,"plt.savefig("+hds_file_stub+"_'chi2Map.pdf')"
printf,unit,"plt.close(0)"
printf,unit,"plt.savefig("+hds_file_stub+"_'fitMap1.pdf')"
printf,unit,"plt.close(1)"
printf,unit,"plt.savefig("+hds_file_stub+"_'fitMap2.pdf')"
printf,unit,"plt.close(1)"
;
;printf,unit,"data.fitMap(fig=1,rmin=0,rmax=10,step=1,fratio=20,"+$
;	"addParam={'diam*': 1.0},doNotFit={'diam*'},"+$
;	"autoSaveFig='"+file_stub+".pdf')"
;
free_lun,unit
;
print,'File '+file_name+' written, please check data file extension!'
;
end
;************************************************************************Block 2
pro fitellipse1
;
; Fit apparent ellipse to position data. Use external library functions.
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; Check data
if n_elements(positions) eq 0 then begin
	print,'***Error(FITELLIPSE): no data!'
	return
endif
;
; Check ellipse parameters
if n_elements(e_parms) eq 0 then begin
	print,'***Error(FITELLIPSE): no ellipse parms!'
	return
endif
;
; Set data. Shift data to array indices 1:n.
index=where(positions.rho gt 0,ndata)
if ndata gt 0 then begin
	p=[0.d0,positions(index).theta]
	r=[0.d0,positions(index).rho]
	sig=dblarr(ndata+1)+1
endif else begin
	print,'***Error(FITELLIPSE): no valid data!'
	return
endelse
;
; Set parameters to fit
ma=n_elements(e_parms)
ia=lonarr(ma+1)
if ellipse_options.all then ia(*)=1 else begin
	if ellipse_options.c then ia(1:2)=1
	if ellipse_options.a then ia(3)=1
	if ellipse_options.b then ia(4)=1
	if ellipse_options.p then ia(5)=1
endelse
index=where(ia ne 0,mfit)
if mfit eq 0 then begin
	print,'***Error(FITELLIPSE): no fit parameters selected!'
	return
endif
a=[0.d0,e_parms]
;
; Store ellipse data
ex=r*sin(p)
ey=r*cos(p)
status=linknload(!external_lib,'store_ell',ex,ey,ndata)
;
; Call marquardt
marquardt1,'ellfuncs',p,r,sig,ndata,a,ia,ma,chisq,covar
;
; Set new parameters
e_parms=a(1:ma)
RAD=180/pi_circle
print,'__________________________________'
print,'Center x = ',e_parms(0)
print,'Center y = ',e_parms(1)
print,'Semi-major axis = ',e_parms(2)
print,'Semi-minor axis = ',e_parms(3)
print,'Position  angle = ',e_parms(4)*RAD
print,'_______________***________________'
;
; Free memory allocated in store_ell
status=linknload(!external_lib,'free_ell',ndata)
;
end
;-------------------------------------------------------------------------------
pro fitorbit1
;
; Fit orbital elements to position data. Use external library functions.
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; Check data
if n_elements(positions) eq 0 then begin
	print,'***Error(FITORBIT): no data!'
	return
endif
;
; Check orbit parameters
if n_elements(o_parms) eq 0 then begin
	print,'***Error(FITORBIT): no orbit parms!'
	return
endif
;
; Check whether orbit parameters have been initialized
if total(o_parms) eq 0 then begin
	print,'***Error(FITORBIT): bad starting model!'
	return
endif
;
; Set data. Shift data to array indices 1:n.
index=where(positions.rho gt 0,ndata)
if ndata gt 0 then begin
	jd=[0.d0,positions(index).jd]
	p=[0.d0,positions(index).theta]
	r=[0.d0,positions(index).rho]
	sig=[1.d0,positions(index).emajor]
endif else begin
	print,'***Error(FITORBIT): no valid data!'
	return
endelse
;
; Set parameters to fit
; RAD=180/3.141592653d0
; o_parms=dblarr(8)
; o_parms(0)=50.0
; o_parms(1)=0.20
; o_parms(2)=127.0/RAD
; o_parms(3)=80.0/RAD
; o_parms(4)=20.0/RAD
; o_parms(5)=103.0
; o_parms(6)=7518.0
;
; Note: o_parms(8) is the orbital sense (not fit).
ma=n_elements(o_parms)-1
ia=lonarr(ma+1)
if orbit_options.all then ia(*)=1 else begin
	if orbit_options.a then ia(1)=1
	if orbit_options.e then ia(2)=1
	if orbit_options.i then ia(3)=1
	if orbit_options.w then ia(4)=1
	if orbit_options.n then ia(5)=1
	if orbit_options.p then ia(6)=1
	if orbit_options.t then ia(7)=1
endelse
index=where(ia ne 0,mfit)
if mfit eq 0 then begin
	print,'***Error(FITORBIT): no fit parameters selected!'
	return
endif
a=[0.d0,o_parms(0:6)]
;
; Store ellipse data
ex=r*sin(p)
ey=r*cos(p)
status=linknload(!external_lib,'store_ell',ex,ey,ndata)
;
; Compute chisq
; xy=true2app(jd,o_parms)
; xy(0,*)=0
; chisq=total(((ex-xy(*,0))/sig)^2+((ey-xy(*,1))/sig)^2)
;
; Set r to zero since orbfuncs does the differences
r=dblarr(ndata+1)
;
; Call marquardt1
marquardt1,'orbfuncs',jd,r,sig,ndata,a,ia,ma,chisq,covar
;
; Set new paramters
o_parms(0:6)=a(1:ma)
RAD=180/pi_circle
print,'__________________________________'
print,'Semi-major axis = ',o_parms(0)
print,'Eccentricity =    ',o_parms(1)
print,'Inclination =     ',o_parms(2)*RAD
print,'Periastron =      ',o_parms(3)*RAD
print,'Ascending node =  ',o_parms(4)*RAD
print,'Period =          ',o_parms(5)
print,'Epoch =           ',o_parms(6)
print,'_______________***________________'
;
; Free memory allocated in store_ell
status=linknload(!external_lib,'free_ell',ndata)
;
end
;-------------------------------------------------------------------------------
pro marquardt1,func,x,y,sig,ndata,a,ia,ma,chisq,covar,alamda
;
; Iterative procedure to do non-linear least-squares fit.
; Use external library functions.
;
common MarquardtFit,fit_options
;
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
;
covar=dblarr(ma,ma)
alpha=dblarr(ma,ma)
;
chifr=fit_options.chifr
tol=fit_options.tolerance
alamda=-fit_options.alamda
;
status=linknload(!external_lib,'marquardt_init',ma)
;
mode=10L				; The default is Marquardt-Levenberg
; if fit_options.s then mode=10L else mode=0L
; if fit_options.e then mode=mode+1	; Not used anymore in Version 8
;
chisq=0.d0
status=linknload(!external_lib,'marquardt', $
		x,y,sig,ndata,a,ia,ma,covar,alpha,chisq,func,alamda,tol,mode)
print,''
print,'Initial chisq=',chisq
;
i=1
repeat begin
	chisqo=chisq
	status=linknload(!external_lib,'marquardt', $
		x,y,sig,ndata,a,ia,ma,covar,alpha,chisq,func,alamda,tol,mode)
	i=i+1
endrep until (1-chisq/chisqo lt chifr and chisq lt chisqo) $
	or alamda gt 10 $
	or i gt 100 $
	or status ne 0 $
	or fit_options.o
print,'Final   chisq=',chisq
print,'Iter=',i,', Alamda=',alamda,', Status=',status, $
	format='(a,i2,a,f8.2,a,i1)'
;
alamda=0.d0
status=linknload(!external_lib,'marquardt', $
		x,y,sig,ndata,a,ia,ma,covar,alpha,chisq,func,alamda,tol,mode)
;
status=linknload(!external_lib,'marquardt_free',ma)
;
end
;-------------------------------------------------------------------------------
pro levenberg1,func,x,y,a
;
; Use external library functions.
;
if n_params() eq 0 then begin
	print,'Procedure LEVENBERG:'
	print,'Please supply: func, x, y, a!'
	return
endif
;
ndata=n_elements(x)
ma=n_elements(a)
;
xx=double([0,x])
yy=double([0,y])
sigma=dblarr(ndata+1)+1
aa=double([0,a])
;
ia=lonarr(ma+1)+1
;
chisq=1d6
init_fit
;
marquardt1,func,xx,yy,sigma,ndata,aa,ia,ma,chisq,covar,alamda
;
end
;************************************************************************Block 3
pro bootstrap_vis,star_models_p,star_models_s,binary_models
;
; Template to carry out a "bootstrap" method to estimate parameter errors.
; Here, we do not replace data, but select a fraction of data (e.g., 90%)
; and repeat the parameter fit. The routine sets everything back when done
; and returns the fitted star and binary models. Selection of fit parameters
; is done via the ModelFit GUI.
;
forward_function cgRandomIndices
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
star_model_bck=star_model
binary_model_bck=binary_model
;
geninfo_bck=geninfo
geoinfo_bck=geoinfo
bufferinfo_bck=bufferinfo
;
num_nights=n_elements(geninfo)
sel_nights=long(0.7*num_nights)
;
n=num_nights
n=30	; Number of data subsets to fit
;
star_models_p=replicate(star_model(0),n)
star_models_s=replicate(star_model(1),n)
binary_models=replicate(binary_model,n)
;
for i=0,n-1 do begin
	indices = cgRandomIndices(num_nights,sel_nights,seed=seed)
	geninfo=geninfo_bck(indices)
	geoinfo=geoinfo_bck(indices)
	bufferinfo=bufferinfo_bck(indices)
	fitmodel
	star_models_p(i)=star_model(0)
	star_models_s(i)=star_model(1)
	binary_models(i)=binary_model
	star_model=star_model_bck
	binary_model=binary_model_bck
endfor
;
geninfo=geninfo_bck
geoinfo=geoinfo_bck
bufferinfo=bufferinfo_bck
;
star_model=star_model_bck
binary_model=binary_model_bck
;
end
;-------------------------------------------------------------------------------
pro fitmodel,chisq,ndata,error=error
;
; Fit model of hierarchical stellar system to combined data sets.
; Note: if a file "Z_h_steps.xdr" exists, use it (fit_options.h=1) to
; read parameter increments for the computation of the numerical derivatives.
; The parameter increments will be stored in gen/star/binary_error structures.
; Please see procedure setincrements for how to create this file.
;
common MarquardtFit,fit_options
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
;
; Check data
if ds_options.i and n_elements(scans) eq 0 then begin
	print,'***Error(FITMODEL): no interferometric data!'
	return
endif
if ds_options.a and n_elements(positions) eq 0 then begin
	print,'***Error(FITMODEL): no astrometric data!'
	return
endif
if ds_options.s and n_elements(velocities) eq 0 then begin
	print,'***Error(FITMODEL): no radial velocities!'
	return
endif
;
; Check parameters
icheck=0
if n_elements(parameters) eq 0 then icheck=1 else $
if fix(total(strlen(parameters.names))) eq 0 then icheck=1
if icheck then begin
	print,'***Error(FITMODEL): no modelfit parameters defined!'
	return
endif
;
; Check model
icheck=0
for i=0,n_elements(binary_model)-1 do begin
	if binary_model(i).method eq 1 then begin
		if binary_model(i).semimajoraxis eq 0 then icheck=1
	endif
	if binary_model(i).method eq 2 then begin
		if binary_model(i).rho eq 0 then icheck=1
	endif
endfor
if icheck then begin
	print,'***Error (FITMODEL): errors in parameter values!'
	return
endif
;
; Initialize parameter array a with hierarchical model
modelparmcopy,-1,a
ma=n_elements(a)
ia=lonarr(ma)+1
mfit=ma
if mfit eq 0 then begin
	print,'***Error(FITMODEL): no fit parameters selected!'
	return
endif
;
; Set data.
;!quiet=1
calcmodel,/quiet
marquardtdata,y,ysig,ymod
ndata=n_elements(y)
if ndata eq 1 and ysig(0) eq 0 then ndata=0
if ndata lt ma then begin
	print,'***Error(FITMODEL): not enough data!'
	return
endif
x=dindgen(ndata)
;
; If this keyword is set, use the parameter steps computed by modelparmcopy
if fit_options.h then begin
	file=file_search('Z_h_steps.xdr')
	if strlen(file) eq 0 then begin
		print,'Error: Z_h_steps.xdr not found!'
		return
	endif else begin
		restore,'Z_h_steps.xdr'
		print,'Using optimized steps in Z_h_steps.xdr.'
	endelse
endif
;
IF keyword_set(error) THEN BEGIN
;
marquardt,'modfuncs',x,y,ysig,ndata,a,ia,ma,chisq,covar,/error
if ndata-mfit ne 0 then chisq=chisq/(ndata-mfit)
;
ENDIF ELSE BEGIN
;
; Run marquardt alogrithm
marquardt,'modfuncs',x,y,ysig,ndata,a,ia,ma,chisq,covar
if ndata-mfit ne 0 then chisq=chisq/(ndata-mfit)
;
if total(abs(covar)) gt 0 then begin
	print,'Normalized covariance matrix:'
	print,'Selected parameters in column-wise order!'
;
;	Establish list order (index)...
	parameters_bck=parameters
	np=n_elements(parameters)
	index=intarr(np)-1
;
	n=0
	p_len=strlen(parameters.component)
	i=where(p_len ge strlen('System A'),ni)
	if ni gt 0 then begin
	if n_elements(nameparse(parameters(i(0)).component,'-')) eq 1 then begin
		index(n)=i(0)
		n=n+1
	endif
	endif
	i=where(p_len eq 1,ni)
	j=sort(parameters(i).component)
	for k=0,ni-1 do begin
		index(n)=i(j(k))
		n=n+1
	endfor
	i=where(strpos(parameters.component,'-') ge 0,ni)
	if ni gt 0 then begin
		j=sort(strlen(parameters(i).component))
		for k=0,ni-1 do begin
			index(n)=i(j(k))
			n=n+1
		endfor
	endif
;
;	Sort the parameters
	parameters=parameters(index(where(index ge 0)))
;
	parms=strarr(ma)
	b=blanks(5)
	print,b,format='(1x,a5,$)'
	k=0
	for i=0,n_elements(parameters)-1 do begin
		j=where(strlen(parameters(i).names) ne 0,n)
		for j=0,n-1 do begin
			print,parameters(i).names(j),format='(1x,a5,$)'
			parms(k)=strmid(parameters(i).names(j),0,5)
			k=k+1
		endfor
	endfor
	print,''	; to start newline
	;
	b='-----'
	bs=strarr(ma)
	for i=1,ma-1 do bs(i)=bs(i-1)+b+'-'
;	Uncomment to swap lines for blanks
;	bs=''
	for i=0,ma-1 do begin
		j=indgen(ma-i)
		values=string(fltarr(ma-i),format='(100(1x,f5.2))')
		covar_ij=sqrt(covar(j+i,j+i)*covar(i,i))
		if total(covar_ij) ne 0 then $
		values=string(covar(j+i,i)/covar_ij,format='(100(1x,f5.2))')
		chars=strarr(6-strlen(parms(i))) & chars(*)=' '
;
;		Swap comments to swap blanks for lines
		print,parms(i)+strjoin(chars),bs(i),values
;		print,parms(i)+blanks(5-strlen(parms(i))),bs,values
;		bs=bs+b+' '
	endfor
	parameters=parameters_bck
endif else begin
	print,'Ill defined parameters for this model!'
	return
endelse
;
; Store new parameters into hierarchical model
modelparmcopy,1,a
calcmodel,/quiet
;!quiet=0
;
ENDELSE
;
; Store diagonal elements of covariance matrix
; ...save the model first, and replace with model errors
gen_model_bck=gen_model
gen_model=gen_error
star_model_bck=star_model
star_model=star_error
binary_model_bck=binary_model
binary_model=binary_error
; ...clear all floats
gen_model.rv=0
for i=0,n_tags(star_model)-1 do $
	if size(star_model.(i),/type) eq 5 $
	or size(star_model.(i),/type) eq 4 then star_model.(i)=0
for i=0,n_tags(binary_model)-1 do $
	if size(binary_model.(i),/type) eq 5 $
	or size(binary_model.(i),/type) eq 4 then binary_model.(i)=0
; ...extract diagonal elements of covariance matrix
index=indgen(n_elements(a))
e=sqrt(covar(index,index)*chisq)	; Scale with sqrt(chisq)!
modelparmcopy,1,e
; ...copy model errors, replace model
gen_error=gen_model
gen_model=gen_model_bck
star_error=star_model
star_model=star_model_bck
binary_error=binary_model
binary_model=binary_model_bck
;
; As of version 8.10 (July 2014)
print,'Errors scaled with sqrt(chisq)!'
;
print,'FitModel complete.'
;
end
;-------------------------------------------------------------------------------
pro modelparmcopy,direction,a,da,init=init
;
; Copies parameters in array a from (direction=-1) and to (direction=1) 
; hierarchical stellar model. Also initializes array da which contains
; increments for the parameters to be used in the differential corrections.
; Procedure determines for each parameter listing the component, whether
; it is a binary component or something else, i.e. either star or general
; model. Since the parameter names are currently unique, this is not really
; necessary for the division of the case statements into two sections.
;
; Some parameter safeguards are implemented (e.g. e<1,a>0)
;
; If init=1 (only when called by set_h_steps), initialize increments
; based on Chi^2 increase when varying the step size. To be combined with
; fit_options.h=1 which causes these results to be used during fitting.
; Note that in this case, the Chi^2 of the mock data is zero!
;
common MarquardtFit,fit_options
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(init) eq 0 then init=0
;
maxparm=1000
if direction eq -1 then a=dblarr(maxparm)
da=a
chisq0=0.1	; target chisq increase for da with mock data
;
rad=180/!pi
;
n=0	; ID in flat parameter array
;
FOR j=0,n_elements(parameters)-1 DO BEGIN
;
if strlen(parameters(j).component) gt 0 then begin
;
binary=strpos(parameters(j).component,'-')
if binary gt 0 then begin
	ic=where(binary_model.component eq parameters(j).component) & ic=ic(0)
endif else begin
	ic=where(star_model.component eq parameters(j).component) & ic=ic(0)
endelse
index=where(strlen(parameters(j).names) gt 0,np)
for i=0,np-1 do begin
	parameter=parameters(j).names(i)
	if binary gt 0 then begin
	case parameter of
		'MassRatio': begin
			if direction eq 1 then $
				binary_model(ic).massratio=abs(a(n)) else $
				a(n)=binary_model(ic).massratio
			da(n)=0.01
			end
		'Semimajoraxis': begin
			if direction eq 1 then $
				binary_model(ic).semimajoraxis=abs(a(n)) else $
				a(n)=binary_model(ic).semimajoraxis
			da(n)=0.01	; mas
			da(n)=0.3	; mas
			if init then begin
				binary_model(ic).semimajoraxis= $
				binary_model(ic).semimajoraxis+da(n)
				calcmodel,/quiet
				binary_model(ic).semimajoraxis= $
				binary_model(ic).semimajoraxis-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				binary_error(ic).semimajoraxis= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and binary_error(ic).semimajoraxis ne 0 then $
				da(n)=binary_error(ic).semimajoraxis
			end
		'Inclination': begin
			if direction eq 1 then $
				binary_model(ic).inclination=a(n) mod 360 else $
				a(n)=binary_model(ic).inclination
			da(n)=0.01	; deg
			da(n)=0.1/(sin(binary_model(ic).inclination/rad)+0.1)
			if init then begin
				binary_model(ic).inclination= $
				binary_model(ic).inclination+da(n)
				calcmodel,/quiet
				binary_model(ic).inclination= $
				binary_model(ic).inclination-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				binary_error(ic).inclination= $
					sqrt(chisq0/(chisq/da(n)^2)) < 1
			endif else if fit_options.h $
				and binary_error(ic).inclination ne 0 then $
				da(n)=binary_error(ic).inclination
			end
		'Ascendingnode': begin
			if direction eq 1 then $
				binary_model(ic).ascendingnode=a(n) mod 360 else $
				a(n)=binary_model(ic).ascendingnode
			da(n)=0.01/(sin(binary_model(ic).inclination)+0.001)
			da(n)=atan(0.3/(binary_model(ic).semimajoraxis>0.01))*rad
			if init then begin
				binary_model(ic).ascendingnode= $
				binary_model(ic).ascendingnode+da(n)
				calcmodel,/quiet
				binary_model(ic).ascendingnode= $
				binary_model(ic).ascendingnode-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				binary_error(ic).ascendingnode= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and binary_error(ic).ascendingnode ne 0 then $
				da(n)=binary_error(ic).ascendingnode
			end
		'Eccentricity': begin
			if direction eq 1 then $
				binary_model(ic).eccentricity=a(n) < 0.99 else $
				a(n)=binary_model(ic).eccentricity
			da(n)=0.001
			if init then begin
				binary_model(ic).eccentricity= $
				binary_model(ic).eccentricity+da(n)
				calcmodel,/quiet
				binary_model(ic).eccentricity= $
				binary_model(ic).eccentricity-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				binary_error(ic).eccentricity= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and binary_error(ic).eccentricity ne 0 then $
				da(n)=binary_error(ic).eccentricity
			end
		'Periastron': begin
			if direction eq 1 then $
				binary_model(ic).periastron=a(n) else $
				a(n)=binary_model(ic).periastron
			da(n)=0.01/(binary_model(ic).eccentricity^2+0.001)
			da(n)=sqrt(atan(0.3 $
				/(binary_model(ic).semimajoraxis>0.01))*rad)
			if init then begin
				binary_model(ic).periastron= $
				binary_model(ic).periastron+da(n)
				calcmodel,/quiet
				binary_model(ic).periastron= $
				binary_model(ic).periastron-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				binary_error(ic).periastron= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and binary_error(ic).periastron ne 0 then $
				da(n)=binary_error(ic).periastron
			end
		'Apsidalmotion': begin
			if direction eq 1 then $
				binary_model(ic).apsidalmotion=a(n) else $
				a(n)=binary_model(ic).apsidalmotion
			da(n)=0.01/(binary_model(ic).eccentricity^2+0.001)
			if init then begin
				binary_model(ic).apsidalmotion= $
				binary_model(ic).apsidalmotion+da(n)
				calcmodel,/quiet
				binary_model(ic).apsidalmotion= $
				binary_model(ic).apsidalmotion-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				binary_error(ic).apsidalmotion= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and binary_error(ic).apsidalmotion ne 0 then $
				da(n)=binary_error(ic).apsidalmotion
			end
		'Epoch': begin
			if direction eq 1 then $
				binary_model(ic).epoch=a(n) else $
				a(n)=binary_model(ic).epoch
			da(n)=binary_model(ic).period/1000
			if init then begin
				binary_model(ic).epoch= $
				binary_model(ic).epoch+da(n)
				calcmodel,/quiet
				binary_model(ic).epoch= $
				binary_model(ic).epoch-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				binary_error(ic).epoch= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and binary_error(ic).epoch ne 0 then $
				da(n)=binary_error(ic).epoch
			end
		'Period': begin
			if direction eq 1 then $
				binary_model(ic).period=abs(a(n)) else $
				a(n)=binary_model(ic).period
			da(n)=binary_model(ic).period/10000
			da(n)=binary_model(ic).period/1000
			if init then begin
				binary_model(ic).period= $
				binary_model(ic).period+da(n)
				calcmodel,/quiet
				binary_model(ic).period= $
				binary_model(ic).period-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				binary_error(ic).period= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and binary_error(ic).period ne 0 then $
				da(n)=binary_error(ic).period
			end
		'Rho': begin
			if direction eq 1 then $
				binary_model(ic).rho=abs(a(n)) else $
				a(n)=binary_model(ic).rho
			da(n)=0.1	; mas
			end
		'Theta': begin
			if direction eq 1 then $
				binary_model(ic).theta=a(n) mod 360 else $
				a(n)=binary_model(ic).theta
			da(n)=0.001
			if binary_model(ic).rho gt 0 then $
			da(n)=atan(0.1/binary_model(ic).rho)*rad
			end
	endcase
	endif else begin ; not binary component
	pos=strpos(parameters(j).names(i),'Magnitude')
	if pos ge 0 then begin
		wave=float(strmid(parameter,10,5))
		ch=where(string(gen_model.wavelengths,format='(f5.2)')  eq wave)
		parameter=strmid(parameter,0,9)
	endif
	case parameter of
		'RV':	begin
			if direction eq 1 then $
				gen_model.rv=a(n) else $
				a(n)=gen_model.rv
			da(n)=1.0
			if init then begin
				gen_model.rv= $
					gen_model.rv+da(n)
				calcmodel,/quiet
				gen_model.rv= $
					gen_model.rv-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				gen_error.rv= $
					sqrt(chisq0/(chisq/da(n)^2)) < da(n)
			endif else if fit_options.h $
				and gen_error.rv ne 0 then $
				da(n)=gen_error.rv
			end
		'Parallax': begin
			if direction eq 1 then $
				gen_model.px=a(n) else $
				a(n)=gen_model.px
			da(n)=0.005*gen_model.px
			end
		'SM': begin
			if direction eq 1 then $
				gen_model.sm=a(n) else $
				a(n)=gen_model.sm
			da(n)=0.05*gen_model.sm
			end
		'Mass':	begin
			if direction eq 1 then $
				star_model(ic).mass=a(n) else $
				a(n)=star_model(ic).mass
			da(n)=star_model(ic).mass/40
			if init then begin
				star_model(ic).mass= $
					star_model(ic).mass+da(n)
				calcmodel,/quiet
				star_model(ic).mass= $
					star_model(ic).mass-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				star_error(ic).mass= $
					sqrt(chisq0/(chisq/da(n)^2)) < da(n)
			endif else if fit_options.h $
				and star_error(ic).mass ne 0 then $
				da(n)=star_error(ic).mass
			end
		'Diameter': begin
			if direction eq 1 then $
				star_model(ic).diameter=abs(a(n)) else $
				a(n)=abs(star_model(ic).diameter)
			da(n)=0.1
			if star_model(ic).type eq 15 then da(n)=1
			if init then begin
				star_model(ic).diameter= $
					star_model(ic).diameter+da(n)
				calcmodel,/quiet
				star_model(ic).diameter= $
					star_model(ic).diameter-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				star_error(ic).diameter= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and star_error(ic).diameter ne 0 then $
				da(n)=star_error(ic).diameter
			end
		'Width': begin
			if direction eq 1 then $
				star_model(ic).width=abs(a(n))>0.05 else $
				a(n)=star_model(ic).width
			da(n)=0.2
			if init then begin
				star_model(ic).width= $
					star_model(ic).width+da(n)
				calcmodel,/quiet
				star_model(ic).width= $
					star_model(ic).width-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				star_error(ic).width= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and star_error(ic).width ne 0 then $
				da(n)=star_error(ic).width
			end
		'Ratio': begin
			if direction eq 1 then $
				star_model(ic).ratio=abs(a(n)) else $
				a(n)=abs(star_model(ic).ratio)
			if a(n) gt 1 then begin
				a(n)=1/a(n)
				star_model(ic).ratio=a(n)
				star_model(ic).pa=(star_model(ic).pa+90) mod 360
			endif
			da(n)=0.05
			if init then begin
				star_model(ic).ratio= $
					star_model(ic).ratio+da(n)
				if a(n) gt 1 then begin
					a(n)=1/a(n)
					star_model(ic).ratio=a(n)
					star_model(ic).pa $
						=(star_model(ic).pa+90) mod 360
				endif
				calcmodel,/quiet
				star_model(ic).ratio= $
					star_model(ic).ratio-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				star_error(ic).ratio= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and star_error(ic).ratio ne 0 then $
				da(n)=star_error(ic).ratio
			end
		'PA': begin
			if direction eq 1 then $
				star_model(ic).pa=a(n) mod 360 else $
				a(n)=star_model(ic).pa mod 360
			da(n)=5.0
			if init then begin
				star_model(ic).pa= $
					star_model(ic).pa+da(n)
				calcmodel,/quiet
				star_model(ic).pa= $
					star_model(ic).pa-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				star_error(ic).pa= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and star_error(ic).pa ne 0 then $
				da(n)=star_error(ic).pa
			end
		'Tilt': begin
			if direction eq 1 then $
				star_model(ic).tilt=a(n) else $
				a(n)=star_model(ic).tilt
			da(n)=5.0
			end
		'Teff': begin
			if direction eq 1 then $
				star_model(ic).teff=a(n) else $
				a(n)=star_model(ic).teff
			da(n)=100
			if init then begin
				star_model(ic).teff= $
					star_model(ic).teff+da(n)
				calcmodel,/quiet
				star_model(ic).teff= $
					star_model(ic).teff-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				star_error(ic).teff= $
					sqrt(chisq0/(chisq/da(n)^2))
			endif else if fit_options.h $
				and star_error(ic).teff ne 0 then $
				da(n)=star_error(ic).teff
			end
		'Alpha': begin
			if direction eq 1 then $
				star_model(ic).alpha=a(n) else $
				a(n)=star_model(ic).alpha
			da(n)=0.1
			end
		'Omega': begin
			if direction eq 1 then $
				star_model(ic).omega=a(n) < 0.99 else $
				a(n)=star_model(ic).omega
			da(n)=0.02
			end
		'Magnitude': begin
			if direction eq 1 then $
				star_model(ic).magnitudes(ch)=a(n) else $
				a(n)=star_model(ic).magnitudes(ch)
			da(n)=0.1
			if init then begin
				star_model(ic).magnitudes(ch)= $
					star_model(ic).magnitudes(ch)+da(n)
				calcmodel,/quiet
				star_model(ic).magnitudes(ch)= $
					star_model(ic).magnitudes(ch)-da(n)
				chisq=modelchisq()
				if chisq ne 0 then $
				star_error(ic).magnitudes(ch)= $
					sqrt(chisq0/(chisq/da(n)^2)) < da(n)
			endif else if fit_options.h $
				and star_error(ic).magnitudes(ch) ne 0 then $
				da(n)=star_error(ic).magnitudes(ch)
			end
	endcase
	endelse
	n=n+1
endfor
;
endif	; valid component
;
ENDFOR
;
if direction eq -1 then begin
	if n eq 0 then a=0 else a=a(0:n-1)
endif
;
end
;-------------------------------------------------------------------------------
pro fitellipse,e_parms_in
;
; Fit apparent ellipse to position data. Pass e_parms_in if not determined
; previously (e_parms in common block).
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common FitEllipse,ex,ey
;
; Check data
if n_elements(positions) eq 0 then begin
	print,'***Error(FITELLIPSE): no data!'
	return
endif
;
; Check ellipse parameters
if n_elements(e_parms) eq 0 then begin
	if n_elements(e_parms_in) eq 5 then e_parms=e_parms_in else begin
	print,'***Error(FITELLIPSE): no ellipse parms!'
	return
	endelse
endif
;
; Set data.
index=where(positions.rho gt 0 $
	and positions.component eq ellipse_options.component,ndata)
if ndata gt 0 then begin
	p=positions(index).theta
	r=positions(index).rho
	sig=dblarr(ndata)+1
endif else begin
	print,'***Error(FITELLIPSE): no valid data!'
	return
endelse
;
; Set parameters to fit
ma=n_elements(e_parms)
ia=lonarr(ma)
if ellipse_options.all then ia(*)=1 else begin
	if ellipse_options.c then ia(0:1)=1
	if ellipse_options.a then ia(2)=1
	if ellipse_options.b then ia(3)=1
	if ellipse_options.p then ia(4)=1
endelse
index=where(ia ne 0,mfit)
if mfit eq 0 then begin
	print,'***Error(FITELLIPSE): no fit parameters selected!'
	return
endif
a=e_parms
;
; Store ellipse data
ex=r*sin(p)
ey=r*cos(p)
; plot,ex,ey
;
; Call marquardt
marquardt,'ellfuncs',p,r,sig,ndata,a,ia,ma,chisq,covar
;
; Set new parameters
e_parms=a
RAD=180/pi_circle
angle=(e_parms(4)*RAD) mod 360
if angle lt 0 then angle=angle+360
print,'__________________________________'
print,'Center x = ',e_parms(0)
print,'Center y = ',e_parms(1)
print,'Semi-major axis = ',e_parms(2)
print,'Semi-minor axis = ',e_parms(3)
print,'Position  angle = ',angle
print,'_______________***________________'
;
end
;-------------------------------------------------------------------------------
pro fitorbit
;
; Fit orbital elements to position data.
;
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common FitEllipse,ex,ey
common ModelFit,parameters,ds_options
;
; Check data
if n_elements(positions) eq 0 then begin
	print,'***Error(FITORBIT): no data!'
	return
endif
;
; Check orbit parameters
if n_elements(o_parms) eq 0 then begin
	print,'***Error(FITORBIT): no orbit parms!'
	return
endif
;
; Check whether orbit parameters have been initialized
if total(o_parms) eq 0 then begin
	print,'***Error(FITORBIT): bad starting model!'
	return
endif
;
; Set data.
index=where(positions.rho gt 0,ndata)
if ndata gt 0 then begin
	jd=positions(index).jd
	p=positions(index).theta
	r=positions(index).rho
	sig=positions(index).emajor
	case ds_options.c of
		0: 
		1: 
		2: sig(*)=1
	endcase
endif else begin
	print,'***Error(FITORBIT): no valid data!'
	return
endelse
;
; Set parameters to fit
; RAD=180/3.141592653d0
; o_parms=dblarr(8)
; o_parms(0)=50.0
; o_parms(1)=0.20
; o_parms(2)=127.0/RAD
; o_parms(3)=80.0/RAD
; o_parms(4)=20.0/RAD
; o_parms(5)=103.0
; o_parms(6)=7518.0
;
; Note: o_parms(8) is the orbital sense (not fit).
ma=n_elements(o_parms)-1
ia=lonarr(ma)
if orbit_options.all then ia(*)=1 else begin
	if orbit_options.a then ia(0)=1
	if orbit_options.e then ia(1)=1
	if orbit_options.i then ia(2)=1
	if orbit_options.w then ia(3)=1
	if orbit_options.n then ia(4)=1
	if orbit_options.p then ia(5)=1
	if orbit_options.t then ia(6)=1
endelse
index=where(ia ne 0,mfit)
if mfit eq 0 then begin
	print,'***Error(FITORBIT): no fit parameters selected!'
	return
endif
a=o_parms(0:6)
;
; Store ellipse data
ex=r*sin(p)
ey=r*cos(p)
;
; Compute chisq
; xy=true2app(jd,o_parms)
; xy(0,*)=0
; chisq=total(((ex-xy(*,0))/sig)^2+((ey-xy(*,1))/sig)^2)
;
; Set r to zero since orbfuncs does the differences
r=dblarr(ndata)
;
; Call marquardt
marquardt,'orbfuncs',jd,r,sig,ndata,a,ia,ma,chisq,covar
;
; Set new paramters
o_parms(0:6)=a(0:ma-1)
RAD=180/pi_circle
print,'__________________________________'
print,'Semi-major axis = ',o_parms(0)
print,'Eccentricity =    ',o_parms(1)
print,'Inclination =     ',o_parms(2)*RAD
print,'Periastron =      ',o_parms(3)*RAD
print,'Ascending node =  ',o_parms(4)*RAD
print,'Period =          ',o_parms(5)
print,'Epoch =           ',o_parms(6)
print,'_______________***________________'
;
end
;************************************************************************Block 4
pro marquardt,func,x,y,sig,ndata,a,ia,ma,chisq,covar,alamda,error=error
;
; Iterative procedure to do non-linear least-squares fit.
;
common MarquardtFit,fit_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
;
chifr=fit_options.chifr
tol=fit_options.tolerance
alamda=-fit_options.alamda
;
; Execute first iteration for setup and check of covariance matrix
chisq=0.d0
marquardtmin,x,y,sig,ndata,a,ia,ma,covar,alpha,chisq,func,alamda,tol
;
if not keyword_set(error) then begin
;
if not !quiet then print,'Running Marquardt...'
if not !quiet then print,'Initial chisq=',chisq
;
i=1
repeat begin
	chisqo=chisq
	marquardtmin,x,y,sig,ndata,a,ia,ma,covar,alpha,chisq,func,alamda,tol
	if not !quiet then print,'Iteration ',i,' completed; chisq = ',chisq
	if total(abs(covar)) eq 0 then begin
		print,'Error: covariance matrix is NULL!'
		return
	endif
	i=i+1
	chifr_crit=1.d0-chisq/chisqo lt chifr
	chisq_crit=chisq ge chisqo
	alamd_crit=(alamda gt 1)
	itera_crit=(i gt 200)
	fit_o_crit=(fit_options.o gt 0)
endrep until (1.d0-chisq/chisqo lt chifr and chisq lt chisqo) $
	or alamda gt 1 $
	or i gt 200 $
	or fit_options.o
print,'Iterations completed due to one or more of these conditions:'
if chifr_crit and not chisq_crit then begin
	print,'Convergence slow.'
	if i gt 15 then print,'To fit small diameters, start with small values!'
endif
if alamd_crit then print,'a_lambda > 1'
if itera_crit then print,'Maxmimum number of iterations reached.'
if fit_o_crit then print,'Just one iteration was requested.'
;
ny=n_elements(y)
if not !quiet then print,'Final   chisq=',chisq,', reduced = ',chisq/ny
if not !quiet then print,'Iter=',i,', Alamda=',alamda,format='(a,i2,a,f8.3)'
;
endif
;
; Compute parameter uncertainties
alamda=0.d0
marquardtmin,x,y,sig,ndata,a,ia,ma,covar,alpha,chisq,func,alamda,tol
;
end
;-------------------------------------------------------------------------------
function fixell,r,t,a,ex,ey
;
dx=ex-a(0)
dy=ey-a(1)
r=sqrt(dx^2+dy^2)
t=atan(dx,dy)-a(4)
esqr=1-(a(3)/a(2))^2
return,ellipse(a(2),a(3),t)
end
;-------------------------------------------------------------------------------
pro ellfuncs,p,r,a,rmod,drda,ma,ndata,ia
;
common FitEllipse,ex,ey
;
deltaa=dblarr(5)
; Fixed steps...
deltaa(0)=0.1	; Center x
deltaa(1)=0.1	; Center y
deltaa(2)=0.1	; Major axis
deltaa(3)=0.1	; Minor axis
deltaa(4)=0.02	; Position angle
; ...fractional step sizes...
deltaa(0)=sqrt(abs(a(2)*a(3)))/10	; Center x
deltaa(1)=sqrt(abs(a(2)*a(3)))/10	; Center y
deltaa(2)=a(2)/10	; Major axis
deltaa(3)=a(3)/10	; Minor axis
deltaa(4)=0.2		; Position angle
;
drda=dblarr(ndata,ma)
;
b=a
;
for j=0,ma-1 do begin
	if ia(j) gt 0 then begin
		b(j)=a(j)-deltaa(j)
		r_low=fixell(r,p,b,ex,ey)
		r_low=r_low-r
		b(j)=a(j)+deltaa(j)
		r_high=fixell(r,p,b,ex,ey)
		r_high=r_high-r
		b(j)=a(j)
		drda(*,j)=(r_high-r_low)/(2*deltaa(j))
	endif
endfor
;
rmod=fixell(r,p,a,ex,ey)
;
end
;-------------------------------------------------------------------------------
pro orbfuncs,jd,a,rmod,drda,ma,ndata,ia
;
common FitEllipse,ex,ey
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
deltaa=dblarr(7)
deltaa(0)=0.1			; Major axis increment
deltaa(1)=0.02			; Eccentricity
deltaa(2)=0.02			; Inclination
deltaa(3)=0.005/(a(1)^2+0.005)	; Periastron
deltaa(4)=0.02/(sin(a(2))+0.02)	; Node
deltaa(5)=a(5)/1000		; Period
deltaa(6)=a(5)/1000		; Epoch
;
if n_elements(gen_model) ne 0 then begin
	ra=gen_model.ra
	dec=gen_model.dec
endif
;
drda=dblarr(ndata,ma)
;
b=a
;
for j=0,ma-1 do begin
	if ia(j) gt 0 then begin
		b(j)=a(j)-deltaa(j) & bb=[b(0:3),0,b(4:6)]
		xy=true2app(jd,bb,rho,theta,gen_model.ra,gen_model.dec)
		r_low=sqrt((xy(*,0)-ex)^2+(xy(*,1)-ey)^2)
		b(j)=a(j)+deltaa(j) & bb=[b(0:3),0,b(4:6)]
		xy=true2app(jd,bb,rho,theta,ra,dec)
		r_high=sqrt((xy(*,0)-ex)^2+(xy(*,1)-ey)^2)
		b(j)=a(j)
		drda(*,j)=(r_high-r_low)/(2*deltaa(j))
	endif
endfor
;
aa=[a(0:3),0,a(4:6)]
xy=true2app(jd,aa,rho,theta,gen_model.ra,gen_model.dec)
rmod=sqrt((xy(*,0)-ex)^2+(xy(*,1)-ey)^2)
;
end
;-------------------------------------------------------------------------------
pro modfuncs,x,y,a,ymod,ysig,dyda,ma,ndata,ia
;
common MarquardtFit,fit_options
;
f=fit_options.f
fit_options.f=0	; Do not scale visibilities when computing derivatives
;
; Store model parameters and get step sizes
modelparmcopy,1,a,deltaa
;
; Compute derivatives
b=a
dyda=dblarr(ndata,ma)
for j=0,ma-1 do begin
	if ia(j) gt 0 then begin
		b(j)=a(j)-deltaa(j)
		modelparmcopy,1,b
		calcmodel,/quiet
		marquardtdata,mrq_y,mrq_ysig,y_low
		b(j)=a(j)+deltaa(j)
		modelparmcopy,1,b
		calcmodel,/quiet
		marquardtdata,mrq_y,mrq_ysig,y_high
		dyda(*,j)=(y_high-y_low)/(2*deltaa(j))
		b(j)=a(j)
	endif
endfor
;
; Compute model
modelparmcopy,1,a
calcmodel,/quiet
marquardtdata,y,ysig,ymod
;
; Astrometry fits vary sigma based on the orientation of the uncertainty 
; ellipse wrt to the model position
;
fit_options.f=f	; Restore original choice
;
end
;-------------------------------------------------------------------------------
pro funct_gauss,x,a,f
;
; Gauss fit, a(2) is FWHM, the relationship between sigma and FWHM is:
; sigma=fwhm/(2*sqrt(2*alog(2))) where Gaussian is exp(-(x^2/(2*sigma^2)))
;
exponents=-(2*sqrt(alog(2))*(x-a(1))/(a(2)))^2
index=where(exponents lt -30,count)
if count gt 0 then exponents(index)=-30
f=a(0)*exp(exponents)
;
end
;-------------------------------------------------------------------------------
function mask2pix,winv,lambda,a,flux=flux
;
; Given polynomial coefficients describing the position of the mask, as well
; as width information, fit mask profiles to each column and return
; the values for an evaluation of how well this masks follows the
; spectrum.
;
; x0 is the reference pixel for the polynom definition: x0(PRISM)=130
; a=[a0,a1,a2], polynomial fit coefficients, a3=width in pixels
;
n=n_elements(winv)
nx=n_elements(lambda)
ny=n/nx
wina=reform(winv,nx,ny)
winf=wina
x=findgen(nx)
y=findgen(ny)
x0=a(4)
y0=poly(x-x0,a(0:2))
w=a(3)*lambda/lambda(fix(x0))
w=a(3)+lambda*0	; no scaling with lambda
flux=fltarr(nx)
;
for j=0,nx-1 do begin
	p=reform(wina(j,*))
	gp=[max(p),y0(j),w(j)]	; parameters (first one to be fit)
	winf(j,*)=curvefit(y,p,p*0+1,gp,fita=[1,0,0], $
		function_name='funct_gauss',/noderivat)
	flux(j)=gp(0)
endfor
;
return,reform(winf,n)
;
end
;-------------------------------------------------------------------------------
pro maskfuncs,x,y,a,ymod,ysig,dyda,ma,ndata,ia
;
common FitMask,lambda,x0,phot_file,det_window,coeffs,num_bad_col,num_bad_row
;
; a: a0,a1,a2: polynomial coefficients, a3=width at x0
;
a(3)=abs(a(3))		; width must not become negative
;
deltaa=dblarr(5)
deltaa(0)=0.5		; a0, the position at x0
deltaa(1)=0.02		; a1, linear term
deltaa(2)=0.0001	; a2, quadratic term
deltaa(3)=0.1		; width
deltaa(4)=1.0		; This position corresponds to x0, which is never fit.
;
dyda=dblarr(ndata,ma)
;
b=a
;
for j=0,ma-1 do begin
	if ia(j) gt 0 then begin
		b(j)=a(j)-deltaa(j)
		y_low=mask2pix(y,lambda,b)
		b(j)=a(j)+deltaa(j)
		y_high=mask2pix(y,lambda,b)
		b(j)=a(j)
		dyda(*,j)=(y_high-y_low)/(2*deltaa(j))
	endif
endfor
;
ymod=mask2pix(y,lambda,a)
;
end
;************************************************************************Block 5
pro set_h_steps,file
;
; One-time initialization of optimal parameter steps for computing the
; numerical derivatives given the data and a model. Instructions: read all
; data, then call this procedure with the name of the model file.
; The observed data are overwritten with model data during this procedure,
; therefore the former have to be reloaded after calling this procedure!
;
forward_function genparms,starparms,binparms,posparms
;
common ModelFit,parameters,ds_options
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(file) eq 0 then begin
	print,'Must specify output file for set_h_steps!'
	return
endif
;
; Determine if the binary fit uses orbit or relative component positions
index=where(parameters.names eq 'Rho' or parameters.names eq 'Theta',count)
if count ge 1 then orbit=0 else orbit=1
;
; Save copy of current parameters
parameters_bck=parameters
;
if init_modelfit() ne 0 then return
if init_fitinterferometry() ne 0 then return
;
parameters(0).component=systemcomp()
parameters(0).names=genparms()
;
for i=0,num_star()-1 do begin
	parameters(i+1).component=star_model(i).component
	parameters(i+1).names=starparms()
endfor
;
for i=0,num_binary()-1 do begin
	parameters(i+1+num_star()).component=binary_model(i).component
	if orbit then parameters(i+1+num_star()).names=binparms() $
		 else parameters(i+1+num_star()).names=posparms()
endfor
;
calcmodel,/quiet
if ds_options.i then mockdata
if ds_options.a then begin
	positions.rho=positions.rhom
	positions.theta=positions.thetam
endif
if ds_options.s then velocities.value=velocities.valuem
;
modelparmcopy,-1,a,/init
;
save,gen_error,star_error,binary_error,filename=file
;
parameters=parameters_bck
;
end
;-------------------------------------------------------------------------------
pro setorbit
;
; Initialize local orbital parameters using the loaded binary model.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
RAD=180/pi_circle
;
if n_elements(binary_model) eq 0 then begin
	print,'***Error(SETORBIT): no binary_model loaded!'
	return
endif
;
index=where(binary_model.component eq orbit_options.component,count)
if count eq 0 then begin
	print,'***Error(SETORBIT): component not found in binary_model!'
	return
endif
o_parms=dblarr(8)
o_parms(0)=binary_model(index).semimajoraxis
o_parms(1)=binary_model(index).eccentricity
o_parms(2)=binary_model(index).inclination/RAD
o_parms(3)=binary_model(index).periastron/RAD
o_parms(4)=binary_model(index).ascendingnode/RAD
o_parms(5)=binary_model(index).period
o_parms(6)=binary_model(index).epoch
;
if not !quiet then print,'Orbit set.'
;
end
;-------------------------------------------------------------------------------
pro setmodel,star
;
; Using the local orbital elements, initialize a binary model.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common StarBase,StarTable,Notes
;
RAD=180/pi_circle
;
if n_elements(binary_model) eq 0 then begin
	print,'***Warning(SETMODEL): initializing model...'
;
	wavelengths=[0.550d0,0.700d0,0.850d0]
	num_wave=n_elements(wavelengths)
	gen_model=alloc_gen_model(num_wave)
	gen_model.starid='CCCNNNN'
	gen_model.wavelengths=wavelengths
	gen_error=gen_model
;
	numspot=2
	star_struct=alloc_star_struct(num_wave,numspot)
	star_model=replicate(star_struct,2)
	star_model(0).component='A'
	star_model(1).component='B'
	star_model(*).mass=1
	star_error=star_model
;
	binary_struct=alloc_binary_struct()
	binary_model=replicate(binary_struct,1)
	binary_model(0).component='A-B'
	binary_model(0).method=1
	binary_error=binary_model
;
endif
;
if n_elements(star) ne 0 then begin
	index=where(StarTable.starid eq star,count)
	if count eq 0 then return
	gen_model.starid=star
	gen_model.ra=startable(index).ra
	gen_model.dec=startable(index).dec
	star_model.mass=1
        binary_model.semimajoraxis=startable(index).a
        binary_model.inclination=startable(index).i
        binary_model.ascendingnode=startable(index).n
        binary_model.eccentricity=startable(index).e
        binary_model.periastron=startable(index).o
        binary_model.apsidalmotion=0
        binary_model.epoch=startable(index).t-2440000L
        binary_model.period=startable(index).p
	return
; 
endif
;
index=where(binary_model.component eq orbit_options.component,count)
if count eq 0 then begin
	print,'***Error(SETMODEL): component not found in binary_model!'
	return
endif
binary_model(index).semimajoraxis	=o_parms(0)
binary_model(index).eccentricity	=o_parms(1)
binary_model(index).inclination		=o_parms(2)*RAD
binary_model(index).periastron		=o_parms(3)*RAD
binary_model(index).ascendingnode	=o_parms(4)*RAD
binary_model(index).period		=o_parms(5)
binary_model(index).epoch		=o_parms(6)
;
if not !quiet then print,'Model set.'
;
end
;************************************************************************Block 6
pro fitnights,plotselection=plotselection,chi2map=chi2map,sigma=sigma,cf=cf
;
; Fit interferometry (usually including separation and position angle,
; possibly with diameters and magnitude differences) to single/multiple nights, 
; obtaining initial estimates for rho and theta from the orbit model. 
;
; The fit parameters must be selected through the widgets (FIT|Interferometry)!
;
; Combines different configurations of the same night into a single fit.
; The reference epoch for the astrometry corresponds to the MIDNIGHT value for
; each interferometer.
;
; plotselection: if true, process only the nights selected in the plot widget
;
; chi2map: if true, use previously created gridchisq maps (YYYY-MM-DD.chisq)
; to fit error ellipses to the local minima. The maps should have a pixel 
; size of 0.01 mas and 100 x 100 or more pixels.
;
; Parameters sigma (default: 1) and cf (default: 7) are passed to fitchisq.
; cf is the correlation factor. For NPOI, we assume a full correlation of the
; first 7 channels (all others are usually not included in the fitting process)
; Selections in papers: 73 Leo sigma=1, cf=10; Alpha Dra sigma=4, cf=1
;
; Note: error ellipse axes are semi-major/minor axes!
;
common MarquardtFit,fit_options
common ModelFit,parameters,ds_options
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common DataSelInfo,class,type,slice,ds_nights,ds_stars,ds_x,ds_y,ds_z,ps_options
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common LocalFitNights,cbcf
;
RAD=180/pi_circle
;
index=where(parameters.names(0) eq 'Rho' or parameters.names(0) eq 'Theta',n)
if n eq 0 then begin
	print,'Neither Rho nor Theta selected to fit!'
	return
endif
for i=0,n-1 do begin
	j=where(binary_model.component eq parameters(index(i)).component,nj)
	if binary_model(j).method eq 1 and binary_model(j).period eq 0 then $
	begin
		print,'Error: selected component has no orbit defined!'
		return
	endif
endfor
;
if n_elements(cbcf) eq 0 then cbcf=7.0
;
; Passed to fitchisq:
if n_elements(cf) eq 0 then cf=7.0 else cf=float(cf)
if n_elements(sigma) eq 0 then sigma=1.0 else sigma=float(sigma)
;
; Save geoinfo, geninfo, and bufferinfo
geoinfo_bck=geoinfo
geninfo_bck=geninfo
if n_elements(bufferinfo) gt 1 then bufferinfo_bck=bufferinfo
list_nights,ds_nights_bck
;
if keyword_set(plotselection) then begin
; 	Here the nights are selected in the plot widget
	if n_elements(ds_nights) eq 0 then plotinit=init_plot('amoeba','pt')
	if set_nightsel() ne 0 then return
	nfit=n_elements(ds_nights)
endif else begin
	files=''
	if n_elements(bufferinfo) gt 1 then $
	files=bufferinfo.file
	dates=geninfo.date
	udates=unique(dates)
	nfit=n_elements(udates)
endelse
;
; Save other current settings
methods=binary_model.method
binary_model_bck=binary_model
weight_i=ds_options.i
weight_s=ds_options.s
weight_p=ds_options.p
weight_a=ds_options.a
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
chifr=fit_options.chifr
;
; Assume all components w/defined orb. elements are used to compute (rho,theta)
index=where(binary_model.period ne 0,count)
if count gt 0 then binary_model(index).method=1
;
; Make sure we find the minimum
fit_options.chifr=0.001
;
; Make sure modelchisq() uses only interferometry
ds_options.i=1
ds_options.s=0
ds_options.p=0
ds_options.a=0
;
; Make backup of parameters
parameters_bck=parameters
;
; Open results file
openw,unit,'fitnights.psn',/get_lun
printf,unit, $
'!                                 Semi-  Semi-                                '
printf,unit, $
'!Comp Julian Year  Rho   Theta    Major  Minor   PA     Rho_err  Theta_err Chi2'
;
for i=0,nfit-1 do begin
	if keyword_set(plotselection) then begin
		print,ds_nights(i)
		k=where(geninfo_bck.date+' ' $
		       +geoinfo_bck.systemid+' ' $
		       +geninfo_bck.configid eq ds_nights(i),count)
;		This ensures only this data set is used
		geoinfo=geoinfo_bck(k)
		geninfo=geninfo_bck(k)
		bufferinfo=bufferinfo_bck(k)
;		Calcmodel does not load data if only one night is in buffer
;		Only read from buffer if mode is AMOEBA
		if count eq 1 and n_elements(bufferinfo_bck) gt 1 $
		then loadnight,ds_nights(i)
	endif else begin
		k=where(dates eq udates(i),count)
		print,'Date = ',udates(i),', files = ',count
		geoinfo=geoinfo_bck(k)
		geninfo=geninfo_bck(k)
		if n_elements(bufferinfo_bck) gt 1 then $
		bufferinfo=bufferinfo_bck(k)
		ds_nights=ds_nights_bck(k)
;		Calcmodel does not load data if only one night is in buffer
;		Only read from buffer if mode is AMOEBA
		if count eq 1 and n_elements(bufferinfo_bck) gt 1 $
		then loadnight,ds_nights(0)
	endelse
	printf,unit,'!     '+date
	midnight=system_config(systemid,'MIDNIGHT')
	parsedate,geninfo(0).date,y,m,d
	d=d+midnight/24
	epoch_jd=julian(y,m,d)
	epoch_jy=jd2jy(epoch_jd)
;	Obtain predictions for (rho,theta) for this night, requires method=1
	for j=0,num_binary()-1 do begin
		rt=binarypos(epoch_jd,binary_model(j).component)
		binary_model(j).rho=rt(0)
		binary_model(j).theta=rt(1)
	endfor
	if binary_model(0).method eq 4 then begin
;		TBC (used only in UX Ari paper
;		parameters.names=['Semimajoraxis','Ascendingnode']
	endif else begin
		binary_model.method=2
	endelse
;	For those components which have orbital elements include orb. motion
	index=where(methods eq 1,count)
	if count gt 0 then binary_model(index).method=3
;	Fit model
	fitmodel,chisq,ndata
;	writemodel,geninfo(0).date+'.model.xdr'
	parameters=parameters_bck
	binary_model.method=methods
;
;	If successful, compute error ellipse and write results
	IF ndata ne 0 THEN BEGIN
;
	locations=replicate(location(),num_binary())
;
	FOR j=0,num_binary()-1 DO BEGIN
;	Don't print components which were not fit
	IF (binary_error(j).rho NE 0 AND $
	    binary_error(j).theta NE 0) OR $
	   (binary_error(j).semimajoraxis NE 0 AND $
	    binary_error(j).ascendingnode) THEN BEGIN
;
;	Compute CLEAN beam size
	print,'------------Computing CLEAN beam size------------'
	if keyword_set(plotselection) then begin
		db=dirtybeam(gen_model.starid)
	endif else begin
		if n_elements(bufferinfo_bck) gt 1 then $
		loadnight,ds_nights(0)
		db=dirtybeam(gen_model.starid)
		for k=1,n_elements(ds_nights)-1 do begin
			loadnight,ds_nights(k)
			db=db+dirtybeam(gen_model.starid)
		endfor
		db=db/n_elements(ds_nights)
	endelse
	rms=abs(min(db)/max(db))	; This does not consider vis. RMS
	e_parms=cleanbeam(db)
	if e_parms(0) lt 0 then begin
;		Clean-up before returning
		if nfit gt 1 then binary_model=binary_model_bck
		geoinfo=geoinfo_bck
		geninfo=geninfo_bck
		if n_elements(bufferinfo_bck) gt 1 then $
		bufferinfo=bufferinfo_bck
		ds_options.i=weight_i
		ds_options.s=weight_s
		ds_options.p=weight_p
		ds_options.a=weight_a
		fit_options.chifr=chifr
	endif
	e_parms(0)=e_parms(0)/2		; Convert to semi-major axis
	e_parms(1)=e_parms(1)/2		; Convert to semi-minor axis
	e_parms(2)=e_parms(2)/RAD
;
	if binary_model(0).method ne 4 then binary_model.method=2
	locations(j).jy=epoch_jy
	locations(j).component=binary_model(j).component
;	These should be the photocenter coordinates
	rt=binarypos(epoch_jd,binary_model(j).component)
	locations(j).rho=rt(0)
	locations(j).theta=rt(1)/RAD
	if locations(j).theta lt 0 then locations(j).theta= $
					locations(j).theta+2*pi_circle
;	
;	In the following, we try different methods to estimate 
;	the uncertainty ellipse 
	locations(j).pa=e_parms(2)
;
;	Derive from CLEAN beam, reduction factor from map RMS
;	Fomalont, Image Analysis, in Perley et al., Synth. Im. Rad. Ast., p.219
	dsp=ds_options.tp
	dsa=ds_options.ta
	ds_options.ta=-1
	ds_options.tp=-1
	marquardtdata,y,ysig,ymod
	ds_options.tp=dsp
	ds_options.ta=dsa
	rms=stddev(y-ymod)/sqrt(2)
	f=2.0/rms
	locations(j).emajor=e_parms(0)/f
	locations(j).eminor=e_parms(1)/f
;
;	Adjust uncertainty ellipse to equal an increase 
;	in total chisq by 1*reduced chisq
	x=locations(j).rho*sin(locations(j).theta) $
	 -locations(j).emajor*sin(locations(j).pa)
	y=locations(j).rho*cos(locations(j).theta) $
	 -locations(j).emajor*cos(locations(j).pa)
	binary_model(j).rho=sqrt(x^2+y^2)
	binary_model(j).theta=atan(x,y)*RAD
	calcmodel,/quiet
	a=abs(modelchisq()-chisq)/locations(j).emajor^2
	f1=locations(j).emajor/sqrt((max([chisq,1])/ndata)/a)
	x=locations(j).rho*sin(locations(j).theta) $
	 -locations(j).eminor*cos(locations(j).pa)
	y=locations(j).rho*cos(locations(j).theta) $
	 +locations(j).eminor*sin(locations(j).pa)
	binary_model(j).rho=sqrt(x^2+y^2)
	binary_model(j).theta=atan(x,y)*RAD
	calcmodel,/quiet
	a=abs(modelchisq()-chisq)/locations(j).eminor^2
	f2=locations(j).eminor/sqrt((max([chisq,1])/ndata)/a)
	f=sqrt(f1*f2)	; maintain beam ratio
	locations(j).emajor=e_parms(0)/f
	locations(j).eminor=e_parms(1)/f
;
;	All of the above may return too optimistic errors
	f=5.0	; Standard radio CLEAN beam/5 criterion
	f=7.0	; Used for Eta Virginis NPOI data
	f=22.0	; Used for UX Ari CHARA data
	f=40.0	; Used in Mizar A NPOI data
	f=40.0	; Used for 41 Eri PIONIER data
	f=cbcf  ; User user-defined conversion factor
	locations(j).emajor=e_parms(0)/f
	locations(j).eminor=e_parms(1)/f
;
;	Grid fit, if requested (not triples or higher)
;	Use fitchisq procedure also to extract error ellipse
	if n_elements(chi2map) ne 0 then begin
	chisqfile=Date+'.chisq'
	fitchisq,chisqfile,sigma=sigma,cf=cf,component=binary_model(j).component
	status=dc_read_free('fitchisq.psn',c,jy,r,t,ea,eb,pa,/col,ignore=['!'])
	locations(j).emajor=ea;/sqrt(ndata)
	locations(j).eminor=eb;/sqrt(ndata)
	locations(j).pa=pa/RAD
	endif
;
	r=ellipse(locations(j).emajor, $
	          locations(j).eminor, $
	          locations(j).theta-e_parms(2))
	factor=(locations(j).rho/100)/r
	if factor gt 1 and geoinfo(0).SystemId eq 'not_obsolete' then begin
		d_pa=e_parms(2)-locations(j).theta
		if d_pa lt 0 then d_pa=d_pa+!pi
		new_parms=apollonius(locations(j).emajor, $
				     locations(j).eminor, $
				     d_pa,factor)
		locations(j).emajor=new_parms(0)
		locations(j).eminor=new_parms(1)
		locations(j).pa=(new_parms(2)+locations(j).theta) mod !pi
	endif
;
; 	Write results to file
	printf,unit,locations(j).component,locations(j).jy, $
	      	locations(j).rho,locations(j).theta*RAD, $
		locations(j).emajor,locations(j).eminor,locations(j).pa*RAD, $
		binary_error(j).rho,binary_error(j).theta,chisq, $
		format='(a5,1x,f10.5,1x,f7.3,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
		       	'3x,f7.3,1x,f8.3,1x,f6.1)'
;
	binary_model.method=methods
	ENDIF
	ENDFOR
	ENDIF
print,'-------------------------------------------------'
print,''
;
endfor
;
; Close file
free_lun,unit
;
; Print information about CBCF used
print,'CLEAN beam conversion factor used: ',cbcf
;
; Set some of the parms back to previous values
if nfit gt 1 then binary_model=binary_model_bck
geoinfo=geoinfo_bck
geninfo=geninfo_bck
if n_elements(bufferinfo_bck) gt 1 then $
bufferinfo=bufferinfo_bck
; if n_elements(files) eq 0 then begin
; 	geoinfo=geoinfo_bck
; 	geninfo=geninfo_bck
; 	bufferinfo=bufferinfo_bck
; endif
;
ds_options.i=weight_i
ds_options.s=weight_s
ds_options.p=weight_p
ds_options.a=weight_a
fit_options.chifr=chifr
;
end
;-------------------------------------------------------------------------------
pro fitnights_rescale,psn_file
;
; Re-scale astrometric error ellipses (from psn_file) to achieve a Chi^2=1
; given a model. Write psn_file (_rescaled).
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_elements(binary_model) eq 0 then begin
	modelfile=''
	read,modelfile,prompt='Please enter name of model file: '
	readmodel,modelfile
endif
;
l=''
status=dc_read_fixed(psn_file,l,/col,format='(a80)')
if status ne 0 then begin
	print,'Please specify name of astrometry (.psn) file!'
	return 
endif else n=n_elements(l)
load_astrometry,psn_file
calcmodel
f=sqrt(modelchisq())
;
new_file=strmid(psn_file,0,strpos(psn_file,'.psn'))+'_rescaled.psn'
openw,unit,new_file,/get_lun
;
for i=0,n-1 do begin
	if strpos(l(i),'!') eq 0 then begin
		printf,unit,l(i)
	endif else begin
		words=nameparse(l(i))
		words(4)=string(float(words(4))*f,format='(f6.3)')
		words(5)=string(float(words(5))*f,format='(f6.3)')
		printf,unit,strjoin(words,'  ')
	endelse
endfor
;
free_lun,unit
;
print,'Rescaled error ellipses, factor =',f	
print,'File written: ',new_file
load_astrometry,new_file
calcmodel
;
end
;-------------------------------------------------------------------------------
pro gridchisq,num_grid,grid_cell,component=component,files=files,chi2map=chi2map
;
; Given a loaded binary model (orbit: method 1), predict rho/theta for the
; observation date, and look for the global minimum in ChiSq by varying r/t
; for the specified component, e.g. 'A-B'. Make sure no data other than
; interferometry are loaded because of the use of modelchisq() to compute ChiSq.
;
; Write Chi^2 surface data to file YYYY-MM-DD.chisq.
;
; Loop over files if given, otherwise use the currently loaded data.
; e.g. gridchisq,150,0.04,files='200?-??-??.cha'
; e.g. gridchisq,150,0.04,files='201?-??-??.cha'
; e.g. gridchisq,150,0.04,files='201[7-8]-??-??.cha'
;
; This procedure can be called also by gridchisq_parallel
;
; If keyword chi2map is set, num_grid=100 and grid_cell=cb/200 are used. 
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common MarquardtFit,fit_options
common OysterBridge,obd
;
rad=180/!pi
;
; Restore data if run within IDL bridge object
if n_elements(obd) ne 0 then restore,obd
;
if n_elements(binary_model) eq 0 then begin
	modelfile=''
	read,modelfile,prompt='Please enter name of model file: '
	readmodel,modelfile
endif
if n_elements(component) eq 0 and num_binary() gt 1 then begin
	component=''
	read,component,prompt='Please enter component (e.g. A-B): '
endif
if n_elements(component) ne 0 then k=where(binary_model.component eq component)$
			      else k=0
method_input=binary_model(k).method
;
if keyword_set(chi2map) then begin
	print,'Parameters for use with fitchisq requested!'
	cb=cleanbeam(dirtybeam(gen_model.starid))
	grid_cell=cb(1)/100
	num_grid=150
	print,'Will use num_grid',num_grid,' and grid_cell ',grid_cell
endif
if n_elements(num_grid) eq 0 then num_grid=100L else num_grid=long(num_grid)
if n_elements(grid_cell) eq 0 then grid_cell=0.5
;
print,'Grid cell [mas]: ',grid_cell
print,'Size of search box [mas]: ',num_grid*grid_cell
;
if n_elements(files) ne 0 then begin
	files=file_search(files)
	if strlen(files(0)) eq 0 then begin
		print,'GRIDFIT: Files not found!'
		return
	endif
	n_files=n_elements(files)
endif else n_files=1
;
FOR i_file=0,n_files-1 DO BEGIN
;
status=0
if n_elements(files) ne 0 then begin
	if extname(files(i_file)) eq 'cha' then get_data,files(i_file),status
	if extname(files(i_file)) eq 'fits' then begin
		get_oifits,files(i_file),status
	endif
endif
if status ne 0 then begin
	print,'Error reading files!'
	retall
endif
;
parsedate,date,y,m,d
d=float(d)+system_config(systemid,'MIDNIGHT')/24.
for i=0,n_elements(binary_model)-1 do begin
	jd=julian(y,m,d)
	rt=binarypos(jd,binary_model(i).component)
	if rt(1) lt 0 then rt(1)=rt(1)+360
	binary_model(i).rho=rt(0)
	binary_model(i).theta=rt(1)
endfor
binary_model.method=2
;
x0=-binary_model(k).rho*sin(binary_model(k).theta/rad)
y0=+binary_model(k).rho*cos(binary_model(k).theta/rad)
;
n=num_grid*num_grid
grid_x=(lindgen(n) mod num_grid)*grid_cell
grid_y=(lindgen(n)/num_grid)*grid_cell
grid_x=reform(grid_x,num_grid,num_grid)-num_grid/2*grid_cell+x0
grid_y=reform(grid_y,num_grid,num_grid)-num_grid/2*grid_cell+y0
;
rho=sqrt(grid_x^2+grid_y^2)
theta=atan(-grid_x,grid_y)*rad
;
chisq=fltarr(num_grid,num_grid)
calcmodel,/quiet
print,'Starting location rho= ',binary_model(k).rho, $
			', theta=',binary_model(k).theta
print,'Start at location RA = ', $
	binary_model(k).rho*sin(binary_model(k).theta/rad), $
	', Dec = ',binary_model(k).rho*cos(binary_model(k).theta/rad)
print,'Starting reduced Chi^2: ',modelchisq(n_deg_free)
;
completed=1
;
for i=0L,n-1 do begin
	binary_model(k).rho=rho(i)
	binary_model(k).theta=theta(i)
	calcmodel,/quiet
	chisq(i)=modelchisq()
	if ((i*100)/n)/10 eq completed then begin
		print,string(13b),completed*10,'% completed',format='(a,i,a,$)'
		completed=completed+1
	endif
endfor
print,''
;
index=where(chisq eq min(chisq)) & index=index(0)
if theta(index) lt 0 then theta(index)=theta(index)+360
print,'Minimum Chi^2 found at rho= ',rho(index),', theta=',theta(index)
print,'Location of minimim at RA = ',rho(index)*sin(theta(index)/rad), $
			   ', Dec = ',rho(index)*cos(theta(index)/rad)
print,'Minimum red. Chi^2 : ',min(chisq)
; print,'Index = ',index mod num_grid,index/num_grid
binary_model(k).rho=rho(index)
binary_model(k).theta=theta(index)
print,'Model values updated.'
print,'-------------------------------------------------------------------'
calcmodel,/quiet
;
magnitudes=star_model.magnitudes
diameters=star_model.diameter
posangles=star_model.pa
rho_best=rho(index)
theta_best=theta(index)
chisq_best=chisq(index)
if n_elements(files) eq 0 then files=Date
filebase=basename(files(i_file)) & filebase=filebase(0)
num_base=strlen(file_search(filebase+'.chisq'))
if num_base(0) eq 0 then begin
	outfile=filebase+'.chisq'
endif else begin
	words=nameparse(systime())
	outfile=filebase+'_'+strjoin(words,'_')+'.chisq'
endelse
; outfile=basename(files(i_file))+'.chisq' & outfile=outfile(0)
save,systemid,date,jd,n_deg_free,chisq,theta,rho,grid_cell, $
	magnitudes,diameters,rho_best,theta_best,chisq_best, $
	filename=outfile
print,"To show the Chi^2 surface, run: gridplot,'"+outfile+"',/rd"
;
binary_model.method=method_input
;
ENDFOR
;
end
;-------------------------------------------------------------------------------
pro gridchisq_parallel,num_grid,grid_cell,component=component,files=files_in, $
	chi2map=chi2map
;
; Multi-processor wrapper of gridchisq.
;
; Must specify files (CHA format).
; If keyword chi2map is set, num_grid=100 and grid_cell=cb/200 are used. 
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common OysterBridge,obd
;
; Setup for gridchisq
;
if n_elements(binary_model) eq 0 then begin
	modelfile=''
	read,modelfile,prompt='Please enter name of model file: '
	readmodel,modelfile
endif
if n_elements(component) eq 0 and num_binary() gt 1 then begin
	component=''
	read,component,prompt='Please enter component (e.g. A-B): '
endif
if n_elements(component) ne 0 then k=where(binary_model.component eq component)$
			      else k=0
component=binary_model(k).component
;
if keyword_set(chi2map) then begin
	print,'Parameters for use with chi2map requested!'
	cb=cleanbeam(dirtybeam(gen_model.starid))
	grid_cell=cb(1)/200
	num_grid=160
	print,'Will use num_grid',num_grid,' and grid_cell ',grid_cell
endif
;
if n_elements(num_grid) eq 0 then num_grid=100L else num_grid=long(num_grid)
if n_elements(grid_cell) eq 0 then grid_cell=0.5
;
print,'Grid cell [mas]: ',grid_cell
print,'Size of search box [mas]: ',num_grid*grid_cell
;
if n_elements(files_in) ne 0 then begin
	all_files=file_search(files_in)
	if strlen(all_files(0)) eq 0 then begin
		print,'GRIDFIT: Files not found!'
		return
	endif
	n=n_elements(all_files)
endif else begin
	print,'No files specified!'
	return
endelse
;
; Determine size of each data set
m=fltarr(n)
for i=0,n-1 do begin
;	get_data,all_files(i)
	if extname(all_files(i)) eq 'cha' then get_data,all_files(i)
	if extname(all_files(i)) eq 'fits' then get_oifits,all_files(i)
	r=modelchisq(nfree)
	m(i)=nfree
	if extname(all_files(i)) eq 'cha' then hds_close

endfor
;
; Re-order files by size and interleaf biggest with smallest to reduce load
si=sort(m)
m=m(si)
all_files=all_files(si)
index_s=indgen(n/2+(n mod 2))
index_b=indgen(n/2)+n/2+(n mod 2)
index=intarr(n)
index(indgen(n/2+(n mod 2))*2)=index_s
index(indgen(n/2)*2+1)=reverse(index_b)
all_files=all_files(index)
m=m(index)
;
; Resources: number of CPUs used (n_cpu) and files per CPU (nf_cpu)
ncpu=!cpu.hw_ncpu < 16	; limit due to IDL error when destroying bridge objects
nf_cpu=nint(float(n)/ncpu) > 1			; Number of files per CPU (tbc)
if ncpu*nf_cpu lt n then nf_cpu=nf_cpu+1	; Add 1 file/CPU if needed
n_cpu=n/nf_cpu < ncpu 				; Number of CPU used (tbc)
if n mod nf_cpu gt 0 then n_cpu=n_cpu+1		; Add one CPU if needed
nf_cpu_last=n mod nf_cpu 			; Files processed by last CPU
print,'       Files,         CPU,  files/CPU, CPU req., files (last CPU)'
print,n,ncpu,nf_cpu,n_cpu,nf_cpu_last
;
; Swap biggest file to last CPU if it has fewer files to process
if nf_cpu_last gt 0 and nf_cpu_last lt (nf_cpu-1) then begin
	m_big=m(1)
	m(1)=m(n-1)
	m(n-1)=m_big
	file_big=all_files(1)
	all_files(1)=all_files(n-1)
	all_files(n-1)=file_big
endif
a_files=strarr(nf_cpu+1,n_cpu)
n_files=intarr(n_cpu)
m_files=fltarr(n_cpu)
for j=0,n_cpu-1 do begin
;	if j*nf_cpu gt n-1 then break
	f=unique(all_files((j*nf_cpu)<(n-1):((j+1)*nf_cpu-1)<(n-1)))
	n_files(j)=n_elements(f)
	a_files(0:n_files(j)-1,j)=f
	index=whereequal(all_files,[f,''])
	m_files(j)=total(m(index))
endfor
print,'Number of measurements to compute per CPU:'
for j=0,n_cpu-1 do print,'CPU'+string(j+1,format='(i2.2)')+':',m_files(j)
;
; Setup OYSTER bridge objects
;
spawn,'pwd',local_dir & local_dir=local_dir(0)
oyster_bridge=objarr(n_cpu)
oyster_bridge_data=strarr(n_cpu)
for j=0,n_cpu-1 do begin
	oyster_bridge(j)=obj_new('IDL_IDLBridge')
	oyster_bridge(j)->execute,".run "+!oyster_dir+"bridge.pro"
	oyster_bridge(j)->execute,"bridge"
	oyster_bridge(j)->execute,"cd,'"+local_dir+"'"
	oyster_bridge_data(j)='bridge_data'+string(j+1,format='(i2.2)')+'.xdr'
	oyster_bridge(j)->setvar,"obd",oyster_bridge_data(j)
endfor
;
; Call gridchisq, n=number of all_files, nf_cpu=number of files per cpu
for j=0,n_cpu-1 do begin
;	if j*nf_cpu gt n-1 then break
;	files=unique(all_files((j*nf_cpu)<(n-1):((j+1)*nf_cpu-1)<(n-1)))
	files=a_files(0:n_files(j)-1,j)
	if n_elements(files) eq 1 then files=files(0)
	save,gen_model,star_model,binary_model, $
		component,files, $
		file=oyster_bridge_data(j)
	parms=string(num_grid)+","+ $
	      string(grid_cell)
	oyster_bridge(j)->execute,"gridchisq,"+parms,/nowait
endfor
;
; Stay in while loop until all processes finish.
notdone = 1
while (notdone ge 1) do begin
	notdone=0
	for j=0,n_elements(oyster_bridge)-1 do $
		notdone = notdone+oyster_bridge[j]->Status()
endwhile
;
print,''
print,'Cleaning up processes...'
for j=0,n_cpu-1 do begin
;	if (oyster_bridge(j)->status() eq 1) then oyster_bridge(j)->abort
	obj_destroy,oyster_bridge(j)
	spawn,'rm -f '+oyster_bridge_data(j)
endfor
;
end
;-------------------------------------------------------------------------------
pro gridplot,file_spec,rd=rd,ps=ps,title_all=title_all,model=model
;
; Given one or more YYYY-MM-DD.chisq files, show contour plots.
; If more than one is specified, do not show but save as PS files,
; create LaTex template and compile to create a PDF with all plots.
; Just create EPS files if ps=1. If model is specified, add companion
; position computed with orbit in plots.
;
common Model,gen_model,star_model,binary_model,gen_error,star_error,binary_error
;
if n_params() eq 0 then begin
	print,'Usage: '+ $
	'gridplot,file_spec,rd=rd,ps=ps,title_all=title_all,model=model'
	return
endif
;
if keyword_set(model) and n_elements(binary_model) eq 0 then begin
	if file_test(model) then begin
		print,'Reading model '+model
		readmodel,model 
	endif else begin 
		print,'Error:model file not found!'
		retall
	endelse
endif
;
if n_elements(rd) eq 0 then rd=0
if n_elements(ps) eq 0 then ps=0
;
rad=180/!pi
;
files=file_search(file_spec)
if strlen(files(0)) eq 0 then begin
	print,'GRIDFIT: Files not found!'
	return
endif
;
if ps eq 0 then buffer=0 else buffer=1
if n_elements(files) gt 1 then begin
	buffer=1
	openw,unit,'gridplot.tex',/get_lun
	printf,unit,'\documentclass[a4paper,12pt]{article}'
	printf,unit,'\pagestyle{empty}'
;	https://www.sharelatex.com/learn/Page_size_and_margins
	printf,unit,'\usepackage{geometry}'
	printf,unit,'\geometry{left=20mm,top=0mm,}'
	printf,unit,'\usepackage{graphicx}'
	printf,unit,'\usepackage{subcaption}'
	printf,unit,'\begin{document}'
	printf,unit,'\begin{figure}[htb]'
	printf,unit,'\centering'
	printf,unit,'\begin{tabular}{@{}cccc@{}}'
	istring='\includegraphics[width=0.32\textwidth]'
endif
;
for i=0,n_elements(files)-1 do begin
;
restore,files(i)	; restores date,n_deg_free,reduced chisq,rho,theta
index=where(chisq eq min(chisq)) & index=index(0)
if rd then begin
	r_min=sin(theta(index)/rad)*rho(index)
	d_min=cos(theta(index)/rad)*rho(index)
	print,'Minimum Chi^2 found at RA = ',r_min,', DEC =',d_min
	print,'Minimum Chi^2 : ',min(chisq)
endif else begin
	print,'Minimum Chi^2 found at rho= ',rho(index),', theta=',theta(index)
	print,'Minimum Chi^2 : ',min(chisq)
endelse
;
rad=180/!pi
r=sin(theta/rad)*rho	; RA
d=cos(theta/rad)*rho	; Dec
dims=size(r,/dim)
cellsize=float(r(0)-r(1))
if n_elements(files) eq 1 then begin
	print,'Cell size = ',cellsize
	print,'Dimensions: ',fix(dims)
endif
;
; Not needed?
; rho=sqrt(r^2+d^2)
; theta=atan(r,d)*rad
; if max(theta) lt 0 then theta=theta+360.
index=where(theta lt 0)
theta(index)=theta(index)+360
;
!except=0
if rd then print,'Positive RA to the East (left).' $
      else print,'Axes are Theta versus Rho.'
if not keyword_set(title_all) then $
	title=strmid(files(i),0,strpos(files(i),'.',/reverse_search)) $
	else title=title_all
if rd then g=contour(chisq,r,d,c_label_show=0,title=title, $
	xtitle='Right Ascension offset',ytitle='Declination offset', $
;	c_value=[5,7,10,15,25,50], $	; sigma
;	c_value=[1.1,1.2,1.46,2.14]*min(chisq)), $
;	c_value=min(chisq)+findgen(7)*(max(chisq)-min(chisq))/7), $
;	rgb_table=39,font_size=16,buffer=buffer,xrange=[min(r),max(r)]) $
;	Switched max and min in the next line (14.1.2025): RA incr. to the left
	rgb_table=39,font_size=16,buffer=buffer,xrange=[max(r),min(r)]) $
      else g=contour(chisq,rho,theta,c_label_show=0,title=title, $
	xtitle='Separation [mas]',ytitle='Position angle [deg]',rgb_table=39, $
	font_size=16,buffer=buffer, $
	xrange=[min(rho),max(rho)],yrange=[min(theta),max(theta)])
!except=1
;
; Obtain observation date either from new chisq file (restore) or from filename
if n_elements(date) eq 0 then date=strmid(files(i),0,10) ; old method
;
; Over-plot concentric rings at the model position of the secondary
two_ssq=0.5
if rd and keyword_set(model) then begin	; "rd" = "RA,DEC"
	rt=binarypos(date2jd(date))	; "rt" = "Radius,Theta"
	r_m=sin(rt(1)/rad)*rt(0)	; compute RA of secondary
	d_m=cos(rt(1)/rad)*rt(0)	; compute DEC of secondary
	gp=max(chisq)*exp(-(((r-r_m)^2+(d-d_m)^2)/two_ssq)); concentric rings
	g=contour(gp,r,d,/overplot,c_label_show=0,color='r')
	print,'Model position indicated by red oval lines.'
endif else if keyword_set(model) then begin
	sp=fltarr(2)
	sp(*)=binarypos(date2jd(date))	; "sp" = "separation,position angle"
	gp=max(chisq)*exp(-(((r-r_m)^2+(d-d_m)^2)/two_ssq)); concentric rings
	g=contour(gp,rho,theta,/overplot,c_label_show=0,color='r')
;	gp=max(chisq)*exp(-(rho^2+theta^2))
;	gp=max(chisq)*exp(-(((r-r_m)^2+(d-d_m)^2)/two_ssq)); concentric rings
;	g=contour(gp,rho,theta,/overplot,c_label_show=0,color='r')
endif
;
if n_elements(files) gt 1 then begin
	if (i+1) mod 3 eq 0 then t=' & \\' else t=' &'
	printf,unit,istring+'{'+files(i)+'.eps}'+t
	if (i+1) mod 21 eq 0 then begin
		printf,unit,'\end{tabular}'
		printf,unit,'\caption{$\chi^2$ surfaces}'
		printf,unit,'\end{figure}'
		printf,unit,'\begin{figure}[htb]'
		printf,unit,'\centering'
		printf,unit,'\begin{tabular}{@{}cccc@{}}'
	endif
endif
if buffer eq 1 then g.save,files(i)+'.eps'
;
endfor
;
if n_elements(files) gt 1 then begin
	printf,unit,'\end{tabular}'
	printf,unit,'\caption{$\chi^2$ surfaces}'
	printf,unit,'\end{figure}'
	printf,unit,'\end{document}'
	free_lun,unit
	spawn,'latex gridplot'
	spawn,'dvips -o gridplot.ps gridplot.dvi'
	spawn,'/usr/bin/ps2pdf -sPAPERSIZE=a4 gridplot.ps'
endif
;
end
;-------------------------------------------------------------------------------
pro fitchisq,chisqfiles,component=component,localfile=file,sigma=sigma,cf=cf
;
; Original version, no longer in use as of 2023
;
; Note: now superceded by next procedure!
;
; Given a list of YYYY-MM-DD.chisq files (from gridchisq), read these and
; fit the (global) minimum position including an error ellipse and write 
; results to file fitchisq.psn. sigma is the desired increase in total chisq, 
; and the ellipse is fit to the corresponding contour. The factor cf allows
; to account for correlated data and reduces the number of degrees of freedom.
; Correlations may involve adjacent channels depending on the calibration. 
;
; Formal errors correspond to sigma=1 and cf=1 (default). 
;
; The gridchisq file must have fine sampling, use option chi2map with gridchisq.
;
; An astrometry file (*.psn) can be specified via file, which evaluates
; the Chi^2 value nearest the position given for the epoch in the localfile.
; This allows to pick a (local) minimum closest to the local astrometry.
; Specify the component if there is more than one when using localfile.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common MarquardtFit,fit_options
;
if n_elements(chisqfiles) eq 0 then begin
	chisqfiles='????-??-??.chisq'
	chisqfiles='*.chisq'
	print,'Processing all CHISQ files...'
endif
;
chisq_files=file_search(chisqfiles)
if strlen(chisq_files(0)) eq 0 then begin
	print,'GRIDPSN: Files not found!'
	return
endif
;
if n_elements(component) eq 0 then component='A-B'
;
if n_elements(localfile) eq 0 then local=0 else local=1
if local then begin
	local_file=file_search(localfile)
	if strlen(local_file(0)) eq 0 then begin
		print,'Local astrometry file not found!'
		return
	endif
	load_astrometry,local_file(0)
	index=where(positions.component eq component)
	local_positions=positions(index)
	r_local=sin(local_positions.theta)*local_positions.rho
	d_local=cos(local_positions.theta)*local_positions.rho
endif
if n_elements(sigma) eq 0 then sigma=1
if n_elements(cf) eq 0 then cf=1 else cf=float(cf)
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
fit_options.tolerance=1e-7
;
openw,unit,'fitchisq.psn',/get_lun
printf,unit, $
'!Comp Julian Year  Rho   Theta    Major  Minor   PA     Rho_err  Theta_err Chi2'
;
rad=180/!pi
ellipse_options=alloc_ellipse_options()
ellipse_options.component=component
ellipse_options.a=1
ellipse_options.b=1
ellipse_options.p=1
positions=replicate({component:'',rho:0.0,theta:0.0},8)
positions.component=component
;
for i_file=0,n_elements(chisq_files)-1 do begin
	print,''
	print,'Analyzing file ',chisq_files(i_file)
	date=''
	restore,chisq_files(i_file); restores n_deg_free,chisq,rho,theta,date
;
	if strlen(date) eq 0 then date=strmid(chisq_files(i_file),0,10) ; old method
	parsedate,date,y,m,d
	midnight=system_config('NPOI','MIDNIGHT')
	midnight=system_config('VLTI','MIDNIGHT')
	d=d+midnight/24
	jy=jd2jy(julian(y,m,d))
;	Defaults
	emajor=1.0
	eminor=emajor
	pa=0
	rho_error=0
	theta_error=0
;
	n_deg_free=nint(n_deg_free/cf)	; cf reduces the number of data points
	min_red_chisq=min(chisq)
	index=where(chisq eq min_red_chisq) & index=index(0)
	ij=whereindex(index,chisq)
	print,'Minimum chi^2 found at rho= ', $
		float(rho(index)),', theta= ',float(theta(index))
	print,'Minimum reduced chi^2: ',min_red_chisq
	print,'Number of degrees of freedom:',n_deg_free
;	print,'Minimum total chi^2: ',min_red_chisq*n_deg_free
	print,'Normalizing reduced chi^2 to 1'
	chisq=chisq/min_red_chisq
;
	r=sin(theta/rad)*rho
	d=cos(theta/rad)*rho
	cellsize=float(r(0)-r(1))
	rr=r(*,0)
	dd=d(0,*) & dd=reform(dd)
	csr=median(rr-shift(rr,-1))
	csd=median(dd-shift(dd,1))
	if nint(csr*1e6) ne nint(csd*1e6) then begin
		print,'Error: different cellsize in x and y directions!'
		return
	endif
	cell_size=csr	; mas
;
	if local then begin
;		Find the grid cell closest to the local astrometry
		i=where(abs(local_positions.jy-jy) lt 1.0/(365.25*24),count) 
		i=i(0)
		if count eq 0 then begin
			print,'Julian Year not found!'
			continue
		endif
		dr=r_local(i)-r
		dd=d_local(i)-d
		dist_local=sqrt(dr^2+dd^2)
		j=where(dist_local eq min(dist_local)) & j=j(0)
		min_dist=dist_local(index)
		print,'Local chi^2 found at rho= ',rho(j),', theta= ',theta(j)
		print,'Local chi^2: ',chisq(j),', min(dist)=',min_dist
		print,'============================================'
		ij=whereindex(j,chisq)
	endif
;
; Attempt to fit ellipses to contours, map size should be less than 2 mas
	dims=size(chisq,/dim)
;
	chisq=chisq*n_deg_free	; Convert to total chisq, 
;				  formal error now corresponds to increase by 1
	print,'Converted to total chi^2:'
       	print,'     Maximum='+strtrim(string(max(chisq)),1)
	print,'     Minimum='+strtrim(string(min(chisq)),1)
	min_chisq=min(chisq)
	c0=1			; Default sigma, used w/polynomial fit
	max_chisq=min_chisq+c0	; Correct, though test shows chi^2(red)>3
	max_chisq=max(chisq)	; Start with highest value, checking edges 
;
;	Closed contour check
	if min(chisq(0,*)) le max_chisq then max_chisq=min(chisq(0,*))
	if min(chisq(*,0)) le max_chisq then max_chisq=min(chisq(*,0))
	print,"Maximum chi^2 of closed cont's.: ",max_chisq
	if (max_chisq-min_chisq) ge sigma then begin
		max_chisq=min_chisq+sigma
		print,'Maximum chi^2 contour to fit (min + sigma): ',max_chisq
	endif else begin
		print,'***Error: requested contour too high:',sigma
		max_chisq=min_chisq+(max_chisq-min_chisq)*0.25	; Use 25%
		print,'Maximum chi^2 contour to fit (min/max=25%): ',max_chisq
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (contour high)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (contour high)']
		print,''
		continue
	endelse
;
	r_ell=fltarr(8)
	d_ell=fltarr(8)
	z_r_ell=fltarr(8)
	z_d_ell=fltarr(8)
; 	Walk from minimum along the axes and diagonals
	mflag=0	; set to 1 if walk hits edge of map
	qflag=0	; set to 1 if quasol fails
;	Cardinal axes
	for i=1,dims(0)/2 do begin	; "left"
		curr_chisq=chisq((ij(0)-i)>0,ij(1))
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij(0) then mflag=1
	r_ell(0)=r((ij(0)-i)>0,ij(1))-r(ij(0),ij(1))
	d_ell(0)=d((ij(0)-i)>0,ij(1))-d(ij(0),ij(1))
	for i=1,dims(0)/2 do begin	; "right"
		curr_chisq=chisq((ij(0)+i)<dims(0)-1,ij(1))
		if curr_chisq ge max_chisq then break
	endfor
	if ij(0)+i ge dims(0) then mflag=1
	r_ell(1)=r((ij(0)+i)<dims(0)-1,ij(1))-r(ij(0),ij(1))
	d_ell(1)=d((ij(0)+i)<dims(0)-1,ij(1))-d(ij(0),ij(1))
;	Fit quadratic polynomial to minimum...
	x=r(*,0)-r(index)
	y=chisq(*,ij(1)) & y0=min(y) & y=y-y0
	j=where(y le max_chisq,nj)
	if nj lt 6 then begin
		print,'Warning: sampling of minimum too coarse!'
		print,'Rerun gridchisq with smaller grid cell size!'
		print,'Current grid cell size is ',cellsize
		return
	endif
	x=x(j) & y=y(j)
	c=poly_fit(x,y,2,yfit)
	c(0)=-c0 & c(1)=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z(si)
	if total(z) eq 0 then qflag=1
	z_r_ell(0)=z(1)
	z_r_ell(1)=z(0)
	z_d_ell(0:1)=0
	for j=1,dims(1)/2 do begin	; "down"
		curr_chisq=chisq(ij(0),(ij(1)-j)>0)
		if curr_chisq ge max_chisq then break
	endfor
	if j gt ij(1) then mflag=1
	r_ell(2)=r(ij(0),(ij(1)-j)>0)-r(ij(0),ij(1))
	d_ell(2)=d(ij(0),(ij(1)-j)>0)-d(ij(0),ij(1))
	for j=1,dims(1)/2 do begin	; "up"
		curr_chisq=chisq(ij(0),(ij(1)+j)<dims(1)-1)
		if curr_chisq ge max_chisq then break
	endfor
	if ij(1)+j ge dims(1) then mflag=1
	r_ell(3)=r(ij(0),(ij(1)+j)<dims(1)-1)-r(ij(0),ij(1))
	d_ell(3)=d(ij(0),(ij(1)+j)<dims(1)-1)-d(ij(0),ij(1))
;	Fit quadratic polynomial to minimum...
	x=reform(d(0,*))-d(index)
	y=reform(chisq(ij(0),*)) & y0=min(y) & y=y-y0
	j=where(y le max_chisq)
	if n_elements(j) lt 6 then begin
		print,'Error: sampling of minimum too coarse!'
		print,'Rerun gridchisq with smaller grid cell size!'
		print,'Current grid cell size is ',cellsize
		return
	endif
	x=x(j) & y=y(j)
	c=poly_fit(x,y,2,yfit)
	c(0)=-c0 & c(1)=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z(si)
	if total(z) eq 0 then qflag=1
	z_d_ell(2)=z(0)
	z_d_ell(3)=z(1)
	z_r_ell(2:3)=0
;	Diagonals
	for i=1,dims(0)/2 do begin	; "lower-left"
		j=i
		curr_chisq=chisq((ij(0)-i)>0,(ij(1)-j)>0)
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij(0) then mflag=1
	if j gt ij(1) then mflag=1
	r_ell(4)=r((ij(0)-i)>0,(ij(1)-j)>0)-r(ij(0),ij(1))
	d_ell(4)=d((ij(0)-i)>0,(ij(1)-j)>0)-d(ij(0),ij(1))
	for i=1,dims(0)/2 do begin	; "upper-right"
		j=i
		curr_chisq=chisq((ij(0)+i)<dims(0)-1,(ij(1)+j)<dims(1)-1)
		if curr_chisq ge max_chisq then break
	endfor
	if ij(0)+i ge dims(0) then mflag=1
	if ij(1)+j ge dims(1) then mflag=1
	r_ell(5)=r((ij(0)+i)<dims(1)-1,(ij(1)+j)<dims(1)-1)-r(ij(0),ij(1))
	d_ell(5)=d((ij(0)+i)<dims(1)-1,(ij(1)+j)<dims(1)-1)-d(ij(0),ij(1))
;	Fit quadratic polynomial to minimum...
	idx=indgen(dims(0))-dims(0)/2
	idx0=ij(0)+idx 
	idx1=ij(1)+idx
	i=where(idx0 ge 0 and idx1 ge 0 and $
		idx0 lt dims(0) and idx1 lt dims(1),j)
	idx0=idx0(i)
	idx1=idx1(i)
	x=sqrt((r(idx0,idx1)-r(index))^2 $
	      +(d(idx0,idx1)-d(index))^2)*signof(idx(i))
	y=chisq(idx0,idx1) & y0=min(y) & y=y-y0
	j=where(y le max_chisq)
	x=x(j) & y=y(j)
	c=poly_fit(x,y,2,yfit)
	c(0)=-c0 & c(1)=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z(si)
	if total(z) eq 0 then qflag=1
	z_d_ell(4)=z(0);/sqrt(2)
	z_d_ell(5)=z(1);/sqrt(2)
	z_r_ell(4)=z(1);/sqrt(2)
	z_r_ell(5)=z(0);/sqrt(2)
	for i=1,dims(1)/2 do begin	; "upper-left"
		j=i
		curr_chisq=chisq((ij(0)-i)>0,(ij(1)+j)<dims(1)-1)
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij(0) then mflag=1
	if ij(1)+j ge dims(1) then mflag=1
	r_ell(6)=r((ij(0)-i)>0,(ij(1)+j)<dims(1)-1)-r(ij(0),ij(1))
	d_ell(6)=d((ij(0)-i)>0,(ij(1)+j)<dims(1)-1)-d(ij(0),ij(1))
	for i=1,dims(1)/2 do begin	; "lower-right"
		j=i
		curr_chisq=chisq((ij(0)+i)<dims(0)-1,(ij(1)-j)>0)
		if curr_chisq ge max_chisq then break
	endfor
	if ij(0)+i ge dims(0) then mflag=1
	if j gt ij(1) then mflag=1
	r_ell(7)=r((ij(0)+i)<dims(0)-1,(ij(1)-j)>0)-r(ij(0),ij(1))
	d_ell(7)=d((ij(0)+i)<dims(0)-1,(ij(1)-j)>0)-d(ij(0),ij(1))
;	Fit quadratic polynomial to minimum...
	idx=indgen(dims(0))-dims(0)/2
	idx0=ij(0)+idx 
	idx1=ij(1)-idx
	i=where(idx0 ge 0 and idx1 ge 0 and $
		idx0 lt dims(0) and idx1 lt dims(1),j)
	idx0=idx0(i)
	idx1=idx1(i)
	x=sqrt((r(idx0,idx1)-r(index))^2 $
	      +(d(idx0,idx1)-d(index))^2)*signof(idx(i))
	y=chisq(idx0,idx1) & y0=min(y) & y=y-y0
	j=where(y le max_chisq)
	if n_elements(j) lt 8 then begin
		print,''
		print,'***Number of contour points: ',n_elements(j)
		print,'***Warning: could not define contour! Skipping '+date
		wait,1
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (contour poor)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (contour poor)']
		continue
	endif
	x=x(j) & y=y(j)
	c=poly_fit(x,y,2,yfit)
	c(0)=-c0 & c(1)=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z(si)
	if total(abs(z)) eq 0 then qflag=1
	z_d_ell(6)=z(1);/sqrt(2)
	z_d_ell(7)=z(0);/sqrt(2)
	z_r_ell(6)=z(1);/sqrt(2)
	z_r_ell(7)=z(0);/sqrt(2)
;
	if local then begin
		if min_dist gt 1 then mflag=1
	endif
;
	if mflag then begin
		print,''
		print,'***Warning: Contour outside map! Skipping '+date
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (contour outside)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (contour outside)']
		wait,1
	endif
	if not mflag then begin
;	Fit error ellipse to contour at sigma
	positions.rho=sqrt(r_ell^2+d_ell^2)
	positions.theta=atan(r_ell,d_ell)
	i=where(positions.rho eq max(positions.rho)) & i=i(0)
	pa=positions(i).theta
	emajor=positions(i).rho	; semi-major axis
	i=where(positions.rho eq min(positions.rho)) & i=i(0)
	eminor=positions(i).rho	; semi-minor axis
	e_parms=[0,0,emajor,eminor,pa]
	ellipse_options.c=1	; fit center 1/0
	print,'Running marquardt to fit ellipse to contour at min.+sigma...'
	fitellipse & e_parms0=e_parms
;
;	emajor=e_parms(2)*cf
;	eminor=e_parms(3)*cf
	emajor=e_parms(2)
	eminor=e_parms(3)
	pa=(e_parms(4)*rad) mod 180
	if eminor gt emajor then begin
		v=emajor
		emajor=eminor
		eminor=v
		pa=(pa+90) mod 180
	endif
	if pa lt 0 then pa=pa+180
	rho_error=0
	theta_error=0
;
	printf,unit,'! '+date+', ellipse at sigma followed by formal error ellipse'
	printf,unit,' '+component,jy, $
      		rho(index),theta(index), $
		emajor,eminor,pa, $
		rho_error,theta_error,min(chisq), $
	format='(a4,2x,f9.4,2x,f6.2,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
	       	'3x,f7.3,1x,f8.3,1x,f6.1)'
;
;	Fit formal error ellipse w/c0=1 and quadratic fit
	if qflag then begin
		print,''
		print,'***Error: sol. of quadratic eq. failed. Skipping '+date
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (solution failed)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (solution failed)']
		wait,1
	endif else begin
		positions.rho=sqrt(z_r_ell^2+z_d_ell^2)
		positions.theta=atan(z_r_ell,z_d_ell)
		i=where(positions.rho eq max(positions.rho)) & i=i(0)
		pa=positions(i).theta
;		pa=e_parms0(4)
		emajor=positions(i).rho	; semi-major axis
		i=where(positions.rho eq min(positions.rho)) & i=i(0)
		eminor=positions(i).rho	; semi-minor axis
		e_parms=[0,0,emajor,eminor,pa]
		ellipse_options.c=0	; fit center 1/0
;		ellipse_options.p=0
		print,''
		print,'Running marquardt to fit formal error ellipse...'
		fitellipse
;
		emajor=e_parms(2)
		eminor=e_parms(3)
		pa=(e_parms(4)*rad) mod 180
		if eminor gt emajor then begin
			v=emajor
			emajor=eminor
			eminor=v
			pa=(pa+90) mod 180
		endif
		if pa lt 0 then pa=pa+180
		rho_error=0
		theta_error=0
;
		printf,unit,'!'+component,jy, $
      		rho(index),theta(index), $
		emajor,eminor,pa, $
		rho_error,theta_error,min(chisq), $
		format='(a4,2x,f9.4,2x,f6.2,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
	       	'3x,f7.3,1x,f8.3,1x,f6.1)'
	endelse	; qflag
	endif	; mflag
endfor
;
free_lun,unit
spawn,'cat fitchisq.psn'
;
print,'------------------------------'
if n_elements(bad_files) gt 0 then begin
	print,'Failed to analyze these files:'
	for i=0,n_elements(bad_files)-1 do print,bad_files(i)
endif
if local then positions=local_positions
;
end
;-------------------------------------------------------------------------------
pro fitchisq,chisqfiles,component=component,localfile=file,sigma=sigma,cf=cf
;
; Note: now superceded by next procedure!
;
; Given a list of YYYY-MM-DD.chisq files (from gridchisq), read these and
; fit the (global) minimum position including an error ellipse and write 
; results to file fitchisq.psn. Sigma is the desired increase in total chisq, 
; and the ellipse is fit to the corresponding contour. The factor cf allows
; to account for correlated data and reduces the number of degrees of freedom.
; Correlations may involve adjacent channels depending on the calibration. 
;
; Formal errors correspond to sigma=1 and cf=1 (default). 
;
; The gridchisq file must have fine sampling, use option chi2map with gridchisq.
;
; An astrometry file (*.psn) can be specified via localfile, which evaluates
; the Chi^2 value nearest the position given for the epoch in the localfile.
; This allows to pick a (local) minimum closest to the local astrometry.
; Specify the component if there is more than one when using localfile.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common MarquardtFit,fit_options
;
debug=1
;
if n_elements(chisqfiles) eq 0 then begin
	chisqfiles='????-??-??.chisq'
	chisqfiles='*.chisq'
	print,'Processing all CHISQ files...'
endif
;
chisq_files=file_search(chisqfiles)
if strlen(chisq_files(0)) eq 0 then begin
	print,'GRIDPSN: Files not found!'
	return
endif
;
if n_elements(component) eq 0 then component='A-B'
;
if n_elements(localfile) eq 0 then local=0 else local=1
if local then begin
	local_file=file_search(localfile)
	if strlen(local_file(0)) eq 0 then begin
		print,'Local astrometry file not found!'
		return
	endif
	load_astrometry,local_file(0)
	index=where(positions.component eq component)
	local_positions=positions(index)
	r_local=sin(local_positions.theta)*local_positions.rho
	d_local=cos(local_positions.theta)*local_positions.rho
endif
if n_elements(sigma) eq 0 then sigma=1
if n_elements(cf) eq 0 then cf=1 else cf=float(cf)
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
fit_options.tolerance=1e-7
;
openw,unit,'fitchisq.psn',/get_lun
printf,unit, $
'!Comp Julian Year  Rho   Theta    Major  Minor   PA     Rho_err  Theta_err Chi2'
;
rad=180/!pi
ellipse_options=alloc_ellipse_options()
ellipse_options.component=component
ellipse_options.a=1
ellipse_options.b=1
ellipse_options.p=1
positions=replicate({component:'',rho:0.0,theta:0.0},8)
positions.component=component
;
;
for i_file=0,n_elements(chisq_files)-1 do begin
	print,''
	print,'=========== Analyzing file '+chisq_files(i_file)+' ==========='
	date=''
	restore,chisq_files(i_file); restores n_deg_free,chisq,rho,theta,date
;
	if strlen(date) eq 0 then date=strmid(chisq_files(i_file),0,10)
	parsedate,date,y,m,d
	midnight=system_config('NPOI','MIDNIGHT')
	midnight=system_config('VLTI','MIDNIGHT')
	d=d+midnight/24
	jy=jd2jy(julian(y,m,d))
;	Defaults
	emajor=1.0
	eminor=emajor
	pa=0
	rho_error=0
	theta_error=0
;
	n_deg_free=nint(n_deg_free/cf)	; cf reduces the number of data points
	min_red_chisq=min(chisq)
	index=where(chisq eq min_red_chisq) & index=index(0)
	ij=whereindex(index,chisq)
	print,'Minimum chi^2 found at rho= ', $
		strtrim(string(rho(index)),1), $
		', theta= ',strtrim(string(theta(index)),1)
	x=sin(theta(index)/rad)*rho(index)
	y=cos(theta(index)/rad)*rho(index)
	print,'Minimum chi^2 found at X [mas]= ', $
		strtrim(string(-float(x)),1), $
		', Y [mas]= ',strtrim(string(float(y)),1)
	print,'Minimum reduced chi^2: ',min_red_chisq
	print,'Number of degrees of freedom:',n_deg_free
	print,'Normalize minimum chi^2 to 1'
	chisq=chisq/min_red_chisq
;	print,'Minimum total chi^2: ',min_red_chisq*n_deg_free
;
	r=sin(theta/rad)*rho
	d=cos(theta/rad)*rho
	cellsize=float(r(0)-r(1))
	rr=r(*,0)
	dd=d(0,*) & dd=reform(dd)
	csr=median(rr-shift(rr,-1))
	csd=median(dd-shift(dd,1))
	if nint(csr*1e6) ne nint(csd*1e6) then begin
		print,'Error: different cellsize in x and y directions!'
		return
	endif
;
	if local then begin
;		Find the grid cell closest to the local astrometry
		i=where(abs(local_positions.jy-jy) lt 1.0/(365.25*24),count) 
		i=i(0)
		if count eq 0 then begin
			print,'Julian Year not found!'
			continue
		endif
		dr=r_local(i)-r
		dd=d_local(i)-d
		dist_local=sqrt(dr^2+dd^2)
		j=where(dist_local eq min(dist_local)) & j=j(0)
		min_dist=dist_local(index)
		print,'Local chi^2 found at rho= ',rho(j),', theta= ',theta(j)
		print,'Local chi^2: ',chisq(j),', min(dist)=',min_dist
		print,'============================================'
;		ij is the index of the minimum of chisq: chis(ij(0),ij(1))
		ij=whereindex(j,chisq)
	endif
;
; Attempt to fit ellipse to contour at sigma, map size should be less than 2 mas
	dims=size(chisq,/dim)
;
	chisq=chisq*n_deg_free	; Convert to total chisq, 
;				  formal error now corresponds to increase by 1
	print,'Converted to total chi^2:'
       	print,'     Maximum='+strtrim(string(max(chisq)),1)
	print,'     Minimum='+strtrim(string(min(chisq)),1)
	min_chisq=min(chisq)
	c0=1			; Default sigma, used w/polynomial fit
	max_chisq=min_chisq+c0	; Correct, though test shows chi^2(red.)>3
	max_chisq=max(chisq)	; Start with highest value, checking edges 
;
;	Closed contour check
	if min(chisq(0,*)) le max_chisq then max_chisq=min(chisq(0,*))
	if min(chisq(*,0)) le max_chisq then max_chisq=min(chisq(*,0))
	print,"Maximum chi^2 of closed cont's.: ",max_chisq
	if (max_chisq-min_chisq) ge sigma then begin
		max_chisq=min_chisq+sigma
		print,'Maximum chi^2 contour to fit (min + sigma): ',max_chisq
	endif else begin
		print,'***Requested contour too high:',min(chisq)+sigma
		max_chisq=min_chisq+(max_chisq-min_chisq)*0.25	; Use 25%
		print,'Maximum chi^2 contour allowed (min/max=25%): ',max_chisq
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (contour high)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (contour high)']
		print,''
		continue
	endelse
;
;	"r": right ascension ("left, right")
;	"d":     declination ("down, up")
	r_ell=fltarr(8)	; left, right, down, up, low.l., up.r., up.l., low.r.
	d_ell=fltarr(8)
; 	Walk from minimum along the axes and diagonals
	mflag=0	; set to 1 if walk hits edge of map
	qflag=0	; set to 1 if quasol fails
;	Cardinal axes: left, right, down, up
	last_chisq=0
	for i=1,dims(0)/2 do begin	; "left"
		curr_chisq=chisq((ij(0)-i)>0,ij(1))
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if i gt ij(0) then mflag=1
	r_ell(0)=r((ij(0)-i)>0,ij(1))-r(ij(0),ij(1))
	d_ell(0)=d((ij(0)-i)>0,ij(1))-d(ij(0),ij(1))
	last_chisq=0
	for i=1,dims(0)/2 do begin	; "right"
		curr_chisq=chisq((ij(0)+i)<dims(0)-1,ij(1))
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if ij(0)+i ge dims(0) then mflag=1
	r_ell(1)=r((ij(0)+i)<dims(0)-1,ij(1))-r(ij(0),ij(1))
	d_ell(1)=d((ij(0)+i)<dims(0)-1,ij(1))-d(ij(0),ij(1))
;	Fit quadratic polynomial to minimum...
	x=r(*,0)-r(index)
	y=chisq(*,ij(1)) & y0=min(y) & y=y-y0
	j=where(y le max_chisq,nj)
	if nj lt 6 then begin
		print,'Error: sampling of minimum too coarse!'
		print,'Rerun gridchisq with smaller grid cell size!'
		print,'Current grid cell size is ',cellsize
		return
	endif
	x=x(j) & y=y(j)
	c=poly_fit(x,y,2,yfit)
	c(0)=-c0 & c(1)=0
;	if debug then plot,x,y,psym=1
;	if debug then oplot,x,yfit,psym=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z(si)
	if total(abs(z)) eq 0 then qflag=1
	z_r_ell=fltarr(8)
	z_d_ell=fltarr(8)
	z_r_ell(0)=z(1)
	z_r_ell(1)=z(0)
	z_d_ell(0:1)=0
	last_chisq=0
	for j=1,dims(1)/2 do begin	; "down"
		curr_chisq=chisq(ij(0),(ij(1)-j)>0)
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if j gt ij(1) then mflag=1
	r_ell(2)=r(ij(0),(ij(1)-j)>0)-r(ij(0),ij(1))
	d_ell(2)=d(ij(0),(ij(1)-j)>0)-d(ij(0),ij(1))
	last_chisq=0
	for j=1,dims(1)/2 do begin	; "up"
		curr_chisq=chisq(ij(0),(ij(1)+j)<dims(1)-1)
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if ij(1)+j ge dims(1) then mflag=1
	r_ell(3)=r(ij(0),(ij(1)+j)<dims(1)-1)-r(ij(0),ij(1))
	d_ell(3)=d(ij(0),(ij(1)+j)<dims(1)-1)-d(ij(0),ij(1))
;	Fit quadratic polynomial to minimum...
	x=reform(d(0,*))-d(index)
	y=reform(chisq(ij(0),*)) & y0=min(y) & y=y-y0
	j=where(y le max_chisq,nj)
	if nj lt 6 then begin
		print,'Error: sampling of minimum too coarse!'
		print,'Rerun gridchisq with smaller grid cell size!'
		print,'Current grid cell size is ',cellsize
		return
	endif
	x=x(j) & y=y(j)
	c=poly_fit(x,y,2,yfit)
	c(0)=-c0 & c(1)=0
;	if debug then plot,x,y,psym=1
;	if debug then oplot,x,yfit,psym=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z(si)
	if total(abs(z)) eq 0 then qflag=1
	z_d_ell(2)=z(0)
	z_d_ell(3)=z(1)
	z_r_ell(2:3)=0
;	Diagonals
	for i=1,dims(0)/2 do begin	; "lower-left"
		j=i
		curr_chisq=chisq((ij(0)-i)>0,(ij(1)-j)>0)
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij(0) then mflag=1
	if j gt ij(1) then mflag=1
	r_ell(4)=r((ij(0)-i)>0,(ij(1)-j)>0)-r(ij(0),ij(1))
	d_ell(4)=d((ij(0)-i)>0,(ij(1)-j)>0)-d(ij(0),ij(1))
	for i=1,dims(0)/2 do begin	; "upper-right"
		j=i
		curr_chisq=chisq((ij(0)+i)<dims(0)-1,(ij(1)+j)<dims(1)-1)
		if curr_chisq ge max_chisq then break
	endfor
	if ij(0)+i ge dims(0) then mflag=1
	if ij(1)+j ge dims(1) then mflag=1
	r_ell(5)=r((ij(0)+i)<dims(1)-1,(ij(1)+j)<dims(1)-1)-r(ij(0),ij(1))
	d_ell(5)=d((ij(0)+i)<dims(1)-1,(ij(1)+j)<dims(1)-1)-d(ij(0),ij(1))
;	Fit quadratic polynomial to minimum...
	idx=indgen(dims(0))-dims(0)/2
	idx0=ij(0)+idx 
	idx1=ij(1)+idx
	i=where(idx0 ge 0 and idx1 ge 0 and $
		idx0 lt dims(0) and idx1 lt dims(1),j)
	idx0=idx0(i)
	idx1=idx1(i)
	x=sqrt((r(idx0,idx1)-r(index))^2 $
	      +(d(idx0,idx1)-d(index))^2)*signof(idx(i))
	y=chisq(idx0,idx1) & y0=min(y) & y=y-y0
	j=where(y le max_chisq)
	x=x(j) & y=y(j)
	c=poly_fit(x,y,2,yfit)
	c(0)=-c0 & c(1)=0
;	if debug then plot,x,y
;	if debug then oplot,x,yfit,psym=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z(si)
	if total(abs(z)) eq 0 then qflag=1
	z_d_ell(4)=z(0);/sqrt(2)
	z_d_ell(5)=z(1);/sqrt(2)
	z_r_ell(4)=z(1);/sqrt(2)
	z_r_ell(5)=z(0);/sqrt(2)
	for i=1,dims(1)/2 do begin	; "upper-left"
		j=i
		curr_chisq=chisq((ij(0)-i)>0,(ij(1)+j)<dims(1)-1)
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij(0) then mflag=1
	if ij(1)+j ge dims(1) then mflag=1
	r_ell(6)=r((ij(0)-i)>0,(ij(1)+j)<dims(1)-1)-r(ij(0),ij(1))
	d_ell(6)=d((ij(0)-i)>0,(ij(1)+j)<dims(1)-1)-d(ij(0),ij(1))
	for i=1,dims(1)/2 do begin	; "lower-right"
		j=i
		curr_chisq=chisq((ij(0)+i)<dims(0)-1,(ij(1)-j)>0)
		if curr_chisq ge max_chisq then break
	endfor
	if ij(0)+i ge dims(0) then mflag=1
	if j gt ij(1) then mflag=1
	r_ell(7)=r((ij(0)+i)<dims(0)-1,(ij(1)-j)>0)-r(ij(0),ij(1))
	d_ell(7)=d((ij(0)+i)<dims(0)-1,(ij(1)-j)>0)-d(ij(0),ij(1))
;	Fit quadratic polynomial to minimum...
	idx=indgen(dims(0))-dims(0)/2
	idx0=ij(0)+idx 
	idx1=ij(1)-idx
	i=where(idx0 ge 0 and idx1 ge 0 and $
		idx0 lt dims(0) and idx1 lt dims(1),j)
	idx0=idx0(i)
	idx1=idx1(i)
	x=sqrt((r(idx0,idx1)-r(index))^2 $
	      +(d(idx0,idx1)-d(index))^2)*signof(idx(i))
	y=chisq(idx0,idx1) & y0=min(y) & y=y-y0
	j=where(y le max_chisq)
	if n_elements(j) lt 8 then begin
		print,''
		print,'***Number of contour points: ',n_elements(j)
		print,'***Warning: could not define contour! Skipping '+date
		wait,1
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (contour poor)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (contour poor)']
		continue
	endif
	x=x(j) & y=y(j)
	c=poly_fit(x,y,2,yfit)
	c(0)=-c0 & c(1)=0
;	if debug then plot,x,y
;	if debug then oplot,x,yfit,psym=0
;	...and solve for contour c0
	z=quasol(c)*2 & si=sort(z) & z=z(si)
	print,'c=',c
	print,'z=',z
	if total(abs(z)) eq 0 then qflag=1
	z_d_ell(6)=z(1);/sqrt(2)
	z_d_ell(7)=z(0);/sqrt(2)
	z_r_ell(6)=z(1);/sqrt(2)
	z_r_ell(7)=z(0);/sqrt(2)
;
	if local then begin
		if min_dist gt 1 then mflag=1
	endif
;
	if mflag then begin
		print,''
		print,'***Warning: Contour outside map! Skipping '+date
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (contour outside)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (contour outside)']
		wait,1
	endif
	if not mflag then begin
;	Fit error ellipse to contour at sigma
	positions.rho=sqrt(r_ell^2+d_ell^2)
	positions.theta=atan(r_ell,d_ell)
	i=where(positions.rho eq max(positions.rho)) & i=i(0)
	pa=positions(i).theta
	emajor=positions(i).rho	; semi-major axis
	i=where(positions.rho eq min(positions.rho)) & i=i(0)
	eminor=positions(i).rho	; semi-minor axis
	e_parms=[0,0,emajor,eminor,pa]
	ellipse_options.c=1	; fit center 1/0
	print,'Running marquardt to fit ellipse to contour at min.+sigma...'
	fitellipse & e_parms0=e_parms
;
;	emajor=e_parms(2)*cf
;	eminor=e_parms(3)*cf
	emajor=e_parms(2)
	eminor=e_parms(3)
	pa=(e_parms(4)*rad) mod 180
	if eminor gt emajor then begin
		v=emajor
		emajor=eminor
		eminor=v
		pa=(pa+90) mod 180
	endif
	if pa lt 0 then pa=pa+180
	rho_error=0
	theta_error=0
;
	printf,unit,'! '+date+', ellipse at sigma followed by formal error ellipse'
	printf,unit,' '+component,jy, $
      		rho(index),theta(index), $
		emajor,eminor,pa, $
		rho_error,theta_error,min(chisq), $
	format='(a4,2x,f9.4,2x,f6.2,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
	       	'3x,f7.3,1x,f8.3,1x,f6.1)'
;
;	Fit formal error ellipse w/c0=1 and quadratic fit
	if qflag then begin
		print,''
		print,'***Error: sol. of quadratic eq. failed. Skipping '+date
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (solution failed)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (solution failed)']
		wait,1
	endif else begin
		positions.rho=sqrt(z_r_ell^2+z_d_ell^2)
		positions.theta=atan(z_r_ell,z_d_ell)
		i=where(positions.rho eq max(positions.rho)) & i=i(0)
		pa=positions(i).theta
;		pa=e_parms0(4)
		emajor=positions(i).rho	; semi-major axis
		i=where(positions.rho eq min(positions.rho)) & i=i(0)
		eminor=positions(i).rho	; semi-minor axis
		e_parms=[0,0,emajor,eminor,pa]
		ellipse_options.c=0	; fit center 1/0
;		ellipse_options.p=0
		print,''
		print,'Running marquardt to fit formal error ellipse...'
		fitellipse
;
		emajor=e_parms(2)
		eminor=e_parms(3)
		pa=(e_parms(4)*rad) mod 180
		if eminor gt emajor then begin
			v=emajor
			emajor=eminor
			eminor=v
			pa=(pa+90) mod 180
		endif
		if pa lt 0 then pa=pa+180
		rho_error=0
		theta_error=0
;
		printf,unit,'!'+component,jy, $
      		rho(index),theta(index), $
		emajor,eminor,pa, $
		rho_error,theta_error,min(chisq), $
		format='(a4,2x,f9.4,2x,f6.2,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
	       	'3x,f7.3,1x,f8.3,1x,f6.1)'
	endelse	; qflag
	endif	; mflag
endfor
;
free_lun,unit
spawn,'cat fitchisq.psn'
;
print,'------------------------------'
if n_elements(bad_files) gt 0 then begin
	print,'Failed to analyze these files:'
	for i=0,n_elements(bad_files)-1 do print,bad_files(i)
endif
if local then positions=local_positions
;
end
;-------------------------------------------------------------------------------
pro fitchisq,chisqfiles,component=component,file=localfile,sigma=sigma,cf=cf
;
; Given a list of YYYY-MM-DD.chisq files (from gridchisq), read these and
; fit the (global) minimum position including an error ellipse and write 
; results to file fitchisq.psn. Sigma is the desired increase in total chisq, 
; and the ellipse is fit to the corresponding contour. The factor cf allows
; to account for correlated data and reduces the number of degrees of freedom.
; Correlations may involve adjacent channels depending on the calibration. 
;
; The ellipse orientation follows the astronomical rule: +RA to the left!
;
; Formal errors correspond to sigma=1 and cf=1 (default). 
;
; The gridchisq file must have fine sampling, use option chi2map with gridchisq.
; More reliable is the error ellipse computed by cleanbeam(dirtybeam(star))!
;
; An astrometry file (*.psn) can be specified via file, which evaluates
; the Chi^2 value nearest the position given for the epoch in the localfile.
; This allows to pick a (local) minimum closest to the local astrometry.
; Specify the component if there is more than one when using localfile.
;
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common FitAstrometry,ellipse_options,orbit_options,e_parms,o_parms
common MarquardtFit,fit_options
;
debug=1
;
if n_elements(chisqfiles) eq 0 then begin
	chisqfiles='????-??-??.chisq'
	chisqfiles='*.chisq'
	print,'Processing all "*.chisq" files...'
endif
;
chisq_files=file_search(chisqfiles)
if strlen(chisq_files(0)) eq 0 then begin
	print,'GRIDPSN: Files not found!'
	retall
endif
for i=0,n_elements(chisq_files)-1 do begin
	r=strsplit(chisq_files(i),'.',/extract)
	if r(n_elements(r)-1) ne 'chisq' then begin
		print,'Error: input files need to have extension "chisq"!'
		retall
	endif
endfor
;
if n_elements(component) eq 0 then component='A-B'
;
if n_elements(localfile) eq 0 then local=0 else local=1
;
if local then begin
	local_file=file_search(localfile)
;	Check extension
	if strpos(local_file(0),'psn') eq -1 then begin
		print,'Error: file does not have .psn extension!'
		return
	endif
	if strlen(local_file(0)) eq 0 then begin
		print,'Local astrometry file not found!'
		return
	endif
	load_astrometry,local_file(0)
	index=where(positions.component eq component)
	local_positions=positions(index)
	r_local=sin(local_positions.theta)*local_positions.rho
	d_local=cos(local_positions.theta)*local_positions.rho
endif
if n_elements(sigma) eq 0 then sigma=1
if n_elements(cf) eq 0 then cf=1 else cf=float(cf)
if n_elements(fit_options) eq 0 then if init_marquardtfit() ne 0 then return
fit_options.tolerance=1e-7
;
openw,unit,'fitchisq.psn',/get_lun
printf,unit, $
'!Comp Julian Year  Rho   Theta    Major  Minor   PA     Rho_err  Theta_err Chi2'
;
rad=180/!pi
ellipse_options=alloc_ellipse_options()
ellipse_options.component=component
ellipse_options.a=1
ellipse_options.b=1
ellipse_options.p=1
positions=replicate({component:'',rho:0.0,theta:0.0},8)
positions.component=component
;
n=n_elements(chisq_files)
ny=long(sqrt(n))
nx=nint(n/float(ny))
if nx*ny lt n then nx=nx+1
!p.multi=0
!p.multi=[0,nx,ny]
!p.multi=[0,4,4,0,0]
if debug then !p.multi=[0,2,2,0,1]
;
for i_file=0,n_elements(chisq_files)-1 do begin
	print,''
	print,'=========== Analyzing file '+chisq_files(i_file)+' ==========='
	date=''
	restore,chisq_files(i_file); restores n_deg_free,chisq,rho,theta,date
;
	if strlen(date) eq 0 then date=strmid(chisq_files(i_file),0,10) ; old method
	parsedate,date,y,m,d
	midnight=system_config('NPOI','MIDNIGHT')
	midnight=system_config('VLTI','MIDNIGHT')
	d=d+midnight/24
	jy=jd2jy(julian(y,m,d))
;	Defaults
	emajor=1.0
	eminor=emajor
	pa=0
	rho_error=0
	theta_error=0
;
	n_deg_free=nint(n_deg_free/cf)	; cf reduces the number of data points
	min_red_chisq=min(chisq)
	index=where(chisq eq min_red_chisq) & index=index(0)
	ij=whereindex(index,chisq)
	print,'Minimum chi^2 found at rho= ', $
		strtrim(string(rho(index)),1), $
		', theta= ',strtrim(string(theta(index)),1)
	x=sin(theta(index)/rad)*rho(index)
	y=cos(theta(index)/rad)*rho(index)
	print,'Minimum chi^2 found at X [mas]= ', $
		strtrim(string(-float(x)),1), $
		', Y [mas]= ',strtrim(string(float(y)),1)
	print,'Minimum reduced chi^2: ',min_red_chisq
	print,'Number of degrees of freedom:',n_deg_free
	print,'Normalize minimum chi^2 to 1'
	chisq=chisq/min_red_chisq
;	print,'Minimum total chi^2: ',min_red_chisq*n_deg_free
;
	r=sin(theta/rad)*rho
	d=cos(theta/rad)*rho
	cellsize=float(r(0)-r(1))
	rr=r(*,0)
	dd=d(0,*) & dd=reform(dd)
	csr=median(rr-shift(rr,-1))
	csd=median(dd-shift(dd,1))
	if nint(csr*1e6) ne nint(csd*1e6) then begin
		print,'Error: different cellsize in x and y directions!'
		return
	endif
;
	if local then begin
;		Find the grid cell closest to the local astrometry
		i=where(abs(local_positions.jy-jy) lt 1.0/(365.25*24),count) 
		i=i(0)
		if count eq 0 then begin
			print,'Julian Year not found!'
			continue
		endif
		dr=r_local(i)-r
		dd=d_local(i)-d
		dist_local=sqrt(dr^2+dd^2)
		j=where(dist_local eq min(dist_local)) & j=j(0)
		min_dist=dist_local(index)
		print,'Local chi^2 found at rho= ',rho(j),', theta= ',theta(j)
		print,'Local chi^2: ',chisq(j),', min(dist)=',min_dist
		print,'============================================'
;		ij is the index of the minimum of chisq: chis(ij(0),ij(1))
		ij=whereindex(j,chisq)
	endif
;
; Attempt to fit ellipse to contour at sigma, map size should be less than 2 mas
	dims=size(chisq,/dim)
;
	chisq=chisq*n_deg_free	; Convert to total chisq, 
;				  formal error now corresponds to increase by 1
	print,'Converted to total chi^2:'
       	print,'     Maximum='+strtrim(string(max(chisq)),1)
	print,'     Minimum='+strtrim(string(min(chisq)),1)
	min_chisq=min(chisq)
	c0=1			; Default sigma, used w/polynomial fit
	max_chisq=min_chisq+c0	; Correct, though test shows chi^2(red.)>3
	max_chisq=max(chisq)	; Start with highest value, checking edges 
;
;	Closed contour check
	if min(chisq(0,*)) le max_chisq then max_chisq=min(chisq(0,*))
	if min(chisq(*,0)) le max_chisq then max_chisq=min(chisq(*,0))
	print,"Maximum chi^2 of closed cont's.: ",max_chisq
	if (max_chisq-min_chisq) ge sigma then begin
		max_chisq=min_chisq+sigma
		print,'Maximum chi^2 contour to fit (min + sigma): ',max_chisq
	endif else begin
		print,'***Requested contour too high:',min(chisq)+sigma
		max_chisq=min_chisq+(max_chisq-min_chisq)*0.25	; Use 25%
		print,'Maximum chi^2 contour allowed (min/max=25%): ',max_chisq
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (contour high)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (contour high)']
		print,''
		continue
	endelse
;
;	"r": right ascension ("left, right")
;	"d":     declination ("down, up")
	r_ell=fltarr(8)	; left, right, down, up, low.l., up.r., up.l., low.r.
	d_ell=fltarr(8)
; 	Walk from minimum along the axes and diagonals
	mflag=0	; set to 1 if walk hits edge of map
;	Cardinal axes: left, right, down, up
	last_chisq=0
	for i=1,dims(0)/2 do begin	; "left"
		curr_chisq=chisq((ij(0)-i)>0,ij(1))
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if i gt ij(0) then mflag=1
	r_ell(0)=r((ij(0)-i)>0,ij(1))-r(ij(0),ij(1))
	d_ell(0)=d((ij(0)-i)>0,ij(1))-d(ij(0),ij(1))
	last_chisq=0
	for i=1,dims(0)/2 do begin	; "right"
		curr_chisq=chisq((ij(0)+i)<dims(0)-1,ij(1))
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if ij(0)+i ge dims(0) then mflag=1
	r_ell(1)=r((ij(0)+i)<dims(0)-1,ij(1))-r(ij(0),ij(1))
	d_ell(1)=d((ij(0)+i)<dims(0)-1,ij(1))-d(ij(0),ij(1))
	last_chisq=0
	for j=1,dims(1)/2 do begin	; "down"
		curr_chisq=chisq(ij(0),(ij(1)-j)>0)
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if j gt ij(1) then mflag=1
	r_ell(2)=r(ij(0),(ij(1)-j)>0)-r(ij(0),ij(1))
	d_ell(2)=d(ij(0),(ij(1)-j)>0)-d(ij(0),ij(1))
	last_chisq=0
	for j=1,dims(1)/2 do begin	; "up"
		curr_chisq=chisq(ij(0),(ij(1)+j)<dims(1)-1)
		if curr_chisq ge max_chisq or curr_chisq le last_chisq $
		then break else last_chisq=curr_chisq
	endfor
	if ij(1)+j ge dims(1) then mflag=1
	r_ell(3)=r(ij(0),(ij(1)+j)<dims(1)-1)-r(ij(0),ij(1))
	d_ell(3)=d(ij(0),(ij(1)+j)<dims(1)-1)-d(ij(0),ij(1))
;	Diagonals
	for i=1,dims(0)/2 do begin	; "lower-left"
		j=i
		curr_chisq=chisq((ij(0)-i)>0,(ij(1)-j)>0)
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij(0) then mflag=1
	if j gt ij(1) then mflag=1
	r_ell(4)=r((ij(0)-i)>0,(ij(1)-j)>0)-r(ij(0),ij(1))
	d_ell(4)=d((ij(0)-i)>0,(ij(1)-j)>0)-d(ij(0),ij(1))
	for i=1,dims(0)/2 do begin	; "upper-right"
		j=i
		curr_chisq=chisq((ij(0)+i)<dims(0)-1,(ij(1)+j)<dims(1)-1)
		if curr_chisq ge max_chisq then break
	endfor
	if ij(0)+i ge dims(0) then mflag=1
	if ij(1)+j ge dims(1) then mflag=1
	r_ell(5)=r((ij(0)+i)<dims(1)-1,(ij(1)+j)<dims(1)-1)-r(ij(0),ij(1))
	d_ell(5)=d((ij(0)+i)<dims(1)-1,(ij(1)+j)<dims(1)-1)-d(ij(0),ij(1))
	for i=1,dims(1)/2 do begin	; "upper-left"
		j=i
		curr_chisq=chisq((ij(0)-i)>0,(ij(1)+j)<dims(1)-1)
		if curr_chisq ge max_chisq then break
	endfor
	if i gt ij(0) then mflag=1
	if ij(1)+j ge dims(1) then mflag=1
	r_ell(6)=r((ij(0)-i)>0,(ij(1)+j)<dims(1)-1)-r(ij(0),ij(1))
	d_ell(6)=d((ij(0)-i)>0,(ij(1)+j)<dims(1)-1)-d(ij(0),ij(1))
	for i=1,dims(1)/2 do begin	; "lower-right"
		j=i
		curr_chisq=chisq((ij(0)+i)<dims(0)-1,(ij(1)-j)>0)
		if curr_chisq ge max_chisq then break
	endfor
	if ij(0)+i ge dims(0) then mflag=1
	if j gt ij(1) then mflag=1
	r_ell(7)=r((ij(0)+i)<dims(0)-1,(ij(1)-j)>0)-r(ij(0),ij(1))
	d_ell(7)=d((ij(0)+i)<dims(0)-1,(ij(1)-j)>0)-d(ij(0),ij(1))
;
	if local then begin
		if min_dist gt 1 then mflag=1
	endif
;
	if mflag then begin
		print,''
		print,'***Warning: Contour outside map! Skipping '+date
		if n_elements(bad_files) eq 0 then $
		bad_files=chisq_files(i_file)+' (contour outside)' else $
		bad_files=[bad_files,chisq_files(i_file)+' (contour outside)']
		wait,1
	endif else begin
;		Fit error ellipse to contour at sigma
		positions.rho=sqrt(r_ell^2+d_ell^2)
		positions.theta=atan(r_ell,d_ell)
		i=where(positions.rho eq max(positions.rho)) & i=i(0)
		pa=positions(i).theta
		emajor=positions(i).rho	; semi-major axis
		i=where(positions.rho eq min(positions.rho)) & i=i(0)
		eminor=positions(i).rho	; semi-minor axis
		e_parms=[0,0,emajor,eminor,pa]
		ellipse_options.c=1	; fit center 1/0
;
		print,'Fitting ellipse to contour at min.+sigma...'
		fitellipse ; & e_parms0=e_parms
;
		if debug then begin
			!x.range=[max(r_ell),min(r_ell)]
			xl=!x.range(0)-!x.range(1)
			!y.range=[min(d_ell),max(d_ell)]
			yl=!y.range(1)-!y.range(0)
			if yl gt xl then !x.range=!x.range*yl/xl
			if yl lt xl then !x.range=!x.range*xl/yl
			if i_file eq 0 then window,xsize=800,ysize=700
			plotellipse,0
			j=sort(positions.theta)
			oplot,[r_ell(j),r_ell(j(0))],[d_ell(j),d_ell(j(0))], $
				psym=6,thick=3
		endif
		emajor=e_parms(2)
		eminor=e_parms(3)
		pa=(e_parms(4)*rad) mod 180
		if eminor gt emajor then begin
			v=emajor
			emajor=eminor
			eminor=v
			pa=(pa+90) mod 180
		endif
		if pa lt 0 then pa=pa+180
		rho_error=0
		theta_error=0
;
		printf,unit,'! '+date+', ellipse at sigma'
		printf,unit,' '+component,jy, $
      			rho(index),theta(index), $
			emajor,eminor,pa, $
			rho_error,theta_error,min(chisq), $
		format='(a4,2x,f9.4,2x,f6.2,1x,f7.2,2x,f6.3,1x,f6.3,1x,f6.1,'+ $
	       		'2x,f7.3,1x,f8.3,1x,f7.1)'
;
	endelse	; mflag
endfor
;
free_lun,unit
spawn,'cat fitchisq.psn'
;
print,'------------------------------'
if n_elements(bad_files) gt 0 then begin
	print,'Failed to analyze these files:'
	for i=0,n_elements(bad_files)-1 do print,bad_files(i)
endif
if local then positions=local_positions
;
end
;-------------------------------------------------------------------------------
