;*******************************************************************************
; File: oyster.pro
;
; Source:
; -------
; Dr. Christian A. Hummel
; European Southern Observatory
; Karl-Schwarzschild-Str. 2
; 85748 Garching
; Germany
;
; Up to Version 5 written while at:
; NPOI Division, Flagstaff Station
; U.S. Naval Observatory
; c/o 3450 Massachusetts Avenue NW
; Washington DC 20392-5420
;
; Versions 6 and later written while at ESO.
; This version only runs well on 24 bit displays or higher.
;
; Very brief version history
; Version 0: PV-WAVE release, includes CHAMELEON and AMOEBA
; Version 1: first IDL version after several releases for PV-WAVE
; Version 2: ?
; Version 3: Imaging
; Version 4: "Michelson version" used for 2001 school, stability tested
; Version 5: Astrometry, raw data analysis, coherent integration
; Version 6: Other interferometer support, MyMidiGui, MyAmberGui
; Version 7: Advanced modeling and imaging support, 64 bit support
; Version 8: Pipeline (npoipipe) and GDL script support, high-S.R. processing
; Version 9: P/G/M-GUIs(OIFITS2), ImgFT buffer, UT=0/time->int_time SC flag rev.
;
; Description:
; ------------
; OISDR (Optical Interferometer Script Data Reduction, pronounced "oyster")
; is the main NPOI data reduction software written in IDL (ITT).
; This package comprises three main parts, CHAMELEON, STARBASE, and AMOEBA. 
; Compile OYSTER using c.pro by typing at the UNIX prompt: idl c. This will 
; start IDL to execute the commands in c.pro, which will compile all 
; libraries for OYSTER. Common blocks to be visible at the MAIN level are 
; also declared. All other common blocks are only available to procedures 
; which have them declared.
;
; There is a manual for OYSTER contained in the file "oyster.tex".
;
; CHAMELEON and AMOEBA have been more or less separate entities before
; they were combined into OYSTER. This has led to some pecularities 
; concerning their simultaneous use. This is mostly because CHAMELEON 
; stores one night's data at a time, whereas AMOEBA uses a buffer to
; store more than one night, from which it can retrieve the data in
; the form used by CHAMELEON. Currently, CHAMELEON procedures which
; load or create scan data also deposit a copy into the buffer, but
; only after clearing it.
;
; Some environment variables which control the behaviour of IDL:
; IDL_DIR
; IDL_PATH: This tells IDL where to look for procedures
; IDL_STARTUP: The commands in this files are executed upon startup
; .idl.startup: another default startup file IDL checks
;
; Block directory:
; ----------------
;
; Block 1: setup_oyster,update_eop,tekcolor,trucolor,saverestore
; Block 2: logo_oyster,logo_chameleon,logo_amoeba,logo_starbase,logo_cobra
; Block 3: checkdata,checkdate,fixstarids,scanconfig,beamconfig
; Block 4: system_id,instrument_id,beamcombiner_id,system_config
; Block 5: quit,bye
;
;************************************************************************Block 1
pro setup_oyster,dir_oyster,dir_catalogs,dir_atmospheres,notv=notv,gdl=gdl
;
; This procedure is called by the OYSTER startup script oyster.pro
; in the OYSTER home directory. The startup script passes as arguments
; the home directories for OYSTER, the catalogs, and atmospheres, e.g. 
; '/home/cah/oyster/', '/home/cah/catalogs/', and '/home/cah/atmospheres'.
;
; Run this procedure before using OYSTER data reduction scripts.
; Initializes all variables in the following common blocks.
; Also calls C-library to store path of catalogs.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
common AstroConstants,m_sun,r_sun,a_unit,s_year
common Mark3,mark3_baselines,mark3_configs,mark3_cat,mark3_bsc
common NPOI,npoi_cat,npoi_hip
common ModelFit,parameters,ds_options
;
if n_elements(notv) eq 0 then notv=0 else notv=notv gt 0
if n_elements(gdl) eq 0 then gdl=0 else gdl=gdl gt 0
;
; Initialize user defined system variables
defsysv,'!oyster_dir',''
defsysv,'!catalogs_dir',''
defsysv,'!atmospheres_dir',''
defsysv,'!evolution_dir',''
defsysv,'!external_lib',''
defsysv,'!xsize',640
defsysv,'!ysize',512
defsysv,'!dxsize',0
defsysv,'!dysize',0
defsysv,'!qiet',0		; Do display informational messages
defsysv,'!display',!d.name
defsysv,'!v2bias',1
defsysv,'!tabias',1
defsysv,'!date_change',0	; 1 if obs. started before UT=0 (OIFITS only)
defsysv,'!offsetcal',0
defsysv,'!int_time',0		; Max. int. time [s] "w/out smearing", 0 = none
defsysv,'!idl',0		; IDL or GDL
defsysv,'!cop','COL'		; Center of phase: default is center of light
				; COP: brightest component, COM: c.o. mass
defsysv,'!owner','chummel'
;
; Initialize OYSTER home directory
;
!oyster_dir=dir_oyster
;
; Initialize catalogs home directory
;
!catalogs_dir=dir_catalogs
;
; Initialize atmospheres home directory
;
!atmospheres_dir=dir_atmospheres
;
; Add oyster source code directories to path
;
if gdl then !idl=0 else !idl=1
if gdl then $
!path=!path+':' $
     +!oyster_dir+'source/gdl:' $
     +!oyster_dir+'source/idl/lib:' $
     +!oyster_dir+'source/idl/cw_lib:' $
     +!oyster_dir+'source/cmsvlib:' $
     +!oyster_dir+'source/vinci:' $
     +!oyster_dir+'source/midi:' $
     +!oyster_dir+'source/amber:' $
     +!oyster_dir+'source/pionier:' $
     +!oyster_dir+'source/gravity:' $
     +!oyster_dir+'source/matisse:' $
     +!oyster_dir+'source/oifits:' $
     +!oyster_dir+'source/mpfit:' $
     +!oyster_dir+'source/pds:' $
     +!oyster_dir+'source/idlastro/pro:' $
     +!oyster_dir+'source/idlastro/pro/coyote:' $
     +!oyster_dir+'source/idlastro/pro/jhuapl:' $
     +!oyster_dir+'source/torun/cw_lib:' $
     +!oyster_dir+'source/torun/gdl:' $
else $
!path=!path+':' $
     +!oyster_dir+'source/idl:' $
     +!oyster_dir+'source/common:' $
     +!oyster_dir+'source/vinci:' $
     +!oyster_dir+'source/midi:' $
     +!oyster_dir+'source/amber:' $
     +!oyster_dir+'source/pionier:' $
     +!oyster_dir+'source/gravity:' $
     +!oyster_dir+'source/matisse:' $
     +!oyster_dir+'source/oifits:' $
     +!oyster_dir+'source/mpfit:' $
     +!oyster_dir+'source/pds:' $
     +!oyster_dir+'source/idlastro/pro:' $
     +!oyster_dir+'source/idlastro/pro/coyote:' $
     +!oyster_dir+'source/idlastro/pro/jhuapl:' $
     +!oyster_dir+'source/proprietary/coyote:' $
     +!oyster_dir+'source/proprietary/amores:' $
     +!oyster_dir+'source/proprietary/ttsre:' $
     +!oyster_dir+'source/proprietary/christian:' $
     +!oyster_dir+'source/proprietary/oliver:' $
     +!oyster_dir+'source/proprietary/oliver/erfa:' $
     +!oyster_dir+'source/proprietary/oliver/math:' $
     +!oyster_dir+'source/proprietary/drimmel:' $
     +!oyster_dir+'source/proprietary/drimmel/cgis:' $
     +!oyster_dir+'source/proprietary/drimmel/cgis/quidl:' $
     +!oyster_dir+'source/proprietary/hochunk3d:' $
     +!oyster_dir+'source/proprietary/tellrem/routines:'
;    +'/data/naco/NACO_SAM_pipeline/starfinder:' $
;    +'/data/naco/NACO_SAM_pipeline/code:'
;
; Initialize external library
!external_lib=''
ostype=strupcase(!version.os)
if strpos(ostype,'LINUX') ne -1 then !external_lib='linux/oyster.so'
if strpos(ostype,'DARWIN') ne -1 then !external_lib='darwin/oyster.so'
if strlen(!external_lib) eq 0 then begin
	print,'Warning(SETUP_OYSTER): possibly incompatible OS ',ostype,'!'
	!external_lib='oyster.so'
endif
!external_lib=!oyster_dir+!external_lib
r=file_search(!external_lib)
if strlen(r(0)) ne 0 then $
	status=linknload(!external_lib,'init_oyster', $
		!catalogs_dir,!atmospheres_dir, $
		pathname(!external_lib))
;
; Initialize mathematical and physical constants
;
c_light=2.997925d8			; Speed of light [m/s]
pi_circle=3.14159265358979323846d0	; PI
e_euler=2.718281828459045d0		; Euler's e
i_complex=sqrt(complex(-1))		; i
m_sun=1.9891d30                         ; kg
r_sun=6.9599d8				; m
a_unit=1.49597870d11                    ; AU [m]
s_year=365.2564d0                       ; Sidereal year [d]
;
; Initialize dispersion constants
; 			; K.P. Birch & M.J. Downs, 1993, Metrologia, 30, 155
; a_disp=dblarr(5)	; n(lambda)-1=(P/96095.43*(1+1e-8*(0.601-0.00972t)*P)/
; a_disp(0)=8343.05d0	;                        (1+0.003661t)*a(lambda)
; a_disp(1)=2406294.d0	;             -f*b(lambda)*1e-2)*1e-8
; a_disp(2)=130.d0	; a(lambda)=a0+a1/(a2-s^2)+a3/(a4-s^2)
; a_disp(3)=15999.d0	; b(lambda)=b0+b1*s^2
; a_disp(4)=38.9d0	; s=1/lambda (lambda in 1e-6 m)
; b_disp=dblarr(2)	; P=pressure in Pa
; b_disp(0)=3.7345d0	; t=t90=temperature in C
; b_disp(1)=-0.0401d0	; f=partial water vapor pressure in Pa
;
a_disp=dblarr(5)	; J.C. Owens, Appl. Opt. 6, 51 (1967)
a_disp(0)=2372.434	; n(lambda)-1=a(lambda)P/T-b(lambda)e/T
a_disp(1)=684255.24	;
a_disp(2)=130.0	;
a_disp(3)=4549.4	; a(lambda)=1e-8*(a0+a1/(a2-s2)+a3/(a4-s2))
a_disp(4)=38.9		; b(lambda)=a(lambda)-1e-8*(b0+b1*s2+b2*s4+b3*s6)
b_disp=dblarr(4)	; with a0=a_disp(0) asf. & b0=b_disp(0) asf.
b_disp(0)=6487.31	; & s2=(1/lambda)^2 asf. (lambda in 1e-6 m)
b_disp(1)=58.058	; P=pressure in mb
b_disp(2)=-0.71150	; T=temperature in K
b_disp(3)=0.08851	; e=partial water vapor pressure in mb
;
;
; Initialize Mark3 common block variables
; Note: FK5 1817 (unknown) and 2001 (Nova Cygni) are still missing!
; FKV2001 is Nova Cygni=V1974 Cyg (USR0006)
;
mark3_baselines=['NAS-SAS','NAS-EAS','NVA-SVA','NVA-SVB','NVB-SVA', $
	'NVB-SVB','NVC-SVA','NVA-SVC','NVC-SVB','NVB-SVC','NVD-SVA','NVC-SVC', $
	'NVA-SVD','NVD-SVB','NVE-SVA','NVB-SVD','NVE-SVB','NVD-SVC','NVF-SVA', $
	'NVC-SVD','NVA-SVE','NVF-SVB','NVE-SVC','NVB-SVE','NVD-SVD','NVC-SVE', $
	'NVA-SVF','NVF-SVC','NVE-SVD','NVB-SVF','NVD-SVE','NVC-SVF','NVF-SVD', $
	'NVE-SVE','NVD-SVF','NVF-SVE','NVE-SVF','NVF-SVF']
mark3_configs=['AQB','HA1','HA2','700','TIO','45A','45B', $
	     '45C','45D','STD','HA2','ST2','HA4']
mark3_cat=['FKV1700','FKV1701','FKV1702','FKV1703','FKV1704','FKV1706', $
           'FKV1707','FKV1708','FKV1709','FKV1710','FKV1711','FKV1713', $
           'FKV1714','FKV1715','FKV1716','FKV1717','FKV1718','FKV1719', $
           'FKV1720','FKV1721','FKV1722','FKV1723','FKV1724','FKV1725', $
           'FKV1726','FKV1727','FKV1728','FKV1729','FKV1730','FKV1731', $
           'FKV1732','FKV1733','FKV1734','FKV1735','FKV1736','FKV1737', $
           'FKV1738','FKV1739','FKV1740','FKV1741','FKV1742','FKV1743', $
           'FKV1744','FKV1745','FKV1746','FKV1747','FKV1748','FKV1749']
mark3_cat=[mark3_cat, $
	  ['FKV1750','FKV1751','FKV1752','FKV1753','FKV1754','FKV1755', $
           'FKV1756','FKV1757','FKV1758','FKV1759','FKV1760','FKV1761', $
           'FKV1762','FKV1763','FKV1764','FKV1765','FKV1766','FKV1767', $
           'FKV1768','FKV1769','FKV1770','FKV1771','FKV1772','FKV1773', $
           'FKV1774','FKV1775','FKV1776','FKV1777','FKV1778','FKV1779', $
           'FKV1781','FKV1782','FKV1783','FKV1784','FKV1785','FKV1786', $
           'FKV1788','FKV1789','FKV1790','FKV1791','FKV1792','FKV1793', $
           'FKV1794','FKV1795','FKV1796','FKV1797','FKV1798','FKV1799']]
mark3_cat=[mark3_cat, $
	  ['FKV1787','FKV1804','FKV1805','FKV1806','FKV1807','FKV1808', $
	   'FKV1809','FKV1810','FKV1811','FKV1812','FKV1813','FKV1814', $
	   'FKV1815','FKV1816','FKV2001']]
mark3_bsc=['BSC0271','BSC0430','BSC0244','BSC1706','BSC3773','BSC2405', $
           'BSC7564','BSC6212','BSC7806','BSC6406','BSC7478','BSC2854', $
           'BSC7528','BSC6132','BSC4057','BSC9004','BSC6752','BSC7882', $
           'BSC1766','BSC8316','BSC0681','BSC0808','BSC0804','BSC0941', $
           'BSC0618','BSC7139','BSC7751','BSC1003','BSC0843','BSC0643', $
           'BSC0219','BSC8130','BSC8752','BSC0994','BSC1601','BSC8796', $
           'BSC1412','BSC1788','BSC1948','BSC2216','BSC2095','BSC2091', $
           'BSC7897','BSC1149','BSC1156','BSC1145','BSC1180','BSC6884']
mark3_bsc=[mark3_bsc, $
          ['BSC3974','BSC5062','BSC3690','BSC5826','BSC3799','BSC4300', $
           'BSC4375','BSC4399','BSC6315','BSC3624','BSC1155','BSC6146', $
           'BSC5080','BSC3882','BSC1637','BSC1726','BSC1879','BSC3482', $
           'BSC5723','BSC7133','BSC5868','BSC4825','BSC4775','BSC8579', $
           'BSC8819','BSC8417','BSC8383','BSC7333','BSC7447','BSC7884', $
           'BSC8315','BSC0660','BSC6153','BSC0788','BSC5958','BSC8123', $
           'BSC1392','BSC7963','BSC6388','BSC1306','BSC0548','BSC1473', $
           'BSC2973','BSC1698','BSC1458','BSC1129','BSC1324','BSC1937']]
mark3_bsc=[mark3_bsc, $
	  ['BSC8865','BSC1087','BSC4248','BSC4362','BSC1496','BSC2508', $
	   'BSC2580','BSC2742','BSC2764','BSC2766','BSC8726','BSC8850', $
	   'BSC8860','BSC5329','USR0006']]
; 
; Initialize NPOI common block variables
; HD200077, Gam Ari, UX Ari, AX Mon, 41 Leo, Nova Del, Alpha Comae 
npoi_cat=['BSC9111',  'BSC9112',  'BSC9113',  'BSC9114',  'BSC9115', $
	  'BSC9116', 'BSC9117']
npoi_hip=['HIP103641','HIP008832','HIP016042','HIP031019','HIP050583', $
	  'USR0007', 'BSC4968']
;
; Initialize data pointer
;
ds_options=alloc_ds_options()
;
; Change prompt
;
!prompt='OYSTER> '
;
; Enable startup without X-server
if not notv then begin
; 	OYSTER now assumed to run on 24 bit or higher displays
;	window,0 & wdelete
	if not gdl then begin
		trucolor
		device,retain=2	; Refresh window when they are uncovered
	endif
;
;	Merge the resources file with the .Xdefaults resources
	if getenv('USER') eq !owner $
        	then spawn,'xrdb -merge '+!oyster_dir+'resources.owner',r $
        	else spawn,'xrdb -merge '+!oyster_dir+'resources',r
endif
; compile_opt idl2
;
print,''
print,'---------------------'
print,' OYSTER Version 9.14'
print,' by Christian Hummel'
print,'---------------------'
print,''
;
end
;-------------------------------------------------------------------------------
pro update_eop,year
;
; This routine will download Earth orientation data provided by the IERS.
; The http address is composed of a static part and a volume string, e.g.:
; https://datacenter.iers.org/data/6/bulletina-xxxiii-025.txt where "025"
; is the week number (this routine will cycle through all weeks until the 
; current one. The output is written to mark.upd and needs to appended to
; mark3.dat.
;
; This routine was used temporarily instead of the update script in folder
; "usno". As of 2023, the latter has been updated and works again.
;
; Roman numerals
;
;  I            1
;  IV           4
;  V            5
;  IX           9
;  X            10
;  XL           40
;  L            50
;  XC           90
;  C            100
;  CD           400
;  D            500
;  CM           900 
;  M            1000
;
years=[2019,2020,2021,2022,2023,2024,2025]
volumes=['xxxii','xxxiii','xxxiv','xxxv','xxxvi','xxxvii','xxxviii']
;
if n_elements(year) eq 0 then begin
	words=nameparse(today(),'_')
	year=words(n_elements(words)-1)
endif
volume=volumes(where(years eq year)) & volume=volume(0)
;
jd_jan1=julian(year,1,1)
jd_today=today(/jd)
;
link='https://datacenter.iers.org/data/6/'
;
cd,!oyster_dir+'/usno',current=old_dir
;
openw,unit,'mark3.upd',/get_lun
;
for i=1,52 do begin
	if jd_jan1+i*7 gt jd_today then break
	week=string(i,format='(i3.3)')
	file='bulletina-'+volume+'-'+week+'.txt'
	print,'Downloading '+file
	spawn,'wget '+link+file
;
	l=''
	status=dc_read_fixed(file,l,format='(a80)',/col)
	index=where(strpos(l,'IERS Rapid Service') ge 0,count)
	if count eq 0 then begin
		print,'Error: could not find line IERS Rapid Service'
		return
	endif
;	n_index=n_elements(index)
	j=3
	repeat begin
;		printf,unit,l(index(n_index-1)+j)
		printf,unit,l(max(index)+j)
		j=j+1
	endrep until strlen(strcompress(l(max(index)+j),/remove_all)) eq 0
;
endfor
free_lun,unit
;
cd,old_dir
;
end
;-------------------------------------------------------------------------------
pro tekcolor
;
; Switch color system to color indices/tables and call tek_color.
; This procedure should *not* be used normally in OYSTER.
;
if !d.name eq 'X' then begin
	device,decompose=0	; Do not compose index into RGB triples
	tek_color		; Load Tektronix color table
	!p.color=1
endif
;
end
;-------------------------------------------------------------------------------
pro trucolor
;
; As of October 24, 2008, OYSTER assumes to be running on 24+ bit displays.
; Therfore, we decompose 24-bit color indices into the three 8-bit
; RGB indices. TrueColor means non-translated indices, DirectColor
; means that the 8-bit indices can also be translated with a table to
; the final value.
;
; Some comments on color in IDL: to use color tables (non-decomposed 
; mode), load RGB indices with TVLCT, e.g. TVLCT, 53, 156, 83, 100.
; Pre-defined color tables are loaded using LOADCT.
;
if !d.name eq 'X' then begin
	if !idl then begin
;		On MacOS, the following call to device cause an exception
		!except=0		; To ignore the exception
		help,/device,output=r
		result=check_math()	; Reset the exception status
		!except=1		; Set back to default
		index=where(strpos(r,'Display') ge 0)
		r=r(index)
		index=where(strpos(r,'Depth') ge 0)
		r=r(index)
		words=nameparse(r(0))
		bit_depth=long(words(3))
		device,true_color=bit_depth,/decompose
	endif
	loadct,0,/silent
	!p.color=tci(1)
endif
;
end
;-------------------------------------------------------------------------------
pro saverestore,file,action
;
; Saves/restores OYSTER session to/from file. Single night data only!
;
; Common blocks included here will be saved/restored from file
common StarBase,startable,notes
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common InchData,Sol0,SolN,Jscan,MetroTime,PivotMotion, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common AuxData,parallaxes,k1,k2,vsini
;
if n_elements(file) eq 0 then begin
	print,'***Error(SAVERESTORE): file undefined!'
	return
endif
if file eq '' then begin
	print,'***Error(SAVERESTORE): file unspecified!'
	return
endif
if n_elements(action) eq 0 then begin
	print,'***Error(SAVERESTORE): action undefined!'
	return
endif
if action eq '' then begin
	print,'***Error(SAVERESTORE): action unspecified!'
	return
endif
;
case action of
	'save'   :begin
	  	  save,filename=file,/variables
	  	  print,'Common block data saved to: ',file
	  	  end
	'restore':begin
		  result=file_search(file,count=fcount)
		  if fcount ne 0 then begin
			restore,filename=file 
			print,'Common block data restored.'
	  	  endif else begin
			print,'***Error(SAVERESTORE): file not found: ',file,'!'
			return
		  endelse
		  end
	     else:begin
		  print,'***Error(SAVERESTORE): unknown action: ',action,'!'
		  return
		  end
endcase
;
end
;************************************************************************Block 2
pro logo_oyster,colors
;
; Display OYSTER logo.
;
if n_elements(colors) eq 0 then colors=256
;
status=dc_read_tiff(!oyster_dir+'logos/oyster.tif',image,colormap=map)
width=n_elements(image(*,0))
length=n_elements(image(0,*))
window,title='Welcome to OYSTER!',/free, $
	xsize=width,ysize=length,colors=colors
device,decompose=0
tvlct,map(*,0),map(*,1),map(*,2)
tv,image,order=1
;
end
;-------------------------------------------------------------------------------
pro logo_chameleon,colors
;
; Display CHAMELEON logo.
;
if n_elements(colors) eq 0 then colors=256
;
status=dc_read_tiff(!oyster_dir+'logos/chameleon.tif',image,colormap=map)
nb=40
width=n_elements(image(*,0))
length=n_elements(image(0,*))
window,title='Welcome to CHAMELEON!',/free, $
	xsize=width,ysize=length-nb,colors=colors
image(10:110,410:430)=image(130:230,410:430)
image(472:532,410:430)=image(130:190,410:430)
image(532:592,410:430)=image(130:190,410:430)
device,decompose=0
tvlct,map(*,0)/2,map(*,1),map(*,2)
tv,image(*,0:length-nb-1),order=1
;
end
;-------------------------------------------------------------------------------
pro logo_amoeba,colors
;
; Display AMOEBA logo.
;
; Kingdom: Protist
; Scientific Name: Amoeba proteus
; Image Courtesy of: Joanne Whallon
; Image Width: 30 microns
; Image Technology: Laser Scanning Confocal Microscopy
;
if n_elements(colors) eq 0 then colors=256
;
status=dc_read_tiff(!oyster_dir+'logos/amoeba.tif',image,colormap=map)
width=n_elements(image(*,0))
length=n_elements(image(0,*))
window,title='Welcome to AMOEBA!',/free, $
	xsize=width,ysize=length,colors=colors
device,decompose=0
tvlct,map(*,0),map(*,1),map(*,2)
tv,image,order=1
;
end
;-------------------------------------------------------------------------------
pro logo_starbase,colors
;
; Display STARBASE logo.
;
if n_elements(colors) eq 0 then colors=256
;
status=dc_read_tiff(!oyster_dir+'logos/starbase.tif',image,colormap=map)
width=n_elements(image(*,0))
length=n_elements(image(0,*))
window,title='Welcome to STARBASE!',/free, $
	xsize=width,ysize=length,colors=colors
device,decompose=0
tvlct,map(*,0),map(*,1),map(*,2)
tv,image,order=1
;
end
;-------------------------------------------------------------------------------
pro logo_cobra,colors
;
; Display COBRA logo.
;
if n_elements(colors) eq 0 then colors=256
;
status=dc_read_tiff(!oyster_dir+'logos/cobra.tif',image,colormap=map)
width=n_elements(image(*,0))
length=n_elements(image(0,*))
window,title='Welcome to COBRA!',/free, $
	xsize=width,ysize=length,colors=colors
device,decompose=0
tvlct,map(*,0),map(*,1),map(*,2)
tv,image,order=1
;
end
;************************************************************************Block 3
function checkdata,items
;
; Check whether particular data item is loaded; return error code otherwise.
;
common Tables,ScanTable,BGTable,StationTable
common FlagTables,pointflagtable,inchflagtable,bgflagtable,scanflagtable
common StarBase,startable,notes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common InchData,Sol0,SolN,Jscan,MetroTime,PivotMotion, $
        ParX,ParXErr,ParY,ParYErr,ParZ,ParZErr,MetroPath,MetroPathErr, $
	MotorAngle,MotorAngleErr
common PointData,Rec0,RecN,Iscan,StarId,PointTime, $
        FDLPos,FDLPosErr,MetroPos,MetroPosErr,GeoDelay,GeoDelayErr, $
        DelayJitter,DelayJitterErr,SoftDelay,SoftDelayErr, $
        NATJitter,NATJitterErr,NATCounts,NATCountsErr, $
        GrpDelay,GrpDelayErr,DryDelay,DryDelayErr,WetDelay,WetDelayErr, $
        PhotonRate,PhotonRateErr,VisSq,VisSqErr, $
        ComplexVis,ComplexVisErr,ComplTriple,ComplTripleErr, $
        VisAmp,VisAmpErr,VisPhase,VisPhaseErr, $
        TripleAmp,TripleAmpErr,TriplePhase,TriplePhaseErr
;
num_items=n_elements(items)
if num_items eq 0 then begin
	print,'***Error(CHECKDATA): no items specified!'
	return,-1
endif
;
returncode=0
for i=0,num_items-1 do begin
	errcode=0
	case items(i) of
		 1:if n_elements(scantable) eq 0 then errcode=items(i)
		 2:if n_elements(bgtable) eq 0 then errcode=items(i)
		 3:if n_elements(startable) eq 0 then errcode=items(i)
		 4:if n_elements(stationtable) eq 0 then errcode=items(i)
		 5:if n_elements(Date) eq 0 then errcode=items(i)
		 6:if n_elements(SystemId) eq 0 then errcode=items(i)
		 7:begin
		   result=size(GeoParms)
		   if result(n_elements(result)-2) ne 8 then errcode=items(i) $
			   else $
			   if n_elements(GeoInfo) eq 0 then GeoInfo=GeoParms
		   end
		 8:begin
		   result=size(GenConfig)
		   if result(n_elements(result)-2) ne 8 then errcode=items(i) $
			   else $
			   if n_elements(GenInfo) eq 0 then GenInfo=GenConfig
		   end
		 9:if n_elements(scans) eq 0 then errcode=items(i)
		10:if n_elements(bgscans) eq 0 then errcode=items(i)
		11:if n_elements(Rec0) eq 0 then errcode=items(i)
		12:if n_elements(bufferinfo) eq 0 then errcode=items(i)
		13:if n_elements(GeoInfo) eq 0 then errcode=items(i)
		14:if n_elements(GenInfo) eq 0 then errcode=items(i)
		15:if n_elements(GrpDelay) eq 0 then errcode=items(i)
		16:if n_elements(MetroTime) eq 0 then errcode=items(i)
		17:if n_elements(pointflagtable) eq 0 then errcode=items(i)
		18:if n_elements(bgflagtable) eq 0 then errcode=items(i)
		19:if n_elements(scanflagtable) eq 0 then errcode=items(i)
		20:if n_elements(inchflagtable) eq 0 then errcode=items(i)
		21:if n_elements(MetroConfig) eq 0 then errcode=items(i)
		22:if n_elements(positions) eq 0 then errcode=items(i)
		23:if n_elements(velocities) eq 0 then errcode=items(i)
		24:if n_elements(magnitudes) eq 0 then errcode=items(i)
	      else:errcode=-1
	endcase
	case errcode of
		-1:print,'***Error(CHECKDATA): unknown item id:',items(i)
		 0:errcode=0
		 1:print,'***Error(CHECKDATA): scantable does not exist!'
		 2:print,'***Error(CHECKDATA): bgtable does not exist!'
		 3:print,'***Error(CHECKDATA): startable does not exist!'
		 4:print,'***Error(CHECKDATA): stationtable does not exist!'
		 5:print,'***Error(CHECKDATA): Date undefined!'
		 6:print,'***Error(CHECKDATA): SystemId undefined!'
		 7:print,'***Error(CHECKDATA): GeoParms undefined or corrupt!'
		 8:print,'***Error(CHECKDATA): GenConfig undefined or corrupt!'
		 9:print,'***Error(CHECKDATA): no ScanData!'
		10:print,'***Error(CHECKDATA): no BackgndData!'
		11:print,'***Error(CHECKDATA): no PointData!'
		12:print,'***Error(CHECKDATA): no bufferinfo!'
		13:print,'***Error(CHECKDATA): no GeoInfo!'
		14:print,'***Error(CHECKDATA): no GenInfo!'
		15:print,'***Error(CHECKDATA): no DelayData!'
		16:print,'***Error(CHECKDATA): no InchData!'
		17:print,'***Error(CHECKDATA): no pointflagtable!'
		18:print,'***Error(CHECKDATA): no bgflagtable!'
		19:print,'***Error(CHECKDATA): no scanflagtable!'
		20:print,'***Error(CHECKDATA): no inchflagtable!'
		21:print,'***Error(CHECKDATA): MetroConfig undefined!'
		22:print,'***Error(CHECKDATA): Positions undefined!'
		23:print,'***Error(CHECKDATA): Velocities undefined!'
		24:print,'***Error(CHECKDATA): Magnitudes undefined!'
	endcase
	if errcode ne 0 then returncode=-1
endfor
return,returncode
end
;-------------------------------------------------------------------------------
function checkdate
;
; If observations began before 0 UT, OYSTER sets the date to the 
; previous day in order to avoid negative timestamps. This functions returns 
; a unique date (the UT date) by detecting these circumstances.
;
; If the timestamps exceed 48 hours, this is a multinight observation
; and in this case we do not change the date. (This case happens with
; VINCI data for example.)
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
;
if n_elements(scantable) ne 0 then time=scantable.starttime
if n_elements(scans) ne 0 then time=scans.time
;
if max(abs(time)/3600) gt 24 and max(abs(time)/3600) lt 48 $
			     then return,nextdate(date) $
			     else return,date
;
end
;-------------------------------------------------------------------------------
function fixstarids,StarId
;
; The Mark3 and also NPOI have been using fake FKV and BSC IDs, respectively,
; for stars not found in these catalogs. Because stellar parameters are not
; saved in the data files but read from external catalogs, we need to translate
; these to other catalogs (Hipparcos), or the USR ("user") catalog.
;
common NPOI,npoi_cat,npoi_hip
common Mark3,mark3_baselines,mark3_configs,mark3_cat,mark3_bsc
;
; Replace Mark3 FKV numbers >= 1700 <= 2001 with corresponding BSC numbers 
for i=0,n_elements(mark3_cat)-1 do begin
	index=where(StarId eq mark3_cat(i),count)
	if count gt 0 then StarId(index)=mark3_bsc(i)
endfor
; Replace NPOI fake BSC numbers >= 9111 with corresponding HIP/USR numbers
for i=0,n_elements(npoi_cat)-1 do begin
	index=where(StarId eq npoi_cat(i),count)
	if count gt 0 then StarId(index)=npoi_hip(i)
endfor
;
; Added for version 9 to get fixed lengths of StarIds
index=where(strmid(StarId,0,3) eq 'HDN' and strlen(StarId) lt 9,count)
for i=0,count-1 do begin
;	ID=long(strmid(StarId(index(i)),3,5))
;	There are cases such as HDN_5300...
	ID=long(strmid(strjoin(nameparse(StarId(index(i)),['_','-'])),3,5))
	StarId(index(i))=strmid(StarId(index(i)),0,3)+string(ID,format='(i6.6)')
endfor
;
index=where(strmid(StarId,0,3) eq 'HIP' and strlen(StarId) lt 9,count)
for i=0,count-1 do begin
;	ID=long(strmid(StarId(index(i)),3,5))
;	There are cases such as HIP_5300...
	ID=long(strmid(strjoin(nameparse(StarId(index(i)),['_','-'])),3,5))
	StarId(index(i))=strmid(StarId(index(i)),0,3)+string(ID,format='(i6.6)')
endfor
;
starids=StarId
return,starids
;
end
;-------------------------------------------------------------------------------
function scanconfig,cf,coherent=coherent,starlist=starlist,maxconfig=maxconfig
;
; The scan configuration specifies which of the configured stations (the ones
; given in the general configuration) were actually used on a scan by scan basis.
; This information is only important for system calibrations such as bias
; and response, the results of which are stored in GenConfig.
;
; Multi-purpose function for scan configuration inquiries.
; Examples: 	cf=<undefined>:	return existing scan configurations
;		cf='111010':	return scan ids with this configuration (code=7)
;		cf=14:		return configuration for this scan (code=2)
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(coherent) eq 0 then coherent=0 else coherent=coherent gt 0
if n_elements(starlist) eq 0 then starlist=0 else starlist=starlist gt 0
if n_elements(maxconfig) eq 0 then maxconfig=0 else maxconfig=maxconfig gt 0
;
if maxconfig then begin
	if n_elements(scantable) eq 0 then return,300
	return,n_elements(scantable)
endif
if n_elements(scantable) eq 0 then return,-1
if n_elements(genconfig) ne 0 then index=indgen(genconfig.numsid) $
			      else index=where(total(scantable.station,2) gt 0)
stations=scantable.station(index) 
stations=strcompress(strjoin(string(stations)),/remove)
starids=' '+scantable.starid
;
r=size(cf) & code=r(n_elements(r)-2)
if code eq 0 then begin
	if coherent then begin
		index=where(scantable.code eq 1,count)
		if count eq 0 then return,''
		stations=stations(index)
		starids=starids(index)
	endif
	if not starlist then starids=''
	return,unique(stations+starids)
endif
;
if code eq 7 then begin
	index=0
	for i=0,n_elements(cf)-1 do begin
		if coherent then index=[index,where(stations eq cf(i) $
						and scantable.code eq 1)+1] $
			    else index=[index,where(stations eq cf(i))+1]
	endfor
	jndex=where(index gt 0,count)
	if count eq 0 then return,0 else return,unique(index(jndex))
endif
;
starid=' '+scantable(cf-1).starid
if not starlist then starid=''
if code eq 2 or code eq 3 then $
return,stations(cf-1)+starid
;
end
;-------------------------------------------------------------------------------
function beamconfig,beam,scanids
;
; Return scan configuration for every scan ID accounting for which
; stations are present in the specified output beam.
;
common Tables,ScanTable,BGTable,StationTable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(scanids) eq 0 then scanids=indgen(n_elements(scantable))+1
;
stations=unique(nameparse(strjoin( $
  genconfig.baselineid(0:genconfig.numbaseline(beam-1)-1,beam-1),'-'),'-'))
;
beams=scantable(scanids-1).station(0:genconfig.numsid-1)
for i=0,genconfig.numsid-1 do begin
	index=where(stations eq genconfig.stationid(i),count)
	if count eq 0 then beams(i,*)=0
endfor
;
return,beams
;
end
;-------------------------------------------------------------------------------
function tripleconfig,tr
;
; Returns the baseline factors (+/-1) for each triple in the data.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
fBaseMatrix=intarr(GenConfig.NumSid,3)
fBaseFactor=fltarr(3)+1
;
for l=0,2 do begin
	ob=GenConfig.TripleBeam(l,tr-1)
	bl=GenConfig.TripleBase(l,tr-1)
	j1=where(GenConfig.StationId eq $
  	strmid(GenConfig.BaselineId(bl,ob),0,3))
	j2=where(GenConfig.StationId eq $
  	strmid(GenConfig.BaselineId(bl,ob),4,3))
	fBaseMatrix(j1,l)=+1
	fBaseMatrix(j2,l)=-1
	if l gt 0 then fBaseFactor(l)=-total(fBaseMatrix(*,0)*fBaseMatrix(*,l))
endfor
;
return,fBaseFactor
;
end
;************************************************************************Block 4
function system_id,sysid
;
; Return the common OYSTER system name for sysid. If not found,
; return sysid unchanged. Return SystemId if no input provided.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(sysid) eq 0 then sysid=''
;
if strpos(strupcase(sysid),'MARK') ge 0 then return,'Mark3'
if strpos(strupcase(sysid),'PTI') ge 0 then return,'PTI'
if strpos(strupcase(sysid),'CHARA') ge 0 then return,'CHARA'
if strpos(strupcase(sysid),'IOTA') ge 0 then return,'IOTA'
if strpos(strupcase(sysid),'KECK') ge 0 then return,'Keck'
if strpos(strupcase(sysid),'T-RECS') ge 0 then return,'T-ReCS'
if strpos(strupcase(sysid),'NACO') ge 0 then return,'NACO'
if strpos(strupcase(sysid),'VLTI') ge 0 then return,'VLTI'
if strpos(strupcase(sysid),'NPOI') ge 0 then return,'NPOI'
if strpos(strupcase(sysid),'COAST') ge 0 then return,'COAST'
if strpos(strupcase(sysid),'SUSI') ge 0 then return,'SUSI'
if strpos(strupcase(sysid),'ISI') ge 0 then return,'ISI'
;
; No match found
return,sysid
;
; No match found; if input was undefined, this function is used to set it
if n_elements(SystemId) ne 0 then return,SystemId
;
end
;-------------------------------------------------------------------------------
function instrument_id,sysid,beamcombinerid=beamcombinerid
;
; Return the instrument ID for the given system and beam combiner ID,
; e.g. VLTI and beamcombinerid=1, or sysid=VLTI/MIDI-N-PRISM.
; If beamcombinerid is not specified, read from genconfig.
; System refers to the interferometer (e.g. NPOI), instrument to the combiner.
; For aperture masks, system should be the instrument name (e.g. NACO),
; and instrument refers to the specific mask (e.g. 7Holes)
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Mark3,mark3_baselines,mark3_configs,mark3_cat,mark3_bsc
;
if n_elements(sysid) eq 0 then sysid=''
if n_elements(beamcombinerid) eq 0 then beamcombinerid=0
;
; This is the easy case, sysid must be composed of array/instrument
i=strpos(sysid,'/')
if i ge 0 then return,strmid(sysid,i+1,strlen(sysid)-(i+1))
;
; Otherwise, we have to look up the beam combiner ID
if n_elements(size(genconfig)) eq 3 then begin
	if beamcombinerid eq 0 then return,''
endif else begin
	if beamcombinerid eq 0 then $
	   beamcombinerid=genconfig.beamcombinerid	
endelse
;
if system_id(sysid) eq 'T-ReCS' then begin
	case beamcombinerid of
	1:return,'7Holes-Ks'
	endcase
endif
if system_id(sysid) eq 'NACO' then begin
	case beamcombinerid of
	1:return,'7Holes'
	endcase
endif
if system_id(sysid) eq 'ISI' then begin
	case beamcombinerid of
	1:return,'4'
	endcase
endif
if system_id(sysid) eq 'Mark3' then begin
	return,mark3_configs(beamcombinerid)
	case beamcombinerid of
	1:return,'450'
	2:return,'500'
	endcase
endif
if system_id(sysid) eq 'SUSI' then begin
	case beamcombinerid of
	1:return,'PAVO'
	endcase
endif
if system_id(sysid) eq 'PTI' then begin
	case beamcombinerid of
	1:return,'K1'
	2:return,'K5'
	endcase
endif
if system_id(sysid) eq 'IOTA' then begin
	case beamcombinerid of
	1:return,'IONIC3'
	endcase
endif
if system_id(sysid) eq 'NPOI' then begin
	case beamcombinerid of
	1:return,'3way'
	2:return,'6way'
	9:return,'nway'
     else:return,'3way'
	endcase
endif
if system_id(sysid) eq 'CHARA' then begin
	case beamcombinerid of
	10:return,'CLASSIC'
	20:return,'CLIMB'
	30:return,'H_PRISM'
	31:return,'MIRC-LR'
	32:return,'MIRC-MR'
	33:return,'MIRC-HR'
	endcase
endif
if system_id(sysid) eq 'VLTI' then begin
	if beamcombinerid eq 0 then return,'UNKNOWN'
	case beamcombinerid of
	 1:return,'VINCI'
	11:return,'VINCI-MONA'
	12:return,'VINCI-IONIC'
	 2:return,'MIDI'
	21:return,'MIDI-N-PRISM'
	22:return,'MIDI-N-GRISM'
	 3:return,'AMBER'
	31:return,'AMBER-LR'
	32:return,'AMBER-MR'
	33:return,'AMBER-HR'
	 4:return,'PIONIER'
	41:return,'PIONIER-GRISM'
	42:return,'PIONIER-FREE'
	 5:return,'GRAVITY'
	51:return,'GRAVITY-LR'
	52:return,'GRAVITY-MR'
	53:return,'GRAVITY-HR'
	 6:return,'MATISSE'
	61:return,'MATISSE-LN'	; generic L+N band files
	62:return,'MATISSE-L-LR'
	63:return,'MATISSE-L-MR'
	64:return,'MATISSE-L-HR'
	65:return,'MATISSE-L-HP'
	66:return,'MATISSE-N-LR'
	67:return,'MATISSE-N-HR'
	endcase
endif
return,''
;
end
;-------------------------------------------------------------------------------
function beamcombiner_id,sysid
;
; Return the beam combiner ID for the given system/instrument, 
; e.g. VLTI/MIDI-N-PRISM. If the instrument is not defined, return 1.
; System refers to the interferometer (e.g. NPOI), instrument to the combiner.
; For aperture masks, system should be the instrument name (e.g. NACO),
; and instrument refers to the specific mask (e.g. 7Holes)
;
; This function is the only one which needs to know non-standard instruments,
; as it is called by get_oifits to return a beam combiner ID known to OYSTER.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common Mark3,mark3_baselines,mark3_configs,mark3_cat,mark3_bsc
;
words=nameparse(sysid,'/')
; if n_elements(words) lt 2 then return,1
;
case words(0) of
'T-ReCS':begin
	if strpos(words(1),'7Holes') ge 0 then return,1
	end
'NACO':	begin
	if strpos(words(1),'7Holes') ge 0 then return,1
	end
'JWST_SIM':begin
	if strpos(words(1),'7Holes') ge 0 then return,1
	end
'SUSI': begin
	return,1
	end
'ISI': 	begin
	if strpos(words(1),'4') ge 0 then return,1
	end
'Mark3':begin
	index=where(mark3_configs eq words(1))
	return,index(0)+1
	end
'PTI':	begin
	if strpos(words(1),'K1') ge 0 then return,1
	if strpos(words(1),'K5') ge 0 then return,2
	if strpos(words(1),'Synthetic_WL') ge 0 then return,1
	end
'IOTA':	begin
	if strpos(words(1),'IONIC3') ge 0 then return,1
	if strpos(words(1),'IONIC') ge 0 then return,2
	end
'COAST':begin
	return,1
	end
'NPOI':	begin
	if strpos(words(1),'3way') ge 0 then return,1
	if strpos(words(1),'6way') ge 0 then return,2
	if strpos(words(1),'nway') ge 0 then return,9
	return,9
	end
'CHARA':begin
	if strpos(words(1),'CLASSIC') ge 0 then return,10
	if strpos(words(1),'CLIMB') ge 0 then return,20
	if strpos(words(1),'H_PRISM') ge 0 then return,30
	if strpos(words(1),'MIRC-LR') ge 0 then return,31
	if strpos(words(1),'MIRC-MR') ge 0 then return,32
	if strpos(words(1),'MIRC-HR') ge 0 then return,33
	if strpos(words(1),'MIRC') ge 0 then return,30
	end
'Keck':	begin
	if strpos(words(1),'K1') ge 0 then return,1
	if strpos(words(1),'KIN') ge 0 then return,2
	end
'VLTI':	begin
	if strpos(words(1),'VINCI-MONA') ge 0 then return,11
	if strpos(words(1),'VINCI-IONIC') ge 0 then return,12
	if strpos(words(1),'MIDI-N-PRISM') ge 0 then return,21
	if strpos(words(1),'MIDI-N-GRISM') ge 0 then return,22
	if strpos(words(1),'AMBER-LR') ge 0 then return,31
	if strpos(words(1),'AMBER-MR') ge 0 then return,32
	if strpos(words(1),'AMBER-HR') ge 0 then return,33
	if strpos(words(1),'PIONIER-GRIS') ge 0 then return,41
	if strpos(words(1),'PIONIER-FREE') ge 0 then return,42
	if strpos(words(1),'PIONIER_P') ge 0 then return,41
	if strpos(words(1),'GRAVITY-LR') ge 0 then return,51
	if strpos(words(1),'GRAVITY-MR') ge 0 then return,52
	if strpos(words(1),'GRAVITY-HR') ge 0 then return,53
	if strpos(words(1),'MATISSE-LN') ge 0 then return,61
	if strpos(words(1),'MATISSE-L-LR') ge 0 then return,62
	if strpos(words(1),'MATISSE-L-MR') ge 0 then return,63
	if strpos(words(1),'MATISSE-L-HR') ge 0 then return,64
	if strpos(words(1),'MATISSE-L-HP') ge 0 then return,65
	if strpos(words(1),'MATISSE-N-LR') ge 0 then return,66
	if strpos(words(1),'MATISSE-N-HR') ge 0 then return,67
;	Deduce config ID from spectrometer layout
	if strpos(words(1),'AMBER') ge 0 then begin
		if max(genconfig.numspecchan) lt 50 then return,31
		if max(genconfig.numspecchan) lt 5000 then return,32
		if max(genconfig.numspecchan) lt 50000 then return,33
	endif
	if strpos(words(1),'PIONIER') ge 0 then begin
		if genconfig.numspecchan(0) gt 1 then return,41
		if genconfig.numspecchan(0) eq 1 then return,42
	endif
	if strpos(words(1),'GRAVITY') ge 0 then begin
		if max(genconfig.numspecchan) le 40 then return,51
		if max(genconfig.numspecchan) le 400 then return,52
		if max(genconfig.numspecchan) le 4000 then return,53
	endif
	if strpos(words(1),'MATISSE') ge 0 then begin
		if genconfig.numoutbeam eq 2 then return,61
		nc=genconfig.numspecchan & nc=nc(0)
		dl=genconfig.wavelength(1:nc-1) $
		  -genconfig.wavelength(0:nc-2)
	  	med_dl=abs(median(dl)*1e6)
		if median(genconfig.wavelength)*1e6 lt 7 then begin
;			L-band
			if med_dl lt 0.0001 then return,65
			if med_dl lt 0.001 then return,64
			if med_dl lt 0.01 then return,63
			return,62
		endif else begin
;			N-band
			if max(genconfig.numspecchan) le 150 then return,66
			if max(genconfig.numspecchan) gt 150 then return,67
		endelse
	endif
	return,1
	end
endcase
;
end
;-------------------------------------------------------------------------------
function unique_station_config,baselineids_in
;
; Based on an empirical algorithm, compute a shorter station ID string which is
; (hopefully) unique given any longer input string of baseline IDs.
;
; Example: 18 Baselines, 9 Stations (Imaging Beauty Contest 2020)
; AA0 AB2
; AA0     AD0
; AA0         AC1
;     AB2 AD0
;     AB2     AC1
;         AD0 AC1
; AA0             AG1
; AA0                 AJ2
; AA0                     AJ3
;                 AG1 AJ2
;                 AG1     AJ3
;                     AJ2 AJ3
;                             AK0 AG2
;        -AD0                 AK0     AD0
;                        -AJ3 AK0         AJ3
;        -AD0                     AG2 AD0
;                        -AJ3     AG2     AJ3
;        -AD0             AJ3         AD0
;  0   1   2   3   4   5   6   7   8   9   10
; ==>
; Unique station ID string?:
; AA0 AB2 AD0 AC1 AG1 AJ2 AJ3 AK0 AG2 AD0 AJ3
;
; UT3 UT4
;         UT1 UT2
;-UT3         UT2 UT3
;    -UT4     UT2     UT4
;-UT3     UT1     UT3 
;    -UT4 UT1         UT4
;
; Another example:
; actual:  AC1-AD0 AB2-AA0 AA0-AC1 AA0-AD0 AB2-AC1 AB2-AD0
; unique:  AA0-AC1 AA0-AD0 AB2-AA0 AB2-AC1 AB2-AD0 AC1-AD0
;MATISSE_L_AA0-AC1-AD0-AB2+AC1+AD0
; actual:  AD0-AC1 AB2-AA0 AA0-AD0 AA0-AC1 AB2-AD0 AB2-AC1
; unique:  AA0-AC1 AA0-AD0 AB2-AA0 AB2-AC1 AB2-AD0 AD0-AC1
;MATISSE_L_AA0-AC1-AD0-AB2+AC1+AD0
; actual:  AC1-AD0 AA0-AB2 AB2-AC1 AB2-AD0 AA0-AC1 AA0-AD0
;MATISSE_L_AA0-AB2-AC1-AD0
; actual:  AD0-AC1 AA0-AB2 AB2-AD0 AB2-AC1 AA0-AD0 AA0-AC1
;MATISSE_L_AA0-AB2-AC1-AD0+AC1

;
; If we re-order, the result will be more compact
baselineids=unique(baselineids_in)
;
for i=0,n_elements(baselineids)-1 do begin
	sij=nameparse(baselineids(i),['-','_'])
	if i eq 0 then begin
		stationids=sij 
		signs='-'
	endif else begin
		j=where(stationids eq sij(0),count)
		if count eq 0 then begin	; station not in list yet
			stationids=[stationids,sij(0)]
			signs=[signs,'-']
		endif else j=j(0)
		k=where(stationids eq sij(1),count)
		if count eq 0 then begin	; station not in list yet
			stationids=[stationids,sij(1)]
			signs=[signs,'-']
		endif
;		n=n_elements(k)
;		if k(0) ge 0 and j(0) ge 0 and k(n-1) lt j(0) then begin
;			stationids=[stationids,sij(1)]
;			signs=[signs,'+']
;		endif
;		To better handle "Another example" (see above)
		j=where(stationids eq sij(0))
		k=where(stationids eq sij(1))
		if k(0) lt j(0) then begin
			sij=nameparse(breve(baselineids(i)),['-','_'])
			stationids=[stationids,sij(1)]
			signs=[signs,'+']
		endif
	endelse
endfor
;
; return,strjoin(stationids(0:n-2)+signs)+stationids(n-1)
;
; Reduce to minimum unique
n=n_elements(stationids)
usc='-'+strjoin(stationids(0:n-2)+'|'+signs)+stationids(n-1)
s=strsplit(usc,'|',/extract)
s=strjoin(s(uniq(s))) 
return,strmid(s,1,strlen(s)-1)
;
end
;-------------------------------------------------------------------------------
function strict_configid,configid_in
;
; Return a standard config id, including the baseline layout instead of just
; the station layout. GenConfig corresponding to loaded data must be defined.
;
; The configid should be composed of:
; array name, instrument, mode, baseline layout
; Examples:
; NPOI, 6way, 
; NPOI, CLASSIC or VISION, Full or Halpha,  
; VLTI, MATISSE, L-HR, UT3-UT4_UT1-UT2_UT2-UT3_UT2-UT4_UT1-UT3_UT1-UT4
; VLTI, GRAVITY, FT/SC, UT3-UT4_UT1-UT2_UT2-UT3_UT2-UT4_UT1-UT3_UT1-UT4
;
; Input examples:
; MATISSE_L_UT1-UT2-UT3-UT4, VLTI/MATISSE-N-LR, MATISSE-N-LR
; GRAVITY_FT_UT1-UT2-UT3-UT4
; GRAVITY_SC_UT1-UT2-UT3-UT4, VLTI, GRAVITY-LR
; PIONIER_GRI6_AD0-AG2-AJ3-AK0
; N00-N05-N10-W06-W08-W10 (NPOI), NPOI/3way
; E03-AC0-AE0-AW0, NPOI/6way
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
configid=configid_in
;
instrumentid=instrument_id(systemid)
words=nameparse(configid,'_')
if n_elements(words) eq 2 and words(0) eq instrument_id(systemid) then $
	instrumentid=configid
;
if !owner eq 'chummel' then begin
	print,'---------------------------------------------------------------'
	print,'This is function strict_configid. Input is: '+configid
	print,'This is function strict_configid. SystemID is: '+systemid
	print,'This is function strict_configid. Instrument is: ' $
						+instrument_id(systemid)
	baselineids=''
	for i=0,genconfig.numoutbeam-1 do begin
	for j=0,genconfig.numbaseline(i)-1 do $
		baselineids=[baselineids,genconfig.baselineid(j,i)]
	endfor
	baselineids=baselineids(1:n_elements(baselineids)-2)
	syswords=nameparse(systemid,'/')
	configid=syswords(0)+'_'+instrumentid+'_' $
		+unique_station_config(baselineids)
	print,'Suggested new configid:'
	print,configid
	print,'---------------------------------------------------------------'
endif
;
; New config ID not yet verified...
return,configid_in
;
end
;-------------------------------------------------------------------------------
function system_config,systemid,key,stations=stations
;
; This function returns general site and system information for the given key.
; System configuration wavelengths are in meters, but grid wavelengths in nm.
;
; key='MIDNIGHT' calls this function 2nd time
;
forward_function system_config
;
common Mark3,mark3_baselines,mark3_configs,mark3_cat,mark3_bsc
common Pionier,pionier_spectral_cal
;
if n_elements(key) eq 0 then begin
	print,'Error(system_config): must also specify key!'
	return,''
endif
key=strupcase(key)
;
; This key creates a new GenConfig (not all systems implemented!)
if key eq 'CONFIG' then begin
	get_systemid,systemid	; store systemid in common block
	stations=strcompress(stations,/remove)
	sid_index=where(strlen(stations) gt 0,NumSid)
endif
;
case system_id(systemid) of
;
; Observatories
'OHP':begin	; Observatoire Haute Provence
	case key of
	'LONGITUDE': 	return,+5.7122D0
	'LATITUDE':  	return,43.9317D0
	'ALTITUDE':  	return,650.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'Ondrejov':begin	; Ondrejov Observatory
	case key of
	'LONGITUDE': 	return,-70.4032D0
	'LATITUDE':  	return,-24.6258D0
	'ALTITUDE':  	return,2635.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'CalarAlto':begin
	case key of
	'LONGITUDE': 	return,-2.546822D0
	'LATITUDE':  	return,37.220888D0
	'ALTITUDE':  	return,2150.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'LaSilla':begin
	case key of
	'LONGITUDE': 	return,-70.051187D0
	'LATITUDE':  	return,-29.255122D0
	'ALTITUDE':  	return,2375.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'Paranal':begin
	case key of
	'LONGITUDE': 	return,-70.4032D0
	'LATITUDE':  	return,-24.6258D0
	'ALTITUDE':  	return,2635.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
'OCA':begin	; Observatorio Cerro Armazones
	case key of
	'LONGITUDE': 	return,-70.196389D0
	'LATITUDE':  	return,24.598056D0
	'ALTITUDE':  	return,2817.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	endcase
	end
;
; Aperture masks
'JWST_SIM':begin	; The JWST aperture mask
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['P01','P02','P03','P04','P05','P06','P07']
	'DIAMETER':	return,9.468544e-1 ; Beauty Contest 2022, JSB
	'LONGITUDE': 	return,-70.7367D0
	'LATITUDE':  	return,-30.241D0
	'ALTITUDE':  	return,2722.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,70
	'ZALIMITMIN': 	return,0
	'GRID':		return,findgen(132)*40+11700-5240.0/2
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	endcase
	end
'T-ReCS':begin	; The TrecS aperture mask
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['A01','A02','A03','A04','A05','A06','A07']
	'DIAMETER':	return,1.5 ; John Monnier, email from 20.3.2005
	'LONGITUDE': 	return,-70.7367D0
	'LATITUDE':  	return,-30.241D0
	'ALTITUDE':  	return,2722.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,70
	'ZALIMITMIN': 	return,0
	'GRID':		return,findgen(132)*40+11700-5240.0/2
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	endcase
	end
'NACO':	begin	; The 7Holes SAM aperture mask, Ks and Lp bands
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['A01','A02','A03','A04','A05','A06','A07']
	'DIAMETER':	return,1.20	; 1.5 mm holes, pupil diameter 10 mm
	'LONGITUDE': 	return,-70.4032D0
	'LATITUDE':  	return,-24.6258D0
	'ALTITUDE':  	return,2635.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,70
	'ZALIMITMIN': 	return,0
	'GRID':		return,findgen(50)*80+1000
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	endcase
	end
;
; Long-baseline interferometers
'Mark3':begin
	case key of
	'EXIST': 	return,1
	'STATIONS': 	return,['NAS','SAS']
	'DIAMETER':	return,0.05
	'LONGITUDE': 	return,-118.05917D0
	'LATITUDE':  	return,34.216694D0
	'ALTITUDE':  	return,1742.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,60
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,35
	'GRID': 	return,findgen(250)*2+401
	'MAXCONFIG':	return,1
	'FILTERS':begin
		index=where(mark3_configs eq instrument_id(systemid))
		case index(0)+1 of
		1:return,[['generic_c','mark3_486'], $
			  ['mark3_486','mark3_486']]
		2:return,[['generic_c','mark3_550'], $
			  ['mark3_656','mark3_550']]
		3:return,[['generic_c','mark3_656'], $
			  ['mark3_657','mark3_550']]
		4:return,[['generic_c','mark3_450'], $
			  ['mark3_700','mark3_550']]
		5:return,[['generic_c','mark3_623'], $
			  ['mark3_754','mark3_712']]
		6:return,[['generic_c','mark3_450'], $
			  ['mark3_799','mark3_550']]
		7:return,[['generic_c','mark3_450'], $
			  ['mark3_800','mark3_500']]
		8:return,[['generic_c','mark3_450'], $
			  ['mark3_800','mark3_550']]
		9:return,[['generic_c','mark3_452'], $
			  ['mark3_800','mark3_500']]
	       10:return,[['generic_c','mark3_500'], $
			  ['mark3_800','mark3_550']]
	       11:return,[['generic_c','mark3_500'], $
			  ['mark3_800','mark3_656']]
	       12:return,[['generic_c','mark3_500'], $
			  ['mark3_800','mark3_550']]
	       13:return,[['generic_c','mark3_656'], $
			  ['mark3_800','mark3_550']]
		endcase
		return,'generic_c'
		case instrument_id(systemid) of
		'450':return,[['generic_c','mark3_450'], $
			      ['mark3_800','mark3_550']]
		'500':return,[['generic_c','mark3_500'], $
			      ['mark3_800','mark3_550']]
		endcase
		end
	endcase
	end
'PTI': 	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['NAS','SAS']
	'DIAMETER':	return,0.40
	'LONGITUDE': 	return,-116.8633D0
	'LATITUDE':  	return,33.3567D0
	'ALTITUDE':  	return,1687.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,60
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,100
	'GRID':		return,findgen(250)*2+2000	; [nm]
	'MAXCONFIG':	return,1
	'FILTERS':	begin
			case instrument_id(systemid) of
			'K1':return,'pti_k'
			'K5':return,['pti_k1','pti_k2', $
				     'pti_k3','pti_k4', $
				     'pti_k5']
			endcase
			end
	'CONFIG':	begin
		maxNumSid=2
		maxNumOutBeam=3
		maxNumTriple=1
		maxBaseline=1
		maxSpecChan=5
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
;
; 		Assume pair-wise beam combination
		case instrument_id(systemid) of
			'K1' :GenConfig.BeamCombinerId=1
			'K5' :GenConfig.BeamCombinerId=2
		endcase
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
		GenConfig.DelayLineId=[1,2]
		GenConfig.BCInputId=[1,2]
		GenConfig.BaselineId(0,0)=genconfig.stationid(0)+'-' $
					 +genconfig.stationid(1)
		GenConfig.NumBaseline(0)=1
;
; 		Remove the instrument string for backward compatibility
		SystemId='PTI'
;
		case GenConfig.BeamCombinerId of
;	  		Single channel K filter
		      1:begin
			GenConfig.NumSpecChan=1
			GenConfig.Wavelength=2.2e-6
			GenConfig.WavelengthErr=1e-9
			GenConfig.ChanWidth=0.03*GenConfig.Wavelength
			GenConfig.ChanWidthErr=1e-9
			end
		      2:begin
			GenConfig.NumSpecChan=5
			GenConfig.Wavelength=[2.009,2.106,2.203,2.299,2.396]*1e-6
			GenConfig.WavelengthErr=1e-9
			GenConfig.ChanWidth=0.097e-6
			GenConfig.ChanWidthErr=1e-9
			end
		endcase
;
; 		Set number of independent triples
		GenConfig.NumTriple=0
;
		return,GenConfig
		end
	endcase
	end
'IOTA':	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['A00','B00','C00']
	'DIAMETER':	return,0.45
	'LONGITUDE': 	return,-110.8849D0
	'LATITUDE':  	return,31.6916D0
	'ALTITUDE':  	return,2564.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,65
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,90
	'GRID':		return,findgen(150)*2+1500; [nm], H-band, 1.51-1.79 mu
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	'CONFIG':begin
		BeamCombinerId=1
;
		maxNumSid=3
		maxNumOutBeam=1
		maxNumTriple=1
		maxBaseline=3
		maxSpecChan=1
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
		case instrument_id(systemid) of
			'IONIC3' :GenConfig.BeamCombinerId=1
			'IONIC'  :GenConfig.BeamCombinerId=2
		endcase
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
;
		if GenConfig.NumSid ne 3 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 3!'
			return,-1
		endif
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.DelayLineId=[1,2,3]
		GenConfig.BCInputId=[1,2,3]
		GenConfig.BaselineId(0,0)=genconfig.stationid(0)+'-' $
				         +genconfig.stationid(1)
		GenConfig.BaselineId(1,0)=genconfig.stationid(1)+'-' $
				         +genconfig.stationid(2)
		GenConfig.BaselineId(2,0)=genconfig.stationid(2)+'-' $
				         +genconfig.stationid(0)
		GenConfig.NumBaseline=3
;
		NumChan=1; must be less than or equal to maxSpecChannel
		for ob=0,GenConfig.NumOutBeam-1 do begin
			GenConfig.NumSpecChan(ob)=NumChan
			GenConfig.Wavelength(*,ob)=1.65e-6
			GenConfig.WavelengthErr(*,ob)=1e-9
			GenConfig.ChanWidth(*,ob)=0.03*GenConfig.Wavelength(*,ob)
			GenConfig.ChanWidthErr(*,ob)=1e-9
		endfor

;		Triple
		GenConfig.NumTriple=1
		GenConfig.TripleBeam=[0,1,2]
		GenConfig.TripleBase=[0,0,0]
		GenConfig.TripleNumChan(0)=NumChan
		for bl=0,2 do begin
		GenConfig.TripleChan(0:NumChan-1,bl,0)=indgen(NumChan)
		endfor
;
; 		Remove the instrument string for backward compatibility
		SystemId='IOTA'
;
		return,GenConfig
		end
	endcase
	end
'COAST':begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['CC0','CW2','CE1','CN4']
	'DIAMETER':	return,0.40
	'LONGITUDE': 	return,-(00.D0+02.D0/60+31.96D0/3600)
	'LATITUDE':  	return,  52.D0+09.D0/60+49.96D0/3600
	'ALTITUDE':  	return,17.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,60
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,115
	'GRID':		return,findgen(400)+630	; [nm]
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	'CONFIG':begin
; 	Pair-wise combination of 4 stations for simulation purposes
;
		NumOutBeam=(NumSid*(NumSid-1))/2
		NumTriple=(NumSid-1)*(NumSid-2)/2
		NumBaseline=1
		NumSpecChan=71
		maxSpecChan=NumSpecChan
		maxConfig=1
		GenConfig=allocGenConfig(NumSid,NumOutBeam,NumTriple, $
					 NumBaseline,maxSpecChan,maxConfig)
;
		GenConfig.BeamCombinerId=1
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
;
		GenConfig.NumSid=NumSid
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.NumSpecChan=NumSpecChan
		GenConfig.TripleNumChan=NumSpecChan
		for i=0,NumOutBeam-1 do $
		GenConfig.Wavelength(*,i)=(findgen(NumSpecChan)*5+650)*1e-9
		GenConfig.WavelengthErr=1e-9
		GenConfig.ChanWidth=5e-9
		GenConfig.ChanWidthErr=1e-9
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		ob=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin

		GenConfig.BaselineId(0,ob)=genconfig.stationid(i)+'-' $
					   +genconfig.stationid(j)
		ob=ob+1
		endfor
		endfor
;
		tr=0
		for i=1,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		ob1=where(GenConfig.BaselineId(0,*) eq genconfig.stationid(0)+'-' $
						      +genconfig.stationid(i))
		ob2=where(GenConfig.BaselineId(0,*) eq genconfig.stationid(i)+'-' $
						      +genconfig.stationid(j))
		ob3=where(GenConfig.BaselineId(0,*) eq genconfig.stationid(0)+'-' $
						      +genconfig.stationid(j))
		GenConfig.TripleBeam(*,tr)=[ob1,ob2,ob3]
		for l=0,2 do begin
			GenConfig.TripleChan(*,l,tr)=indgen(NumSpecChan)
			GenConfig.TripleChan(*,l,tr)=indgen(NumSpecChan)
		endfor
		tr=tr+1
		endfor
		endfor
;
		return,GenConfig
		end
	endcase
	end
'NPOI':	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['AC0','AE0','AW0','AN0']
	'DIAMETER':	return,0.12
	'LONGITUDE': 	return,-(111.D0+32.D0/60+ 6.D0/3600)
	'LATITUDE':  	return,   35.D0+ 5.D0/60+48.D0/3600
	'ALTITUDE':  	return,2200.66D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,60
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,35
	'GRID': begin
		case instrument_id(systemid) of
		'3way':return,findgen(250)*2+401
		'6way':return,findgen(250)*2+401
		'nway':return,findgen(250)*2+401
		'else':return,findgen(250)*2+401
		endcase
		end
	'FILTERS':begin
		return,'generic_c'
		end
	'MAXCONFIG':begin
		case instrument_id(systemid) of
		'3way':return,'stars'
		'nway':return,'stars'
		  else:return,'scans'
		endcase
		end
	'CONFIG':begin
;
; 		Set beam combiner ID, which is identical to instrument ID
		case instrument_id(systemid) of
			'3way' :BeamCombinerId=1
			'6way' :BeamCombinerId=2
			'nway' :BeamCombinerId=9
			else:   BeamCombinerId=1
		endcase
;
; 		Remove the instrument string for backward compatibility
		SystemId='NPOI'
;
		case BeamCombinerId of
;
;		3way
	      1:begin
		maxNumSid=3
		maxNumOutBeam=3
		maxNumTriple=(maxNumSid-1)*(maxNumSid-2)/2
		maxBaseline=1
		maxSpecChan=32
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
		GenConfig.BeamCombinerId=BeamCombinerId
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
;
		if GenConfig.NumSid ne 3 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 3!'
			return,-1
		endif
;
		NumOutBeam=3
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.DelayLineId=[1,2,3]
		GenConfig.BCInputId=[1,2,3]
		GenConfig.BaselineId(0,0)=genconfig.stationid(0)+'-' $
				         +genconfig.stationid(1)
		GenConfig.BaselineId(0,1)=genconfig.stationid(1)+'-' $
				         +genconfig.stationid(2)
		GenConfig.BaselineId(0,2)=genconfig.stationid(2)+'-' $
				         +genconfig.stationid(0)
		GenConfig.NumBaseline=1
;
		NumChan=32; must be less than or equal to maxSpecChannel
		for ob=0,GenConfig.NumOutBeam-1 do begin
			GenConfig.NumSpecChan(ob)=NumChan
			GenConfig.Wavelength(*,ob)= $
			1e-9/((1./850)+findgen(GenConfig.NumSpecChan(ob))*4e-5)
			GenConfig.WavelengthErr(*,ob)=1e-9
			GenConfig.ChanWidth(*,ob)=0.03*GenConfig.Wavelength(*,ob)
			GenConfig.ChanWidthErr(*,ob)=1e-9
		endfor

;		Triple
		GenConfig.NumTriple=1
		GenConfig.TripleBeam=[0,1,2]
		GenConfig.TripleBase=[0,0,0]
		GenConfig.TripleNumChan(0)=NumChan
		for bl=0,2 do begin
		GenConfig.TripleChan(0:NumChan-1,bl,0)=indgen(NumChan)
		endfor
;
		return,GenConfig
		end
;
;		6way
	      2:begin
;
		maxNumSid=6
		maxNumOutBeam=3
		maxNumTriple=(maxNumSid-1)*(maxNumSid-2)/2
		maxBaseline=6	; 4 stations per output beam
		maxSpecChan=16
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
;
		GenConfig.BeamCombinerId=BeamCombinerId
		GenConfig.NumBin=64
		strokes=1e-6*[-1,4,-3,3,-4,2]
; 		3-way combination sequence for spectrometers 1-3
		outputs=[[2,3],[1,3],[1,2]]
; 		The 6-way combiner is two 3-way combiners
		stations6=strarr(6)
		stations6(0:n_elements(stations)-1)=stations
		doublet=reform(stations6,3,2)
; 		Each spectrometer has these inputs
		quad=strarr(4,3)
		for i=0,2 do quad(*,i)=[doublet(outputs(*,i)-1,0), $
					doublet(outputs(*,i)-1,1)]
;
		for ob=0,3-1 do begin
		bl=0
		for s1=0,4-2 do begin
		for s2=s1+1,4-1 do begin
		if strlen(quad(s1,ob)) ne 0 and strlen(quad(s2,ob)) ne 0 $
			then begin
			m=where(stations eq quad(s1,ob)) & m=m(0)
			n=where(stations eq quad(s2,ob)) & n=n(0)
			k=nint((strokes(m)-strokes(n))/1e-6)
			if k gt 0 then $
			GenConfig.BaselineId(bl,ob)=quad(s1,ob)+'-'+quad(s2,ob)
			if k lt 0 then $
			GenConfig.BaselineId(bl,ob)=quad(s2,ob)+'-'+quad(s1,ob)
			GenConfig.FringeMod(bl,ob)=abs(k)
			bl=bl+1
		endif
		endfor
		endfor
		GenConfig.NumBaseline(ob)=bl
		endfor
;
		GenConfig.NumOutBeam=3
		GenConfig.NumSid=NumSid
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.Stroke(0:NumSid-1)=strokes(sid_index)
		GenConfig.RefStation=1
		GenConfig.DelayLineId(0:NumSid-1)=sid_index+1
		GenConfig.BCInputId(0:NumSid-1)=sid_index+1
;
		for ob=0,GenConfig.NumOutBeam-1 do begin
			GenConfig.NumSpecChan(ob)=maxSpecChan
			GenConfig.Wavelength(*,ob)= $
			1e-9/((1./850)+findgen(GenConfig.NumSpecChan(ob))*4e-5)
			GenConfig.WavelengthErr(*,ob)=1e-9
			GenConfig.ChanWidth(*,ob)=0.03*GenConfig.Wavelength(*,ob)
			GenConfig.ChanWidthErr(*,ob)=1e-9
		endfor
;
; 		Get space for (n-1)(n-2)/2 independent triples
		GenConfig.NumTriple=(NumSid-1)*(NumSid-2)/2
		TripleBeam=lonarr(3,maxNumTriple)
		TripleBase=lonarr(3,maxNumTriple)
		TripleChan=lonarr(maxSpecChan,3,maxNumTriple)
		TripleNumChan=lonarr(maxNumTriple)
		tr=0
		s1=0
		FOR s2=1,NumSid-2 DO BEGIN
		FOR s3=s2+1,NumSid-1 DO BEGIN
;	 	This is the first baseline in the triple
		for ob=0,GenConfig.NumOutBeam-1 do begin
		for bl=0,GenConfig.NumBaseline(ob)-1 do begin
			if strpos(GenConfig.BaselineId(bl,ob), $
				GenConfig.StationId(s1)) ne -1 $
		        and strpos(GenConfig.BaselineId(bl,ob), $
		       		GenConfig.StationId(s2)) ne -1 $
			then begin
				TripleBeam(0,tr)=ob
				TripleBase(0,tr)=bl
				TripleChan(*,0,tr)=indgen(maxSpecChan)
			endif
		endfor
		endfor
;	 	This is the second baseline in the triple
		for ob=0,GenConfig.NumOutBeam-1 do begin
		for bl=0,GenConfig.NumBaseline(ob)-1 do begin
			if strpos(GenConfig.BaselineId(bl,ob), $
				GenConfig.StationId(s1)) ne -1 $
		        and strpos(GenConfig.BaselineId(bl,ob), $
				GenConfig.StationId(s3)) ne -1 $
		      	then begin
				TripleBeam(1,tr)=ob
				TripleBase(1,tr)=bl
				TripleChan(*,1,tr)=indgen(maxSpecChan)
			endif
		endfor
		endfor
;	 	This is the third baseline in the triple
		for ob=0,GenConfig.NumOutBeam-1 do begin
		for bl=0,GenConfig.NumBaseline(ob)-1 do begin
			if strpos(GenConfig.BaselineId(bl,ob), $
				GenConfig.StationId(s2)) ne -1 $
		       and strpos(GenConfig.BaselineId(bl,ob), $
		       		GenConfig.StationId(s3)) ne -1 $
		      then begin
				TripleBeam(2,tr)=ob
				TripleBase(2,tr)=bl
				TripleChan(*,2,tr)=indgen(maxSpecChan)
			endif
		endfor
		endfor
		TripleNumChan(tr)=maxSpecChan
		tr=tr+1
		ENDFOR
		ENDFOR
;
		GenConfig.TripleBeam=TripleBeam
		GenConfig.TripleBase=Triplebase
		GenConfig.TripleChan=TripleChan
		GenConfig.TripleNumChan=TripleNumChan
;
		return,GenConfig
		end
;
; 		Pair-wise combination of 6 stations for simulation purposes
	     9:	begin
;
		NumOutBeam=(NumSid*(NumSid-1))/2
		NumTriple=(NumSid-1)*(NumSid-2)/2
		NumBaseline=1
		NumSpecChan=1
		maxSpecChan=2
		maxConfig=1
		GenConfig=allocGenConfig(NumSid,NumOutBeam,NumTriple, $
					 NumBaseline,maxSpecChan,maxConfig)
;
		GenConfig.BeamCombinerId=9
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
;
		GenConfig.NumSid=NumSid
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.NumSpecChan=NumSpecChan
		GenConfig.TripleNumChan=NumSpecChan
		GenConfig.Wavelength=550e-9
		GenConfig.WavelengthErr=1e-9
		GenConfig.ChanWidth=20e-9
		GenConfig.ChanWidthErr=1e-9
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		ob=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin

		GenConfig.BaselineId(0,ob)=genconfig.stationid(i)+'-' $
					   +genconfig.stationid(j)
		ob=ob+1
		if i eq 0 then begin
		endif
		endfor
		endfor
;
		tr=0
		for i=1,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		ob1=where(GenConfig.BaselineId(0,*) eq $
			genconfig.stationid(0)+'-'+genconfig.stationid(i))
		ob2=where(GenConfig.BaselineId(0,*) eq $
			genconfig.stationid(i)+'-'+genconfig.stationid(j))
		ob3=where(GenConfig.BaselineId(0,*) eq $
			genconfig.stationid(0)+'-'+genconfig.stationid(j))
		GenConfig.TripleBeam(*,tr)=[ob1,ob2,ob3]
		tr=tr+1
		endfor
		endfor
;
		return,GenConfig
		end
;
		endcase
		end
	endcase
	end
'CHARA':begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['S1_','S2_','E1_','E2_','W1_','W2_']
	'DIAMETER':	return,1.0
	'LONGITUDE': 	return,-118.0570D0
	'LATITUDE':  	return,34.2244D0
	'ALTITUDE':  	return,1725.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,80
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,400
;	'GRID':		return,findgen(400)*4+1000
	'GRID':		return,'SPECTROMETER'
	'MAXCONFIG':	return,1
	'FILTERS':	begin
			return,'generic_c'	; we don't have filter curves yet
			case instrument_id(systemid) of
			'CLASSIC':return,'kprime'
			'CLIMB'  :return,'kprime'
			     else:return,'generic_c'
			endcase
			end
	'CONFIG':	begin
;
		maxNumSid=6
		maxNumOutBeam=2
		maxNumTriple=(maxNumSid-1)*(maxNumSid-2)	; H+K combined
		maxBaseline=(maxNumSid*(maxNumSid-1))/2
		maxSpecChan=500
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
		GenConfig.BeamCombinerId=beamcombiner_id(systemid)
;
		case GenConfig.BeamCombinerId/10 of
;
;		CLASSIC
		1:begin
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
;
		GenConfig.NumSpecChan(0)=1
		case instrument_id(systemid) of
			'CLASSIC':GenConfig.Wavelength=2.13e-6
			     else:GenConfig.Wavelength=2.13e-6
		endcase
		GenConfig.WavelengthErr=1e-9
		GenConfig.ChanWidth=0.35e-6
		GenConfig.ChanWidthErr=1e-9
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.NumBaseline=(NumSid*(NumSid-1))/2
		GenConfig.RefStation=1
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin

		GenConfig.BaselineId(bl,*)=genconfig.stationid(i)+'-' $
					  +genconfig.stationid(j)
		bl=bl+1
		endfor
		endfor
;
; 		Set number of independent triples
		GenConfig.NumTriple=0
;
		end
;
;		CLIMB
	     	2:begin
;
		NumOutBeam=2
		GenConfig.NumOutBeam=NumOutBeam
;
		NumChan=1
		GenConfig.NumSpecChan=NumChan
		GenConfig.Wavelength(0,*)=[1.65,2.13]*1e-6
		GenConfig.WavelengthErr(0,*)=1e-9
		GenConfig.ChanWidth(0,*)=0.03*GenConfig.Wavelength(0,*)
		GenConfig.ChanWidthErr(0,*)=1e-9

		GenConfig.NumSid=NumSid
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.NumBaseline=3
		GenConfig.RefStation=1
		GenConfig.DelayLineId=[1,2,3]
		GenConfig.BCInputId=[1,2,3]
		GenConfig.BaselineId(0,*)=genconfig.stationid(0)+'-' $
				         +genconfig.stationid(1)
		GenConfig.BaselineId(1,*)=genconfig.stationid(1)+'-' $
				         +genconfig.stationid(2)
		GenConfig.BaselineId(2,*)=genconfig.stationid(2)+'-' $
				         +genconfig.stationid(0)

;		Triple
		GenConfig.NumTriple=2
		GenConfig.TripleBeam=[[0,0,0],[1,1,1]]
		GenConfig.TripleBase=[[0,1,2],[0,1,2]]
		GenConfig.TripleNumChan(*)=NumChan
		for bl=0,2 do begin
		GenConfig.TripleChan(0:NumChan-1,bl,0)=indgen(NumChan)
		GenConfig.TripleChan(0:NumChan-1,bl,1)=indgen(NumChan)
		endfor
;
		end
;
;		MIRC
		3:begin
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
;
		lambda=[1.79+1.51,2.38+1.97]/2	; Center wavelengths
		widths=[1.79-1.51,2.38-1.97]	; H and K bandwidths
		lambda=lambda[0]		; MIRC is only H-band
		widths=widths[0]
		case GenConfig.BeamCombinerId mod 10 of
			1: NumChan=widths/lambda*35	; R=35
			2: NumChan=widths/lambda*150	; R=150
			3: NumChan=widths/lambda*450	; R=450
		endcase
		NumChan=fix(NumChan)+1
		GenConfig.NumSpecChan(0)=NumChan
;		H-band, 1.51-1.79 microns
		l=findgen(NumChan(0))/(NumChan(0)-1)
		GenConfig.Wavelength(0:NumChan(0)-1,0)=(l*0.29+1.51)*1e-6
		GenConfig.ChanWidth(0:NumChan(0)-1,0)=0.29e-6/NumChan(0)
;		K-band, 1.97-2.38 microns
;		l=findgen(NumChan(1))/(NumChan(1)-1)
;		GenConfig.Wavelength(0:NumChan(1)-1,1)=(l*0.41+1.97)*1e-6
;		GenConfig.ChanWidth(0:NumChan(1)-1,1)=0.41e-6/NumChan(1)
;
		GenConfig.WavelengthErr=0.1e-6
		GenConfig.ChanWidthErr=1e-9
;
		NumTriple=(NumSid-1)*(NumSid-2)	; H+K combined
		NumBaseline=(NumSid*(NumSid-1))/2
;
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
;
		GenConfig.NumSid=NumSid
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.NumSpecChan=NumChan
		GenConfig.TripleNumChan=NumChan
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin

		GenConfig.BaselineId(bl,*)=genconfig.stationid(i)+'-' $
					  +genconfig.stationid(j)
		bl=bl+1
		endfor
		endfor
;
		tr=0
		for k=0,NumOutBeam-1 do begin
		for i=1,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId(*,k) eq genconfig.stationid(0)+'-' $
						      +genconfig.stationid(i))
		bl2=where(GenConfig.BaselineId(*,k) eq genconfig.stationid(i)+'-' $
						      +genconfig.stationid(j))
		bl3=where(GenConfig.BaselineId(*,k) eq genconfig.stationid(0)+'-' $
						      +genconfig.stationid(j))
		GenConfig.TripleBeam(*,tr)=k
		GenConfig.TripleBase(*,tr)=[bl1,bl2,bl3]
		GenConfig.TripleNumChan(tr)=NumChan(k)
		for l=0,2 do begin
			GenConfig.TripleChan(0:NumChan(k)-1,l,tr)=indgen(NumChan(k))
		endfor
		tr=tr+1
		endfor
		endfor
		endfor
;
		end
	endcase
;
; 	Remove the instrument string for backward compatibility
;	SystemId='CHARA'
	return,GenConfig
	end
;
	endcase
	end
'Keck':	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':	return,['K01','K02']
	'DIAMETER':	return,10.0
	'LONGITUDE': 	return,-155.4733D0
	'LATITUDE':  	return,19.8267D0
	'ALTITUDE':  	return,4123.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,45
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	return,90
	'GRID':		return,findgen(250)*2+2000	; [nm]
	'MAXCONFIG':	return,1
	'FILTERS':	return,'generic_c'
	'CONFIG':begin
		maxNumSid=2
		maxNumOutBeam=3
		maxNumTriple=1
		maxBaseline=1
		maxSpecChan=1
		maxConfig=1
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
;
; 		Assume pair-wise beam combination
		GenConfig.BeamCombinerId=1
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId(0:NumSid-1)=stations
		GenConfig.RefStation=1
		GenConfig.DelayLineId=[1,2]
		GenConfig.BCInputId=[1,2]
		GenConfig.BaselineId(0,0)='K01-K02'
		GenConfig.NumBaseline(0)=1
;
		GenConfig.NumSpecChan=1
		GenConfig.Wavelength=2.2e-6
		GenConfig.WavelengthErr=1e-9
		GenConfig.ChanWidth=0.03*GenConfig.Wavelength
		GenConfig.ChanWidthErr=1e-9
;
; 		Set number of independent triples
		GenConfig.NumTriple=0
;
		return,GenConfig
		end
	endcase
	end
'VLTI':	begin
	case key of
	'EXIST': 	return,1
	'STATIONS':begin
		instrument=nameparse(instrument_id(systemid),'-')
		case instrument(0) of
		'VINCI':  return,['VE0','VG0']
		'IONIC':  return,['VE0','VG0']
		'MIDI' :  return,['UT2','UT3']
		'AMBER':  return,['UT1','UT2','UT3']
		'AMBER':  return,['AA0','AG1','AJ3']
		'PIONIER':return,['UT1','UT2','UT3','UT4']
		'GRAVITY':return,['AA0','AB2','AD0','AC1']
		'GRAVITY':return,['AD0','AG2','AJ3','AK0']
		'MATISSE':return,['UT1','UT2','UT3','UT4']
		   else:  return,['UT1','UT2']
		endcase
		end
	'DIAMETER':	begin
			case strmid(stations(0),0,1) of
			'A' :return,1.8
			'U' :return,8.0
			else:return,1.8
			endcase
			end
	'LONGITUDE': 	return,-70.4032D0
	'LATITUDE':  	return,-24.6258D0
	'ALTITUDE':  	return,2635.D0
	'MIDNIGHT':	return,nint(-system_config(systemid,'LONGITUDE')/15.)
	'ZALIMITMAX': 	return,70
	'ZALIMITMIN': 	return,0
	'FDLLIMIT': 	begin
			if n_elements(stations) ne 0 then begin
			case stations(0) of
;			'AA0':return,65
;			'AG1':return,64
			 else:return,99
			endcase
			endif else return,99	; was 105 m
			end
	'GRID':begin
		case instrument_id(systemid) of
		'VINCI-MONA':	return,findgen(100)*6+1900
		'VINCI-IONIC':	return,findgen(100)*6+1900
		'MIDI-N-PRISM':	return,findgen(264)*40+10340-5240.0
		'MIDI-N-GRISM':	return,'SPECTROMETER'
		'AMBER-LR':	return,findgen(400)*5+1000
		'AMBER-MR':	return,'SPECTROMETER'
		'AMBER-HR':	return,'SPECTROMETER'
		'PIONIER':	return,findgen(200)*2+1500; [nm], H-band
		'PIONIER-FREE':	return,findgen(200)*2+1500; [nm], H-band
		'PIONIER-GRISM':return,findgen(200)*2+1500; [nm], H-band
		'GRAVITY-LR':	return,findgen(100)*5+1950; K-band, R=40
		'GRAVITY-MR':	return,findgen(600)*1+1950; K-band, R=400
		'GRAVITY-HR':	return,'SPECTROMETER'	  ; K-band, R=4000
		'MATISSE-LN':   return,findgen(230)*11+2600 ; LM,R=34
		'MATISSE-L-LR': return,findgen(230)*11+2600 ; LM,R=34
		'MATISSE-L-MR': return,'SPECTROMETER'; R=506
		'MATISSE-L-HR': return,'SPECTROMETER'; R=959
		'MATISSE-L-HP': return,'SPECTROMETER'; R=?
		'MATISSE-N-LR': return,findgen(510)*10+7950 ; N, R=30
		'MATISSE-N-HR': return,'SPECTROMETER'; R=218
		endcase
		end
	'FILTERS':begin
		case instrument_id(systemid) of
		'VINCI-MONA':	return,'vinci_k'
		'VINCI-IONIC':	return,'vinci_k'
		'PIONIER-FREE':	return,pionier_spectral_cal
		else:		return,'generic_c'
		endcase
		end
	'MAXCONFIG':return,20
	'CONFIG':begin
		maxNumSid=4
		maxNumTriple=(maxNumSid-1)*(maxNumSid-2)/2+1
		maxBaseline=maxNumSid*(maxNumSid-1)/2
		maxNumOutBeam=3
		case instrument_id(systemid) of
;			These limits specified again later!
			'AMBER-LR': maxSpecChan=30
			'AMBER-MR': maxSpecChan=600
			'AMBER-HR': maxSpecChan=1000
			'MIDI-N-PRISM': maxSpecChan=60
			'MIDI-N-GRISM': maxSpecChan=160
			'PIONIER-GRISM': maxSpecChan=10
			'PIONIER-FREE': maxSpecChan=1
			'GRAVITY-LR': maxSpecChan=20
			'GRAVITY-MR': maxSpecChan=200
			'GRAVITY-HR': maxSpecChan=2000
			'MATISSE-LN':maxSpecChan=100
			'MATISSE-L-LR':maxSpecChan=100
			'MATISSE-L-MR':maxSpecChan=500
			'MATISSE-L-HR':maxSpecChan=1000
			'MATISSE-L-HP':maxSpecChan=1000
			'MATISSE-N-LR':maxSpecChan=100
			'MATISSE-N-HR':maxSpecChan=1000
			else:	maxSpecChan=100
		endcase
		maxConfig=20
		GenConfig=allocGenConfig(maxNumSid,maxNumOutBeam,maxNumTriple, $
					 maxBaseline,maxSpecChan,maxConfig)
;
		GenConfig.NumSid=NumSid
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
;
; 		Set beam combiner ID, which is identical to instrument ID
		case instrument_id(systemid) of
			'VINCI-MONA' :GenConfig.BeamCombinerId=11
			'VINCI-IONIC':GenConfig.BeamCombinerId=12
			'MIDI-N-PRISM' :GenConfig.BeamCombinerId=21
			'MIDI-N-GRISM' :GenConfig.BeamCombinerId=22
			'AMBER-LR'   :GenConfig.BeamCombinerId=31
			'AMBER-MR'   :GenConfig.BeamCombinerId=32
			'AMBER-HR'   :GenConfig.BeamCombinerId=33
			'PIONIER-GRISM':GenConfig.BeamCombinerId=41
			'PIONIER-FREE' :GenConfig.BeamCombinerId=42
			'GRAVITY-LR' :GenConfig.BeamCombinerId=51
			'GRAVITY-MR' :GenConfig.BeamCombinerId=52
			'GRAVITY-HR' :GenConfig.BeamCombinerId=53
			'MATISSE-LN'  :GenConfig.BeamCombinerId=61
			'MATISSE-L-LR':GenConfig.BeamCombinerId=62
			'MATISSE-L-MR':GenConfig.BeamCombinerId=63
			'MATISSE-L-HR':GenConfig.BeamCombinerId=64
			'MATISSE-L-HP':GenConfig.BeamCombinerId=65
			'MATISSE-N-LR':GenConfig.BeamCombinerId=66
			'MATISSE-N-HR':GenConfig.BeamCombinerId=67
			         else:GenConfig.BeamCombinerId=1
		endcase
;
; 		Remove the instrument string for backward compatibility
;		SystemId='VLTI'
;
		case GenConfig.BeamCombinerId/10 of
;	  	VINCI
	      1:begin
		if GenConfig.NumSid ne 2 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 2!'
			return,-1
		endif
	  	NumOutBeam=1
		NumChan=1
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=0
;
		GenConfig.DelayLineId=[1,2]
		GenConfig.BCInputId=[1,2]
		GenConfig.BaselineId(0,0)=genconfig.stationid(0)+'-' $
				         +genconfig.stationid(1)
		GenConfig.NumBaseline(0)=1
;
		GenConfig.NumSpecChan=NumChan
		GenConfig.Wavelength(0:NumChan-1,0)=2.2e-6
		GenConfig.WavelengthErr(0:NumChan-1,0)=0.1e-6
		GenConfig.ChanWidth=0.03*GenConfig.Wavelength
		GenConfig.ChanWidthErr(0:NumChan-1,0)=1e-9
;
		return,GenConfig
		end
;		MIDI
	      2:begin
		if GenConfig.NumSid ne 2 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 2!'
			return,-1
		endif
		NumOutBeam=1
		NumChan=60
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=0
;
		GenConfig.DelayLineId=[2,3]
		GenConfig.BCInputId=[1,2]
		GenConfig.BaselineId(0,0)=genconfig.stationid(0)+'-' $
				         +genconfig.stationid(1)
		GenConfig.NumBaseline(0)=1
;
		GenConfig.NumSpecChan=NumChan
		GenConfig.Wavelength(0:NumChan-1,0)=(findgen(NumChan)*0.1+7.5)*1e-6
		GenConfig.WavelengthErr(0:NumChan-1,0)=0.05e-6
		GenConfig.ChanWidth=0.1e-6
		GenConfig.ChanWidthErr(0:NumChan-1,0)=1e-9
;
		return,GenConfig
		end
;		AMBER
	      3:begin
		if GenConfig.NumSid ne 3 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 3!'
			return,-1
		endif
		NumOutBeam=3
;		3-way combination
		GenConfig.NumOutBeam=NumOutBeam
;
		GenConfig.DelayLineId=[1,2,3]
		GenConfig.BCInputId=[1,3,5]
		GenConfig.BaselineId(0,*)=genconfig.stationid(0)+'-' $
				         +genconfig.stationid(1)
		GenConfig.BaselineId(1,*)=genconfig.stationid(1)+'-' $
				         +genconfig.stationid(2)
		GenConfig.BaselineId(2,*)=genconfig.stationid(2)+'-' $
				         +genconfig.stationid(0)
		GenConfig.NumBaseline=3
;
		case GenConfig.BeamCombinerId mod 10 of
			1: NumChan=[30,30,30]		; R=35
			2: NumChan=[600,600,600]	; R=1500
			3: NumChan=[1000,1000,1000]	; R=10000
		endcase
		GenConfig.NumSpecChan=NumChan
;		J-band, 1.08-1.34 microns
		l=findgen(NumChan(0))/(NumChan(0)-1)
		GenConfig.Wavelength(0:NumChan(0)-1,0)=(l*0.26+1.08)*1e-6
		GenConfig.ChanWidth(0:NumChan(0)-1,0)=0.26e-6/NumChan(0)
;		H-band, 1.51-1.79 microns
		l=findgen(NumChan(1))/(NumChan(1)-1)
		GenConfig.Wavelength(0:NumChan(1)-1,1)=(l*0.29+1.51)*1e-6
		GenConfig.ChanWidth(0:NumChan(1)-1,1)=0.29e-6/NumChan(1)
;		K-band, 1.97-2.38 microns
		l=findgen(NumChan(2))/(NumChan(2)-1)
		GenConfig.Wavelength(0:NumChan(2)-1,2)=(l*0.41+1.97)*1e-6
		GenConfig.ChanWidth(0:NumChan(2)-1,2)=0.41e-6/NumChan(2)
;
		GenConfig.WavelengthErr=0.1e-6
		GenConfig.ChanWidthErr=1e-9
;
;		Triple
		GenConfig.NumTriple=3
		GenConfig.TripleBeam=[[0,0,0],[1,1,1],[2,2,2]]
		GenConfig.TripleBase=[[0,1,2],[0,1,2],[0,1,2]]
		GenConfig.TripleChan=0
		for tr=0,2 do begin
		GenConfig.TripleNumChan(tr)=NumChan(tr)
		for bl=0,2 do begin
		GenConfig.TripleChan(0:NumChan(tr)-1,bl,tr)=indgen(NumChan(tr))
		endfor
		endfor
;
		return,GenConfig
		end
;		PIONIER
	      4:begin
		if GenConfig.NumSid ne 4 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 4!'
			return,-1
		endif
		NumChan=7
		GenConfig.NumSpecChan=NumChan
;		H-band, 1.51-1.79 microns
		l=findgen(NumChan(0))/(NumChan(0)-1)
		GenConfig.Wavelength(0:NumChan(0)-1,0)=(l*0.29+1.51)*1e-6
		GenConfig.ChanWidth(0:NumChan(0)-1,0)=0.29e-6/NumChan(0)
;
		GenConfig.WavelengthErr=0.1e-6
		GenConfig.ChanWidthErr=1e-9
;
		NumOutBeam=1
		NumTriple=((NumSid-1)*(NumSid-2))/2+1
		NumBaseline=(NumSid*(NumSid-1))/2
;
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
;
		GenConfig.NumOutBeam=NumOutBeam
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.NumSpecChan=NumChan
		GenConfig.TripleNumChan=NumChan
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
;
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin

		GenConfig.BaselineId(bl,*)=genconfig.stationid(i)+'-' $
					  +genconfig.stationid(j)
		bl=bl+1
		endfor
		endfor
;
		if 0 then begin
		tr=0
		k=0
		for i=1,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId(*,k) eq genconfig.stationid(0)+'-' $
						      +genconfig.stationid(i))
		bl2=where(GenConfig.BaselineId(*,k) eq genconfig.stationid(i)+'-' $
						      +genconfig.stationid(j))
		bl3=where(GenConfig.BaselineId(*,k) eq genconfig.stationid(0)+'-' $
						      +genconfig.stationid(j))
		GenConfig.TripleBeam(*,tr)=k
		GenConfig.TripleBase(*,tr)=[bl1,bl2,bl3]
		GenConfig.TripleNumChan(tr)=NumChan(k)
		for l=0,2 do begin
			GenConfig.TripleChan(0:NumChan(k)-1,l,tr)=indgen(NumChan(k))
		endfor
		tr=tr+1
		endfor
		endfor
		endif
;
;		Borrowed from GRAVITY (below) 
;
		tr=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		for k=j+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId(*,0) eq genconfig.stationid(i)+'-' $
						      +genconfig.stationid(j))
		bl2=where(GenConfig.BaselineId(*,0) eq genconfig.stationid(j)+'-' $
						      +genconfig.stationid(k))
		bl3=where(GenConfig.BaselineId(*,0) eq genconfig.stationid(i)+'-' $
						      +genconfig.stationid(k))
		GenConfig.TripleBeam(*,tr)=0
		GenConfig.TripleBase(*,tr)=[bl1,bl2,bl3]
		GenConfig.TripleNumChan(tr)=NumChan
		for l=0,2 do begin
			GenConfig.TripleChan(0:NumChan-1,l,tr) $
				=indgen(NumChan)
		endfor
		tr=tr+1
		endfor
		endfor
		endfor
;
		return,GenConfig
		end
;		GRAVITY
	      5:begin
		if GenConfig.NumSid ne 4 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 4!'
			return,-1
		endif
		NumTriple=1*(((NumSid-1)*(NumSid-2))/2+1)
		NumBaseline=(NumSid*(NumSid-1))/2
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
		case GenConfig.BeamCombinerId mod 10 of
			1: NumChan=20	; R=40
			2: NumChan=200	; R=400
			3: NumChan=2000	; R=4000
		endcase
		GenConfig.NumSpecChan=0
		GenConfig.NumSpecChan(0)=NumChan
;		K-band, 2.05-2.45 microns
		l=findgen(NumChan)/(NumChan-1)
		GenConfig.Wavelength(0:NumChan-1,0)=(l*0.4+2.05)*1e-6
		GenConfig.ChanWidth(0:NumChan-1,0)=0.4e-6/NumChan
;
		GenConfig.WavelengthErr(0:NumChan-1,0)=0.1e-6
		GenConfig.ChanWidthErr(0:NumChan-1,0)=1e-9
;
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.TripleNumChan(0:3)=NumChan
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin

		GenConfig.BaselineId(bl,*)=genconfig.stationid(i)+'-' $
					  +genconfig.stationid(j)
		bl=bl+1
		endfor
		endfor
;
		tr=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		for k=j+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId(*,0) eq genconfig.stationid(i)+'-' $
						      +genconfig.stationid(j))
		bl2=where(GenConfig.BaselineId(*,0) eq genconfig.stationid(j)+'-' $
						      +genconfig.stationid(k))
		bl3=where(GenConfig.BaselineId(*,0) eq genconfig.stationid(i)+'-' $
						      +genconfig.stationid(k))
		GenConfig.TripleBeam(*,tr)=0
		GenConfig.TripleBase(*,tr)=[bl1,bl2,bl3]
		GenConfig.TripleNumChan(tr)=NumChan
		for l=0,2 do begin
			GenConfig.TripleChan(0:NumChan-1,l,tr) $
				=indgen(NumChan)
		endfor
		tr=tr+1
		endfor
		endfor
		endfor
;
		return,GenConfig
		end
;		MATISSE
	      6:begin
		if GenConfig.NumSid ne 4 then begin
			print,'***Error(SYSTEM_CONFIG): NumSid must be 4!'
			return,-1
		endif
		NumTriple=1*(((NumSid-1)*(NumSid-2))/2+1)
		NumBaseline=(NumSid*(NumSid-1))/2
		GenConfig.StationId(0:NumSid-1)=stations(sid_index)
		GenConfig.RefStation=1
;
		NumOutBeam=1
		GenConfig.NumOutBeam=NumOutBeam
		case GenConfig.BeamCombinerId mod 10 of
			1: NumChan=15	; R=30
			2: NumChan=250	; R=500
			3: NumChan=500	; R=1000
			4: NumChan=15	; R=30
			5: NumChan=110	; R=220
		endcase
		case GenConfig.BeamCombinerId mod 10 of
			1: l_min=2.5	; R=30
			2: l_min=2.5	; R=500
			3: l_min=2.5	; R=1000
			4: l_min=8.0	; R=30
			5: l_min=8.0	; R=220
		endcase
		case GenConfig.BeamCombinerId mod 10 of
			1: l_max=5.5	; R=30
			2: l_max=5.5	; R=500
			3: l_max=5.5	; R=1000
			4: l_max=13.0	; R=30
			5: l_max=13.0	; R=220
		endcase
		width=(l_max-l_min)/NumChan
		case GenConfig.BeamCombinerId mod 10 of
			1: width=15	; R=30
			2: width=250	; R=500
			3: width=500	; R=1000
			4: width=15	; R=30
			5: width=110	; R=220
		endcase
		GenConfig.NumSpecChan=0
		GenConfig.NumSpecChan(0)=NumChan
		l=findgen(NumChan)/(NumChan-1)
		GenConfig.Wavelength(0:NumChan-1,0)=(l*width+l_min)*1e-6
		GenConfig.ChanWidth(0:NumChan-1,0)=width
;
		GenConfig.WavelengthErr(0:NumChan-1,0)=0.1e-6
		GenConfig.ChanWidthErr(0:NumChan-1,0)=1e-9
;
		GenConfig.NumTriple=NumTriple
		GenConfig.NumBaseline=NumBaseline
		GenConfig.TripleNumChan(0:3)=NumChan
;
		GenConfig.DelayLineId=indgen(NumSid)+1
		GenConfig.BCInputId=indgen(NumSid)+1
		GenConfig.FringeMod=1
		bl=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin

		GenConfig.BaselineId(bl,*)=genconfig.stationid(i)+'-' $
					  +genconfig.stationid(j)
		bl=bl+1
		endfor
		endfor
;
		tr=0
		for i=0,numsid-2 do begin
		for j=i+1,numsid-1 do begin
		for k=j+1,numsid-1 do begin
		bl1=where(GenConfig.BaselineId(*,0) eq genconfig.stationid(i)+'-' $
						      +genconfig.stationid(j))
		bl2=where(GenConfig.BaselineId(*,0) eq genconfig.stationid(j)+'-' $
						      +genconfig.stationid(k))
		bl3=where(GenConfig.BaselineId(*,0) eq genconfig.stationid(i)+'-' $
						      +genconfig.stationid(k))
		GenConfig.TripleBeam(*,tr)=0
		GenConfig.TripleBase(*,tr)=[bl1,bl2,bl3]
		GenConfig.TripleNumChan(tr)=NumChan
		for l=0,2 do begin
			GenConfig.TripleChan(0:NumChan-1,l,tr) $
				=indgen(NumChan)
		endfor
		tr=tr+1
		endfor
		endfor
		endfor
;
		return,GenConfig
		end
;
		endcase
		end
	endcase
	end
else: 	begin
	if key eq 'MIDNIGHT' then begin
	print,'Warning: SystemId ('+systemid+') not known, assume midnight=8 UT'
	return,8. 
	endif else $
        print,'***Error(SYSTEM_CONFIG): unknown SystemId: '+ $
                                          systemid+' !'
	end
endcase
;
return,-1
;
end
;************************************************************************Block 5
pro quit
;
print,''
print,' Bye-bye, leaving OYSTER...'
print,''
;
freememory
spawn,'rm -f temp_do_not_keep.fits'
;
exit
;
end
;-------------------------------------------------------------------------------
pro bye
;
print,''
print,' Bye-bye, leaving OYSTER...'
print,''
;
freememory
;
exit
;
end
;-------------------------------------------------------------------------------
