;*******************************************************************************
; File: pearl.pro
;
; Part of OYSTER, written by C. A. Hummel, ESO
;
; Description:
; ------------
; Container of IDL scripts for imaging interferometric data.
; PEARL implements an algorithm for broadband aperture synthesis by
; assigning effective temperatures to CLEAN components. This works well in
; the case of composite spectrum binaries. It should work in all circumstances
; where blackbody emission regions can be defined across an image.
;
; Sanity check as of May 2015, should produce 0.5% countours: 
; cd ~/oyster/oyster/lab/data/2001
; oyster
; get_data,'1997-05-01.cha'
; readmodel,'zetuma.model'
; binary_model.period=200 	; Binary will move during obs. otherwise
; scans.vissqcerr=0.05		; Use all data
; scans.triplephasecerr=0.01	; Use all data
; calcmodel			; Compute complex visibilities
; mockdata			; Overwrite observed values, incl photometry!
; pearl,'FKV0497',cellsize=0.3,imsize=129
;
; --- begin (obsolete)
; Ideas on how to implement the T fitting tool: For a given pixel, adopt
; a T. Compare predicted brightness with actual brightness (lambda) and
; derive calibration factor C.  This could be done for each channel, but
; in the initial step, since all pixels have the same T, the new T would
; not depend on channel.  If T was correct, then all other wavelengths,
; using the same C, would reproduce the SED of this pixel.  Given a C,
; compute for pixels of a different brightness the T (might be ambiguous!).
; Instead of using the pixels, it is best to convolve the CM with a beam
; at least the size of the CB.  Then iterate once more. Now the Teff
; computation is done in each channel.  If T was correct, and we have a
; good map, then the pixel T should be same in each channel.
;
; Problem: pixels also have a "diameter" Procedure: Fit photometry
; and derive Teff, assign to brightest pixel, the "reference".
; Assign diameter=1 to all pixels above threshold 5% of maxmimum.
; Compute teff for all pixels based on flux relative to reference pixel.
; Rescale flux of each pixel given teff Compute chisq Cycle through all
; diameter combinations and find the best chisq
; --- end (obsolete)
;
; Quote from AIPS help system:
; The Steer-Dewdney-Ito CLEAN (Steer et al. 1984), is also a variety
; of CLEAN, but one which diverges more radically from the original
; Högbom form than the algorithms just described. This algorithm is
; one of several which attempts to avoid the production of stripes in
; the CLEAN map, as described by (Schwarz 1984). The solution adopted in
; SDI CLEAN is simply to shave off all residuals greater than some trim
; threshold times the maximum residual and accept them as components. If
; the original source is smooth, the residual map will largely be so and
; thus the model components as well. The components selected in the trim
; stage must be scaled in some manner to account for the different effects
; of the dirty beam on resolved and unresolved sources. One possibility
; is to convolve the potential components by the dirty beam and scale
; the components such that the peak of the dirty components is equal to
; the peak dirty residual. The re-scaled components are then multiplied
; by a conventional loop gain and the new residual image formed just as
; in a major cycle of the CLARK CLEAN. The algorithm described by Steer
; et. al. scales to the peak residual and also includes a provision for
; processing point sources separately from the main emission.
;
; Block directory:
; ----------------
; Block 1: pearlimage,pearlflux,pearlmap,set_scale,set_scales,
;	   dft,mft,cleankernel,cleankernels,crbox,cmbox,cmpix,
;          clean,cleanmovie,residualmap,scramblemap,diffmap,dirtymap,selfcal,
;	   tvdb,tvrm,tvfm,tvcc,tvrg,tvcl,tkfm,
;          clean_event,pearl_event,
;	   pearl,pearlquit,pearlinit,pearldata,pearlmodel
; Block 2: blackbody_pro,teffmap
; Block 3: set_uvc,pearson
;
;************************************************************************Block 1
function pearlimage,imsze,cellsze,nc,relax=relax
;
; Return an image of dimension imsize(0)*imsize(1)*nc, i.e. column by row,
; width by height, and cellsize in mas. The image coordinates x and y
; are measured in radians, and x increases to the left (east) just as
; the right ascension does.
;
; Both dimensions must be odd numbers, so that there will be a pixel on the
; center of the image. The bottom left corner is at pixel (0,0), the top right
; corner is (imsize(0)-1,imsize(1)-1)
;
; This is the special PEARL image format including effective temperatures.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
MAS2RAD=pi_circle/(180l*3600000l)
;
if n_elements(relax) eq 0 then relax=0
if n_elements(imsze) eq 1 then imsze=[imsze,imsze]
;
if not relax then begin
if imsze(0) mod 2 ne 1 then begin
        print,'***Error(SET_MAP): imsize(0) must be odd number!'
        return,-1
endif
if imsze(1) mod 2 ne 1 then begin
        print,'***Error(SET_MAP): imsize(1) must be odd number!'
        return,-1
endif
endif
if n_elements(nc) eq 0 then nc=1
;
imsize=long(imsze)
cellsize=float(cellsze*MAS2RAD)
;
maps=replicate({map:fltarr(imsize(0),imsize(1))},nc)	; data cube
teff=fltarr(imsize(0),imsize(1))
logg=fltarr(imsize(0),imsize(1))
x=(lindgen(imsize(0)*imsize(1)) mod imsize(0)) - (imsize(0)/2)
y=(lindgen(imsize(0)*imsize(1)) / imsize(0)) - (imsize(1)/2)
x=-x*cellsize	; RA increases to the left (East)
y=+y*cellsize
;
return,{maps:maps,teff:teff,logg:logg,x:x,y:y, $
	zero:fltarr(nc),scale:fltarr(nc)+1.0}
;
end
;-------------------------------------------------------------------------------
function pearlflux,teff,clambda,dlambda
;
; Average blackbody flux over rectangular band pass
; defined through center wavelength clambda [nm] and full width dlambda [nm].
;
n=n_elements(clambda)
flux=fltarr(n)
i=where(dlambda le 1,count)
if count gt 0 then flux(i)=blackbody(teff,clambda(i)*1e-9)
i=where(dlambda gt 1,count)
for j=0,count-1 do begin
	l=(findgen(nint(dlambda(i(j))))+clambda(i(j))-dlambda(i(j))/2)*1e-9
	flux(i(j))=avg(blackbody(teff,l))
endfor
;
return,flux
;
end
;-------------------------------------------------------------------------------
function pearlmap,lambda,channel=channel
;
; For a given wavelength, return image. Convolve with circular Gaussian
; assuming the width is 3 pixels. Lambda is in meters.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
MAS2RAD=!pi/(180l*3600000l)
;
; nr is the number of non-zero effective temperature regions
ir=where(rt ne 0,nr)
;
cmap=cm.maps(0).map+cc.maps(0).map
;
for i=0,nr-1 do begin
        flux=stellarfluxes( $
              {teff:-rt(i),logg:rg(i), $
               diameter:1.0,ratio:1.0, $
               code:13}, $
               lambda,ld_coeffs)
	index=where(cm.teff eq rt(i))
	cmap(index)=cmap(index)*flux
endfor
;
cellsze=cm.x(0)-cm.x(1)
;
; When read from a file, we don't know the CLEAN beam for this PEARL image
if not keyword_set(channel) then begin
	r=size(cmap,/dim)
	imsze=r(0)/4+1
	cbm=pearlimage(imsze,cellsze/MAS2RAD,1)
	r=sqrt(cbm.x^2+cbm.y^2)
	a=3*cellsze
	result=machar() & tiny=result.xmin*1e6 & toosmall=alog(tiny)
	y=(-(r/a)^2)
	index=where(y lt toosmall,count)
	if count gt 0 then y(index)=1
	cbm.maps(0).map(*)=exp(y)
	if count gt 0 then cbm.maps(0).map(index)=0
	ch=0
endif else begin
	cbm=cb
	ch=channel
endelse
;
fmap=convol(cmap,cbm.maps(ch).map,/edge_wrap,/center)
;
return,fmap
;
end
;-------------------------------------------------------------------------------
function set_scale,image,j
;
; In pearl, tvscl is not used normally in order to avoid the rescaling effect 
; when CLEANing for example. Therefore, we determine the scaling parameters
; at the beginning and store them with the images, which are then displayed 
; with tv.
;
; Image can be a Pearl image or a flat map.
;
if n_elements(j) eq 0 then j=0
;
r=size(image)
if r(n_elements(r)-2) eq 8 then map=image.maps(j).map else map=image
;
maxm=max(map)
minm=min(map)
;
zero=minm
if maxm gt minm then scale=(!d.table_size-1)/(maxm-minm) $
		else scale=0
;
if r(n_elements(r)-2) eq 8 then begin
	image.zero(j)=zero
	image.scale(j)=scale
endif
;
return,[zero,scale]
;
end
;-------------------------------------------------------------------------------
function set_scales,image
;
; Store scaling parameters for all maps in a PEARL image cube.
;
nc=n_elements(image.maps)
r=fltarr(2,nc)
;
for j=0,nc-1 do r(*,j)=set_scale(image,j)
;
return,r
;
end
;-------------------------------------------------------------------------------
function dft,image,c,u,v,ch,db=db
;
; Given a map and complex visibilities at coordinates (u,v), add to the map
; the direct Fourier transform of the visibilities evaluated at the map 
; coordinates. Zero the map before if needed.
;
; Index ch specifies which plane of the map cube to compute.
; 
if n_elements(ch) eq 0 then ch=0
if n_elements(db) eq 0 then db=0
;
img=image
map=img.maps(ch).map
r=size(map)
;
ur=-2*!pi*u
vr=-2*!pi*v
;
cr=float(c)/n_elements(c)
ci=imaginary(c)/n_elements(c)
;
if db then begin
;
	dbxsize=r(1)
	dbysize=r(2)
;
	k=dbysize/2+1
	l=k*dbxsize-1
;
	m2=map(*,0:k-1)
	x=image.x(0:l)
	y=image.y(0:l)
;	for i=0L,n_elements(c)-1 do begin
;		p=ur(i)*x+vr(i)*y
;		m2=m2+cr(i)*cos(p)
;	endfor
	m2=m2+reform(cr#cos(ur#x+vr#y),size(m2,/dim))
;
	map(*,0:k-1)=m2
	map(*,k:dbysize-1)=reverse(reverse(map(*,0:k-2),2),1)
	img.maps(ch).map=map
;
endif else begin
;
	for i=0L,n_elements(c)-1 do begin
        	p=ur(i)*image.x+vr(i)*image.y
        	map=map+(cr(i)*cos(p)-ci(i)*sin(p))
	endfor
	img.maps(ch).map=map
;
endelse
;
return,img
;
end
;-------------------------------------------------------------------------------
function mft,image,c,u,v,ci,rt,rf,ch,normalize=normalize
;
; Given a map and coordinates (u,v), add to the complex visibilities
; the direct Fourier transform of the map evaluated at (u,v) and return
; the result. (u,v) can be vectors. Normalization is only necessary for
; the computation of visibilities, but for the computation of a difference
; map, only correlated fluxes should be compared.
;
if n_elements(normalize) eq 0 then normalize=0
;
if n_elements(ch) eq 0 then ch=0
;
uci=unique(ci)
nc=n_elements(uci)
if ch ge 1 then index=where(ci eq uci(ch-1)) $
	   else index=lindgen(n_elements(ci))
mv=c(index)
uc=u(index)
vc=v(index)
cj=ci(index)
uci=unique(cj)
nc=n_elements(uci)
;
cl=image.maps(ch).map
index=where(cl ne 0,count)
if count eq 0 then begin
	print,'No clean components!'
	return,-1
endif
cl=cl(index)
et=image.teff(index)
x=image.x(index)
y=image.y(index)
ir=where(rt ne 0,nr)
;
tf=abs(mv)*0
for i=0L,n_elements(cl)-1 do begin
	p=2*!pi*(uc*x(i)+vc*y(i))
	m=where(rt(ir) eq et(i)) & m=m(0)
	for j=0,nc-1 do begin
		f=(rf(j,m)/total(rf(j,ir)))*cl(i)
		f=rf(j,m)*cl(i)
		n=where(cj eq uci(j))
		tf(n)=tf(n)+f
		mv(n)=mv(n)+f*complex(cos(p(n)),sin(p(n)))
	endfor
endfor
;
if normalize then return,mv/tf $
             else return,mv
;
end
;-------------------------------------------------------------------------------
function cleankernel,db,ch,aparms=aparms,mparms=mparms
;
; Return the CLEAN beam, which is the central peak in the dirty beam.
; The latter is converted to a white light beam if the input is a cube.
; With aparms=[major (mas),minor (mas),pa (deg)] beam can be specified.
;
common FitEllipse,ex,ey
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(ch) eq 0 then ch=0
;
mas2rad=pi_circle/(180l*3600000l)
rad=180/!pi
; 
map=db.maps(ch).map
index=where(map eq max(map))
;
imsize=n_elements(map(*,0))
jmsize=n_elements(map(0,*))
;
cellsize=db.x(0)-db.x(1)
;
; Fit the beam size
IF n_elements(aparms) eq 0 THEN BEGIN
;
i=index(0) mod imsize
j=index(0)  /  imsize
index=[i-2,i-1,i,i+1,i+2]
jndex=[j-2,j-1,j,j+1,j+2]
;
r=findgen(5)-2
h=[0.5,0.0,0.0]*max(map)
x=fltarr(8)
y=x
; Fit quadratic polynomial, get FWHM by lowering 0.5 (max=1),
; and then solving the quadratic equation for r=0. This can
; fail if there are too few pixels across the maximum.
x(0:1)=quasol(poly_fit(r,map(index,j),2)-h)
y(2:3)=quasol(poly_fit(r,map(i,jndex),2)-h)
x(4:5)=quasol(poly_fit(r,map(index,jndex),2)-h)
y(4:5)=x(4:5)
x(6:7)=quasol(poly_fit(r,map(index,reverse(jndex)),2)-h)
y(6:7)=x([7,6])
;
; Set data
x=-x*cellsize/MAS2RAD
y=+y*cellsize/MAS2RAD
ndata=8
;
r=sqrt(x^2+y^2)
p=!pi-atan(x,y)
;
; Set parameters to fit
ma=5
ia=lonarr(ma)+1 & ia(0)=0 & ia(1)=0
index=where(ia ne 0,mfit)
e_parms=fltarr(ma)
e_parms(0)=0
e_parms(1)=0
e_parms(2)=max(r)
e_parms(3)=min(r)
i=where(r eq max(r)) & i=i(0)
e_parms(4)=p(i)
if e_parms(4) gt !pi then e_parms(4)=e_parms(4)-!pi
a=e_parms
;
; Store ellipse data
ex=r*sin(p)
ey=r*cos(p)
;
; Call marquardt
sig=fltarr(ndata)+0.1
pa=p    ; ellfuncs changes p!
quiet=!quiet
!quiet=1
marquardt,'ellfuncs',pa,r,sig,ndata,a,ia,ma,chisq,covar
!quiet=quiet
; Set new parameters
e_parms=a
e_parms(2)=a(2)*2*mas2rad
e_parms(3)=a(3)*2*mas2rad
e_parms(4)=a(4)
if not !quiet then $
print,'Synthesized beam ',a(2)*2,' by ',a(3)*2, $
	' mas, pa = ',360-a(4)*rad,' degrees.'
;
ENDIF ELSE BEGIN
;
; Use the supplied beam size parameters
e_parms=fltarr(5)
e_parms(2)=aparms(0)*mas2rad
e_parms(3)=aparms(1)*mas2rad
e_parms(4)=aparms(2)/rad
;
ENDELSE
;
; Return the fit results in aparms
aparms=fltarr(3)
aparms(0)=e_parms(2)/mas2rad	; major axis in mas
aparms(1)=e_parms(3)/mas2rad	; minor axis in mas
aparms(2)=e_parms(4)*rad	; PA in degrees
;
; Determine size of kernel
csize=db.x(0)-db.x(1)	; cellsize in radians
msize=intarr(2)
msize(0)=fix(max([e_parms(3)*abs(cos(e_parms(4))), $
		  e_parms(2)*abs(sin(e_parms(4)))]/csize)) < imsize
msize(1)=fix(max([e_parms(2)*abs(cos(e_parms(4))), $
		  e_parms(3)*abs(sin(e_parms(4)))]/csize)) < jmsize
msize=msize*3		; axis is FWHM, and we the kernel further down
msize=2*(msize/2)+1	; make size an odd number
if n_elements(mparms) eq 2 then msize=mparms
mparms=msize
cbm=pearlimage(msize,csize/mas2rad)
;
el=sqrt(e_parms(2)^2-e_parms(3)^2)
en=el/e_parms(2)	; numeric eccentricity
;
r=sqrt(cbm.x^2+cbm.y^2)
a=sqrt((e_parms(3)/2)^2/(1-en^2*cos(-atan(cbm.y,cbm.x)+e_parms(4)+!pi/2)^2))
result=machar() & tiny=result.xmin*1e6 & toosmall=alog(tiny)
y=(-(r/a)^2);> toosmall
index=where(y lt toosmall,count)
if count gt 0 then y(index)=1
cbm.maps(0).map(*)=exp(y)
if count gt 0 then cbm.maps(0).map(index)=0
;
return,cbm
;
end
;-------------------------------------------------------------------------------
function cleankernels,db,aparms=aparms
;
; With aparms=[major (mas),minor (mas),pa (deg)] beam can be specified.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
mas2rad=pi_circle/(180l*3600000l)
;
nc=n_elements(db.maps)
;
cbm=cleankernel(db,aparms=aparms)
r=size(cbm.maps(0).map)
cbmxsize=n_elements(cbm.maps(0).map(*,0))
cbmysize=n_elements(cbm.maps(0).map(0,*))
cellsize=(db.x(0)-db.x(1))/mas2rad
cb=pearlimage(r(1:2),cellsize,nc)
cb.maps(0).map=cbm.maps(0).map
;
!quiet=1
for j=1,nc-1 do begin
	cbm=cleankernel(db,j,aparms=aparms,mparms=r(1:2))
	cb.maps(j).map=cbm.maps(0).map
endfor
!quiet=0
;
return,cb
;
end
;-------------------------------------------------------------------------------
function crbox,image,ch
;
; Set to zero all pixels outside the crboxes.
;
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
;
if n_elements(ch) eq 0 then ch=0
if ncr eq 0 then return,image.maps(ch).map
;
wmap=image.maps(ch).map*0
for i=0,ncr-1 do wmap(crbox_x(0,i):crbox_x(1,i),crbox_y(3,i):crbox_y(4,i))=1
;
return,wmap*image.maps(ch).map
;
end
;-------------------------------------------------------------------------------
function clbox,image,ch
;
; Set to zero all pixels outside the crboxes.
;
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
;
if n_elements(ch) eq 0 then ch=0
if ncl eq 0 then return,image.maps(ch).map
;
wmap=image.maps(ch).map*0
for i=0,ncl-1 do wmap(clbox_x(0,i):clbox_x(1,i),clbox_y(3,i):clbox_y(4,i))=1
;
return,wmap*image.maps(ch).map
;
end
;-------------------------------------------------------------------------------
function cmpix,image,ch,cm
;
; Return index of pixels in combined map with flux, otherwise all pixels within
; clean windows.
;
; common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if n_elements(ch) eq 0 then ch=0
;
index=where(cm.maps(0).map ne 0,count)
if count eq 0 then return,clbox(image,ch)
;
wmap=image.maps(ch).map*0
if count gt 0 then wmap(index)=1
;
return,wmap*image.maps(ch).map
;
end
;-------------------------------------------------------------------------------
pro clean_parallel
;
common LocalClean,rmmaps,ccmaps,niter,niter_total
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlWids,draw_wid,clean_wid,channel_wid
;
spawn,'pwd',local_dir & local_dir=local_dir(0)
;
n_cpu=!cpu.hw_ncpu
clean_bridge=objarr(n_cpu)
clean_bridge_data=strarr(n_cpu)
for j=0,n_cpu-1 do begin
	clean_bridge(j)=obj_new('IDL_IDLBridge')
	clean_bridge(j)->execute,"cd,'"+local_dir+"'"
	clean_bridge(j)->execute,".run ~/oyster/source/common/pearl.pro"
	clean_bridge_data(j)='clean_bridge_'+string(j+1,format='(i1)')+'.xdr'
endfor
;
index=where(cm.maps(0).map ne 0,count)
if count eq 0 then niter=300 $
	      else niter=niter_total(0)
print,'Number of iterations to clean: ',niter
;
nc=n_elements(db.maps)
n_iter=nc/n_cpu
;
FOR i=0,n_iter DO BEGIN
;
print,string(13b),'Cleaning channels ' $
	+string((i*n_cpu+1) < (nc-1),format='(i4)')+' - ' $
	+string((i+1)*n_cpu < (nc-1),format='(i4)'), $
				format='(a,a,a,a,a,$)'
if i eq 0 then clock=tic('clock')
;
for j=0,n_cpu-1 do begin
	ch=i*n_cpu+j+1
	if ch lt nc then begin
		dirtymap,ch
		cc.maps(ch).map=0
		save,ch,rm,db,cc,rt,rf,cm,niter,file=clean_bridge_data(j)
		clean_bridge(j)->execute,"clean,"+string(j+1),/nowait
	endif
endfor
;
; After sleeping (no CPU used for main process), 
; stay in while loop until all processes finish.
if i ne 0 then wait,sleep
notdone = 1
while notdone do begin
	done=0
	for j=0,n_elements(clean_bridge)-1 do $
		done = done+clean_bridge[j]->Status()
	if done eq 0 then notdone=done
endwhile
if i eq 0 then sleep=fix(toc(clock)*0.6) > 1
;
; print,'Getting results...'
;
for j=0,n_cpu-1 do begin
	ch=i*n_cpu+j+1
	if ch lt nc then begin
		rm_bck=rm
		cc_bck=cc
		restore,file=clean_bridge_data(j)	; restores rm, cc
		dm.maps(ch).map=rm.maps(ch).map
		cm.maps(ch).map=cc.maps(ch).map
		cc.maps(ch).map=0
		widget_control,channel_wid,set_value=ch
		tvdb,ch
		tvfm,ch
		tvrm,ch
		rm_bck.maps(ch).map=rm.maps(ch).map
		cc_bck.maps(ch).map=cc.maps(ch).map
		rm=rm_bck
		cc=cc_bck
	endif
endfor
;
ENDFOR
;
print,''
print,'Cleaning up processes...'
for j=0,n_cpu-1 do begin
	obj_destroy,clean_bridge(j)
	spawn,'rm -f '+clean_bridge_data(j)
endfor
;
widget_control,channel_wid,set_value=1
;
end
;-------------------------------------------------------------------------------
pro clean,ch,cm,rm,db,cc,rt,rf,quiet=quiet;,fl
;
; Deconvolve the residual map (rm) from the dirty beam (db) and return a map
; of the subtracted clean components (cc) and the residual map (rm).
; Do exactly niter iterations, and store these locally.
;
; Note: this is a multiwavelength CLEAN.
;
common LocalClean,rmmaps,ccmaps,niter,niter_total
;
if n_params() eq 1 then begin
	quiet=1
	j=ch
	clean_bridge_data='clean_bridge_'+string(j,format='(i1)')+'.xdr'
	restore,clean_bridge_data	; also restore ch
endif
;
if n_elements(ch) eq 0 then ch=0
if n_elements(quiet) eq 0 then quiet=0
;
niter_max=100
;
if niter eq 0 then niter=niter_max
;
if niter le niter_max then begin
	rmmaps=replicate({map:rm.maps(ch).map*0},niter+1)
	ccmaps=replicate({map:cc.maps(ch).map*0},niter+1)
	rmmaps(0).map=rm.maps(ch).map
endif
;
dbxsize=n_elements(db.maps(ch).map(*,0))
dbysize=n_elements(db.maps(ch).map(0,*))
rmxsize=n_elements(rm.maps(ch).map(*,0))
rmysize=n_elements(rm.maps(ch).map(0,*))
;
nc=n_elements(rf(*,0))
nr=where(rt ne 0)
gain=0.1
;
for k=0,niter-1 do begin
	if ch eq 0 then rm_map=clbox(rm,ch) $
		   else rm_map=cmpix(rm,ch,cm)
	maxrm=max(rm_map)
	scale=maxrm*gain
;	Find maximum in current residual map
	index=where(rm_map eq maxrm) & index=index(0)
	i=(index mod rmxsize)-rmxsize/2
	j=(index/rmxsize)-rmysize/2
;	Cut-out patch of dirty beam centered on peak and size of RM
	dbs=db.maps.map(dbxsize/2-i-rmxsize/2:dbxsize/2-i+rmxsize/2, $
		   	dbysize/2-j-rmysize/2:dbysize/2-j+rmysize/2)*scale
	IF ch eq 0 THEN BEGIN
;	Broad-band CLEAN for ch=0 (white light channel)
	m=where(rt(nr) eq cc.teff(i+rmxsize/2,j+rmysize/2)) & m=m(0)
;	New code using SED, rf normalized when defining region
	flux_scale=total(rf(*,m))/nc
;	Subtract dirty beam in each channel from RM
;	flux=rf(*,0)*0+1
;	for l=0,nc-1 do $
;	rm.maps(ch).map=rm.maps(ch).map $
;		       -dbs(*,*,l+1)*(rf(l,m)/flux(l))/flux_scale
	dbs=dbs/flux_scale
	for l=0,nc-1 do $
	rm.maps(ch).map-=dbs(*,*,l+1)*rf(l,m)
	ENDIF ELSE BEGIN
;	Classic CLEAN
	flux_scale=1
	rm.maps(ch).map-=dbs(*,*,ch)*nc
	ENDELSE
	cc.maps(ch).map(i+rmxsize/2,j+rmysize/2)+=scale/flux_scale
	if ch eq 0 and niter le niter_max then begin
		rmmaps(k+1).map=rm.maps(ch).map
		ccmaps(k+1).map=cc.maps(ch).map
	endif
endfor
;
if not quiet then $
print,'CLEAN complete.'
;
if n_params() eq 1 then save,rm,cc,file=clean_bridge_data
;
end
;-------------------------------------------------------------------------------
pro cleanmovie,ccmaps,rmmaps,cb
;
mpegID=mpeg_open(size(rmmaps(0).map,/dimensions))
for i=0,n_elements(rmmaps)-1 do begin
map=convol(ccmaps(i).map,cb.maps(0).map,/edge_wrap,/center)+rmmaps(i).map
map=bytscl(sqrt(float(bytscl(map))))
mpeg_put,mpegID,image=map,frame=i
endfor
mpeg_save,mpegID,filename='clean.mpg'
mpeg_close,mpegID
;
end
;-------------------------------------------------------------------------------
pro residualmap,ch
;
; Compute the model phases for the current CLEAN map, and perform a phase
; self-calibration. Compute a new dirty/residual map.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if n_elements(ch) eq 0 then ch=0
;
; Compute the model phases from the current CLEAN map
if total(abs(cm.maps(ch).map)) ne 0 then mv=mft(cm,mv*0,uc,vc,ci,rt,rf,ch)
if mv(0) eq -1 then return
;
selfcal
diffmap,ch
;
end
;-------------------------------------------------------------------------------
pro scramblemap,ch
;
; Scramble the observed phases using random station phases, and compute a new
; residual map. Used for testing the selfcal routine.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if n_elements(ch) eq 0 then ch=0
;
; Compute the model phases from the current CLEAN map
if total(cm.maps(ch).map) ne 0 then mv=mft(cm,mv*0,uc,vc,ci,rt,rf,ch)
if mv(0) eq -1 then return
;
scramble
diffmap,ch
;
end
;-------------------------------------------------------------------------------
pro diffmap,ch
;
; Compute differential map, DM (same as DM if no model is available). 
; Set RM to equal DM.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if n_elements(ch) eq 0 then ch=0
;
uci=unique(ci)
if ch ge 1 then index=where(ci eq uci(ch-1)) $
	   else index=lindgen(n_elements(ci))
;
dm.maps(ch).map=0
dm=dft(dm,cv(index)-mv(index),uc(index),vc(index),ch)
status=set_scale(dm,ch)
rm.maps(ch).map=dm.maps(ch).map
r=set_scale(rm,ch)
;
print,'Residual map rms: ' $
	+string(stddev(dm.maps(ch).map)*100,format='(f5.1)')+' %'
;
end
;-------------------------------------------------------------------------------
pro dirtymap,ch
;
; Obsolete! See next procedure.
; This procedure does not work if the number of data points per channel is
; not the same. Procedure dft weights by number of visibilities! (24.2.2017)
;
; Compute DM. Set RM to equal DM.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if n_elements(ch) eq 0 then ch=0
;
uci=unique(ci)
if ch ge 1 then index=where(ci eq uci(ch-1)) $
	   else index=lindgen(n_elements(ci))
;
dm.maps(ch).map=0
dm=dft(dm,cv(index),uc(index),vc(index),ch)
;
status=set_scale(dm,ch)
rm.maps(ch).map=dm.maps(ch).map
r=set_scale(rm,ch)
;
end
;-------------------------------------------------------------------------------
pro dirtymap,ch
;
; Compute DM. Set RM to equal DM.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
md=dm
;
if n_elements(ch) eq 0 then ch=0
md.maps(*).map=0
;
if ch ge 1 then begin
	chan=ch
	nc=1 
endif else begin
	chan=unique(ci)
	nc=n_elements(chan)
endelse
;
for j=0,nc-1 do begin
	index=where(ci eq chan(j))
	md=dft(md,cv(index),uc(index),vc(index),j+1)
endfor
md.maps.map=md.maps.map/nc
dm.maps(ch).map=total(md.maps.map,3)
;
status=set_scale(dm,ch)
rm.maps(ch).map=dm.maps(ch).map
r=set_scale(rm,ch)
;
end
;-------------------------------------------------------------------------------
pro selfcal,phase_rms
;
; Original version
;
; Perform a phase self-calibration of complex visibilities (c) using model
; visibilities (m). This algorithm uses a weighting "mask" and SVD to remove 
; phase wraps first by computing an exact solution, then by considering more
; baselines to improve the solution up to all available data.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Determine number of spectral channels
uci=unique(ci)
numchan=n_elements(uci)
;
; Determine number of stations
numsid=max([unique(bi) mod 1000,unique(bi)/1000])
;
; Determine number of scans
numscan=n_elements(unique(si))
numscan=max(si)
;
; Illumination factor
weight=100
;
; SVD eigenvalue tolerance
tolerance=1e-5
;
for i=0,numscan-1 do begin
for j=0,numchan-1 do begin
	index=where(ci eq uci(j) and si eq i+1,count)
	if count gt 0 then begin
	dmatrix=fltarr(count,numsid)
	for l=0,count-1 do begin
		dmatrix(l,(bi(index(l)) mod 1000)-1)=+1
		dmatrix(l,(bi(index(l))  /  1000)-1)=-1
	endfor
	dmatriz=dmatrix
	yweight=fltarr(count)+1
	for k=0,numsid-2 do begin
		y=cphase(cv(index))-cphase(mv(index))
		jndex=where(y gt !pi,jc)
		if jc gt 0 then y(jndex)=y(jndex)-2*!pi
		jndex=where(y lt -!pi,jc)
		if jc gt 0 then y(jndex)=y(jndex)+2*!pi
;		Select all baselines involving station k
		bindex=where((bi(index)  /  1000) eq k+1 $
			  or (bi(index) mod 1000) eq k+1,count)
		if count gt 0 then begin
			dmatriz(bindex,*)=dmatriz(bindex,*)*weight
			dmatriz=(dmatriz<weight)>(-weight)
			yweight(bindex)=yweight(bindex)*weight
			yweight=(yweight<weight)>(-weight)
			y=y*yweight
                        tmatriz=transpose(dmatriz)
                        nmatriz=tmatriz#dmatriz
                        r=tmatriz#y
			svd8,nmatriz,nw,nu,nv
			windex=where(nw/max(nw) lt tolerance,wcount)
			if wcount gt 0 then nw(windex)=0
			svb8,nu,nw,nv,r,s
			p=dmatrix#s
			cv(index)=cv(index)*conj(complex(cos(p),sin(p)))
		endif
	endfor
delta=180/!pi*stddev(cphase(cv(index))-cphase(mv(index)))
; print,i,j,delta,abs(cv(index))
	endif	; End of: "Are there any data for this channel?"
endfor
endfor
;
y=cphase(cv)-cphase(mv)
jndex=where(y gt !pi,jc)
if jc gt 0 then y(jndex)=y(jndex)-2*!pi
jndex=where(y lt -!pi,jc)
if jc gt 0 then y(jndex)=y(jndex)+2*!pi
phase_rms=stddev(y)*180/!pi
print,'Solution computed, phase RMS [deg] = ',phase_rms
;
end
;-------------------------------------------------------------------------------
pro selfcal,phase_rms
;
; Experimental version for datasets containing sub-arrays, i.e., station
; indices in the full array need to be translated to sub-array IDs.
;
; Perform a phase self-calibration of complex visibilities (c) using model
; visibilities (m). This algorithm uses a weighting "mask" and SVD to remove 
; phase wraps first by computing an exact solution, then by considering more
; baselines to improve the solution up to all available data.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Determine number of spectral channels
uci=unique(ci)
numchan=n_elements(uci)
;
; Determine number of stations in data set
maxsid=max([unique(bi) mod 1000,unique(bi)/1000])
;
; Station index (converted to range [1,2,3,...]
sta_index=intarr(maxsid)
;
; Determine number of scans
numscan=n_elements(unique(si))
numscan=max(si)
;
; Illumination factor
weight=100
;
; SVD eigenvalue tolerance
tolerance=1e-5
;
for i=0,numscan-1 do begin
for j=0,numchan-1 do begin
	index=where(ci eq uci(j) and si eq i+1,count)
	if count gt 0 then begin
;	Compute numsid for sub-array (e.g., Beauty Contest 2020)
	sub_index=unique([bi(index)/1000,bi(index) mod 1000])
	numsid=n_elements(sub_index)
;	Prepare station index for sub-array station IDs
	sta_index=intarr(max(sub_index))
	sta_index(sub_index-1)=indgen(numsid)
	dmatrix=fltarr(count,numsid)
	for l=0,count-1 do begin
		dmatrix(l,sta_index(bi(index(l)) mod 1000-1))=+1
		dmatrix(l,sta_index(bi(index(l))  /  1000-1))=-1
	endfor
	dmatriz=dmatrix
	yweight=fltarr(count)+1
	for k=0,numsid-2 do begin
		y=cphase(cv(index))-cphase(mv(index))
		jndex=where(y gt !pi,jc)
		if jc gt 0 then y(jndex)=y(jndex)-2*!pi
		jndex=where(y lt -!pi,jc)
		if jc gt 0 then y(jndex)=y(jndex)+2*!pi
;		Select all baselines involving station k
		index1=bi(index) mod 1000-1
		index2=bi(index)  /  1000-1
		bindex=where(sta_index(index1) eq k $
			  or sta_index(index2) eq k,count)
;		bindex=where(sub_index(bi(index) mod 1000-1) eq k $
;			  or sub_index(bi(index)  /  1000-1) eq k,count)
		if count gt 0 then begin
			dmatriz(bindex,*)=dmatriz(bindex,*)*weight
			dmatriz=(dmatriz<weight)>(-weight)
			yweight(bindex)=yweight(bindex)*weight
			yweight=(yweight<weight)>(-weight)
			y=y*yweight
                        tmatriz=transpose(dmatriz)
                        nmatriz=tmatriz#dmatriz
                        r=tmatriz#y
			svd8,nmatriz,nw,nu,nv
			windex=where(nw/max(nw) lt tolerance,wcount)
			if wcount gt 0 then nw(windex)=0
			svb8,nu,nw,nv,r,s
			p=dmatrix#s
			cv(index)=cv(index)*conj(complex(cos(p),sin(p)))
		endif
	endfor
delta=180/!pi*stddev(cphase(cv(index))-cphase(mv(index)))
; print,i,j,delta,abs(cv(index))
	endif	; End of: "Are there any data for this channel?"
endfor
endfor
;
y=cphase(cv)-cphase(mv)
jndex=where(y gt !pi,jc)
if jc gt 0 then y(jndex)=y(jndex)-2*!pi
jndex=where(y lt -!pi,jc)
if jc gt 0 then y(jndex)=y(jndex)+2*!pi
phase_rms=stddev(y)*180/!pi
print,'Solution computed, phase RMS [deg] = ',phase_rms
;
end
;-------------------------------------------------------------------------------
pro scramble
;
; Create a random vector of station phases and use them to scramble the observed
; phases. For testing purposes.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
; Determine number of spectral channels
uci=unique(ci)
numchan=n_elements(uci)
;
; Determine number of stations
numsid=max(unique(bi) mod 1000)
;
; Determine number of scans
numscan=n_elements(unique(si))
numscan=max(si)
;
for i=0,numscan-1 do begin
for j=0,numchan-1 do begin
	index=where(ci eq uci(j) and si eq i+1,count)
	if count gt 0 then begin
	dmatrix=fltarr(count,numsid)
	for l=0,count-1 do begin
		dmatrix(l,(bi(index(l)) mod 1000)-1)=+1
		dmatrix(l,(bi(index(l))  /  1000)-1)=-1
	endfor
	sp=randomu(seed,numsid)*2*!pi
	rp=dmatrix#sp
	cv(index)=cv(index)*conj(complex(cos(rp),sin(rp)))
	endif	; End of: "Are there any data for this channel?"
endfor
endfor
;
y=cphase(cv)-cphase(mv)
jndex=where(y gt !pi,jc)
if jc gt 0 then y(jndex)=y(jndex)-2*!pi
jndex=where(y lt -!pi,jc)
if jc gt 0 then y(jndex)=y(jndex)+2*!pi
phase_rms=stddev(y)*180/!pi
print,'Station phases scrambled, phase RMS [deg] = ',phase_rms
;
end
;-------------------------------------------------------------------------------
pro tvdb,ch
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlWids,draw_wid,clean_wid,channel_wid
;
if n_elements(ch) eq 0 then ch=0
;
widget_control,draw_wid,get_value=win_num & wset,win_num
;
tv,(db.maps(ch).map-db.zero(ch))*db.scale(ch)
if ch eq 0 then label='Whitelight' $
	   else label=greek('lambda')+'='+string(wl(ch-1),format='(f6.1)')+' nm'
xyouts,0,0,label,/normal,charsize=1.4
;
end
;-------------------------------------------------------------------------------
pro tvrm,ch
;
; Display residual map (lower right), using scaling parameters of RM.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlWids,draw_wid,clean_wid,channel_wid
;
if n_elements(ch) eq 0 then ch=0
;
widget_control,draw_wid,get_value=win_num & wset,win_num
;
xpos=n_elements(db.maps(0).map(*,0))
;
tv,(rm.maps(ch).map-rm.zero(ch))*rm.scale(ch),xpos,0
;
end
;-------------------------------------------------------------------------------
pro tvet
;
; Displays the temperature map
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlWids,draw_wid,clean_wid,channel_wid
;
widget_control,draw_wid,get_value=win_num & wset,win_num
;
xpos=n_elements(db.maps(0).map(*,0))
;
loadct,13,/silent
; Transform range [tmin,tmax] to [255,0]
tmin=0.
tmax=10000.
cmin=0.
cmax=!d.table_size-1
;
a1=(cmax-cmin)/(tmin-tmax)
a0=cmax-a1*tmin
teff=(a0+a1*cm.teff) > 0
tv,teff < cmax,xpos,0
loadct,0,/silent
;
end
;-------------------------------------------------------------------------------
pro etrg,fitsfile
;
; Allow user to define irregularly shaped regions in the residual map
; and specify the effective temperature for them.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if n_elements(fitsfile) eq 0 then begin
;
image=(rm.maps(0).map-rm.zero(0))*rm.scale(0)
imsze=size(image,/dim)
nx=imsze(0)
ny=imsze(1)
scale_factor=4
image=rebin(image,imsze*scale_factor)
;
print,'-----------------------------------------------------------------'
print,'Use Freehand tool to draw regions, starting with the LARGEST one.'
print,'Select Quit to finish and assign temperatures at command prompt.'
;
xroi,bytscl(image,top=200),roi_geometry=roigeo,regions_out=regout,/block
if n_elements(roigeo) eq 0 then return
;
;print,'Now enter Teff for each region in order of decreasing size.'
;si=reverse(sort(roigeo.area))
;roigeo=roigeo(si)
;regout=regout(si)
print,'Now enter Teff for each region starting with first one:'
etmap=fltarr(nx,ny)+5000
lgmap=fltarr(nx,ny)+4.5
nr=n_elements(regout)
for ireg=0,nr-1 do begin
	read,teff, $
	prompt='Please enter Teff for region ' $
		+string(ireg+1,format='(i1)')+': '
	regout(ireg)->scale,fltarr(2)+1./float(scale_factor)
;	Make sure there are no pixels without initialization
; 	The following command is not compiled by GDL
	mask=regout(ireg)->computemask(dimension=[nx,ny])
	etmap(where(mask eq 255))=teff
endfor
;
; Save to disk
ext_header=["XTENSION= 'IMAGE' / IMAGE extension", $
	    "BITPIX  = -32 /  Number of bits per data pixel", $
	    "NAXIS   = 2 / Number of data axes", $
	    "NAXIS1  = "+string(nx)+" /", $
	    "NAXIS2  = "+string(ny)+" /", $
	    "PCOUNT  = 0 / No Group Parameters", $
	    "GCOUNT  = 1 / One Data Group", $
	    "EXTNAME = 'Teff' /", $
	    "END     "]
fitsfile='Teff_map.fits'
writefits,fitsfile,etmap,ext_header,/append
ext_header(where(strpos(ext_header,'EXTNAME') ge 0))="EXTNAME = 'log(g)' /"
writefits,fitsfile,lgmap,ext_header,/append
print,'Regions save to '+fitsfile+'. Click T-Map to display.' 
;
endif else begin
;
etmap=mrdfits(fitsfile,1)
;
endelse
;
; Copy to CM
cm.teff=etmap
cc.teff=cm.teff
;
; (Re-)initialize the effective temperature regions
pearlinit,cm.teff
;
end
;-------------------------------------------------------------------------------
pro tvcc,ch
;
; Overplots on the RM the positions of CLEAN components in CC
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common PearlWids,draw_wid,clean_wid,channel_wid
;
if n_elements(ch) eq 0 then ch=0
;
widget_control,draw_wid,get_value=win_num & wset,win_num
;
cellsize=db.x(0)-db.x(1)
dbxsize=n_elements(db.maps(ch).map(*,0))
dbysize=n_elements(db.maps(ch).map(0,*))
dmxsize=n_elements(dm.maps(ch).map(*,0))
dmysize=n_elements(dm.maps(ch).map(0,*))
;
index=where(cc.maps(ch).map ne 0,count)
tek_color
if count gt 0 then plots,(index mod dmxsize)+dbxsize,index/dmxsize, $
	/device,color=3,psym=3
loadct,0,/silent
;
tvcl
;
end
;-------------------------------------------------------------------------------
pro tvcm,ch
;
; Overplots on the RM the positions of CLEAN components in CM
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common PearlWids,draw_wid,clean_wid,channel_wid
;
if n_elements(ch) eq 0 then ch=0
;
widget_control,draw_wid,get_value=win_num & wset,win_num
;
cellsize=db.x(0)-db.x(1)
dbxsize=n_elements(db.maps(ch).map(*,0))
dbysize=n_elements(db.maps(ch).map(0,*))
dmxsize=n_elements(dm.maps(ch).map(*,0))
dmysize=n_elements(dm.maps(ch).map(0,*))
;
tek_color
index=where(cm.maps(ch).map ne 0,count)
if count gt 0 then plots,(index mod dmxsize)+dbxsize,index/dmxsize, $
	/device,color=3,psym=3
;
for i=0,ncl-1 do begin
	plots,clbox_x(*,i)+dbxsize,clbox_y(*,i), $
	/device,psym=0,color=3
endfor
loadct,0,/silent
;
end
;-------------------------------------------------------------------------------
pro tvrg,ch
;
; Overplot the Teff regions on the RM.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common PearlWids,draw_wid,clean_wid,channel_wid
;
if n_elements(ch) eq 0 then ch=0
;
widget_control,draw_wid,get_value=win_num & wset,win_num
;
cellsize=db.x(0)-db.x(1)
dbxsize=n_elements(db.maps(ch).map(*,0))
dbysize=n_elements(db.maps(ch).map(0,*))
dmxsize=n_elements(dm.maps(ch).map(*,0))
dmysize=n_elements(dm.maps(ch).map(0,*))
;
tek_color
for i=0,ncr-1 do begin
	plots,crbox_x(*,i)+dbxsize,crbox_y(*,i)+dmysize*0, $
	/device,psym=0,color=3
endfor
loadct,0,/silent
;
end
;-------------------------------------------------------------------------------
pro tvcl
;
; Overplot the CLEAN windows on the RM.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common PearlWids,draw_wid,clean_wid,channel_wid
;
widget_control,draw_wid,get_value=win_num & wset,win_num
;
cellsize=db.x(0)-db.x(1)
dbxsize=n_elements(db.maps(0).map(*,0))
dbysize=n_elements(db.maps(0).map(0,*))
dmxsize=n_elements(dm.maps(0).map(*,0))
dmysize=n_elements(dm.maps(0).map(0,*))
;
tek_color
for i=0,ncl-1 do begin
	plots,clbox_x(*,i)+dbxsize,clbox_y(*,i)+dmysize*0, $
	/device,psym=0,color=3
endfor
loadct,0,/silent
;
end
;-------------------------------------------------------------------------------
pro tvfm,ch
;
; Display final map (upper right).
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlWids,draw_wid,clean_wid,channel_wid
;
if n_elements(ch) eq 0 then ch=0
;
widget_control,draw_wid,get_value=win_num & wset,win_num
;
xpos=n_elements(db.maps(0).map(*,0))
ypos=n_elements(dm.maps(0).map(0,*))
;
dims_cm=size(cm.maps.map,/dim)
dims_cb=size(cb.maps.map,/dim)
if dims_cb(0) ge dims_cm(0) then begin
	print,'Warning: map too small for convolution with CLEAN beam!'
	return
endif
map=convol(cm.maps(ch).map+cc.maps(ch).map,cb.maps(ch).map,/edge_wrap,/center) $
   +rm.maps(ch).map
r=set_scale(map)
; tv,(map-r(1))*r(0),xpos,ypos
tvscl,sqrt((map-r(0))*r(1)),xpos,ypos
;
end
;-------------------------------------------------------------------------------
pro tkfm,ch
;
; Plots contours of map of channel ch.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if n_elements(ch) eq 0 then ch=0
;
map=convol(cm.maps(ch).map+cc.maps(ch).map,cb.maps(ch).map,/edge_wrap,/center) $
   +rm.maps(ch).map
r=set_scale(map)
;
rad2mas=180l*3600l*1000l/!pi
;
plevels=[0.05,0.1,0.2,0.5,1,2,5,10,20,50,80]
index=where(plevels gt abs(min(map)/max(map))*100)
if index(0) gt 0 then index=[index(0)-1,index]
if index(0) eq -1 then index=[9]
plevels=plevels(index)
nlevels=[-plevels(0)]
;
levels=[nlevels,plevels]
levels_label=''
x=-unique(cm.x)*rad2mas
y=+unique(cm.y)*rad2mas
;
for i=0,n_elements(levels)-1 do begin
	if abs(levels(i)) lt 1 then $
		if levels(i) lt 0 then format='(f5.1)' else format='(f4.1)'
	if abs(levels(i)) ge 1 then $
		if levels(i) lt 0 then format='(i3)' else format='(i2)'
	if abs(levels(i)) ge 10 then $
		if levels(i) lt 0 then format='(i4)' else format='(i3)'
	levels_label=levels_label+string(levels(i),format=format)
endfor
;
!p.multi=0
!x.range=0
!y.range=0
;
contour,map,x,y,levels=plevels*max(map)/100,xstyle=1,ystyle=1, $
	xtitle='Contours: '+levels_label+' % of peak',ytitle='[mas]', $
	xrange=[max(x),min(x)]
contour,map,x,y,levels=nlevels*max(map)/100,c_linestyle=1,/overplot
;
end
;-------------------------------------------------------------------------------
pro tkpm,ch
;
; Plots contours of the PEARL image computed for channel ch.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if ch eq 0 then begin
	tkfm
	return
endif
map=pearlmap(wl(ch-1)*1e-9,channel=ch)
map=map/max(map)*max(cm.maps(0).map+cc.maps(0).map)+rm.maps(0).map
;
rad2mas=180l*3600l*1000l/!pi
;
plevels=[0.05,0.1,0.2,0.5,1,2,5,10,20,50,80]
index=where(plevels gt abs(min(map)/max(map))*100)
if index(0) gt 0 then index=[index(0)-1,index]
if index(0) eq -1 then index=[9]
plevels=plevels(index)
nlevels=[-plevels(0)]
;
levels=[nlevels,plevels]
levels_label=''
x=-unique(cm.x)*rad2mas
y=+unique(cm.y)*rad2mas
;
for i=0,n_elements(levels)-1 do begin
	if abs(levels(i)) lt 1 then $
		if levels(i) lt 0 then format='(f5.1)' else format='(f4.1)'
	if abs(levels(i)) ge 1 then $
		if levels(i) lt 0 then format='(i3)' else format='(i2)'
	if abs(levels(i)) ge 10 then $
		if levels(i) lt 0 then format='(i4)' else format='(i3)'
	levels_label=levels_label+string(levels(i),format=format)
endfor
;
!p.multi=0
!x.range=0
!y.range=0
;
contour,map,x,y,levels=plevels*max(map)/100,xstyle=1,ystyle=1, $
	xtitle='Contours: '+levels_label+' % of peak',ytitle='[mas]', $
	xrange=[max(x),min(x)]
contour,map,x,y,levels=nlevels*max(map)/100,c_linestyle=1,/overplot
;
end
;-------------------------------------------------------------------------------
function clean_event,event
;
; CLEAN is run whenever the cc map is empty. The "Take" option allows
; to clean more than 100 components (the default range for CLEAN) which
; are aggregated in CM.
;
common LocalPearl,star,pearl_channel
common PearlWids,draw_wid,clean_wid,channel_wid
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common LocalClean,rmmaps,ccmaps,niter,niter_total
;
if widget_info(draw_wid,/valid) ne 0 and !d.name ne 'PS' then begin
	widget_control,draw_wid,get_value=win_num
	wset,win_num
endif
ch=pearl_channel
;
if n_elements(niter) eq 0 then niter=100
if n_elements(niter_total) eq 0 then niter_total=intarr(n_elements(unique(ci))+1)
if n_elements(ccmaps) eq 0 then $
	ccmaps=replicate({map:cc.maps(ch).map*0},1)
;
case string(event.value) of
;	Call-back for combined CLEAN buttons
	'Take' :begin
		dm.maps(ch).map=rm.maps(ch).map
		cm.maps(ch).map=cm.maps(ch).map+cc.maps(ch).map
		cc.maps(ch).map=0
		ccmaps.map=0
		niter_total(ch)=niter_total(ch)+niter
		niter=0
		r=set_scale(rm,ch)
		tvrm,ch
		tvcc,ch
		tvfm,ch
	       	widget_control,clean_wid,set_value=0
		widget_control,channel_wid,sensitive=1
		end
	'Update':begin
		cm.maps(ch).map=cm.maps(ch).map+cc.maps(ch).map
		cc.maps(ch).map=0
		ccmaps.map=0
		niter_total(ch)=niter_total(ch)+niter
		niter=0
		ir=where(rt ne 0,nr)
		if nr gt 0 then residualmap,ch
		r=set_scale(rm,ch)
		tvrm,ch
		tvfm,ch
		tvcc,ch
	       	widget_control,clean_wid,set_value=0
		widget_control,channel_wid,sensitive=1
	       	end
	'Cycle':begin
		if total(abs(ccmaps.map)) ne 0 then begin
			cm.maps(ch).map=cm.maps(ch).map+cc.maps(ch).map
			cc.maps(ch).map=0
			ccmaps.map=0
			niter_total(ch)=niter_total(ch)+niter
	       		widget_control,clean_wid,set_value=0
			mv=mft(cm,mv*0,uc,vc,ci,rt,rf,ch)
			selfcal
			print,'Computing dirty map...'
			dirtymap,ch
		endif
		device,cursor_standard=150
		niter=niter_total(ch)
		print,'Running '+string(niter,format='(i3)') $
			+' CLEAN iterations...',format='(a,$)'
		clean,ch,cm,rm,db,cc,rt,rf;,fl
		print,'Finished CLEAN.'
		device,/cursor_original
		cm.maps(ch).map=cc.maps(ch).map
		cc.maps(ch).map=0
		ccmaps.map=0
		print,'Running self cal...'
		mv=mft(cm,mv*0,uc,vc,ci,rt,rf,ch)
		selfcal
		print,'Computing dirty map...'
		dirtymap,ch
		tvfm
		tvrm
		cm.maps(ch).map=0
	       	end
	'Help': begin
		print,'Combined CLEAN slider (drag to start):'
		print,'Take: add components to model'
		print,'Update: add components and run selfcal'
		print,'Cycle: add components, selfcal, CLEAN for new DM'
		end
;	Call-back for channel buttons
	'Remove' :begin
		if ch ge 1 then begin
		 	print,'Removing clean components from map...',format='(a,$)'
		 	cc.maps(ch).map=0
		 	cm.maps(ch).map=0
		 	rm.maps(ch).map=0
		 	diffmap,ch
		 	tvrm,ch
		 	tvfm,ch
		 	tvcm,ch
		 	print,'done.'
		endif else begin
			print,'Cannot clear combined map with this button!'
		endelse
		end
	'Clean':begin
		index=where(cm.maps(0).map ne 0,count)
		if count eq 0 then niter=300 $
			      else niter=niter_total(ch)
		print,'Number of iterations to clean: ',niter
		nc=n_elements(db.maps)
		for ch=1,nc-1 do begin
			widget_control,channel_wid,set_value=ch
			dirtymap,ch
			cc.maps(ch).map=0
			print,string(13b), $
				'Cleaning channel '+string(ch)+'...', $
				format='(a,a,$)'
			clean,ch,cm,rm,db,cc,rt,rf,/quiet
			dm.maps(ch).map=rm.maps(ch).map
			cm.maps(ch).map=cc.maps(ch).map
			cc.maps(ch).map=0
			tvdb,ch
			tvfm,ch
			tvrm,ch
		endfor
		print,''
;		widget_control,channel_wid,set_value=pearl_channel
		widget_control,channel_wid,set_value=1
		end
	'Contour':begin
		if !d.name ne 'PS' then window,/free,xsize=550,ysize=500
		if ch ge 1 then $
		!p.title='Channel '+string(ch,format='(i2)')+' map' $
		else $
		!p.title='PEARL CLEAN map'
		tkfm,ch
		if !d.name eq 'PS' then begin
			device,/close
;			set_plot,!display
			set_screen
		endif
		widget_control,draw_wid,get_value=id
		wset,id
		device,decompose=0
		end
	'Calibrate':begin
		teffmap
		end
	'T-Map':begin
		tvet
		rtu=unique(rt(where(rt ne 0)))
		n=n_elements(rtu)
		rtc=intarr(n)
		for i=0,n-1 do begin
			index=where(cm.teff eq rtu(i),count)
			rtc(i)=count
		endfor
		print,'Effective temperature regions and areas (pixel): '
		for i=0,n-1 do print,rtu(i),rtc(i)
		end
	'T-Init': begin
		pearlinit
		end
	'HELP': begin
		print,'Remove: remove clean components from selected channel map'
		print,'Clean: clean *all* channels; use windows if combined map was cleared'
		print,'Contour: plot a contour map of the selected channel'
		print,'Calibrate: compute and install new Teff map'
		print,'T-Map: display color Teff map'
		print,'T-Init: initialize Teff map'
		end
	'Cleanup': begin
;		Call back for when PEARL GUI is closed
		if n_elements(ccmaps) ne 0 then ccmaps.map=0
		niter_total(*)=0
		end
	else: 	begin
;		Call-back for clean and channel sliders 
		if event.id eq channel_wid then begin
			pearl_channel=event.value
			ch=pearl_channel
; 			Reset the total CLEAN counter if we start with an empty map
			if total(rm.maps(ch).map) eq 0 then begin
				niter_total(ch)=0
				dirtymap,ch
				r=set_scale(rm,ch)
			endif
			tvdb,ch
			tvrm,ch
			tvfm,ch
		endif
		if event.id eq clean_wid then begin
			widget_control,channel_wid,sensitive=0
			if n_elements(ccmaps) eq 0 then $
			ccmaps=replicate({map:cc.maps(ch).map*0},1)
			if total(abs(ccmaps.map)) eq 0 then begin
				device,cursor_standard=150
				print,'Running CLEAN...',format='(a,$)'
				niter=0
				clean,ch,cm,rm,db,cc,rt,rf;,fl
				print,'Finished CLEAN.'
				device,/cursor_original
;				cleanmovie,ccmaps,rmmaps,cb
			endif
			widget_control,event.id,get_value=value
			niter=fix(value)
			rm.maps(ch).map=rmmaps(niter).map
			cc.maps(ch).map=ccmaps(niter).map
			tvrm,ch
			tvcc,ch
			tvfm,ch
		endif
		end
endcase
;
end
;-------------------------------------------------------------------------------
function pearl_event,event
;
; Call back for the main Pearl GUI.
;
forward_function set_boxes
;
common LocalPearl,star,pearl_channel
common PearlWids,draw_wid,clean_wid,channel_wid
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common LocalClean,rmmaps,ccmaps,niter,niter_total
;
ch=pearl_channel
nc=n_elements(db.maps)
;
if n_elements(niter_total) eq 0 $
then niter_total=intarr(n_elements(unique(ci))+1)
if n_elements(ccmaps) eq 0 then $
	ccmaps=replicate({map:cc.maps(ch).map*0},1)
;
cellsize=db.x(0)-db.x(1)
dbxsize=n_elements(db.maps(ch).map(*,0))
dbysize=n_elements(db.maps(ch).map(0,*))
dmxsize=n_elements(dm.maps(ch).map(*,0))
dmysize=n_elements(dm.maps(ch).map(0,*))
;
command=event.value
case command of
	'Layout':begin
		print,'Large box left: DB, small upper box: FM, lower: RM/Tmap'
		end
	'Reset':begin
		print,'Resetting data and maps...',format='(a,$)'
		cv=ov
		mv=ov*0
		niter=0
;
		niter_total(*)=0
		cc.maps.map=0
		cm.maps.map=0
		rm.maps.map=0
		dirtymap
		tvfm
		tvrm
		tvcm
	       	widget_control,clean_wid,set_value=0
		ccmaps.map=0
		print,'done.'
		end
	'Remove':begin
		print,'Removing clean components from combined map...',format='(a,$)'
		cc.maps(0).map=0
		cm.maps(0).map=0
		rm.maps(0).map=0
		dirtymap
		tvfm
		tvrm
		tvcm
		ccmaps.map=0
		print,'done.'
		end
	'Window':begin
		print,'----------------------------------------------------'
		print,'Place one window in final map; right click to reset!'
		widget_control,draw_wid,get_value=win_num & wset,win_num
		!x.s=[0,1] & !y.s=[0,1] ; Normalized coordinates: (0,0)=BLC
		icom=set_boxes(x0,y0)
		if icom eq 4 then begin
;			Remove all windows, do not initialize Teff map!
			ncl=0
			icom=set_boxes(x0,y0,/clear)
			tvfm
			tvrm
			print,'Windows removed.'
			return,0
		endif
		xsize=dbxsize+dmxsize
		ysize=dmysize*2
;		(x,y) in map pixels
		x=(x0*xsize-dbxsize) > 0
		y=(y0*ysize-dmysize) > 0
		if total(x) eq 0 or total(y) eq 0 then begin
;			print,'***Error: window placed outside FM!'
;			return,-1
;			Assume it was placed in CM, move up to FM
			y0=y0+float(dmysize)/ysize
			y=(y0*ysize-dmysize) > 0
		endif
		r=size(cc.maps(0).map)
;		Make sure the box does not extend over the borders
		x=x>0
		x=x<r(1)
		y=y>0
		y=y<r(2)
		icom=cw_form('0,LIST,Window only|Window with Teff|Cancel,set_value=0,QUIT')
		case icom.tag0 of
;			  Setting a window
			0:begin
			  clbox_x(*,ncl)=x(*,0)
			  clbox_y(*,ncl)=y(*,0)
			  ncl=ncl+1
			  widget_control,clean_wid,/sensitive
			  tvcl
			  end
;			  Defining a region
		 	1:begin
			  clbox_x(*,ncl)=x(*,0)
			  clbox_y(*,ncl)=y(*,0)
			  ncl=ncl+1
			  read,teff,prompt='Enter Teff [K]: '
			  logg=4.5
			  cm.teff(x(0,0):x(1,0),y(3,0):y(4,0))=teff
			  cm.logg(x(0,0):x(1,0),y(3,0):y(4,0))=logg
			  cc.teff=cm.teff
			  cc.logg=cm.logg
			  rt(ncr)=teff
			  rg(ncr)=logg
			  index=unique(ci)-1
			  rf(*,ncr)=pearlflux(teff,wl(index),bw(index))
;			  Normalize rf with total flux
			  rf(*,ncr)=rf(*,ncr)/fl(index)
; 			  Scale with geometric mean
			  rf=rf/gavg(rf(*,0:ncr-1))
			  ncr=ncr+1
			  tvcl
			  end
			2:begin
			  icom=set_boxes(x0,y0,/clear)
			  end
		endcase
		print,'Number of active regions: ',ncr
		print,'Number of active windows: ',ncl
		end
	'Regions':begin
		etrg
		end
;	CLEAN window (currently not used, functionality included in Region)
	'Window_old':begin
		widget_control,draw_wid,get_value=win_num & wset,win_num
		!x.s=[0,1] & !y.s=[0,1] ; Normalized coordinates, (0,0)=BLC
		icom=set_boxes(x0,y0,n)
		xsize=dbxsize+dmxsize
		ysize=dmysize*2
;		(x,y) in map pixels
		x=(x0*xsize-dbxsize)
		y=(y0*ysize)
		icom=cw_form('0,LIST,Keep new windows|Keep old windows|Remove windows,set_value=0,QUIT')
		case icom.tag0 of
		 	0:begin
			  clbox_x=x
			  clbox_y=y
			  ncl=n
			  end
			1:
			2:begin
			  ncl=0
			  icom=set_boxes(x0,y0,n,/clear)
			  end
		endcase
		print,'Number of active windows: ',ncl
		end
	'Selfcal':begin
		selfcal
		dirtymap
                tvrm,ch
		tvcm,ch
;               tvfm,ch
		print,'Display updated.'
		end
	'Scramble':begin
		scramblemap
		r=set_scale(rm,ch)
                tvrm,ch
		tvcm,ch
;               tvfm,ch
		print,'Display updated.'
		end
	'Shift':begin
;		Execute "Take" command
		status=clean_event({value:'Take'})
;		Now get shift vector
		read,x,y,prompt='Enter shift [mas] (right, up): '
		mas2rad=!pi/(180l*3600000l)
		i=fix(x*mas2rad/cellsize)
		j=fix(y*mas2rad/cellsize)
		for k=0,nc-1 do begin
		 	cm.maps(k).map=shift(cm.maps(k).map,i,j)
		 	rm.maps(k).map=shift(rm.maps(k).map,i,j)
		 	dm.maps(k).map=shift(dm.maps(k).map,i,j)
			cm.teff=shift(cm.teff,i,j)
			cc.teff=shift(cc.teff,i,j)
		endfor
		tvrm,ch
		tvcm,ch
		tvfm,ch
		x=i*cellsize
		y=j*cellsize
		p=2*!pi*(-uc*x+vc*y)
		cv(*)=cv*complex(cos(p),sin(p))
		mv(*)=mv*complex(cos(p),sin(p))
		end
	'Contour':begin
		if !d.name ne 'PS' then window,/free,xsize=550,ysize=500
		if ch ge 1 then $
		!p.title=star+' map for channel '+string(ch,format='(i2)') $
		else $
		!p.title=star+' CLEAN map'
		if !d.name eq 'PS' then device,filename=star+'.ps'
		tkpm,ch
		if !d.name eq 'PS' then begin
			device,/close
;			set_plot,!display
			set_screen
			print,'Plot saved in '+star+'.ps'
		endif
		widget_control,draw_wid,get_value=id
		wset,id
		device,decompose=0
		end
	'Convolve':begin
		cb0=cb
		read,a,b,p,prompt='Enter a,b[mas], pa[deg]: '
		if b gt a then begin
			print,'Minor axis must be shortar than major axis!'
		endif else begin
		 	cb=cleankernel(db,ch,aparms=[a,b,p])
;		 	tvfm
			tkfm
		 	cb=cb0
		endelse
		end
	'Save': begin
		fitsfile='pearl.fits'
		mas2rad=180/!pi*3600000.d0
		naxis1=dmxsize
		naxis2=dmysize
		parseidldate,systime(),y,m,d
		datum=constrictordate(y,m,d)
		cdelt1=cellsize
		cdelt2=cellsize
		header=["SIMPLE  =                    T / Written by IDL", $
		         "BITPIX  =                  -32 / Number of bits per data pixel", $
		         "NAXIS   =                    2 / Number of data axes", $
		         "NAXIS1  =   "+string(naxis1)+" /", $
        		 "NAXIS2  =   "+string(naxis2)+" /", $
        		 "EXTEND  =                    T / FITS data may contain extensions", $
        		 "DATE    = '"+datum+"'         / Creation date of FITS header", $
        		 "BSCALE  =    1.00000000000E+00 /REAL = TAPE * BSCALE + BZERO", $
        		 "BZERO   =    0.00000000000E+00 /", $
        		 "CTYPE1  = 'RA---TAN'           /", $
        		 "CRVAL1  =    0.00000000000E+00 /", $
        		 "CDELT1  =   "+string(cdelt1)+" / radians", $
        		 "CRPIX1  =      1.000000000E+00 /", $
        		 "CROTA1  =      0.000000000E+00 /", $
        		 "CTYPE2  = 'DEC--TAN'           /", $
        		 "CRVAL2  =    0.00000000000E+00 /", $
        		 "CDELT2  =   "+string(cdelt2)+" / radians", $
        		 "CRPIX2  =      1.000000000E+00 /", $
        		 "CROTA2  =      0.000000000E+00 /", $
        		 "END     "]
		writefits,fitsfile,cm.maps(0).map+cc.maps(0).map,header
	        ext_header=["XTENSION= 'IMAGE' / IMAGE extension", $
	                     "BITPIX  = -32 /  Number of bits per data pixel", $
	                     "NAXIS   = 2 / Number of data axes", $
	                     "NAXIS1  = "+string(naxis1)+" /", $
	                     "NAXIS2  = "+string(naxis2)+" /", $
	                     "PCOUNT  = 0 / No Group Parameters", $
	                     "GCOUNT  = 1 / One Data Group", $
	                     "EXTNAME = 'Teff' /", $
	                     "END     "]
	        writefits,fitsfile,cm.teff,ext_header,/append
	        ext_header(where(strpos(ext_header,'EXTNAME') ge 0)) $
			="EXTNAME = 'log(g)' /"
	        writefits,fitsfile,cm.logg,ext_header,/append
		print,'Saved to file '+fitsfile
		end
	'Read':	begin
		readimage,'pearl.fits'
		pearlinit,cm.teff
		residualmap
		tvfm
		tvrm
		tvcm
		end
	'Clear':begin
		set_screen
		device,decompose=0
		wdall
		end
	'HELP':	begin
		print,'Layout: click to have explanation of maps'
		print,'Reset: start over, but keep the windows'
		print,'Remove: remove all clean components from combined map'
		print,'Window: place or remove windows, optionally assign Teff'
		print,'Regions: draw irregularly-shaped region and assign Teff'
		print,'Selfcal: run phase self-calibration, display FM and DM/CM'
;		print,'Scramble: scramble observed phases (for testing purposes)'
		print,'Shift: shift map'
		print,'Contour: plot a contour PEARL map for the selected channel'
		print,'Convolve: convolve selected clean map with Gaussian'
		print,'Save: write PEARL map to disk (pearl.fits)'
		print,'Read: read PEARL map from disk (pearl.fits)'
		print,'Clear: remove all plot windows'
		end
endcase
;
end
;-------------------------------------------------------------------------------
pro pearl,starid,cellsize=cellsize,imsize=imsize,lmin=lmin,lmax=lmax
;
; Setup and start an imaging session.
; Cellsize [mas], imsize=[nx,ny] or imsize=nxy (square map)
;
; lmin/lmax are the minimum and maximum wavelength [nm] to use.
; If lmax < lmin, take these values as center and width [nm].
;
common PearlWids,draw_wid,clean_wid,channel_wid
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common LocalPearl,star,pearl_channel
common LocalClean,rmmaps,ccmaps,niter,niter_total
;
if n_elements(lmin) eq 0 then lmin=100.		; [nm]
if n_elements(lmax) eq 0 then lmax=100000.	; [nm]
flag=0
if lmin lt 100 then flag=1
if lmax lt 100 then flag=1
if flag then begin
	print,'Error: lmin and lmax should be in [nm]!'
	return
endif
lmin=float(lmin)
lmax=float(lmax)
if lmax lt lmin then begin
	lmin=lmin-lmax/2
	lmax=lmin+lmax
endif
;
; Initialize imaging data
pearldata,starid,status,lmin=lmin,lmax=lmax
if status ne 0 then return
niter_total=intarr(n_elements(unique(ci))+1)
;
; Initialize map parameters
uvr=sqrt(uc^2+vc^2)
fringespacing_min=(180/!pi)*3600000/max(uvr)
skip_cb=0
if n_elements(cellsize) eq 0 then begin
	cellsize=fringespacing_min/4
	print,'Cellsize [mas] set to: ',cellsize
endif else begin
	if cellsize gt fringespacing_min/3 then begin
	print,'Warning: cellsize too large!'
	skip_cb=1
	aparm=[fringespacing_min,fringespacing_min,0]
aparm=[20,20,0]
;	return
	endif
endelse
fringespacing_max=(180/!pi)*3600000/min(uvr(where(uvr ne 0)))
if n_elements(imsize) eq 0 then begin
	imsize=(4*fringespacing_max)/cellsize
	imsize=fix(imsize/2)*2+1
endif
if n_elements(imsize) eq 1 then imsize=[imsize,imsize]
if (imsize(0) mod 2) eq 0 then begin
	imsize(0)=imsize(0)+1
	print,'Warning(PEARL): imsize(0) must be odd, set to: ',imsize(0)
endif
if (imsize(1) mod 2) eq 0 then begin
	imsize(1)=imsize(1)+1
	print,'Warning(PEARL): imsize(1) must be odd, set to: ',imsize(1)
endif
print,'Mapsize is ',imsize(0)*cellsize,' mas by ',imsize(1)*cellsize,' mas'
dmsize=imsize
dbsize=imsize*2-1
;
; Compute maps
uci=unique(ci)
nc=n_elements(uci)
db=pearlimage(dbsize,cellsize,nc+1)	; 0: white light, 1-nc: channels
cb=pearlimage(dbsize,cellsize,nc+1)	; 0: white light, 1-nc: channels
;
print,'Computing dirty beam in '+string(nc,format='(i0)')+' channels...'
cv_1=cv & cv_1(*)=1
for j=0,nc-1 do begin
	index=where(ci eq uci(j))
	db=dft(db,cv_1(index),uc(index),vc(index),j+1,/db)
endfor
db.maps.map=db.maps.map/nc
db.maps(0).map=total(db.maps.map,3)		; white light dirty beam
status=set_scales(db)
if skip_cb then cb=cleankernels(db,aparms=aparm) $
	   else cb=cleankernels(db)		; clean beam
;
print,'Computing white light dirty map...'
dm=pearlimage(dmsize,cellsize,nc+1)
rm=dm
dirtymap
;
cm=pearlimage(dmsize,cellsize,nc+1)	; clean map
cc=cm					; clean component map
;
; Inititalize regions and effective temperature map
pearlinit
;
base_wid=widget_base(/row,title='Pearl',resource_name='oyster', $
	kill_notify='pearlquit')
column1_wid=widget_base(base_wid,/col,event_func='clean_event')
column2_wid=widget_base(base_wid,/col,event_func='pearl_event')
;
draw_wid=widget_draw(column1_wid, $
        scr_ysize=dbsize(1)+1,scr_xsize=dbsize(0)+dmsize(0))
color_wid=widget_draw(column1_wid, $
        scr_ysize=30,scr_xsize=256)
row1_wid=widget_base(column1_wid,/row,/base_align_bottom)
clean_wid=cw_fslider(row1_wid,minimum=0,maximum=100,value=0,/drag, $
        xsize=dbsize[0],format='(i3)')
button_wid=cw_bgroup(row1_wid,/row,['Take','Update','Cycle','Help'], $
	/return_name)
channel_wid=widget_slider(column1_wid,minimum=0,maximum=nc,value=0,/drag, $
        xsize=dbsize[0]+dmsize[0],title='Channel slider (0=white light)')
button_wid=cw_bgroup(column1_wid,/row,/return_name, $
	['Remove','Clean','Contour','Calibrate','T-Map','T-Init','HELP'])
;
pearl_channel=0	; Initialize to white light channel
;
; Right-hand column of buttons
buttons=['Layout','Reset','Remove','Window','Regions','Selfcal','Shift', $
	 'Contour','Convolve','Save','Read','Clear','HELP']
button_wid=cw_bgroup(column2_wid,/col,buttons,/return_name)
;
widget_control,base_wid,/realize
xmanager,'pearl',base_wid,/no_block
;
; For now, PEARL uses color maps only
device,decomposed=0
;
; Display Teff color bar
bar = reverse(BINDGEN(256)) # REPLICATE(1B, 30)
loadct,13,/silent
tv,bar
tek_color
xyouts,1*256/5-!d.x_ch_size,!d.y_ch_size,'2K',/device,color=5
xyouts,2*256/5-!d.x_ch_size,!d.y_ch_size,'4K',/device,color=1
xyouts,3*256/5-!d.x_ch_size,!d.y_ch_size,'6K',/device,color=1
xyouts,4*256/5-!d.x_ch_size,!d.y_ch_size,'8K',/device,color=1
loadct,0,/silent
;
tvdb	; dirty beam (left)
tvrm	; residual map (upper right)
tvfm	; final map (lower right)
;
end
;-------------------------------------------------------------------------------
pro pearlquit,event
;
common LocalClean,rmmaps,ccmaps,niter,niter_total
;
r=clean_event({value:'Cleanup'})
;
end
;-------------------------------------------------------------------------------
pro pearlinit,teff
;
; Procedure to initialize effective temperature regions and maps.
; If Teff map is not specified, fit a blackbody to the SED in "fl".
;
common PearlBoxes,clbox_x,clbox_y,ncl,crbox_x,crbox_y,ncr
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
nc=n_elements(fl)
logg=4.5	; This is not used anymore
tgran=50.0
;
;  Try to fit Teff to photometry
if n_elements(teff) eq 0 then begin
	ncl=0
	mcl=20
	clbox_x=fltarr(5,mcl)
	clbox_y=fltarr(5,mcl)
	teff=5000
	a=[teff,1.0]
	index=unique(ci)-1
	xc=wl(index)*1d-9
	yc=fl(index)
	if n_elements(xc) ge 2 then begin
		r=curvefit(xc,yc,yc*0+1,a, $
			function_name='blackbody_pro',/noderiv)
		teff=nint(a(0)/tgran)*tgran
		print,'Fitted Teff = ',teff
	endif
	cm.teff=teff
	cm.logg=logg
	cc.teff=cm.teff
	cc.logg=cm.logg
endif
;
tu=unique(teff)
ncr=n_elements(tu)
mcr=ncr > 20
rt=fltarr(mcr)
rg=fltarr(mcr)+logg
rf=fltarr(nc,mcr)
crbox_x=fltarr(5,mcr)
crbox_y=fltarr(5,mcr)
;
index=unique(ci)-1
;
for i=0,ncr-1 do begin
	rt(i)=tu(i)
	rf(*,i)=pearlflux(tu(i),wl(index),bw(index))
; 	Normalize rf with total flux
	rf(*,i)=rf(*,i)/fl(index)
endfor
; Scale with geometric mean
rf=rf/gavg(rf(*,0:ncr-1))
;
end
;-------------------------------------------------------------------------------
pro pearldata,starid,status,lmin=lmin,lmax=lmax
;
; Initialize imaging data, but not maps.
;
; ======== WARNING:
; Original version, has bugs if processing more than one nights' data.
; Superceded by version following this one.
;
; ov: observed (complex) visibility, never changed
; ow: visibility weight
; uc: u-coordinate
; vc: v-coordinate
; cv: corrected visibility, initially copy of ov, then phase calibrated
; mv: model visibility
; si: scan index
; bi: baseline index, e.g. 12, 56, no more than 9 stations
; ci: channel index
; wl: wavelength [nm]
; bw: bandwidth [nm]
; fl: total flux = SED, read from scans.photometryc
; db: dirty beam, cube
; cb: clean beam, not a cube
; dm: dirty map, not a cube
; rm: residual map, not a cube
; cm: clean map, not a cube, accumulates all cc
; cc: clean components corresponding to one run of CLEAN, not a cube
; rt: temperature map
; rg: log(g) map
; rf: stellar atmosphere flux normalized by observed SED
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Tables,scantable,bgtable,stationtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PearlBaselines,ob,bl
common LocalPearl,star,pearl_channel
;
if n_elements(lmin) eq 0 then lmin=100.		; [nm]
if n_elements(lmax) eq 0 then lmax=100000.	; [nm]
flag=0
if lmin lt 100 then flag=1
if lmax lt 100 then flag=1
if flag then begin
	print,'Error: lmin and lmax should be in [nm]!'
	return
endif
lmin=float(lmin)
lmax=float(lmax)
if lmax lt lmin then begin
	lmin=lmin-lmax/2
	lmax=lmin+lmax
endif
;
if n_elements(starid) eq 0 then begin
	ustars=unique(scans.starid)
	if n_elements(ustars) eq 1 then begin
		starid=ustars(0)
	endif else begin
        	print,'***Error(PEARLDATA): you have to specify a star!'
		status=-1
        	return
	endelse
endif
star=strupcase(starid)
;
for night=0,n_elements(bufferinfo)-1 > 0 do begin
if n_elements(bufferinfo) gt 1 then $
loadnight,GenInfo(night(0)).date, $
	  GeoInfo(night(0)).systemid, $
	  GenInfo(night(0)).configid
if night eq 0 then stationids=genconfig.stationid(0:genconfig.numsid-1) $
	      else stationids=[stationids, $
			      genconfig.stationid(0:genconfig.numsid-1)]
endfor
if n_elements(bufferinfo) gt 1 then $
stationids=unique(stationids)
;
sio=0
nci=1	; channel 0 reserved for white light
;
FOR night=0,n_elements(bufferinfo)-1 > 0 DO BEGIN
;
if n_elements(bufferinfo) gt 1 then $
loadnight,GenInfo(night(0)).date, $
	  GeoInfo(night(0)).systemid, $
	  GenInfo(night(0)).configid
;
index=where(scans.starid eq star and scantable.code eq 1,count)
if count eq 0 then begin
        print,'***Error(PEARLDATA): no scans on this star for this date!'
	status=-1
        return
endif
;
; These are reformed into flat arrays
ov=reform(scans(index).complexvis,n_elements(scans(index).complexvis))
ow=reform(scans(index).complexweight,n_elements(scans(index).complexweight))
uc=reform(scans(index).uvw(*,*,*,0),n_elements(scans(index).uvw(*,*,*,0)))
vc=reform(scans(index).uvw(*,*,*,1),n_elements(scans(index).uvw(*,*,*,1)))
;
; Create scan index, has same dimensions as ov, i.e. si(ob,ch,bl,sc)
; si=fix(scans(index).complexvis < 10) ; Removed 2020, may produce underflow
si=fix(scans(index).complexvis)
for i=0,count-1 do si(*,*,*,i)=i+1+sio
si=reform(si,n_elements(si))
sio=sio+count
;
; Create baseline index, has same dimensions as ov, i.e. bi(ob,ch,bl,sc)
baselineid=intarr(genconfig.numoutbeam,max(genconfig.numbaseline))
ob_index=baselineid
bl_index=baselineid
for i=0,genconfig.numoutbeam-1 do begin
for l=0,genconfig.numbaseline(i)-1 do begin
        m=where(stationids eq strmid(genconfig.baselineid(l,i),0,3))+1
        n=where(stationids eq strmid(genconfig.baselineid(l,i),4,3))+1
        baselineid(i,l)=1000*m+n
	ob_index(i,l)=i
	bl_index(i,l)=l
endfor
endfor
; bi=fix(scans(index).complexvis < 10) ; Removed 2020, may produce underflow
bi=fix(scans(index).complexvis)
ob=bi
bl=bi
for j=0,max(genconfig.numspecchan)-1 do begin
	for i=0,count-1 do bi(*,j,*,i)=baselineid
	for i=0,count-1 do ob(*,j,*,i)=ob_index
	for i=0,count-1 do bl(*,j,*,i)=bl_index
endfor
bi=reform(bi,n_elements(bi))
ob=reform(ob,n_elements(ob))
bl=reform(bl,n_elements(bl))
;
; Create channel index, has same dimensions as ov, i.e. ci(ob,ch,bl,sc)
; ci=fix(scans(index).complexvis < 10) ; Removed 2020, may produce underflow
ci=fix(scans(index).complexvis)
; Not all scans have photometry computed
ti=3
if n_elements(size(scans(index).photometryc(0,0,*),/dim)) eq 2 then ti=2
fi=where(total(scans(index).photometryc(0,0,*),ti) gt 0)
spectrometers=strarr(genconfig.numoutbeam)
w=genconfig.wavelength	; save wavelength info
if system_id(systemid) eq 'NPOI' then begin
; 	Assume same channel layout for NPOI!
	mean_w=total(w,2)/genconfig.numoutbeam
	for i=0,genconfig.numoutbeam-1 do genconfig.wavelength(*,i)=mean_w
endif
for i=0,genconfig.numoutbeam-1 do $
	spectrometers(i)=strjoin(string( $
		genconfig.wavelength(0:genconfig.numspecchan(i)-1,i)*1e6))
uspms=unique(spectrometers)
for i=0L,n_elements(uspms)-1 do begin
	k=where(spectrometers eq uspms(i))
	wl0=genconfig.wavelength(0:genconfig.numspecchan(i)-1,k(0))*1e9
	bw0=genconfig.chanwidth(0:genconfig.numspecchan(i)-1,k(0))*1e9
;	Get SED
	fl0=reform( $
	   scans(index(fi(0))).photometryc(0,k(0),0:genconfig.numspecchan(i)-1))
;	Make sure the SED is not identical to zero
	if total(fl0) eq 0 then fl0=blackbody(5000.,wl0*1e-9)
	if night eq 0 and i eq 0 then wl=wl0 else wl=[wl,wl0]
	if night eq 0 and i eq 0 then bw=bw0 else bw=[bw,bw0]
	if night eq 0 and i eq 0 then fl=fl0 else fl=[fl,fl0]
	for j=0L,genconfig.numspecchan(i)-1 do ci(k,j,*,*)=j+nci
	nci=nci+genconfig.numspecchan(i)
endfor
genconfig.wavelength=w
ci=reform(ci,n_elements(ci))
;
; Select only data within requested wavelength range
jndex=where(wl lt lmin or wl gt lmax,count)
if count gt 0 then begin
 	index=whereequal(ci-1,jndex)
	ow(index)=0
endif	
;
; Edit the uv-coverage
index=where(ow gt 0,count)
if count eq 0 then begin
        print,'***Error(PEARLDATA): no data for this star!'
	status=-1
        return
endif
uc=float(uc(index))
vc=float(vc(index))
ov=ov(index)
ow=ow(index)
bi=bi(index)
ci=ci(index)
si=si(index)
ob=ob(index)
bl=bl(index)
;
if night eq 0 then begin
	uc0=uc
	vc0=vc
	ov0=ov
	ow0=ow
	bi0=bi
	ci0=ci
	si0=si
	ob0=ob
	bl0=bl
endif else begin
	uc0=[uc0,uc]
	vc0=[vc0,vc]
	ov0=[ov0,ov]
	ow0=[ow0,ow]
	bi0=[bi0,bi]
	ci0=[ci0,ci]
	si0=[si0,si]
	ob0=[ob0,ob]
	bl0=[bl0,bl]
endelse
;
ENDFOR
;
uc=uc0
vc=vc0
ov=ov0
ow=ow0
bi=bi0
ci=ci0
si=si0
ob=ob0
bl=bl0
;
cv=ov
mv=ov*0
;
; We need to remove channel indices without data
uci=unique(ci)
while n_elements(uci) lt max(uci) do begin
for j=1,n_elements(uci) do begin
	if uci(j-1) gt j then begin
		index=indgen(n_elements(wl))+1
		k=where(index ne j)
		wl=wl(k)
		fl=fl(k)
		bw=bw(k)
		k=where(ci gt j)
		ci(k)=ci(k)-1
		uci=unique(ci)
		break
	endif
endfor
endwhile
;
wl=wl(0:max(uci)-1)
fl=fl(0:max(uci)-1)
bw=bw(0:max(uci)-1)
;
status=0
if max(abs(ov)) eq 0 then begin
        print,'***Error(PEARLDATA): no visibility data!'
	status=-1
        return
endif
;
end
;-------------------------------------------------------------------------------
pro pearldata,starid,status,lmin=lmin,lmax=lmax
;
; Initialize imaging data, but not maps.
;
; Combines several data sets, but they all have to have 
; the same spectrometer setup. 
;
; ov: observed (complex) visibility, never changed
; ow: visibility weight
; uc: u-coordinate
; vc: v-coordinate
; cv: corrected visibility, initially copy of ov, then phase calibrated
; mv: model visibility
; si: scan index
; bi: baseline index, e.g. 12, 56, no more than 9 stations
; ci: channel index
; wl: wavelength [nm]
; bw: bandwidth [nm]
; fl: total flux = SED, read from scans.photometryc
; db: dirty beam, cube
; cb: clean beam, not a cube
; dm: dirty map, not a cube
; rm: residual map, not a cube
; cm: clean map, not a cube, accumulates all cc
; cc: clean components corresponding to one run of CLEAN, not a cube
; rt: temperature map
; rg: log(g) map
; rf: stellar atmosphere flux normalized by observed SED
;
; ob: OB index with same length as ov
; bl: Baseline index (1000*m+n) of station IDs m and n
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common Tables,scantable,bgtable,stationtable
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
common PearlBaselines,ob,bl
common LocalPearl,star,pearl_channel
;
if n_elements(lmin) eq 0 then lmin=100.		; [nm]
if n_elements(lmax) eq 0 then lmax=100000.	; [nm]
flag=0
if lmin lt 100 then flag=1
if lmax lt 100 then flag=1
if flag then begin
	print,'Error: lmin and lmax should be in [nm]!'
	return
endif
lmin=float(lmin)
lmax=float(lmax)
if lmax lt lmin then begin
	lmin=lmin-lmax/2
	lmax=lmin+lmax
endif
;
if n_elements(starid) eq 0 then begin
	ustars=unique(scans.starid)
	if n_elements(ustars) eq 1 then begin
		starid=ustars(0)
	endif else begin
        	print,'***Error(PEARLDATA): you have to specify a star!'
		status=-1
        	return
	endelse
endif
star=strupcase(starid)
;
for night=0,n_elements(bufferinfo)-1 > 0 do begin
if n_elements(bufferinfo) gt 1 then $
loadnight,GenInfo(night(0)).date, $
	  GeoInfo(night(0)).systemid, $
	  GenInfo(night(0)).configid
if night eq 0 then stationids=genconfig.stationid(0:genconfig.numsid-1) $
	      else stationids=[stationids, $
			      genconfig.stationid(0:genconfig.numsid-1)]
endfor
if n_elements(bufferinfo) gt 1 then $
stationids=unique(stationids)
;
nsi=0	; scan counter
;
FOR night=0,n_elements(bufferinfo)-1 > 0 DO BEGIN
;
nci=1	; channel counter
;
if n_elements(bufferinfo) gt 1 then $
loadnight,GenInfo(night(0)).date, $
	  GeoInfo(night(0)).systemid, $
	  GenInfo(night(0)).configid
;
index=where(scans.starid eq star and scantable.code eq 1,count)
if count eq 0 then begin
        print,'***Error(PEARLDATA): no scans on this star for this date!'
	status=-1
        return
endif
;
; These are reformed into flat arrays
ov=reform(scans(index).complexvis,n_elements(scans(index).complexvis))
ow=reform(scans(index).complexweight,n_elements(scans(index).complexweight))
uc=reform(scans(index).uvw(*,*,*,0),n_elements(scans(index).uvw(*,*,*,0)))
vc=reform(scans(index).uvw(*,*,*,1),n_elements(scans(index).uvw(*,*,*,1)))
ow(*)=1
;
; Create scan index, has same dimensions as ov, i.e. si(ob,ch,bl,sc)
; si=fix(scans(index).complexvis < 10) ; Removed 2020, may produce underflow
si=fix(scans(index).complexvis)
for i=0,count-1 do si(*,*,*,i)=i+1+nsi
si=reform(si,n_elements(si))
nsi=nsi+count
;
; Create baseline index, has same dimensions as ov, i.e. bi(ob,ch,bl,sc)
baselineid=intarr(genconfig.numoutbeam,max(genconfig.numbaseline))
ob_index=baselineid
bl_index=baselineid
for i=0,genconfig.numoutbeam-1 do begin
for l=0,genconfig.numbaseline(i)-1 do begin
        m=where(stationids eq strmid(genconfig.baselineid(l,i),0,3))+1
        n=where(stationids eq strmid(genconfig.baselineid(l,i),4,3))+1
        baselineid(i,l)=1000*m+n
	ob_index(i,l)=i
	bl_index(i,l)=l
endfor
endfor
; bi=fix(scans(index).complexvis < 10) ; Removed 2020, may produce underflow
bi=fix(scans(index).complexvis)
ob=bi
bl=bi
for j=0,max(genconfig.numspecchan)-1 do begin
	for i=0,count-1 do bi(*,j,*,i)=baselineid
	for i=0,count-1 do ob(*,j,*,i)=ob_index
	for i=0,count-1 do bl(*,j,*,i)=bl_index
endfor
bi=reform(bi,n_elements(bi))
ob=reform(ob,n_elements(ob))
bl=reform(bl,n_elements(bl))
;
; Create channel index, has same dimensions as ov, i.e. ci(ob,ch,bl,sc)
; ci=fix(scans(index).complexvis < 10) ; Removed 2020, may produce underflow
ci=fix(scans(index).complexvis)+1
;
; Not all scans have photometry computed
ti=3
if n_elements(size(scans(index).photometryc(0,0,*),/dim)) eq 2 then ti=2
fi=where(total(scans(index).photometryc(0,0,*),ti) gt 0)
spectrometers=strarr(genconfig.numoutbeam)
w=genconfig.wavelength	; save wavelength info
if system_id(systemid) eq 'NPOI' then begin
; 	Assume same channel layout for NPOI!
	mean_w=total(w,2)/genconfig.numoutbeam
	for i=0,genconfig.numoutbeam-1 do genconfig.wavelength(*,i)=mean_w
endif
for i=0,genconfig.numoutbeam-1 do $
	spectrometers(i)=strjoin(string( $
		genconfig.wavelength(0:genconfig.numspecchan(i)-1,i)*1e6))
uspms=unique(spectrometers)
;
for i=0L,n_elements(uspms)-1 do begin
	k=where(spectrometers eq uspms(i)) ; k=output beam index
	wl=genconfig.wavelength(0:genconfig.numspecchan(i)-1,k(0))*1e9
	bw=genconfig.chanwidth(0:genconfig.numspecchan(i)-1,k(0))*1e9
;	Get (calibrated) SED from telescope photometry
	fl=reform( $
	    scans(index(fi(0))).photometryc(0,k,0:genconfig.numspecchan(i)-1))
	for j=1,genconfig.numsid-1 do fl=fl+reform( $
	    scans(index(fi(0))).photometryc(j,k,0:genconfig.numspecchan(i)-1))
;	Smooth the SED (needs more work here!)
	n_sample=max([genconfig.numspecchan(i)/200,1])
	if n_sample gt 1 then fl=median(fl,n_sample)
;	Make sure the SED is not identical to zero
	if total(fl) eq 0 then fl=blackbody(5000.,wl*1e-9)
	if i eq 0 then wl0=wl else wl0=[wl0,wl]
	if i eq 0 then bw0=bw else bw0=[bw0,bw]
	if i eq 0 then fl0=fl else fl0=[fl0,fl]
;	Channel 0 reserved for white light: for-loop starts with j=1
	for j=0L,genconfig.numspecchan(i)-1 do ci(k,j,*,*)=j+nci
	nci=nci+genconfig.numspecchan(i)
endfor
genconfig.wavelength=w
ci=reform(ci,n_elements(ci))
;
; Select only data within requested wavelength range
jndex=where(wl lt lmin or wl gt lmax,count)
if count gt 0 then begin
 	index=whereequal(ci-1,jndex)
	ow(index)=0
endif	
;
; Edit the uv-coverage
index=where(ow gt 0,count)
if count eq 0 then begin
        print,'***Error(PEARLDATA): no data for this star!'
	status=-1
        return
endif
ov=ov(index)
ow=ow(index)
uc=float(uc(index))
vc=float(vc(index))
si=si(index)
bi=bi(index)
ci=ci(index)
ob=ob(index)
bl=bl(index)
;
if night eq 0 then begin
	ov_all=ov
	ow_all=ow
	uc_all=uc
	vc_all=vc
	si_all=si
	bi_all=bi
	ci_all=ci
	ob_all=ob
	bl_all=bl
	wl_all=wl0
	bw_all=bw0
	fl_all=fl0
endif else begin
	ov_all=[ov_all,ov]
	ow_all=[ow_all,ow]
	uc_all=[uc_all,uc]
	vc_all=[vc_all,vc]
	si_all=[si_all,si]
	bi_all=[bi_all,bi]
	ci_all=[ci_all,ci]
	ob_all=[ob_all,ob]
	bl_all=[bl_all,bl]
	if n_elements(unique([wl_all,wl0])) ne n_elements(wl0) then begin
	print,'Warning: spectrometer wavelength differ [nm]: ', $
		max(abs(wl0-wl_all))
	endif
;	The following 3 lines are commented as the setups should be the same
;	wl_all=[wl_all,wl0]
;	bw_all=[bw_all,bw0]
;	fl_all=[bw_all,fl0]
endelse
;
ENDFOR
;
ov=ov_all
ow=ow_all
uc=uc_all
vc=vc_all
si=si_all
bi=bi_all
ci=ci_all
ob=ob_all
bl=bl_all
;
wl=wl_all
fl=fl_all
bw=bw_all
;
cv=ov
mv=ov*0
;
; We need to remove channel indices without data
scan_index=unique(si)
scan_count=n_elements(scan_index)
for i=1,scan_count do begin
	index=where(si eq i)
	uci=unique(ci)
	while n_elements(uci) lt max(uci) do begin
	for j=1,n_elements(uci) do begin
		if uci(j-1) gt j then begin
			index=indgen(n_elements(wl))+1
			k=where(index ne j)
			wl=wl(k)
			fl=fl(k)
			bw=bw(k)
			k=where(ci gt j)
			ci(k)=ci(k)-1
			uci=unique(ci)
			break
		endif
	endfor
	endwhile
endfor
;
wl=wl(0:max(uci)-1)
fl=fl(0:max(uci)-1)
bw=bw(0:max(uci)-1)
;
status=0
if max(abs(ov)) eq 0 then begin
        print,'***Error(PEARLDATA): no visibility data!'
	print,'(Did you not run set_complexvis before?)'
	status=-1
        return
endif
;
end
;-------------------------------------------------------------------------------
pro pearlmodel,vis,flux,mv_in=mv_in,cv_in=cv_in,night=n
;
; Interface for calcmodel to pass complex model visibilities for
; self-calibration and component subtraction. The two options 
; cannot be used together.
;
; Option /mv: initialize mv (pearl) to vis, must then run selfcal in Pearl
; Option /cv: must specify flux, convert cv (pearl) and mv to corr. fluxes,
;	      then subtract vis*flux from cv*flux and re-normalize.
;	      This option is useful for the analysis of triple stars.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
;
if keyword_set(mv_in) then begin
	mv=vis
	print,'Please run selfcal!'
endif
if keyword_set(cv_in) then begin
	print,'Subtracting model...'
	uci=unique(ci)
;	Convert to correlated fluxes
	for j=0,n_elements(uci)-1 do begin
		index=where(ci eq uci(j))
		cv(index)=cv(index)*fl(j)
		vis(index)=vis(index)*flux(j)
	endfor
;	Subtract correlated flux from model
	cv=cv-vis
	fl=fl-flux
;	Re-normalize visibilities
	for j=0,n_elements(uci)-1 do begin
		index=where(ci eq uci(j))
		cv(index)=cv(index)/fl(j)
	endfor
	if n_elements(dm) ne 0 then begin
		print,'Computing new dirty map...'
		dirtymap
		tvrm
		tvcm
	endif
endif
;
end
;************************************************************************Block 2
pro blackbody_pro,x,a,f
;
; x=wavelength [m]
; a[0]=teff, a[1]=scaling factor
; f=flux on output
;
if a(0) lt 100 then f=x*0 $
	       else f=((1.191d-22/x^5)/(exp(1.439d-2/(x*a(0)))-1))*a(1)
;
end
;-------------------------------------------------------------------------------
pro teffmap
;
; Derive new effective temperatures for each non-zero image pixel.
; Multiply each channel image by the SED so that the total flux is the SED.
;
common PearlData,ov,ow,uc,vc,cv,mv,si,bi,ci,wl,bw,fl,db,cb,dm,rm,cm,cc,rt,rg,rf
common ScanData,scans,bgscans,bufferinfo,positions,velocities,magnitudes
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if total(cm.maps.map) eq 0 then return
;
; Initialize Teff map
teff0=median(cm.teff)
cm.teff=teff0
;
print,'Computing new effective temperature map...',format='(a,$)'
;
fm=cm
nc=n_elements(fm.maps)-1
; Average clean beam
cbt=total(cb.maps.map,3)/nc
; Largest clean beam
j=where(wl eq max(wl)) & j=j(0)
cbt=cb.maps(j+1).map
;
; It is important to convolve all channels maps with the same CB
for ch=1,nc do begin
	fm.maps(ch).map=convol(cm.maps(ch).map,cbt,/edge_wrap,/center)
	tfm=total(fm.maps(ch).map)
	fm.maps(ch).map=fm.maps(ch).map/tfm*fl(ch-1)
endfor
;
; Find the indices of clean components in the combined map
tfm=total(fm.maps.map,3)
index=where(tfm gt max(tfm)*0.01,count)
;
teff=fltarr(count)
x=wl*1e-9
tgran=50.0
;
; Weighting vector to give more weight to the edges
w=cos(abs(x-x(0))/(max(x)-min(x))*2*!pi)+2
;
; wset,34	; is that needed?
for k=0,count-1 do begin
	ij=whereindex(index(k),tfm)
	y=fm.maps(1:nc).map(ij(0),ij(1))
	z=where(y eq 0,count)
	if count eq 0 then begin
		z=where(y eq max(y))
		a=[wien(x(z(0))),1]
		blackbody_pro,x,a,f
		a(1)=median(y/f)
		r=curvefit(x,y,w,a,function_name='blackbody_pro', $
			/noderiv,itmax=100,status=s)
		teff(k)=nint(a(0)/tgran)*tgran
	endif else begin
		teff(k)=teff0
	endelse
endfor
cm.teff(index)=teff
;
; Apply median filter over half the kernel patch
r=size(cbt)
r=fix(sqrt(r(1)*r(2))/2)
cm.teff=median(cm.teff,r)
cc.teff=cm.teff
;
; (Re)Initialize rt and rf
pearlinit,cm.teff
;
print,'done.'
;
end
;************************************************************************Block 3
function set_uvc,imsze,cellsze,nc,relax=relax
;
; Return a uv-coverage of dimension imsize(0)*imsize(1), i.e. column by row,
; width by height, and cellsize in Mlambda.
;
; Both dimensions must be odd numbers, so that there will be a pixel on the
; center of the image. The bottom left corner is at pixel (0,0), the top right
; corner is (imsize(0)-1,imsize(1)-1)
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
MAS2RAD=pi_circle/(180l*3600000l)
;
if n_elements(relax) eq 0 then relax=0
;
if not relax then begin
if imsze(0) mod 2 ne 1 then begin
        print,'***Error(SET_UVC): imsize(0) must be odd number!'
        return,-1
endif
if imsze(1) mod 2 ne 1 then begin
        print,'***Error(SET_UVC): imsize(1) must be odd number!'
        return,-1
endif
endif
if n_elements(nc) eq 0 then nc=1
;
imsize=long(imsze)
cellsize=float(cellsze)
;
map=complexarr(imsize(0),imsize(1),nc)
u=(lindgen(imsize(0)*imsize(1)) mod imsize(0)) - (imsize(0)/2)
v=(lindgen(imsize(0)*imsize(1)) / imsize(0)) - (imsize(1)/2)
u=-u*cellsize
v=+v*cellsize
;
return,{map:map,u:u,v:v}
;
end
;-------------------------------------------------------------------------------
function pearson_funct,x,a
;
return,a(0)/(1.+((2.*x*sqrt(2.^(1./a(1))-1))/a(2))^2)^a(1)
;
end
;-------------------------------------------------------------------------------
pro pearson_pro,x,a,f
;
f=a(0)/(1.+((2.*x*sqrt(2.^(1./a(1))-1))/a(2))^2)^a(1)
;
end
;-------------------------------------------------------------------------------
pro pearson
;
nuv=61L
uvc=set_uvc([nuv,nuv],1.0)
uvc.map=sqrt(exp(-0.40*sqrt(uvc.u^2+uvc.v^2)))
c=float(uvc.map)
nim=127L
map=pearlimage([nim,nim],2.0d0)
uvc.u=uvc.u*1e6
uvc.v=uvc.v*1e6
img=dft(map,reform(uvc.map,nuv*nuv),uvc.u,uvc.v)
m=img.map
x=findgen(nim)-nim/2
y=m(*,nim/2)
ym=max(y)
index=where(y gt ym/2,fwhm)
a=[max(y),1.50,40.0]
a=[ym,1.5,fwhm]
xc=x
yc=y
r=curvefit(xc,yc,yc*0+1,a,function_name='pearson_pro',/noderiv)
!p.multi=[0,1,2]
plot,findgen(nuv)-nuv/2,c(*,nuv/2),psym=0
plot,x,y
oplot,x,pearson_funct(x,a),psym=0
print,a
;
; Check the back transformation
imsize=[nim,nim]
x=(lindgen(imsize(0)*imsize(1)) mod imsize(0)) - (imsize(0)/2)
y=(lindgen(imsize(0)*imsize(1)) / imsize(0)) - (imsize(1)/2)
img.map=pearson_funct(sqrt(float(x)^2+float(y)^2),a)
img.teff=1
t=mft(img,reform(uvc.map*0,nuv*nuv),uvc.u,uvc.v,1,1,lonarr(nuv*nuv)+1)
t=reform(t,nuv,nuv)
window,/free
!p.multi=0
y1=float(uvc.map(*,nuv/2))
plot,y1,psym=0
y2=float(t(*,nuv/2))
f=median(y1/y2)
oplot,y2*f,psym=1
;
end
;-------------------------------------------------------------------------------
