;*******************************************************************************
; File: stars.pro
;
; System: NPOI Data Reduction
; Subsystem: CHAMELEON, AMOEBA, STARBASE
;
; Description:
; ------------
; Contains procedures to estimates stellar parameters from calibrations
; based on spectral type. The naming general naming scheme is "result"_"input"
;
; Block directory:
; ----------------
; Block 1: khh_mass,lum_mass,rad_mass,teff_mass,logg_mass,lum_teff,rad_teff,
;	   rad_rsun,rad_vsini,radius_dwarf,vcrit_mr,
;	   teff_vmk,vmk_teff,bc_teff,teff_bv,mass_vcp
; Block 2: ft_dyck,ft_gerard,ft_tabby,
;	   dwarfs,subgiants,giants,brightgiants,supergiants,
;	   Iasupergiants,Iabsupergiants,Ibsupergiants, wolfrayets
; Block 3: mass_dwarf,.._subgiant,.._giant,.._brightgiant,
;	   .._supergiant,
;          logg_dwarf,.._subgiant,.._giant,.._brightgiant,
;	   .._Iasupergiant,.._Iabsupergiant,.._Ibsupergiant
;	   bv_dwarf,bv_giant,bv_supergiant,
;	   teff_dwarf,teff_giant,teff_supergiant,
;	   diam_dwarf,diam_giant,diam_supergiant
; Block 4: mass_star,mass_stars,mass_binaries,mass2_mf,
;          logg_star,logg_stars,
;	   teff_star,teff_stars,
;	   diam_star,diam_stars,
;	   diam_binaries,
;	   bv_star,bv_stars,
; Block 5: amv_dwarf,amv_subgiant,amv_giant,amv_brightgiant,amv_supergiant,
;	   amv_star,amv_stars,amv_dwarfs,mk_dwarfs
;	   limb_star,limb_stars,
;	   av_stars
;
;************************************************************************Block 1
function khh_mass,mass
;
; Return Kelvin-Helmholtz timescale (years) for stars to contract before 
; reaching the MS. Input mass in solar units.
;
m=[7.,1.,0.5]
t=[60e3,60e6,150e6]
y=alog10(t)
r=poly_fit(m,y,2,yfit)
;
return,10^poly(mass,r)
;
end
;-------------------------------------------------------------------------------
function lum_mass,mass
;
; Use mass-luminosity relationship for main-sequence stars to return
; luminosity (in solar units) from mass. Input data from Torres et al. 2010
; Use, e.g., l=lum_mass() to get a plot of the Torres et al. data and fit.
;
; Luminosity in solar units.
;
; New version using data from Torres et al. 2010
status=dc_read_free(!oyster_dir+'starbase/torres.dat', $
	m,em,rad,erad,teff,eteff,logg,elogg,logl,elogl,mv,emv, $
	/col,ignore=['!'])
; Exclude slightly evolved stars with log(g) < 4.3/4.1 for mass <10/>10
index=where((m le 10 and logg gt 4.3) or (m gt 10 and logg gt 4.1))
r=poly_fit(alog10(m(index)),logl(index),4,yfit)
if n_elements(mass) eq 0 then begin
	plot,alog10(m),logl,psym=1,title='Data from Torres et al. 2010', $
		xtitle='log(mass)',ytitle='log(luminosity)',charsize=1.5
	oplot,alog10(m(index)),logl(index),psym=1,color=tci(3)
	oplot,alog10(m(index)),yfit,psym=0,color=tci(3)
;	Overplot numerical relation from Allen's A. Q., p. 382
	lm=findgen(100)/50-0.5
	ll=3.8*lm+0.08
	oplot,lm,ll,psym=0,color=tci(1)
	return,0
endif else $
return,10^poly(alog10(mass),r)
;
; Old version using data from Andersen 1991
status=dc_read_free(!oyster_dir+'starbase/andersen.dat',m,lm,bv,mv,ll,lteff, $
	/col,ignore=['!'])
r=poly_fit(lm,ll,2,yfit)
return,10^poly(alog10(mass),r)
;
; From Table 4 of Eker et al. 2015
lm=alog10(mass)
logl=-0.705*lm^2+4.655*lm-0.025
return,10^logl
;
; From Allen's Astrophysical Quantities
n=60
logm=-1.0+2.5*findgen(n)/(n-1)
llum=3.8*logm+0.08
;
end
;-------------------------------------------------------------------------------
function rad_mass,mass
;
; Use mass-radius relationship for main-sequence stars to predict
; effective radius (solar) from mass. Input data from Torres et al. 2010
;
; Radius in solar units.
;
; New version using data from Torres et al. 2010
status=dc_read_free(!oyster_dir+'starbase/torres.dat', $
	m,em,rad,erad,teff,eteff,logg,elogg,logl,elogl,mv,emv, $
	/col,ignore=['!'])
; Exclude slightly evolved stars with log(g) < 4.3/4.1 for mass <10/>10
index=where((m le 10 and logg gt 4.3) or (m gt 10 and logg gt 4.1))
r=poly_fit(alog10(m(index)),rad(index),4,yfit)
if n_elements(mass) eq 0 then begin
	plot,alog10(m),rad,psym=1,title='Data from Torres et al. 2010', $
		xtitle='log(mass)',ytitle='Radius',charsize=1.5
	oplot,alog10(m(index)),rad(index),psym=1,color=tci(3)
	oplot,alog10(m(index)),yfit,psym=0,color=tci(3)
;	Overplot numerical relation from Allen's A. Q., p. 382
	lm=findgen(100)/50-0.5
	lr=fltarr(100)
	index=where(lm gt 0.12)
	lr(index)=0.640*lm(index)+0.011
	index=where(lm le 0.12)
	lr(index)=0.917*lm(index)-0.020
	oplot,lm,10^lr,psym=0,color=tci(1)
	return,0
endif else $
return,poly(alog10(mass),r)
;
end
;-------------------------------------------------------------------------------
function teff_mass,mass
;
; Use mass-effective temperature relationship for main-sequence stars to predict
; effective temperature (K) from mass. Input data from Torres et al. 2010.
;
; The Sun is 5777 K, but we get 5454 K...?
;
; New version using data from Torres et al. 2010
status=dc_read_free(!oyster_dir+'starbase/torres.dat', $
	m,em,rad,erad,teff,eteff,logg,elogg,logl,elogl,mv,emv, $
	/col,ignore=['!'])
; Exclude slightly evolved stars with log(g) < 4.3/4.1 for mass <10/>10
index=where((m le 10 and logg gt 4.3) or (m gt 10 and logg gt 4.1))
r=poly_fit(alog10(m(index)),alog10(teff(index)),3,yfit)
if n_elements(mass) eq 0 then begin
	plot,alog10(m),alog10(teff),psym=1,yrange=[3,5], $
		title='Data from Torres et al. 2010', $
		xtitle='log(mass)',ytitle='log(Teff)',charsize=1.5
	oplot,alog10(m(index)),alog10(teff(index)),psym=1,color=tci(3)
	oplot,alog10(m(index)),yfit,psym=0,color=tci(3)
;	Overplot numerical relation from Allen's A. Q., p. 382
	lm=findgen(100)/50-0.5
	lr=fltarr(100)
	index=where(lm gt 0.12)
	lr(index)=0.640*lm(index)+0.011
	index=where(lm le 0.12)
	lr(index)=0.917*lm(index)-0.020
	ll=3.8*lm+0.08
	ltf=(ll+4*alog10(5777)-2*lr)/4
	oplot,lm,ltf,psym=0,color=tci(1)
	return,0
endif else $
return,10^poly(alog10(mass),r)
;
end
;-------------------------------------------------------------------------------
function logg_mass,mass,radius
;
; Return log(g) computed from mass and radius of a star (solar units on input).
;
return,alog10(mass)-2*alog10(radius)+4.44
;
end
;-------------------------------------------------------------------------------
function lum_teff,teff,radius
;
; Compute luminosity from eff. temperature and radius (Stefan-Boltzmann law),
; based on Allens's Astrophysical Quantities, p. 382, if radius specified,
; else on Boyajian et al. 2012 (equation 6), MS only!, valid for 3200 < T < 5500
; Solar units for radius. Return in solar luminosities.
;
if n_elements(radius) ne 0 then begin
;	Allens's Astrophysical Quantities, p. 382
	return,10^(2*alog10(radius)+4*alog10(teff)-4*alog10(5777))
endif else begin
;	Boyajian et al. 2012 (equation 6), valid for 3200 < T < 5500
	index=where(teff lt 3200 or teff gt 5500,count)
	if count gt 0 then $
		print,'Warning: this function only valid for 3200< T< 5500'
	logt=alog10(teff)
;
	lum=-5960.5710+4831.6912*logt-1306.9966*logt^2+117.9716*logt^3
	return,10^lum
endelse
;
end
;-------------------------------------------------------------------------------
function rad_teff,teff
;
; Return main-sequence radius computed as a function of effective temperature.
; Based on Boyajian et al. 2017 (erratum), equation 8 valid for 3200 < T < 5500 
; and equation 9 for more massive stars. Solar units.
;
t=float(teff)
;
index=where(t lt 3200,count)
if count gt 0 then print,'Warning: this function only valid for 3200 < T'
;
r=teff
;
index=where(t ge 3200 and t le 5500,count)
if count gt 0 then r(index)=-8.133+5.09342e-3*t-9.86602e-7*t^2+6.47963e-11*t^3
;
index=where(t gt 5500,count)
if count gt 0 then r(index)=-10.8828+7.18727e-3*t-1.50957e-6*t^2+1.07572e-10*t^3
;
return,r
;
end
;-------------------------------------------------------------------------------
function rad_rsun,r_sol,d_pc
;
; Compute stellar radius in mas from input in solar units and distance in pc.
; 1 pc = 206266.3 AU; the diameter of the Sun is about 0.01 AU.
; 1 pc = distance where 1 AU appears under an angle of 1".
;
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
RAD=180.d0/!pi
;
; Convert r_sol to AU
r_au=r_sol*r_sun/a_unit
;
; Convert r_au to pc (Jean Meeus, Astronomical Algorithms, p.366)
r_pc=r_au/206265.d0
;
; Convert r_sol[pc] at distance d_pc to angle in mas
r_mas=(atan(r_pc/d_pc)*RAD)*3600.d0*1000.d0
;
return,r_mas
;
end
;-------------------------------------------------------------------------------
function rad_vsini,vsini,i,period
;
; Compute stellar radius in solar units from v*sin(i), i, and period
; vsini [km/s], incl [deg], period [d]
;
common AstroConstants,m_sun,r_sun,a_unit,s_year
;
rad=180/!pi
p=period*86400/(2*!pi)
;
return,(vsini/sin(i/rad))*(p/(r_sun/1000))
;
end
;-------------------------------------------------------------------------------
function radius_dwarf,spectrum
;
; Compute stellar radius using the log(M)-log(R) diagram, 
; based on Torres et al. 2010
;
; Measured coordinates
coords=[ $
[187,375], [209,359], [225,351], [245,340], [293,310], [328,290], [341,285], $
[350,282], [359,281], [362,277], [373,267], [380,261], [394,245], [413,234], $
[432,214], [437,208], [473,182]] 
coords_refp=[[178,133], [519,387]]
coords_refd=[[1.5,-1.0], [-1.0,1.0]]
scale_m=(coords_refp(*,1)-coords_refp(*,0)) $
       /(coords_refd(*,1)-coords_refd(*,0))
n=17
lmass=fltarr(n)
lradi=fltarr(n)
for i=0,n-1 do begin
	lmass(i)=(coords(0,i)-coords_refp(0,0))/scale_m(0)+coords_refd(0,0)
	lradi(i)=(coords(1,i)-coords_refp(1,0))/scale_m(1)+coords_refd(1,0)
endfor
;
spec_parse,spectrum,type,class,type2,class2
lmasses=alog10(mass_dwarf(type))
return,10^interpol(lradi,lmass,lmasses)
;
end
;-------------------------------------------------------------------------------
function vcrit_mr,mass,radius_p
;
; Compute the critical (i.e., break-up) velocity of a star (km/s).
; See: Townsend, Owocki, and Howarth 2004, MNRAS 350, 189, Table 1
; Input mass & radius in solar units. Radius is the polar radius.
;
G=191e3
return,sqrt(2*G*mass/(3*radius_p))
;
end
;-------------------------------------------------------------------------------
function teff_vmk,vmk
;
; Compute teff=f(V-K) from Mozurkewich et al. 2003.
;
; Equation 4
lteff=3.972-0.176*vmk+0.024*vmk^2-0.0013*vmk^3
;
min=-0.5 ; Minimum allowable (V-K)
index=where(vmk lt min,count)
count=0 ; Do not limit at this time
if count gt 0 then begin
	ltmax=3.972-0.176*min+0.024*min^2-0.0013*min^3
	print,'Warning(TEFF_VMK): results not reliable for (V-K) < -0.25!'
	lteff(index)=ltmax
	print,'Maximum T_eff limited to: ',10^ltmax
endif
;
return,10^lteff
;
end
;-------------------------------------------------------------------------------
function vmk_teff,teff_in
;
; Compute (V-K)=f(teff) from Mozurkewich et al. 2003. Only for T <= 15000 K
;
; Invert fit from Mozurkewich et al. 2003
vmk=findgen(9)-1
teff=teff_vmk(vmk)
r=poly_fit(teff,vmk,5)
; plot,teff,vmk,psym=1
; oplot,teff,poly(teff,r)
;
teff=teff_in
limit=10000.
index=where(teff ge limit,count)
; count=0 ; Do not limit at this time
if count gt 0 then begin
	print,'Warning(VMK_TEFF): results not reliable for T > 10000 K!'
	teff(index)=limit
endif
;
return,poly(teff,r)
;
end
;-------------------------------------------------------------------------------
pro gordon
;
; Related to function teff_vmk in that it computes a new set of coefficients
; by including the work by Gordon et al. 2018 on O-star colors and T_eff's.
;
; Not currently in use to derive T_eff from V-K (flat T dependence of V-K)
;
status=dc_read_free(!oyster_dir+'starbase/vmk_teff.dat', $
	HR,dots,constellation,v,vr,vk,flux,SpT,LC,t,e,/col,ignore=['!'])
t=double(t)	; Teff
e=double(e)	; sigma(Teff)
;
si=sort(t)
si=sort(vk)
plot,t(si),vk(si),psym=4,ytitle='(V-K)',xtitle='T_eff [K]'
oplot,teff_vmk(vk(si)),vk(si),psym=0
;
r=poly_fit(vk,t,3)
;
end
;-------------------------------------------------------------------------------
function bc_teff,teff
;
; Returns bolometric correction bc (m_bol=m+bc) according to polynomial fits
; given in Flower, 1996 and confirmed by Torres 2010.
;
teff=float(teff)
lte=alog10(teff)
;
c1=[-0.190537291496456D+05, $
     0.155144866764412D+05, $
    -0.421278819301717D+04, $
     0.381476328422343D+03]
c2=[-0.370510203809015D+05, $
     0.385672629965804D+05, $
    -0.150651486316025D+05, $
     0.261724637119416D+04, $
    -0.170623810323864D+03]
c3=[-0.118115450538963D+06, $
     0.137145973583929D+06, $
    -0.636233812100225D+05, $
     0.147412923562646D+05, $
    -0.170587278406872D+04, $
     0.788731721804990D+02]
;
i1=where(lte lt 3.7,n1)
i2=where(lte ge 3.7 and lte le 3.9,n2)
i3=where(lte gt 3.9,n3)
;
bcorr=teff
if n1 gt 0 then bcorr(i1)=poly(lte(i1),c1)
if n2 gt 0 then bcorr(i2)=poly(lte(i2),c2)
if n3 gt 0 then bcorr(i3)=poly(lte(i3),c3)
;
return,bcorr
;
end
;-------------------------------------------------------------------------------
function teff_bv,bv,sg=sg
;
; Returns Teff as a function of (B-V) according to polynomial fits
; given in Flower, 1996 (Table 5), corrected by Torres 2010 (Table 2).
;
; For main-sequence stars, use the newer coefficients of Boyajian et al. 2013
; for spectral types later then A0.
;
; For supergiants, use the Flower & Torres polynomial.
;
if n_elements(sg) eq 0 then sg=0
;
bv=double(bv)
logt=bv*0
;
; Coefficients for AFGK dwarfs (Boyajian et al. 2013)
c=[9552,-17443,44350,-68940,57338,-24072,4009] & c=double(c)
index=where(bv ge 0,count)
if count gt 0 then logt(index)=alog10(poly(bv(index),c))
;
; Coefficients for dwarfs, subgiants, and giants (Flower 1996, Torres 2010)
c=[3.979145,-0.654499,1.740690,-4.608815,6.792600,-5.396910,2.192970,-0.359496]
c=double(c)
index=where(bv lt 0,count)
if count gt 0 then logt(index)=poly(bv(index),c)
;
if not sg then return,10^logt
;
; Coefficients for supergiants
if sg then $
c=[4.0125597,-1.055043,2.133395,-2.459770,1.349424,-0.283943]
c=double(c)
logt=poly(bv,c)
;
return,10^logt
;
end
;-------------------------------------------------------------------------------
function mass_vcp,mv,bv,px
;
; Estimate mass from proximity of star to stellar tracks in Teff/Lum plane.
; Tracks are taken from Schaller et al. 1992 models.
;
; mv=V, bv=(B-V), px=parallax ["]
;
amv=mv-(-5+5*alog10(1./px))
teff=teff_bv(bv)
mbol=amv+bc_teff(teff)
llum=-(mbol-4.74)/2.5
ltff=alog10(teff)
;
restore,!oyster_dir+'starbase/tracks.xdr'
logl=float(logl)
logt=float(logt)
;
d=sqrt((logl-llum)^2+(logt-ltff)^2)
index=where(d eq min(d))
print,'Derived log(lum), log(Teff):',llum,ltff
print,'Track   log(lum), log(Teff):',logl(index(0)),logt(index(0))
;
return,mass(index)
;
end
;-------------------------------------------------------------------------------
pro tracks
;
; Prepare tracks.xdr for mass_vcp function. Run in geneva/Schaller92/Z020.
;
files='table'+strcompress(string(indgen(19)+4),/remove_all)
;
status=dc_read_free(files(0),n,a,m,l,t,/col)
;
mass=m
logl=l
logt=t
;
for i=1,n_elements(files)-1 do begin
	status=dc_read_free(files(i),n,a,m,l,t,/col)
	mass=[mass,m]
	logl=[logl,l]
	logt=[logt,t]
endfor
;
save,mass,logl,logt,filename='tracks.xdr'
;
end
;************************************************************************Block 2
function ft_dyck,type
;
; Return floating spectral type defined by M. Dyck (AJ 116, 981),
; ft = -2, ..., 0, ..., 5, 6, ..., 8 for G8, ..., K0, ..., K5, ..., M0, ..., M8
;
; Input type as used by OYSTER: O0:00, B0:10, A0:20, F0:30, G0:40, K0:50, M0:60
;
st=float([-2,0,5,6,8])
ot=float([48,50,55,60,68])
;
return,interpol(st,ot,type)
;
end
;-------------------------------------------------------------------------------
function ft_gerard,type
;
; Return floating spectral type defined by van Belle et al. 1999 (AJ 117, 521),
; ft = -2, ..., 0, ..., 5, 6, ..., 14 for G8, ..., K0, ..., K5, ..., M0, ..., M8
;
; Input type as used by OYSTER: O0:00, B0:10, A0:20, F0:30, G0:40, K0:50, M0:60
;
st=float([-2,0,5,6,14])
ot=float([48,50,55,60,68])
;
return,interpol(st,ot,type)
;
end
;-------------------------------------------------------------------------------
function ft_tabby,type
;
; Return floating spectral type defined by Boyajan et al. 2012 (ApJ 746, 101),
; ft = 0,  1,  2, ..., 10, 11, 12, ..., 20, 21, 22, ..., 30
; for A0, A1, A2, ..., F0, F1, F2, ..., G0, G1, G2, ..., K0
;
; Input type as used by OYSTER: O0:00, B0:10, A0:20, F0:30, G0:40, K0:50, M0:60
;
st=float([ 0, 1, 2,10,11,12,20,21,22,30])
ot=float([20,21,22,30,31,32,40,41,42,50])
;
return,interpol(st,ot,type)
;
end
;-------------------------------------------------------------------------------
function dwarfs,class,type,extended
;
; Return index of dwarfs.
;
if n_elements(extended) eq 0 then extended=0 else extended=extended gt 0
;
if extended then return,where(class ge 4.5 and type ne 0) $
	    else return,where(fix(class) eq 5 and type ne 0)
;
end
;-------------------------------------------------------------------------------
function subgiants,class,type
;
; Return index of sub giants.
;
return,where(nint(class) eq 4 and type ne 0)
;
end
;-------------------------------------------------------------------------------
function giants,class,type,extended
;
; Return index of giants.
;
if n_elements(extended) eq 0 then extended=0 else extended=extended gt 0
;
if extended then return,where(class lt 4.5 and class gt 1.75 and type ne 0) $
	    else return,where(fix(class) eq 3 and type ne 0)
;
end
;-------------------------------------------------------------------------------
function brightgiants,class,type
;
; Return index of bright giants.
;
return,where(nint(class) eq 2 and type ne 0)
;
end
;-------------------------------------------------------------------------------
function supergiants,class,type,extended
;
; Return index of super giants.
;
if n_elements(extended) eq 0 then extended=0 else extended=extended gt 0
;
if extended then return,where(class gt 0 and class le 1.75 and type ne 0) $
	    else return,where(fix(class) eq 1 and type ne 0)
;
end
;-------------------------------------------------------------------------------
function Iasupergiants,class,type
;
return,where(class gt 0 and class lt 1.15 and type ne 0)
;
end
;-------------------------------------------------------------------------------
function Iabsupergiants,class,type
;
return,where(class ge 1.15 and class lt 1.35 and type ne 0)
;
end
;-------------------------------------------------------------------------------
function Ibsupergiants,class,type
;
return,where(class ge 1.35 and class lt 1.75 and type ne 0)
;
end
;-------------------------------------------------------------------------------
function wolfrayets,class,type
;
; Return index of Wolf Rayets.
;
return,where(type lt 0)
;
end
;************************************************************************Block 3
function mass_dwarf,type
;
; Estimate mass for dwarfs from spectral type.
;
common StarBase,startable,notes
;
s=''
m=0.
t=0.
; Schmidt-Kaler 1982
; status=dc_read_free(!oyster_dir+'starbase/spec_mass.V',s,m,t, $
; 	/col,resize=[1,2,3],ignore=['!'])
;
; Masses from Anderson 1991
status=dc_read_free(!oyster_dir+'starbase/masses.V',m,t, $
	/col,resize=[1,2],ignore=['!'])
r=poly_fit(t,alog(m),5)
;
plot=0
if plot then begin
	!p.charsize=1.5
	plot,findgen(70),poly(findgen(70),r),title='Dwarfs', $
		xtitle='Integer Spectral type',ytitle='ln(mass)'
	oplot,t,alog(m),psym=6
endif
;
return,exp(poly(type,r))
;
end
;-------------------------------------------------------------------------------
function mass_subgiant,type
;
; Estimate mass for subgiants from spectral type. Very uncertain!
;
common StarBase,startable,notes
;
s=''
m=0.
t=0.
status=dc_read_free(!oyster_dir+'starbase/spec_mass.IV',s,m,t, $
	/col,resize=[1,2,3],ignore=['!'])
;
; Read this table as the sub-giant masses are more like the dwarfs? (CHU 2017)
status=dc_read_free(!oyster_dir+'starbase/spec_mass.V',s,m,t, $
	/col,resize=[1,2,3],ignore=['!'])
;
r=poly_fit(t,alog(m),5)
;
plot=0
if plot then begin
	!p.charsize=1.5
	plot,findgen(70),poly(findgen(70),r),title='Sub-Giants', $
 		xtitle='Integer Spectral type',ytitle='ln(mass)'
	oplot,t,alog(m),psym=6
endif
;
return,exp(poly(type,r))
;
end
;-------------------------------------------------------------------------------
function mass_giant,type
;
; Estimate mass for giants from spectral type.
;
common StarBase,startable,notes
;
s=''
m1=0.
t1=0.
; Schmidt-Kaler 1982
status=dc_read_free(!oyster_dir+'starbase/spec_mass.III',s,m,t, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,alog(m),3)
;
plot=0
if plot then begin
	!p.charsize=1.5
	plot,findgen(70),poly(findgen(70),r),title='Giants', $
 		xtitle='Integer Spectral type',ytitle='ln(mass)'
	oplot,t,alog(m),psym=6
endif
;
return,exp(poly(type,r))
;
end
;-------------------------------------------------------------------------------
function mass_brightgiant,type
;
; Estimate mass for bright giants from spectral type. Very uncertain!
;
common StarBase,startable,notes
;
s=''
m=0.
t=0.
status=dc_read_free(!oyster_dir+'starbase/spec_mass.II',s,m,t, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,alog(m),4)
;
plot=0
if plot then begin
	!p.charsize=1.5
	plot,findgen(70),poly(findgen(70),r),title='Bright-Giants', $
 		xtitle='Integer Spectral type',ytitle='ln(mass)'
	oplot,t,alog(m),psym=6
endif
;
return,exp(poly(type,r))
;
end
;-------------------------------------------------------------------------------
function mass_supergiant,type
;
; Estimate mass for super giants from spectral type.
;
common StarBase,startable,notes
;
s=''
m=0.
t=0.
; Schmidt-Kaler 1982
status=dc_read_free(!oyster_dir+'starbase/spec_mass.I',s,m,t, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,alog(m),6)
;
plot=0
if plot then begin
	!p.charsize=1.5
	plot,findgen(70),poly(findgen(70),r),title='Super-Giants', $
 		xtitle='Integer Spectral type',ytitle='ln(mass)'
	oplot,t,alog(m),psym=6
endif
;
return,exp(poly(type,r))
;
end
;-------------------------------------------------------------------------------
function logg_dwarf,type
;
; Estimate log(g) [cgs] for dwarfs from spectral type.
; V. Straizys, "Multicolor Stellar Photometry"
;
common StarBase,startable,notes
;
s=''
t=0.
g=0.
status=dc_read_free(!oyster_dir+'starbase/spec_logg.V',s,t,g, $
 	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,g,3)
;
; Plot of log(g)-spectral type relationship
plot=0
if plot then begin
	!p.charsize=1.5
	plot,findgen(70),poly(findgen(70),r),title='Dwarfs', $
 		xtitle='Integer Spectral type',ytitle='log(g) [cgs]'
	oplot,t,g,psym=6
endif
;
return,poly(type,r)
;
end
;-------------------------------------------------------------------------------
function logg_subgiant,type
;
; Estimate log(g) [cgs] for subgiants from spectral type.
; V. Straizys, "Multicolor Stellar Photometry"
;
common StarBase,startable,notes
;
s=''
t=0.
g=0.
status=dc_read_free(!oyster_dir+'starbase/spec_logg.IV',s,t,g, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,g,3)
;
; Plot of log(g)-spectral type relationship
plot=0
if plot then begin
	!p.charsize=1.5
	plot,findgen(70),poly(findgen(70),r),title='Sub-Giants', $
 		xtitle='Integer Spectral type',ytitle='log(g) [cgs]'
	oplot,t,g,psym=6
endif
;
return,poly(type,r)
;
end
;-------------------------------------------------------------------------------
function logg_giant,type
;
; Estimate log(g) [cgs] for giants from spectral type.
; V. Straizys, "Multicolor Stellar Photometry"
;
common StarBase,startable,notes
;
s=''
t=0.
g=0.
status=dc_read_free(!oyster_dir+'starbase/spec_logg.III',s,t,g, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,g,3)
;
; Plot of log(g)-spectral type relationship
plot=0
if plot then begin
	!p.charsize=1.5
	plot,findgen(70),poly(findgen(70),r),title='Giants', $
 		xtitle='Integer Spectral type',ytitle='log(g) [cgs]'
	oplot,t,g,psym=6
endif
;
return,poly(type,r)
;
end
;-------------------------------------------------------------------------------
function logg_brightgiant,type
;
; Estimate log(g) [cgs] for bright giants from spectral type.
; V. Straizys, "Multicolor Stellar Photometry"
;
common StarBase,startable,notes
;
s=''
t=0.
g=0.
status=dc_read_free(!oyster_dir+'starbase/spec_logg.II',s,t,g, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,g,3)
;
; Plot of log(g)-spectral type relationship
plot=0
if plot then begin
	!p.charsize=1.5
	plot,findgen(70),poly(findgen(70),r),title='Bright-Giants', $
 		xtitle='Integer Spectral type',ytitle='log(g) [cgs]'
	oplot,t,g,psym=6
endif
;
return,poly(type,r)
;
end
;-------------------------------------------------------------------------------
function logg_Iasupergiant,type
;
; Estimate log(g) [cgs] for super giants from spectral type.
; V. Straizys, "Multicolor Stellar Photometry"
;
common StarBase,startable,notes
;
; Super giants class Ia
s=''
t=0.
g=0.
status=dc_read_free(!oyster_dir+'starbase/spec_logg.Ia',s,t,g, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,g,3)
;
; Plot of log(g)-spectral type relationship
plot=0
if plot then begin
	!p.charsize=1.5
	window,/free
	plot,findgen(70),poly(findgen(70),r),title='Super-Giants Ia', $
 		xtitle='Integer Spectral type',ytitle='log(g) [cgs]'
	oplot,t,g,psym=6
endif
;
return,poly(type,r)
;
end
;-------------------------------------------------------------------------------
function logg_Iabsupergiant,type
;
; Estimate log(g) [cgs] for super giants from spectral type.
; V. Straizys, "Multicolor Stellar Photometry"
;
common StarBase,startable,notes
;
; Super giants class Iab
s=''
t=0.
g=0.
status=dc_read_free(!oyster_dir+'starbase/spec_logg.Iab',s,t,g, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,g,3)
;
plot=0
if plot then begin
	!p.charsize=1.5
	window,/free
	plot,findgen(70),poly(findgen(70),r),title='Super-Giants Ia', $
 		xtitle='Integer Spectral type',ytitle='log(g) [cgs]'
	oplot,t,g,psym=6
endif
;
return,poly(type,r)
;
end
;-------------------------------------------------------------------------------
function logg_Ibsupergiant,type
;
; Estimate log(g) [cgs] for super giants from spectral type.
; V. Straizys, "Multicolor Stellar Photometry"
;
common StarBase,startable,notes
; Super giants class Ib
s=''
t=0.
g=0.
status=dc_read_free(!oyster_dir+'starbase/spec_logg.Ib',s,t,g, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(t,g,3)
;
plot=0
if plot then begin
	!p.charsize=1.5
	window,/free
	plot,findgen(70),poly(findgen(70),r),title='Super-Giants Ia', $
 		xtitle='Integer Spectral type',ytitle='log(g) [cgs]'
	oplot,t,g,psym=6
endif
;
return,poly(type,r)
;
end
;-------------------------------------------------------------------------------
function bv_dwarf,type
;
; Return (B-V)0 from calibration of MK spectral type (Schmidt-Kaler 1982)
;
status=dc_read_free(!oyster_dir+'starbase/schmidt-kaler.dat', $
			sp,bv5,bv3,bv2,bv1b,bv1ab,bv1a, $
			/col,ignore=['!'])
spec_parse,sp,ft
;
; plot,ft,bv5,psym=1
; ft1=findgen(70)
; oplot,ft1,spline(ft,bv5,ft1)
;
si=sort(type)	; spline requires sorted vectors!
bv=type
bv(si)=spline(ft,bv5,type(si))
return,bv
;
end
;-------------------------------------------------------------------------------
function bv_giant,type
;
; Return (B-V)0 from calibration of MK spectral type (Schmidt-Kaler 1982)
;
status=dc_read_free(!oyster_dir+'starbase/schmidt-kaler.dat', $
			sp,bv5,bv3,bv2,bv1b,bv1ab,bv1a, $
			/col,ignore=['!'])
spec_parse,sp,ft
;
; plot,ft,bv3,psym=1
; ft1=findgen(70)
; oplot,ft1,spline(ft,bv3,ft1)
;
si=sort(type)	; spline requires sorted vectors!
bv=type
bv(si)=spline(ft,bv3,type(si))
return,bv
;
end
;-------------------------------------------------------------------------------
function bv_supergiant,type
;
; Return (B-V)0 from calibration of MK spectral type (Schmidt-Kaler 1982)
;
status=dc_read_free(!oyster_dir+'starbase/schmidt-kaler.dat', $
			sp,bv5,bv3,bv2,bv1b,bv1ab,bv1a, $
			/col,ignore=['!'])
spec_parse,sp,ft
;
; plot,ft,bv1ab,psym=1
; ft1=findgen(70)
; oplot,ft1,spline(ft,bv1ab,ft1)
;
si=sort(type)	; spline requires sorted vectors!
bv=type
bv(si)=spline(ft,bv1ab,type(si))
return,bv
;
end
;-------------------------------------------------------------------------------
function teff_dwarf,type
;
; Estimate Teff [K] for dwarfs from spectral type.
; Input type as used by OYSTER: O0:00, B0:10, A0:20, F0:30, G0:40, K0:50, M0:60
;
type=float(type)
teff=fltarr(n_elements(type))
;
; A0 to K0 (Boyajian et al. 2012a)
index=where(type ge 20 and type lt 50,count)
c=[9393.59,-490.25,36.44,-1.44,0.0208]
if count gt 0 then teff(index)=poly(ft_tabby(type(index)),c)
;
; K0 to M5.5 (Boyajian et al. 2012b)
ft=[findgen(6)+50,57,findgen(9)*0.5+60,65.5]
tf=[5347,5147,5013,4680,4507,4436,3961,3907, $
    3684,3497,3674,3464,3442,3412,3104,3222,3054]
tf=float(tf)
index=where(type ge 50,count)
if count gt 0 then teff(index)=interpol(tf,ft,type(index))
;
; All other spectral types (via B-V calibration)
index=where(type lt 20 or type gt 65.5,count)
if count gt 0 then teff(index)=teff_bv(bv_dwarf(type(index)))
;
return,teff
;
; Old code
d_type=''       ; Spectral type
d_ft=0.         ; Floating spectral type
d_t=0.          ; log(T_eff/K)
d_c=0.          ; Color (U-B, B-V, etc.)
d_mv=0.         ; M_V
d_mb=0.         ; M_bol
status=dc_read_free(!oyster_dir+'starbase/spec_par.V', $
        d_type,d_ft,d_t,d_c,d_mv,d_mb, $
        /col,resize=[1,2,3,4,5],ignore=['!'])
;
; First version (polynomial interpolation)
d_r=poly_fit(d_ft,d_t,4)
; return,10^poly(type,d_r)
;
; Plot of Teff-spectral type relationship
plot=0
if plot then begin
!p.charsize=1.5
window,/free
x=findgen(70)
y=poly(x,d_r)
y=spline(d_ft,d_t,x)
plot,x,y,title='Dwarfs',psym=0, $
       xtitle='Integer Spectral type',ytitle='log(Effective temperature/K)'
oplot,d_ft,d_t,psym=6
endif
;
; Second version (spline interpolation)
si=sort(type)
teff=fltarr(n_elements(type))
teff(si)=10^spline(d_ft,d_t,type(si))
return,teff
;
end
;-------------------------------------------------------------------------------
function teff_giant,type
;
; Estimate Teff [K] for giants from spectral type.
; Input type as used by OYSTER: O0:00, B0:10, A0:20, F0:30, G0:40, K0:50, M0:60
;
type=float(type)
teff=fltarr(n_elements(type))
;
; G8 to M8 (van Belle et al. 1999)
index=where(type ge 48 and type le 68,count)
if count gt 0 then teff(index)=-109.0*ft_gerard(type(index))+4570
;
; All other spectral types (via B-V calibration)
index=where(type lt 48 or type gt 68,count)
if count gt 0 then teff(index)=teff_bv(bv_giant(type(index)))
;
return,teff
;
; Old code
g_type=''
g_ft=0.
g_t=0.
g_c=0.
g_mv=0.
g_mb=0.
status=dc_read_free(!oyster_dir+'starbase/spec_par.III', $
        g_type,g_ft,g_t,g_c,g_mv,g_mb, $
        /col,resize=[1,2,3,4,5],ignore=['!'])
;
; First version (polynomial interpolation)
g_r=poly_fit(g_ft,g_t,4)
; return,10^poly(type,g_r)
;
; Plot of M_V-spectral type relationship
plot=0
if plot then begin
!p.charsize=1.5
window,/free
x=findgen(70)
; y=poly(x,g_r)
y=spline(g_ft,g_t,x)
plot,x,y,title='Giants',psym=0, $
        xtitle='Integer Spectral type',ytitle='log(Effective temperature/K)'
oplot,g_ft,g_t,psym=6
endif
;
; Second version (spline interpolation)
si=sort(type)
teff=fltarr(n_elements(type))
teff(si)=10^spline(g_ft,g_t,type(si))
return,teff
;
end
;-------------------------------------------------------------------------------
function teff_supergiant,type
;
; Estimate Teff [K] for super giants from spectral type.
; Input type as used by OYSTER: O0:00, B0:10, A0:20, F0:30, G0:40, K0:50, M0:60
;
type=float(type)
teff=fltarr(n_elements(type))
;
; G8 to M8 (van Belle et al. 1999)
index=where(type ge 48 and type le 68,count)
if count gt 0 then teff(index)=-109.0*ft_gerard(type(index))+4570
;
; All other spectral types (via B-V calibration), limit type < 75 
index=where(type lt 48 or type gt 68,count)
if count gt 0 then teff(index)=teff_bv(bv_supergiant(type(index) < 75),/sg)
;
return,teff
;
; Old code
sg_type=''
sg_ft=0.
sg_t=0.
sg_c=0.
sg_mv=0.
sg_mb=0.
status=dc_read_free(!oyster_dir+'starbase/spec_par.Iab', $
        sg_type,sg_ft,sg_t,sg_c,sg_mv,sg_mb, $
        /col,resize=[1,2,3,4,5],ignore=['!'])
;
; sg_r=poly_fit(sg_ft,sg_t,4)
;
; Plot of M_V-spectral type relationship
plot=0
if plot then begin
!p.charsize=1.5
window,/free
x=findgen(70)
; y=poly(x,sg_r)
y=spline(sg_ft,sg_t,x)
plot,x,y,title='Super-Giants',psym=0, $
        xtitle='Integer Spectral type',ytitle='log(Effective temperature/K)'
oplot,sg_ft,sg_t,psym=6
endif
;
si=sort(type)
teff=fltarr(n_elements(type))
teff(si)=10^spline(sg_ft,sg_t,type(si))
return,teff
; return,10^poly(type,sg_r)
;
end
;-------------------------------------------------------------------------------
function teff_wr,type_in
;
; Estimate Teff [K] for Wolf-Rayet stars based on Schmutz et al. 1982
; We assume that T_eff can be extrapolated linearly from WC 9,8,7 to WC 6 and 5.
;
type=float(abs(type_in))
teff=fltarr(n_elements(type))
;
wc_teff=[22e3,26e3,30e3,34e3,38e3]
wc_type=[9,8,7,6,5]
r=poly_fit(wc_type,wc_teff,1)
;
return,poly(type,r)
;
return,teff
;
end
;-------------------------------------------------------------------------------
pro baines
;
; Read tables from Baines et al., 2021, "Angular Diameters and Fundamental
; Parameters of Forty-four Stars from the Navy Precision Optical Interferometer"
;
; Table 5
hd=long(0)                                                           
st=''
lc=''
r=0.
sr=0.
fbol=0.
av=0.
teff=0.
steff=0.
l=0.
s=dc_read_free('table5.dat',hd,st,lc,r,sr,fbol,av,teff,steff,l, $
	resize=[1,2,3,4,5,6,7,8,9,10],/col)

end
;-------------------------------------------------------------------------------
function diam_dwarf,type
;
; Estimate stellar diameter from bolometric magnitude and effective temperature.
; Unit is the radius of the Sun
;
; Input type as used by OYSTER: O0:00, B0:10, A0:20, F0:30, G0:40, K0:50, M0:60
;
type=float(type)
;
d_type=''       ; Spectral type
d_ft=0.         ; Floating spectral type
d_t=0.          ; log(T_eff/K)
d_c=0.          ; Color (U-B, B-V, etc.)
d_mv=0.         ; M_V
d_mb=0.         ; M_bol
status=dc_read_free(!oyster_dir+'starbase/spec_par.V', $
        d_type,d_ft,d_t,d_c,d_mv,d_mb, $
        /col,resize=[1,2,3,4,5],ignore=['!'])
;
si=sort(type)
teff=fltarr(n_elements(type))
mbol=fltarr(n_elements(type))
teff(si)=10^spline(d_ft,d_t,type(si))
mbol(si)=spline(d_ft,d_mb,type(si))
;
logl=(4.74-mbol)/2.5	; Allen's Astrophysical Quantities, p.382, 15.1.1
diam=10^((logl-4*(alog10(teff)-3.762))/2)*2
;
return,diam
;
end
;-------------------------------------------------------------------------------
function diam_subgiant,type
;
; Estimate diameters of bright giants based on type, obtained from an
; experimental calibration using the surface brightness method on the 
; Sky Catalog data. Uncertainty of estimates 25%.
;
; Unit is the radius of the Sun
;
common StarBase,StarTable,Notes
;
type=float(type)
;
if 0 then begin		; Run only to compute coefficients
	get_sky
	i=where(startable.class eq 4)
	startable=startable(i)
	get_jhk
	i=where(startable.mk ne 100)
	startable=startable(i)
	diameter_vk
	sv=startable.mv+5.*alog10(startable.diameter)
	plot,startable.type,sv,psym=1
	n=40
	t=findgen(n)+12
	v=t*0
	for i=0,n-1 do v(i)=median(sv(where(startable.type eq t(i))))
	for i=0,n-1 do begin
		j=where(startable.type eq t(i),count)
		if count ge 2 then v(i)=median(sv(j))
	endfor
	index=where(v ne 0)
	v=v(index)
	t=t(index)
	r=poly_fit(t,v,2,vfit)
	x=findgen(70)
	oplot,x,poly(x,r),psym=0,color=tci(2)
endif
r=[1.41244,0.0614751,0.000487209]
sv=poly(type,r)
return,10^(0.2*(sv-amv_subgiant(type)))
;
end
;-------------------------------------------------------------------------------
function diam_giant,type
;
; Estimate stellar diameter from bolometric magnitude and effective temperature.
; Unit is the radius of the Sun
;
; type: 0,  1,  2, ..., 10, 11, 12, ..., 20, 21, 22, ..., 30
; for: A0, A1, A2, ..., F0, F1, F2, ..., G0, G1, G2, ..., K0
;
type=float(type)
;
d_type=''       ; Spectral type
d_ft=0.         ; Floating spectral type
d_t=0.          ; log(T_eff/K)
d_c=0.          ; Color (U-B, B-V, etc.)
d_mv=0.         ; M_V
d_mb=0.         ; M_bol
status=dc_read_free(!oyster_dir+'starbase/spec_par.III', $
        d_type,d_ft,d_t,d_c,d_mv,d_mb, $
        /col,resize=[1,2,3,4,5],ignore=['!'])
;
si=sort(type)
teff=fltarr(n_elements(type))
mbol=fltarr(n_elements(type))
teff(si)=10^spline(d_ft,d_t,type(si))
mbol(si)=spline(d_ft,d_mb,type(si))
;
logl=(4.74-mbol)/2.5	; Allen's Astrophysical Quantities, p.382, 15.1.1
diam=10^((logl-4*(alog10(teff)-3.762))/2)*2
;
return,diam
;
end
;-------------------------------------------------------------------------------
function diam_brightgiant,type
;
; Estimate diameters of bright giants based on type, obtained from an
; experimental calibration using the surface brightness method on the 
; Sky Catalog data. Uncertainty of estimates 25%.
;
; Unit is the radius of the Sun
;
common StarBase,StarTable,Notes
;
type=float(type)
;
if 0 then begin		; Run only to compute coefficients
	get_sky
	i=where(startable.class eq 2)
	startable=startable(i)
	get_jhk
	i=where(startable.mk ne 100)
	startable=startable(i)
	diameter_vk
	sv=startable.mv+5.*alog10(startable.diameter)
	plot,startable.type,sv,psym=1
	n=60
	t=findgen(n)+12
	v=t*0
	for i=0,n-1 do begin
		j=where(startable.type eq t(i),count)
		if count ge 2 then v(i)=median(sv(j))
	endfor
	index=where(v ne 0)
	v=v(index)
	t=t(index)
	r=poly_fit(t,v,2,vfit)
	x=findgen(70)
	oplot,x,poly(x,r),psym=0,color=tci(2)
endif
r=[3.42353,-0.0252380,0.00177207]
sv=poly(type,r)
return,10^(0.2*(sv-amv_brightgiant(type)))
;
end
;-------------------------------------------------------------------------------
function diam_supergiant,type
;
; Estimate stellar diameter from bolometric magnitude and effective temperature.
; Unit is the radius of the Sun
;
type=float(type)
;
d_type=''       ; Spectral type
d_ft=0.         ; Floating spectral type
d_t=0.          ; log(T_eff/K)
d_c=0.          ; Color (U-B, B-V, etc.)
d_mv=0.         ; M_V
d_mb=0.         ; M_bol
status=dc_read_free(!oyster_dir+'starbase/spec_par.Iab', $
        d_type,d_ft,d_t,d_c,d_mv,d_mb, $
        /col,resize=[1,2,3,4,5],ignore=['!'])
;
si=sort(type)
teff=fltarr(n_elements(type))
mbol=fltarr(n_elements(type))
teff(si)=10^spline(d_ft,d_t,type(si))
mbol(si)=spline(d_ft,d_mb,type(si))
;
logl=(4.74-mbol)/2.5	; Allen's Astrophysical Quantities, p.382, 15.1.1
diam=10^((logl-4*(alog10(teff)-3.762))/2)*2
;
return,diam
;
end
;************************************************************************Block 4
function mass_star,spectrum
;
n=n_elements(spectrum)
if n eq 0 then return,-1
mass=fltarr(n)
spec_parse,spectrum,type,class,type2,class2
;
index=dwarfs(class,type,1)
if index(0) ne -1 then mass(index)=mass_dwarf(type(index))
;
index=subgiants(class,type)
if index(0) ne -1 then mass(index)=mass_subgiant(type(index))
;
index=giants(class,type,1)
if index(0) ne -1 then mass(index)=mass_giant(type(index))
;
index=brightgiants(class,type)
if index(0) ne -1 then mass(index)=mass_brightgiant(type(index))
;
index=supergiants(class,type,1)
if index(0) ne -1 then mass(index)=mass_supergiant(type(index))
;
if n eq 1 then mass=mass(0)
return,mass
;
end
;-------------------------------------------------------------------------------
pro mass_stars,extended
;
; Compound procedure to estimate stellar masses for all spectral types.
; Store results in startable.
;
common StarBase,startable,notes
;
; If extended=1, allow floating point types.
if n_elements(extended) eq 0 then extended=0 else extended=extended gt 0
;
index=dwarfs(startable.class,startable.type,extended)
if index(0) ne -1 then $
	startable(index).mass=mass_dwarf(startable(index).type)
index=dwarfs(startable.class2,startable.type2,extended)
if index(0) ne -1 $
	then startable(index).mass2=mass_dwarf(startable(index).type2)
;
index=subgiants(startable.class,startable.type)
if index(0) ne -1 then $
	startable(index).mass=mass_subgiant(startable(index).type)
index=subgiants(startable.class2,startable.type2)
if index(0) ne -1 then $
	startable(index).mass2=mass_subgiant(startable(index).type2)
;
index=giants(startable.class,startable.type,extended)
if index(0) ne -1 then $
	startable(index).mass=mass_giant(startable(index).type)
index=giants(startable.class2,startable.type2,extended)
if index(0) ne -1 then $
	startable(index).mass2=mass_giant(startable(index).type2)
;
index=brightgiants(startable.class,startable.type)
if index(0) ne -1 then $
	startable(index).mass=mass_brightgiant(startable(index).type)
index=brightgiants(startable.class2,startable.type2)
if index(0) ne -1 then $
	startable(index).mass2=mass_brightgiant(startable(index).type2)
;
index=supergiants(startable.class,startable.type,extended)
if index(0) ne -1 then $
	startable(index).mass=mass_supergiant(startable(index).type)
index=supergiants(startable.class2,startable.type2,extended)
if index(0) ne -1 then $
	startable(index).mass2=mass_supergiant(startable(index).type2)
;
end
;-------------------------------------------------------------------------------
pro mass_binaries
;
; Applying Kepler's 3rd law to visual binaries with measured parallaxes
; and the primary spectral types, compute component masses and magnitude
; differences. Reduces the startable to just those systems!
;
common StarBase,StarTable,Notes
;
index=where(startable.px ne 0 and $
	    startable.p ne 0 and $
	    startable.type gt 0,count)
if count eq 0 then begin
	print,'***Error(MASS_BINARIES): not enough data to compute!'
	return
endif
startable=startable(index)
;
; Assume dwarfs if class is unknown
startable(where(startable.class eq 0)).class=5
;
; Set secondary class and type to unknown
type2=startable.type2
class2=startable.class2
startable.type2=0
startable.class2=0
;
; Compute total masses from parallax, a, and P
m1pm2=(startable.a/startable.px)^3/(startable.p/365.25)^2
;
; Estimate primary mass from spectral classification
mass_stars
m1=startable.mass
;
; Compute secondary mass as total minus primary mass
m2=m1pm2-m1
startable.mass2=m2
;
; Compute mass function
; Beware: f(m) from SB9 cannot be taken at face value as for example
; K1 (HIC 6564) is actually K1+k2!
startable.mf=1.0385e-7*(1.0-startable.e^2)^(3./2)*startable.k1^3*startable.p
mf=(m2*sin(startable.i*!pi/180))^3/m1pm2^2 ; should have same value as above
startable.mf=mf	; Store this value for f(m)
;
; Estimate secondary mass with mass function
; tbd...
return
;
; Estimate primary absolute magnitude from spectral classification
amv_stars
amv1=startable.amv
;
; Derive secondary (integer) spectral type
s=''
m=0.
t=0.
status=dc_read_free(!oyster_dir+'starbase/masses.V',m,t, $
	/col,resize=[1,2],ignore=['!'])
r=poly_fit(alog(m),t,5)
t2=fix(poly(alog(m2),r))
startable.type2=t2
startable.class2=5
;
; Estimate secondary absolute magnitude assuming they are dwarfs
d_type=''
d_ft=0.
d_t=0.
d_c=0.
d_mv=0.
d_mb=0.
status=dc_read_free(!oyster_dir+'starbase/spec_par.V', $
	d_type,d_ft,d_t,d_c,d_mv,d_mb, $
	/col,resize=[1,2,3,4,5],ignore=['!'])
d_r=poly_fit(d_ft,d_mv,5)
amv2=poly(t2,d_r)
;
dmv=amv2-amv1
startable.dmv=dmv
;
end
;-------------------------------------------------------------------------------
function mass2_mf,mf,m1
;
; Compute minimum secondary mass for given velue mf of mass function and
; primary mass.
;
; Definition of mass function
; mf=m2^3*sin(i)^3/(m1+m2)^2
;
; The minimum mass estimate is for i=90
inclination=90.
;
m2=m1
;
repeat m2=(mf*(m1+m2)^2)^(1./3) until abs(mf-m2^3/(m1+m2)^2) lt 0.001
;
return,m2
;
end
;-------------------------------------------------------------------------------
function logg_star,spectrum
;
n=n_elements(spectrum)
if n eq 0 then return,-1
logg=fltarr(n)
spec_parse,spectrum,type,class,type2,class2
;
index=dwarfs(class,type)
if index(0) ne -1 then logg(index)=logg_dwarf(type(index))
;
index=subgiants(class,type)
if index(0) ne -1 then logg(index)=logg_subgiant(type(index))
;
index=giants(class,type)
if index(0) ne -1 then logg(index)=logg_giant(type(index))
;
index=brightgiants(class,type)
if index(0) ne -1 then logg(index)=logg_brightgiant(type(index))
;
index=Iasupergiants(class,type)
if index(0) ne -1 then logg(index)=logg_Iasupergiant(type(index))
;
index=Iabsupergiants(class,type)
if index(0) ne -1 then logg(index)=logg_Iabsupergiant(type(index))
;
index=wolfrayets(class,type)
if index(0) ne -1 then logg(index)=logg_Ibsupergiant(type(index))
;
if n eq 1 then logg=logg(0)
;
return,logg
;
end
;-------------------------------------------------------------------------------
pro logg_stars
;
; Compound procedure to estimate stellar gravities for all spectral types.
; Store results in startable.
;
common StarBase,startable,notes
;
index=dwarfs(startable.class,startable.type)
if index(0) ne -1 then $
	startable(index).logg=logg_dwarf(startable(index).type)
index=dwarfs(startable.class2,startable.type2)
if index(0) ne -1 then $
	startable(index).logg2=logg_dwarf(startable(index).type2)
;
index=subgiants(startable.class,startable.type)
if index(0) ne -1 then $
	startable(index).logg=logg_subgiant(startable(index).type)
index=subgiants(startable.class2,startable.type2)
if index(0) ne -1 then $
	startable(index).logg2=logg_subgiant(startable(index).type2)
;
index=giants(startable.class,startable.type)
if index(0) ne -1 then $
	startable(index).logg=logg_giant(startable(index).type)
index=giants(startable.class2,startable.type2)
if index(0) ne -1 then $
	startable(index).logg2=logg_giant(startable(index).type2)
;
index=supergiants(startable.class,startable.type)
if index(0) ne -1 then $
	startable(index).logg=logg_brightgiant(startable(index).type)
index=supergiants(startable.class2,startable.type2)
if index(0) ne -1 then $
	startable(index).logg2=logg_brightgiant(startable(index).type2)
;
index=Iasupergiants(startable.class,startable.type)
if index(0) ne -1 then $
	startable(index).logg=logg_Iasupergiant(startable(index).type)
index=Iasupergiants(startable.class2,startable.type2)
if index(0) ne -1 then $
	startable(index).logg2=logg_Iasupergiant(startable(index).type2)
;
index=Iabsupergiants(startable.class,startable.type)
if index(0) ne -1 then $
	startable(index).logg=logg_Iabsupergiant(startable(index).type)
index=Iabsupergiants(startable.class2,startable.type2)
if index(0) ne -1 then $
	startable(index).logg2=logg_Iabsupergiant(startable(index).type2)
;
index=Ibsupergiants(startable.class,startable.type)
if index(0) ne -1 then $
	startable(index).logg=logg_Ibsupergiant(startable(index).type)
index=Ibsupergiants(startable.class2,startable.type2)
if index(0) ne -1 then $
	startable(index).logg2=logg_Ibsupergiant(startable(index).type2)
;
end
;-------------------------------------------------------------------------------
function teff_star,spectrum
;
n=n_elements(spectrum)
if n eq 0 then return,-1
;
teff=fltarr(n)
spec_parse,spectrum,type,class,type2,class2
;
; Dwarfs
index=dwarfs(class,type,1)
if index(0) ne -1 then teff(index)=teff_dwarf(type(index))
;
; Giants
index=giants(class,type,1)
if index(0) ne -1 then teff(index)=teff_giant(type(index))
;
; Super giants
index=supergiants(class,type,1)
if index(0) ne -1 then teff(index)=teff_supergiant(type(index))
;
; Wolf-Rayet
index=supergiants(class,type,1)
if index(0) ne -1 then teff(index)=teff_wr(type(index))
;
if n eq 1 then teff=teff(0)
return,teff
;
end
;-------------------------------------------------------------------------------
pro teff_stars
;
; Derive effective temperature for all stars from spectral classification.
; Store results in startable.
;
common StarBase,startable,notes
;
; We should initialize Teff to zero
startable.teff=0
;
index=dwarfs(startable.class,startable.type,1)
if index(0) ne -1 then $
startable(index).teff=teff_dwarf(startable(index).type)
;
index=giants(startable.class,startable.type,1)
if index(0) ne -1 then $
startable(index).teff=teff_giant(startable(index).type)
;
index=supergiants(startable.class,startable.type,1)
if index(0) ne -1 then $
startable(index).teff=teff_supergiant(startable(index).type)
;
end
;-------------------------------------------------------------------------------
function diam_star,spectrum
;
; Stellar diameter in units of the solar radius (R_Sun)
;
n=n_elements(spectrum)
if n eq 0 then return,-1
diam=fltarr(n)
spec_parse,spectrum,type,class,type2,class2
;
; Dwarfs
index=dwarfs(class,type,1)
if index(0) ne -1 then diam(index)=diam_dwarf(type(index))
;
; Giants
index=giants(class,type,1)
if index(0) ne -1 then diam(index)=diam_giant(type(index))
;
; Sub-giants
index=subgiants(class,type)
if index(0) ne -1 then diam(index)=diam_subgiant(type(index))
;
; Super giants
index=supergiants(class,type,1)
if index(0) ne -1 then diam(index)=diam_supergiant(type(index))
;
if n eq 1 then diam=diam(0)
return,diam
;
end
;-------------------------------------------------------------------------------
pro diam_stars
;
; Derive diameters for all stars from spectral classification.
; Unit is the radius of the Sun
;
common StarBase,startable,notes
;
; We should initialize to zero
startable.diameter=0
;
index=dwarfs(startable.class,startable.type,1)
if index(0) ne -1 then $
startable(index).diameter=diam_dwarf(startable(index).type)
;
index=giants(startable.class,startable.type,1)
if index(0) ne -1 then $
startable(index).diameter=diam_giant(startable(index).type)
;
index=supergiants(startable.class,startable.type,1)
if index(0) ne -1 then $
startable(index).diameter=diam_supergiant(startable(index).type)
;
end
;-------------------------------------------------------------------------------
pro diam_binaries,teff_a,teff_b
;
end
;-------------------------------------------------------------------------------
function bv_star,spectrum
;
n=n_elements(spectrum)
if n eq 0 then return,-1
bv=fltarr(n)
spec_parse,spectrum,type,class,type2,class2
;
; Dwarfs
index=dwarfs(class,type,1)
if index(0) ne -1 then bv(index)=bv_dwarf(type(index))
;
; Giants
index=giants(class,type,1)
if index(0) ne -1 then bv(index)=bv_giant(type(index))
;
; Super giants
index=supergiants(class,type,1)
if index(0) ne -1 then bv(index)=bv_supergiant(type(index))
;
if n eq 1 then bv=bv(0)
return,bv
;
end
;-------------------------------------------------------------------------------
pro bv_stars
;
; Derive (B-V) for all stars from spectral classification.
;
common StarBase,startable,notes
;
; We should initialize to zero
startable.bv=0
;
index=dwarfs(startable.class,startable.type,1)
if index(0) ne -1 then $
startable(index).bv=bv_dwarf(startable(index).type)
;
index=giants(startable.class,startable.type,1)
if index(0) ne -1 then $
startable(index).bv=bv_giant(startable(index).type)
;
index=supergiants(startable.class,startable.type,1)
if index(0) ne -1 then $
startable(index).bv=bv_supergiant(startable(index).type)
;
end
;************************************************************************Block 5
function amv_dwarf,type
;
; Absolute visual magnitude of dwarfs
;
d_type=''
d_ft=0.
d_t=0.
d_c=0.
d_mv=0.
d_mb=0.
status=dc_read_free(!oyster_dir+'starbase/spec_par.V', $
	d_type,d_ft,d_t,d_c,d_mv,d_mb, $
	/col,resize=[1,2,3,4,5],ignore=['!'])
d_r=poly_fit(d_ft,d_mv,5)
;
return,spline(d_ft,d_mv,type)
return,poly(type,d_r)
;
end
;-------------------------------------------------------------------------------
function amv_subgiant,type
;
; Absolute visual magnitude of subgiants
;
subg_type=''
subg_ft=0.
subg_mv=0.
status=dc_read_free(!oyster_dir+'starbase/spec_MV.IV', $
	subg_type,subg_ft,subg_mv, $
	/col,resize=[1,2,3],ignore=['!'])
subg_r=poly_fit(subg_ft,subg_mv,3)
;
return,spline(subg_ft,subg_mv,type)
return,poly(type,subg_r)
;
end
;-------------------------------------------------------------------------------
function amv_giant,type
;
; Absolute visual magnitude of giants
;
g_type=''
g_ft=0.
g_t=0.
g_c=0.
g_mv=0.
g_mb=0.
status=dc_read_free(!oyster_dir+'starbase/spec_par.III', $
	g_type,g_ft,g_t,g_c,g_mv,g_mb, $
	/col,resize=[1,2,3,4,5],ignore=['!'])
g_r=poly_fit(g_ft,g_mv,5)
;
return,spline(g_ft,g_mv,type)
return,poly(type,g_r)
;
end
;-------------------------------------------------------------------------------
function amv_brightgiant,type
;
; Absolute visual magnitude of bright giants
;
bg_type=''
bg_ft=0.
bg_mv=0.
status=dc_read_free(!oyster_dir+'starbase/spec_MV.II',bg_type,bg_ft,bg_mv, $
	/col,resize=[1,2,3],ignore=['!'])
bg_r=poly_fit(bg_ft,bg_mv,4)
;
return,spline(bg_ft,bg_mv,type)
return,poly(type,bg_r)
;
end
;-------------------------------------------------------------------------------
function amv_supergiant,type
;
; Absolute visual magnitude of supergiants
;
sgIab_type=''
sgIab_ft=0.
sgIab_mv=0.
status=dc_read_free(!oyster_dir+'starbase/spec_MV.Iab', $
	sgIab_type,sgIab_ft,sgIab_mv, $
	/col,resize=[1,2,3],ignore=['!'])
sgIab_r=poly_fit(sgIab_ft,sgIab_mv,4)
;
return,spline(sgIab_ft,sgIab_mv,type)
return,poly(type,sgIab_r)
;
end
;-------------------------------------------------------------------------------
function amv_star,spectrum
;
; Return absolute V-magnitude M_V from spectral classification.
n=n_elements(spectrum)
if n eq 0 then return,-1
pmv=fltarr(n)
spec_parse,spectrum,type,class,type2,class2
;
index=dwarfs(class,type)
if index(0) ne -1 then pmv(index)=amv_dwarf(type(index))
;
index=subgiants(class,type)
if index(0) ne -1 then pmv(index)=amv_subgiant(type(index))
;
index=giants(class,type)
if index(0) ne -1 then pmv(index)=amv_giant(type(index))
;
index=brightgiants(class,type)
if index(0) ne -1 then pmv(index)=amv_brightgiant(type(index))
;
index=supergiants(class,type)
if index(0) ne -1 then pmv(index)=amv_supergiant(type(index))
;
if n eq 1 then pmv=pmv(0)
return,pmv
;
end
;-------------------------------------------------------------------------------
pro amv_stars
;
; Derive absolute V-magnitude M_V for all stars from spectral classification.
; For double stars, M_V will be the combined absolute magnitude.
;
common StarBase,startable,notes
;
; Dwarfs
d_type=''
d_ft=0.
d_t=0.
d_c=0.
d_mv=0.
d_mb=0.
status=dc_read_free(!oyster_dir+'starbase/spec_par.V', $
	d_type,d_ft,d_t,d_c,d_mv,d_mb, $
	/col,resize=[1,2,3,4,5],ignore=['!'])
d_r=poly_fit(d_ft,d_mv,5)
;
; Uncomment next lines for plot of M_V-spectral type relationship
; !p.charsize=1.5
; plot,findgen(70),poly(findgen(70),d_r),title='Dwarfs',yrange=[20,-8], $
; 	xtitle='Integer Spectral type',ytitle='Absolute magnitude'
; oplot,d_ft,d_mv,psym=6
;
index=where(startable.class gt 4.5 $
		and startable.type ne 0,count)
if count gt 0 then startable(index).amv=poly(startable(index).type,d_r)
;
; Sub-giants
subg_type=''
subg_ft=0.
subg_mv=0.
status=dc_read_free(!oyster_dir+'starbase/spec_MV.IV', $
	subg_type,subg_ft,subg_mv, $
	/col,resize=[1,2,3],ignore=['!'])
subg_r=poly_fit(subg_ft,subg_mv,3)
;
; Uncomment next lines for plot of M_V-spectral type relationship
; !p.charsize=1.5
; plot,findgen(70),poly(findgen(70),subg_r),title='Sub-giants',yrange=[20,-8], $
; 	xtitle='Integer Spectral type',ytitle='Absolute magnitude'
; oplot,subg_ft,subg_mv,psym=6
;
index=where(startable.class le 4.5 and startable.class gt 3.5 $
		and startable.type ne 0,count)
if count gt 0 then startable(index).amv=poly(startable(index).type,d_r)
;
; Giants
g_type=''
g_ft=0.
g_t=0.
g_c=0.
g_mv=0.
g_mb=0.
status=dc_read_free(!oyster_dir+'starbase/spec_par.III', $
	g_type,g_ft,g_t,g_c,g_mv,g_mb, $
	/col,resize=[1,2,3,4,5],ignore=['!'])
g_r=poly_fit(g_ft,g_mv,5)
;
; Uncomment next lines for plot of M_V-spectral type relationship
; !p.charsize=1.5
; plot,findgen(70),poly(findgen(70),g_r),title='Giants',yrange=[20,-8], $
; 	xtitle='Integer Spectral type',ytitle='Absolute magnitude'
; oplot,g_ft,g_mv,psym=6
;
index=where(startable.class le 3.5 and startable.class gt 2.5 $
		and startable.type ne 0,count)
if count gt 0 then startable(index).amv=poly(startable(index).type,g_r)
;
; Bright-Giants
bg_type=''
bg_ft=0.
bg_mv=0.
status=dc_read_free(!oyster_dir+'starbase/spec_MV.II',bg_type,bg_ft,bg_mv, $
	/col,resize=[1,2,3],ignore=['!'])
bg_r=poly_fit(bg_ft,bg_mv,4)
;
; Uncomment next lines for plot of M_V-spectral type relationship
; !p.charsize=1.5
; plot,findgen(70),poly(findgen(70),bg_r),title='Bright-giants',yrange=[20,-8], $
; 	xtitle='Integer Spectral type',ytitle='Absolute magnitude'
; oplot,bg_ft,bg_mv,psym=6
;
index=where(startable.class le 2.5 and startable.class gt 1.75 $
		and startable.type ne 0,count)
if count gt 0 then startable(index).amv=poly(startable(index).type,g_r)
;
; Super-Giants
; sg_type=''
; sg_ft=0.
; sg_t=0.
; sg_c=0.
; sg_mv=0.
; sg_mb=0.
; status=dc_read_free(!oyster_dir+'starbase/spec_par.Iab', $
; 	sg_type,sg_ft,sg_t,sg_c,sg_mv,sg_mb, $
; 	/col,resize=[1,2,3,4,5],ignore=['!'])
; sg_r=poly_fit(sg_ft,sg_mv,5)
;
; Uncomment next lines for plot of M_V-spectral type relationship
; !p.charsize=1.5
; plot,findgen(70),poly(findgen(70),sg_r),title='Super-Giants',yrange=[20,-8], $
; 	xtitle='Integer Spectral type',ytitle='Absolute magnitude'
; oplot,sg_ft,sg_mv,psym=6
;
; index=where(fix(startable.class) eq 1,count)
; if count gt 0 then startable(index).amv=poly(startable(index).type,sg_r)
;
; Super-giants Ia
sgIa_type=''
sgIa_ft=0.
sgIa_mv=0.
status=dc_read_free(!oyster_dir+'starbase/spec_MV.Ia', $
	sgIa_type,sgIa_ft,sgIa_mv, $
	/col,resize=[1,2,3],ignore=['!'])
sgIa_r=poly_fit(sgIa_ft,sgIa_mv,4)
index=where(startable.class gt 0 and startable.class lt 1.15 $
		and startable.type ne 0,count)
if count gt 0 then startable(index).amv=poly(startable(index).type,sgIa_r)
;
; Uncomment next lines for plot of M_V-spectral type relationship
; !p.charsize=1.5
; plot,findgen(70),poly(findgen(70),sgIa_r),title='Super-Giants Ia',yrange=[20,-8], $
; 	xtitle='Integer Spectral type',ytitle='Absolute magnitude'
; oplot,sgIa_ft,sgIa_mv,psym=6
;
; Super-giants Iab
sgIab_type=''
sgIab_ft=0.
sgIab_mv=0.
status=dc_read_free(!oyster_dir+'starbase/spec_MV.Iab', $
	sgIab_type,sgIab_ft,sgIab_mv, $
	/col,resize=[1,2,3],ignore=['!'])
sgIab_r=poly_fit(sgIab_ft,sgIab_mv,4)
index=where(startable.class ge 1.15 and startable.class lt 1.35 $
		and startable.type ne 0,count)
if count gt 0 then startable(index).amv=poly(startable(index).type,sgIab_r)
;
; Uncomment next lines for plot of M_V-spectral type relationship
; !p.charsize=1.5
; plot,findgen(70),poly(findgen(70),sgIab_r),title='Super-Giants Iab',yrange=[20,-8], $
; 	xtitle='Integer Spectral type',ytitle='Absolute magnitude'
; oplot,sgIab_ft,sgIab_mv,psym=6
;
; Super-giants Ib
sgIb_type=''
sgIb_ft=0.
sgIb_mv=0.
status=dc_read_free(!oyster_dir+'starbase/spec_MV.Ib', $
	sgIb_type,sgIb_ft,sgIb_mv, $
	/col,resize=[1,2,3],ignore=['!'])
sgIb_r=poly_fit(sgIb_ft,sgIb_mv,4)
index=where(startable.class ge 1.35 and startable.class le 1.75 $
		and startable.type ne 0,count)
if count gt 0 then startable(index).amv=poly(startable(index).type,sgIb_r)
;
; Uncomment next lines for plot of M_V-spectral type relationship
; !p.charsize=1.5
; plot,findgen(70),poly(findgen(70),sgIb_r),title='Super-Giants Ib',yrange=[20,-8], $
; 	xtitle='Integer Spectral type',ytitle='Absolute magnitude'
; oplot,sgIb_ft,sgIb_mv,psym=6
;
;...............................................................................
; Check for secondaries and make necessary adjustments
; Dwarfs...
index=where(startable.class ne 0 and startable.type2 ne 0 $
	and startable.class2 gt 4.5,count)
if count gt 0 then begin
	MV2=poly(startable(index).type2,d_r)
	startable(index).dmv=MV2-startable(index).amv
	startable(index).amv= $
	cmag(startable(index).amv,MV2)
endif
;
; Sub-giants...
index=where(startable.class ne 0 and startable.type2 ne 0 $
	and startable.class2 le 4.5 and startable.class2 gt 3.5,count)
if count gt 0 then begin
	MV2=poly(startable(index).type2,subg_r)
	startable(index).dmv=MV2-startable(index).amv
	startable(index).amv= $
	cmag(startable(index).amv,MV2)
endif
;
; Giants...
index=where(startable.class ne 0 and startable.type2 ne 0 $
	and startable.class2 le 3.5 and startable.class2 gt 2.5,count)
if count gt 0 then begin
	MV2=poly(startable(index).type2,g_r)
	startable(index).dmv=MV2-startable(index).amv
	startable(index).amv= $
	cmag(startable(index).amv,MV2)
endif
;
; Bright-giants...
index=where(startable.class ne 0 and startable.type2 ne 0 $
	and startable.class2 le 2.5 and startable.class2 gt 1.75,count)
if count gt 0 then begin
	MV2=poly(startable(index).type2,bg_r)
	startable(index).dmv=MV2-startable(index).amv
	startable(index).amv= $
	cmag(startable(index).amv,MV2)
endif
;
; Super-Giants...
index=where(startable.class ne 0 and startable.type2 ne 0 $
	and startable.class2 le 1.75 and startable.class2 gt 1.35,count)
if count gt 0 then begin
	MV2=poly(startable(index).type2,sgIb_r)
	startable(index).dmv=MV2-startable(index).amv
	startable(index).amv= $
	cmag(startable(index).amv,MV2)
endif
;
index=where(startable.class ne 0 and startable.type2 ne 0 $
	and startable.class2 le 1.35 and startable.class2 gt 1.15,count)
if count gt 0 then begin
	MV2=poly(startable(index).type2,sgIab_r)
	startable(index).dmv=MV2-startable(index).amv
	startable(index).amv= $
	cmag(startable(index).amv,MV2)
endif
;
index=where(startable.class ne 0 and startable.type2 ne 0 $
	and startable.class2 le 1.15 and startable.class2 gt 0,count)
if count gt 0 then begin
	MV2=poly(startable(index).type2,sgIa_r)
	startable(index).dmv=MV2-startable(index).amv
	startable(index).amv= $
	cmag(startable(index).amv,MV2)
endif
;
end
;-------------------------------------------------------------------------------
pro amv_dwarfs
;
; Derive absolute V-magnitude M_V for single dwarfs from (B-V)-M_V calibration.
;
common StarBase,startable,notes
;
index=where((startable.class eq 5) $
	and (startable.class2 eq 0) $
	and (startable.bv ne +100.0),count)
;
status=dc_read_free(!oyster_dir+'starbase/BV_MV.dwarfs',bv,mv,/col)
r=poly_fit(bv,mv,5)
amv=poly(startable(index).bv,r)
startable(index).amv=amv
;
end
;-------------------------------------------------------------------------------
pro mk_dwarfs
;
; Compute K from V and V-K from spectrum, rather rough probably!
;
common StarBase,startable,notes
;
index=where((startable.class eq 5) $
	and (startable.type ne 0) $
	and (startable.mv ne +100.0),count)
;
type=''
ft=0.
vk=0.
status=dc_read_free(!oyster_dir+'starbase/spec_VK.V', $
	type,ft,vk, $
	/col,resize=[1,2,3],ignore=['!'])
r=poly_fit(ft,vk,6)
vk=poly(startable(index).type < 67,r)
startable(index).mk=startable(index).mv-vk
;
end
;-------------------------------------------------------------------------------
function limb_star,spectrum,filter
;
n=n_elements(spectrum)
if n eq 0 then return,-1
if n_elements(filter) eq 0 then return,-1
coeffs=fltarr(n)
spec_parse,spectrum,type,class,type2,class2
;
for i=0,n-1 do $
	coeffs(i)=limbfilter(teff_star(spectrum(i)), $
			     logg_star(spectrum(i)),filter)
;
return,coeffs
;
end
;-------------------------------------------------------------------------------
pro limb_stars,range,logarithmic=logarithmic
;
; For all stars in the StarTable, compute the quadratic fit coefficients
; to the linear limb darkening coefficients in the interval [450,850] nm,
; unless a different range [nm] is specified.
;
; For larger ranges, e.g. [450,2500], it is recommended to use the
; fit to the logarithmic values of lambda and u.
;
common StarBase,startable,notes
;
if total(startable.teff) eq 0 then begin
	print,'***Error(LIMB_STARS): all effective temperatures zero!'
	return
endif
if total(startable.class) eq 0 then begin
	print,'***Error(LIMB_STARS): all classes equal to zero!'
	return
endif
;
if n_elements(range) eq 0 then range=[450.0,850.0]
if not keyword_set(logarithmic) then logarithmic=0
;
for i=0L,n_elements(startable)-1 do begin
	if startable(i).teff ne 0 and startable(i).class ne 0 then begin
        	if limbgrid(startable(i).teff,startable(i).logg, $
			lamda,limbdu,fluxes) ne 0 then return
		index=where(lamda gt range(0) and lamda lt range(1))
		x=lamda(index)
		y=limbdu(index)
		if logarithmic then begin
;			In this case, abs(A1) will be larger than 0.01
			x=alog10(x)
			y=alog10(y)
		endif
		r=poly_fit(x,y,2)
        	startable(i).a0=r(0)
        	startable(i).a1=r(1)
        	startable(i).a2=r(2)
	endif
endfor
;
end
;-------------------------------------------------------------------------------
pro av_stars,amores=amores,drimmel=drimmel
;
; Compute Av for all stars in startable using tools by Amores and Lepine 2005
; (AJ, 130, 679, http://www.astro.iag.usp.br/~amores/modextin.html) or Drimmel
; et al. 2003 (A&A, 409, 205, ftp://ftp.oato.inaf.it/astrometria/extinction/).
;
common StarBase,startable,notes
;
if n_elements(amores) eq 0 then amores=0
if n_elements(drimmel) eq 0 then drimmel=0
if amores eq 0 and drimmel eq 0 then begin
	print,'Using model by Amores and Lepine 2005.'
	amores=1
endif
;
index=where(startable.px gt 0,count)
if count eq 0 then begin
	print,'Error: all parallaxes zero or negative!'
	return
endif
table=startable(index)
d=1.0/table.px & d=d/1000	; kpc
;
ra=table.ra*15
dec=table.dec
euler,ra,dec,glong,glat,1
;
if drimmel then begin
	findext,glong,glat,d,av,avmod
	startable(index).av=av
;	for i=0,count-1 do begin
;	findext,glong(i),glat(i),d(i),av,avmod
;	startable(index(i)).av=av
;	endfor
endif
if amores then begin
	for i=0,count-1 do begin
	extin,glong(i),glat(i),d(i),av,ebv
	startable(index(i)).av=av
	endfor
endif
;
end
;-------------------------------------------------------------------------------
