;*******************************************************************************
; File: time.pro
;
; Description:
; ------------
; Contains time/date functions.
;
; Block directory:
; ----------------
; Block 1: constrictordate,rawdate,aipsdate,mark3date,julian,jd2date,date2jd,
;          jd2mjd,mjd2jd,by2jd,jd2by,by2jy,jy2jd,jd2jy,jy2by,date2jy,
;	   reversedate,previousdate,nextdate,
;	   parsedate,parsemark3,parsepti,
;	   parsehpuxdate,parseidldate,parseisodate,parsewbddate
;
;************************************************************************Block 1
function constrictordate,y,m,d
;
; Returns date string in CONSTRICTOR format (e.g.: 1994-05-31)
; This is the standard date string format.
;
Date=string(y,format='(i4.4)')+'-' $
    +string(m,format='(i2.2)')+'-' $
    +string(d,format='(i2.2)')
return,Date
;
end
;-------------------------------------------------------------------------------
function rawdate,y,m,d
;
; Returns date string in NPOI site format (e.g.: 1994-0531)
;
Date=string(y,format='(i4.4)')+'-' $
    +string(m,format='(i2.2)') $
    +string(d,format='(i2.2)')
return,Date
;
end
;-------------------------------------------------------------------------------
function aipsdate,y,m,d
;
; Returns date string in AIPS format (e.g.: 31/05/94)
;
if y lt 2000 then y0=1900 else y0=2000
Date=string(d,format='(i2.2)')+'/' $
    +string(m,format='(i2.2)')+'/' $
    +string(y-y0,format='(i2.2)')
return,Date
end
;-------------------------------------------------------------------------------
function mark3date,y,m,d
;
; Returns date string in Mark3 format (e.g.: 940531)
;
Date=string(y-1900,format='(i3.2)') $
    +string(m,format='(i2.2)') $
    +string(d,format='(i2.2)')
return,Date
;
end
;-------------------------------------------------------------------------------
; function julian,y,m,d
;
; Returns Julian Day Number that begins at NOON OF THE CALENDAR DATE
; specified by year y, month m, and day d, all long integer variables.
; Positive year signifies A.D.; negative, B.C. Remember that the year
; after 1 B.C. was 1 A.D.! (From: Numerical Recipes, CUP)
;
; if y lt 0 then iy=y+1 else iy=y
; if m gt 2 then im=m+1 else begin
; 	iy=iy-1
; 	im=m+13
; endelse
; if (d+31L*(m+12L*y)) ge (15L+31L*(10L+12L*1582L)) then begin
; 	aa=long(iy/100)
; 	bb=2-aa+long(aa/4)
; endif else bb=0
; if iy lt 0 then cc=long((365.25*iy)-0.75) else cc=long(365.25*iy)
; dd=long(30.6001*im)
; return,bb+cc+dd+d+1720995L
;
; end
;-------------------------------------------------------------------------------
function julian,y,m,day,mjd=mjd,rjd=rjd,jdn=jdn
;
; Returns Julian Day at 0 hours UT1 (def.) or noon (/jdn).
; Please note that the default is used by the NPOI team.
; The MJD gives the number of days since November 17, 1858, midnight
; The RJD gives the number of days since November 16, 1858, noon
; The latter two options are not affected by the jdn parameter.
; Inputs can be arrays!  Input day can be fractional day.
; (Modified from: Numerical Recipes, CUP)
;
; Also go here for testing:
; https://aa.usno.navy.mil/data/JulianDate
; e.g., 2025-09-29 00:00:00h => 2460947.5 = julian(2025,9,29)
; e.g., 2025-09-29 00:00:00h =>   60947.5 = julian(2025,9,29,/mjd)
;
; https://core1.gsfc.nasa.gov/time/julian.html
; e.g., 2025-09-29  
;
if n_elements(mjd) eq 0 then mjd=0
if n_elements(rjd) eq 0 then rjd=0
if n_elements(jdn) eq 0 then jdn=0
;
n=n_elements(y)
y=long(y)
m=long(m)
;
d=long(day)
df=day-d
;
iy=y
index=where(y lt 0,count)
if count gt 0 then iy(index)=iy(index)+1
im=m+1
index=where(m le 2,count)
if count gt 0 then begin
	iy(index)=iy(index)-1
	im(index)=im(index)+12
endif
bb=lonarr(n)
index=where(d+31L*(m+12L*y) ge (15L+31L*(10L+12L*1582L)),count)
if count gt 0 then begin
	aa=long(iy(index)/100)
	bb(index)=2-aa+long(aa/4)
endif
cc=lonarr(n)
index=where(iy lt 0,count)
if count gt 0 then cc(index)=long((365.25*iy(index))-0.75)
index=where(iy ge 0,count)
if count gt 0 then cc(index)=long(365.25*iy(index))
dd=long(30.6001*im)
jd=double(bb+cc+dd+d+1720995L)+df-0.5	; JD relative to 0 UT
if mjd then begin
	jd=jd-2400000.5d0		; Modified Julian Date
endif else if rjd then begin
	jd=jd-2440000.0d0		; Reduced Julian Date
endif else begin
	if jdn then begin
		jd=jd+0.5		; JD relative to noon
	endif
endelse
;
;
if n eq 1 then return,jd(0) else return,jd
;
end
;-------------------------------------------------------------------------------
function julian_pds,y,m,day,mjd=mjd,rjd=rjd
;
; Returns Julian Day (Number + fractional part) at 0 hours UT.
; Inputs can be arrays!  Input day can be fractional day.
; (From: Practical Astronomy with your calculator, by Peter Duffet-Smith)
; The MJD gives the number of days since November 17, 1858, midnight
; The RJD gives the number of days since November 16, 1858, noon
;
; Introduced 2025, as check on previous implementation
;
if n_elements(mjd) eq 0 then mjd=0
if n_elements(rjd) eq 0 then rjd=0
;
; Save a copy of input
y_in=y
m_in=m
day_in=day
;
y=double(y)
m=double(m)
d=double(day)
;
n=n_elements(y)
;
index_m0=where(m ne 1 and m ne 2)
index=where(m eq 1 or m eq 2,count)
if count gt 0 then begin
	y(index)=y(index)-1
	m(index)=m(index)+12
endif
;
a=long(n)
b=long(n)
c=long(n)
index=where(y ge 1582 and m ge 10 and day ge 15,count)
if count gt 0 then begin
	a(index)=long(float(y(index))/100)
	b(index)=2-a(index)+long(a(index)/4)
endif
;
index=where(y lt 0,count)
if count gt 0 then c(index)=long((365.25*y(index))-0.75)
index=where(y ge 0,count)
if count gt 0 then c(index)=long(365.25*y(index))
;
di=long(30.6001*(m(index)+1))
;
; Restore input parameters
y=y_in
m=m_in
day=day_in
;
jd=b+c+di+d+1720994.5
;
if mjd then jd=jd-2400000.5d0	; Modified JD
if rjd then jd=jd-2440000.0d0	;  Reduced JD
;
if n eq 1 then return,jd(0) else return,jd
;
end
;-------------------------------------------------------------------------------
function jd2date,jd_in,df,mjd=mjd,rjd=rjd
;
; Program returns year, month, day+fraction corresponding to specified Julian
; Day epoch (double precision!, also modified and "reduced" OYSTER epochs). 
; Input can be arrays! (From: Numerical Recipes, CUP) 
;
; Make sure we compute with the highest precision
jd_in=double(jd_in)
;
jd=jd_in
if n_elements(mjd) eq 0 then mjd=0
if n_elements(rjd) eq 0 then rjd=0
if mjd then jd=jd_in+2400000.5d0
if rjd then jd=jd_in+2440000.5d0
;
jd_long=long(jd)
jdf=jd-jd_long
index=where(jdf ge 0.5,count)
if count gt 0 then begin
	jd_long(index)=jd_long(index)+1
	jdf(index)=jdf(index)-1
endif
a=jd_long
index=where(jd_long ge 2299161L,count)
if count gt 0 then begin
	a(index)=long((jd_long(index)-1867216-0.25)/36524.25)
	a(index)=jd_long(index)+1+a(index)-a(index)/4
endif
b=a+1524
c=long(6680+(b-2439870-122.1)/365.25)
d=long(365*c+c/4)
e=long((b-d)/30.6001)
day=b-d-long(30.6001*e)
month=e-1
index=where(month gt 12,count)
if count gt 0 then month(index)=month(index)-12
year=c-4715
index=where(month gt 2,count)
if count gt 0 then year(index)=year(index)-1
index=where(year lt 0,count)
if count gt 0 then year(index)=year(index)-1
day=day+0.5+jdf
df=day-long(day)
;
return,constrictordate(year,month,day)
;
end
;-------------------------------------------------------------------------------
function date2jd,dates,mjd=mjd
;
; Return JD at 0h UT
; Input dates have format YYYY-MM-DD
;
parsedate,dates,y,m,d
return,julian(y,m,d,mjd=mjd)
;
end
;-------------------------------------------------------------------------------
function jd2mjd,jd_in
;
; Return mjd for jd
;
return,jd_in-2400000.5d0
;
end
;-------------------------------------------------------------------------------
function mjd2jd,mjd_in
;
; Return jd for mjd
;
return,mjd_in+2400000.5d0
;
end
;-------------------------------------------------------------------------------
function by2jd,year
;
; Compute Julian date corresponding to Besselian year.
;
return,(year-1900.d0)*365.242198781d0+2415020.31352d0
;
end
;-------------------------------------------------------------------------------
function jd2by,jd
;
; Compute Besselian year from Julian date
;
return,1900.d0+(jd-2415020.31352d0)/365.242198781d0
;
end
;-------------------------------------------------------------------------------
function by2jy,by
;
; Return Julian Year for Besselian Year.
; Reference: Mason & Hartkopf, IAU Comm. G1, Circular #192
;
return,(by*0.999978641d0)+0.041439661d0
;
end
;-------------------------------------------------------------------------------
function jy2jd,year
;
; Compute Julian date corresponding to Julian year.
;
return,(year-2000.0d0)*365.25d0+2451545.0d0
;
end
;-------------------------------------------------------------------------------
function jd2jy,jd,mjd=mjd
;
; Compute Julian year from Julian date
; if mjd then jd=jd+2400000.5d0
;
if keyword_set(mjd) then begin
	return,2000.0d0+((jd+2400000.5d0)-2451545.0d0)/365.25
endif else begin
	return,2000.0d0+(jd-2451545.0d0)/365.25
endelse
;
end
;-------------------------------------------------------------------------------
function jy2by,jy
;
; Return Besselian Year for Julian Year
; Reference: Mason & Hartkopf, IAU Comm. G1, Circular #192
;
return,(jy-0.041439661d0)/0.999978641d0
;
end
;-------------------------------------------------------------------------------
function date2jy,dates
;
; Return JY for JD at 0h UT
; Input dates have format YYYY-MM-DD
;
return,jd2jy(date2jd(dates))
;
end
;-------------------------------------------------------------------------------
function reversedate,date,ymd=ymd
;
; Return YYYY-MM-DD from input DD-MM-YYYY or vice-versa if ymd=1
;
if keyword_set(ymd) then $ 
return,strmid(date,8,2)+'-'+strmid(date,5,2)+'-'+strmid(date,0,4)
;
return,strmid(date,6,4)+'-'+strmid(date,3,2)+'-'+strmid(date,0,2)
;
end
;-------------------------------------------------------------------------------
function previousdate,date,ndays
;
if n_elements(ndays) eq 0 then ndays=1
;
parsedate,date,y,m,d
return,jd2date(julian(y,m,d)-ndays)
;
jd2date,julian(y,m,d)-ndays,year,month,day
return,constrictordate(year,month,day)
;
end
;-------------------------------------------------------------------------------
function nextdate,date,ndays
;
if n_elements(ndays) eq 0 then ndays=1
;
parsedate,date,y,m,d
return,jd2date(julian(y,m,d)+ndays)
;
jd2date,julian(y,m,d)+ndays,year,month,day
return,constrictordate(year,month,day)
;
end
;-------------------------------------------------------------------------------
pro parsedate,date,y,m,d,h
;
; Extracts year, month, and day (long integers) from CONSTRICTOR format date
; string (e.g.: 1994-05-31). Optionally return hours in h if single date
; string has fractional day.
;
y=long(strmid(date,0,4))
m=long(strmid(date,5,2))
d=long(strmid(date,8,2))
h=0.0
;
; Add a test using STRTRIM to remove trailing blanks from date. 
; If the STRLEN after removal of trailing blanks > 10 then a
; fractional day number is present.
;
if n_elements(date) eq 1 and strlen(strtrim(date(0))) gt 10 then begin
	h=float(strmid(date,10,strlen(date)-10))*24
endif
;
end
;-------------------------------------------------------------------------------
pro parsemark3,mark3_date,y,m,d
;
; Extracts year, month, and day (long integers) from Mark3 date (YYMMDD).
;
m3d=long(mark3_date)
y=m3d/10000
m3d=abs(m3d)
m=(m3d-abs(y)*10000)/100
d=(m3d-abs(y)*10000-m*100)
y=1900+y
;
end
;-------------------------------------------------------------------------------
pro parsepti,pti_date,y,m,d
;
; Extracts year, month, and day (long integers) from PTI date,
; e.g. 3/1/99/4:34:5
;
n=n_elements(pti_date)
y=lonarr(n)
m=y
d=y
for i=0,n-1 do begin
	words=nameparse(pti_date(i),'/')
	y(i)=long(words(2))+1900 & if y(i) lt 1990 then y(i)=y(i)+100
	m(i)=long(words(0))
	d(i)=long(words(1))
endfor
;
end
;-------------------------------------------------------------------------------
pro parsehpuxdate,hpux_date,y,m,d
;
; Extracts year, month, and day (long integers) from HPUX date,
; e.g. Fri Apr 18 16:42:34 MST 1997.
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
y=long(strmid(hpux_date,24,4))
m=where(months eq strmid(hpux_date,4,3),count)+1 & m=m(0)
d=long(strmid(hpux_date,8,2))
;
end
;-------------------------------------------------------------------------------
pro parseidldate,idl_date,y,m,d
;
; Extracts year, month, and day (long integers) from IDL date,
; e.g. Mon Oct 27 17:17:13 2003. If called with just 2 arguments,
; return date/time string in variable y.
;
; Array input OK.
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
y=long(strmid(idl_date,20,4))
;
n=n_elements(idl_date)
m=lonarr(n)
for i=0L,n-1 do m(i)=where(months eq strmid(idl_date(i),4,3),count)+1
;
d=long(strmid(idl_date,8,2))
;
if n_params() eq 2 then begin
	time=strmid(idl_date,11,8)
	date=string(y)+'-'+string(m)+'-'+string(d)+'T'+strmid(idl_date,11,8)
	y=strcompress(date,/remove_all)
endif
if n_elements(idl_date) eq 1 then m=m(0)
;
end
;-------------------------------------------------------------------------------
pro parseisodate,iso_date,y,m,d
;
; Extracts year, month, and day (long integers) from ISO date,
; e.g. 2011-Sep-29
;
; Array input OK.
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
;
y=long(strmid(iso_date,0,4))
;
n=n_elements(iso_date)
m=lonarr(n)
for i=0L,n-1 do m(i)=where(months eq strmid(iso_date(i),5,3),count)+1
;
d=long(strmid(iso_date,9,2))
;
end
;-------------------------------------------------------------------------------
pro parseesodate,wdb_date,y,m,d
;
; Extracts year, month, and day (long integers) from WDB date ("03 Oct 2020")
;
months=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']
n=n_elements(wdb_date)
;
d=long(strmid(wdb_date,0,2))
m=lonarr(n)
for i=0L,n-1 do m(i)=where(months eq strmid(wdb_date(i),3,3),count)+1
if n eq 1 then m=m(0)
;
y=long(strmid(wdb_date,7,4))
;
end
;-------------------------------------------------------------------------------
