;*******************************************************************************
; File: atmospheres.pro
;
; Description:
; ------------
; Mostly contains functions related to stellar atmospheres and spectra.
; Note that for all of the functions here, lambda [nm], except for Block 7 [m].
; Note that "lambda" is an IDL function, so we use "lammda" for wavelength!
; 
; Block directory:
; ----------------
; Block 1: limbdata,limbhamme
; Block 2: limbmodel,limbgrid,limbcoeffs,limbfluxes,limblinear
;	   limbfilter,magfilter,limbfactor,limbcoeff,limbband,fluxband,limbmaps
; Block 3: limbnpoi
; Block 4: kurudata,jasondata
; Block 5: kurumodel,kurugrid,kurucoeffs,kurufluxes,
;	   kuruband
; Block 6: jasongrid,jasoncoeffs,jasonfluxes
; Block 7: pickles,glushneva,cohen,vanboekel
;
;************************************************************************Block 1
pro limbdata,restore=do_restore
;
; Read Van Hamme's monochromatic limb darkening coefficients and fluxes
; and interpolate onto grid from B to N bands (2 nm stepsize).
; Also add NextGen stellar fluxes for cooler stars. Save all data to
; XDR file. NextGen data from https://phoenix.ens-lyon.fr/Grids/
;
; If called with restore=1, just re-read the limbdata.xdr file
;
common LimbBase,limb_data
;
limbfile=!atmospheres_dir+'vanhamme/limbdata.xdr'
f=file_search(limbfile) & f=f[0]
if n_elements(do_restore) eq 0 then do_restore=0
if do_restore and strlen(f) ne 0 then begin
	restore,limbfile
	print,'Restored '+limbfile+'.'
	print,'Min. Teff = ',min(limb_data.t),', max. Teff = ',max(limb_data.t)
	m=n_elements(limb_data.m)
	min_m=min(limb_data.m)
	u=fltarr(m)
	t=limb_data.t
	for i=0,m-1 do begin
		index=where(limb_data.n eq (i+min_m))
		u[i]=mean(limb_data.u[index])
	endfor
	index=where(u ne 0,count)
	if count gt 0 then begin
		print,'Limb-darkening coefficients non-zero for ' $
		+string(nint(min(t[index])),format='(i5)')+' < T < ' $
		+string(nint(max(t[index]),/long),format='(i5)')
	endif else begin
		print,'Warning: all limb-darkening coefficients zero!'
	endelse
	print,'See '+!atmospheres_dir+'vanhamme/Notes.txt for more information.'
	return
endif else print,'File limbdata.xdr not found, recreate instead...'
;
print,'Reading VanHamme data...'
modelfile=!atmospheres_dir+'vanhamme/table3a'
status=dc_read_free(modelfile,m,t,g,/col)
m=fix(m)	; m = Model number for t = Teff, g = log(g)
;
datafile=!atmospheres_dir+'vanhamme/table3c.xdr'
f=file_search(datafile) & f=f[0]
if strlen(f) ne 0 then restore,datafile else begin
	datafile=!atmospheres_dir+'vanhamme/table3c'
	status=dc_read_free(datafile,n,w,u,qlin,x,y,qlog,a,b,qsqr,f,/col)
	save,n,w,u,qlin,x,y,qlog,a,b,qsqr,f,filename=datafile+'.xdr'
endelse
n=fix(n)
; n = model number, w = wavelength, f = flux, u = lin. coeff, x,y = log coeff.
;
; Remove all models with Teff < teff_limit 
; (only 3500, 4000, 8000, 10000 K allowed)
teff_limit=10000
if getenv('USER') eq !owner then begin
	index=where(t lt teff_limit,count)
	for i=0,count-1 do begin
		nndex=where(n eq m[index[i]])
		n[nndex]=0	; The lowest valid model number is 1
	endfor
	if count gt 0 then begin
		m[index]=0
		index=where(m ne 0)
		m=m[index]
		t=t[index]
		g=g[index]
	endif
endif
;
; Use 300 nm < lambda < 20 mu for interpolation
i=where(w ge 300 and w le 20000 and n ne 0)
n0=n[i] & w0=w[i] & f0=f[i] & u0=u[i] & x0=x[i] & y0=y[i] & a0=a[i] & b0=b[i]
nw=n_elements(unique(w0))
nm=n_elements(unique(n0))
if n_elements(w0)/nw ne nm then begin
	print,'File corrupt: '+datafile
	return
endif
l0=unique(w0)
; Prepare grid from B to N bands
l=findgen(((14000-min(w0))/2)+1)*2+min(w0)
nl=n_elements(l)
n=intarr(n_elements(l)*nm)
w=fltarr(n_elements(l)*nm)
f=fltarr(n_elements(l)*nm)
u=fltarr(n_elements(l)*nm)
x=fltarr(n_elements(l)*nm)
y=fltarr(n_elements(l)*nm)
a=fltarr(n_elements(l)*nm)
b=fltarr(n_elements(l)*nm)
; Interpolate onto grid variables
for i=0,nm-1 do begin
	i0=i*nw
	i1=(i+1)*nw-1
	j0=i*nl
	j1=(i+1)*nl-1
	w[j0:j1]=l
;	n(j0:j1)=i+1 bug!
	n[j0:j1]=n0[i0]
	f[j0:j1]=interpol(f0[i0:i1],l0,l)
	u[j0:j1]=interpol(u0[i0:i1],l0,l)
	x[j0:j1]=interpol(x0[i0:i1],l0,l)
	y[j0:j1]=interpol(y0[i0:i1],l0,l)
	a[j0:j1]=interpol(a0[i0:i1],l0,l)
	b[j0:j1]=interpol(b0[i0:i1],l0,l)
endfor
;
; Replace all models for Teff < teff_limit with NextGen models
if getenv('USER') eq !owner then begin
	print,'Reading NextGen data...'
;	Use only NextGen models cooler than teff_limit
	if teff_limit eq 3500 then $
	modelfiles=file_search(!atmospheres_dir+'nextgen/lte[2-3][0-4]*.xdr')
	if teff_limit eq 4000 then $
	modelfiles=file_search(!atmospheres_dir+'nextgen/lte[2-3][0-9]*.xdr')
	if teff_limit eq 8000 then $
	modelfiles=file_search(!atmospheres_dir+'nextgen/lte[2-7][0-9]*.xdr')
	if teff_limit eq 10000 then $
	modelfiles=file_search(!atmospheres_dir+'nextgen/lte[2-9][0-9]*.xdr')
	models=specname(modelfiles)
	k=n_elements(modelfiles)
	m1=max(m)+1+indgen(k)		; assign model numbers
;	Get Teff and Log(g) from the filename!
	t1=float(strmid(models,3,2))*100
	g1=float(strmid(models,6,3))
	m=[m,m1]
	t=[t,t1]
	g=[g,g1]
	restore,modelfiles[0]		; restores l,s
	nw=n_elements(l)
	n1=intarr(k*nw)
	w1=fltarr(k*nw)
	f1=fltarr(k*nw)
	u1=fltarr(k*nw)
	x1=fltarr(k*nw)
	y1=fltarr(k*nw)
	a1=fltarr(k*nw)
	b1=fltarr(k*nw)
	for i=0,k-1 do begin
		restore,modelfiles[i]	; restores l,s
		n1[i*nw:(i+1)*nw-1]=m1[i]
		w1[i*nw:(i+1)*nw-1]=l
;		Apply scale factor (nextgen_scale)
		f1[i*nw:(i+1)*nw-1]=s*3.61267e-08
	endfor
	n=[n,n1]
	w=[w,w1]
	f=[f,f1]
	u=[u,u1]
	x=[x,x1]
	y=[y,y1]
	a=[a,a1]
	b=[b,b1]
endif
;
limb_data=alloc_limb_data(m,t,g,n,w,f,u,x,y,a,b)
;
; Save the XDR file
save,limb_data,filename=limbfile
;
end
;-------------------------------------------------------------------------------
pro limbhamme,limb_table
;
; Read Van Hamme's integrated limb darkening coefficients and return in table.
;
data=''
status=dc_read_fixed(!atmospheres_dir+'vanhamme/limb.dat',data,/col, $
	resize=[1],format='(a132)')
num_mod=408
num_bnd=17
num_rec=num_mod*num_bnd
;
band=strarr(num_rec)
ulin=fltarr(num_rec,/nozero)
teff=fltarr(num_rec,/nozero)
logg=fltarr(num_rec,/nozero)
;
record=0
n=0
for i=0,num_mod-1 do begin
	pos=strpos(data[record],'Teff')
	t_eff=float(strmid(data[record],pos+6,6))
	pos=strpos(data[record],'log g')
	log_g=float(strmid(data[record],pos+7,4))
	for j=record+1,record+num_bnd do begin
		band[n]=strmid(data[j],2,1)
		ulin[n]=float(strmid(data[j],8,5))
		teff[n]=t_eff
		logg[n]=log_g
		n=n+1
	endfor
	record=record+1+num_bnd
endfor
;
limb_table=alloc_limb_table(band,ulin,teff,logg)
;
end
;************************************************************************Block 2
function limbmodel,m,t,g,teff,logg,teff_model,closest=closest
;
; Return the model number closest to teff and logg. Now modified to
; return two models bracketing requested Teff with closest in log(g).
;
common LocalLimbModel,model_old
;
if n_elements(closest) eq 0 then closest=0
;
if n_elements(model_old) eq 0 then model_old=-1
n=n_elements(m)
;
IF not closest THEN BEGIN
;
; Find closest bracketing models for Teff (closest in logg)
j=where(t le teff[0],count)
if count eq 0 then j=indgen(n)	; use closest if outside available range
d_t=abs(teff[0]-t[j])
jndex=where(d_t eq min(d_t))
k=where(g le logg[0])
k=indgen(n)		; just use closest for log(g)
d_g=abs(logg[0]-g[k])
kndex=where(d_g eq min(d_g))
i=whereequal(j[jndex],k[kndex])
i1=j[jndex[i]]
;
j=where(t ge teff[0],count)
if count eq 0 then j=indgen(n)	; use closest if outside available range
d_t=abs(teff[0]-t[j])
jndex=where(d_t eq min(d_t))
k=where(g ge logg[0])
k=indgen(n)		; just use closest for log(g)
d_g=abs(logg[0]-g[k])
kndex=where(d_g eq min(d_g))
i=whereequal(j[jndex],k[kndex])
i2=j[jndex[i]]
;
i=where(k[kndex[0]] eq model_old,count)
if count eq 0 and long(t[i1]) ne long(t[i2]) then begin
print,'Model: Teff [K] interpolated between ',long(t[i1]),',',long(t[i2]), $
		', log(g)=',g[k[kndex[0]]],format='(a,i5,a,i5,a,f3.1)'
model_old=[model_old,k[kndex[0]]]
endif
;
return,m[[i1,i2]]
;
ENDIF ELSE BEGIN
;
; Find closest model
d_t=abs(teff[0]-t)
d_g=abs(logg[0]-g)
index=where(d_t eq min(d_t)) & j=where(d_g[index] eq min(d_g[index])) & j=j[0]
teff_model=t[index[j]]
k=where(index[j] eq model_old,count)
if count eq 0 then begin
        print,'Using model: Teff=',long(teff_model), $
                       ', log(g)=',g[index[j]],format='(a,i5,a,f3.1)'
	model_old=[model_old,index[j]]
endif
return,m[index[j]]
;
ENDELSE
;
end
;-------------------------------------------------------------------------------
function limbgrid,teff,logg,lammda,limbdu,fluxes,quiet=quiet,closest=closest
;
; Return linear limb darkening coefficients on a grid of wavelengths for
; a model specified through teff and logg. These coefficients can be integrated
; by the calling routine over an arbitrary band pass. Note that a model from
; a grid which best matches the inputs is selected. The model atmosphere
; data was computed by Van Hamme. "Interpolate" to the requested Teff
; using the ratio of the black body spectra at the requested and grid Teff.
;
common LimbBase,limb_data
common LocalLimbgrid,blackbody_notification
;
if n_elements(blackbody_notification) eq 0 then blackbody_notification=1 $
					   else blackbody_notification=0
;
if n_elements(quiet) eq 0 then quiet=0
if n_elements(closest) eq 0 then closest=0
;
if n_elements(limb_data) eq 0 then limbdata,/restore
;
if teff eq 0 then begin
	print,'Error (limbgrid): Teff zero on input!'
	return,0
endif
;
; Find closest model(s)
models=limbmodel(limb_data.m,limb_data.t,limb_data.g,teff,logg,teff_model, $
	closest=closest)
;
; Extract corresponding grid
index=where(limb_data.n eq models[0])
lammda=limb_data.w[index]
limbdu=limb_data.u[index]
fluxes=limb_data.f[index]
; The following is experimental when a cool star grid is requested
mndex=where(limb_data.m eq models[0])
if teff lt unique(limb_data.t[mndex]) then begin
	if not quiet and blackbody_notification then $
		print,'Using matching blackbody!'
;	Use scale black body when Kurucz template is not available
	rt=[-703.571,1.15714]		; Coefficients for blackbody Teff
	rs=[1.72650,-0.000128975]	; Coefficients for scale factor
	l=lammda*1e-9
	fluxes=blackbody(poly(teff,rt),l)*poly(teff,rs)
	return,0
endif
;
; If two (different) models are returned...
if n_elements(models) eq 2 then begin
if models[0] ne models[1] then begin
	index=where(limb_data.n eq models[1])
	lammda2=limb_data.w[index]	; should be identical to lammda
	limbdu2=limb_data.u[index]
	fluxes2=limb_data.f[index]
	t1=alog10(limb_data.t[where(limb_data.m eq models[0])]) & t1=t1[0]
	t2=alog10(limb_data.t[where(limb_data.m eq models[1])]) & t2=t2[0]
; 	Interpolate fluxes between the two
	f1=alog10(fluxes)
	f2=alog10(fluxes2)
	f=f1+(f2-f1)*(alog10(teff)-t1)/(t2-t1)
	fluxes=10^f
; 	Interpolate limb darkening coefficient between the two
	u1=limbdu
	u2=limbdu2
	u=u1+(u2-u1)*(alog10(teff)-t1)/(t2-t1)
endif
endif
;
return,0
;
end
;-------------------------------------------------------------------------------
function limbcoeffs,teff,logg,lammda
;
; Return interpolated array of linear limb darkening coefficients for
; input array of wavelengths lambda, and stellar model atmosphere for
; T_eff=teff and log(g)=logg. Lambda is in nm.
;
; Get coefficients on Van Hamme's grid
s=limbgrid(teff,logg,mlambda,mlimbdu,mfluxes)
;
; High tension (10) spline, i.e. nearly equal linear interpolation
return,spline(mlambda,mlimbdu,lammda,10)
;
end
;-------------------------------------------------------------------------------
function limbfluxes,teff,logg,lammda
;
; Return interpolated array of model atmosphere fluxes for
; input array of wavelengths lambda, and stellar model atmosphere for
; T_eff=teff and log(g)=logg. Lambda is in nm.
;
; Get fluxes on Van Hamme's grid
s=limbgrid(teff,logg,mlambda,mlimbdu,mfluxes)
;
; High tension (10) spline, i.e. nearly equal linear interpolation
return,spline(mlambda,mfluxes,lammda,10)
;
end
;-------------------------------------------------------------------------------
function limblinear,teff,logg,lammda,fluxes
;
; Combines the functions limbcoeffs and limbfluxes.
;
; Get coefficients and fluxes from Van Hamme's grid
s=limbgrid(teff,logg,glambda,glimbdu,gfluxes)
;
; High tension (10) spline, i.e. nearly equal linear interpolation
fluxes=interpol(gfluxes,glambda,lammda)
return,interpol(glimbdu,glambda,lammda)
;
end
;-------------------------------------------------------------------------------
function limbfilter,teff,logg,filter
;
; Integrate linear limb darkening coefficients over a named filter.
; Model atmosphere parameters teff and logg in K and cgs, respectively.
;
if limbgrid(teff,logg,lammda,limbdu,fluxes) ne 0 then return,-1
;
if n_elements(filter) eq 0 then begin
	print,'***Error(LIMBFILTER): no filter specified!'
	return,-1
endif
;
case filter of
'U':	begin
	tm=johnson_u(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'B':	begin
	tm=johnson_b(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'V':	begin
	tm=johnson_v(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'R':	begin
	tm=johnson_r(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'I':	begin
	tm=johnson_i(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'J':	begin
	tm=johnson_j(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'H':	begin
	tm=johnson_h(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'K':	begin
	tm=johnson_k(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'Hp':	begin
	tm=hipparcos_hp(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'u': 	begin
	tm=stroemgren_u(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'v': 	begin
	tm=stroemgren_v(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'b': 	begin
	tm=stroemgren_b(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'y': 	begin
	tm=stroemgren_y(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'm800': begin
	tm=mark3_800(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'm550': begin
	tm=mark3_550(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'm500': begin
	tm=mark3_500(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'm450': begin
	tm=mark3_450(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'mhan': begin
	tm=mark3_han(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'mhab': begin
	tm=mark3_hab(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
'm500o':begin
	tm=mark3_500o(lammda)*fluxes
	tu=total(tm*limbdu)/total(tm)
	end
else:	begin
	print,'Warning(LIMBFILTER): unknown filter!'
	tu=-1
	end
endcase
;
return,tu
;
end
;-------------------------------------------------------------------------------
function magfilter,teff,logg,filter,radius,parallax
;
; Integrate model stellar atmosphere over a named filter.
; Atmosphere parameters teff and logg in K and cgs, respectively.
; Radius in solar units, parallax in mas.
; Subtract Vega zero points.
;
forward_function magfilter
;
; Vega zero point data
vega_filter=['U','B','V','R','I']
vega_colors=[0.03,0.03,0.03,0.07,0.10]		; from Johnson et al. 1966
vega_colors=[0.026,0.026,0.026,0.066,0.096]	; Deane Peterson (version 2)
; Get Vega zero points with model T=9500 K, d=3.24 mas, px=129 mas, (mass=2)
vega_zp=[-22.7879,-22.4989,-21.9033,-21.1903,-20.5634]-vega_colors
; Get Vega zero points with model T=9442 K, d=3.24 mas, px=128.9 mas, (mass=2.4)
vega_zp=[-22.7633,-22.4774,-21.8882,-21.1780,-20.5523]-vega_colors
; Calibration with measured flux in V (Allen's Astrophysical Quantities, p.150)
vega_flux_v=3.44e-8
vega_zp=vega_zp-vega_zp[2]-2.5*alog10(vega_flux_v)
; Adding data for infrared filters
vega_filter=[vega_filter,'J','H','K','L','M','N','Q']
vega_fluxes=[3.31e-9,1.15e-9,4.14e-10,6.59e-11,2.11e-11,9.63e-13,7.18e-14]
vega_zp=[vega_zp,-2.5*alog10(vega_fluxes)]
;
; Here, we choose to derive Vega zero points from the model atmosphere
if n_params() eq 1 then begin
;	This call is for Vega
	filter=teff
	teff=9500.
	logg=4.1
	radius=2.73
	parallax=128.9
endif else begin
	if n_elements(radius) eq 0 then radius=2.73
	if n_elements(parallax) eq 0 then parallax=128.9
endelse
if limbgrid(teff,logg,lammda,limbdu,fluxes) ne 0 then return,-1
;
fluxes=fluxes*(radius/2.73)^2*(parallax/128.9)^2
;
if n_elements(filter) eq 0 then begin
	print,'***Error(MAGFILTER): no filter specified!'
	return,-1
endif
;
case filter of
'U':	begin
	tm=johnson_u(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[0]
	end
'B':	begin
	tm=johnson_b(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[1]
	end
'V':	begin
	tm=johnson_v(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[2]
	end
'R':	begin
	tm=johnson_r(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[3]
	end
'I':	begin
	tm=johnson_i(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[4]
	end
'J':	begin
	tm=johnson_j(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[5]
	end
'H':	begin
	tm=johnson_h(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[6]
	end
'K':	begin
	tm=johnson_k(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[7]
	end
'L':	begin
	tm=johnson_lp(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[8]
	end
'M':	begin
	tm=johnson_m(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[9]
	end
'N':	begin
	tm=johnson_n(lammda)
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[10]
	end
'Q':	begin
	tm=johnson_q[lammda]
	flux=total(tm*fluxes)/total(tm)
	zp=vega_zp[11]
	end
'Hp':	begin
	tm=hipparcos_hp(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'u': 	begin
	tm=stroemgren_u(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'v': 	begin
	tm=stroemgren_v(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'b': 	begin
	tm=stroemgren_b(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'y': 	begin
	tm=stroemgren_y(lammda)*atmosphere(lammda)*aluminium(lammda)^2 $
	  *pm1p21(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm800': begin
	tm=mark3_800(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm550': begin
	tm=mark3_550(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm500': begin
	tm=mark3_500(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm450': begin
	tm=mark3_450(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'mhan': begin
	tm=mark3_han(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'mhab': begin
	tm=mark3_hab(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
'm500o':begin
	tm=mark3_500o(lammda)
	flux=total(tm*fluxes)/total(tm)
	end
else:	begin
	print,'Warning(MAGFILTER): unknown filter!'
	flux=-1
	end
endcase
;
if n_params() eq 1 then vega_zp=0 $
		   else vega_zp=magfilter(filter)
return,-2.5*alog10(flux)-vega_zp
;
; Using Vega zero points from literature
; return,-2.5*alog10(flux)+zp
;
end
;-------------------------------------------------------------------------------
function limbfactor,coeff
;
; Return the factor which converts a uniform disk diameter
; to a limb darkened diameter using Hanbury Browns formula
; (Hanbury Brown et al. 1974, "The effect of limb-darkening...")
;
return,sqrt((1-coeff/3)/(1-7*coeff/15))
;
end
;-------------------------------------------------------------------------------
function limbcoeff,factor
;
; Inverse function to limbfactor. Factor is LDD/UDD (>1).
;
f=factor^2
return,(1-f)/(1./3-7*f/15)
;
end
;-------------------------------------------------------------------------------
function limbband,teff,logg,clambda,dlambda
;
; Integrate linear limb darkening coefficients over rectangular band pass
; defined through center wavelength clambda [nm] and full width dlambda.
;
if limbgrid(teff,logg,lammda,limbdu,fluxes) ne 0 then return,-1
;
num=n_elements(lammda)
transmission=fltarr(num)
;
n=n_elements(clambda)
tu=fltarr(n)
;
for i=0,n-1 do begin
;
	transmission[*]=0
	index=where(abs(lammda-clambda[i]) lt dlambda[i]/2,count)
	if count gt 0 then begin
		transmission[index]=1
		tm=transmission*fluxes
		tu[i]=total(tm*limbdu)/total(tm)
	endif else begin
;		If the grid is not fine enough
		tu[i]=interpol(limbdu,lammda,clambda[i])
	endelse
;
endfor
;
return,tu
;
end
;-------------------------------------------------------------------------------
function fluxband,teff,logg,clambda,dlambda
;
; Average stellar flux over rectangular band pass
; defined through center wavelength clambda [nm] and full width dlambda [nm].
;
if limbgrid(teff,logg,lammda,limbdu,fluxes) ne 0 then return,-1
;
num=n_elements(lammda)
tm=fltarr(num)
;
n=n_elements(clambda)
flux=fltarr(n)+1
for i=0,n-1 do begin
	index=where(abs(lammda-clambda[i]) lt dlambda[i]/2,count)
	if count gt 0 then tm[index]=1
	if total(tm) ne 0 then flux[i]=total(tm*fluxes)/total(tm)
	tm=tm*0
endfor
;
return,flux
;
end
;-------------------------------------------------------------------------------
function limbmaps,model,lammda,num=num
;
; Return an array of stellar disk maps for each wavelength using various
; limb darkening laws. Note: input lambda is in [nm].
;
common LimbBase,limb_data
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
if n_elements(limb_data) eq 0 then limbdata,/restore
;
; Find closest model
m=limbmodel(limb_data.m,limb_data.t,limb_data.g,model.teff,model.logg,/closest)
;
; Find the model data
index=where(limb_data.n eq m)
;
; Interpolate values for observed wavelengths
limbdu=interpol(limb_data.u[index],limb_data.w[index],lammda)
limbdx=interpol(limb_data.x[index],limb_data.w[index],lammda)
limbdy=interpol(limb_data.y[index],limb_data.w[index],lammda)
fluxes=interpol(limb_data.f[index],limb_data.w[index],lammda)
; limbdu=spline(limb_data.w(index),limb_data.u(index),lammda,10)
; limbdx=spline(limb_data.w(index),limb_data.x(index),lammda,10)
; limbdy=spline(limb_data.w(index),limb_data.y(index),lammda,10)
; fluxes=spline(limb_data.w(index),limb_data.f(index),lammda,10)
;
; Create a map with element identifications
if n_elements(num) eq 0 then num=33 	; must be odd number
num=2*(num/2)+1
map=indgen(num*num)
;
; Copy coordinates
xcc=float((map mod num)-(num-1)/2)/((num-1)/2)	; [-1,1]
ycc=float((map  /  num)-(num-1)/2)/((num-1)/2)	; [-1,1]
;
; Compute radius squared
rsq=xcc^2+ycc^2
;
; Use only the visible hemisphere
index=where(rsq lt 1,count)
rsq=rsq[index]
xcc=xcc[index]
ycc=ycc[index]
;
; Compute mu, the cosine of the angle between the LOS and the surface normal
mu=sqrt(1-rsq)
;
; Create array of to hold maps
map=alloc_map(count,lonarr(count),fltarr(count),fltarr(count),fltarr(count))
maps=replicate(map,n_elements(lammda))
;
for i=0,n_elements(lammda)-1 do begin
	case model.type of
	3:	begin
;		Linear limb darkening law
		fcc=fluxes[i]*(1-limbdu[i]*(1-mu))
		end
	4:	begin
;		Logarithmic limb darkening law
		fcc=fluxes[i]*(1-limbdx[i]*(1-mu)-limbdy[i]*mu*alog(mu))
		end
	5:	begin
;		Linear limb darkening law
		fcc=fluxes[i]*(1-limbdu[i]*(1-mu))
		end
	endcase
	maps[i].xcc=xcc*model.diameter/2
	maps[i].ycc=ycc*model.diameter/2
	maps[i].fcc=fcc
endfor
;
return,maps
;
end
;************************************************************************Block 3
function limbnpoi,teff,logg
;
; Return array of integrated linear limb darkening coefficients for the
; channels in GenConfig.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(GenConfig) eq 0 then begin
	print,'***Error(LIMBNPOI): GenConfig undefined!'
	return,-1
endif
;
if limbgrid(teff,logg,lammda,limbdu,fluxes) ne 0 then return,-1
;
tu=fltarr(max(GenConfig.NumSpecChan),GenConfig.NumOutBeam)
for i=0,GenConfig.NumOutBeam-1 do begin
for j=0,GenConfig.NumSpecChan[i]-1 do begin
	tm=generic_c(lammda,i,j)*fluxes
	tu[j,i]=total(tm*limbdu)/total(tm)
endfor
endfor
;
return,tu
;
end
;-------------------------------------------------------------------------------
function fluxnpoi,teff,logg
;
; Return array of integrated stellar fluxes for the
; channels in GenConfig.
;
common SysConfig,SystemId,Date,MetroConfig,GenConfig,GeoParms,GenInfo,GeoInfo
;
if n_elements(GenConfig) eq 0 then begin
	print,'***Error(FLUXNPOI): GenConfig undefined!'
	return,-1
endif
;
if limbgrid(teff,logg,lammda,limbdu,fluxes) ne 0 then return,-1
;
tf=fltarr(max(GenConfig.NumSpecChan),GenConfig.NumOutBeam)
for i=0,GenConfig.NumOutBeam-1 do begin
for j=0,GenConfig.NumSpecChan[i]-1 do begin
	tm=generic_c(lammda,i,j)
	tf[j,i]=total(tm*fluxes)/total(tm)
endfor
endfor
;
return,tf
;
end
;************************************************************************Block 4
pro kurudata,filename
;
; Given a Kurucz atmosphere model file (e.g. ip00k2.pck19), read all
; models and save each one using the XDR format. This procedure needs
; to be called only once for installing OYSTER.
;
; The Kurucz models give I(ergs/cm**2/s/hz/ster) whereas filter bandpasses
; in OYSTER are usually expressed in and integrated over wavelength.
; Therefore, the fluxes have to be multiplied by c/lambda^2.
;
common Constants,c_light,pi_circle,e_euler,i_complex,a_disp,b_disp
;
; First see if 477 xdr files exist. If so, we're done. If not,
; try and extract the xdr files from a Kurucz atmosphere file
SPAWN,'ls '+!atmospheres_dir+'kurucz/*xdr > xdr.list'
nlines = FILE_LINES('xdr.list')
if nlines eq 477 then begin
   print,' '
   print,' Kurucz atmosphere files exist in '+!atmospheres_dir+'kurucz/'
   print,' '
   return
endif
;
file=!atmospheres_dir+'kurucz/'+filename
r=file_search(file)
if strlen(r[0]) eq 0 then begin
	print,'***Error(KURUDATA): file does not exist!'
	return
endif
spawn,'grep TEFF '+file,r
num_model=n_elements(r)
kur_teffs=fltarr(num_model)
kur_loggs=fltarr(num_model)
kur_files=strarr(num_model)
;
case filename of
	'Kurucz_model':	begin
			end
	else:		begin
			num_header=3
			num_rec=1221
			format='(F9.2,E10.3,16F6.0)'
			end
endcase
;
openr,unit,file,/get_lun
line=''
;
for i=0,num_model-1 do begin
	if eof(unit) then begin
		print,'***Error(KURUDATA): unexpected End-of-file!'
		return
	endif
	readf,unit,line,format='(a77)'
	words=nameparse(line)
	kur_teff=float(words[1])
	kur_logg=float(words[3])
	kur_teffs[i]=kur_teff
	kur_loggs[i]=kur_logg
	extn='.'+string(fix(10*kur_logg),format='(i2.2)') $
		+string(long(kur_teff),format='(i5.5)')+'.xdr'
	readf,unit,line
	readf,unit,line,format='(a115)'
	kur_mu=[0.0,reverse(float(nameparse(line)))]
	num_mu=n_elements(kur_mu)
	kur_I=fltarr(num_mu,num_rec)
	kur_l=fltarr(num_rec)
	values=fltarr(num_mu-1+1)
	for j=0,num_rec-1 do begin
		readf,unit,values,format=format
		kur_l[j]=values[0]
		values[2:num_mu-1]=values[2:num_mu-1]/1e5*values[1]
		kur_I[*,j]=[0.0,reverse(values[1:num_mu-1])]*c_light/kur_l[j]^2
	endfor
	index=where(kur_l gt 400 and kur_l le 20000)
	kur_l=kur_l[index]
	kur_I=kur_I[*,index]
	save,kur_teff,kur_logg,kur_mu,kur_l,kur_I,filename=filename+extn
	kur_files[i]=filename+extn
endfor
;
kur_model={kur_teffs:kur_teffs,kur_loggs:kur_loggs,kur_files:kur_files}
save,kur_model,filename=filename+'.models.xdr'
;
end
;-------------------------------------------------------------------------------
pro jasondata,model
;
; Reads an atmospheric model by Jason Aufdenberg. These models are computed
; for specific stars and are not general models for a (Teff,log(g)) combination.
; The model parameter must not include the ".dat" suffix!
;
; This procedure writes a ".xdr" file for faster access.
;
; Adapted from M. Wittkowski.
;
common Jason,jason_mu,jason_l,jason_I
;
file=!atmospheres_dir+'aufdenberg/'+model
r=file_search(file)
if strlen(r[0]) eq 0 then begin
	print,'***Error(JASONDATA): file does not exist!'
	return
endif
;
openr,unit,file,/get_lun
;
case model of
	'gamsge':begin
		 num_l=4199
		 end
	else	:begin
		 num_l=4200
		 end
endcase
;
readf,unit,numberangles
angles=fltarr(numberangles)
readf,unit,angles
jason_mu=[0.0,sqrt(1.-angles[numberangles/2:numberangles-1]^2)]
;
; lr=fltarr(2)
fluxes=fltarr(numberangles)
;
jason_I=fltarr(1+numberangles/2,num_l)
jason_l=fltarr(num_l)
;
for j=0L,num_l-1 do begin
	readf,unit,lr
	lammda=lr[0]
;	radius=lr(1)
	jason_l[j]=lammda/10.	; nm
	readf,unit,fluxes
;	Convert flux into Joule/m^2/s/m/sr
	jason_I[*,j]=[0.0,fluxes[numberangles/2:numberangles-1]/1e16]
endfor
;
free_lun,unit
;
if strupcase(getenv('USER')) eq 'CAH' then $
save,jason_mu,jason_l,jason_I, $
	filename=!atmospheres_dir+'aufdenberg/'+model+'.xdr'
;
end
;************************************************************************Block 5
function kurumodel,model,teff,logg
;
; Return the model number closest to teff and logg.
;
common LocalKuruModel,model_file_old
common Kurucz,kur_model,kur_teff,kur_logg,kur_mu,kur_l,kur_I
;
if n_elements(model_file_old) eq 0 then model_file_old=''
;
restore,!atmospheres_dir+'kurucz/'+model+'.models.xdr'
;
; Find closest model
d_t=abs(teff[0]-kur_model.kur_teffs)
d_g=abs(logg[0]-kur_model.kur_loggs)
index=where(d_t eq min(d_t)) & j=where(d_g[index] eq min(d_g[index]))
index=index[j[0]]	; in case two equidistant models were found
;
model_file=!atmospheres_dir+'kurucz/'+kur_model.kur_files[index[j]]
if model_file ne model_file_old then begin
        print,'Using model: Teff=',long(kur_model.kur_teffs[index[j]]), $
                ', log(g)=',kur_model.kur_loggs[index[j]],format='(a,i5,a,f3.1)'
	model_file_old=model_file
endif else model_file=''
;
return,model_file
;
end
;-------------------------------------------------------------------------------
function kurugrid,model,lammda,limbdu,fluxes
;
; Return linear or power law limb darkening coefficients on a grid of 
; wavelengths for a model specified through teff and logg. These coefficients 
; can be integrated by the calling routine over an arbitrary band pass. Note 
; that a model from a grid which best matches the inputs is selected. 
;
common Kurucz,kur_model,kur_teff,kur_logg,kur_mu,kur_l,kur_I
;
; Find closest model
model_file=kurumodel(model.model,model.teff,model.logg)
if strlen(model_file) ne 0 then restore,model_file
;
; Obtain coefficients through fit to intensity profile
kur_alpha=fltarr(n_elements(kur_l))
for i=0,n_elements(kur_l)-1 do begin
	y=kur_I[*,i]/kur_I[n_elements(kur_mu)-1,i]*kur_mu
	kur_alpha[i]=-1.0
	integ=1.*int_tabulated(kur_mu,y)
	if model.type eq 5 then kur_alpha[i]=(1./integ)-2.
	if model.type eq 6 then kur_alpha[i]=3.-6.*integ
endfor
;
lammda=kur_l
limbdu=kur_alpha
fluxes=kur_I[n_elements(kur_mu)-1,*]
;
return,0
;
end
;-------------------------------------------------------------------------------
function kurucoeffs,model,lammda,fluxes
;
s=kurugrid(model,mlambda,mlimbdu,mfluxes)
;
fluxes=spline8(mlambda,mfluxes,lammda)
return,spline8(mlambda,mlimbdu,lammda)
;
end
;-------------------------------------------------------------------------------
function kurufluxes,model,lammda,mu
;
common Kurucz,kur_model,kur_teff,kur_logg,kur_mu,kur_l,kur_I
;
; Find closest model
model_file=kurumodel(model.model,model.teff,model.logg)
if strlen(model_file) ne 0 then restore,model_file
;
profiles=fltarr(n_elements(kur_mu),n_elements(lammda))
for i=0,n_elements(kur_mu)-1 do $
	profiles[i,*]=spline8(kur_l,reform(kur_I[i,*]),lammda*1d9)
;
mu=kur_mu
return,profiles
;
end
;-------------------------------------------------------------------------------
function kuruband,model,clambda,dlambda
;
; Integrate linear limb darkening coefficients over rectangular band pass
; defined through center wavelength clambda [nm] and full width dlambda.
;
if kurugrid(model,lammda,limbdu,fluxes) ne 0 then return,-1
;
num=n_elements(lammda)
transmission=fltarr(num)
;
n=n_elements(clambda)
tu=fltarr(n)
;
for i=0,n-1 do begin
;
	transmission[*]=0
	index=where(abs(lammda-clambda[i]) lt dlambda[i]/2,count)
	if count gt 0 then begin
		transmission[index]=1
		tm=transmission*fluxes
		tu[i]=total(tm*limbdu)/total(tm)
	endif else begin
;		If the grid is not fine enough
		tu[i]=interpol(limbdu,lammda,clambda[i])
	endelse
;
endfor
;
return,tu
;
end
;************************************************************************Block 6
function jasongrid,model,lammda,limbdu,fluxes
;
; Return linear or power law limb darkening coefficients on a grid of 
; wavelengths for a model specified through teff and logg. These coefficients 
; can be integrated by the calling routine over an arbitrary band pass. Note 
; that a model from a grid which best matches the inputs is selected. 
;
common Jason,jason_mu,jason_l,jason_I
;
; Read atmosphere model
model_file=!atmospheres_dir+'aufdenberg/'+model.model+'.xdr'
r=file_search(model_file)
if strlen(r[0]) ne 0 then restore,model_file $
		     else jasondata,model.model
;
; Obtain coefficients through fit to intensity profile
jason_alpha=fltarr(n_elements(jason_l))
for i=0,n_elements(jason_l)-1 do begin
	y=jason_I[*,i]/jason_I[n_elements(jason_mu)-1,i]*jason_mu
	jason_alpha[i]=-1.0
	integ=1.*int_tabulated(jason_mu,y)
	if model.type eq 5 then jason_alpha[i]=(1./integ)-2.
	if model.type eq 6 then jason_alpha[i]=3.-6.*integ
endfor
;
lammda=jason_l
limbdu=jason_alpha
fluxes=jason_I[n_elements(jason_mu)-1,*]
;
return,0
;
end
;-------------------------------------------------------------------------------
function jasoncoeffs,model,lammda,fluxes
;
s=jasongrid(model,mlambda,mlimbdu,mfluxes)
;
fluxes=spline8(mlambda,mfluxes,lammda)
return,spline8(mlambda,mlimbdu,lammda)
;
end
;-------------------------------------------------------------------------------
function jasonfluxes,model,lammda,mu
;
common Jason,jason_mu,jason_l,jason_I
;
model_file=!atmospheres_dir+'aufdenberg/'+model.model+'.xdr'
r=file_search(model_file)
if strlen(r[0]) ne 0 then restore,model_file $
		     else jasondata,model.model
;
profiles=fltarr(n_elements(jason_mu),n_elements(lammda))
for i=0,n_elements(jason_mu)-1 do $
	profiles[i,*]=spline8(jason_l,jason_I[i,*],lammda*1d9)
;
mu=jason_mu
return,profiles
;
end
;************************************************************************Block 7
function pickles,starid,template=template
;
; Return the count rate as a function of wavelength for a given star.
; Spectra are taken from Pickles library for closest match in class and type.
; We use the UVK spectra in dat_uvk which are extended to 25 microns.
;
; A Stellar Spectral Flux Library: 1150–25000 Å
; A.J. Pickles, 1998
; https://iopscience.iop.org/article/10.1086/316197
;
; Keyword template: must not include "uk", e.g., "g2iv" for "ukg2iv"!
; If template is specified, use it. If not available, suggest alternatives.
; If template is not specified, use type and class from StarTable and
; find the nearest template/templates. 
;
; Return [lammda [m], flux]
;
forward_function teff_star
;
common StarBase,startable,notes
;
; 	Pickles library (file names, e.g.: a47iv )
	files= $
	["o5v"    ,"o9v"    ,"b0v"    ,"b1v"    ,"b3v", $
	 "b57v"   ,"b8v"    ,"b9v"    ,"a0v"    ,"a2v", $
	 "a3v"    ,"a5v"    ,"a7v"    ,"f0v"    ,"f2v", $
	 "f5v"    ,"f6v"    ,"f8v", $
	 "g0v"    , $
	 "g2v"    ,"g5v"    ,"g8v", $
	 "k0v"    ,"k2v"    ,"k3v"    ,"k4v", $
	 "k5v"    ,"k7v"    ,"m0v"    ,"m1v"    ,"m2v", $
	 "m2.5v"  ,"m3v"    ,"m4v"    ,"m5v"    ,"m6v", $
	 "b2iv"   ,"b6iv"   ,"a0iv"   ,"a47iv"  ,"f02iv", $
	 "f5iv"   ,"f8iv"   ,"g0iv"   ,"g2iv"   ,"g5iv", $
	 "g8iv"   ,"k0iv"   ,"k1iv"   ,"k3iv"   ,"o8iii", $
	 "b12iii" ,"b3iii"  ,"b5iii"  ,"b9iii"  ,"a0iii", $
	 "a3iii"  ,"a5iii"  ,"a7iii"  ,"f0iii"  ,"f2iii", $
	 "f5iii"  ,"g0iii"  ,"g5iii"  , $
	 "g8iii"  ,"k0iii"  , $
	 "k1iii"  ,"k2iii"  , $
	 "k3iii"  ,"k4iii", $
	 "k5iii"  ,"m0iii", $
	 "m1iii"  ,"m2iii"  ,"m3iii"  ,"m4iii"  ,"m5iii", $
	 "m6iii"  ,"m7iii"  ,"m8iii"  ,"m9iii"  , $
	 "b2ii"   ,"b5ii"   ,"f0ii"   ,"f2ii"   ,"g5ii", $
	 "k01ii"  ,"k34ii"  ,"m3ii"   ,"b0i"    ,"b1i", $
	 "b3i"    ,"b5i"    ,"b8i"    ,"a0i"    ,"a2i", $
	 "f0i"    ,"f5i"    ,"f8i"    ,"g0i"    ,"g2i", $
	 "g5i"    ,"g8i"    ,"k2i"    ,"k3i"    ,"k4i",  $
	 "m2i"]
;
; 	Pickles library (spectral types, e.g., a4.7iv)
	spectra= $
	["o5v"    ,"o9v"    ,"b0v"    ,"b1v"    ,"b3v", $
	 "b5.7v"   ,"b8v"    ,"b9v"    ,"a0v"    ,"a2v", $
	 "a3v"    ,"a5v"    ,"a7v"    ,"f0v"    ,"f2v", $
	 "f5v"    ,"f6v"    ,"f8v", $
	 "g0v"    , $
	 "g2v"    ,"g5v"    ,"g8v", $
	 "k0v"    ,"k2v"    ,"k3v"    ,"k4v", $
	 "k5v"    ,"k7v"    ,"m0v"    ,"m1v"    ,"m2v", $
	 "m2.5v"  ,"m3v"    ,"m4v"    ,"m5v"    ,"m6v", $
	 "b2iv"   ,"b6iv"   ,"a0iv"   ,"a4.7iv"  ,"f0.2iv", $
	 "f5iv"   ,"f8iv"   ,"g0iv"   ,"g2iv"   ,"g5iv", $
	 "g8iv"   ,"k0iv"   ,"k1iv"   ,"k3iv"   ,"o8iii", $
	 "b1.2iii" ,"b3iii"  ,"b5iii"  ,"b9iii"  ,"a0iii", $
	 "a3iii"  ,"a5iii"  ,"a7iii"  ,"f0iii"  ,"f2iii", $
	 "f5iii"  ,"g0iii"  ,"g5iii"  , $
	 "g8iii"  ,"k0iii"  , $
	 "k1iii"  ,"k2iii"  , $
	 "k3iii"  ,"k4iii", $
	 "k5iii"  ,"m0iii", $
	 "m1iii"  ,"m2iii"  ,"m3iii"  ,"m4iii"  ,"m5iii", $
	 "m6iii"  ,"m7iii"  ,"m8iii"  ,"m9iii"  , $
	 "b2ii"   ,"b5ii"   ,"f0ii"   ,"f2ii"   ,"g5ii", $
	 "k0.1ii" ,"k3.4ii" ,"m3ii"   ,"b0i"    ,"b1i", $
	 "b3i"    ,"b5i"    ,"b8i"    ,"a0i"    ,"a2i", $
	 "f0i"    ,"f5i"    ,"f8i"    ,"g0i"    ,"g2i", $
	 "g5i"    ,"g8i"    ,"k2i"    ,"k3i"    ,"k4i",  $
	 "m2i"]
;
;	Translate spectral types into grid cells
;	o,b,a,f,g,k,m => 1,2,3,4,5,6,7
;	i,ii,iii,iv,v => 1,2,3,4,5
;
; 	Info: library file name syntax and spectra differ in a few cases:
; 	index=where(files ne spectra)
; 	print,files(index)
; 	b57v a47iv f02iv b12iii k01ii k34ii
; 	print,spectra(index)
; 	b5.7v a4.7iv f0.2iv b1.2iii k0.1ii k3.4ii
;
;	Extract stellar types and classes of the library templates 
	spec_parse,spectra,types,classes,t2,c2
	types_bck=types
	classes_bck=classes
;
IF keyword_set(template) THEN  BEGIN
;	Use specified template, if available
	index=where(spectra eq template,count)
	if count eq 1 then begin
		sfile=!atmospheres_dir+'pickles/dat_uvk/uk'+spectra[index[0]]
	endif else begin
		spec_parse,template,type,class
		print,''
		print,'Template not available, consider these alternatives:'
		index=where(types eq type[0],count)
		print,'Same type (line profiles differ): ',spectra[index]
		index=where(classes eq class[0],count)
		print,'Same class (spectral shapes differ):'
		print,spectra[index]
		return,0
	endelse
;
ENDIF ELSE BEGIN
;
;	Find closest template matching type and class of star
	if n_elements(startable) eq 0 then get_startable,starid
; 	Make backup copy of startable
	startable_bck=startable
	st_index=where(startable.starid eq starid,count)
	if count eq 0 then begin
		get_startable,starid
		st_index=0
	endif
	mv=startable[st_index].mv
;
;	Extract stellar type (e.g., B9) and class (e.g., V) of target
	type=fix(startable[st_index].type)
	class=fix(startable[st_index].class)
;	Compute effective temperature of star
	teff_stars
	teff_star=startable[st_index].teff
;
; 	Find closest match for target class
	index=where(classes eq class,count)
	if count eq 0 then begin
		class=' class '+string(class,format='(i2)')+' ('+starid+')!'
		print,'Warning(PICKLES): substituting class V for unknown'+class
			class=5
		index=where(classes eq class)
	endif
; 	Find closest match for target type
	dt=abs(types[index]-type)
	jndex=where(dt eq min(dt))
;	Make index to index into files/spectra
	index=index[jndex]
	print,'Pickles: loading file for class '+strupcase(files[index])
	sfile=!atmospheres_dir+'pickles/dat_uvk/uk'+files[index]
	teff_temp=teff_star[spectra[index]]
	print,'Effective temperature of closest template:' $
		+strcompress(string(teff_temp))
	template='uk'+files[index]
ENDELSE
;
pfile=file_search(sfile+'.xdr') & pfile=pfile[0]
if strlen(pfile) eq 0 then begin
	pfile=sfile+'.dat'
;	print,'Reading file: '+pfile
	s=dc_read_free(pfile,lammda,fluxes,/col,ignore=['#'])
	if getenv('USER') eq !owner then $
		save,lammda,fluxes,filename=sfile+'.xdr'
endif else begin
	restore,pfile
	print,'Restored file: ',pfile
endelse
lammda=lammda*1e-10
nl=n_elements(lammda)
lammda=lammda[0:nl-2]
fluxes=fluxes[0:nl-2]
;
; If not requesting a specific template, make adjustments
IF NOT keyword_set(template) THEN BEGIN
;
; Calculate the shape correction for a mismatch in temperature and apply
teff_temp=teff_star[spectra[index]]
print,'Calculating spectral shape corr. for T_eff = ' $
	+strcompress(string(teff_star))
f_ratio=(exp(1.439d-2/(lammda*abs(teff_temp)))-1) $
       /(exp(1.439d-2/(lammda*abs(teff_star)))-1)
fluxes=fluxes*f_ratio
;
; Normalize V flux to 3.75e-9 erg/cm^2/s/A
fv0=total(fluxes*johnson_v(lammda*1e9))*10^(mv/2.5)/total(johnson_v(lammda*1e9))
factor=0.375/fv0
fluxes=fluxes*factor
;
; Extend to 25 microns to include N and Q bands, using black body
; lammda_nq=(findgen(230)/10+2.5)*1e-6
; fluxes_nq=(3.742d-30/lammda_nq^5)/(exp(1.439d-2/(lammda_nq*abs(teff)))-1)
; index=where(lammda gt 2.45e-6)
; factor_nq=median(fluxes(index))/fluxes_nq(0)
; lammda=[lammda,lammda_nq]
; fluxes=[fluxes,fluxes_nq*factor_nq]
; Obsolete: the UVK spectra already go to 25 microns!
;
ENDIF
;
; Restore original startable, if applicable
if n_elements(startable_bck) ne 0 then startable=startable_bck
;
; This is the energy (ergs) of a photon of wavelength lammda
hnu=6.6262e-27*2.997925e8/lammda
;
; Replace zero's in flux with smallest value in flux
index=where(fluxes ne 0,count)
min_flux=min(fluxes[index])
index=where(fluxes eq 0,count)
if count gt 0 then fluxes[index]=min_flux
;
; Return rate/cm^2/s/nm
; return,(fluxes/hnu)*1e-7
;
; Return spectrum
return,[[lammda],[fluxes]]
;
end
;-------------------------------------------------------------------------------
function glushneva,starid,lammda
;
; Return the count rate as a function of wavelength for a given star.
; Find the closest match in class and type from a data base of secondary
; photometric standards (Glushneva et al. 1992, A&AS 92, 1), and scale
; with the blackbody flux ratio arising from the temperature mismatch, if any.
; Also extrapolate the flux from 767.5 nm to 850 nm using the black body law.
;
; Return lammda [m]
;
forward_function teff_star
;
common StarBase,startable,notes
;
; Save startable if it exists
if n_elements(startable) eq 0 then get_startable,starid
table=startable
;
; Get the information for the star
index=where(startable.starid eq starid,count)
if count eq 0 then begin
	get_startable,starid
	index=0
endif
type=startable[index[0]].type
class=startable[index[0]].class
mv=startable[index[0]].mv
teff_stars
teff=startable[index[0]].teff
;
; Get the stars for which spectra are available
s=dc_read_free(!atmospheres_dir+'glushneva/table3',bsc,/col)
stars='BSC'+string(bsc,format='(i4.4)')
get_startable,stars
teff_stars
;
; Find the closest match and read flux data
index=where(startable.class eq class,count)
if count eq 0 then begin
	print,'***Error(GLUSHNEVA): class not found!'
	return,0
endif
startable=startable[index]
sep=abs(startable.type-type)
index=where(sep eq min(sep)) & index=index[0]
jndex=where(stars eq startable[index].starid) & jndex=jndex[0]
print,'Found matching star: ',stars[jndex]
var=fltarr(91)
s=dc_read_free(!atmospheres_dir+'glushneva/table3',var,nrec=1,nskip=jndex)
fluxes=median(var[1:85],3)
lammda=(findgen(85)*5+322.5)*1e-9
;
; Calculate the shape correction for a mismatch in temperature, normalize to V
f_ratio=(exp(1.439d-2/(lammda*abs(startable[index].teff)))-1) $
       /(exp(1.439d-2/(lammda*abs(teff)))-1)
f_ratio=f_ratio/f_ratio[46]
;
; Adjust fluxes for temperature and brightness mismatch
fluxes=fluxes*f_ratio*10^((startable[index].mv-mv)/2.5)
;
; Check the V flux against absolute calibration of 3.75e-9 erg/cm^2/s/A
fv0=total(fluxes*johnson_v(lammda*1e9))*10^(mv/2.5)/total(johnson_v(lammda*1e9))
print,'Predicted absolute flux V=0: ',fv0*10,', should be 3.75'
factor=0.375/fv0
fluxes=fluxes*factor
;
; Extend arrays for the long wavelength extrapolation
lammda=[lammda,(findgen(25)*5+747.5)*1e-9]
f_ratio=median(fluxes[80:84])/ $
	((3.742d-30/lammda[82]^5)/(exp(1.439d-2/(lammda[82]*abs(teff)))-1))
fluxes=median([fluxes,f_ratio*((3.742d-30/lammda[85:109]^5) $
		      /(exp(1.439d-2/(lammda[85:109]*abs(teff)))-1))],7)
;
; Restore original startable, if needed
if n_elements(table) ne 0 then startable=table
;
; This is the energy (ergs) of a photon of wavelength lammda
hnu=6.6262e-27*2.997925e8/lammda
;
; Return rate/cm^2/s/nm
return,(fluxes/hnu)*1e-7
;
end
;-------------------------------------------------------------------------------
function cohen,starid,lammda,template=template
;
; This function returns flux [W/cm^2/micron]
;
; Return the count rate as a function of wavelength for a given star.
; Find the closest match in class and type from a data base of spectro-
; photometric standards (Cohen), unless a template is specified, and scale
; with the blackbody flux ratio arising from the temperature mismatch, if any.
; 
; Currently, scaling fluxes with Teff is disabled!
;
; Return lammda [m]
;
common StarBase,startable,notes
;
forward_function teff_star
;
; Save any existing startable
if n_elements(startable) eq 0 then get_startable,starid
table=startable
;
; Get the information for the star
index=where(startable.starid eq starid,count)
if count eq 0 then begin
	get_startable,starid
	index=0
endif
type=startable[index[0]].type
class=startable[index[0]].class
mv=startable[index[0]].mv
teff_stars
teff=startable[index[0]].teff
;
; Check if choice of template is forced...
if n_elements(template) eq 0 then template=''
if strlen(template) gt 0 then begin
	words=nameparse(template,'_')
	template=words[0]
	f=file_search(!atmospheres_dir+'cohen/'+template+'_*.tem')
	if strlen(f[0]) eq 0 then begin
		print,'***Error: template not found!'
		return,-1
	endif
	words=nameparse(f[0],'.')
	cfile=words[0]
	xfile=file_search(!atmospheres_dir+'cohen/'+cfile+'.xdr') & xfile=xfile[0]
endif else begin
; or get the stars for which spectra are available
	f=file_search(!atmospheres_dir+'cohen/*.tem') & f=specname(f)
	n=n_elements(f)
	stars=strarr(n)
	spectra=strarr(n)
	for i=0,n-1 do begin
		words=nameparse(f[i],['_','.'])
		stars[i]=words[0]
		spectra[i]=words[1]
		if n_elements(words) eq 4 then $
		spectra[i]=spectra[i]+'.'+words[2]
	endfor
	spec_parse,spectra,types,classes,t2,c2
;
; 	Find closest match in type for this class
	index=where(classes eq class,count)
	if count eq 0 then begin
		class=' class '+string(class,format='(i2)')+' ('+starid+')!'
		print,'Warning: substituting 5 (V) for unknown'+class
		class=5
		index=where(classes eq class)
	endif
;
	dt=abs(types[index]-type)
	jndex=where(dt eq min(dt)) & jndex=jndex[0]
	cfile=!atmospheres_dir+'cohen/'+stars[index[jndex]]+'_' $
				    +spectra[index[jndex]]
	xfile=file_search(!atmospheres_dir+'cohen/'+cfile+'.xdr') & xfile=xfile[0]
endelse
if strlen(xfile) eq 0 then begin
	tfile=cfile+'.tem'
	xfile=cfile+'.xdr'
	print,'Reading file: '+tfile
	lines=''
	s=dc_read_fixed(tfile,lines,format='(a80)',/col)
;	index=where(strpos(lines,'Wavelength  Irradiance') ge 0) 
	index=where(strpos(lines,'Wavelength') ge 0 and $
		    strpos(lines,'Irradiance') ge 0) 
	index=index[0]
	m=n_elements(lines)-index-3
	lammda=fltarr(m)
	fluxes=fltarr(m)
	for i=0,m-1 do begin
		words=nameparse(lines[index+3+i])
		lammda[i]=float(words[0]); IDL 8.4 introduced function lammda
		fluxes[i]=float(words[1])
	endfor
	if getenv('USER') eq !owner then $
		save,lammda,fluxes,filename=xfile
endif else begin
	print,'Restoring file: ',xfile
	restore,xfile
endelse
lammda=lammda*1e-6
nl=n_elements(lammda)
;
; Restore original startable, if needed
if n_elements(table) ne 0 then startable=table
;
; The following is not well tested...hence we return here!
return,fluxes	; [W/cm^2/micron]
;
; Calculate the shape correction for a mismatch in temperature and apply
f_ratio=(exp(1.439d-2/(lammda*abs(teff_star(spectra[index[jndex[0]]]))))-1) $
       /(exp(1.439d-2/(lammda*abs(teff)))-1)
fluxes=fluxes*f_ratio
;
; Check the V flux against absolute calibration of 3.75e-9 erg/cm^2/s/A
fv0=total(fluxes*johnson_v(lammda*1e9))*10^(mv/2.5)/total(johnson_v(lammda*1e9))
factor=0.375/fv0
fluxes=fluxes*factor
;
; Extend to 25 microns to include N and Q bands, using black body
lammda_nq=(findgen(230)/10+2.5)*1e-6
fluxes_nq=(3.742d-30/lammda_nq^5)/(exp(1.439d-2/(lammda_nq*abs(teff)))-1)
index=where(lammda gt 2.45e-6)
factor_nq=median(fluxes[index])/fluxes_nq[0]
lammda=[lammda,lammda_nq]
fluxes=[fluxes,fluxes_nq*factor_nq]
;
; Restore original startable, if needed
if n_elements(table) ne 0 then startable=table
;
; This is the energy (ergs) of a photon of wavelength lammda
hnu=6.6262e-27*2.997925e8/lammda
;
; Return rate/cm^2/s/nm
return,(fluxes/hnu)*1e-7
;
end
;-------------------------------------------------------------------------------
function vanboekel,starid,lammda
;
; Return flux [Jy?] and lammda [m]
;
forward_function mrdfits
;
common StarBase,startable,notes
;
; Save any existing startable
if n_elements(startable) eq 0 then get_startable,starid
table=startable
;
; Get the information for the star
index=where(startable.starid eq starid,count)
if count eq 0 then begin
	get_startable,starid
	index=0
endif
type=startable[index[0]].type
class=startable[index[0]].class
;
xdrfile=!atmospheres_dir+'proprietary/vanboekel/'+'vBoekelDatabase.xdr'
file=file_search(xdrfile) & file=file[0]
if strlen(file) ne 0 then restore,file else begin
	fitsfile=!atmospheres_dir+'proprietary/vanboekel/'+'vBoekelDatabase.fits'
	fits_open,fitsfile,fcb
	index=where(fcb.extname eq 'SPECTROPHOTOMETRY',n)
	stars=strarr(n)
	fits_close,fcb
;
	d=mrdfits(fitsfile,index[0],header,/silent)
	wave=fltarr(n_elements(d.wavelength),n)
	flux=fltarr(n_elements(d.flux),n)
;
	for i=0,n-1 do begin
		d=mrdfits(fitsfile,index[i],header,/silent)
		stars[i]=cri_vlti(fitshparse(header,'NAME'))
		wave[*,i]=d.wavelength
		flux[*,i]=d.flux
	endfor
;
	get_startable,stars
	for i=0,n-1 do begin
		startable[i].wvl=wave[*,i]
		startable[i].sed=flux[*,i]
	endfor
	save,startable,filename=xdrfile
endelse
;
index=where(startable.starid eq starid,count)
if count eq 1 then begin
	print,'Found star in data base.'
	lammda=startable[index].wvl
	fluxes=startable[index].sed
endif else begin
;
	spec_parse,startable.spectrum,types,classes,t2,c2
;
; 	Find closest match in type for this class
	index=where(classes eq class,count)
	if count eq 0 then begin
		class=' class '+string(class,format='(i2)')+' ('+starid+')!'
		print,'Warning: substituting 5 (V) for unknown'+class
		class=5
		index=where(classes eq class)
	endif
;
	dt=abs(types[index]-type)
	jndex=where(dt eq min(dt)) & jndex=jndex[0]
	print,'Star not found, closest match: '+startable[jndex].starid
	lammda=startable[jndex].wvl
	fluxes=startable[jndex].sed
endelse
;
; Restore original startable, if needed
if n_elements(table) ne 0 then startable=table
;
; Return SED
return,fluxes
;
end
;-------------------------------------------------------------------------------
